/* drops.hpp
 * 
 * Copyright (C) 2010 Sylwester Arabas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#ifndef DROPS_HPP
#  define DROPS_HPP

#include <boost/units/systems/si.hpp>
#include <boost/units/dimensionless_quantity.hpp>
#include <boost/units/cmath.hpp>
using namespace boost::units;

#  include <cvodes/cvodes.h>
#  include <cvode/cvode_direct.h>

class root
{
  public: virtual ~root() {}; // to enforce execution of child-class destructors
};

class Updraft : public root
{
  public: virtual quantity<si::velocity> U(quantity<si::time>) = 0;
  public: virtual quantity<si::time> t(quantity<si::length> alt) = 0;
};

#  include "constants.hpp"

class SpectraMemLayout : public root
{
  public: virtual size_t length() = 0;
  public: virtual size_t max_length() = 0;

  public: virtual size_t n_specs() = 0;
  public: virtual size_t n_bins(size_t spec) = 0;

  public: virtual size_t r_l_ix(size_t spec, size_t bin) = 0;
  public: virtual size_t r_r_ix(size_t spec, size_t bin) = 0;
  public: virtual size_t N_ix(size_t spec, size_t bin) = 0;
  public: virtual size_t To_l_ix(size_t spec, size_t bin) = 0;
  public: virtual size_t To_r_ix(size_t spec, size_t bin) = 0;
  public: virtual size_t r_d_l_ix(size_t spec, size_t bin) = 0;
  public: virtual size_t r_d_r_ix(size_t spec, size_t bin) = 0;

  public: virtual size_t bin_r(size_t spec, size_t bin) = 0;

  public: virtual void split(size_t spec, size_t bin, size_t into) = 0;
};

class Solute : public root
{
  public: virtual quantity<si::dimensionless> i() = 0; // van 't Hoff factor
  public: virtual quantity<si::mass_density> rho_s() = 0; // density of pure solute
  public: virtual quantity<si::constants::codata::mass_over_amount> M_s() = 0; // molar mass of the solute
  public: virtual quantity<si::dimensionless> kappa() = 0; // Kohler-kappa model parameter
};

class ModelParams : public constants
{
  public: SpectraMemLayout *ML; 
  public: vector<Solute*> *S;
  public: N_Vector pp; // N, r_d_l, r_d_r
  public: void *cvode_mem;
};

class InitpTq : public root
{
  public: virtual quantity<si::pressure> getPressure() = 0;
  public: virtual quantity<si::temperature> getTemperature() = 0;
  public: virtual quantity<si::dimensionless> getSpecificHumidity() = 0;
};

class InitSpectrum : public root
{
  private: bool dryNotWet;
  public: bool isDryNotWet() { return dryNotWet; }
  public: InitSpectrum(bool dryNotWet_ ) { dryNotWet = dryNotWet_; }
  public: virtual quantity<concentration_density> n_n(quantity<si::length> r) = 0;
};

class InitBinLayout : public root
{
  public: virtual quantity<si::length> getLeftEdge(size_t n) = 0;
  public: virtual quantity<si::length> getRightEdge(size_t n) = 0;
  public: virtual bool binNeedsSplitting(
    quantity<si::length> leftEdge, quantity<si::length> rightEdge, 
    quantity<si::length> lastLeftEdge, quantity<si::length> lastRightEdge,
    quantity<specific_concentration> specConc, size_t binNumber
  ) = 0;
  public: virtual size_t splitInto(quantity<specific_concentration> specConc) = 0;
  public: virtual size_t numOfBins() = 0;
};

class Solver : public root
{
  public: virtual void run() = 0;
  public: virtual void postinit() = 0;
};

class Tolerances : public root
{
  public: virtual void setTolerances(void *cvode_mem) = 0;
};

class InBinSpectrum : public root
{
  public: virtual quantity<si::area> kappa(quantity<si::length> a, quantity<si::length> b) = 0;
  public: virtual quantity<si::length> d_kappa__d_a(quantity<si::length> a, quantity<si::length> b) = 0;
  public: virtual quantity<si::length> d_kappa__d_b(quantity<si::length> a, quantity<si::length> b) = 0;
};

class SatVapPresMltplr : public root
{
  public: virtual quantity<si::dimensionless> p_v_s_r__p_v_s_infty(
    quantity<si::temperature>, quantity<si::length> radius, quantity<si::length> r_d, Solute *solute
  ) = 0;
  public: virtual quantity<si::length> r_eq(
    quantity<si::dimensionless> svpm_eq, quantity<si::temperature>, quantity<si::length> r_d, Solute *solute
  ) = 0; 
  public: virtual quantity<si::length> rd_eq(
    quantity<si::dimensionless> svpm_eq, quantity<si::temperature>, quantity<si::length> r_w, Solute *solute
  ) = 0; 
  public: virtual quantity<inverse_length> d_mltplr__d_r(
    quantity<si::temperature>, quantity<si::length> radius, quantity<si::length> r_d, Solute *solute
  ) = 0;
};

class DiffCoeffMltplr : public root
{
  public: virtual quantity<si::dimensionless> D_transition__D_continuum(
    quantity<si::dimensionless> knudsen
  ) = 0;
};

class DropGrowthEq : public root
{
  public: virtual quantity<si::velocity> dr__dt(
    quantity<si::length> r, quantity<si::length> r_d, quantity<si::temperature> T, quantity<si::temperature> T_w, 
    quantity<si::pressure> p, quantity<si::dimensionless> q_v, Solute* solute, 
    DiffCoeffMltplr* dcmv, DiffCoeffMltplr* dcmh, SatVapPresMltplr* svpm
  ) = 0;
};

class Model : public root
{
  // TODO: move into MemLayout?
  public: virtual size_t getStateVectorIndexPressure() = 0;
  public: virtual size_t getStateVectorIndexTemperature() = 0;
  public: virtual size_t getStateVectorIndexSpecificHumidity() = 0;
  public: virtual size_t getStateVectorIndexRadii() = 0;
  public: virtual bool isBulk() = 0;

  public: virtual ModelParams* getParams(Updraft*, InBinSpectrum*, SpectraMemLayout*, 
    SatVapPresMltplr*, DiffCoeffMltplr*, DiffCoeffMltplr*, vector<Solute*>*, DropGrowthEq*) = 0;
  public: virtual CVRhsFn getODERhsFnPtr() = 0;
  public: virtual CVDlsDenseJacFn getODEJacFnPtr() = 0;
};

class Output : public root
{
  public: virtual void head() = 0;
  public: virtual void record(quantity<si::time>, N_Vector x, N_Vector p) = 0;
  public: virtual void foot(quantity<si::time>, N_Vector x, N_Vector p) = 0;
};

#  include "sundials.hpp"
#  include "gnuplot.hpp"

#endif
