/* dropc.cpp
 * 
 * Copyright (C) 2010 Sylwester Arabas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include <exception>
#include <iostream>
#include <vector>
using namespace std;

#define msgprefix "-- calc: "
#include "drops.hpp"

#include "ModelPseudoAdiabaticBulk.hpp"
#include "ModelPseudoAdiabaticMovingSectional.hpp"

#include "SolverCVODESAdamsFunctional.hpp"
#include "SolverCVODESBDFNewton.hpp"

#include "OutputGnuplotStuve.hpp"
#include "OutputGnuplotOneLine.hpp"
#include "OutputGnuplotSpecEvol.hpp"

#include "UpdraftSteady.hpp"
#include "UpdraftBells.hpp"

#include "InitpTqCmdLine.hpp"
#include "InitpTqCmdLinepTRH.hpp"

#include "InBinSpectrumConstNr.hpp"

#include "InitSpectrumWhitby1978.hpp"
#include "InitSpectrumJaenicke1993.hpp"
#include "InitSpectrumODowd1997.hpp"
#include "InitSpectrumFromFile.hpp"

#include "InitBinLayoutLogLinear.hpp"
#include "InitBinLayoutInstrument.hpp"

#include "SpectraMemLayoutVariable.hpp"

#include "SatVapPresMltplrKelvin.hpp"
#include "SatVapPresMltplrUnity.hpp"
#include "SatVapPresMltplrRaoult.hpp"
#include "SatVapPresMltplrRaoultKelvin.hpp"
#include "SatVapPresMltplrRaoultPetters_2007.hpp"

#include "DiffCoeffMltpltFuchsSutugin.hpp"
#include "DiffCoeffMltpltUnity.hpp"

#include "TolerancesUniform.hpp"

#include "Solutes.hpp"

#include "DropGrowthEqFick.hpp"
#include "DropGrowthEqMaxwellMason.hpp"

#include <string>

#undef msgprefix
#define msgprefix "-- init: "

template<typename T> void str2num(char* str, T* num)
{
  stringstream ss;
  ss.exceptions(stringstream::failbit | stringstream::badbit);
  ss << str;
  try 
  {
    ss >> *num;
  }
  catch (...)
  {
    cerr << msgprefix << "failed to parse argument into a number: " << str << endl;
    throw exception();
  }
}

realtype str2real(char* str) 
{
  realtype val;
  str2num(str, &val); 
  return val; 
}

int str2int(char* str) 
{ 
  int val;
  str2num(str, &val); 
  return val; 
}

int str2bool(char* str, char* t = "t", char* f = "f")
{
  if (string(str) == t) return true;
  if (string(str) == f) return false;
  cerr << msgprefix << "failed to parse argument into a boolean value: " << str << endl;
  throw exception();
}

int main(int argc, char *argv[]) 
{

  try 
  {
    
    cerr << msgprefix << "parsing command-line options..." << endl;

    if (argc == 1) 
    {
      cerr << msgprefix << "to get help, try: " << argv[0] << " --help " << endl;
      throw exception();
    } 

    size_t model_name = 0, solver_name = 0, output_name = 0, updraft_name = 0, 
      initptq_name = 0, memlayout_name = 0, initspec_names[2] = {0, 0}, specprop_names[2] = {0, 0},
      svpmltp_name = 0, dcmltplr_mass_name = 0, dcmltplr_heat_name = 0, initbinl_name = 0, 
      tol_name = 0, stop_name = 0;
    vector<bool> parsed(argc);
 
    // first walk through the options to find solver/model/output/... choices
#define arg string(argv[a])
#define handle(opt_long, name) \
if (arg == opt_long)  { \
  if (name != 0) \
  { \
    cerr << msgprefix << opt_long << " specified more than once!" << endl; \
    throw exception(); \
  } \
  parsed[a] = true; \
  if (++a == argc) \
  { \
    cerr << msgprefix << opt_long << " needs an argument" << endl; \
    throw exception(); \
  } \
  parsed[a] = true; \
  name = a; \
}
    for (size_t a = 1; a < argc; ++a)
    {
      if (arg == "--help" || arg == "-h")
      {
#undef msgprefix
#define msgprefix "-- help: "
        cerr << msgprefix << endl;
        cerr << msgprefix << "drops -- an air-parcel CCN activation model using adaptive MOL" << endl;
        cerr << msgprefix << "         based on the SUNDIALS ODE solvers" << endl;
        cerr << msgprefix << "         by Sylwester Arabas <slayoo@igf.fuw.edu.pl>" << endl;
        cerr << msgprefix << endl;
        cerr << msgprefix << "usage: " << argv[0] << " --model|-m X --solver|-s Y --output|-o Z ..." << endl;
        cerr << msgprefix << endl;
        cerr << msgprefix << "model:" << endl;
        cerr << msgprefix << "- PseudoAdiabaticMovingSectional* Fick* | MaxwellMason" << endl;
        cerr << msgprefix << "stop:" << endl;
        cerr << msgprefix << "- t_max* t[s]*250" << endl;
        cerr << msgprefix << "- alt_max h[m]" << endl;
        cerr << msgprefix << "updraft:" << endl;
        cerr << msgprefix << "- Steady* U[m/s]*.25" << endl;
        cerr << msgprefix << "- SteadyPredefined .1|.25|.5|1|4|5|10" << endl;
        cerr << msgprefix << "- Bells U_max[m/s] period[s]" << endl;
        cerr << msgprefix << "initptq:" << endl;
        cerr << msgprefix << "- CmdLine p[Pa] T[K] q_v[kg/kg]" << endl;
        cerr << msgprefix << "- CmdLinepTRH* p[Pa]*100000 T[K]*280 RH[1]*.99" << endl;
        cerr << msgprefix << "memlayout:"<< endl;
        cerr << msgprefix << "- Variable* max_bins*100 max_mult[1]*2 noise_spec_conc[ug-1]*5" << endl;
        cerr << msgprefix << "- Constant" << endl;

        for (int i = 0; i < 2; ++i)
        {
          cerr << msgprefix << "specprop_" << (i?"1":"0") << ":" << endl;
          if (i != 0) cerr << msgprefix << "- None" << endl;
          cerr << msgprefix << "- Constant* NH42SO4" << (i?"":"*") << " | NaCl" << (i?"*":"") << " | kappa=1.28 | kappa=0.64 | kappa=0.32 | kappa=0.16 | kappa=0.08 | kappa=0.04 | kappa=0.02 | kappa=0.01 | kappa=0.005 dry* | wet" << endl;

          cerr << msgprefix << "initspec_" << (i?"1":"0") << ":" << endl;
          if (i != 0) cerr << msgprefix << "- None" << endl;
          cerr << msgprefix << "- Whitby1978 marine | clean_continental | average_background | urban" << endl;
          cerr << msgprefix << "- Jaenicke1993 polar | background | maritime | remote_continental | desert_dust_storm | rural | urban" << endl;
          cerr << msgprefix << "- ODowd1997* nss_sulphate" << (i?"":"*") << " | sea_salt" << (i?"*":"")  << endl;
          cerr << msgprefix << "- Lognormal r[um] sgma[1] N[cm-3]" << endl;
          cerr << msgprefix << "- FromFile filename" << endl;
        }

        cerr << msgprefix << "svpmltp:" << endl;
        cerr << msgprefix << "- Raoult" << endl;
        cerr << msgprefix << "- RaoultKelvin* brent* | falsepos | bisection relative_tolerance[1]*1e-12 noise_level[1]*1e-24 max_iter[1]*20 Konopka_1996 | Petters_2007*" << endl;
        cerr << msgprefix << "dcmltplr_mass:" << endl;
        cerr << msgprefix << "- FuchsSutugin* .04 | .042 | .1 | .5 | 1*" << endl;
        cerr << msgprefix << "- Unity" << endl;
        cerr << msgprefix << "dcmltplr_heat:" << endl;
        cerr << msgprefix << "- FuchsSutugin* .04 | .042 | .1 | .5 | 1*" << endl;
        cerr << msgprefix << "- Unity" << endl;
        cerr << msgprefix << "initbinl:" << endl;
        cerr << msgprefix << "- LogLinear* init_bins*40 min[m]*1e-9 max[m]*1e-4" << endl;
        cerr << msgprefix << "- Instrument SMPS | OPC | SMPS+OPC | SMPS+gap+OPC" << endl;
        cerr << msgprefix << "solver:" << endl;
        cerr << msgprefix << "- CVODESAdamsFunctional" << endl;
        cerr << msgprefix << "- CVODESBDFNewton* CVDIAG | CVDENSE* | CVDENSE-JAC | CVSPGMR | CVSPBCG | CVSPTFQMR" << endl;
        cerr << msgprefix << "tolerances:" << endl;
        cerr << msgprefix << "- Uniform* relative_tolerance[1]*1e-8 noise_level[1]*0" << endl;
        cerr << msgprefix << "output:" << endl;
        cerr << msgprefix << "- GnuplotStuve* cloud_min[um]*1 cloud_max[um]*25 N_min[ug-1]*50 n_spec[1]*6 div_spec[1]*4 dot_step[1]*50 rec_step[1]*1 range_r_min[um]*1 range_r_max[um]*6 range_delta_t_min[K]*-.05 range_delta_t_max[K]*.125" << endl;
        cerr << msgprefix << "- GnuplotSpecEvol n_spec[1]*20 div_spec[1]*1 rec_step[1]*1" << endl;
        cerr << msgprefix << endl;
        throw exception();
      }
      handle("--model", model_name)
      handle("--dcmltplr_mass", dcmltplr_mass_name)
      handle("--dcmltplr_heat", dcmltplr_heat_name)
      handle("--solver", solver_name)
      handle("--stop", stop_name)
      handle("--tolerances", tol_name)
      handle("--output", output_name)
      handle("--updraft", updraft_name)
      handle("--initptq", initptq_name)
      handle("--memlayout", memlayout_name)
      handle("--specprop_0", specprop_names[0])
      handle("--initspec_0", initspec_names[0])
      handle("--specprop_1", specprop_names[1])
      handle("--initspec_1", initspec_names[1])
      handle("--svpmltp", svpmltp_name)
      handle("--initbinl", initbinl_name)
    }
#undef arg
#undef handle

    if (
      model_name == 0 || 
      dcmltplr_mass_name == 0 ||
      dcmltplr_heat_name == 0 ||
      solver_name == 0 || 
      stop_name == 0 || 
      tol_name == 0 || 
      output_name == 0 ||
      updraft_name == 0 ||
      initptq_name == 0 ||
      memlayout_name == 0 ||
      initspec_names[0] == 0 ||
      specprop_names[0] == 0 ||
      initspec_names[1] == 0 ||
      specprop_names[1] == 0 ||
      svpmltp_name == 0 ||
      initbinl_name == 0
    )
    {
      cerr << msgprefix << "not all options specified!" << endl;
      throw exception();
    }

    // instantiate the chosen solver/model/output objects
#define assureargs(offset, total) \
  for (size_t a = offset + 1; a <= offset + total; ++a) \
  { \
    if (a >= argc || parsed[a]) \
    { \
      cerr << msgprefix << argv[offset - 1] << " " << argv[offset] << " requires " << total << " argument(s)" << endl; \
      throw exception(); \
    } \
    parsed[a] = true; \
  }
#define arg(idx) string(argv[idx])
#define unknown(what) \
  { \
    cerr << msgprefix << "unknown " << #what << " type: " << arg(what##_name) << endl; \
    throw exception(); \
  }

    // -------- -------- -------- -------- -------- -------- //
    // -------- -------- -------- -------- -------- -------- //

    Model *model;
    DropGrowthEq *dropgrowth;
    if (arg(model_name) == "PseudoAdiabaticMovingSectional")
    {
      assureargs(model_name, 1);
      if (string(argv[model_name + 1]) == "Fick") 
      {
        dropgrowth = new DropGrowthEqFick();
      }
      else if (string(argv[model_name + 1]) == "MaxwellMason") 
      {
        dropgrowth = new DropGrowthEqMaxwellMason();
      }
      else 
      {
        cerr << "unknown drop growth equation" << endl;
        throw exception();
      }
      model = new ModelPseudoAdiabaticMovingSectional();
    }
    else unknown(model);

    // -------- -------- -------- -------- -------- -------- //

    Updraft* updraft;
    if (arg(updraft_name) == "Steady" || arg(updraft_name) == "SteadyPredefined")
    {
      assureargs(updraft_name, 1);
      updraft = new UpdraftSteady(str2real(argv[updraft_name + 1]) * si::metres_per_second);
    }
    else if (arg(updraft_name) == "Bells")
    {
      assureargs(updraft_name, 2);
      updraft = new UpdraftBells(
        str2real(argv[updraft_name + 1]) * si::metres_per_second,
        str2real(argv[updraft_name + 2]) * si::seconds
      );
    }
    else unknown(updraft);

    // -------- -------- -------- -------- -------- -------- //

    quantity<si::time> t_max;
    if (arg(stop_name) == "t_max")
    {
      assureargs(stop_name, 1);
      t_max = str2real(argv[stop_name + 1]) * si::seconds;
    }
    else if (arg(stop_name) == "alt_max")
    {
      assureargs(stop_name, 1);
      t_max = updraft->t(str2real(argv[stop_name + 1]) * si::metres);
    }
    else unknown(stop);

    // -------- -------- -------- -------- -------- -------- //

    InitpTq* initptq;
    if (arg(initptq_name) == "CmdLine")
    {
      assureargs(initptq_name, 3);
      initptq = new InitpTqCmdLine(
        str2real(argv[initptq_name + 1]) * si::pascals,
        str2real(argv[initptq_name + 2]) * si::kelvins,
        str2real(argv[initptq_name + 3]) 
      );
    }
    else if (arg(initptq_name) == "CmdLinepTRH")
    {
      assureargs(initptq_name, 3);
      initptq = new InitpTqCmdLinepTRH(
        str2real(argv[initptq_name + 1]) * si::pascals,
        str2real(argv[initptq_name + 2]) * si::kelvins,
        str2real(argv[initptq_name + 3]) 
      );
    }
    else unknown(initptq);

    // -------- -------- -------- -------- -------- -------- //

    size_t n_specs = 2;
    if (arg(initspec_names[0]) == "None") 
    {
      cerr << "initspec[0] must be defined" << endl;
      throw exception();
    }
    if (arg(initspec_names[1]) == "None") --n_specs;

    vector<Solute*>* svec = new vector<Solute*>(n_specs);
    vector<InitSpectrum*>* initspecs = new vector<InitSpectrum*>(n_specs);
    vector<bool>* dryNotWets = new vector<bool>(n_specs);

    for (int i = 0; i < n_specs; ++i) 
    {

      assureargs(specprop_names[i], 2);
      if (arg(specprop_names[i] + 1) == "NH42SO4") svec->at(i) = new SoluteAmmoniumSulphate(); 
      else if (arg(specprop_names[i] + 1) == "NaCl") svec->at(i) = new SoluteSodiumChloride(); 
      else if (arg(specprop_names[i] + 1) == "kappa=1.28") svec->at(i) = new SoluteKappa1_28(); 
      else if (arg(specprop_names[i] + 1) == "kappa=0.64") svec->at(i) = new SoluteKappa0_64(); 
      else if (arg(specprop_names[i] + 1) == "kappa=0.32") svec->at(i) = new SoluteKappa0_32(); 
      else if (arg(specprop_names[i] + 1) == "kappa=0.16") svec->at(i) = new SoluteKappa0_16(); 
      else if (arg(specprop_names[i] + 1) == "kappa=0.08") svec->at(i) = new SoluteKappa0_08(); 
      else if (arg(specprop_names[i] + 1) == "kappa=0.04") svec->at(i) = new SoluteKappa0_04(); 
      else if (arg(specprop_names[i] + 1) == "kappa=0.02") svec->at(i) = new SoluteKappa0_02(); 
      else if (arg(specprop_names[i] + 1) == "kappa=0.01") svec->at(i) = new SoluteKappa0_01(); 
      else if (arg(specprop_names[i] + 1) == "kappa=0.005") svec->at(i) = new SoluteKappa0_005(); 
      else 
      { 
        cerr << msgprefix << "unknown Solute:" << arg(specprop_names[i] + 1) << endl; 
        throw exception(); 
      } 
      dryNotWets->at(i) = str2bool(argv[specprop_names[i] + 2], (char*)"dry", (char*)"wet");

      if (arg(initspec_names[i]) == "Whitby1978") 
      {
        assureargs(initspec_names[i], 1);
        initspecs->at(i) = new InitSpectrumWhitby1978(
          argv[initspec_names[i] + 1], dryNotWets->at(i)
        );
      }
      else if (arg(initspec_names[i]) == "Jaenicke1993") 
      {
        assureargs(initspec_names[i], 1);
        initspecs->at(i) = new InitSpectrumJaenicke1993(
          argv[initspec_names[i] + 1], dryNotWets->at(i)
        );
      }
      else if (arg(initspec_names[i]) == "ODowd1997")
      {
        assureargs(initspec_names[i], 1);
        initspecs->at(i) = new InitSpectrumODowd1997(
          argv[initspec_names[i] + 1], dryNotWets->at(i)
        );
      }
      else if (arg(initspec_names[i]) == "Lognormal")
      {
        assureargs(initspec_names[i], 3)
        initspecs->at(i) = new InitSpectrumLognormal(
          str2real(argv[initspec_names[i] + 1]) * 1e-6 * si::metres,
          str2real(argv[initspec_names[i] + 2]),
          str2real(argv[initspec_names[i] + 3]) * pow<-3>(1e-2 * si::meters),
          dryNotWets->at(i)
        );
      }
      else if (arg(initspec_names[i]) == "FromFile")
      {
        assureargs(initspec_names[i], 1)
        initspecs->at(i) = new InitSpectrumFromFile(
          argv[initspec_names[i] + 1], dryNotWets->at(i)
        );
      }
      else 
      {
        cerr << msgprefix << "unknown initspec: " << argv[initspec_names[i]] << endl;
        throw exception();
      }
    }
#undef handle_solute

    // -------- -------- -------- -------- -------- -------- //

    SpectraMemLayout *memlayout;
    realtype max_mult = 1e8;
    // TODO: kludge!
    quantity<specific_concentration> noise_spec_conc = -1 / si::kilogrammes;
    size_t n_bins, max_bins = -1;

    if (arg(memlayout_name) == "Variable")
    {
      assureargs(memlayout_name, 3);
      max_bins = str2int(argv[memlayout_name + 1]);
      max_mult = str2real(argv[memlayout_name + 2]);
      noise_spec_conc = str2real(argv[memlayout_name + 3]) / si::kilogrammes * 1e6;
    }
    else if (arg(memlayout_name) == "Constant")
    {
      assureargs(memlayout_name, 0);
    }
    else unknown(memlayout);

    // -------- -------- -------- -------- -------- -------- //

    // assumes the same number of bins in each spectrum
    InitBinLayout *initbinl;
    if (arg(initbinl_name) == "LogLinear")
    {
      assureargs(initbinl_name, 3);
      initbinl = new InitBinLayoutLogLinear(
        str2int(argv[initbinl_name + 1]), 
        str2real(argv[initbinl_name + 2]) * si::metres, 
        str2real(argv[initbinl_name + 3]) * si::metres,
        max_mult, 
        noise_spec_conc
      ); 
    }
    else if (arg(initbinl_name) == "Instrument")
    {
      assureargs(initbinl_name, 1);
      initbinl = new InitBinLayoutInstrument(
        string(argv[initbinl_name + 1]),
        max_mult, 
        noise_spec_conc
      );
    }
    else unknown(initbinl);

    if (max_bins == -1) max_bins = initbinl->numOfBins();
    memlayout = new SpectraMemLayoutVariable(n_specs, initbinl->numOfBins(), max_bins);

    // -------- -------- -------- -------- -------- -------- //

    Output *output;
    if (arg(output_name) == "GnuplotStuve")
    {
      assureargs(output_name, 11);
      output = new OutputGnuplotStuve(model, initptq, memlayout, updraft,
        str2real(argv[output_name + 1]) * 1e-6 * si::metres,
        str2real(argv[output_name + 2]) * 1e-6 * si::metres,
        str2real(argv[output_name + 3]) / si::kilogrammes * 1e6,
        str2int(argv[output_name + 4]),
        str2real(argv[output_name + 5]),
        str2int(argv[output_name + 6]),
        str2int(argv[output_name + 7]),
        str2real(argv[output_name + 8]) * 1e-6 * si::metres,
        str2real(argv[output_name + 9]) * 1e-6 * si::metres,
        str2real(argv[output_name + 10]) * si::kelvins,
        str2real(argv[output_name + 11]) * si::kelvins
      );
    } 
    else if (arg(output_name) == "GnuplotOneLine")
    {
      assureargs(output_name, 3);
      output = new OutputGnuplotOneLine(model, initptq, memlayout, updraft,
        str2real(argv[output_name + 1]) * 1e-6 * si::metres,
        str2real(argv[output_name + 2]) * 1e-6 * si::metres,
        str2real(argv[output_name + 3]) / si::kilogrammes * 1e6
      );
    }
    else if (arg(output_name) == "GnuplotSpecEvol")
    {
      assureargs(output_name, 3);
      output = new OutputGnuplotSpecEvol(model, initptq, memlayout, updraft,
        str2int(argv[output_name + 1]),
        str2real(argv[output_name + 2]),
        str2int(argv[output_name + 3])
      );
    } 
    else unknown(output);

    // -------- -------- -------- -------- -------- -------- //

    InBinSpectrum *inbinspec;
    inbinspec = new InBinSpectrumConstNr();

    // -------- -------- -------- -------- -------- -------- //

    DiffCoeffMltplr *dcmltplr_mass;
    if (arg(dcmltplr_mass_name) == "Unity")
    {
      dcmltplr_mass = new DiffCoeffMltplrUnity();
    }
    else if (arg(dcmltplr_mass_name) == "FuchsSutugin")
    {
      assureargs(dcmltplr_mass_name, 1);
      dcmltplr_mass = new DiffCoeffMltplrFuchsSutugin(
        str2real(argv[dcmltplr_mass_name + 1])
      );
    }
    else unknown(dcmltplr_mass);

    DiffCoeffMltplr *dcmltplr_heat;
    if (arg(dcmltplr_heat_name) == "Unity")
    {
      dcmltplr_heat = new DiffCoeffMltplrUnity();
    }
    else if (arg(dcmltplr_heat_name) == "FuchsSutugin")
    {
      assureargs(dcmltplr_heat_name, 1);
      dcmltplr_heat = new DiffCoeffMltplrFuchsSutugin(
        str2real(argv[dcmltplr_heat_name + 1])
      );
    }
    else unknown(dcmltplr_heat);

    // -------- -------- -------- -------- -------- -------- //

    SatVapPresMltplr *svpmltp;
    if (arg(svpmltp_name) == "Unity")
    {
      svpmltp = new SatVapPresMltplrUnity();
    }
    else if (arg(svpmltp_name) == "Kelvin")
    {
      svpmltp = new SatVapPresMltplrKelvin();
    }
    else if (arg(svpmltp_name) == "Raoult")
    {
      svpmltp = new SatVapPresMltplrRaoult();
    }
    else if (arg(svpmltp_name) == "RaoultKelvin")
    {
      assureargs(svpmltp_name, 5);
      
      SatVapPresMltplr* raoult;
      if (arg(svpmltp_name + 5) == "Konopka_1996") raoult = new SatVapPresMltplrRaoult();
      else if (arg(svpmltp_name + 5) == "Petters_2007") raoult = new SatVapPresMltplrRaoultPetters_2007();
      else 
      {
        cerr << "unknown Roult term form!" << endl;
        throw exception();
      }

      svpmltp = new SatVapPresMltplrRaoultKelvin(
        string(argv[svpmltp_name + 1]),
        str2real(argv[svpmltp_name + 2]), 
        str2real(argv[svpmltp_name + 3]),
        str2int(argv[svpmltp_name + 4]),
        raoult
      );
    }
    else unknown(svpmltp);

    // -------- -------- -------- -------- -------- -------- //

    Tolerances *tol;
    if (arg(tol_name) == "Uniform")
    {
      assureargs(tol_name, 2)
      tol = new TolerancesUniform(
        str2real(argv[tol_name + 1]), 
        str2real(argv[tol_name + 2])
      );
    }
    else unknown(tol);

    // -------- -------- -------- -------- -------- -------- //

    ModelParams* params = model->getParams(updraft, inbinspec, memlayout, svpmltp, 
      dcmltplr_mass, dcmltplr_heat, svec, dropgrowth);

    Solver *solver;
    if (arg(solver_name) == "CVODESBDFNewton") 
    {
      assureargs(solver_name, 1)
      solver = new SolverCVODESBDFNewton(model, output, params, initptq, initspecs, 
        initbinl, memlayout, svpmltp, tol, t_max,
        string(argv[solver_name + 1])
      );
    }
    else if (arg(solver_name) == "CVODESAdamsFunctional") 
    {
      solver = new SolverCVODESAdamsFunctional(model, output, params, initptq, initspecs, 
        initbinl, memlayout, svpmltp, tol, t_max);
    }
    else unknown(solver);

    // -------- -------- -------- -------- -------- -------- //
    // -------- -------- -------- -------- -------- -------- //

#undef unknown
#undef arg
#undef assureargs

    for (size_t opt = 1; opt < parsed.size(); opt++)
    {
      if (!parsed[opt])
      {
        cerr << msgprefix << "unparsed option: " << argv[opt] << endl; 
        throw exception();
      }
    }

#undef msgprefix
#define msgprefix "-- calc: "

    // do the number crunching
    solver->run();

#undef msgprefix
#define msgprefix "-- exit: "

    // cleanup TODO: update to cover all objects... 
    //               perhaps better with auto_ptr?... 
    //               but who cares just before exit...
/*
    delete initptq;
    delete params;
    delete solver;
    delete model;
    delete output;
    delete updraft;
*/

  }
  catch (...)
  {
    cerr << msgprefix << "KO" << endl;
    exit(EXIT_FAILURE);
  }

  cerr << msgprefix << "OK" << endl;
  exit(EXIT_SUCCESS);

}
