/* constants.hpp
 * 
 * Copyright (C) 2010 Sylwester Arabas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#ifndef CONSTANTS_HPP
#  define CONSTANTS_HPP

// for getting DWPTELEV by inverting p_vs
#  include <gsl/gsl_errno.h>
#  include <gsl/gsl_roots.h>

#include <boost/units/systems/si/codata/physico-chemical_constants.hpp>
#include <boost/units/systems/si.hpp>

typedef divide_typeof_helper<
    si::constants::codata::energy_over_temperature_amount,
    si::constants::codata::mass_over_amount
  >::type energy_over_temperature_mass;

typedef divide_typeof_helper<
    si::energy,
    si::mass
  >::type energy_over_mass;

typedef multiply_typeof_helper<
    si::velocity,
    si::length
  >::type diffusivity;

typedef multiply_typeof_helper<
    si::time,
    si::area
  >::type time_area;

typedef divide_typeof_helper<
    si::mass,
    time_area
  >::type mass_flux;

typedef multiply_typeof_helper<
    energy_over_mass,
    mass_flux
  >::type energy_flux;

typedef divide_typeof_helper<
    si::temperature,
    si::length
  >::type temperature_gradient;

typedef divide_typeof_helper<
    energy_flux,
    temperature_gradient
  >::type thermal_conductivity;

typedef divide_typeof_helper<
    si::dimensionless,
    si::mass
  >::type specific_concentration;

typedef divide_typeof_helper<
    specific_concentration,
    si::length
  >::type specific_concentration_density;

typedef divide_typeof_helper<
    si::dimensionless,
    si::volume
  >::type concentration;

typedef divide_typeof_helper<
    concentration,
    si::length
  >::type concentration_density;

typedef divide_typeof_helper<
    si::dimensionless,
    si::mass_density
  >::type inverse_mass_density;

typedef divide_typeof_helper<
    inverse_mass_density,
    si::time
  >::type inverse_mass_density_over_time;

typedef divide_typeof_helper<
    si::length,
    si::mass
  >::type length_over_mass;

typedef divide_typeof_helper<
    si::area,
    si::mass
  >::type area_over_mass;

typedef divide_typeof_helper<
    si::volume,
    si::mass
  >::type volume_over_mass;

typedef divide_typeof_helper<
    si::dimensionless,
    si::length
  >::type inverse_length;

typedef divide_typeof_helper<
    si::dimensionless,
    si::area
  >::type inverse_area;

typedef divide_typeof_helper<
    si::pressure,
    si::temperature
  >::type pressure_over_temperature;

#  define self constants
class self : public root
{
 
  // water triple point parameters
  public: static quantity<si::temperature> T_0;
  public: static quantity<si::pressure> p_0;
  public: static quantity<energy_over_mass> l_v_0;
  public: static quantity<energy_over_temperature_mass> c_p_w;
  
  // dry air parameters
  public: static quantity<si::constants::codata::mass_over_amount> M_d;
  public: static quantity<energy_over_temperature_mass> c_p_d;

  // water vapour parameters
  public: static quantity<si::constants::codata::mass_over_amount> M_v;
  public: static quantity<energy_over_temperature_mass> c_p_v;

  // diffusion coefficient of water vapour in air at O C, 1000 hPa
  public: static quantity<diffusivity> D_0;

  // coefficient of thermal condictivity of air at O C, 1000 hPa
  public: static quantity<thermal_conductivity> K_0;

  // water density at 0 C, 1000 hPa
  public: static quantity<si::mass_density> rho_w_0;

  // water surface tension at ... TODO
  public: static quantity<si::surface_tension> sigma_w_0;

  // Earth parameters
  public: static quantity<si::acceleration> g_0;

  // helper derived quantities
  public: static quantity<si::dimensionless> epsilon;
  public: static quantity<energy_over_temperature_mass> R_d, R_v;

  // inline (because defined in a header file) functions
  public: static quantity<energy_over_temperature_mass> R(quantity<si::dimensionless> q_v)
  {   
    return q_v * R_v + (1 - q_v) * R_d;
  }   

  public: static quantity<energy_over_temperature_mass> c_p(quantity<si::dimensionless> q_v)
  {   
    return q_v * c_p_v + (1 - q_v) * c_p_d;
  }   

  public: static quantity<energy_over_mass> l_v(quantity<si::temperature> T)
  {
    return l_v_0 + (c_p_v - c_p_w) * (T - T_0);
  }

  // solution to the C-C eq. with c_p_w and c_p_v constant
  public: static quantity<si::pressure> p_v_s(quantity<si::temperature> T)
  {
    return p_0 * exp(
      (l_v_0 + (c_p_w - c_p_v) * T_0) / R_v * (1. / T_0 - 1. / T)
      - (c_p_w - c_p_v) / R_v * log(T / T_0)
    );
  }

  // the C-C eq.
  public: static quantity<pressure_over_temperature> dp_v_s__dT(quantity<si::temperature> T, 
    quantity<si::pressure> p_v_s)
  {
    return p_v_s * l_v(T) * pow<-2>(T) / R_v;
  } 

  // e.g. for calculating dew-point
  private: static double p_v_s_inv_f(double T_, void *p_v_)
  {
    quantity<si::temperature> T = T_ * si::kelvins;
    quantity<si::pressure> p_v = *((realtype*)p_v_) * si::pascals;
    return (p_v_s(T) - p_v) / si::pascals;
  }
  private: static double p_v_s_inv_df(double T_, void*)
  {
    quantity<si::temperature> T = T_ * si::kelvins;
    return (dp_v_s__dT(T, p_v_s(T))) / si::pascals * si::kelvins;
  }
  private: static void p_v_s_inv_fdf(double T_, void *p_v_, double *f, double *df)
  {
    quantity<si::temperature> T = T_ * si::kelvins;
    quantity<si::pressure> p_v = *((realtype*)p_v_) * si::pascals;
    quantity<si::pressure> p_v_s_ = p_v_s(T);
    *f = (p_v_s_ - p_v) / si::pascals;
    *df = (dp_v_s__dT(T, p_v_s_)) / si::pascals * si::kelvins;
  }
  public: static quantity<si::temperature> p_v_s_inv(quantity<si::pressure> p_v, 
    realtype reltol, realtype abstol, int max_iter)
  {
    gsl_function_fdf rtslvr_struct;
    rtslvr_struct.f = self::p_v_s_inv_f; 
    rtslvr_struct.df = self::p_v_s_inv_df;
    rtslvr_struct.fdf = self::p_v_s_inv_fdf;
    realtype p_v_ = p_v / si::pascals;
    rtslvr_struct.params = &p_v_;
    quantity<si::temperature> p_v_s_inv_approx = pow<-1>(1./T_0 - R_v/l_v_0*log(p_v/p_0)); // l_v=l_v_0

    gsl_root_fdfsolver *rtslvr = gsl_root_fdfsolver_alloc(gsl_root_fdfsolver_steffenson);
    double root = p_v_s_inv_approx / si::kelvins;
    gsl_root_fdfsolver_set(rtslvr, &rtslvr_struct, root);
    int status, iter = 0;
    double root_last;
    do
    {
      ++iter;
      if (gsl_root_fdfsolver_iterate(rtslvr) != GSL_SUCCESS)
      {
        cerr << msgprefix << "GSL root solver (p_v_s_inv) failed to iterate" << endl;
        throw exception();
      }
      root_last = root;
      root = gsl_root_fdfsolver_root(rtslvr);
      status = gsl_root_test_delta(root, root_last, abstol, reltol);  
    }
    while (status == GSL_CONTINUE && iter < max_iter);
    if (status != GSL_SUCCESS)
    {
      cerr << msgprefix << "GSL root solver (p_v_s_inv) failed to converge" << endl;
      throw exception(); 
    }
    gsl_root_fdfsolver_free(rtslvr);
    return root * si::kelvins;
  }

  public: static quantity<si::dimensionless> q_v_s(quantity<si::temperature> T, quantity<si::pressure> p)
  {
    return 1. / (1 - 1./epsilon * (1 - p / p_v_s(T)));
  }

  // Knudsen number for condensation (eq. 6.6 in Williams/Loyalka)
  public: static quantity<si::dimensionless> Kn_D(
    quantity<si::length> r_w, quantity<si::temperature> T_w
  )
  {
    return 2. * D_0 * sqrt(1. / 2. / R_v / T_w) / r_w;
  }
 
  // Knudsen number for heat transfer
  public: static quantity<si::dimensionless> Kn_K(
    quantity<si::length> r_w, quantity<si::temperature> T, quantity<si::pressure> p
  )
  {
    return 4. / 5. * K_0 * T / p * sqrt(1. / 2. / R_d / T) / r_w;
  }


};
#  undef self

// numeric values
quantity<si::constants::codata::mass_over_amount> constants::M_v(0.01802 * si::kilograms / si::moles);
quantity<si::constants::codata::mass_over_amount> constants::M_d(0.02896 * si::kilograms / si::moles);
quantity<si::temperature> constants::T_0(273.16 * si::kelvins);
quantity<si::pressure> constants::p_0(611.73 * si::pascals);
quantity<energy_over_mass> constants::l_v_0(2.5e6 * si::joules / si::kilograms);
quantity<energy_over_temperature_mass> constants::c_p_v(1850 * si::joules / si::kilograms / si::kelvins);
quantity<energy_over_temperature_mass> constants::c_p_d(1005 * si::joules / si::kilograms / si::kelvins);
quantity<energy_over_temperature_mass> constants::c_p_w(4218 * si::joules / si::kilograms / si::kelvins);
quantity<si::acceleration> constants::g_0(9.80665 * si::metres_per_second_squared);
quantity<diffusivity> constants::D_0(2.21e-5 * si::metres_per_second * si::metres);
quantity<thermal_conductivity> constants::K_0(2.4e-2 * si::joules / si::metres / si::seconds / si::kelvins);
quantity<si::mass_density> constants::rho_w_0(1000 * si::kilogrammes_per_cubic_metre);
quantity<si::surface_tension> constants::sigma_w_0(72 * 1e-3 * si::newtons_per_meter);

// derived quantities
quantity<energy_over_temperature_mass> constants::R_v(si::constants::codata::R / constants::M_v);
quantity<energy_over_temperature_mass> constants::R_d(si::constants::codata::R / constants::M_d);
quantity<si::dimensionless> constants::epsilon(constants::M_v / constants::M_d);

#endif
