/* SpectraMemLayoutVariable.hpp
 * 
 * Copyright (C) 2010 Sylwester Arabas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#ifndef SPECTRA_MEM_LAYOUT_VARIABLE_HPP
#  define SPECTRA_MEM_LAYOUT_VARIABLE_HPP

#  include <stack>

#  define self SpectraMemLayoutVariable
#  define parent SpectraMemLayout
class self : public parent
{

  private: vector< vector<size_t> > offsets; // one offset might be negative!
  private: vector< vector<int> > neighbours;
  private: vector<size_t> free;
  private: size_t storage_space;

#  define idx_N 0
#  define idx_r_d_l 1

#  define idx_To_l 0
#  define idx_r_l 1

// assumes p- and x-vectors have the same block size
#  define block_size 2

  public: size_t max_length() { return storage_space; }
  public: size_t length() { return storage_space - free.size() * block_size; }

  public: size_t n_specs() { return offsets.size(); }
  public: size_t n_bins(size_t spec) { return offsets[spec].size() - 1; }

  // TODO: n_bins_max is specified here per spectrum, a global value would be better perhaps?
  public: self(size_t n_specs, size_t n_bins_init, size_t n_bins_max) 
  {
    storage_space = n_specs * (n_bins_max + 1 ) * block_size; 
    free.reserve(storage_space / block_size); // just an approximation
    for (int i = storage_space - block_size; i >= 0; i -= block_size) free.push_back(i);
    assert(length() == 0);

    offsets.resize(n_specs);
    neighbours.resize(n_specs);
    for (size_t s = 0; s < n_specs; ++s) 
    {
      offsets[s].reserve(n_bins_max); // just an approximation
      neighbours[s].reserve(n_bins_max); // just an approximation
      for (size_t i = 0; i < n_bins_init + 1; ++i) 
      {
        offsets[s].push_back(free.back());
        if (i > 0) neighbours[s].push_back(offsets[s].size() - 1);
        free.pop_back();
      }
      neighbours[s].push_back(-1);
    }
  }

  public: void split(size_t spec, size_t bin, size_t into)
  {
    assert(spec < n_specs());
    assert(into > 1);

    if (free.size() == 0) 
    {
      cerr << msgprefix << "storage space exhausted!" << endl;
      throw exception();
    }

    offsets[spec].push_back(free.back());              // adding a new block to the spectrum
    free.pop_back();                                   // removing the freshly-added block from the free stack
    neighbours[spec].push_back(neighbours[spec][bin]); // defining the neighbour of the freshly-added bin
    neighbours[spec][bin] = offsets[spec].size() - 1;  // altering the neighbour of the initial bin

    if (into > 2) split(spec, neighbours[spec][bin], into - 1); // recursively adding subsequent bins
  }

  public: size_t bin_r(size_t spec, size_t bin)
  {
    return neighbours[spec][bin];
  }

  public: size_t N_ix(size_t spec, size_t bin) 
  { 
    assert(bin >= 0);
    assert(bin_r(spec, bin) != -1);
    return offsets[spec][bin] + idx_N; 
  }

  public: size_t r_d_l_ix(size_t spec, size_t bin) 
  { 
    assert(bin >= 0);
    return offsets[spec][bin] + idx_r_d_l; 
  }

  public: size_t r_d_r_ix(size_t spec, size_t bin) 
  { 
    assert(bin >= 0);
    return offsets[spec][bin_r(spec, bin)] + idx_r_d_l; 
  }

  public: size_t r_l_ix(size_t spec, size_t bin) 
  { 
    assert(bin >= 0);
    return offsets[spec][bin] + idx_r_l; 
  }

  public: size_t To_l_ix(size_t spec, size_t bin) 
  { 
    assert(bin >= 0);
    return offsets[spec][bin] + idx_To_l; 
  }

  public: size_t r_r_ix(size_t spec, size_t bin) 
  { 
    assert(bin >= 0);
    return offsets[spec][bin_r(spec, bin)] + idx_r_l; 
  }

  public: size_t To_r_ix(size_t spec, size_t bin) 
  { 
    assert(bin >= 0);
    return offsets[spec][bin_r(spec, bin)] + idx_To_l; 
  }

#  undef idx_r_l
#  undef idx_r_d_l
#  undef idx_To_l
#  undef idx_N
#  undef block_size

};
#  undef parent
#  undef self

#endif
