/* SatVapPresMltplrRaoultKelvin.hpp 
 * 
 * Copyright (C) 2010 Sylwester Arabas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#ifndef SAT_VAP_PRES_MLTPLR_RAOULT_KELVIN_HPP
#  define SAT_VAP_PRES_MLTPLR_RAOULT_KELVIN_HPP

#  include <gsl/gsl_errno.h>
#  include <gsl/gsl_math.h>
#  include <gsl/gsl_roots.h>

#  define self SatVapPresMltplrRaoultKelvin
class self : public SatVapPresMltplr
{
  private: SatVapPresMltplr* raoult;
  private: SatVapPresMltplrKelvin* kelvin; 

  private: const gsl_root_fsolver_type *rtslvr_type;
  private: gsl_root_fsolver *rtslvr;
  private: gsl_function rtslvr_struct;
  private: double rtslvr_relative_tolerance, rtslvr_noise_level;
  private: int rtslvr_max_iter;

  public: self(string type, double relative_tolerance, double noise_level, int max_iter, SatVapPresMltplr* raoult_) 
  {
    raoult = raoult_;
    kelvin = new SatVapPresMltplrKelvin();

    if (type == "brent")
      rtslvr_type = gsl_root_fsolver_brent;
    else if (type == "falsepos")
      rtslvr_type = gsl_root_fsolver_falsepos;
    else if (type == "bisection")
      rtslvr_type = gsl_root_fsolver_bisection;
    else 
    {   
      cerr << msgprefix << "unknown root bracketer: " << type << endl;
      throw exception();
    } 

    rtslvr = gsl_root_fsolver_alloc(rtslvr_type); // TODO: override GSL error handler and handle NULL pointer return
    rtslvr_struct.function = self::rtslvr_function;
    rtslvr_struct.params = new self::rtslvr_params();
    static_cast<rtslvr_params*>(rtslvr_struct.params)->this_ = this;

    rtslvr_relative_tolerance = relative_tolerance; 
    rtslvr_noise_level = noise_level;
    rtslvr_max_iter = max_iter;
  }

  private: class rtslvr_params
  {
    public: quantity<si::temperature> T;
    public: quantity<si::length> r_d;
    public: Solute* solute;
    public: quantity<si::dimensionless> svpm_eq;
    public: self* this_;
  };

  public: quantity<si::dimensionless> p_v_s_r__p_v_s_infty(
    quantity<si::temperature> T, quantity<si::length> r, 
    quantity<si::length> r_d, Solute *solute
  ) 
  {
    return 
      raoult->p_v_s_r__p_v_s_infty(T, r, r_d, solute) *
      kelvin->p_v_s_r__p_v_s_infty(T, r, r_d, solute);
  }

  private: static double rtslvr_function(double r, void* p_)
  {
    self::rtslvr_params* p = static_cast<self::rtslvr_params*>(p_);
    return (
      p->this_->p_v_s_r__p_v_s_infty(p->T, r * si::metres, p->r_d, p->solute) - p->svpm_eq
    );
  }

  public: ~self()
  {
    delete static_cast<self::rtslvr_params*>(rtslvr_struct.params);
    gsl_root_fsolver_free(rtslvr);

    delete raoult;
    delete kelvin;
  }

  public: quantity<si::length> r_eq(
    quantity<si::dimensionless> svpm_eq, quantity<si::temperature> T,
    quantity<si::length> r_d, Solute *solute
  )
  {
    static_cast<rtslvr_params*>(rtslvr_struct.params)->T = T;
    static_cast<rtslvr_params*>(rtslvr_struct.params)->r_d = r_d;
    static_cast<rtslvr_params*>(rtslvr_struct.params)->solute = solute;
    static_cast<rtslvr_params*>(rtslvr_struct.params)->svpm_eq = svpm_eq;

    gsl_root_fsolver_set(rtslvr, &rtslvr_struct,
      r_d / si::metres, raoult->r_eq(svpm_eq, T, r_d, solute) / si::metres);
    int status, iter = 0;
    do
    {
      ++iter;
      if (gsl_root_fsolver_iterate(rtslvr) != GSL_SUCCESS) 
      {
        cerr << msgprefix << "GSL root solver (KelvinRaoult) failed to iterate" << endl;
        throw exception();
      }
      status = gsl_root_test_interval(
        gsl_root_fsolver_x_lower(rtslvr),
        gsl_root_fsolver_x_upper(rtslvr),
        rtslvr_noise_level, rtslvr_relative_tolerance
      );
    }
    while (status == GSL_CONTINUE && iter < rtslvr_max_iter);
    if (status != GSL_SUCCESS) 
    {
      cerr << msgprefix << "GSL root solver (KelvinRaoult) failed to converge" << endl;
      throw exception(); 
    }
    return gsl_root_fsolver_root(rtslvr) * si::metres;
  }

  public: quantity<inverse_length> d_mltplr__d_r(
    quantity<si::temperature> T, quantity<si::length> radius, quantity<si::length> r_d, Solute *solute
  )   
  {
    return (
      raoult->p_v_s_r__p_v_s_infty(T, radius, r_d, solute) * kelvin->d_mltplr__d_r(T, radius, r_d, solute) +
      kelvin->p_v_s_r__p_v_s_infty(T, radius, r_d, solute) * raoult->d_mltplr__d_r(T, radius, r_d, solute)
    );
  }

  public: quantity<si::length> rd_eq(
    quantity<si::dimensionless> svpm_eq, quantity<si::temperature> T,
    quantity<si::length> r_w, Solute *solute
  )
  {
    return raoult->rd_eq(svpm_eq / kelvin->p_v_s_r__p_v_s_infty(T, r_w, r_w, solute), T, r_w, solute);
    //                                                                  ^^^ -> dummy argument
  }

};
#  undef self

#endif
