/* OutputGnuplotStuve.hpp
 * 
 * Copyright (C) 2010 Sylwester Arabas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#ifndef OUTPUT_GNUPLOT_STUVE_HPP
#  define OUTPUT_GNUPLOT_STUVE_HPP

#  define self OutputGnuplotStuve
class self : public OutputGnuplot
{

  protected: quantity<si::length> cloud_min, cloud_max;
  protected: quantity<specific_concentration> N_min;

  // plot ranges etc
  protected: int n_spec, dot_step;
  protected: quantity<si::dimensionless> div_spec;
  protected: quantity<si::length> range_r_min, range_r_max;
  protected: quantity<si::temperature> range_delta_t_min, range_delta_t_max;

  public: self(Model *model_, InitpTq *initptq_, SpectraMemLayout *ml_, Updraft *updraft,
    quantity<si::length> cloud_min_, quantity<si::length> cloud_max_,
    quantity<specific_concentration> N_min_, int n_spec_, quantity<si::dimensionless> div_spec_,
    int dot_step_, size_t rec_step,
    quantity<si::length> range_r_min_, quantity<si::length> range_r_max_,
    quantity<si::temperature> range_delta_t_min_, quantity<si::temperature> range_delta_t_max_
  ) : OutputGnuplot(model_, initptq_, ml_, updraft, rec_step) 
  { 
    cloud_min = cloud_min_;
    cloud_max = cloud_max_;
    N_min = N_min_;
    n_spec = n_spec_;
    div_spec = div_spec_;
    dot_step = dot_step_;
    range_r_min = range_r_min_;
    range_r_max = range_r_max_;
    range_delta_t_min = range_delta_t_min_;
    range_delta_t_max = range_delta_t_max_;
  }


  public: void record(quantity<si::time> t, N_Vector y, N_Vector p)
  {
    ++step;
    if (step - step_last < rec_step) return;
    step_last = step;
    tmpfile_write(t, y, p); 
  }

  public: void head() {}

  public: void foot(quantity<si::time>, N_Vector, N_Vector) 
  {
    lib();
    cout << "set multiplot" << endl;
    cout << "set size .38, .48" << endl;
    cout << "set origin 0, .53" << endl;
    cout << "set title 'Stuve diagram'" << endl;
    cout << "set ylabel 'pressure [hPa]'" << endl;
    cout << "set grid" << endl;
    cout << "set key top left" << endl;
    cout << "yaxis(p) = -p ** (R_d / c_p_d)" << endl;
    cout << "set yrange [yaxis(100000) : yaxis(" << realtype(p_last / si::pascals) << ")]" << endl;

    // ticks
    cout << "set ytics (\"1000\" yaxis(100000)"; 
    for (int i = 900; i; i -= 100) cout << ", \"" << i <<  "\" yaxis(" << 100 * i << ")";
    cout << ")" << endl;
    cout << "set xtics 1 rotate by -45 format '%g K'" << endl;

    // theoretical curves
    cout << "const_th(T, th) = - T / th * (100000)**(R_d / c_p_d)" << endl;
    cout << "const_rs(T, rs) = -(p_s(T) + epsilon * p_s(T) / rs)**(R_d / c_p_d)" << endl;

    // model output
    cout << "plot '-' using " 
      << model->getStateVectorIndexTemperature() + 2
      << ":(yaxis($"
      << model->getStateVectorIndexPressure() + 2
      << ")) with lines lt 1 title 'bulk model' \\" 
      << endl;
    cout << ", '-' using " 
      << model->getStateVectorIndexTemperature() + 2
      << ":(yaxis($"
      << model->getStateVectorIndexPressure() + 2
      << ")) with lines lt 2 title 'sectional model' \\" 
      << endl;

    // dry adiabats
    for (int i = 100; i < 400; i += 10) 
      cout << ", const_th(x, " << i << ") lt 3 t " << (i==100 ? "'dry adiabats'" : "''") << " \\" << endl;
    // one isohume
    cout << ", const_rs(x, qs2rs(" << realtype(initptq->getSpecificHumidity()) << ")) lt 4 t '100% isohume' \\" << endl;

    cout << endl;

    moist_adiabat();
    cout << "e" << endl;
    dataSounding();
    cout << "e" << endl;

    cout << "reset" << endl;
    cout << "set key samplen 1" << endl;
    cout << "set origin .4,.5" << endl;
    cout << "set size .6,.5" << endl;
    cout << "set title 'aerosol/droplet spectrum'" << endl;
    cout << "set logscale xy" << endl;
    cout << "set xtics (1e-9, 10e-9, 100e-9, 1e-6, 10e-6, 100e-6) format '' out nomirror" << endl;
    cout << "set ytics" << endl;
    cout << "set ytics format '%.1e'" << endl;
    cout << "set xrange [10e-10:100e-6]" << endl;
    cout << "set ylabel \"specific concentration density [mg-1 um-1]\"" << endl;
    cout << "set yrange [1e-3 : 1e7]" << endl;
    cout << "plot ";
    for (size_t i = 0; i < ml->n_specs(); ++i)
    {
      if (i != 0) cout << ", ";
      cout << "'-' using 5:($3/($6-$5)*1e-12) with steps t ";
      if (i == 0) cout << "'initial dry radii (a, " << n_bins(0, i) << " bins)'";
      else cout << "'ditto (b, " << n_bins(0, i) << " -> " << n_bins(numRecords() - 1, i) << " bins)'";
      cout << " lt 1 lw " << 3 - 2 * i;

      //cout << ", '-' using 5:($3/($6-$5)*1e-12 * pi/6.*$5**3) axes x1y2 with steps t '";
      //if (i == 0) cout << "initial dry radii (volume, arbitrary unit)";
      //cout << "' lc rgbcolor 'grey' lw " << 3 - 2 * i;

      cout << ", '-' using 2:($3/($4-$2)*1e-12) with steps t '";
      if (i == 0) cout << "initial wet radii (a, " << n_bins(0, i) << " bins)";
      cout << "' lt 2 lw " << 3 - 2 * i;

      cout << ", '-' using 2:($3/($4-$2)*1e-12) with steps t '";
      if (i == 0) cout << "final wet radii (a, " << n_bins(numRecords() - 1, i) << " bins)"; 
      cout << "' lt 3 lw " << 3 - 2 * i;
    }
    cout << endl;

    for (size_t i = 0; i < ml->n_specs(); ++i)
    {
      dataSpectrum(0, i);
      cout << "e" << endl;
      //dataSpectrum(0, i);
      //cout << "e" << endl;
      dataSpectrum(0, i); 
      cout << "e" << endl;
      dataSpectrum(numRecords() - 1, i);
      cout << "e" << endl;
    }

    cout << "reset" << endl;
    cout << "set ylabel 'time [s]'" << endl;
    cout << "set yrange [-" 
      << realtype(times.back() / si::seconds / 40) << ":" 
      << realtype(times.back() / si::seconds) << "]" << endl;
    cout << "set x2tics (\"1nm\" 1e-9, \"10 nm\" 10e-9, \"100 nm\" 100e-9, \"1 um\" 1e-6, \"10 um\" 10e-6, \"100 um\" 100e-6) out" << endl;
    cout << "set xtics out nomirror" << endl;
    cout << "set x2range [10e-10:100e-6]" << endl;
    cout << "set ytics format '%.1e'" << endl;
    cout << "set xlabel 'RH [%]'" << endl;
    cout << "set origin .4,0" << endl;
    cout << "set size .6,.55" << endl;
    cout << "set logscale x2" << endl;
    cout << "set key below reverse Left left" << endl;
    cout << "set arrow from " 
      << "second " << realtype(cloud_min / si::metres) << ",0" << " to " 
      << "second " << realtype(cloud_min / si::metres) << "," << realtype(times.back() / si::seconds) 
      << " nohead lw .5 back" << endl;
    cout << "set arrow from " 
      << "second " << realtype(cloud_max / si::metres) << ",0" << " to " 
      << "second " << realtype(cloud_max / si::metres) << "," << realtype(times.back() / si::seconds) 
      << " nohead lw .5  back" << endl;
    cout << "set arrow from " 
      << "first 100,0" << " to " 
      << "first 100," << realtype(times.back() / si::seconds) 
      << " nohead lw 2 lt 2 front" << endl;
    cout << "set label 'RH=100%' at first 100," << realtype(times.back() / si::seconds) 
      << " rotate by 270 front" << endl;
    cout << "set colorbox horizontal user origin .82,.08 size .16,.015" << endl;
    cout << "set cbtics " << realtype(2. * (range_delta_t_max - range_delta_t_min) / si::kelvins / 7) << endl;
    cout << "set cblabel 'T_o - T' offset 0,.5" << endl;
    cout << "set palette maxcolors 7 defined (" 
      << realtype(range_delta_t_min / si::kelvins) 
      << " \"blue\", 0 \"dark-grey\", "
      << realtype(2. * (range_delta_t_max - range_delta_t_min) / si::kelvins / 7) << " \"orange\", " 
      << realtype(range_delta_t_max / si::kelvins) 
      << " \"red\")" << endl;
    cout << "set cbrange [" 
      << realtype(range_delta_t_min / si::kelvins) << ":" 
      << realtype(range_delta_t_max / si::kelvins) << "]" << endl;
    cout << "plot";
    for (size_t s = 0; s < ml->n_specs(); ++s)
    {
      for (size_t i = 0; i < numRecords(); i += dot_step)
      {
        if (i+s != 0) cout << ", ";
        cout << "'-' using 2:1:3";
        // TODO: pt should be supplied as an option (was 19 vs. 12 ps .4) 
        cout << " with points pt " << (s == 0 ? "19" : "12") << " ps .25 lc palette axes x2y1 t ";
        if (i == 0) cout << "'wet radii (" << (s == 0 ? "a" : "b") << ", every " << dot_step << " steps)'";
        else cout << "''";
      }
    }
    cout << ", '-' using (";
    cout << "p_v($" << model->getStateVectorIndexPressure() + 2;
    cout << ", $" << model->getStateVectorIndexSpecificHumidity() + 2;
    cout << ")/p_s($" << model->getStateVectorIndexTemperature() + 2;
    cout << ")*100):1 with lines lt 2 lw 2 axes x1y1 t 'RH'";
    cout << endl;

    for (size_t s = 0; s < ml->n_specs(); ++s)
    {
      for (size_t i = 0; i < numRecords(); i += dot_step)
      {
        dataSpectrum(i, s, true);
        cout << "e" << endl;
      }
    }
    dataSounding();
    cout << "e" << endl;

    cout << "reset" << endl;
    cout << "set origin 0, 0" << endl;
    cout << "set size .45,.55" << endl;
    cout << "set xrange [" << realtype(range_r_min / si::metres) << ":" << realtype(range_r_max / si::meters) << "]" << endl;
    cout << "set y2range [-"
      << realtype(times.back() / si::seconds / 40) << ":" 
      << realtype(times.back() / si::seconds) << "]" << endl;
    cout << "set x2range [0:3000e6]" << endl;
    cout << "set ytics format ''" << endl;
    cout << "set x2tics (\"0/ug\" 0, \"500\" 500e6, \"1000\" 1000e6, \"1500\" 1500e6, \"2000\" 2000e6, \"2500\" 2500e6, \"3000\" 3000e6) out" << endl;
    cout << "set y2tics format '' out" << endl;
    cout << "set xtics (\"1 um\" 1e-6, \"2\" 2e-6, \"3\" 3e-6, \"4\" 4e-6, \"5\" 5e-6, \"6\" 6e-6) out nomirror" << endl;
    cout << "set xlabel ''" << endl;
    cout << "set key below reverse Left left spacing 1.055" << endl;
    cout << "plot ";
    for (int i = n_spec - 1; i >= 0; --i) // going from top to botton to allow filledcurves act as overlays for previous spectra
    {
      for (int s = 0; s < ml->n_specs(); ++s)
      {
        if (s != 0 || i != n_spec - 1) cout << ",";
        cout << "'-' using ($2+.5*($4-$2)):";
        cout << "($3/($4-$2)*1e-12 / " << realtype(div_spec) << " + $1)";
        cout << ":($4-$2) with boxes axes x1y2 lt 3 lw " << 3 - 2 * s << " t '";
        if (i == n_spec - 1) 
        {
          cout << "wet spectra (";
          if (s == 0) cout << "a";
          else cout << "b";
          cout << ", every " << lrint(realtype(times.back() / si::seconds) / n_spec) << " s)";
        }
        cout << "'";
      }
      cout << ", '-' using 2:1";
      cout << " with filledcurves x1 lc rgbcolor 'white' axes x1y2 t ''";
      cout << ", '-' using 2:1 with lines lt 3 lw .15 axes x1y2 t '";
      cout << "'";
    }
    cout << ", '-' using " << model->getStateVectorIndexRadii() + 2 + 0;
    cout << ":1 with lines lt 8 lw 4 axes x2y2 t 'N [1/ug]'";
    cout << ", '-' using ($" << model->getStateVectorIndexRadii() + 2 + 0 << " < " 
      << realtype(N_min * si::kilogrammes) << " ? NaN :"
      << "$" 
      << model->getStateVectorIndexRadii() + 2 + 1 // m1
      << "/$" 
      << model->getStateVectorIndexRadii() + 2 + 0 // m0
      << ")";
    cout << ":1 with lines lt 7 lw 4 axes x1y2 t '<r> +/- sgma [um] (N > " 
      << lrint(realtype(N_min * si::kilogrammes * 1e-6)) << "/ug)'";
    cout << ", '-' using ($" << model->getStateVectorIndexRadii() + 2 + 0 << " < " 
      << realtype(N_min * si::kilogrammes) << " ? NaN :"
      << "$" 
      << model->getStateVectorIndexRadii() + 2 + 1 // m1
      << "/$" 
      << model->getStateVectorIndexRadii() + 2 + 0 // m0
      << "-sqrt($"
      << model->getStateVectorIndexRadii() + 2 + 2 // m2
      << "/$" 
      << model->getStateVectorIndexRadii() + 2 + 0 // m0
      << "-$"
      << model->getStateVectorIndexRadii() + 2 + 1 // m1
      << "/$" 
      << model->getStateVectorIndexRadii() + 2 + 0 // m0
      << "*$"
      << model->getStateVectorIndexRadii() + 2 + 1 // m1
      << "/$" 
      << model->getStateVectorIndexRadii() + 2 + 0 // m0
      << "))";
    cout << ":1 with lines lt 7 lw 1 axes x1y2 t ''";
    cout << ", '-' using ($" << model->getStateVectorIndexRadii() + 2 + 0 << " < " 
      << realtype(N_min * si::kilogrammes) << " ? NaN :"
      << "$" 
      << model->getStateVectorIndexRadii() + 2 + 1 // m1
      << "/$" 
      << model->getStateVectorIndexRadii() + 2 + 0 // m0
      << "+sqrt($"
      << model->getStateVectorIndexRadii() + 2 + 2 // m2
      << "/$" 
      << model->getStateVectorIndexRadii() + 2 + 0 // m0
      << "-$"
      << model->getStateVectorIndexRadii() + 2 + 1 // m1
      << "/$" 
      << model->getStateVectorIndexRadii() + 2 + 0 // m0
      << "*$"
      << model->getStateVectorIndexRadii() + 2 + 1 // m1
      << "/$" 
      << model->getStateVectorIndexRadii() + 2 + 0 // m0
      << "))";
    cout << ":1 with lines lt 7 lw .5 axes x1y2 t ''";
/*
    // r_eff:
    cout << ", '-' using ($" << model->getStateVectorIndexRadii() + 2 + 0 << " < " 
      << realtype(N_min * si::kilogrammes) << " ? NaN :"
      << "$" 
      << model->getStateVectorIndexRadii() + 2 + 3 // m3
      << "/$" 
      << model->getStateVectorIndexRadii() + 2 + 2 // m2
      << ")";
    cout << ":1 with lines lt 4 lw 1 axes x1y2 t 'r_{eff} [um] (N > "
      << lrint(realtype(N_min * si::kilogrammes * 1e-6)) << "/ug)'";
*/
/*
    // LWC:: TODO: there is an error in q_c/LWC convertion here !
    cout << ", '-' using ($" << model->getStateVectorIndexRadii() + 2 + 0 << " < " 
      << realtype(N_min * si::kilogrammes) << " ? NaN :"
      << "rho_w_0 * 4. / 3. * pi *  $" 
      << model->getStateVectorIndexRadii() + 2 + 3 // m3
      << "/$" 
      << model->getStateVectorIndexPressure() + 2
      << "*$" 
      << model->getStateVectorIndexTemperature() + 2
      << "*R($" 
      << model->getStateVectorIndexSpecificHumidity() + 2
      << ")*1e3*1e-6*1e2)";
    cout << ":1 with lines lt 4 lw 4 axes x1y2 t 'q_{c} [g/kg] (N > "
      << lrint(realtype(N_min * si::kilogrammes * 1e-6)) << "/ug)'";
*/
    cout << endl;

    { // scope for times_spec
      vector<size_t> times_spec(n_spec); 
      { // scope for j
        int j = 0;
        for (size_t i = 1; i < times.size() - 1; ++i) // starting with 1 so that (i - 1) works from the start, 
        {                                             // ending at (length() - 1) so that tmax is never reached
          if (times[i] > (j / realtype(n_spec)) * times.back()) 
          { 
            times_spec[j] = i - 1;
            ++j;
          }
        }
      }

      for (int i = n_spec - 1; i >= 0; --i)
      {
        for (int s = 0; s < ml->n_specs(); ++s)
        {
          dataSpectrum(times_spec[i], s); 
          cout << "e" << endl;
        }
        dataSpectrum(times_spec[i], 0, true); 
        cout << "e" << endl;
        dataSpectrum(times_spec[i], 0, true);
        cout << "e" << endl;
      }
    }
    dataSounding(true, cloud_min, cloud_max);
    cout << "e" << endl;
    dataSounding(true, cloud_min, cloud_max);
    cout << "e" << endl;
    dataSounding(true, cloud_min, cloud_max);
    cout << "e" << endl;
    dataSounding(true, cloud_min, cloud_max);
    cout << "e" << endl;
    /*
    // r_eff:
    dataSounding(true, cloud_min, cloud_max);
    cout << "e" << endl;
    */
    /*
    // LWC:
    dataSounding(true, cloud_min, cloud_max);
    cout << "e" << endl;
    */

    cout << "unset multiplot" << endl;
  }

};
#  undef self

#endif
