/* OutputGnuplotOneLine.hpp
 * 
 * Copyright (C) 2010 Sylwester Arabas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#ifndef OUTPUT_GNUPLOT_ONE_LINE_HPP
#  define OUTPUT_GNUPLOT_ONE_LINE_HPP

#  define self OutputGnuplotOneLine
class self : public OutputGnuplot
{

  protected: quantity<si::length> cloud_min, cloud_max;
  protected: quantity<specific_concentration> N_min;

  protected: quantity<si::dimensionless> RH_max;
  protected: quantity<si::temperature> DWPTELEV_max;

  public: self(Model *model_, InitpTq *initptq_, SpectraMemLayout *ml_, Updraft *updraft,
    quantity<si::length> cloud_min_, quantity<si::length> cloud_max_,
    quantity<specific_concentration> N_min_
  ) : OutputGnuplot(model_, initptq_, ml_, updraft, rec_step) 
  { 
    cloud_min = cloud_min_;
    cloud_max = cloud_max_;
    N_min = N_min_;
    RH_max = 0.;
    DWPTELEV_max = -100. * si::kelvins;
  }

  public: void record(quantity<si::time>, N_Vector y, N_Vector) 
  {
#  define p (NV_Ith_S(y, model->getStateVectorIndexPressure()) * si::pascals)
#  define q_v (NV_Ith_S(y, model->getStateVectorIndexSpecificHumidity()))
#  define T (NV_Ith_S(y, model->getStateVectorIndexTemperature()) * si::kelvins)
    quantity<si::dimensionless> 
      RH_curr = p * q_v / (constants::epsilon + q_v * (1. - constants::epsilon)) / constants::p_v_s(T);
    if (RH_curr > RH_max)
    {
      RH_max = RH_curr;

      quantity<si::pressure> p_v = RH_curr * constants::p_v_s(T);
      quantity<si::temperature> DWPT_curr = constants::p_v_s_inv(p_v, 1e-8, 1e-16, 100);
      DWPTELEV_max = DWPT_curr - T;
    }
#  undef p
#  undef q_v
#  undef T
  }

  public: void head() {}

  public: void foot(quantity<si::time> t, N_Vector y, N_Vector p) 
  {
    tmpfile_write(t, y, p);
    dataSounding(true, cloud_min, cloud_max, false);
    cout << "\t" << realtype(RH_max);
    cout << "\t" << realtype(DWPTELEV_max / si::kelvins);
    cout << "\t" << ml->n_bins(0);
    cout << endl;
  }

};
#  undef self

#endif
