/* InitSpectrumFromFile.hpp 
 * 
 * Copyright (C) 2010 Sylwester Arabas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#ifndef INIT_SPECTRUM_FROM_FILE_HPP
#  define INIT_SPECTRUM_FROM_FILE_HPP

#  include <fstream>
#  include <valarray>
#  include <gsl/gsl_interp.h>
#  include <gsl/gsl_errno.h>

#  define self InitSpectrumFromFile
#  define parent InitSpectrum
class self : public InitSpectrum
{

  private: valarray<double> *xx, *yy;
  private: gsl_interp *interp;
  private: gsl_interp_accel *interp_accel;
  private: quantity<si::length> x_unit;
  private: quantity<concentration_density> y_unit;

  public: quantity<concentration_density> n_n(quantity<si::length> radius)
  {
    if (radius < (*xx)[0] * x_unit || radius > (*xx)[xx->size() - 1] * x_unit) return 0. * pow<-4>(si::metres);
    return y_unit * gsl_interp_eval(interp, &((*xx)[0]), &((*yy)[0]), radius / x_unit, interp_accel);
  }

  public: self(string filename, bool dryNotWet) : parent(dryNotWet)
  {
    // TODO: opcja
    x_unit = 1e-9 * si::metres;
    y_unit = pow<-3>(1e-2 * si::metres) / x_unit;

    fstream f;
    f.open(filename.c_str(), fstream::in);
    if (f.fail()) 
    {
      cerr << msgprefix << "failed to open file: " << filename << endl;
      throw exception();
    }

    // counting number of lines
    int lines = 1;
    string line;
    while (getline(f, line)) ++lines;
    if (line.length() == 0) --lines;
    
    xx = new valarray<double>(lines);
    yy = new valarray<double>(lines);
    f.clear();
    f.seekg(0, fstream::beg);       

    for (int i = 0; i < lines; ++i) 
    {
      double x, y;
      f >> x;
      f >> y;
      (*xx)[i] = x;
      (*yy)[i] = y;
    }

    f.close();

    interp_init();
  }

  public: ~self()
  {
    interp_free();
    delete xx;
    delete yy;
  }

  private: void interp_init() 
  {
    interp_accel = gsl_interp_accel_alloc();
    if (interp_accel == NULL) 
    {
      cerr << msgprefix << "failed to allocate GSL accellerator" << endl;
      throw exception();
    }
// TODO interp type jako opcja
    interp = gsl_interp_alloc(gsl_interp_linear, xx->size());
    if (interp == NULL)
    {
      cerr << msgprefix << "failed to allocate GSL interpolator" << endl;
      throw exception();
    }
    if (GSL_SUCCESS != gsl_interp_init(interp, &((*xx)[0]), &((*yy)[0]), xx->size()))
    {
      cerr << msgprefix << "failed to initialize GSL interpolator" << endl;
      throw exception();
    }
  }

  private: void interp_free()
  {
    gsl_interp_free(interp);
    gsl_interp_accel_free(interp_accel);
  }

};
#  undef self
#  undef parent

#endif
