/* InitBinLayoutLogLinear.hpp 
 * 
 * Copyright (C) 2010 Sylwester Arabas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#ifndef INIT_BIN_LAYOUT_LOG_LINEAR_HPP
#  define INIT_BIN_LAYOUT_LOG_LINEAR_HPP

#  define self InitBinLayoutLogLinear
class self : public InitBinLayout
{

  private: quantity<si::dimensionless> leftEndLn, rightEndLn, binWidthLn, binWidthLnMax;
  private: quantity<specific_concentration> noiseConc;
  private: size_t n_bins;

  public: self(size_t n_bins_, quantity<si::length> leftEnd_, 
    quantity<si::length> rightEnd_, quantity<si::dimensionless> maxMult,
    quantity<specific_concentration> noiseConc_)
  {
    assert(n_bins_ > 0);

    n_bins = n_bins_;
    leftEndLn = log(leftEnd_ / si::metres);
    rightEndLn = log(rightEnd_ / si::metres);
    binWidthLn = (rightEndLn - leftEndLn) / realtype(n_bins);
    binWidthLnMax = maxMult * binWidthLn;
    noiseConc = noiseConc_;
  }

  public: quantity<si::length> getLeftEdge(size_t n)
  {
    assert(n >= 0 && n < n_bins + 1);
    return exp(leftEndLn + realtype(n) * binWidthLn) * si::metres;
  }

  public: quantity<si::length> getRightEdge(size_t n)
  {
    assert(n >=0 && n < n_bins);
    return exp(rightEndLn - realtype(n_bins - n - 1) * binWidthLn) * si::metres;
  }

  public: bool binNeedsSplitting(quantity<si::length> leftEdge, quantity<si::length> rightEdge, 
    quantity<si::length> lastLeftEdge, quantity<si::length> lastRightEdge,
    quantity<specific_concentration> specConc, size_t) 
  {
    if (specConc < noiseConc) return false;
    return (log(rightEdge / si::metres) - log(leftEdge / si::metres)) > binWidthLnMax;
  }

  public: size_t splitInto(quantity<specific_concentration> specConc)
  {
    return max(2, int(ceil(specConc / noiseConc)));
  }

  public: size_t numOfBins() 
  {
    return n_bins;
  }

};
#  undef self

#endif
