/* InitBinLayoutInstrument.hpp 
 * 
 * Copyright (C) 2010 Sylwester Arabas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#ifndef INIT_BIN_LAYOUT_INTRUMENT_HPP
#  define INIT_BIN_LAYOUT_INSTRUMENT_HPP

#  define self InitBinLayoutInstrument
class self : public InitBinLayout
{

  private: quantity<specific_concentration> noiseConc;
  private: quantity<si::dimensionless> maxMult;
  private: vector<quantity<si::length> > l_edges, r_edges;

  public: self(string instruments,
    quantity<si::dimensionless> maxMult_,
    quantity<specific_concentration> noiseConc_)
  {
    // GRIMM OPC bins [radii in meters]
    size_t opc_bins_num = 3;
    realtype opc_bins_centers[] = {.725, .9, 1.3};
    realtype opc_bins_widths[] = {.15, .2, .6}; 
    for (int i = 0; i < opc_bins_num; ++i) 
    {
      opc_bins_centers[i] /= 2e6;
      opc_bins_widths[i] /= 2e6;
    }

    // PCASP bins [radii in meters]
    size_t pcasp_bins_num = 29;
    realtype pcasp_bins_centers[] = {.105,.115,.125,.135,.145,.155,.165,.175, .19,.21,.23,.25,.27,.29, .35,.45,.55, 0.7,0.9,1.1,1.3,1.5,1.7,1.9,2.1,2.3,2.5,2.7,2.9};
    realtype pcasp_bins_widths[] = {.005,.005,.005,.005,.005,.005,.005,.005, .01,.01,.01,.01,.01,.01, .05,.05,.05, 0.1,0.1,0.1,0.1,0.1,0.1,0.1,0.1,0.1,0.1,0.1,0.1};
    for (int i = 0; i < pcasp_bins_num; ++i) 
    {
      pcasp_bins_centers[i] /= 2e6;
      pcasp_bins_widths[i] /= 1e6; // 1 is correct!
    }
    
    // SMPS bins [radii in meters] 
    size_t smps_bins_num = 161;
    realtype smps_bins_centers[] = {
      0.003035, 0.003147, 0.003262, 0.003381, 0.003505, 0.003634, 0.003767, 0.003905, 0.004048, 
      0.004196, 0.004350, 0.004509, 0.004674, 0.004845, 0.005023, 0.005207, 0.005398, 0.005595, 
      0.005800, 0.006013, 0.006233, 0.006461, 0.006698, 0.006944, 0.007198, 0.007462, 0.007735, 
      0.008018, 0.008312, 0.008617, 0.008932, 0.009259, 0.009599, 0.009950, 0.010310, 0.010690, 
      0.011080, 0.011490, 0.011910, 0.012350, 0.012800, 0.013270, 0.013750, 0.014260, 0.014780, 
      0.015320, 0.015880, 0.016470, 0.017070, 0.017690, 0.018340, 0.019010, 0.019710, 0.020430, 
      0.021180, 0.021960, 0.022760, 0.023600, 0.024460, 0.025360, 0.026280, 0.027250, 0.028250, 
      0.029280, 0.030350, 0.031470, 0.032620, 0.033810, 0.035050, 0.036340, 0.037670, 0.039050, 
      0.040480, 0.041960, 0.043500, 0.045090, 0.046740, 0.048450, 0.050230, 0.052070, 0.053980, 
      0.055950, 0.058000, 0.060130, 0.062330, 0.064610, 0.066980, 0.069440, 0.071980, 0.074620, 
      0.077350, 0.080180, 0.083120, 0.086170, 0.089320, 0.092590, 0.095990, 0.099500, 0.103100, 
      0.106900, 0.110800, 0.114900, 0.119100, 0.123500, 0.128000, 0.132700, 0.137500, 0.142600, 
      0.147800, 0.153200, 0.158800, 0.164700, 0.170700, 0.176900, 0.183400, 0.190100, 0.197100, 
      0.204300, 0.211800, 0.219600, 0.227600, 0.236000, 0.244600, 0.253600, 0.262800, 0.272500, 
      0.282500, 0.292800, 0.303500, 0.314700, 0.326200, 0.338100, 0.350500, 0.363400, 0.376700, 
      0.390500, 0.404800, 0.419600, 0.435000, 0.450900, 0.467400, 0.484500, 0.502300, 0.520700, 
      0.539800, 0.559500, 0.580000, 0.601300, 0.623300, 0.646100, 0.669800, 0.694400, 0.719800, 
      0.746200, 0.773500, 0.801800, 0.831200, 0.861700, 0.893200, 0.925900, 0.959900               
    };
    realtype smps_bins_widths[smps_bins_num];
    for (int i = 0; i < smps_bins_num; ++i) smps_bins_widths[i] = 
      smps_bins_centers[i] - (i == 0 ? .003 : smps_bins_centers[i - 1]);
    for (int i = 0; i < smps_bins_num; ++i) 
    {
      smps_bins_centers[i] /= 2e6;
      smps_bins_widths[i] /= 2e6;
    }

    l_edges.reserve(smps_bins_num + opc_bins_num + 1); // SMPS+gap+OPC case
    r_edges.reserve(smps_bins_num + opc_bins_num + 1); // SMPS+gap+OPC case

    // SMPS
    if (instruments.find("SMPS") != string::npos)
    {
      for (int i = 53; i < 141; ++i) 
      {
        l_edges.push_back((smps_bins_centers[i] - .5 * smps_bins_widths[i]) * si::metres);
        r_edges.push_back((smps_bins_centers[i] + .5 * smps_bins_widths[i]) * si::metres);
      }
    }

    // SMPS-OPC gap (for interpolation)
    if (instruments.find("gap") != string::npos)
    {
      l_edges.push_back((smps_bins_centers[140] + .5 * smps_bins_widths[smps_bins_num - 1]) * si::metres);
      r_edges.push_back((opc_bins_centers[0] - .5 * opc_bins_widths[0]) * si::metres);
    }

    // OPC
    if (instruments.find("OPC") != string::npos)
    {
      for (int i = 0; i < opc_bins_num; ++i) 
      {
        l_edges.push_back((opc_bins_centers[i] - .5 * opc_bins_widths[i]) * si::metres);
        r_edges.push_back((opc_bins_centers[i] + .5 * opc_bins_widths[i]) * si::metres);
      }
    }

    // TODO: dirty
    l_edges.push_back(r_edges.back());

    maxMult = maxMult_;
    noiseConc = noiseConc_;
  }

  public: quantity<si::length> getLeftEdge(size_t n)
  {
    assert(n >= 0 && n < l_edges.size());
    return l_edges[n];
  }

  public: quantity<si::length> getRightEdge(size_t n)
  {
    assert(n >=0 && n < r_edges.size());
    return r_edges[n];
  }

  public: bool binNeedsSplitting(quantity<si::length> leftEdge, quantity<si::length> rightEdge, 
    quantity<si::length> lastLeftEdge, quantity<si::length> lastRightEdge,
    quantity<specific_concentration> specConc, size_t bin) 
  {
    if (specConc < noiseConc) return false;
    realtype binWidthLnMax = maxMult * (log(r_edges[bin] / si::metres) - log(l_edges[bin] / si::metres));
    return (log(rightEdge / si::metres) - log(leftEdge / si::metres)) > binWidthLnMax;
  }

  public: size_t splitInto(quantity<specific_concentration> specConc)
  {
    return max(2, int(ceil(specConc / noiseConc)));
  }

  public: size_t numOfBins() 
  {
    return r_edges.size();
  }

};
#  undef self

#endif
