!    Copyright (C) 2007 Imperial College London and others.
!
!    Please see the AUTHORS file in the main source directory for a full list
!    of copyright holders.
!
!    Applied Modelling and Computation Group
!    Department of Earth Science and Engineering
!    Imperial College London
!
!    David.Ham@Imperial.ac.uk
!
!    This library is free software; you can redistribute it and/or
!    modify it under the terms of the GNU Lesser General Public
!    License as published by the Free Software Foundation,
!    version 2.1 of the License.
!
!    This library is distributed in the hope that it will be useful,
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
!    Lesser General Public License for more details.
!
!    You should have received a copy of the GNU Lesser General Public
!    License along with this library; if not, write to the Free Software
!    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
!    USA

subroutine test_fspud

  use spud
  use unittest_tools

  implicit none

  integer, parameter :: D = kind(0.0D0)
  real(D), parameter :: tol = 1.0e-6_D
  
  print *, "*** Testing clear_options ***"
  call test_clear_options("/type_none")
    
  print *, "*** Testing set_option and get_option for real scalar ***"
  call test_set_and_get_real_scalar("/real_scalar", 42.0_D)
  
  print *, "*** Testing set_option and get_option for real vector ***"
  call test_set_and_get_real_vector("/real_vector", (/42.0_D, 43.0_D/))
  
  print *, "*** Testing set_option and get_option for real tensor ***"
  call test_set_and_get_real_tensor("/real_tensor", reshape((/42.0_D, 43.0_D, 44.0_D, 45.0_D, 46.0_D, 47.0_D/), (/2, 3/)))
  
  print *, "*** Testing set_option and get_option for integer scalar ***"
  call test_set_and_get_integer_scalar("/integer_scalar", 42)
  
  print *, "*** Testing set_option and get_option for integer vector ***"
  call test_set_and_get_integer_vector("/integer_vector", (/42, 43/))
  
  print *, "*** Testing set_option and get_option for integer tensor ***"
  call test_set_and_get_integer_tensor("/integer_tensor", reshape((/42, 43, 44, 45, 46, 47/), (/2, 3/)))
  
  print *, "*** Testing set_option and get_option for character ***"
  call test_set_and_get_character("/character", "Forty Two")
  
  print *, "*** Testing add_option and get_option ***"
  call test_set_and_get_type_none("/type_none")
  
contains
  
  subroutine test_key_errors(key)
    character(len = *), intent(in) :: key
  
    character(len = 255) :: test_char
    integer :: test_integer_scalar
    integer, dimension(3) :: test_integer_vector, integer_vector_default
    integer, dimension(3, 4) :: test_integer_tensor, integer_tensor_default
    real(D) :: test_real_scalar
    real(D), dimension(3) :: test_real_vector, real_vector_default
    real(D), dimension(3, 4) :: test_real_tensor, real_tensor_default
    integer :: rank, type, stat
    integer, dimension(2) :: shape
    
    integer :: i, j
    
    do i = 1, size(real_vector_default)
      real_vector_default = 42.0_D + i
    end do      
    do i = 1, size(real_tensor_default, 1)
      do j = 1, size(real_tensor_default, 2)
        real_tensor_default = 42.0_D + i * size(real_tensor_default, 2) + j
      end do
    end do
    do i = 1, size(integer_vector_default)
      integer_vector_default = 42.0_D + i
    end do 
    do i = 1, size(real_tensor_default, 1)
      do j = 1, size(integer_tensor_default, 2)
        integer_tensor_default = 42.0_D + i * size(integer_tensor_default, 2) + j
      end do
    end do
  
    call report_test("[Missing option]", have_option(trim(key)), .false., "Missing option reported present")
    type = option_type(trim(key), stat)
    call report_test("[Key error when extracting option type]", stat /= SPUD_KEY_ERROR, .false., "Returned incorrect error code when retrieving option type")
    rank = option_rank(trim(key), stat)
    call report_test("[Key error when extracting option rank]", stat /= SPUD_KEY_ERROR, .false., "Returned incorrect error code when retrieving option rank")
    shape = option_shape(trim(key), stat)
    call report_test("[Key error when extracting option shape]", stat /= SPUD_KEY_ERROR, .false., "Returned incorrect error code when retrieving option shape")
    call get_option(trim(key), test_real_scalar, stat)
    call report_test("[Key error when extracting option data]", stat /= SPUD_KEY_ERROR, .false., "Returned incorrect error code when retrieving option type")
    call get_option(trim(key), test_real_scalar, stat, default = 42.0_D)
    call report_test("[Extracted option data with default argument]", stat /= SPUD_NO_ERROR, .false., "Returned error code when retrieving option data")
    call report_test("[Extracted correct option data (default)]", abs(test_real_scalar - 42.0_D) > tol, .false., "Retrieved incorrect option data")
    call get_option(trim(key), test_real_vector, stat)
    call report_test("[Key error when extracting option data]", stat /= SPUD_KEY_ERROR, .false., "Returned incorrect error code when retrieving option type")
    call get_option(trim(key), test_real_vector, stat, default = real_vector_default)
    call report_test("[Extracted option data with default argument]", stat /= SPUD_NO_ERROR, .false., "Returned error code when retrieving option data")
    call report_test("[Extracted correct option data (default)]", maxval(abs(test_real_vector - real_vector_default)) > tol, .false., "Retrieved incorrect option data")
    call get_option(trim(key), test_real_tensor, stat)
    call report_test("[Key error when extracting option data]", stat /= SPUD_KEY_ERROR, .false., "Returned incorrect error code when retrieving option type")
    call get_option(trim(key), test_real_tensor, stat, default = real_tensor_default)
    call report_test("[Extracted option data with default argument]", stat /= SPUD_NO_ERROR, .false., "Returned error code when retrieving option data")
    call report_test("[Extracted correct option data (default)]", maxval(abs(test_real_tensor - real_tensor_default)) > tol, .false., "Retrieved incorrect option data")
    call get_option(trim(key), test_integer_scalar, stat)
    call report_test("[Key error when extracting option data]", stat /= SPUD_KEY_ERROR, .false., "Returned incorrect error code when retrieving option type")
    call get_option(trim(key), test_integer_scalar, stat, default = 42)
    call report_test("[Extracted option data with default argument]", stat /= SPUD_NO_ERROR, .false., "Returned error code when retrieving option data")
    call report_test("[Extracted correct option data (default)]", test_integer_scalar /= 42, .false., "Retrieved incorrect option data")
    call get_option(trim(key), test_integer_vector, stat)
    call report_test("[Key error when extracting option data]", stat /= SPUD_KEY_ERROR, .false., "Returned incorrect error code when retrieving option type")
    call get_option(trim(key), test_integer_vector, stat, default = integer_vector_default)
    call report_test("[Extracted option data with default argument]", stat /= SPUD_NO_ERROR, .false., "Returned error code when retrieving option data")
    call report_test("[Extracted correct option data (default)]", count(test_integer_vector /= integer_vector_default) > 0, .false., "Retrieved incorrect option data")
    call get_option(trim(key), test_integer_tensor, stat)
    call report_test("[Key error when extracting option data]", stat /= SPUD_KEY_ERROR, .false., "Returned incorrect error code when retrieving option type")
    call get_option(trim(key), test_integer_tensor, stat, default = integer_tensor_default)
    call report_test("[Extracted option data with default argument]", stat /= SPUD_NO_ERROR, .false., "Returned error code when retrieving option data")
    call report_test("[Extracted correct option data (default)]", count(test_integer_tensor /= integer_tensor_default) > 0, .false., "Retrieved incorrect option data")
    call get_option(trim(key), test_char, stat)
    call report_test("[Key error when extracting option data]", stat /= SPUD_KEY_ERROR, .false., "Returned incorrect error code when retrieving option type")
    call get_option(trim(key), test_char, stat, default = "Forty Two")
    call report_test("[Extracted option data with default argument]", stat /= SPUD_NO_ERROR, .false., "Returned error code when retrieving option data")
    call report_test("[Extracted correct option data (default)]", test_char /= "Forty Two", .false., "Retrieved incorrect option data")
    
  end subroutine test_key_errors
  
  subroutine test_key_present(key)
    character(len = *), intent(in) :: key
    
    call report_test("[Option present]", .not. have_option(trim(key)), .false., "Present option reported missing")
    
  end subroutine test_key_present
  
  subroutine test_type(key, type)
    character(len = *), intent(in) :: key
    integer, intent(in) :: type
    
    integer :: stat, type_ret
    
    type_ret = option_type(trim(key), stat)
    call report_test("[Extracted option type]", stat /= SPUD_NO_ERROR, .false., "Returned error code when retrieving option type")
    call report_test("[Correct option type]", type_ret /= type, .false., "Incorrect option type returned")
    
  end subroutine test_type
  
  subroutine test_rank(key, rank)
    character(len = *), intent(in) :: key
    integer, intent(in) :: rank
    
    integer :: stat, rank_ret
    
    rank_ret = option_rank(trim(key), stat)
    call report_test("[Extracted option rank]", stat /= SPUD_NO_ERROR, .false., "Returned error code when retrieving option rank")
    call report_test("[Correct option rank]", rank_ret /= rank, .false., "Incorrect option rank returned")
    
  end subroutine test_rank
  
  subroutine test_shape(key, shape)
    character(len = *), intent(in) :: key
    integer, dimension(2), intent(in) :: shape
    
    integer :: stat
    integer, dimension(2) :: shape_ret
    
    shape_ret = option_shape(trim(key), stat)
    call report_test("[Extracted option shape]", stat /= SPUD_NO_ERROR, .false., "Returned error code when retrieving option shape")
    call report_test("[Correct option shape]", count(shape_ret /= shape) /= 0, .false., "Incorrect option shape returned")
    
  end subroutine test_shape
  
  subroutine test_type_errors_real(key)
    character(len = *), intent(in) :: key
    
    integer :: stat
    real(D) :: real_scalar_val
    real(D), dimension(3) :: real_vector_default, real_vector_val
    real(D), dimension(3, 4) :: real_tensor_default, real_tensor_val

    call get_option(trim(key), real_scalar_val, stat)
    call report_test("[Type error when extracting option data]", stat /= SPUD_TYPE_ERROR, .false., "Returned incorrect error code when retrieving option data")
    call get_option(trim(key), real_scalar_val, stat, default = 0.0_D)
    call report_test("[Type error when extracting option data with default argument]", stat /= SPUD_TYPE_ERROR, .false., "Returned incorrect error code when retrieving option data")
    call get_option(trim(key), real_vector_val, stat)
    call report_test("[Type error when extracting option data]", stat /= SPUD_TYPE_ERROR, .false., "Returned incorrect error code when retrieving option data")
    call get_option(trim(key), real_vector_val, stat, default = real_vector_default)
    call report_test("[Type error when extracting option data with default argument]", stat /= SPUD_TYPE_ERROR, .false., "Returned incorrect error code when retrieving option data")
    call get_option(trim(key), real_tensor_val, stat)
    call report_test("[Type error when extracting option data]", stat /= SPUD_TYPE_ERROR, .false., "Returned incorrect error code when retrieving option data")
    call get_option(trim(key), real_tensor_val, stat, default = real_tensor_default)
    call report_test("[Type error when extracting option data with default argument]", stat /= SPUD_TYPE_ERROR, .false., "Returned incorrect error code when retrieving option data")

  end subroutine test_type_errors_real
  
  subroutine test_type_errors_integer(key)
    character(len = *), intent(in) :: key
    
    integer :: integer_scalar_val, stat
    integer, dimension(3) :: integer_vector_default, integer_vector_val
    integer, dimension(3, 4) :: integer_tensor_default, integer_tensor_val

    call get_option(trim(key), integer_scalar_val, stat)
    call report_test("[Type error when extracting option data]", stat /= SPUD_TYPE_ERROR, .false., "Returned incorrect error code when retrieving option data")
    call get_option(trim(key), integer_scalar_val, stat, default = 0)
    call report_test("[Type error when extracting option data with default argument]", stat /= SPUD_TYPE_ERROR, .false., "Returned incorrect error code when retrieving option data")
    call get_option(trim(key), integer_vector_val, stat)
    call report_test("[Type error when extracting option data]", stat /= SPUD_TYPE_ERROR, .false., "Returned incorrect error code when retrieving option data")
    call get_option(trim(key), integer_vector_val, stat, default = integer_vector_default)
    call report_test("[Type error when extracting option data with default argument]", stat /= SPUD_TYPE_ERROR, .false., "Returned incorrect error code when retrieving option data")
    call get_option(trim(key), integer_tensor_val, stat)
    call report_test("[Type error when extracting option data]", stat /= SPUD_TYPE_ERROR, .false., "Returned incorrect error code when retrieving option data")
    call get_option(trim(key), integer_tensor_val, stat, default = integer_tensor_default)
    call report_test("[Type error when extracting option data with default argument]", stat /= SPUD_TYPE_ERROR, .false., "Returned incorrect error code when retrieving option data")
  
  end subroutine test_type_errors_integer
  
  subroutine test_type_errors_character(key)
    character(len = *), intent(in) :: key
    
    character(len = 0) :: character_val
    integer :: stat
    
    call get_option(trim(key), character_val, stat)
    call report_test("[Type error when extracting option data]", stat /= SPUD_TYPE_ERROR, .false., "Returned incorrect error code when retrieving option data")
    call get_option(trim(key), character_val, stat, default = "")
    call report_test("[Type error when extracting option data with default argument]", stat /= SPUD_TYPE_ERROR, .false., "Returned incorrect error code when retrieving option data")
  
  end subroutine test_type_errors_character
  
  subroutine test_rank_errors_real_scalar(key)
    character(len = *), intent(in) :: key
    
    integer :: stat
    real(D) :: real_scalar_val
    
    call get_option(trim(key), real_scalar_val, stat)
    call report_test("[Rank error when extracting option data]", stat /= SPUD_RANK_ERROR, .false., "Returned incorrect error code when retrieving option data")
    call get_option(trim(key), real_scalar_val, stat, default = 0.0_D)
    call report_test("[Rank error when extracting option data with default argument]", stat /= SPUD_RANK_ERROR, .false., "Returned error code when retrieving option data")
    
  end subroutine test_rank_errors_real_scalar
  
  subroutine test_rank_errors_real_vector(key)
    character(len = *), intent(in) :: key
    
    integer :: stat
    real(D), dimension(3) :: real_vector_default, real_vector_val
    
    call get_option(trim(key), real_vector_val, stat)
    call report_test("[Rank error when extracting option data]", stat /= SPUD_RANK_ERROR, .false., "Returned incorrect error code when retrieving option data")
    call get_option(trim(key), real_vector_val, stat, default = real_vector_default)
    call report_test("[Rank error when extracting option data with default argument]", stat /= SPUD_RANK_ERROR, .false., "Returned error code when retrieving option data")

  end subroutine test_rank_errors_real_vector
  
  subroutine test_rank_errors_real_tensor(key)
    character(len = *), intent(in) :: key
    
    integer :: stat
    real(D), dimension(3, 4) :: real_tensor_default, real_tensor_val
   
    call get_option(trim(key), real_tensor_val, stat)
    call report_test("[Rank error when extracting option data]", stat /= SPUD_RANK_ERROR, .false., "Returned incorrect error code when retrieving option data")   
    call get_option(trim(key), real_tensor_val, stat, default = real_tensor_default)
    call report_test("[Rank error when extracting option data with default argument]", stat /= SPUD_RANK_ERROR, .false., "Returned error code when retrieving option data")

  end subroutine test_rank_errors_real_tensor
  
  subroutine test_rank_errors_integer_scalar(key)
    character(len = *), intent(in) :: key
    
    integer :: integer_scalar_val, stat
    
    call get_option(trim(key), integer_scalar_val, stat)
    call report_test("[Rank error when extracting option data]", stat /= SPUD_RANK_ERROR, .false., "Returned incorrect error code when retrieving option data")
    call get_option(trim(key), integer_scalar_val, stat, default = 0)
    call report_test("[Rank error when extracting option data with default argument]", stat /= SPUD_RANK_ERROR, .false., "Returned error code when retrieving option data")
    
  end subroutine test_rank_errors_integer_scalar
  
  subroutine test_rank_errors_integer_vector(key)
    character(len = *), intent(in) :: key
    
    integer :: stat
    integer, dimension(3) :: integer_vector_default, integer_vector_val
    
    call get_option(trim(key), integer_vector_val, stat)
    call report_test("[Rank error when extracting option data]", stat /= SPUD_RANK_ERROR, .false., "Returned incorrect error code when retrieving option data")
    call get_option(trim(key), integer_vector_val, stat, default = integer_vector_default)
    call report_test("[Rank error when extracting option data with default argument]", stat /= SPUD_RANK_ERROR, .false., "Returned error code when retrieving option data")

  end subroutine test_rank_errors_integer_vector
  
  subroutine test_rank_errors_integer_tensor(key)
    character(len = *), intent(in) :: key
    
    integer :: stat
    integer, dimension(3, 4) :: integer_tensor_default, integer_tensor_val
   
    call get_option(trim(key), integer_tensor_val, stat)
    call report_test("[Rank error when extracting option data]", stat /= SPUD_RANK_ERROR, .false., "Returned incorrect error code when retrieving option data")
    call get_option(trim(key), integer_tensor_val, stat, default = integer_tensor_default)
    call report_test("[Rank error when extracting option data with default argument]", stat /= SPUD_RANK_ERROR, .false., "Returned error code when retrieving option data")

  end subroutine test_rank_errors_integer_tensor
  
  subroutine test_add_new_option(key)
    character(len = *), intent(in) :: key
    
    integer :: stat
    
    call add_option(trim(key), stat)
    call report_test("[New option]", stat /= SPUD_NEW_KEY_WARNING, .false., "Failed to return new key warning when adding option")
  
  end subroutine test_add_new_option
  
  subroutine test_delete_option(key)
    character(len = *), intent(in) :: key
    
    integer :: stat
    
    call delete_option(trim(key), stat)
    call report_test("[Deleted option]", stat /= SPUD_NO_ERROR, .false., "Returned error code when deleting option")
    
  end subroutine test_delete_option
  
  subroutine test_clear_options(key)
    character(len = *), intent(in) :: key
    
    call test_key_errors(key)
    call test_add_new_option(key)
    call clear_options()
    call test_key_errors(key)
    
  end subroutine test_clear_options
  
  subroutine test_set_and_get_real_scalar(key, test_real_scalar)
    character(len = *), intent(in) :: key
    real(D), intent(in) :: test_real_scalar

    integer :: i, stat
    real(D) :: ltest_real_scalar, real_scalar_val
    
    call test_key_errors(key)
    
    do i = 1, 2
      select case(i)
        case(1)
          ltest_real_scalar = test_real_scalar
          call set_option(trim(key), ltest_real_scalar, stat)
          call report_test("[New option]", stat /= SPUD_NEW_KEY_WARNING, .false., "Failed to return new key warning when setting option")
        case default
          ltest_real_scalar = ltest_real_scalar * 1.1_D
          call set_option(trim(key), ltest_real_scalar, stat)
          call report_test("[Set existing option]", stat /= SPUD_NO_ERROR, .false., "Returned error code when setting option")
      end select
    
      call test_key_present(key)
      call test_type(key, SPUD_REAL)
      call test_rank(key, 0)
      call test_shape(key, (/-1, -1/) )
      
      call get_option(trim(key), real_scalar_val, stat)
      call report_test("[Extracted option data]", stat /= SPUD_NO_ERROR, .false., "Returned error code when retrieving option data")
      call report_test("[Extracted correct option data]", abs(real_scalar_val - ltest_real_scalar) > tol, .false., "Retrieved incorrect option data")
      call get_option(trim(key), real_scalar_val, stat, default = ltest_real_scalar * 1.1_D)
      call report_test("[Extracted option data with default argument]", stat /= SPUD_NO_ERROR, .false., "Returned error code when retrieving option data")
      call report_test("[Extracted correct option data with default argument]", abs(real_scalar_val - ltest_real_scalar) > tol, .false., "Retrieved incorrect option data")
      
      call test_rank_errors_real_vector(key)
      call test_rank_errors_real_tensor(key)
      call test_type_errors_integer(key)
      call test_type_errors_character(key)
    
    end do
    
    call test_delete_option(key)

    call test_key_errors(key)
  
  end subroutine test_set_and_get_real_scalar
  
  subroutine test_set_and_get_real_vector(key, test_real_vector)
    character(len = *), intent(in) :: key
    real(D), dimension(:), intent(in) :: test_real_vector

    integer :: i, stat
    real(D), dimension(size(test_real_vector)) :: ltest_real_vector
    real(D), dimension(:), allocatable :: real_vector_default, real_vector_val
    
    call test_key_errors(key)
    
    do i = 1, 2
      select case(i)
        case(1)
          ltest_real_vector = test_real_vector
          call set_option(trim(key), ltest_real_vector, stat)
          call report_test("[New option]", stat /= SPUD_NEW_KEY_WARNING, .false., "Failed to return new key warning when setting option")
        case default
          ltest_real_vector = ltest_real_vector * 1.1_D
          call set_option(trim(key), ltest_real_vector, stat)
          call report_test("[Set existing option]", stat /= SPUD_NO_ERROR, .false., "Returned error code when setting option")
      end select
    
      call test_key_present(key)
      call test_type(key, SPUD_REAL)
      call test_rank(key, 1)
      call test_shape(key, (/size(ltest_real_vector), -1/))
      
      allocate(real_vector_val(size(ltest_real_vector)))
      allocate(real_vector_default(size(ltest_real_vector)))
      call get_option(trim(key), real_vector_val, stat)
      call report_test("[Extracted option data]", stat /= SPUD_NO_ERROR, .false., "Returned error code when retrieving option data")
      call report_test("[Extracted correct option data]", maxval(abs(real_vector_val - ltest_real_vector)) > tol, .false., "Retrieved incorrect option data")
      call get_option(trim(key), real_vector_val, stat, default = ltest_real_vector * 1.1_D)
      call report_test("[Extracted option data with default argument]", stat /= SPUD_NO_ERROR, .false., "Returned error code when retrieving option data")
      call report_test("[Extracted correct option data with default argument]", maxval(abs(real_vector_val - ltest_real_vector)) > tol, .false., "Retrieved incorrect option data")
      deallocate(real_vector_val)
      deallocate(real_vector_default)
      allocate(real_vector_val(size(ltest_real_vector) + 1))
      allocate(real_vector_default(size(ltest_real_vector) + 1))
      call get_option(trim(key), real_vector_val, stat)
      call report_test("[Shape error when extracting option data]", stat /= SPUD_SHAPE_ERROR, .false., "Returned error code when retrieving option data")
      call get_option(trim(key), real_vector_val, stat, default = real_vector_default)
      call report_test("[Shape error when extracting option data with default argument]", stat /= SPUD_SHAPE_ERROR, .false., "Returned error code when retrieving option data")
      deallocate(real_vector_val)
      deallocate(real_vector_default)
      
      call test_rank_errors_real_scalar(key)
      call test_rank_errors_real_tensor(key)
      call test_type_errors_integer(key)
      call test_type_errors_character(key)
    
    end do
    
    call test_delete_option(key)

    call test_key_errors(key)
  
  end subroutine test_set_and_get_real_vector
  
  subroutine test_set_and_get_real_tensor(key, test_real_tensor)
    character(len = *), intent(in) :: key
    real(D), dimension(:, :), intent(in) :: test_real_tensor

    integer :: i, stat
    real(D), dimension(size(test_real_tensor, 1), size(test_real_tensor, 2)) :: ltest_real_tensor
    real(D), dimension(:, :), allocatable :: real_tensor_default, real_tensor_val
    
    call test_key_errors(key)
    
    do i = 1, 2
      select case(i)
        case(1)
          ltest_real_tensor = test_real_tensor
          call set_option(trim(key), ltest_real_tensor, stat)
          call report_test("[New option]", stat /= SPUD_NEW_KEY_WARNING, .false., "Failed to return new key warning when setting option")
        case default
          ltest_real_tensor = ltest_real_tensor * 1.1_D
          call set_option(trim(key), ltest_real_tensor, stat)
          call report_test("[Set existing option]", stat /= SPUD_NO_ERROR, .false., "Returned error code when setting option")
      end select
    
      call test_key_present(key)
      call test_type(key, SPUD_REAL)
      call test_rank(key, 2)
      call test_shape(key, (/size(ltest_real_tensor, 1), size(ltest_real_tensor, 2)/))
      
      allocate(real_tensor_val(size(ltest_real_tensor, 1), size(ltest_real_tensor, 2)))
      allocate(real_tensor_default(size(ltest_real_tensor, 1), size(ltest_real_tensor, 2)))
      call get_option(trim(key), real_tensor_val, stat)
      call report_test("[Extracted option data]", stat /= SPUD_NO_ERROR, .false., "Returned error code when retrieving option data")
      call report_test("[Extracted correct option data]", maxval(abs(real_tensor_val - ltest_real_tensor)) > tol, .false., "Retrieved incorrect option data")
      call get_option(trim(key), real_tensor_val, stat, default = ltest_real_tensor * 1.1_D)
      call report_test("[Extracted option data with default argument]", stat /= SPUD_NO_ERROR, .false., "Returned error code when retrieving option data")
      call report_test("[Extracted correct option data with default argument]", maxval(abs(real_tensor_val - ltest_real_tensor)) > tol, .false., "Retrieved incorrect option data")
      deallocate(real_tensor_val)
      deallocate(real_tensor_default)
      allocate(real_tensor_val(size(ltest_real_tensor, 1) + 1, size(ltest_real_tensor, 2) + 1))
      allocate(real_tensor_default(size(ltest_real_tensor, 1) + 1, size(ltest_real_tensor, 2) + 1))
      call get_option(trim(key), real_tensor_val, stat)
      call report_test("[Shape error when extracting option data]", stat /= SPUD_SHAPE_ERROR, .false., "Returned error code when retrieving option data")
      call get_option(trim(key), real_tensor_val, stat, default = real_tensor_default)
      call report_test("[Shape error when extracting option data with default argument]", stat /= SPUD_SHAPE_ERROR, .false., "Returned error code when retrieving option data")
      deallocate(real_tensor_val)
      deallocate(real_tensor_default)
      
      call test_rank_errors_real_scalar(key)
      call test_rank_errors_real_vector(key)
      call test_type_errors_integer(key)
      call test_type_errors_character(key)
    
    end do
    
    call test_delete_option(key)

    call test_key_errors(key)
  
  end subroutine test_set_and_get_real_tensor
  
  subroutine test_set_and_get_integer_scalar(key, test_integer_scalar)
    character(len = *), intent(in) :: key
    integer, intent(in) :: test_integer_scalar

    integer :: i, integer_scalar_val, ltest_integer_scalar, stat
    
    call test_key_errors(key)
    
    do i = 1, 2
      select case(i)
        case(1)
          ltest_integer_scalar = test_integer_scalar
          call set_option(trim(key), ltest_integer_scalar, stat)
          call report_test("[New option]", stat /= SPUD_NEW_KEY_WARNING, .false., "Failed to return new key warning when setting option")
        case default
          ltest_integer_scalar = ltest_integer_scalar + 1
          call set_option(trim(key), ltest_integer_scalar, stat)
          call report_test("[Set existing option]", stat /= SPUD_NO_ERROR, .false., "Returned error code when setting option")
      end select
    
      call test_key_present(key)
      call test_type(key, SPUD_INTEGER)
      call test_rank(key, 0)
      call test_shape(key, (/-1, -1/))
      
      call get_option(trim(key), integer_scalar_val, stat)
      call report_test("[Extracted option data]", stat /= SPUD_NO_ERROR, .false., "Returned error code when retrieving option data")
      call report_test("[Extracted correct option data]", integer_scalar_val /= ltest_integer_scalar, .false., "Retrieved incorrect option data")
      call get_option(trim(key), integer_scalar_val, stat, default = ltest_integer_scalar + 1)
      call report_test("[Extracted option data with default argument]", stat /= SPUD_NO_ERROR, .false., "Returned error code when retrieving option data")
      call report_test("[Extracted correct option data with default argument]", integer_scalar_val /= ltest_integer_scalar, .false., "Retrieved incorrect option data")
      
      call test_type_errors_real(key)
      call test_rank_errors_integer_vector(key)
      call test_rank_errors_integer_tensor(key)
      call test_type_errors_character(key)
   
    end do
    
    call test_delete_option(key)

    call test_key_errors(key)
  
  end subroutine test_set_and_get_integer_scalar
  
  subroutine test_set_and_get_integer_vector(key, test_integer_vector)
    character(len = *), intent(in) :: key
    integer, dimension(:), intent(in) :: test_integer_vector

    integer :: i, stat
    integer, dimension(size(test_integer_vector)) :: ltest_integer_vector
    integer, dimension(:), allocatable :: integer_vector_default, integer_vector_val
    
    call test_key_errors(key)
    
    do i = 1, 2
      select case(i)
        case(1)
          ltest_integer_vector = test_integer_vector
          call set_option(trim(key), ltest_integer_vector, stat)
          call report_test("[New option]", stat /= SPUD_NEW_KEY_WARNING, .false., "Failed to return new key warning when setting option")
        case default
          ltest_integer_vector = ltest_integer_vector + 1
          call set_option(trim(key), ltest_integer_vector, stat)
          call report_test("[Set existing option]", stat /= SPUD_NO_ERROR, .false., "Returned error code when setting option")
      end select
    
      call test_key_present(key)
      call test_type(key, SPUD_INTEGER)
      call test_rank(key, 1)
      call test_shape(key, (/size(ltest_integer_vector), -1/))
      
      allocate(integer_vector_val(size(ltest_integer_vector)))
      allocate(integer_vector_default(size(ltest_integer_vector)))
      call get_option(trim(key), integer_vector_val, stat)
      call report_test("[Extracted option data]", stat /= SPUD_NO_ERROR, .false., "Returned error code when retrieving option data")
      call report_test("[Extracted correct option data]", count(integer_vector_val /= ltest_integer_vector) > 1, .false., "Retrieved incorrect option data")
      call get_option(trim(key), integer_vector_val, stat, default = ltest_integer_vector + 1)
      call report_test("[Extracted option data with default argument]", stat /= SPUD_NO_ERROR, .false., "Returned error code when retrieving option data")
      call report_test("[Extracted correct option data with default argument]", count(integer_vector_val /= ltest_integer_vector) > 1, .false., "Retrieved incorrect option data")
      deallocate(integer_vector_val)
      deallocate(integer_vector_default)
      allocate(integer_vector_val(size(ltest_integer_vector) + 1))
      allocate(integer_vector_default(size(ltest_integer_vector) + 1))
      call get_option(trim(key), integer_vector_val, stat)
      call report_test("[Shape error when extracting option data]", stat /= SPUD_SHAPE_ERROR, .false., "Returned error code when retrieving option data")
      call get_option(trim(key), integer_vector_val, stat, default = integer_vector_default)
      call report_test("[Shape error when extracting option data with default argument]", stat /= SPUD_SHAPE_ERROR, .false., "Returned error code when retrieving option data")
      deallocate(integer_vector_val)
      deallocate(integer_vector_default)
      
      call test_type_errors_real(key)
      call test_rank_errors_integer_scalar(key)
      call test_rank_errors_integer_tensor(key)
      call test_type_errors_character(key)
    
    end do
    
    call test_delete_option(key)

    call test_key_errors(key)
  
  end subroutine test_set_and_get_integer_vector
  
  subroutine test_set_and_get_integer_tensor(key, test_integer_tensor)
    character(len = *), intent(in) :: key
    integer, dimension(:, :), intent(in) :: test_integer_tensor

    integer :: i, stat 
    integer, dimension(size(test_integer_tensor, 1), size(test_integer_tensor, 2)) :: ltest_integer_tensor
    integer, dimension(:, :), allocatable :: integer_tensor_default, integer_tensor_val
        
    call test_key_errors(key)
    
    do i = 1, 2
      select case(i)
        case(1)
          ltest_integer_tensor = test_integer_tensor
          call set_option(trim(key), ltest_integer_tensor, stat)
          call report_test("[New option]", stat /= SPUD_NEW_KEY_WARNING, .false., "Failed to return new key warning when setting option")
        case default
          ltest_integer_tensor = ltest_integer_tensor + 1
          call set_option(trim(key), ltest_integer_tensor, stat)
          call report_test("[Set existing option]", stat /= SPUD_NO_ERROR, .false., "Returned error code when setting option")
      end select
    
      call test_key_present(key)
      call test_type(key, SPUD_INTEGER)
      call test_rank(key, 2)
      call test_shape(key, (/size(ltest_integer_tensor, 1), size(ltest_integer_tensor, 2)/))
      
      allocate(integer_tensor_val(size(ltest_integer_tensor, 1), size(ltest_integer_tensor, 2)))
      allocate(integer_tensor_default(size(ltest_integer_tensor, 1), size(ltest_integer_tensor, 2)))
      call get_option(trim(key), integer_tensor_val, stat)
      call report_test("[Extracted option data]", stat /= SPUD_NO_ERROR, .false., "Returned error code when retrieving option data")
      call report_test("[Extracted correct option data]", count(integer_tensor_val /= ltest_integer_tensor) > 0, .false., "Retrieved incorrect option data")
      call get_option(trim(key), integer_tensor_val, stat, default = ltest_integer_tensor + 1)
      call report_test("[Extracted option data with default argument]", stat /= SPUD_NO_ERROR, .false., "Returned error code when retrieving option data")
      call report_test("[Extracted correct option data with default argument]", count(integer_tensor_val /= ltest_integer_tensor) > 0, .false., "Retrieved incorrect option data")
      deallocate(integer_tensor_val)
      deallocate(integer_tensor_default)
      allocate(integer_tensor_val(size(ltest_integer_tensor, 1) + 1, size(ltest_integer_tensor, 2) + 1))
      allocate(integer_tensor_default(size(ltest_integer_tensor, 1) + 1, size(ltest_integer_tensor, 2) + 1))
      call get_option(trim(key), integer_tensor_val, stat)
      call report_test("[Shape error when extracting option data]", stat /= SPUD_SHAPE_ERROR, .false., "Returned error code when retrieving option data")
      call get_option(trim(key), integer_tensor_val, stat, default = integer_tensor_default)
      call report_test("[Shape error when extracting option data with default argument]", stat /= SPUD_SHAPE_ERROR, .false., "Returned error code when retrieving option data")
      deallocate(integer_tensor_val)
      deallocate(integer_tensor_default)
      
      call test_type_errors_real(key)
      call test_rank_errors_integer_scalar(key)
      call test_rank_errors_integer_vector(key)
      call test_type_errors_character(key)
    
    end do

    call test_delete_option(key)

    call test_key_errors(key)
  
  end subroutine test_set_and_get_integer_tensor
  
  subroutine test_set_and_get_character(key, test_character)
    character(len = *), intent(in) :: key
    character(len = *), intent(in) :: test_character

    character(len = 0) :: short_character
    character(len = len_trim(test_character) + len(" Plus One")) :: character_val, ltest_character
    integer :: i, stat
    
    call test_key_errors(key)
    
    do i = 1, 2
      select case(i)
        case(1)
          ltest_character = trim(test_character)
          call set_option(trim(key), ltest_character, stat)
          call report_test("[New option]", stat /= SPUD_NEW_KEY_WARNING, .false., "Failed to return new key warning when setting option")
        case default
          ltest_character = trim(ltest_character) // " Plus One"
          call set_option(trim(key), ltest_character, stat)
          call report_test("[Set existing option]", stat /= SPUD_NO_ERROR, .false., "Returned error code when setting option")
      end select
      
      call test_key_present(key)
      call test_type(key, SPUD_CHARACTER)
      call test_rank(key, 1)
      call test_shape(key, (/len_trim(ltest_character), -1/))
      
      call get_option(trim(key), character_val, stat)
      call report_test("[Extracted option data]", stat /= SPUD_NO_ERROR, .false., "Returned error code when retrieving option data")
      call report_test("[Extracted correct option data]", trim(character_val) /= trim(ltest_character), .false., "Retrieved incorrect option data")
      call get_option(trim(key), character_val, stat, default = trim(ltest_character) // " Plus One")
      call report_test("[Extracted option data with default argument]", stat /= SPUD_NO_ERROR, .false., "Returned error code when retrieving option data")
      call report_test("[Extracted correct option data with default argument]", trim(character_val) /= trim(ltest_character), .false., "Retrieved incorrect option data") 
      if(len_trim(ltest_character) > 0) then
        call get_option(trim(key), short_character, stat)
        call report_test("[Shape error when extracting option data]", stat /= SPUD_SHAPE_ERROR, .false., "Returned error code when retrieving option data")
        call get_option(trim(key), short_character, stat, default = "")
        call report_test("[Shape error when extracting option data with default argument]", stat /= SPUD_SHAPE_ERROR, .false., "Returned error code when retrieving option data")
      else
        write(0, *) "Warning: Zero length test character supplied - character shape test skipped"
      end if
      
      call test_type_errors_real(key)
      call test_type_errors_integer(key)
    
    end do

    call test_delete_option(key)

    call test_key_errors(key)
  
  end subroutine test_set_and_get_character
  
  subroutine test_set_and_get_type_none(key)
    character(len = *), intent(in) :: key

    integer :: i, stat
    
    call test_key_errors(key)
    
    do i = 1, 2
      select case(i)
        case(1)
          call test_add_new_option(key)
        case default
          call add_option(trim(key), stat)
          call report_test("[Add existing option]", stat /= SPUD_NO_ERROR, .false., "Returned error code when adding option")
      end select
      
      call test_key_present(key)
      call test_type(key, SPUD_NONE)
      call test_rank(key, -1)
      call test_shape(key, (/-1, -1/))
      
      call test_type_errors_real(key)
      call test_type_errors_integer(key)
      call test_type_errors_character(key)
    
    end do
    
    call test_delete_option(key)

    call test_key_errors(key)
    
  end subroutine test_set_and_get_type_none

end subroutine test_fspud
