rm(list = ls())
gc()


# Weight fraction of carbon in crude protein (Eq. 7)
w_C_CP <- 0.53
# Weight fraction of nitrogen in crude protein (Eq. 8)
w_N_CP <- 0.16
# Weight fraction of carbon in carbohydrates (Eq. 9)
w_C_CHO <- 0.44
# Weight fraction of carbon in lignin (Eq. 10)
w_C_L <- 0.66
# Weight fraction of carbon in fat (Eq. 11)
w_C_FA <- 0.78

# assumed milk composition (section 2.7)
w_fat_milk <- 0.04
w_protein_milk <- 0.032
w_CHO_milk <- 0.0485


###############################

# Weight fraction of digestible carbon in feed (Eq. 20)
calc_fC <- function(w_N_CN) {
    0.561 + 2.190 * w_N_CN
}

# Weight fraction of digestible nitrogen in feed (Eq. 21)
calc_fN <- function(w_N_CN) {
    0.914 - 0.494 * exp(-59.559 * w_N_CN)
}

# Digestible energy in feed (Eq. 22)
calc_de <- function(w_N_CN) {
    1.952 + 11.438 * w_N_CN
}

# Weight fraction of C in feed dry matter (Eq. 23)
w_C_DM <- 0.424

###############################

# Weight fraction of N in feed dry matter (Eq. 26)
calc_w_N_DM <- function(w_N_CN) {
    w_C_DM * w_N_CN / (1 - w_N_CN)
}

# Weight fraction of crude protein in feed dry matter (Eq. 25)
calc_w_CP_DM <- function(w_N_CN) {
    calc_w_N_DM(w_N_CN) / w_N_CP
}

# Maximum daily dry matter intake (Eq. 24)
calc_DMI_max <- function(w_N_CN, BW = 500) {
    dmi <- 1.33 * BW * (0.0235 - 0.0385 * exp(-32 * calc_w_CP_DM(w_N_CN)))
    pmax(dmi, 0)
}

# Dry matter intake (Eq. 27+28)
calc_DMI <- function(w_N_CN, BW = 500, LAI = 99) {
    K <- 0.229 * BW^0.36
    calc_DMI_max(w_N_CN, BW) * (LAI / K)^3 / (1 + (LAI / K)^3)
}

# N intake (Eq. 29)
calc_m_N_intake <- function(w_N_CN, BW = 500, LAI = 99) {
    calc_DMI(w_N_CN, BW, LAI) * calc_w_N_DM(w_N_CN)
}

# C intake (Eq. 30)
calc_m_C_intake <- function(w_N_CN, BW = 500, LAI = 99) {
    calc_DMI(w_N_CN, BW, LAI) * w_C_DM
}

# Metabolizable energy in forage dry matter (Eq. 31)
calc_me <- function(w_N_CN) {
    1.01 * calc_de(w_N_CN) - 0.45
}

# Net energy in forage dry matter (Eq. 32)
calc_ne <- function(w_N_CN) {
    0.73 * calc_me(w_N_CN) - 0.19
}

# Daily net energy required requirements for maintenance (Eq. 33)
calc_NE_m <- function(BW = 500) {
    0.08 * BW^0.75
}

# Metabolized protein for endogenous urinary protein requirement (Eq. 34)
calc_MP_up <- function(w_N_CN, BW = 500) {
    0.0041 * BW^0.5
}

# Metabolized protein requirement for metabolic fecal protein (Eq. 35)
calc_MP_mfp <- function(w_N_CN, BW = 500, LAI = 99) {
    0.03 * calc_DMI(w_N_CN, BW, LAI)
}

# Net energy requirements for milk (Eq. 36)
NE_milk <- 0.36 + 9.69 * w_fat_milk

# Metabolizable protein requirements for milk (Eq. 38)
MP_milk <- w_protein_milk / 0.67

# Total available metabolizable protein (Eq. 40)
calc_MP_avail <- function(w_N_CN, BW = 500, LAI = 99) {
    calc_DMI(w_N_CN, BW, LAI) * calc_w_N_DM(w_N_CN) * calc_fN(w_N_CN) / w_N_CP
}

# Milk production
calc_m_milk <- function(w_N_CN, BW = 500, LAI = 99) {
    # Potentiual milk production from MP (Eq. 37)
    m_milk_NE <-
    (calc_DMI(w_N_CN, BW, LAI) * calc_ne(w_N_CN) - calc_NE_m(BW)) / NE_milk
    # Potentiual milk production from MP (Eq. 39)
    m_milk_MP <-
    (calc_MP_avail(w_N_CN, BW, LAI) -
    calc_MP_up(w_N_CN, BW) -
    calc_MP_mfp(w_N_CN, BW, LAI)) /
    MP_milk
    # Actual milk production (Eq. 41)
    pmax(pmin(m_milk_NE, m_milk_MP), 0)
}

# Weight fraction of nitrogen in milk (Eq. 44)
w_N_milk <- w_protein_milk * w_N_CP

# Weight fraction of carbon in milk (Eq. 45)
w_C_milk <- w_fat_milk * w_C_FA + w_protein_milk * w_C_CP + w_CHO_milk * w_C_CHO

# N milk (Eq. 42)
calc_m_N_milk <- function(w_N_CN, BW = 500, LAI = 99) {
    w_N_milk * calc_m_milk(w_N_CN, BW, LAI)
}

# C milk (Eq. 43)
calc_m_C_milk <- function(w_N_CN, BW = 500, LAI = 99) {
    w_C_milk * calc_m_milk(df$w_N_CN, BW, LAI)
}

# Methane production (Eq. 46)
calc_m_methane <- function(w_N_CN, BW = 500, LAI = 99) {
    calc_DMI(w_N_CN, BW, LAI) * 18.4 * 0.065 / 55.6
}

# C methane (Eq. 47)
calc_m_C_methane <- function(w_N_CN, BW = 500, LAI = 99) {
    0.75 * calc_m_methane(w_N_CN, BW, LAI)
}

# N feces (Eq. 48)
calc_m_N_feces <- function(w_N_CN, BW = 500, LAI = 99) {
    calc_m_N_intake(w_N_CN, BW, LAI) *
    (1 - calc_fN(w_N_CN)) +
    w_N_CP *
    calc_MP_mfp(w_N_CN, BW, LAI)
}

# C feces (Eq. 49)
calc_m_C_feces <- function(w_N_CN, BW = 500, LAI = 99) {
    calc_m_C_intake(w_N_CN, BW, LAI) *
    (1 - calc_fC(w_N_CN)) +
    w_C_CP * calc_MP_mfp(w_N_CN, BW, LAI)
}

# C urine (Eq. 50)
calc_m_C_urine <- function(w_N_CN, BW = 500, LAI = 99) {
    1 * df$N_urine
}

# N urine (Eq. 1)
calc_m_N_urine <- function(w_N_CN, BW = 500, LAI = 99) {
    calc_m_N_intake(w_N_CN, BW, LAI) -
    calc_m_N_milk(w_N_CN, BW, LAI) -
    calc_m_N_feces(w_N_CN, BW, LAI)
}

# C respiration (Eq. 2)
calc_m_C_respiration <- function(w_N_CN, BW = 500, LAI = 99) {
    calc_m_C_intake(w_N_CN, BW, LAI) -
    calc_m_C_feces(w_N_CN, BW, LAI) -
    calc_m_C_urine(w_N_CN, BW, LAI) -
    calc_m_C_methane(w_N_CN, BW, LAI) -
    calc_m_C_milk(w_N_CN, BW, LAI)
}

####### Apply functions to generate Fig. 2 ########

# calulate N and C balances for a range of w_N_CN
# Body weight
BW <- 500
# Leaf area index (very high value so that DMI = DMI_max)
LAI <- 99
# range of w_N_CN
xrange <- c(0.015, 0.09)
df <- data.frame(w_N_CN = seq(xrange[1], xrange[2], len = 1000))
df$N_intake <- calc_m_N_intake(df$w_N_CN, BW, LAI)
df$N_milk <- calc_m_N_milk(df$w_N_CN, BW, LAI)
df$N_feces <- calc_m_N_feces(df$w_N_CN, BW, LAI)
df$N_urine <- calc_m_N_urine(df$w_N_CN, BW, LAI)
df$C_intake <- calc_m_C_intake(df$w_N_CN, BW, LAI)
df$C_feces <- calc_m_C_feces(df$w_N_CN, BW, LAI)
df$C_urine <- calc_m_C_urine(df$w_N_CN, BW, LAI)
df$C_methane <- calc_m_C_methane(df$w_N_CN, BW, LAI)
df$C_milk <- calc_m_C_milk(df$w_N_CN, BW, LAI)
df$C_respiration <- calc_m_C_respiration(df$w_N_CN, BW, LAI)

# plotorder (from top to bottom)
plotorder_N <- c("milk", "urine", "feces")
plotorder_C <- c("milk", "respiration", "urine", "methane", "feces")

# plotting colors
colors <- character()
colors["milk"] <- "red"
colors["urine"] <- "purple"
colors["feces"] <- "orange"
colors["respiration"] <- "blue"
colors["methane"] <- "green"

# open image file
png("balances.png", h = 2200, w = 6000, res = 400)

layout(
    matrix(c(1, 2, 3, 3), nrow = 2, ncol = 2, byrow = TRUE),
    h = c(1, 0.1),
    w = c(1, 1)
)
par(mar = c(5.1, 4.1, 2.1, 1.1), cex = 1.1)

# plot N balances
plot(NA,
    xlim = xrange,
    ylim = c(0, 0.7),
    xaxs = "i",
    yaxs = "i",
    xlab = expression("w"["N,CN"] * " in forage"),
    ylab = expression("m"[N] * " in kg day"^-1)
)
for (i in seq_along(plotorder_N)) {
    columns_sum <- paste0("N_", plotorder_N[i:length(plotorder_N)])
    polygon(
        x = c(xrange[1], df$w_N_CN, xrange[2], xrange[1]),
        y = c(0, apply(df[, columns_sum, drop = FALSE], 1, sum), 0, 0),
        col = colors[plotorder_N[i]]
    )
}

# plot C balances
plot(NA,
    xlim = xrange,
    ylim = c(0, 7),
    xaxs = "i",
    yaxs = "i",
    xlab = expression("w"["N,CN"] * " in forage"),
    ylab = expression("m"[C] * " in kg day"^-1)
)
for (i in seq_along(plotorder_C)) {
    columns_sum <- paste0("C_", plotorder_C[i:length(plotorder_C)])
    polygon(
        x = c(xrange[1], df$w_N_CN, xrange[2], xrange[1]),
        y = c(0, apply(df[, columns_sum, drop = FALSE], 1, sum), 0, 0),
        col = colors[plotorder_C[i]]
    )
}

#plot legend
par(mar = c(0.1, 4.1, 0.1, 1.1))
llx <- 0.02
lly <- 0.2
width_box <- 0.035
height_box <- 0.45
plot(NA, xlim = c(0, 1), ylim = c(0, 1), xlab = "", ylab = "", axes = FALSE)
for (elem in c("milk", "urine", "feces", "respiration", "methane")) {
    polygon(
        x = c(llx, llx, llx + width_box, llx + width_box, llx),
        y = c(lly, lly + height_box, lly + height_box, lly, lly),
        border = "black",
        col = colors[elem],
        xpd = NA
    )
    text(
        x = llx + 1.5 * width_box,
        y = lly + 0.5 * height_box,
        label = tools::toTitleCase(elem),
        cex = 1.1,
        adj = 0,
        xpd = NA
    )
    llx <- llx + 0.2
}

dev.off()
