#include "cppdefs.h"
!-----------------------------------------------------------------------
!BOP
!
! !ROUTINE: gotm - a wrapper to call GOTM \label{sec-gotm}
!
! !INTERFACE:
   subroutine gotm()
!
! !DESCRIPTION:
!
! Here, the turbulence module of the General Ocean Turbulence Model (GOTM,
! see {\tt www.gotm.net} and \cite{UMLAUFea05}) is called. First, all
! necessary parameters are transformed to suit with a 1D water column model,
! i.e., 3D fields are transformed to a vertical vector, 2D horizontal
! fields are converted to a scalar. The transformed 3D fields are
! the layer heights {\tt hn $\rightarrow$ h}, the shear squared
! {\tt SS $\rightarrow$ SS1d},
! the buoyancy frequency squared {\tt NN $\rightarrow$ NN1d},
! the turbulent kinetic energy {\tt tke $\rightarrow$ tke1d},
! the dissipation rate {\tt eps $\rightarrow$ eps1d}
! (from which the integral length scale {\tt L1d} is calculated), the
! eddy viscosity {\tt num $\rightarrow$ num1d}, and the eddy diffusivity
! {\tt nuh $\rightarrow$ nuh1d}. The scalars are the surface and bottom friction
! velocities, {\tt u\_taus} and {\tt u\_taub}, respectively, the
! surface roughness parameter {\tt z0s} (which is currently hard-coded),
! and the bottom roughess parameter {\tt z0b}.
! Then, the GOTM turbulence module {\tt do\_turbulence} is called with
! all the transformed parameters discussed above. Finally, the
! vertical vectors {\tt tke1d}, {\tt eps1d}, {\tt num1d} and {\tt nuh1d}
! are transformed back to 3D fields.
!
! In case that the compiler option {\tt STRUCTURE\_FRICTION} is switched on,
! the additional turbulence production by structures in the water column is calculated
! by calculating the total production as
! \begin{equation}
! P_{tot} = P +C \left(u^2+v^2\right)^{3/2},
! \end{equation}
! with the shear production $P$, and the structure friction coefficient $C$. The
! latter is calculated in the routine {\tt structure\_friction\_3d.F90}.
!
! There are furthermore a number of compiler options provided, e.g.\
! for an older GOTM version, for barotropic calcuations,
! and for simple parabolic viscosity profiles circumventing the GOTM
! turbulence module.
!
! !USES:
   use halo_zones, only: update_3d_halo,wait_halo,H_TAG
   use domain, only: imin,imax,jmin,jmax,kmax,az,min_depth,crit_depth,z0,lonc,latc
   use variables_2d, only: D,z
   use variables_3d, only: dt,kmin,ho,hn,tke,eps,SS,num,taus,taub,zub,zvb,xxP
#ifndef NO_BAROCLINIC
   use variables_3d, only: NN,nuh,mld_tke
#endif
   use variables_3d, only: avmback,avhback
#ifdef STRUCTURE_FRICTION
   use variables_3d, only: uu,vv,hun,hvn,sf
#endif
   use turbulence, only: do_turbulence,cde
   use turbulence, only: tke1d => tke, eps1d => eps, L1d => L
   use turbulence, only: num1d => num, nuh1d => nuh
   use getm_timers,only: tic, toc, TIM_GOTM, TIM_GOTMTURB, TIM_GOTMH
   use meteo,      only: wind
   use getm_ice,   only: ice_hi
   IMPLICIT NONE
!
! !REVISION HISTORY:
!  Original author(s): Karsten Bolding & Hans Burchard
!
! !LOCAL VARIABLES:
   integer                   :: i,j,k
   REALTYPE                  :: u_taus,u_taub,z0s,z0b
   REALTYPE                  :: h(0:kmax),dry,zz
   REALTYPE                  :: NN1d(0:kmax),SS1d(0:kmax)
   REALTYPE                  :: xP(0:kmax)
   REALTYPE                  :: LCP(0:kmax)
   REALTYPE                  :: scaleflux,iw_factor,lc_factor,w10
   REALTYPE                  :: iw_ngob, iw_sgob, iw_gof
   REALTYPE                  :: iw_ice, dice
!EOP
!-----------------------------------------------------------------------
!BOC
! Note: For ifort we need to explicitly state that this routine is
! single-thread only. Presently I don't know why that is necessary,
! but if I use ifort -omp without any OMP-statements in this file,
! then the result is garbage.
! The OMP SINGLE or OMP MASTER statements helps, but sometimes it *still*
! messes up, in the sense that NaN "suddenly" appears on output.
! Apparently, writing out array-copy explicitly helps.
!    BJB 2009-09-17.
#ifdef DEBUG
   integer, save :: Ncall = 0
   Ncall = Ncall+1
   write(debug,*) 'gotm() # ',Ncall
#endif
   call tic(TIM_GOTM)

   xP = _ZERO_
#ifdef NO_BAROCLINIC
   NN1d = _ZERO_
#endif
   do j=jmin,jmax
      do i=imin,imax

         if (az(i,j) .ge. 1 ) then

#ifdef STRUCTURE_FRICTION
! BJB-TODO: Change all constants to double
            do k=1,kmax
               xP(k)= _QUART_*(                                                &
               (uu(i  ,j  ,k)/hun(i  ,j  ,k))**2*(sf(i  ,j  ,k)+sf(i+1,j  ,k)) &
              +(uu(i-1,j  ,k)/hun(i-1,j  ,k))**2*(sf(i-1,j  ,k)+sf(i  ,j  ,k)) &
              +(vv(i  ,j  ,k)/hvn(i  ,j  ,k))**2*(sf(i  ,j  ,k)+sf(i  ,j+1,k)) &
              +(vv(i  ,j-1,k)/hvn(i  ,j-1,k))**2*(sf(i  ,j-1,k)+sf(i  ,j  ,k)))
            end do
#endif
            
            u_taus    = sqrt(taus(i,j))
            u_taub    = sqrt(taub(i,j))
            
            ! account for missing mixing in the northern Gulf of Bothnia
            iw_ngob   = min( max( latc(i,j)-63.0d0 ,_ZERO_),_ONE_)
            ! account for missing mixing in the southern Gulf of Bothnia
            iw_sgob   = min( max( latc(i,j)-60.0d0 ,_ZERO_),_ONE_)
            ! account for missing mixing in the Gulf of Finnland
            iw_gof    = min( max( lonc(i,j)-22.0d0 ,_ZERO_),_ONE_)
            ! account for missing mixing in the Gotland Basin
            iw_factor = min( max( lonc(i,j)-14.0d0 ,_ZERO_),_ONE_)
            
            ! only icrease min_tke if ice is present
            iw_ice    = min( max( ice_hi(i,j)*20.0d0 ,_ZERO_),_ONE_)
            
            do k=0,kmax
               h(k)    = hn(i,j,k)
               SS1d(k) = SS(i,j,k)
#ifndef NO_BAROCLINIC
               NN1d(k) = NN(i,j,k)
#endif
               tke1d(k) = tke(i,j,k)
               ! account for missing mixing in the northern Gulf of Bothnia
               !tke1d(k) = max(30e-6*iw_ngob*iw_ice,tke1d(k))
               tke1d(k) = max(30e-6*iw_ngob*iw_ice,tke1d(k))
               ! account for missing mixing in the southern Gulf of Bothnia               
               !tke1d(k) = max(15e-6*iw_sgob*iw_ice,tke1d(k))                
               tke1d(k) = max(30e-6*iw_sgob*iw_ice,tke1d(k))               
               ! account for missing mixing in the Gulf of Finnland
               !tke1d(k) = max(4e-6*iw_gof*iw_ice,tke1d(k))  
               ! account for missing mixing in the Gotland Basin                            
               !tke1d(k) = max(6e-6*iw_factor,tke1d(k))                             
               tke1d(k) = max(2e-6*iw_factor,tke1d(k))  
                            
               eps1d(k) = eps(i,j,k)
               L1d(k)   = cde*tke1d(k)**1.5/eps1d(k)
               num1d(k) = num(i,j,k)
#ifndef NO_BAROCLINIC
               nuh1d(k) = nuh(i,j,k)
#endif
            end do            
            
            ! do some fake internal wave mixing Mellor (1994)
            iw_factor = min( max( lonc(i,j)-16.0d0 ,_ZERO_),_ONE_)
            xP        = _ZERO_
            xP        = abs(num1d * iw_factor * 0.7d0 * NN1d)
            
            ! do Langmuir turbulence according to Axell (2002)
            LCP = _ZERO_            
            ! Abschaltung des Windes ab 10 cm Eis.
            !w10 = ( _ONE_ - min( ice_hi(i,j)*10.0 , _ONE_ )) * wind(i,j)  
            w10 = wind(i,j)  
            call langmuir(kmax,w10,h,NN1d,LCP)
            xP  = xP + abs(LCP)
            
            xxP(i,j,:) = xP
            
            z0s = _TENTH_
            z0b = _HALF_*( max( z0(i,j) , zub(i-1,j  ) , zub(i,j) ) &
                          +max( z0(i,j) , zvb(i  ,j-1) , zvb(i,j) ) )
            if (z0s .gt. D(i,j)/10.) z0s= D(i,j)/10.

#ifdef PARABOLIC_VISCOSITY
            zz = _ZERO_
            do k=1,kmax-1
               zz=zz+hn(i,j,k)
! BJB-TODO: Get rid of **1.5 and **2
               tke1d(k)=max(1.e-10,3.333333d0*taub(i,j)*(_ONE_-zz/D(i,j)))
               L1d(k)=0.4d0*(zz+z0b)*sqrt(_ONE_-zz/D(i,j))
               eps1d(k)=0.16431677d0*sqrt(tke1d(k)*tke1d(k)*tke1d(k))/L1d(k)
               num1d(k)=0.09d0*tke1d(k)*tke1d(k)/eps1d(k)
#ifndef NO_BAROCLINIC
               nuh1d(k)=num1d(k)
#endif
            end do
#else
            ! If we do tic/toc for do_turbulence, then we can
            ! easily get into the millions of system_clock calls,
            ! as the call is deeply in loops
            !call tic(TIM_GOTMTURB)
            call do_turbulence(kmax,dt,D(i,j),u_taus,u_taub,z0s,z0b,h, &
                               NN1d,SS1d,xP)
            !call toc(TIM_GOTMTURB)
#endif
            do k=0,kmax
               tke(i,j,k) = tke1d(k)
               eps(i,j,k) = eps1d(k)
               num(i,j,k) = num1d(k) + avmback
#ifndef NO_BAROCLINIC
               nuh(i,j,k) = nuh1d(k) + avhback
#endif
            end do
            
            
         end if
      end do
   end do
   
   call mld_from_tke(1d-5,mld_tke,tke)

   call tic(TIM_GOTMH)
   call update_3d_halo(num,num,az,imin,jmin,imax,jmax,kmax,H_TAG)
   call wait_halo(H_TAG)
#ifndef NO_BAROCLINIC
   call update_3d_halo(nuh,nuh,az,imin,jmin,imax,jmax,kmax,H_TAG)
   call wait_halo(H_TAG)
#endif
   call toc(TIM_GOTMH)

   call toc(TIM_GOTM)
#ifdef DEBUG
   write(debug,*) 'Leaving gotm()'
   write(debug,*)
#endif
   return
   end subroutine gotm
   

!-----------------------------------------------------------------------
!BOP
!
! !ROUTINE: langmuir - source term parameterisation for Langmuir turbulence
!
! !INTERFACE:
   subroutine langmuir(nlev,wind,h,NN,xP)
!
! !DESCRIPTION:
!
! !USES:
   IMPLICIT NONE
!
! !INPUT PARAMETERS:
   integer , intent(in)  :: nlev
   REALTYPE, intent(in)  :: wind
   REALTYPE, intent(in)  :: h(0:nlev)
   REALTYPE, intent(in)  :: NN(0:nlev)
!
! !INPUT/OUTPUT PARAMETERS:
   REALTYPE, intent(inout) :: xP(0:nlev)
!
! !REVISION HISTORY:
!  Original author(s): Knut Klingbeil
!
! !LOCAL VARIABLES:
   integer, parameter        :: rk = kind(_ONE_)
   REALTYPE, parameter       :: grav = 9.81_rk
   REALTYPE, parameter       :: pi = 3.14159265358979323846_rk
   integer                   :: i,ii,ii0
   REALTYPE                  :: zi(0:nlev),L_lc,us,epot,H_lc,ekin,w_lc,H_lc_inv
!EOP
!-----------------------------------------------------------------------
!BOC
#ifdef DEBUG
   integer, save :: Ncall = 0
   Ncall = Ncall+1
   write(debug,*) 'langmuir() # ',Ncall
#endif

   zi(nlev) = _ZERO_
   do i=nlev-1,0,-1
     zi(i) = zi(i+1) - h(i+1)
   end do

   epot = _ZERO_
   H_lc = _ZERO_
   us   = 0.016_rk * wind 
   ekin = 0.5_rk * us*us
   
   do ii=nlev-1,1,-1
      if ( epot .gt. ekin ) then
         exit
      end if
      epot = epot - NN(ii)*zi(ii)*h(ii+1)
      H_lc = H_lc + h(ii+1)
   end do

   H_lc_inv = _ONE_ / H_lc
   do i=1,nlev-1
      if ( -H_lc .lt. zi(i) ) then
         w_lc = abs(0.15_rk * us * sin( -pi * zi(i) * H_lc_inv ))
         xP(i) = xP(i) + w_lc*w_lc*w_lc * H_lc_inv
      end if
   end do

#ifdef DEBUG
   write(debug,*) 'Leaving langmuir()'
   write(debug,*)
#endif
   return
   end subroutine langmuir
!EOC

!-----------------------------------------------------------------------
! Copyright (C) 2018 - Knut Klingbeil (IOW)                            !
!-----------------------------------------------------------------------

   
   
   
   ! !IROUTINE: depth_of_isoline - compute the depth of an isovalue
! \label{sec-depth_of_isoline}

! !INTERFACE:
   subroutine mld_from_tke(isolevel,isodepth,varin)
!
! !DESCRIPTION:
! interpolate out of a 3d field the depth of a isoline
!
! !USES:

   use variables_3d, only: hn
   use domain,       only: imin,imax,jmin,jmax,kmax,az, H
   
   IMPLICIT NONE
!
! !INPUT PARAMETERS:
   REALTYPE, intent(in)                :: isolevel
   REALTYPE, intent(out)               :: isodepth(E2DFIELD)
   REALTYPE, intent(in)                :: varin(I3DFIELD)
!
! !INPUT/OUTPUT PARAMETERS:
!
! !OUTPUT PARAMETERS:
!
! !LOCAL VARIABLES:
   integer                   :: i,j,k,n
   REALTYPE                  :: zlev(kmax)
   REALTYPE                  :: dz, s1, s2, z1 ,z2
   REALTYPE                  :: ss, cc
   REALTYPE                  :: var(I3DFIELD)
!EOP
!-------------------------------------------------------------------------
!BOC

   var = varin

   do j=jmin,jmax
      do i=imin,imax
         if( az(i,j) .eq. 1) then

            zlev(1) = -H(i,j) + _HALF_*hn(i,j,1)
            do k=2,kmax
               zlev(k) = zlev(k-1) + _HALF_*(hn(i,j,k-1)+hn(i,j,k))
            end do
                         
            isodepth(i,j) = H(i,j)
                        
            do k=(kmax-1),2,-1
               if (var(i,j,k) .le. isolevel ) then
                  s1 = var(i,j,k+1)
                  s2 = var(i,j,k)
                  z1 = zlev(k+1)
                  z2 = zlev(k)
                  
                  ss = (s1-s2)/(z1-z2)
                  cc = s1 - (s1-s2)/(z1-z2)*z1
                  
                  if ( ss .ne. _ZERO_) then
                     isodepth(i,j) = -(isolevel - cc)/ss
                  end if
                  
                  exit
               end if
            end do
                        
         else
            isodepth(i,j) = H(i,j)
         endif
         
         
      end do
   end do
   
            

   return
   end subroutine mld_from_tke
!EOC
!EOC

!-----------------------------------------------------------------------
! Copyright (C) 2001 - Hans Burchard and Karsten Bolding               !
!-----------------------------------------------------------------------

