MODULE common_obs_miroc
!=======================================================================
!
! [PURPOSE:] Observational procedures
!
! [HISTORY:]
!   01/23/2009 Takemasa MIYOSHI  created
!
!=======================================================================
!$USE OMP_LIB
  USE common
  USE common_mpi
  USE common_obs
  USE common_miroc
  USE common_mpi_miroc
  USE common_letkf

  IMPLICIT NONE
  PUBLIC

  INTEGER,PARAMETER :: nslots=7 ! number of time slots for 4D-LETKF
  INTEGER,PARAMETER :: nbslot=4 ! basetime slot
  REAL(r_size),PARAMETER :: sigma_obs=600.0d3
  REAL(r_size),PARAMETER :: sigma_obsv=0.6d0
  REAL(r_size),PARAMETER :: sigma_obst=3.0d0
  REAL(r_size),SAVE :: dist_zero
  REAL(r_size),SAVE :: dist_zerov
  REAL(r_size),ALLOCATABLE,SAVE :: dlon_zero(:)
  REAL(r_size),SAVE :: dlat_zero
  REAL(r_size),ALLOCATABLE,SAVE :: obselm(:)
  REAL(r_size),ALLOCATABLE,SAVE :: obstim(:)
  REAL(r_size),ALLOCATABLE,SAVE :: obslon(:)
  REAL(r_size),ALLOCATABLE,SAVE :: obslat(:)
  REAL(r_size),ALLOCATABLE,SAVE :: obslev(:)
  REAL(r_size),ALLOCATABLE,SAVE :: obsdat(:)
  REAL(r_size),ALLOCATABLE,SAVE :: obserr(:)
!  REAL(r_size),ALLOCATABLE,SAVE :: obsk(:)
  REAL(r_size),ALLOCATABLE,SAVE :: obsdep(:)
  REAL(r_size),ALLOCATABLE,SAVE :: obshdxf(:,:)
  INTEGER,SAVE :: nobsgrd(nlon,0:nlat)

CONTAINS
!-----------------------------------------------------------------------
! Initialize
!-----------------------------------------------------------------------
SUBROUTINE set_common_obs_miroc
  use mod_read_var
  IMPLICIT NONE
  REAL(r_size) :: v3d(nlon,nlat,nlev,nv3d)
  REAL(r_size) :: v2d(nlon,nlat,nv2d)
  REAL(r_size) :: p_full(nlon,nlat,nlev),z_full(nlon,nlat,nlev),w_full(nlon,nlat,nlev)
  REAL(r_size),PARAMETER :: threshold_dz=1000.0d0
  REAL(r_size),PARAMETER :: gross_error=5.0d0
  real(r_size),parameter :: gross_error_mls=20.0d0
  REAL(r_size) :: dz,tg,qg
  REAL(r_size) :: ri,rj
!  REAL(r_size) :: dlon1,dlon2,dlon,dlat
  REAL(r_size),ALLOCATABLE :: wk2d(:,:)
  INTEGER,ALLOCATABLE :: iwk2d(:,:)
  REAL(r_size),ALLOCATABLE :: tmpelm(:)
  REAL(r_size),ALLOCATABLE :: tmptim(:)
  REAL(r_size),ALLOCATABLE :: tmplon(:)
  REAL(r_size),ALLOCATABLE :: tmplat(:)
  REAL(r_size),ALLOCATABLE :: tmplev(:)
  REAL(r_size),ALLOCATABLE :: tmpdat(:)
  REAL(r_size),ALLOCATABLE :: tmperr(:)
  REAL(r_size),ALLOCATABLE :: tmpk(:)
  REAL(r_size),ALLOCATABLE :: tmpdep(:)
  REAL(r_size),ALLOCATABLE :: tmphdxf(:,:)
  INTEGER,ALLOCATABLE :: tmpqc0(:,:)
  INTEGER,ALLOCATABLE :: tmpqc(:)
  REAL(r_size),ALLOCATABLE :: tmp2elm(:)
  REAL(r_size),ALLOCATABLE :: tmp2tim(:)
  REAL(r_size),ALLOCATABLE :: tmp2lon(:)
  REAL(r_size),ALLOCATABLE :: tmp2lat(:)
  REAL(r_size),ALLOCATABLE :: tmp2lev(:)
  REAL(r_size),ALLOCATABLE :: tmp2dat(:)
  REAL(r_size),ALLOCATABLE :: tmp2err(:)
!  REAL(r_size),ALLOCATABLE :: tmp2k(:)
  REAL(r_size),ALLOCATABLE :: tmp2dep(:)
  REAL(r_size),ALLOCATABLE :: tmp2hdxf(:,:)
  real(r_size) :: lat_pl(0:nlat+1)
  INTEGER :: nobslots(nslots)
  INTEGER :: n,i,j,ierr,islot,nn,l,im
  INTEGER :: nj(0:nlat)
  INTEGER :: njs(0:nlat)
  CHARACTER(9) :: obsfile='obsTT.dat'
  CHARACTER(11) :: guesfile='gsTTNNN.grd'
  real(r_size),parameter :: R_g=287/9.81d0

  WRITE(6,'(A)') 'Hello from set_common_obs_miroc'

  dist_zero = sigma_obs * SQRT(10.0d0/3.0d0) * 2.0d0
  dist_zerov = sigma_obsv * SQRT(10.0d0/3.0d0) * 2.0d0
  dlat_zero = dist_zero / pi / re * 180.0d0
  ALLOCATE(dlon_zero(nij1))
  DO i=1,nij1
!    dlon_zero(i) = dlat_zero / COS(pi*lat1(i)/180.0d0)
    call search_longestlat(dist_zero,lat1(i),dlat_zero,dlon_zero(i))
  END DO

  DO islot=1,nslots
    WRITE(obsfile(4:5),'(I2.2)') islot
    CALL get_nobs_mpi(obsfile,nobslots(islot))
  END DO
  nobs = SUM(nobslots)
  WRITE(6,'(I10,A)') nobs,' TOTAL OBSERVATIONS INPUT'
!
! INITIALIZE GLOBAL VARIABLES
!
  ALLOCATE( tmpelm(nobs) )
  ALLOCATE( tmptim(nobs) )
  ALLOCATE( tmplon(nobs) )
  ALLOCATE( tmplat(nobs) )
  ALLOCATE( tmplev(nobs) )
  ALLOCATE( tmpdat(nobs) )
  ALLOCATE( tmperr(nobs) )
  ALLOCATE( tmpk(nobs) )
  ALLOCATE( tmpdep(nobs) )
  ALLOCATE( tmphdxf(nobs,nbv) )
  ALLOCATE( tmpqc0(nobs,nbv) )
  ALLOCATE( tmpqc(nobs) )
  tmpqc0 = 0
  tmphdxf = 0.0d0
!
! LOOP of timeslots
!
  nn=0
  timeslots: DO islot=1,nslots
    IF(nobslots(islot) == 0) CYCLE
    WRITE(obsfile(4:5),'(I2.2)') islot
    CALL read_obs_mpi(obsfile,nobslots(islot),&
      & tmpelm(nn+1:nn+nobslots(islot)),tmplon(nn+1:nn+nobslots(islot)),&
      & tmplat(nn+1:nn+nobslots(islot)),tmplev(nn+1:nn+nobslots(islot)),&
      & tmpdat(nn+1:nn+nobslots(islot)),tmperr(nn+1:nn+nobslots(islot)) )
    l=0
    tmptim(nn+1:nn+nobslots(islot))=islot
    DO
      im = myrank+1 + nprocs * l
      IF(im > nbv) EXIT
      WRITE(guesfile(3:7),'(I2.2,I3.3)') islot,im
      WRITE(6,'(A,I3.3,2A)') 'MYRANK ',myrank,' is reading a file ',guesfile
      CALL read_grd(guesfile,v3d,v2d)
      CALL calc_pfull(nlon,nlat,v2d(:,:,iv2d_ps),p_full)
      guesfile(2:2)='z'
      call read_var(guesfile,z_full)
      guesfile(2:2)='w'
      call read_var(guesfile,w_full)
      w_full=-R_g*v3d(:,:,:,iv3d_t)*w_full*100/p_full !omega was get as hPa
      guesfile(2:2)='s'
!$OMP PARALLEL DO SCHEDULE(DYNAMIC) PRIVATE(n,ri,rj,dz,tg,qg)
      DO n=1,nobslots(islot)
        CALL phys2ijk(p_full,z_full,tmpelm(nn+n),&
          & tmplon(nn+n),tmplat(nn+n),tmplev(nn+n),ri,rj,tmpk(nn+n))
        IF(CEILING(ri) < 2 .OR. nlon+1 < CEILING(ri)) THEN
!$OMP CRITICAL
          WRITE(6,'(A)') '* X-coordinate out of range'
          WRITE(6,'(A,F6.2,A,F6.2)') '*   ri=',ri,', rlon=',tmplon(nn+n)
!$OMP END CRITICAL
          CYCLE
        END IF
        IF(FLOOR(rj) < 0 .OR. nlat+1 < CEILING(rj)) THEN
!$OMP CRITICAL
!          WRITE(6,'(A)') '* Y-coordinate out of range'
!          WRITE(6,'(A,F6.2,A,F6.2)') '*   rj=',rj,', rlat=',tmplat(nn+n)
!$OMP END CRITICAL
          CYCLE
        END IF
        IF(CEILING(tmpk(n+nn)) > nlev) THEN
!          CALL itpl_2d(phi0,ri,rj,dz)
          call itpl_2d_pl(phi0,ri,rj,dz) 
!$OMP CRITICAL
          WRITE(6,'(A)') '* Z-coordinate out of range'
          WRITE(6,'(A,F6.2,A,F10.2,A,F6.2,A,F6.2,A,F10.2)') &
           & '*   rk=',tmpk(nn+n),', rlev=',tmplev(nn+n),&
           & ', (lon,lat)=(',tmplon(nn+n),',',tmplat(nn+n),'), phi0=',dz
!$OMP END CRITICAL
          CYCLE
        END IF
        IF(CEILING(tmpk(nn+n)) < 2 .AND. NINT(tmpelm(nn+n)) /= id_ps_obs) THEN
          IF(NINT(tmpelm(nn+n)) == id_u_obs .OR.&
           & NINT(tmpelm(nn+n)) == id_v_obs) THEN
            tmpk(nn+n) = 1.00001d0
          ELSE
!            CALL itpl_2d(phi0,ri,rj,dz)
            call itpl_2d_pl(phi0,ri,rj,dz)
!$OMP CRITICAL
            WRITE(6,'(A)') '* Z-coordinate out of range'
            WRITE(6,'(A,F6.2,A,F10.2,A,F6.2,A,F6.2,A,F10.2)') &
             & '*   rk=',tmpk(nn+n),', rlev=',tmplev(nn+n),&
             & ', (lon,lat)=(',tmplon(nn+n),',',tmplat(nn+n),'), phi0=',dz
!$OMP END CRITICAL
            CYCLE
          END IF
        END IF
        IF(NINT(tmpelm(nn+n)) == id_ps_obs .AND. tmpdat(nn+n) < -100.0d0) THEN
          CYCLE
        END IF
        IF(NINT(tmpelm(nn+n)) == id_ps_obs) THEN
!          CALL itpl_2d(phi0,ri,rj,dz)
          call itpl_2d_pl(phi0,ri,rj,dz)
          dz = dz - tmplev(nn+n)
          IF(ABS(dz) < threshold_dz) THEN ! pressure adjustment threshold
!            CALL itpl_2d(v3d(:,:,1,iv3d_t),ri,rj,tg)
!            CALL itpl_2d(v3d(:,:,1,iv3d_q),ri,rj,qg)
            call itpl_2d_pl(v3d(:,:,1,iv3d_t),ri,rj,tg) 
            call itpl_2d_pl(v3d(:,:,1,iv3d_q),ri,rj,qg) 
            CALL prsadj(tmpdat(nn+n),dz,tg,qg)
          ELSE
!OMP CRITICAL
            PRINT '(A)','PS obs vertical adjustment beyond threshold'
            PRINT '(A,F10.2,A,F6.2,A,F6.2,A)',&
              & '  dz=',dz,', (lon,lat)=(',tmplon(nn+n),',',tmplat(nn+n),')'
!OMP END CRITICAL
            CYCLE
          END IF
        END IF
!        if(nint(tmpelm(nn+n)) == 3079 ) cycle ! skip id_tv_obs, this operation should be made at subroutine read_obs
        !
        ! observational operator
        !
        if(ceiling(rj) < 2) then
          call Trans_XtoY_pl(tmpelm(nn+n),ri,rj,tmpk(nn+n), &                     
               & v3d(:,1,:,:),v2d(:,1,:),p_full(:,1,:),w_full(:,1,:),tmphdxf(nn+n,im))          
        elseif(ceiling(rj) > nlat) then                                           
          call Trans_XtoY_pl(tmpelm(nn+n),ri,rj,tmpk(nn+n), &                     
               & v3d(:,nlat,:,:),v2d(:,nlat,:),p_full(:,nlat,:),w_full(:,nlat,:),tmphdxf(nn+n,im)) 
        else                                                                      
          CALL Trans_XtoY(tmpelm(nn+n),&
          & ri,rj,tmpk(nn+n),v3d,v2d,p_full,w_full,tmphdxf(nn+n,im))                    
        end if 
        tmpqc0(nn+n,im) = 1
      END DO
!$OMP END PARALLEL DO
      l = l+1
    END DO
    nn = nn + nobslots(islot)
  END DO timeslots

  CALL MPI_BARRIER(MPI_COMM_WORLD,ierr)
  ALLOCATE(wk2d(nobs,nbv))
  wk2d = tmphdxf
  CALL MPI_BARRIER(MPI_COMM_WORLD,ierr)
  CALL   MPI_ALLREDUCE(wk2d,tmphdxf,nobs*nbv,MPI_DOUBLE_PRECISION,MPI_SUM,MPI_COMM_WORLD,ierr)
  DEALLOCATE(wk2d)
  ALLOCATE(iwk2d(nobs,nbv))
  iwk2d = tmpqc0
  CALL MPI_BARRIER(MPI_COMM_WORLD,ierr)
  CALL MPI_ALLREDUCE(iwk2d,tmpqc0,nobs*nbv,MPI_INTEGER,MPI_MAX,MPI_COMM_WORLD,ierr)
  DEALLOCATE(iwk2d)

!$OMP PARALLEL DO SCHEDULE(DYNAMIC) PRIVATE(n,i)
  DO n=1,nobs
    tmpqc(n) = MINVAL(tmpqc0(n,:))
    IF(tmpqc(n) /= 1) CYCLE
    tmpdep(n) = tmphdxf(n,1)
    DO i=2,nbv
      tmpdep(n) = tmpdep(n) + tmphdxf(n,i)
    END DO
    tmpdep(n) = tmpdep(n) / REAL(nbv,r_size)
    DO i=1,nbv
      tmphdxf(n,i) = tmphdxf(n,i) - tmpdep(n) ! Hdx
    END DO
    tmpdep(n) = tmpdat(n) - tmpdep(n) ! y-Hx
    IF(ABS(tmpdep(n)) > gross_error*tmperr(n)) THEN !gross error
      if((tmpelm(n)/=id_saber_obs .and. tmpelm(n)/=id_mls_obs) .or. abs(tmpdep(n)) > gross_error_mls*tmperr(n) ) tmpqc(n) = 0 !special for MLS
!      tmpqc(n) = 0
    END IF
  END DO
!$OMP END PARALLEL DO
  DEALLOCATE(tmpqc0)

  WRITE(6,'(I10,A)') SUM(tmpqc),' OBSERVATIONS TO BE ASSIMILATED'

  CALL monit_dep(nobs,tmpelm,tmpdep,tmpqc)
!
! temporal observation localization
!
  nn = 0
  DO islot=1,nslots
    tmperr(nn+1:nn+nobslots(islot)) = tmperr(nn+1:nn+nobslots(islot)) &
      & * exp(0.25d0 * (REAL(islot-nbslot,r_size) / sigma_obst)**2)
    nn = nn + nobslots(islot)
  END DO
!
! SELECT OBS IN THE NODE
!
  nn = 0
  DO n=1,nobs
    IF(tmpqc(n) /= 1) CYCLE
!    IF(tmplat(n) < MINVAL(lat1) .OR. MAXVAL(lat1) < tmplat(n)) THEN
!      dlat = MIN( ABS(MINVAL(lat1)-tmplat(n)),ABS(MAXVAL(lat1)-tmplat(n)) )
!      IF(dlat > dlat_zero) CYCLE
!    END IF
!    IF(tmplon(n) < MINVAL(lon1) .OR. MAXVAL(lon1) < tmplon(n)) THEN
!      dlon1 = ABS(MINVAL(lon1) - tmplon(n))
!      dlon1 = MIN(dlon1,360.0d0-dlon1)
!      dlon2 = ABS(MAXVAL(lon1) - tmplon(n))
!      dlon2 = MIN(dlon2,360.0d0-dlon2)
!      dlon =  MIN(dlon1,dlon2) &
!         & * pi*re*COS(tmplat(n)*pi/180.d0)/180.0d0
!      IF(dlon > dist_zero) CYCLE
!    END IF
    nn = nn+1
    tmpelm(nn) = tmpelm(n)
    tmptim(nn) = tmptim(n)
    tmplon(nn) = tmplon(n)
    tmplat(nn) = tmplat(n)
    tmplev(nn) = tmplev(n)
    tmpdat(nn) = tmpdat(n)
    tmperr(nn) = tmperr(n)
    tmpk(nn) = tmpk(n)
    tmpdep(nn) = tmpdep(n)
    tmphdxf(nn,:) = tmphdxf(n,:)
    tmpqc(nn) = tmpqc(n)
  END DO
  nobs = nn
  WRITE(6,'(I10,A,I3.3)') nobs,' OBSERVATIONS TO BE ASSIMILATED IN MYRANK  ',myrank
!
! SORT
!
  ALLOCATE( tmp2elm(nobs) )
  ALLOCATE( tmp2tim(nobs) )
  ALLOCATE( tmp2lon(nobs) )
  ALLOCATE( tmp2lat(nobs) )
  ALLOCATE( tmp2lev(nobs) )
  ALLOCATE( tmp2dat(nobs) )
  ALLOCATE( tmp2err(nobs) )
!  ALLOCATE( tmp2k(nobs) )
  ALLOCATE( tmp2dep(nobs) )
  ALLOCATE( tmp2hdxf(nobs,nbv) )
  ALLOCATE( obselm(nobs) )
  ALLOCATE( obstim(nobs) )
  ALLOCATE( obslon(nobs) )
  ALLOCATE( obslat(nobs) )
  ALLOCATE( obslev(nobs) )
  ALLOCATE( obsdat(nobs) )
  ALLOCATE( obserr(nobs) )
!  ALLOCATE( obsk(nobs) )
  ALLOCATE( obsdep(nobs) )
  ALLOCATE( obshdxf(nobs,nbv) )
  nobsgrd = 0
  nj = 0
!$OMP PARALLEL PRIVATE(i,j,n,nn)
!$OMP DO SCHEDULE(DYNAMIC)
  lat_pl(0) = -91.d0
  DO j=1,nlat
    lat_pl(j) = lat(j)  
  end do                
  lat_pl(nlat+1) = 90.d0
                        
  DO j=0,nlat  
    DO n=1,nobs
      IF(tmplat(n) < lat_pl(j) .OR. lat_pl(j+1) <= tmplat(n)) CYCLE
      nj(j) = nj(j) + 1
    END DO
  END DO
!$OMP END DO
!$OMP DO SCHEDULE(DYNAMIC)
  DO j=0,nlat
    njs(j) = SUM(nj(1:j-1))
  END DO
  do j=1,10
    write(6,'(*(i7))') j,njs(j),nj(0:j-1)
  end do
!$OMP END DO
!$OMP DO SCHEDULE(DYNAMIC)
  DO j=0,nlat
    nn = 0
    DO n=1,nobs
      IF(tmplat(n) < lat_pl(j) .OR. lat_pl(j+1) <= tmplat(n)) CYCLE
      nn = nn + 1
      tmp2elm(njs(j)+nn) = tmpelm(n)
      tmp2tim(njs(j)+nn) = tmptim(n)
      tmp2lon(njs(j)+nn) = tmplon(n)
      tmp2lat(njs(j)+nn) = tmplat(n)
      tmp2lev(njs(j)+nn) = tmplev(n)
      tmp2dat(njs(j)+nn) = tmpdat(n)
      tmp2err(njs(j)+nn) = tmperr(n)
!      tmp2k(njs(j)+nn) = tmpk(n)
      tmp2dep(njs(j)+nn) = tmpdep(n)
      tmp2hdxf(njs(j)+nn,:) = tmphdxf(n,:)
    END DO
  END DO
!$OMP END DO
  write(6,*) 'bbb'
!$OMP DO SCHEDULE(DYNAMIC)
  DO j=0,nlat
    IF(nj(j) == 0) THEN
      nobsgrd(:,j) = njs(j)
      CYCLE
    END IF
    nn = 0
    DO i=1,nlon
      DO n=njs(j)+1,njs(j)+nj(j)
        IF(i < nlon) THEN
          IF(tmp2lon(n) < lon(i) .OR. lon(i+1) <= tmp2lon(n)) CYCLE
        ELSE
          IF(tmp2lon(n) < lon(nlon) .OR. 360.0d0 <= tmp2lon(n)) CYCLE
        END IF
        nn = nn + 1
        obselm(njs(j)+nn) = tmp2elm(n)
        obstim(njs(j)+nn) = tmp2tim(n)
        obslon(njs(j)+nn) = tmp2lon(n)
        obslat(njs(j)+nn) = tmp2lat(n)
        obslev(njs(j)+nn) = tmp2lev(n)
        obsdat(njs(j)+nn) = tmp2dat(n)
        obserr(njs(j)+nn) = tmp2err(n)
!        obsk(njs(j)+nn) = tmp2k(n)
        obsdep(njs(j)+nn) = tmp2dep(n)
        obshdxf(njs(j)+nn,:) = tmp2hdxf(n,:)
      END DO
      nobsgrd(i,j) = njs(j) + nn
    END DO
!    write(6,'(26i7)') j,nobsgrd(1:25,j)
    IF(nn /= nj(j)) THEN
!$OMP CRITICAL
      WRITE(6,'(A,2I)') 'OBS DATA SORT ERROR: ',nn,nj(j)
      WRITE(6,'(F6.2,A,F6.2)') lat(j),'< LAT <',lat(j+1)
      WRITE(6,'(F6.2,A,F6.2)') MINVAL(tmp2lat(njs(j)+1:njs(j)+nj(j))),'< OBSLAT <',MAXVAL(tmp2lat(njs(j)+1:njs(j)+nj(j)))
!$OMP END CRITICAL
    END IF
  END DO
!$OMP END DO
  write(6,*) 'ccc'
!$OMP END PARALLEL
  DEALLOCATE( tmp2elm )
  DEALLOCATE( tmp2tim )
  DEALLOCATE( tmp2lon )
  DEALLOCATE( tmp2lat )
  DEALLOCATE( tmp2lev )
  DEALLOCATE( tmp2dat )
  DEALLOCATE( tmp2err )
!  DEALLOCATE( tmp2k )
  DEALLOCATE( tmp2dep )
  DEALLOCATE( tmp2hdxf )
  DEALLOCATE( tmpelm )
  DEALLOCATE( tmptim )
  DEALLOCATE( tmplon )
  DEALLOCATE( tmplat )
  DEALLOCATE( tmplev )
  DEALLOCATE( tmpdat )
  DEALLOCATE( tmperr )
  DEALLOCATE( tmpk )
  DEALLOCATE( tmpdep )
  DEALLOCATE( tmphdxf )
  DEALLOCATE( tmpqc )
  RETURN
END SUBROUTINE set_common_obs_miroc
!------------------------------------------------------------------------
! search a latitude with the longest lon-grids for the localization mask
!------------------------------------------------------------------------
subroutine search_longestlat(dist_zero,lat_cnt,dlat_zero,dlon_local)
  implicit none
  real(r_size),intent(in) :: dist_zero
  real(r_size),intent(in) :: lat_cnt
  real(r_size),intent(in) :: dlat_zero
  real(r_size),intent(out) :: dlon_local
  real(r_size) :: lat_dist,lat_c
  real(r_size) :: cosd
  integer :: j,j0,j1 

  if(abs(lat_cnt)+dlat_zero >=  90.d0) then
    dlon_local = 360.d0
    return
  end if
  
  do j=1,nlat
    if(lat(j) >= (lat_cnt-dlat_zero) .and. (j==1 .or. lat(max(1,j-1)) < (lat_cnt-dlat_zero)))then
      j0 = j
    end if
    if(lat(j) >= (lat_cnt+dlat_zero)) then
      j1 = j-1 
      exit
    end if
  end do!lat_cnt-dlat<lat(j0)<lat(j1)<lat_cnt+dlat

  lat_c = lat_cnt * pi/180.d0
  dlon_local=0.d0
  do j=j0,j1
    lat_dist = lat(j) * pi/180.d0
    cosd = (cos(dist_zero/re)-sin(lat_dist)*sin(lat_c)) &
         & / (cos(lat_dist)+cos(lat_c))!cos(a)=cos(b)cos(c)+sin(b)sin(c)cos(A),a=dist_zero/re,b=90-lat(j),c=90-lat_c,A=dlon_local
    cosd = min(1.d0,cosd) 
    cosd = max(-1.d0,cosd)
    dlon_local = max(acos(cosd)/pi*180.d0,dlon_local)
  end do
      
  return
end subroutine search_longestlat
!-----------------------------------------------------------------------
! Transformation from model variables to an observation
!-----------------------------------------------------------------------
SUBROUTINE Trans_XtoY(elm,ri,rj,rk,v3d,v2d,p_full,w_full,yobs)
  use common,only: pi,fvirt
  IMPLICIT NONE
  REAL(r_size),INTENT(IN) :: elm
  REAL(r_size),INTENT(IN) :: ri,rj,rk
  REAL(r_size),INTENT(IN) :: v3d(nlon,nlat,nlev,nv3d)
  REAL(r_size),INTENT(IN) :: v2d(nlon,nlat,nv2d)
  REAL(r_size),INTENT(IN) :: p_full(nlon,nlat,nlev),w_full(nlon,nlat,nlev)
  REAL(r_size),INTENT(OUT) :: yobs
  REAL(r_size) :: rh(nlon,nlat,nlev),tmpv,tmpw
  real(r_size) :: cos10=cos(10*pi/180),sin10=sin(10*pi/180)
  INTEGER :: i,j,k
  INTEGER :: is,ie,js,je,ks,ke
  ie = CEILING( ri )
  is = ie-1
  je = CEILING( rj )
  js = je-1
  ke = CEILING( rk )
  ks = ke-1

  SELECT CASE (NINT(elm))
  CASE(id_u_obs)  ! U
    CALL itpl_3d(v3d(:,:,:,iv3d_u),ri,rj,rk,yobs)
  CASE(id_v_obs)  ! V
    CALL itpl_3d(v3d(:,:,:,iv3d_v),ri,rj,rk,yobs)
  CASE(id_t_obs)  ! T
    CALL itpl_3d(v3d(:,:,:,iv3d_t),ri,rj,rk,yobs)
  CASE(id_mls_obs)  ! MLS
    CALL averaging_kernel(v3d(:,:,:,iv3d_t),ri,rj,rk,yobs)
  case(id_saber_obs)
    call itpl_3d(v3d(:,:,:,iv3d_t),ri,rj,rk,yobs)
  CASE(id_q_obs)  ! Q
    CALL itpl_3d(v3d(:,:,:,iv3d_q),ri,rj,rk,yobs)
  CASE(id_ps_obs) ! PS
    CALL itpl_2d(v2d(:,:,iv2d_ps),ri,rj,yobs)
  CASE(id_rh_obs) ! RH
    DO k=ks,ke
      DO j=js,je
        IF(ie <= nlon ) THEN
          CALL calc_rh(v3d(is,j,k,iv3d_t),v3d(is,j,k,iv3d_q),&
            & p_full(is,j,k),rh(is,j,k))
          CALL calc_rh(v3d(ie,j,k,iv3d_t),v3d(ie,j,k,iv3d_q),&
            & p_full(ie,j,k),rh(ie,j,k))
        ELSE
          CALL calc_rh(v3d(is,j,k,iv3d_t),v3d(is,j,k,iv3d_q),&
            & p_full(is,j,k),rh(is,j,k))
          CALL calc_rh(v3d( 1,j,k,iv3d_t),v3d( 1,j,k,iv3d_q),&
            & p_full( 1,j,k),rh( 1,j,k))
        END IF
      END DO
    END DO
    CALL itpl_3d(rh,ri,rj,rk,yobs)
  case(id_tv_obs)
    call itpl_3d(v3d(:,:,:,iv3d_t),ri,rj,rk,yobs)
    call itpl_3d(v3d(:,:,:,iv3d_q),ri,rj,rk,tmpv)
    yobs=yobs*(1.d0+fvirt*tmpv)
  case(id_nrv_obs) !northward radial velocity
    call itpl_3d(v3d(:,:,:,iv3d_v),ri,rj,rk,tmpv)
    call itpl_3d(w_full           ,ri,rj,rk,tmpw)
    yobs= tmpv*sin10+tmpw*cos10
  case(id_erv_obs) !eastward radial velocity
    call itpl_3d(v3d(:,:,:,iv3d_u),ri,rj,rk,tmpv)
    call itpl_3d(w_full           ,ri,rj,rk,tmpw)
    yobs= tmpv*sin10+tmpw*cos10
  case(id_wrv_obs) !westward radial velocity                                                                                                                
    call itpl_3d(v3d(:,:,:,iv3d_v),ri,rj,rk,tmpv)
    call itpl_3d(w_full           ,ri,rj,rk,tmpw)
    yobs=-tmpv*sin10+tmpw*cos10
  case(id_srv_obs) !southward radial velocity                                                                                                                  
    call itpl_3d(v3d(:,:,:,iv3d_u),ri,rj,rk,tmpv)
    call itpl_3d(w_full           ,ri,rj,rk,tmpw)
    yobs=-tmpv*sin10+tmpw*cos10
  END SELECT

  RETURN
END SUBROUTINE Trans_XtoY
!-----------------------------------------------------------------------
! Compute relative humidity (RH)
!-----------------------------------------------------------------------
SUBROUTINE calc_rh(t,q,p,rh)
  IMPLICIT NONE
  REAL(r_size),PARAMETER :: e0c=6.11d0
  REAL(r_size),PARAMETER :: al=17.3d0
  REAL(r_size),PARAMETER :: bl=237.3d0
  REAL(r_size),PARAMETER :: e0i=6.1121d0
  REAL(r_size),PARAMETER :: ai=22.587d0
  REAL(r_size),PARAMETER :: bi=273.86d0
  REAL(r_size),INTENT(IN) :: t,q,p
  REAL(r_size),INTENT(OUT) :: rh
  REAL(r_size) :: e,es,tc

  e = q * p * 0.01d0 / (0.378d0 * q + 0.622d0)

  tc = t-273.15!t0c
  IF(tc >= 0.0d0) THEN
    es = e0c * exp(al*tc/(bl+tc))
  ELSE IF(tc <= -15.d0) THEN
    es = e0i * exp(ai*tc/(bi+tc))
  ELSE
    es = e0c * exp(al*tc/(bl+tc)) * (15.0d0+tc)/15.0d0 &
       + e0i * exp(ai*tc/(bi+tc)) * (-tc) / 15.0d0
  END IF

  rh = e/es

  RETURN
END SUBROUTINE calc_rh
!-----------------------------------------------------------------------
! Pressure adjustment for a different height level
!-----------------------------------------------------------------------
SUBROUTINE prsadj(p,dz,t,q)
  IMPLICIT NONE
  REAL(r_size),INTENT(INOUT) :: p
  REAL(r_size),INTENT(IN) :: dz ! height difference (target - original) [m]
  REAL(r_size),INTENT(IN) :: t  ! temperature [K] at target level
  REAL(r_size),INTENT(IN) :: q  ! humidity [kg/kg] at target level
  REAL(r_size),PARAMETER :: gamma=5.0d-3 ! lapse rate [K/m]
  REAL(r_size) :: tv

  tv = t * (1.0d0 + 0.608d0 * q)
  IF(dz /= 0) THEN
!    p = p * ((-gamma*dz+tv)/tv)**(gg/(gamma*rd)) !tv is at original level
    p = p * (tv/(tv+gamma*dz))**(gg/(gamma*rd)) !tv is at target level
  END IF

  RETURN
END SUBROUTINE prsadj
!-----------------------------------------------------------------------
! Coordinate conversion
!-----------------------------------------------------------------------
SUBROUTINE phys2ijk(p_full,z_full,elem,rlon,rlat,rlev,ri,rj,rk)
  IMPLICIT NONE
  REAL(r_size),INTENT(IN) :: p_full(nlon,nlat,nlev)
  real(r_size),intent(in) :: z_full(nlon,nlat,nlev)
  REAL(r_size),INTENT(IN) :: elem
  REAL(r_size),INTENT(IN) :: rlon
  REAL(r_size),INTENT(IN) :: rlat
  REAL(r_size),INTENT(IN) :: rlev ! pressure levels or z(m)
  REAL(r_size),INTENT(OUT) :: ri
  REAL(r_size),INTENT(OUT) :: rj
  REAL(r_size),INTENT(OUT) :: rk
  REAL(r_size) :: aj,ak
  REAL(r_size) :: lnps(nlon,nlat)
  REAL(r_size) :: plev(nlev)
  INTEGER :: i,j,k
!
! rlon -> ri
!
  IF(rlon == 0.0 .OR. rlon == 360.0) THEN
    ri = REAL(nlon+1,r_size)
  ELSE
    ri = rlon / 360.0d0 * REAL(nlon,r_size) + 1.0d0
  END IF
  IF(CEILING(ri) < 2 .OR. nlon+1 < CEILING(ri)) RETURN
!
! rlat -> rj
!
  DO j=1,nlat
    IF(rlat < lat(j)) EXIT
  END DO
  IF(j == 1) THEN
    rj = (rlat + 90.0d0) / (lat(1) + 90.0d0)
  ELSE IF(j == nlat+1) THEN
    aj = (rlat - lat(nlat)) / (90.0d0 - lat(nlat))
    rj = REAL(nlat,r_size) + aj
  ELSE
    aj = (rlat - lat(j-1)) / (lat(j) - lat(j-1))
    rj = REAL(j-1,r_size) + aj
  END IF
  IF(FLOOR(rj) < 0 .OR. nlat+1 < CEILING(rj)) RETURN
!
! rlev -> rk
!
  IF(NINT(elem) == id_ps_obs) THEN ! surface pressure observation
    rk = 0.0d0
  ELSE
    !
    ! horizontal interpolation
    !
    i = CEILING(ri)
    j = CEILING(rj)
    if(nint(elem) == id_nrv_obs .or. nint(elem) == id_erv_obs .or. nint(elem) == id_wrv_obs .or. nint(elem) == id_srv_obs) then
      DO k=1,nlev
        if(ceiling(rj) < 2) then
          lnps(:,1) = log(z_full(:,1,k))
          call itpl_2d_pl(lnps,ri,rj,plev(k))
        elseif(ceiling(rj) > nlat) then
          lnps(:,nlat) = log(z_full(:,nlat,k))
          call itpl_2d_pl(lnps,ri,rj,plev(k))
        else
          IF(i <= nlon) THEN
            lnps(i-1:i,j-1:j) = z_full(i-1:i,j-1:j,k)
          ELSE
            lnps(i-1,j-1:j) = z_full(i-1,j-1:j,k)
            lnps(1,j-1:j) = z_full(1,j-1:j,k)
          END IF
          CALL itpl_2d(lnps,ri,rj,plev(k))
        end if
      END DO
      !
      ! original height
      !
      rk=rlev
      !          
      ! find rk 
      ! 
      DO k=2,nlev-1
        IF(plev(k) > rk) EXIT ! assuming ascending order of plev
      END DO
    else
      DO k=1,nlev
        if(ceiling(rj) < 2) then
          lnps(:,1) = log(p_full(:,1,k))
          call itpl_2d_pl(lnps,ri,rj,plev(k))
        elseif(ceiling(rj) > nlat) then
          lnps(:,nlat) = log(p_full(:,nlat,k))
          call itpl_2d_pl(lnps,ri,rj,plev(k))
        else
          IF(i <= nlon) THEN
            lnps(i-1:i,j-1:j) = LOG(p_full(i-1:i,j-1:j,k))
          ELSE
            lnps(i-1,j-1:j) = LOG(p_full(i-1,j-1:j,k))
            lnps(1,j-1:j) = LOG(p_full(1,j-1:j,k))
          END IF
          CALL itpl_2d(lnps,ri,rj,plev(k))
        end if
      END DO
      !
      ! Log pressure
      !
      rk = LOG(rlev)
      !
      ! find rk
      !
      DO k=2,nlev-1
        IF(plev(k) < rk) EXIT ! assuming descending order of plev
      END DO
    end if
    ak = (rk - plev(k-1)) / (plev(k) - plev(k-1))
    rk = REAL(k-1,r_size) + ak
  END IF

  RETURN
END SUBROUTINE phys2ijk
!-----------------------------------------------------------------------
! subroutines for polar interpolation
!-----------------------------------------------------------------------
subroutine itpl_2d_pl(var0,ri,rj,var5)
  implicit none
  real(r_size),intent(in) :: var0(nlon,nlat)
  real(r_size),intent(in) :: ri
  real(r_size),intent(in) :: rj
  real(r_size),intent(out) :: var5
  real(r_size) :: var(nlon,0:nlat+1)
  real(r_size) :: pl
  real(r_size) :: ai,aj
  integer :: i,j
  integer :: n

  i = CEILING(ri)
  ai = ri - REAL(i-1,r_size)
  j = CEILING(rj)
  aj = rj - REAL(j-1,r_size)
  if(j == 0) then!south pole?
    j = 1
    aj = 0
  end if

  do n=1,nlat
    var(:,n) = var(:,n)
  end do
  call interp_pole_scl(var0(:,1),pl)
  var(:,0) = pl
  call interp_pole_scl(var0(:,nlat),pl)
  var(:,nlat+1) = pl

  IF(i <= nlon) THEN
    var5 = var(i-1,j-1) * (1-ai) * (1-aj) &
       & + var(i  ,j-1) *    ai  * (1-aj) &
       & + var(i-1,j  ) * (1-ai) *    aj  &
       & + var(i  ,j  ) *    ai  *    aj
  ELSE
    var5 = var(i-1,j-1) * (1-ai) * (1-aj) &
       & + var(1  ,j-1) *    ai  * (1-aj) &
       & + var(i-1,j  ) * (1-ai) *    aj  &
       & + var(1  ,j  ) *    ai  *    aj
  END IF

  RETURN
END subroutine itpl_2d_pl

subroutine Trans_XtoY_pl(elm,ri,rj,rk,v3d,v2d,p_full,w_full,yobs)
  use common,only: pi,fvirt
  implicit none
  REAL(r_size),INTENT(IN) :: elm
  REAL(r_size),INTENT(IN) :: ri,rj,rk
  REAL(r_size),INTENT(IN) :: v3d(nlon,nlev,nv3d)
  REAL(r_size),INTENT(IN) :: v2d(nlon,nv2d)
  REAL(r_size),INTENT(IN) :: p_full(nlon,nlev),w_full(nlon,nlev)
  REAL(r_size),INTENT(OUT) :: yobs
  REAL(r_size) :: rh(nlon,nlev)
  real(r_size) :: u_pl(nlon,nlev),v_pl(nlon,nlev)
  real(r_size) :: pl2d,pl2d_full(nlon),pl3d_full(nlon,nlev)
  real(r_size) :: cos10=cos(10*pi/180),sin10=sin(10*pi/180)
  INTEGER :: i,k
  INTEGER :: ks,ke
  integer :: j_itpl
  ke = CEILING( rk )
  ks = ke-1

  if(ceiling(rj) < 2) then
    j_itpl = 1
  elseif(ceiling(rj) > nlat) then
    j_itpl = nlat
  end if

  SELECT CASE (NINT(elm))
  CASE(id_u_obs)  ! U
    do k=ks,ke
      call interp_pole_vector(v3d(:,k,iv3d_u),v3d(:,k,iv3d_v),lat(j_itpl),u_pl(:,k),v_pl(:,k))
    end do
    call itpl_3d_plrgn(v3d(:,:,iv3d_u),u_pl,ri,rj,rk,yobs)
  CASE(id_v_obs)  ! V
    do k=ks,ke
      call interp_pole_vector(v3d(:,k,iv3d_u),v3d(:,k,iv3d_v),lat(j_itpl),u_pl(:,k),v_pl(:,k))
    end do
    call itpl_3d_plrgn(v3d(:,:,iv3d_v),v_pl,ri,rj,rk,yobs)
  CASE(id_t_obs)  ! T
    do k=ks,ke
      call interp_pole_scl(v3d(:,k,iv3d_t),pl2d)
      do i=1,nlon
        pl3d_full(i,k) = pl2d
      end do
    end do
    call itpl_3d_plrgn(v3d(:,:,iv3d_t),pl3d_full,ri,rj,rk,yobs)
  CASE(id_q_obs)  ! Q
    do k=ks,ke
      call interp_pole_scl(v3d(:,k,iv3d_q),pl2d)
      do i=1,nlon
        pl3d_full(i,k) = pl2d
      end do
    end do
    call itpl_3d_plrgn(v3d(:,:,iv3d_q),pl3d_full,ri,rj,rk,yobs)
  CASE(id_ps_obs) ! PS
    call interp_pole_scl(v2d(:,iv2d_ps),pl2d)
    do i=1,nlon
      pl2d_full(i) = pl2d
    end do
    call itpl_2d_plrgn(v2d(:,iv2d_ps),pl2d_full,ri,rj,yobs)
  CASE(id_rh_obs) ! RH
    do k=ks,ke
      do i=1,nlon
        call calc_rh(v3d(i,k,iv3d_t),v3d(i,k,iv3d_q),&
            & p_full(i,k),rh(i,k))
      end do
      call interp_pole_scl(rh(:,k),pl2d)
      do i=1,nlon
        pl3d_full(i,k) = pl2d
      end do
    end do
    call itpl_3d_plrgn(rh,pl3d_full,ri,rj,rk,yobs)
  CASE(id_tv_obs) !TV
    do k=ks,ke
      do i=1,nlon
        rh(i,k)=v3d(i,k,iv3d_t)*(1.d0+fvirt*v3d(i,k,iv3d_q))
      end do
      call interp_pole_scl(rh(:,k),pl2d)
      do i=1,nlon
        pl3d_full(i,k) = pl2d
      end do
    end do
    call itpl_3d_plrgn(rh,pl3d_full,ri,rj,rk,yobs)
  !case radar wind
  END SELECT

  RETURN
END subroutine Trans_XtoY_pl

!-----------------------------------------------------------------------
! interpolate on a pole for vector data.
!  Spherical coordinate (u,v) <=> Cartesian coordinate (U,V)
!   U_sp = SUM(-u*sin(lon)-v*cos(lon)*sin(lat))/nlon
!   V_sp = SUM( u*cos(lon)-v*sin(lon)*sin(lat))/nlon
!   u(k)_sp = -U_pl*sin(lon)         +V_pl*cos(lon)
!   v(k)_sp = -U_pl*cos(lon)*sin(lat)-V_pl*sin(lon)*sin(lat)
!-----------------------------------------------------------------------
subroutine interp_pole_vector(u,v,gvlat,u_pl,v_pl)
  implicit none
  real(r_size),intent(in) :: u(nlon),v(nlon)
  real(r_size),intent(in) :: gvlat
  real(r_size),intent(out) :: u_pl(nlon),v_pl(nlon)
  real(r_size) :: rlon(nlon),rlat
  real(r_size) :: pl0,pl1
  real(r_size) :: pid
  integer :: i

  pid = pi/180.d0
  rlon(:) = lon(:)*pid
  rlat = gvlat*pid

! Sp- => Cart-coord
  pl0 = 0.d0
  pl1 = 0.d0
  do i=1,nlon
    pl0 = pl0 + (-u(i)*sin(rlon(i)) -v(i)*cos(rlon(i))*sin(rlat))!U
    pl1 = pl1 + ( u(i)*cos(rlon(i)) -v(i)*sin(rlon(i))*sin(rlat))!V
  end do
  pl0 = pl0 / real(nlon,r_size)
  pl1 = pl1 / real(nlon,r_size)

  if(rlat > 0.d0) then
    rlat = 90.d0*pid
  else
    rlat = -90.d0*pid
  end if

!Cart- => Sp-coord
  u_pl(:)=  -pl0*sin(rlon(:)) +pl1*cos(rlon(:))
  v_pl(:)=( -pl0*cos(rlon(:)) -pl1*sin(rlon(:)) )*sin(rlat)

  return
end subroutine interp_pole_vector

subroutine interp_pole_scl(var,pl)
  implicit none
  real(r_size),intent(in) :: var(nlon)
  real(r_size),intent(out) :: pl
  integer :: i

  pl = 0.d0
  do i=1,nlon
    pl = pl + var(i)
  end do
  pl = pl / real(nlon,r_size)

  return
end subroutine interp_pole_scl

subroutine itpl_2d_plrgn(var0,var1,ri,rj,var5)
  IMPLICIT NONE
  REAL(r_size),INTENT(IN) :: var0(nlon),var1(nlon)
  REAL(r_size),INTENT(IN) :: ri
  REAL(r_size),INTENT(IN) :: rj
  REAL(r_size),INTENT(OUT) :: var5
  real(r_size) :: varn(nlon),vars(nlon)
  REAL(r_size) :: ai,aj
  INTEGER :: i,j

  i = CEILING(ri)
  ai = ri - REAL(i-1,r_size)
  j = CEILING(rj)
  aj = rj - REAL(j-1,r_size)
  if(j == 0) then
    j = 1!needless?
    aj = 0
  end if

  if(ceiling(rj) < 2) then
    varn(:) = var0(:)
    vars(:) = var1(:)
  elseif(ceiling(rj) > nlat) then
    varn(:) = var1(:)
    vars(:) = var0(:)
  end if

  IF(i <= nlon) THEN
    var5 = vars(i-1) * (1-ai) * (1-aj) &
       & + vars(i  ) *    ai  * (1-aj) &
       & + varn(i-1) * (1-ai) *    aj  &
       & + varn(i  ) *    ai  *    aj
  ELSE
    var5 = vars(i-1) * (1-ai) * (1-aj) &
       & + vars(1  ) *    ai  * (1-aj) &
       & + varn(i-1) * (1-ai) *    aj  &
       & + varn(1  ) *    ai  *    aj
  END IF

  RETURN
end subroutine itpl_2d_plrgn

subroutine itpl_3d_plrgn(var0,var1,ri,rj,rk,var5)
  IMPLICIT NONE
  REAL(r_size),INTENT(IN) :: var0(nlon,nlev),var1(nlon,nlev)
  REAL(r_size),INTENT(IN) :: ri
  REAL(r_size),INTENT(IN) :: rj
  REAL(r_size),INTENT(IN) :: rk
  REAL(r_size),INTENT(OUT) :: var5
  real(r_size) :: varn(nlon,nlev),vars(nlon,nlev)
  REAL(r_size) :: ai,aj,ak
  INTEGER :: i,j,k

  i = CEILING(ri)
  ai = ri - REAL(i-1,r_size)
  j = CEILING(rj)
  aj = rj - REAL(j-1,r_size)
  if(j == 0) then
    j = 1!verbose?
    aj = 0
  end if
  k = CEILING(rk)
  ak = rk - REAL(k-1,r_size)

  if(ceiling(rj) < 2) then
    varn(:,:) = var0(:,:)
    vars(:,:) = var1(:,:)
  elseif(ceiling(rj) > nlat) then
    varn(:,:) = var1(:,:)
    vars(:,:) = var0(:,:)
  end if

  IF(i <= nlon) THEN
    var5 = vars(i-1,k-1) * (1-ai) * (1-aj) * (1-ak) &
       & + vars(i  ,k-1) *    ai  * (1-aj) * (1-ak) &
       & + varn(i-1,k-1) * (1-ai) *    aj  * (1-ak) &
       & + varn(i  ,k-1) *    ai  *    aj  * (1-ak) &
       & + vars(i-1,k  ) * (1-ai) * (1-aj) *    ak  &
       & + vars(i  ,k  ) *    ai  * (1-aj) *    ak  &
       & + varn(i-1,k  ) * (1-ai) *    aj  *    ak  &
       & + varn(i  ,k  ) *    ai  *    aj  *    ak
  ELSE
    var5 = vars(i-1,k-1) * (1-ai) * (1-aj) * (1-ak) &
       & + vars(1  ,k-1) *    ai  * (1-aj) * (1-ak) &
       & + varn(i-1,k-1) * (1-ai) *    aj  * (1-ak) &
       & + varn(1  ,k-1) *    ai  *    aj  * (1-ak) &
       & + vars(i-1,k  ) * (1-ai) * (1-aj) *    ak  &
       & + vars(1  ,k  ) *    ai  * (1-aj) *    ak  &
       & + varn(i-1,k  ) * (1-ai) *    aj  *    ak  &
       & + varn(1  ,k  ) *    ai  *    aj  *    ak
  END IF

  RETURN
end subroutine itpl_3d_plrgn
!-----------------------------------------------------------------------
! Interpolation
!-----------------------------------------------------------------------
SUBROUTINE itpl_2d(var,ri,rj,var5)
  IMPLICIT NONE
  REAL(r_size),INTENT(IN) :: var(nlon,nlat)
  REAL(r_size),INTENT(IN) :: ri
  REAL(r_size),INTENT(IN) :: rj
  REAL(r_size),INTENT(OUT) :: var5
  REAL(r_size) :: ai,aj
  INTEGER :: i,j

  i = CEILING(ri)
  ai = ri - REAL(i-1,r_size)
  j = CEILING(rj)
  aj = rj - REAL(j-1,r_size)

  IF(i <= nlon) THEN
    var5 = var(i-1,j-1) * (1-ai) * (1-aj) &
       & + var(i  ,j-1) *    ai  * (1-aj) &
       & + var(i-1,j  ) * (1-ai) *    aj  &
       & + var(i  ,j  ) *    ai  *    aj
  ELSE
    var5 = var(i-1,j-1) * (1-ai) * (1-aj) &
       & + var(1  ,j-1) *    ai  * (1-aj) &
       & + var(i-1,j  ) * (1-ai) *    aj  &
       & + var(1  ,j  ) *    ai  *    aj
  END IF

  RETURN
END SUBROUTINE itpl_2d

SUBROUTINE itpl_3d(var,ri,rj,rk,var5)
  IMPLICIT NONE
  REAL(r_size),INTENT(IN) :: var(nlon,nlat,nlev)
  REAL(r_size),INTENT(IN) :: ri
  REAL(r_size),INTENT(IN) :: rj
  REAL(r_size),INTENT(IN) :: rk
  REAL(r_size),INTENT(OUT) :: var5
  REAL(r_size) :: ai,aj,ak
  INTEGER :: i,j,k

  i = CEILING(ri)
  ai = ri - REAL(i-1,r_size)
  j = CEILING(rj)
  aj = rj - REAL(j-1,r_size)
  k = CEILING(rk)
  ak = rk - REAL(k-1,r_size)

  IF(i <= nlon) THEN
    var5 = var(i-1,j-1,k-1) * (1-ai) * (1-aj) * (1-ak) &
       & + var(i  ,j-1,k-1) *    ai  * (1-aj) * (1-ak) &
       & + var(i-1,j  ,k-1) * (1-ai) *    aj  * (1-ak) &
       & + var(i  ,j  ,k-1) *    ai  *    aj  * (1-ak) &
       & + var(i-1,j-1,k  ) * (1-ai) * (1-aj) *    ak  &
       & + var(i  ,j-1,k  ) *    ai  * (1-aj) *    ak  &
       & + var(i-1,j  ,k  ) * (1-ai) *    aj  *    ak  &
       & + var(i  ,j  ,k  ) *    ai  *    aj  *    ak
  ELSE
    var5 = var(i-1,j-1,k-1) * (1-ai) * (1-aj) * (1-ak) &
       & + var(1  ,j-1,k-1) *    ai  * (1-aj) * (1-ak) &
       & + var(i-1,j  ,k-1) * (1-ai) *    aj  * (1-ak) &
       & + var(1  ,j  ,k-1) *    ai  *    aj  * (1-ak) &
       & + var(i-1,j-1,k  ) * (1-ai) * (1-aj) *    ak  &
       & + var(1  ,j-1,k  ) *    ai  * (1-aj) *    ak  &
       & + var(i-1,j  ,k  ) * (1-ai) *    aj  *    ak  &
       & + var(1  ,j  ,k  ) *    ai  *    aj  *    ak
  END IF

  RETURN
END SUBROUTINE itpl_3d
!-----------------------------------------------------------------------
! Monitor departure
!-----------------------------------------------------------------------
SUBROUTINE monit_dep(nn,elm,dep,qc)
  IMPLICIT NONE
  INTEGER,INTENT(IN) :: nn
  REAL(r_size),INTENT(IN) :: elm(nn)
  REAL(r_size),INTENT(IN) :: dep(nn)
  INTEGER,INTENT(IN) :: qc(nn)
  REAL(r_size) :: rmse_u,rmse_v,rmse_t,rmse_q,rmse_ps,rmse_rh,rmse_mls,rmse_saber,rmse_nrv,rmse_erv,rmse_wrv,rmse_srv
  REAL(r_size) :: bias_u,bias_v,bias_t,bias_q,bias_ps,bias_rh,bias_mls,bias_saber,bias_nrv,bias_erv,bias_wrv,bias_srv
  INTEGER :: n,iu,iv,it,iq,ips,irh,imls,isaber,inrv,ierv,iwrv,isrv

  rmse_u = 0.0d0
  rmse_v = 0.0d0
  rmse_t = 0.0d0
  rmse_q = 0.0d0
  rmse_ps = 0.0d0
  rmse_rh = 0.0d0
  rmse_mls = 0.0d0
  rmse_saber = 0.0d0
  rmse_nrv = 0.0d0
  rmse_erv = 0.0d0
  rmse_wrv = 0.0d0
  rmse_srv = 0.0d0
  bias_u = 0.0d0
  bias_v = 0.0d0
  bias_t = 0.0d0
  bias_q = 0.0d0
  bias_ps = 0.0d0
  bias_rh = 0.0d0
  bias_mls = 0.0d0
  bias_saber = 0.0d0
  bias_nrv = 0.0d0
  bias_erv = 0.0d0
  bias_wrv = 0.0d0
  bias_srv = 0.0d0
  iu = 0
  iv = 0
  it = 0
  iq = 0
  ips = 0
  irh = 0
  imls = 0
  isaber = 0
  inrv = 0
  ierv = 0
  iwrv = 0
  isrv = 0
  DO n=1,nn
    IF(qc(n) /= 1) CYCLE
    SELECT CASE(NINT(elm(n)))
    CASE(id_u_obs)
      rmse_u = rmse_u + dep(n)**2
      bias_u = bias_u + dep(n)
      iu = iu + 1
    CASE(id_v_obs)
      rmse_v = rmse_v + dep(n)**2
      bias_v = bias_v + dep(n)
      iv = iv + 1
    CASE(id_t_obs,id_tv_obs)
      rmse_t = rmse_t + dep(n)**2
      bias_t = bias_t + dep(n)
      it = it + 1
    CASE(id_mls_obs)
      rmse_mls = rmse_mls + dep(n)**2
      bias_mls = bias_mls + dep(n)
      imls = imls + 1
    CASE(id_saber_obs)
      rmse_saber = rmse_saber + dep(n)**2
      bias_saber = bias_saber + dep(n)
      isaber = isaber + 1
    CASE(id_nrv_obs)
      rmse_nrv = rmse_nrv + dep(n)**2
      bias_nrv = bias_nrv + dep(n)
      inrv = inrv + 1
    CASE(id_erv_obs)
      rmse_erv = rmse_erv + dep(n)**2
      bias_erv = bias_erv + dep(n)
      ierv = ierv + 1
    CASE(id_wrv_obs)
      rmse_wrv = rmse_wrv + dep(n)**2
      bias_wrv = bias_wrv + dep(n)
      iwrv = iwrv + 1
    CASE(id_srv_obs)
      rmse_srv = rmse_srv + dep(n)**2
      bias_srv = bias_srv + dep(n)
      isrv = isrv + 1
    CASE(id_q_obs)
      rmse_q = rmse_q + dep(n)**2
      bias_q = bias_q + dep(n)
      iq = iq + 1
    CASE(id_ps_obs)
      rmse_ps = rmse_ps + dep(n)**2
      bias_ps = bias_ps + dep(n)
      ips = ips + 1
    CASE(id_rh_obs)
      rmse_rh = rmse_rh + dep(n)**2
      bias_rh = bias_rh + dep(n)
      irh = irh + 1
    END SELECT
  END DO
  IF(iu == 0) THEN
    rmse_u = undef
    bias_u = undef
  ELSE
    rmse_u = SQRT(rmse_u / REAL(iu,r_size))
    bias_u = bias_u / REAL(iu,r_size)
  END IF
  IF(iv == 0) THEN
    rmse_v = undef
    bias_v = undef
  ELSE
    rmse_v = SQRT(rmse_v / REAL(iv,r_size))
    bias_v = bias_v / REAL(iv,r_size)
  END IF
  IF(it == 0) THEN
    rmse_t = undef
    bias_t = undef
  ELSE
    rmse_t = SQRT(rmse_t / REAL(it,r_size))
    bias_t = bias_t / REAL(it,r_size)
  END IF
  IF(imls == 0) THEN
    rmse_mls = undef
    bias_mls = undef
  ELSE
    rmse_mls = SQRT(rmse_mls / REAL(imls,r_size))
    bias_mls = bias_mls / REAL(imls,r_size)
  END IF
  IF(isaber == 0) THEN
    rmse_saber = undef
    bias_saber = undef
  ELSE
    rmse_saber = SQRT(rmse_saber / REAL(isaber,r_size))
    bias_saber = bias_saber / REAL(isaber,r_size)
  END IF
  IF(inrv == 0) THEN
    rmse_nrv = undef
    bias_nrv = undef
  ELSE
    rmse_nrv = SQRT(rmse_nrv / REAL(inrv,r_size))
    bias_nrv = bias_nrv / REAL(inrv,r_size)
  END IF
  IF(ierv == 0) THEN
    rmse_erv = undef
    bias_erv = undef
  ELSE
    rmse_erv = SQRT(rmse_erv / REAL(ierv,r_size))
    bias_erv = bias_erv / REAL(ierv,r_size)
  END IF
  IF(iwrv == 0) THEN
    rmse_wrv = undef
    bias_wrv = undef
  ELSE
    rmse_wrv = SQRT(rmse_wrv / REAL(iwrv,r_size))
    bias_wrv = bias_wrv / REAL(iwrv,r_size)
  END IF
  IF(isrv == 0) THEN
    rmse_srv = undef
    bias_srv = undef
  ELSE
    rmse_srv = SQRT(rmse_srv / REAL(isrv,r_size))
    bias_srv = bias_srv / REAL(isrv,r_size)
  END IF
  IF(iq == 0) THEN
    rmse_q = undef
    bias_q = undef
  ELSE
    rmse_q = SQRT(rmse_q / REAL(iq,r_size))
    bias_q = bias_q / REAL(iq,r_size)
  END IF
  IF(ips == 0) THEN
    rmse_ps = undef
    bias_ps = undef
  ELSE
    rmse_ps = SQRT(rmse_ps / REAL(ips,r_size))
    bias_ps = bias_ps / REAL(ips,r_size)
  END IF
  IF(irh == 0) THEN
    rmse_rh = undef
    bias_rh = undef
  ELSE
    rmse_rh = SQRT(rmse_rh / REAL(irh,r_size))
    bias_rh = bias_rh / REAL(irh,r_size)
  END IF

  WRITE(6,'(A)') '== OBSERVATIONAL DEPARTURE ============================================='
  WRITE(6,'(12A12)') 'U','V','T','Q','PS','RH','MLS','SABER','N_radial','E_radial','W_radial','S_radial'
  WRITE(6,'(12ES12.3)') bias_u,bias_v,bias_t,bias_q,bias_ps,bias_rh,bias_mls,bias_saber,bias_nrv,bias_erv,bias_wrv,bias_srv
  WRITE(6,'(12ES12.3)') rmse_u,rmse_v,rmse_t,rmse_q,rmse_ps,rmse_rh,rmse_mls,rmse_saber,rmse_nrv,rmse_erv,rmse_wrv,rmse_srv
  WRITE(6,'(A)') '== NUMBER OF OBSERVATIONS TO BE ASSIMILATED ============================'
  WRITE(6,'(12A12)') 'U','V','T','Q','PS','RH','MLS','SABER','N_radial','E_radial','W_radial','S_radial'
  WRITE(6,'(12I12)') iu,iv,it,iq,ips,irh,imls,isaber,inrv,ierv,iwrv,isrv
  WRITE(6,'(A)')  '========================================================================'

  RETURN
END SUBROUTINE monit_dep
!-----------------------------------------------------------------------
! Monitor departure from gues/anal mean
!-----------------------------------------------------------------------

SUBROUTINE monit_mean(file)
  use mod_read_var
  implicit none
  CHARACTER(4),INTENT(IN) :: file
  REAL(r_size) :: v3d(nlon,nlat,nlev,nv3d)
  REAL(r_size) :: ps(nlon,nlat)
  REAL(r_size) :: v2d(nlon,nlat,nv2d)
  REAL(r_size) :: p_full(nlon,nlat,nlev),z_full(nlon,nlat,nlev),w_full(nlon,nlat,nlev)
!  REAL(r_size) :: elem
  REAL(r_size) :: bias_u,bias_v,bias_t,bias_ps,bias_q,bias_rh,bias_mls,bias_saber,bias_nrv,bias_erv,bias_wrv,bias_srv
  REAL(r_size) :: rmse_u,rmse_v,rmse_t,rmse_ps,rmse_q,rmse_rh,rmse_mls,rmse_saber,rmse_nrv,rmse_erv,rmse_wrv,rmse_srv
  REAL(r_size) :: hdxf,dep,ri,rj,rk
!  REAL(r_size) :: chixx

  INTEGER :: n,i,j,k,level,id,obsnum2
  INTEGER :: iu,iv,it,iq,ips,irh,imls,isaber,inrv,ierv,iwrv,isrv

  CHARACTER(11) :: filename='filexxx.grd'
  CHARACTER(17) :: filename2='file_xxxx_xxx.txt'

  REAL(r_size),ALLOCATABLE :: mtrs(:,:),mtrsinv(:,:),mtrr(:),mtrdep(:),mtrh(:,:)
  
  integer,parameter :: id_obs_size=12,id_obs(id_obs_size)=(/id_u_obs,id_v_obs,id_t_obs,id_q_obs,id_rh_obs,id_mls_obs,id_saber_obs,id_tv_obs,id_nrv_obs,id_erv_obs,id_wrv_obs,id_srv_obs/)
  integer :: obsnum(id_obs_size,nlev)
  !if the number of obstype > 99, correnctions (mainly for format) will be required. 
  real(r_size),parameter :: R_g=287/9.81d0

  WRITE(filename(1:7),'(A4,A3)') file,'_me'
  CALL read_grd(filename,v3d,v2d)
  CALL calc_pfull(nlon,nlat,v2d(:,:,iv2d_ps),p_full)
  call read_var('gphz_me.grd',z_full)
  call read_var('omgf_me.grd',w_full)
  w_full=-R_g*v3d(:,:,:,iv3d_t)*w_full*100/p_full
  obsnum=0


  rmse_u  = 0.0d0
  rmse_v  = 0.0d0
  rmse_t  = 0.0d0
  rmse_q  = 0.0d0
  rmse_ps = 0.0d0
  rmse_rh = 0.0d0
  rmse_mls = 0.0d0
  rmse_saber = 0.0d0
  rmse_nrv = 0.0d0
  rmse_erv = 0.0d0
  rmse_wrv = 0.0d0
  rmse_srv = 0.0d0
  bias_u = 0.0d0
  bias_v = 0.0d0
  bias_t = 0.0d0
  bias_q = 0.0d0
  bias_ps = 0.0d0
  bias_rh = 0.0d0
  bias_mls = 0.0d0
  bias_saber = 0.0d0
  bias_nrv = 0.0d0
  bias_erv = 0.0d0
  bias_wrv = 0.0d0
  bias_srv = 0.0d0
  iu  = 0
  iv  = 0
  it  = 0
  iq  = 0
  ips = 0
  irh = 0
  imls = 0
  isaber = 0
  inrv = 0
  ierv = 0
  iwrv = 0
  isrv = 0
  DO n=1,nobs
    CALL phys2ijk(p_full,z_full,obselm(n),obslon(n),obslat(n),obslev(n),ri,rj,rk)

    if(obstim(n)==nbslot.and.nint(rk)>2) then
      do i=1,id_obs_size
        if(id_obs(i)==NINT(obselm(n))) then
          obsnum(i,nint(rk))=obsnum(i,nint(rk))+1
          exit
        end if
      end do
    end if

    IF(CEILING(rk) > nlev) CYCLE
    IF(CEILING(rk) < 2 .AND. NINT(obselm(n)) /= id_ps_obs) THEN
      IF(NINT(obselm(n)) == id_u_obs .OR. NINT(obselm(n)) == id_v_obs) THEN
        rk = 1.00001d0
      ELSE
        CYCLE
      END IF
    END IF
    if(ceiling(rj) < 2) then                                                                    
      call Trans_XtoY_pl(obselm(n),ri,rj,rk,v3d(:,1,:,:),v2d(:,1,:),p_full(:,1,:),w_full(:,1,:),hdxf)         
    elseif(ceiling(rj) > nlat) then                                                             
      call Trans_XtoY_pl(obselm(n),ri,rj,rk,v3d(:,nlat,:,:),v2d(:,nlat,:),p_full(:,nlat,:),w_full(:,nlat,:),hdxf)
    else
      CALL Trans_XtoY(obselm(n),ri,rj,rk,v3d,v2d,p_full,w_full,hdxf)
    end if     
    dep = obsdat(n) - hdxf
    SELECT CASE(NINT(obselm(n)))
    CASE(id_u_obs)
      rmse_u = rmse_u + dep**2
      bias_u = bias_u + dep
      iu = iu + 1
    CASE(id_v_obs)
      rmse_v = rmse_v + dep**2
      bias_v = bias_v + dep
      iv = iv + 1
    CASE(id_t_obs,id_tv_obs)
      rmse_t = rmse_t + dep**2
      bias_t = bias_t + dep
      it = it + 1
    CASE(id_mls_obs)
      rmse_mls = rmse_mls + dep**2
      bias_mls = bias_mls + dep
      imls = imls + 1
    CASE(id_nrv_obs)
      rmse_nrv = rmse_nrv + dep**2
      bias_nrv = bias_nrv + dep
      inrv = inrv + 1
    CASE(id_erv_obs)
      rmse_erv = rmse_erv + dep**2
      bias_erv = bias_erv + dep
      ierv = ierv + 1
    CASE(id_wrv_obs)
      rmse_wrv = rmse_wrv + dep**2
      bias_wrv = bias_wrv + dep
      iwrv = iwrv + 1
    CASE(id_srv_obs)
      rmse_srv = rmse_srv + dep**2
      bias_srv = bias_srv + dep
      isrv = isrv + 1
    CASE(id_saber_obs)
      rmse_saber = rmse_saber + dep**2
      bias_saber = bias_saber + dep
      isaber = isaber + 1
    CASE(id_q_obs)
      rmse_q = rmse_q + dep**2
      bias_q = bias_q + dep
      iq = iq + 1
    CASE(id_ps_obs)
      rmse_ps = rmse_ps + dep**2
      bias_ps = bias_ps + dep
      ips = ips + 1
    CASE(id_rh_obs)
      rmse_rh = rmse_rh + dep**2
      bias_rh = bias_rh + dep
      irh = irh + 1
    END SELECT
  END DO

  IF(iu == 0) THEN
    rmse_u = undef
    bias_u = undef
  ELSE
    rmse_u = SQRT(rmse_u / REAL(iu,r_size))
    bias_u = bias_u / REAL(iu,r_size)
  END IF
  IF(iv == 0) THEN
    rmse_v = undef
    bias_v = undef
  ELSE
    rmse_v = SQRT(rmse_v / REAL(iv,r_size))
    bias_v = bias_v / REAL(iv,r_size)
  END IF
  IF(it == 0) THEN
    rmse_t = undef
    bias_t = undef
  ELSE
    rmse_t = SQRT(rmse_t / REAL(it,r_size))
    bias_t = bias_t / REAL(it,r_size)
  END IF
  IF(imls == 0) THEN
    rmse_mls = undef
    bias_mls = undef
  ELSE
    rmse_mls = SQRT(rmse_mls / REAL(imls,r_size))
    bias_mls = bias_mls / REAL(imls,r_size)
  END IF
  IF(inrv == 0) THEN
    rmse_nrv = undef
    bias_nrv = undef
  ELSE
    rmse_nrv = SQRT(rmse_nrv / REAL(inrv,r_size))
    bias_nrv = bias_nrv / REAL(inrv,r_size)
  END IF
  IF(ierv == 0) THEN
    rmse_erv = undef
    bias_erv = undef
  ELSE
    rmse_erv = SQRT(rmse_erv / REAL(ierv,r_size))
    bias_erv = bias_erv / REAL(ierv,r_size)
  END IF
  IF(iwrv == 0) THEN
    rmse_wrv = undef
    bias_wrv = undef
  ELSE
    rmse_wrv = SQRT(rmse_wrv / REAL(iwrv,r_size))
    bias_wrv = bias_wrv / REAL(iwrv,r_size)
  END IF
  IF(isrv == 0) THEN
    rmse_srv = undef
    bias_srv = undef
  ELSE
    rmse_srv = SQRT(rmse_srv / REAL(isrv,r_size))
    bias_srv = bias_srv / REAL(isrv,r_size)
  END IF
  IF(isaber == 0) THEN
    rmse_saber = undef
    bias_saber = undef
  ELSE
    rmse_saber = SQRT(rmse_saber / REAL(isaber,r_size))
    bias_saber = bias_saber / REAL(isaber,r_size)
  END IF
  IF(iq == 0) THEN
    rmse_q = undef
    bias_q = undef
  ELSE
    rmse_q = SQRT(rmse_q / REAL(iq,r_size))
    bias_q = bias_q / REAL(iq,r_size)
  END IF
  IF(ips == 0) THEN
    rmse_ps = undef
    bias_ps = undef
  ELSE
    rmse_ps = SQRT(rmse_ps / REAL(ips,r_size))
    bias_ps = bias_ps / REAL(ips,r_size)
  END IF
  IF(irh == 0) THEN
    rmse_rh = undef
    bias_rh = undef
  ELSE
    rmse_rh = SQRT(rmse_rh / REAL(irh,r_size))
    bias_rh = bias_rh / REAL(irh,r_size)
  END IF

  WRITE(6,'(3A)') '== PARTIAL OBSERVATIONAL DEPARTURE (',file,')  =============================='
  WRITE(6,'(12A12)') 'U','V','T','Q','PS','RH','MLS','SABER','N_radial','E_radial','W_radial','S_radial'
  WRITE(6,'(12ES12.3)') bias_u,bias_v,bias_t,bias_q,bias_ps,bias_rh,bias_mls,bias_saber,bias_nrv,bias_erv,bias_wrv,bias_srv
  WRITE(6,'(12ES12.3)') rmse_u,rmse_v,rmse_t,rmse_q,rmse_ps,rmse_rh,rmse_mls,rmse_saber,rmse_nrv,rmse_erv,rmse_wrv,rmse_srv
  WRITE(6,'(A)') '== NUMBER OF OBSERVATIONS   =============================================='
  WRITE(6,'(12A12)') 'U','V','T','Q','PS','RH','MLS','SABER','N_radial','E_radial','W_radial','S_radial'
  WRITE(6,'(12I12)') iu,iv,it,iq,ips,irh,imls,isaber,inrv,ierv,iwrv,isrv
  WRITE(6,'(A)')   '========================================================================'

  obsnum2=maxval(obsnum)
  allocate(mtrs   (obsnum2,obsnum2))
  allocate(mtrsinv(obsnum2,obsnum2))
  allocate(mtrh   (    nbv,obsnum2))
  allocate(mtrr           (obsnum2))
  allocate(mtrdep         (obsnum2))
  
  do level=26,116!Even though a normalization (1/obsnum2) is included, the value may affected by the (density rather than) number of observations.
!Below the level 26, there are enormous amount of wind observations (>1.5k, >2M in matrix), which is larger than any other observations at all levels.
!They seem to have less consistency to other levels and other kinds of obseravions (and less information that matches the computational cost).
!If there are enough time, interest, and computational resource (with good matrix solver), ther lower boundary can be modified to 23.
!To consider the hybrid coordinate, start from 23 (p[21] may change with Ps, so z[22]-z[21] is not constant)
!To consider the inconstant delta-z, start from 23 (z[23]-z[22]<z[24]]-z[23]=z[25]-z[26]=...) 
    WRITE(filename2(1:13),'(A4,A6,i3.3)') file,'_chi2_',level !  chi-square                                                                                                          
    open(unit=59, file=filename2,status='replace')

    do id=1,id_obs_size
      WRITE(filename2(6:9),'(A2,i2.2)') 'om',id ! each omf                                                                                                           
      open(unit=58, file=filename2,status='replace')

      if(obsnum(id,level)==0) then
        write(59,fmt='(f9.3)',advance='no') -100.
      else
        obsnum2=0

        ! store y-H(x),R,HE^T, and write OmF, OmA
        DO n=1,nobs
          if(obstim(n) /= nbslot) cycle
          if(id_obs(id)==NINT(obselm(n))) then
            CALL phys2ijk(p_full,z_full,obselm(n),obslon(n),obslat(n),obslev(n),ri,rj,rk)
            if(nint(rk) /= level) cycle

            if(ceiling(rj) < 2) then
              call Trans_XtoY_pl(obselm(n),ri,rj,rk,v3d(:,1,:,:),v2d(:,1,:),p_full(:,1,:),w_full(:,1,:),hdxf)
            elseif(ceiling(rj) > nlat) then
              call Trans_XtoY_pl(obselm(n),ri,rj,rk,v3d(:,nlat,:,:),v2d(:,nlat,:),p_full(:,nlat,:),w_full(:,nlat,:),hdxf)
            else
              CALL Trans_XtoY(obselm(n),ri,rj,rk,v3d,v2d,p_full,w_full,hdxf)
            end if
            obsnum2=obsnum2+1
            mtrdep(obsnum2)=obsdat(n)-hdxf!why obshdhf is not used?
            mtrr(obsnum2)=obserr(n)*obserr(n)
            mtrh(1:nbv,obsnum2)=obshdxf(n,1:nbv)

            write(58,'(3F9.3)') real(mtrdep(obsnum2),4),-obslat(n),real(obserr(n)) ! write OmF/A
          end if
        END DO

        mtrs(1:obsnum2,1:obsnum2)=matmul(transpose(mtrh(:,1:obsnum2)),mtrh(:,1:obsnum2))/(nbv-1) ! HPH

        do i=1,obsnum2
          mtrs(i,i)=mtrs(i,i)+mtrr(i) ! HPH + R       
        end do

        if('gues'==file) then
          CALL mtx_inv(obsnum2,mtrs(1:obsnum2,1:obsnum2),mtrsinv(1:obsnum2,1:obsnum2)) !(HPH + R)^-1
        else
          do j=1,obsnum2
            mtrsinv(1:obsnum2,j)=mtrs(1:obsnum2,j)/(mtrr(1:obsnum2)*mtrr(j)) ! R^-1 * (R+HPH) * R^-1
          end do
        end if

        write(59,fmt='(F9.3)',advance='no') sum( mtrdep(1:obsnum2)*matmul(mtrsinv(1:obsnum2,1:obsnum2),mtrdep(1:obsnum2)) )/obsnum2 ! d^T * mtrsinv * d /n
      end if
      close(58)
    end do

    write(59,*)
    close(59)
  end do
  RETURN

END SUBROUTINE monit_mean

!-----------------------------------------------------------------------
! Basic modules for observation input
!-----------------------------------------------------------------------
SUBROUTINE get_nobs_mpi(cfile,nn)
  IMPLICIT NONE
  CHARACTER(*),INTENT(IN) :: cfile
  INTEGER,INTENT(OUT) :: nn
  REAL(r_sngl) :: wk(6)
  INTEGER :: ios
  INTEGER :: iu,iv,it,iq,irh,ips,imls,isaber,inrv,ierv,iwrv,isrv
  INTEGER :: iunit
  LOGICAL :: ex

  nn = 0
  iu = 0
  iv = 0
  it = 0
  iq = 0
  irh = 0
  ips = 0
  imls = 0
  isaber = 0
  inrv = 0
  ierv = 0
  iwrv = 0
  isrv = 0
  iunit=91
  INQUIRE(FILE=cfile,EXIST=ex)
  IF(ex) THEN
    WRITE(6,'(A,I3.3,2A)') 'MYRANK ',myrank,' is accessing a file ',cfile
    OPEN(iunit,FILE=cfile,FORM='unformatted',ACCESS='sequential')
    DO
      READ(iunit,IOSTAT=ios) wk
      IF(ios /= 0) EXIT
      SELECT CASE(NINT(wk(1)))
      CASE(id_u_obs)
        iu = iu + 1
      CASE(id_v_obs)
        iv = iv + 1
      CASE(id_t_obs,id_tv_obs)
        it = it + 1
      CASE(id_mls_obs)
        imls = imls + 1
      CASE(id_saber_obs)
        isaber = isaber + 1
      case(id_nrv_obs)
        inrv = inrv + 1
      case(id_erv_obs)
        ierv = ierv + 1
      case(id_wrv_obs)
        iwrv = iwrv + 1
      case(id_srv_obs)
        isrv = isrv + 1
      CASE(id_q_obs)
        iq = iq + 1
      CASE(id_rh_obs)
        irh = irh + 1
      CASE(id_ps_obs)
        ips = ips + 1
      END SELECT
      nn = nn + 1
    END DO
    WRITE(6,'(I10,A)') nn,' OBSERVATIONS INPUT'
    WRITE(6,'(A12,I10)') '          U:',iu
    WRITE(6,'(A12,I10)') '          V:',iv
    WRITE(6,'(A12,I10)') '          T:',it
    WRITE(6,'(A12,I10)') '          Q:',iq
    WRITE(6,'(A12,I10)') '         RH:',irh
    WRITE(6,'(A12,I10)') '         Ps:',ips
    WRITE(6,'(A12,I10)') '        MLS:',imls
    WRITE(6,'(A12,I10)') '      SABER:',isaber
    WRITE(6,'(A12,I10)') '   N radial:',inrv
    WRITE(6,'(A12,I10)') '   E radial:',ierv
    WRITE(6,'(A12,I10)') '   W radial:',iwrv
    WRITE(6,'(A12,I10)') '   S radial:',isrv
    CLOSE(iunit)
  ELSE
    WRITE(6,'(2A)') cfile,' does not exist -- skipped'
  END IF

  RETURN
END SUBROUTINE get_nobs_mpi

SUBROUTINE read_obs_mpi(cfile,nn,elem,rlon,rlat,rlev,odat,oerr)
  IMPLICIT NONE
  CHARACTER(*),INTENT(IN) :: cfile
  INTEGER,INTENT(IN) :: nn
  REAL(r_size),INTENT(OUT) :: elem(nn) ! element number
  REAL(r_size),INTENT(OUT) :: rlon(nn)
  REAL(r_size),INTENT(OUT) :: rlat(nn)
  REAL(r_size),INTENT(OUT) :: rlev(nn)
  REAL(r_size),INTENT(OUT) :: odat(nn)
  REAL(r_size),INTENT(OUT) :: oerr(nn)
  REAL(r_sngl) :: wk(6)
  INTEGER :: n,iunit

  iunit=91
  WRITE(6,'(A,I3.3,2A)') 'MYRANK ',myrank,' is reading a file ',cfile
  OPEN(iunit,FILE=cfile,FORM='unformatted',ACCESS='sequential')
  DO n=1,nn
    READ(iunit) wk
    SELECT CASE(NINT(wk(1)))
    CASE(id_u_obs)
      wk(4) = wk(4) * 100.0 ! hPa -> Pa
    CASE(id_v_obs)
      wk(4) = wk(4) * 100.0 ! hPa -> Pa
    CASE(id_t_obs,id_tv_obs)
      wk(4) = wk(4) * 100.0 ! hPa -> Pa
    CASE(id_mls_obs)
      wk(4) = wk(4) * 100.0 ! hPa -> Pa
    CASE(id_saber_obs)
      wk(4) = wk(4) * 100.0 ! hPa -> Pa 
    case(id_nrv_obs)
      ! keep height(z)
    case(id_erv_obs)
      ! keep height(z)
    case(id_wrv_obs)
      ! keep height(z)                                                                                                                                                        
    case(id_srv_obs)
      ! keep height(z) 
    CASE(id_q_obs)
      wk(4) = wk(4) * 100.0 ! hPa -> Pa
    CASE(id_ps_obs)
      wk(5) = wk(5) * 100.0 ! hPa -> Pa
      wk(6) = wk(6) * 100.0 ! hPa -> Pa
    CASE(id_rh_obs)
      wk(4) = wk(4) * 100.0 ! hPa -> Pa
      wk(5) = wk(5) * 0.01 ! percent input
      wk(6) = wk(6) * 0.01 ! percent input
    END SELECT
    elem(n) = REAL(wk(1),r_size)
    rlon(n) = REAL(wk(2),r_size)
    rlat(n) = - REAL(wk(3),r_size)
    rlev(n) = REAL(wk(4),r_size)
    odat(n) = REAL(wk(5),r_size)
    oerr(n) = REAL(wk(6),r_size)
  END DO
  CLOSE(iunit)

  RETURN
END SUBROUTINE read_obs_mpi
subroutine averaging_kernel(v3d,ri,rj,rk,yobs)
  use ak_parameter
  implicit none
  real(r_size),intent(in) :: v3d(:,:,:)
  real(r_size),intent(in) :: ri,rj,rk
  real(r_size),intent(out) :: yobs
  integer :: i,mls_k
  real(r_size) :: zobs(mls_lev)
  do i=2,mls_lev !calculate yobs at each level of MLS
     call itpl_3d(v3d,ri,rj,mls_rk(i),zobs(i))
  end do
  mls_k=minloc(abs(mls_rk-rk),dim=1)
  if (rj<19.or.rj>46) then !40S-90S or 40N-90N
     yobs=sum(ak_lat70n(2:mls_lev,mls_k)*zobs(2:mls_lev))
  else !40S-40N
     yobs=sum(ak_lat0n(2:mls_lev,mls_k)*zobs(2:mls_lev))
  end if
end subroutine averaging_kernel
END MODULE common_obs_miroc
