!
!    Copyright  2013, 2014, 2020, 2021 Guy Munhoven
!
!    This file is part of SolveSAPHE-r2.

!    SolveSAPHE-r2 is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Lesser General Public License as published by
!    the Free Software Foundation, either version 3 of the License, or
!    (at your option) any later version.
!
!    SolveSAPHE is distributed in the hope that it will be useful,
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU Lesser General Public License for more details.
!
!    You should have received a copy of the GNU Lesser General Public License
!    along with SolveSAPHE-r2.  If not, see <http://www.gnu.org/licenses/>.
!





! **********************
! Precompiler directives
! **********************

! CREATEFILES:
! -- if defined, result files are written
! -- if not defined, result files are *not* written

#define CREATEFILES


! CASE_TSP =  1: Surface cold (for SW1, SW2, SW3)
!             2: Surface warm (for SW1, SW2, SW3)
!             3: Deep cold (for SW1, SW2, SW3)
!             4: Surface cold, brackish (for BW4)
!             5: Intermediate cold, brackish (for ABW5)

#define CASE_TSP 1


! CASE_SW  =  1:  SW1  -- common present-day seawater
!             2:  SW2  -- common present-day and future seawater
!             3:  SW3  -- extreme seawater samples
!             4:  BW4  -- brackish water and dilute seawater
!             5:  ABW5 -- brackish anoxic fjord water

#define CASE_SW 2


! CASE_DIC =  1: DIC
!             2: CO2
!             3: HCO3
!             4: CO3
!            12: CO2,  values from DIC (1) - used for comparative benchmarks in the paper
!            13: HCO3, values from DIC (1) - used for comparative benchmarks in the paper
!            14: CO3,  values from DIC (1) - used for comparative benchmarks in the paper

#define CASE_DIC 1


! CASE_ZHINI:
! -- if not defined, leave the initialisation to the solver
!    (solver default)
! -- if defined, use explicit initialisation (pH=8 for all)

#undef CASE_ZHINI


! **************************
! End precompiler directives
! **************************


!===============================================================================
PROGRAM DRIVER_AT_GENERAL2
!===============================================================================

USE MOD_PRECISION
USE MOD_CHEMCONST
USE MOD_PHSOLVERS_LOGGING


IMPLICIT NONE

REAL(KIND=wp)                       :: z_alktot, z_dicvar, z_bortot
REAL(KIND=wp)                       :: z_po4tot, z_siltot, z_nh4tot, z_h2stot
REAL(KIND=wp)                       :: z_so4tot, z_flutot
REAL(KIND=wp), DIMENSION(2)         :: z_hini, z_h, z_val
REAL(KIND=wp), DIMENSION(2)         :: z_hini2

REAL                                :: z_cputime_start, z_cputime_end

INTEGER                             :: ji, ji_alk, ji_dic
INTEGER                             :: n_roots

INTEGER, PARAMETER                  :: jp_resunit = 1



! Temperature, salinity, pressure data
! ====================================

#if CASE_TSP == 1
REAL(KIND=wp), PARAMETER            :: z_t_k    = 275.15_wp
REAL(KIND=wp), PARAMETER            :: z_s      =  35.00_wp
REAL(KIND=wp), PARAMETER            :: z_p_bar  =   0.00_wp
 CHARACTER(LEN=*), PARAMETER        :: cp_tspid = 'sc'
#endif

#if CASE_TSP == 2
REAL(KIND=wp), PARAMETER            :: z_t_k    = 298.15_wp
REAL(KIND=wp), PARAMETER            :: z_s      =  35.00_wp
REAL(KIND=wp), PARAMETER            :: z_p_bar  =   0.00_wp
 CHARACTER(LEN=*), PARAMETER        :: cp_tspid = 'sw'
#endif

#if CASE_TSP == 3
REAL(KIND=wp), PARAMETER            :: z_t_k    = 275.15_wp
REAL(KIND=wp), PARAMETER            :: z_s      =  35.00_wp
REAL(KIND=wp), PARAMETER            :: z_p_bar  = 300.00_wp
 CHARACTER(LEN=*), PARAMETER        :: cp_tspid = 'dc'
#endif

#if CASE_TSP == 4
REAL(KIND=wp), PARAMETER            :: z_t_k   = 275.15_wp
REAL(KIND=wp), PARAMETER            :: z_s     =   3.50_wp
REAL(KIND=wp), PARAMETER            :: z_p_bar =   0.00_wp
 CHARACTER(LEN=*), PARAMETER        :: cp_tspid = 'sb'
#endif

#if CASE_TSP == 5
REAL(KIND=wp), PARAMETER            :: z_t_k   = 280.71_wp  ! 7.56 degC
REAL(KIND=wp), PARAMETER            :: z_s     =  22.82_wp  ! 22.82
REAL(KIND=wp), PARAMETER            :: z_p_bar =  13.50_wp  ! 135 m depth
 CHARACTER(LEN=*), PARAMETER        :: cp_tspid = 'ib'
#endif


! DIC selection
! =============

#if (CASE_DIC == 1) || (CASE_DIC == 12) || (CASE_DIC == 13) || (CASE_DIC == 14)
# if CASE_DIC == 1
 CHARACTER(LEN=*), PARAMETER        :: cp_dicid   = 'ct-'
 INTEGER,          PARAMETER        :: jp_dicsel  = jp_dic
# else

 CHARACTER(LEN=*), PARAMETER        :: cp_dicidin = 'ct-'

#  if CASE_DIC == 12
 CHARACTER(LEN=*), PARAMETER        :: cp_dicid   = 'ctco2-'
 INTEGER,          PARAMETER        :: jp_dicsel  = jp_co2
#  endif

#  if CASE_DIC == 13
 CHARACTER(LEN=*), PARAMETER        :: cp_dicid   = 'cthco3-'
 INTEGER,          PARAMETER        :: jp_dicsel  = jp_hco3
#  endif

#  if CASE_DIC == 14
 CHARACTER(LEN=*), PARAMETER        :: cp_dicid   = 'ctco3-'
 INTEGER,          PARAMETER        :: jp_dicsel  = jp_co3
#  endif
# endif
#endif

#if CASE_DIC == 2
 CHARACTER(LEN=*), PARAMETER        :: cp_dicid   = 'co2-'
 INTEGER,          PARAMETER        :: jp_dicsel  = jp_co2
#endif

#if CASE_DIC == 3
 CHARACTER(LEN=*), PARAMETER        :: cp_dicid   = 'hco3-'
 INTEGER,          PARAMETER        :: jp_dicsel  = jp_hco3
#endif

#if CASE_DIC == 4
 CHARACTER(LEN=*), PARAMETER        :: cp_dicid   = 'co3-'
 INTEGER,          PARAMETER        :: jp_dicsel  = jp_co3
#endif



! DIC and Alkalinity distributions (SWx cases)
! ============================================

#if CASE_SW == 1
! Settings for common present-day seawater samples
# if (CASE_DIC == 1) || (CASE_DIC == 12) || (CASE_DIC == 13) || (CASE_DIC == 14)
! DIC interval extent (linear)
REAL(KIND=wp), PARAMETER            :: z_dicvar_min =   1.85E-3_wp
REAL(KIND=wp), PARAMETER            :: z_dicvar_max =   2.45E-3_wp
INTEGER, PARAMETER                  :: jp_ndic = 600
# endif

# if CASE_DIC == 2
! CO2 interval extent (log scale)
REAL(KIND=wp), PARAMETER            :: z_dicvar_min =  -6.00_wp
REAL(KIND=wp), PARAMETER            :: z_dicvar_max =  -3.00_wp
INTEGER, PARAMETER                  :: jp_ndic = 600
# endif

# if CASE_DIC == 3
! HCO3 interval extent (linear)
REAL(KIND=wp), PARAMETER            :: z_dicvar_min =   1.20E-3_wp
REAL(KIND=wp), PARAMETER            :: z_dicvar_max =   2.40E-3_wp
INTEGER, PARAMETER                  :: jp_ndic = 600
# endif

# if CASE_DIC == 4
! CO3 interval extent (log scale)
REAL(KIND=wp), PARAMETER            :: z_dicvar_min =  -5.00_wp
REAL(KIND=wp), PARAMETER            :: z_dicvar_max =  -3.00_wp
INTEGER, PARAMETER                  :: jp_ndic = 600
# endif

! ALK interval extent (linear)
REAL(KIND=wp), PARAMETER            :: z_alktot_min =  2.20E-3_wp
REAL(KIND=wp), PARAMETER            :: z_alktot_max =  2.50E-3_wp
INTEGER, PARAMETER                  :: jp_nalk = 300

CHARACTER(LEN=*), PARAMETER        :: cp_fileid = 'sw1-' // cp_dicid // cp_tspid
# if (CASE_DIC == 12) || (CASE_DIC == 13) || (CASE_DIC == 14)
CHARACTER(LEN=*), PARAMETER        :: cp_fileidin = 'sw1-' // cp_dicidin // cp_tspid
# endif
#endif



#if CASE_SW == 2
! Settings for common present-day and future seawater
! (derived from SP750 simulation experiment with MBM-Medusa)
# if (CASE_DIC == 1) || (CASE_DIC == 12) || (CASE_DIC == 13) || (CASE_DIC == 14)
! DIC interval extent (linear)
REAL(KIND=wp), PARAMETER            :: z_dicvar_min =  1.85E-3_wp
REAL(KIND=wp), PARAMETER            :: z_dicvar_max =  3.35E-3_wp
INTEGER, PARAMETER                  :: jp_ndic = 1500
# endif

# if CASE_DIC == 2
! CO2 interval extent (log scale)
REAL(KIND=wp), PARAMETER            :: z_dicvar_min =  -7.00E+0_wp
REAL(KIND=wp), PARAMETER            :: z_dicvar_max =  -3.00E+0_wp
INTEGER, PARAMETER                  :: jp_ndic = 1500
# endif

# if CASE_DIC == 3
! HCO3 interval extent (linear)
REAL(KIND=wp), PARAMETER            :: z_dicvar_min =  0.60E-3_wp
REAL(KIND=wp), PARAMETER            :: z_dicvar_max =  3.20E-3_wp
INTEGER, PARAMETER                  :: jp_ndic = 1500
# endif

# if CASE_DIC == 4
! CO3 interval extent (log scale)
REAL(KIND=wp), PARAMETER            :: z_dicvar_min =  -6._wp
REAL(KIND=wp), PARAMETER            :: z_dicvar_max =  -3._wp
INTEGER, PARAMETER                  :: jp_ndic = 1500
# endif

! ALK interval extent (linear)
REAL(KIND=wp), PARAMETER            :: z_alktot_min =  2.2E-3_wp
REAL(KIND=wp), PARAMETER            :: z_alktot_max =  3.5E-3_wp
INTEGER, PARAMETER                  :: jp_nalk = 1300

CHARACTER(LEN=*), PARAMETER         :: cp_fileid = 'sw2-' // cp_dicid // cp_tspid
# if (CASE_DIC == 12) || (CASE_DIC == 13) || (CASE_DIC == 14)
CHARACTER(LEN=*), PARAMETER        :: cp_fileidin = 'sw2-' // cp_dicidin // cp_tspid
# endif
#endif



#if CASE_SW == 3
! Settings for extreme seawater samples
# if (CASE_DIC == 1) || (CASE_DIC == 12) || (CASE_DIC == 13) || (CASE_DIC == 14)
! DIC interval extent (linear)
REAL(KIND=wp), PARAMETER            :: z_dicvar_min =  0.0E-3_wp
REAL(KIND=wp), PARAMETER            :: z_dicvar_max =  6.0E-3_wp
INTEGER, PARAMETER                  :: jp_ndic = 600
# endif

# if CASE_DIC == 2
! CO2 interval extent (log scale)
REAL(KIND=wp), PARAMETER            :: z_dicvar_min = -14._wp
REAL(KIND=wp), PARAMETER            :: z_dicvar_max =  -2._wp
INTEGER, PARAMETER                  :: jp_ndic = 600
# endif

# if CASE_DIC == 3
! HCO3 interval extent (linear)
REAL(KIND=wp), PARAMETER            :: z_dicvar_min =  0.0E-3_wp
REAL(KIND=wp), PARAMETER            :: z_dicvar_max =  5.0E-3_wp
INTEGER, PARAMETER                  :: jp_ndic = 600
# endif

# if CASE_DIC == 4
! CO3 interval extent (log scale)
REAL(KIND=wp), PARAMETER            :: z_dicvar_min = -14._wp !-9.4_wp !
REAL(KIND=wp), PARAMETER            :: z_dicvar_max =  -2._wp !-9.3_wp ! limit at -9.39_wp
INTEGER, PARAMETER                  :: jp_ndic = 600
# endif

! ALK interval extent (linear)
REAL(KIND=wp), PARAMETER            :: z_alktot_min = -1.0E-3_wp
REAL(KIND=wp), PARAMETER            :: z_alktot_max =  5.0E-3_wp
INTEGER, PARAMETER                  :: jp_nalk = 600

CHARACTER(LEN=*), PARAMETER        :: cp_fileid = 'sw3-' // cp_dicid // cp_tspid
# if (CASE_DIC == 12) || (CASE_DIC == 13) || (CASE_DIC == 14)
CHARACTER(LEN=*), PARAMETER        :: cp_fileidin = 'sw3-' // cp_dicidin // cp_tspid
# endif
#endif



#if CASE_SW == 4
! Settings for dilute seawater samples
# if (CASE_DIC == 1) || (CASE_DIC == 12) || (CASE_DIC == 13) || (CASE_DIC == 14)
! DIC interval extent (linear)
REAL(KIND=wp), PARAMETER            :: z_dicvar_min =  0.0E-3_wp
REAL(KIND=wp), PARAMETER            :: z_dicvar_max =  1.2E-3_wp
INTEGER, PARAMETER                  :: jp_ndic = 600
# endif

# if CASE_DIC == 2
! CO2 interval extent (log scale)
REAL(KIND=wp), PARAMETER            :: z_dicvar_min = -12.00E+0_wp
REAL(KIND=wp), PARAMETER            :: z_dicvar_max =  -3.00E+0_wp
INTEGER, PARAMETER                  :: jp_ndic = 600
# endif

# if CASE_DIC == 3
! HCO3 interval extent (linear)
REAL(KIND=wp), PARAMETER            :: z_dicvar_min =  0.0E-3_wp
REAL(KIND=wp), PARAMETER            :: z_dicvar_max =  1.0E-3_wp
INTEGER, PARAMETER                  :: jp_ndic = 600
# endif

# if CASE_DIC == 4
! CO3 interval extent (log scale)
REAL(KIND=wp), PARAMETER            :: z_dicvar_min =  -9.00E+0_wp
REAL(KIND=wp), PARAMETER            :: z_dicvar_max =  -3.00E+0_wp
INTEGER, PARAMETER                  :: jp_ndic = 600
# endif

! ALK interval extent (linear)
REAL(KIND=wp), PARAMETER            :: z_alktot_min =  0.0E-3_wp
REAL(KIND=wp), PARAMETER            :: z_alktot_max =  1.5E-3_wp
INTEGER, PARAMETER                  :: jp_nalk = 600

CHARACTER(LEN=*), PARAMETER        :: cp_fileid = 'bw4-' // cp_dicid // cp_tspid
# if (CASE_DIC == 12) || (CASE_DIC == 13) || (CASE_DIC == 14)
CHARACTER(LEN=*), PARAMETER        :: cp_fileidin = 'bw4-' // cp_dicidin // cp_tspid
# endif
#endif



#if CASE_SW == 5
! Settings for brackish anoxic fjord water samples
# if (CASE_DIC == 1) || (CASE_DIC == 12) || (CASE_DIC == 13) || (CASE_DIC == 14)
! DIC interval extent (linear)
REAL(KIND=wp), PARAMETER            :: z_dicvar_min =  15.0E-3_wp
REAL(KIND=wp), PARAMETER            :: z_dicvar_max =  17.5E-3_wp
INTEGER, PARAMETER                  :: jp_ndic = 500
# endif

# if CASE_DIC == 2
! CO2 interval extent (log scale)
REAL(KIND=wp), PARAMETER            :: z_dicvar_min =  -4.00E+0_wp
REAL(KIND=wp), PARAMETER            :: z_dicvar_max =  -2.00E+0_wp
INTEGER, PARAMETER                  :: jp_ndic = 500
# endif

# if CASE_DIC == 3
! HCO3 interval extent (linear)
REAL(KIND=wp), PARAMETER            :: z_dicvar_min =  13.0E-3_wp
REAL(KIND=wp), PARAMETER            :: z_dicvar_max =  17.0E-3_wp
INTEGER, PARAMETER                  :: jp_ndic = 500
# endif

# if CASE_DIC == 4
! CO3 interval extent (log scale)
REAL(KIND=wp), PARAMETER            :: z_dicvar_min =  -5.00E+0_wp
REAL(KIND=wp), PARAMETER            :: z_dicvar_max =  -3.00E+0_wp
INTEGER, PARAMETER                  :: jp_ndic = 500
# endif

! ALK interval extent (linear)
REAL(KIND=wp), PARAMETER            :: z_alktot_min =  17.0E-3_wp
REAL(KIND=wp), PARAMETER            :: z_alktot_max =  20.0E-3_wp
INTEGER, PARAMETER                  :: jp_nalk = 600

CHARACTER(LEN=*), PARAMETER        :: cp_fileid = 'abw5-' // cp_dicid // cp_tspid
# if (CASE_DIC == 12) || (CASE_DIC == 13) || (CASE_DIC == 14)
CHARACTER(LEN=*), PARAMETER        :: cp_fileidin = 'abw5-' // cp_dicidin // cp_tspid
# endif
#endif


! Work variables and arrays
! =========================

REAL(KIND=wp), DIMENSION(jp_ndic)            :: z_dicvar_arr
REAL(KIND=wp), DIMENSION(jp_nalk)            :: z_alktot_arr
REAL(KIND=wp)                                :: z_dicvar_del
REAL(KIND=wp)                                :: z_alktot_del


REAL(KIND=wp), DIMENSION(jp_ndic,jp_nalk, 2) :: z_h_atgen
REAL(KIND=wp), DIMENSION(jp_ndic,jp_nalk, 2) :: z_val_atgen
REAL(KIND=wp), DIMENSION(jp_ndic,jp_nalk, 2) :: z_hmin_atgen_ini
REAL(KIND=wp), DIMENSION(jp_ndic,jp_nalk, 2) :: z_hmax_atgen_ini
REAL(KIND=wp), DIMENSION(jp_ndic,jp_nalk, 2) :: z_h_atgen_ini
REAL(KIND=wp), DIMENSION(jp_ndic,jp_nalk, 2) :: zval_hmin_atgen_ini
REAL(KIND=wp), DIMENSION(jp_ndic,jp_nalk, 2) :: zval_hmax_atgen_ini

INTEGER, DIMENSION(jp_ndic,jp_nalk, 2)       :: jniter_atgen
INTEGER, DIMENSION(jp_ndic,jp_nalk, 2)       :: jniter_atgen_limin
INTEGER, DIMENSION(jp_ndic,jp_nalk, 2)       :: jniter_atgen_limax
INTEGER, DIMENSION(jp_ndic,jp_nalk, 2)       :: jniter_atgen_limabs
INTEGER, DIMENSION(jp_ndic,jp_nalk, 2)       :: jndivg_atgen
INTEGER, DIMENSION(jp_ndic,jp_nalk, 2)       :: jniter_div_atgen
INTEGER, DIMENSION(jp_ndic,jp_nalk)          :: jnroots_atgen


REAL(KIND=wp), DIMENSION(jp_ndic,jp_nalk, 2) :: z_h_atsec
REAL(KIND=wp), DIMENSION(jp_ndic,jp_nalk, 2) :: z_val_atsec
REAL(KIND=wp), DIMENSION(jp_ndic,jp_nalk, 2) :: z_hmin_atsec_ini
REAL(KIND=wp), DIMENSION(jp_ndic,jp_nalk, 2) :: z_hmax_atsec_ini
REAL(KIND=wp), DIMENSION(jp_ndic,jp_nalk, 2) :: z_h_atsec_ini
REAL(KIND=wp), DIMENSION(jp_ndic,jp_nalk, 2) :: zval_hmin_atsec_ini
REAL(KIND=wp), DIMENSION(jp_ndic,jp_nalk, 2) :: zval_hmax_atsec_ini

INTEGER, DIMENSION(jp_ndic,jp_nalk, 2)       :: jniter_atsec
INTEGER, DIMENSION(jp_ndic,jp_nalk, 2)       :: jniter_atsec_limin
INTEGER, DIMENSION(jp_ndic,jp_nalk, 2)       :: jniter_atsec_limax
INTEGER, DIMENSION(jp_ndic,jp_nalk, 2)       :: jniter_atsec_limabs
INTEGER, DIMENSION(jp_ndic,jp_nalk, 2)       :: jndivg_atsec
INTEGER, DIMENSION(jp_ndic,jp_nalk, 2)       :: jniter_div_atsec
INTEGER, DIMENSION(jp_ndic,jp_nalk)          :: jnroots_atsec


#if (CASE_DIC == 12) || (CASE_DIC == 13) || (CASE_DIC == 14)
INTEGER                                      :: j_ndicin, j_nalkin
REAL(KIND=wp)                                :: z_dummy
INTEGER,       DIMENSION(jp_ndic,jp_nalk)    :: j_dummy_arr
REAL(KIND=wp), DIMENSION(jp_ndic,jp_nalk, 2) :: z_dummy_arr
REAL(KIND=wp), DIMENSION(jp_ndic,jp_nalk, 2) :: z_cth
REAL(KIND=wp), DIMENSION(jp_ndic,jp_nalk)    :: z_ctdicvar
#endif

! Miscellaneous diagnostics
! -------------------------

INTEGER, DIMENSION(jp_ndic,jp_nalk)          :: jniter_alktan
REAL(KIND=wp), DIMENSION(jp_ndic)            :: z_htan_arr
REAL(KIND=wp), DIMENSION(jp_ndic)            :: z_atan_arr
REAL(KIND=wp), DIMENSION(jp_ndic)            :: z_hmin_gampos, z_lmin_gampos
REAL(KIND=wp), DIMENSION(jp_ndic)            :: z_crit2_gampos, z_crit0_gampos



 PRINT*
 PRINT*, 'T(K)          : ', z_t_k
 PRINT*, 'Salinity      : ', z_s
 PRINT*, 'Pressure(bar) : ', z_p_bar


 z_bortot =  A_BTOT_SALIN(z_s)
#if CASE_SW == 5
 z_po4tot =  0.1E-3_wp  ! 97.6 µM +/- 5.7 µM
 z_siltot =  0.6E-3_wp  !  606 µM +/-  31 µM
 z_nh4tot =  1.5E-3_wp  ! 1534 µM +/- 120 µM
 z_h2stot =  5.1E-3_wp  ! 5100 µM +/- 460 µM
#else
 z_po4tot =  0.5E-6_wp
 z_siltot =  5.E-6_wp
 z_nh4tot =  0.E-3_wp
 z_h2stot =  0.E-3_wp
#endif
 z_so4tot =  A_SO4TOT_SALIN(z_s)
 z_flutot =  A_FTOT_SALIN(z_s)



 PRINT*
 PRINT*, 'SigB          : ', z_bortot
 PRINT*, 'SigPO4        : ', z_po4tot
 PRINT*, 'SigSil        : ', z_siltot
 PRINT*, 'SigAmmonium   : ', z_nh4tot
 PRINT*, 'SigSulfides   : ', z_h2stot
 PRINT*, 'SigSulfates   : ', z_so4tot
 PRINT*, 'SigF          : ', z_flutot


 CALL SETUP_API4PHSWS(z_t_k, z_s, z_p_bar)

 PRINT*
 PRINT*, 'Pi_1 DIC : ', api1_dic
 PRINT*, 'Pi_2 DIC : ', api2_dic
 PRINT*, 'Pi_1 BT  : ', api1_bor
 PRINT*, 'Pi_1 PO4 : ', api1_po4
 PRINT*, 'Pi_2 PO4 : ', api2_po4
 PRINT*, 'Pi_3 PO4 : ', api3_po4
 PRINT*, 'Pi_1 Sil : ', api1_sil
 PRINT*, 'Pi_1 NH4 : ', api1_nh4
 PRINT*, 'Pi_1 H2S : ', api1_h2s
 PRINT*, 'Pi_1 HSO4: ', api1_so4
 PRINT*, 'Pi_1 HF  : ', api1_flu
 PRINT*, 'Pi_1 H2O : ', api1_wat
 PRINT*, 'pHscale : ',  aphscale

#if CASE_DIC == 4
 PRINT*
 PRINT*, 'gamma = 0 @ CO3 = ', (api2_dic/api1_dic)/aphscale, &
                               LOG10((api2_dic/api1_dic)/aphscale)
#endif

 z_dicvar_del = (z_dicvar_max-z_dicvar_min)/REAL(jp_ndic,KIND=wp)
 z_alktot_del = (z_alktot_max-z_alktot_min)/REAL(jp_nalk,KIND=wp)

 PRINT*
 PRINT*, 'Running variant "'//cp_fileid//'"'
#if defined(CASE_ZHINI)
 PRINT*, 'Initialisation procedure: explicit (pH=8)'
#else
 PRINT*, 'Initialisation procedure: default from solver (cubic polynomials)'
#endif

 PRINT*
#if (CASE_DIC == 1) || (CASE_DIC == 12) || (CASE_DIC == 13) || (CASE_DIC == 14)
 PRINT*, 'DIC Interval       : ', z_dicvar_min, z_dicvar_max
 PRINT*, 'DIC Step           : ', z_dicvar_del

 DO ji_dic = 1, jp_ndic
  z_dicvar_arr(ji_dic) =  z_dicvar_min+(REAL(ji_dic,KIND=wp)-0.5_wp) * z_dicvar_del
 ENDDO
#endif

#if CASE_DIC == 2
 PRINT*, 'log(CO2) Interval  : ', z_dicvar_min, z_dicvar_max
 PRINT*, 'log(CO2) Step      : ', z_dicvar_del

 DO ji_dic = 1, jp_ndic
  z_dicvar_arr(ji_dic) =  10._wp**(z_dicvar_min+(REAL(ji_dic,KIND=wp)-0.5_wp) * z_dicvar_del)
 ENDDO
#endif

#if CASE_DIC == 3
 PRINT*, 'HCO3 Interval      : ', z_dicvar_min, z_dicvar_max
 PRINT*, 'HCO3 Step          : ', z_dicvar_del

 DO ji_dic = 1, jp_ndic
  z_dicvar_arr(ji_dic) =  z_dicvar_min+(REAL(ji_dic,KIND=wp)-0.5_wp) * z_dicvar_del
 ENDDO
#endif

#if CASE_DIC == 4
 PRINT*, 'log(CO3) Interval  : ', z_dicvar_min, z_dicvar_max
 PRINT*, 'log(CO3) Step      : ', z_dicvar_del

 DO ji_dic = 1, jp_ndic
  z_dicvar_arr(ji_dic) =  10._wp**(z_dicvar_min+(REAL(ji_dic,KIND=wp)-0.5_wp) * z_dicvar_del)
 ENDDO
#endif


 PRINT*, 'ALK Interval       : ', z_alktot_min, z_alktot_max
 PRINT*, 'ALK Step           : ', z_alktot_del

 DO ji_alk = 1, jp_nalk
  z_alktot_arr(ji_alk) =  z_alktot_min+(REAL(ji_alk,KIND=wp)-0.5_wp) * z_alktot_del
 ENDDO

# if (CASE_DIC == 1) || (CASE_DIC == 12) || (CASE_DIC == 13) || (CASE_DIC == 14)
 PRINT*, 'DIC First/Last : ', z_dicvar_arr(1), z_dicvar_arr(jp_ndic)
#endif

#if CASE_DIC == 2
 PRINT*, 'CO2 First/Last : ', z_dicvar_arr(1), z_dicvar_arr(jp_ndic)
#endif

#if CASE_DIC == 3
 PRINT*, 'HCO3 First/Last: ', z_dicvar_arr(1), z_dicvar_arr(jp_ndic)
#endif

#if CASE_DIC == 4
 PRINT*, 'CO3 First/Last : ', z_dicvar_arr(1), z_dicvar_arr(jp_ndic)
#endif

 PRINT*, 'ALK First/Last : ', z_alktot_arr(1), z_alktot_arr(jp_nalk)


#if (CASE_DIC == 12) || (CASE_DIC == 13) || (CASE_DIC == 14)
 OPEN(UNIT=jp_resunit, FILE='atlogsec_'//cp_fileidin//'.res',FORM='UNFORMATTED',STATUS='OLD')
 READ(UNIT=jp_resunit) j_ndicin, j_nalkin
 READ(UNIT=jp_resunit) z_dummy ! z_dicvar_min
 READ(UNIT=jp_resunit) z_dummy ! z_dicvar_max
 READ(UNIT=jp_resunit) z_dummy ! z_alktot_min
 READ(UNIT=jp_resunit) z_dummy ! z_alktot_max
 READ(UNIT=jp_resunit) z_dummy ! z_dicvar_arr
 READ(UNIT=jp_resunit) z_dummy ! z_alktot_arr
 READ(UNIT=jp_resunit) z_dummy ! z_bortot
 READ(UNIT=jp_resunit) z_dummy ! z_po4tot
 READ(UNIT=jp_resunit) z_dummy ! z_siltot
 READ(UNIT=jp_resunit) z_dummy ! z_nh4tot
 READ(UNIT=jp_resunit) z_dummy ! z_h2stot
 READ(UNIT=jp_resunit) z_dummy ! z_so4tot
 READ(UNIT=jp_resunit) z_dummy ! z_flutot

                               ! jnroots_atsec
 READ(UNIT=jp_resunit) ((j_dummy_arr(ji_dic,ji_alk),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk)

                               ! z_hmin_atsec_ini
 READ(UNIT=jp_resunit) (((z_dummy_arr(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)
                               ! z_hmax_atsec_ini
 READ(UNIT=jp_resunit) (((z_dummy_arr(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)
                               ! z_h_atsec_ini
 READ(UNIT=jp_resunit) (((z_dummy_arr(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)

                               ! zval_hmax_atsec_ini
 READ(UNIT=jp_resunit) (((z_dummy_arr(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)
                               ! zval_hmax_atsec_ini
 READ(UNIT=jp_resunit) (((z_dummy_arr(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)

                               ! z_h_atsec
 READ(UNIT=jp_resunit) (((z_cth(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)

 CLOSE(UNIT=jp_resunit)

# if CASE_DIC == 12
 z_ctdicvar(:,:) =   (z_cth(:,:,1)**2) &
                   / ( api2_dic + z_cth(:,:,1)*(api1_dic + z_cth(:,:,1)) )
# endif
# if CASE_DIC == 13
 z_ctdicvar(:,:) =   (api1_dic * z_cth(:,:,1)) &
                   / ( api2_dic + z_cth(:,:,1)*(api1_dic + z_cth(:,:,1)) )
# endif
# if CASE_DIC == 14
 z_ctdicvar(:,:) =    api2_dic &
                   / ( api2_dic + z_cth(:,:,1)*(api1_dic + z_cth(:,:,1)) )
# endif

 DO ji_dic = 1, jp_ndic
   z_ctdicvar(ji_dic,:) = z_ctdicvar(ji_dic,:) * z_dicvar_arr(ji_dic)
 ENDDO

#endif


 PRINT*
 PRINT*, 'Calling SOLVE_AT_GENERAL2'
 PRINT*, '-------------------------'


 z_hini = 1.E-8_wp

 jndivg_atgen(:,:, :) = 0

 jniter_alktan(:,:) = 0
 z_htan_arr(:) = pp_hnan
 z_atan_arr(:) = HUGE(1._wp)

 z_hmin_gampos(:) = pp_hnan
 z_lmin_gampos(:) = HUGE(1._wp)
 z_crit2_gampos(:) = HUGE(1._wp)
 z_crit0_gampos(:) = HUGE(1._wp)


 CALL CPU_TIME(z_cputime_start)


#if defined(DEBUG_PHSOLVERS)
DO ji_alk = 1040, 1040

  z_alktot = z_alktot_arr(ji_alk)

  DO ji_dic = 971, 971

    print*, '----------', ji_dic, ji_alk

#else
DO ji_alk = 1, jp_nalk

  z_alktot = z_alktot_arr(ji_alk)

  DO ji_dic = 1, jp_ndic

#endif


#if (CASE_DIC == 1) || (CASE_DIC == 2) || (CASE_DIC == 3) || (CASE_DIC == 4)
    z_dicvar = z_dicvar_arr(ji_dic)
#endif

#if (CASE_DIC == 12) || (CASE_DIC == 13) || (CASE_DIC == 14)
    z_dicvar = z_ctdicvar(ji_dic, ji_alk)
#endif

    z_h = SOLVE_AT_GENERAL2(z_alktot, z_dicvar,  z_bortot, &
                                      z_po4tot,  z_siltot, &
                                      z_nh4tot,  z_h2stot, &
                                      z_so4tot,  z_flutot, &
                                      jp_dicsel, n_roots,  &
#ifdef CASE_ZHINI
                                      p_hini=z_hini,       &
#endif
                                      p_val=z_val          )

    jnroots_atgen(ji_dic,ji_alk)      = n_roots

    z_hmin_atgen_ini(ji_dic,ji_alk,:)     = ahmin_atgen_ini(:)    ! from mod_phsolvers_logging
    z_hmax_atgen_ini(ji_dic,ji_alk,:)     = ahmax_atgen_ini(:)    ! from mod_phsolvers_logging
    z_h_atgen_ini(ji_dic,ji_alk,:)        = ah_atgen_ini(:)       ! from mod_phsolvers_logging

    zval_hmin_atgen_ini(ji_dic,ji_alk,:)  = vhmin_atgen_ini(:)    ! from mod_phsolvers_logging
    zval_hmax_atgen_ini(ji_dic,ji_alk,:)  = vhmax_atgen_ini(:)    ! from mod_phsolvers_logging

    z_h_atgen(ji_dic,ji_alk, :)           = z_h(:)
    z_val_atgen(ji_dic,ji_alk, :)         = z_val(:)

    jniter_atgen(ji_dic,ji_alk, :)        = niter_atgen(:)        ! from mod_phsolvers_logging
    jniter_atgen_limin(ji_dic,ji_alk, :)  = niter_atgen_limin(:)  ! from mod_phsolvers_logging
    jniter_atgen_limax(ji_dic,ji_alk, :)  = niter_atgen_limax(:)  ! from mod_phsolvers_logging
    jniter_atgen_limabs(ji_dic,ji_alk, :) = niter_atgen_limabs(:) ! from mod_phsolvers_logging

    jniter_alktan(ji_dic,ji_alk)          = niter_alktan
    IF (z_htan_arr(ji_dic) == pp_hnan) THEN
      z_htan_arr(ji_dic)                  = ahtan
      z_atan_arr(ji_dic)                  = aatan
    ENDIF

    IF (z_hmin_gampos(ji_dic) == pp_hnan) THEN
      z_hmin_gampos(ji_dic) = ahmin_gampos
      z_lmin_gampos(ji_dic) = almin_gampos
      z_crit2_gampos(ji_dic) = acrit2_gampos
      z_crit0_gampos(ji_dic) = acrit0_gampos
    ENDIF

    IF (n_roots > 0) THEN
      IF (z_h(1) == pp_hnan) &
      jndivg_atgen(ji_dic,ji_alk, 1)  = 1
    ENDIF

    IF (n_roots > 1) THEN
      IF (z_h(2) == pp_hnan) &
      jndivg_atgen(ji_dic,ji_alk, 2)  = 1
    ENDIF

  ENDDO

ENDDO

#if defined(DEBUG_PHSOLVERS)
!STOP
#endif

 CALL CPU_TIME(z_cputime_end)

 PRINT*
 PRINT*, '[DRIVER_AT_LOGGING] elapsed time [s]           : ', z_cputime_end - z_cputime_start
 PRINT*, '[DRIVER_AT_LOGGING] total number of calls      : ', jp_nalk*jp_ndic
 PRINT*, '[DRIVER_AT_LOGGING] total number of divergences: ', SUM(jndivg_atgen)


#if defined(CREATEFILES)
 OPEN(UNIT=jp_resunit, FILE='atloggen_'//cp_fileid//'.res',FORM='UNFORMATTED')

 WRITE(UNIT=jp_resunit) jp_ndic, jp_nalk
 WRITE(UNIT=jp_resunit) z_dicvar_min
 WRITE(UNIT=jp_resunit) z_dicvar_max
 WRITE(UNIT=jp_resunit) z_alktot_min
 WRITE(UNIT=jp_resunit) z_alktot_max
 WRITE(UNIT=jp_resunit) z_dicvar_arr
 WRITE(UNIT=jp_resunit) z_alktot_arr
 WRITE(UNIT=jp_resunit) z_bortot
 WRITE(UNIT=jp_resunit) z_po4tot
 WRITE(UNIT=jp_resunit) z_siltot
 WRITE(UNIT=jp_resunit) z_nh4tot
 WRITE(UNIT=jp_resunit) z_h2stot
 WRITE(UNIT=jp_resunit) z_so4tot
 WRITE(UNIT=jp_resunit) z_flutot

 WRITE(UNIT=jp_resunit) ((jnroots_atgen(ji_dic,ji_alk),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk)

 WRITE(UNIT=jp_resunit) (((z_hmin_atgen_ini(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)
 WRITE(UNIT=jp_resunit) (((z_hmax_atgen_ini(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)
 WRITE(UNIT=jp_resunit) (((z_h_atgen_ini(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)

 WRITE(UNIT=jp_resunit) (((zval_hmin_atgen_ini(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)
 WRITE(UNIT=jp_resunit) (((zval_hmax_atgen_ini(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)

 WRITE(UNIT=jp_resunit) (((z_h_atgen(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)
 WRITE(UNIT=jp_resunit) (((z_val_atgen(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)

 WRITE(UNIT=jp_resunit) (((jndivg_atgen(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)
 WRITE(UNIT=jp_resunit) (((jniter_atgen(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)
 WRITE(UNIT=jp_resunit) (((jniter_atgen_limin(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)
 WRITE(UNIT=jp_resunit) (((jniter_atgen_limax(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)
 WRITE(UNIT=jp_resunit) (((jniter_atgen_limabs(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)

 WRITE(UNIT=jp_resunit)  ((jniter_alktan(ji_dic,ji_alk),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk)
 WRITE(UNIT=jp_resunit)   (z_htan_arr(ji_dic),ji_dic=1,jp_ndic)
 WRITE(UNIT=jp_resunit)   (z_atan_arr(ji_dic),ji_dic=1,jp_ndic)

 WRITE(UNIT=jp_resunit)   (z_hmin_gampos(ji_dic),ji_dic=1,jp_ndic)
 WRITE(UNIT=jp_resunit)   (z_lmin_gampos(ji_dic),ji_dic=1,jp_ndic)
 WRITE(UNIT=jp_resunit)   (z_crit2_gampos(ji_dic),ji_dic=1,jp_ndic)
 WRITE(UNIT=jp_resunit)   (z_crit0_gampos(ji_dic),ji_dic=1,jp_ndic)

 CLOSE(UNIT=jp_resunit)
#endif




 PRINT*
 PRINT*, 'Calling SOLVE_AT_GENERAL2_SEC'
 PRINT*, '-----------------------------'


 z_hini = 1.E-8_wp

 jndivg_atsec(:,:, :) = 0

 jniter_alktan(:,:) = 0

 z_hmin_gampos(:) = pp_hnan
 z_lmin_gampos(:) = HUGE(1._wp)
 z_crit2_gampos(:) = HUGE(1._wp)
 z_crit0_gampos(:) = HUGE(1._wp)

 CALL CPU_TIME(z_cputime_start)


#if defined(DEBUG_PHSOLVERS)
DO ji_alk = 1040, 1040

  z_alktot = z_alktot_arr(ji_alk)

  DO ji_dic = 971, 971

    print*, '----------', ji_dic, ji_alk

#else
DO ji_alk = 1, jp_nalk

  z_alktot = z_alktot_arr(ji_alk)

  DO ji_dic = 1, jp_ndic

#endif


#if (CASE_DIC == 1) || (CASE_DIC == 2) || (CASE_DIC == 3) || (CASE_DIC == 4)
    z_dicvar = z_dicvar_arr(ji_dic)
#endif

#if (CASE_DIC == 12) || (CASE_DIC == 13) || (CASE_DIC == 14)
    z_dicvar = z_ctdicvar(ji_dic, ji_alk)
#endif

    z_h = SOLVE_AT_GENERAL2_SEC(z_alktot, z_dicvar, z_bortot, &
                                          z_po4tot, z_siltot, &
                                          z_nh4tot, z_h2stot, &
                                          z_so4tot, z_flutot, &
                                          jp_dicsel, n_roots, &
#ifdef CASE_ZHINI
                                          p_hini=z_hini,      &
#endif
                                          p_val=z_val         )


    jnroots_atsec(ji_dic,ji_alk)          = n_roots

    z_hmin_atsec_ini(ji_dic,ji_alk,:)     = ahmin_atsec_ini(:)    ! from mod_phsolvers_logging
    z_hmax_atsec_ini(ji_dic,ji_alk,:)     = ahmax_atsec_ini(:)    ! from mod_phsolvers_logging
    z_h_atsec_ini(ji_dic,ji_alk,:)        = ah_atsec_ini(:)       ! from mod_phsolvers_logging

    zval_hmin_atsec_ini(ji_dic,ji_alk,:)  = vhmin_atsec_ini(:)    ! from mod_phsolvers_logging
    zval_hmax_atsec_ini(ji_dic,ji_alk,:)  = vhmax_atsec_ini(:)    ! from mod_phsolvers_logging

    z_h_atsec(ji_dic,ji_alk, :)           = z_h(:)
    z_val_atsec(ji_dic,ji_alk, :)         = z_val(:)

    jniter_atsec(ji_dic,ji_alk, :)        = niter_atsec(:)        ! from mod_phsolvers_logging
    jniter_atsec_limin(ji_dic,ji_alk, :)  = niter_atsec_limin(:)  ! from mod_phsolvers_logging
    jniter_atsec_limax(ji_dic,ji_alk, :)  = niter_atsec_limax(:)  ! from mod_phsolvers_logging
    jniter_atsec_limabs(ji_dic,ji_alk, :) = niter_atsec_limabs(:) ! from mod_phsolvers_logging

    jniter_alktan(ji_dic,ji_alk)          = niter_alktan
    IF (z_htan_arr(ji_dic) == pp_hnan) THEN
      z_htan_arr(ji_dic)                  = ahtan
      z_atan_arr(ji_dic)                  = aatan
    ENDIF

    IF (z_hmin_gampos(ji_dic) == pp_hnan) THEN
      z_hmin_gampos(ji_dic) = ahmin_gampos
      z_lmin_gampos(ji_dic) = almin_gampos
      z_crit2_gampos(ji_dic) = acrit2_gampos
      z_crit0_gampos(ji_dic) = acrit0_gampos
    ENDIF


    IF (n_roots > 0) THEN
      IF (z_h(1) == pp_hnan) &
      jndivg_atsec(ji_dic,ji_alk, 1)  = 1
    ENDIF

    IF (n_roots > 1) THEN
      IF (z_h(2) == pp_hnan) &
      jndivg_atsec(ji_dic,ji_alk, 2)  = 1
    ENDIF

  ENDDO

ENDDO


 CALL CPU_TIME(z_cputime_end)

 PRINT*
 PRINT*, '[DRIVER_AT_LOGGING] elapsed time [s]           : ', z_cputime_end - z_cputime_start
 PRINT*, '[DRIVER_AT_LOGGING] total number of calls      : ', jp_nalk*jp_ndic
 PRINT*, '[DRIVER_AT_LOGGING] total number of divergences: ', SUM(jndivg_atsec)


#if defined(CREATEFILES)
 OPEN(UNIT=jp_resunit, FILE='atlogsec_'//cp_fileid//'.res',FORM='UNFORMATTED')

 WRITE(UNIT=jp_resunit) jp_ndic, jp_nalk
 WRITE(UNIT=jp_resunit) z_dicvar_min
 WRITE(UNIT=jp_resunit) z_dicvar_max
 WRITE(UNIT=jp_resunit) z_alktot_min
 WRITE(UNIT=jp_resunit) z_alktot_max
 WRITE(UNIT=jp_resunit) z_dicvar_arr
 WRITE(UNIT=jp_resunit) z_alktot_arr
 WRITE(UNIT=jp_resunit) z_bortot
 WRITE(UNIT=jp_resunit) z_po4tot
 WRITE(UNIT=jp_resunit) z_siltot
 WRITE(UNIT=jp_resunit) z_nh4tot
 WRITE(UNIT=jp_resunit) z_h2stot
 WRITE(UNIT=jp_resunit) z_so4tot
 WRITE(UNIT=jp_resunit) z_flutot

 WRITE(UNIT=jp_resunit) ((jnroots_atsec(ji_dic,ji_alk),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk)

 WRITE(UNIT=jp_resunit) (((z_hmin_atsec_ini(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)
 WRITE(UNIT=jp_resunit) (((z_hmax_atsec_ini(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)
 WRITE(UNIT=jp_resunit) (((z_h_atsec_ini(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)

 WRITE(UNIT=jp_resunit) (((zval_hmin_atsec_ini(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)
 WRITE(UNIT=jp_resunit) (((zval_hmax_atsec_ini(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)

 WRITE(UNIT=jp_resunit) (((z_h_atsec(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)
 WRITE(UNIT=jp_resunit) (((z_val_atsec(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)

 WRITE(UNIT=jp_resunit) (((jndivg_atsec(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)
 WRITE(UNIT=jp_resunit) (((jniter_atsec(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)
 WRITE(UNIT=jp_resunit) (((jniter_atsec_limin(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)
 WRITE(UNIT=jp_resunit) (((jniter_atsec_limax(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)
 WRITE(UNIT=jp_resunit) (((jniter_atsec_limabs(ji_dic,ji_alk,ji),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk),ji=1,2)

 WRITE(UNIT=jp_resunit)  ((jniter_alktan(ji_dic,ji_alk),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk)
 WRITE(UNIT=jp_resunit)   (z_htan_arr(ji_dic),ji_dic=1,jp_ndic)
 WRITE(UNIT=jp_resunit)   (z_atan_arr(ji_dic),ji_dic=1,jp_ndic)

 WRITE(UNIT=jp_resunit)   (z_hmin_gampos(ji_dic),ji_dic=1,jp_ndic)
 WRITE(UNIT=jp_resunit)   (z_lmin_gampos(ji_dic),ji_dic=1,jp_ndic)
 WRITE(UNIT=jp_resunit)   (z_crit2_gampos(ji_dic),ji_dic=1,jp_ndic)
 WRITE(UNIT=jp_resunit)   (z_crit0_gampos(ji_dic),ji_dic=1,jp_ndic)

 CLOSE(UNIT=jp_resunit)
#endif



!==============================================================================================
END PROGRAM
!==============================================================================================

