!
!    Copyright 2014, 2015, 2017, 2018, 2020 Guy Munhoven
!
!    This file is part of µXML.
!
!    µXML is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    µXML is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with µXML.  If not, see <https://www.gnu.org/licenses/>.
!


#include <modmxm.h>

!=======================================================================
 MODULE MODMXM_STRUCTLOAD
!=======================================================================

IMPLICIT NONE

INTEGER, PARAMETER, PRIVATE :: jp_stderr = (MXM_STDERR)
#ifdef DEBUG
INTEGER, PARAMETER, PRIVATE :: jp_stddbg = (MXM_STDDBG)
#endif

CONTAINS

!===============================================================================
 FUNCTION XMLSTRUCT(str_xmlfilename,                                   &
                      n_maxlen_eltname,                                &
                      n_maxlen_attname, n_maxlen_attcont)              &
 RESULT(stkxe_events)
!===============================================================================

USE MODMXM_GENERAL, m_unit => p_unit, m_logunit => p_logunit
USE MODMXM_TASKS
USE MODMXM_STKXE

IMPLICIT NONE


! Argument variables
! ------------------

CHARACTER(LEN=*), INTENT(IN) :: str_xmlfilename
TYPE(stack_xmlevents), POINTER  :: stkxe_events

INTEGER, INTENT(OUT) :: n_maxlen_eltname
INTEGER, INTENT(OUT) :: n_maxlen_attname
INTEGER, INTENT(OUT) :: n_maxlen_attcont


! Local variables
! ---------------
INTEGER :: ntot_events=0

INTEGER :: i_iostat_curr, i_iostat_prev
CHARACTER(LEN=p_maxlen_chunk)   :: str_chunk_prev, str_chunk_curr
INTEGER, PARAMETER :: p_nchunk_buffer = 2
CHARACTER(LEN=p_nchunk_buffer*p_maxlen_chunk) :: str_chunk

INTEGER :: n_rec_prev, i_chunk_prev, i_size_prev
INTEGER :: n_rec_curr, i_chunk_curr, i_size_curr
INTEGER :: n_rec,      i_chunk
INTEGER :: i_size, i_char, nwork_chunks, nwork_left
INTEGER :: i_evrec, i_evchunk, i_evchar, i_evid

INTEGER :: i_s, i_e

TYPE(stack_xmlevents), POINTER  :: stkxe_work

CHARACTER(LEN=1) :: c_onechar
INTEGER :: ipos_nextdd   ! --   dd := Double-Dash
INTEGER :: ipos_dgt      ! ->   dgt := Dash--Greater-Than
INTEGER :: ipos_nextbbgt ! ]]>  bbgt := Bracket--Bracket--Greater-Than
INTEGER :: ipos_nextsq   ! '    sq := Single Quote
INTEGER :: ipos_nextdq   ! >    dq := Double Quote
INTEGER :: ipos_kw       ! KeyWord

INTEGER :: i_times

! Variables related to the processing of tasks
INTEGER :: i_err, n_skip_control, i_task

TYPE(integer_lifo), POINTER  :: ili_tasks
TYPE(integer_lifo), POINTER  :: ili_work

CHARACTER(LEN=*), PARAMETER &
  :: str_fmt  = '("Discarding RecChunk ", I0,".",I0, " (",A,")")'

INTEGER :: n_elts
INTEGER :: nmax_atts


! End of declarations
! -------------------


IF (p_maxlen_chunk < 10) THEN
  WRITE(jp_stderr,'("Configuration error: p_maxlen_chunk must be >= 10!")')
  WRITE(jp_stderr,'("  p_maxlen_chunk = ", I0)') p_maxlen_chunk
  WRITE(jp_stderr,'()')
  WRITE(jp_stderr,'("Please change the source code and rebuild the application.")')

  CALL ABORT()
ENDIF


ntot_events = 0

                  ! Put stkxe_events into a controlled state
NULLIFY(stkxe_events)
CALL CREATE_EVENTSTACK(stkxe_events)

stkxe_work => stkxe_events


! To start, we have to proceed onto the position of the first
! non-SPC character, and from there get the start of a tag.
! This way, we may already filter out junk from the file

NULLIFY(ili_tasks)   ! Put ili_tasks into a determined state (NULL)
                     ! and initialise it with the second task, ...
CALL PUSH_TASK(ili_tasks, p_itask_get_tag, i_err)
ili_work => ili_tasks

                     ! ... then add the first to do task
CALL PUSH_TASK(ili_work, p_itask_proceed_to_nonSPC, i_err)


#ifdef DEBUG
WRITE(jp_stddbg, '("[XMLSTRUCT]: Starting to process file """, A, """")') &
                        TRIM(str_xmlfilename)
#endif

OPEN(UNIT=m_unit, FILE=str_xmlfilename)


nwork_chunks = 0

i_size_prev  = 0
i_chunk_prev = 0
n_rec_prev   = 0


CALL READ_NEXTCHUNK(m_unit, i_iostat_curr, str_chunk_curr, i_size_curr, &
                    n_rec_curr, i_chunk_curr, l_resetcounters=.TRUE.)

IF(i_iostat_curr == p_iostat_eof) THEN

  WRITE(jp_stderr,'("Error: ", A)') '  Empty file -- aborting'
  CALL ABORT_XMLSTRUCT()

ENDIF


IF (i_size_curr > 0) THEN

  str_chunk(1:i_size_curr) = str_chunk_curr(1:i_size_curr)
  nwork_chunks = 1
  nwork_left = i_size_curr
  i_size = i_size_curr

ELSE

  i_char = 0

  IF (i_iostat_curr == p_iostat_eor) THEN
    CALL REGISTER_EVENT(n_rec_curr, i_chunk_curr, i_char, &
                        p_evid_eor, ntot_events, stkxe_work)
  ELSE
    CALL REGISTER_EVENT(n_rec_curr, i_chunk_curr, i_char, &
                        p_evid_void, ntot_events, stkxe_work)
  ENDIF

  nwork_chunks = 0

ENDIF


i_e = 0

DO

  ! <PrepareBuffer>
  ! Complete the buffer to hold as many chunks as possible
  ! up to the next EOR at most, however
  DO WHILE (     (nwork_chunks == 0)                     &
            .OR. (      (nwork_chunks < p_nchunk_buffer) &
                  .AND. (i_iostat_curr /= p_iostat_eor)))

    i_chunk_prev  = i_chunk_curr
    n_rec_prev    = n_rec_curr
    i_size_prev   = i_size_curr
    i_iostat_prev = i_iostat_curr

!~     WRITE(jp_stderr,'("-- Building str_chunk: nwork_chunks = ",I0)') &
!~       nwork_chunks

    CALL READ_NEXTCHUNK(m_unit, i_iostat_curr, str_chunk_curr, i_size_curr, &
                        n_rec_curr, i_chunk_curr)

                    ! If we have come across the EOF, exit
    IF(i_iostat_curr == p_iostat_eof) EXIT

    IF ((i_size_curr == 0) .AND. (nwork_chunks == 0)) THEN
      i_char = 0
      IF (i_iostat_curr == p_iostat_eor) THEN
        CALL REGISTER_EVENT(n_rec_curr, i_chunk_curr, i_char, &
                            p_evid_eor, ntot_events, stkxe_work)
      ELSE
        CALL REGISTER_EVENT(n_rec_curr, i_chunk_curr, i_char, &
                            p_evid_void, ntot_events, stkxe_work)
      ENDIF
      CYCLE
    ENDIF

    IF (nwork_chunks == 0) THEN
      i_size = i_size_curr
      str_chunk(1:i_size) = str_chunk_curr(1:i_size_curr)
      nwork_chunks = 1
      i_size_prev = 0
      i_e = 0
    ELSE
      IF(i_size_curr > 0) THEN
        i_size = i_size_prev + i_size_curr
        str_chunk(i_size_prev+1:i_size) = str_chunk_curr(1:i_size_curr)
      ENDIF
      nwork_chunks = nwork_chunks + 1
    ENDIF

    nwork_left = i_size - i_e

  END DO

  ! Exit if EOF
  IF(i_iostat_curr == p_iostat_eof) THEN
                    ! If we have come across the EOF while trying to
                    ! fill up the buffer, register the event and exit
    i_char = i_size_prev
    CALL REGISTER_EVENT(n_rec_curr, i_chunk_curr, i_char, &
                        p_evid_eof, ntot_events, stkxe_work)
    EXIT

  ENDIF
  ! </PrepareBuffer


  ! <ExecuteTasks>
  CALL CURR_TASK(ili_work, i_task, i_err)
  IF (i_err > 0) CALL ABORT_XMLSTRUCT


  currenttask: SELECT CASE(i_task)
  ! %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  CASE(p_itask_proceed_to_nonSPC) currenttask
    CALL TASK_PROCEED_TO_NONSPC(n_skip_control)
                  ! Proceed to next non-SPC character
                  ! n_skip_control will indicate the number
                  ! of SPC and EOR cumulatated until


  CASE(p_itask_get_tag) currenttask
    CALL TASK_GET_TAG
                  ! Check if a tag follows on the very next position (i_e+1)


  CASE(p_itask_search_next_tag) currenttask
    CALL TASK_SEARCH_NEXT_TAG
                  ! Search for the next "<"


  CASE(p_itask_identify_tag_type) currenttask
    CALL TASK_IDENTIFY_TAG_TYPE
                  ! A tag-opening "<" has been queued before
                  ! and the two-chunk buffer replenished.
                  ! We must now find out which event it initiates.
                  ! The resulting event will be located at
                  ! (i_evrec,i_evchunk:i_evchar), i.e., the position of
                  ! the opening "<".


  CASE(p_itask_process_comment) currenttask
    CALL TASK_PROCESS_COMMENT
                  ! Opening sequence of a comment tag has been identified.
                  ! Search for the corresponding closing sequence.


  CASE(p_itask_process_cdata) currenttask
    CALL TASK_PROCESS_CDATA
                  ! Opening sequence of a CDATA tag has been identified.
                  ! Search for the corresponding closing sequence.


  CASE(p_itask_process_doctype) currenttask
    CALL TASK_PROCESS_DOCTYPE
                  ! Opening sequence of a !DOCTYPE tag has been identified.
                  ! Search for the corresponding closing sequence.


  CASE(p_itask_process_desc_tag) currenttask
    CALL TASK_PROCESS_DESC_TAG
                  ! Opening sequence of a description tag (<?...?>
                  ! has been identified. Analyse the contents and
                  ! search for the corresponding closing sequence.


  CASE(p_itask_get_name) currenttask
    CALL TASK_GET_NAME
                  ! Check for a valid name right after the current
                  ! i_e position and get its extension


  CASE(p_itask_get_quoted_string) currenttask
    CALL TASK_GET_QUOTED_STRING
                  ! Get quoted string


  CASE(p_itask_process_sqstr) currenttask
    CALL TASK_PROCESS_SQSTR
                  ! Get ending of a single-quoted character string


  CASE(p_itask_process_dqstr) currenttask
    CALL TASK_PROCESS_DQSTR
                  ! Get ending of a double-quoted character string


  CASE(p_itask_process_elt_starttag) currenttask
    CALL TASK_PROCESS_ELT_STARTTAG
                  ! Process start-tag of an element


  CASE(p_itask_process_elt_endtag) currenttask
    CALL TASK_PROCESS_ELT_ENDTAG
                  ! Process end-tag of an element


  CASE(p_itask_process_attribute) currenttask
    CALL TASK_PROCESS_ATTRIBUTE
                  ! Process one attribute


  CASE(p_itask_none) currenttask
    EXIT          ! Nothing to be done: EXIT


  CASE DEFAULT currenttask
    ! Unknow task
    WRITE(jp_stderr,'("Error: Unknown task requested -- aborting")')
    CALL ABORT_XMLSTRUCT()

  ! %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  END SELECT currenttask
  ! </ExecuteTasks>


  ! <RegularizeBuffer>

  IF (i_e == i_size) THEN
                  ! Discard both chunks
    IF (i_iostat_curr == p_iostat_eor) THEN
      i_char = i_size_curr
      CALL REGISTER_EVENT(n_rec_curr, i_chunk_curr, i_char, &
                          p_evid_eor, ntot_events, stkxe_work)
    ENDIF

#   ifdef DEBUG_XMLSTRUCT_DISCARDRECCHUNKS
    SELECT CASE(nwork_chunks)
    CASE(2)
      WRITE(jp_stddbg,str_fmt) n_rec_prev, i_chunk_prev, '1 of all'
      WRITE(jp_stddbg,str_fmt) n_rec_curr, i_chunk_curr, '2 of all'
    CASE(1)
      WRITE(jp_stddbg,str_fmt) n_rec_curr, i_chunk_curr, '1 of all'
    CASE(0)
      WRITE(jp_stddbg,'(A)') ' ?? No chunk left to discard ??'
    END SELECT
#   endif

    i_size = 0
    i_size_prev = 0
    nwork_chunks = 0

    i_e = 0

    CYCLE

  ENDIF


  IF (i_e > i_size_prev) THEN

    IF (nwork_chunks == 2) THEN

#     ifdef DEBUG_XMLSTRUCT_DISCARDRECCHUNKS
      WRITE(jp_stddbg,str_fmt) n_rec_prev, i_chunk_prev, '1 of 2'
#     endif

      i_e = i_e - i_size_prev
      str_chunk(1:i_size_curr) = str_chunk_curr(1:i_size_curr)
      i_size = i_size_curr
      i_size_prev = i_size_curr
      n_rec_prev = n_rec_curr
      i_chunk_prev = i_chunk_curr
      i_iostat_prev = i_iostat_curr

      nwork_chunks = 1

    ENDIF

  ENDIF

  ! </RegularizeBuffer>

ENDDO

CLOSE(UNIT=m_unit)

                  ! Log the detected events to file
OPEN(UNIT=m_logunit, FILE=TRIM(str_xmlfilename)//'.log')

                  ! First, calculate some basic statistics.
                  ! At the same time, complete the element start-tags
                  ! with the number of attributes information
                  ! (in %idesc(p_n_supp))
CALL STAT_EVENTSTACK(stkxe_events, n_elts, nmax_atts, &
                     n_maxlen_eltname, n_maxlen_attname, &
                     n_maxlen_attcont)

                  ! Print out the summary statistical information.
WRITE(m_logunit,'("Statistics")')
WRITE(m_logunit,'("----------")')
WRITE(m_logunit,'()')
WRITE(m_logunit,'("Number of elements: ", I0)') n_elts
WRITE(m_logunit,'("Max. number of attributes per element: ", I0)') nmax_atts
WRITE(m_logunit,'("Longest element name: ", I0, " characters")') n_maxlen_eltname
WRITE(m_logunit,'("Longest attribute name: ", I0, " characters")') n_maxlen_attname
WRITE(m_logunit,'("Longest attribute content: ", I0, " characters")') n_maxlen_attcont

                  ! Dump the event sequence to the log-file
WRITE(m_logunit,'()')
WRITE(m_logunit,'()')
WRITE(m_logunit,'("Stack dump")')
WRITE(m_logunit,'("----------")')
WRITE(m_logunit,'()')

CALL DUMP_EVENTSTACK(stkxe_events, m_logunit)

CLOSE(UNIT=m_logunit)

                  ! Clean up:
NULLIFY(stkxe_work)
CALL DEALLOCATE_TASKSTACK(ili_tasks)
NULLIFY(ili_work)



RETURN


!=======================================================================
 CONTAINS
!=======================================================================


!-----------------------------------------------------------------------
 SUBROUTINE ABORT_XMLSTRUCT
!-----------------------------------------------------------------------

IMPLICIT NONE

OPEN(UNIT=m_logunit, FILE='xmlstruct_abort.log')
CALL DUMP_EVENTSTACK(stkxe_events, m_logunit)
CLOSE(UNIT=m_logunit)

CALL ABORT()

!-----------------------------------------------------------------------
 END SUBROUTINE ABORT_XMLSTRUCT
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE TASK_PROCEED_TO_NONSPC(n_skip_ret)
!-----------------------------------------------------------------------

    IMPLICIT NONE

    INTEGER :: n_skip_ret

    INTEGER, SAVE :: n_skip_internal = 0

#   ifdef DEBUG_PROCEED_TO_NONSPC
    WRITE(jp_stddbg,'("Starting TASK_PROCEED_TO_NONSPC at i_e = ", ' // &
      'I0, " with i_size = ", I0)')  i_e, i_size
#   endif

    n_skip_ret = 0

    DO WHILE(i_e < i_size)
      IF (str_chunk(i_e+1:i_e+1) == ' ') THEN
        n_skip_internal = n_skip_internal + 1
        i_e = i_e + 1
#       ifdef DEBUG_PROCEED_TO_NONSPC
        IF (i_e <= i_size_prev) THEN
          n_rec   = n_rec_prev
          i_chunk = i_chunk_prev
          i_char  = i_e
        ELSE
          n_rec   = n_rec_curr
          i_chunk = i_chunk_curr
          i_char  = i_e - i_size_prev
        ENDIF

#       ifdef DEBUG_TASKS_PROGRESS
        WRITE(jp_stddbg,'(" Info: skipping SPC at Rec.Chunk:Char ", ' // &
          'I0,".",I0,":",I0)') n_rec, i_chunk, i_char
#       endif
#       endif

        CYCLE
      ELSE

#       ifdef DEBUG_PROCEED_TO_NONSPC
        IF (i_e <= i_size_prev) THEN
          n_rec   = n_rec_prev
          i_chunk = i_chunk_prev
          i_char  = i_e
        ELSE
          n_rec   = n_rec_curr
          i_chunk = i_chunk_curr
          i_char  = i_e - i_size_prev
        ENDIF
#       ifdef DEBUG_TASKS_PROGRESS
        WRITE(jp_stddbg,'(" Info: stopping at Rec.Chunk:Char ", &
          &I0,".",I0,":",I0, " in front of """,A1, """")')
        WRITE(jp_stddbg,'("  n_skip_internal = ", I0)') n_skip_internal
#       endif
#       endif

        EXIT

      ENDIF

    ENDDO

    IF (i_e < i_size) THEN
                  ! Some non-SPC character was found
                  ! i_e points to the position right in front of it.
                  ! The task is complete and it can be taken off from
                  ! the task stack.
      n_skip_ret = n_skip_internal
      n_skip_internal = 0

      CALL PULL_TASK(ili_work, p_itask_proceed_to_nonSPC, i_err)
      IF (i_err > 0) CALL ABORT_XMLSTRUCT

    ELSE

                  ! If there is an EOR ahead, it is also considered
                  ! and we increment the n_skip_internal counter
      IF (i_iostat_curr == p_iostat_eor) n_skip_internal = n_skip_internal + 1

    ENDIF
                  ! If i_e == i_size, then nothing was found.
                  ! The task needs to be repeated with an
                  ! updated buffer.


!-----------------------------------------------------------------------
 END SUBROUTINE TASK_PROCEED_TO_NONSPC
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE TASK_GET_TAG
!-----------------------------------------------------------------------

    IMPLICIT NONE

    i_e = i_e + 1

    IF (i_e <= i_size_prev) THEN
                  ! i_e points to a char in the leftmost
                  ! (second-latest read) chunk
      i_evrec = n_rec_prev
      i_evchunk = i_chunk_prev
      i_evchar = i_e

    ELSE
                  ! i_e points to a char in the rightmost
                  ! (latest read) chunk
      i_evrec   = n_rec_curr
      i_evchunk = i_chunk_curr
      i_evchar  = i_e - i_size_prev

    ENDIF

    IF (str_chunk(i_e:i_e) == '<') THEN
                  ! There is indeed a "<" at the expected position

      CALL REGISTER_EVENT(i_evrec, i_evchunk, i_evchar, &
                          p_evid_infolt, ntot_events, stkxe_work)

                  ! Task done: pull it and push next one instead:
                  ! Identification of the tag type
      CALL PULL_TASK(ili_work, p_itask_get_tag, i_err)
      IF (i_err > 0) CALL ABORT_XMLSTRUCT
      CALL PUSH_TASK(ili_work, p_itask_identify_tag_type, i_err)

    ELSE
                  ! There is none: ERROR
      WRITE(jp_stderr,'("Error: Expected ""<"" at Rec.Chunk:Char ",' // &
        ' I0,".",I0,":",I0, ", but found """, A1, """ -- aborting")') &
        i_evrec, i_evchunk, i_evchar, str_chunk(i_e:i_e)
      CALL ABORT_XMLSTRUCT()

    ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE TASK_GET_TAG
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE TASK_SEARCH_NEXT_TAG
!-----------------------------------------------------------------------

    IMPLICIT NONE

    INTEGER :: ipos_nextlt   ! <    lt := Lower-Than
    INTEGER :: ipos_nextgt   ! >    gt := Greater-Than

    INTEGER :: i_rec, i_chunk, i_char


                  ! Get the positions of the next "<" and of the next ">"
    IF (i_e < i_size) THEN
      ipos_nextlt   = i_e + INDEX(str_chunk(i_e+1:i_size),'<')
      ipos_nextgt   = i_e + INDEX(str_chunk(i_e+1:i_size),'>')
    ELSE
      ipos_nextlt   = i_e
      ipos_nextgt   = i_e
    ENDIF


    IF (ipos_nextlt > i_e) THEN
                  ! If we have now found one, ...

      IF (ipos_nextlt <= i_size_prev) THEN
                  ! ... and if it is in the leftmost (second-latest read) chunk
        i_evrec = n_rec_prev
        i_evchunk = i_chunk_prev
        i_evchar = ipos_nextlt

      ELSE
                  ! ... and if it is in the rightmost (latest read) chunk
        i_evrec   = n_rec_curr
        i_evchunk = i_chunk_curr
        i_evchar  = ipos_nextlt - i_size_prev

      ENDIF

      CALL REGISTER_EVENT(i_evrec, i_evchunk, i_evchar, &
                          p_evid_infolt, ntot_events, stkxe_work)


      IF ((ipos_nextgt > i_e) .AND. (ipos_nextgt < ipos_nextlt)) THEN
                          ! Unfortunately, there is a ">" that comes before the "<"
        IF (ipos_nextgt <= i_size_prev) THEN
                  ! ... and if it is in the leftmost (second-latest read) chunk
          i_rec   = n_rec_prev
          i_chunk = i_chunk_prev
          i_char  = ipos_nextgt

        ELSE
                  ! ... and if it is in the rightmost (latest read) chunk
          i_rec   = n_rec_curr
          i_chunk = i_chunk_curr
          i_char  = ipos_nextgt - i_size_prev

        ENDIF

        CALL REGISTER_EVENT(i_rec, i_chunk, i_char, &
                            p_evid_infogt, ntot_events, stkxe_work)

        WRITE(jp_stderr,'("Error: ", A)') '">" found before a "<" -- aborting'
        CALL ABORT_XMLSTRUCT()

      ENDIF

                  ! OK: we got a "<" first. We may now
                  ! update the current scanning position.
      i_e = ipos_nextlt

                  ! Task "p_itask_search_next_tag" is complete:
                  ! - pull it off the stack of tasks
                  ! - push "p_itask_identify_tag_type" as the next task
      CALL PULL_TASK(ili_work, p_itask_search_next_tag, i_err)
      IF (i_err > 0) CALL ABORT_XMLSTRUCT
      CALL PUSH_TASK(ili_work, p_itask_identify_tag_type, i_err)

    ELSE
                  ! If we have not yet found any

                  ! We may however have found a ">": this is not allowed
      IF (ipos_nextgt > i_e) THEN
                  ! Unfortunately, there is a ">"
        IF (ipos_nextgt <= i_size_prev) THEN
                  ! ... and if it is in the leftmost (second-latest read) chunk
          i_rec   = n_rec_prev
          i_chunk = i_chunk_prev
          i_char  = ipos_nextgt

        ELSE
                  ! ... and if it is in the rightmost (latest read) chunk
          i_rec   = n_rec_curr
          i_chunk = i_chunk_curr
          i_char  = ipos_nextgt - i_size_prev

        ENDIF

        CALL REGISTER_EVENT(i_rec, i_chunk, i_char, &
                            p_evid_infogt, ntot_events, stkxe_work)

        WRITE(jp_stderr,'("Error: ", A)') '">" found before any "<" -- aborting'
        CALL ABORT_XMLSTRUCT()

      ELSE
                  ! OK: neither "<" nor ">".  Update the current
                  ! scanning position to repeat the the sarch.
        i_e = i_size

      ENDIF

    ENDIF


    RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE TASK_SEARCH_NEXT_TAG
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE TASK_IDENTIFY_TAG_TYPE
!-----------------------------------------------------------------------

    IMPLICIT NONE

    INTEGER          :: i_onechar
    CHARACTER(LEN=1) :: c_onechar
    CHARACTER(LEN=2) :: c_sequence
    INTEGER          :: i_task_next, i_evid_info, i_evid_tag

    LOGICAL :: l_case_xm_done = .FALSE.  ! xm := eXclamation Mark


                  ! First check if it is on the last character of the
                  ! buffer. If so, check if the last read operation
                  ! terminated by an EOR. In this case, the found "<" is
                  ! followed by the EOR, which is an error.

    ! If "<" is on the last position in the buffer, ...
    IF (i_e == i_size) THEN

      i_char = i_size_curr

      ! ... and the last read terminated by an EOR
      IF (i_iostat_curr == p_iostat_eor) THEN

        CALL REGISTER_EVENT(n_rec_curr, i_chunk_curr, i_char, &
                            p_evid_eor, ntot_events, stkxe_work)
        WRITE(jp_stderr,'("Error: ", A)') 'EOR not allowed right after "<" -- aborting'
        CALL ABORT_XMLSTRUCT()

      ELSE

        CALL REGISTER_EVENT(n_rec_curr, i_chunk_curr, i_char, &
                            p_evid_void, ntot_events, stkxe_work)
        WRITE(jp_stderr,'("Error: ", A)') 'no chars left on record (reason unknown) -- aborting'
        CALL ABORT_XMLSTRUCT()

      ENDIF

    ENDIF

                  ! There are some characters (possibly blanks, though)
                  ! left in the buffer after the position of the "<"

                  ! Probe the next character
    i_onechar = i_e + 1
    c_onechar = str_chunk(i_onechar:i_onechar)

                  ! Locate c_onechar in the chunk it belongs to
    IF (i_onechar <= i_size_prev) THEN
      i_char = i_onechar
      n_rec = n_rec_prev
      i_chunk = i_chunk_prev
    ELSE
      i_char = i_onechar - i_size_prev
      n_rec = n_rec_curr
      i_chunk = i_chunk_curr
    ENDIF

    SELECT CASE(c_onechar)
    CASE(' ')     ! %%%%%%   SPC after tag opening "<"   %%%%%%
      CALL REGISTER_EVENT(n_rec, i_chunk, i_char, &
                          p_evid_spc, ntot_events, stkxe_work)
      WRITE(jp_stderr,'("Error: ", A)') 'SPC not allowed right after "<" -- aborting'
      CALL ABORT_XMLSTRUCT()


    CASE('/','?') ! "/" after tag-opening "<": possibly element end-tag ahead
                  ! "?" after tag-opening "<": possibly description-tag ahead

      SELECT CASE(c_onechar)
      CASE ('/')
        i_evid_info = p_evid_infosl
        c_sequence  = '</'
        i_evid_tag  = p_evid_elteto
        i_task_next = p_itask_process_elt_endtag
      CASE('?')
        i_evid_info = p_evid_infoqm
        c_sequence  = '<?'
        i_evid_tag  = p_evid_descto
        i_task_next = p_itask_process_desc_tag
      END SELECT

      CALL REGISTER_EVENT(n_rec, i_chunk, i_char, &
                          i_evid_info, ntot_events, stkxe_work)

      ! If "/" (resp. "?") is on the last position in the buffer, ...
      IF (i_onechar == i_size) THEN

        i_char = i_size_curr

        ! ... and the last read terminated by an EOR
        IF (i_iostat_curr == p_iostat_eor) THEN

          CALL REGISTER_EVENT(n_rec_curr, i_chunk_curr, i_char, &
                              p_evid_eor, ntot_events, stkxe_work)
          WRITE(jp_stderr,'("Error: EOR not allowed &
            &right after """, A2, """ -- aborting")') c_sequence
          CALL ABORT_XMLSTRUCT()

        ELSE

          CALL REGISTER_EVENT(n_rec_curr, i_chunk_curr, i_char, &
                              p_evid_void, ntot_events, stkxe_work)
          WRITE(jp_stderr,'("Error: no chars left on record (why?) -- aborting")')
          CALL ABORT_XMLSTRUCT()

        ENDIF

      ENDIF

                  ! "/" (resp. "?") is not rejected: advance i_e to i_onechar
      i_e = i_onechar

                  ! There are some characters (possibly blanks, though)
                  ! left in the buffer after the position i_e
      IF (str_chunk(i_e+1:i_e+1) == ' ') THEN

        IF (i_char == p_maxlen_chunk) THEN
                  ! i_char, which currently points to the "/" in "</"
                  ! is on the last character of a chunk.
                  ! So the detected SPC is one char later, i.e., on the
                  ! first char of the following chunk.
          i_char = 1
          i_chunk = i_chunk + 1
        ELSE
          i_char = i_char + 1
        ENDIF

        CALL REGISTER_EVENT(n_rec, i_chunk, i_char, &
                            p_evid_spc, ntot_events, stkxe_work)

        WRITE(jp_stderr,'("Error: SPC not allowed &
          &right after """, A2, """ -- aborting")') c_sequence
        CALL ABORT_XMLSTRUCT()

      ENDIF


      CALL REGISTER_EVENT(i_evrec, i_evchunk, i_evchar, &
                          i_evid_tag, ntot_events, stkxe_work)

                  ! Task "p_itask_identify_tag_type" completed:
                  ! - pull it off the stack
                  ! - push "p_itask_process_elt_endtag" as next task
      CALL PULL_TASK(ili_work, p_itask_identify_tag_type, i_err)
      IF (i_err > 0) CALL ABORT_XMLSTRUCT
      CALL PUSH_TASK(ili_work, i_task_next, i_err)


    CASE('!')     ! %%%%%%   "!" after tag opening "<"   %%%%%%
                  ! Possibly
                  ! - "<!--" comment ahead. Conditions:
                  !   * "--" right behind, on the same record.
                  !   * the next "--" on this same record, or
                  !     on any following records, must be followed, on
                  !     the same record that it stands on, either by ">"
                  !     or by "->". If so, this ends the comment,
                  !     if not: Error(no "--" allowed within comments)
                  ! - "<![CDATA[" tag ahead. Conditions:
                  !   * ended by the first "]]>" on the same or any of
                  !     the following records.
                  ! - "<!DOCTYPE" ahead. Only SYSTEM DOCTYPE definitions
                  !   are currently supported. expected structure:
                  !   "<!DOCTYPE root_elt_name SYSTEM "file.dtd">
                  !   DOCTYPE compliance will not be validated!
      CALL REGISTER_EVENT(n_rec, i_chunk, i_char, &
                          p_evid_infoxm, ntot_events, stkxe_work)

      l_case_xm_done = .FALSE.

      ! Check for comment-tag opening: need at least 2 chars left
      IF (((i_size - i_onechar) > 1) .AND. (.NOT. l_case_xm_done)) THEN

        IF (str_chunk(i_onechar+1:i_onechar+2) == '--') THEN

                  ! We have found the opening of a comment tag
          CALL REGISTER_EVENT(i_evrec, i_evchunk, i_evchar, &
                              p_evid_commto, ntot_events, stkxe_work)

                  ! Task "p_itask_identify_tag_type" completed:
                  ! - discard it
                  ! - push new task "p_itask_process_comment"
          CALL PULL_TASK(ili_work, p_itask_identify_tag_type, i_err)
          IF (i_err > 0) CALL ABORT_XMLSTRUCT
          CALL PUSH_TASK(ili_work, p_itask_process_comment, i_err)

                  ! Position i_e 2 chars after the "!" (which is at i_onechar)
          i_e = i_onechar + 2

          l_case_xm_done = .TRUE.

        ENDIF

      ENDIF

      ! Check for CDATA-tag opening: need at least 7 chars left
      IF (((i_size - i_onechar) > 6) .AND. (.NOT. l_case_xm_done)) THEN

        IF (str_chunk(i_onechar+1:i_onechar+7) == '[CDATA[') THEN
                  ! We have found the opening sequence of a CDATA tag
          CALL REGISTER_EVENT(i_evrec, i_evchunk, i_evchar, &
                              p_evid_cdatao, ntot_events, stkxe_work)

                  ! Task "p_itask_identify_tag_type" completed:
                  ! - discard it
                  ! - push new task "p_itask_process_cdata"
          CALL PULL_TASK(ili_work, p_itask_identify_tag_type, i_err)
          IF (i_err > 0) CALL ABORT_XMLSTRUCT
          CALL PUSH_TASK(ili_work, p_itask_process_cdata, i_err)

                  ! Position i_e 7 chars after the "!" (which is at i_onechar)
          i_e = i_onechar + 7

          l_case_xm_done = .TRUE.

        ENDIF

      ENDIF

      ! Check for DOCTYPE-tag opening: need at least 7 chars left
      IF (((i_size - i_onechar) > 6) .AND. (.NOT. l_case_xm_done)) THEN

        IF (str_chunk(i_onechar+1:i_onechar+7) == 'DOCTYPE') THEN

                  ! We have found the opening sequence of a !DOCTYPE tag
          CALL REGISTER_EVENT(i_evrec, i_evchunk, i_evchar, &
                              p_evid_doctto, ntot_events, stkxe_work)

                  ! Position i_e 7 chars after the "!" (which is at i_onechar)
          i_e = i_onechar + 7

                  ! "<!DOCTYPE" must be followed by SPC or EOR. If not: ERROR
          IF (i_e == i_size) THEN
            IF(i_iostat_curr /= p_iostat_eor) THEN
              WRITE(jp_stderr,'("Error: ",A)') &
                '"<!DOCTYPE" not followed by SPC or EOR  -- aborting'
              CALL ABORT_XMLSTRUCT()
            ENDIF
          ELSE
            IF (str_chunk(i_e+1:i_e+1) /= ' ') THEN
              WRITE(jp_stderr,'("Error: ",A)') &
                '"<!DOCTYPE" not followed by SPC or EOR  -- aborting'
              CALL ABORT_XMLSTRUCT()
            ENDIF
          ENDIF

          l_case_xm_done = .TRUE.

                  ! Task "p_itask_identify_tag_type" completed:
                  ! - discard it
                  ! - push new task "p_itask_process_doctype"
                  ! - push new task "p_itask_proceed_to_nonSPC"
          CALL PULL_TASK(ili_work, p_itask_identify_tag_type, i_err)
          IF (i_err > 0) CALL ABORT_XMLSTRUCT
          CALL PUSH_TASK(ili_work, p_itask_process_doctype, i_err)
          CALL PUSH_TASK(ili_work, p_itask_proceed_to_nonSPC, i_err)

        ENDIF

      ENDIF


    CASE DEFAULT
                  ! Should be opening an element tag
      CALL REGISTER_EVENT(i_evrec, i_evchunk, i_evchar, &
                          p_evid_eltsto, ntot_events, stkxe_work)

      CALL PULL_TASK(ili_work, p_itask_identify_tag_type, i_err)
      IF (i_err > 0) CALL ABORT_XMLSTRUCT
      CALL PUSH_TASK(ili_work, p_itask_process_elt_starttag, i_err)


    END SELECT

    RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE TASK_IDENTIFY_TAG_TYPE
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE TASK_PROCESS_COMMENT
!-----------------------------------------------------------------------

    IMPLICIT NONE

    LOGICAL, SAVE :: l_dd_found = .FALSE.  ! dd := Double-Dash


    ! search for '--'
    IF (.NOT. l_dd_found) THEN

      ipos_nextdd = i_e + INDEX(str_chunk(i_e+1:i_size), '--')

      ! If we have now found one instance, ...
      IF (ipos_nextdd > i_e) THEN

        IF (ipos_nextdd <= i_size_prev) THEN

          i_evrec   = n_rec_prev
          i_evchunk = i_chunk_prev
          i_evchar  = ipos_nextdd

        ELSE

          i_evrec   = n_rec_curr
          i_evchunk = i_chunk_curr
          i_evchar  = ipos_nextdd - i_size_prev

        ENDIF

        CALL REGISTER_EVENT(i_evrec, i_evchunk, i_evchar, &
                            p_evid_infodd, ntot_events, stkxe_work)


        IF ((ipos_nextdd == i_size-1) .AND. (i_iostat_curr == p_iostat_eor)) THEN
          WRITE(jp_stderr,'("Error: ", A)') '"--" not allowed in a comment -- aborting'
          CALL ABORT_XMLSTRUCT()
        ENDIF

        l_dd_found = .TRUE.
        i_e = ipos_nextdd
        ipos_dgt = 0

      ELSE

                  ! Allow to continue searching for "--" from the last
                  ! char of the buffer on, except if the buffer reaches
                  ! to an EOR.
        ipos_nextdd = 0
        IF (i_iostat_curr /= p_iostat_eor) THEN
          i_e = i_size - 1
        ELSE
          i_e = i_size
        ENDIF
      ENDIF

    ELSE

      ipos_dgt = INDEX(str_chunk(i_e+1:i_size), '->')

      ! If we have now found one instance, ...
      IF ((ipos_dgt == 1) .OR. (ipos_dgt == 2)) THEN

        i_e = i_e + ipos_dgt + 1

        IF (i_e <= i_size_prev) THEN
          i_evrec   = n_rec_prev
          i_evchunk = i_chunk_prev
          i_evchar  = i_e
        ELSE
          i_evrec   = n_rec_curr
          i_evchunk = i_chunk_curr
          i_evchar  = i_e - i_size_prev
        ENDIF

        CALL REGISTER_EVENT(i_evrec, i_evchunk, i_evchar, &
                            p_evid_commtc, ntot_events, stkxe_work)

        l_dd_found = .FALSE.

                  ! Task "p_itask_process_comment" completed:
                  ! - discard it
                  ! - next task is "p_itask_search_next_tag"
        CALL PULL_TASK(ili_work, p_itask_process_comment, i_err)
        IF (i_err > 0) CALL ABORT_XMLSTRUCT
        CALL PUSH_TASK(ili_work, p_itask_search_next_tag, i_err)

      ELSE

        WRITE(jp_stderr,'("Error: ",A)') '"--" not allowed in a comment -- aborting'
        CALL ABORT_XMLSTRUCT()

      ENDIF

    ENDIF


    RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE TASK_PROCESS_COMMENT
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE TASK_PROCESS_CDATA
!-----------------------------------------------------------------------

    IMPLICIT NONE

                  ! search for ']]>'
      ipos_nextbbgt = i_e + INDEX(str_chunk(i_e+1:i_size), ']]>')

      ! If we have now found one instance, ...
      IF (ipos_nextbbgt > i_e) THEN

        i_e = ipos_nextbbgt + 2

        IF (i_e <= i_size_prev) THEN
          i_evrec = n_rec_prev
          i_evchunk = i_chunk_prev
          i_evchar = i_e
        ELSE
          i_evrec   = n_rec_curr
          i_evchunk = i_chunk_curr
          i_evchar  = i_e - i_size_prev
        ENDIF

        CALL REGISTER_EVENT(i_evrec, i_evchunk, i_evchar, &
                            p_evid_cdatac, ntot_events, stkxe_work)

                  ! Task "p_itask_process_cdata" completed:
                  ! - discard it
                  ! - push new "p_itask_search_next_tag"
        CALL PULL_TASK(ili_work, p_itask_process_cdata, i_err)
        IF (i_err > 0) CALL ABORT_XMLSTRUCT
        CALL PUSH_TASK(ili_work, p_itask_search_next_tag, i_err)

      ELSE

                  ! Allow to continue searching for "]]>" from the second-last
                  ! char of the buffer on, except if the buffer reaches
                  ! to an EOR.
        ipos_nextbbgt = 0
        IF (i_iostat_curr /= p_iostat_eor) THEN
          i_e = i_size - 2
        ELSE
          i_e = i_size
        ENDIF
      ENDIF


    RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE TASK_PROCESS_CDATA
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE TASK_PROCESS_DOCTYPE
!-----------------------------------------------------------------------

    IMPLICIT NONE

    INTEGER, SAVE :: i_stage_process_doctype = 1



    ! It is expected that i_e already points to the position before the
    ! first non-SPC character
    ! Three stages (i_stage_process_doctype):
    ! 1. Get the name of the root element (task "p_itask_get_element_name")
    !    Have i_e point to the position preceeding the next following
    !    non-SPC char. Must at least skip one separator!
    ! 2. Check if keyword "SYSTEM" follows at position i_e
    !    Have i_e point to the position preceeding the next following
    !    non-SPC char.
    ! 3. Search for quoted filename: at i_e+1, there must be a " or a '
    !    Have i_e point to the position preceeding the first following
    !    non-SPC char.
    ! 4. Find closing ">" and discard task from stack and push new task
    !    "p_itask_search_next_tag"

#     ifdef DEBUG_TASKS_PROGRESS
      WRITE(jp_stddbg,'(" Info: !DOCTYPE processing stage ",I0)') i_stage_process_doctype
#     endif

    SELECT CASE(i_stage_process_doctype)

    CASE(0)       ! Should never get here
      WRITE(jp_stderr,'("Error: ",A)') 'Process !DOCTYPE stage 0 (why?) -- aborting'
      CALL ABORT_XMLSTRUCT()

    CASE(1)
      CALL PUSH_TASK(ili_work, p_itask_proceed_to_nonspc, i_err)
      CALL PUSH_TASK(ili_work, p_itask_get_name, i_err)

                  ! the following stage will be stage 2
      i_stage_process_doctype = 2

    CASE(2)       ! The "SYSTEM" keyword must follow starting on i_e+1
      IF (n_skip_control == 0) THEN
        WRITE(jp_stderr,'("Error: root element name in !DOCTYPE")')
        WRITE(jp_stderr,'("is not followed by SPC or EOR -- aborting")')
        CALL ABORT_XMLSTRUCT()
      ENDIF

      IF (((i_size - i_e) >= 6)) THEN

        IF (str_chunk(i_e+1:i_e+6) == 'SYSTEM') THEN

          ipos_kw = i_e + 1
          i_e = i_e + 6  ! i_e := i_e + LEN('SYSTEM')

          IF (ipos_kw <= i_size_prev) THEN
            i_evrec = n_rec_prev
            i_evchunk = i_chunk_prev
            i_evchar = ipos_kw
          ELSE
            i_evrec   = n_rec_curr
            i_evchunk = i_chunk_curr
            i_evchar  = ipos_kw - i_size_prev
          ENDIF

                  ! Keyword must be followed by SPC or EOR. If not: ERROR
          IF (i_e == i_size) THEN
            IF(i_iostat_curr /= p_iostat_eor) THEN
              WRITE(jp_stderr,'("Error: ",A)') &
                '"SYSTEM" keyword in !DOCTYPE not followed by SPC or EOR  -- aborting'
              CALL ABORT_XMLSTRUCT()
            ENDIF
          ELSE
            IF (str_chunk(i_e+1:i_e+1) /= ' ') THEN
              WRITE(jp_stderr,'("Error: ",A)') &
                '"SYSTEM" keyword in !DOCTYPE not followed by SPC or EOR  -- aborting'
              CALL ABORT_XMLSTRUCT()
            ENDIF
          ENDIF

          CALL REGISTER_EVENT(i_evrec, i_evchunk, i_evchar, &
                              p_evid_kw_system, ntot_events, stkxe_work)

          CALL PUSH_TASK(ili_work, p_itask_proceed_to_nonSPC, i_err)

                  ! Afterwards, continue with stage 3
          i_stage_process_doctype = 3

        ENDIF

      ELSE
                  ! Keyword not found: ERROR
        WRITE(jp_stderr,'("Error: ",A)') '!DOCTYPE keyword "SYSTEM" not found -- aborting'
        CALL ABORT_XMLSTRUCT()

      ENDIF

    CASE(3)       ! Search for quoted filename (single or double quotes)
                  ! and proceed to the next non-SPC character
      CALL PUSH_TASK(ili_work, p_itask_proceed_to_nonSPC, i_err)
      CALL PUSH_TASK(ili_work, p_itask_get_quoted_string, i_err)

      i_stage_process_doctype = 4

    CASE(4)
                  ! Close off, if possible
      i_e = i_e + 1

      IF (i_e <= i_size_prev) THEN
        i_evrec = n_rec_prev
        i_evchunk = i_chunk_prev
        i_evchar = i_e
      ELSE
        i_evrec   = n_rec_curr
        i_evchunk = i_chunk_curr
        i_evchar  = i_e - i_size_prev
      ENDIF

      IF (str_chunk(i_e:i_e) == '>') THEN

        CALL REGISTER_EVENT(i_evrec, i_evchunk, i_evchar, &
                            p_evid_docttc, ntot_events, stkxe_work)

                  ! Task complete: pull it
                  ! Next task: proceed to non-SPC, then get tag

        CALL PULL_TASK(ili_work, p_itask_process_doctype, i_err)
        IF (i_err > 0) CALL ABORT_XMLSTRUCT
        CALL PUSH_TASK(ili_work, p_itask_get_tag, i_err)
        CALL PUSH_TASK(ili_work, p_itask_proceed_to_nonSPC, i_err)

#       ifdef DEBUG_TASKS_PROGRESS
        WRITE(jp_stddbg,'("Terminating !DOCTYPE processing at i_e = ", ' // &
          'I0, " with i_size = ", I0)')  i_e, i_size
#       endif

        i_stage_process_doctype = 1

      ELSE

        WRITE(jp_stderr,'("Error: ",A)') '!DOCTYPE-tag not correctly closed -- aborting'
        WRITE(jp_stderr,'("  expecting "">"" instead of """, A1, ' // &
          ' """ at Rec.Chunk:Char ", I0,".",I0,":",I0, " -- aborting")') &
          str_chunk(i_e:i_e), i_evrec, i_evchunk, i_evchar
        CALL ABORT_XMLSTRUCT()

      ENDIF

    END SELECT


    RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE TASK_PROCESS_DOCTYPE
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE TASK_GET_NAME
!-----------------------------------------------------------------------

    IMPLICIT NONE

    INTEGER :: n_chars_name = 0

    ! Check first char if OK for element-name start
    ! - If so register p_evid_name_start, else ERROR
    ! - read all up to EOR or non allowed char ++++++
    ! - Register p_evid_name_end
    ! - then proceed to the first non-SPC char
    ! - CALL PULL_TASK(ili_work)

    LOGICAL, SAVE :: l_firstletter = .TRUE.
    INTEGER       :: icharc

    INTEGER, SAVE :: i_


    DO WHILE (i_e < i_size)


      c_onechar = str_chunk(i_e+1:i_e+1)
      icharc = ICHAR(c_onechar)

      SELECT CASE(icharc)
        ! 48:57, 58, 65:90, 95, 97:122 = 0-9, :, A-Z, _, a-z

        CASE(48:57, 58, 65:90, 95, 97:122)

          IF (l_firstletter) THEN

            IF ((icharc >= 48) .AND. (icharc <= 57 )) THEN

              WRITE(jp_stderr,'("Error: Number literal """, A1, ' // &
                         ' """ not allowed to start a name -- aborting")') c_onechar
              CALL ABORT_XMLSTRUCT()

            ELSE
                  ! Valid first character for a name: advance one step
              i_e = i_e + 1
              n_chars_name = 1

              IF (i_e <= i_size_prev) THEN
                i_evrec   = n_rec_prev
                i_evchunk = i_chunk_prev
                i_evchar  = i_e
              ELSE
                i_evrec   = n_rec_curr
                i_evchunk = i_chunk_curr
                i_evchar  = i_e - i_size_prev
              ENDIF


              CALL REGISTER_EVENT(i_evrec, i_evchunk, i_evchar, &
                                  p_evid_nameo, ntot_events, stkxe_work)

                  ! Next letter will not be the first anymore
              l_firstletter = .FALSE.

            ENDIF

          ELSE
                  ! Valid character for inside a name: advance one step
            i_e = i_e + 1
            n_chars_name = n_chars_name + 1

                  ! Update Rec.Chunk:Char of latest valid name character
            IF (i_e <= i_size_prev) THEN
              i_evrec   = n_rec_prev
              i_evchunk = i_chunk_prev
              i_evchar  = i_e
            ELSE
              i_evrec   = n_rec_curr
              i_evchunk = i_chunk_curr
              i_evchar  = i_e - i_size_prev
            ENDIF


          ENDIF

        CASE DEFAULT
          IF (l_firstletter) THEN

            WRITE(jp_stderr,'("Error: Character """, A1, ' // &
                         ' """ not allowed to start a name -- aborting")') c_onechar
            WRITE(jp_stderr,'("  ICHAR(c_onechar) = ", I0)') icharc
            CALL ABORT_XMLSTRUCT()

          ELSE

            CALL REGISTER_EVENT(i_evrec, i_evchunk, i_evchar, &
                                p_evid_namec, ntot_events, stkxe_work, &
                                n_supp = n_chars_name)

            CALL PULL_TASK(ili_work, p_itask_get_name, i_err)
            IF (i_err > 0) CALL ABORT_XMLSTRUCT

            n_chars_name = 0
            l_firstletter = .TRUE.
            RETURN

          ENDIF

      END SELECT

    END DO

                ! If we get here, the name is not yet finished,
                ! except if an EOR follows immediately
    IF (i_e == i_size) THEN

      IF (i_iostat_curr == p_iostat_eor) THEN

        CALL REGISTER_EVENT(i_evrec, i_evchunk, i_evchar, &
                            p_evid_namec, ntot_events, stkxe_work, &
                            n_supp = n_chars_name)

        CALL PULL_TASK(ili_work, p_itask_get_name, i_err)
        IF (i_err > 0) CALL ABORT_XMLSTRUCT
        l_firstletter = .TRUE.

      ELSE
                  ! the final char in the buffer has been reached,
                  ! which was not yet terminated by EOR.
                  ! Keep l_firstletter at .FALSE.,
                  ! and leave the current task active
        CONTINUE

      ENDIF

    ENDIF


    RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE TASK_GET_NAME
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE TASK_GET_QUOTED_STRING
!-----------------------------------------------------------------------

    IMPLICIT NONE

    i_e = i_e + 1

    IF (i_e <= i_size_prev) THEN
      i_evrec = n_rec_prev
      i_evchunk = i_chunk_prev
      i_evchar = i_e
    ELSE
      i_evrec   = n_rec_curr
      i_evchunk = i_chunk_curr
      i_evchar  = i_e - i_size_prev
    ENDIF

    SELECT CASE (str_chunk(i_e:i_e))
    CASE('"')
      CALL REGISTER_EVENT(i_evrec, i_evchunk, i_evchar, &
                          p_evid_dqstro, ntot_events, stkxe_work)
      CALL PULL_TASK(ili_work, p_itask_get_quoted_string, i_err)
      IF (i_err > 0) CALL ABORT_XMLSTRUCT
      CALL PUSH_TASK(ili_work, p_itask_process_dqstr, i_err)

    CASE("'")
      CALL REGISTER_EVENT(i_evrec, i_evchunk, i_evchar, &
                          p_evid_sqstro, ntot_events, stkxe_work)
      CALL PULL_TASK(ili_work, p_itask_get_quoted_string, i_err)
      IF (i_err > 0) CALL ABORT_XMLSTRUCT
      CALL PUSH_TASK(ili_work, p_itask_process_sqstr, i_err)

    CASE DEFAULT
      WRITE(jp_stderr,'("Error: expecting single or double quotes instead '// &
        ' of """, A1, """ at Rec.Chunk:Char ", I0,".",I0,":",I0, " -- aborting")') &
        str_chunk(i_e:i_e), i_evrec, i_evchunk, i_evchar
      CALL ABORT_XMLSTRUCT()

    END SELECT


    RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE TASK_GET_QUOTED_STRING
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE TASK_PROCESS_SQSTR
!-----------------------------------------------------------------------

    IMPLICIT NONE

    INTEGER :: n_chars_sqstr = 0


    IF (i_e < i_size) THEN

      ipos_nextsq = i_e + INDEX(str_chunk(i_e+1:i_size), "'")

      IF (ipos_nextsq > i_e) THEN

        n_chars_sqstr = n_chars_sqstr + ipos_nextsq - i_e - 1

        IF (ipos_nextsq <= i_size_prev) THEN
          i_evrec = n_rec_prev
          i_evchunk = i_chunk_prev
          i_evchar = ipos_nextsq
        ELSE
          i_evrec   = n_rec_curr
          i_evchunk = i_chunk_curr
          i_evchar  = ipos_nextsq - i_size_prev
        ENDIF

        CALL REGISTER_EVENT(i_evrec, i_evchunk, i_evchar, &
                            p_evid_sqstrc, ntot_events, stkxe_work, &
                            n_supp = n_chars_sqstr)

                  ! Task "p_itask_process_sqstr" is complete
        CALL PULL_TASK(ili_work, p_itask_process_sqstr, i_err)
        IF (i_err > 0) CALL ABORT_XMLSTRUCT

                  ! Update current scanning position
        i_e = ipos_nextsq
        n_chars_sqstr = 0

      ELSE
        n_chars_sqstr = n_chars_sqstr + i_size - i_e
        i_e = i_size
      ENDIF

    ENDIF


    RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE TASK_PROCESS_SQSTR
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE TASK_PROCESS_DQSTR
!-----------------------------------------------------------------------

    IMPLICIT NONE

    INTEGER :: n_chars_dqstr = 0


    IF (i_e < i_size) THEN

      ipos_nextdq = i_e + INDEX(str_chunk(i_e+1:i_size), '"')

      IF (ipos_nextdq > i_e) THEN

        n_chars_dqstr = n_chars_dqstr + ipos_nextdq - i_e - 1

        IF (ipos_nextdq <= i_size_prev) THEN
          i_evrec = n_rec_prev
          i_evchunk = i_chunk_prev
          i_evchar = ipos_nextdq
        ELSE
          i_evrec   = n_rec_curr
          i_evchunk = i_chunk_curr
          i_evchar  = ipos_nextdq - i_size_prev
        ENDIF

        CALL REGISTER_EVENT(i_evrec, i_evchunk, i_evchar, &
                            p_evid_dqstrc, ntot_events, stkxe_work, &
                            n_supp = n_chars_dqstr)

                  ! Task "p_itask_process_dqstr" is complete
        CALL PULL_TASK(ili_work, p_itask_process_dqstr, i_err)
        IF (i_err > 0) CALL ABORT_XMLSTRUCT

                  ! Update current scanning position
        i_e = ipos_nextdq
        n_chars_dqstr = 0

      ELSE
        n_chars_dqstr = n_chars_dqstr + i_size - i_e
        i_e = i_size
      ENDIF

    ENDIF


    RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE TASK_PROCESS_DQSTR
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE TASK_PROCESS_ELT_STARTTAG
!-----------------------------------------------------------------------

    IMPLICIT NONE

    INTEGER, SAVE :: i_stage_process_elt_starttag = 1
    INTEGER :: ipos_att
    CHARACTER(LEN=1) :: c_onechar

    ! It is expected that i_e already points to the position before the
    ! first character of the name
    ! Three stages (i_stage_process_elt_starttag):
    ! 1. Get the name of the element (task "p_itask_get_name"). At the
    !    end, have i_e point to the position preceeding the next
    !    following non-SPC char.
    ! 2. Go for the first attribute if any (p_itask_get_attribute)
    !    Have i_e point to the position preceeding the next following
    !    non-SPC char.
    ! 3. Search for quoted filename: at i_e+1, there must be a " or a '
    !    Have i_e point to the position preceeding the first following
    !    non-SPC char.
    ! 4. Find closing ">" and discard task from stack and push new task
    !    "p_itask_search_next_tag"

#   ifdef DEBUG_TASKS_PROGRESS
    WRITE(jp_stddbg,'(" Info: element start-tag processing stage ",I0)') &
      i_stage_process_elt_starttag
#   endif

    SELECT CASE(i_stage_process_elt_starttag)

    CASE(1)
      CALL PUSH_TASK(ili_work, p_itask_proceed_to_nonSPC, i_err)
      CALL PUSH_TASK(ili_work, p_itask_get_name, i_err)

                  ! The following stage will be stage 2
      i_stage_process_elt_starttag = 2

    CASE(2)
      IF (n_skip_control == 0) THEN
                  ! If no SPC or EOR have been skipped, there cannot be
                  ! any attributes: proceed to the last stage (closing).
        i_stage_process_elt_starttag = 3

      ELSE
                  ! If the character at i_e+1 is not "/" or ">",
                  ! then check for attributes.
        ipos_att = i_e + 1
        c_onechar = str_chunk(ipos_att:ipos_att)
        IF ((c_onechar == '/') .OR. (c_onechar == '>')) THEN
                  ! Current char not possibly a start of a name,
                  ! but rather the closing of the tag: proceed to last stage
          i_stage_process_elt_starttag = 3

        ELSE

          IF (ipos_att <= i_size_prev) THEN
            i_evrec   = n_rec_prev
            i_evchunk = i_chunk_prev
            i_evchar  = ipos_att
          ELSE
            i_evrec   = n_rec_curr
            i_evchunk = i_chunk_curr
            i_evchar  = ipos_att - i_size_prev
          ENDIF

          CALL PUSH_TASK(ili_work, p_itask_proceed_to_nonSPC, i_err)
          CALL PUSH_TASK(ili_work, p_itask_process_attribute, i_err)

        ENDIF

      ENDIF

    CASE(3)
                  ! Close off, if possible
      i_e = i_e + 1

      IF (i_e <= i_size_prev) THEN
        i_evrec   = n_rec_prev
        i_evchunk = i_chunk_prev
        i_evchar  = i_e
      ELSE
        i_evrec   = n_rec_curr
        i_evchunk = i_chunk_curr
        i_evchar  = i_e - i_size_prev
      ENDIF


      IF (str_chunk(i_e:i_e) == '>') THEN

        CALL REGISTER_EVENT(i_evrec, i_evchunk, i_evchar, &
                            p_evid_eltstc, ntot_events, stkxe_work)

                  ! Task complete: pull it
                  ! Next task: proceed to non-SPC, then get tag

        CALL PULL_TASK(ili_work, p_itask_process_elt_starttag, i_err)
        IF (i_err > 0) CALL ABORT_XMLSTRUCT
        CALL PUSH_TASK(ili_work, p_itask_search_next_tag, i_err)

#       ifdef DEBUG_TASKS_PROGRESS
        WRITE(jp_stddbg,'("Terminating element start-tag processing &
          &at i_e = ", I0, " with i_size = ", I0)')  i_e, i_size
#       endif

        i_stage_process_elt_starttag = 1

      ELSEIF ((i_size - i_e) > 0) THEN

        IF (str_chunk(i_e:i_e+1) == '/>') THEN

          i_e = i_e + 1

          IF (i_e <= i_size_prev) THEN
            i_evrec   = n_rec_prev
            i_evchunk = i_chunk_prev
            i_evchar  = i_e
          ELSE
            i_evrec   = n_rec_curr
            i_evchunk = i_chunk_curr
            i_evchar  = i_e - i_size_prev
          ENDIF

          CALL REGISTER_EVENT(i_evrec, i_evchunk, i_evchar, &
                              p_evid_eeltstc, ntot_events, stkxe_work)

                  ! Task complete: pull it
                  ! Next task: proceed to non-SPC, then get tag

          CALL PULL_TASK(ili_work, p_itask_process_elt_starttag, i_err)
          IF (i_err > 0) CALL ABORT_XMLSTRUCT
          CALL PUSH_TASK(ili_work, p_itask_search_next_tag, i_err)

#         ifdef DEBUG_TASKS_PROGRESS
          WRITE(jp_stddbg,'("Terminating (empty) element start-tag processing &
            &at i_e = ", I0, " with i_size = ", I0)')  i_e, i_size
#         endif

          i_stage_process_elt_starttag = 1

        ENDIF

      ENDIF

                  ! If the task is not yet complete: Error
      IF (i_stage_process_elt_starttag /= 1) THEN

        WRITE(jp_stderr,'("Error: ",A)') 'Element start-tag not correctly closed -- aborting'
        WRITE(jp_stderr,'("  expecting "">"" instead of """, A1, ' // &
          ' """ at Rec.Chunk:Char ", I0,".",I0,":",I0)') &
          str_chunk(i_e:i_e), i_evrec, i_evchunk, i_evchar
        IF ((i_size - i_e) > 0) THEN
          WRITE(jp_stderr,'("  or ""/>"" instead of """, A2, ' // &
            ' """ at Rec.Chunk:Char ", I0,".",I0,":",I0)') &
            str_chunk(i_e:i_e+1), i_evrec, i_evchunk, i_evchar
        ENDIF
        CALL ABORT_XMLSTRUCT()

      ENDIF

    CASE DEFAULT
      WRITE(jp_stderr,'("Error: ",A)') 'Unknown stage for element start-tag processing -- aborting'
      WRITE(jp_stderr,'("  i_stage_process_elt_starttag = ", I0)')  i_stage_process_elt_starttag

    END SELECT


    RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE TASK_PROCESS_ELT_STARTTAG
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE TASK_PROCESS_DESC_TAG
!-----------------------------------------------------------------------

    IMPLICIT NONE

    INTEGER, SAVE :: i_stage_process_desc_tag = 1
    INTEGER :: ipos_att
    CHARACTER(LEN=1) :: c_onechar

    ! It is expected that i_e already points to the position before the
    ! first character of the name (i.e., on the "?" in "<?"
    ! Three stages (i_stage_process_descriptiontag):
    ! 1. Get the name of the element (task "p_itask_get_name"). At the
    !    end, have i_e point to the position preceeding the next
    !    following non-SPC char.
    ! 2. Go for the first attribute if any (p_itask_get_attribute)
    !    and require i_e point to the position preceeding the next
    !    following non-SPC char.
    ! 3. Find the closing "?>", pull task off the stack and push
    !    new task "p_itask_search_next_tag"

#   ifdef DEBUG_TASKS_PROGRESS
    WRITE(jp_stddbg,'(" Info: description-tag processing stage ",I0)') &
      i_stage_process_desc_tag
#   endif

    SELECT CASE(i_stage_process_desc_tag)

    CASE(1)
                  ! Second task to execute (LIFO stack!)
      CALL PUSH_TASK(ili_work, p_itask_proceed_to_nonSPC, i_err)
                  ! First task to execute (LIFO stack!)
      CALL PUSH_TASK(ili_work, p_itask_get_name, i_err)

                  ! The following stage will be stage 2
      i_stage_process_desc_tag = 2

    CASE(2)
      IF (n_skip_control == 0) THEN
                  ! If no SPC or EOR have been skipped, there cannot be
                  ! any attributes: proceed to the last stage (closing).
        i_stage_process_desc_tag = 3

      ELSE
                  ! If the character at i_e+1 is not "?"
                  ! then check for attributes.
        ipos_att = i_e + 1
        c_onechar = str_chunk(ipos_att:ipos_att)
        IF (c_onechar == '?') THEN
                  ! Current char not possibly a start of a name,
                  ! but rather the closing of the tag:
                  ! proceed to the last stage
          i_stage_process_desc_tag = 3

        ELSE

          IF (ipos_att <= i_size_prev) THEN
            i_evrec   = n_rec_prev
            i_evchunk = i_chunk_prev
            i_evchar  = ipos_att
          ELSE
            i_evrec   = n_rec_curr
            i_evchunk = i_chunk_curr
            i_evchar  = ipos_att - i_size_prev
          ENDIF

                  ! Second task to execute (LIFO stack!)
          CALL PUSH_TASK(ili_work, p_itask_proceed_to_nonSPC, i_err)
                  ! First task to execute (LIFO stack!)
          CALL PUSH_TASK(ili_work, p_itask_process_attribute, i_err)

        ENDIF

      ENDIF

    CASE(3)
                  ! Close off, if possible
      i_e = i_e + 1

      IF (i_e <= i_size_prev) THEN
        i_evrec   = n_rec_prev
        i_evchunk = i_chunk_prev
        i_evchar  = i_e
      ELSE
        i_evrec   = n_rec_curr
        i_evchunk = i_chunk_curr
        i_evchar  = i_e - i_size_prev
      ENDIF


      IF ((i_size - i_e) > 0) THEN

        IF (str_chunk(i_e:i_e+1) == '?>') THEN

          i_e = i_e + 1

          IF (i_e <= i_size_prev) THEN
            i_evrec   = n_rec_prev
            i_evchunk = i_chunk_prev
            i_evchar  = i_e
          ELSE
            i_evrec   = n_rec_curr
            i_evchunk = i_chunk_curr
            i_evchar  = i_e - i_size_prev
          ENDIF

          CALL REGISTER_EVENT(i_evrec, i_evchunk, i_evchar, &
                              p_evid_desctc, ntot_events, stkxe_work)

                  ! Task complete: pull it off the stack
          CALL PULL_TASK(ili_work, p_itask_process_desc_tag, i_err)
          IF (i_err > 0) CALL ABORT_XMLSTRUCT
                  ! Next tasks: proceed to non-SPC, then get new tag
                  ! Second task to execute (LIFO stack!)
          CALL PUSH_TASK(ili_work, p_itask_get_tag, i_err)
                  ! First task to execute (LIFO stack!)
          CALL PUSH_TASK(ili_work, p_itask_proceed_to_nonSPC, i_err)

#         ifdef DEBUG_TASKS_PROGRESS
          WRITE(jp_stddbg,'("Terminating description-tag processing&
            & at i_e = ", I0, " with i_size = ", I0)')  i_e, i_size
#         endif

          i_stage_process_desc_tag = 1

        ENDIF

      ENDIF

                  ! If the task is not yet complete: Error
      IF (i_stage_process_desc_tag /= 1) THEN

        WRITE(jp_stderr,'("Error: ",A)') 'Description-tag not correctly closed -- aborting'
        IF ((i_size - i_e) > 0) THEN
          WRITE(jp_stderr,'("  expecting ""?>"" instead of &
            &""", A2, """ at Rec.Chunk:Char ", I0,".",I0,":",I0)') &
          str_chunk(i_e:i_e+1), i_evrec, i_evchunk, i_evchar
        ELSE
          WRITE(jp_stderr,'("  expecting ""?>"" instead of &
            &""", A1, """ at Rec.Chunk:Char ", I0,".",I0,":",I0)') &
          str_chunk(i_e:i_e), i_evrec, i_evchunk, i_evchar
        ENDIF
        CALL ABORT_XMLSTRUCT()

      ENDIF

    CASE DEFAULT
      WRITE(jp_stderr,'("Error: ",A)') 'Unknown stage for description-tag processing -- aborting'
      WRITE(jp_stderr,'("  i_stage_process_elt_starttag = ", I0)')  i_stage_process_desc_tag

    END SELECT


    RETURN

!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 END SUBROUTINE TASK_PROCESS_DESC_TAG
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -



!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 SUBROUTINE TASK_PROCESS_ELT_ENDTAG
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    IMPLICIT NONE

    INTEGER, SAVE :: i_stage_process_elt_endtag = 1
    CHARACTER(LEN=1) :: c_onechar

    ! It is expected that i_e already points to the position before the
    ! first character of the name
    ! Two stages (i_stage_process_elt_endtag):
    ! 1. Get the name of the element (task "p_itask_get_name"). At the
    !    end, have i_e point to the position preceeding the next
    !    following non-SPC char.
    ! 4. Find closing ">" and discard task from stack and push new task
    !    "p_itask_search_next_tag"

#   ifdef DEBUG_TASKS_PROGRESS
    WRITE(jp_stddbg,'(" Info: element end-tag processing stage ",I0)') &
      i_stage_process_elt_endtag
#   endif

    SELECT CASE(i_stage_process_elt_endtag)

    CASE(1)
      CALL PUSH_TASK(ili_work, p_itask_proceed_to_nonSPC, i_err)
      CALL PUSH_TASK(ili_work, p_itask_get_name, i_err)

                  ! The following stage will be stage 2
      i_stage_process_elt_endtag = 2

    CASE(2)
                  ! Close off, if possible
      i_e = i_e + 1

      IF (i_e <= i_size_prev) THEN
        i_evrec   = n_rec_prev
        i_evchunk = i_chunk_prev
        i_evchar  = i_e
      ELSE
        i_evrec   = n_rec_curr
        i_evchunk = i_chunk_curr
        i_evchar  = i_e - i_size_prev
      ENDIF


      IF (str_chunk(i_e:i_e) == '>') THEN

        CALL REGISTER_EVENT(i_evrec, i_evchunk, i_evchar, &
                            p_evid_eltetc, ntot_events, stkxe_work)

                  ! Task complete: pull it
                  ! Next task: proceed to non-SPC, then get tag

        CALL PULL_TASK(ili_work, p_itask_process_elt_endtag, i_err)
        IF (i_err > 0) CALL ABORT_XMLSTRUCT
        CALL PUSH_TASK(ili_work, p_itask_search_next_tag, i_err)

#       ifdef DEBUG_TASKS_PROGRESS
        WRITE(jp_stddbg,'("Terminating element end-tag processing &
          &at i_e = ", I0, " with i_size = ", I0)')  i_e, i_size
#       endif

        i_stage_process_elt_endtag = 1

      ELSE

        WRITE(jp_stderr,'("Error: ",A)') 'Element end-tag not correctly closed -- aborting'
        WRITE(jp_stderr,'("  expecting "">"" instead of """, A1, ' // &
          ' """ at Rec.Chunk:Char ", I0,".",I0,":",I0)') &
          str_chunk(i_e:i_e), i_evrec, i_evchunk, i_evchar
        CALL ABORT_XMLSTRUCT()

      ENDIF

    CASE DEFAULT
      WRITE(jp_stderr,'("Error: ",A)') 'Unknown stage for element end-tag processing -- aborting'
      WRITE(jp_stderr,'("  i_stage_process_elt_endtag = ", I0)')  i_stage_process_elt_endtag

    END SELECT


    RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE TASK_PROCESS_ELT_ENDTAG
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE TASK_PROCESS_ATTRIBUTE
!-----------------------------------------------------------------------

    IMPLICIT NONE

    INTEGER, SAVE :: i_stage_process_attribute = 1

    CHARACTER(LEN=1) :: c_onechar

    ! It is expected that i_e already points to the position before the
    ! first character of the name
    ! Three stages (i_stage_process_elt_starttag):
    ! 1. Get the name of the element (task "p_itask_get_name"). At the
    !    end, have i_e point to the position preceeding the next
    !    following non-SPC char.
    ! 2. Go for the first attribute if any (p_itask_get_attribute)
    !    Have i_e point to the position preceeding the next following
    !    non-SPC char.
    ! 3. Search for quoted filename: at i_e+1, there must be a " or a '
    !    Have i_e point to the position preceeding the first following
    !    non-SPC char.
    ! 4. Find closing ">" and discard task from stack and push new task
    !    "p_itask_search_next_tag"

#   ifdef DEBUG_TASKS_PROGRESS
    WRITE(jp_stddbg,'(" Info: attribute processing stage ",I0)') &
      i_stage_process_attribute
#   endif

    SELECT CASE(i_stage_process_attribute)
    CASE(1)
      IF ((i_e+1) <= i_size_prev) THEN
        i_evrec   = n_rec_prev
        i_evchunk = i_chunk_prev
        i_evchar  = i_e + 1
      ELSE
        i_evrec   = n_rec_curr
        i_evchunk = i_chunk_curr
        i_evchar  = i_e + 1 - i_size_prev
      ENDIF

      CALL REGISTER_EVENT(i_evrec, i_evchunk, i_evchar, &
                          p_evid_atto, ntot_events, stkxe_work)


      CALL PUSH_TASK(ili_work, p_itask_proceed_to_nonspc, i_err)
      CALL PUSH_TASK(ili_work, p_itask_get_name, i_err)

      i_stage_process_attribute = 2

    CASE(2)
      c_onechar = str_chunk(i_e+1:i_e+1)
      IF (c_onechar == '=') THEN
        i_e = i_e + 1
        IF (i_e <= i_size_prev) THEN
          i_evrec   = n_rec_prev
          i_evchunk = i_chunk_prev
          i_evchar  = i_e
        ELSE
          i_evrec   = n_rec_curr
          i_evchunk = i_chunk_curr
          i_evchar  = i_e - i_size_prev
        ENDIF

        CALL REGISTER_EVENT(i_evrec, i_evchunk, i_evchar, &
                            p_evid_infoeq, ntot_events, stkxe_work)


        CALL PUSH_TASK(ili_work, p_itask_proceed_to_nonspc, i_err)

        i_stage_process_attribute = 3
      ELSE
        WRITE(jp_stderr,'("Error: ""="" sign not found after attribute name -- aborting")')
        WRITE(jp_stderr,'("  found """, A1, """ instead")') c_onechar
        CALL ABORT_XMLSTRUCT()
      ENDIF

    CASE(3)
      CALL PUSH_TASK(ili_work, p_itask_get_quoted_string, i_err)
      i_stage_process_attribute = 4

    CASE(4)
      IF (i_e <= i_size_prev) THEN
        i_evrec   = n_rec_prev
        i_evchunk = i_chunk_prev
        i_evchar  = i_e
      ELSE
        i_evrec   = n_rec_curr
        i_evchunk = i_chunk_curr
        i_evchar  = i_e - i_size_prev
      ENDIF

      CALL REGISTER_EVENT(i_evrec, i_evchunk, i_evchar, &
                          p_evid_attc, ntot_events, stkxe_work)

                  ! Task completed: pull it off the stack
      CALL PULL_TASK(ili_work, p_itask_process_attribute, i_err)
      IF (i_err > 0) CALL ABORT_XMLSTRUCT

      i_stage_process_attribute = 1

    END SELECT


    RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE TASK_PROCESS_ATTRIBUTE
!-----------------------------------------------------------------------


!===============================================================================
 END FUNCTION XMLSTRUCT
!===============================================================================




!=======================================================================
 FUNCTION XMLLOAD(str_xmlfilename, stkxe_root, n_maxdepth)             &
 RESULT(stkmx_root)
!=======================================================================

USE MODMXM_GENERAL, m_unit => p_unit, m_logunit => p_logunit
USE MODMXM_STKXE
USE MODMXM_STKRC
USE MODMXM_STKMX


IMPLICIT NONE

! Type definitions
! ----------------

TYPE stkrc_lifo
  TYPE(stack_recchunks), POINTER :: stkrc
  TYPE(stkrc_lifo), POINTER      :: prev
  TYPE(stkrc_lifo), POINTER      :: next
END TYPE


! Argument list variables
! -----------------------

CHARACTER(LEN=*), INTENT(IN)    :: str_xmlfilename
TYPE(stack_minixml), POINTER    :: stkmx_root
TYPE(stack_xmlevents), POINTER  :: stkxe_root
INTEGER, OPTIONAL, INTENT(OUT)  :: n_maxdepth


! Local variables
! ---------------

CHARACTER(LEN=*), PARAMETER :: str_errprefix = '[XMLLOAD] Error: '
CHARACTER(LEN=*), PARAMETER :: str_infoprefix = '[XMLLOAD] Info: '

TYPE(stack_minixml), POINTER   :: stkmx_work
TYPE(stack_xmlevents), POINTER :: stkxe_work
TYPE(stack_recchunks), POINTER :: stkrc_data
INTEGER :: nlentrim_data, nlen_data

INTEGER :: nloc_maxdepth

                  ! Subroutine-wide (COMmon) counters, indices and content
                  ! relative to information read in from the XML-file
INTEGER :: icom_iostat, icom_size, ncom_rec, icom_chunk
CHARACTER(LEN=p_maxlen_chunk) :: strcom_chunk
INTEGER :: nprevcom_rec, iprevcom_chunk, iprevcom_char
INTEGER :: nprevcom_supp

LOGICAL :: l_resetcounters

INTEGER :: n_rec, i_chunk, i_char

TYPE(stkrc_lifo), POINTER :: stksl_eltnames
INTEGER :: nopen_elts
INTEGER :: ifound_evid

INTEGER :: i_evid, n_atts, n_supp
INTEGER :: nlay_stkrc, nlen_stkrc


! Operations
! ----------

IF (.NOT. ASSOCIATED(stkxe_root)) THEN
  WRITE(jp_stderr, '("[XMLLOAD] Error: ", A)') &
    'XML-event stack "stkxe_root" is not associated -- aborting'
  CALL ABORT_XMLLOAD()
ENDIF


NULLIFY(stkmx_root)
CALL STKMX_CREATE_ROOT(stkmx_root)


nopen_elts = 0
nloc_maxdepth = 0


OPEN(UNIT=m_unit, FILE=str_xmlfilename)

stkxe_work => stkxe_root
stkmx_work => stkmx_root


! Set up the MiniXML stack: initialize the root element
! -----------------------------------------------------

                  ! Read the first chunk from file into "strcom_chunk"
CALL READ_NEXTCHUNK(m_unit, icom_iostat, strcom_chunk,   &
                    icom_size, ncom_rec, icom_chunk, &
                    l_resetcounters=.TRUE.)

                  ! Search for the first start-tag opening event
CALL STKXE_SEEK_ONE_FROMCURR(stkxe_work, p_evid_eltsto, ifound_evid, &
                             n_rec, i_chunk, i_char, n_atts)

IF (ifound_evid == p_evid_void) THEN
  WRITE(jp_stderr, '("'//str_errprefix//'", A)') &
    'No elements found in file -- aborting'
  CALL ABORT_XMLLOAD()
ENDIF
! Found the first element start-tag
#ifdef DEBUG_XMLLOADTASKS_PROGRESS
WRITE(jp_stddbg,'("'//str_infoprefix//'stkxe indicates Elt start-tag opening at &
              &Rec.Chunk:Char ", I0,".",I0,":",I0)') n_rec, i_chunk, i_char
WRITE(jp_stddbg,'("'//str_infoprefix//'Calling XMLLOADTASK_PROCESS_STARTTAG to process start-tag")')
#endif

! Process it: it must be the root element

CALL XMLLOADTASK_PROCESS_STARTTAG(n_atts, nopen_elts)
nloc_maxdepth = MAX(nloc_maxdepth, nopen_elts)


DO WHILE (nopen_elts > 0)

  ! get the first following non-info tag

  CALL STKXE_SEEK_NONINFO_AFTERCURR(stkxe_work, ifound_evid,           &
                                    n_rec, i_chunk, i_char, n_supp)

  IF (ifound_evid == p_evid_void) THEN
    WRITE(jp_stderr, '("'//str_errprefix//'", A)') &
      'Only info elements left in XML-file -- aborting'
    CALL ABORT_XMLLOAD()
  ENDIF

#   ifdef DEBUG_XMLLOADTASKS_PROGRESS
    WRITE(jp_stddbg,'("'//str_infoprefix//'stkxe indicates Non-info-tag at &
              &Rec.Chunk:Char ", I0,".",I0,":",I0)') n_rec, i_chunk, i_char
    WRITE(jp_stddbg,'("'//str_infoprefix//'Calling XMLLOADTASK_PROCESS_STKRC to read PCDATA")')
#   endif
                  ! Provisionally read in data into "stkrc_data"
  NULLIFY(stkrc_data)
  CALL XMLLOADTASK_READ_STKRC(stkrc_data, nlay_stkrc, nlen_stkrc,      &
                              nprevcom_rec,                        &
                              iprevcom_chunk, iprevcom_char,   &
                              n_rec, i_chunk, i_char,                  &
                              l_exclude_ends = .TRUE.)

  CALL STKRC_lentrim(stkrc_data, nlentrim_data, nlen_data)

  IF (nlentrim_data /= 0) THEN
#   ifdef DEBUG_XMLLOADTASKS_PROGRESS
    WRITE(jp_stddbg,'("'//str_infoprefix//'Registering PCDATA into stkmx_work")')
#   endif
                  ! If the data is not blank, add a data-child to the
                  ! curently open element, attach the read
                  ! stkrc_data to it and switch it to PCDATA.
    CALL STKMX_ADD_CHILD(stkmx_work, l_datachild=.TRUE.)
    stkmx_work%lastchild%stkrc_data => stkrc_data
    stkmx_work%lastchild%i_type = 3

#   ifdef DEBUG_XMLLOAD_INFONODES
    CALL STKMX_INFO_NODE(stkmx_work%lastchild, jp_stddbg)
#   endif

  ELSE
#   ifdef DEBUG_XMLLOADTASKS_PROGRESS
    WRITE(jp_stddbg,'("'//str_infoprefix//'Discarding blank PCDATA")')
    WRITE(jp_stddbg,'("'//str_infoprefix//' - nlen_data = ", I0)')     nlen_data
    WRITE(jp_stddbg,'("'//str_infoprefix//' - nlentrim_data = ", I0)') nlentrim_data
#   endif
    IF (ASSOCIATED(stkrc_data)) CALL STKRC_deallocateStkrc(stkrc_data)
  ENDIF


  i_evid = ifound_evid


  SELECT CASE(i_evid)
  CASE(p_evid_commto)
#   ifdef DEBUG_XMLLOADTASKS_PROGRESS
    WRITE(jp_stddbg,'("'//str_infoprefix//'stkxe indicates Comment-tag opening at &
              &Rec.Chunk:Char ", I0,".",I0,":",I0)') n_rec, i_chunk, i_char
#   endif
                  ! Comment tag found: skip to i_evid = p_evid_commtc
    CALL STKXE_SEEK_ONE_FROMCURR(stkxe_work, p_evid_commtc, ifound_evid, &
                                 nprevcom_rec, iprevcom_chunk,  &
                                 iprevcom_char)

    IF (ifound_evid == p_evid_void) THEN
      WRITE(jp_stderr, '("'//str_errprefix//'", A)') &
        'Comment tag not correctly closed -- aborting'
      CALL ABORT_XMLLOAD()
    ENDIF

#   ifdef DEBUG_XMLLOADTASKS_PROGRESS
    WRITE(jp_stddbg,'("'//str_infoprefix//'stkxe indicates Comment-tag closing at &
              &Rec.Chunk:Char ", I0,".",I0,":",I0)') &
              nprevcom_rec, iprevcom_chunk, iprevcom_char
#   endif

  CASE(p_evid_eltsto)
                  ! Element start-tag found:
#   ifdef DEBUG_XMLLOADTASKS_PROGRESS
    WRITE(jp_stddbg,'("'//str_infoprefix//'stkxe indicates Elt start-tag opening at &
              &Rec.Chunk:Char ", I0,".",I0,":",I0)') &
              n_rec, i_chunk, i_char
    WRITE(jp_stddbg,'("'//str_infoprefix//'Calling XMLLOADTASK_PROCESS_STARTTAG to process start-tag")')
#   endif
    n_atts = n_supp
    CALL XMLLOADTASK_PROCESS_STARTTAG(n_atts, nopen_elts)
    nloc_maxdepth = MAX(nloc_maxdepth, nopen_elts)


  CASE(p_evid_elteto)
                  ! Element end-tag found
#   ifdef DEBUG_XMLLOADTASKS_PROGRESS
    WRITE(jp_stddbg,'("'//str_infoprefix//'stkxe indicates Elt end-tag opening at &
              &Rec.Chunk:Char ", I0,".",I0,":",I0)') &
              n_rec, i_chunk, i_char
    WRITE(jp_stddbg,'("'//str_infoprefix//'Calling XMLLOADTASK_PROCESS_ENDTAG to process start-tag")')
#   endif
    CALL XMLLOADTASK_PROCESS_ENDTAG(nopen_elts)


  CASE(p_evid_cdatao)
                  ! CDATA opening tag found
#   ifdef DEBUG_XMLLOADTASKS_PROGRESS
    WRITE(jp_stddbg,'("'//str_infoprefix//'stkxe indicates CDATA tag opening at &
              &Rec.Chunk:Char ", I0,".",I0,":",I0)') &
              n_rec, i_chunk, i_char
    WRITE(jp_stddbg,'("'//str_infoprefix//'Calling XMLLOADTASK_PROCESS_CDATATAG to process start-tag")')
#   endif
    nprevcom_rec   = n_rec
    iprevcom_chunk = i_chunk
    iprevcom_char  = i_char
    CALL XMLLOADTASK_PROCESS_CDATATAG()


  CASE DEFAULT
    nopen_elts = 1
  END SELECT

ENDDO

CLOSE(UNIT=m_unit)

IF (PRESENT(n_maxdepth)) THEN
                  ! Set n_maxdepth to the largest number of
                  ! simultaneously opened elements plus one, to
                  ! allow for possible data it might contain.
  n_maxdepth = nloc_maxdepth + 1
ENDIF

RETURN


!=======================================================================
 CONTAINS
!=======================================================================


!-----------------------------------------------------------------------
 SUBROUTINE ABORT_XMLLOAD
!-----------------------------------------------------------------------

IMPLICIT NONE

CALL ABORT()

!-----------------------------------------------------------------------
 END SUBROUTINE ABORT_XMLLOAD
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE XMLLOADTASK_PROCESS_STARTTAG(n_atts, nopen_elts)
!-----------------------------------------------------------------------

IMPLICIT NONE

! Argument list variables
! -----------------------

INTEGER, INTENT(IN)    :: n_atts
INTEGER, INTENT(INOUT) :: nopen_elts


! Local variables
! ---------------

CHARACTER(LEN=*), PARAMETER &
  :: str_errprefix = '[XMLLOAD/TASK_PROCESS_STARTTAG] Error: '
CHARACTER(LEN=*), PARAMETER &
  :: str_infoprefix = '[XMLLOAD/TASK_PROCESS_STARTTAG] Info: '

INTEGER :: nlay_name, nlay_cntt
INTEGER :: nlen_name, nlen_cntt

TYPE(stack_recchunks), POINTER, SAVE :: stkrc_name
TYPE(stack_recchunks), POINTER, SAVE :: stkrc_cntt

                  ! IDs of possible element start tag closing events
INTEGER, DIMENSION(2), PARAMETER &
  :: p_evidlist_eltstc = (/ p_evid_eltstc, p_evid_eeltstc/)
INTEGER :: ifound_evid
INTEGER :: n_rec, i_chunk, i_char
INTEGER :: i_att
TYPE(stack_recchunks) :: stkrc_eltname
INTEGER :: nlay_stkrc, nlen_stkrc


                  ! Create new element name child except if
                  ! we are currently initializing the root element
IF (nopen_elts /= 0) THEN
  CALL STKMX_ADD_CHILD(stkmx_work, l_datachild=.FALSE.)
                  ! and make the working pointer point to it
  stkmx_work => stkmx_work%lastchild
ENDIF


! ==== Read Name ====

#ifdef DEBUG_XMLLOADTASKS_PROGRESS
WRITE(jp_stddbg,'("'//str_infoprefix//'Calling XMLLOADTASK_READ_NAME to read element name")')
#endif
NULLIFY(stkrc_name)
CALL XMLLOADTASK_READ_NAME(stkrc_name, nlay_name, nlen_name)

stkmx_work%stkrc_eltname => stkrc_name
stkmx_work%nlen_eltname  =   nlen_name


! ==== Read attributes if any ====
IF (n_atts > 0) THEN

! Allocate space in stkmx_work for the attributes
  ALLOCATE(stkmx_work%stkrc_attnames(n_atts))
  ALLOCATE(stkmx_work%nlen_attnames(n_atts))
  ALLOCATE(stkmx_work%stkrc_attcntts(n_atts))
  ALLOCATE(stkmx_work%nlen_attcntts(n_atts))

                  ! Read "n_atts" consecutive attributes
  DO i_att = 1, n_atts

#   ifdef DEBUG_XMLLOADTASKS_PROGRESS
    WRITE(jp_stddbg,'("'//str_infoprefix//'", A, 1X, I0 )') &
      'Processing attribute Nr.', i_att
#   endif

    NULLIFY(stkrc_name)
    NULLIFY(stkrc_cntt)
    CALL XMLLOADTASK_READ_ATT(stkrc_name, nlay_name, nlen_name, &
                              stkrc_cntt, nlay_cntt, nlen_cntt)

    stkmx_work%stkrc_attnames(i_att)%ptr => stkrc_name
    stkmx_work%nlen_attnames(i_att)      =   nlen_name
    stkmx_work%stkrc_attcntts(i_att)%ptr => stkrc_cntt
    stkmx_work%nlen_attcntts(i_att)      =   nlen_cntt

  ENDDO

ENDIF
                  ! Goto i_evid = p_evid_eltstc or p_evid_eeltstc
                  ! (element start-tag closing events): keep the
                  !
CALL STKXE_SEEK_LIST_FROMCURR(stkxe_work, p_evidlist_eltstc, ifound_evid, &
                              nprevcom_rec, iprevcom_chunk,    &
                              iprevcom_char, nprevcom_supp)


SELECT CASE(ifound_evid)
CASE(p_evid_eltstc)
  IF (nopen_elts /= 0) THEN
    ALLOCATE(stksl_eltnames%next)
    stksl_eltnames%next%prev => stksl_eltnames
    stksl_eltnames => stksl_eltnames%next
    NULLIFY(stksl_eltnames%stkrc)
    NULLIFY(stksl_eltnames%next)
  ELSE
    ALLOCATE(stksl_eltnames)
    NULLIFY(stksl_eltnames%stkrc)
    NULLIFY(stksl_eltnames%prev)
    NULLIFY(stksl_eltnames%next)
  ENDIF

# ifdef DEBUG_XMLLOAD_INFOELEMENTS
  IF (nopen_elts == 0) THEN
    WRITE(jp_stddbg,'("'//str_infoprefix//'Root element <")', ADVANCE='NO')
  ELSE
    WRITE(jp_stddbg,'("'//str_infoprefix//'Element <")', ADVANCE='NO')
  ENDIF
  CALL STKRC_writeStkrc(stkmx_work%stkrc_eltname, i_unit=jp_stddbg, &
                        l_pruneateor=.TRUE., l_advance=.FALSE.)
  WRITE(jp_stddbg,'("> started")')
# endif

                ! Copy element name onto the lifo stack
  stksl_eltnames%stkrc => STKRC_createCopyStkrc(stkmx_work%stkrc_eltname)

  nopen_elts = nopen_elts + 1

# ifdef DEBUG_XMLLOAD_INFONODES
  CALL STKMX_INFO_NODE(stkmx_work, jp_stddbg)
# endif

CASE(p_evid_eeltstc)
# ifdef DEBUG_XMLLOAD_INFOELEMENTS
  IF (nopen_elts == 0) THEN
    WRITE(jp_stddbg,'("'//str_infoprefix//'Root element <")', ADVANCE='NO')
  ELSE
    WRITE(jp_stddbg,'("'//str_infoprefix//'Element <")', ADVANCE='NO')
  ENDIF
  CALL STKRC_writeStkrc(stkmx_work%stkrc_eltname, i_unit=jp_stddbg, &
                        l_pruneateor=.TRUE., l_advance=.FALSE.)
  WRITE(jp_stddbg,'("> is empty")')
# endif

# ifdef DEBUG_XMLLOAD_INFONODES
  CALL STKMX_INFO_NODE(stkmx_work, jp_stddbg)
# endif

  IF (nopen_elts > 0) stkmx_work => stkmx_work%parelt


CASE DEFAULT
  WRITE(jp_stderr,'("'//str_errprefix//'", A)') &
    'Element start tag not properly closed -- aborting'
  CALL ABORT_XMLLOAD()

END SELECT



RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE XMLLOADTASK_PROCESS_STARTTAG
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
 SUBROUTINE XMLLOADTASK_PROCESS_ENDTAG(nopen_elts)
!-----------------------------------------------------------------------

IMPLICIT NONE

! Argument list variables
! -----------------------

INTEGER, INTENT(INOUT) :: nopen_elts


! Local variables
! ---------------

CHARACTER(LEN=*), PARAMETER &
  :: str_errprefix = '[XMLLOAD/TASK_PROCESS_ENDTAG] Error: '
CHARACTER(LEN=*), PARAMETER &
  :: str_infoprefix = '[XMLLOAD/TASK_PROCESS_ENDTAG] Info: '

INTEGER :: nlay_name, nlen_name

TYPE(stack_recchunks), POINTER :: stkrc_name


! ==== Read Name ====

#ifdef DEBUG_XMLLOADTASKS_PROGRESS
WRITE(jp_stddbg,'("'//str_infoprefix//'Calling XMLLOADTASK_READ_NAME to read element name")')
#endif
NULLIFY(stkrc_name)
CALL XMLLOADTASK_READ_NAME(stkrc_name, nlay_name, nlen_name)


                  ! Goto i_evid = p_evid_eltetc
                  ! (element end-tag closing events): keep the
                  ! n_rec, i_chunk, i_char and n_supp data available
CALL STKXE_SEEK_ONE_FROMCURR(stkxe_work, p_evid_eltetc, ifound_evid, &
                             nprevcom_rec, iprevcom_chunk,   &
                             iprevcom_char, nprevcom_supp)


IF (ifound_evid == p_evid_eltetc) THEN

  IF (STKRC_STKRC_EQ_STKRC(stkrc_name, stksl_eltnames%stkrc)) THEN

    IF (nopen_elts > 1) THEN

#     ifdef DEBUG_XMLLOAD_INFOELEMENTS
      WRITE(jp_stddbg,'("'//str_infoprefix//'Element <")', ADVANCE='NO')
      CALL STKRC_writeStkrc(stkrc_name, i_unit=jp_stddbg, &
                            l_pruneateor=.TRUE., l_advance=.FALSE.)
      WRITE(jp_stddbg,'("> ended")')
#     endif

                  ! Update the pointer to point to the parent element of
                  ! the just closed element, which now becomes the
                  ! top-level element-

      CALL STKRC_deallocateStkrc(stksl_eltnames%stkrc)
      stksl_eltnames => stksl_eltnames%prev
      DEALLOCATE(stksl_eltnames%next)
      nopen_elts = nopen_elts - 1

      stkmx_work => stkmx_work%parelt

    ELSE

#     ifdef DEBUG_XMLLOAD_INFOELEMENTS
      WRITE(jp_stddbg,'("'//str_infoprefix//'Root element <")', ADVANCE='NO')
      CALL STKRC_writeStkrc(stkrc_name, i_unit=jp_stddbg, &
                            l_pruneateor=.TRUE., l_advance=.FALSE.)
      WRITE(jp_stddbg,'("> ended")')
#     endif

      CALL STKRC_deallocateStkrc(stksl_eltnames%stkrc)
      DEALLOCATE(stksl_eltnames)
      nopen_elts = 0

    ENDIF

  ELSE

    WRITE(jp_stderr,'("'//str_errprefix//'", A)') &
      'Incompatible element end-tag -- aborting'

    WRITE(jp_stderr, '(" - expected ", A)', ADVANCE='NO') '</'
    CALL STKRC_writeStkrc(stksl_eltnames%stkrc, i_unit=jp_stderr, &
                          l_pruneateor=.TRUE., l_advance=.FALSE.)
    WRITE(jp_stderr, '(A)') '>'

    WRITE(jp_stderr, '(" - found ", A)', ADVANCE='NO') '</'
    CALL STKRC_writeStkrc(stkrc_name, i_unit=jp_stderr, &
                          l_pruneateor=.TRUE., l_advance=.FALSE.)
    WRITE(jp_stderr, '(A)') '>'
    CALL ABORT_XMLLOAD()

  ENDIF

ELSE

  WRITE(jp_stderr,'("'//str_errprefix//'", A)') &
    'Element end tag not closed -- aborting'
  CALL ABORT_XMLLOAD()

ENDIF

CALL STKRC_deallocateStkrc(stkrc_name)

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE XMLLOADTASK_PROCESS_ENDTAG
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
 SUBROUTINE XMLLOADTASK_PROCESS_CDATATAG
!-----------------------------------------------------------------------

IMPLICIT NONE

! Argument list variables
! -----------------------

! N/A


! Local variables
! ---------------

CHARACTER(LEN=*), PARAMETER &
  :: str_errprefix = '[XMLLOAD/TASK_PROCESS_CDATATAG] Error: '
CHARACTER(LEN=*), PARAMETER &
  :: str_infoprefix = '[XMLLOAD/TASK_PROCESS_CDATATAG] Info: '

INTEGER :: nlay_name
INTEGER :: nlen_name
INTEGER :: nleft_skip

INTEGER :: i_rec_o, i_chunk_o, i_char_o
INTEGER :: i_rec_c, i_chunk_c, i_char_c
INTEGER :: n_rec, i_chunk, i_char

TYPE(stack_recchunks), POINTER, SAVE :: stkrc_data


                  ! Add new data child to the current element
CALL STKMX_ADD_CHILD(stkmx_work, l_datachild=.TRUE.)


! ==== Read CDATA ====

NULLIFY(stkrc_data)

                  ! Save positions of i_evid = p_evid_cdatao
i_rec_o   = nprevcom_rec
i_chunk_o = iprevcom_chunk
i_char_o  = iprevcom_char


nleft_skip = 8   ! need to shift the i_char_o position by 8 chars

                  ! Forward read in the file until the opening
                  ! record "n_rec_o" and the chunk "i_chunk_o"
DO WHILE ((ncom_rec < i_rec_o) .OR. (icom_chunk < i_chunk_o))
  CALL READ_NEXTCHUNK(m_unit, icom_iostat, strcom_chunk, &
                      icom_size, ncom_rec, icom_chunk)
ENDDO

DO WHILE (nleft_skip > 0)
  IF ((icom_size - i_char_o) >= nleft_skip) THEN
    i_char_o = i_char_o + nleft_skip
    EXIT
  ELSE
    nleft_skip = nleft_skip - (icom_size - i_char_o)
    i_char_o = 0
    CALL READ_NEXTCHUNK(m_unit, icom_iostat, strcom_chunk, &
                        icom_size, ncom_rec, icom_chunk)
    i_rec_o   = ncom_rec
    i_chunk_o = icom_chunk
  ENDIF
ENDDO

                  ! Goto i_evid = p_evid_eltetc
                  ! (element end-tag closing events): keep the
                  ! n_rec, i_chunk, i_char and n_supp data available
CALL STKXE_SEEK_ONE_FROMCURR(stkxe_work, p_evid_cdatac, ifound_evid, &
                             n_rec, i_chunk, i_char, n_supp)

IF (ifound_evid /= p_evid_cdatac) THEN
  WRITE(jp_stderr,'("'//str_errprefix//'", A)') &
    'CDATA tag not properly closed -- aborting'
  CALL ABORT_XMLLOAD()
# ifdef DEBUG_XMLLOADTASKS_PROGRESS
ELSE
  WRITE(jp_stddbg,'("'//str_infoprefix//'stkxe indicates CDATA tag closing at &
              &Rec.Chunk:Char ", I0,".",I0,":",I0)') &
              n_rec, i_chunk, i_char
  WRITE(jp_stddbg,'("'//str_infoprefix//'Calling XMLLOADTASK_PROCESS_CDATATAG to process start-tag")')
# endif
ENDIF

                  ! Rec.Chunk:Char = n_rec.i_chunk:i_char points to the
                  ! ">" in the CDATA closing tag "]]>". We need to find the
                  ! position of the opening "]". "]]>" must be on one record,
                  ! so the n_rec points to the record; however, it may be
                  ! split over two chunks. If this is the case, the preceeding
                  ! chunk must have maximum length
i_rec_c = n_rec
IF (i_char > 2) THEN
  i_chunk_c = i_chunk
  i_char_c = i_char - 2
ELSE
  i_chunk_c = i_chunk - 1
  i_char_c = p_maxlen_chunk + i_char -2
ENDIF

                  ! Read in data into "stkrc_data", link it into
                  ! the new child, and switch the child type from 2
                  ! (PCDATA or CDATA) to 4 (CDATA)
#ifdef DEBUG_XMLLOADTASKS_PROGRESS
WRITE(jp_stddbg,'("'//str_infoprefix//'Calling XMLLOADTASK_PROCESS_STKRC to read CDATA:")')
WRITE(jp_stddbg,'("'//str_infoprefix//'reading up to (excl.) &
              &Rec.Chunk:Char ", I0,".",I0,":",I0)') i_rec_c, i_chunk_c, i_char_c
#endif



NULLIFY(stkrc_data)
CALL XMLLOADTASK_READ_STKRC(stkrc_data, nlay_stkrc, nlen_stkrc, &
                            i_rec_o, i_chunk_o, i_char_o,       &
                            i_rec_c, i_chunk_c, i_char_c,       &
                            l_exclude_ends = .TRUE.)

stkmx_work%lastchild%stkrc_data => stkrc_data
stkmx_work%lastchild%i_type = 4

nleft_skip = 2   ! need to shift the i_char_c position by 2 chars
                 ! (from the first "]" of "]]>" to the ending ">"

DO WHILE (nleft_skip > 0)
  IF ((icom_size - i_char_c) >= nleft_skip) THEN
    i_char_c = i_char_c + nleft_skip
    EXIT
  ELSE
    nleft_skip = nleft_skip - (icom_size - i_char_c)
    i_char_c = 0
    CALL READ_NEXTCHUNK(m_unit, icom_iostat, strcom_chunk, &
                        icom_size, ncom_rec, icom_chunk)
    i_rec_c   = ncom_rec
    i_chunk_c = icom_chunk
  ENDIF
ENDDO

                  ! Transfer the most recent event info to the subroutine-wide vars.
nprevcom_rec   = i_rec_c
iprevcom_chunk = i_chunk_c
iprevcom_char  = i_char_c
nprevcom_supp  = n_supp


#ifdef DEBUG_XMLLOAD_INFONODES
CALL STKMX_INFO_NODE(stkmx_work%lastchild, jp_stddbg)
#endif

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE XMLLOADTASK_PROCESS_CDATATAG
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE XMLLOADTASK_READ_NAME(stkrc_name, nlay_name, nlen_name)
!-----------------------------------------------------------------------

! Read Name from file


IMPLICIT NONE

! Argument list variables
! -----------------------

TYPE(stack_recchunks), POINTER :: stkrc_name
INTEGER, INTENT(OUT)           :: nlay_name
INTEGER, INTENT(OUT)           :: nlen_name



! Local variables
! ---------------

CHARACTER(LEN=*), PARAMETER &
  :: str_errprefix = '[XMLLOAD/XMLLOADTASK_READ_NAME] Error: '
CHARACTER(LEN=*), PARAMETER &
  :: str_warnprefix = '[XMLLOAD/XMLLOADTASK_READ_NAME] Warning: '
CHARACTER(LEN=*), PARAMETER &
  :: str_infoprefix = '[XMLLOAD/XMLLOADTASK_READ_NAME] Info: '

INTEGER :: nlen_check
INTEGER :: n_rec_o, i_chunk_o, i_char_o
INTEGER :: n_rec_c, i_chunk_c, i_char_c
INTEGER :: ifound_evid


                  ! Go to i_evid = p_evid_nameo (name opening)
                  ! in the event stack: get n_rec_o, i_chunk_o, i_char_o
CALL STKXE_SEEK_ONE_FROMCURR(stkxe_work, p_evid_nameo, ifound_evid, &
                            n_rec_o, i_chunk_o, i_char_o)

IF (ifound_evid == p_evid_void) THEN
  WRITE(jp_stderr,'("'//str_errprefix//'", A, " -- aborting")') &
    'Name openng event not found'
  CALL ABORT_XMLLOAD()
ENDIF

#ifdef DEBUG_XMLLOADTASKS_PROGRESS
WRITE(jp_stddbg,'("'//str_infoprefix//'stkxe indicates Name opening at &
              &Rec.Chunk:Char ", I0,".",I0,":",I0)') n_rec_o, i_chunk_o, i_char_o
#endif

                  ! Go to i_evid = p_evid_namec (name closing)
                  ! in the event stack: get n_rec_c, i_chunk_c, i_char_c
                  ! and nlen_check = n_supp
CALL STKXE_SEEK_ONE_FROMCURR(stkxe_work, p_evid_namec, ifound_evid, &
                             n_rec_c, i_chunk_c, i_char_c, nlen_check)

IF (ifound_evid == p_evid_void) THEN
  WRITE(jp_stderr,'("'//str_errprefix//'", A, " -- aborting")') &
    'Name closing event not found'
  CALL ABORT_XMLLOAD()
ENDIF

#ifdef DEBUG_XMLLOADTASKS_PROGRESS
WRITE(jp_stddbg,'("'//str_infoprefix//'stkxe indicates Name closing at &
                 &Rec.Chunk:Char ", I0,".",I0,":",I0)') n_rec_c, i_chunk_c, i_char_c
#endif

#ifdef DEBUG_XMLLOADTASKS_PROGRESS
WRITE(jp_stddbg,'("'//str_infoprefix//'Calling XMLLOADTASK_READ_STKRC to read name")')
#endif
                  ! Retrieve content from file and store it
                  ! into stkrc_name
CALL XMLLOADTASK_READ_STKRC(stkrc_name, nlay_name, nlen_name, &
                            n_rec_o, i_chunk_o, i_char_o,     &
                            n_rec_c, i_chunk_c, i_char_c,     &
                            l_exclude_ends = .FALSE.)

                  ! Issue warning if name lengths do not agree
IF (nlen_name /= nlen_check) THEN
  WRITE(jp_stderr,'("'//str_warnprefix//'", A, " -- continuing")') &
    'Name length mismatch'
  WRITE(jp_stderr,'("  expected:  ", I0)') nlen_check
  WRITE(jp_stderr,'("  retrieved: ", I0)') nlen_name
ENDIF

#ifdef DEBUG_XMLLOADTASKS_PROGRESS
WRITE(jp_stddbg,'("'//str_infoprefix//'Done!")')
#endif

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE XMLLOADTASK_READ_NAME
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE XMLLOADTASK_READ_QSTR(stkrc_qstr, nlay_qstr, nlen_qstr)
!-----------------------------------------------------------------------

! Read Quoted STRing from file


IMPLICIT NONE

! Argument list variables
! -----------------------

TYPE(stack_recchunks), POINTER :: stkrc_qstr
INTEGER, INTENT(OUT)           :: nlay_qstr
INTEGER, INTENT(OUT)           :: nlen_qstr



! Local variables
! ---------------

CHARACTER(LEN=*), PARAMETER &
  :: str_errprefix = '[XMLLOAD/XMLLOADTASK_READ_QSTR] Error: '
CHARACTER(LEN=*), PARAMETER &
  :: str_warnprefix = '[XMLLOAD/XMLLOADTASK_READ_QSTR] Warning: '

INTEGER, DIMENSION(2), PARAMETER &
  :: p_evidlist_stro = (/ p_evid_sqstro, p_evid_dqstro /)

INTEGER :: itarget_evid, ifound_evid

INTEGER :: nlen_check
INTEGER :: n_rec_o, i_chunk_o, i_char_o
INTEGER :: n_rec_c, i_chunk_c, i_char_c


                  ! Go to i_evid = p_evid_sqstro or p_evid_dqstro
                  ! in the event stack: get n_rec_o, i_chunk_o, i_char_o
  CALL STKXE_SEEK_LIST_FROMCURR(stkxe_work, p_evidlist_stro, ifound_evid, &
                                n_rec_o, i_chunk_o, i_char_o)

  SELECT CASE(ifound_evid)
  CASE(p_evid_sqstro)
    itarget_evid = p_evid_sqstrc
  CASE(p_evid_dqstro)
    itarget_evid = p_evid_dqstrc
  CASE(p_evid_void)
    WRITE(jp_stderr,'("'//str_errprefix//'", A, " -- aborting")') &
      'Character string not correctly quoted'
    CALL ABORT_XMLLOAD()
  END SELECT

                  ! Go to i_evid = p_evid_[s,d]qstrc
                  ! in the event stack: get n_rec_c, i_chunk_c, i_char_c,
                  ! and nlen_check = n_supp
  CALL STKXE_SEEK_ONE_FROMCURR(stkxe_work, itarget_evid, ifound_evid,  &
                               n_rec_c, i_chunk_c, i_char_c, nlen_check)

  IF (ifound_evid == p_evid_void) THEN
    WRITE(jp_stderr,'("'//str_errprefix//'", A, " -- aborting")') &
      'Name openng event not found'
    CALL ABORT_XMLLOAD()
  ENDIF


                  ! Read in contents fromt the quoted string
                  ! quotes not included and store into stkrc_qstr
  NULLIFY(stkrc_qstr)
  CALL XMLLOADTASK_READ_STKRC(stkrc_qstr, nlay_qstr, nlen_qstr, &
                              n_rec_o, i_chunk_o, i_char_o,     &
                              n_rec_c, i_chunk_c, i_char_c,     &
                              l_exclude_ends = .TRUE.)

  IF (nlen_qstr /= nlen_check) THEN
    WRITE(jp_stderr,'("'//str_warnprefix//'", A, 1X, I0, " -- continuing")') &
      'Quoted string length mismatch'
    WRITE(jp_stderr,'("  expected:  ", I0)') nlen_check
    WRITE(jp_stderr,'("  retrieved: ", I0)') nlen_qstr
  ENDIF


RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE XMLLOADTASK_READ_QSTR
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
 SUBROUTINE XMLLOADTASK_READ_ATT(stkrc_name, nlay_name, nlen_name, &
                                 stkrc_cntt, nlay_cntt, nlen_cntt)
!-----------------------------------------------------------------------


IMPLICIT NONE

! Argument list variables
! -----------------------

TYPE(stack_recchunks), POINTER :: stkrc_name
INTEGER, INTENT(OUT) :: nlay_name
INTEGER, INTENT(OUT) :: nlen_name
TYPE(stack_recchunks), POINTER :: stkrc_cntt
INTEGER, INTENT(OUT) :: nlay_cntt
INTEGER, INTENT(OUT) :: nlen_cntt


! Local variables
! ---------------

INTEGER :: n_rec, i_chunk, i_char
CHARACTER(LEN=*), PARAMETER &
  :: str_errprefix = '[XMLLOAD/XMLLOADTASK_READ_ATT] Error: '
CHARACTER(LEN=*), PARAMETER &
  :: str_infoprefix = '[XMLLOAD/XMLLOADTASK_READ_ATT] Info: '


                  ! Go to i_evid = p_evid_atto (attribute opening)
                  ! in the event stack
CALL STKXE_SEEK_ONE_FROMCURR(stkxe_work, p_evid_atto, &
                             ifound_evid,             &
                             n_rec, i_chunk, i_char)

IF (ifound_evid == p_evid_void) THEN
  WRITE(jp_stderr,'("'//str_errprefix//'", A, " -- aborting")') &
    'Name opening event not found'
  CALL ABORT_XMLLOAD()
ENDIF

#ifdef DEBUG_XMLLOADTASKS_PROGRESS
WRITE(jp_stddbg,'("'//str_infoprefix//'stkxe indicates Attribute opening at &
              &Rec.Chunk:Char ", I0,".",I0,":",I0)') n_rec, i_chunk, i_char
#endif


#ifdef DEBUG_XMLLOADTASKS_PROGRESS
WRITE(jp_stddbg,'("'//str_infoprefix//'Calling XMLLOADTASK_READ_NAME to read attribute name")')
#endif
CALL XMLLOADTASK_READ_NAME(stkrc_name, nlay_name, nlen_name)

#ifdef DEBUG_XMLLOADTASKS_PROGRESS
WRITE(jp_stddbg,'("'//str_infoprefix//'Calling XMLLOADTASK_READ_QSTR to read attribute content")')
#endif
CALL XMLLOADTASK_READ_QSTR(stkrc_cntt, nlay_cntt, nlen_cntt)


                  ! Goto i_evid = p_evid_attc (attribute closing)
                  ! in the event stack
CALL STKXE_SEEK_ONE_FROMCURR(stkxe_work, p_evid_attc, ifound_evid, &
                             n_rec, i_chunk, i_char)

IF (ifound_evid == p_evid_void) THEN
  WRITE(jp_stderr,'("'//str_errprefix//'", A, " -- aborting")') &
    'Name opening event not found'
  CALL ABORT_XMLLOAD()
ENDIF


#ifdef DEBUG_XMLLOADTASKS_PROGRESS
WRITE(jp_stddbg,'("'//str_infoprefix//'stkxe indicates Attribute closing at &
                 &Rec.Chunk:Char ", I0,".",I0,":",I0)') n_rec, i_chunk, i_char
#endif

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE XMLLOADTASK_READ_ATT
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
 SUBROUTINE XMLLOADTASK_READ_STKRC(stkrc_any, nlay_stkrc, nlen_stkrc,  &
                                   n_rec_o, i_chunk_o, i_char_o,       &
                                   n_rec_c, i_chunk_c, i_char_c,       &
                                   l_exclude_ends)
!-----------------------------------------------------------------------

! Read in data from the XML-file,
! - starting from  Rec.Chunk:Char = n_rec_o.i_chunk_o:i_char_o (opening pos.)
! - and ending at  Rec.Chunk:Char = n_rec_c.i_chunk_c:i_char_c (closing pos.),
! - including those two positions (l_exclude_ends = .TRUE., optional, default)
!   or excluding them (l_exclude_ends = .FALSE., to be stated)

IMPLICIT NONE


! Argument list variables
! -----------------------

TYPE(stack_recchunks), POINTER :: stkrc_any
INTEGER, INTENT(OUT) :: nlay_stkrc
INTEGER, INTENT(OUT) :: nlen_stkrc
INTEGER, INTENT(IN)  :: n_rec_o, i_chunk_o, i_char_o
INTEGER, INTENT(IN)  :: n_rec_c, i_chunk_c, i_char_c
LOGICAL, INTENT(IN), OPTIONAL :: l_exclude_ends


! Local variables
! ---------------

CHARACTER(LEN=*), PARAMETER &
  :: str_errprefix = '[XMLLOAD/XMLLOADTASK_READ_STKRC] Error: '
CHARACTER(LEN=*), PARAMETER &
  :: str_infoprefix = '[XMLLOAD/XMLLOADTASK_READ_STKRC] Info: '

TYPE(stack_recchunks), POINTER :: stkrc_work
TYPE(stack_recchunks), POINTER :: stkrc_discard
LOGICAL :: lloc_exclude_ends
INTEGER :: nlen_chunk
INTEGER :: nlay_discard, nlen_discard


!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
! Operations
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

IF (ASSOCIATED(stkrc_any)) THEN
  WRITE(jp_stderr,'("'//str_errprefix//'", A)') &
    '"stkrc_any" must not be associated -- aborting.'
  CALL ABORT_XMLLOAD()
ELSE
  CALL STKRC_createRoot(stkrc_any, nlay_stkrc, nlen_stkrc)
  stkrc_work => stkrc_any
ENDIF
#ifdef DEBUG_XMLLOADTASKS_PROGRESS
WRITE(jp_stddbg,'("'//str_infoprefix//'After STKRC_createRoot, nlen_stkrc = ",I0)') nlen_stkrc
WRITE(jp_stddbg,'("- nlay_stkrc = ",I0)') nlay_stkrc
WRITE(jp_stddbg,'("- nlen_stkrc = ",I0)') nlen_stkrc
#endif


IF (PRESENT(l_exclude_ends)) THEN
  lloc_exclude_ends = l_exclude_ends
ELSE
  lloc_exclude_ends = .FALSE.
ENDIF

                  ! Forward read in the file until the opening
                  ! record "n_rec_o" and the chunk "i_chunk_o"
#ifdef DEBUG_XMLLOADTASKS_PROGRESS
WRITE(jp_stddbg,'("'//str_infoprefix//'Forwarding to &
              &Rec.Chunk:Char ", I0,".",I0,":",I0)') n_rec_o, i_chunk_o, i_char_o
WRITE(jp_stddbg,'("'//str_infoprefix//'Current read position is &
              &Rec.Chunk ", I0,".",I0)') ncom_rec, icom_chunk
#endif


DO WHILE ((ncom_rec < n_rec_o) .OR. (icom_chunk < i_chunk_o))
  CALL READ_NEXTCHUNK(m_unit, icom_iostat, strcom_chunk, &
                      icom_size, ncom_rec, icom_chunk)

# ifdef DEBUG_XMLLOADTASKS_PROGRESS
  WRITE(jp_stddbg,'("'//str_infoprefix//'Read &
               &Rec.Chunk ", I0,".",I0)') ncom_rec, icom_chunk
# endif

ENDDO

                  ! If the requested set of characters is
                  ! included within a single chunk
IF ((n_rec_o == n_rec_c) .AND. (i_chunk_o == i_chunk_c)) THEN

# ifdef DEBUG_XMLLOADTASKS_PROGRESS
  WRITE(jp_stddbg,'("'//str_infoprefix//'Content in one chunk")')
# endif

  IF (lloc_exclude_ends) THEN

    IF (i_char_c - i_char_o <= 1) THEN
      stkrc_work%str_chunk = ''
      stkrc_work%n_chars = 0
    ELSE
      nlen_chunk = i_char_c - i_char_o - 1
      stkrc_work%str_chunk(1:nlen_chunk) = strcom_chunk(i_char_o+1:i_char_c-1)
      stkrc_work%n_chars = nlen_chunk
    ENDIF
                  ! do not set EOR in this case
    stkrc_work%l_eor = .FALSE.

    nlen_stkrc = stkrc_work%n_chars

    RETURN        ! and we are done


  ELSE

    nlen_chunk = i_char_c - i_char_o + 1
    stkrc_work%str_chunk(1:nlen_chunk) = strcom_chunk(i_char_o:i_char_c)
    stkrc_work%n_chars = nlen_chunk

                  ! set EOR only if i_char_c == i_size and EOR was triggered
    IF ((i_char_c == icom_size) .AND. (icom_iostat == p_iostat_eor)) THEN
      stkrc_work%l_eor = .TRUE.
    ELSE
      stkrc_work%l_eor = .FALSE.
    ENDIF

    nlen_stkrc = stkrc_work%n_chars

    RETURN        ! and we are done

  ENDIF


ELSE

# ifdef DEBUG_XMLLOADTASKS_PROGRESS
  WRITE(jp_stddbg,'("'//str_infoprefix//'Content over multiple chunks")')
# endif

  IF (lloc_exclude_ends) THEN

    IF (i_char_o == icom_size) THEN
      stkrc_work%str_chunk = ''
      stkrc_work%n_chars = 0
    ELSE
      nlen_chunk = icom_size - i_char_o
      stkrc_work%str_chunk(1:nlen_chunk) = strcom_chunk(i_char_o+1:icom_size)
      stkrc_work%n_chars = nlen_chunk
    ENDIF

  ELSE

    nlen_chunk = icom_size - i_char_o + 1
    stkrc_work%str_chunk(1:nlen_chunk) = strcom_chunk(i_char_o:icom_size)
    stkrc_work%n_chars = nlen_chunk

  ENDIF

  IF (icom_iostat == p_iostat_eor) THEN
    stkrc_work%l_eor = .TRUE.
  ELSE
    stkrc_work%l_eor = .FALSE.
  ENDIF

  CALL STKRC_createNext(stkrc_work, nlay_stkrc, nlen_stkrc)
  stkrc_work => stkrc_work%next

# ifdef DEBUG_XMLLOADTASKS_PROGRESS
  WRITE(jp_stddbg,'("'//str_infoprefix//'After STKRC_createNext [1]")')
  WRITE(jp_stddbg,'("- nlay_stkrc = ",I0)') nlay_stkrc
  WRITE(jp_stddbg,'("- nlen_stkrc = ",I0)') nlen_stkrc
# endif

ENDIF


DO

  CALL READ_NEXTCHUNK(m_unit, icom_iostat, strcom_chunk, &
                      icom_size, ncom_rec, icom_chunk)

  IF ((ncom_rec == n_rec_c) .AND. (icom_chunk == i_chunk_c)) THEN

    IF (lloc_exclude_ends) THEN

      IF (i_char_c <= 1) THEN

        stkrc_work => stkrc_work%prev
        CALL STKRC_deallocateStkrc(stkrc_work%next, nlay_discard, nlen_discard)
!~         stkrc_discard => stkrc_work%next
!~         CALL STKRC_deallocateStkrc(stkrc_discard, nlay_discard, nlen_discard)
        nlay_stkrc = nlay_stkrc - nlay_discard
        nlen_stkrc = nlen_stkrc - nlen_discard

      ELSE

        nlen_chunk = i_char_c - 1
        stkrc_work%str_chunk(1:nlen_chunk) = strcom_chunk(1:nlen_chunk)
        stkrc_work%n_chars = nlen_chunk

                  ! in this case, we do not set EOR
        stkrc_work%l_eor = .FALSE.

        nlen_stkrc = nlen_stkrc + nlen_chunk

      ENDIF

      RETURN      ! and we are done

    ELSE

      IF (i_char_c == 0) THEN

!~         WRITE(jp_stderr,'("[XXX] ", I0, 1X, I0)') nlay_stkrc, nlen_stkrc
        stkrc_work => stkrc_work%prev
        CALL STKRC_deallocateStkrc(stkrc_work%next, nlay_discard, nlen_discard)
        nlay_stkrc = nlay_stkrc - nlay_discard
        nlen_stkrc = nlen_stkrc - nlen_discard
!~         WRITE(jp_stderr,'("[XXX] ", I0, 1X, I0)') nlay_stkrc, nlen_stkrc

      ELSE

        nlen_chunk = i_char_c
        stkrc_work%str_chunk(1:nlen_chunk) = strcom_chunk(1:i_char_c)
        stkrc_work%n_chars = nlen_chunk
                    ! To finish, we set EOR only if i_char_c == i_size
                    ! and EOR was triggered
        IF ((i_char_c == icom_size) .AND. (icom_iostat == p_iostat_eor)) THEN
          stkrc_work%l_eor = .TRUE.
        ELSE
          stkrc_work%l_eor = .FALSE.
        ENDIF

        nlen_stkrc = nlen_stkrc + nlen_chunk

      ENDIF


      RETURN      ! and we are done

    ENDIF

  ELSE

    stkrc_work%str_chunk(1:icom_size) = strcom_chunk(1:icom_size)
    stkrc_work%n_chars   = icom_size

    IF (icom_iostat == p_iostat_eor) THEN
      stkrc_work%l_eor = .TRUE.
    ELSE
      stkrc_work%l_eor = .FALSE.
    ENDIF

    CALL STKRC_createNext(stkrc_work, nlay_stkrc, nlen_stkrc)
    stkrc_work => stkrc_work%next

#   ifdef DEBUG_XMLLOADTASKS_PROGRESS
    WRITE(jp_stddbg,'("'//str_infoprefix//'After STKRC_createNext [2]")')
    WRITE(jp_stddbg,'("- nlay_stkrc = ",I0)') nlay_stkrc
    WRITE(jp_stddbg,'("- nlen_stkrc = ",I0)') nlen_stkrc
#   endif

  ENDIF

ENDDO


RETURN


!-----------------------------------------------------------------------
 END SUBROUTINE XMLLOADTASK_READ_STKRC
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
 SUBROUTINE DUMP_LIFOSTACK(stksl_eltnames, i_unit)
!-----------------------------------------------------------------------

! Prints out the contents fo the stack, starting from stksl_eltnames to
! its root

IMPLICIT NONE

TYPE(stkrc_lifo), POINTER :: stksl_eltnames
INTEGER, OPTIONAL         :: i_unit

TYPE(stkrc_lifo), POINTER :: stksl_work


IF (PRESENT(i_unit)) THEN
  IF (.NOT. ASSOCIATED(stksl_eltnames)) THEN
    WRITE(i_unit, '("--- Stack empty ---")')
  ELSE
    WRITE(i_unit, '("---- Stack Top ----")')
    stksl_work => stksl_eltnames
    DO
      WRITE(i_unit, '("<")', ADVANCE='NO')
      CALL STKRC_writeStkrc(stksl_work%stkrc, i_unit=i_unit, &
                            l_pruneateor=.TRUE., l_advance=.FALSE.)
      WRITE(i_unit,'(">")')
      IF (ASSOCIATED(stksl_work%prev)) THEN
        stksl_work => stksl_work%prev
        CYCLE
      ELSE
        WRITE(i_unit, '("--- Stack Bottom --")')
        EXIT
      ENDIF
    ENDDO
  ENDIF
ELSE
  IF (.NOT. ASSOCIATED(stksl_eltnames)) THEN
    WRITE(*, '("--- Stack empty ---")')
  ELSE
    WRITE(*, '("---- Stack Top ----")')
    stksl_work => stksl_eltnames
    DO
      WRITE(*, '("<")', ADVANCE='NO')
      CALL STKRC_writeStkrc(stksl_work%stkrc, l_pruneateor=.TRUE., l_advance=.FALSE.)
      WRITE(*,'(">")')
      IF (ASSOCIATED(stksl_work%prev)) THEN
        stksl_work => stksl_work%prev
        CYCLE
      ELSE
        WRITE(*, '("--- Stack Bottom --")')
        EXIT
      ENDIF
    ENDDO
  ENDIF
ENDIF

RETURN


!-----------------------------------------------------------------------
 END SUBROUTINE DUMP_LIFOSTACK
!-----------------------------------------------------------------------


!=======================================================================
 END FUNCTION XMLLOAD
!=======================================================================


!=======================================================================
 END MODULE MODMXM_STRUCTLOAD
!=======================================================================
