!
!    Copyright 2014, 2015, 2017, 2018, 2020 Guy Munhoven
!
!    This file is part of µXML.
!
!    µXML is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    µXML is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with µXML.  If not, see <https://www.gnu.org/licenses/>.
!


#include <modmxm.h>
!=======================================================================
 MODULE MODMXM_STKXE
!=======================================================================

IMPLICIT NONE

INTEGER, PARAMETER, PRIVATE :: jp_stderr = (MXM_STDERR)
#ifdef DEBUG
INTEGER, PARAMETER, PRIVATE :: jp_stddbg = (MXM_STDDBG)
#endif


INTEGER, PARAMETER :: p_ndescs = 6


INTEGER, PARAMETER :: p_i_rec   = 1  ! record index
INTEGER, PARAMETER :: p_i_chunk = 2  ! index of chunk in the record
INTEGER, PARAMETER :: p_i_char  = 3  ! index of char in the record
INTEGER, PARAMETER :: p_i_evid  = 4  ! event ID
INTEGER, PARAMETER :: p_n_supp  = 5  ! supplementary info (some events only)
                                     !   (n_chars, n_atts)
INTEGER, PARAMETER :: p_n_times = 6  ! nbr of consecutive occurrences
                                     !   on the current record or afterwards
                                     !   (NB: Rec.Chunk:Char only gives the
                                     !   location of the first occurrence)

! Event names
! -----------

INTEGER,          PARAMETER :: p_nlenmax_evname = 31

INTEGER,          PARAMETER :: p_evid_void = -1

CHARACTER(LEN=*), PARAMETER :: p_evname_00    = 'None'
INTEGER,          PARAMETER :: p_evid_none    = 0
CHARACTER(LEN=*), PARAMETER :: p_evname_01    = 'End-of-file (EOF)'
INTEGER,          PARAMETER :: p_evid_eof     = 1
CHARACTER(LEN=*), PARAMETER :: p_evname_02    = 'Description tag opening'
INTEGER,          PARAMETER :: p_evid_descto  = 2
CHARACTER(LEN=*), PARAMETER :: p_evname_03    = 'Description tag closing'
INTEGER,          PARAMETER :: p_evid_desctc  = 3
CHARACTER(LEN=*), PARAMETER :: p_evname_04    = 'Single-Quoted String opening'
INTEGER,          PARAMETER :: p_evid_sqstro  = 4
CHARACTER(LEN=*), PARAMETER :: p_evname_05    = 'Single-Quoted String closing'
INTEGER,          PARAMETER :: p_evid_sqstrc  = 5
CHARACTER(LEN=*), PARAMETER :: p_evname_06    = 'Double-Quoted String opening'
INTEGER,          PARAMETER :: p_evid_dqstro  = 6
CHARACTER(LEN=*), PARAMETER :: p_evname_07    = 'Double-Quoted String closing'
INTEGER,          PARAMETER :: p_evid_dqstrc  = 7
CHARACTER(LEN=*), PARAMETER :: p_evname_10    = 'Comment tag opening'
INTEGER,          PARAMETER :: p_evid_commto  = 10
CHARACTER(LEN=*), PARAMETER :: p_evname_11    = 'Comment tag closing'
INTEGER,          PARAMETER :: p_evid_commtc  = 11
CHARACTER(LEN=*), PARAMETER :: p_evname_12    = 'Element start-tag opening'
INTEGER,          PARAMETER :: p_evid_eltsto  = 12
CHARACTER(LEN=*), PARAMETER :: p_evname_13    = 'Element start-tag closing'
INTEGER,          PARAMETER :: p_evid_eltstc  = 13
CHARACTER(LEN=*), PARAMETER :: p_evname_15    = 'Element start-tag closing empty'
INTEGER,          PARAMETER :: p_evid_eeltstc = 15
CHARACTER(LEN=*), PARAMETER :: p_evname_16    = 'Element end-tag opening'
INTEGER,          PARAMETER :: p_evid_elteto  = 16
CHARACTER(LEN=*), PARAMETER :: p_evname_17    = 'Element end-tag closing'
INTEGER,          PARAMETER :: p_evid_eltetc  = 17
CHARACTER(LEN=*), PARAMETER :: p_evname_18    = '!DOCTYPE tag opening'
INTEGER,          PARAMETER :: p_evid_doctto  = 18
CHARACTER(LEN=*), PARAMETER :: p_evname_19    = '!DOCTYPE tag closing'
INTEGER,          PARAMETER :: p_evid_docttc  = 19
CHARACTER(LEN=*), PARAMETER :: p_evname_20    = '![CDATA[ tag opening'
INTEGER,          PARAMETER :: p_evid_cdatao  = 20
CHARACTER(LEN=*), PARAMETER :: p_evname_21    = '![CDATA[ tag closing'
INTEGER,          PARAMETER :: p_evid_cdatac  = 21
CHARACTER(LEN=*), PARAMETER :: p_evname_22    = 'Name starting'
INTEGER,          PARAMETER :: p_evid_nameo   = 22
CHARACTER(LEN=*), PARAMETER :: p_evname_23    = 'Name ending'
INTEGER,          PARAMETER :: p_evid_namec   = 23
CHARACTER(LEN=*), PARAMETER :: p_evname_24    = 'Attribute starting'
INTEGER,          PARAMETER :: p_evid_atto    = 24
CHARACTER(LEN=*), PARAMETER :: p_evname_25    = 'Attribute ending'
INTEGER,          PARAMETER :: p_evid_attc    = 25

CHARACTER(LEN=*), PARAMETER :: p_evname_51    = 'Keyword "SYSTEM" found'
INTEGER,          PARAMETER :: p_evid_kw_system = 51

INTEGER,          PARAMETER :: p_evid_infoinf = 100

CHARACTER(LEN=*), PARAMETER :: p_evname_101 = 'Info: "<" found'
INTEGER,          PARAMETER :: p_evid_infolt = 101
CHARACTER(LEN=*), PARAMETER :: p_evname_102 = 'Info: "!" found'
INTEGER,          PARAMETER :: p_evid_infoxm = 102
CHARACTER(LEN=*), PARAMETER :: p_evname_103 = 'Info: "?" found'
INTEGER,          PARAMETER :: p_evid_infoqm = 103
CHARACTER(LEN=*), PARAMETER :: p_evname_104 = 'Info: "/" found'
INTEGER,          PARAMETER :: p_evid_infosl = 104
CHARACTER(LEN=*), PARAMETER :: p_evname_105 = 'Info: "--" found'
INTEGER,          PARAMETER :: p_evid_infodd = 105
CHARACTER(LEN=*), PARAMETER :: p_evname_106 = 'Info: "=" found'
INTEGER,          PARAMETER :: p_evid_infoeq = 106
CHARACTER(LEN=*), PARAMETER :: p_evname_107 = 'Info: ">" found'
INTEGER,          PARAMETER :: p_evid_infogt = 107
CHARACTER(LEN=*), PARAMETER :: p_evname_108 = 'Info: EOR found'
INTEGER,          PARAMETER :: p_evid_eor    = 108
CHARACTER(LEN=*), PARAMETER :: p_evname_109 = 'Info:SPC found'
INTEGER,          PARAMETER :: p_evid_spc    = 109



TYPE stack_xmlevents
  INTEGER, DIMENSION(p_ndescs)   :: idesc
  TYPE(stack_xmlevents), POINTER :: prev
  TYPE(stack_xmlevents), POINTER :: next
END TYPE

TYPE stkxe_ptr
  TYPE(stack_xmlevents), POINTER :: ptr
END TYPE

CONTAINS


!-----------------------------------------------------------------------
 FUNCTION EVENT_NAME(i_event) RESULT(str_event)
!-----------------------------------------------------------------------

IMPLICIT NONE

INTEGER, INTENT(IN)             :: i_event
CHARACTER(LEN=p_nlenmax_evname) :: str_event

SELECT CASE(i_event)
CASE( 0)
  str_event = p_evname_00
CASE( 1)
  str_event = p_evname_01
CASE( 2)
  str_event = p_evname_02
CASE( 3)
  str_event = p_evname_03
CASE( 4)
  str_event = p_evname_04
CASE( 5)
  str_event = p_evname_05
CASE( 6)
  str_event = p_evname_06
CASE( 7)
  str_event = p_evname_07
CASE(10)
  str_event = p_evname_10
CASE(11)
  str_event = p_evname_11
CASE(12)
  str_event = p_evname_12
CASE(13)
  str_event = p_evname_13
CASE(15)
  str_event = p_evname_15
CASE(16)
  str_event = p_evname_16
CASE(17)
  str_event = p_evname_17
CASE(18)
  str_event = p_evname_18
CASE(19)
  str_event = p_evname_19
CASE(20)
  str_event = p_evname_20
CASE(21)
  str_event = p_evname_21
CASE(22)
  str_event = p_evname_22
CASE(23)
  str_event = p_evname_23
CASE(24)
  str_event = p_evname_24
CASE(25)
  str_event = p_evname_25
CASE(51)
  str_event = p_evname_51
CASE(101)
  str_event = p_evname_101
CASE(102)
  str_event = p_evname_102
CASE(103)
  str_event = p_evname_103
CASE(104)
  str_event = p_evname_104
CASE(105)
  str_event = p_evname_105
CASE(106)
  str_event = p_evname_106
CASE(107)
  str_event = p_evname_107
CASE(108)
  str_event = p_evname_108
CASE(109)
  str_event = p_evname_109
CASE DEFAULT
  str_event = ''
  WRITE(str_event,'("Unknown event ", I0)') i_event
END SELECT

RETURN

!-----------------------------------------------------------------------
 END FUNCTION EVENT_NAME
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE CREATE_EVENTSTACK(stkxe_events)
!-----------------------------------------------------------------------

IMPLICIT NONE

! Argument list variables
! -----------------------

TYPE(stack_xmlevents), POINTER :: stkxe_events

IF (ASSOCIATED(stkxe_events)) THEN
  WRITE(jp_stderr, '("[CREATE_EVENTSTACK] Error: &
                      &event stack already allocated -- aborting")')
  CALL ABORT()
ENDIF


ALLOCATE(stkxe_events)
NULLIFY(stkxe_events%prev)
NULLIFY(stkxe_events%next)
stkxe_events%idesc(p_i_rec)   = -1
stkxe_events%idesc(p_i_chunk) = -1
stkxe_events%idesc(p_i_char)  = -1
stkxe_events%idesc(p_i_evid)  =  p_evid_none
stkxe_events%idesc(p_n_supp)  = -1
stkxe_events%idesc(p_n_times) =  0


RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE CREATE_EVENTSTACK
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE EXTEND_EVENTSTACK(stkxe_events)
!-----------------------------------------------------------------------

IMPLICIT NONE

! Argument list variables
! -----------------------

TYPE(stack_xmlevents), POINTER :: stkxe_events

IF (.NOT. ASSOCIATED(stkxe_events)) THEN
  WRITE(jp_stderr, '("[CREATE_EVENTSTACK] Error: &
                    &event stack not yet allocated -- aborting")')
  CALL ABORT()
ELSEIF (ASSOCIATED(stkxe_events%next)) THEN
  WRITE(jp_stderr, '("[CREATE_EVENTSTACK] Error: &
                    &%next already allocated -- aborting")')
  CALL ABORT()
ENDIF


ALLOCATE(stkxe_events%next)
stkxe_events%next%prev => stkxe_events
NULLIFY(stkxe_events%next%next)
stkxe_events => stkxe_events%next

stkxe_events%idesc(p_i_rec)   = -1
stkxe_events%idesc(p_i_chunk) = -1
stkxe_events%idesc(p_i_char)  = -1
stkxe_events%idesc(p_i_evid)  = p_evid_none
stkxe_events%idesc(p_n_supp)  = -1
stkxe_events%idesc(p_n_times) =  0


RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE EXTEND_EVENTSTACK
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
 SUBROUTINE REGISTER_EVENT(i_rec, i_chunk, i_char,                     &
                           i_evid, n_events, stkxe_events, n_supp)
!-----------------------------------------------------------------------

IMPLICIT NONE

! Argument list variables
! -----------------------

INTEGER, INTENT(IN)         :: i_rec
INTEGER, INTENT(IN)         :: i_chunk
INTEGER, INTENT(IN)         :: i_char
INTEGER, INTENT(IN)         :: i_evid
INTEGER, INTENT(INOUT)      :: n_events
TYPE(stack_xmlevents), POINTER :: stkxe_events
INTEGER, INTENT(IN)         :: n_supp

OPTIONAL n_supp


! Local variables
! ---------------

INTEGER, DIMENSION(p_ndescs) :: idesc
INTEGER :: i_times
CHARACTER(LEN=*), PARAMETER &
  :: str_fmt0 = '(" At Rec.Chunk:Char ", I0,".",I0,":",I0, ": &
                  &event Nr. ", I0, ", EvID=", I0)'
CHARACTER(LEN=*), PARAMETER &
  :: str_fmt1 = '(" At Rec.Chunk:Char ", I0,".",I0,":",I0, ": &
                  &event Nr. ", I0, ", EvID=", I0, &
                  &" with n_supp=", I0)'

INTEGER :: n_supp_internal

IF (.NOT. ASSOCIATED(stkxe_events)) THEN
  WRITE(jp_stderr, '("[REGISTER_EVENT] Error: event stack not allocated -- aborting")')
  CALL ABORT()
ENDIF


#ifdef REGISTER_EVENT_IGNOREINFO
IF (i_evid > p_evid_infoinf) RETURN
#endif

                  ! If n_supp is not present, use a default value of -1
IF (PRESENT(n_supp)) THEN
  n_supp_internal = n_supp
ELSE
  n_supp_internal = 0
ENDIF

n_events = n_events + 1

#ifdef DEBUG_REGISTER_EVENT
!WRITE(jp_stddbg, '(5I6, " - ", 2I6)')  i_rec, i_chunk, i_char, &
!                                    n_events, i_evid, n_supp_internal, 1
!#else
IF (n_supp_internal == 0) THEN
  WRITE(jp_stddbg, str_fmt0)  i_rec, i_chunk, i_char, &
                               n_events, i_evid
ELSE
  WRITE(jp_stddbg, str_fmt1)  i_rec, i_chunk, i_char, &
                               n_events, i_evid, n_supp_internal
ENDIF
#endif

#ifdef REGISTER_EVENT_COMPACT
IF (.NOT. ASSOCIATED(stkxe_events%prev)) THEN

# ifdef DEBUG_REGISTER_EVENT
  WRITE(jp_stddbg, '("----> ", 4I6, 1X,L1,1X, 2I6)')  1 , -1, -1, -1, &
    ASSOCIATED(stkxe_events%prev), -1, 0
# else
  CONTINUE
# endif

ELSE

  i_times = stkxe_events%prev%idesc(p_n_times)

  IF (      (stkxe_events%prev%idesc(p_i_evid) == i_evid) &
       .AND.(stkxe_events%prev%idesc(p_n_supp) ==      0) ) THEN

    stkxe_events%prev%idesc(p_n_times) = i_times + 1

#   ifdef DEBUG_REGISTER_EVENT
    WRITE(jp_stddbg, '("----> ", 4I6, 1X,L1,1X, 3I6)') &
      stkxe_events%prev%idesc(p_i_rec),    &
      stkxe_events%prev%idesc(p_i_chunk),  &
      stkxe_events%prev%idesc(p_i_char),   &
      stkxe_events%prev%idesc(p_i_evid),   &
      ASSOCIATED(stkxe_events%prev),       &
      stkxe_events%prev%idesc(p_n_supp),   &
      i_times, i_times + 1
#   endif

    RETURN

  ENDIF

# ifdef DEBUG_REGISTER_EVENT
  WRITE(jp_stddbg, '("----> ", 4I6, 1X,L1,1X, 3I6)') &
    stkxe_events%prev%idesc(p_i_rec),  &
    stkxe_events%prev%idesc(p_i_chunk),&
    stkxe_events%prev%idesc(p_i_char), &
    stkxe_events%prev%idesc(p_i_evid), &
    ASSOCIATED(stkxe_events%prev),     &
    stkxe_events%prev%idesc(p_n_supp), &
    i_times, i_times
# endif

ENDIF
#endif


idesc( (/ p_i_rec, p_i_chunk, p_i_char, p_i_evid, p_n_supp,          p_n_times /) ) &
 =     (/   i_rec,   i_chunk,   i_char,   i_evid,   n_supp_internal,         1 /)

#ifdef DEBUG_REGISTER_EVENT
WRITE(jp_stddbg, '(6X, 4I6, 1X,L1,1X, 21I6)')  &
  idesc(p_i_rec),                   &
  idesc(p_i_chunk),                 &
  idesc(p_i_char),                  &
  idesc(p_i_evid),                  &
  ASSOCIATED(stkxe_events%prev),    &
  idesc(p_n_supp),                  &
  idesc(p_n_times)
#endif

stkxe_events%idesc(1:p_ndescs) = idesc(1:p_ndescs)

#ifdef DEBUG_REGISTER_EVENT
WRITE(jp_stddbg, '(6X, 4I6, 1X,L1,1X, 2I6)') &
  stkxe_events%idesc(p_i_rec),        &
  stkxe_events%idesc(p_i_chunk),      &
  stkxe_events%idesc(p_i_char),       &
  stkxe_events%idesc(p_i_evid),       &
  ASSOCIATED(stkxe_events%prev),      &
  stkxe_events%idesc(p_n_supp),       &
  stkxe_events%idesc(p_n_times)
#endif


CALL EXTEND_EVENTSTACK(stkxe_events)


RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE REGISTER_EVENT
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
 SUBROUTINE DUMP_EVENTSTACK(stkxe_events, i_unit)
!-----------------------------------------------------------------------

IMPLICIT NONE

! Argument list variables
! -----------------------

TYPE(stack_xmlevents), POINTER :: stkxe_events
INTEGER, INTENT(IN)         :: i_unit


! Local variables
! ---------------

TYPE(stack_xmlevents), POINTER :: stkxe_work
INTEGER :: n_times
INTEGER :: n_supp
INTEGER :: i_evid

CHARACTER(LEN=*), PARAMETER &
  :: str_fmt0  = '("Rec.Chunk:Char ", I0,".",I0,":", I0, &
                  &" -- Event ", I0, " (", A, ")")'
CHARACTER(LEN=*), PARAMETER &
  :: str_fmt1  = '("Rec.Chunk:Char ", I0,".",I0,":", I0, &
                  &" -- Event ", I0, " (", A, ")", &
                  &", n_chars=", I0)'
CHARACTER(LEN=*), PARAMETER &
  :: str_fmt2  = '("Rec.Chunk:Char ", I0,".",I0,":", I0, &
                  &" -- Event ", I0, " (", A, ")", &
                  &", n_atts=", I0)'
CHARACTER(LEN=*), PARAMETER &
  :: str_fmt3  = '("Rec.Chunk:Char ", I0,".",I0,":", I0, &
                  &" -- Event ", I0, " (", A, ")", &
                  &", n_atts=0")'
CHARACTER(LEN=*), PARAMETER &
  :: str_fmto  = '("Rec.Chunk:Char ", I0,".",I0,":", I0, &
                  &" -- Event ", I0, " (", A, "), ", I0, " occurrences")'

CHARACTER(LEN=LEN(str_fmt1)) :: str_fmt

! End of declarations
! -------------------


NULLIFY(stkxe_work)

IF (ASSOCIATED(stkxe_events)) THEN

  n_times = stkxe_events%idesc(p_n_times)
  n_supp = stkxe_events%idesc(p_n_supp)
  i_evid = stkxe_events%idesc(p_i_evid)

  SELECT CASE(i_evid)
  CASE(p_evid_eltsto)
    IF (n_supp /= 0) THEN
      str_fmt = str_fmt2
    ELSE
      str_fmt = str_fmt3
    ENDIF
  CASE(p_evid_sqstrc, p_evid_dqstrc, p_evid_namec)
      str_fmt = str_fmt1
  CASE DEFAULT
    str_fmt = str_fmt0
  END SELECT

  SELECT CASE(n_times)
  CASE(0)
    WRITE(i_unit, '("Event-stack empty")')
    RETURN

  CASE(1)
    SELECT CASE(n_supp)
    CASE(:0)
      WRITE(i_unit, str_fmt)                          &
        stkxe_events%idesc(p_i_rec),                  &
        stkxe_events%idesc(p_i_chunk),                &
        stkxe_events%idesc(p_i_char),                 &
        stkxe_events%idesc(p_i_evid),                 &
        TRIM(EVENT_NAME(stkxe_events%idesc(p_i_evid)))
    CASE(1:)
      WRITE(i_unit, str_fmt)                          &
        stkxe_events%idesc(p_i_rec),                  &
        stkxe_events%idesc(p_i_chunk),                &
        stkxe_events%idesc(p_i_char),                 &
        stkxe_events%idesc(p_i_evid),                 &
        TRIM(EVENT_NAME(stkxe_events%idesc(p_i_evid))), n_supp
    END SELECT
  CASE (2:)
    WRITE(i_unit, str_fmto)                           &
      stkxe_events%idesc(p_i_rec),                    &
      stkxe_events%idesc(p_i_chunk),                  &
      stkxe_events%idesc(p_i_char),                   &
      stkxe_events%idesc(p_i_evid),                   &
      TRIM(EVENT_NAME(stkxe_events%idesc(p_i_evid))), n_times

  CASE DEFAULT
    WRITE(i_unit, '("Meaningless data (n_times<0) in the stack -- Returning")')
    RETURN

  END SELECT

  stkxe_work => stkxe_events

  DO WHILE(ASSOCIATED(stkxe_work%next))

    stkxe_work => stkxe_work%next

    n_times = stkxe_work%idesc(p_n_times)
    n_supp = stkxe_work%idesc(p_n_supp)
    i_evid = stkxe_work%idesc(p_i_evid)

    SELECT CASE(i_evid)
    CASE(p_evid_eltsto)
      IF (n_supp /= 0) THEN
        str_fmt = str_fmt2
      ELSE
        str_fmt = str_fmt3
      ENDIF
    CASE(p_evid_sqstrc, p_evid_dqstrc, p_evid_namec)
      str_fmt = str_fmt1
    CASE DEFAULT
      str_fmt = str_fmt0
    END SELECT

    SELECT CASE(n_times)
    CASE(0)
      EXIT

    CASE(1)
      SELECT CASE(n_supp)
      CASE(:0)
        WRITE(i_unit, str_fmt)                         &
          stkxe_work%idesc(p_i_rec),                   &
          stkxe_work%idesc(p_i_chunk),                 &
          stkxe_work%idesc(p_i_char),                  &
          stkxe_work%idesc(p_i_evid),                  &
          TRIM(EVENT_NAME(stkxe_work%idesc(p_i_evid)))

      CASE(1:)
        WRITE(i_unit, str_fmt)                         &
          stkxe_work%idesc(p_i_rec),                   &
          stkxe_work%idesc(p_i_chunk),                 &
          stkxe_work%idesc(p_i_char),                  &
          stkxe_work%idesc(p_i_evid),                  &
          TRIM(EVENT_NAME(stkxe_work%idesc(p_i_evid))), n_supp
      END SELECT

    CASE DEFAULT
      WRITE(i_unit, str_fmto)                          &
        stkxe_work%idesc(p_i_rec),                     &
        stkxe_work%idesc(p_i_chunk),                   &
        stkxe_work%idesc(p_i_char),                    &
        stkxe_work%idesc(p_i_evid),                    &
        TRIM(EVENT_NAME(stkxe_work%idesc(p_i_evid))), n_times

    END SELECT

  END DO

ELSE

  WRITE(i_unit, '("Event-stack NULL")')

ENDIF

NULLIFY(stkxe_work)

RETURN

!-----------------------------------------------------------------------
END SUBROUTINE DUMP_EVENTSTACK
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
SUBROUTINE STAT_EVENTSTACK(stkxe_events, n_elts, nmax_elt_atts,            &
                           n_maxlen_eltname,                           &
                           n_maxlen_attname, n_maxlen_attcont)
!-----------------------------------------------------------------------

IMPLICIT NONE

! Argument list variables
! -----------------------

TYPE(stack_xmlevents), POINTER :: stkxe_events
INTEGER, INTENT(OUT)        :: n_elts
INTEGER, INTENT(OUT)        :: nmax_elt_atts
INTEGER, INTENT(OUT)        :: n_maxlen_eltname
INTEGER, INTENT(OUT)        :: n_maxlen_attname
INTEGER, INTENT(OUT)        :: n_maxlen_attcont


! Local variables
! ---------------

TYPE(stack_xmlevents), POINTER :: stkxe_work
TYPE(stack_xmlevents), POINTER :: stkxe_descto
TYPE(stack_xmlevents), POINTER :: stkxe_last_eltsto

LOGICAL :: l_desc
LOGICAL :: l_eltname
LOGICAL :: l_attname
LOGICAL :: l_attcont

INTEGER :: i_evid

INTEGER :: n_atts
INTEGER :: n_len_eltname
INTEGER :: n_len_attname
INTEGER :: n_len_attcont


! End of declarations
! -------------------

#ifdef DEBUG_STAT_EVENTSTACK
WRITE(jp_stddbg, '("[STAT_EVENTSTACK]: start")')
#endif


n_maxlen_eltname = 0
n_maxlen_attname = 0
n_maxlen_attcont = 0

n_elts = 0
nmax_elt_atts = 0

IF (.NOT. ASSOCIATED(stkxe_events)) RETURN

l_desc    = .FALSE.
l_eltname = .FALSE.
l_attname = .FALSE.
l_attcont = .FALSE.

stkxe_work => stkxe_events
NULLIFY(stkxe_descto)
NULLIFY(stkxe_last_eltsto)

DO WHILE(ASSOCIATED(stkxe_work))

  i_evid = stkxe_work%idesc(p_i_evid)

  SELECT CASE(i_evid)

  CASE(p_evid_descto)               ! "Description tag opening"
#   ifdef DEBUG_STAT_EVENTSTACK
    WRITE(jp_stddbg, '(" Description tag opening: reset n_atts")')
#   endif
    stkxe_descto => stkxe_work
    l_desc = .TRUE.
    n_atts = 0

  CASE(p_evid_desctc)               ! "Description tag closing"
#   ifdef DEBUG_STAT_EVENTSTACK
    WRITE(jp_stddbg, '(" Description tag closing")')
#   endif
    l_desc = .FALSE.
    !stkxe_descto%idesc(p_n_supp) = n_atts
    NULLIFY(stkxe_descto)

  CASE(p_evid_eltsto)               ! "Element start-tag opening"
#   ifdef DEBUG_STAT_EVENTSTACK
    WRITE(jp_stddbg, '(" Element start-tag opening: reset n_atts")')
#   endif
    stkxe_last_eltsto => stkxe_work
    l_eltname = .TRUE.
    n_atts = 0

  CASE(p_evid_elteto)               ! "Element end-tag opening"
#   ifdef DEBUG_STAT_EVENTSTACK
    WRITE(jp_stddbg, '(" Element end-tag opening")')
#   endif
    l_eltname = .TRUE.

  CASE(p_evid_eltstc, p_evid_eeltstc)       ! "Element start-tag closing" or
    n_elts = n_elts + 1                     ! "Element start-tag closing empty"
#   ifdef DEBUG_STAT_EVENTSTACK
    WRITE(jp_stddbg, '(" Element start-tag closing (empty): update nmax_elt_atts")')
#   endif
    l_eltname = .FALSE.
    nmax_elt_atts = MAX(n_atts, nmax_elt_atts)
    stkxe_last_eltsto%idesc(p_n_supp) = n_atts
    NULLIFY(stkxe_last_eltsto)

  CASE(p_evid_eltetc)               ! "Element end-tag closing"
#   ifdef DEBUG_STAT_EVENTSTACK
    WRITE(jp_stddbg, '(" Element end-tag closing")')
#   endif
    l_eltname = .FALSE.

  CASE(p_evid_atto)                 ! "Attribute starting"
#   ifdef DEBUG_STAT_EVENTSTACK
    WRITE(jp_stddbg, '("  Attribute starting")')
#   endif
    l_attname = .TRUE.
    l_attcont = .TRUE.

  CASE(p_evid_attc)                 ! "Attribute ending"
#   ifdef DEBUG_STAT_EVENTSTACK
    WRITE(jp_stddbg, '("  Attribute ending: increment n_atts")')
#   endif
    n_atts = n_atts + 1
    l_attname = .FALSE.
    l_attcont = .FALSE.

# ifdef DEBUG_STAT_EVENTSTACK
  CASE(p_evid_nameo)                ! "Name starting"
    IF (l_eltname) THEN
      WRITE(jp_stddbg, '("  Element name starting")')
    ELSEIF (l_attname) THEN
      WRITE(jp_stddbg, '("   Attribute name starting")')
    ELSEIF (l_desc) THEN
      WRITE(jp_stddbg, '("  Descriptor name starting")')
    ENDIF
# endif

  CASE(p_evid_namec)                ! "Name ending"
    IF (l_eltname) THEN
#     ifdef DEBUG_STAT_EVENTSTACK
      WRITE(jp_stddbg, '("  Element name ending: update n_maxlen_eltname")')
#     endif
      n_len_eltname = stkxe_work%idesc(p_n_supp)
      n_maxlen_eltname = MAX(n_maxlen_eltname, n_len_eltname)
      l_eltname = .FALSE.
    ELSEIF (l_attname) THEN
      IF (.NOT. l_desc) THEN
#       ifdef DEBUG_STAT_EVENTSTACK
        WRITE(jp_stddbg, '("   Attribute name ending: update n_maxlen_attname")')
#       endif
        n_len_attname = stkxe_work%idesc(p_n_supp)
        n_maxlen_attname = MAX(n_len_attname, n_maxlen_attname)
#     ifdef DEBUG_STAT_EVENTSTACK
      ELSE
        WRITE(jp_stddbg, '("   Attribute name ending: not updating n_maxlen_attname")')
#     endif
      ENDIF
      l_attname = .FALSE.
#   ifdef DEBUG_STAT_EVENTSTACK
    ELSEIF (l_desc) THEN
      WRITE(jp_stddbg, '("  Descriptor name ending")')
#   endif
    ENDIF

  CASE(p_evid_dqstrc, p_evid_sqstrc)        ! "Double-Quoted String closing" or
    IF (l_attcont) THEN                     ! "Single-Quoted String closing"
      IF (.NOT. l_desc) THEN
#       ifdef DEBUG_STAT_EVENTSTACK
        WRITE(jp_stddbg, '("   Attribute content ending: update n_maxlen_attcont")')
#       endif
        n_len_attcont = stkxe_work%idesc(p_n_supp)
        n_maxlen_attcont = MAX(n_len_attcont, n_maxlen_attcont)
#     ifdef DEBUG_STAT_EVENTSTACK
      ELSE
        WRITE(jp_stddbg, '("   Attribute content ending: not updating n_maxlen_attcont")')
#     endif
      ENDIF
      l_attcont = .FALSE.
    ENDIF

# ifdef DEBUG_STAT_EVENTSTACK
  CASE(p_evid_dqstro, p_evid_sqstro)        ! "Double-Quoted String opening" or
    IF (l_attcont) THEN                     ! "Single-Quoted String opening"
      WRITE(jp_stddbg, '("   Attribute content starting")')
    ENDIF
# endif

  END SELECT

  stkxe_work => stkxe_work%next

ENDDO


NULLIFY(stkxe_work)

#   ifdef DEBUG_STAT_EVENTSTACK
    WRITE(jp_stddbg, '("[STAT_EVENTSTACK]: end")')
#   endif

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE STAT_EVENTSTACK
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
SUBROUTINE STKXE_SEEK_NONINFO_AFTERCURR(stkxe_start, ifound_evid,      &
                                        i_rec, i_chunk, i_char, n_supp)
!-----------------------------------------------------------------------

IMPLICIT NONE

! Argument list variables
! -----------------------

TYPE(stack_xmlevents), POINTER :: stkxe_start
INTEGER, INTENT(OUT)           :: ifound_evid
INTEGER, INTENT(OUT), OPTIONAL :: i_rec
INTEGER, INTENT(OUT), OPTIONAL :: i_chunk
INTEGER, INTENT(OUT), OPTIONAL :: i_char
INTEGER, INTENT(OUT), OPTIONAL :: n_supp


! Local  variables
! ----------------

CHARACTER(LEN=*), PARAMETER &
  :: str_errprefix = '[MODMXM_EVENTS/STKXE_SEEK_NONINFO_AFTERCURR] Error: '

TYPE(stack_xmlevents), POINTER :: stkxe_work
INTEGER :: i_evid


IF (.NOT. ASSOCIATED(stkxe_start)) THEN
  WRITE(jp_stderr,'("'//str_errprefix//'", A)') &
    '"stkxe_start" not associated -- aborting'
  CALL ABORT()
ENDIF

stkxe_work => stkxe_start

DO

  IF (ASSOCIATED(stkxe_work%next)) THEN

    stkxe_work => stkxe_work%next
    i_evid = stkxe_work%idesc(p_i_evid)

    IF ((i_evid < p_evid_infoinf) .AND. (i_evid /= p_evid_eor)) EXIT

  ELSE

    ifound_evid = p_evid_void
    IF (PRESENT(i_rec))   i_rec   = -1
    IF (PRESENT(i_chunk)) i_chunk = -1
    IF (PRESENT(i_char))  i_char  = -1
    IF (PRESENT(n_supp))  n_supp  = -1

    RETURN

  ENDIF

ENDDO

ifound_evid = i_evid
IF (PRESENT(i_rec))   i_rec   = stkxe_work%idesc(p_i_rec)
IF (PRESENT(i_chunk)) i_chunk = stkxe_work%idesc(p_i_chunk)
IF (PRESENT(i_char))  i_char  = stkxe_work%idesc(p_i_char)
IF (PRESENT(n_supp))  n_supp  = stkxe_work%idesc(p_n_supp)

                  ! Re-point the initial pointer to the
                  ! most recent position
stkxe_start => stkxe_work

NULLIFY(stkxe_work)


RETURN

!-----------------------------------------------------------------------
END SUBROUTINE STKXE_SEEK_NONINFO_AFTERCURR
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
SUBROUTINE STKXE_SEEK_ONE_FROMCURR(stkxe_start, itarget_evid,          &
                                   ifound_evid,                        &
                                   i_rec, i_chunk, i_char, n_supp)
!-----------------------------------------------------------------------

IMPLICIT NONE

! Argument list variables
! -----------------------

TYPE(stack_xmlevents), POINTER :: stkxe_start
INTEGER, INTENT(IN)            :: itarget_evid
INTEGER, INTENT(OUT)           :: ifound_evid
INTEGER, INTENT(OUT), OPTIONAL :: i_rec
INTEGER, INTENT(OUT), OPTIONAL :: i_chunk
INTEGER, INTENT(OUT), OPTIONAL :: i_char
INTEGER, INTENT(OUT), OPTIONAL :: n_supp


! Local  variables
! ----------------

CHARACTER(LEN=*), PARAMETER &
  :: str_errprefix = '[MODMXM_EVENTS/STKXE_SEEK_ONE_FROMCURR] Error: '

TYPE(stack_xmlevents), POINTER :: stkxe_work
INTEGER :: i_evid


IF (.NOT. ASSOCIATED(stkxe_start)) THEN
  WRITE(jp_stderr,'("'//str_errprefix//'", A)') &
    '"stkxe_start" not associated -- aborting'
  CALL ABORT()
ENDIF

stkxe_work => stkxe_start

i_evid = stkxe_work%idesc(p_i_evid)

DO WHILE (i_evid /= itarget_evid)
  IF (ASSOCIATED(stkxe_work%next)) THEN
    stkxe_work => stkxe_work%next
    i_evid = stkxe_work%idesc(p_i_evid)
    CYCLE
  ELSE
    ifound_evid = p_evid_void
    IF (PRESENT(i_rec))   i_rec   = -1
    IF (PRESENT(i_chunk)) i_chunk = -1
    IF (PRESENT(i_char))  i_char  = -1
    IF (PRESENT(n_supp))  n_supp  = -1
    RETURN
  ENDIF
ENDDO

ifound_evid = i_evid
IF (PRESENT(i_rec))   i_rec   = stkxe_work%idesc(p_i_rec)
IF (PRESENT(i_chunk)) i_chunk = stkxe_work%idesc(p_i_chunk)
IF (PRESENT(i_char))  i_char  = stkxe_work%idesc(p_i_char)
IF (PRESENT(n_supp))  n_supp  = stkxe_work%idesc(p_n_supp)

stkxe_start => stkxe_work

NULLIFY(stkxe_work)


RETURN

!-----------------------------------------------------------------------
END SUBROUTINE STKXE_SEEK_ONE_FROMCURR
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
SUBROUTINE STKXE_SEEK_LIST_FROMCURR(stkxe_start, ilisttarget_evids,    &
                                    ifound_evid,                       &
                                    i_rec, i_chunk, i_char, n_supp)
!-----------------------------------------------------------------------

IMPLICIT NONE

! Argument list variables
! -----------------------

TYPE(stack_xmlevents), POINTER    :: stkxe_start
INTEGER, DIMENSION(:), INTENT(IN) :: ilisttarget_evids
INTEGER, INTENT(OUT)              :: ifound_evid
INTEGER, INTENT(OUT), OPTIONAL    :: i_rec
INTEGER, INTENT(OUT), OPTIONAL    :: i_chunk
INTEGER, INTENT(OUT), OPTIONAL    :: i_char
INTEGER, INTENT(OUT), OPTIONAL    :: n_supp


! Local  variables
! ----------------

CHARACTER(LEN=*), PARAMETER &
  :: str_errprefix = '[MODMXM_EVENTS/STKXE_SEEK_LIST_FROMCURR] Error: '

TYPE(stack_xmlevents), POINTER :: stkxe_work
INTEGER :: i_evid


IF (.NOT. ASSOCIATED(stkxe_start)) THEN
  WRITE(jp_stderr,'("'//str_errprefix//'", A)') &
    '"stkxe_start" not associated -- aborting'
  CALL ABORT()
ENDIF

stkxe_work => stkxe_start

i_evid = stkxe_work%idesc(p_i_evid)

DO WHILE (.NOT. ANY(ilisttarget_evids == i_evid))
  IF (ASSOCIATED(stkxe_work%next)) THEN
    stkxe_work => stkxe_work%next
    i_evid = stkxe_work%idesc(p_i_evid)
    CYCLE
  ELSE
    ifound_evid = p_evid_void
    IF (PRESENT(i_rec))   i_rec   = -1
    IF (PRESENT(i_chunk)) i_chunk = -1
    IF (PRESENT(i_char))  i_char  = -1
    IF (PRESENT(n_supp))  n_supp = -1
    RETURN
  ENDIF
ENDDO

ifound_evid = i_evid
IF (PRESENT(i_rec))   i_rec   = stkxe_work%idesc(p_i_rec)
IF (PRESENT(i_chunk)) i_chunk = stkxe_work%idesc(p_i_chunk)
IF (PRESENT(i_char))  i_char  = stkxe_work%idesc(p_i_char)
IF (PRESENT(n_supp))  n_supp  = stkxe_work%idesc(p_n_supp)

stkxe_start => stkxe_work

NULLIFY(stkxe_work)


RETURN

!-----------------------------------------------------------------------
END SUBROUTINE STKXE_SEEK_LIST_FROMCURR
!-----------------------------------------------------------------------

!=======================================================================
 END MODULE MODMXM_STKXE
!=======================================================================
