!
!    Copyright 2014, 2015, 2017, 2018, 2020 Guy Munhoven
!
!    This file is part of µXML.
!
!    µXML is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    µXML is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with µXML.  If not, see <https://www.gnu.org/licenses/>.
!


#include <modmxm.h>
!=======================================================================
 MODULE MODMXM_STKMX
!=======================================================================

USE MODMXM_STKRC

IMPLICIT NONE

INTEGER, PARAMETER, PRIVATE :: jp_stderr = (MXM_STDERR)
#ifdef DEBUG
INTEGER, PARAMETER, PRIVATE :: jp_stddbg = (MXM_STDDBG)
#endif


! Type definitions
! ----------------

TYPE stack_minixml
  INTEGER                          :: i_type
                  ! %i_type
                  !   = -1: undetermined
                  !   =  0: root element, name and attributes
                  !   =  1: element name with attributes
                  !   =  2: PCDATA or CDATA of an element
                  !         (implies that %n_children=0,
                  !         %str_eltname => NULL, %str_attname => NULL,
                  !         %mxe_child_a => NULL, %mxe_child_z => NULL)
                  !   =  3: PCDATA
                  !   =  4: CDATA

  INTEGER                          :: n_children
                  ! %n_children: only for elements

  INTEGER                          :: n_order
                  ! %n_order: order of the information
                  !  = 0: root element name and its attributes
                  !  = 1: sub-elements of the root element or
                  !       data in the root element
                  !  = 2: sub-elements of order-1-elements

  INTEGER, DIMENSION(:), POINTER   :: i_chain
                  ! %i_chain: integer array indicating the information sequence
                  ! RANK(i_chain) = %n_order

  TYPE(stack_recchunks), POINTER   :: stkrc_eltname
  INTEGER                          :: nlen_eltname
                  ! %stkrc_eltname: element name (only for element nodes)
                  ! %nlen_eltname: element name length (only for element nodes)

  TYPE(stkrc_ptr), &
    DIMENSION(:), ALLOCATABLE      :: stkrc_attnames
  INTEGER, &
    DIMENSION(:), ALLOCATABLE      :: nlen_attnames
                  ! %stkrc_attnames: attribute names (only for element nodes,
                  !   only allocated if any attributes present)
                  ! %nlen_attnames: attribute name lengths (only for element
                  !   nodes only allocated if any attributes present)

  TYPE(stkrc_ptr), &
    DIMENSION(:), ALLOCATABLE      :: stkrc_attcntts
  INTEGER, &
    DIMENSION(:), ALLOCATABLE      :: nlen_attcntts
                  ! %stkrc_attnames: attribute contents (only for element nodes,
                  !   only allocated if any attributes present)
                  ! %nlen_attnames: attribute content lengths (only for element
                  !   nodes only allocated if any attributes present)

  TYPE(stack_recchunks), POINTER   :: stkrc_data
  INTEGER                          :: nlen_data
                  ! %stkrc_data: data content (only for data nodes)
                  ! %nlen_data: data content length (only for data nodes)

  TYPE(stack_minixml), POINTER     :: parelt    ! pointer to parent element

  TYPE(stack_minixml), POINTER     :: prevsib   ! pointer to previous sibling
  TYPE(stack_minixml), POINTER     :: nextsib   ! pointer to next sibling
  TYPE(stack_minixml), POINTER     :: frstchild ! pointer to first child
  TYPE(stack_minixml), POINTER     :: lastchild ! pointer to last child
END TYPE


                  ! Encapsulate POINTER to TYPE(stack_minixml),
                  ! in order to use arrays of pointers
TYPE stkmx_ptr
  TYPE(stack_minixml), POINTER :: ptr
END TYPE

TYPE stkmx_ptrlist
  TYPE(stack_minixml), POINTER :: stkmx
  TYPE(stkmx_ptrlist), POINTER :: next
END TYPE



!=======================================================================
 CONTAINS
!=======================================================================


!-----------------------------------------------------------------------
 SUBROUTINE STKMX_CREATE_ROOT(stkmx_root)
!-----------------------------------------------------------------------

IMPLICIT NONE

! Argument list variables
! -----------------------

TYPE(stack_minixml), POINTER ::  stkmx_root


!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
! Operations
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

IF (ASSOCIATED(stkmx_root)) THEN
  WRITE(jp_stderr,'("[STKMX_CREATE_ROOT] Error: ", A)') &
    'stkmx_root already ASSOCIATED -- aborting'
  CALL ABORT()
ENDIF

ALLOCATE(stkmx_root)

                  ! initialise:
                  !  - i_type ==> root element name
                  !  - no children
                  !  - order 0
stkmx_root%i_type     = 0
stkmx_root%n_children = 0
stkmx_root%n_order    = 0

                  ! The root of a minixml stack always has
                  ! the type element-name
NULLIFY(stkmx_root%stkrc_eltname)
CALL STKRC_createRoot(stkmx_root%stkrc_eltname)
stkmx_root%nlen_eltname = 0

                  ! Must not be filled here
NULLIFY(stkmx_root%stkrc_data)

                  ! Will never have any parent
NULLIFY(stkmx_root%parelt)

                  ! Will never have siblings
NULLIFY(stkmx_root%prevsib)
NULLIFY(stkmx_root%nextsib)
                  ! May later on have children (except if empty)
NULLIFY(stkmx_root%frstchild)
NULLIFY(stkmx_root%lastchild)

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE STKMX_CREATE_ROOT
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
 SUBROUTINE STKMX_ADD_CHILD(stkmx_parelt, l_datachild)
!-----------------------------------------------------------------------

IMPLICIT NONE

! Argument list variables
! -----------------------

TYPE(stack_minixml), POINTER ::  stkmx_parelt
LOGICAL, INTENT(IN), OPTIONAL :: l_datachild


! Local variables
! ---------------

TYPE(stack_minixml), POINTER, SAVE   :: stkmx_child
INTEGER                        :: new_n_order
LOGICAL                        :: lloc_datachild


!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
! Operations
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

IF (.NOT. ASSOCIATED(stkmx_parelt)) THEN
  WRITE(jp_stderr,'("[STKMX_ADD_CHILD] Error: ", A)') &
    'stkmx_parelt not ASSOCIATED -- aborting'
  CALL ABORT()
ENDIF

                 ! By default (l_datachild not present in the call),
                 ! add an element-name child (not a data child)
IF (PRESENT(l_datachild)) THEN
  lloc_datachild = l_datachild
ELSE
  lloc_datachild = .FALSE.
ENDIF

new_n_order = stkmx_parelt%n_order + 1

ALLOCATE(stkmx_child)

! initialise
IF (lloc_datachild) THEN
                  ! Data-type node for PCDATA or CDATA
                  ! to be revised to 3 for PCDATA or 4 for CDATA,
  stkmx_child%i_type   = 2
ELSE
                  ! Element-name and attribute information node
  stkmx_child%i_type   = 1
ENDIF
stkmx_child%n_children =  0
stkmx_child%n_order    =  new_n_order

NULLIFY(stkmx_child%stkrc_eltname)
IF (.NOT. lloc_datachild) THEN
  CALL STKRC_createRoot(stkmx_child%stkrc_eltname)
ENDIF
stkmx_child%nlen_eltname = 0

! stkrc_attnames -- to be allocated later
! nlen_attnames -- to be allocated later
! stkrc_attcntts -- to be allocated later
! nlen_attcntts -- to be allocated later

NULLIFY(stkmx_child%stkrc_data)
!~ IF (lloc_datachild) THEN
!~   CALL STKRC_createRoot(stkmx_child%stkrc_data)
!~ ENDIF

                  ! Set the parent of the child
stkmx_child%parelt => stkmx_parelt
                  ! New child has not yet any children
NULLIFY(stkmx_child%frstchild)
NULLIFY(stkmx_child%lastchild)
                  ! New child has not yet any "younger" sibling
NULLIFY(stkmx_child%nextsib)

IF (.NOT. ASSOCIATED(stkmx_parelt%frstchild)) THEN
                  ! No children so far for the parent:

                  !  - there is no "elder" (%prevsib) sibling
  NULLIFY(stkmx_child%prevsib)

                  !  - the new child is the first and thus
                  !    also the last child for the parent;
  stkmx_parelt%frstchild => stkmx_child
  stkmx_parelt%lastchild => stkmx_child
  stkmx_parelt%n_children = 1

ELSE
                  ! Parent has already at least one child

                  !  - link the new child's "youngest elder"
                  !    sibling (%prevsib) to the previously last
                  !    child of the parent
  stkmx_child%prevsib => stkmx_parelt%lastchild

                  !  - link the parent's last child's %nextsib
                  !    to the new child
  stkmx_parelt%lastchild%nextsib => stkmx_child
                  !  - now update the parent's last child to the new one
  stkmx_parelt%lastchild => stkmx_child
                  !  - update the parent's number of children
  stkmx_parelt%n_children = stkmx_parelt%n_children + 1

ENDIF

ALLOCATE(stkmx_child%i_chain(new_n_order))

IF (new_n_order > 1) THEN
  stkmx_child%i_chain(1:new_n_order-1) = stkmx_parelt%i_chain
ENDIF
stkmx_child%i_chain(new_n_order) = stkmx_parelt%n_children

RETURN 


!-----------------------------------------------------------------------
 END SUBROUTINE STKMX_ADD_CHILD
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE STKMX_DEALLOCATE(stkmx_any)
!-----------------------------------------------------------------------

IMPLICIT NONE

! Argument list variables
! -----------------------

TYPE(stack_minixml), POINTER   :: stkmx_any


! Local variables
! ---------------

TYPE(stack_minixml), POINTER   :: stkmx_work
TYPE(stack_minixml), POINTER   :: stkmx_saveparelt
INTEGER :: i, n_atts
TYPE(stack_recchunks), POINTER, SAVE :: stkrc_i


!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
! Operations
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

IF (.NOT.ASSOCIATED(stkmx_any)) THEN
  WRITE(jp_stderr,'("[STKMX_DEALLOCATE] Warning: ", A)') &
    'stkmx_any not ASSOCIATED -- returning'
  RETURN
ENDIF

stkmx_work => stkmx_any

NULLIFY(stkrc_i)

                  ! Save the pointer to the possible parent
                  ! element of stkmx_any, and nullify stkmx_any%parelt
IF (ASSOCIATED(stkmx_any%parelt)) THEN
  stkmx_saveparelt => stkmx_any%parelt
  NULLIFY(stkmx_any%parelt)

  IF (.NOT.ASSOCIATED(stkmx_any%prevsib)) THEN
                  ! stkmx_any is the first child
    IF (.NOT. ASSOCIATED(stkmx_any%nextsib)) THEN
                  ! stkmx_any is actually the only child
      NULLIFY(stkmx_saveparelt%frstchild)
      NULLIFY(stkmx_saveparelt%lastchild)
      stkmx_saveparelt%n_children = 0
    ELSE
      stkmx_saveparelt%frstchild => stkmx_any%nextsib
      NULLIFY(stkmx_any%nextsib%prevsib)
      NULLIFY(stkmx_any%nextsib)
      stkmx_saveparelt%n_children = stkmx_saveparelt%n_children - 1
    ENDIF
  ELSE
                  ! stkmx_any is not the first child
    IF (.NOT. ASSOCIATED(stkmx_any%nextsib)) THEN
                  ! stkmx_any is the last child
      stkmx_saveparelt%lastchild => stkmx_any%prevsib
      NULLIFY(stkmx_any%prevsib%nextsib)
      NULLIFY(stkmx_any%prevsib)
      stkmx_saveparelt%n_children = stkmx_saveparelt%n_children - 1
    ELSE
                  ! stkmx_any is neither the first nor the last child
      stkmx_any%prevsib%nextsib => stkmx_any%nextsib
      stkmx_any%nextsib%prevsib => stkmx_any%prevsib
      NULLIFY(stkmx_any%prevsib)
      NULLIFY(stkmx_any%nextsib)
      stkmx_saveparelt%n_children = stkmx_saveparelt%n_children - 1
    ENDIF
  ENDIF
ELSE
                  ! We are discarding a root element, which does not have siblings
  NULLIFY(stkmx_saveparelt)
ENDIF

                  ! Unlink stkmx_any from its siblings as well


DO WHILE (ASSOCIATED(stkmx_work%frstchild))
  stkmx_work => stkmx_work%frstchild
ENDDO

                  ! stkmx_work now points to an element that has no
                  ! children. We may clean it up
DO

  IF (ASSOCIATED(stkmx_work%i_chain)) DEALLOCATE(stkmx_work%i_chain)

  n_atts = SIZE(stkmx_work%nlen_attnames)

  CALL STKRC_deallocateStkrc(stkmx_work%stkrc_eltname)

  DO i = 1, n_atts
    stkrc_i => stkmx_work%stkrc_attnames(i)%ptr
    CALL STKRC_deallocateStkrc(stkrc_i)
    NULLIFY(stkmx_work%stkrc_attnames(i)%ptr)
    stkrc_i => stkmx_work%stkrc_attcntts(i)%ptr
    CALL STKRC_deallocateStkrc(stkrc_i)
    NULLIFY(stkmx_work%stkrc_attcntts(i)%ptr)
    NULLIFY(stkrc_i)
  ENDDO

  DEALLOCATE(stkmx_work%nlen_attnames)
  DEALLOCATE(stkmx_work%nlen_attcntts)

  CALL STKRC_deallocateStkrc(stkmx_work%stkrc_data)


                  ! If the parent element of stkmx_work is not NULL
                  ! (i.e., if we have not yet retruned to stkmx_any)
                  ! then check for siblings
  IF (ASSOCIATED(stkmx_work%parelt)) THEN
  
                  ! If there are siblings, they must also be deallocated
    stkmx_work%parelt%n_children = stkmx_work%parelt%n_children - 1
  
    IF (ASSOCIATED(stkmx_work%nextsib)) THEN
      stkmx_work => stkmx_work%nextsib
      DEALLOCATE(stkmx_work%prevsib)
      NULLIFY(stkmx_work%prevsib)
      stkmx_work%parelt%frstchild => stkmx_work
  
                  ! Move to the deepest possible first child
      DO WHILE (ASSOCIATED(stkmx_work%frstchild))
        stkmx_work => stkmx_work%frstchild
      ENDDO
  
    ELSE
                  ! No siblings left (stkmx_work%parelt%n_children should be 0)
                  ! we return to the parent
      stkmx_work => stkmx_work%parelt
      DEALLOCATE(stkmx_work%frstchild)
      NULLIFY(stkmx_work%frstchild)
      NULLIFY(stkmx_work%lastchild)

    ENDIF

                    ! and discard the current work
    CYCLE

  ELSE
                    ! The parent element is not associated
                    ! all the children should have been discard, and
                    ! the content of the root element as well
    EXIT
  
  ENDIF

ENDDO

DEALLOCATE(stkmx_any)
NULLIFY(stkmx_any)
NULLIFY(stkmx_work)
NULLIFY(stkmx_saveparelt)

RETURN


!-----------------------------------------------------------------------
 END SUBROUTINE STKMX_DEALLOCATE
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE STKMX_INFO_NODE(stkmx_node, i_unit)
!-----------------------------------------------------------------------

IMPLICIT NONE

! Argument list variables
! -----------------------

TYPE(stack_minixml), POINTER ::  stkmx_node
INTEGER, INTENT(IN), OPTIONAL :: i_unit


! Local variables
! ---------------
CHARACTER(LEN=255) str_fmt

!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
! Operations
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

IF (PRESENT(i_unit)) THEN

  WRITE(i_unit,'("[STKMX_INFO_NODE]")')
  WRITE(i_unit,'(" - %i_type = ", I0)') stkmx_node%i_type
!~   IF ((stkmx_node%i_type == 0) .OR. (stkmx_node%i_type == 1)) THEN
  WRITE(i_unit,'(" - %n_order = ", I0)') stkmx_node%n_order
  IF (stkmx_node%n_order > 1) THEN
    WRITE(str_fmt, '(A,I0,A)') &
      '(" - %i_chain = (",', stkmx_node%n_order-1, '(I0, ","), I0, ")")'
  ELSE
    WRITE(str_fmt, '(A)') '(" - %i_chain = (", I0, ")")'
  ENDIF
  IF (stkmx_node%n_order > 0) WRITE(i_unit,str_fmt) stkmx_node%i_chain
!~   WRITE(i_unit, '(" - ASSOCIATED(%parelt): ", L1)') ASSOCIATED(stkmx_node%parelt)
!~   WRITE(i_unit, '(" - ASSOCIATED(%prevsib): ", L1)') ASSOCIATED(stkmx_node%prevsib)
!~   WRITE(i_unit, '(" - ASSOCIATED(%nextsib): ", L1)') ASSOCIATED(stkmx_node%nextsib)
!~   WRITE(i_unit, '(" - ASSOCIATED(%frstchild): ", L1)') ASSOCIATED(stkmx_node%frstchild)
!~   WRITE(i_unit, '(" - ASSOCIATED(%lastchild): ", L1)') ASSOCIATED(stkmx_node%lastchild)

ELSE

  WRITE(*,'("[STKMX_INFO_NODE]")')
  WRITE(*,'(" - %i_type = ", I0)') stkmx_node%i_type
!~   IF ((stkmx_node%i_type == 0) .OR. (stkmx_node%i_type == 1)) THEN
  WRITE(*,'(" - %n_order = ", I0)') stkmx_node%n_order
  IF (stkmx_node%n_order > 1) THEN
    WRITE(str_fmt, '(A,I0,A)') &
      '(" - %i_chain = (",', stkmx_node%n_order-1, '(I0, ","), I0, ")")'
  ELSE
    WRITE(str_fmt, '(A)') '(" - %i_chain = (", I0, ")")'
  ENDIF
  IF (stkmx_node%n_order > 0) WRITE(*,str_fmt) stkmx_node%i_chain
!~   WRITE(*, '(" - ASSOCIATED(%parelt): ", L1)') ASSOCIATED(stkmx_node%parelt)
!~   WRITE(*, '(" - ASSOCIATED(%prevsib): ", L1)') ASSOCIATED(stkmx_node%prevsib)
!~   WRITE(*, '(" - ASSOCIATED(%nextsib): ", L1)') ASSOCIATED(stkmx_node%nextsib)
!~   WRITE(*, '(" - ASSOCIATED(%frstchild): ", L1)') ASSOCIATED(stkmx_node%frstchild)
!~   WRITE(*, '(" - ASSOCIATED(%lastchild): ", L1)') ASSOCIATED(stkmx_node%lastchild)

ENDIF


RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE STKMX_INFO_NODE
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
FUNCTION STKMX_getRootElement(stkmx_any) &
RESULT(stkmx_rootelt)
!-----------------------------------------------------------------------

! Returns a pointers to the root element node of <stkmx_any>
! Returns a NULL pointer if
! 1. <stkmx_any> is not associated

IMPLICIT NONE

! Argument list variables
! -----------------------

TYPE(stack_minixml), POINTER :: stkmx_any
TYPE(stack_minixml), POINTER :: stkmx_rootelt


! Pre-set the RESULT to NULL
NULLIFY(stkmx_rootelt)

IF (.NOT. ASSOCIATED(stkmx_any)) RETURN

stkmx_rootelt => stkmx_any

DO WHILE (ASSOCIATED(stkmx_rootelt%parelt))
  stkmx_rootelt => stkmx_rootelt%parelt
ENDDO

IF (stkmx_rootelt%i_type /= 0) THEN
  WRITE(jp_stderr,'("[STKMX_getRootElement] Error: ", A)') &
    'stkmx_any does not have a legal root element (%i_type /= 0) -- aborting'
  CALL ABORT()  
ENDIF

RETURN

!-----------------------------------------------------------------------
END FUNCTION STKMX_getRootElement
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
FUNCTION STKMX_getElementNodeByName(stkmx_any, str_eltname) &
RESULT(stkmx_nodearr)
!-----------------------------------------------------------------------

! Returns an array of Pointers to the child nodes of <stkmx_any>
! that are
! 1. element nodes
! 2. have the name <str_name>
! Returns a NULL pointer if
! 1. no children
! 2. no children with that name

IMPLICIT NONE

! Argument list variables
! -----------------------

TYPE(stack_minixml), POINTER :: stkmx_any
CHARACTER(LEN=*), INTENT(IN) :: str_eltname

TYPE(stkmx_ptr), DIMENSION(:), POINTER ::  stkmx_nodearr

! Local variables
! ---------------

TYPE(stack_minixml), POINTER :: stkmx_child
TYPE(stkmx_ptr), DIMENSION(:), POINTER ::  stkmx_nodearrtmp
INTEGER :: n_children, i_child, n_dimres
TYPE(stack_recchunks), POINTER :: stkrc_childname



! Pre-set the RESULT to NULL
NULLIFY(stkmx_nodearr)

IF (.NOT. ASSOCIATED(stkmx_any)) THEN
  WRITE(jp_stderr,'("[STKMX_getElementNodeByName] Error: ", A)') &
    'stkmx_any not ASSOCIATED -- aborting'
  CALL ABORT()
ENDIF

                 ! If str_name is empty, there is nothing to look for
IF (LEN_TRIM(str_eltname) == 0) RETURN

                 ! Get the number of children
n_children = stkmx_any%n_children

                 ! If stkmx_any does not have any children, there is
                 ! nothing to look for
IF (stkmx_any%n_children == 0) RETURN


                 ! There are children: pointer to the first one
stkmx_child => stkmx_any%frstchild

! Provisionally ALLOCATE stkmx_nodearrtmp with <n_children> elements
ALLOCATE(stkmx_nodearrtmp(n_children))


n_dimres = 0
DO i_child = 1, n_children

  IF (i_child > 1) stkmx_child => stkmx_child%nextsib
  IF (stkmx_child%i_type /= 1) CYCLE

  stkrc_childname => stkmx_child%stkrc_eltname
  IF (STKRC_STKRC_EQ_STR(stkrc_childname, str_eltname)) THEN
    n_dimres = n_dimres+1
    stkmx_nodearrtmp(n_dimres)%ptr => stkmx_child
  ENDIF

ENDDO

IF (n_dimres /= 0) THEN
  ALLOCATE(stkmx_nodearr(n_dimres))
  DO i_child = 1, n_dimres
    stkmx_nodearr(i_child)%ptr => stkmx_nodearrtmp(i_child)%ptr
  ENDDO
ENDIF

DEALLOCATE(stkmx_nodearrtmp)


RETURN

!-----------------------------------------------------------------------
END FUNCTION STKMX_getElementNodeByName
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
FUNCTION STKMX_getUniqueChildEltByName(stkmx_any, str_eltname)        &
RESULT(stkmx_resnode)
!-----------------------------------------------------------------------

! Returns a pointer to the child node of <stkmx_any> that has
! exactly one child element node named <str_eltname>
! Returns a NULL pointer if
! 1. <stkmx_any> does not have no children
! 2. or does no have children named <str_eltname>
! Triggers an error if there is more than one child element named <str_eltname>

IMPLICIT NONE

! Argument list variables
! -----------------------

TYPE(stack_minixml), POINTER   :: stkmx_any
CHARACTER(LEN=*), INTENT(IN)   :: str_eltname

TYPE(stack_minixml), POINTER   :: stkmx_resnode


! Local variables
! ---------------

TYPE(stack_minixml), POINTER   :: stkmx_child
INTEGER                        :: n_children, i_child, n_hits
TYPE(stack_recchunks), POINTER :: stkrc_childname
LOGICAL                        :: l_foundone


! Pre-set the RESULT to NULL
NULLIFY(stkmx_resnode)
l_foundone = .FALSE.

IF (.NOT. ASSOCIATED(stkmx_any)) THEN
  WRITE(jp_stderr,'("[STKMX_getUniqueChildNodeByName] Error: ", A)') &
    'stkmx_any not ASSOCIATED -- aborting'
  CALL ABORT()
ENDIF

                 ! If str_name is empty, there is nothing to look for
IF (LEN_TRIM(str_eltname) == 0) RETURN

                 ! Get the number of children
n_children = stkmx_any%n_children

                 ! If stkmx_any does not have any children, there is
                 ! nothing to look for
IF (stkmx_any%n_children == 0) RETURN


                 ! There are children: pointer to the first one
stkmx_child => stkmx_any%frstchild


n_hits = 0
DO i_child = 1, n_children

  IF (i_child > 1) stkmx_child => stkmx_child%nextsib
  IF (stkmx_child%i_type /= 1) CYCLE

  stkrc_childname => stkmx_child%stkrc_eltname
  IF (STKRC_STKRC_EQ_STR(stkrc_childname, str_eltname)) THEN
    n_hits = n_hits+1
    IF (.NOT. l_foundone) THEN
      stkmx_resnode => stkmx_child
      l_foundone = .TRUE.
    ENDIF
  ENDIF

ENDDO

SELECT CASE(n_hits)
CASE(0,1)
  RETURN
CASE DEFAULT
  ! Found more than one
  WRITE(jp_stderr,'("[STKMX_getUniqueChildNodeByName] Error: ", A)') &
    'found more than one child element named <' // TRIM(str_eltname) //'> -- aborting'
  CALL ABORT()
END SELECT

RETURN

!-----------------------------------------------------------------------
END FUNCTION STKMX_getUniqueChildEltByName
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
FUNCTION STKMX_getChildElementNodes(stkmx_any) RESULT(stkmx_nodearr)
!-----------------------------------------------------------------------

! Returns an array of pointers to the child nodes of <stkmx_any>
! that are element nodes
! Returns a NULL pointer if <stkmx_any> does not have any children


IMPLICIT NONE

! Argument list variables
! -----------------------

TYPE(stack_minixml), POINTER :: stkmx_any

TYPE(stkmx_ptr), DIMENSION(:), POINTER ::  stkmx_nodearr

! Local variables
! ---------------

TYPE(stack_minixml), POINTER :: stkmx_child
TYPE(stkmx_ptr), DIMENSION(:), POINTER ::  stkmx_nodearrtmp
INTEGER :: n_children, i_child, n_dimres
TYPE(stack_recchunks), POINTER :: stkrc_childname



! Pre-set the RESULT to NULL
NULLIFY(stkmx_nodearr)

IF (.NOT. ASSOCIATED(stkmx_any)) THEN
  WRITE(jp_stderr,'("[STKMX_getChildElementNodes] Error: ", A)') &
    'stkmx_any not ASSOCIATED -- aborting'
  CALL ABORT()
ENDIF


                 ! Get the number of child nodes
n_children = stkmx_any%n_children

                 ! If stkmx_any does not have any children, there is
                 ! nothing to look for
IF (stkmx_any%n_children == 0) RETURN


                 ! There are children: pointer to the first one
stkmx_child => stkmx_any%frstchild

! Provisionally ALLOCATE stkmx_nodearrtmp with <n_children> elements
ALLOCATE(stkmx_nodearrtmp(n_children))


n_dimres = 0
DO i_child = 1, n_children

  IF (i_child > 1) stkmx_child => stkmx_child%nextsib
  IF (stkmx_child%i_type /= 1) CYCLE

  n_dimres = n_dimres+1
  stkmx_nodearrtmp(n_dimres)%ptr => stkmx_child

ENDDO

IF (n_dimres /= 0) THEN
  ALLOCATE(stkmx_nodearr(n_dimres))
  DO i_child = 1, n_dimres
    stkmx_nodearr(i_child)%ptr => stkmx_nodearrtmp(i_child)%ptr
  ENDDO
ENDIF

DEALLOCATE(stkmx_nodearrtmp)


RETURN

!-----------------------------------------------------------------------
END FUNCTION STKMX_getChildElementNodes
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
FUNCTION STKMX_getPCDatacntt(stkmx_any) RESULT(stkrc_datacntt)
!-----------------------------------------------------------------------

! Returns an array of pointers to the child nodes of <stkmx_any>
! that are data nodes (with PCDATA)
! Returns a NULL pointer if <stkmx_any> does not have any children


IMPLICIT NONE

! Argument list variables
! -----------------------

TYPE(stack_minixml), POINTER :: stkmx_any

TYPE(stkrc_ptr), DIMENSION(:), POINTER   :: stkrc_datacntt


! Local variables
! ---------------

TYPE(stack_minixml), POINTER :: stkmx_child
TYPE(stkmx_ptr), DIMENSION(:), POINTER ::  stkmx_nodearrtmp
INTEGER :: n_children, i_child, n_dimres
TYPE(stack_recchunks), POINTER :: stkrc_childname



! Pre-set the RESULT to NULL
NULLIFY(stkrc_datacntt)

IF (.NOT. ASSOCIATED(stkmx_any)) THEN
  WRITE(jp_stderr,'("[STKMX_getPCDatacntt] Error: ", A)') &
    'stkmx_any not ASSOCIATED -- aborting'
  CALL ABORT()
ENDIF


                 ! Get the number of child nodes
n_children = stkmx_any%n_children

                 ! If stkmx_any does not have any children, there is
                 ! nothing to look for
IF (stkmx_any%n_children == 0) RETURN


                 ! There are children: pointer to the first one
stkmx_child => stkmx_any%frstchild

! Provisionally ALLOCATE stkmx_nodearrtmp with <n_children> elements
ALLOCATE(stkmx_nodearrtmp(n_children))


n_dimres = 0
DO i_child = 1, n_children

  IF (i_child > 1) stkmx_child => stkmx_child%nextsib
  IF (stkmx_child%i_type /= 3) CYCLE

  n_dimres = n_dimres+1
  stkmx_nodearrtmp(n_dimres)%ptr => stkmx_child

ENDDO

IF (n_dimres /= 0) THEN
  ALLOCATE(stkrc_datacntt(n_dimres))
  DO i_child = 1, n_dimres
    stkrc_datacntt(i_child)%ptr => stkmx_nodearrtmp(i_child)%ptr%stkrc_data
  ENDDO
ENDIF

DEALLOCATE(stkmx_nodearrtmp)


RETURN

!-----------------------------------------------------------------------
END FUNCTION STKMX_getPCDatacntt
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
FUNCTION STKMX_getCDatacntt(stkmx_any) RESULT(stkrc_datacntt)
!-----------------------------------------------------------------------

! Returns an array of pointers to the child nodes of <stkmx_any>
! that are data nodes (with CDATA)
! Returns a NULL pointer if <stkmx_any> does not have any children


IMPLICIT NONE

! Argument list variables
! -----------------------

TYPE(stack_minixml), POINTER :: stkmx_any

TYPE(stkrc_ptr), DIMENSION(:), POINTER   :: stkrc_datacntt


! Local variables
! ---------------

TYPE(stack_minixml), POINTER :: stkmx_child
TYPE(stkmx_ptr), DIMENSION(:), POINTER ::  stkmx_nodearrtmp
INTEGER :: n_children, i_child, n_dimres
TYPE(stack_recchunks), POINTER :: stkrc_childname



! Pre-set the RESULT to NULL
NULLIFY(stkrc_datacntt)

IF (.NOT. ASSOCIATED(stkmx_any)) THEN
  WRITE(jp_stderr,'("[STKMX_getPCDatacntt] Error: ", A)') &
    'stkmx_any not ASSOCIATED -- aborting'
  CALL ABORT()
ENDIF


                 ! Get the number of child nodes
n_children = stkmx_any%n_children

                 ! If stkmx_any does not have any children, there is
                 ! nothing to look for
IF (stkmx_any%n_children == 0) RETURN


                 ! There are children: pointer to the first one
stkmx_child => stkmx_any%frstchild

! Provisionally ALLOCATE stkmx_nodearrtmp with <n_children> elements
ALLOCATE(stkmx_nodearrtmp(n_children))


n_dimres = 0
DO i_child = 1, n_children

  IF (i_child > 1) stkmx_child => stkmx_child%nextsib
  IF (stkmx_child%i_type /= 4) CYCLE

  n_dimres = n_dimres+1
  stkmx_nodearrtmp(n_dimres)%ptr => stkmx_child

ENDDO

IF (n_dimres /= 0) THEN
  ALLOCATE(stkrc_datacntt(n_dimres))
  DO i_child = 1, n_dimres
    stkrc_datacntt(i_child)%ptr => stkmx_nodearrtmp(i_child)%ptr%stkrc_data
  ENDDO
ENDIF

DEALLOCATE(stkmx_nodearrtmp)


RETURN

!-----------------------------------------------------------------------
END FUNCTION STKMX_getCDatacntt
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
FUNCTION STKMX_getElementName(stkmx_any) RESULT(stkrc_eltname)
!-----------------------------------------------------------------------

! Returns the element name of the node that <stkmx_any> points to
! Returns an empty chain if <stkmx_any> does not point to an element
! node (i.e., if it points to a PCDATA or a CDATA node.

IMPLICIT NONE

! Argument list variables
! -----------------------

TYPE(stack_minixml), POINTER   :: stkmx_any
TYPE(stack_recchunks), POINTER :: stkrc_eltname


! Pre-set stkrc_eltname to NULL
NULLIFY(stkrc_eltname)


IF (.NOT. ASSOCIATED(stkmx_any)) THEN
  WRITE(jp_stderr,'("[STKMX_getElementName] Error: ", A)') &
    'stkmx_any not ASSOCIATED -- aborting'
  CALL ABORT()
ENDIF

                 ! If <stkmx_any> does not point to an element node, return
IF ((stkmx_any%i_type < 0) .OR. (stkmx_any%i_type > 1)) RETURN

stkrc_eltname => stkmx_any%stkrc_eltname

RETURN

!-----------------------------------------------------------------------
END FUNCTION STKMX_getElementName
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
INTEGER FUNCTION STKMX_getAttIdxByName(stkmx_node, str_attname)
!-----------------------------------------------------------------------

! Returns the index of the attribute named <str_attname>
! in the arrays in <stkmx_eltnode>
! Returns 
! -1 if <stkmx_node> is not an element node;
!  0 if <stkmx_node> is an element node, but does not have
!    any attribute named <str_attname> (or no attributes at all);
!  the index of the attribute named <str_attname> if it exists.

IMPLICIT NONE

! Argument list variables
! -----------------------

TYPE(stack_minixml), POINTER :: stkmx_node
CHARACTER(LEN=*), INTENT(IN) :: str_attname


! Local variables
! ---------------

INTEGER :: n_atts, i_att
TYPE(stack_recchunks), POINTER :: stkrc_attname


IF (.NOT. ASSOCIATED(stkmx_node)) THEN
  WRITE(jp_stderr,'("[STKMX_getElementNodeByName] Error: ", A)') &
    'stkmx_node is not ASSOCIATED -- aborting'
  CALL ABORT()
ENDIF

                 ! Pre-set RESULT with 0
STKMX_getAttIdxByName = 0

                 ! If str_name is empty, there is nothing to look for
IF (LEN_TRIM(str_attname) == 0) RETURN

IF ((stkmx_node%i_type /= 0) .AND. (stkmx_node%i_type /= 1)) THEN
                 ! Wrong node type
  STKMX_getAttIdxByName = -1
  RETURN
ENDIF

                 ! If no attributes, RETURN
IF (.NOT. ALLOCATED(stkmx_node%nlen_attnames)) RETURN

                 ! There are attributes

n_atts = SIZE(stkmx_node%nlen_attnames)


DO i_att = 1, n_atts

  IF (STKRC_STKRC_EQ_STR(stkmx_node%stkrc_attnames(i_att)%ptr, str_attname)) THEN
    STKMX_getAttIdxByName = i_att
    RETURN
  ENDIF

ENDDO


RETURN

!-----------------------------------------------------------------------
END FUNCTION STKMX_getAttIdxByName
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
FUNCTION STKMX_getAttcnttByIdx(stkmx_node, i_att) RESULT(stkrc_attcntt)
!-----------------------------------------------------------------------

! Returns a pointer to the content of attribute <i_att>
! of the node pointed to by <stkmx_eltnode>
! Returns 
! NULL if <stkmx_node> is not an element node
! NULL if <stkmx_node> is an element node, but does not have
!    any attribute with index <i_att>

IMPLICIT NONE

! Argument list variables
! -----------------------

TYPE(stack_minixml), POINTER :: stkmx_node
INTEGER, INTENT(IN)          :: i_att


! Result variable
! -----------------------
TYPE(stack_recchunks), POINTER :: stkrc_attcntt


! Local variables
! ---------------

INTEGER :: n_atts


IF (.NOT. ASSOCIATED(stkmx_node)) THEN
  WRITE(jp_stderr,'("[STKMX_getAttcnttByIdx] Error: ", A)') &
    'stkmx_node is not ASSOCIATED -- aborting'
  CALL ABORT()
ENDIF

                 ! Pre-set RESULT with NULL
NULLIFY(stkrc_attcntt)


IF ((stkmx_node%i_type /= 0) .AND. (stkmx_node%i_type /= 1)) THEN
                 ! Wrong node type
  RETURN
ENDIF

                 ! If no attributes, RETURN
IF (.NOT. ALLOCATED(stkmx_node%nlen_attnames)) RETURN

                 ! There are attributes
n_atts = SIZE(stkmx_node%nlen_attnames)

IF ( (i_att > 0) .AND. (i_att <= n_atts) ) THEN
  stkrc_attcntt => stkmx_node%stkrc_attcntts(i_att)%ptr
ENDIF


RETURN

!-----------------------------------------------------------------------
END FUNCTION STKMX_getAttcnttByIdx
!-----------------------------------------------------------------------



!=======================================================================
 END MODULE MODMXM_STKMX
!=======================================================================
