;
;    Copyright 2007-2020 Guy Munhoven
;
;    This file is part of Medusa.
;
;    Medusa is free software: you can redistribute it and/or modify
;    it under the terms of the GNU Affero General Public License as
;    published by the Free Software Foundation, either version 3 of
;    the License, or (at your option) any later version.
;
;    Medusa is distributed in the hope that it will be useful, but
;    WITHOUT ANY WARRANTY; without even the implied warranty of
;    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
;    See the GNU Affero General Public License for more details.
;
;    You should have received a copy of the Affero GNU General Public
;    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
;


FUNCTION INT_SOLIDS, dbswi, phi, xconc_solid, dbswi_max

idx = WHERE(dbswi LE dbswi_max, count)

int_solid = xconc_solid[0] * (1D0 - phi[0]) * (dbswi[1] - dbswi[0])/ 2D0 

FOR i = 1, count - 1 DO BEGIN

  int_solid = int_solid + xconc_solid[i] * (1D0 - phi[i]) * (dbswi[i+1] - dbswi[i-1])/ 2D0

ENDFOR

RETURN, int_solid

END




FUNCTION MEDUSA_H_FITFUN, x, p

; Jeasim parameter and forcing values that are not under MPFIT control
; ====================================================================

COMMON JEASIM_OTHER, i_vl, i_pc, dbswi_max, c_site, c_headerline1, c_headerline2, $
  wlon, wlat, wbdry_dbsl, wbdry_omega_calc, wbdry_tmpc, wbdry_sali, $
  pc_calc, bbdry_wtot, db_0, phi_0, phi_infty, phi_alpha, $
  wconc_o2, wconc_no3, wconc_hco3, wconc_co3, wconc_co2, wbdry_ph, wconc_ca, $
  wconc_nh4, wconc_so4, wconc_mn, wconc_fe, wconc_h2s, wconc_hs, wconc_boh3, wconc_boh4, $
  wflux_mno2, wflux_feoh3, wflux_orgc, $
  wbdry_frac_orgcf, ak_orgmf, $
  n_ph

; Please adapt the name of executable below if necessary:
IF i_vl EQ 1 THEN BEGIN
  ; here the "volumeless solids" executable (compiled with -DSOLIDS_VOLUMELESS)
  cmd_medusa_exe = './medusa_jeasim_vl.0_321_0'
ENDIF ELSE BEGIN
  ; here the "normal solids" executable (compiled without that option)
  cmd_medusa_exe = './medusa_jeasim.0_321_0'
ENDELSE


; plug P values into the relevant extra variables
; ===============================================

wflux_clay       = p(0)
an_calcdiss      = p(1)
ak_calcdiss      = p(2)
wflux_calc       = p(3)


; Use WRITE_JEASIM_CSV to create new jeasim_adj.csv
; =================================================
WRITE_JEASIM_CSV, 'jeasim_adj.csv', $
  c_site, c_headerline1, c_headerline2, $
  wlon, wlat, wbdry_dbsl, wbdry_omega_calc, wbdry_tmpc, wbdry_sali, $
  pc_calc, bbdry_wtot, db_0, phi_0, phi_infty, phi_alpha, $
  wconc_o2, wconc_no3, wconc_hco3, wconc_co3, wconc_co2, wbdry_ph, wconc_ca, $
  wconc_nh4, wconc_so4, wconc_mn, wconc_fe, wconc_h2s, wconc_hs, wconc_boh3, wconc_boh4, $
  wflux_mno2, wflux_feoh3, wflux_orgc, $
  wbdry_frac_orgcf, ak_orgmf, wflux_calc, $
  an_calcdiss, ak_calcdiss, wflux_clay


; run Medusa
; ==========
; Make sure beforehand that cfn_csvin_jeasim is set
; to 'jeasim_adj.csv' in medusa_jeasim_files.nml

SPAWN, cmd_medusa_exe


;; Interrupt the program to allow for inspection of the results
;; ============================================================
;PRINT, FORMAT = '(A, $)', 'Continue ? '
;c = ''
;READ, c


; Load results
; ============

fn_reaclay = 'medusa_reaclay.nc'
fn_flx     = 'medusa_flx.nc'
fn_bc      = 'medusa_bc.nc'

; xgeom_dbswi, xconc_o2 from medusa_reaclay.nc
; ---------------------

id = NCDF_OPEN(fn_reaclay, /NOWRITE)

NCDF_VARGET, id, NCDF_VARID(id, 'xgeom_dbswi'),   dbswi
NCDF_VARGET, id, NCDF_VARID(id, 'xgeom_phi'),     phi
NCDF_VARGET, id, NCDF_VARID(id, 'xconc_o2'),      o2
NCDF_VARGET, id, NCDF_VARID(id, 'xconc_co2'),     co2
NCDF_VARGET, id, NCDF_VARID(id, 'xconc_hco3'),    hco3
NCDF_VARGET, id, NCDF_VARID(id, 'xconc_co3'),     co3
NCDF_VARGET, id, NCDF_VARID(id, 'xconc_calc'),    calc
NCDF_VARGET, id, NCDF_VARID(id, 'xconc_clay'),    clay
NCDF_VARGET, id, NCDF_VARID(id, 'xconc_omf'),     omf
NCDF_VARGET, id, NCDF_VARID(id, 'xconc_oms'),     oms
NCDF_VARGET, id, NCDF_VARID(id, 'xconc_mno2'),    mno2
NCDF_VARGET, id, NCDF_VARID(id, 'xconc_feoh3'),   feoh3

NCDF_CLOSE, id



IF i_pc EQ 1 THEN BEGIN

  tcalc = INT_SOLIDS(dbswi, phi, calc, dbswi_max)
  tclay = INT_SOLIDS(dbswi, phi, clay, dbswi_max)

  IF i_vl EQ 1 THEN BEGIN

    ; with the volumeless setup, "clay" actually represents
    ; the bulk solids, and all the other components behave
    ; as (volumeless/infinite density) colour tracers
    pc_calc_res = 100D0*tcalc/tclay

  ENDIF ELSE BEGIN

    tomf  = INT_SOLIDS(dbswi, phi, omf,   dbswi_max)
    toms  = INT_SOLIDS(dbswi, phi, oms,   dbswi_max)
    mno2  = INT_SOLIDS(dbswi, phi, mno2,  dbswi_max)
    feoh3 = INT_SOLIDS(dbswi, phi, feoh3, dbswi_max)
  
    pc_calc_res = 100D0*tcalc/(tclay+tcalc+tomf+toms+tmno2+tfeoh3)

  ENDELSE

ENDIF ELSE BEGIN

  ; ignore the pc_calc constraint by setting pc_calc_res = pc_calc right away
  pc_calc_res = pc_calc

ENDELSE


; xw from medusa_flx.nc
; --

id = NCDF_OPEN(fn_flx, /NOWRITE)

NCDF_VARGET, id, NCDF_VARID(id, 'xgeom_dbswi'),   dbswi_vtx
NCDF_VARGET, id, NCDF_VARID(id, 'xw_tot'),        xw_tot

NCDF_CLOSE, id


; xw from medusa_bc.nc
; --

id = NCDF_OPEN(fn_bc, /NOWRITE)

NCDF_VARGET, id, NCDF_VARID(id, 'wconc_co2'),       wco2
NCDF_VARGET, id, NCDF_VARID(id, 'wconc_hco3'),      whco3
NCDF_VARGET, id, NCDF_VARID(id, 'wconc_co3'),       wco3
NCDF_VARGET, id, NCDF_VARID(id, 'ccttotc_h3o_swi'), wh3o

NCDF_CLOSE, id

k1 = wh3o*(whco3/wco2)
k2 = wh3o*(wco3/whco3)

h = (k1(0)*(co2/hco3) + k2(0)*(hco3/co3))/2D0

; interpolate results onto X values
; ---------------------------------

n = N_ELEMENTS(x)

h_res = INTERPOL(h, dbswi, x(0:n_ph-1))


; take deepest available xw_tot value for xw
; ------------------------------------------

w_res = xw_tot(N_ELEMENTS(xw_tot)-1)


; Done
; ====

RETURN, [h_res, pc_calc_res, w_res]

END
