!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#ifdef FN_THISFILE
#undef FN_THISFILE
#endif
#define FN_THISFILE "sedcore_extract.F90"
#ifndef __LINE__
#define __LINE__ 0
#endif
!=======================================================================
 MODULE MOD_SEDCORE_EXTRACT
!=======================================================================

USE MOD_UTICOMMON


IMPLICIT NONE


CHARACTER(LEN=*), PARAMETER :: cfn_thisfile = FN_THISFILE


                                    ! From MOD_MEDUSA_COCOGEN
                                    ! -----------------------

                                    !  * maximum length of
                                    !    component type names
INTEGER, PARAMETER :: n_lmaxcomptyp =  30
      
                                    !  * solid production time class name
CHARACTER(LEN=*), PARAMETER :: cp_classsolidpt = 'SolidProductionTime'


                                    ! From MOD_MEDUSA_NETCDFPARAM
                                    ! ---------------------------

                                    !  * LAYer dimension name
CHARACTER(LEN=*), PARAMETER :: ddn_lay  = 'lay'

                                    !  * LAYer dimension variable name
CHARACTER(LEN=*), PARAMETER :: vsn_lay  = ddn_lay

                                    !  * COLumn dimension name
CHARACTER(LEN=*), PARAMETER :: ddn_col   = 'col'

                                    !  * COLumn dimension variable name
CHARACTER(LEN=*), PARAMETER :: vsn_col   = ddn_col


                                    ! From MOD_SEDCORE
                                    ! ----------------

CHARACTER(LEN=*), PARAMETER :: cp_nbcorelays = 'nlay_corelay'
CHARACTER(LEN=*), PARAMETER :: cp_nbrecs     = 'nrecs'
CHARACTER(LEN=*), PARAMETER :: cp_itcorelay  = 'irec_top_corelay'
CHARACTER(LEN=*), PARAMETER :: cp_ilerodlay  = 'irec_last_erodlay'
CHARACTER(LEN=*), PARAMETER :: cp_coreid     = 'core_id'
CHARACTER(LEN=*), PARAMETER :: cp_layid      = 'lay_id'
CHARACTER(LEN=*), PARAMETER :: cp_ilastlayid = 'last_lay_id'
CHARACTER(LEN=*), PARAMETER :: cp_ipreclay   = 'irec_preclay'
CHARACTER(LEN=*), PARAMETER :: cp_burtime    = 'burial_time'
CHARACTER(LEN=*), PARAMETER :: cp_erotime    = 'erosion_time'

CHARACTER(LEN=*), PARAMETER :: cp_prefix_hcont = 'hcont_'

                                    ! special flag for "no record"
INTEGER, PARAMETER ::  jp_pt_norecord            = -1


#include <netcdf.inc>

                                    ! SEDCORE file related stuff
                                    ! --------------------------

                                    ! - open/not open flag
LOGICAL, SAVE :: l_sedfil_isopen = .FALSE.

                                    ! - number of solids included
INTEGER, SAVE :: nsolid = 0

                                    ! - NetCDF ID
INTEGER, SAVE :: sedfil_ncid

                                    ! - NetCDF dimension and variable IDs
 
INTEGER, SAVE :: iddim_col_in, idvar_col_in
INTEGER, SAVE :: iddim_lay_in, idvar_lay_in


INTEGER, SAVE :: idvar_nbcorelays
INTEGER, SAVE :: idvar_nbrecs
INTEGER, SAVE :: idvar_itcorelay
INTEGER, SAVE :: idvar_ilerodlay
INTEGER, SAVE :: idvar_coreid
INTEGER, SAVE :: idvar_layid
INTEGER, SAVE :: idvar_ilastlayid
INTEGER, SAVE :: idvar_ipreclay
INTEGER, SAVE :: idvar_burtime_in
INTEGER, SAVE :: idvar_erotime_in

CHARACTER(LEN = NF_MAX_NAME), DIMENSION(:), ALLOCATABLE, SAVE :: cname_hcont
INTEGER, DIMENSION(:), ALLOCATABLE, SAVE :: idvar_hcont_in
INTEGER, DIMENSION(:), ALLOCATABLE, SAVE :: jf_ptxref


                                    ! - total number of columns
                                    !   considered in file
INTEGER, SAVE :: sedfil_ncols

                                    ! - total number of records
                                    !   included in the file
INTEGER, SAVE :: sedfil_nrecs

                                    ! - for each core, number records
                                    !   in the file (all records:
                                    !   CORELAY, ERODLAY)
INTEGER, DIMENSION(:), ALLOCATABLE, SAVE :: sedflay_nrecs

                                    ! - for each core, index of the
                                    !   record for the latest layer that
                                    !   still exists only in the file
INTEGER, DIMENSION(:), ALLOCATABLE, SAVE :: sedflay_irectop

                                    ! - for each core, index of the
                                    !   record for the latest eroded
                                    !   layer that has been written to
                                    !   the file
INTEGER, DIMENSION(:), ALLOCATABLE, SAVE :: sedflay_irecero

                                    ! - for each core, number of not yet
                                    !   eroded layers in the file
                                    !   (sets the maximum number of
                                    !   records that are available for
                                    !   transfer back).
INTEGER, DIMENSION(:), ALLOCATABLE, SAVE :: sedflay_nlay

                                    ! - LAST ID, ID of the most recent
                                    !   layer that was created
INTEGER, DIMENSION(:), ALLOCATABLE, SAVE :: corelay_lastid


                                    ! CORE file related stuff
                                    ! -----------------------

                                    ! - open/not open flag
LOGICAL, SAVE :: l_corefil_isopen = .FALSE.

                                    ! - NetCDF ID
INTEGER, SAVE :: corefil_ncid

                                    ! - NetCDF dimension and variable IDs
 
INTEGER, SAVE :: iddim_lay_out, idvar_lay_out

INTEGER, SAVE :: idvar_burtime_out
INTEGER, SAVE :: idvar_erotime_out
INTEGER, DIMENSION(:), ALLOCATABLE, SAVE :: idvar_hcont_out


CONTAINS

!-----------------------------------------------------------------------
 SUBROUTINE GET_MENU(cfn_nc_sedcore, cdn_corefiles, ilist_cols2extract)
!-----------------------------------------------------------------------


IMPLICIT NONE


CHARACTER(LEN=*),   INTENT(OUT) :: cfn_nc_sedcore
CHARACTER(LEN=*),   INTENT(OUT) :: cdn_corefiles
INTEGER, DIMENSION(:), POINTER  :: ilist_cols2extract

CHARACTER(LEN=255)              :: cl_corelist


cfn_nc_sedcore = 'medmbm_sedcore.nc'
cdn_corefiles  = 'corefiles'


WRITE(*, '("Please enter (no quotes required)")')

WRITE(*, '(A)', ADVANCE="NO") ' - the name of the SEDCORE file to process: '
READ(*,'(A)') cfn_nc_sedcore

WRITE(*, '(A)', ADVANCE="NO") ' - the name of the directory to store the CORE files: '
READ(*,'(A)') cdn_corefiles


WRITE(*, '(A)', ADVANCE="NO") ' - the list of the cores to extract ' // &
                              '(e.g. 1-3,5,6-9), max. 255 characters: '
READ(*,'(A)') cl_corelist

#ifdef DEBUG
WRITE(*,'("SEDCORE name """, A,"""")')         TRIM(cfn_nc_sedcore)
WRITE(*,'("dest. directory name """, A,"""")') TRIM(cdn_corefiles)
WRITE(*,'("List read in = """, A,"""")')       TRIM(cl_corelist)
#endif


CALL EXPAND_LIST(cl_corelist, ilist_cols2extract)


RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE GET_MENU
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE SEDFIL_OPEN(cfn_ncin_sedfil)
!-----------------------------------------------------------------------

! Opens an existing SEDCORE file named <cfn_ncin_sedfil>



IMPLICIT NONE


CHARACTER(LEN=*), INTENT(IN) :: cfn_ncin_sedfil


! Local variables
! ===============

INTEGER :: ncid, istatus
INTEGER :: i, j, jsolid, jcompo
INTEGER :: len_coldim, len_laydim
INTEGER :: nvars
INTEGER :: idvar_hcont_in_ptxref

CHARACTER(LEN = NF_MAX_NAME) :: cname_var
CHARACTER(LEN = n_lmaxcomptyp) :: c_class
INTEGER                      :: nlen


CHARACTER(LEN=*), PARAMETER :: c_fmterr_a = '("[SEDFIL_OPEN] error: ", A)'


! ===================
! End of declarations
! ===================


IF (l_sedfil_isopen) THEN
  WRITE(jp_stderr, c_fmterr_a) 'SEDCORE file already open -- aborting!'
  CALL ABORT_EXECUTION()
ENDIF



!--------------------
! Open the data file
!--------------------

istatus = NF_OPEN(TRIM(cfn_ncin_sedfil), NF_NOWRITE, ncid)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))


!----------------------------------------------------
! Get dimension id's and lengths + consistency check
!----------------------------------------------------


                                    ! Columns
istatus = NF_INQ_DIMID(ncid, ddn_col, iddim_col_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

istatus = NF_INQ_DIMLEN(ncid, iddim_col_in, len_coldim)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

istatus = NF_INQ_VARID(ncid, vsn_col, idvar_col_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))


                                    ! Layers
istatus = NF_INQ_DIMID(ncid, ddn_lay, iddim_lay_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

istatus = NF_INQ_DIMLEN(ncid, iddim_lay_in, len_laydim)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

istatus = NF_INQ_VARID(ncid, vsn_lay, idvar_lay_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))


                                    ! Data variables:
                                    
                                    ! 1. General core characteristics

                                    !  - number of corelay's
istatus = NF_INQ_VARID(ncid, cp_nbcorelays, idvar_nbcorelays)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

                                    !  - number of actual records
istatus = NF_INQ_VARID(ncid, cp_nbrecs, idvar_nbrecs)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

                                    !  - irec of topmost CORELAY
istatus = NF_INQ_VARID(ncid, cp_itcorelay, idvar_itcorelay)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

                                    !  - irec of last written ERODLAY
istatus = NF_INQ_VARID(ncid, cp_ilerodlay, idvar_ilerodlay)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

                                    !  - ilay of last physical layer
istatus = NF_INQ_VARID(ncid, cp_ilastlayid, idvar_ilastlayid)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))


                                    ! 2. Layer related data

                                    !  - Core ID of a layer
istatus = NF_INQ_VARID(ncid, cp_coreid, idvar_coreid)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

                                    !  - Layer ID in a core
istatus = NF_INQ_VARID(ncid, cp_layid, idvar_layid)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

                                    !  - irec of the preceeding layer
istatus = NF_INQ_VARID(ncid, cp_ipreclay, idvar_ipreclay)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

                                    !  - burial time
istatus = NF_INQ_VARID(ncid, cp_burtime, idvar_burtime_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

                                    !  - erosion time
istatus = NF_INQ_VARID(ncid, cp_erotime, idvar_erotime_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

                                    !  - hcont
                                    !    * get the total number of variables
istatus = NF_INQ_NVARS(ncid, nvars)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

                                    !    * scan the file to get the
                                    !      number of 'hcont*' variables
nsolid = 0
nlen = LEN_TRIM(cp_prefix_hcont)
DO i = 1, nvars

  istatus = NF_INQ_VARNAME (ncid, i, cname_var)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

  IF (cname_var(1:nlen) == cp_prefix_hcont) THEN
    nsolid = nsolid + 1
  ENDIF

ENDDO

                                    !    * no solids ??? cannot be!
IF (nsolid == 0) THEN
  WRITE(jp_stderr, c_fmterr_a) 'No data for solids included -- aborting'
  CALL ABORT_EXECUTION()
ENDIF


ALLOCATE(idvar_hcont_in(nsolid))
ALLOCATE(cname_hcont(nsolid))
ALLOCATE(jf_ptxref(nsolid))

jf_ptxref(:) = -1

                                    !    * get "hcont_*" variable ID's
jsolid = 0
DO i = 1, nvars

  istatus = NF_INQ_VARNAME(ncid, i, cname_var)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

  nlen = LEN_TRIM(cp_prefix_hcont)

  IF (cname_var(1:nlen) == cp_prefix_hcont) THEN

    jsolid = jsolid + 1
    cname_hcont(jsolid) = cname_var

    idvar_hcont_in(jsolid) = i

                                    !    * get class
    istatus = NF_INQ_ATTLEN(ncid, i, 'class', nlen)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))
    istatus = NF_GET_ATT_TEXT(ncid, i, 'class', c_class)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

                                    !    * if solid PT, ...
    IF (c_class(1:nlen) == cp_classsolidpt) THEN
                                    !    * ... get 'pt_xref' attribute
      istatus = NF_INQ_ATTLEN(ncid, i, 'pt_xref', nlen)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))
      istatus = NF_GET_ATT_TEXT(ncid, i, 'pt_xref', cname_var)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))


                                    !    * ... get var ID referring to
                                    !      'pt_xref' attribute
      istatus = NF_INQ_VARID(ncid, cname_var(1:nlen), idvar_hcont_in_ptxref)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

                                    !    * ... set up X-ref link
      DO j = 1, nsolid
        IF (idvar_hcont_in(j) ==  idvar_hcont_in_ptxref) THEN
          jf_ptxref(jsolid) = j
          EXIT
        ENDIF
      ENDDO

      IF (j == (nsolid+1)) THEN
        WRITE(jp_stderr, c_fmterr_a) ' cannot find pt_xref variable "' // &
          cname_var(1:nlen) // '" -- aborting'
        CALL ABORT_EXECUTION()
      ENDIF

    ENDIF

  ENDIF

ENDDO



ALLOCATE(sedflay_nrecs(len_coldim))
ALLOCATE(sedflay_nlay(len_coldim))
ALLOCATE(sedflay_irectop(len_coldim))
ALLOCATE(sedflay_irecero(len_coldim))
ALLOCATE(corelay_lastid(len_coldim))



                                    ! Read back pointers and counters
                                    ! - number of CORELAY layers available
                                    !   in the file for core <icore>
istatus = NF_GET_VAR_INT(ncid, idvar_nbcorelays, sedflay_nlay(:))
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

                                    ! - get the number of records already
                                    !   stored in the file for core <icore>
istatus = NF_GET_VAR_INT(ncid, idvar_nbrecs, sedflay_nrecs(:))
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

                                    ! - record number of the youngest
                                    !   stored CORELAY
istatus = NF_GET_VAR_INT(ncid, idvar_itcorelay, sedflay_irectop(:))
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

                                    ! - record number of the most
                                    !   recently stored ERODLAY
istatus = NF_GET_VAR_INT(ncid, idvar_ilerodlay, sedflay_irecero(:))
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

                                    ! - adjust the last ID of the layers
                                    !   stored in the file for core <icore>
istatus = NF_GET_VAR_INT(ncid, idvar_ilastlayid, corelay_lastid(:))
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))


IF (ANY(sedflay_irectop(:) == jp_pt_norecord) .OR.   &
    ANY(sedflay_irecero(:) == jp_pt_norecord)) THEN

  WRITE(jp_stderr, c_fmterr_a) 'Some cores have not been correctly initialised'
  WRITE(jp_stderr, '(A)')      'File "'//TRIM(cfn_ncin_sedfil)//'" is not ' // &
                               'correctly formatted -- aborting.'
  CALL ABORT_EXECUTION()

ENDIF

                                    ! Transcribe info into module main part
sedfil_ncols  = len_coldim
sedfil_nrecs  = len_laydim

sedfil_ncid   = ncid

l_sedfil_isopen = .TRUE.


RETURN


!-----------------------------------------------------------------------
 END SUBROUTINE SEDFIL_OPEN
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE SEDFIL_CLOSE
!-----------------------------------------------------------------------

! - Close the currently open SEDCORE file

IMPLICIT NONE


INTEGER :: istatus

CHARACTER(LEN=*), PARAMETER :: cfmt_err_a = '("[SEDFIL_CLOSE] error: ", A)'


! Check if the file is actually open (if not -- fatal error)

IF (.NOT. l_sedfil_isopen) THEN
  WRITE(jp_stderr,cfmt_err_a) 'Trying to close a SEDCORE file that is not open -- aborting!.'
  CALL ABORT_EXECUTION()
ENDIF


istatus = NF_CLOSE(sedfil_ncid)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

DEALLOCATE(sedflay_nrecs)
DEALLOCATE(sedflay_nlay)
DEALLOCATE(sedflay_irectop)
DEALLOCATE(sedflay_irecero)
DEALLOCATE(corelay_lastid)

nsolid = 0
DEALLOCATE(idvar_hcont_in)
DEALLOCATE(cname_hcont)
DEALLOCATE(jf_ptxref)

l_sedfil_isopen = .FALSE.


RETURN


!-----------------------------------------------------------------------
 END SUBROUTINE SEDFIL_CLOSE
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE SEDFIL_GET_RECORDINFO(irec, icore, ilay, irec_prevlay)
!-----------------------------------------------------------------------

! - get record info

IMPLICIT NONE


INTEGER, INTENT(IN)  :: irec
INTEGER, INTENT(OUT) :: icore
INTEGER, INTENT(OUT) :: ilay
INTEGER, INTENT(OUT) :: irec_prevlay


INTEGER :: istatus

CHARACTER(LEN=*), PARAMETER :: cfmt_err_a = '("[SEDFIL_GET_RECORDINFO] error: ", A)'


! ===================
! End of declarations
! ===================


IF (.NOT. l_sedfil_isopen) THEN
  WRITE(jp_stderr, cfmt_err_a) 'cannot read record!'
  WRITE(jp_stderr,'("no SEDFIL file open")')
  WRITE(jp_stderr,'("Aborting!")')
  CALL ABORT_EXECUTION()
ENDIF


IF ((irec <= 0) .OR. (irec > sedfil_nrecs)) THEN
  WRITE(jp_stderr, cfmt_err_a) 'invalid record number:'
  WRITE(jp_stderr, '(" - required: 0 < irec <= ", I0)') sedfil_nrecs
  WRITE(jp_stderr, '(" - is equal to ", I0)') irec
  WRITE(jp_stderr, '("Aborting!")')
  CALL ABORT_EXECUTION()
ENDIF

                                    ! - record number of the youngest
                                    !   stored CORELAY
istatus = NF_GET_VAR1_INT(sedfil_ncid, idvar_coreid, irec, icore)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

istatus = NF_GET_VAR1_INT(sedfil_ncid, idvar_layid, irec, ilay)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

istatus = NF_GET_VAR1_INT(sedfil_ncid, idvar_ipreclay, irec, irec_prevlay)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))


RETURN


!-----------------------------------------------------------------------
 END SUBROUTINE SEDFIL_GET_RECORDINFO
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE SEDFIL_GET_LAYERDATA(irec, hcont_rec,                      &
                                    burial_time_rec, erosion_time_rec)
!-----------------------------------------------------------------------

! - get record info

IMPLICIT NONE


INTEGER, INTENT(IN)  :: irec
DOUBLE PRECISION, DIMENSION(:), INTENT(OUT) :: hcont_rec
DOUBLE PRECISION, INTENT(OUT) :: burial_time_rec
DOUBLE PRECISION, INTENT(OUT) :: erosion_time_rec

INTEGER :: jsolid
INTEGER :: istatus

CHARACTER(LEN=*), PARAMETER :: cfmt_err_a = '("[SEDFIL_GET_LAYERDATA] error: ", A)'


! ===================
! End of declarations
! ===================


IF (.NOT. l_sedfil_isopen) THEN
  WRITE(jp_stderr, cfmt_err_a) 'cannot read record!'
  WRITE(jp_stderr,'("no SEDCORE file open")')
  WRITE(jp_stderr,'("Aborting!")')
  CALL ABORT_EXECUTION()
ENDIF


IF ((irec <= 0) .OR. (irec > sedfil_nrecs)) THEN
  WRITE(jp_stderr, cfmt_err_a) 'invalid record number:'
  WRITE(jp_stderr, '(" - required: 0 < irec <= ", I0)') sedfil_nrecs
  WRITE(jp_stderr, '(" - is equal to ", I0)') irec
  WRITE(jp_stderr, '("Aborting!")')
  CALL ABORT_EXECUTION()
ENDIF

                                    ! - record number of the youngest
                                    !   stored CORELAY
DO jsolid = 1, nsolid
  istatus = NF_GET_VAR1_DOUBLE(sedfil_ncid, idvar_hcont_in(jsolid), irec, &
                                    hcont_rec(jsolid))
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-2))
ENDDO

istatus = NF_GET_VAR1_DOUBLE(sedfil_ncid, idvar_burtime_in, irec, burial_time_rec)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

istatus = NF_GET_VAR1_DOUBLE(sedfil_ncid, idvar_erotime_in, irec, erosion_time_rec)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))


RETURN


!-----------------------------------------------------------------------
 END SUBROUTINE SEDFIL_GET_LAYERDATA
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE COREFIL_CREATE(cfn_ncout_corefil, k_core)
!-----------------------------------------------------------------------

! Creates and opens a CORE file named <cfn_nc_core>.


IMPLICIT NONE


CHARACTER(LEN=*), INTENT(IN) :: cfn_ncout_corefil
INTEGER,          INTENT(IN) :: k_core


! Local variables
! ===============

INTEGER :: ncid, istatus
INTEGER :: i, j
INTEGER :: iatt, natts
INTEGER :: jsolid, jcompo
INTEGER :: idvar_in, idvar_out


CHARACTER(LEN=NF_MAX_NAME) :: cname_var
CHARACTER(LEN=NF_MAX_NAME) :: cname_att
INTEGER                    :: nlen

CHARACTER(LEN=*), PARAMETER :: cfmt_err_a = '("[COREFIL_CREATE] error: ", A)'


! ===================
! End of declarations
! ===================


IF (.NOT. l_sedfil_isopen) THEN
  WRITE(jp_stderr, cfmt_err_a) 'Cannot create CORE file:'
  WRITE(jp_stderr,'("no SEDCORE file open -- Aborting!")')
  CALL ABORT_EXECUTION()
ENDIF


IF (l_corefil_isopen) THEN
  WRITE(jp_stderr, cfmt_err_a) 'Cannot create CORE file:'
  WRITE(jp_stderr,'("CORE file already open -- Aborting!")')
  CALL ABORT_EXECUTION()
ENDIF


ALLOCATE(idvar_hcont_out(nsolid))


!-----------------------------------------------
! Create the data file and put global attributes
!-----------------------------------------------


istatus = NF_CREATE(TRIM(cfn_ncout_corefil), NF_CLOBBER, ncid)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))


                                    ! Core ID
istatus = NF_PUT_ATT_INT(ncid, NF_GLOBAL, 'core_id', NF_INT, 1, k_core)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))



!---------------------------------
! Define dimensions and variables
!---------------------------------


                                    ! Layers
istatus = NF_DEF_DIM(ncid, ddn_lay, NF_UNLIMITED, iddim_lay_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

istatus = NF_DEF_VAR(ncid, vsn_lay, NF_INT, 1, iddim_lay_out, idvar_lay_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))


                                    ! Define variables for

                                    !  - burial time
                                    !    * create variable
istatus = NF_DEF_VAR(ncid, cp_burtime, NF_DOUBLE, 1, iddim_lay_out, idvar_burtime_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

                                    !    * copy over attributes
istatus = NF_INQ_VARNATTS(sedfil_ncid, idvar_burtime_in, natts)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

DO iatt = 1, natts

  istatus = NF_INQ_ATTNAME(sedfil_ncid, idvar_burtime_in, iatt, cname_att)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

  istatus = NF_COPY_ATT(sedfil_ncid, idvar_burtime_in, TRIM(cname_att), ncid, idvar_burtime_out)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

ENDDO

                                    !  - erosion time
                                    !    * create variable
istatus = NF_DEF_VAR(ncid, cp_erotime, NF_DOUBLE, 1, iddim_lay_out, idvar_erotime_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))


                                    !    * copy over attributes
istatus = NF_INQ_VARNATTS(sedfil_ncid, idvar_erotime_in, natts)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

DO iatt = 1, natts

  istatus = NF_INQ_ATTNAME(sedfil_ncid, idvar_erotime_in, iatt, cname_att)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

  istatus = NF_COPY_ATT(sedfil_ncid, idvar_erotime_in, TRIM(cname_att), ncid, idvar_erotime_out)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

ENDDO


                                    !  - hcont
DO jsolid = 1, nsolid

  cname_var = cname_hcont(jsolid)
  nlen = LEN_TRIM(cname_var)

                                    !    * create variable
  istatus = NF_DEF_VAR(ncid, cname_var(1:nlen), NF_DOUBLE, 1, iddim_lay_out, idvar_out)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

  idvar_hcont_out(jsolid) = idvar_out 


                                    !    * copy over attributes from
                                    !      idvar_hcont_in to idvar_hcont_out
  idvar_in = idvar_hcont_in(jsolid)

  istatus = NF_INQ_VARNATTS(sedfil_ncid, idvar_in, natts)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

  DO iatt = 1, natts

    istatus = NF_INQ_ATTNAME(sedfil_ncid, idvar_in, iatt, cname_att)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

    istatus = NF_COPY_ATT(sedfil_ncid, idvar_in, TRIM(cname_att), ncid, idvar_out)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

  ENDDO


ENDDO


!-----------------
! End define mode
!-----------------

istatus = NF_ENDDEF(ncid)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))


corefil_ncid = ncid

l_corefil_isopen = .TRUE.


RETURN


!-----------------------------------------------------------------------
 END SUBROUTINE COREFIL_CREATE
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE COREFIL_CLOSE
!-----------------------------------------------------------------------

! - close the CORE file

IMPLICIT NONE


INTEGER :: istatus

CHARACTER(LEN=*), PARAMETER :: cfmt_err_a = '("[COREFIL_CLOSE] error: ", A)'


! Check if the file is actually open (if not -- fatal error)

IF (.NOT. l_corefil_isopen) THEN
  WRITE(jp_stderr,cfmt_err_a) &
    'Trying to close a CORE file that is not open -- aborting!.'
  CALL ABORT_EXECUTION()
ENDIF


istatus = NF_CLOSE(corefil_ncid)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

DEALLOCATE(idvar_hcont_out)

l_corefil_isopen = .FALSE.


RETURN


!-----------------------------------------------------------------------
 END SUBROUTINE COREFIL_CLOSE
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE COREFIL_WRITE_LAYERDATA(k_lay,                             &
                                    corefil_burtime, corefil_erotime,  &
                                    corefil_solid)
!-----------------------------------------------------------------------


IMPLICIT NONE


INTEGER,                        INTENT(IN) :: k_lay
DOUBLE PRECISION,               INTENT(IN) :: corefil_burtime
DOUBLE PRECISION,               INTENT(IN) :: corefil_erotime
DOUBLE PRECISION, DIMENSION(:), INTENT(IN) :: corefil_solid

INTEGER :: jsolid
INTEGER :: istatus

CHARACTER(LEN=*), PARAMETER :: cfmt_err_a = '("[COREFIL_WRITE_LAYERDATA] error: ", A)'


IF (.NOT. l_corefil_isopen) THEN
  WRITE(jp_stderr, cfmt_err_a) 'cannot WRITE layer data!'
  WRITE(jp_stderr,'("No CORE file open -- Aborting!")')
  CALL ABORT_EXECUTION()
ENDIF


IF (k_lay <= 0) THEN
  WRITE(jp_stderr, cfmt_err_a) 'invalid layer number:'
  WRITE(jp_stderr, '(" - required: 0 < k_lay")')
  WRITE(jp_stderr, '(" - is equal to ", I0)') k_lay
  WRITE(jp_stderr, '("Aborting!")')
  CALL ABORT_EXECUTION()
ENDIF


istatus = NF_PUT_VAR1_INT(corefil_ncid, idvar_lay_out, k_lay, k_lay)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))
 

istatus = NF_PUT_VAR1_DOUBLE(corefil_ncid, idvar_burtime_out, k_lay, corefil_burtime)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))

istatus = NF_PUT_VAR1_DOUBLE(corefil_ncid, idvar_erotime_out, k_lay, corefil_erotime)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-1))


DO jsolid = 1, nsolid
  istatus = NF_PUT_VAR1_DOUBLE(corefil_ncid, idvar_hcont_out(jsolid), k_lay,       &
                                    corefil_solid(jsolid))
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, (__LINE__-2))

ENDDO


RETURN


!-----------------------------------------------------------------------
 END SUBROUTINE COREFIL_WRITE_LAYERDATA
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE HANDLE_NCERRORS(istatus, iline)
!-----------------------------------------------------------------------

IMPLICIT NONE

INTEGER, INTENT(IN) :: istatus
INTEGER, INTENT(IN) :: iline

CHARACTER(LEN=*), PARAMETER :: cfn_thisfile = FN_THISFILE


IF (istatus /= NF_NOERR) THEN
  IF(iline > 0) THEN
    WRITE(jp_stderr,"('[',A,':',I0,']: ', A)") cfn_thisfile, iline,    &
                                    TRIM(NF_STRERROR(istatus))
  ELSE
    WRITE(jp_stderr,"('[',A,':???]: ', A)")    cfn_thisfile,           &
                                    TRIM(NF_STRERROR(istatus))
  ENDIF
  PRINT *, 'NetCDF error detected; aborting.'
  
  CALL ABORT_EXECUTION()

ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE HANDLE_NCERRORS
!-----------------------------------------------------------------------



!=======================================================================
 END MODULE MOD_SEDCORE_EXTRACT
!=======================================================================



!=======================================================================
 PROGRAM SEDCORE_EXTRACT
!=======================================================================

USE mod_sedcore_extract

IMPLICIT NONE


CHARACTER(LEN=jp_lmaxpathname) :: cfn_ncin_sedfil
CHARACTER(LEN=jp_lmaxpathname) :: cdn_corefiles
INTEGER, DIMENSION(:), POINTER :: ilist_cols2extract

LOGICAL :: l_exists

DOUBLE PRECISION, DIMENSION(:,:), ALLOCATABLE :: hcont_vars
DOUBLE PRECISION, DIMENSION(:),   ALLOCATABLE :: burial_time
DOUBLE PRECISION, DIMENSION(:),   ALLOCATABLE :: erosion_time

CHARACTER(LEN=jp_lmaxpathname) :: cfn_ncout_corefil


INTEGER :: i, n
INTEGER :: icol
INTEGER :: irec_nextcorelay, irec_nexterodlay
INTEGER :: irec, nrecs, irec2read
INTEGER :: icore, ilay

INTEGER :: jsolid, j

DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE :: hcont_rec
DOUBLE PRECISION :: burial_time_rec
DOUBLE PRECISION :: erosion_time_rec 

CHARACTER(LEN=*), PARAMETER :: cfmt_err_a = '("[SEDCORE_EXTRACT] error: ", A)'
#ifdef DEBUG
CHARACTER(LEN=*), PARAMETER :: cfmt_dbg_a = '("[SEDCORE_EXTRACT] debug: ", A)'
#endif

CALL GET_MENU(cfn_ncin_sedfil, cdn_corefiles, ilist_cols2extract)

#ifdef DEBUG
WRITE(*,'("Processing SEDCORE file """, A, """")')       TRIM(cfn_ncin_sedfil)
WRITE(*,'("CORE files to be collected in """, A, """")') TRIM(cdn_corefiles)
WRITE(*,'()')
#endif

l_exists = TESTDIR(cdn_corefiles)

IF (.NOT. l_exists) THEN
  WRITE(jp_stderr, cfmt_err_a) '"' // TRIM(cdn_corefiles) // '": no such directory.'
  WRITE(jp_stderr, '("Aborting!")')
  CALL ABORT_EXECUTION()
ENDIF

n = SIZE(ilist_cols2extract)


CALL SEDFIL_OPEN(cfn_ncin_sedfil)

ALLOCATE(hcont_rec(nsolid))

DO i = 1, n

  icol = ilist_cols2extract(i)

# ifdef DEBUG
  WRITE(*, '("Processing column ",I0)', ADVANCE="NO") icol
# endif

!!!#if (1 == 2)

  nrecs = sedflay_nrecs(icol)

# ifdef DEBUG
  IF (nrecs == 1) THEN
    WRITE(*, '(" (has 1 record, incl. ", I0, " eroded one)")') nrecs - sedflay_nlay(icol)
  ELSE
    WRITE(*, '(" (has ", I0, " records, incl. ", I0, " eroded ones)")') nrecs, nrecs - sedflay_nlay(icol)
  ENDIF
# endif

  ALLOCATE(hcont_vars  (nsolid, nrecs))
  ALLOCATE(burial_time (nrecs))
  ALLOCATE(erosion_time(nrecs))

  irec_nextcorelay = sedflay_irectop(icol)
  irec_nexterodlay = sedflay_irecero(icol)

#ifdef DEBUG
  WRITE(*, '(" topmost CORELAY record:", I0)') irec_nextcorelay
  WRITE(*, '(" topmost ERODLAY record:", I0)') irec_nexterodlay
#endif

  DO irec = 1, nrecs

    IF (irec_nextcorelay > irec_nexterodlay) THEN
      irec2read = irec_nextcorelay
      CALL SEDFIL_GET_RECORDINFO(irec2read, icore, ilay, irec_nextcorelay)
#     ifdef DEBUG
      WRITE(*, '(" core ", I0, ": layer ", I0, " (CORELAY)")', ADVANCE="NO") icore, ilay
#     endif
    ELSE
      irec2read = irec_nexterodlay
      CALL SEDFIL_GET_RECORDINFO(irec2read, icore, ilay, irec_nexterodlay)
#     ifdef DEBUG
      WRITE(*, '(" core ", I0, ": layer ", I0, " (ERODLAY)")', ADVANCE="NO") icore, ilay
#     endif
    ENDIF

    CALL SEDFIL_GET_LAYERDATA(irec2read, hcont_rec,                    &
                                    burial_time_rec, erosion_time_rec)

#   ifdef DEBUG
    WRITE(*, '(" -- burial time = ", F10.2,' //                        &
             ' ", erosion time = ", F10.2)',    ADVANCE="NO")          &
                                    burial_time_rec, erosion_time_rec
    DO jsolid = 1, nsolid
      IF (jf_ptxref(jsolid) /= -1) THEN
        WRITE(*, '(" -- avg. product. time = ", F10.2)', ADVANCE="NO") &
                hcont_rec(jsolid)/hcont_rec(jf_ptxref(jsolid))*1.0D+06
      ENDIF
    ENDDO

    WRITE(*,'()')
#   endif

    hcont_vars(:, ilay) = hcont_rec(:)
    burial_time(ilay)   = burial_time_rec
    erosion_time(ilay)  = erosion_time_rec

  ENDDO


  WRITE(cfn_ncout_corefil, '(A, "/", "core", I5.5, ".nc")') TRIM(cdn_corefiles), icore

  CALL COREFIL_CREATE(cfn_ncout_corefil, icore)

  DO ilay = 1, nrecs

    CALL COREFIL_WRITE_LAYERDATA(ilay, burial_time(ilay), erosion_time(ilay), &
                                    hcont_vars(:, ilay))

  ENDDO


  CALL COREFIL_CLOSE()


  DEALLOCATE(hcont_vars)
  DEALLOCATE(burial_time)
  DEALLOCATE(erosion_time)

!!!#endif

# ifdef DEBUG
  WRITE(*,'()')
# endif

ENDDO


DEALLOCATE(ilist_cols2extract)
DEALLOCATE(hcont_rec)
NULLIFY(ilist_cols2extract)


CALL SEDFIL_CLOSE

!=======================================================================
 END PROGRAM SEDCORE_EXTRACT
!=======================================================================
