!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#ifdef CFN_THISFILE
#undef CFN_THISFILE
#endif
#define CFN_THISFILE "medxy2col.F90"
#ifndef __LINE__
#define __LINE__ 0
#endif

#ifdef DEBUG
#ifndef VERBOSE
#define VERBOSE
#endif
#endif
PROGRAM MEDCOL2XY

IMPLICIT NONE

CHARACTER(LEN=4095) :: cfn_ncin_res, cfn_ncin_aux
CHARACTER(LEN=4095) :: cfn_ncout_res
CHARACTER(LEN=4095) :: cfn_tmp

#include <netcdf.inc>

INTEGER :: ncid_file_in
INTEGER :: ncid_file_aux
INTEGER :: ncid_file_out

INTEGER :: ncid_dim_lev_in,   ncid_var_lev_in,  nlen_dim_lev
INTEGER :: ncid_dim_col_in,   ncid_var_col_in,  nlen_dim_col
INTEGER :: ncid_dim_time_in,  ncid_var_time_in, nlen_dim_time

INTEGER :: ncid_dim,          ncid_var

INTEGER :: ncid_dim_lev_out,  ncid_var_lev_out

INTEGER ::                    ncid_var_col4ij
INTEGER :: ndims_in, nvars_in
INTEGER :: ndims_out
INTEGER, DIMENSION(:), ALLOCATABLE :: nlen_dims_out
INTEGER, DIMENSION(:), ALLOCATABLE :: nlen_dims_in
INTEGER :: ndims, nvars, natts, itype
INTEGER :: ndims_var_out
CHARACTER(LEN=NF_MAX_NAME) :: cname_dim_x_aux, cname_dim_y_aux, cname_dim
CHARACTER(LEN=4096) :: cname_vars_extra
CHARACTER(LEN=NF_MAX_NAME) :: cname_extra
CHARACTER(LEN=NF_MAX_NAME) :: cname_var
CHARACTER(LEN=NF_MAX_NAME) :: cname_att
INTEGER, DIMENSION(:), POINTER  :: iptr_begin, nptr_len
INTEGER :: i_timeslice, itime

CHARACTER(LEN=NF_MAX_NAME), DIMENSION(:),   ALLOCATABLE :: cname_dims_in
CHARACTER(LEN=NF_MAX_NAME), DIMENSION(:),   ALLOCATABLE :: cname_vars_in
INTEGER,                    DIMENSION(:),   ALLOCATABLE :: ihow2processvars_in
INTEGER,                    DIMENSION(:),   ALLOCATABLE :: ncid_vars_out
INTEGER,                    DIMENSION(:),   ALLOCATABLE :: itype_vars_in

INTEGER,                    DIMENSION(3) :: istart, ncount


INTEGER :: ncid_dim_x_aux,  ncid_var_x_aux, ix_aux, nlen_dim_x_aux
INTEGER :: ncid_dim_y_aux,  ncid_var_y_aux, jy_aux, nlen_dim_y_aux

INTEGER :: ncid_dim_x_in,   ncid_var_x_in,          nlen_dim_x_in
INTEGER :: ncid_dim_y_in,   ncid_var_y_in,          nlen_dim_y_in

INTEGER :: ncid_dim_col_out,  ncid_var_col_out
INTEGER :: istatus

INTEGER, DIMENSION(2) :: ncid_dim2
INTEGER, DIMENSION(3) :: ncid_dim3

LOGICAL :: l_exists
INTEGER :: ix, i1, i2
INTEGER :: i, j, k, n, ivar

NAMELIST /nml_medxy2col/ cfn_ncin_res, cfn_ncin_aux, cfn_ncout_res, &
                                    cname_dim_x_in, cname_dim_y_in, &
                                    cname_dim_x_aux, cname_dim_y_aux, &
                                    cname_vars_extra, &
                                    i_timeslice

INTEGER, DIMENSION(:), ALLOCATABLE :: i4col
INTEGER, DIMENSION(:), ALLOCATABLE :: j4col
INTEGER :: i_nonesuch

INTEGER                                         :: iscalar
INTEGER,          DIMENSION(:),     ALLOCATABLE :: iarr1_in
INTEGER,          DIMENSION(:,:),   ALLOCATABLE :: iarr2_out
INTEGER,          DIMENSION(:,:),   ALLOCATABLE :: iarr2_in
INTEGER,          DIMENSION(:,:,:), ALLOCATABLE :: iarr3_out
DOUBLE PRECISION                                :: dscalar
DOUBLE PRECISION, DIMENSION(:),     ALLOCATABLE :: darr1_in
DOUBLE PRECISION, DIMENSION(:,:),   ALLOCATABLE :: darr2_out
DOUBLE PRECISION, DIMENSION(:,:),   ALLOCATABLE :: darr2_in
DOUBLE PRECISION, DIMENSION(:,:,:), ALLOCATABLE :: darr3_out

INTEGER, PARAMETER :: jp_lunit  = 1
INTEGER, PARAMETER :: jp_stderr = 0
CHARACTER(LEN=*), PARAMETER :: cfn_thisfile = CFN_THISFILE


cfn_ncin_res  = "/dev/null"
cfn_ncin_aux  = "medusa_aux.nc"
cfn_ncout_res = "/dev/null"

cname_dim_x_in       = 'X'
cname_dim_y_in       = 'Y'
cname_dim_x_aux      = 'X'
cname_dim_y_aux      = 'Y'
cname_vars_extra = ''

i_timeslice      = 0

OPEN(UNIT=jp_lunit, FILE='medxy2col.nml', STATUS='OLD')
READ(UNIT=jp_lunit, NML=nml_medcol2xy)
CLOSE(UNIT=jp_lunit)

#ifdef VERBOSE
WRITE(*,'("After reading in NAMELIST:")')
WRITE(*,'(" cfn_ncin_res = """,     A, """")') TRIM(cfn_ncin_res)
WRITE(*,'(" cfn_ncin_aux = """,     A, """")') TRIM(cfn_ncin_aux)
WRITE(*,'(" cfn_ncout_res = """,    A, """")') TRIM(cfn_ncout_res)
WRITE(*,'(" cname_dim_x_in = """,   A, """")') TRIM(cname_dim_x_in)
WRITE(*,'(" cname_dim_y_in = """,   A, """")') TRIM(cname_dim_y_in)
WRITE(*,'(" cname_dim_x_aux = """,  A, """")') TRIM(cname_dim_x_aux)
WRITE(*,'(" cname_dim_y_aux = """,  A, """")') TRIM(cname_dim_y_aux)
WRITE(*,'(" cname_vars_extra = """, A, """")') TRIM(cname_vars_extra)
WRITE(*,'(" i_timeslice = ", I0)') i_timeslice
WRITE(*,'()')
#endif

                                    ! Check validity of filenames
IF (cfn_ncin_res == "/dev/null") THEN
  WRITE(jp_stderr, '("NetCDF IN file missing - Aborting!")')
  CALL ABORT()
ELSE
  INQUIRE(FILE=cfn_ncin_res, EXIST=l_exists)
  IF (.NOT.l_exists) THEN
    WRITE(jp_stderr, '("""", A, """: no such file - Aborting!")') TRIM(cfn_ncin_res)
    CALL ABORT()
  ENDIF
ENDIF


IF (cfn_ncin_aux == "/dev/null") THEN
  WRITE(jp_stderr, '("NetCDF AUX file name missing - Aborting!")')
  CALL ABORT()
ELSE
  INQUIRE(FILE=cfn_ncin_aux, EXIST=l_exists)
  IF (.NOT.l_exists) THEN
    WRITE(jp_stderr, '("""", A, """: no such file - Aborting!")') TRIM(cfn_ncin_aux)
    CALL ABORT()
  ENDIF
ENDIF

IF (cfn_ncout_res == "/dev/null") THEN
# ifdef VERBOSE
  WRITE(*, '("Missing name for the NetCDF OUT file")')
# endif
  ix = INDEX(cfn_ncin_res, '.nc', BACK=.TRUE.)
  IF (ix > 1) THEN
    cfn_ncout_res = cfn_ncin_res(1:ix-1) // '_mapped.nc'
#   ifdef VERBOSE
    WRITE(*, '("Using default name """, A, """")') TRIM(cfn_ncout_res)
#   endif
  ELSE
#   ifdef VERBOSE
    WRITE(*, '("Unable to generate name for NetCDF OUT file - Aborting!")')
#   endif
    CALL ABORT()
  ENDIF
ELSE
  INQUIRE(FILE=cfn_ncout_res, EXIST=l_exists)
  IF (l_exists) THEN
    WRITE(jp_stderr, '("Cannot write to """, A, """: file exists already - Aborting!")') TRIM(cfn_ncout_res)
    CALL ABORT()
  ENDIF
ENDIF

#ifdef VERBOSE
WRITE(*, '()')
WRITE(*, '("Using NCIN file """,A,"""")') TRIM(cfn_ncin_res)
WRITE(*, '("Using AUX file """,A,"""")') TRIM(cfn_ncin_aux)
WRITE(*, '("Creating NCOUT file """,A,"""")') TRIM(cfn_ncout_res)
WRITE(*, '()')
#endif


                                    ! Open NCIN file
                                    ! ==============
istatus = NF_OPEN(cfn_ncin_res, NF_NOWRITE, ncid_file_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! Query characteristics of the 'lev' dimension
istatus = NF_INQ_DIMID(ncid_file_in, 'lev', ncid_dim_lev_in)
IF (istatus /= NF_NOERR) THEN
  IF (istatus == NF_EBADDIM) THEN
    ncid_dim_lev_in = -1
  ELSE
    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-5))
  ENDIF
ENDIF

IF (ncid_dim_lev_in /= -1) THEN
  istatus = NF_INQ_DIMLEN(ncid_file_in, ncid_dim_lev_in, nlen_dim_lev)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
ELSE
  nlen_dim_lev = 0
ENDIF



                                    ! Query characteristics of the X dimension
istatus = NF_INQ_DIMID(ncid_file_in, cname_dim_x_in, ncid_dim_x_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_INQ_DIMLEN(ncid_file_in, ncid_dim_x_in, nlen_dim_x_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


                                    ! Query characteristics of the Y dimension
istatus = NF_INQ_DIMID(ncid_file_in, cname_dim_y_in, ncid_dim_y_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_INQ_DIMLEN(ncid_file_in, ncid_dim_y_in, nlen_dim_y_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


                                    ! Query characteristics of the 'time' dimension
istatus = NF_INQ_DIMID(ncid_file_in, 'time', ncid_dim_time_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_INQ_DIMLEN(ncid_file_in, ncid_dim_time_in, nlen_dim_time)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))



                                    ! Open AUX file
                                    ! =============
istatus = NF_OPEN(cfn_ncin_aux, NF_NOWRITE, ncid_file_aux)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! Query characteristics of the X dimension
istatus = NF_INQ_DIMID(ncid_file_aux, 'col', ncid_dim_col_aux)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_INQ_DIMLEN(ncid_file_aux, ncid_dim_col_aux, nlen_dim_col)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! Get the var ID of the dimension
                                    ! variable of the 'col' dimension
istatus = NF_INQ_VARID(ncid_file_aux, cname_dim_col_aux, ncid_var_col_aux)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

#ifdef DEBUG
WRITE(*,'(" ""col"" dimension length: ", I0)') nlen_dim_col
WRITE(*,'()')
#endif


                                    ! Read in conversion tables

ALLOCATE(i4col(nlen_dim_col))
ALLOCATE(j4col(nlen_dim_col))

istatus = NF_INQ_VARID(ncid_file_aux, 'i4col', ncid_var_i4col)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_GET_VAR_INT(ncid_file_aux, ncid_var_i4col, i4col)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_INQ_VARID(ncid_file_aux, 'j4col', ncid_var_j4col)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_GET_VAR_INT(ncid_file_aux, ncid_var_j4col, j4col)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))




! Create NCOUT file
                                    ! Create Output file
                                    ! ==================
istatus = NF_CREATE(cfn_ncout_res, NF_CLOBBER, ncid_file_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! Define dimension 'lev'
IF (nlen_dim_lev > 0) THEN
  istatus = NF_DEF_DIM(ncid_file_out, 'lev', nlen_dim_lev, ncid_dim_lev_out)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
ENDIF

                                    ! Define dimension 'col'
istatus = NF_DEF_DIM(ncid_file_out, cname_dim_x_aux, nlen_dim_col, ncid_dim_col_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


                                    ! Copy dimension variable 'col' from AUX to OUT
istatus = NF_COPY_VAR(ncid_file_aux, ncid_var_col_aux, ncid_file_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_INQ_VARID(ncid_file_out, cname_dim_col_aux, ncid_var_col_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


                                    ! Put back file to "define" mode -- NF_COPY_VAR
                                    ! has ended this.
istatus = NF_REDEF(ncid_file_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


                                    ! Copy over additional variables, if any
IF (LEN_TRIM(cname_vars_extra) > 0) THEN
# ifdef DEBUG
  WRITE(*, '("Copying extra variables from the AUX to the NCOUT file")')
# endif
  CALL DELIMIT_STRING_TOKENS(cname_vars_extra, ' ,;:', iptr_begin, nptr_len)

  DO j = 1, SIZE(iptr_begin)

    i1 = iptr_begin(j)
    i2 = iptr_begin(j) + nptr_len(j)-1
    cname_extra = cname_vars_extra(i1:i2)

    IF (cname_extra == cname_dim_x_aux) THEN
#     ifdef VERBOSE
      WRITE(*, '("Skipping X dimension variable """, A, """")') TRIM(cname_extra)
#     endif
      CYCLE
    ENDIF

    IF (cname_extra == cname_dim_y_aux) THEN
#     ifdef VERBOSE
      WRITE(*, '("Skipping Y dimension variable """, A, """")') TRIM(cname_extra)
#     endif
      CYCLE
    ENDIF

#   ifdef DEBUG
    WRITE(*, '("Querying variable """, A, """ in the AUX file")', ADVANCE='NO') TRIM(cname_extra)
#   endif
    istatus = NF_INQ_VARID(ncid_file_aux, cname_extra, ncid_var)
    IF (istatus /= NF_NOERR) THEN
#     ifdef DEBUG
      WRITE(*,'(" - not found, skipping.")')
#     else
      CONTINUE
#     endif
    ELSE
#     ifdef DEBUG
      WRITE(*,'(" - found, copying.")')
#     endif
      istatus = NF_COPY_VAR(ncid_file_aux, ncid_var, ncid_file_out)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
                                    ! Put back the OUT file into define mode
                                    ! NF_COPY_VAR ends the define mode.
      istatus = NF_REDEF(ncid_file_out)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
    ENDIF

  ENDDO

# ifdef DEBUG
ELSE

  WRITE(*, '("No extra variables to copy from the AUX to the NCOUT file")')
# endif

ENDIF


! For each variable that
!  - is not a dimension variable (ndim = 1, varname = dimname)
!  - has 'col' among its dimensions
!  - create mapped variable where 'X,Y' couple of dimensions is replaced
!    by the single 'col' dimension, with the 'time' dimension omitted
!  - copy over attributes from original file
! extract time slice
! convert from 'X,Y'-ordering to 'col'-ordering


! Variable in the NCIN file may be of the following shapes:
! and are remapped as follows
! 0 scalar           --> scalar    -- (0) copy
! 1 (lev)            --> (lev)     -- (0) copy
! 2 (x,y)            --> (col)     -- (2) remap and write
! 3 (time)           --> scalar    -- (3) extract and write
! 4 (lev, time)      --> (lev)     -- (4) extract and write
! 5 (x,y, time)      --> (col)     -- (5) extract and remap first and second index

                                    ! Get number of dims 
istatus = NF_INQ_NDIMS(ncid_file_in, ndims_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


ALLOCATE(cname_dims_in(ndims_in))

DO i = 1, ndims_in

  istatus = NF_INQ_DIMNAME(ncid_file_in, i, cname_dims_in(i))
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

ENDDO

                                    ! Get number of vars and classify them.
istatus = NF_INQ_NVARS(ncid_file_in, nvars_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

#ifdef DEBUG
WRITE(*, '("Found ", I0, " variable definitions in NCIN file")') nvars_in
#endif

ALLOCATE(cname_vars_in(nvars_in))
ALLOCATE(ihow2processvars_in(nvars_in))
ALLOCATE(ncid_vars_out(nvars_in))
ALLOCATE(itype_vars_in(nvars_in))

ncid_vars_out(:) = -9999

DO j = 1, nvars_in

  istatus = NF_INQ_VARNAME(ncid_file_in, j, cname_var)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
  cname_vars_in(j) = TRIM(cname_var)

  istatus = NF_INQ_VARNDIMS(ncid_file_in, j, ndims)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

  istatus = NF_INQ_VARTYPE(ncid_file_in, j, itype)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
  itype_vars_in(j) = itype


  IF (cname_var == 'col') THEN
#   ifdef DEBUG
    WRITE(*, '("Skipping dimension variable ""col"" in the NCIN file")')
#   endif
    ihow2processvars_in(j) = 0      ! No processing required
    CYCLE
  ENDIF

  IF (cname_var == 'time') THEN
#   ifdef DEBUG
    WRITE(*, '("Skipping dimension variable ""time"" in the NCIN file")')
#   endif
    ihow2processvars_in(j) = 0      ! No processing required
    CYCLE
  ENDIF


# ifdef DEBUG
  WRITE(*, '("Analysing variable """, A, """ in the NCIN file")', ADVANCE='NO') TRIM(cname_var)
# endif


  SELECT CASE(ndims)

  CASE(0)
#   ifdef DEBUG
    WRITE(*,'(" - scalar variable, copying")')
#   endif
    istatus = NF_COPY_VAR(ncid_file_in, j, ncid_file_out)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
    istatus = NF_REDEF(ncid_file_out)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

    ihow2processvars_in(j) = 0    ! Done - no more processing required

  CASE(1)
    istatus = NF_INQ_VARDIMID(ncid_file_in, j, ncid_dim)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

    IF (cname_dims_in(ncid_dim) == cname_var) THEN

#     ifdef DEBUG
      WRITE(*,'(" - dimension variable, copying")')
#     endif
      istatus = NF_COPY_VAR(ncid_file_in, j, ncid_file_out)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      istatus = NF_REDEF(ncid_file_out)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      ihow2processvars_in(j) = 0    ! Done - no more processing required

    ELSE

      IF (ncid_dim == ncid_dim_lev_in) THEN

#       ifdef DEBUG
        WRITE(*,'(" - dimensions (lev), copying")')
#       endif
        istatus = NF_COPY_VAR(ncid_file_in, j, ncid_file_out)
        IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

        istatus = NF_REDEF(ncid_file_out)
        IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

        ihow2processvars_in(j) = 0    ! Done - no more processing required

      ELSEIF (ncid_dim == ncid_dim_col_in) THEN
#       ifdef DEBUG
        WRITE(*,'(" - dimensions (col), needs remapping [2]")')
#       endif
        istatus  = NF_DEF_VAR(ncid_file_out, TRIM(cname_var), itype, &
                                    2, (/ ncid_dim_x_out, ncid_dim_y_out /), &
                                    ncid_vars_out(j))
        IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        ihow2processvars_in(j) = 2    ! Schedule remapping

      ELSEIF (ncid_dim == ncid_dim_time_in) THEN
#       ifdef DEBUG
        WRITE(*,'(" - dimensions (time), needs extraction [3]")')
#       endif
        istatus  = NF_DEF_VAR(ncid_file_out, TRIM(cname_var), itype, &
                                    0, (/ (i,i=1,0) /), &
                                    ncid_vars_out(j))
        IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        ihow2processvars_in(j) = 3    ! Schedule extraction

      ELSE

        WRITE(jp_stderr,'(" - unknown DIM_ID ", I0, " - aborting")') ncid_dim
        CALL ABORT()
        
      ENDIF

    ENDIF

    
  CASE(2)

    istatus = NF_INQ_VARDIMID(ncid_file_in, j, ncid_dim2)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

    IF (ALL(ncid_dim2(:) == (/ ncid_dim_lev_in, ncid_dim_time_in /))) THEN

#     ifdef DEBUG
      WRITE(*,'(" - dimensions (lev,time), needs extraction [4]")')
#     endif
      istatus  = NF_DEF_VAR(ncid_file_out, TRIM(cname_var), itype, &
                                    1, (/ ncid_dim_lev_out /), &
                                    ncid_vars_out(j))
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

      ihow2processvars_in(j) = 4    ! Schedule extraction

    ELSEIF (ALL(ncid_dim2(:) == (/ ncid_dim_col_in, ncid_dim_time_in /))) THEN

#     ifdef DEBUG
      WRITE(*,'(" - dimensions (x,y), needs remapping [5]")')
#     endif
      istatus  = NF_DEF_VAR(ncid_file_out, TRIM(cname_var), itype, &
                                    2, (/ ncid_dim_x_out, ncid_dim_y_out /), &
                                    ncid_vars_out(j))
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

      ihow2processvars_in(j) = 5    ! Schedule extraction

    ELSE

        WRITE(jp_stderr,'(" - unknown DIM_IDs (", I0, ",", I0, ") - aborting")') ncid_dim2
        CALL ABORT()

    ENDIF


  CASE(3)

    istatus = NF_INQ_VARDIMID(ncid_file_in, j, ncid_dim3)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

    IF (ALL(ncid_dim3(:) == (/ ncid_dim_lev_in, ncid_dim_col_in, ncid_dim_time_in /))) THEN

#     ifdef DEBUG
      WRITE(*,'(" - dimensions (lev, x,y), needs extraction and remapping [6]")')
#     endif
      istatus  = NF_DEF_VAR(ncid_file_out, TRIM(cname_var), itype, &
                                    3, (/ ncid_dim_lev_out, ncid_dim_x_out, ncid_dim_y_out /), &
                                    ncid_vars_out(j))
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

      ihow2processvars_in(j) = 6    ! Schedule extraction and remapping

    ELSE

      WRITE(jp_stderr,'(" - unknown DIM_IDs (", I0, ",", I0, ",", I0, ") - aborting")') ncid_dim3
      CALL ABORT()

    ENDIF

  CASE DEFAULT

    WRITE(jp_stderr,'(" - arrays with ", I0, " cannot yet be processed - aborting")') ndims
    CALL ABORT()

  END SELECT


  SELECT CASE(ihow2processvars_in(j))
  CASE(0)
    CYCLE

  CASE(2:6)
                                    ! Copy over the attributes
    istatus = NF_INQ_VARNATTS(ncid_file_in, j, natts)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

    DO k = 1, natts

      istatus = NF_INQ_ATTNAME(ncid_file_in, j, k, cname_att)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      istatus = NF_COPY_ATT(ncid_file_in, j, TRIM(cname_att), ncid_file_out, ncid_vars_out(j))
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

    ENDDO

                                    ! Add _FillValue attribute
    SELECT CASE(itype_vars_in(j))
    CASE(NF_INT)
      istatus = NF_PUT_ATT_INT(ncid_file_out, ncid_vars_out(j), &
                                    '_FillValue', NF_INT, 1, (/ NF_FILL_INT /))
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
    CASE(NF_REAL)
      istatus = NF_PUT_ATT_REAL(ncid_file_out, ncid_vars_out(j), &
                                    '_FillValue', NF_REAL, 1, (/ NF_FILL_REAL /))
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
    CASE(NF_DOUBLE)
      istatus = NF_PUT_ATT_DOUBLE(ncid_file_out, ncid_vars_out(j), &
                                    '_FillValue', NF_DOUBLE, 1, (/ NF_FILL_DOUBLE /))
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
    END SELECT

  CASE DEFAULT

    WRITE(jp_stderr,'("Unknown scheduled action [", I0, "] - aborting")') ihow2processvars_in(j)
    CALL ABORT()

  END SELECT

ENDDO

                                    ! End define mode
istatus = NF_ENDDEF(ncid_file_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! Select time slice
IF ((i_timeslice < 1) .OR. (i_timeslice > nlen_dim_time)) THEN
  itime = nlen_dim_time
ELSE
  itime = i_timeslice
ENDIF


DO ivar = 1, nvars_in

  SELECT CASE(ihow2processvars_in(ivar))

  CASE(0)                           ! Already copied - done!

    CYCLE


  CASE(2)                           ! (x,y) --> (col) - (2) remap and write

    SELECT CASE(itype_vars_in(ivar))

    CASE(NF_INT)

      ALLOCATE(iarr2in(nlen_dim_x, nlen_dim_y))

      istatus = NF_GET_VAR_INT(ncid_file_in, ivar, iarr1_in)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      ALLOCATE(iarr1_out(nlen_dim_col))
      iarr1_out(:,:) = NF_FILL_INT

      DO n = 1, nlen_dim_col
        i = i4col(n)
        j = j4col(n)
        iarr1_out(n) = iarr2_in(i,j)
      ENDDO

      istatus = NF_PUT_VAR_INT(ncid_file_out, ncid_vars_out(ivar), iarr1_out)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      DEALLOCATE(iarr2_in)
      DEALLOCATE(iarr1_out)

    CASE(NF_DOUBLE)

      ALLOCATE(darr2_in(nlen_dim_x, nlen_dim_y))

      istatus = NF_GET_VAR_DOUBLE(ncid_file_in, ivar, darr2_in)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      ALLOCATE(darr1_out(nlen_dim_col))
      darr1_out(:,:) = NF_FILL_DOUBLE

      DO n = 1, nlen_dim_col
        i = i4col(n)
        j = j4col(n)
        darr1_out(n) = darr2_in(i,j)
      ENDDO

      istatus = NF_PUT_VAR_DOUBLE(ncid_file_out, ncid_vars_out(ivar), darr1_out)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      DEALLOCATE(darr2_in)
      DEALLOCATE(darr1_out)

    CASE DEFAULT

      WRITE(jp_stderr, '("Case for NetCDF type ", I0, " not yet covered")') itype_vars_in(ivar)
      WRITE(jp_stderr, '("This program needs an extension - aborting!")')
      CALL ABORT()

    END SELECT


  CASE(3)                           ! (time) --> scalar - (3) extract and write

    istart(1) = itime
    ncount(1) = 1

    SELECT CASE(itype_vars_in(ivar))

    CASE(NF_INT)

      istatus = NF_GET_VARA_INT(ncid_file_in, ivar, istart(1:1), ncount(1:1), iscalar)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      istatus = NF_PUT_VAR_INT(ncid_file_out, ncid_vars_out(ivar), iscalar)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

    CASE(NF_DOUBLE)

      istatus = NF_GET_VARA_DOUBLE(ncid_file_in, ivar, istart(1:1), ncount(1:1), dscalar)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      istatus = NF_PUT_VAR_DOUBLE(ncid_file_out, ncid_vars_out(ivar), dscalar)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

    CASE DEFAULT

      WRITE(jp_stderr, '("Case for NetCDF type ", I0, " not yet covered")') itype_vars_in(ivar)
      WRITE(jp_stderr, '("This program needs an extension - aborting!")')
      CALL ABORT()

    END SELECT

  CASE(4)                           ! (lev, time) --> (lev) - (4) extract and write

    istart(1:2) = (/            1, itime /)
    ncount(1:2) = (/ nlen_dim_lev,     1 /)

    SELECT CASE(itype_vars_in(ivar))

    CASE(NF_INT)

      ALLOCATE(iarr1_in(nlen_dim_lev))

      istatus = NF_GET_VARA_INT(ncid_file_in, ivar, istart(1:2), ncount(1:2), iarr1_in)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      istatus = NF_PUT_VAR_INT(ncid_file_out, ncid_vars_out(ivar), iarr1_in)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      DEALLOCATE(iarr1_in)


    CASE(NF_DOUBLE)

      ALLOCATE(darr1_in(nlen_dim_lev))

      istatus = NF_GET_VARA_DOUBLE(ncid_file_in, ivar, istart(1:2), ncount(1:2), darr1_in)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      istatus = NF_PUT_VAR_DOUBLE(ncid_file_out, ncid_vars_out(ivar), darr1_in)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      DEALLOCATE(darr1_in)

    CASE DEFAULT

      WRITE(jp_stderr, '("Case for NetCDF type ", I0, " not yet covered")') itype_vars_in(ivar)
      WRITE(jp_stderr, '("This program needs an extension - aborting!")')
      CALL ABORT()

    END SELECT


  CASE(5)                           ! (x,y, time) --> (col) - (5) extract and remap first index

    istart(1:2) = (/              1,              1, itime /)
    ncount(1:2) = (/ nlen_dim_x_aux, nlen_dim_y_aux,     1 /)

    SELECT CASE(itype_vars_in(ivar))

    CASE(NF_INT)

      ALLOCATE(iarr2_in(nlen_dim_x_aux, nlen_dim_y_aux))

      istatus = NF_GET_VARA_INT(ncid_file_in, ivar, istart(1:3), ncount(1:3), iarr2_in)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      ALLOCATE(iarr1_out(nlen_dim_col))
      iarr1_out(:) = NF_FILL_INT

      DO n = 1, nlen_dim_col
        i = i4col(n)
        j = j4col(n)
        iarr1_out(n) = iarr2_in(i,j)
      ENDDO

      istatus = NF_PUT_VAR_INT(ncid_file_out, ncid_vars_out(ivar), iarr1_out)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      DEALLOCATE(iarr2_in)
      DEALLOCATE(iarr1_out)


    CASE(NF_DOUBLE)

      ALLOCATE(darr2_in(nlen_dim_x_aux, nlen_dim_y_aux))

      istatus = NF_GET_VARA_DOUBLE(ncid_file_in, ivar, istart(1:3), ncount(1:3), darr2_in)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      ALLOCATE(darr1_out(nlen_dim_col))
      darr1_out(:,:) = NF_FILL_DOUBLE

      DO n = 1, nlen_dim_col
        i = i4col(n)
        j = j4col(n)
        darr1_out(n) = darr2_in(i,j)
      ENDDO

      istatus = NF_PUT_VAR_DOUBLE(ncid_file_out, ncid_vars_out(ivar), darr2_out)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      DEALLOCATE(darr2_in)
      DEALLOCATE(darr1_out)

    CASE DEFAULT

      WRITE(jp_stderr, '("Case for NetCDF type ", I0, " not yet covered")') itype_vars_in(ivar)
      WRITE(jp_stderr, '("This program needs an extension - aborting!")')
      CALL ABORT()

    END SELECT

  CASE(6)                           ! (lev, x,y) --> (lev, col) - (6) extract and remap second index

    SELECT CASE(itype_vars_in(ivar))

    CASE(NF_INT)

      ALLOCATE(iarr3_in(nlen_dim_lev, nlen_dim_x, nlen_dim_y))

      istatus = NF_GET_VAR_INT(ncid_file_in, ivar, iarr3_in)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      ALLOCATE(iarr2_out(nlen_dim_lev, nlen_dim_col))
      iarr2_out(:,:) = NF_FILL_INT

      DO n = 1, nlen_dim_col
        i = i4col(n)
        j = j4col(n)
        iarr2_out(:, n) = iarr3_in(:, i,j)
      ENDDO

      istatus = NF_PUT_VAR_INT(ncid_file_in, ncid_vars_out(ivar), iarr2_out)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      DEALLOCATE(iarr3_in)
      DEALLOCATE(iarr2_out)


    CASE(NF_DOUBLE)

      ALLOCATE(darr3_in(nlen_dim_lev, nlen_dim_x, nlen_dim_y))

      istatus = NF_GET_VAR_DOUBLE(ncid_file_in, ivar, darr2_in)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      ALLOCATE(darr2_out(nlen_dim_lev, nlen_dim_col))
      darr2_out(:,:) = NF_FILL_DOUBLE

      DO n = 1, nlen_dim_col
        i = i4col(n)
        j = j4col(n)
        darr2_out(:, n) = darr3_in(:, i,j)
      ENDDO

      istatus = NF_PUT_VAR_DOUBLE(ncid_file_out, ncid_vars_out(ivar), darr2_out)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      DEALLOCATE(darr3_in)
      DEALLOCATE(darr2_out)

    CASE DEFAULT

      WRITE(jp_stderr, '("Case for NetCDF type ", I0, " not yet covered")') itype_vars_in(ivar)
      WRITE(jp_stderr, '("This program needs an extension - aborting!")')
      CALL ABORT()

    END SELECT

  END SELECT

ENDDO

                                    ! Close files
istatus = NF_CLOSE(ncid_file_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_CLOSE(ncid_file_aux)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_CLOSE(ncid_file_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))



!========!
 CONTAINS
!========!

!-----------------------------------------------------------------------
 SUBROUTINE HANDLE_NCERRORS(istatus, whatfile, iline)
!-----------------------------------------------------------------------

IMPLICIT NONE

#include <netcdf.inc>

INTEGER, INTENT(IN) :: istatus
CHARACTER(LEN=*)    :: whatfile
INTEGER, INTENT(IN) :: iline

IF (istatus /= NF_NOERR) THEN
  IF (iline > 0) THEN
    WRITE(jp_stderr,"('[',A,':',I0,']: ', A)") &
      TRIM(whatfile), iline, TRIM(NF_STRERROR(istatus))
  ELSE
    WRITE(jp_stderr,"('[',A,':???]: ', A)") &
      TRIM(whatfile), TRIM(NF_STRERROR(istatus))
  ENDIF

  PRINT *, 'NetCDF error detected; aborting.'

  CALL ABORT()

ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE HANDLE_NCERRORS
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE DELIMIT_STRING_TOKENS(c_in, c_separators, i_begin, n_len)
!-----------------------------------------------------------------------

CHARACTER(LEN=*), INTENT(IN)   :: c_in
CHARACTER(LEN=*), INTENT(IN)   :: c_separators
INTEGER, DIMENSION(:), POINTER :: i_begin
INTEGER, DIMENSION(:), POINTER :: n_len


INTEGER, PARAMETER :: jp_stderr = 0
CHARACTER(LEN=LEN(c_in)) :: c_tmp

INTEGER :: i, ii, i1, i2, i3, n_words

IF(LEN_TRIM(c_separators) == 0) THEN
  WRITE(jp_stderr, '("[DELIMIT_STRING_TOKENS]: No separators provided -- aborting")')
  CALL ABORT()
ENDIF

c_tmp = ' '
n_words = 0
DO i = 1, LEN_TRIM(c_in)
  c_tmp(i:i) = ' '
  ii = INDEX(c_separators, c_in(i:i))
  IF(ii > 0) THEN
    IF (i > 1) THEN
      SELECT CASE(c_tmp(i-1:i-1))
      CASE(' ')
        CYCLE
      CASE('1')
        c_tmp(i-1:i-1) = '3'  ! one-letter word on previous character
        n_words = n_words + 1
      CASE('2')
        n_words = n_words + 1
      END SELECT
    ELSE
      CYCLE
    ENDIF  
  ELSE
    IF (i == 1) THEN
      c_tmp(i:i) = '1'
    ELSE
      SELECT CASE(c_tmp(i-1:i-1))
      CASE(' ')           ! SPC in pos=i-1
        c_tmp(i:i) = '1'  ! => new word starts at pos=i
      CASE('1')           ! new word has begun at pos=i-1
        c_tmp(i:i) = '2'  ! => provisionally end it at pos=i
      CASE('2')           ! word ended provisionally at pos=i-1
        c_tmp(i-1:i-1) = ' '
        c_tmp(i:i) = '2'  ! => report end to pos=i
      END SELECT
    ENDIF
  END IF
  IF (i == LEN_TRIM(c_in)) THEN
    SELECT CASE(c_tmp(i:i))
    CASE('1')
      c_tmp(i:i) = '3'  ! one-letter word on last character
      n_words = n_words + 1
    CASE('2')
      n_words = n_words + 1
    END SELECT
  ENDIF
END DO


IF(n_words > 0) THEN
  ALLOCATE(i_begin(n_words))
  ALLOCATE(n_len(n_words))

  ii = 1
  DO i = 1, n_words
    i1 = INDEX(c_tmp(ii:),'1')
    i3 = INDEX(c_tmp(ii:),'3')

    IF(i3 > 0) THEN
      IF(i1 > 0) THEN 
        IF(i1 < i3) THEN
          i_begin(i) = i1 + ii - 1
          i2 = INDEX(c_tmp(ii:),'2')
          n_len(i) = i2 - i1 + 1
          ii = ii + i2
        ELSE
          i_begin(i) = i3 + ii - 1
          n_len(i) = 1
          ii = ii + i3
        ENDIF
      ELSE
        i_begin(i) = i3 + ii - 1
        n_len(i) = 1
        ii = ii + i3
      ENDIF
    ELSE
      i_begin(i) = i1 + ii - 1
      i2 = INDEX(c_tmp(ii:),'2')
      n_len(i) = i2 - i1 + 1
      ii = ii + i2
    ENDIF
  ENDDO

ELSE

  NULLIFY(i_begin)
  NULLIFY(n_len)

ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE DELIMIT_STRING_TOKENS
!-----------------------------------------------------------------------

END
