!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#ifdef CFN_THISFILE
#undef CFN_THISFILE
#endif
#define CFN_THISFILE "medintegralev.F90"
#ifndef __LINE__
#define __LINE__ 0
#endif

#ifdef DEBUG
#ifndef VERBOSE
#define VERBOSE
#endif
#endif
!=======================================================================
 PROGRAM MEDINTEGRALEV
!=======================================================================


USE MOD_UTICOMMON,                  ONLY: ABORT_EXECUTION, jp_stderr,  &
                                          c_dirsep, jp_lmaxpathname


IMPLICIT NONE


CHARACTER(LEN=*), PARAMETER    :: cfn_nml          = "medintegralev.nml"


CHARACTER(LEN=*), PARAMETER    :: cdn_nml_defaults = "/tmp"
CHARACTER(LEN=*), PARAMETER    :: cfn_nml_defaults = "medintegralev_def.nml"
CHARACTER(LEN=jp_lmaxpathname) :: cpn_nml_defaults


CHARACTER(LEN=jp_lmaxpathname) :: cfn_ncin_res
CHARACTER(LEN=jp_lmaxpathname) :: cfn_ncin_reaclay
CHARACTER(LEN=jp_lmaxpathname) :: cfn_ncout_res

#include <netcdf.inc>

INTEGER :: ncid_file_in
INTEGER :: ncid_file_reaclay
INTEGER :: ncid_file_out

INTEGER :: ncid_dim_lev_in,   ncid_var_lev_in,  nlen_dim_lev_in
INTEGER :: ncid_dim_vtx_in
INTEGER :: ncid_dim_col_in,   ncid_var_col_in,  nlen_dim_col_in
INTEGER :: ncid_dim_time_in,  ncid_var_time_in, nlen_dim_time_in

INTEGER :: ncid_dim_lev_reaclay,   ncid_var_lev_reaclay,  nlen_dim_lev_reaclay
INTEGER :: ncid_dim_col_reaclay,                          nlen_dim_col_reaclay
INTEGER :: ncid_dim_time_reaclay,                         nlen_dim_time_reaclay

INTEGER :: ncid_dim_col_out,  ncid_var_col_out,  nlen_dim_col_out
INTEGER :: ncid_dim_time_out, ncid_var_time_out, nlen_dim_time_out

INTEGER :: ncid_var_xzdn, ndim_var_xzdn
INTEGER :: ncid_var_xphi, ndim_var_xphi


INTEGER :: ncid_dim

INTEGER, DIMENSION(3) :: istart, ncount
INTEGER, DIMENSION(2) :: ncid_dim2
INTEGER, DIMENSION(3) :: ncid_dim3


INTEGER :: ndims_in, nvars_in
INTEGER :: ndims, natts, itype
CHARACTER(LEN=NF_MAX_NAME) :: cname_var
CHARACTER(LEN=NF_MAX_NAME) :: cname_att
CHARACTER(LEN=255)         :: cval_att
INTEGER :: nlen_att
INTEGER :: istatus
INTEGER :: i, j, k, ivar, irec, ix
LOGICAL :: l_exists, l_ncin_is_reaclay
CHARACTER(LEN=10) :: cname_filetype
INTEGER :: iv_file_in_fmt, iv_file_r_fmt
DOUBLE PRECISION :: thetatop, thetabot
DOUBLE PRECISION :: ageom_swiloc

INTEGER :: ndi_w2t_in, ndi_t2b_in, ndi_w2t_r, ndi_t2b_r
INTEGER :: ndn_w2s_in, ndn_s2b_in, ndn_w2s_r, ndn_s2b_r

INTEGER :: idnw, idnt, idnb

LOGICAL :: l_nb_dbl_tofind, l_nb_rcl_tofind

CHARACTER(LEN=NF_MAX_NAME), DIMENSION(:),   ALLOCATABLE :: cname_dims_in
CHARACTER(LEN=NF_MAX_NAME), DIMENSION(:),   ALLOCATABLE :: cname_vars_in
INTEGER,                    DIMENSION(:),   ALLOCATABLE :: ihow2processvars_in
INTEGER,                    DIMENSION(:),   ALLOCATABLE :: ncid_vars_out
INTEGER,                    DIMENSION(:),   ALLOCATABLE :: itype_vars_in
CHARACTER(LEN=2),           DIMENSION(:),   ALLOCATABLE :: cphasid_vars_in


INTEGER :: n2, n3
DOUBLE PRECISION, DIMENSION(:,:,:), ALLOCATABLE :: xgeom_xzdn
DOUBLE PRECISION, DIMENSION(:,:,:), ALLOCATABLE :: xgeom_xphi

DOUBLE PRECISION, DIMENSION(:,:,:), ALLOCATABLE :: xgeom_vxzdn_dbl_c
DOUBLE PRECISION, DIMENSION(:,:,:), ALLOCATABLE :: xgeom_vxzdn_rcl_c

DOUBLE PRECISION, DIMENSION(:),     ALLOCATABLE :: xphi
DOUBLE PRECISION, DIMENSION(:),     ALLOCATABLE :: vxzdn_dbl_c
DOUBLE PRECISION, DIMENSION(:),     ALLOCATABLE :: vxzdn_rcl_c

DOUBLE PRECISION, DIMENSION(:,:),   ALLOCATABLE :: darr_lc
DOUBLE PRECISION, DIMENSION(:,:),   ALLOCATABLE :: darr_ct

DOUBLE PRECISION :: dfillvalue

INTEGER, PARAMETER :: jp_lunit  = 100
CHARACTER(LEN=*), PARAMETER :: cfn_thisfile = CFN_THISFILE


#ifdef VERBOSE
WRITE(*,'()')
WRITE(*,'("This is MEDINTEGRALEV")')
WRITE(*,'("=====================")')
WRITE(*,'()')
WRITE(*,'("MEDINTEGRALEV integrates the results stored in MEDUSA NetCDF files")')
WRITE(*,'("along the vertical (lev) dimension, for each ''time'' record and each")')
WRITE(*,'("column included.")')
WRITE(*,'()')
#endif


CALL GET_MENU(cfn_ncin_res, cfn_ncin_reaclay, cfn_ncout_res)


                                    ! Check validity of filenames
IF (cfn_ncin_res == "/dev/null") THEN
  WRITE(jp_stderr, '("NetCDF IN file missing - Aborting!")')
  CALL ABORT_EXECUTION()
ELSE
  INQUIRE(FILE=cfn_ncin_res, EXIST=l_exists)
  IF (.NOT.l_exists) THEN
    WRITE(jp_stderr, '("""", A, """: no such file - Aborting!")') TRIM(cfn_ncin_res)
    CALL ABORT_EXECUTION()
  ENDIF
ENDIF


IF (cfn_ncin_reaclay == "/dev/null") THEN
  WRITE(jp_stderr, '("NetCDF REACLAY file missing: checking if IN file is a REACLAY file...")')
  istatus = NF_OPEN(cfn_ncin_res, NF_NOWRITE, ncid_file_in)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

  l_ncin_is_reaclay = .FALSE.
                                    ! Test if there is a global 'file_type' attribute
  istatus = NF_INQ_ATTLEN(ncid_file_in, NF_GLOBAL, 'file_type', nlen_att)
  IF (istatus /= NF_NOERR) THEN
    IF (istatus == NF_ENOTATT) THEN
      WRITE(jp_stderr, '(" NetCDF IN file does not have a ""file_type"" attribute")')
    ELSE
      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
    ENDIF
  ELSE
    istatus = NF_GET_ATT_TEXT(ncid_file_in, NF_GLOBAL, 'file_type', cname_filetype)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

    WRITE(jp_stderr, '(" NetCDF IN file has file_type=""",A,"""")') cname_filetype(1:nlen_att)
    IF (cname_filetype(1:nlen_att) /= 'REACLAY') THEN
      WRITE(jp_stderr, '(" NetCDF IN file is not a REACLAY file - Aborting!")')
      CALL ABORT_EXECUTION()
    ELSE
      WRITE(jp_stderr, '(" Using NetCDF IN file as the REACLAY file.")')
      l_ncin_is_reaclay = .TRUE.
      cfn_ncin_reaclay = cfn_ncin_res
    ENDIF
  ENDIF

  IF (.NOT. l_ncin_is_reaclay) THEN
                                    ! 'file_type' attribute may be missing as it
                                    ! was only introduced lately
                                    ! Check for 'xgeom_dbswi' and 'xgeom_phi' variables
    istatus = NF_INQ_VARID(ncid_file_in, 'xgeom_dbswi', ncid_var_xzdn)
    IF (istatus /= NF_NOERR) THEN
      IF (istatus == NF_ENOTVAR) THEN
        WRITE(jp_stderr, '(" NetCDF IN file does not contain a ""xgeom_dbswi"" variable.")')
        WRITE(jp_stderr, '(" It cannot be a REACLAY file - Aborting!")')
        CALL ABORT_EXECUTION()
      ELSE
        CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-7))
      ENDIF
    ELSE
      WRITE(jp_stderr, '(" NetCDF IN file contains a ""xgeom_dbswi"" variable")')
      istatus = NF_INQ_VARID(ncid_file_in, 'xgeom_phi', ncid_var_xphi)
      IF (istatus /= NF_NOERR) THEN
        IF (istatus == NF_ENOTVAR) THEN
          WRITE(jp_stderr, '(" NetCDF IN file does not contain a ""xgeom_phi"" variable.")')
          WRITE(jp_stderr, '(" It cannot be a REACLAY file - Aborting!")')
          CALL ABORT_EXECUTION()
        ELSE
          CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-7))
        ENDIF
      ELSE
        WRITE(jp_stderr, '(" NetCDF IN file contains a ""xgeom_phi"" variable")')
        WRITE(jp_stderr, '(" and appears to be a REACLAY file.")')
        l_ncin_is_reaclay = .TRUE.
        cfn_ncin_reaclay = cfn_ncin_res
      ENDIF
    ENDIF
  ENDIF

  istatus = NF_CLOSE(ncid_file_in)
  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

  IF (.NOT. l_ncin_is_reaclay) THEN
    WRITE(jp_stderr, '(" Could not conclude whether NetCDF IN file is a REACLAY file - Aborting!")')
    CALL ABORT_EXECUTION()
  ENDIF

ELSE

  INQUIRE(FILE=cfn_ncin_reaclay, EXIST=l_exists)
  IF (.NOT.l_exists) THEN
    WRITE(jp_stderr, '("""", A, """: no such file - Aborting!")') TRIM(cfn_ncin_reaclay)
    CALL ABORT_EXECUTION()
  ENDIF
ENDIF

IF (cfn_ncout_res == "/dev/null") THEN
# ifdef VERBOSE
  WRITE(*, '("Missing name for the NetCDF OUT file")')
# endif
  ix = INDEX(cfn_ncin_res, '.nc', BACK=.TRUE.)
  IF (ix > 1) THEN
    cfn_ncout_res = cfn_ncin_res(1:ix-1) // '_intlev.nc'
#   ifdef VERBOSE
    WRITE(*, '(" switching to default name """, A, """")') TRIM(cfn_ncout_res)
#   endif
  ELSE
#   ifdef VERBOSE
    WRITE(*, '(" unable to generate name for NetCDF OUT file - Aborting!")')
#   endif
    CALL ABORT_EXECUTION()
  ENDIF
ENDIF

#ifdef VERBOSE
WRITE(*, '()')
WRITE(*, '("Using NCIN file """,A,"""")')     TRIM(cfn_ncin_res)
WRITE(*, '("Using REACLAY file """,A,"""")')  TRIM(cfn_ncin_reaclay)
WRITE(*, '("Creating NCOUT file """,A,"""")') TRIM(cfn_ncout_res)
WRITE(*, '()')
#endif


                                    ! Open NCIN file
                                    ! ==============

istatus = NF_OPEN(cfn_ncin_res, NF_NOWRITE, ncid_file_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! Query file format version
istatus = NF_GET_ATT_INT(ncid_file_in, NF_GLOBAL, 'format_version', iv_file_in_fmt)
IF (istatus /= NF_NOERR)  THEN
  IF (istatus == NF_ENOTATT) THEN   ! Attribute not found,
    iv_file_in_fmt = 1              ! use default version 1!
  ELSE
                                    ! Other error: process it!
    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-6))
  ENDIF
ENDIF

SELECT CASE(iv_file_in_fmt)
CASE(1)
  WRITE(jp_stderr, '("File format version 1 not supported -- aborting!")')
  CALL ABORT_EXECUTION()

CASE(2, 3)
# ifdef VERBOSE
  WRITE(jp_stderr, '("NCIN file has format version ", I0)') iv_file_in_fmt
  WRITE(jp_stderr, '()')
# else
  CONTINUE
# endif

CASE DEFAULT
  WRITE(jp_stderr, '("File format version ", I0, " cannot (yet) be handled -- aborting!")') iv_file_in_fmt
  CALL ABORT_EXECUTION()

END SELECT

                                    ! Query characteristics of the 'lev' dimension
istatus = NF_INQ_DIMID(ncid_file_in, 'lev', ncid_dim_lev_in)
IF (istatus /= NF_NOERR) THEN

  IF (istatus == NF_EBADDIM) THEN   ! This can also happen with if there is only a 'vtx'
                                    ! dimension, (FLX file), which we do not (yet) integrate.
    WRITE(*, '("File """,A,""" does not include a ""lev"" dimension.")') TRIM(cfn_ncin_res)
    istatus = NF_INQ_DIMID(ncid_file_in, 'vtx', ncid_dim_vtx_in)

    IF (istatus /= NF_NOERR) THEN

      IF (istatus == NF_EBADDIM) THEN ! No dimension called 'vtx'
        CONTINUE
      ELSE
        CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-5))
      ENDIF

    ELSE

      WRITE(*, '("It does include a ""vtx"" dimension")')
      WRITE(*, '(" Notice, however, that integration will only be carried out")')
      WRITE(*, '(" along a ""lev"" dimension.")')

    ENDIF

    WRITE(*, '(" There is nothing to integrate. Done!")')

    istatus = NF_CLOSE(ncid_file_in)
    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
    GOTO 9999

  ELSE

    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-38))

  ENDIF

ENDIF

istatus = NF_INQ_DIMLEN(ncid_file_in, ncid_dim_lev_in, nlen_dim_lev_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_INQ_VARID(ncid_file_in, 'lev', ncid_var_lev_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

SELECT CASE(iv_file_in_fmt)
CASE(2)
  istatus = NF_GET_ATT_INT(ncid_file_in, ncid_var_lev_in, 'nb_depth_intervals_DBL', ndi_w2t_in)
  IF (istatus /= NF_NOERR) THEN
    IF (istatus == NF_ENOTATT) THEN
      WRITE(jp_stderr, '("The ""lev"" variable in the NCIN file does not have a ""nb_depth_intervals_DBL"" attribute")')
      WRITE(jp_stderr, '("Please run fixncatt to fix this and rerun medintegralev after that. Aborting!")')
      CALL ABORT_EXECUTION()
    ELSE
      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-8))
    ENDIF
  ENDIF

  istatus = NF_GET_ATT_INT(ncid_file_in, ncid_var_lev_in, 'nb_depth_intervals_Reaclay', ndi_t2b_in)
  IF (istatus /= NF_NOERR)  THEN
    IF (istatus == NF_ENOTATT) THEN
      WRITE(jp_stderr, '("The ""lev"" variable in the NCIN file does not have a ""nb_depth_intervals_Reaclay"" attribute")')
      WRITE(jp_stderr, '("Please run fixncatt to fix this and rerun medintegralev after that. Aborting!")')
      CALL ABORT_EXECUTION()
    ELSE
      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-8))
    ENDIF
  ENDIF

CASE(3)
  istatus = NF_GET_ATT_INT(ncid_file_in, ncid_var_lev_in, 'nb_nodes_DBL', ndn_w2s_in)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

  istatus = NF_GET_ATT_INT(ncid_file_in, ncid_var_lev_in, 'nb_nodes_Reaclay', ndn_s2b_in)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

CASE DEFAULT
  WRITE(jp_stderr, '("And yet another part to be completed ... Aborting!")')
  CALL ABORT_EXECUTION()

END SELECT


istatus = NF_GET_ATT_DOUBLE(ncid_file_in, ncid_var_lev_in, 'thetatop', thetatop)
IF (istatus /= NF_NOERR)  THEN
  IF (istatus == NF_ENOTATT) THEN   ! can only happen with file format 2
    WRITE(jp_stderr, '("The ""lev"" variable in the NCIN file does not have a ""thetatop"" attribute")')
    WRITE(jp_stderr, '("Please run fixncatt to fix this and rerun medintegralev after that. Aborting!")')
    CALL ABORT_EXECUTION()
  ELSE
    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-8))
  ENDIF
ENDIF

istatus = NF_GET_ATT_DOUBLE(ncid_file_in, ncid_var_lev_in, 'thetabot', thetabot)
IF (istatus /= NF_NOERR)  THEN
  IF (istatus == NF_ENOTATT) THEN   ! can only happen with file format 2
                                    ! which can be fixed with fixncatt
    WRITE(jp_stderr, '("The ""lev"" variable in the NCIN file does not have a ""thetabot"" attribute")')
    WRITE(jp_stderr, '("Please run fixncatt to fix this and rerun medintegralev after that. Aborting!")')
    CALL ABORT_EXECUTION()
  ELSE
    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-8))
  ENDIF
ENDIF


                                    ! Query characteristics of the 'col' dimension
istatus = NF_INQ_DIMID(ncid_file_in, 'col', ncid_dim_col_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_INQ_DIMLEN(ncid_file_in, ncid_dim_col_in, nlen_dim_col_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_INQ_VARID(ncid_file_in, 'col', ncid_var_col_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


                                    ! Query characteristics of the 'time' dimension
istatus = NF_INQ_DIMID(ncid_file_in, 'time', ncid_dim_time_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_INQ_DIMLEN(ncid_file_in, ncid_dim_time_in, nlen_dim_time_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_INQ_VARID(ncid_file_in, 'time', ncid_var_time_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


#ifdef VERBOSE
WRITE(*, '("Dimension lengths of the NCIN file:")')
WRITE(*, '(" - ""lev"" dimension length = ", I0)') nlen_dim_lev_in
WRITE(*, '(" - ""col"" dimension length = ", I0)') nlen_dim_col_in
WRITE(*, '(" - ""time"" dimension length = ", I0)') nlen_dim_time_in
WRITE(*, '()')
#endif


IF (cfn_ncin_reaclay == cfn_ncin_res) THEN

  ncid_file_reaclay     = ncid_file_in
  ncid_dim_lev_reaclay  = ncid_dim_lev_in
  nlen_dim_lev_reaclay  = nlen_dim_lev_in

  ncid_dim_col_reaclay  = ncid_dim_col_in
  nlen_dim_col_reaclay  = nlen_dim_col_in

  ncid_dim_time_reaclay  = ncid_dim_time_in
  nlen_dim_time_reaclay  = nlen_dim_time_in

  iv_file_r_fmt          = iv_file_in_fmt

  SELECT CASE(iv_file_r_fmt)
  CASE(2)
    ndi_w2t_r              = ndi_w2t_in
    ndi_t2b_r              = ndi_t2b_in

  CASE(3)
    ndn_w2s_r              = ndn_w2s_in
    ndn_s2b_r              = ndn_s2b_in

  CASE DEFAULT
    WRITE(jp_stderr, '("And yet another part to be completed ... Aborting!")')
    CALL ABORT_EXECUTION()

  END SELECT

  l_nb_dbl_tofind        = .FALSE.
  l_nb_rcl_tofind        = .FALSE.

ELSE
                                    ! Open REACLAY file
                                    ! =================

  istatus = NF_OPEN(cfn_ncin_reaclay, NF_NOWRITE, ncid_file_reaclay)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! Query file format version
  istatus = NF_GET_ATT_INT(ncid_file_reaclay, NF_GLOBAL, 'format_version', iv_file_r_fmt)
  IF (istatus /= NF_NOERR)  THEN
    IF (istatus == NF_ENOTATT) THEN ! Attribute not found,
      iv_file_r_fmt = 1             ! use default version 1!
    ELSE
                                    ! Other error: process it!
      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-6))
    ENDIF
  ENDIF


  IF (iv_file_r_fmt /= iv_file_in_fmt) THEN
    WRITE(jp_stderr, '("NCIN and REACLAY files have different file format versions:")')
    WRITE(jp_stderr, '(" - NCIN file:    ", I0)') iv_file_in_fmt
    WRITE(jp_stderr, '(" - REACLAY file: ", I0)') iv_file_r_fmt
    WRITE(jp_stderr, '("This cannot be handled -- aborting!")')
    CALL ABORT_EXECUTION()
  ENDIF


                                    ! Query characteristics of the 'lev' dimension
                                    ! (REACLAY files always have a lev dimension)
  istatus = NF_INQ_DIMID(ncid_file_reaclay, 'lev', ncid_dim_lev_reaclay)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

  istatus = NF_INQ_DIMLEN(ncid_file_reaclay, ncid_dim_lev_reaclay, nlen_dim_lev_reaclay)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

  IF (nlen_dim_lev_reaclay /= nlen_dim_lev_in) THEN
    WRITE(jp_stderr, '("Incompatible ""lev"" dimension lengths in NCIN and REACLAY files:")')
    WRITE(jp_stderr, '(" - NCIN file:    ", I0)') nlen_dim_lev_in
    WRITE(jp_stderr, '(" - REACLAY file: ", I0)') nlen_dim_lev_reaclay
    WRITE(jp_stderr, '("Abort!")')
    CALL ABORT_EXECUTION()
  ENDIF

  istatus = NF_INQ_VARID(ncid_file_reaclay, 'lev', ncid_var_lev_reaclay)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

  SELECT CASE(iv_file_r_fmt)
  CASE(2)
    istatus = NF_GET_ATT_INT(ncid_file_reaclay, ncid_var_lev_reaclay, 'nb_depth_intervals_DBL', ndi_w2t_r)
    IF (istatus /= NF_NOERR) THEN
      IF (istatus == NF_ENOTATT) THEN
        l_nb_dbl_tofind = .TRUE.
      ELSE
        CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-5))
      ENDIF
    ELSE
      l_nb_dbl_tofind = .FALSE.
    ENDIF

    istatus = NF_GET_ATT_INT(ncid_file_reaclay, ncid_var_lev_reaclay, 'nb_depth_intervals_Reaclay', ndi_t2b_r)
    IF (istatus /= NF_NOERR) THEN
      IF (istatus == NF_ENOTATT) THEN
        l_nb_rcl_tofind = .TRUE.
      ELSE
        CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-5))
      ENDIF
    ELSE
      l_nb_rcl_tofind = .FALSE.
    ENDIF

  CASE(3)
    istatus = NF_GET_ATT_INT(ncid_file_reaclay, ncid_var_lev_reaclay, 'nb_nodes_DBL', ndn_w2s_r)
    IF (istatus /= NF_NOERR) THEN
      IF (istatus == NF_ENOTATT) THEN
        l_nb_dbl_tofind = .TRUE.
      ELSE
        CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-5))
      ENDIF
    ELSE
      l_nb_dbl_tofind = .FALSE.
    ENDIF

    istatus = NF_GET_ATT_INT(ncid_file_reaclay, ncid_var_lev_reaclay, 'nb_nodes_Reaclay', ndn_s2b_r)
    IF (istatus /= NF_NOERR) THEN
      IF (istatus == NF_ENOTATT) THEN
        l_nb_rcl_tofind = .TRUE.
      ELSE
        CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-5))
      ENDIF
    ELSE
      l_nb_rcl_tofind = .FALSE.
    ENDIF

  CASE DEFAULT
    WRITE(jp_stderr, '("And yet another part to be completed ... Aborting!")')
    CALL ABORT_EXECUTION()

  END SELECT

                                    ! Query characteristics of the 'col' dimension
  istatus = NF_INQ_DIMID(ncid_file_reaclay, 'col', ncid_dim_col_reaclay)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

  istatus = NF_INQ_DIMLEN(ncid_file_reaclay, ncid_dim_col_reaclay, nlen_dim_col_reaclay)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


                                    ! Query characteristics of the 'time' dimension
  istatus = NF_INQ_DIMID(ncid_file_reaclay, 'time', ncid_dim_time_reaclay)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

  istatus = NF_INQ_DIMLEN(ncid_file_reaclay, ncid_dim_time_in, nlen_dim_time_reaclay)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

ENDIF


                                    ! Query characteristics of the 'xgeom_dbswi'
                                    ! variable in the REACLAY file
istatus = NF_INQ_VARID(ncid_file_reaclay, 'xgeom_dbswi', ncid_var_xzdn)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_INQ_VARNDIMS(ncid_file_reaclay, ncid_var_xzdn, ndim_var_xzdn)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


                                    ! Query characteristics of the 'xgeom_phi'
                                    ! variable in the REACLAY file
istatus = NF_INQ_VARID(ncid_file_reaclay, 'xgeom_phi', ncid_var_xphi)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_INQ_VARNDIMS(ncid_file_reaclay, ncid_var_xphi, ndim_var_xphi)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! Complete DBL/REACLAY partitioning if necessary
SELECT CASE(iv_file_r_fmt)
CASE(2)
  IF (l_nb_dbl_tofind) THEN
    istatus = NF_GET_ATT_INT(ncid_file_reaclay, ncid_var_xzdn, 'nb_depth_intervals_DBL', ndi_w2t_r)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
  ENDIF

  IF (l_nb_rcl_tofind) THEN  
    istatus = NF_GET_ATT_INT(ncid_file_reaclay, ncid_var_xzdn, 'nb_depth_intervals_Reaclay', ndi_t2b_r)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
  ENDIF


  IF ((ndi_w2t_r /= ndi_w2t_in) .OR. (ndi_t2b_r /= ndi_t2b_in)) THEN
    WRITE(jp_stderr, '("Incompatible DBL/Reaclay partitioning in IN and REACLAY files:")')
    WRITE(jp_stderr, '(" - IN file:      nb_depth_intervals_DBL=", I0)') ndi_w2t_in
    WRITE(jp_stderr, '(" - REACLAY file: nb_depth_intervals_DBL=", I0)') ndi_w2t_r
    WRITE(jp_stderr, '(" - IN file:      nb_depth_intervals_Reaclay=", I0)') ndi_t2b_in
    WRITE(jp_stderr, '(" - REACLAY file: nb_depth_intervals_Reaclay=", I0)') ndi_t2b_r
    WRITE(jp_stderr, '("Abort!")')
    CALL ABORT_EXECUTION()
  ENDIF

  idnw = -ndi_w2t_in
  idnt =  0
  idnb =  ndi_t2b_in

  ageom_swiloc = 0.0D+00

CASE(3)

  IF ((ndn_w2s_r /= ndn_w2s_in) .OR. (ndn_s2b_r /= ndn_s2b_in)) THEN
    WRITE(jp_stderr, '("Incompatible DBL/Reaclay partitioning in IN and REACLAY files:")')
    WRITE(jp_stderr, '(" - IN file:      nb_nodes_DBL=", I0)') ndn_w2s_in
    WRITE(jp_stderr, '(" - REACLAY file: nb_nodes_DBL=", I0)') ndn_w2s_r
    WRITE(jp_stderr, '(" - IN file:      nb_nodes_Reaclay=", I0)') ndn_s2b_in
    WRITE(jp_stderr, '(" - REACLAY file: nb_nodes_Reaclay=", I0)') ndn_s2b_r
    WRITE(jp_stderr, '("Abort!")')
    CALL ABORT_EXECUTION()
  ENDIF

  idnt =  0
  idnw = idnt - ndn_w2s_in
  idnb = idnt + ndn_s2b_in - 1

  istatus = NF_GET_ATT_DOUBLE(ncid_file_reaclay, ncid_var_xzdn, 'swi_location', ageom_swiloc)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

END SELECT

IF (cfn_ncin_res /= cfn_ncin_reaclay) THEN

  ! Check col dimension compatibility
  ! Check time dimension compatibility

ENDIF

SELECT CASE(ndim_var_xzdn)
CASE(1)
  n2 = 1
  n3 = 1

CASE(2)
  n2 = nlen_dim_col_reaclay
  n3 = 1

CASE(3)
  n2 = nlen_dim_col_reaclay
  n3 = nlen_dim_time_reaclay

CASE DEFAULT
  WRITE(jp_stderr, '("""xgeom_dbswi"" arrays with ", I0, " dimensions &
                   &cannot yet be processed - aborting")') ndim_var_xzdn
  CALL ABORT_EXECUTION()

END SELECT
!====================================================

#ifdef DEBUG
WRITE(*,'("Allocating xgeom_xzdn(", I0, ":", I0, ",", I0, ",", I0, ")")') idnw, idnb, n2, n3
#endif
ALLOCATE(xgeom_xzdn       (idnw:idnb, n2, n3))

#ifdef DEBUG
WRITE(*,'("Allocating xgeom_vxzdn_dbl_c(", I0, ":", I0, ",", I0, ",", I0, ")")') idnw, idnt, n2, n3
#endif
ALLOCATE(xgeom_vxzdn_dbl_c(idnw:idnt, n2, n3))

#ifdef DEBUG
WRITE(*,'("Allocating xgeom_vxzdn_rcl_c(", I0, ":", I0, ",", I0, ",", I0, ")")') idnt, idnb, n2, n3
#endif
ALLOCATE(xgeom_vxzdn_rcl_c(idnt:idnb, n2, n3))

SELECT CASE(ndim_var_xphi)
CASE(1)
  n2 = 1
  n3 = 1

CASE(2)
  n2 = nlen_dim_col_reaclay
  n3 = 1

CASE(3)
  n2 = nlen_dim_col_reaclay
  n3 = nlen_dim_time_reaclay

CASE DEFAULT
  WRITE(jp_stderr, '("""xgeom_xphi"" arrays with ", I0, " dimensions &
                   &cannot yet be processed - aborting")') ndim_var_xphi
  CALL ABORT_EXECUTION()

END SELECT

#ifdef DEBUG
WRITE(*,'("Allocating xgeom_xphi(", I0, ":", I0, ",", I0, ",", I0, ")")') idnw, idnb, n2, n3
#endif
ALLOCATE(xgeom_xphi(idnw:idnb, n2, n3))


istatus = NF_GET_VAR_DOUBLE(ncid_file_reaclay, ncid_var_xzdn, xgeom_xzdn)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


ALLOCATE(xphi(idnw:idnb))
istatus = NF_GET_VAR_DOUBLE(ncid_file_reaclay, ncid_var_xphi, xgeom_xphi)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

xphi(:) = xgeom_xphi(:,1,1)

SELECT CASE(iv_file_r_fmt)
CASE(2)
  CALL CALC_VXZDN2(idnw, idnt, idnb, xgeom_xzdn, &
                xgeom_vxzdn_dbl_c, xgeom_vxzdn_rcl_c)
CASE(3)
  CALL CALC_VXZDN3(idnw, idnt, idnb, ageom_swiloc, xgeom_xzdn, &
                xgeom_vxzdn_dbl_c, xgeom_vxzdn_rcl_c)
END SELECT

ALLOCATE(vxzdn_dbl_c(idnw:idnt))
ALLOCATE(vxzdn_rcl_c(idnt:idnb))

vxzdn_dbl_c(:) = xgeom_vxzdn_dbl_c(:,1,1)
vxzdn_rcl_c(:) = xgeom_vxzdn_rcl_c(:,1,1)


! Create NCOUT file
                                    ! Create Output file
                                    ! ==================
istatus = NF_CREATE(cfn_ncout_res, NF_CLOBBER, ncid_file_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


                                    ! Define dimension 'col'
istatus = NF_DEF_DIM(ncid_file_out, 'col', nlen_dim_col_in, ncid_dim_col_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! Define dimension 'time'
istatus = NF_DEF_DIM(ncid_file_out, 'time', NF_UNLIMITED, ncid_dim_time_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


                                    ! Copy dimension variable 'col' from IN to OUT
istatus = NF_COPY_VAR(ncid_file_in, ncid_var_col_in, ncid_file_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

nlen_dim_col_out = nlen_dim_col_in

istatus = NF_INQ_VARID(ncid_file_out, 'col', ncid_var_col_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! Copy dimension variable 'time' from IN to OUT
istatus = NF_COPY_VAR(ncid_file_in, ncid_var_time_in, ncid_file_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

nlen_dim_time_out = nlen_dim_time_in

istatus = NF_INQ_VARID(ncid_file_out, 'time', ncid_var_time_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! Put back file to "define" mode
                                    ! -- NF_COPY_VAR has ended this.
istatus = NF_REDEF(ncid_file_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


! Variable in the NCIN file may be of the following shapes:
! and are integrated as follows
! 0 scalar           --> scalar      -- (0) copy
! 1 (lev)            -->             -- (-) skip
! 2 (col)            --> (col)       -- (0) copy
! 3 (time)           --> (time)      -- (0) copy
! 4 (lev, time)      --> (time)      -- (1) integrate and write
! 5 (col, time)      --> (col, time) -- (0) copy
! 6 (lev, col, time) --> (col, time) -- (2) integrate and write


                                    ! Get number of dims 
istatus = NF_INQ_NDIMS(ncid_file_in, ndims_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


ALLOCATE(cname_dims_in(ndims_in))

DO i = 1, ndims_in
  istatus = NF_INQ_DIMNAME(ncid_file_in, i, cname_dims_in(i))
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
ENDDO


                                    ! Get number of vars and classify them.
istatus = NF_INQ_NVARS(ncid_file_in, nvars_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

#ifdef DEBUG
WRITE(*, '("Found ", I0, " variable definitions in NCIN file")') nvars_in
#endif

ALLOCATE(cname_vars_in(nvars_in))
ALLOCATE(ihow2processvars_in(nvars_in))
ALLOCATE(ncid_vars_out(nvars_in))
ALLOCATE(itype_vars_in(nvars_in))
ALLOCATE(cphasid_vars_in(nvars_in))

ALLOCATE(darr_lc(nlen_dim_lev_in, nlen_dim_col_in))
ALLOCATE(darr_ct(nlen_dim_col_out, nlen_dim_time_out))


ncid_vars_out(:) = -9999

DO j = 1, nvars_in

  istatus = NF_INQ_VARNAME(ncid_file_in, j, cname_var)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
  cname_vars_in(j) = TRIM(cname_var)

  istatus = NF_INQ_VARNDIMS(ncid_file_in, j, ndims)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

  istatus = NF_INQ_VARTYPE(ncid_file_in, j, itype)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
  itype_vars_in(j) = itype

  istatus = NF_GET_ATT_TEXT(ncid_file_in, j, 'phasid', cphasid_vars_in(j))
  IF (istatus /= NF_NOERR) THEN
    IF (istatus == NF_ENOTATT) THEN
      cphasid_vars_in(j) = '??'
    ELSE
      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-5))
    ENDIF
  ENDIF

  SELECT CASE(cname_var)
  CASE('lev', 'col', 'time')
#   ifdef DEBUG
    WRITE(*, '("Skipping dimension variable """, A, """ in the NCIN file")') TRIM(cname_var)
#   endif
    ihow2processvars_in(j) = 0      ! No processing required
    CYCLE

  CASE('xgeom_dbswi', 'xgeom_phi', 'xgeom_dphi')
#   ifdef DEBUG
    WRITE(*, '("Skipping geometric variable """, A, """ in the NCIN file")') TRIM(cname_var)
#   endif
    ihow2processvars_in(j) = 0      ! No processing required
    CYCLE

  CASE DEFAULT
#   ifdef DEBUG
    WRITE(*, '("Analysing variable """, A, """ in the NCIN file")', ADVANCE='NO') TRIM(cname_var)
#   endif
    CONTINUE

  END SELECT



  SELECT CASE(ndims)

  CASE(0)
#   ifdef DEBUG
    WRITE(*,'(" - scalar variable, copying")')
#   endif
    istatus = NF_COPY_VAR(ncid_file_in, j, ncid_file_out)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
    istatus = NF_REDEF(ncid_file_out)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

    ihow2processvars_in(j) = 0    ! Done - no more processing required

  CASE(1)
    istatus = NF_INQ_VARDIMID(ncid_file_in, j, ncid_dim)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

    IF (cname_dims_in(ncid_dim) == cname_var) THEN
                                  ! Other dimension variables than 'lev', 'col' or 'time'
#     ifdef DEBUG
      WRITE(*,'(" - dimension variable, copying")')
#     endif
      istatus = NF_COPY_VAR(ncid_file_in, j, ncid_file_out)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      istatus = NF_REDEF(ncid_file_out)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      ihow2processvars_in(j) = 0    ! Done - no more processing required

    ELSE

      IF ( (ncid_dim == ncid_dim_col_in) .OR. &
           (ncid_dim == ncid_dim_time_in) ) THEN

#       ifdef DEBUG
        WRITE(*,'(" - dimensions (col) or (time), copying")')
#       endif
        istatus = NF_COPY_VAR(ncid_file_in, j, ncid_file_out)
        IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

        istatus = NF_REDEF(ncid_file_out)
        IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

        ihow2processvars_in(j) = 0    ! Done - no more processing required

      ELSEIF (ncid_dim == ncid_dim_lev_in) THEN
#       ifdef DEBUG
        WRITE(*,'(" - dimensions (lev), skipping")')
#       endif

        ihow2processvars_in(j) = 0    ! Done - no more processing required
        CYCLE

      ELSE

        WRITE(jp_stderr,'(" - unknown DIM_ID ", I0, " - aborting")') ncid_dim
        CALL ABORT_EXECUTION()
        
      ENDIF

    ENDIF

    
  CASE(2)

    istatus = NF_INQ_VARDIMID(ncid_file_in, j, ncid_dim2)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

    IF (ALL(ncid_dim2(:) == (/ ncid_dim_lev_in, ncid_dim_time_in /))) THEN

#     ifdef DEBUG
      WRITE(*,'(" - dimensions (lev,time), needs integration [1]")')
#     endif
      istatus  = NF_DEF_VAR(ncid_file_out, TRIM(cname_var), itype, &
                                    1, (/ ncid_dim_time_out /), &
                                    ncid_vars_out(j))
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

      ihow2processvars_in(j) = 1    ! Schedule integration dim 1 of 2

    ELSEIF (ALL(ncid_dim2(:) == (/ ncid_dim_col_in, ncid_dim_time_in /))) THEN

#     ifdef DEBUG
      WRITE(*,'(" - dimensions (col,time), copying")')
#     endif
      istatus = NF_COPY_VAR(ncid_file_in, j, ncid_file_out)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      istatus = NF_REDEF(ncid_file_out)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      ihow2processvars_in(j) = 0    ! Done - no more processing required

    ELSE

        WRITE(jp_stderr,'(" - unknown DIM_IDs (", I0, ",", I0, ") - aborting")') ncid_dim2
        CALL ABORT_EXECUTION()

    ENDIF


  CASE(3)

    istatus = NF_INQ_VARDIMID(ncid_file_in, j, ncid_dim3)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

    IF (ALL(ncid_dim3(:) == (/ ncid_dim_lev_in, ncid_dim_col_in, ncid_dim_time_in /))) THEN

#     ifdef DEBUG
      WRITE(*,'(" - dimensions (lev,col,time), needs integration [2]")')
#     endif
      istatus  = NF_DEF_VAR(ncid_file_out, TRIM(cname_var), itype, &
                                    2, (/ ncid_dim_col_out, ncid_dim_time_out /), &
                                    ncid_vars_out(j))
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

      ihow2processvars_in(j) = 2    ! Schedule integration on dimension 1 of 3

    ELSE

      WRITE(jp_stderr,'(" - unknown DIM_IDs (", I0, ",", I0, ",", I0, ") - aborting")') ncid_dim3
      CALL ABORT_EXECUTION()

    ENDIF

  CASE DEFAULT

    WRITE(jp_stderr,'(" - arrays with ", I0, " cannot yet be processed - aborting")') ndims
    CALL ABORT_EXECUTION()

  END SELECT



  SELECT CASE(ihow2processvars_in(j))
  CASE(0)
    CYCLE

  CASE(1:2)
                                    ! Copy over the attributes
    istatus = NF_INQ_VARNATTS(ncid_file_in, j, natts)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

    DO k = 1, natts

      istatus = NF_INQ_ATTNAME(ncid_file_in, j, k, cname_att)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      IF (cname_att == 'units') THEN

        istatus = NF_INQ_ATTLEN(ncid_file_in, j, 'units', nlen_att)
        IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
        istatus = NF_GET_ATT_TEXT(ncid_file_in, j, 'units', cval_att)
        IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

        IF (cval_att(1:nlen_att) == 'mol m-3 bulk sed yr-1') THEN
          cval_att = 'mol m-2 bulk sed yr-1'
          nlen_att = LEN_TRIM(cval_att)
        ELSEIF (cval_att(1:nlen_att) == 'kg m-3 bulk sed yr-1') THEN
          cval_att = 'kg m-2 bulk sed yr-1'
          nlen_att = LEN_TRIM(cval_att)
        ELSEIF (cval_att(1:nlen_att) == 'mol m-3 porewater') THEN
          cval_att = 'mol m-2 bulk sed'
          nlen_att = LEN_TRIM(cval_att)
        ELSEIF (cval_att(1:nlen_att) == 'kg m-3 solid sed') THEN
          cval_att = 'kg m-2 bulk sed'
          nlen_att = LEN_TRIM(cval_att)
        ENDIF
        istatus = NF_PUT_ATT_TEXT(ncid_file_out, ncid_vars_out(j), 'units', nlen_att, cval_att)
        IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
          
      ELSE

        istatus = NF_COPY_ATT(ncid_file_in, j, TRIM(cname_att), ncid_file_out, ncid_vars_out(j))
        IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      ENDIF

    ENDDO

                                    ! Add _FillValue attribute
    SELECT CASE(itype_vars_in(j))
    CASE(NF_INT)
      istatus = NF_PUT_ATT_INT(ncid_file_out, ncid_vars_out(j), &
                                    '_FillValue', NF_INT, 1, (/ NF_FILL_INT /))
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
    CASE(NF_REAL)
      istatus = NF_PUT_ATT_REAL(ncid_file_out, ncid_vars_out(j), &
                                    '_FillValue', NF_REAL, 1, (/ NF_FILL_REAL /))
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
    CASE(NF_DOUBLE)
      istatus = NF_PUT_ATT_DOUBLE(ncid_file_out, ncid_vars_out(j), &
                                    '_FillValue', NF_DOUBLE, 1, (/ NF_FILL_DOUBLE /))
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
    END SELECT

  CASE DEFAULT

    WRITE(jp_stderr,'("Unknown scheduled action [", I0, "] - aborting")') ihow2processvars_in(j)
    CALL ABORT_EXECUTION()

  END SELECT

ENDDO

                                    ! End define mode
istatus = NF_ENDDEF(ncid_file_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))



#ifdef DEBUG
WRITE(*,'()')
#endif

DO ivar = 1, nvars_in

  SELECT CASE(ihow2processvars_in(ivar))

  CASE(0)                           ! Already copied - done!

    CYCLE


  CASE(1)                           ! (lev, time) --> (time) - (1) integrate

    WRITE(*,'()')
    WRITE(*, '("(lev, time) --> (time) integration not yet implemented -- skipping")')

  CASE(2)                           ! (lev, col, time) --> (col, time) - (2) integrate

#   ifdef DEBUG
    WRITE(*, '("Processing variable """, A, """ in the NCIN file")') TRIM(cname_vars_in(ivar))
#   endif
    SELECT CASE(itype_vars_in(ivar))

    CASE(NF_DOUBLE)

      istatus = NF_GET_ATT_DOUBLE(ncid_file_in, ivar, '_FillValue', dfillvalue)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


      DO irec = 1, nlen_dim_time_in

        istart(1:3) = (/              1,               1, irec /)
        ncount(1:3) = (/nlen_dim_lev_in, nlen_dim_col_in,    1 /)

        istatus = NF_GET_VARA_DOUBLE(ncid_file_in, ivar, istart, ncount, darr_lc)
        IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! Check for _FillValue content
                                    ! (can occur in REACLAY files where
                                    ! the first record may be kept empty).
        IF (ALL(darr_lc == dfillvalue)) THEN

#         ifdef DEBUG
          WRITE(*,'(" record ", I0, " has only _FillValue content -- &
                      &also using _FillValue for integrated record")') irec
#         endif
          darr_ct(:,irec) = NF_FILL_DOUBLE

        ELSE

          DO j = 1, nlen_dim_col_in

            SELECT CASE(ndim_var_xzdn)
            CASE(2)
              vxzdn_dbl_c(:) = xgeom_vxzdn_dbl_c(:,j,1)
              vxzdn_rcl_c(:) = xgeom_vxzdn_rcl_c(:,j,1)
            CASE(3)
              vxzdn_dbl_c(:) = xgeom_vxzdn_dbl_c(:,j,irec)
              vxzdn_rcl_c(:) = xgeom_vxzdn_rcl_c(:,j,irec)
            END SELECT

            SELECT CASE(ndim_var_xphi)
            CASE(2)
              xphi(:) = xgeom_xphi(:,j,1)
            CASE(3)
              xphi(:) = xgeom_xphi(:,j,irec)
            END SELECT

            SELECT CASE(cphasid_vars_in(ivar))
            CASE('ic')
              CALL INTEGRATE_SOLUTE(iv_file_in_fmt, idnw, idnt, idnb,  &
                                      vxzdn_dbl_c, vxzdn_rcl_c, xphi,  &
                                      darr_lc(:,j), darr_ct(j,irec))
            CASE('if')
              CALL INTEGRATE_SOLID(iv_file_in_fmt, idnw, idnt, idnb,   &
                                      vxzdn_rcl_c, xphi,               &
                                      darr_lc(:,j), darr_ct(j,irec))
            CASE DEFAULT
              CALL INTEGRATE_BULK(iv_file_in_fmt, idnw, idnt, idnb,    &
                                      vxzdn_dbl_c, vxzdn_rcl_c,        &
                                      darr_lc(:,j), darr_ct(j,irec))
            END SELECT
          ENDDO

        ENDIF

      ENDDO

      istatus = NF_PUT_VAR_DOUBLE(ncid_file_out, ncid_vars_out(ivar), darr_ct)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


    CASE DEFAULT
      ! Others than NF_DOUBLE cannot yet be handled
      WRITE(jp_stderr, '("Variable types other than DOUBLE cannot yet be handled - skipping")')

    END SELECT

  END SELECT


ENDDO

                                    ! Close files
istatus = NF_CLOSE(ncid_file_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

IF (cfn_ncin_reaclay /= cfn_ncin_res) THEN
  istatus = NF_CLOSE(ncid_file_reaclay)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
ENDIF

istatus = NF_CLOSE(ncid_file_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

!print*, ' -1: darr_lc :', ALLOCATED(darr_lc)
!print*, ' -1: darr_ct :', ALLOCATED(darr_ct)
!print*, ' -1: xgeom_xzdn :', ALLOCATED(xgeom_xzdn)
!print*, ' -1: xgeom_xphi :', ALLOCATED(xgeom_xphi)
!print*, ' -1: xgeom_vxzdn_dbl_c :', ALLOCATED(xgeom_vxzdn_dbl_c)
!print*, ' -1: xgeom_vxzdn_rcl_c :', ALLOCATED(xgeom_vxzdn_rcl_c)
!print*, ' -1: xphi :', ALLOCATED(xphi)
!print*, ' -1: vxzdn_dbl_c :', ALLOCATED(vxzdn_dbl_c)
!print*, ' -1: vxzdn_rcl_c :', ALLOCATED(vxzdn_rcl_c)

DEALLOCATE(darr_lc)
DEALLOCATE(darr_ct)
DEALLOCATE(xgeom_xzdn)
DEALLOCATE(xgeom_xphi)
DEALLOCATE(xgeom_vxzdn_dbl_c)
DEALLOCATE(xgeom_vxzdn_rcl_c)
DEALLOCATE(xphi)
DEALLOCATE(vxzdn_dbl_c)
DEALLOCATE(vxzdn_rcl_c)

9999 CONTINUE





cpn_nml_defaults = TRIM(cdn_nml_defaults) // TRIM(c_dirsep) // TRIM(cfn_nml_defaults)

#ifdef VERBOSE
WRITE(*, '()')
WRITE(*, '("Saving the current configuration to """, A, """")') TRIM(cpn_nml_defaults)
WRITE(*, '("for usage as default next time. Please move that file to the")')
WRITE(*, '("current directory and rename it to """, A, """")') cfn_nml
WRITE(*, '("to use the same configuration the next times (without asking).")')
#endif

OPEN(UNIT=jp_lunit, FILE=cpn_nml_defaults)

WRITE(jp_lunit, '("!=======================================================================")')
WRITE(jp_lunit, '("! MEDINTEGRALEV requests namelist from the most recent successful run.")')
WRITE(jp_lunit, '("! Content from this file will be used to derive default values next")')
WRITE(jp_lunit, '("! time, unless the working directory contains a namelist file called")')
WRITE(jp_lunit, '("! """, A, """, which will then always be used,")') cfn_nml
WRITE(jp_lunit, '("! without any questions asked.")')
WRITE(jp_lunit, '("!=======================================================================")')
WRITE(jp_lunit, '("&nml_medintegralev")')
WRITE(jp_lunit, '("! Input data for MEDINTEGRALEV")')
WRITE(jp_lunit, '("!  - cfn_ncin_res:     name of Medusa generated NetCDF results file")')
WRITE(jp_lunit, '("!                      (REACLAY, REACTION, PROCRATE, FLX) to integrate")')
WRITE(jp_lunit, '("!                      [mandatory, no default]")')
WRITE(jp_lunit, '("!  - cfn_ncin_reaclay: name of the REACLAY file with the geometrical")')
WRITE(jp_lunit, '("!                      information [default: cfn_ncin_reaclay =")')
WRITE(jp_lunit, '("!                      cfn_ncin_res if cfn_ncin_res is a REACLAY file]")')
WRITE(jp_lunit, '("!  - cfn_ncout_res:    name of the NetCDF file to write the integrated")')
WRITE(jp_lunit, '("!                      results to [default: ""_intlev"" appended to the")')
WRITE(jp_lunit, '("!                      base name of <cfn_ncin_res>, i e., inserted")')
WRITE(jp_lunit, '("!                      before the extension "".nc""]")')
WRITE(jp_lunit, '("cfn_ncin_res          = """, A, """")') TRIM(cfn_ncin_res)
WRITE(jp_lunit, '("cfn_ncin_reaclay      = """, A, """")') TRIM(cfn_ncin_reaclay)
WRITE(jp_lunit, '("cfn_ncout_res         = """, A, """")') TRIM(cfn_ncout_res)
WRITE(jp_lunit, '("/")')
CLOSE(jp_lunit)



!========!
 CONTAINS
!========!

!-----------------------------------------------------------------------
 SUBROUTINE GET_MENU(cfn_ncin_res, cfn_ncin_reaclay, cfn_ncout_res)
!-----------------------------------------------------------------------


IMPLICIT NONE


CHARACTER(LEN=*), INTENT(OUT) ::    cfn_ncin_res
CHARACTER(LEN=*), INTENT(OUT) ::    cfn_ncin_reaclay
CHARACTER(LEN=*), INTENT(OUT) ::    cfn_ncout_res

NAMELIST /nml_medintegralev/ cfn_ncin_res, cfn_ncin_reaclay, cfn_ncout_res

CHARACTER(LEN=jp_lmaxpathname) :: c_dummy



cfn_ncin_res      = "/dev/null"
cfn_ncin_reaclay  = "/dev/null"
cfn_ncout_res     = "/dev/null"

                                    ! First try to read from a NAMELIST
                                    ! file in the working directory
INQUIRE(FILE=cfn_nml, EXIST=l_exists)

IF (l_exists) THEN                  ! - if there is one, use it without
                                    !   asking questions

  OPEN(UNIT=jp_lunit, FILE=cfn_nml, STATUS='OLD')
  READ(UNIT=jp_lunit, NML=nml_medintegralev)
  CLOSE(UNIT=jp_lunit)


# ifdef VERBOSE
  WRITE(*,'("Found NAMELIST file """, A, """")') TRIM(cfn_nml)
  WRITE(*,'("and retrieved the following information from it:")')
  WRITE(*,'(" cfn_ncin_res     = """, A, """")') TRIM(cfn_ncin_res)
  WRITE(*,'(" cfn_ncin_reaclay = """, A, """")') TRIM(cfn_ncin_reaclay)
  WRITE(*,'(" cfn_ncout_res    = """, A, """")') TRIM(cfn_ncout_res)
  WRITE(*,'()')
  WRITE(*,'()')
  WRITE(*,'("If any of the entries above are unsuitable,")')
  WRITE(*,'("please delete """, A, """ and re-run me!")') TRIM(cfn_nml)
  WRITE(*,'()')
# endif

ELSE                                ! - if no NAMELIST file is present
                                    !   in the current working directory,
                                    !   go for an interactive configuration.
  cpn_nml_defaults = TRIM(cdn_nml_defaults) // TRIM(c_dirsep) // TRIM(cfn_nml_defaults)
  INQUIRE(FILE=cpn_nml_defaults, EXIST=l_exists)

  IF (l_exists) THEN                !   if there is one, read its contents
                                    !   and use that as default values
    OPEN(UNIT=jp_lunit, FILE=cpn_nml_defaults, STATUS='OLD')
    READ(UNIT=jp_lunit, NML=nml_medintegralev)
    CLOSE(UNIT=jp_lunit)
  ENDIF

                                    !   Now, we may start the configuration
# ifndef VERBOSE
  WRITE(*, '()')
  WRITE(*, '("This is MEDINTEGRALEV")')
  WRITE(*, '("=====================")')
  WRITE(*, '()')
# endif

  WRITE(*, '("----------------------------------------------------------")')
  WRITE(*, '("Please enter the following required information")')
  WRITE(*, '("----------------------------------------------------------")')
  WRITE(*, '(" * enter file, directory and names without")')
  WRITE(*, '("   delimiting quotes")')
  WRITE(*, '(" * defaults given between brackets ([...]) will")')
  WRITE(*, '("   be used for fields left empty")')
  WRITE(*, '("----------------------------------------------------------")')
  WRITE(*, '()')

  WRITE(*, '(" - the name of the NCIN file to read from")')
  IF (cfn_ncin_res == "/dev/null") THEN
    WRITE(*,'("   > ")', ADVANCE="NO")
  ELSE
    WRITE(*,'("   [", A, "] > ")', ADVANCE="NO") TRIM(cfn_ncin_res)
  ENDIF

  DO
    READ(*, '(A)') c_dummy

    IF (LEN_TRIM(c_dummy) == 0) THEN
      IF (cfn_ncin_res == "/dev/null") THEN
        WRITE(*, '("   sorry, there is no valid default - please try again > ")', ADVANCE = "NO")
        CYCLE
      ELSE
        EXIT
      ENDIF
    ELSE
      READ(c_dummy, '(A)') cfn_ncin_res
      EXIT
    ENDIF
  ENDDO

  WRITE(*, '("   requesting NCIN file """, A, """")') TRIM(cfn_ncin_res)
  WRITE(*, '()')


  WRITE(*,' (" - the name of the REACLAY file to read from")')
  IF (cfn_ncin_reaclay == "/dev/null") THEN
    WRITE(*, '("   [", A, "] > ")', ADVANCE="NO") TRIM(cfn_ncin_res)
  ELSE
    WRITE(*, '("   [", A, "] > ")', ADVANCE="NO") TRIM(cfn_ncin_reaclay)
  ENDIF

  READ(*, '(A)') c_dummy
  IF (LEN_TRIM(c_dummy) == 0) THEN
    IF (cfn_ncin_reaclay == "/dev/null") cfn_ncin_reaclay = TRIM(cfn_ncin_res)
  ELSE
    READ(c_dummy, '(A)') cfn_ncin_reaclay
  ENDIF
  WRITE(*, '("   requesting REACLAY file """, A, """")') TRIM(cfn_ncin_reaclay)
  WRITE(*, '()')


  WRITE(*,' (" - the name of the NCOUT file to write to")')
  IF (cfn_ncout_res == "/dev/null") THEN
    ix = INDEX(cfn_ncin_res, '.nc', BACK=.TRUE.)
    IF (ix > 1) THEN
      cfn_ncout_res = cfn_ncin_res(1:ix-1) // '_intlev.nc'
      WRITE(*,'("   [", A, "] > ")', ADVANCE="NO") TRIM(cfn_ncout_res)
    ELSE
      WRITE(*,'("   > ")', ADVANCE="NO")
    ENDIF
  ELSE
    WRITE(*,'("   [", A, "] > ")', ADVANCE="NO") TRIM(cfn_ncout_res)
  ENDIF

  DO
    READ(*, '(A)') c_dummy

    IF (LEN_TRIM(c_dummy) == 0) THEN
      IF (cfn_ncout_res == "/dev/null") THEN
        WRITE(*, '("   sorry, there is no valid default - please try again > ")', ADVANCE = "NO")
        CYCLE
      ELSE
        EXIT
      ENDIF
    ELSE
      READ(c_dummy, '(A)') cfn_ncout_res
      EXIT
    ENDIF
  ENDDO

  WRITE(*, '("   requesting NCOUT file """, A, """")') TRIM(cfn_ncout_res)
  WRITE(*, '()')


# ifdef DEBUG
  WRITE(*, '("NCIN name """, A,"""")')    TRIM(cfn_ncin_res)
  WRITE(*, '("REACLAY name """, A,"""")') TRIM(cfn_ncin_reaclay)
  WRITE(*, '("NCOUT name """, A,"""")')   TRIM(cfn_ncout_res)
# endif

  WRITE(*, '()')
  WRITE(*, '("----------------------------------------------------------")')
  WRITE(*, '()')


ENDIF


RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE GET_MENU
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE INTEGRATE_SOLUTE(iv_filefmt, idnw, idnt, idnb,             &
                                    vxzdn_dbl_c, vxzdn_rcl_c, xphi,    &
                                    xc_solute, tc_solute)
!-----------------------------------------------------------------------

IMPLICIT NONE


INTEGER,                                INTENT(IN)  :: iv_filefmt
INTEGER,                                INTENT(IN)  :: idnw
INTEGER,                                INTENT(IN)  :: idnt
INTEGER,                                INTENT(IN)  :: idnb
DOUBLE PRECISION, DIMENSION(idnw:idnt), INTENT(IN)  :: vxzdn_dbl_c
DOUBLE PRECISION, DIMENSION(idnt:idnb), INTENT(IN)  :: vxzdn_rcl_c
DOUBLE PRECISION, DIMENSION(idnw:idnb), INTENT(IN)  :: xphi
DOUBLE PRECISION, DIMENSION(idnw:idnb), INTENT(IN)  :: xc_solute
DOUBLE PRECISION,                       INTENT(OUT) :: tc_solute

INTEGER :: inode


IF (idnw < idnt) THEN

  SELECT CASE(iv_filefmt)
  CASE(2)
                                    ! Initialize with the content of the
                                    ! half-cell below the DBL-Water interface
    tc_solute = ( (1.0D+00-thetatop) * xc_solute(idnw)    &
                 +     thetatop      * xc_solute(idnw+1)) &
                * vxzdn_dbl_c(idnw)

                                    ! Add contents of the full cells in the DBL
    DO inode = idnw + 1, idnt - 1
      tc_solute =   tc_solute + xc_solute(inode)*vxzdn_dbl_c(inode)
    ENDDO

                                    ! Add content of the half-cell above the SWI
    tc_solute = tc_solute +                               &
                (      thetatop      * xc_solute(idnt-1)  &
                 +(1.0D+00-thetatop) * xc_solute(idnt  )) &
                * vxzdn_dbl_c(idnt)

                                    ! Add content of the half-cell below the SWI
    tc_solute = tc_solute +                                        &
            ( (1.0D+00-thetatop) * xc_solute(idnt  )*xphi(idnt  )  &
             +     thetatop      * xc_solute(idnt+1)*xphi(idnt+1)) &
            * vxzdn_rcl_c(idnt)
  CASE(3)

                                    ! Initialize with the content of the
                                    ! half-cell below the DBL-Water interface
                                    ! (no thetatop)
    tc_solute = xc_solute(idnw) * vxzdn_dbl_c(idnw)

                                    ! Add the contents of the remaining
                                    ! (full) cells in the DBL
    DO inode = idnw + 1, idnt - 1
      tc_solute =   tc_solute + xc_solute(inode)*vxzdn_dbl_c(inode)
    ENDDO
                                    ! Add the content of the (full) cell
                                    ! right below the SWI
    tc_solute = tc_solute + xc_solute(idnt)*xphi(idnt) * vxzdn_rcl_c(idnt)

  END SELECT

ELSE
                                    ! Initialize with the content of the
                                    ! half-cell below the DBL-Water interface
                                    ! (with thetatop)
    tc_solute = ( (1.0D+00-thetatop) * xc_solute(idnt  )*xphi(idnt  )  &
                 +     thetatop      * xc_solute(idnt+1)*xphi(idnt+1)) &
                * vxzdn_rcl_c(idnt)

ENDIF

                                    ! Add contents of all the nodes
                                    ! below idnt (excl.) and above idnb (excl.)
DO inode = idnt + 1, idnb - 1
  tc_solute = tc_solute + xc_solute(inode)*xphi(inode)* vxzdn_rcl_c(inode)
ENDDO

                                    ! Add half-cell content above the bottom
                                    ! of REACLAY
tc_solute = tc_solute +                                            &
            (      thetabot      * xc_solute(idnb-1)*xphi(idnb-1)  &
             +(1.0D+00-thetabot) * xc_solute(idnb  )*xphi(idnb  )) &
            * vxzdn_rcl_c(idnb)

RETURN


!-----------------------------------------------------------------------
 END SUBROUTINE INTEGRATE_SOLUTE
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE INTEGRATE_SOLID(iv_filefmt, idnw, idnt, idnb,              &
                                    vxzdn_rcl_c, xphi,                 &
                                    xc_solid, tc_solid)
!-----------------------------------------------------------------------

IMPLICIT NONE


INTEGER,                                INTENT(IN)  :: iv_filefmt
INTEGER,                                INTENT(IN)  :: idnw
INTEGER,                                INTENT(IN)  :: idnt
INTEGER,                                INTENT(IN)  :: idnb
DOUBLE PRECISION, DIMENSION(idnt:idnb), INTENT(IN)  :: vxzdn_rcl_c
DOUBLE PRECISION, DIMENSION(idnw:idnb), INTENT(IN)  :: xphi
DOUBLE PRECISION, DIMENSION(idnw:idnb), INTENT(IN)  :: xc_solid
DOUBLE PRECISION,                       INTENT(OUT) :: tc_solid

INTEGER :: inode


IF (idnw < idnt) THEN

  SELECT CASE(iv_filefmt)
  CASE(2)
                                    ! Initialize with the content of the
                                    ! half-cell below the SWI
    tc_solid = ( (1.0D+00-thetatop) * xc_solid(idnt  ) * (1.0D+00-xphi(idnt  ))   &
                +     thetatop      * xc_solid(idnt+1) * (1.0D+00-xphi(idnt+1)) ) &
               * vxzdn_rcl_c(idnt)

  CASE(3)
                                    ! Initialize with the content of the
                                    ! (full) cell below the SWI
    tc_solid = xc_solid(idnt) * (1.0D+00-xphi(idnt)) * vxzdn_rcl_c(idnt)
  END SELECT

ELSE
                                    ! Initialize with the content of the
                                    ! half-cell below the SWI
    tc_solid = ( (1.0D+00-thetatop) * xc_solid(idnt  ) * (1.0D+00-xphi(idnt  ))   &
                +     thetatop      * xc_solid(idnt+1) * (1.0D+00-xphi(idnt+1)) ) &
               * vxzdn_rcl_c(idnt)

ENDIF

                                    ! Add contents of all the nodes
                                    ! below idnt (excl.) and above idnb (excl.)
DO inode = idnt + 1, idnb - 1
  tc_solid = tc_solid + xc_solid(inode)*(1.0D+00-xphi(inode)) * vxzdn_rcl_c(inode)
ENDDO


                                    ! Add half-cell content above the bottom
                                    ! of REACLAY
tc_solid = tc_solid +                                                          &
           (       thetabot      * xc_solid(idnb-1) * (1.0D+00-xphi(idnb-1))   &
            + (1.0D+00-thetabot) * xc_solid(idnb  ) * (1.0D+00-xphi(idnb  )) ) &
           * vxzdn_rcl_c(idnb)

RETURN


!-----------------------------------------------------------------------
 END SUBROUTINE INTEGRATE_SOLID
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE INTEGRATE_BULK(iv_filefmt, idnw, idnt, idnb,               &
                                    vxzdn_dbl_c, vxzdn_rcl_c,          &
                                    xc_bulk, tc_bulk)
!-----------------------------------------------------------------------

IMPLICIT NONE

INTEGER,                                INTENT(IN)  :: iv_filefmt
INTEGER,                                INTENT(IN)  :: idnw
INTEGER,                                INTENT(IN)  :: idnt
INTEGER,                                INTENT(IN)  :: idnb
DOUBLE PRECISION, DIMENSION(idnw:idnt), INTENT(IN)  :: vxzdn_dbl_c
DOUBLE PRECISION, DIMENSION(idnt:idnb), INTENT(IN)  :: vxzdn_rcl_c
DOUBLE PRECISION, DIMENSION(idnw:idnb), INTENT(IN)  :: xc_bulk
DOUBLE PRECISION,                       INTENT(OUT) :: tc_bulk

INTEGER :: inode


IF (idnw < idnt) THEN

  SELECT CASE(iv_filefmt)
  CASE(2)
                                    ! Initialize with the content of the
                                    ! half-cell below the DBL-Water interface
    tc_bulk = xc_bulk(idnw) * vxzdn_dbl_c(idnw)

                                    ! Add contents of the full cells in the DBL
    DO inode = idnw + 1, idnt - 1
      tc_bulk =   tc_bulk + xc_bulk(inode)*vxzdn_dbl_c(inode)
    ENDDO

                                    ! Add content of the half-cell above the SWI
    tc_bulk = tc_bulk + xc_bulk(idnt) * vxzdn_dbl_c(idnt)

                                    ! Add content of the half-cell below the SWI
    tc_bulk = tc_bulk + xc_bulk(idnt) * vxzdn_rcl_c(idnt)

  CASE(3)

                                    ! Initialize with the content of the
                                    ! half-cell below the DBL-Water interface
                                    ! (without thetatop)
    tc_bulk = xc_bulk(idnw) * vxzdn_dbl_c(idnw)

                                    ! Add the contents of the remaining
                                    ! (full) cells in the DBL
    DO inode = idnw + 1, idnt - 1
      tc_bulk =   tc_bulk + xc_bulk(inode)*vxzdn_dbl_c(inode)
    ENDDO
                                    ! Add the content of the (full) cell
                                    ! right below the SWI
    tc_bulk = tc_bulk + xc_bulk(idnt) * vxzdn_rcl_c(idnt)

  END SELECT

ELSE
                                    ! Initialize with the content of the
                                    ! half-cell below the DBL-Water interface
                                    ! (without thetatop)
  tc_bulk = xc_bulk(idnt) * vxzdn_rcl_c(idnt)

ENDIF

                                    ! Add contents of all the celles
                                    ! below idnt (excl.) and above idnb (incl.)
DO inode = idnt + 1, idnb
  tc_bulk = tc_bulk + xc_bulk(inode)* vxzdn_rcl_c(inode)
ENDDO

RETURN


RETURN


!-----------------------------------------------------------------------
 END SUBROUTINE INTEGRATE_BULK
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE CALC_VXZDN2(idnw, idnt, idnb, xzdn, vxzdn_dbl_c, vxzdn_rcl_c)
!-----------------------------------------------------------------------

IMPLICIT NONE

INTEGER,                            INTENT(IN)  :: idnw
INTEGER,                            INTENT(IN)  :: idnt
INTEGER,                            INTENT(IN)  :: idnb
DOUBLE PRECISION, DIMENSION(:,:,:), INTENT(IN)  :: xzdn
DOUBLE PRECISION, DIMENSION(:,:,:), INTENT(OUT) :: vxzdn_dbl_c
DOUBLE PRECISION, DIMENSION(:,:,:), INTENT(OUT) :: vxzdn_rcl_c

INTEGER :: kdnw,     kdnt,     kdnb
INTEGER :: kdnw_dbl, kdnt_dbl
INTEGER ::           kdnt_rcl, kdnb_rcl


kdnw = 1
kdnt = 1 + (idnt-idnw)
kdnb = 1 + (idnb-idnw)

! vxzdn_dbl_c is dimensioned (idnw:idnt,:,:)

kdnw_dbl = kdnw
kdnt_dbl = kdnt


IF (idnw < idnt) THEN

  vxzdn_dbl_c(kdnw_dbl,:,:)              = (xzdn(kdnw+1,:,:)-xzdn(kdnw,:,:))/2.0D+00
  vxzdn_dbl_c(kdnw_dbl+1:kdnt_dbl-1,:,:) = (xzdn(kdnw+2:idnt,:,:)-xzdn(kdnw:kdnt-2,:,:))/2.0D+00
  vxzdn_dbl_c(kdnt_dbl,:,:)              = (xzdn(kdnt,:,:)-xzdn(kdnt-1,:,:))/2.0D+00

ELSE

  vxzdn_dbl_c(kdnt_dbl,:,:)              = 0.0D+00

ENDIF


! vxzdn_rcl_c is dimensioned (idnt:idnb,:,:)

kdnt_rcl = kdnt - (idnt-idnw)
kdnb_rcl = kdnb - (idnt-idnw)

vxzdn_rcl_c(kdnt_rcl,:,:)                = (xzdn(kdnt+1,:,:)-xzdn(kdnt,:,:))/2.0D+00
vxzdn_rcl_c(kdnt_rcl+1:kdnb_rcl-1,:,:)   = (xzdn(kdnt+2:kdnb,:,:)-xzdn(kdnt:kdnb-2,:,:))/2.0D+00
vxzdn_rcl_c(kdnb_rcl,:,:)                = (xzdn(kdnb,:,:)-xzdn(kdnb-1,:,:))/2.0D+00

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE CALC_VXZDN2
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
 SUBROUTINE CALC_VXZDN3(idnw, idnt, idnb, aswi, xzdn, vxzdn_dbl_c, vxzdn_rcl_c)
!-----------------------------------------------------------------------

IMPLICIT NONE

INTEGER,                            INTENT(IN)  :: idnw
INTEGER,                            INTENT(IN)  :: idnt
INTEGER,                            INTENT(IN)  :: idnb
DOUBLE PRECISION,                   INTENT(IN)  :: aswi
DOUBLE PRECISION, DIMENSION(:,:,:), INTENT(IN)  :: xzdn
DOUBLE PRECISION, DIMENSION(:,:,:), INTENT(OUT) :: vxzdn_dbl_c
DOUBLE PRECISION, DIMENSION(:,:,:), INTENT(OUT) :: vxzdn_rcl_c

INTEGER :: kdnw, kdnt, kdnb
INTEGER :: kdnw_dbl, kdnt_dbl
INTEGER ::           kdnt_rcl, kdnb_rcl


kdnw = 1
kdnt = 1 + (idnt-idnw)
kdnb = 1 + (idnb-idnw)

! vxzdn_dbl_c is dimensioned (idnw:idnt,:,:)

kdnw_dbl = kdnw
kdnt_dbl = kdnt


IF (idnw == idnt) THEN

  vxzdn_dbl_c(kdnt_dbl,:,:)              = 0.0D+00 ! not to be used

ELSEIF (idnw == (idnt - 1)) THEN

  vxzdn_dbl_c(kdnw_dbl,:,:)              =  aswi - xzdn(kdnw,:,:)
  vxzdn_dbl_c(kdnt_dbl,:,:)              =  0.0D+00 ! not to be used

ELSE

  vxzdn_dbl_c(kdnw_dbl,:,:)              = (xzdn(kdnw+1,:,:)-xzdn(kdnw,:,:))/2.0D+00
  vxzdn_dbl_c(kdnw_dbl+1:kdnt_dbl-2,:,:) = (xzdn(kdnw+2:kdnt-1,:,:)-xzdn(kdnw:kdnt-3,:,:))/2.0D+00
  vxzdn_dbl_c(kdnt_dbl-1,:,:)            = aswi - (xzdn(kdnt,:,:)+xzdn(kdnt-1,:,:))/2.0D+00
  vxzdn_dbl_c(kdnt_dbl,:,:)              =  0.0D+00 ! not to be used

ENDIF


! vxzdn_rcl_c is dimensioned (idnt:idnb,:,:)

kdnt_rcl = kdnt - (idnt-idnw)
kdnb_rcl = kdnb - (idnt-idnw)

IF (idnw == idnt) THEN

  vxzdn_rcl_c(kdnt_rcl,:,:)            = (xzdn(kdnt+1,:,:)-xzdn(kdnt,:,:))/2.0D+00

ELSE

  vxzdn_rcl_c(kdnt_rcl,:,:)            = (xzdn(kdnt+1,:,:)+xzdn(kdnt,:,:))/2.0D+00 - aswi

ENDIF

vxzdn_rcl_c(kdnt_rcl+1:kdnb_rcl-1,:,:) = (xzdn(kdnt+2:kdnb,:,:)-xzdn(kdnt:kdnb-2,:,:))/2.0D+00
vxzdn_rcl_c(kdnb_rcl,:,:)              = (xzdn(kdnb,:,:)-xzdn(kdnb-1,:,:))/2.0D+00


RETURN


!-----------------------------------------------------------------------
 END SUBROUTINE CALC_VXZDN3
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
 SUBROUTINE HANDLE_NCERRORS(istatus, whatfile, iline)
!-----------------------------------------------------------------------

IMPLICIT NONE

#include <netcdf.inc>

INTEGER, INTENT(IN) :: istatus
CHARACTER(LEN=*)    :: whatfile
INTEGER, INTENT(IN) :: iline

IF (istatus /= NF_NOERR) THEN
  IF (iline > 0) THEN
    WRITE(jp_stderr,"('[',A,':',I0,']: ', A)") &
      TRIM(whatfile), iline, TRIM(NF_STRERROR(istatus))
  ELSE
    WRITE(jp_stderr,"('[',A,':???]: ', A)") &
      TRIM(whatfile), TRIM(NF_STRERROR(istatus))
  ENDIF

  PRINT *, 'NetCDF error detected; aborting.'

  CALL ABORT_EXECUTION()

ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE HANDLE_NCERRORS
!-----------------------------------------------------------------------


END
