!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#ifdef CFN_THISFILE
#undef CFN_THISFILE
#endif
#define CFN_THISFILE "medcol2xy.F90"
#ifndef __LINE__
#define __LINE__ 0
#endif

#ifdef DEBUG
#ifndef VERBOSE
#define VERBOSE
#endif
#endif
!=======================================================================
 PROGRAM MEDCOL2XY
!=======================================================================


USE MOD_UTICOMMON,                  ONLY: ABORT_EXECUTION, jp_stderr, &
                                          c_dirsep, jp_lmaxpathname


IMPLICIT NONE


CHARACTER(LEN=*), PARAMETER    :: cfn_nml          = "medcol2xy.nml"

CHARACTER(LEN=*), PARAMETER    :: cdn_nml_defaults = "/tmp"
CHARACTER(LEN=*), PARAMETER    :: cfn_nml_defaults = "medcol2xy_def.nml"
CHARACTER(LEN=jp_lmaxpathname) :: cpn_nml_defaults


CHARACTER(LEN=jp_lmaxpathname) :: cfn_ncin_res, cfn_ncin_aux
CHARACTER(LEN=jp_lmaxpathname) :: cfn_ncout_res
CHARACTER(LEN=jp_lmaxpathname) :: cfn_tmp


#include <netcdf.inc>

INTEGER :: ncid_file_in
INTEGER :: ncid_file_aux
INTEGER :: ncid_file_out

INTEGER :: ncid_dim_lev_in,   ncid_var_lev_in,  nlen_dim_lev
INTEGER :: ncid_dim_vtx_in,   ncid_var_vtx_in,  nlen_dim_vtx
INTEGER :: ncid_dim_col_in,   ncid_var_col_in,  nlen_dim_col
INTEGER :: ncid_dim_time_in,  ncid_var_time_in, nlen_dim_time

INTEGER :: ncid_dim,          ncid_var

INTEGER :: ncid_dim_lev_out,  ncid_var_lev_out
INTEGER :: ncid_dim_vtx_out,  ncid_var_vtx_out

INTEGER ::                    ncid_var_col4ij
INTEGER :: ndims_in, nvars_in
INTEGER :: ndims_out
INTEGER, DIMENSION(:), ALLOCATABLE :: nlen_dims_out
INTEGER, DIMENSION(:), ALLOCATABLE :: nlen_dims_in
INTEGER :: ndims, nvars, natts, itype
INTEGER :: ndims_var_out
CHARACTER(LEN=NF_MAX_NAME) :: cname_dim_x, cname_dim_y
CHARACTER(LEN=NF_MAX_NAME) :: cname_dim
CHARACTER(LEN=256) :: cname_vars_extra
CHARACTER(LEN=NF_MAX_NAME) :: cname_extra
CHARACTER(LEN=NF_MAX_NAME) :: cname_var
CHARACTER(LEN=NF_MAX_NAME) :: cname_att
INTEGER, DIMENSION(:), POINTER  :: iptr_begin, nptr_len
INTEGER :: i_timeslice, itime

CHARACTER(LEN=NF_MAX_NAME), DIMENSION(:),   ALLOCATABLE :: cname_dims_in
CHARACTER(LEN=NF_MAX_NAME), DIMENSION(:),   ALLOCATABLE :: cname_vars_in
INTEGER,                    DIMENSION(:),   ALLOCATABLE :: ihow2processvars_in
INTEGER,                    DIMENSION(:),   ALLOCATABLE :: ncid_vars_out
INTEGER,                    DIMENSION(:),   ALLOCATABLE :: itype_vars_in

INTEGER,                    DIMENSION(3) :: istart, ncount


INTEGER :: ncid_dim_x_aux,  ncid_var_x_aux, ix_aux, nlen_dim_x_aux
INTEGER :: ncid_dim_y_aux,  ncid_var_y_aux, jy_aux, nlen_dim_y_aux

INTEGER :: ncid_dim_x_out,  ncid_var_x_out, ix_out, nlen_dim_x_out
INTEGER :: ncid_dim_y_out,  ncid_var_y_out, jy_out, nlen_dim_y_out
INTEGER :: istatus

INTEGER, DIMENSION(2) :: ncid_dim2
INTEGER, DIMENSION(3) :: ncid_dim3

LOGICAL :: l_exists
INTEGER :: ix, i1, i2
INTEGER :: i, j, k, n, ivar, nlen_levtx

INTEGER, DIMENSION(:,:), ALLOCATABLE :: icol4ij
INTEGER :: i_nonesuch

INTEGER                                         :: iscalar
INTEGER,          DIMENSION(:),     ALLOCATABLE :: iarr1_in
INTEGER,          DIMENSION(:,:),   ALLOCATABLE :: iarr2_out
INTEGER,          DIMENSION(:,:),   ALLOCATABLE :: iarr2_in
INTEGER,          DIMENSION(:,:,:), ALLOCATABLE :: iarr3_out
DOUBLE PRECISION                                :: dscalar
DOUBLE PRECISION, DIMENSION(:),     ALLOCATABLE :: darr1_in
DOUBLE PRECISION, DIMENSION(:,:),   ALLOCATABLE :: darr2_out
DOUBLE PRECISION, DIMENSION(:,:),   ALLOCATABLE :: darr2_in
DOUBLE PRECISION, DIMENSION(:,:,:), ALLOCATABLE :: darr3_out

INTEGER, PARAMETER  :: jp_lunit  = 100
CHARACTER(LEN=*), PARAMETER :: cfn_thisfile = CFN_THISFILE


#ifdef VERBOSE
WRITE(*, '()')
WRITE(*, '("This is MEDCOL2XY")')
WRITE(*, '("=================")')
WRITE(*, '()')
#endif


CALL GET_MENU(cfn_ncin_res, cfn_ncin_aux,  &
                                    cfn_ncout_res, &
                                    cname_dim_x, cname_dim_y, &
                                    cname_vars_extra, i_timeslice)


                                    ! Check validity of filenames
IF (cfn_ncin_res == "/dev/null") THEN
  WRITE(jp_stderr, '("NetCDF IN file missing - Aborting!")')
  CALL ABORT_EXECUTION()
ELSE
  INQUIRE(FILE=cfn_ncin_res, EXIST=l_exists)
  IF (.NOT.l_exists) THEN
    WRITE(jp_stderr, '("""", A, """: no such file - Aborting!")') TRIM(cfn_ncin_res)
    CALL ABORT_EXECUTION()
  ENDIF
ENDIF


IF (cfn_ncin_aux == "/dev/null") THEN
  WRITE(jp_stderr, '("NetCDF AUX file name missing - Aborting!")')
  CALL ABORT_EXECUTION()
ELSE
  INQUIRE(FILE=cfn_ncin_aux, EXIST=l_exists)
  IF (.NOT.l_exists) THEN
    WRITE(jp_stderr, '("""", A, """: no such file - Aborting!")') TRIM(cfn_ncin_aux)
    CALL ABORT_EXECUTION()
  ENDIF
ENDIF


IF (cfn_ncout_res == "/dev/null") THEN
# ifdef VERBOSE
  WRITE(*, '("Missing name for the NetCDF OUT file")')
# endif
  ix = INDEX(cfn_ncin_res, '.nc', BACK=.TRUE.)
  IF (ix > 1) THEN
    cfn_ncout_res = cfn_ncin_res(1:ix-1) // '_mapped.nc'
#   ifdef VERBOSE
    WRITE(*, '("Using default name """, A, """")') TRIM(cfn_ncout_res)
#   endif
  ELSE
#   ifdef VERBOSE
    WRITE(*, '("Unable to generate name for NetCDF OUT file - Aborting!")')
#   endif
    CALL ABORT_EXECUTION()
  ENDIF
ENDIF


#ifdef VERBOSE
WRITE(*, '()')
WRITE(*, '("Using NCIN file """,A,"""")') TRIM(cfn_ncin_res)
WRITE(*, '("Using AUX file """,A,"""")')  TRIM(cfn_ncin_aux)
WRITE(*, '("Creating NCOUT file """,A,"""")') TRIM(cfn_ncout_res)
WRITE(*, '("Using dimension """, A,""" as X-dimension in AUX")') TRIM(cname_dim_x)
WRITE(*, '("Using dimension """, A,""" as Y-dimension in AUX")') TRIM(cname_dim_y)
WRITE(*, '("Extra variables to copy from AUX: """, A,"""")')  TRIM(cname_vars_extra)
WRITE(*, '("Time slice = ", I0)')                  i_timeslice
WRITE(*, '()')
#endif


                                    ! Open NCIN file
                                    ! ==============
istatus = NF_OPEN(cfn_ncin_res, NF_NOWRITE, ncid_file_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! Query characteristics of the 'lev' dimension
istatus = NF_INQ_DIMID(ncid_file_in, 'lev', ncid_dim_lev_in)
IF (istatus /= NF_NOERR) THEN
  IF (istatus == NF_EBADDIM) THEN
    ncid_dim_lev_in = -1
  ELSE
    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-5))
  ENDIF
ENDIF

IF (ncid_dim_lev_in /= -1) THEN
  istatus = NF_INQ_DIMLEN(ncid_file_in, ncid_dim_lev_in, nlen_dim_lev)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
ELSE
  nlen_dim_lev = 0
ENDIF

                                    ! Query characteristics of the 'vtx' dimension
istatus = NF_INQ_DIMID(ncid_file_in, 'vtx', ncid_dim_vtx_in)
IF (istatus /= NF_NOERR) THEN
  IF (istatus == NF_EBADDIM) THEN
    ncid_dim_vtx_in = -1
  ELSE
    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-5))
  ENDIF
ENDIF

IF (ncid_dim_vtx_in /= -1) THEN
  istatus = NF_INQ_DIMLEN(ncid_file_in, ncid_dim_vtx_in, nlen_dim_vtx)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
ELSE
  nlen_dim_vtx = 0
ENDIF



                                    ! Query characteristics of the 'col' dimension
istatus = NF_INQ_DIMID(ncid_file_in, 'col', ncid_dim_col_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_INQ_DIMLEN(ncid_file_in, ncid_dim_col_in, nlen_dim_col)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


                                    ! Query characteristics of the 'time' dimension
istatus = NF_INQ_DIMID(ncid_file_in, 'time', ncid_dim_time_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_INQ_DIMLEN(ncid_file_in, ncid_dim_time_in, nlen_dim_time)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))



                                    ! Open AUX file
                                    ! =============
istatus = NF_OPEN(cfn_ncin_aux, NF_NOWRITE, ncid_file_aux)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! Query characteristics of the X dimension
istatus = NF_INQ_DIMID(ncid_file_aux, cname_dim_x, ncid_dim_x_aux)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_INQ_DIMLEN(ncid_file_aux, ncid_dim_x_aux, nlen_dim_x_aux)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! Get the var ID of the dimension
                                    ! variable of the X dimension
istatus = NF_INQ_VARID(ncid_file_aux, cname_dim_x, ncid_var_x_aux)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

#ifdef DEBUG
WRITE(*,'(" X dimension length in AUX: ", I0)') nlen_dim_x_aux
#endif

                                    ! Query characteristics of the Y dimension
istatus = NF_INQ_DIMID(ncid_file_aux, cname_dim_y, ncid_dim_y_aux)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_INQ_DIMLEN(ncid_file_aux, ncid_dim_y_aux, nlen_dim_y_aux)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! Get the var ID of the dimension
                                    ! variable of the Y dimension
istatus = NF_INQ_VARID(ncid_file_aux, cname_dim_y, ncid_var_y_aux)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

#ifdef DEBUG
WRITE(*,'(" Y dimension length in AUX: ", I0)') nlen_dim_y_aux
WRITE(*,'()')
#endif


                                    ! Read in conversion tables

ALLOCATE(icol4ij(nlen_dim_x_aux, nlen_dim_y_aux))

istatus = NF_INQ_VARID(ncid_file_aux, 'col4ij', ncid_var_col4ij)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_GET_VAR_INT(ncid_file_aux, ncid_var_col4ij, icol4ij)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_GET_ATT_INT(ncid_file_aux, ncid_var_col4ij, '_FillValue', i_nonesuch)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))





! Create NCOUT file

nlen_dim_x_out = nlen_dim_x_aux
nlen_dim_y_out = nlen_dim_y_aux


#ifdef DEBUG
WRITE(*,'(" X dimension length in OUT: ", I0)') nlen_dim_x_out
WRITE(*,'(" Y dimension length in OUT: ", I0)') nlen_dim_y_out
WRITE(*,'()')
#endif

                                    ! Create Output file
                                    ! ==================
istatus = NF_CREATE(cfn_ncout_res, NF_CLOBBER, ncid_file_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! Define dimension 'lev'
IF (nlen_dim_lev > 0) THEN
  istatus = NF_DEF_DIM(ncid_file_out, 'lev', nlen_dim_lev, ncid_dim_lev_out)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
ENDIF

                                    ! Define dimension 'vtx'
IF (nlen_dim_vtx > 0) THEN
  istatus = NF_DEF_DIM(ncid_file_out, 'vtx', nlen_dim_vtx, ncid_dim_vtx_out)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
ENDIF

                                    ! Dimension 'X'
                                    ! - define dimension
istatus = NF_DEF_DIM(ncid_file_out, cname_dim_x, nlen_dim_x_out, ncid_dim_x_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! - copy dimension variable from AUX to OUT
istatus = NF_COPY_VAR(ncid_file_aux, ncid_var_x_aux, ncid_file_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! - put back file to "define" mode
                                    !   (NF_COPY_VAR has ended this).
istatus = NF_REDEF(ncid_file_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! - retrieve variable ID in OUT
istatus = NF_INQ_VARID(ncid_file_out, cname_dim_x, ncid_var_x_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! Dimension 'Y'
                                    ! - define dimension
istatus = NF_DEF_DIM(ncid_file_out, cname_dim_y, nlen_dim_y_out, ncid_dim_y_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! - copy dimension variable from AUX to OUT
istatus = NF_COPY_VAR(ncid_file_aux, ncid_var_y_aux, ncid_file_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! - put back file to "define" mode
                                    !   (NF_COPY_VAR has ended this).
istatus = NF_REDEF(ncid_file_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! - retrieve variable ID in OUT
istatus = NF_INQ_VARID(ncid_file_out, cname_dim_y, ncid_var_y_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


                                    ! Copy over additional variables, if any
IF (LEN_TRIM(cname_vars_extra) > 0) THEN
# ifdef DEBUG
  WRITE(*, '("Copying extra variables from the AUX to the NCOUT file")')
# endif
  CALL DELIMIT_STRING_TOKENS(cname_vars_extra, ' ,;:', iptr_begin, nptr_len)

  DO j = 1, SIZE(iptr_begin)

    i1 = iptr_begin(j)
    i2 = iptr_begin(j) + nptr_len(j)-1
    cname_extra = cname_vars_extra(i1:i2)

    IF (cname_extra == cname_dim_x) THEN
#     ifdef VERBOSE
      WRITE(*, '("Skipping X dimension variable """, A, """ - already included")') TRIM(cname_extra)
#     endif
      CYCLE
    ENDIF

    IF (cname_extra == cname_dim_y) THEN
#     ifdef VERBOSE
      WRITE(*, '("Skipping Y dimension variable """, A, """ - already included")') TRIM(cname_extra)
#     endif
      CYCLE
    ENDIF

#   ifdef DEBUG
    WRITE(*, '("Querying variable """, A, """ in the AUX file")', ADVANCE='NO') TRIM(cname_extra)
#   endif
    istatus = NF_INQ_VARID(ncid_file_aux, cname_extra, ncid_var)
    IF (istatus /= NF_NOERR) THEN
#     ifdef DEBUG
      WRITE(*,'(" - not found, skipping.")')
#     else
      CONTINUE
#     endif
    ELSE
#     ifdef DEBUG
      WRITE(*,'(" - found, copying.")')
#     endif
      istatus = NF_COPY_VAR(ncid_file_aux, ncid_var, ncid_file_out)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
                                    ! Put back the OUT file into define mode
                                    ! NF_COPY_VAR ends the define mode.
      istatus = NF_REDEF(ncid_file_out)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
    ENDIF

  ENDDO

# ifdef DEBUG
ELSE

  WRITE(*, '("No extra variables to copy from the AUX to the NCOUT file")')
# endif

ENDIF

                                    ! Add 'axis' attribute
istatus = NF_PUT_ATT_TEXT(ncid_file_out, ncid_var_x_out, 'axis', 1, 'X')
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_PUT_ATT_TEXT(ncid_file_out, ncid_var_y_out, 'axis', 1, 'Y')
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


! For each variable that
!  - is not a dimension variable (ndim = 1, varname = dimname)
!  - has 'col' among its dimensions
! we 
!  - create a mapped variable where the 'col' dimension is replaced
!    by the 'X,Y' couple of dimensions, with 'time' dimension omitted.
!  - copy over the attributes from the original file
!  - extract time slice
!  - convert from 'col'-ordering to 'X,Y'-ordering
! Add col_id(x,y) array for reference purposes.


! Variable in the NCIN file may be of the following shapes:
! and are remapped as follows
! 0 scalar           --> scalar    -- (0) copy
! 1 (lev)            --> (lev)     -- (0) copy
! 2 (vtx)            --> (vtx)     -- (0) copy
! 3 (col)            --> (x,y)     -- (3) remap and write
! 4 (time)           --> scalar    -- (4) extract and write
! 5 (lev, time)      --> (lev)     -- (5) extract and write
! 6 (vtx, time)      --> (vtx)     -- (6) extract and write
! 7 (col, time)      --> (x,y)     -- (7) extract and remap first index
! 8 (lev, col, time) --> (lev,x,y) -- (8) extract and remap second index
! 9 (vtx, col, time) --> (vtx,x,y) -- (9) extract and remap second index

                                    ! Get number of dims 
istatus = NF_INQ_NDIMS(ncid_file_in, ndims_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


ALLOCATE(cname_dims_in(ndims_in))

DO i = 1, ndims_in

  istatus = NF_INQ_DIMNAME(ncid_file_in, i, cname_dims_in(i))
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

ENDDO

                                    ! Get number of vars and classify them.
istatus = NF_INQ_NVARS(ncid_file_in, nvars_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

#ifdef DEBUG
WRITE(*, '("Found ", I0, " variable definitions in NCIN file")') nvars_in
#endif

ALLOCATE(cname_vars_in(nvars_in))
ALLOCATE(ihow2processvars_in(nvars_in))
ALLOCATE(ncid_vars_out(nvars_in))
ALLOCATE(itype_vars_in(nvars_in))

ncid_vars_out(:) = -9999

DO j = 1, nvars_in

  istatus = NF_INQ_VARNAME(ncid_file_in, j, cname_var)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
  cname_vars_in(j) = TRIM(cname_var)

  istatus = NF_INQ_VARNDIMS(ncid_file_in, j, ndims)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

  istatus = NF_INQ_VARTYPE(ncid_file_in, j, itype)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
  itype_vars_in(j) = itype


  IF (cname_var == 'col') THEN
#   ifdef DEBUG
    WRITE(*, '("Skipping dimension variable ""col"" in the NCIN file")')
#   endif
    ihow2processvars_in(j) = 0      ! No processing required
    CYCLE
  ENDIF

  IF (cname_var == 'time') THEN
#   ifdef DEBUG
    WRITE(*, '("Skipping dimension variable ""time"" in the NCIN file")')
#   endif
    ihow2processvars_in(j) = 0      ! No processing required
    CYCLE
  ENDIF


# ifdef DEBUG
  WRITE(*, '("Analysing variable """, A, """ in the NCIN file")', ADVANCE='NO') TRIM(cname_var)
# endif


  SELECT CASE(ndims)

  CASE(0)
#   ifdef DEBUG
    WRITE(*,'(" - scalar variable, copying")')
#   endif
    istatus = NF_COPY_VAR(ncid_file_in, j, ncid_file_out)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
    istatus = NF_REDEF(ncid_file_out)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

    ihow2processvars_in(j) = 0    ! Done - no more processing required

  CASE(1)
    istatus = NF_INQ_VARDIMID(ncid_file_in, j, ncid_dim)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

    IF (cname_dims_in(ncid_dim) == cname_var) THEN

#     ifdef DEBUG
      WRITE(*,'(" - dimension variable, copying")')
#     endif
      istatus = NF_COPY_VAR(ncid_file_in, j, ncid_file_out)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      istatus = NF_REDEF(ncid_file_out)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      ihow2processvars_in(j) = 0    ! Done - no more processing required

    ELSE

      IF ((ncid_dim == ncid_dim_lev_in) .OR. (ncid_dim == ncid_dim_vtx_in)) THEN

#       ifdef DEBUG
        WRITE(*,'(" - dimensions (lev) or (vtx), copying")')
#       endif
        istatus = NF_COPY_VAR(ncid_file_in, j, ncid_file_out)
        IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

        istatus = NF_REDEF(ncid_file_out)
        IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

        ihow2processvars_in(j) = 0    ! Done - no more processing required

      ELSEIF (ncid_dim == ncid_dim_col_in) THEN

#       ifdef DEBUG
        WRITE(*,'(" - dimensions (col), needs remapping [3]")')
#       endif
        istatus  = NF_DEF_VAR(ncid_file_out, TRIM(cname_var), itype, &
                                    2, (/ ncid_dim_x_out, ncid_dim_y_out /), &
                                    ncid_vars_out(j))
        IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        ihow2processvars_in(j) = 3    ! Schedule remapping

      ELSEIF (ncid_dim == ncid_dim_time_in) THEN

#       ifdef DEBUG
        WRITE(*,'(" - dimensions (time), needs extraction [4]")')
#       endif
        istatus  = NF_DEF_VAR(ncid_file_out, TRIM(cname_var), itype, &
                                    0, (/ (i,i=1,0) /), &
                                    ncid_vars_out(j))
        IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

        ihow2processvars_in(j) = 4    ! Schedule extraction

      ELSE

        WRITE(jp_stderr,'(" - unknown DIM_ID ", I0, " - aborting")') ncid_dim
        CALL ABORT_EXECUTION()
        
      ENDIF

    ENDIF

    
  CASE(2)

    istatus = NF_INQ_VARDIMID(ncid_file_in, j, ncid_dim2)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

    IF (ALL(ncid_dim2(:) == (/ ncid_dim_lev_in, ncid_dim_time_in /))) THEN

#     ifdef DEBUG
      WRITE(*,'(" - dimensions (lev,time), needs extraction [5]")')
#     endif
      istatus  = NF_DEF_VAR(ncid_file_out, TRIM(cname_var), itype, &
                                    1, (/ ncid_dim_lev_out /), &
                                    ncid_vars_out(j))
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

      ihow2processvars_in(j) = 5    ! Schedule extraction

    ELSEIF (ALL(ncid_dim2(:) == (/ ncid_dim_vtx_in, ncid_dim_time_in /))) THEN

#     ifdef DEBUG
      WRITE(*,'(" - dimensions (vtx,time), needs extraction [6]")')
#     endif
      istatus  = NF_DEF_VAR(ncid_file_out, TRIM(cname_var), itype, &
                                    1, (/ ncid_dim_vtx_out /), &
                                    ncid_vars_out(j))
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

      ihow2processvars_in(j) = 6    ! Schedule extraction

    ELSEIF (ALL(ncid_dim2(:) == (/ ncid_dim_col_in, ncid_dim_time_in /))) THEN

#     ifdef DEBUG
      WRITE(*,'(" - dimensions (col,time), needs extraction and remapping [7]")')
#     endif
      istatus  = NF_DEF_VAR(ncid_file_out, TRIM(cname_var), itype, &
                                    2, (/ ncid_dim_x_out, ncid_dim_y_out /), &
                                    ncid_vars_out(j))
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

      ihow2processvars_in(j) = 7    ! Schedule extraction

    ELSE

        WRITE(jp_stderr,'(" - unknown DIM_IDs (", I0, ",", I0, ") - aborting")') ncid_dim2
        CALL ABORT_EXECUTION()

    ENDIF


  CASE(3)

    istatus = NF_INQ_VARDIMID(ncid_file_in, j, ncid_dim3)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

    IF (ALL(ncid_dim3(:) == (/ ncid_dim_lev_in, ncid_dim_col_in, ncid_dim_time_in /))) THEN

#     ifdef DEBUG
      WRITE(*,'(" - dimensions (lev,col,time), needs extraction and remapping [8]")')
#     endif
      istatus  = NF_DEF_VAR(ncid_file_out, TRIM(cname_var), itype, &
                                    3, (/ ncid_dim_lev_out, ncid_dim_x_out, ncid_dim_y_out /), &
                                    ncid_vars_out(j))
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

      ihow2processvars_in(j) = 8    ! Schedule extraction and remapping

    ELSEIF (ALL(ncid_dim3(:) == (/ ncid_dim_vtx_in, ncid_dim_col_in, ncid_dim_time_in /))) THEN

#     ifdef DEBUG
      WRITE(*,'(" - dimensions (vtx,col,time), needs extraction and remapping [9]")')
#     endif
      istatus  = NF_DEF_VAR(ncid_file_out, TRIM(cname_var), itype, &
                                    3, (/ ncid_dim_vtx_out, ncid_dim_x_out, ncid_dim_y_out /), &
                                    ncid_vars_out(j))
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-3))

      ihow2processvars_in(j) = 9    ! Schedule extraction and remapping

    ELSE

      WRITE(jp_stderr,'(" - unknown DIM_IDs (", I0, ",", I0, ",", I0, ") - aborting")') ncid_dim3
      CALL ABORT_EXECUTION()

    ENDIF

  CASE DEFAULT

    WRITE(jp_stderr,'(" - arrays with ", I0, " cannot yet be processed - aborting")') ndims
    CALL ABORT_EXECUTION()

  END SELECT


  SELECT CASE(ihow2processvars_in(j))
  CASE(0)
    CYCLE

  CASE(3:9)
                                    ! Copy over the attributes
    istatus = NF_INQ_VARNATTS(ncid_file_in, j, natts)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

    DO k = 1, natts

      istatus = NF_INQ_ATTNAME(ncid_file_in, j, k, cname_att)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      istatus = NF_COPY_ATT(ncid_file_in, j, TRIM(cname_att), ncid_file_out, ncid_vars_out(j))
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

    ENDDO

                                    ! Add _FillValue attribute
    SELECT CASE(itype_vars_in(j))
    CASE(NF_INT)
      istatus = NF_PUT_ATT_INT(ncid_file_out, ncid_vars_out(j), &
                                    '_FillValue', NF_INT, 1, (/ NF_FILL_INT /))
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
    CASE(NF_REAL)
      istatus = NF_PUT_ATT_REAL(ncid_file_out, ncid_vars_out(j), &
                                    '_FillValue', NF_REAL, 1, (/ NF_FILL_REAL /))
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
    CASE(NF_DOUBLE)
      istatus = NF_PUT_ATT_DOUBLE(ncid_file_out, ncid_vars_out(j), &
                                    '_FillValue', NF_DOUBLE, 1, (/ NF_FILL_DOUBLE /))
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-2))
    END SELECT

  CASE DEFAULT

    WRITE(jp_stderr,'("Unknown scheduled action [", I0, "] - aborting")') ihow2processvars_in(j)
    CALL ABORT_EXECUTION()

  END SELECT

ENDDO

                                    ! End define mode
istatus = NF_ENDDEF(ncid_file_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! Select time slice
IF ((i_timeslice < 1) .OR. (i_timeslice > nlen_dim_time)) THEN
  itime = nlen_dim_time
ELSE
  itime = i_timeslice
ENDIF


DO ivar = 1, nvars_in

  SELECT CASE(ihow2processvars_in(ivar))

  CASE(0)                           ! Already copied - done!

    CYCLE


  CASE(3)                           ! (col) --> (x,y) - (3) remap and write

    SELECT CASE(itype_vars_in(ivar))

    CASE(NF_INT)

      ALLOCATE(iarr1_in(nlen_dim_col))

      istatus = NF_GET_VAR_INT(ncid_file_in, ivar, iarr1_in)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      ALLOCATE(iarr2_out(nlen_dim_x_out, nlen_dim_y_out))
      iarr2_out(:,:) = NF_FILL_INT

      DO jy_aux = 1, nlen_dim_y_aux
        jy_out = jy_aux
        DO ix_aux = 1, nlen_dim_x_aux
          ix_out = ix_aux  
          n = icol4ij(ix_aux,jy_aux)
          IF (n /= i_nonesuch) iarr2_out(ix_out,jy_out) = iarr1_in(n)
        ENDDO
      ENDDO

      istatus = NF_PUT_VAR_INT(ncid_file_out, ncid_vars_out(ivar), iarr2_out)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      DEALLOCATE(iarr1_in)
      DEALLOCATE(iarr2_out)

    CASE(NF_DOUBLE)

      ALLOCATE(darr1_in(nlen_dim_col))

      istatus = NF_GET_VAR_DOUBLE(ncid_file_in, ivar, darr1_in)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      ALLOCATE(darr2_out(nlen_dim_x_out, nlen_dim_y_out))
      darr2_out(:,:) = NF_FILL_DOUBLE

      DO jy_aux = 1, nlen_dim_y_aux
        jy_out = jy_aux
        DO ix_aux = 1, nlen_dim_x_aux
          ix_out = ix_aux  
          n = icol4ij(ix_aux,jy_aux)
          IF (n /= i_nonesuch) darr2_out(ix_out,jy_out) = darr1_in(n)
        ENDDO
      ENDDO

      istatus = NF_PUT_VAR_DOUBLE(ncid_file_out, ncid_vars_out(ivar), darr2_out)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      DEALLOCATE(darr1_in)
      DEALLOCATE(darr2_out)

    CASE DEFAULT

      WRITE(jp_stderr, '("Case for NetCDF type ", I0, " not yet covered")') itype_vars_in(ivar)
      WRITE(jp_stderr, '("This program needs an extension - aborting!")')
      CALL ABORT_EXECUTION()

    END SELECT


  CASE(4)                           ! (time) --> scalar - (4) extract and write

    istart(1) = itime
    ncount(1) = 1

    SELECT CASE(itype_vars_in(ivar))

    CASE(NF_INT)

      istatus = NF_GET_VARA_INT(ncid_file_in, ivar, istart(1:1), ncount(1:1), iscalar)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      istatus = NF_PUT_VAR_INT(ncid_file_out, ncid_vars_out(ivar), iscalar)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

    CASE(NF_DOUBLE)

      istatus = NF_GET_VARA_DOUBLE(ncid_file_in, ivar, istart(1:1), ncount(1:1), dscalar)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      istatus = NF_PUT_VAR_DOUBLE(ncid_file_out, ncid_vars_out(ivar), dscalar)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

    CASE DEFAULT

      WRITE(jp_stderr, '("Case for NetCDF type ", I0, " not yet covered")') itype_vars_in(ivar)
      WRITE(jp_stderr, '("This program needs an extension - aborting!")')
      CALL ABORT_EXECUTION()

    END SELECT

  CASE(5, 6)                        ! (lev, time) --> (lev) - (5) extract and write
                                    ! (vtx, time) --> (vtx) - (6) extract and write

    IF (ihow2processvars_in(ivar) == 5) THEN
      nlen_levtx = nlen_dim_lev
    ELSE
      nlen_levtx = nlen_dim_vtx
    ENDIF

    istart(1:2) = (/          1, itime /)
    ncount(1:2) = (/ nlen_levtx,     1 /)

    SELECT CASE(itype_vars_in(ivar))

    CASE(NF_INT)

      ALLOCATE(iarr1_in(nlen_levtx))

      istatus = NF_GET_VARA_INT(ncid_file_in, ivar, istart(1:2), ncount(1:2), iarr1_in)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      istatus = NF_PUT_VAR_INT(ncid_file_out, ncid_vars_out(ivar), iarr1_in)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      DEALLOCATE(iarr1_in)


    CASE(NF_DOUBLE)

      ALLOCATE(darr1_in(nlen_levtx))

      istatus = NF_GET_VARA_DOUBLE(ncid_file_in, ivar, istart(1:2), ncount(1:2), darr1_in)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      istatus = NF_PUT_VAR_DOUBLE(ncid_file_out, ncid_vars_out(ivar), darr1_in)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      DEALLOCATE(darr1_in)

    CASE DEFAULT

      WRITE(jp_stderr, '("Case for NetCDF type ", I0, " not yet covered")') itype_vars_in(ivar)
      WRITE(jp_stderr, '("This program needs an extension - aborting!")')
      CALL ABORT_EXECUTION()

    END SELECT


  CASE(7)                           ! (col, time) --> (x,y) - (7) extract and remap first index

    istart(1:2) = (/            1, itime /)
    ncount(1:2) = (/ nlen_dim_col,     1 /)

    SELECT CASE(itype_vars_in(ivar))

    CASE(NF_INT)

      ALLOCATE(iarr1_in(nlen_dim_col))

      istatus = NF_GET_VARA_INT(ncid_file_in, ivar, istart(1:2), ncount(1:2), iarr1_in)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      ALLOCATE(iarr2_out(nlen_dim_x_out, nlen_dim_y_out))
      iarr2_out(:,:) = NF_FILL_INT

      DO jy_aux = 1, nlen_dim_y_aux
        jy_out = jy_aux
        DO ix_aux = 1, nlen_dim_x_aux
          ix_out = ix_aux  
          n = icol4ij(ix_aux,jy_aux)
          IF (n /= i_nonesuch) iarr2_out(ix_out,jy_out) = iarr1_in(n)
        ENDDO
      ENDDO

      istatus = NF_PUT_VAR_INT(ncid_file_out, ncid_vars_out(ivar), iarr2_out)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      DEALLOCATE(iarr1_in)
      DEALLOCATE(iarr2_out)


    CASE(NF_DOUBLE)

      ALLOCATE(darr1_in(nlen_dim_col))

      istatus = NF_GET_VARA_DOUBLE(ncid_file_in, ivar, istart(1:2), ncount(1:2), darr1_in)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      ALLOCATE(darr2_out(nlen_dim_x_out, nlen_dim_y_out))
      darr2_out(:,:) = NF_FILL_DOUBLE

      DO jy_aux = 1, nlen_dim_y_aux
        jy_out = jy_aux
        DO ix_aux = 1, nlen_dim_x_aux
          ix_out = ix_aux  
          n = icol4ij(ix_aux,jy_aux)
          IF (n /= i_nonesuch) darr2_out(ix_out,jy_out) = darr1_in(n)
        ENDDO
      ENDDO

      istatus = NF_PUT_VAR_DOUBLE(ncid_file_out, ncid_vars_out(ivar), darr2_out)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      DEALLOCATE(darr1_in)
      DEALLOCATE(darr2_out)

    CASE DEFAULT

      WRITE(jp_stderr, '("Case for NetCDF type ", I0, " not yet covered")') itype_vars_in(ivar)
      WRITE(jp_stderr, '("This program needs an extension - aborting!")')
      CALL ABORT_EXECUTION()

    END SELECT

  CASE(8, 9)                        ! (lev, col, time) --> (lev,x,y) - (8) extract and remap second index
                                    ! (vtx, col, time) --> (vtx,x,y) - (9) extract and remap second index

    IF (ihow2processvars_in(ivar) == 8) THEN
      nlen_levtx = nlen_dim_lev
    ELSE
      nlen_levtx = nlen_dim_vtx
    ENDIF

    istart(1:3) = (/          1,            1, itime /)
    ncount(1:3) = (/ nlen_levtx, nlen_dim_col,     1 /)

    SELECT CASE(itype_vars_in(ivar))

    CASE(NF_INT)

      ALLOCATE(iarr2_in(nlen_levtx, nlen_dim_col))

      istatus = NF_GET_VARA_INT(ncid_file_in, ivar, istart(1:3), ncount(1:3), iarr2_in)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      ALLOCATE(iarr3_out(nlen_levtx, nlen_dim_x_out, nlen_dim_y_out))
      iarr3_out(:,:,:) = NF_FILL_INT

      DO jy_aux = 1, nlen_dim_y_aux
        jy_out = jy_aux
        DO ix_aux = 1, nlen_dim_x_aux
          ix_out = ix_aux  
          n = icol4ij(ix_aux,jy_aux)
          IF (n /= i_nonesuch) iarr3_out(:, ix_out,jy_out) = iarr2_in(:, n)
      ENDDO
      ENDDO

      istatus = NF_PUT_VAR_INT(ncid_file_in, ncid_vars_out(ivar), iarr3_out)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      DEALLOCATE(iarr2_in)
      DEALLOCATE(iarr3_out)


    CASE(NF_DOUBLE)

      ALLOCATE(darr2_in(nlen_levtx, nlen_dim_col))

      istatus = NF_GET_VARA_DOUBLE(ncid_file_in, ivar, istart(1:3), ncount(1:3), darr2_in)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      ALLOCATE(darr3_out(nlen_levtx, nlen_dim_x_out, nlen_dim_y_out))
      darr3_out(:,:,:) = NF_FILL_DOUBLE

      DO jy_aux = 1, nlen_dim_y_aux
        jy_out = jy_aux
        DO ix_aux = 1, nlen_dim_x_aux
          ix_out = ix_aux  
          n = icol4ij(ix_aux,jy_aux)
          IF (n /= i_nonesuch) darr3_out(:, ix_out,jy_out) = darr2_in(:, n)
      ENDDO
      ENDDO

      istatus = NF_PUT_VAR_DOUBLE(ncid_file_out, ncid_vars_out(ivar), darr3_out)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

      DEALLOCATE(darr2_in)
      DEALLOCATE(darr3_out)

    CASE DEFAULT

      WRITE(jp_stderr, '("Case for NetCDF type ", I0, " not yet covered")') itype_vars_in(ivar)
      WRITE(jp_stderr, '("This program needs an extension - aborting!")')
      CALL ABORT_EXECUTION()

    END SELECT

  END SELECT

ENDDO

                                    ! Close files
istatus = NF_CLOSE(ncid_file_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_CLOSE(ncid_file_aux)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_CLOSE(ncid_file_out)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))



cpn_nml_defaults = TRIM(cdn_nml_defaults) // TRIM(c_dirsep) // TRIM(cfn_nml_defaults)

#ifdef VERBOSE
WRITE(*, '()')
WRITE(*, '("Saving the current configuration to """, A, """")') TRIM(cpn_nml_defaults)
WRITE(*, '("for usage as default next time. Please move that file to the")')
WRITE(*, '("current directory and rename it to """, A, """")') cfn_nml
WRITE(*, '("to use the same configuration the next times (without asking).")')
#endif

OPEN(UNIT=jp_lunit, FILE=cpn_nml_defaults)

WRITE(jp_lunit, '("!=======================================================================")')
WRITE(jp_lunit, '("! MEDCOL2XY requests namelist from the most recent successful run.")')
WRITE(jp_lunit, '("! Content from this file will be used to derive default values next")')
WRITE(jp_lunit, '("! time, unless the working directory contains a namelist file called")')
WRITE(jp_lunit, '("! """, A, """, which will then always be used,")') cfn_nml
WRITE(jp_lunit, '("! without any questions asked.")')
WRITE(jp_lunit, '("!=======================================================================")')
WRITE(jp_lunit, '("&nml_medcol2xy")')
WRITE(jp_lunit, '("! Input data for MEDCOL2XY")')
WRITE(jp_lunit, '("!  - cfn_ncin_res:     name of Medusa generated NetCDF results file")')
WRITE(jp_lunit, '("!                      (REACLAY, REACTION, PROCRATE, FLX) to remap to 2D")')
WRITE(jp_lunit, '("!                      [mandatory, no default]")')
WRITE(jp_lunit, '("!  - cfn_ncin_aux:     name of the AUX file with the mapping information")')
WRITE(jp_lunit, '("!                      [default: cfn_ncin_aux = ""medusa_aux.nc""]")')
WRITE(jp_lunit, '("!  - cfn_ncout_res:    name of the the NetCDF file to write the remapped")')
WRITE(jp_lunit, '("!                      results to [default: ""_mapped"" appended to the")')
WRITE(jp_lunit, '("!                      base name of <cfn_ncin_res>, i e., inserted")')
WRITE(jp_lunit, '("!                      before the extension "".nc""]")')
WRITE(jp_lunit, '("!  - cname_dim_x:      name of the X-dimension to use in the AUX file")')
WRITE(jp_lunit, '("!                      [mandatory, no default - check AUX file for actual name]")')
WRITE(jp_lunit, '("!  - cname_dim_y:      name of the Y-dimension to use in the AUX file")')
WRITE(jp_lunit, '("!                      [mandatory, no default - check AUX file for actual name]")')
WRITE(jp_lunit, '("!  - cname_vars_extra: list of additional variables to copy from the AUX")')
WRITE(jp_lunit, '("!                      to the remapped file, separated by SPC, commas or")')
WRITE(jp_lunit, '("!                      semicolons [default: cname_vars_extra = """"]")')
WRITE(jp_lunit, '("!  - i_timeslice:      index of the time slice to retrieve from")')
WRITE(jp_lunit, '("!                      <cfn_ncin_res>; when <i_timeslice> is 0 or")')
WRITE(jp_lunit, '("!                      negative, or has a value that exceeds the number")')
WRITE(jp_lunit, '("!                      of ''time'' records in <cfn_ncin_res>, the last")')
WRITE(jp_lunit, '("!                      record is used [default: i_timeslice = 0]")')
WRITE(jp_lunit, '("cfn_ncin_res          = """, A, """")') TRIM(cfn_ncin_res)
WRITE(jp_lunit, '("cfn_ncin_aux          = """, A, """")') TRIM(cfn_ncin_aux)
WRITE(jp_lunit, '("cfn_ncout_res         = """, A, """")') TRIM(cfn_ncout_res)
WRITE(jp_lunit, '("cname_dim_x           = """, A, """")') TRIM(cname_dim_x)
WRITE(jp_lunit, '("cname_dim_y           = """, A, """")') TRIM(cname_dim_y)
WRITE(jp_lunit, '("cname_vars_extra      = """, A, """")') TRIM(cname_vars_extra)
WRITE(jp_lunit, '("i_timeslice           = ", I0)')        i_timeslice
WRITE(jp_lunit, '("/")')
CLOSE(jp_lunit)



!========!
 CONTAINS
!========!

!-----------------------------------------------------------------------
 SUBROUTINE GET_MENU(cfn_ncin_res, cfn_ncin_aux,                       &
                                    cfn_ncout_res,                     &
                                    cname_dim_x, cname_dim_y,          &
                                    cname_vars_extra, i_timeslice      )
!-----------------------------------------------------------------------


IMPLICIT NONE


CHARACTER(LEN=*), INTENT(OUT) ::    cfn_ncin_res
CHARACTER(LEN=*), INTENT(OUT) ::    cfn_ncin_aux
CHARACTER(LEN=*), INTENT(OUT) ::    cfn_ncout_res
CHARACTER(LEN=*), INTENT(OUT) ::    cname_dim_x, cname_dim_y
CHARACTER(LEN=*), INTENT(OUT) ::    cname_vars_extra
INTEGER,          INTENT(OUT) ::    i_timeslice

NAMELIST /nml_medcol2xy/ cfn_ncin_res, cfn_ncin_aux, cfn_ncout_res, &
                                    cname_dim_x, cname_dim_y, &
                                    cname_vars_extra, &
                                    i_timeslice

CHARACTER(LEN=jp_lmaxpathname) :: c_dummy



cfn_ncin_res     = "/dev/null"
cfn_ncin_aux     = "medusa_aux.nc"
cfn_ncout_res    = "/dev/null"

cname_dim_x      = ''
cname_dim_y      = ''

cname_vars_extra = ''

i_timeslice      = 0

                                    ! First try to read from a NAMELIST
                                    ! file in the working directory
INQUIRE(FILE=cfn_nml, EXIST=l_exists)

IF (l_exists) THEN                  ! - if there is one, use it without
                                    !   asking questions

  OPEN(UNIT=jp_lunit, FILE=cfn_nml, STATUS='OLD')
  READ(UNIT=jp_lunit, NML=nml_medcol2xy)
  CLOSE(UNIT=jp_lunit)

# ifdef VERBOSE
  WRITE(*,'("Found NAMELIST file """, A, """")') TRIM(cfn_nml)
  WRITE(*,'("and retrieved the following information from it:")')
  WRITE(*,'(" cfn_ncin_res = """, A, """")') TRIM(cfn_ncin_res)
  WRITE(*,'(" cfn_ncin_aux = """, A, """")') TRIM(cfn_ncin_aux)
  WRITE(*,'(" cfn_ncout_res = """, A, """")') TRIM(cfn_ncout_res)
  WRITE(*,'(" cname_dim_x = """, A, """")') TRIM(cname_dim_x)
  WRITE(*,'(" cname_dim_y = """, A, """")') TRIM(cname_dim_y)
  WRITE(*,'(" cname_vars_extra = """, A, """")') TRIM(cname_vars_extra)
  WRITE(*,'(" i_timeslice = ", I0)') i_timeslice
  WRITE(*,'()')
  WRITE(*,'("If any of the entries above are unsuitable,")')
  WRITE(*,'("please delete """, A, """ and re-run me!")') TRIM(cfn_nml)
  WRITE(*,'()')
# endif

ELSE                                ! - if no NAMELIST file is present
                                    !   in the current working directory,
                                    !   go for an interactive configuration.
  cpn_nml_defaults = TRIM(cdn_nml_defaults) // TRIM(c_dirsep) // TRIM(cfn_nml_defaults)
  INQUIRE(FILE=cpn_nml_defaults, EXIST=l_exists)

  IF (l_exists) THEN                !   if there is one, read its contents
                                    !   and use that as default values
    OPEN(UNIT=jp_lunit, FILE=cpn_nml_defaults, STATUS='OLD')
    READ(UNIT=jp_lunit, NML=nml_medcol2xy)
    CLOSE(UNIT=jp_lunit)
  ENDIF

                                    !   Now, we may start the configuration
# ifndef VERBOSE
  WRITE(*, '()')
  WRITE(*, '("This is MEDCOL2XY")')
  WRITE(*, '("=================")')
  WRITE(*, '()')
# endif

  WRITE(*, '("----------------------------------------------------------")')
  WRITE(*, '("Please enter the following required information")')
  WRITE(*, '("----------------------------------------------------------")')
  WRITE(*, '(" * enter file, directory and names without")')
  WRITE(*, '("   delimiting quotes")')
  WRITE(*, '(" * defaults given between brackets ([...]) will")')
  WRITE(*, '("   be used for fields left empty")')
  WRITE(*, '("----------------------------------------------------------")')
  WRITE(*, '()')

  WRITE(*, '(" - the name of the NCIN file to read from")')
  IF (cfn_ncin_res == "/dev/null") THEN
    WRITE(*,'("   > ")', ADVANCE="NO")
  ELSE
    WRITE(*,'("   [", A, "] > ")', ADVANCE="NO") TRIM(cfn_ncin_res)
  ENDIF

  DO
    READ(*, '(A)') c_dummy

    IF (LEN_TRIM(c_dummy) == 0) THEN
      IF (cfn_ncin_res == "/dev/null") THEN
        WRITE(*, '("   sorry, there is no valid default - please try again > ")', ADVANCE = "NO")
        CYCLE
      ELSE
        EXIT
      ENDIF
    ELSE
      READ(c_dummy, '(A)') cfn_ncin_res
      EXIT
    ENDIF
  ENDDO

  WRITE(*, '("   requesting NCIN file """, A, """")') TRIM(cfn_ncin_res)
  WRITE(*, '()')


  WRITE(*,' (" - the name of the AUX file to read from")')
  IF (cfn_ncin_aux == "/dev/null") THEN
    WRITE(*, '("   [", A, "] > ")', ADVANCE="NO") "medusa_aux.nc"
  ELSE
    WRITE(*, '("   [", A, "] > ")', ADVANCE="NO") TRIM(cfn_ncin_aux)
  ENDIF

  READ(*, '(A)') c_dummy
  IF (LEN_TRIM(c_dummy) == 0) THEN
    IF (cfn_ncin_aux == "/dev/null") cfn_ncin_aux = "medusa_aux.nc"
  ELSE
    READ(c_dummy, '(A)') cfn_ncin_aux
  ENDIF
  WRITE(*, '("   requesting AUX file """, A, """")') TRIM(cfn_ncin_aux)
  WRITE(*, '()')


  WRITE(*,' (" - the name of the NCOUT file to write to")')
  IF (cfn_ncout_res == "/dev/null") THEN
    ix = INDEX(cfn_ncin_res, '.nc', BACK=.TRUE.)
    IF (ix > 1) THEN
      cfn_ncout_res = cfn_ncin_res(1:ix-1) // '_mapped.nc'
      WRITE(*,'("   [", A, "] > ")', ADVANCE="NO") TRIM(cfn_ncout_res)
    ELSE
      WRITE(*,'("   > ")', ADVANCE="NO")
    ENDIF
  ELSE
    WRITE(*,'("   [", A, "] > ")', ADVANCE="NO") TRIM(cfn_ncout_res)
  ENDIF

  DO
    READ(*, '(A)') c_dummy

    IF (LEN_TRIM(c_dummy) == 0) THEN
      IF (cfn_ncout_res == "/dev/null") THEN
        WRITE(*, '("   sorry, there is no valid default - please try again > ")', ADVANCE = "NO")
        CYCLE
      ELSE
        EXIT
      ENDIF
    ELSE
      READ(c_dummy, '(A)') cfn_ncout_res
      EXIT
    ENDIF
  ENDDO

  WRITE(*, '("   requesting NCOUT file """, A, """")') TRIM(cfn_ncout_res)
  WRITE(*, '()')


  WRITE(*,' (" - the name of the X-dimension to use in the AUX file")')
  IF (LEN_TRIM(cname_dim_x) == 0) THEN
    WRITE(*,'("   > ")', ADVANCE="NO")
  ELSE
    WRITE(*,'("   [", A, "] > ")', ADVANCE="NO") TRIM(cname_dim_x)
  ENDIF

  DO
    READ(*, '(A)') c_dummy

    IF (LEN_TRIM(c_dummy) == 0) THEN
      IF (LEN_TRIM(cname_dim_x) == 0) THEN
        WRITE(*, '("   sorry, there is no valid default - please try again > ")', ADVANCE = "NO")
        CYCLE
      ELSE
        EXIT
      ENDIF
    ELSE
      READ(c_dummy, '(A)') cname_dim_x
      EXIT
    ENDIF
  ENDDO

  WRITE(*, '("   requesting dimension """, A, """ as X-dimension in the AUX file")') TRIM(cname_dim_x)
  WRITE(*, '()')


  WRITE(*,' (" - the name of the Y-dimension to use in the AUX file")')
  IF (LEN_TRIM(cname_dim_y) == 0) THEN
    WRITE(*,'("   > ")', ADVANCE="NO")
  ELSE
    WRITE(*,'("   [", A, "] > ")', ADVANCE="NO") TRIM(cname_dim_y)
  ENDIF

  DO
    READ(*, '(A)') c_dummy

    IF (LEN_TRIM(c_dummy) == 0) THEN
      IF (LEN_TRIM(cname_dim_y) == 0) THEN
        WRITE(*, '("   sorry, there is no valid default - please try again > ")', ADVANCE = "NO")
        CYCLE
      ELSE
        EXIT
      ENDIF
    ELSE
      READ(c_dummy, '(A)') cname_dim_y
      EXIT
    ENDIF
  ENDDO

  WRITE(*, '("   requesting dimension """, A, """ as Y-dimension in the AUX file")') TRIM(cname_dim_y)
  WRITE(*, '()')


  WRITE(*,' (" - the list of additional variables to copy from the AUX file")')
  WRITE(*,' ("   (comma-separated list of variable names)")')
  IF (LEN_TRIM(cname_vars_extra) == 0) THEN
    WRITE(*,'("   [none]> ")', ADVANCE="NO")
  ELSE
    WRITE(*,'("   [", A, "] > ")', ADVANCE="NO") TRIM(cname_vars_extra)
  ENDIF

  READ(*, '(A)') c_dummy

  IF (LEN_TRIM(c_dummy) /= 0) THEN
    READ(c_dummy, '(A)') cname_vars_extra
  ENDIF

  IF (LEN_TRIM(cname_vars_extra) == 0) THEN
    WRITE(*, '("   not requesting any additional variables")')
  ELSE
    WRITE(*, '("   requesting the additional variables """, A, """")') TRIM(cname_vars_extra)
  ENDIF
  WRITE(*, '()')


  WRITE(*, '(" - the number of the time slice to extract ")')
  IF (i_timeslice == 0) THEN
    WRITE(*, '("   [0 - last one] > ")', ADVANCE="NO")
  ELSE
    WRITE(*, '("   [", I0, "] > ")', ADVANCE="NO") i_timeslice
  ENDIF

  READ(*, '(A)') c_dummy

  IF (LEN_TRIM(c_dummy) /= 0) THEN
    READ(c_dummy, *) i_timeslice
  ENDIF
  WRITE(*, '("   requesting time slice ", I0)') i_timeslice



# ifdef DEBUG
  WRITE(*, '("NCIN name """, A,"""")')               TRIM(cfn_ncin_res)
  WRITE(*, '("AUX name """, A,"""")')                TRIM(cfn_ncin_aux)
  WRITE(*, '("NCOUT name """, A,"""")')              TRIM(cfn_ncout_res)
  WRITE(*, '("X-dim. name in AUX """, A,"""")')      TRIM(cname_dim_x)
  WRITE(*, '("Y-dim. name in AUX """, A,"""")')      TRIM(cname_dim_y)
  WRITE(*, '("Extra AUX vars to copy """, A,"""")')  TRIM(cname_vars_extra)
  WRITE(*, '("Time slice = ", I0)')                  i_timeslice
# endif

  WRITE(*, '()')
  WRITE(*, '("----------------------------------------------------------")')
  WRITE(*, '()')


ENDIF


RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE GET_MENU
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE HANDLE_NCERRORS(istatus, whatfile, iline)
!-----------------------------------------------------------------------

IMPLICIT NONE

#include <netcdf.inc>

INTEGER, INTENT(IN) :: istatus
CHARACTER(LEN=*)    :: whatfile
INTEGER, INTENT(IN) :: iline

IF (istatus /= NF_NOERR) THEN
  IF (iline > 0) THEN
    WRITE(jp_stderr,"('[',A,':',I0,']: ', A)") &
      TRIM(whatfile), iline, TRIM(NF_STRERROR(istatus))
  ELSE
    WRITE(jp_stderr,"('[',A,':???]: ', A)") &
      TRIM(whatfile), TRIM(NF_STRERROR(istatus))
  ENDIF

  PRINT *, 'NetCDF error detected; aborting.'

  CALL ABORT_EXECUTION()

ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE HANDLE_NCERRORS
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE DELIMIT_STRING_TOKENS(c_in, c_separators, i_begin, n_len)
!-----------------------------------------------------------------------

CHARACTER(LEN=*), INTENT(IN)   :: c_in
CHARACTER(LEN=*), INTENT(IN)   :: c_separators
INTEGER, DIMENSION(:), POINTER :: i_begin
INTEGER, DIMENSION(:), POINTER :: n_len


INTEGER, PARAMETER :: jp_stderr = 0
CHARACTER(LEN=LEN(c_in)) :: c_tmp

INTEGER :: i, ii, i1, i2, i3, n_words

IF(LEN_TRIM(c_separators) == 0) THEN
  WRITE(jp_stderr, '("[DELIMIT_STRING_TOKENS]: No separators provided -- aborting")')
  CALL ABORT_EXECUTION()
ENDIF

c_tmp = ' '
n_words = 0
DO i = 1, LEN_TRIM(c_in)
  c_tmp(i:i) = ' '
  ii = INDEX(c_separators, c_in(i:i))
  IF(ii > 0) THEN
    IF (i > 1) THEN
      SELECT CASE(c_tmp(i-1:i-1))
      CASE(' ')
        CYCLE
      CASE('1')
        c_tmp(i-1:i-1) = '3'  ! one-letter word on previous character
        n_words = n_words + 1
      CASE('2')
        n_words = n_words + 1
      END SELECT
    ELSE
      CYCLE
    ENDIF  
  ELSE
    IF (i == 1) THEN
      c_tmp(i:i) = '1'
    ELSE
      SELECT CASE(c_tmp(i-1:i-1))
      CASE(' ')           ! SPC in pos=i-1
        c_tmp(i:i) = '1'  ! => new word starts at pos=i
      CASE('1')           ! new word has begun at pos=i-1
        c_tmp(i:i) = '2'  ! => provisionally end it at pos=i
      CASE('2')           ! word ended provisionally at pos=i-1
        c_tmp(i-1:i-1) = ' '
        c_tmp(i:i) = '2'  ! => report end to pos=i
      END SELECT
    ENDIF
  END IF
  IF (i == LEN_TRIM(c_in)) THEN
    SELECT CASE(c_tmp(i:i))
    CASE('1')
      c_tmp(i:i) = '3'  ! one-letter word on last character
      n_words = n_words + 1
    CASE('2')
      n_words = n_words + 1
    END SELECT
  ENDIF
END DO


IF(n_words > 0) THEN
  ALLOCATE(i_begin(n_words))
  ALLOCATE(n_len(n_words))

  ii = 1
  DO i = 1, n_words
    i1 = INDEX(c_tmp(ii:),'1')
    i3 = INDEX(c_tmp(ii:),'3')

    IF(i3 > 0) THEN
      IF(i1 > 0) THEN 
        IF(i1 < i3) THEN
          i_begin(i) = i1 + ii - 1
          i2 = INDEX(c_tmp(ii:),'2')
          n_len(i) = i2 - i1 + 1
          ii = ii + i2
        ELSE
          i_begin(i) = i3 + ii - 1
          n_len(i) = 1
          ii = ii + i3
        ENDIF
      ELSE
        i_begin(i) = i3 + ii - 1
        n_len(i) = 1
        ii = ii + i3
      ENDIF
    ELSE
      i_begin(i) = i1 + ii - 1
      i2 = INDEX(c_tmp(ii:),'2')
      n_len(i) = i2 - i1 + 1
      ii = ii + i2
    ENDIF
  ENDDO

ELSE

  NULLIFY(i_begin)
  NULLIFY(n_len)

ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE DELIMIT_STRING_TOKENS
!-----------------------------------------------------------------------

END
