!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#ifdef CFN_THISFILE
#undef CFN_THISFILE
#endif
#define CFN_THISFILE "column_extract2csv.F90"
#ifndef __LINE__
#define __LINE__ 0
#endif

#ifdef DEBUG
#ifndef VERBOSE
#define VERBOSE
#endif
#endif
!=======================================================================
 PROGRAM COLUMN_EXTRACT2CSV
!=======================================================================


   !================================================================!
   ! This version of column_extract2csv handles MEDUSA NetCDF files !
   ! compliant to the File Format Version 3 only. File Format       !
   ! Version 2 is handled by column_extract2csv_ffv2; later File    !
   ! Format Versions require later versions of column_extract2csv.  !
   !================================================================!


USE MOD_UTICOMMON,                  ONLY: ABORT_EXECUTION, TESTDIR, &
                                          EXPAND_LIST, &
                                          jp_stderr, c_dirsep


IMPLICIT NONE


CHARACTER(LEN=*), PARAMETER :: cfn_nml          = "column_extract2csv.nml"

CHARACTER(LEN=*), PARAMETER :: cdn_nml_defaults = "/tmp"
CHARACTER(LEN=*), PARAMETER :: cfn_nml_defaults = "column_extract2csv_def.nml"
CHARACTER(LEN=*), PARAMETER :: cfmt_dbl         = "E23.17"


CHARACTER(LEN=4095) :: cfn_ncin_res
CHARACTER(LEN=4095) :: cfn_ncin_reaclay
CHARACTER(LEN=4095) :: cfn_csvout_res_single
CHARACTER(LEN=4095) :: cdn_columnfiles
CHARACTER(LEN=255)  :: cl_columnlist

INTEGER             :: i_timeslice

INTEGER, DIMENSION(:), POINTER :: ilist_cols2extract

CHARACTER(LEN=4095) :: cfn_csvout_res
CHARACTER(LEN=4095) :: cpn_csvout_res
CHARACTER(LEN=4095) :: cpn_nml_defaults

CHARACTER(LEN=*), PARAMETER :: cfmt_append_dbl = '(","' // cfmt_dbl // ')'

#include <netcdf.inc>

INTEGER :: iv_file_in_fmt

INTEGER :: ncid_file_in
INTEGER :: ncid_file_reaclay
INTEGER :: ncid_file4geom

INTEGER :: ncid_dim_levtx_in, ncid_var_levtx_in, nlen_dim_levtx_in
INTEGER :: ncid_dim_col_in,                      nlen_dim_col_in
INTEGER :: ncid_dim_time_in,  ncid_var_time_in,  nlen_dim_time_in

INTEGER :: ncid_dim_lev_reaclay,   ncid_var_lev_reaclay,  nlen_dim_lev_reaclay
INTEGER :: ncid_dim_col_reaclay,                          nlen_dim_col_reaclay
INTEGER :: ncid_dim_time_reaclay,                         nlen_dim_time_reaclay

INTEGER :: ncid_var


INTEGER :: ncid_dim

INTEGER, DIMENSION(3) :: istart
INTEGER, DIMENSION(2) :: ncid_dim2
INTEGER, DIMENSION(3) :: ncid_dim3


INTEGER :: ndims_in, nvars_in

INTEGER :: ndims, itype

CHARACTER(LEN=NF_MAX_NAME) :: cname_var
INTEGER :: nlen_att
INTEGER :: istatus
INTEGER :: i, j, n, ivar, ilevtx
INTEGER :: ix, is
LOGICAL :: l_exists
LOGICAL :: l_ncin_is_reaclay, l_ncin_is_flx
LOGICAL :: l_csvout_isopen
CHARACTER(LEN=10) :: cname_filetype

INTEGER :: ndn_w2s_in, ndn_s2b_in, ndv_w2s_in, ndv_s2b_in
INTEGER :: ndn_w2s_r,  ndn_s2b_r
INTEGER :: idw
LOGICAL :: l_nb_dbl_tofind, l_nb_rcl_tofind

CHARACTER(LEN=NF_MAX_NAME), DIMENSION(:), ALLOCATABLE :: cname_dims_in
CHARACTER(LEN=NF_MAX_NAME), DIMENSION(:), ALLOCATABLE :: cname_vars_in
INTEGER,                    DIMENSION(:), ALLOCATABLE :: ihow2processvars_in
INTEGER,                    DIMENSION(:), ALLOCATABLE :: itype_vars_in
INTEGER,                    DIMENSION(:), ALLOCATABLE :: ncid_vars_in
INTEGER,                    DIMENSION(:), ALLOCATABLE :: ndims_vars_in
INTEGER :: ihow2process_xgeom_dbswi
INTEGER :: ihow2process_xgeom_phi
INTEGER :: ihow2process_xgeom_dphi
INTEGER :: ncid_var_xgeom_dbswi, ndims_xgeom_dbswi
INTEGER :: ncid_var_xgeom_phi,   ndims_xgeom_phi
INTEGER :: ncid_var_xgeom_dphi , ndims_xgeom_dphi

INTEGER,          DIMENSION(:), ALLOCATABLE :: ifillvalue
REAL,             DIMENSION(:), ALLOCATABLE :: ffillvalue
DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE :: dfillvalue
INTEGER          :: ivalue
REAL             :: fvalue
DOUBLE PRECISION :: dvalue, azdn
DOUBLE PRECISION :: swi_location

INTEGER, PARAMETER  :: jp_lunit  = 100
CHARACTER(LEN=4095) :: c_dummy

INTEGER             :: i_column

CHARACTER(LEN=*), PARAMETER :: cfn_thisfile = CFN_THISFILE


#ifdef VERBOSE
WRITE(*, '()')
WRITE(*, '("This is COLUMN_EXTRACT2CSV")')
WRITE(*, '("==========================")')
WRITE(*, '()')
#endif


CALL GET_MENU(cfn_ncin_res, cfn_ncin_reaclay,  &
                                    cfn_csvout_res_single, &
                                    cdn_columnfiles, &
                                    cl_columnlist, i_timeslice)


CALL EXPAND_LIST(cl_columnlist, ilist_cols2extract)


#ifdef VERBOSE
WRITE(*, '()')
WRITE(*, '("Using NCIN file """,A,"""")')        TRIM(cfn_ncin_res)
WRITE(*, '("Using REACLAY file """,A,"""")')     TRIM(cfn_ncin_reaclay)
WRITE(*, '("Using CSVOUT directory """,A,"""")') TRIM(cdn_columnfiles)
WRITE(*, '()')
#endif

l_exists = TESTDIR(cdn_columnfiles)

IF (.NOT. l_exists) THEN
  WRITE(jp_stderr, '("CSVOUT directory """, A, """ does not exist.")') TRIM(cdn_columnfiles)
  WRITE(jp_stderr, '("Aborting!")')
  CALL ABORT_EXECUTION()
ENDIF


                                    ! Check validity of filenames
IF (cfn_ncin_res == "/dev/null") THEN

  WRITE(jp_stderr, '("NCIN file name missing - Aborting!")')
  CALL ABORT_EXECUTION()

ELSE
                                    ! Does the requested NCIN file exist?
  INQUIRE(FILE=cfn_ncin_res, EXIST=l_exists)
  IF (.NOT.l_exists) THEN
    WRITE(jp_stderr, '("""", A, """: no such file - Aborting!")') TRIM(cfn_ncin_res)
    CALL ABORT_EXECUTION()
  ENDIF

                                    ! Does the requested NCIN file
                                    ! have the format version 3 ?
  istatus = NF_OPEN(cfn_ncin_res, NF_NOWRITE, ncid_file_in)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


                                    !  - Query file format version
  istatus = NF_GET_ATT_INT(ncid_file_in, NF_GLOBAL, 'format_version', iv_file_in_fmt)
  IF (istatus /= NF_NOERR)  THEN
    IF (istatus == NF_ENOTATT) THEN !    Attribute not found, assume version 1!
    ELSE
                                    !    Other error: process it!
      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-5))
    ENDIF
  ENDIF

# ifdef DEBUG
  WRITE(jp_stderr, '("NCIN file has format version ", I0)') iv_file_in_fmt
# endif

  SELECT CASE(iv_file_in_fmt)
  CASE(1)
    WRITE(jp_stderr, '(" File format version 1 is not supported by any")')
    WRITE(jp_stderr, '(" version of ""column_extract2csv"" -- aborting!")')
    CALL ABORT_EXECUTION()

  CASE(2)
    WRITE(jp_stderr, '()')
    WRITE(jp_stderr, '(" This version of COLUMN_EXTRACT2CSV does not handle format version 2.")')
    WRITE(jp_stderr, '(" Please use ""column_extract2csv_ffv2"" to handle that format version.")')
    WRITE(jp_stderr, '(" Aborting!")')
    CALL ABORT_EXECUTION()

  CASE(3)
    CONTINUE

  CASE DEFAULT
    WRITE(jp_stderr, '()')
    WRITE(jp_stderr, '(" This version of COLUMN_EXTRACT2CSV does not handle format version ", I0, ".")') iv_file_in_fmt
    WRITE(jp_stderr, '(" Please check if there is a later version that can handle it.")')
    WRITE(jp_stderr, '(" Aborting!")')
    CALL ABORT_EXECUTION()

  END SELECT

                                    ! Is the NCIN file possibly a REACLAY
                                    ! or a FLX file?

                                    ! All FFV3 compliant files have a
                                    ! global 'file_type' attribute: we use it
  istatus = NF_INQ_ATTLEN(ncid_file_in, NF_GLOBAL, 'file_type', nlen_att)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

  istatus = NF_GET_ATT_TEXT(ncid_file_in, NF_GLOBAL, 'file_type', cname_filetype)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


# ifdef DEBUG
  WRITE(jp_stderr, '("NCIN file has file_type=""", A, """")') cname_filetype(1:nlen_att)
# endif
  SELECT CASE(cname_filetype(1:nlen_att))
  CASE('REACLAY')
    l_ncin_is_reaclay = .TRUE.
    l_ncin_is_flx     = .FALSE.
  CASE('FLX')
    l_ncin_is_reaclay = .FALSE.
    l_ncin_is_flx     = .TRUE.
  CASE DEFAULT
    l_ncin_is_reaclay = .FALSE.
    l_ncin_is_flx     = .FALSE.
  END SELECT

  istatus = NF_CLOSE(ncid_file_in)
  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


ENDIF


IF (cfn_ncin_reaclay == "/dev/null") THEN

  WRITE(jp_stderr, '("NetCDF REACLAY file missing:")')

  IF (l_ncin_is_reaclay) THEN
    WRITE(jp_stderr, '(" NCIN file has file_type=""REACLAY""")')
    cfn_ncin_reaclay = cfn_ncin_res
  ELSE
    WRITE(jp_stderr, '(" NCIN file is not a REACLAY file - Aborting!")')
    CALL ABORT_EXECUTION()
  ENDIF


ELSE

  INQUIRE(FILE=cfn_ncin_reaclay, EXIST=l_exists)
  IF (.NOT.l_exists) THEN
    WRITE(jp_stderr, '("""", A, """: no such file - Aborting!")') TRIM(cfn_ncin_reaclay)
    CALL ABORT_EXECUTION()
  ENDIF

                                    ! Is cfn_ncin_reaclay indeed a REACLAY file
  istatus = NF_OPEN(cfn_ncin_reaclay, NF_NOWRITE, ncid_file_reaclay)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

  istatus = NF_INQ_ATTLEN(ncid_file_reaclay, NF_GLOBAL, 'file_type', nlen_att)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

  istatus = NF_GET_ATT_TEXT(ncid_file_reaclay, NF_GLOBAL, 'file_type', cname_filetype)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

  IF (cname_filetype(1:nlen_att) == 'REACLAY') THEN
    CONTINUE
  ELSE
    WRITE(jp_stderr, '("""", A, """ is not a REACLAY file - Aborting!")') TRIM(cfn_ncin_reaclay)
    CALL ABORT_EXECUTION()
  ENDIF

  istatus = NF_CLOSE(ncid_file_reaclay)
  CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

ENDIF


                                    ! Open NCIN file
                                    ! ==============

istatus = NF_OPEN(cfn_ncin_res, NF_NOWRITE, ncid_file_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


                                    ! Query characteristics of the 'col' dimension
istatus = NF_INQ_DIMID(ncid_file_in, 'col', ncid_dim_col_in)
IF (istatus /= NF_NOERR) THEN

  IF (istatus == NF_EBADDIM) THEN
    WRITE(*, '("File """,A,""" does not include a ""col"" dimension")') TRIM(cfn_ncin_res)
    WRITE(*, '(" There is nothing to extract - Done!")')
    istatus = NF_CLOSE(ncid_file_in)
    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
    GOTO 9999
  ELSE
    CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-10))
  ENDIF

ELSE
                                    ! Retrieve extent of column numbers
  istatus = NF_INQ_DIMLEN(ncid_file_in, ncid_dim_col_in, nlen_dim_col_in)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

ENDIF

                                    ! If the NCIN file s a FLX file, its
                                    ! vertical dimension is 'vtx', not 'lev'
IF (l_ncin_is_flx) THEN

  istatus = NF_INQ_DIMID(ncid_file_in, 'vtx', ncid_dim_levtx_in)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

  istatus = NF_INQ_DIMLEN(ncid_file_in, ncid_dim_levtx_in, nlen_dim_levtx_in)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

  istatus = NF_INQ_VARID(ncid_file_in, 'vtx', ncid_var_levtx_in)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

  istatus = NF_GET_ATT_INT(ncid_file_in, ncid_var_levtx_in, 'nb_vertices_DBL', ndv_w2s_in)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

  istatus = NF_GET_ATT_INT(ncid_file_in, ncid_var_levtx_in, 'nb_vertices_Reaclay', ndv_s2b_in)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

  ndn_w2s_in = ndv_w2s_in
  ndn_s2b_in = ndv_s2b_in - 1

ELSE
                                    ! Try to query characteristics of the 'lev' dimension
  istatus = NF_INQ_DIMID(ncid_file_in, 'lev', ncid_dim_levtx_in)
  IF (istatus /= NF_NOERR) THEN

    IF (istatus == NF_EBADDIM) THEN

      nlen_dim_levtx_in = 0
      ndn_w2s_in = 0
      ndn_s2b_in = 0

    ELSE

      CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-38))

    ENDIF

  ELSE

    istatus = NF_INQ_DIMLEN(ncid_file_in, ncid_dim_levtx_in, nlen_dim_levtx_in)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

    istatus = NF_INQ_VARID(ncid_file_in, 'lev', ncid_var_levtx_in)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

    istatus = NF_GET_ATT_INT(ncid_file_in, ncid_var_levtx_in, 'nb_nodes_DBL', ndn_w2s_in)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

    istatus = NF_GET_ATT_INT(ncid_file_in, ncid_var_levtx_in, 'nb_nodes_Reaclay', ndn_s2b_in)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

  ENDIF


ENDIF


                                    ! Query characteristics of the 'time' dimension
istatus = NF_INQ_DIMID(ncid_file_in, 'time', ncid_dim_time_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_INQ_DIMLEN(ncid_file_in, ncid_dim_time_in, nlen_dim_time_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_INQ_VARID(ncid_file_in, 'time', ncid_var_time_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


#ifdef VERBOSE
WRITE(*, '("Dimension lengths of the NCIN file:")')
IF (nlen_dim_levtx_in /= 0) THEN
  IF (l_ncin_is_flx) THEN
    WRITE(*, '(" - ""vtx"" dimension length = ", I0)') nlen_dim_levtx_in
  ELSE
    WRITE(*, '(" - ""lev"" dimension length = ", I0)') nlen_dim_levtx_in
  ENDIF
ELSE
  WRITE(*, '(" - no ""vtx"" dimension")')
  WRITE(*, '(" - no ""lev"" dimension")')
 
ENDIF
WRITE(*, '(" - ""col"" dimension length = ", I0)') nlen_dim_col_in
WRITE(*, '(" - ""time"" dimension length = ", I0)') nlen_dim_time_in
WRITE(*, '()')
#endif


IF (cfn_ncin_reaclay == cfn_ncin_res) THEN

  ncid_file_reaclay     = ncid_file_in

  ncid_dim_lev_reaclay  = ncid_dim_levtx_in
  nlen_dim_lev_reaclay  = nlen_dim_levtx_in

  ncid_dim_col_reaclay  = ncid_dim_col_in
  nlen_dim_col_reaclay  = nlen_dim_col_in

  ncid_dim_time_reaclay  = ncid_dim_time_in
  nlen_dim_time_reaclay  = nlen_dim_time_in

  ndn_w2s_r              = ndn_w2s_in
  ndn_s2b_r              = ndn_s2b_in

ELSE
                                    ! Open REACLAY file
                                    ! ==============
  istatus = NF_OPEN(cfn_ncin_reaclay, NF_NOWRITE, ncid_file_reaclay)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! Query characteristics of the 'lev' dimension
  istatus = NF_INQ_DIMID(ncid_file_reaclay, 'lev', ncid_dim_lev_reaclay)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

  istatus = NF_INQ_DIMLEN(ncid_file_reaclay, ncid_dim_lev_reaclay, nlen_dim_lev_reaclay)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

  IF (l_ncin_is_flx) THEN
    IF (nlen_dim_lev_reaclay /= (nlen_dim_levtx_in - 1)) THEN
      WRITE(jp_stderr, '("Incompatible ""vtx"" and ""lev"" dimension lengths in NCIN and REACLAY files:")')
      WRITE(jp_stderr, '(" - NCIN file:    ", I0)') nlen_dim_levtx_in - 1
      WRITE(jp_stderr, '(" - REACLAY file: ", I0)') nlen_dim_lev_reaclay
      WRITE(jp_stderr, '("Abort!")')
      CALL ABORT_EXECUTION()
    ENDIF
  ELSE
    IF (     (nlen_dim_lev_reaclay /= nlen_dim_levtx_in) &
        .AND. (nlen_dim_levtx_in /= 0)) THEN
      WRITE(jp_stderr, '("Incompatible ""lev"" dimension lengths in NCIN and REACLAY files:")')
      WRITE(jp_stderr, '(" - NCIN file:    ", I0)') nlen_dim_levtx_in
      WRITE(jp_stderr, '(" - REACLAY file: ", I0)') nlen_dim_lev_reaclay
      WRITE(jp_stderr, '("Abort!")')
      CALL ABORT_EXECUTION()
    ENDIF
  ENDIF

  istatus = NF_INQ_VARID(ncid_file_reaclay, 'lev', ncid_var_lev_reaclay)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

  istatus = NF_GET_ATT_INT(ncid_file_reaclay, ncid_var_lev_reaclay, 'nb_nodes_DBL', ndn_w2s_r)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

  istatus = NF_GET_ATT_INT(ncid_file_reaclay, ncid_var_lev_reaclay, 'nb_nodes_Reaclay', ndn_s2b_r)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


                                    ! Query characteristics of the 'col' dimension
  istatus = NF_INQ_DIMID(ncid_file_reaclay, 'col', ncid_dim_col_reaclay)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

  istatus = NF_INQ_DIMLEN(ncid_file_reaclay, ncid_dim_col_reaclay, nlen_dim_col_reaclay)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


                                    ! Query characteristics of the 'time' dimension
  istatus = NF_INQ_DIMID(ncid_file_reaclay, 'time', ncid_dim_time_reaclay)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

  istatus = NF_INQ_DIMLEN(ncid_file_reaclay, ncid_dim_time_reaclay, nlen_dim_time_reaclay)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

#ifdef VERBOSE
  WRITE(*, '("Dimension lengths of the REACLAY file:")')
  WRITE(*, '(" - ""lev"" dimension length = ", I0)') nlen_dim_lev_reaclay
  WRITE(*, '(" - ""col"" dimension length = ", I0)') nlen_dim_col_reaclay
  WRITE(*, '(" - ""time"" dimension length = ", I0)') nlen_dim_time_reaclay
  WRITE(*, '()')
#endif


ENDIF

IF (l_ncin_is_flx) THEN
  ncid_file4geom = ncid_file_in
ELSE
  ncid_file4geom = ncid_file_reaclay
ENDIF
                                    ! Query characteristics of the 'xgeom_dbswi'
                                    ! variable in the REACLAY file
istatus = NF_INQ_VARID(ncid_file4geom, 'xgeom_dbswi', ncid_var_xgeom_dbswi)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_INQ_VARNDIMS(ncid_file4geom, ncid_var_xgeom_dbswi, ndims_xgeom_dbswi)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

SELECT CASE(ndims_xgeom_dbswi)
CASE(1)
  ihow2process_xgeom_dbswi = 1
CASE(2)
  ihow2process_xgeom_dbswi = 3
CASE(3)
  ihow2process_xgeom_dbswi = 7
END SELECT


                                    ! Query characteristics of the 'xgeom_phi'
                                    ! variable in the REACLAY file
istatus = NF_INQ_VARID(ncid_file4geom, 'xgeom_phi', ncid_var_xgeom_phi)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_INQ_VARNDIMS(ncid_file4geom, ncid_var_xgeom_phi, ndims_xgeom_phi)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

SELECT CASE(ndims_xgeom_phi)
CASE(1)
  ihow2process_xgeom_phi  = 1
CASE(2)
  ihow2process_xgeom_phi  = 3
CASE(3)
  ihow2process_xgeom_phi  = 7
END SELECT


                                    ! Query characteristics of the 'xgeom_dphi'
                                    ! variable in the REACLAY file
istatus = NF_INQ_VARID(ncid_file4geom, 'xgeom_dphi', ncid_var_xgeom_dphi)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

istatus = NF_INQ_VARNDIMS(ncid_file4geom, ncid_var_xgeom_dphi, ndims_xgeom_dphi)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

SELECT CASE(ndims_xgeom_dphi)
CASE(1)
  ihow2process_xgeom_dphi  = 1
CASE(2)
  ihow2process_xgeom_dphi  = 3
CASE(3)
  ihow2process_xgeom_dphi  = 7
END SELECT


!                                    ! Complete DBL/REACLAY partitioning if necessary
!istatus = NF_GET_ATT_DOUBLE(ncid_file4geom, ncid_var_xgeom_dbswi, 'swi_location', swi_location)
!IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))


IF (((ndn_w2s_r /= ndn_w2s_in) .OR. (ndn_s2b_r /= ndn_s2b_in)) &
    .AND. (nlen_dim_levtx_in /= 0)) THEN
  WRITE(jp_stderr, '("Incompatible DBL/Reaclay partitioning in IN and REACLAY files:")')
  WRITE(jp_stderr, '(" - NCIN file:    nb_nodes_DBL=", I0)') ndn_w2s_in
  WRITE(jp_stderr, '(" - REACLAY file: nb_nodes_DBL=", I0)') ndn_w2s_r
  WRITE(jp_stderr, '(" - NCIN file:    nb_nodes_Reaclay=", I0)') ndn_s2b_in
  WRITE(jp_stderr, '(" - REACLAY file: nb_nodes_Reaclay=", I0)') ndn_s2b_r
  WRITE(jp_stderr, '("Abort!")')
  CALL ABORT_EXECUTION()
ENDIF

IF (l_ncin_is_flx) THEN
  idw = -ndn_w2s_in - 1
ELSE
  idw = -ndn_w2s_in
ENDIF


IF (cfn_ncin_res /= cfn_ncin_reaclay) THEN

  ! Check col dimension compatibility
  ! Check time dimension compatibility

ENDIF

IF (ndims_xgeom_dbswi > 3) THEN
  WRITE(jp_stderr, '("""xgeom_dbswi"" arrays with ", I0, " dimensions cannot yet be processed - aborting")') ndims_xgeom_dbswi
  CALL ABORT_EXECUTION()
ENDIF


IF (ndims_xgeom_phi > 3) THEN
  WRITE(jp_stderr, '("""xgeom_xphi"" arrays with ", I0, " dimensions cannot yet be processed - aborting")') ndims_xgeom_phi
  CALL ABORT_EXECUTION()
ENDIF


                                    ! If i_timeslice out of bounds use last
                                    ! time record as default
IF ((i_timeslice < 1) .OR. (i_timeslice > nlen_dim_time_reaclay)) THEN
  i_timeslice = nlen_dim_time_reaclay
ENDIF


                                    ! Create CSVOUT file
                                    ! ==================

! Variable in the NCIN file may be of the following shapes:
! and are extracted as follows
! -1 lev, col, time vars         -- (-1) skip
!  0 scalar           --> scalar -- (0) copy and write value
!  1 (lev)            --> (lev)  -- (1) write lev-array
!  2 (col)            --> scalar -- (2) extract and write value
!  3 (time)           --> scalar -- (4) extract and write value
!  4 (lev, time)      --> (lev)  -- (5) extract and write lev-array
!  5 (lev, col)       --> (lev)  -- (3) extract and write lev-array
!  5 (col, time)      --> scalar -- (6) extract and write value
!  6 (lev, col, time) --> (lev)  -- (7) extract and write lev-array


                                    ! Get number of dims
istatus = NF_INQ_NDIMS(ncid_file_in, ndims_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

ALLOCATE(cname_dims_in(ndims_in))

DO i = 1, ndims_in
  istatus = NF_INQ_DIMNAME(ncid_file_in, i, cname_dims_in(i))
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
ENDDO


                                    ! Get number of vars and classify them.
istatus = NF_INQ_NVARS(ncid_file_in, nvars_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

#ifdef DEBUG
WRITE(*, '("Found ", I0, " variable definitions in NCIN file")') nvars_in
#endif

ALLOCATE(ncid_vars_in(nvars_in))
ALLOCATE(cname_vars_in(nvars_in))
ALLOCATE(ihow2processvars_in(nvars_in))
ALLOCATE(itype_vars_in(nvars_in))
ALLOCATE(ndims_vars_in(nvars_in))
ALLOCATE(dfillvalue(nvars_in))
ALLOCATE(ffillvalue(nvars_in))
ALLOCATE(ifillvalue(nvars_in))


DO j = 1, nvars_in

  istatus = NF_INQ_VARNAME(ncid_file_in, j, cname_var)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
  cname_vars_in(j) = TRIM(cname_var)

  istatus = NF_INQ_VARNDIMS(ncid_file_in, j, ndims)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
  ndims_vars_in(j) = ndims

  istatus = NF_INQ_VARTYPE(ncid_file_in, j, itype)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
  itype_vars_in(j) = itype

  SELECT CASE(itype)
  CASE(NF_INT)
    istatus = NF_GET_ATT_INT(ncid_file_in, j, '_FillValue', ifillvalue(j))
    !IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
  CASE(NF_REAL)
    istatus = NF_GET_ATT_REAL(ncid_file_in, j, '_FillValue', ffillvalue(j))
    !IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
  CASE(NF_DOUBLE)
    istatus = NF_GET_ATT_DOUBLE(ncid_file_in, j, '_FillValue', dfillvalue(j))
    !IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
  CASE DEFAULT
    CONTINUE
  END SELECT

  SELECT CASE(cname_var)
  CASE('vtx', 'lev', 'col', 'time')
#   ifdef DEBUG
    WRITE(*, '("Skipping dimension variables in the NCIN file")')
#   endif
    ihow2processvars_in(j) = -1      ! No processing required
    CYCLE

  CASE('xgeom_phi', 'xgeom_dphi','xgeom_dbswi')
    ! Can be (lev), (lev,col) or (lev,col,time)
    ! Skip in the loop, processed separately
#   ifdef DEBUG
    WRITE(*, '("Skipping ""xgeom_dbswi"" - processed separately")')
#   endif
    ihow2processvars_in(j) = -1      ! No processing required
    CYCLE

  CASE DEFAULT
#   ifdef DEBUG
    WRITE(*, '("Analysing variable """, A, """ in the NCIN file")', ADVANCE="NO") TRIM(cname_var)
#   endif
    CONTINUE

  END SELECT


  SELECT CASE(ndims)

  CASE(0)
#   ifdef DEBUG
    WRITE(*, '(" - scalar variable, to copy")')
#   endif
    ihow2processvars_in(j) = 0    ! schedule copying

  CASE(1)
    istatus = NF_INQ_VARDIMID(ncid_file_in, j, ncid_dim)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))



    IF (cname_dims_in(ncid_dim) == cname_var) THEN

#     ifdef DEBUG
      WRITE(*, '(" - dimension variable, already set")')
#     else
      CONTINUE
#     endif


    ELSE

      IF (ncid_dim == ncid_dim_col_in) THEN

#       ifdef DEBUG
        WRITE(*, '(" - dimension (col), extract one single value")')
#       endif
        ihow2processvars_in(j) = 2    ! Schedule single (col) value extraction

      ELSEIF (ncid_dim == ncid_dim_time_in) THEN

#       ifdef DEBUG
        WRITE(*, '(" - dimension (time), extract one single value")')
#       endif
        ihow2processvars_in(j) = 4    ! Schedule single (time) value extraction

      ELSEIF (ncid_dim == ncid_dim_levtx_in) THEN

#       ifdef DEBUG
        IF (l_ncin_is_flx) THEN
          WRITE(*, '(" - dimensions (vtx), copying array(vtx)")')
        ELSE
          WRITE(*, '(" - dimensions (lev), copying array(lev)")')
        ENDIF
#       endif
        ihow2processvars_in(j) = 1    ! Schedule array(lev) copying

      ELSE

        WRITE(jp_stderr, '(" - unknown DIM_ID ", I0, " - aborting")') ncid_dim
        CALL ABORT_EXECUTION()

      ENDIF

    ENDIF


  CASE(2)

    istatus = NF_INQ_VARDIMID(ncid_file_in, j, ncid_dim2)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

    IF     (ALL(ncid_dim2(:) == (/ ncid_dim_levtx_in, ncid_dim_col_in /))) THEN

#     ifdef DEBUG
      IF (l_ncin_is_flx) THEN
        WRITE(*, '(" - dimensions (vtx,col), extract vtx-array [3]")')
      ELSE
        WRITE(*, '(" - dimensions (lev,col), extract lev-array [3]")')
      ENDIF
#     endif

      ihow2processvars_in(j) = 3    ! Schedule array(lev) extraction

    ELSEIF (ALL(ncid_dim2(:) == (/ ncid_dim_levtx_in, ncid_dim_time_in /))) THEN

#     ifdef DEBUG
      IF (l_ncin_is_flx) THEN
        WRITE(*, '(" - dimensions (vtx ,time), extract array(vtx)")')
      ELSE
        WRITE(*, '(" - dimensions (lev ,time), extract array(lev)")')
      ENDIF
#     endif

      ihow2processvars_in(j) = 5    ! Schedule array(lev) extraction

    ELSEIF (ALL(ncid_dim2(:) == (/ ncid_dim_col_in, ncid_dim_time_in /))) THEN

#     ifdef DEBUG
      WRITE(*, '(" - dimensions (col,time), extracting single value")')
#     endif

      ihow2processvars_in(j) = 6    ! Schedule single (col,time) value extraction

    ELSE

        WRITE(jp_stderr, '(" - unknown DIM_IDs (", I0, ",", I0, ") - aborting")') ncid_dim2
        CALL ABORT_EXECUTION()

    ENDIF


  CASE(3)

    istatus = NF_INQ_VARDIMID(ncid_file_in, j, ncid_dim3)
    IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

    IF (ALL(ncid_dim3(:) == (/ ncid_dim_levtx_in, ncid_dim_col_in, ncid_dim_time_in /))) THEN

#     ifdef DEBUG
      IF (l_ncin_is_flx) THEN
        WRITE(*, '(" - dimensions (vtx,col,time), extract vtx-array [7]")')
      ELSE
        WRITE(*, '(" - dimensions (lev,col,time), extract lev-array [7]")')
      ENDIF
#     endif

      ihow2processvars_in(j) = 7    ! Schedule array(lev) extraction

    ELSE

      WRITE(jp_stderr, '(" - unknown DIM_IDs (", I0, ",", I0, ",", I0, ") - aborting")') ncid_dim3
      CALL ABORT_EXECUTION()

    ENDIF

  CASE DEFAULT

    WRITE(jp_stderr, '(" - arrays with ", I0, " dimensions cannot yet be processed - aborting")') ndims
    CALL ABORT_EXECUTION()

  END SELECT

ENDDO


l_csvout_isopen = .FALSE.           ! No CSVOUT file should currently be open

DO i = 1, SIZE(ilist_cols2extract)

  i_column = ilist_cols2extract(i)

  IF ((i_column < 1) .OR. (i_column > nlen_dim_col_in)) THEN
    WRITE(jp_stderr, '("The requested column number ", I0, " is not valid - skipping!")') i_column
    CYCLE
  ENDIF


                                    ! Generate CSVOUT file(s)
                                    ! -----------------------

                                    ! If a single file is to be produced,
                                    ! open it if not yet done.

  IF (cfn_csvout_res_single /= "/dev/null") THEN

    IF (.NOT. l_csvout_isopen) THEN
      cpn_csvout_res = TRIM(cdn_columnfiles) // c_dirsep // TRIM(cfn_csvout_res_single)
      OPEN(UNIT=jp_lunit, FILE=cpn_csvout_res)
      l_csvout_isopen = .TRUE.
    ENDIF

  ELSE
                                    ! If per-column CSVOUT files are to
                                    ! be produced, generate their names
                                    ! on the basis of the name of the
                                    ! NCIN file:

    n = LEN_TRIM(cfn_ncin_res)      ! - get length of actual NCIN file name

                                    ! - check if it has a subdir part
    is = INDEX(cfn_ncin_res, c_dirsep, BACK=.TRUE.)
    IF (is <= 0) is = 0             !   if not, assume a virtual '/'
                                    !   directory separator at position 0

                                    ! - check if the name that follows the
                                    !   (possibly virtual) '/' sign has
                                    !   the extension '.nc':
    ix = INDEX(cfn_ncin_res(is+1:n), '.nc', BACK=.TRUE.)

    IF (ix > 0) THEN                !   * if so, we can use the adopted
                                    !     scheme to generate a column-
                                    !     specific CSVOUT file name;
      WRITE(cfn_csvout_res, '(A, "_col", I5.5, ".csv")') &
        cfn_ncin_res(is+1:is+ix-1) , i_column
#     ifdef VERBOSE
      WRITE(*, '("Adopting default name """, A, """")') TRIM(cfn_csvout_res)
#     endif

    ELSE                            !   * if not, we have, unfortunately,
                                    !     to abort here.
#     ifdef VERBOSE
      WRITE(*, '("Unable to generate name(s) for the CSVOUT file(s) - Aborting!")')
#     endif
      CALL ABORT_EXECUTION()

    ENDIF

                                    ! - and finally open the file
    cpn_csvout_res = TRIM(cdn_columnfiles) // c_dirsep // TRIM(cfn_csvout_res)
    OPEN(UNIT=jp_lunit, FILE=cpn_csvout_res)
    l_csvout_isopen = .TRUE.

  ENDIF

                                    ! Fill CSVOUT file(s)
                                    ! -------------------

                                    ! Write header for scalar data
  WRITE(jp_lunit, '("''col'',''time''")', ADVANCE="NO")

  DO ivar = 1, nvars_in

    SELECT CASE(ihow2processvars_in(ivar))
    CASE(0, 2, 4, 6)
      WRITE(jp_lunit, '(",''", A, "''")', ADVANCE="NO") TRIM(cname_vars_in(ivar))

    CASE DEFAULT
      CYCLE

    END SELECT

  ENDDO

  WRITE(jp_lunit, '()')             ! and end header line with a linefeed

                                    ! Write scalar data

  WRITE(jp_lunit, '(I0)', ADVANCE="NO") i_column

  istatus = NF_GET_VAR1_DOUBLE(ncid_file_in, ncid_var_time_in, i_timeslice, dvalue)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

  WRITE(jp_lunit, cfmt_append_dbl, ADVANCE="NO") dvalue


  DO ivar = 1, nvars_in

    n = -1

    SELECT CASE(ihow2processvars_in(ivar))
    CASE(0)                         ! scalar --> copy
      n = 0

    CASE(2)                         ! 1D (col) array, extract col value
      n = 1
      istart(1) = i_column

    CASE(4)                         ! 1D (time) array, extract time value
      n = 1
      istart(1) = i_timeslice

    CASE(6)                         ! 2D (col,time) array, extract time value
      n = 2
      istart(1:2) = (/ i_column, i_timeslice /)

    END SELECT


    IF (n >= 0) THEN
#     ifdef DEBUG
      WRITE(*, '("Processing variable """, A, """ in the NCIN file")') TRIM(cname_vars_in(ivar))
#     endif

      SELECT CASE(itype_vars_in(ivar))

      CASE(NF_DOUBLE)

        IF (n == 0) THEN
          istatus = NF_GET_VAR_DOUBLE(ncid_file_in, ivar, dvalue)
          IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
        ELSE
          istatus = NF_GET_VAR1_DOUBLE(ncid_file_in, ivar, istart(1:n), dvalue)
          IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
        ENDIF

                                    ! Check for _FillValue content
                                    ! (can occur in Medusa files where
                                    ! the first record has been left empty).
      IF (dvalue == dfillvalue(ivar)) THEN
          WRITE(jp_lunit, '(",")', ADVANCE="NO")
        ELSE
          WRITE(jp_lunit, cfmt_append_dbl, ADVANCE="NO") dvalue
        ENDIF


      CASE DEFAULT
                                    ! Others than NF_DOUBLE cannot yet be handled, sorry
        WRITE(jp_stderr, '("Variable types other than DOUBLE cannot yet be handled - skipping")')
        WRITE(jp_lunit, '(",")', ADVANCE="NO")

      END SELECT

    ENDIF

  ENDDO

  WRITE(jp_lunit, '()')             ! Linefeed for the data line

                                    ! If the NCIN file has a lev or a vtx
                                    ! dimension, write the spatial data out next
  IF (nlen_dim_levtx_in /= 0) THEN

    WRITE(jp_lunit, '()')           ! Linefeed to insert one blank line between
                                    ! the scalar and the spatially resolved data

                                    ! Write header for non-scalar data
    IF (l_ncin_is_flx) THEN
      WRITE(jp_lunit, '("''Vertices''")', ADVANCE="NO")
    ELSE
      WRITE(jp_lunit, '("''Level''")', ADVANCE="NO")
    ENDIF
    WRITE(jp_lunit, '(",''xgeom_dbswi''")', ADVANCE="NO")
    WRITE(jp_lunit, '(",''xgeom_phi''")', ADVANCE="NO")
    WRITE(jp_lunit, '(",''xgeom_dphi''")', ADVANCE="NO")

    DO ivar = 1, nvars_in

      SELECT CASE(ihow2processvars_in(ivar))
      CASE(1, 3, 5, 7)
        WRITE(jp_lunit, '(",''", A, "''")', ADVANCE="NO") TRIM(cname_vars_in(ivar))
      CASE DEFAULT
        CYCLE
      END SELECT

    ENDDO

                                    ! Repeat DBSWI value to simplify profile charts
    WRITE(jp_lunit, '(",''xgeom_dbswi''")', ADVANCE="NO")

    WRITE(jp_lunit, '()')           ! Linefeed for the data line

#   ifdef DEBUG
    WRITE(*,'()')
#   endif

    DO ilevtx = 1, nlen_dim_levtx_in

      WRITE(jp_lunit, '(I0)', ADVANCE="NO") (ilevtx - 1) + idw

      istart(:) = (/ ilevtx, i_column, i_timeslice /)

#     ifdef DEBUG
      IF (l_ncin_is_flx) THEN
        WRITE(*, '("Processing vertex ", I0, "(", I0, ") of ", I0, ": ")') &
          ilevtx, (ilevtx - 1) + idw, nlen_dim_levtx_in
      ELSE
        WRITE(*, '("Processing level ", I0, "(", I0, ") of ", I0, ": ")') &
          ilevtx, (ilevtx - 1) + idw, nlen_dim_levtx_in
      ENDIF
      WRITE(*, '("extracting VAR1 at (", I0, ", ", I0, ", " I0, ")")') istart(:)
#     endif
      n = ndims_xgeom_dbswi
      istatus = NF_GET_VAR1_DOUBLE(ncid_file4geom, ncid_var_xgeom_dbswi, istart(1:n), azdn)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
      WRITE(jp_lunit, cfmt_append_dbl, ADVANCE="NO") azdn

      n = ndims_xgeom_phi
      istatus = NF_GET_VAR1_DOUBLE(ncid_file4geom, ncid_var_xgeom_phi, istart(1:n), dvalue)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
      WRITE(jp_lunit, cfmt_append_dbl, ADVANCE="NO") dvalue

      n = ndims_xgeom_dphi
      istatus = NF_GET_VAR1_DOUBLE(ncid_file4geom, ncid_var_xgeom_dphi, istart(1:n), dvalue)
      IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
      WRITE(jp_lunit, cfmt_append_dbl, ADVANCE="NO") dvalue


      DO ivar = 1, nvars_in

        n = -1

        SELECT CASE(ihow2processvars_in(ivar))
        CASE(1)                     ! (lev) or (vtx) --> get one value from 1D array
          n = 1
          istart(1)   =    ilevtx
        CASE(3)                     ! (lev,col) or (vtx,col) --> get one value from 2D array
          n = 2
          istart(1:2) = (/ ilevtx, i_column /)
        CASE(5)                     ! (lev,time) or (vtx,time)--> get one value from 2D array
          n = 2
          istart(1:2) = (/ ilevtx, i_timeslice /)
        CASE(7)                     ! (lev,col,time) or (vtx,col,time) --> get one value from 3D array
          n = 3
          istart(1:3) = (/ ilevtx, i_column, i_timeslice /)
        END SELECT


        IF (n >= 0) THEN
#         ifdef DEBUG
          WRITE(*, '("Processing variable """, A, """ in the NCIN file")') TRIM(cname_vars_in(ivar))
#         endif

          SELECT CASE(itype_vars_in(ivar))

          CASE(NF_INT)
            istatus = NF_GET_VAR1_INT(ncid_file_in, ivar, istart(1:n), ivalue)
            IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! Check for _FillValue content
                                    ! (can occur in REACLAY files where
                                    ! the first record may be kept empty).
            IF (ivalue == ifillvalue(ivar)) THEN
              WRITE(jp_lunit, '(",")', ADVANCE="NO")
            ELSE
              WRITE(jp_lunit, '(",", I0)', ADVANCE="NO") ivalue
            ENDIF

          CASE(NF_REAL)
            istatus = NF_GET_VAR1_REAL(ncid_file_in, ivar, istart(1:n), fvalue)
            IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! Check for _FillValue content
            IF (fvalue == ffillvalue(ivar)) THEN
              WRITE(jp_lunit, '(",")', ADVANCE="NO")
            ELSE
              WRITE(jp_lunit, '(",", E13.7)', ADVANCE="NO") ivalue
            ENDIF

          CASE(NF_DOUBLE)
            istatus = NF_GET_VAR1_DOUBLE(ncid_file_in, ivar, istart(1:n), dvalue)
            IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! Check for _FillValue content
            IF (dvalue == dfillvalue(ivar)) THEN
              WRITE(jp_lunit, '(",")', ADVANCE="NO")
            ELSE
              WRITE(jp_lunit, cfmt_append_dbl, ADVANCE="NO") dvalue
            ENDIF


          CASE DEFAULT
            WRITE(jp_stderr, '("Variable types of this type cannot yet be handled - skipping")')
            WRITE(jp_lunit, '(",")', ADVANCE="NO")

          END SELECT

        ENDIF

      ENDDO

                                    ! Repeat DBSWI value to simplify profile charts
      WRITE(jp_lunit, cfmt_append_dbl, ADVANCE="NO") azdn
      WRITE(jp_lunit, '()')         ! Linefeed for the data line

    ENDDO

  ENDIF


  WRITE(jp_lunit, '()')             ! Add two linefeed so that individual
  WRITE(jp_lunit, '()')             ! files can be concatenated later on
                                    ! if required

                                    ! Close CSVOUT file if per-column files are used
  IF (cfn_csvout_res_single == "/dev/null") THEN
    CLOSE(jp_lunit)
    l_csvout_isopen = .FALSE.
  ENDIF

ENDDO

                                    ! Close CSVOUT file if single file is used
IF (cfn_csvout_res_single /= "/dev/null")THEN
  CLOSE(jp_lunit)
  l_csvout_isopen = .FALSE.
ENDIF


                                    ! Close NCIN file
istatus = NF_CLOSE(ncid_file_in)
IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))

                                    ! Close REACLAY file if different from NCIN file
IF (cfn_ncin_reaclay /= cfn_ncin_res) THEN
  istatus = NF_CLOSE(ncid_file_reaclay)
  IF (istatus /= NF_NOERR) CALL HANDLE_NCERRORS(istatus, cfn_thisfile, (__LINE__-1))
ENDIF


DEALLOCATE(ncid_vars_in)
DEALLOCATE(cname_vars_in)
DEALLOCATE(ihow2processvars_in)
DEALLOCATE(itype_vars_in)
DEALLOCATE(ndims_vars_in)
DEALLOCATE(dfillvalue)
DEALLOCATE(ffillvalue)
DEALLOCATE(ifillvalue)


9999 CONTINUE


cpn_nml_defaults = TRIM(cdn_nml_defaults) // TRIM(c_dirsep) // TRIM(cfn_nml_defaults)

WRITE(*, '()')
WRITE(*, '("Saving current configuration to """, A, """")') TRIM(cpn_nml_defaults)
WRITE(*, '("for usage as default next time. Please move that file to the")')
WRITE(*, '("current directory and rename it to """, A, """")') cfn_nml
WRITE(*, '("to use the same configuration the next times (without asking).")')

OPEN(UNIT=jp_lunit, FILE=cpn_nml_defaults)

WRITE(jp_lunit, '("! COLUMN_EXTRACT2CSV requests namelist from the most")')
WRITE(jp_lunit, '("! recent successful run. Content from this file will")')
WRITE(jp_lunit, '("! be used to derive default values next time, unless")')
WRITE(jp_lunit, '("! the working directory contains a namelist file")')
WRITE(jp_lunit, '("! called """, A, """, which will always")') cfn_nml
WRITE(jp_lunit, '("! be used without any questions asked.")')
WRITE(jp_lunit, '("&nml_column_extract2csv")')
WRITE(jp_lunit, '("cfn_ncin_res          = """, A, """")') TRIM(cfn_ncin_res)
WRITE(jp_lunit, '("cfn_ncin_reaclay      = """, A, """")') TRIM(cfn_ncin_reaclay)
IF (cfn_csvout_res_single /= "/dev/null") THEN
  WRITE(jp_lunit, '("cfn_csvout_res_single = """, A, """")') TRIM(cfn_csvout_res_single)
ENDIF
WRITE(jp_lunit, '("cdn_columnfiles       = """, A, """")') TRIM(cdn_columnfiles)
WRITE(jp_lunit, '("cl_columnlist         = """, A, """")') TRIM(cl_columnlist)
WRITE(jp_lunit, '("i_timeslice           = ", I0)')        i_timeslice
WRITE(jp_lunit, '("/")')
CLOSE(jp_lunit)


CONTAINS


!-----------------------------------------------------------------------
 SUBROUTINE GET_MENU(cfn_ncin_res, cfn_ncin_reaclay,                   &
                                    cfn_csvout_res_single,             &
                                    cdn_columnfiles,                   &
                                    cl_columnlist, i_timeslice         )
!-----------------------------------------------------------------------


USE MOD_UTICOMMON, ONLY: EXPAND_LIST


IMPLICIT NONE


CHARACTER(LEN=*), INTENT(OUT) ::    cfn_ncin_res
CHARACTER(LEN=*), INTENT(OUT) ::    cfn_ncin_reaclay
CHARACTER(LEN=*), INTENT(OUT) ::    cfn_csvout_res_single
CHARACTER(LEN=*), INTENT(OUT) ::    cdn_columnfiles
CHARACTER(LEN=*), INTENT(OUT) ::    cl_columnlist
INTEGER,          INTENT(OUT) ::    i_timeslice


NAMELIST /nml_column_extract2csv/   cfn_ncin_res, cfn_ncin_reaclay,    &
                                    cfn_csvout_res_single,             &
                                    cdn_columnfiles,                   &
                                    cl_columnlist, i_timeslice


cfn_ncin_res          = "/dev/null"
cfn_ncin_reaclay      = "/dev/null"
cfn_csvout_res_single = "/dev/null"
cdn_columnfiles       = "."
cl_columnlist         = "0"
i_timeslice           =  0

                                    ! First try to read from a NAMELIST
                                    ! file in the working directory
INQUIRE(FILE=cfn_nml, EXIST=l_exists)

IF (l_exists) THEN                  ! - if there is one, use it without
                                    !   asking questions

  OPEN(UNIT=jp_lunit, FILE=cfn_nml, STATUS='OLD')
  READ(UNIT=jp_lunit, NML=nml_column_extract2csv)
  CLOSE(UNIT=jp_lunit)

# ifdef VERBOSE
  WRITE(*, '("After reading in NAMELIST file """, A, """")') cfn_nml
  WRITE(*, '(" cfn_ncin_res = """, A, """")')     TRIM(cfn_ncin_res)
  WRITE(*, '(" cfn_ncin_reaclay = """, A, """")') TRIM(cfn_ncin_reaclay)
  WRITE(*, '(" cfn_csvout_res_single = """, A, """")') &
                                                  TRIM(cfn_csvout_res_single)
  WRITE(*, '(" cdn_columnfiles = """, A, """")')  TRIM(cdn_columnfiles)
  WRITE(*, '(" cl_columnlist = """, A, """")')    TRIM(cl_columnlist)
  WRITE(*, '(" i_timeslice = ", I0)')             i_timeslice
  WRITE(*, '()')
# endif

ELSE                                ! - if no NAMELIST file is present
                                    !   in the current working directory,
                                    !   go for an interactive configuration.
                                    !
                                    !   First check if there is a defaults
                                    !   NAMELIST file available that was
                                    !   produced at the end of a previous
                                    !   successful execution of this
                                    !   program
  cpn_nml_defaults = TRIM(cdn_nml_defaults) // TRIM(c_dirsep) // TRIM(cfn_nml_defaults)
  INQUIRE(FILE=cpn_nml_defaults, EXIST=l_exists)

  IF (l_exists) THEN                !   if there is one, read its contents
                                    !   and use that as default values
    OPEN(UNIT=jp_lunit, FILE=cpn_nml_defaults, STATUS='OLD')
    READ(UNIT=jp_lunit, NML=nml_column_extract2csv)
    CLOSE(UNIT=jp_lunit)
  ENDIF

                                    !   Now, we may start the configuration
# ifndef VERBOSE
  WRITE(*, '()')
  WRITE(*, '("This is COLUMN_EXTRACT2CSV")')
  WRITE(*, '("==========================")')
  WRITE(*, '()')
# endif

  WRITE(*, '("----------------------------------------------------------")')
  WRITE(*, '("Please enter the following required information")')
  WRITE(*, '("----------------------------------------------------------")')
  WRITE(*, '(" * enter file and directory names without")')
  WRITE(*, '("   delimiting quotes")')
  WRITE(*, '(" * defaults given between brackets ([...]) will")')
  WRITE(*, '("   be used for fields left empty")')
  WRITE(*, '("----------------------------------------------------------")')
  WRITE(*, '()')

  WRITE(*, '(" - the name of the NCIN file to read from")')
  IF (cfn_ncin_res == "/dev/null") THEN
    WRITE(*,'("   > ")', ADVANCE="NO")
  ELSE
    WRITE(*,'("   [", A, "] > ")', ADVANCE="NO") TRIM(cfn_ncin_res)
  ENDIF

  DO
    READ(*, '(A)') c_dummy

    IF (LEN_TRIM(c_dummy) == 0) THEN
      IF (cfn_ncin_res == "/dev/null") THEN
        WRITE(*, '("   sorry, there is no valid default - please try again > ")', ADVANCE = "NO")
        CYCLE
      ELSE
        EXIT
      ENDIF
    ELSE
      READ(c_dummy, '(A)') cfn_ncin_res
      EXIT
    ENDIF
  ENDDO

  WRITE(*, '("   requesting NCIN file """, A, """")') TRIM(cfn_ncin_res)
  WRITE(*, '()')


  WRITE(*,' (" - the name of the REACLAY file to read from")')
  IF (cfn_ncin_reaclay == "/dev/null") THEN
    WRITE(*, '("   [", A, "] > ")', ADVANCE="NO") TRIM(cfn_ncin_res)
  ELSE
    WRITE(*, '("   [", A, "] > ")', ADVANCE="NO") TRIM(cfn_ncin_reaclay)
  ENDIF

  READ(*, '(A)') c_dummy
  IF (LEN_TRIM(c_dummy) == 0) THEN
    IF (cfn_ncin_reaclay == "/dev/null") cfn_ncin_reaclay = TRIM(cfn_ncin_res)
  ELSE
    READ(c_dummy, '(A)') cfn_ncin_reaclay
  ENDIF
  WRITE(*, '("   requesting REACLAY file """, A, """")') TRIM(cfn_ncin_reaclay)
  WRITE(*, '()')


  WRITE(*, '(" - the name of the CSVOUT file in case a single file is wanted")')
  WRITE(*, '("   * leave empty if per-column CSVOUT files should be generated")')
  WRITE(*, '("   * please use ""/dev/null"" to override a non-void default")')
  IF (cfn_csvout_res_single == "/dev/null") THEN
    WRITE(*, '("   >  ")', ADVANCE="NO")
  ELSE
    WRITE(*, '("   [", A, "] > ")', ADVANCE="NO") TRIM(cfn_csvout_res_single)
  ENDIF

  READ(*, '(A)') c_dummy
  IF (LEN_TRIM(c_dummy) /= 0) THEN
    READ(c_dummy, '(A)') cfn_csvout_res_single
  ENDIF

  IF (cfn_csvout_res_single == "/dev/null") THEN
    WRITE(*, '("   using per-column CSVOUT files")')
  ELSE
    WRITE(*, '("   requesting single CSVOUT file """, A, """")') TRIM(cfn_csvout_res_single)
  ENDIF
  WRITE(*, '()')


  WRITE(*, '(" - the name of the directory to store the CSVOUT file(s)")')
  WRITE(*, '("   [", A, "] > ")', ADVANCE="NO") TRIM(cdn_columnfiles)

  READ(*, '(A)') c_dummy
  IF (LEN_TRIM(c_dummy) /= 0) THEN
    READ(c_dummy, '(A)') cdn_columnfiles
  ENDIF
  WRITE(*, '("   requesting CSVOUT directory """, A, """")') TRIM(cdn_columnfiles)
  WRITE(*, '()')


  WRITE(*, '(" - the list of the cores to extract &
               &(e.g. 1-3,5,6-9), max. ",I0, " characters ")') LEN(cl_columnlist)
  IF (cl_columnlist == "0") THEN
    WRITE(*, '("   > ")', ADVANCE="NO")
  ELSE
    WRITE(*, '("   [", A, "] > ")', ADVANCE="NO") TRIM(cl_columnlist)
  ENDIF

  DO
    READ(*, '(A)') c_dummy

    IF (LEN_TRIM(c_dummy) == 0) THEN
      IF (cl_columnlist == "0") THEN
        WRITE(*, '("   no valid default - please try again > ")', ADVANCE = "NO")
        CYCLE
      ELSE
        EXIT
      ENDIF
    ELSE
      READ(c_dummy, '(A)') cl_columnlist
      EXIT
    ENDIF
  ENDDO
  WRITE(*, '("   requesting to extract columns (", A, ")")') TRIM(cl_columnlist)
  WRITE(*, '()')

  WRITE(*, '(" - the number of the time slice to extract ")')
  IF (i_timeslice == 0) THEN
    WRITE(*, '("   [0 - last one] > ")', ADVANCE="NO")
  ELSE
    WRITE(*, '("   [", I0, "] > ")', ADVANCE="NO") i_timeslice
  ENDIF

  READ(*, '(A)') c_dummy

  IF (LEN_TRIM(c_dummy) /= 0) THEN
    READ(c_dummy, *) i_timeslice
  ENDIF
  WRITE(*, '("   requesting time slice ", I0)') i_timeslice


# ifdef DEBUG
  WRITE(*, '("NCIN name """, A,"""")')               TRIM(cfn_ncin_res)
  WRITE(*, '("REACLAY name """, A,"""")')            TRIM(cfn_ncin_reaclay)
  WRITE(*, '("CSVOUT single name """, A,"""")')      TRIM(cfn_csvout_res_single)
  WRITE(*, '("dest. directory name """, A,"""")')    TRIM(cdn_columnfiles)
  WRITE(*, '("List of cols. to read = """, A,"""")') TRIM(cl_columnlist)
  WRITE(*, '("Time slice = ", I0)')                  i_timeslice
# endif

  WRITE(*, '()')
  WRITE(*, '("----------------------------------------------------------")')
  WRITE(*, '()')

ENDIF


RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE GET_MENU
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE HANDLE_NCERRORS(istatus, whatfile, iline)
!-----------------------------------------------------------------------

IMPLICIT NONE

#include <netcdf.inc>

INTEGER, INTENT(IN) :: istatus
CHARACTER(LEN=*)    :: whatfile
INTEGER, INTENT(IN) :: iline

IF (istatus /= NF_NOERR) THEN
  IF (iline > 0) THEN
    WRITE(jp_stderr,"('[',A,':',I0,']: ', A)") &
      TRIM(whatfile), iline, TRIM(NF_STRERROR(istatus))
  ELSE
    WRITE(jp_stderr,"('[',A,':???]: ', A)") &
      TRIM(whatfile), TRIM(NF_STRERROR(istatus))
  ENDIF

  PRINT *, 'NetCDF error detected; aborting.'

  CALL ABORT_EXECUTION()

ENDIF


RETURN


!-----------------------------------------------------------------------
 END SUBROUTINE HANDLE_NCERRORS
!-----------------------------------------------------------------------


!=======================================================================
 END
!=======================================================================
