!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


PROGRAM CIRCUGRAPH

! CIRCUGRAPH generates the TeX code for drawing a diagram of
! the water circulation from an MBM *.dat file.
! The program includes tentative code for processing the
! generated circugraph.tex and convert it to PostScript.
! The required SYSTEM calls are, however, non standard, and
! it might be required to comment them out, and process
! circugraph.tex manually.

IMPLICIT NONE

INTEGER, PARAMETER :: nres=10
INTEGER, PARAMETER :: linestoskip=11
INTEGER, PARAMETER :: unit=10

CHARACTER(LEN=5), DIMENSION(nres), PARAMETER :: resnames = &
 (/ 'SNATL','SEATL',' SANT','SEI-P','SNPAC', &
            'TEATL',        'TEI-P',   &
    ' DATL',        ' DANT',' DI-P' /)
INTEGER, DIMENSION(nres,nres) :: fluxesint
DOUBLE PRECISION, DIMENSION(nres,nres) :: fluxes
CHARACTER(LEN=4), DIMENSION(nres,nres) :: formats = 'f3.1'

CHARACTER(LEN=255) :: datfilename, latexdatfilename
CHARACTER(LEN=255) :: format
INTEGER :: i, j
CHARACTER(LEN=1) :: dummy

INTEGER :: nvert, nhorz
INTEGER :: n_from, n_to
INTEGER :: fvnet, fvmix, fhnet, fhmix, netflux, mixflux

DOUBLE PRECISION, DIMENSION(nres) :: fluxin, fluxout, fluxio


WRITE(*,'(A)') 'Name of dat-file to process (not quoted):'
READ(*,'(A)') datfilename

OPEN(unit,FILE=datfilename)
DO i=1,linestoskip
  READ(unit,'(A1)') dummy
ENDDO

READ(unit,*) nvert, fvnet, fvmix
DO i=1, nvert
    READ(unit,*) n_from, n_to, netflux, mixflux
    fluxesint(n_from, n_to) = fvnet*netflux+fvmix*mixflux
    fluxesint(n_to, n_from) = fvmix*mixflux
ENDDO
READ(unit,*) nhorz, fhnet, fhmix
DO i=1, nhorz
    READ(unit,*) n_from, n_to, netflux, mixflux
    fluxesint(n_from, n_to) = fhnet*netflux+fhmix*mixflux
    fluxesint(n_to, n_from) = fhmix*mixflux
ENDDO
CLOSE(unit)

fluxes=fluxesint/1000.D0

WHERE(fluxesint(:,:) > 9950)
  formats(:,:)='f4.1'
ENDWHERE

WHERE(fluxesint(:,:) >= 99950)
  formats(:,:)='f5.1'
ENDWHERE


DO i=1,nres
  fluxin(i)=SUM(fluxesint(:,i))/1000.D0
  fluxout(i)=SUM(fluxesint(i,:))/1000.D0
END DO
DO i=1,nres
  fluxio(i)=(SUM(fluxesint(:,i))-SUM(fluxesint(i,:)))/1000.
END DO

j=1
DO i = 1, LEN(datfilename)
  SELECT CASE(datfilename(i:i))
  CASE('_')
    latexdatfilename(j:j+1) = '\_'
    j=j+2
  CASE DEFAULT
    latexdatfilename(j:j) = datfilename(i:i)
    j=j+1
  END SELECT
ENDDO


OPEN(unit, FILE='circugraph.tex')
WRITE(unit,'(A)') '\documentclass[12pt,a4paper]{article}'
WRITE(unit,'(A)') '\pagestyle{empty}'
WRITE(unit,'(A)') '\usepackage{times}'
WRITE(unit,'(A)') '\begin{document}'
WRITE(unit,'(A)') '\begin{center}'
WRITE(unit,'(A)') '{\Large \textbf{Water circulation summary for \texttt{'// &
                  TRIM(latexdatfilename) // '}}\par}'
WRITE(unit,'(A)') '\end{center}'
WRITE(unit,'(A)') '%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%'
WRITE(unit,'(A)') '\begin{figure}[h]                                                           %'
WRITE(unit,'(A)') '\mbox{}\hfill\mbox{\unitlength0.73mm                                        %'
WRITE(unit,'(A)') '\begin{picture}(187,90)                                                     %'
WRITE(unit,'(A)') '{\scriptsize                                                                %'
WRITE(unit,'(A)') '%                                                                           %'
WRITE(unit,'(A)') '%                                                                           %'
WRITE(unit,'(A)') '% Bo{\^\i}tes                                                               %'
WRITE(unit,'(A)') '%                                                                           %'
WRITE(unit,'(A)') '\put(4,0){\framebox(66,45){DATL}}     % Atl prof                            %'
WRITE(unit,'(A)') '\put(80,0){\framebox(20,45){DANT}}    % Antarct prof                        %'
WRITE(unit,'(A)') '\put(110,0){\framebox(66,45){DI-P}}   % Indo-Pac prof                       %'
WRITE(unit,'(A)') '\put(4,50){\framebox(16,30){SNATL}}   % Atl Nord superf                     %'
WRITE(unit,'(A)') '\put(30,50){\framebox(40,15){TEATL}}  % Atl Equat thermocl                  %'
WRITE(unit,'(A)') '\put(30,70){\framebox(40,10){SEATL}}  % Atl Equat superf                    %'
WRITE(unit,'(A)') '\put(80,50){\framebox(20,30){SANT}}   % Antarct superf                      %'
WRITE(unit,'(A)') '\put(110,50){\framebox(40,15){TEI-P}} % Indo-Pac Equat thermocl             %'
WRITE(unit,'(A)') '\put(110,70){\framebox(40,10){SEI-P}} % Indo-Pac Equat superf               %'
WRITE(unit,'(A)') '\put(160,50){\framebox(16,30){SNPAC}} % Pac Nord superf                     %'
WRITE(unit,'(A)') '\put(3,-1){\dashbox(68,82){\ }}                                             %'
WRITE(unit,'(A)') '\put(3,-2){\makebox(68,0)[t]{\large Atlantic}}                              %'
WRITE(unit,'(A)') '\put(79,-1){\dashbox(22,82){\ }}                                            %'
WRITE(unit,'(A)') '\put(79,-2){\makebox(22,0)[t]{\large Antarctic}}                            %'
WRITE(unit,'(A)') '\put(109,-1){\dashbox(68,82){\ }}                                           %'
WRITE(unit,'(A)') '\put(109,-2){\makebox(68,0)[t]{\large Indo-Pacific}}                        %'
WRITE(unit,'(A)') '%                                                                           %'
WRITE(unit,'(A)') '%                                                                           %'
WRITE(unit,'(A)') '% Fleches                                                                   %'
WRITE(unit,'(A)') '%                                                                           %'
WRITE(unit,'(A)') '\put(68,23.5){\vector(1,0){14}}      % ATLp->ANTp      8 to  9              %'
format="('\put(75,24.5){\makebox(0,0)[b]{',"//formats(8,9)//",'}}')"
WRITE(unit,format) fluxes(8,9)
WRITE(unit,'(A)') '\put(82,21.5){\vector(-1,0){14}}     % ANTp->ATLp      9 to  8              %'
format="('\put(75,20.5){\makebox(0,0)[t]{',"//formats(9,8)//",'}}')"
WRITE(unit,format) fluxes(9,8)
WRITE(unit,'(A)') '%                                                                           %'
WRITE(unit,'(A)') '\put(98,23.5){\vector(1,0){14}}      % ANTp->IPp       9 to 10              %'
format="('\put(105,24.5){\makebox(0,0)[b]{',"//formats(9,10)//",'}}')"
WRITE(unit,format) fluxes(9,10)
WRITE(unit,'(A)') '\put(112,21.5){\vector(-1,0){14}}    % IPp->ANTp      10 to  9              %'
format="('\put(105,20.5){\makebox(0,0)[t]{',"//formats(10,9)//",'}}')"
WRITE(unit,format) fluxes(10,9)
WRITE(unit,'(A)') '%                                                                           %'
WRITE(unit,'(A)') '\put(68,58.5){\vector(1,0){14}}      % ATEth->ANTs     6 to  3              %'
format="('\put(75,59.5){\makebox(0,0)[b]{',"//formats(6,3)//",'}}')"
WRITE(unit,format) fluxes(6,3)
WRITE(unit,'(A)') '\put(82,56.5){\vector(-1,0){14}}     % ANTs->ATEth     3 to  6              %'
format="('\put(75,55.5){\makebox(0,0)[t]{',"//formats(3,6)//",'}}')"
WRITE(unit,format) fluxes(3,6)
WRITE(unit,'(A)') '%                                                                           %'
WRITE(unit,'(A)') '\put(98,58.5){\vector(1,0){14}}      % ANTs->IPEth     3 to  7              %'
format="('\put(105,59.5){\makebox(0,0)[b]{',"//formats(3,7)//",'}}')"
WRITE(unit,format) fluxes(3,7)
WRITE(unit,'(A)') '\put(112,56.5){\vector(-1,0){14}}    % IPEth->ANTs     7 to  3              %'
format="('\put(105,55.5){\makebox(0,0)[t]{',"//formats(7,3)//",'}}')"
WRITE(unit,format) fluxes(7,3)
WRITE(unit,'(A)') '%                                                                           %'
WRITE(unit,'(A)') '\put(68,76){\vector(1,0){14}}        % ATEs->ANTs      2 to  3              %'
format="('\put(75,77){\makebox(0,0)[b]{',"//formats(2,3)//",'}}')"
WRITE(unit,format) fluxes(2,3)
WRITE(unit,'(A)') '\put(82,74){\vector(-1,0){14}}       % ANTs->ATEs      3 to  2              %'
format="('\put(75,73){\makebox(0,0)[t]{',"//formats(3,2)//",'}}')"
WRITE(unit,format) fluxes(3,2)
WRITE(unit,'(A)') '%                                                                           %'
WRITE(unit,'(A)') '\put(98,76){\vector(1,0){14}}        % ANTs->IPEs      3 to  4              %'
format="('\put(105,77){\makebox(0,0)[b]{',"//formats(3,4)//",'}}')"
WRITE(unit,format) fluxes(3,4)
WRITE(unit,'(A)') '\put(112,74){\vector(-1,0){14}}      % IPEs->ANTs      4 to  3              %'
format="('\put(105,73){\makebox(0,0)[t]{',"//formats(4,3)//",'}}')"
WRITE(unit,format) fluxes(4,3)
WRITE(unit,'(A)') '%                                                                           %'
WRITE(unit,'(A)') '\put(18,76){\vector(1,0){14}}        % ATNs->ATEs      1 to  2              %'
format="('\put(25,77){\makebox(0,0)[b]{',"//formats(1,2)//",'}}')"
WRITE(unit,format) fluxes(1,2)
WRITE(unit,'(A)') '\put(32,74){\vector(-1,0){14}}       % ATEs->ATNs      2 to  1              %'
format="('\put(25,73){\makebox(0,0)[t]{',"//formats(2,1)//",'}}')"
WRITE(unit,format) fluxes(2,1)
WRITE(unit,'(A)') '%                                                                           %'
WRITE(unit,'(A)') '\put(148,76){\vector(1,0){14}}       % IPEs->PANs      4 to  5              %'
format="('\put(155,77){\makebox(0,0)[b]{',"//formats(4,5)//",'}}')"
WRITE(unit,format) fluxes(4,5)
WRITE(unit,'(A)') '\put(162,74){\vector(-1,0){14}}      % PANs->IPEs      5 to  4              %'
format="('\put(155,73){\makebox(0,0)[t]{',"//formats(5,4)//",'}}')"
WRITE(unit,format) fluxes(5,4)
WRITE(unit,'(A)') '%                                                                           %'
WRITE(unit,'(A)') '\put(174,76){\vector(1,0){10}}       % PANs->ATNs      5 to  1              %'
WRITE(unit,'(A)') '\put(185.5,76){\circle{3}}                                                  %'
WRITE(unit,'(A)') '\put(185.6,76.1){\makebox(0,0){a}}                                          %'
format="('\put(180.5,77){\makebox(0,0)[b]{',"//formats(5,1)//",'}}')"
WRITE(unit,format) fluxes(5,1)
WRITE(unit,'(A)') '\put(2,76){\vector(1,0){4}}                                                 %'
WRITE(unit,'(A)') '\put(0.5,76){\circle{3}}                                                    %'
WRITE(unit,'(A)') '\put(0.4,76.1){\makebox(0,0){a}}                                            %'
WRITE(unit,'(A)') '%                                                                           %'
WRITE(unit,'(A)') '\put(18,58.5){\vector(1,0){14}}      % ATNs->ATEth     1 to  6              %'
format="('\put(25,59.5){\makebox(0,0)[b]{',"//formats(1,6)//",'}}')"
WRITE(unit,format) fluxes(1,6)
WRITE(unit,'(A)') '\put(32,56.5){\vector(-1,0){14}}     % ATEth->ATNs     6 to  1              %'
format="('\put(25,55.5){\makebox(0,0)[t]{',"//formats(6,1)//",'}}')"
WRITE(unit,format) fluxes(6,1)
WRITE(unit,'(A)') '%                                                                           %'
WRITE(unit,'(A)') '\put(148,58.5){\vector(1,0){14}}     % IPEth->PANs     7 to  5              %'
format="('\put(155,59.5){\makebox(0,0)[b]{',"//formats(7,5)//",'}}')"
WRITE(unit,format) fluxes(7,5)
WRITE(unit,'(A)') '\put(162,56.5){\vector(-1,0){14}}    % PANs->IPEth     5 to  7              %'
format="('\put(155,55.5){\makebox(0,0)[t]{',"//formats(5,7)//",'}}')"
WRITE(unit,format) fluxes(5,7)
WRITE(unit,'(A)') '%                                                                           %'
WRITE(unit,'(A)') '\put(11,43){\vector(0,1){9}}         % ATLp->ATNs      8 to  1              %'
format="('\put(10,47.5){\makebox(0,0)[r]{',"//formats(8,1)//",'}}')"
WRITE(unit,format) fluxes(8,1)
WRITE(unit,'(A)') '\put(13,52){\vector(0,-1){9}}        % ATNs->ATLp      1 to  8              %'
format="('\put(14,47.5){\makebox(0,0)[l]{',"//formats(1,8)//",'}}')"
WRITE(unit,format) fluxes(1,8)
WRITE(unit,'(A)') '%                                                                           %'
WRITE(unit,'(A)') '\put(49,43){\vector(0,1){9}}         % ATLp->ATEth     8 to  6              %'
format="('\put(48,47.5){\makebox(0,0)[r]{',"//formats(8,6)//",'}}')"
WRITE(unit,format) fluxes(8,6)
WRITE(unit,'(A)') '\put(51,52){\vector(0,-1){9}}        % ATEth->ATLp     6 to  8              %'
format="('\put(52,47.5){\makebox(0,0)[l]{',"//formats(6,8)//",'}}')"
WRITE(unit,format) fluxes(6,8)
WRITE(unit,'(A)') '%                                                                           %'
WRITE(unit,'(A)') '\put(49,63){\vector(0,1){9}}         % ATEth->ATEs     6 to  2              %'
format="('\put(48,67.5){\makebox(0,0)[r]{',"//formats(6,2)//",'}}')"
WRITE(unit,format) fluxes(6,2)
WRITE(unit,'(A)') '\put(51,72){\vector(0,-1){9}}        % ATEs->ATEth     2 to  6              %'
format="('\put(52,67.5){\makebox(0,0)[l]{',"//formats(2,6)//",'}}')"
WRITE(unit,format) fluxes(2,6)
WRITE(unit,'(A)') '%                                                                           %'
WRITE(unit,'(A)') '\put(89,43){\vector(0,1){9}}         % ANTp->ANTs      9 to  3              %'
format="('\put(88,47.5){\makebox(0,0)[r]{',"//formats(9,3)//",'}}')"
WRITE(unit,format) fluxes(9,3)
WRITE(unit,'(A)') '\put(91,52){\vector(0,-1){9}}        % ANTs->ANTp      3 to  9              %'
format="('\put(92,47.5){\makebox(0,0)[l]{',"//formats(3,9)//",'}}')"
WRITE(unit,format) fluxes(3,9)
WRITE(unit,'(A)') '%                                                                           %'
WRITE(unit,'(A)') '\put(129,43){\vector(0,1){9}}        % IPp->IPEth     10 to  7              %'
format="('\put(128,47.5){\makebox(0,0)[r]{',"//formats(10,7)//",'}}')"
WRITE(unit,format) fluxes(10,7)
WRITE(unit,'(A)') '\put(131,52){\vector(0,-1){9}}       % IPEth->IPp      7 to 10              %'
format="('\put(132,47.5){\makebox(0,0)[l]{',"//formats(7,10)//",'}}')"
WRITE(unit,format) fluxes(7,10)
WRITE(unit,'(A)') '%                                                                           %'
WRITE(unit,'(A)') '\put(129,63){\vector(0,1){9}}        % IPEth->IPEs     7 to  4              %'
format="('\put(128,67.5){\makebox(0,0)[r]{',"//formats(7,4)//",'}}')"
WRITE(unit,format) fluxes(7,4)
WRITE(unit,'(A)') '\put(131,72){\vector(0,-1){9}}       % IPEs->IPEth     4 to  7              %'
format="('\put(132,67.5){\makebox(0,0)[l]{',"//formats(4,7)//",'}}')"
WRITE(unit,format) fluxes(4,7)
WRITE(unit,'(A)') '%                                                                           %'
WRITE(unit,'(A)') '\put(167,43){\vector(0,1){9}}        % IPp->PANs      10 to  5              %'
format="('\put(166,47.5){\makebox(0,0)[r]{',"//formats(10,5)//",'}}')"
WRITE(unit,format) fluxes(10,5)
WRITE(unit,'(A)') '\put(169,52){\vector(0,-1){9}}       % PANs->IPp       5 to 10              %'
format="('\put(170,47.5){\makebox(0,0)[l]{',"//formats(5,10)//",'}}')"
WRITE(unit,format) fluxes(5,10)
WRITE(unit,'(A)') '%                                                                           %'
WRITE(unit,'(A)') '}% \end scriptsize                                                          %'
WRITE(unit,'(A)') '\end{picture}%                                                              %'
WRITE(unit,'(A)') '}\hfill\mbox{}                                                              %'
WRITE(unit,'(A)') '\vspace*{5mm}                                                               %'
WRITE(unit,'(A)') '\end{figure}                                                                %'
WRITE(unit,'(A)') '%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%'
WRITE(unit,'(A)') '%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%'
WRITE(unit,'(A)') '\begin{table}[h]'
WRITE(unit,'(A)') '\begin{center}'
WRITE(unit,'(A)') '\begin{tabular}{crrr}'
WRITE(unit,'(A)') '\hline'
WRITE(unit,'(A)') 'Reservoir & \multicolumn{1}{c}{In} & \multicolumn{1}{c}{Out} & \multicolumn{1}{c}{In$-$Out} \\'
WRITE(unit,'(A)') '\hline'

format="(a6, 3(' & ', f7.3), ' \\')"
DO i=1,nres
  WRITE(unit,format) resnames(i), fluxin(i), fluxout(i), fluxio(i)
ENDDO
WRITE(unit,'(A)') '\hline'
WRITE(unit,'(A)') '\end{tabular}'
WRITE(unit,'(A)') '\end{center}'
WRITE(unit,'(A)') '\end{table}'
WRITE(unit,'(A)') '\end{document}'
CLOSE(unit)

WRITE(*, '(A)', ADVANCE='NO') 'Generating summary sheet'
CALL SYSTEM('latex circugraph') !>& /dev/null')
CALL SYSTEM('dvips circugraph -o')! >& /dev/null')
WRITE(*,'(A)') '. Done.'
WRITE(*,'(A)') 'Displaying summary sheet.'
WRITE(*,'(A)') 'Quit PS viewer to continue!'
CALL SYSTEM('gv circugraph.ps')! >& /dev/null')

END
