!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


!***********************************************************************
PROGRAM BATHYM
!***********************************************************************

! PROGRAM TO DETERMINE DEPTH PROFILES FOR THE DEEP OCEAN BASINS
! BASED ON PROGRAM DS750.1 FROM SCRIPPS
! See http://dss.ucar.edu/datasets/ds750.1 for info on data used

! Now also produces map of ocean basin areas

IMPLICIT NONE
INTEGER, DIMENSION(360,180) :: le, map
INTEGER X1
INTEGER, PARAMETER :: BOTTOM=8000, TOP=0, KK=100, NN=BOTTOM/KK, NNTOP=TOP/KK
INTEGER, PARAMETER :: izero = 0
CHARACTER(LEN=1), DIMENSION(360,180) :: f
CHARACTER(LEN=3) :: BOX
DOUBLE PRECISION, PARAMETER :: pi = 3.1415927D+00, &
                               dss = (6370000.D+00*PI/180.D+00)**2, &
                               maxdth = 0.999D0

CHARACTER(LEN=32) :: resdef_filename

DOUBLE PRECISION :: DS, X
DOUBLE PRECISION, DIMENSION(top:BOTTOM) :: ATE,ATN, ANT, ipe, ipn
DOUBLE PRECISION, DIMENSION(nntop:NN)   :: SATE, SATN, SANT, SIPE, SIPN
DOUBLE PRECISION, DIMENSION(nntop:NN)   :: VATE, VATN, VANT, VIPE, VIPN
DOUBLE PRECISION, DIMENSION(nntop:NN,5) :: s_all, v_all
INTEGER, DIMENSION(5) :: maxdep

INTEGER :: i, i1, i2, j, j1, j2, k, nbr

! Netcdf related declarations

INCLUDE "netcdf.inc" 

CHARACTER(LEN=32) :: nc_filename, nc_varname, nc_dimname, nc_unitname, nc_varlongname
CHARACTER(LEN=64) :: nc_attname

INTEGER :: ll, dim(2), i_value, i_index
DOUBLE PRECISION :: d_value
INTEGER :: nc_fileid, nc_status, nc_varndims
INTEGER :: nc_dimlen_dpi, nc_dimid_dpi, nc_varid_dpi
INTEGER :: nc_dimlen_dep, nc_dimid_dep, nc_varid_dep
INTEGER :: nc_varid_sfa, nc_varid_vol, nc_varid_max





! LE(I,J): I=1..360 AS LONG=0.5E..359.5E
!          J=1..180 AS LAT=89.5S..89.5N

WRITE(*,*)'Resolution set to ',KK,'M (',NN,' subdivisions).'
WRITE(*,*)'Now reading depth file'
OPEN(1,FILE='ds750-1.dat')
READ(1,1000)((LE(I,J),F(I,J),I=1,360),J=1,180)
CLOSE(1)

ATN(:)=0.D+00
ATE(:)=0.D+00
ANT(:)=0.D+00
IPE(:)=0.D+00
IPN(:)=0.D+00

SATN(:)=0.D+00
SATE(:)=0.D+00
SANT(:)=0.D+00
SIPE(:)=0.D+00
SIPN(:)=0.D+00

VATN(:)=0.D+00
VATE(:)=0.D+00
VANT(:)=0.D+00
VIPE(:)=0.D+00
VIPN(:)=0.D+00

MAP(:,:) = -1
WHERE(F(:,:) == 'D')
  MAP(:,:) = 0
END WHERE

WRITE(*,*)'Now reading basin definition file'
WRITE(*,*)'and calculating surface areas'

WRITE(*,*)
WRITE(*,*)'Please specify reservoir definition file name'
WRITE(*,*)'(default: reservoirs.def) :'

READ(*,'(A)') resdef_filename
IF(LEN_TRIM(resdef_filename) == 0) resdef_filename = 'reservoirs.def'
OPEN(2,FILE=resdef_filename)
READ(2,*)NBR

DO K=1,NBR
  READ(2,1001)J1,J2,I1,I2,BOX
  DO J=J1,J2
!   DS = R*R*SIN(TH)*DTH*DPHI=DSS*SIN(TH)
!   R = 6370 KM
!   TH = (180.5D+00-J) DEG
!   DTH = 1 DEG
!   DPHI = 1 DEG
    DS=SIN((180.5D+00-J)*PI/180.D+00)*DSS
    SELECT CASE(box)
      CASE('ATN')
        DO I=I1,I2
          IF(F(I,J).EQ.'D') THEN
            ATN(-LE(I,J))=ATN(-LE(I,J))+DS
            MAP(I,J) = 2
          ENDIF
        ENDDO
      CASE('ATE')
        DO I=I1,I2
          IF(F(I,J).EQ.'D') THEN
            ATE(-LE(I,J))=ATE(-LE(I,J))+DS
            MAP(I,J) = 3
          ENDIF
        ENDDO
      CASE('ANT')
        DO I=I1,I2
          IF(F(I,J).EQ.'D') THEN
            ANT(-LE(I,J))=ANT(-LE(I,J))+DS
            MAP(I,J) = 4
          ENDIF
        ENDDO
      CASE('IPE')
        DO I=I1,I2
          IF(F(I,J).EQ.'D') THEN
            IPE(-LE(I,J))=IPE(-LE(I,J))+DS
            MAP(I,J) = 5
          ENDIF
        ENDDO
      CASE('IPN')
        DO I=I1,I2
          IF(F(I,J).EQ.'D') THEN
            IPN(-LE(I,J))=IPN(-LE(I,J))+DS
            MAP(I,J) = 6
          ENDIF
        ENDDO
      CASE DEFAULT
        WRITE(*,*) 'Unknown box ',BOX,' in line',K+1
        STOP
      END SELECT
   ENDDO
ENDDO
CLOSE(2)

! SXXX(K)=SURFACE AREA OF WATER WITH DEPTH <= K IN BASIN XXX
! VXXX(K)=VOLUME OF WATER WITH DEPTH <= K IN BASIN XXX
WRITE(*,*)'Now calculating cumulative surfaces and volumes'
SATN(izero)=ATN(izero)
SATE(izero)=ATE(izero)
SIPN(izero)=IPN(izero)
SIPE(izero)=IPE(izero)
SANT(izero)=ANT(izero)


DO K=izero+1,izero+NN

  SATN(K)=SATN(K-1)
  SATE(K)=SATE(K-1)
  SIPN(K)=SIPN(K-1)
  SIPE(K)=SIPE(K-1)
  SANT(K)=SANT(K-1)

  VATN(K)=VATN(K-1)
  VATE(K)=VATE(K-1)
  VIPN(K)=VIPN(K-1)
  VIPE(K)=VIPE(K-1)
  VANT(K)=VANT(K-1)

  DO I=1,KK

    SATN(K)=SATN(K)+ATN((K-1)*KK+I)
    SATE(K)=SATE(K)+ATE((K-1)*KK+I)
    SANT(K)=SANT(K)+ANT((K-1)*KK+I)
    SIPN(K)=SIPN(K)+IPN((K-1)*KK+I)
    SIPE(K)=SIPE(K)+IPE((K-1)*KK+I)

    VATN(K)=VATN(K)+ATN((K-1)*KK+I)*REAL((K-1)*KK+I)
    VATE(K)=VATE(K)+ATE((K-1)*KK+I)*REAL((K-1)*KK+I)
    VANT(K)=VANT(K)+ANT((K-1)*KK+I)*REAL((K-1)*KK+I)
    VIPN(K)=VIPN(K)+IPN((K-1)*KK+I)*REAL((K-1)*KK+I)
    VIPE(K)=VIPE(K)+IPE((K-1)*KK+I)*REAL((K-1)*KK+I)

  ENDDO

ENDDO

WRITE(*,"('Areas, total area to  200m (10^6 km^2):', 6F8.1)") &
      SATN(2)*1E-12,SATE(2)*1E-12,SANT(2)*1E-12,SIPE(2)*1E-12,SIPN(2)*1E-12, &
      (SATN(2)+SATE(2)+SANT(2)+SIPE(2)+SIPN(2))*1E-12
WRITE(*,"('Areas, total area to 1000m (10^6 km^2):', 6F8.1)") &
      SATN(10)*1E-12,SATE(10)*1E-12,SANT(10)*1E-12,SIPE(10)*1E-12,SIPN(10)*1E-12, &
      (SATN(10)+SATE(10)+SANT(10)+SIPE(10)+SIPN(10))*1E-12
WRITE(*,"('Areas, total area to 3700m (10^6 km^2):', 6F8.1)") &
      SATN(37)*1E-12,SATE(37)*1E-12,SANT(37)*1E-12,SIPE(37)*1E-12,SIPN(37)*1E-12, &
      (SATN(37)+SATE(37)+SANT(37)+SIPE(37)+SIPN(37))*1E-12
WRITE(*,"('Areas, total area to 7000m (10^6 km^2):', 6F8.1)") &
      SATN(70)*1E-12,SATE(70)*1E-12,SANT(70)*1E-12,SIPE(70)*1E-12,SIPN(70)*1E-12, &
      (SATN(70)+SATE(70)+SANT(70)+SIPE(70)+SIPN(70))*1E-12
WRITE(*,"('Areas, total area (10^6 km^2):         ', 6F8.1)") &
      SATN(NN)*1E-12,SATE(NN)*1E-12,SANT(NN)*1E-12,SIPE(NN)*1E-12,SIPN(NN)*1E-12, &
      (SATN(NN)+SATE(NN)+SANT(NN)+SIPE(NN)+SIPN(NN))*1E-12
WRITE(*,"('Volumes, total volume (10^6 km^3):     ', 6F8.1)") &
      VATN(NN)*1E-15,VATE(NN)*1E-15,VANT(NN)*1E-15,VIPE(NN)*1E-15,VIPN(NN)*1E-15, &
      (VATN(NN)+VATE(NN)+VANT(NN)+VIPE(NN)+VIPN(NN))*1E-15


WRITE(*,*)'Now searching greatest depths per profile.'
WRITE(*,*)'Criterion: Sigma Volume > ',MAXDTH*1D2,'%'

I=NN
DO WHILE (vATN(I).GT.MAXDTH*vATN(NN))
  I=I-1
ENDDO
WRITE(*,'("Northern Atlantic: ",I0," (",I0,"m)")') I, I*KK
WRITE(*,'(I3,2E13.5)') (j, vATN(J),sATN(J), J=I-2,MIN(I+2,NN-1))
WRITE(*,'(I3,2E13.5)') (j, vATN(J),sATN(J), J=nn,nn)
I=44
maxdep(1) = I-nntop+1
WRITE(*,"('MAXDEP Northern Atlantic ',i0,' (',i0,')')"), I, maxdep(1)

I=NN
DO WHILE (vATE(I).GT.MAXDTH*vATE(NN))
  I=I-1
ENDDO
WRITE(*,'("Equatorial Atlantic: ",I3," (",I4,"m)")') I, I*KK
WRITE(*,'(I3,2E13.5)') (j, vATE(J),sATE(J), J=I-2,MIN(I+2,NN))
WRITE(*,'(I3,2E13.5)') (j, vATE(J),sATE(J), J=nn,nn)
I=64
maxdep(2) = I-nntop+1
WRITE(*,"('MAXDEP Equatorial Atlantic ',i0,' (',i0,')')"), I, maxdep(2)

I=NN
DO WHILE (vANT(I).GT.MAXDTH*vANT(NN))
  I=I-1
ENDDO
WRITE(*,'("Antarctic: ",I3," (",I4,"m)")') I, I*KK
WRITE(*,'(I3,2E13.5)') (j, vANT(J),sANT(J), J=I-2,MIN(I+2,NN))
WRITE(*,'(I3,2E13.5)') (j, vANT(J),sANT(J), J=nn,nn)
I=59
maxdep(3) = I-nntop+1
WRITE(*,"('MAXDEP Antarctic ',i0,' (',i0,')')"), I, maxdep(3)

I=NN
DO WHILE (vIPE(I).GT.MAXDTH*vIPE(NN))
  I=I-1
ENDDO
WRITE(*,'("Equatorial Indo-Pacific: ",I3," (",I4,"m)")') I, I*KK
WRITE(*,'(I3,2E13.5)') (j, vIPE(J),sIPE(J), J=I-2,MIN(I+2,NN))
WRITE(*,'(I3,2E13.5)') (j, vIPE(J),sIPE(J), J=nn,nn)
I=69
maxdep(4) = I-nntop+1
WRITE(*,"('MAXDEP Equatorial Indo-Pacific ',i0,' (',i0,')')"), I, maxdep(4)

I=NN
DO WHILE (vIPN(I).GT.MAXDTH*vIPN(NN))
  I=I-1
ENDDO
WRITE(*,'("Northern Pacific: ",I3," (",I4,"m)")') I, I*KK
WRITE(*,'(I3,2E13.5)') (j, vIPN(J),sIPN(J), J=I-2,MIN(I+2,NN))
WRITE(*,'(I3,2E13.5)') (j, vIPN(J),sIPN(J), J=nn,nn)
I=68
maxdep(5) = I-nntop+1
WRITE(*,"('MAXDEP Northern Pacific ',i0,' (',i0,')')"), I, maxdep(5)

WRITE(*,*)
WRITE(*,*)'Now writing results to file'

WRITE(*,*)
WRITE(*,*)'Please specify output file name, incl. filename extension'
WRITE(*,*)'(default: depthprofiles.nc) :'

READ(*,'(A)') nc_filename
IF(LEN_TRIM(nc_filename) == 0) nc_filename = 'depthprofiles.nc'

nc_status = NF_CREATE(nc_filename, NF_CLOBBER, nc_fileid)
IF(nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)


! Dimension 'dpi' (Depth Profile ID)
! ----------------------------------
nc_dimname = 'dpi'                  ! Dimension name
nc_dimlen_dpi  = 5                  ! Dimension length
                                    ! Define the dimension
nc_status = NF_DEF_DIM(nc_fileid, &
                       nc_dimname, nc_dimlen_dpi, nc_dimid_dpi)
IF (nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)


                                    ! Define the variable that holds it
nc_varname = 'dpi'                  ! Dimension variable name
      
nc_status = NF_DEF_VAR(nc_fileid, &
                       nc_varname, NF_INT, 1, nc_dimid_dpi, nc_varid_dpi)
IF (nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)

nc_varlongname = 'Depth Profile ID' ! long name of the variable
ll = LEN_TRIM(nc_varlongname)

nc_status = NF_PUT_ATT_TEXT(nc_fileid, nc_varid_dpi, &
                            'long_name', ll, nc_varlongname(1:ll))
IF (nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)


! Dimension 'depth' (Depth)
! -------------------------
nc_dimname = 'dep'                  ! Dimension name
nc_dimlen_dep  = nntop+nn+1         ! Dimension length
                                    ! Define the dimension
nc_status = NF_DEF_DIM(nc_fileid, &
                       nc_dimname, nc_dimlen_dep, nc_dimid_dep)
IF (nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)


                                    ! Define the variable that holds it
nc_varname = 'depth'                ! Dimension variable name
      
nc_status = NF_DEF_VAR(nc_fileid, &
                       nc_varname, NF_DOUBLE, 1, nc_dimid_dep, nc_varid_dep)
IF (nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)

nc_varlongname = 'Depth'            ! long name of the variable
ll = LEN_TRIM(nc_varlongname)

nc_status = NF_PUT_ATT_TEXT(nc_fileid, nc_varid_dep, &
                            'long_name', ll, nc_varlongname(1:ll))
IF (nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)

nc_unitname = 'm'
ll = 1
nc_status = NF_PUT_ATT_TEXT(nc_fileid, nc_varid_dep, &
                            'units', ll, nc_unitname)
IF (nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)

nc_status = NF_PUT_ATT_TEXT(nc_fileid, nc_dimid_dep, 'positive', 4, 'down')
IF (nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)

ll = -nntop+1
nc_status = NF_PUT_ATT_INT(nc_fileid, nc_varid_dep, &
                           'reference_sealevel_index', NF_INT, 1, ll)
IF (nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)


! Data variables
! --------------

dim(1:2) = (/ nc_dimid_dep, nc_dimid_dpi /)

                                    ! Surface Areas

nc_varname = 'sfa'                  ! short name of the variable
nc_varndims = 2                     ! number of dimensions (Fortran rank)

nc_status = NF_DEF_VAR(nc_fileid, nc_varname, &
                       NF_DOUBLE, nc_varndims, dim(1:2), nc_varid_sfa)
IF (nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)

nc_varlongname = 'Cumulative Surface Area' ! long name of the variable
ll = LEN_TRIM(nc_varlongname)
nc_status = NF_PUT_ATT_TEXT(nc_fileid, nc_varid_sfa, &
                            'long_name', ll, nc_varlongname(1:ll))
IF (nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)

nc_unitname = 'm^2'
ll = 3
nc_status = NF_PUT_ATT_TEXT(nc_fileid, nc_varid_sfa, &
                            'units', ll, nc_unitname)
IF (nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)


                                    ! Volumes

nc_varname = 'vol'                  ! short name of the variable
nc_varndims = 2                     ! number of dimensions (Fortran rank)

nc_status = NF_DEF_VAR(nc_fileid, nc_varname, &
                       NF_DOUBLE, nc_varndims, dim(1:2), nc_varid_vol)
IF (nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)

nc_varlongname = 'Cumulative Volumes' ! long name of the variable
ll = LEN_TRIM(nc_varlongname)
nc_status = NF_PUT_ATT_TEXT(nc_fileid, nc_varid_vol, &
                            'long_name', ll, nc_varlongname(1:ll))
IF (nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)

nc_unitname = 'm^3'
ll = 3
nc_status = NF_PUT_ATT_TEXT(nc_fileid, nc_varid_vol, &
                            'units', ll, nc_unitname)
IF (nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)


                                    ! Volumes

nc_varname = 'maxdep'               ! short name of the variable
nc_varndims = 1                     ! number of dimensions (Fortran rank)

nc_status = NF_DEF_VAR(nc_fileid, nc_varname, &
                       NF_INT, nc_varndims, dim(2:2), nc_varid_max)
IF (nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)

nc_varlongname = 'Maximum Significant Depth Index' ! long name of the variable
ll = LEN_TRIM(nc_varlongname)
nc_status = NF_PUT_ATT_TEXT(nc_fileid, nc_varid_max, &
                            'long_name', ll, nc_varlongname(1:ll))
IF (nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)



!----------------------
! Put global attributes
!----------------------

nc_attname = 'Geometric data for MBM with 5 Profiles'
ll = LEN_TRIM(nc_attname)
nc_status=NF_PUT_ATT_TEXT(nc_fileid, NF_GLOBAL, 'title', ll, nc_attname(1:ll))
IF (nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)

WRITE(*,*)
WRITE(*,*)'Please give a description of the variant of the profiles'
WRITE(*,*)'(default: "standard") :'

READ(*,'(A)') nc_attname
IF(LEN_TRIM(nc_attname) == 0) nc_attname = 'standard'
ll = LEN_TRIM(nc_attname)
nc_status=NF_PUT_ATT_TEXT(nc_fileid, NF_GLOBAL, 'variant', ll, nc_attname(1:ll))
IF (nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)


ll = LEN_TRIM(resdef_filename)
nc_status=NF_PUT_ATT_TEXT(nc_fileid, NF_GLOBAL, 'reservoir_defs_file', ll, resdef_filename(1:ll))
IF (nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)


!--------------------
! End define mode
!--------------------
      
nc_status = NF_ENDDEF(nc_fileid)
IF (nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)

        
! Set 'dpi' coordinate equal to its index value + 1
! as is done in MBM
DO i = 1, nc_dimlen_dpi
  i_index = i
  i_value = i+1
  nc_status = NF_PUT_VAR1_INT(nc_fileid, nc_varid_dpi, i_index, i_value)
  IF (nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)
ENDDO

! Set 'dep' coordinate equal to the depths
DO i = 1, nc_dimlen_dep
  i_index = i
  d_value = DBLE(i-1+nntop-izero)*DBLE(kk)
  nc_status = NF_PUT_VAR1_DOUBLE(nc_fileid, nc_varid_dep, i_index, d_value)
  IF (nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)
ENDDO


! Done with file creation and setup


! Save data into the file
        
! Write a complete array (2D here)

s_all(:,1) = satn
s_all(:,2) = sate
s_all(:,3) = sant
s_all(:,4) = sipe
s_all(:,5) = sipn

v_all(:,1) = vatn
v_all(:,2) = vate
v_all(:,3) = vant
v_all(:,4) = vipe
v_all(:,5) = vipn


nc_status = NF_PUT_VAR_DOUBLE(nc_fileid, nc_varid_sfa, s_all(:,:))
IF (nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)

nc_status = NF_PUT_VAR_DOUBLE(nc_fileid, nc_varid_vol, v_all(:,:))
IF (nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)

nc_status = NF_PUT_VAR_INT(nc_fileid, nc_varid_max, maxdep(:))
IF (nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)



nc_status = NF_SYNC(nc_fileid)
IF (nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)


nc_status = NF_CLOSE(nc_fileid)
IF (nc_status /= NF_NOERR) CALL HANDLE_ERRORS(nc_status)
 

OPEN(3,FILE='depthprofiles.res',FORM='UNFORMATTED')
WRITE(3)SATN,SATE,SANT,SIPE,SIPN
WRITE(3)VATN,VATE,VANT,VIPE,VIPN
I=44
WRITE(3)I
I=64
WRITE(3)I
I=59
WRITE(3)I
I=69
WRITE(3)I
I=68
WRITE(3)I
CLOSE(3)


OPEN(3,FILE='depthprofiles_map.res')
WRITE(3,'(360I3)') MAP
CLOSE(3)

!     WRITE(*,*)' '
!     WRITE(*,*)'Characteristic areas (ATN ATE ANT IPE IPN):'
!     WRITE(*,*)'Depth 0m :',SATN(NN)-SATN(0),
!    >                       SATE(NN)-SATE(0),
!    >                       SANT(NN)-SANT(0),
!    >                       SIPE(NN)-SIPE(0),
!    >                       SIPN(NN)-SIPN(0)
      X1=100/KK
      X=100.D+00/KK
!     WRITE(*,*)'Depth 100m :',
!    >           SATN(NN)-SATN(X1)-(X-X1)*(SATN(X1+1)-SATN(X1)),
!    >           SATE(NN)-SATE(X1)-(X-X1)*(SATE(X1+1)-SATE(X1)),
!    >           SANT(NN)-SANT(X1)-(X-X1)*(SANT(X1+1)-SANT(X1)),
!    >           SIPE(NN)-SIPE(X1)-(X-X1)*(SIPE(X1+1)-SIPE(X1)),
!    >           SIPN(NN)-SIPN(X1)-(X-X1)*(SIPN(X1+1)-SIPN(X1))
      X1=1000/KK
      X=1000.D+00/KK
!     WRITE(*,*)'Depth 1000m :',
!    >           SATN(NN)-SATN(X1)-(X-X1)*(SATN(X1+1)-SATN(X1)),
!    >           SATE(NN)-SATE(X1)-(X-X1)*(SATE(X1+1)-SATE(X1)),
!    >           SANT(NN)-SANT(X1)-(X-X1)*(SANT(X1+1)-SANT(X1)),
!    >           SIPE(NN)-SIPE(X1)-(X-X1)*(SIPE(X1+1)-SIPE(X1)),
!    >           SIPN(NN)-SIPN(X1)-(X-X1)*(SIPN(X1+1)-SIPN(X1))
!     WRITE(*,*)' '
!     WRITE(*,*)'Characteristic volumes (ATN ATE ANT IPE IPN):'
!     WRITE(*,*)'Depth 0m-',BOTTOM,'m :',VATN(NN)-VATN(0),
!    >                       VATE(NN)-VATE(0),
!    >                       VANT(NN)-VANT(0),
!    >                       VIPE(NN)-VIPE(0),
!    >                       VIPN(NN)-VIPN(0)
!     WRITE(*,*)' '
WRITE(*,*)'All done.'

 1000 FORMAT(12(I5,A1),8X)
 1001 FORMAT(2(I3,1X,I3,2X),A3)

CONTAINS

      !*****************************************************************
       SUBROUTINE HANDLE_ERRORS(status)
      !*****************************************************************

      INCLUDE "netcdf.inc" 

      INTEGER :: status

      IF (STATUS.NE.NF_NOERR) THEN
        PRINT *, NF_STRERROR(STATUS)
        PRINT *, 'NetCDF error detected; aborting.'
        CALL ABORT()
      ENDIF
      !*****************************************************************
      END SUBROUTINE HANDLE_ERRORS
      !*****************************************************************


      !*****************************************************************
       SUBROUTINE HANDLE_NCERRORS(status, whatfile, whatline)
      !*****************************************************************

      IMPLICIT NONE

      INCLUDE "netcdf.inc" 

      INTEGER, INTENT(IN) :: status
      CHARACTER(LEN=*)    :: whatfile
      INTEGER, INTENT(IN) :: whatline

      IF (STATUS /= NF_NOERR) THEN
        IF(whatline > 0) THEN
          WRITE(0,"('[',A,':',I0,']: ', A)") &
            TRIM(whatfile), whatline, TRIM(NF_STRERROR(status))
        ELSE
          WRITE(0,"('[',A,':???]: ', A)") &
            TRIM(whatfile), TRIM(NF_STRERROR(status))
        ENDIF
        PRINT *, 'NetCDF error detected; aborting.'
        CALL ABORT()
      ENDIF

      RETURN

      !*****************************************************************
      END SUBROUTINE HANDLE_NCERRORS
      !*****************************************************************

END
