!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#ifdef DEBUG
#include <debug.h>
#endif

!=======================================================================
      SUBROUTINE SOLVSED_ONESTEP(atime, datime, n_columns,
     &                           soflag, n_trouble)
!=======================================================================


      USE mod_seafloor_wdata,    ONLY: WDATA_CONTAINER
      USE mod_defines_medusa
      USE mod_execontrol_medusa, ONLY: ABORT_MEDUSA, MEDEXE_RANKFILENAME
      USE mod_seafloor_central
      USE mod_sedcore,           ONLY: CORELAY_PACKMASS,
     &                                 CORELAY_NLAY2ERODE_CLEAR,
     &                                 CORELAY_NLAY2ERODE_SET
      USE mod_indexparam
      USE mod_gridparam
      USE mod_materialcharas,    ONLY: apsv, apsv_factor
      USE mod_logunits

#ifdef SAVE_COLUMN_BYCALLER
      USE mod_seafloor_temp
#endif

      USE mod_transport,         ONLY: PROFIU, PROFIW
      USE mod_rreac,             ONLY: rreac_factor_max, rreac_factor


      IMPLICIT NONE


! =====================
! Subroutine Parameters
! =====================

                                    ! Indices and order of execution of
                                    ! the different methods implemented
                                    ! to solve the equation system
      INTEGER, PARAMETER :: jp_its_basic = 1
      INTEGER, PARAMETER :: jp_its_solut = 2
      INTEGER, PARAMETER :: jp_its_psvol = 3
      INTEGER, PARAMETER :: jp_its_wfflx = 4
      INTEGER, PARAMETER :: jp_its_rreac = 5
      INTEGER, PARAMETER :: jp_its_timst = 6
      INTEGER, PARAMETER :: jp_its_colla = 7

                                    ! Index of the last method to consider
      INTEGER, PARAMETER :: jp_its_last  = 7

                                    ! Number of steps to try for the
                                    ! continuation methods among the above.
      INTEGER, PARAMETER :: jp_ntrysteps_psvol = 10
      INTEGER, PARAMETER :: jp_ntrysteps_wfflx = 10
      INTEGER, PARAMETER :: jp_ntrysteps_rreac = 30
      INTEGER, PARAMETER :: jp_ntrysteps_timst = 21
      INTEGER, PARAMETER :: jp_ntrysteps_colla = 10

                                    ! Additional parameters
                                    !  - TIMST minimum and maximum of the
                                    !    range of time step lengths to
                                    !    adopt before switching to Infinity.
      DOUBLE PRECISION, PARAMETER :: dp_timst_dtmin = 1.0D+01
      DOUBLE PRECISION, PARAMETER :: dp_timst_dtmax = 1.0D+06

                                    ! Preprocessor token to enable (#define)
                                    ! or disable (#undef) adoption of partial
                                    ! results from some method even if the
                                    ! complete sequence fails before completion.

#define ITS_SALVAGING



! ============================
! End of Subroutine Parameters
! ============================


      INTEGER, INTENT(IN) :: n_columns
      INTEGER, INTENT(OUT) :: soflag, n_trouble
      DOUBLE PRECISION, INTENT(IN) :: atime, datime


      INTEGER :: i_column
      INTEGER :: iflag

                                    ! Boundary conditions for
                                    ! the new time instant (_gn: given new)
      TYPE(WDATA_CONTAINER) :: wdata_gn
      DOUBLE PRECISION, DIMENSION(1:nsolut) :: wconc_gn
      DOUBLE PRECISION, DIMENSION(1:nsolid) :: wfflx_gn

                                    ! Trial (adapted) new boundary conditions
      TYPE(WDATA_CONTAINER)                 :: wdata_tn
      DOUBLE PRECISION                      :: atime_tn, datime_tn
      DOUBLE PRECISION, DIMENSION(1:nsolut) :: wconc_tn
      DOUBLE PRECISION, DIMENSION(1:nsolid) :: wfflx_tn


                                    ! Variables for initial conditions
                                    ! (_go: given old)
      DOUBLE PRECISION, DIMENSION(idnw:idnb) :: xzdn_go
      DOUBLE PRECISION, DIMENSION(idvw:idvb) :: xzdv_go
      DOUBLE PRECISION, DIMENSION(1:nsolut)  :: wconc_go
      DOUBLE PRECISION, DIMENSION(idnw:idnb) :: xzdn_n
      DOUBLE PRECISION, DIMENSION(idvw:idvb) :: xzdv_n
      DOUBLE PRECISION, DIMENSION(idnw:idnb, ncompo)
     &                                       :: xc_tn, xc_go, xc_cn
      DOUBLE PRECISION, DIMENSION(nsolid)    :: ysolid_go, ysolid_to,
     &                                          ysolid_cn

                                    ! Variables for velocities
      DOUBLE PRECISION, DIMENSION(idvs:idvb) :: xwtot_go, xwtot_cn
      DOUBLE PRECISION, DIMENSION(idvw:idvb) :: xutot_go, xutot_cn
      DOUBLE PRECISION                       :: aw_bplus

#ifdef ITS_SALVAGING
      DOUBLE PRECISION, DIMENSION(idnw:idnb, ncompo)
     &                                       :: xc_cs
      DOUBLE PRECISION, DIMENSION(nsolid)    :: ysolid_cs
      DOUBLE PRECISION, DIMENSION(idvs:idnb) :: xwtot_cs
      DOUBLE PRECISION, DIMENSION(idvw:idnb) :: xutot_cs
      DOUBLE PRECISION, DIMENSION(1:nsolut)  :: bconc_cs
      DOUBLE PRECISION, DIMENSION(1:nsolut)  :: wcflx_cs
      DOUBLE PRECISION, DIMENSION(1:nsolut)  :: wcflx_bi_cs
      DOUBLE PRECISION, DIMENSION(1:nsolid)  :: bfflx_cs
      DOUBLE PRECISION                       :: f_its_best
      DOUBLE PRECISION                       :: f_its_curr
      INTEGER                                :: i_its_best
      INTEGER                                :: ibest_step
      INTEGER                                :: nbest_steps
#endif


      DOUBLE PRECISION :: factor
      INTEGER :: itsflag, i_request, failure_flag
      INTEGER :: itry_step, ntry_steps, method

      DOUBLE PRECISION, DIMENSION(nsolid)   :: bfflx_sv

                                    ! Boundary fluxes and calculated
                                    ! bottom concentration
      DOUBLE PRECISION, DIMENSION(1:nsolut) :: wcflx_cn, wcflx_bi_cn
      DOUBLE PRECISION, DIMENSION(1:nsolut) :: bconc_cn

      DOUBLE PRECISION, DIMENSION(1:nsolid) :: bfflx_cn

      DOUBLE PRECISION :: atime_n

                                    ! Progress log-file
      INTEGER, SAVE :: progresslog_unit = -1

      INTEGER :: n_packlay, cpmflag
      DOUBLE PRECISION, DIMENSION(nsolid) :: psolid

      CHARACTER(LEN=*), PARAMETER ::
     &       tinfo_fromto_fmt = '("Time interval ", F0.2, " to ", F0.2)'
      CHARACTER(LEN=*), PARAMETER ::
     &       tinfo_steadyst_fmt = '("Steady-state, with time = ", F0.2)'
      CHARACTER(LEN=48) ::
     &       tinfo_str
      INTEGER :: tinfo_flg
      CHARACTER(LEN=*), PARAMETER ::
     &       convg_fmt = '("  Column ", I0, ": No convergence")'
      CHARACTER(LEN=100) ::
     &       chero_fmt
      CHARACTER(LEN=*), PARAMETER ::  cfmt_success =
     &  '("  Column ",I0,": ", A, " method successful")'
      CHARACTER(LEN=*), PARAMETER ::  cfmt_success_stage =
     &  '("  Column ",I0,": ", A, " method successful at stage ", ' //
     &  'I0, "/", I0)'
      CHARACTER(LEN=*), PARAMETER ::  cfmt_failed =
     &  '("  Column ",I0,": ", A, " method failed")'
      CHARACTER(LEN=*), PARAMETER ::  cfmt_failed_stage =
     &  '("  Column ",I0,": ", A, " method failed at stage ", ' //
     &  'I0, "/", I0)'
      CHARACTER(LEN=*), PARAMETER ::  cfmt_accept_stage =
     &  '("  Column ",I0,": results obtained with method ", A,' //
     &  '" at stage ", I0, "/", I0, " accepted")'
#ifdef DEBUG
      CHARACTER(LEN=*), PARAMETER ::
     &  cfmt_iterations4stage =
     &  '("Column ", I0,' //
     &  ' ": ", A, " iterations for stage ", I0, "/", I0)'
#ifdef DEBUG_SOLVSED_CHECK_WCONC
      CHARACTER(LEN=*), PARAMETER ::
     &       wconc_fmt = '("  Column ",I0,'
     &                // '": Negative wconc for jsolut=",'
     &                // 'I2, ": ", EN25.15)'
#endif
#endif
      INTEGER :: jsolut
      LOGICAL :: l_std_corr_act = .FALSE.

                                    ! Note: <cfn_progresslog> must not
                                    ! be given the PARAMETER attribute
                                    ! as it will have to be modified by
                                    ! MEDEXE_RANKFILENAME under MPI
                                    ! execution.
      CHARACTER(LEN=jp_lmaxpathname) :: cfn_progresslog =
     &                              "medusa-convergence.log"

      CHARACTER(LEN=*), PARAMETER :: cfmt_modprocname_a =
     &                              '("[SOLVSED_ONESTEP]: ", A)'
      CHARACTER(LEN=*), PARAMETER :: cfmt_a     = '(" - ", A)'
      CHARACTER(LEN=*), PARAMETER :: cfmt_a_ind = '("   ", A)'


! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
! End of declarations
! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


!----------------------------------------------------------------------
! Program Start
!----------------------------------------------------------------------

#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_modprocname_a, ADVANCE="NO") 'Start'

      IF (datime > 0.0D+00) THEN
        WRITE(jp_stddbg, '(" (time interval ", F0.2," to ",F0.2, ")")')
     &    atime, atime + datime
      ELSE
        WRITE(jp_stddbg, '(" (steady-state, with time = ", F0.2, ")")')
     &    atime
      ENDIF
#endif


                                    ! Open progress log-file if not yet done

      IF (progresslog_unit == -1) THEN

        iflag = RESERVE_LOGUNIT(progresslog_unit)

        IF(iflag /= 0) THEN
          WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
          WRITE(jp_stderr, cfmt_a)
     &      'Unable to reserve a logical unit number ' //
     &      'for "progresslog_unit" -- aborting!'
          CALL ABORT_MEDUSA()
#ifdef DEBUG
        ELSE
          WRITE(jp_stddbg, cfmt_a, ADVANCE="NO")
     &      'Assigning logical unit number '
          WRITE(jp_stddbg, '(I0, A)')
     &      progresslog_unit, ' to "progresslog_unit".'
#endif
        ENDIF

        CALL MEDEXE_RANKFILENAME(cfn_progresslog)
        OPEN(progresslog_unit, FILE=cfn_progresslog)

        WRITE(progresslog_unit, '(A)')
     &    '------------------------------------------------------------'
        WRITE(progresslog_unit, '(A)')
     &    '        SOLVSED_ONESTEP progress and convergence log'
        WRITE(progresslog_unit, '(A)')
     &    '------------------------------------------------------------'
        WRITE(progresslog_unit, '(A)')
     &    'SOLVSED_ONESTEP logs here convergence information for each'
        WRITE(progresslog_unit, '(A)')
     &    'time step and each sediment column for which one or several'
        WRITE(progresslog_unit, '(A)')
     &    'calculation methods failed to converge. Further inspection'
        WRITE(progresslog_unit, '(A)')
     &    'is only required in case there are columns for which'
        WRITE(progresslog_unit, '(A)')
     &    'NO method is reported as "successful".'
        WRITE(progresslog_unit, '(A)')
     &    'Please notice that only difficulties encountered are logged;'
        WRITE(progresslog_unit, '(A)')
     &    'columns for which there are no messages could be processed'
        WRITE(progresslog_unit, '(A)')
     &    'in a straightforward way. Ideally, there are no messages'
        WRITE(progresslog_unit, '(A)')
     &    'below the next line ("No news is good news").'
        WRITE(progresslog_unit, '(A)')
     &    '------------------------------------------------------------'

      ENDIF

                                    ! Progress log-file opening done


      soflag = 0
      n_trouble = 0

      atime_n = atime + datime

      
      IF (datime > 0.0D+00) THEN
        WRITE(tinfo_str, tinfo_fromto_fmt) atime, atime_n
      ELSE
        WRITE(tinfo_str, tinfo_steadyst_fmt) atime
      ENDIF
      tinfo_flg = -1                ! Not yet printed; will only be
                                    ! printed in case of trouble

                                    ! Clear the list of core layers
                                    ! scheduled for erosion in MOD_SEDCORE
      CALL CORELAY_NLAY2ERODE_CLEAR


      do_columns: DO i_column = 1, n_columns

                                    ! Retrieve the given new boundary
                                    ! conditions
        CALL GET_BOUNDARY_CONDS(i_column, iflag,
     &               wdata_gn, wconc_gn(:), wfflx_gn(:))

                                    ! Load new values om_c, om_n, om_p, ...
                                    ! into MOD_MATERIALPARAMS if required
        CALL GET_MATERIALCHARAS(i_column, iflag)

        l_std_corr_act = .FALSE.
        DO jsolut = 1, nsolut
          IF (wconc_gn(jsolut) .LT. 0.0D+00) THEN
#ifdef DEBUG
#ifdef DEBUG_SOLVSED_CHECK_WCONC
            IF(tinfo_flg == -1) THEN
              WRITE(progresslog_unit, *) TRIM(tinfo_str)
              tinfo_flg = 0
            ENDIF
            WRITE(progresslog_unit, wconc_fmt) i_column, jsolut,
     &                                         wconc_gn(jsolut)
            WRITE(progresslog_unit,
     &        '("Standard corrective action taken -' //
     &        ' execution continuing!")')
#endif
#endif
            l_std_corr_act = .TRUE.
            wconc_gn(jsolut) = 0.0D+00
          ENDIF
        ENDDO

                                    ! Retrieve the initial conditions
                                    ! (xx_go: 'g'iven 'o'ld of xx)
        CALL GET_COLUMN(i_column, iflag,
     &                              xzdn = xzdn_go, xzdv = xzdv_go,
     &                              xc = xc_go, ysolid = ysolid_go)

        xzdn_n    = xzdn_go
        xzdv_n    = xzdv_go

                                    ! Load xphi, xdphi, ... into
                                    ! MOD_MILIEUCHARAS and adjust xvtor2
        CALL GET_MILIEUCHARAS(i_column, iflag)

        CALL PROFIW(wfflx_gn, xc_go, xzdn_go, xwtot_go, aw_bplus)
        CALL PROFIU(xwtot_go, aw_bplus, xutot_go)
                                    ! The CALL to PROFIW/U at this stage
                                    ! is not fully coherent,
                                    ! as xc = xc(atime)
                                    ! but wfflx = wfflx(atime+datime).
                                    ! On the other hand, it is not
                                    ! indispensable either
        n_packlay = 0
        psolid = 0D0

#ifdef ITS_SALVAGING
        i_its_best = 0
        f_its_best = 0.0D+00
#endif

        chem_erosion: DO

          method = 1                ! Start with the first method
          itry_step = 0             ! Execute Step 0 for the first method
          failure_flag = -1         ! --> no failure so far

          try_methods: DO WHILE (method <= jp_its_last)

                                    ! If we are not currently in the middle
                                    ! of one of the continuation methods,
                                    ! then we intialise from scratch.

            IF(itry_step == 0) THEN
                                    ! Make sure that
                                    !  - rreac_factor is set to its maximum
              rreac_factor(:) = rreac_factor_max(:)
                                    !  - apsv_factor is set to 1
              apsv_factor(:)  = 1.0D+00
                                    !  - datime_tn is set to datime
              datime_tn       = datime

#ifdef SAVE_COLUMN_BYCALLER
                                    ! Use the values stored in mod_seafloor_tmp as
                                    ! default starting values instead of xx_go
              xc_cn      = solvsed_xc(:,:,i_column)
              ysolid_cn  = solvsed_ysolid(:,i_column)

                                    ! Calculate corresponding velocities
              CALL PROFIW(wfflx_gn, xc_cn, xzdn_n, xwtot_cn, aw_bplus)
              CALL PROFIU(xwtot_cn, aw_bplus, xutot_cn)
#else
                                    ! Use xx_go as starting values for xx_cn
                  xc_cn =     xc_go
              ysolid_cn = ysolid_go
               xwtot_cn =  xwtot_go
               xutot_cn =  xutot_go
#endif
            ENDIF


                                    ! Use actual given new values as
                                    ! default "trial new" values.
            wdata_tn  = wdata_gn
            wconc_tn  = wconc_gn
            wfflx_tn  = wfflx_gn
            atime_tn  = atime
            datime_tn = datime

                                    ! Correct initial ysolid value
                                    ! for potential unburial.
            ysolid_to = ysolid_go + psolid

            SELECT CASE(method)
            CASE(jp_its_basic)
                                    ! For the BASIC method, only one
                                    ! step is carried out, and the
                                    ! default initialisation is appropriate.
              ntry_steps = 0

            CASE(jp_its_solut)
                                    ! For the SOLUT method, only one
                                    ! step is carried out.
              ntry_steps = 0

                                    ! Selected solute profiles are initially set
                                    ! homogeneously equal to their top values.
              CALL SOLVSED_ONESTEP_ITS_SOLUT


            CASE(jp_its_rreac)
                                    ! Set the number of trial steps.
              ntry_steps = jp_ntrysteps_rreac

                                    ! Correction factor for this trial step.
              IF (itry_step < ntry_steps) THEN
                rreac_factor(:) =
     &            (2.0D+00*(DBLE(itry_step)/DBLE(ntry_steps))
     &              /(1.0D+00 + (DBLE(itry_step)/DBLE(ntry_steps)**2))
     &            )* rreac_factor_max(:)
              ELSE
                rreac_factor(:) = rreac_factor_max(:)
              ENDIF

#ifdef ITS_SALVAGING
                                    ! If salvaging active, backup the
                                    ! currently accepted best result.
              IF (jp_its_rreac <= jp_its_last) THEN

                f_its_curr = DBLE(itry_step-1)/DBLE(ntry_steps)

                IF (f_its_curr >= f_its_best) THEN

                       i_its_best  = jp_its_rreac
                       f_its_best  = f_its_curr
                       ibest_step  = itry_step-1
                       nbest_steps = ntry_steps

                        xc_cs(:,:) =     xc_cn(:,:)
                     xwtot_cs(:)   =  xwtot_cn(:)
                     xutot_cs(:)   =  xutot_cn(:)
                    ysolid_cs(:)   = ysolid_cn(:)
                     bconc_cs(:)   =  bconc_cn(:)
                     wcflx_cs(:)   =  wcflx_cn(:)
                  wcflx_bi_cs(:)   =  wcflx_bi_cn(:)
                     bfflx_cs(:)   =  bfflx_cn(:)

                ENDIF

              ENDIF
#endif

            CASE(jp_its_psvol)
                                    ! Set the number of trial steps.
              ntry_steps = jp_ntrysteps_psvol

                                    ! Correction factor for this trial step.
              IF (itry_step < ntry_steps) THEN
                apsv_factor(:)      = (DBLE(itry_step)/DBLE(ntry_steps))
                apsv_factor(jf_mud) = 1.0D+00
              ELSE
                apsv_factor(:)      = 1.0D+00
              ENDIF

#ifdef ITS_SALVAGING
                                    ! If salvaging active, backup the
                                    ! currently accepted best result.
              IF (jp_its_psvol <= jp_its_last) THEN

                f_its_curr = DBLE(itry_step-1)/DBLE(ntry_steps)

                IF (f_its_curr >= f_its_best) THEN

                       i_its_best  = jp_its_psvol
                       f_its_best  = f_its_curr
                       ibest_step  = itry_step-1
                       nbest_steps = ntry_steps

                        xc_cs(:,:) =     xc_cn(:,:)
                     xwtot_cs(:)   =  xwtot_cn(:)
                     xutot_cs(:)   =  xutot_cn(:)
                    ysolid_cs(:)   = ysolid_cn(:)
                     bconc_cs(:)   =  bconc_cn(:)
                     wcflx_cs(:)   =  wcflx_cn(:)
                  wcflx_bi_cs(:)   =  wcflx_bi_cn(:)
                     bfflx_cs(:)   =  bfflx_cn(:)

                ENDIF

              ENDIF
#endif

            CASE(jp_its_wfflx)
                                    ! Set the number of trial steps.
              ntry_steps = jp_ntrysteps_wfflx

                                    ! Correction factor for this trial step.
              IF (itry_step < ntry_steps) THEN
                factor = DBLE(itry_step)/DBLE(ntry_steps)
              ELSE
                factor = 1.0D+00
              ENDIF

              IF(n_packlay == 0) THEN
                                    ! If we are not currently trying to
                                    ! rescue a chemically eroding
                                    ! (collapsing) column then, apply the
                                    ! correction factor to generate new
                                    ! wfflx forcing values.

                IF (itry_step == ntry_steps) THEN
                  wfflx_tn = wfflx_gn
                ELSE
                  wfflx_tn = wfflx_gn*factor
                  wfflx_tn(jf_mud) = wfflx_gn(jf_mud)
                ENDIF
                CALL PROFIW(wfflx_tn, xc_cn, xzdn_n, xwtot_cn, aw_bplus)
                CALL PROFIU(xwtot_cn, aw_bplus, xutot_cn)

              ELSE                  ! if n_packlay /= 0

                                    ! If n_packlay /= 0, the column is
                                    ! currently collapsing and we are
                                    ! facing a chemical erosion epsisode.
                                    ! Increase the amount of clay, such
                                    ! that there should be no erosion
                                    ! going on initially.
                                    ! Notice that bfflx_sv < 0.
                wfflx_tn = wfflx_gn
                IF(itry_step == 0) THEN
                  wfflx_tn(jf_mud) = wfflx_gn(jf_mud)
     &                - SUM(bfflx_sv(:)*apsv(:))/apsv(jf_mud)
                ELSEIF(itry_step == ntry_steps) THEN
                  CONTINUE
                ELSE
                  wfflx_tn(jf_mud) = wfflx_gn(jf_mud)
     &             -(1D0-factor)*SUM(bfflx_sv(:)*apsv(:))/apsv(jf_mud)
                ENDIF
                CALL PROFIW(wfflx_tn, xc_cn, xzdn_n, xwtot_cn, aw_bplus)
                CALL PROFIU(xwtot_cn, aw_bplus, xutot_cn)

              ENDIF

#ifdef ITS_SALVAGING
                                    ! If salvaging active, backup the
                                    ! currently accepted best result
              IF (jp_its_wfflx <= jp_its_last) THEN

                f_its_curr = DBLE(itry_step-1)/DBLE(ntry_steps)

                IF (f_its_curr > f_its_best) THEN

                       i_its_best  = jp_its_wfflx
                       f_its_best  = f_its_curr
                       ibest_step  = itry_step-1
                       nbest_steps = ntry_steps

                        xc_cs(:,:) =     xc_cn(:,:)
                     xwtot_cs(:)   =  xwtot_cn(:)
                     xutot_cs(:)   =  xutot_cn(:)
                    ysolid_cs(:)   = ysolid_cn(:)
                     bconc_cs(:)   =  bconc_cn(:)
                     wcflx_cs(:)   =  wcflx_cn(:)
                  wcflx_bi_cs(:)   =  wcflx_bi_cn(:)
                     bfflx_cs(:)   =  bfflx_cn(:)

                ENDIF

              ENDIF
#endif

            CASE(jp_its_timst)
                                    ! This method is only used
                                    ! for steady-state time steps
              IF (datime > 0.0D+00) THEN
                method = method + 1
                itry_step = 0
                CYCLE try_methods
              ENDIF
                                    ! Set the number of trial steps.
              ntry_steps = jp_ntrysteps_timst

                                    ! Correction factor for this trial step.
              IF (itry_step == 0) THEN
                datime_tn = dp_timst_dtmin
              ELSEIF (itry_step < ntry_steps) THEN
                datime_tn = dp_timst_dtmin
     &                      * (dp_timst_dtmax/dp_timst_dtmin)
     &                        **(DBLE(itry_step)/DBLE(ntry_steps-1))
              ELSE
                datime_tn = datime
              ENDIF

#ifdef ITS_SALVAGING
                                    ! If salvaging active, backup the
                                    ! currently accepted best result.
              IF (jp_its_timst <= jp_its_last) THEN

                f_its_curr = DBLE(itry_step-1)/DBLE(ntry_steps)

                IF (f_its_curr >= f_its_best) THEN

                       i_its_best  = jp_its_timst
                       f_its_best  = f_its_curr
                       ibest_step  = itry_step-1
                       nbest_steps = ntry_steps

                        xc_cs(:,:) =     xc_cn(:,:)
                     xwtot_cs(:)   =  xwtot_cn(:)
                     xutot_cs(:)   =  xutot_cn(:)
                    ysolid_cs(:)   = ysolid_cn(:)
                     bconc_cs(:)   =  bconc_cn(:)
                     wcflx_cs(:)   =  wcflx_cn(:)
                  wcflx_bi_cs(:)   =  wcflx_bi_cn(:)
                     bfflx_cs(:)   =  bfflx_cn(:)

                ENDIF

              ENDIF
#endif

            CASE(jp_its_colla)
                                    ! This method is only used
                                    ! for collapsing columns.
              IF (n_packlay == 0) THEN
                method = method + 1
                itry_step = 0
                CYCLE try_methods
              ELSE
                ntry_steps = jp_ntrysteps_colla

                                    ! Correction factor for this trial step.
                IF (itry_step < ntry_steps) THEN
                  factor = DBLE(itry_step)/DBLE(ntry_steps)
                ELSE
                  factor = 1.0D+00
                ENDIF

                ysolid_to(:) = (ysolid_go(:) + psolid(:))

                IF(itry_step == 0) THEN
                  ysolid_to(:) = 0D0
                  ysolid_to(jf_mud) =
     &              SUM((ysolid_go(:) + psolid(:))*apsv(:))/apsv(jf_mud)
                ELSEIF(itry_step == ntry_steps) THEN
                  ysolid_to(:) = ysolid_go(:) + psolid(:)
                ELSE
                  ysolid_to(:) = factor * (ysolid_go(:) + psolid(:))
                  ysolid_to(jf_mud) = ysolid_to(jf_mud)
     &               + (1D0 - factor)
     &                 *SUM((ysolid_go(:) + psolid(:))*apsv(:))
     &                 /apsv(jf_mud)
                ENDIF
              ENDIF

            CASE DEFAULT

              CONTINUE

            END SELECT

            ysolid_cn = ysolid_to   ! Both need to be the same initially,
                                    ! as they are used to calculate
                                    ! the concentration for transport.

#ifdef DEBUG
#  ifdef DEBUG_SOLVSED
            WRITE(jp_stddbg, cfmt_a, ADVANCE="NO")
            SELECT CASE(method)
            CASE(jp_its_basic)
              WRITE(jp_stddbg,
     &              '("Column ", I0, ": BASIC iterations")') i_column
#             ifdef DEBUG_SOLVSED_EQN_SUMMARY
              WRITE(jp_trace_eqn_summary,
     &              '("Column ", I0, ": BASIC iterations")') i_column
#             endif
#             ifdef DEBUG_SOLVSED_EQN_DETAIL
              WRITE(jp_trace_eqn_detail,
     &              '("Column ", I0, ": BASIC iterations")') i_column
#             endif

            CASE(jp_its_solut)
              WRITE(jp_stddbg,
     &              '("Column ", I0, ": SOLUT iterations")') i_column
#             ifdef DEBUG_SOLVSED_EQN_SUMMARY
              WRITE(jp_trace_eqn_summary,
     &              '("Column ", I0, ": SOLUT iterations")') i_column
#             endif
#             ifdef DEBUG_SOLVSED_EQN_DETAIL
              WRITE(jp_trace_eqn_detail,
     &              '("Column ", I0, ": SOLUT iterations")') i_column
#             endif

            CASE(jp_its_psvol)
              WRITE(jp_stddbg, cfmt_iterations4stage)
     &          i_column, 'PSVOL', itry_step, ntry_steps
#             ifdef DEBUG_SOLVSED_EQN_SUMMARY
              WRITE(jp_trace_eqn_summary, cfmt_iterations4stage)
     &          i_column, 'PSVOL', itry_step, ntry_steps
#             endif
#             ifdef DEBUG_SOLVSED_EQN_DETAIL
              WRITE(jp_trace_eqn_detail, cfmt_iterations4stage)
     &          i_column, 'PSVOL', itry_step, ntry_steps
#             endif

            CASE(jp_its_rreac)
              WRITE(jp_stddbg, cfmt_iterations4stage)
     &          i_column, 'RREAC', itry_step, ntry_steps
#             ifdef DEBUG_SOLVSED_EQN_SUMMARY
              WRITE(jp_trace_eqn_summary, cfmt_iterations4stage)
     &          i_column, 'RREAC', itry_step, ntry_steps
#             endif
#             ifdef DEBUG_SOLVSED_EQN_DETAIL
              WRITE(jp_trace_eqn_detail, cfmt_iterations4stage)
     &          i_column, 'RREAC', itry_step, ntry_steps
#             endif

            CASE(jp_its_wfflx)
              WRITE(jp_stddbg, cfmt_iterations4stage)
     &          i_column, 'WFFLX', itry_step, ntry_steps
#             ifdef DEBUG_SOLVSED_EQN_SUMMARY
              WRITE(jp_trace_eqn_summary, cfmt_iterations4stage)
     &          i_column, 'WFFLX', itry_step, ntry_steps
#             endif
#             ifdef DEBUG_SOLVSED_EQN_DETAIL
              WRITE(jp_trace_eqn_detail, cfmt_iterations4stage)
     &          i_column, 'WFFLX', itry_step, ntry_steps
#             endif

            CASE(jp_its_timst)
              WRITE(jp_stddbg, cfmt_iterations4stage)
     &          i_column, 'TIMST', itry_step, ntry_steps
#             ifdef DEBUG_SOLVSED_EQN_SUMMARY
              WRITE(jp_trace_eqn_summary, cfmt_iterations4stage)
     &          i_column, 'TIMST', itry_step, ntry_steps
#             endif
#             ifdef DEBUG_SOLVSED_EQN_DETAIL
              WRITE(jp_trace_eqn_detail, cfmt_iterations4stage)
     &          i_column, 'TIMST', itry_step, ntry_steps
#             endif

            CASE(jp_its_colla)
              WRITE(jp_stddbg, cfmt_iterations4stage)
     &          i_column, 'COLLA', itry_step, ntry_steps
#             ifdef DEBUG_SOLVSED_EQN_SUMMARY
              WRITE(jp_trace_eqn_summary, cfmt_iterations4stage)
     &          i_column, 'COLLA', itry_step, ntry_steps
#             endif
#             ifdef DEBUG_SOLVSED_EQN_DETAIL
              WRITE(jp_trace_eqn_detail, cfmt_iterations4stage)
     &          i_column, 'COLLA', itry_step, ntry_steps
#             endif

            END SELECT
#  endif
#endif

            CALL ImplicitTimeStep(
     I            xzdn_n, xzdv_n,
     I            wdata_tn,
     I            atime_tn, datime_tn, wconc_tn, wfflx_tn,
     X            xc_go, ysolid_to, xwtot_go, xutot_go,
     X            xc_cn, ysolid_cn, xwtot_cn, xutot_cn,
     O            bconc_cn, wcflx_cn, wcflx_bi_cn, bfflx_cn, itsflag)

            IF(itsflag == 0) THEN
                                    ! If ImplicitTimeStep completed
                                    ! sucessfully, then:
              IF(itry_step == ntry_steps) THEN
                                    !  - if the method's last step just
                                    !    completed, we are OK.
                                    !    Proceed to exit.
                                    !    Make sure rreac_factor is set
                                    !    to rreac_factor_max ...
                rreac_factor(:) = rreac_factor_max(:)
                                    !    ... and apsv_factor is set to 1.
                apsv_factor(:)  = 1.0D+00
                                    !    ... and datime_tn is set to datime
                datime_tn  = datime
                                    !  - if we encountered a failure
                                    !    previously, report success.
                IF(failure_flag == 0) THEN
                  SELECT CASE(method)
                  CASE(jp_its_basic)
                    WRITE(progresslog_unit, cfmt_success)
     &                              i_column, 'BASIC'
                  CASE(jp_its_solut)
                    WRITE(progresslog_unit, cfmt_success)
     &                              i_column, 'SOLUT'
                  CASE(jp_its_psvol)
                    WRITE(progresslog_unit, cfmt_success_stage)
     &                              i_column, 'PSVOL',
     &                              itry_step, ntry_steps
                  CASE(jp_its_rreac)
                    WRITE(progresslog_unit, cfmt_success_stage)
     &                              i_column, 'RREAC',
     &                              itry_step, ntry_steps
                  CASE(jp_its_wfflx)
                    WRITE(progresslog_unit, cfmt_success_stage)
     &                              i_column, 'WFFLX',
     &                              itry_step, ntry_steps
                  CASE(jp_its_timst)
                    WRITE(progresslog_unit, cfmt_success_stage)
     &                              i_column, 'TIMST',
     &                              itry_step, ntry_steps
                  CASE(jp_its_colla)
                    WRITE(progresslog_unit, cfmt_success_stage)
     &                              i_column, 'COLLA',
     &                              itry_step, ntry_steps
                  CASE DEFAULT
                    CONTINUE
                  END SELECT
                ENDIF
                EXIT try_methods

              ELSE

                                    ! If the current method's last step
                                    ! has not yet completed, increment step
                                    ! counter and carry out the next step
                                    ! with the same method.

                itry_step = itry_step + 1
                CYCLE try_methods

              ENDIF

            ELSE                    ! ImplicitTimeStep did not succeed, so ...:

              failure_flag = 0      ! we have got one failure now.
              IF(tinfo_flg == -1) THEN
                WRITE(progresslog_unit, *) TRIM(tinfo_str)
                tinfo_flg = 0
              ENDIF
              SELECT CASE(method)
              CASE(jp_its_basic)
                WRITE(progresslog_unit, cfmt_failed)
     &                              i_column, 'BASIC'
              CASE(jp_its_solut)
                WRITE(progresslog_unit, cfmt_failed)
     &                              i_column, 'SOLUT'
              CASE(jp_its_psvol)
                WRITE(progresslog_unit, cfmt_failed_stage)
     &                              i_column, 'PSVOL',
     &                              itry_step, ntry_steps
              CASE(jp_its_rreac)
                WRITE(progresslog_unit, cfmt_failed_stage)
     &                              i_column, 'RREAC',
     &                              itry_step, ntry_steps
              CASE(jp_its_wfflx)
                WRITE(progresslog_unit, cfmt_failed_stage)
     &                              i_column, 'WFFLX',
     &                              itry_step, ntry_steps
              CASE(jp_its_timst)
                WRITE(progresslog_unit, cfmt_failed_stage)
     &                              i_column, 'TIMST',
     &                              itry_step, ntry_steps
              CASE(jp_its_colla)
                WRITE(progresslog_unit, cfmt_failed_stage)
     &                              i_column, 'COLLA',
     &                              itry_step, ntry_steps
              CASE DEFAULT
                CONTINUE
              END SELECT

#ifdef DEBUG_SOLVSED
#ifdef DEBUG_SOLVSED_TRACE_ALL
              i_request = 0
              CALL SOLVSED_TRACE(i_column, i_request, iflag, xzdn_n,
     &                           wdata_gn,
     &                           wconc_gn, wfflx_gn,
     &                           xc_go, xc_cn,
     &                           xwtot_go, xwtot_cn,
     &                           xutot_go, xutot_cn)
#endif
#endif

                                    ! If the current method was not yet
                                    ! the last one, increment the method
                                    ! counter and proceed with step 0 of
                                    ! the next method.
              method = method + 1
              itry_step = 0
              CYCLE try_methods
            ENDIF
          ENDDO try_methods

#ifdef ITS_SALVAGING
          IF (itsflag /= 0) THEN
            IF (i_its_best /= 0) THEN
                                    ! None of the methods tried was successful
                                    ! However, salvaging is active, and we
                                    ! accept the best backup values before
                                    ! failure.
                    xc_cn(:,:) =     xc_cs(:,:)
                 xwtot_cn(:)   =  xwtot_cs(:)
                 xutot_cn(:)   =  xutot_cs(:)
                ysolid_cn(:)   = ysolid_cs(:)
                 bconc_cn(:)   =  bconc_cs(:)
                 wcflx_cn(:)   =  wcflx_cs(:)
              wcflx_bi_cn(:)   =  wcflx_bi_cs(:)
                 bfflx_cn(:)   =  bfflx_cs(:)

              SELECT CASE(i_its_best)
              CASE(jp_its_psvol)
                WRITE(progresslog_unit, cfmt_accept_stage)
     &                              i_column, 'PSVOL',
     &                              ibest_step, nbest_steps
              CASE(jp_its_rreac)
                WRITE(progresslog_unit, cfmt_accept_stage)
     &                              i_column, 'RREAC',
     &                              ibest_step, nbest_steps
              CASE(jp_its_wfflx)
                WRITE(progresslog_unit, cfmt_accept_stage)
     &                              i_column, 'WFFLX',
     &                              ibest_step, nbest_steps
              CASE(jp_its_timst)
                WRITE(progresslog_unit, cfmt_accept_stage)
     &                              i_column, 'TIMST',
     &                              ibest_step, nbest_steps
              END SELECT
              itsflag = 0           ! correct itsflag for below
            ENDIF
          ENDIF
#endif

          itsflag_zero: IF (itsflag /= 0) THEN
                                    ! If none of the methods tried was
                                    ! successful, report this.
            WRITE(progresslog_unit,
     &       '("  Column ",i4,": No convergence for time interval ",' //
     &       ' f12.2, " to ", f12.2)')
     &       i_column, atime, atime + datime
            WRITE(jp_stderr, convg_fmt) i_column
            n_trouble = n_trouble + 1
            soflag = 1

#ifdef DEBUG
#ifdef DEBUG_SOLVSED
#ifdef DEBUG_SOLVSED_SAVE_ANYWAY
                                    ! For debugging reasons, it is sometimes
                                    ! interesting to proceed anyway.
            CALL SAVE_COLUMN(i_column, iflag,
     &                       xzdn=xzdn_n, xzdv=xzdn_n,
     &                       xc=xc_cn, ysolid=ysolid_cn)
            CALL SAVE_BOUNDARY_FLUXES(i_column, iflag,
     &                       bconc_cn, wcflx_cn, wcflx_bi_cn, bfflx_cn)
            CALL SAVE_MILIEUCHARAS(i_column, iflag)
#endif
#endif
#endif
                                    ! And quit chem_erosion loop
                                    ! to proceed with next column.
            EXIT chem_erosion

          ELSE

                                    ! If at least one method yielded an
                                    ! acceptable result, test whether
                                    ! chemical erosion was possibly too
                                    ! strong.

            ys_neg: IF ( ANY(ysolid_cn(jmf_to_if(:)) < 0.0D+00) ) THEN

                                    ! If chemical erosion was too strong,
                                    ! ysolid_cn is negative for some
                                    ! material solid components (i.e., for
                                    ! solid components with finite density).
                                    ! Report this fact.

              chero_fmt = '("  Column ",i0,": Strong chemical erosion")'
              WRITE(progresslog_unit,chero_fmt) i_column

                                    ! Consider one more historical (core)
                                    ! layer than before.
              n_packlay = n_packlay + 1

                                    ! Get the masses of solids in the
                                    ! n_packlay top core layers.
              CALL CORELAY_PACKMASS(i_column, n_packlay, atime,
     &                              psolid, cpmflag)

                                    ! Report the details to the log file ...
              chero_fmt =  '("  Column ",i0,": Mixing ", i0,'
     &                   // '" core layer(s), incl. ", i0,'
     &                   // '" phantom(s), into Y")'
              WRITE(progresslog_unit,chero_fmt)
     &             i_column, n_packlay, cpmflag
              WRITE(progresslog_unit,
     &                     '("  Repeating methods for this step")')

                                    ! ... and restart the calculation with
                                    ! the first method. psolid will be
                                    ! added onto ysolid_go, to yield the
                                    ! ysolid_to used for the calculation.

                                    ! First save the bfflx so that it can
                                    ! possibly be used for evaluation purposes.
              bfflx_sv(:) = bfflx_cn(:)

              CYCLE chem_erosion

            ELSE                    ! i.e., if ALL(ysolid_cn(jmf_to_if) >= 0.0D+00)

              IF ( ANY(bfflx_cn(jmf_to_if(:)) < 0.0D+00) ) THEN
                                    ! Print out notice
                chero_fmt = '("  Column ",i0,": Chemical erosion")'
                WRITE(progresslog_unit,chero_fmt) i_column
              ENDIF

              IF (n_packlay /= 0) THEN

                                    ! Register the number of layers to
                                    ! erode in MOD_SEDCORE so that the
                                    ! call of REACLAY_X_CORELAY by the
                                    ! caller of the present procedure will
                                    ! correctly update the sediment core
                                    ! layer stack, with the full bookkeeping.


                CALL CORELAY_NLAY2ERODE_SET(i_column, n_packlay)

              ENDIF


              CALL PROFIW(wfflx_gn, xc_cn, xzdn_n, xwtot_cn, aw_bplus)
              CALL PROFIU(xwtot_cn, aw_bplus, xutot_cn)

#ifdef SAVE_COLUMN_BYCALLER
              solvsed_xc(:,:,i_column)    = xc_cn
              solvsed_ysolid(:,i_column)  = ysolid_cn
                                    ! Still save xwtot, xutot
                                    ! (in case a save to file would be made).
              CALL SAVE_COLUMN(i_column, iflag,
     &                         xwtot=xwtot_cn, xutot=xutot_cn)

#else
              CALL SAVE_COLUMN(i_column, iflag,
     &               xc=xc_cn, ysolid=ysolid_cn,
     &               xwtot=xwtot_cn, xutot=xutot_cn)
#endif
              CALL SAVE_BOUNDARY_FLUXES(i_column, iflag,
     &               bconc = bconc_cn, wcflx = wcflx_cn,
     &               wcflx_bi = wcflx_bi_cn, bfflx = bfflx_cn)

              CALL SAVE_MILIEUCHARAS(i_column, iflag)

              EXIT chem_erosion

            ENDIF ys_neg

          ENDIF itsflag_zero

        ENDDO chem_erosion


      ENDDO do_columns

#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_modprocname_a) 'End'
      WRITE(jp_stddbg,'()')
#endif


      RETURN


      CONTAINS

#include <solvsed_onestep_its_solut.F>


!=======================================================================
      END SUBROUTINE SOLVSED_ONESTEP
!=======================================================================

#ifdef DEBUG
#ifdef DEBUG_SOLVSED_TRACE
#include <debug/solvsed-solvsed_trace.F>
#endif
#endif


