!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


!-----------------------------------------------------------------------
      SUBROUTINE ImplicitTimeStep(
     I            xzdn, xzdv,
     I            wdata,
     I            atime, datime, wconc, wfflx,
     X            xco, ysolido, xwtoto, xutoto,
     X            xcn, ysolidn, xwtotn, xutotn,
     O            bconc, wcflx, wcflx_bi, bfflx, itsflag)
!-----------------------------------------------------------------------


      USE mod_defines_medusa
      USE mod_seafloor_wdata
      USE mod_fvupwind_params
      USE mod_gridparam
      USE mod_indexparam
      USE mod_materialcharas
      USE mod_milieucharas
      USE mod_transport
      USE mod_gauss

      USE mod_equilibcontrol
      USE mod_chemicalconsts,       ONLY: SETCCT
      USE mod_processcontrol
      USE mod_medinterfaces,        ONLY: FVUPWIND_EQUATIONS,
     &                                    FVUPWIND_JACOBIAN,
     &                                    FVUPWIND_TRANLAY_RADIODECAY
      USE mod_rreac,                ONLY: REACRATE

#ifdef DEBUG
#  include <debug.h>
#endif


      IMPLICIT NONE


! =====================
! Subroutine Parameters
! =====================

      ! Fine-tuning parameters for the Newton iterations
      DOUBLE PRECISION, PARAMETER   :: dp_sigma_eqn2_threshold
     &                                  = neqns_total*(1.0D-06)**2
      DOUBLE PRECISION, PARAMETER   :: dp_delta_var_threshold = 1.0D-9

      INTEGER, PARAMETER            :: jp_itermax            = 120
      INTEGER, PARAMETER            :: jp_newton_test_min    =   0
      INTEGER, PARAMETER            :: jp_newton_test_minmax =   0
      INTEGER, PARAMETER            :: jp_newton_test_max    =   4


! ============================
! End of Subroutine Parameters
! ============================


      DOUBLE PRECISION,
     &  DIMENSION(idnw:idnb)        :: xzdn
      DOUBLE PRECISION,
     &  DIMENSION(idvw:idvb)        :: xzdv

      TYPE(WDATA_CONTAINER)         :: wdata

      DOUBLE PRECISION,
     &  DIMENSION (nsolut)          :: wconc

      DOUBLE PRECISION,
     &  DIMENSION (nsolid)          :: wfflx

      DOUBLE PRECISION              :: atime, datime

      DOUBLE PRECISION,
     &  DIMENSION(idnw:idnb, ncompo)
     &                              :: xco, xcn

      DOUBLE PRECISION,
     &  DIMENSION(idvs:idvb)        :: xwtoto, xwtotn
      DOUBLE PRECISION,
     &  DIMENSION(idvw:idvb)        :: xutoto, xutotn

      DOUBLE PRECISION,
     &  DIMENSION(nsolid)           :: ysolido, ysolidn

      INTEGER                       :: itsflag

      DOUBLE PRECISION,
     &  DIMENSION(nsolut)           :: bconc, wcflx, wcflx_bi
      DOUBLE PRECISION,
     &  DIMENSION(nsolid)           :: bfflx

      INTENT(IN)    :: xzdn, xzdv, wdata, atime, datime
      INTENT(IN)    :: wconc, wfflx
      INTENT(IN)    :: xco, ysolido, xwtoto, xutoto
      INTENT(INOUT) :: xcn, ysolidn, xwtotn, xutotn
      INTENT(OUT)   :: itsflag
      INTENT(OUT)   :: bconc, wcflx, wcflx_bi, bfflx

! Local variables
! General purpose
      INTEGER                       :: jsolid, jsolut, jcompo
      INTEGER                       :: jmatsolid

      INTEGER, DIMENSION(idnt:idnb) :: jf_svc
      INTEGER                          io_svc

      INTEGER, PARAMETER            :: idntp1 = idnt + 1
      INTEGER, PARAMETER            :: idnwp1 = idnw + 1
      INTEGER                       :: inode

      DOUBLE PRECISION              :: atimen
      DOUBLE PRECISION              :: one_datime     ! = 1/datime if datime > 0
                                                      ! = 0 if datime = 0 (steady-state calculations)

      DOUBLE PRECISION,
     &  DIMENSION(nsolid)           :: yco, ycn

      DOUBLE PRECISION              :: ysolido_vol, ysolidn_vol

      DOUBLE PRECISION,
     &  DIMENSION(neqns_node, idnw:idnb)
     &                              :: eqn_syst

      DOUBLE PRECISION, TARGET,
     &  DIMENSION(neqns_node, nvars_node, idnw:idnb, -1:1)
     &                              :: jcb_syst

! Variables related to Gauss linear system resolution
      INTEGER                       :: nrs

#ifdef DEBUG
      INTEGER                       :: nused, nkequ, nkvar
#endif

      INTEGER                       :: info

! Variables related to Block Tridiagonal system resolution
      DOUBLE PRECISION,
     &   DIMENSION(nvars_node, nvars_node)
     &                              :: WorkA, WorkB
      DOUBLE PRECISION, POINTER,
     &   DIMENSION(:, :)            :: PtrWorkA1, PtrWorkA2

      DOUBLE PRECISION,
     &   DIMENSION(nvars_node, nvars_node+1)
     &                              :: WorkX, WorkY

      DOUBLE PRECISION,
     &   DIMENSION(nvars_node)      :: WorkX1, WorkY1
      DOUBLE PRECISION, POINTER,
     &   DIMENSION(:)               :: PtrWorkC1

      DOUBLE PRECISION, TARGET,
     &   DIMENSION(nvars_node, idnw:idnb)
     &                              :: WorkC
      DOUBLE PRECISION,
     &   DIMENSION(nvars_node, idnw:idnb)
     &                              :: deltac, deltac_best

      DOUBLE PRECISION, TARGET,
     &   DIMENSION(nvars_node, nvars_node, idnw:idnb)
     &                              :: WorkG

! Variables related to Newton iterations
      INTEGER                       :: ii, iequ0, iequ, i, jvar

      DOUBLE PRECISION,
     &   DIMENSION(ncompo,idnw:idnb)
     &                              :: xc_scale
      DOUBLE PRECISION,
     &   DIMENSION(neqns_node,idnw:idnb)
     &                              :: eqn_scale

      DOUBLE PRECISION              :: sigma_eqn2,
     &                                 sigma_eqn2_tst,
     &                                 sigma_eqn2_best,
     &                                 sigma_eqn2_columnbest

      DOUBLE PRECISION              :: newton_factor

      INTEGER                       :: newton_i, newton_i_best

      INTEGER                       :: niter_conv_sigmaeqn
      INTEGER                       :: iflag_conv_delta

      DOUBLE PRECISION,
     &   DIMENSION(idnw:idnb, ncompo)
     &                              :: xcn_0, xcn_best

      DOUBLE PRECISION,
     &   DIMENSION(idvs:idvb)       :: xwtotn_best
      DOUBLE PRECISION,
     &   DIMENSION(idvw:idvb)       :: xutotn_best
      DOUBLE PRECISION              :: aw_bplus_best


! Variables related to the flux calculations

      DOUBLE PRECISION, DIMENSION(ncompo)
     &                              :: dt_xc,
     &                                 flxadvdif,
     &                                 reaction_na, reaction_nb,
     &                                 reaction,
     &                                 flxnonloc_na, flxnonloc_nb,
     &                                 flxnonloc
      DOUBLE PRECISION, DIMENSION(ncompo)
     &                              :: ac
      DOUBLE PRECISION              :: azdn, aphi, azdnz
      DOUBLE PRECISION              :: aw_bplus
      INTEGER                       :: isign_aw_bplus


#ifdef DEBUG
#  ifdef DEBUG_SOLVSED
#    ifdef DEBUG_SOLVSED_EQN_DETAIL
      CHARACTER(LEN=15)             :: cs_fmt9200
#    endif
#    ifdef DEBUG_SOLVSED_JACOBIAN
      CHARACTER(LEN=63)             :: cs_fmt99992
      CHARACTER(LEN=63)             :: cs_fmt99993
#    endif
#  endif
#endif


      ! Standard I/O related data
      ! -------------------------

      CHARACTER(LEN=*), PARAMETER :: cfmt_err_a =
     &  '("[solvsedFVFullUpwind.F/ImplicitTimeStep] error: ", A)'


! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
! End of declarations
! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


!----------------------------------------------------------------------
! Program Start
!----------------------------------------------------------------------

      IF (datime > 0.0D+00) THEN
        one_datime = 1.0D+00/datime
      ELSE
        one_datime = 0.0D+00
      ENDIF


      itsflag = 0

                                    ! Calculate yco from ysolido
      ysolido_vol = SUM(ysolido(:)*apsv(:)*apsv_factor(:))
      yco(:) = ysolido(:)/ysolido_vol

                                    ! Set process parameters and
                                    ! equilibrium constants
      CALL SETCCT(wdata, wconc(:))
      CALL SetEquilibParameters(wconc(:))
      CALL SetProcessParameters

                                    ! Set the diffusion coefficients
                                    ! (SAVE'd via MOD_DIFFUSIONCOEFFS):
      CALL MDIFFC(wdata)            !  - molecular and ionic diffusion


      atimen = atime + datime       ! Set target time


      ii = 0                        ! Reset iteration counter
      niter_conv_sigmaeqn = 0
      iflag_conv_delta = -1
      newton_i = jp_newton_test_min - 1
      isign_aw_bplus = 0            ! Reset sign flag for aw_bplus


 1    CONTINUE
      ii = ii + 1

                                    ! Update biodiffusion (bioturbation)
                                    ! and bioirrigation coefficients,
                                    ! which possibly depend on xc.
      CALL BDIFFC(xzdn, xzdv, xcn, wconc, wfflx)  ! Bioturbation (biodiffusion)
      CALL BIRRIC(xzdn, xcn, wconc, wfflx)  ! Bioirrigation

!
! Equation to solve:
!
!  \partial{\hat{C}}     \partial{\hat{F}}
!  -----------------  +  -----------------  - \hat{R}^V  = 0
!  \partial{t}           \partial{z}
!
! +-----------------+   +------------------------------+
!  Chosen by time        Given by eqn_dflx_rea, from
!  integration           EQNS_DZFLUX_REAC, as a function
!  method                of xc=C, xwtot=(phi_s*w) and xutot=(phi*u).
!  (implicit here)
!
! in conjunction with algebraic constraints. The general
! system is formulated as EQN(C, C', C", xwtot, xwtot', xutot, xutot') = 0
! and solved by a Newton iteration:
! EQN(xc_i, xwtot_i, xutot_i) = -JAC(xc_i, xwtot_i, xutot_i) * (xc_{i+1}-xc_i)
! The current solution method expresses d2x_xc as a function of xc.


! jf_svc is currently locked to jf_mud, as in certain
! circumstances, updating it to the index of the most abundant
! solid may lead to oscillatory behaviour, when it switches
! from one index to another between iterations.
!      DO inode = idnt, idnb
!        jf_svc(inode) = 1
!        io_svc = jf_to_io(1)
!        DO jsolid = 2, nsolid
!           jcompo = jf_to_io(jsolid)
!           IF(xcn(inode,jcompo) > xcn(inode,io_svc)) THEN
!             jf_svc(inode) = jsolid
!             io_svc = jcompo
!           ENDIF
!        ENDDO
!      ENDDO
       jf_svc(:) = jf_mud

                                    ! Update xwtotn and aw_bplus to match wfflx and xcn
      CALL PROFIW(wfflx, xcn, xzdn, xwtotn, aw_bplus)
      CALL PROFIU(xwtotn, aw_bplus, xutotn)

      IF (aw_bplus < 0.0D+00) THEN
        isign_aw_bplus = -1
        CALL FVUPWIND_TRANLAY_RADIODECAY(datime, yco, ycn)
      ELSE
        isign_aw_bplus = +1
        ycn(:) = yco(:)             ! pro forma initialization
      ENDIF


      CALL FVUPWIND_SCALES(xzdn, jf_svc, xc_scale, eqn_scale)


      CALL FVUPWIND_EQUATIONS(atimen, datime, xzdn, xzdv,
     &  wconc, wfflx,
     &  xco, yco, xcn, ycn, xwtotn, xutotn,
     &  jf_svc,
     &  eqn_syst)

      IF ((ii == 1) .AND. (newton_i < jp_newton_test_min)) THEN
        sigma_eqn2_columnbest = SUM((eqn_syst/eqn_scale)**2)
      ENDIF

#ifdef DEBUG
#  ifdef DEBUG_SOLVSED
#    include <debug/solvsed-eqn_summary.F>
#  endif
#endif

      CALL FVUPWIND_JACOBIAN(atimen, datime, xzdn, xzdv,
     &  wconc, wfflx,
     &  xco, yco, xcn, ycn, xwtotn, xutotn,
     &  jf_svc, jcb_syst)

#ifdef DEBUG
#  ifdef DEBUG_SOLVSED
      nkequ = neqns_total
      nkvar = nvars_total
      nused = neqns_total
#    include <debug/solvsed-jacobian.F>
#  endif
#endif

      nrs = 1+nvars_node

#ifdef DEBUG
#  ifdef DEBUG_SOLVSED
#    ifdef DEBUG_SOLVSED_ITERAT_II
      WRITE(jp_stddbg, '(3X, I4, " ---- initial aw_bplus is", 6X)',
     &                              ADVANCE="NO") ii
      WRITE(jp_stddbg,*) aw_bplus
#    endif
#  endif
#endif

      DO inode = idnw, idnb

        IF(inode == idnw) THEN
          WorkA      = jcb_syst(:, :, inode, 0)
          WorkY(:,1) = eqn_syst(:, inode)
        ELSE
          PtrWorkA1  => jcb_syst(:, :, inode, -1)
          PtrWorkA2  => WorkG(:,:,inode-1)
          WorkA      =  jcb_syst(:, :, inode, 0)
     &                              - MATMUL(PtrWorkA1, PtrWorkA2)
          PtrWorkC1  => WorkC(:,inode-1)
          WorkY(:,1) =  eqn_syst(:, inode)
     &                              - MATMUL(PtrWorkA1, PtrWorkC1)
        ENDIF

        IF(inode < idnb) THEN
          WorkY(:,2:nrs) = jcb_syst(:, :, inode, 1)
          WorkX(:,:)     = WorkY(:,:)
          CALL GM_GESV(WorkA, WorkX, INFO=info)
          IF (info /= 0) THEN
            itsflag = 2             ! Gaussian inversion in trouble
            WRITE(jp_stderr, cfmt_err_a, ADVANCE="NO")
     &        'Trouble [2] at node '
            WRITE(jp_stderr, '(I0, " (info=", I0, ")")') inode, info
#ifdef DEBUG
#  ifdef DEBUG_SOLVSED
            WRITE(jp_stddbg, '(3X, A, I0, A, I0, A, I0)')
     &        ' Aborting iteration ', ii,
     &        ' (return after GM_GESV [2], info=', info,
     &        ') because of trouble at node ', inode
#  endif
#endif
             RETURN
          ENDIF
          WorkG(:,:,inode) = WorkX(:,2:nrs)
          WorkC(:, inode)  = WorkX(:,1)
        ELSE
          WorkY1(:) = WorkY(:,1)
          WorkX1(:) = WorkY1(:)
          CALL GM_GESV(WorkA, WorkX1, INFO=info)
          IF (info /= 0) THEN
            itsflag = 3             ! Gaussian inversion in trouble
            WRITE(jp_stderr, cfmt_err_a, ADVANCE="NO")
     &        'Trouble [3] at node '
            WRITE(jp_stderr, '(I0)') inode
#ifdef DEBUG
#  ifdef DEBUG_SOLVSED
            WRITE(jp_stddbg, '(3X, A, I0, A, I0)')
     &       ' Aborting iteration ', ii,
     &       ' (return after GM_GESV [3]) because of trouble at node ',
     &       inode
#  endif
#endif
            RETURN
          ENDIF
          WorkC(:, inode)  = WorkX1
        ENDIF

      ENDDO

      deltac(:,idnb) = WorkC(:,idnb)

      DO inode = idnb-1, idnw, -1
        PtrWorkA1 => WorkG(:,:,inode)
        PtrWorkC1 => WorkC(:, inode+1)
        deltac(:,inode) = WorkC(:, inode) - MATMUL(PtrWorkA1, PtrWorkC1)
        IF (inode > idnw)  WorkC(:, inode) = deltac(:,inode)
      ENDDO


      sigma_eqn2 = SUM((eqn_syst/eqn_scale)**2)
      sigma_eqn2_best = sigma_eqn2
                xcn_0 =    xcn
        newton_i_best = jp_newton_test_min - 1

      DO newton_i = jp_newton_test_min, jp_newton_test_max

        newton_factor = 2.0D+00**(newton_i)

        DO inode = idnw, idnb
          xcn(inode,:)    =    xcn_0(inode,:)
     &                      -deltac(:,inode) / newton_factor

                                    ! First test for possibly unphysical
                                    ! material concentrations and bring
                                    ! back into allowable interval
                                    !  - test material solids for xc < 0
          DO jmatsolid = 1, nmatsolid
            jcompo = jmf_to_io(jmatsolid)
            IF (xcn(inode, jcompo) < 0.0D+00) THEN
#ifdef DEBUG
#  ifdef DEBUG_SOLVSED
#    ifdef DEBUG_SOLVSED_ITERAT_OUTOFRANGE
              WRITE(jp_stddbg, '(4X, "At iteration ", I0, "(",I0, "), '
     &          // 'xc(",I0,") < 0 at node ", I0, ": ")', ADVANCE="NO")
     &                              ii, newton_i, jcompo, inode
              WRITE(jp_stddbg, *) xcn(inode,jcompo), ' = ',
     &                            xcn_0(inode, jcompo), ' - ',
     &                            deltac(jcompo, inode) / newton_factor
#    endif
#  endif
#endif
              xcn(inode, jcompo) = 0.0D+00
            ENDIF
          ENDDO

                                    !  - test solutes for xc < 0
          DO jsolut = 1, nsolut
            jcompo = jc_to_io(jsolut)
            IF (xcn(inode, jcompo) < 0.0D+00) THEN
#ifdef DEBUG
#  ifdef DEBUG_SOLVSED
#    ifdef DEBUG_SOLVSED_ITERAT_OUTOFRANGE
              WRITE(jp_stddbg, '(4X, "At iteration ", I0, "(",I0, "), '
     &          // 'xc(",I0,") < 0 at node ", I0, ": ")', ADVANCE="NO")
     &                              ii, newton_i, jcompo, inode
              WRITE(jp_stddbg, *) xcn(inode, jcompo), ' = ',
     &                            xcn_0(inode, jcompo), ' - ',
     &                            deltac(jcompo, inode) / newton_factor
#    endif
#  endif
#endif
              xcn(inode, jcompo) = 0.0D+00
            ENDIF
          ENDDO

                                    !  - test material solids for > rho
          DO jmatsolid = 1, nmatsolid
            jsolid = jmf_to_if(jmatsolid)
            jcompo = jmf_to_io(jmatsolid)
            IF ( (xcn(inode, jcompo)*apsv(jsolid)*apsv_factor(jsolid))
     &          > 1.0D+00) THEN
#ifdef DEBUG
#  ifdef DEBUG_SOLVSED
#    ifdef DEBUG_SOLVSED_ITERAT_OUTOFRANGE
              WRITE(jp_stddbg, '(4X, "At iteration ", I0, "(",I0, "), '
     &          //'xc(",I0,") > rho at node ", I0, ": ")', ADVANCE="NO")
     &                              ii, newton_i, jcompo, inode
              WRITE(jp_stddbg, *) xcn(inode,jcompo), ' = ',
     &                            xcn_0(inode, jcompo), ' - ',
     &                            deltac(jcompo, inode) / newton_factor
#    endif
#  endif
#endif
              xcn(inode, jcompo) =
     &          1.0D+00/(apsv(jsolid)*apsv_factor(jsolid))
            ENDIF
          ENDDO

        ENDDO


        ! Update xwtotn and xutotn to match wfflx and xcn
        CALL PROFIW(wfflx, xcn, xzdn, xwtotn, aw_bplus)
        CALL PROFIU(xwtotn, aw_bplus, xutotn)

                                    ! Check if ycn does not have to be
                                    ! re-evaluated at this stage.
                                    ! This is necessary when the initial
                                    ! xwtot profile had aw_bplus >= 0
                                    ! and the updated trial xwtotn now
                                    ! has aw_bplus < 0.
        IF (aw_bplus < 0.0D+00) THEN
          IF (isign_aw_bplus /= -1) THEN
            isign_aw_bplus = -1
            CALL FVUPWIND_TRANLAY_RADIODECAY(datime, yco, ycn)
          ENDIF
        ELSE
          isign_aw_bplus =  1
          ycn(:) = yco(:)
        ENDIF


                                    ! Evaluate Equation System
                                    ! for the new xcn estimate
        CALL FVUPWIND_EQUATIONS(atimen, datime, xzdn, xzdv,
     &    wconc, wfflx,
     &    xco, yco, xcn, ycn, xwtotn, xutotn,
     &    jf_svc,
     &    eqn_syst)

        sigma_eqn2_tst = SUM((eqn_syst/eqn_scale)**2)

#ifdef DEBUG_SOLVSED
#  ifdef DEBUG_SOLVSED_ITERAT_NEWTONI
        WRITE(jp_stddbg, '(3X, I4, 1X, I4)', ADVANCE ="NO") ii, newton_i
        WRITE(jp_stddbg, '(3X)', ADVANCE="NO")
        WRITE(jp_stddbg, *) sigma_eqn2_tst, aw_bplus,
     &                      MAXVAL(ABS(deltac))/newton_factor
#  endif
#endif

        IF ((newton_i == jp_newton_test_min) .OR.
     &     (sigma_eqn2_tst < sigma_eqn2_best)) THEN
          sigma_eqn2_best = sigma_eqn2_tst
                 xcn_best =    xcn
              xwtotn_best = xwtotn
            aw_bplus_best = aw_bplus
              xutotn_best = xutotn
            newton_i_best = newton_i
              deltac_best = -deltac / newton_factor
        ENDIF

        IF (newton_i > jp_newton_test_minmax) THEN
          IF (sigma_eqn2_tst > sigma_eqn2_best) EXIT
        ENDIF

      ENDDO

               xcn =    xcn_best
            xwtotn = xwtotn_best
          aw_bplus = aw_bplus_best
            xutotn = xutotn_best
        sigma_eqn2 = sigma_eqn2_best

#ifdef DEBUG
#  ifdef DEBUG_SOLVSED
      CALL FVUPWIND_EQUATIONS(atimen, datime, xzdn, xzdv,
     &  wconc, wfflx,
     &  xco, yco, xcn, ycn, xwtotn, xutotn,
     &  jf_svc,
     &  eqn_syst)

#    ifdef DEBUG_SOLVSED_EQN_SUMMARY
      WRITE(jp_trace_eqn_summary,*) '   Best EQN_SYST (Sigma^2 =',
     &                              SUM((eqn_syst/eqn_scale)**2),'):',
     &                              ii, newton_i_best
#    endif
#    ifdef DEBUG_SOLVSED_EQN_DETAIL
      WRITE(jp_trace_eqn_detail,*)  '   Best EQN_SYST (Sigma^2 =',
     &                              SUM((eqn_syst/eqn_scale)**2),'):',
     &                              ii, newton_i_best
      DO inode = idnw, idnt-2, jp_trace_step
        WRITE(jp_trace_eqn_detail,cs_fmt9200)
     &                              eqn_syst(:,inode)
      ENDDO
      IF (idnw < idnt) THEN
        inode = idnt-1
        WRITE(jp_trace_eqn_detail,cs_fmt9200)
     &                              eqn_syst(:,inode), xwtotn(inode)
      ENDIF
      DO inode = idnt, idnb, jp_trace_step
        WRITE(jp_trace_eqn_detail,cs_fmt9200)
     &                              eqn_syst(:,inode), xwtotn(inode)
      ENDDO
      WRITE(jp_trace_eqn_detail,*)  '   Best Delta_C'
      DO inode = idnw, idnt-1
        WRITE(jp_trace_eqn_detail,cs_fmt9200)
     &                              deltac_best(:,inode)
      ENDDO
      DO inode = idnt, idnb, jp_trace_step
        WRITE(jp_trace_eqn_detail,cs_fmt9200)
     &                              deltac_best(:,inode)
      ENDDO
      WRITE(jp_trace_eqn_detail,*)  '   Best C'
      DO inode = idnw, idnt-1
        WRITE(jp_trace_eqn_detail,cs_fmt9200)
     &                              xcn_best(inode, :)
      ENDDO
      DO inode = idnt, idnb, jp_trace_step
        WRITE(jp_trace_eqn_detail,cs_fmt9200)
     &                              xcn_best(inode, :)
      ENDDO
      WRITE(jp_trace_eqn_detail,'()')
#    endif
#  endif
#endif

      IF (sigma_eqn2 > dp_sigma_eqn2_threshold) THEN
        niter_conv_sigmaeqn = 0
        IF (ii < jp_itermax) THEN
          GOTO 1
        ELSE
          itsflag = 10  ! Max number of iterations exceeded
        ENDIF
      ELSE
        IF (niter_conv_sigmaeqn == 0) niter_conv_sigmaeqn = ii
        IF (ANY(ABS(deltac_best(:,:))
     &          > dp_delta_var_threshold*xc_scale(:,:))) THEN
          IF (ii < jp_itermax) GOTO 1 ! Try to improve
        ELSE
          iflag_conv_delta = 0
        ENDIF
        itsflag = 0    ! convergence reached
      ENDIF

! Finish up with the various boundary charcteristics
! * Bottom solute concentrations

      bconc = xcn(idnb,jc_to_io(:))

! * Top fluxes of solutes
!   =====================
!   These should normally be equal to
!      wcflx = xphi(idnw)*(dcf_molion(:)/xtor2(idnw))
!     &                       *dx_xcn(idnw,jc_to_io(:))
!   if no DBL is included (note that idnt==idnw in this case), and
!      wcflx = dcf_molion(:)*dx_xcn(idnw,jc_to_io(:))
!   if such a DBL is included
!
!   Here, we calculate them by mass balance considerations
!   dC/dt + (F_{1/2}-F_{0})/(xv_{1/2} - xv_{0}) - R - Fnonloc = 0
!   => F_0 = F_{1/2} + (xv_{1/2} - xv_{0}) * (dC/dt - R - Fnonloc
!   where C = (1-thetatop)*C_0 + thetatop*C_1
!   and   R = (1-thetatop)*R_0 + thetatop*R_1
!   and   Fnonloc = (1-thetatop)*Fnonloc_0 + thetatop*Fnonloc_1
!   and thetatop in [0; 1/2]

!   The above equation is not completely solved below. The interconversion
!   reaction rates have not been considered. The calculated "fluxes"
!   are thus actually equal to the true fluxes reduced by the
!   respective sums of the interconversion rate reactions.
!   These latter cancel out when total fluxes (of DIC, ALK, ...)
!   are considered rather than single species fluxes.

                                    ! F_{1/2} calculation

      CALL TRANSPORT
     &  (atimen, xzdn, idnw, xcn, xwtotn, xutotn, wconc, wfflx, ycn,
     &   flxadvdif)


      IF (ndn_w2s == 0) THEN
                                    ! If there is no DBL included in the
                                    ! model configuration, we use the
                                    ! thetatop based approximation
                                    ! at the topmost half-cell.

        dt_xc(:) =                 !  - dC/dt calculation
     &      (1.0D+00-thetatop)
     &      *(xcn(idnt  ,:)-xco(idnt  ,:))*(xphi(idnt  )*one_datime)
     &    + thetatop
     &      *(xcn(idntp1,:)-xco(idntp1,:))*(xphi(idntp1)*one_datime)

                                    !  - R, via ...
        azdn  = xzdn(idnt)
        aphi  = xphi(idnt)
        ac(:) =  xcn(idnt,:)
        CALL REACRATE(jp_realm_reaclay, azdn, aphi, ac, reaction_na) ! ... R_0 = R(idnt)

        azdn  = xzdn(idntp1)
        aphi  = xphi(idntp1)
        ac(:) =  xcn(idntp1,:)
        CALL REACRATE(jp_realm_reaclay, azdn, aphi, ac, reaction_nb) ! ... R_1 = R(idnt+1)

                                    !  - Fnonloc, via ...
        CALL NONLOCAL_TRANSPORT(jp_realm_reaclay, xzdn,   idnt,      ! ... Fnonloc_0
     &                              xcn, wconc, flxnonloc_na)        !     = Fnonloc(idnt)

        CALL NONLOCAL_TRANSPORT(jp_realm_reaclay, xzdn, idntp1,      ! ... Fnonloc_1
     &                              xcn, wconc, flxnonloc_nb)        !     = Fnonloc(idnt+1)

        reaction(:) =
     &    (1.0D+00-thetatop)*reaction_na(:) + thetatop*reaction_nb(:)

        flxnonloc(:) =
     &    (1.0D+00-thetatop)*flxnonloc_na(:) + thetatop*flxnonloc_nb(:)

      ELSE
                                    ! If there is a DBL included in the
                                    ! current model configuration, we
                                    ! force thetatop = 0 as things become
                                    ! unmanagable else.

                                    !  - dC/dt calculation
        dt_xc(:) = (xcn(idnw  ,:)-xco(idnw  ,:))*one_datime

        azdn  = xzdn(idnw)          !  - R
        aphi  = 1.0D+00
        ac(:) =  xcn(idnw,:)
        CALL REACRATE(jp_realm_difblay, azdn, aphi, ac, reaction) 

                                    !  - Fnonloc
        CALL NONLOCAL_TRANSPORT(jp_realm_difblay, xzdn,   idnw,
     &                              xcn, wconc, flxnonloc)

      ENDIF


      wcflx(:) =                    ! to get a first estimate of wcflx,
     &  flxadvdif(jc_to_io(:))
     &  + (xzdv(idvw+1) - xzdv(idvw))
     &    * ( dt_xc(jc_to_io(:))
     &        - reaction(jc_to_io(:))
     &        - flxnonloc(jc_to_io(:)))

                                    ! This estimate still needs to be
                                    ! corrected for the implicitly
                                    ! contained interconversion rate
                                    ! terms.
      CALL SPECIATE_WCFLX(atimen, xzdv, xcn, xutotn, wconc, wcflx, info)
      IF (info /= 0) THEN
                                    ! SPECIATE_WCFLX has not been able
                                    ! to carry out the correction
        itsflag = -1                ! Issue warning!
      ENDIF

                                    ! Finally, we add the total
                                    ! non-local (bioirrigation) flux
                                    ! to the top solute fluxes
      CALL NONLOCAL_TRANSPORT_TOTALS(xzdn, xcn, wconc, flxnonloc)

      wcflx_bi(:) = flxnonloc(jc_to_io(:))
      wcflx(:)    = wcflx(:) + wcflx_bi(:)

! * Bottom unburial (<0) and burial (>0) fluxes of solids

      CALL TRANSPORT
     &  (atimen, xzdn, idnb, xcn, xwtotn, xutotn, wconc, wfflx, ycn,
     &   flxadvdif)


      bfflx(:) = flxadvdif(jf_to_io(:))

! And finally, also calculate ysolidn
! This may have to be changed as datime * flxadvdif(jf_to_io(:))
! might exceed ysolido(:)
! Solvsed_onestep now takes this into account

      IF (aw_bplus < 0.0D+00) THEN

        IF (datime > 0.0D+00) THEN  ! Transient case
          ysolidn_vol =
     &      ysolido_vol + aw_bplus
     &                    * (1.0D+00 - yphi)/(1.0D+00 - xphi(idnb))
     &                    * datime
          ysolidn(:) = ycn(:)*ysolidn_vol

        ELSE                        ! Steady-state case

                                    ! This should actually not happen
                                    ! as aw_bplus > 0 in a steady-state
                                    ! sediment. Leave some meaningful
                                    ! value, though, but issue warning
          ysolidn(:) = ycn(:)*da_gpd_dcorelay

#ifdef DEBUG
          IF (itsflag == 0) THEN
            WRITE(jp_stddbg, '(3X, A)')
     &               'Warning: aw_bplus < 0 encountered at steady state'
          ENDIF
#endif

        ENDIF

      ELSE

        CALL FVUPWIND_TRANLAY_RADIODECAY(datime, ysolido, ysolidn,
     &                                   bfflx)
      ENDIF


! Process LOST_COMPO definitions
      CALL solvsedLossConversions(bfflx, ysolidn)


#ifdef DEBUG
#  ifdef DEBUG_SOLVSED
      SELECT CASE(itsflag)
      CASE (0)                      ! Convergence reached
        IF (iflag_conv_delta == 0) THEN
          WRITE(jp_stddbg, '(4X, "Convergence (Sigma2 + Delta) after ",'
     &                  // 'I0, " (", I0, "+", I0, ") iterations")')
     &                              ii, niter_conv_sigmaeqn,
     &                              ii - niter_conv_sigmaeqn
        ELSE
          WRITE(jp_stddbg, '(4X, "Convergence (Sigma2 only) after ",'
     &                  // 'I0, " (", I0, ") iterations")')
     &                              ii, niter_conv_sigmaeqn
        ENDIF

      CASE (10)                     ! maximum number of iterations reached
        WRITE(jp_stddbg, '(4X, "Aborting after ", I0, " iterations")')
     &                              ii
      CASE (-1)                      ! Convergence reached
        IF (iflag_conv_delta == 0) THEN
          WRITE(jp_stddbg, '(4X, "Convergence (Sigma2 + Delta) after ",'
     &                  // 'I0, " (", I0, "+", I0, ") iterations")')
     &                              ii, niter_conv_sigmaeqn,
     &                              ii - niter_conv_sigmaeqn
        ELSE
          WRITE(jp_stddbg, '(4X, "Convergence (Sigma2 only) after  ",'
     &                  // 'I0, " (", I0, ") iterations")')
     &                              ii, niter_conv_sigmaeqn
        ENDIF
        WRITE(jp_stddbg, '(4X, "Warning: SPECIATE_WCFLX failed.")')
      END SELECT

      WRITE(jp_stddbg, '(4X, "Sigma Eqn^2 (FVUpwind): ")', ADVANCE="NO")
      WRITE(jp_stddbg, *) sigma_eqn2
      WRITE(jp_stddbg, '(4X, "Max of last correction: ")', ADVANCE="NO")
      WRITE(jp_stddbg, *) MAXVAL(ABS(deltac_best(:,:))/xc_scale(:,:))
#  endif
#endif


      RETURN


      CONTAINS

#include <solvsedLossConversions.F>


!-----------------------------------------------------------------------
      END SUBROUTINE ImplicitTimeStep
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE FVUPWIND_EQUATIONS(atimen, datime, xzdn, xzdv,
     &  wconc, wfflx,
     &  xco, yco, xcn, ycn, xwtotn, xutotn,
     &  jf_svc,
     &  eqn_syst)
!-----------------------------------------------------------------------

      USE mod_defines_medusa
      USE mod_fvupwind_params
      USE mod_indexparam,      ONLY: nsolut, nsolid, ncompo,
     &                               jf_to_io, jc_to_io,
     &                               jmf_to_io, jmf_to_if
      USE mod_gridparam,       ONLY: ndn_w2s,
     &                               idnw, idnt, idnb,
     &                               idvw, idvs, idvb,
     &                               thetatop, thetabot
      USE mod_milieucharas,    ONLY: xphi
      USE mod_materialcharas
      USE mod_transport

      USE mod_rreac,           ONLY: REACRATE


      IMPLICIT NONE


      DOUBLE PRECISION                       :: atimen, datime
      DOUBLE PRECISION, DIMENSION(idnw:idnb) :: xzdn
      DOUBLE PRECISION, DIMENSION(idvw:idvb) :: xzdv
      DOUBLE PRECISION, DIMENSION(1:nsolut)  :: wconc
      DOUBLE PRECISION, DIMENSION(1:nsolid)  :: wfflx
      DOUBLE PRECISION, DIMENSION(idnw:idnb, ncompo) :: xco
      DOUBLE PRECISION, DIMENSION(nsolid)    :: yco, ycn
      DOUBLE PRECISION, DIMENSION(idnw:idnb, ncompo) :: xcn
      DOUBLE PRECISION, DIMENSION(idvs:idvb) :: xwtotn
      DOUBLE PRECISION, DIMENSION(idvw:idvb) :: xutotn
      INTEGER, DIMENSION(idnt:idnb)          :: jf_svc

      DOUBLE PRECISION, DIMENSION(neqns_node,idnw:idnb) :: eqn_syst

      INTENT(IN) :: atimen, datime, xzdn, xzdv, wconc, wfflx, xco,
     &              jf_svc
      INTENT(INOUT) :: xcn, xwtotn, xutotn, eqn_syst


      INTEGER, PARAMETER :: idntp1 = idnt + 1
      INTEGER, PARAMETER :: idnbm1 = idnb - 1
      INTEGER :: inode
      INTEGER :: ivrtx_a, ivrtx_b

      INTEGER :: ieqn, io_svc

      INTEGER :: jsolid, jsolut, jcompo

      DOUBLE PRECISION :: one_datime

      DOUBLE PRECISION, DIMENSION(ncompo) :: ac
      DOUBLE PRECISION :: azdn, aphi


                                    ! *_ni -> node i
                                    ! *_na -> node i-1   (node above)
                                    ! *_nb -> node i+1   (node below)
                                    ! *_va -> "node" i-1/2 (vertex above)
                                    ! *_vb -> "node" i+1/2 (vertex below)
      DOUBLE PRECISION, DIMENSION(ncompo) :: flxadvdif_va,
     &                                       flxadvdif_vb
      DOUBLE PRECISION, DIMENSION(ncompo) :: flxnonloc_na,
     &                                       flxnonloc_nb,
     &                                       flxnonloc_ni
      DOUBLE PRECISION, DIMENSION(ncompo) :: reaction_na,
     &                                       reaction_nb,
     &                                       reaction_ni

      DOUBLE PRECISION, DIMENSION(ncompo) :: phix_datime_na,
     &                                       phix_datime_nb,
     &                                       phix_datime_ni

      DOUBLE PRECISION                    :: dx_ni


                                    ! Time control
                                    ! ============
      IF (datime > 0.0D+00) THEN
        one_datime = 1.0D+00/datime
      ELSE
        one_datime = 0.0D+00
      ENDIF

      xcn(idnw, jc_to_io(:)) = wconc(:)

      IF (ndn_w2s /= 0) THEN        ! If there is a DBL ...

                                    ! Equations at node idnw
                                    ! ==================================
                                    ! Solutes: xcn - wconc = 0

        eqn_syst(jc_to_io(:), idnw) = xcn(idnw, jc_to_io(:)) - wconc(:)

                                    ! Solids: xcn = 0

        eqn_syst(jf_to_io(:), idnw) = xcn(idnw, jf_to_io(:))


                                    ! Flux across the vertex below node
                                    ! idnw (will be required for the
                                    ! equations of the next steps)
                                    ! (-> flxadvdif_vb)
        CALL TRANSPORT
     &    (atimen, xzdn, idnw, xcn, xwtotn, xutotn, wconc, wfflx, ycn,
     &     flxadvdif_vb)


                                    ! Prepare next equation block
        flxadvdif_va(:) = flxadvdif_vb(:)


                                    ! Equations at nodes idnw+1, ..., idnt-1
                                    ! ==================================

        DO inode = idnw + 1, idnt - 1

                                    ! Solutes: normal ADR equations for the DBL

          CALL TRANSPORT            ! - flux across the vertex below node 'inode'
     &      (atimen, xzdn, inode, xcn, xwtotn, xutotn, wconc, wfflx,
     &       ycn, flxadvdif_vb)


          azdn  = xzdn(inode)       ! - reaction at node 'inode' (-> reaction_ni)
          aphi  = 1.0D+00
          ac(:) =  xcn(inode,:)
          CALL REACRATE(jp_realm_difblay, azdn, aphi, ac, reaction_ni)


          ivrtx_b = inode           ! - spacing
          ivrtx_a = ivrtx_b - 1
          dx_ni = xzdv(ivrtx_b) - xzdv(ivrtx_a)

                                    ! - equation
          eqn_syst(jc_to_io(:), inode) =
     &        (xcn(inode, jc_to_io(:)) - xco(inode, jc_to_io(:)))
     &        * one_datime
     &      + (flxadvdif_vb(jc_to_io(:)) - flxadvdif_va(jc_to_io(:)))
     &        / dx_ni
     &      - reaction_ni(jc_to_io(:))


                                    ! Solids: xcn = 0

          eqn_syst(jf_to_io(:), inode) = xcn(inode, jf_to_io(:))


                                    ! Prepare next equation block
          flxadvdif_va(:) = flxadvdif_vb(:)

        ENDDO


                                    ! Equations at node 'idnt' (inside
                                    ! the topmost REACLAY  cell)
                                    ! ==================================

                                    ! Solutes: normal ADR-NLT equations

        CALL TRANSPORT              ! - flux across the vertex below node 'idnt'
     &    (atimen, xzdn, idnt, xcn, xwtotn, xutotn, wconc, wfflx,
     &     ycn, flxadvdif_vb)

                                    ! - non local flux at idnt (-> flxnonloc_ni)
        CALL NONLOCAL_TRANSPORT
     &    (jp_realm_reaclay, xzdn,   idnt, xcn, wconc, flxnonloc_ni)


        azdn  = xzdn(idnt)          ! - reaction at idnt (-> reaction_ni)
        aphi  = xphi(idnt)
        ac(:) =  xcn(idnt,:)
        CALL REACRATE(jp_realm_reaclay, azdn, aphi, ac, reaction_ni)


                                    ! - \phi^{x}/\Delta t for node 'idnt'
        phix_datime_ni(jf_to_io(:)) = (1.0D+00-aphi) * one_datime
        phix_datime_ni(jc_to_io(:)) =      aphi      * one_datime

        ivrtx_b = idnt              ! - spacing
        ivrtx_a = ivrtx_b - 1
        dx_ni = xzdv(ivrtx_b) - xzdv(ivrtx_a)

                                    ! - equation
        eqn_syst(:, idnt) =
     &      (xcn(idnt,:) - xco(idnt,:)) * phix_datime_ni(:)
     &    + (flxadvdif_vb(:) - flxadvdif_va(:)) / dx_ni
     &    - reaction_ni(:) - flxnonloc_ni(:)


      ELSE
                                    ! If there is no DBL ...
                                    ! ... there is only a half cell at the
                                    ! top of the REACLAY
                                    ! ==================================

                                    ! Solutes: xcn - wconc = 0

        eqn_syst(jc_to_io(:), idnw) = xcn(idnw, jc_to_io(:)) - wconc(:)


                                    ! Solids: normal equations with
                                    ! thetatop weighting

        CALL TRANSPORT              ! - flux across the vertex below the node 'idnt'
     &    (atimen, xzdn, idnt, xcn, xwtotn, xutotn, wconc, wfflx, ycn,
     &     flxadvdif_vb)


        CALL NONLOCAL_TRANSPORT     ! - non local flux at idnt^+ (-> flxnonloc_na)
     &    (jp_realm_reaclay, xzdn,   idnt, xcn, wconc, flxnonloc_na)

        CALL NONLOCAL_TRANSPORT     ! - non local flux at idnt+1 (-> flxnonloc_nb)
     &    (jp_realm_reaclay, xzdn, idntp1, xcn, wconc, flxnonloc_nb)

        flxnonloc_ni(:) =
     &    (1.0D+00-thetatop) * flxnonloc_na(:)
     &  +      thetatop      * flxnonloc_nb(:)


        azdn  = xzdn(idnt)          ! - reaction at node <idnt>
        aphi  = xphi(idnt)          !   (-> reaction_na)
        ac(:) = xcn(idnt,:)
        CALL REACRATE(jp_realm_reaclay, azdn, aphi, ac, reaction_na)

        azdn  = xzdn(idntp1)        ! - reaction at node <idnt+1>
        aphi  = xphi(idntp1)        !   (-> reaction_nb)
        ac(:) = xcn(idntp1,:)
        CALL REACRATE(jp_realm_reaclay, azdn, aphi, ac, reaction_nb)

        reaction_ni(:) =
     &    (1.0D+00-thetatop) * reaction_na(:)
     &  +      thetatop      * reaction_nb(:)


                                    ! - $\phi^{x}/\Delta t$ for nodes
                                    !   <idnt> and <idnt+1>
                                    !   (only required for solids here)
        phix_datime_na(jf_to_io(:)) = (1.0D+00-xphi(idnt  ))*one_datime
        phix_datime_nb(jf_to_io(:)) = (1.0D+00-xphi(idntp1))*one_datime


        ivrtx_b = idvs + 1          ! - spacing
        dx_ni = xzdv(ivrtx_b) - xzdv(idvs)

                                    ! - equations at node <idnt>
        eqn_syst(jf_to_io(:), idnt) =
     &      (1.0D+00-thetatop)
     &      * (xcn(idnt  , jf_to_io(:)) - xco(idnt  , jf_to_io(:)))
     &        * phix_datime_na(jf_to_io(:))
     &    + thetatop
     &      * (xcn(idntp1, jf_to_io(:)) - xco(idntp1, jf_to_io(:)))
     &        * phix_datime_nb(jf_to_io(:))
     &    + (flxadvdif_vb(jf_to_io(:)) - wfflx(:)) / dx_ni
     &    - reaction_ni(jf_to_io(:)) - flxnonloc_ni(jf_to_io(:))


      ENDIF


#ifndef SOLVSED_NO_SVC_EQUATION
                                    ! Static volume conservation equation
                                    ! for node 'idnt' (replaces the
                                    ! equation of the most abundant solid)
                                    ! if more than one material solid
                                    ! component is considered in the model
      IF (nmatsolid > 1) THEN
        io_svc = jf_to_io(jf_svc(idnt))
        eqn_syst(io_svc, idnt) =
     &    SUM( xcn(idnt, jmf_to_io(:))
     &        *apsv(jmf_to_if(:))
     &        *apsv_factor(jmf_to_if(:)))
     &    - 1.0D+00
      ENDIF
#endif

                                    ! Prepare for next equation block
      flxadvdif_va(:) = flxadvdif_vb(:)


                                    ! Equations for nodes idnt+1, ..., idnb-1
                                    ! =====================================

      DO inode = idnt + 1, idnb - 1

        CALL TRANSPORT              ! - flux across the vertex below the node <inode>
     &    (atimen, xzdn, inode, xcn, xwtotn, xutotn, wconc, wfflx, ycn,
     &     flxadvdif_vb)

        CALL NONLOCAL_TRANSPORT     ! - non local flux at node <inode>
     &    (jp_realm_reaclay, xzdn, inode, xcn, wconc, flxnonloc_ni)


        azdn  = xzdn(inode)         ! - reaction at node <inode>
        aphi  = xphi(inode)
        ac(:) =  xcn(inode,:)
        CALL REACRATE(jp_realm_reaclay, azdn, aphi, ac, reaction_ni)

                                    ! - $\phi^{x}/\Delta t$ for node <iinode>
        phix_datime_ni(jf_to_io(:)) = (1.0D+00-aphi) * one_datime
        phix_datime_ni(jc_to_io(:)) =      aphi      * one_datime


        ivrtx_b = inode             ! - spacing
        ivrtx_a = ivrtx_b - 1
        dx_ni = xzdv(ivrtx_b) - xzdv(ivrtx_a)

                                    ! - equations
        eqn_syst(:,inode) =
     &    (xcn(inode,:) - xco(inode,:))*phix_datime_ni(:)
     &      + (flxadvdif_vb(:) - flxadvdif_va(:)) / dx_ni
     &      - reaction_ni(:) - flxnonloc_ni(:)

#ifndef SOLVSED_NO_SVC_EQUATION
                                    ! Static volume conservation equation
        IF (nmatsolid > 1) THEN
          io_svc = jf_to_io(jf_svc(inode))
          eqn_syst(io_svc, inode)
     &      = SUM( xcn(inode,jmf_to_io(:))
     &            *apsv(jmf_to_if(:))
     &            *apsv_factor(jmf_to_if(:)))
     &        - 1.0D+00
        ENDIF
#endif

                                    ! Prepare processing of next inode
        flxadvdif_va(:) = flxadvdif_vb(:)

      ENDDO

                                    ! Finally the equations for node idnb
                                    ! ===================================

                                    ! - flux across the (virtual) vertex
      CALL TRANSPORT                !   at node <idnb>
     &  (atimen, xzdn, idnb, xcn, xwtotn, xutotn, wconc, wfflx, ycn,
     &   flxadvdif_vb)


      CALL NONLOCAL_TRANSPORT       ! - non local flux at node <idnb-1>
     &    (jp_realm_reaclay, xzdn, idnbm1, xcn, wconc, flxnonloc_na)

      CALL NONLOCAL_TRANSPORT       ! - non local flux at node <idnb>
     &    (jp_realm_reaclay, xzdn, idnb,   xcn, wconc, flxnonloc_nb)

      flxnonloc_ni(:) =
     &           thetabot     * flxnonloc_na(:)
     &  +  (1.0D+00-thetabot) * flxnonloc_nb(:)


      azdn  = xzdn(idnbm1)          ! - reaction at node <idnb-1>
      aphi  = xphi(idnbm1)          !   (-> reaction_na)
      ac(:) =  xcn(idnbm1,:)
      CALL REACRATE(jp_realm_reaclay, azdn, aphi, ac, reaction_na)

      azdn  = xzdn(idnb)            ! - reaction at node <idnb>
      aphi  = xphi(idnb)            !   (-> reaction_nb)
      ac(:) =  xcn(idnb,:)
      CALL REACRATE(jp_realm_reaclay, azdn, aphi, ac, reaction_nb)

      reaction_ni(:) =
     &           thetabot     * reaction_na(:)
     &  +  (1.0D+00-thetabot) * reaction_nb(:)


      phix_datime_na(jf_to_io(:)) = (1.0D+00-xphi(idnbm1)) * one_datime
      phix_datime_na(jc_to_io(:)) =      xphi(idnbm1)      * one_datime
      phix_datime_nb(jf_to_io(:)) = (1.0D+00-xphi(idnb  )) * one_datime
      phix_datime_nb(jc_to_io(:)) =      xphi(idnb  )      * one_datime


      ivrtx_a = idvb - 1            ! - spacing
      dx_ni = xzdv(idvb) - xzdv(ivrtx_a)


                                    ! - equations
      eqn_syst(:,idnb) =
     &    thetabot
     &      * (xcn(idnbm1,:) - xco(idnbm1,:)) * phix_datime_na(:)
     &  + (1.0D+00-thetabot)
     &      * (xcn(idnb  ,:) - xco(idnb  ,:)) * phix_datime_nb(:)
     &  + (flxadvdif_vb(:) - flxadvdif_va(:)) / dx_ni
     &  - reaction_ni(:) - flxnonloc_ni(:)


#ifndef SOLVSED_NO_SVC_EQUATION
                                    ! Static volume conservation equation
      IF (nmatsolid > 1) THEN
        io_svc = jf_to_io(jf_svc(idnb))
        eqn_syst(io_svc, idnb) =
     &    SUM( xcn(idnb,jmf_to_io(:))
     &        *apsv(jmf_to_if(:))
     &        *apsv_factor(jmf_to_if(:))) - 1.0D+00
      ENDIF
#endif
                                    ! Finally, substitute DAEs for
                                    ! solutes' equations where equilibria
                                    ! have to be taken into account.
      CALL EQN_FLUX2DAE(xcn, eqn_syst, neqns_node)


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE FVUPWIND_EQUATIONS
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE FVUPWIND_JACOBIAN(atimen, datime, xzdn, xzdv,
     &  wconc, wfflx,
     &  xco, yco, xcn, ycn, xwtotn, xutotn,
     &  jf_svc, jcb_syst)
!-----------------------------------------------------------------------

      USE mod_defines_medusa
      USE mod_transport
      USE mod_fvupwind_params
      USE mod_indexparam,      ONLY: nsolut, nsolid, ncompo,
     &                               jf_to_io, jc_to_io,
     &                               jmf_to_io, jmf_to_if
      USE mod_gridparam,       ONLY: ndn_w2s,
     &                               idnw, idnt, idnb,
     &                               idvw, idvs, idvb,
     &                               thetatop, thetabot
      USE mod_milieucharas,    ONLY: xphi
      USE mod_materialcharas
      USE mod_rreac,           ONLY: DREACRATE


      IMPLICIT NONE


      DOUBLE PRECISION ::                               atimen,
     &                                                  datime
      DOUBLE PRECISION, DIMENSION(idnw:idnb) ::         xzdn
      DOUBLE PRECISION, DIMENSION(idvw:idvb) ::         xzdv
      DOUBLE PRECISION, DIMENSION(1:nsolut) ::          wconc
      DOUBLE PRECISION, DIMENSION(1:nsolid) ::          wfflx
      DOUBLE PRECISION, DIMENSION(idnw:idnb, ncompo) :: xco, xcn
      DOUBLE PRECISION, DIMENSION(nsolid) ::            yco, ycn
      DOUBLE PRECISION, DIMENSION(idvs:idvb) ::         xwtotn
      DOUBLE PRECISION, DIMENSION(idvw:idvb) ::         xutotn
      INTEGER, DIMENSION(idnt:idnb) ::                  jf_svc
      DOUBLE PRECISION, DIMENSION(neqns_node,
     &  nvars_node, idnw:idnb, -1:1) ::                 jcb_syst

      INTENT(IN)    :: atimen, datime, xzdn, xzdv, wconc, wfflx, xco,
     &                 yco, ycn, jf_svc
      INTENT(INOUT) :: xcn, xwtotn, xutotn
      INTENT(OUT)   :: jcb_syst


      INTEGER, PARAMETER :: idntp1 = idnt + 1
      INTEGER, PARAMETER :: idnbm1 = idnb - 1
      INTEGER :: inode
      INTEGER :: ivrtx_a, ivrtx_b

      INTEGER :: ieqn, iequ, jvar, iequ0, io_svc

      INTEGER :: jsolid, jsolut, jcompo


      DOUBLE PRECISION, DIMENSION(ncompo) :: jcb_svc
      DOUBLE PRECISION, DIMENSION(ncompo) :: jcb_dsvc

      DOUBLE PRECISION                    :: one_datime

      DOUBLE PRECISION, DIMENSION(ncompo) :: ac
      DOUBLE PRECISION                    :: azdn, aphi

                                    !     *_ni -> node i
                                    !     *_na -> node i-1   (node above)
                                    !     *_nb -> node i+1   (node below)
                                    !     *_va -> "node" i-1/2 (vertex above)
                                    !     *_vb -> "node" i+1/2 (vertex below)
      DOUBLE PRECISION, DIMENSION(ncompo, -1:1)   :: d_flxadvdif_va,
     &                                               d_flxadvdif_vb
      DOUBLE PRECISION, DIMENSION(ncompo)         :: dxc_flxnonloc_na,
     &                                               dxc_flxnonloc_nb,
     &                                               dxc_flxnonloc_ni
      DOUBLE PRECISION, DIMENSION(ncompo, ncompo) :: dxc_reaction_na,
     &                                               dxc_reaction_nb,
     &                                               dxc_reaction_ni

      DOUBLE PRECISION, DIMENSION(ncompo)         :: phix_datime_na,
     &                                               phix_datime_nb,
     &                                               phix_datime_ni

      DOUBLE PRECISION                            :: dx_ni


                                    ! Start with a zero Jacobian
      jcb_syst(:,:,:,:) = 0.0D+00

                                    ! Time control
                                    ! ============
      IF (datime > 0.0D+00) THEN
        one_datime = 1.0D+00/datime
      ELSE
        one_datime = 0.0D+00
      ENDIF



      IF (ndn_w2s /= 0) THEN        ! If there is a DBL ...

                                    ! Equations at node idnw
                                    ! ==================================
                                    ! Solutes: xcn - wconc = 0

        ! FVUPWIND_EQUATIONS equation:
        !  eqn_syst(jc_to_io(:), idnw) = xcn(idnw, jc_to_io(:)) - wconc(:)

        DO jsolut = 1, nsolut
          jcompo = jc_to_io(jsolut)
          ! Components in jacobian w/r to concentrations at
          ! current node (<inode>, idx_dim4=0)
          jcb_syst(jcompo, jcompo, idnw, 0) = 1.0D+00
        ENDDO


                                    ! Solids: xcn = 0
        ! FVUPWIND_EQUATIONS equation:
        !  eqn_syst(jf_to_io(:), idnw) = xcn(idnw, jf_to_io(:))
        DO jsolid = 1, nsolid
          jcompo = jf_to_io(jsolid)
          ! Components in jacobian w/r to concentrations at
          ! current node (<inode>, idx_dim4=0)
          jcb_syst(jcompo, jcompo, idnw, 0) = 1.0D+00
        ENDDO


                                    ! Flux across the vertex below node
                                    ! idnw (will be required for the
                                    ! equations of the next steps)
                                    ! Derivative -> d_flxadvdif_vb
        ! FVUPWIND_EQUATIONS instruction:
        !  CALL TRANSPORT
        !  &  (atimen, xzdn, idnw, xcn, xwtotn, xutotn, wconc, wfflx, ycn,
        !  &   flxadvdif_vb)
        CALL DERIVFORM_TRANSPORT
     &    (atimen, xzdn, idnw, xcn, xwtotn, xutotn, d_flxadvdif_vb)

                                    ! Prepare d_flxadvdif_va values for next inode.
                                    ! d_flxadvdif_va has been calculated for
                                    ! inode = cn + 1/2. Hence, currently:
                                    !   d_flxadvdif_vb(:,-1) -> cn-1 (= 0 here)
                                    !   d_flxadvdif_vb(:, 0) -> cn
                                    !   d_flxadvdif_vb(:, 1) -> cn+1
                                    ! Now, flxadvdif_vb at cn  is equal to
                                    ! flxadvdif_va at cn+1. However, the reference
                                    ! node changes from cn to cn+1.
                                    ! For inode = cn+1, we will therefore need
                                    !   d_flxadvdif_va(:,-1) -> cn
                                    !   d_flxadvdif_va(:, 0) -> cn+1
                                    !   d_flxadvdif_va(:, 1) -> cn+2 (= 0 here)
                                    ! Hence:
                                    !   d_flxadvdif_va(:,-1) := d_flxadvdif_vb(:,0)
                                    !   d_flxadvdif_va(:, 0) := d_flxadvdif_vb(:,1)
                                    !   d_flxadvdif_va(:, 1) := 0.0D+00

        ! FVUPWIND_EQUATIONS instruction:
        !  flxadvdif_va(:) = flxadvdif_vb(:)
        d_flxadvdif_va(:, -1) = d_flxadvdif_vb(:, 0)
        d_flxadvdif_va(:,  0) = d_flxadvdif_vb(:, 1)
        d_flxadvdif_va(:,  1) = 0.0D+00


                                    ! Equations at nodes idnw+1, ..., idnt-1
                                    ! ==================================

        DO inode = idnw + 1, idnt - 1

                                    ! Solutes: normal ADR equations for the DBL

          ! FVUPWIND_EQUATIONS instruction:
          !  CALL TRANSPORT
          ! & (atimen, xzdn, inode, xcn, xwtotn, xutotn, wconc, wfflx,
          ! &  ycn, flxadvdif_vb)
          CALL DERIVFORM_TRANSPORT
     &      (atimen, xzdn, inode, xcn, xwtotn, xutotn, d_flxadvdif_vb)


                                    ! - derivatives of reaction at inode
                                    !   (-> dxc_reaction_ni)
          ! FVUPWIND_EQUATIONS instructions:
          !  azdn  = xzdn(inode)
          !  aphi  = 1.0D+00
          !  ac(:) =  xcn(inode,:)
          !  CALL REACRATE(jp_realm_difblay, azdn, aphi, ac, reaction_ni)
          azdn  = xzdn(inode)
          aphi  = 1.0D+00
          ac(:) =  xcn(inode,:)
          CALL DREACRATE(jp_realm_difblay, azdn, aphi, ac,
     &      dxc_reaction_ni)


          ivrtx_b = inode           ! - spacing
          ivrtx_a = ivrtx_b - 1
          dx_ni = xzdv(ivrtx_b) - xzdv(ivrtx_a)

          ! FVUPWIND_EQUATIONS equation:
          !  eqn_syst(jc_to_io(:), inode) =
          ! &   (xcn(inode, jc_to_io(:)) - xco(inode, jc_to_io(:)))
          ! &   * one_datime
          ! & + (flxadvdif_vb(jc_to_io(:)) - flxadvdif_va(jc_to_io(:)))
          ! &   / dx_ni
          ! & - reaction_ni(jc_to_io(:))

                                    ! Preset jacobian values for concentrations
                                    ! at current node (dim4=0) with reaction
                                    ! rate related derivative values
          jcb_syst(jc_to_io(:), :, inode,  0) =
     &      -dxc_reaction_ni(jc_to_io(:), :)

          DO jsolut = 1, nsolut

            jcompo = jc_to_io(jsolut)

                                    ! Components in jacobian w/r to
                                    ! concentrations at the node
                                    ! above the current one
            jcb_syst(jcompo, jcompo, inode, -1) =
     &        (d_flxadvdif_vb(jcompo, -1) - d_flxadvdif_va(jcompo, -1))
     &        / dx_ni

                                    ! Components in jacobian w/r to
                                    ! concentrations at the current node
            jcb_syst(jcompo, jcompo, inode,  0) =
     &        jcb_syst(jcompo, jcompo, inode,  0)
     &        + one_datime
     &        + (d_flxadvdif_vb(jcompo, 0) - d_flxadvdif_va(jcompo, 0))
     &          / dx_ni

                                    ! Components in jacobian w/r to
                                    ! concentrations at the node below
                                    ! the current one
            jcb_syst(jcompo, jcompo, inode,  1) =
     &        (d_flxadvdif_vb(jcompo, 1) - d_flxadvdif_va(jcompo, 1))
     &        / dx_ni

          ENDDO

                                    ! Solids: xcn = 0

          ! FVUPWIND_EQUATIONS equation:
          !  eqn_syst(jf_to_io(:), inode) = xcn(inode, jf_to_io(:))
          DO jsolid = 1, nsolid
            jcompo = jf_to_io(jsolid)
            ! Components in jacobian w/r to concentrations at
            ! current node (<inode>, idx_dim4=0)
            jcb_syst(jcompo, jcompo, inode, 0) = 1.0D+00
          ENDDO


                                    ! Prepare teh next equation block
          ! FVUPWIND_EQUATIONS instruction:
          !  flxadvdif_va(:) = flxadvdif_vb(:)
          d_flxadvdif_va(:, -1) = d_flxadvdif_vb(:, 0)
          d_flxadvdif_va(:,  0) = d_flxadvdif_vb(:, 1)
          d_flxadvdif_va(:,  1) = 0.0D+00


        ENDDO


                                    ! Equations at node 'idnt' (inside
                                    ! the topmost REACLAY  cell)
                                    ! ==================================

                                    ! Solutes: normal ADR-NLT equations
        ! FVUPWIND_EQUATIONS instruction:
        !  CALL TRANSPORT
        ! &    (atimen, xzdn, idnt, xcn, xwtotn, xutotn, wconc, wfflx,
        ! &     ycn, flxadvdif_vb)
        CALL DERIVFORM_TRANSPORT
     &    (atimen, xzdn, idnt, xcn, xwtotn, xutotn, d_flxadvdif_vb)


        ! FVUPWIND_EQUATIONS instruction:
        !  CALL NONLOCAL_TRANSPORT
        ! &  (jp_realm_reaclay, xzdn,   idnt, xcn, wconc, flxnonloc_ni)
        CALL DERIVFORM_NONLOCAL_TRANSPORT
     &    (jp_realm_reaclay, xzdn,  idnt, xcn, dxc_flxnonloc_ni)


        ! FVUPWIND_EQUATIONS instructions:
        !  azdn  = xzdn(idnt)       ! - reaction at idnt (-> reaction_ni)
        !  aphi  = xphi(idnt)
        !  ac(:) =  xcn(idnt,:)
        !  CALL REACRATE(jp_realm_reaclay, azdn, aphi, ac, reaction_ni)
        azdn  = xzdn(idnt)
        aphi  = xphi(idnt)
        ac(:) =  xcn(idnt,:)
        CALL DREACRATE(jp_realm_reaclay, azdn, aphi, ac,
     &    dxc_reaction_ni)

                                    ! - \phi^{x}/\Delta t for node 'idnt'
        phix_datime_ni(jf_to_io(:)) = (1.0D+00-aphi) * one_datime
        phix_datime_ni(jc_to_io(:)) =      aphi      * one_datime

        ivrtx_b = idnt              ! - spacing
        ivrtx_a = ivrtx_b - 1
        dx_ni = xzdv(ivrtx_b) - xzdv(ivrtx_a)

        ! FVUPWIND_EQUATIONS equation:
        !  eqn_syst(:, idnt) =
        ! & (xcn(idnt,:) - xco(idnt,:) * phix_datime_ni(:)
        ! & + (flxadvdif_vb(:) - flxadvdif_va(:)) / dx_ni
        ! & - reaction_ni(:) - flxnonloc_ni(:)

                                    ! Preset jacobian values for concentrations
                                    ! at current node (dim4=0) with reaction
                                    ! rate related derivative values
        jcb_syst(:, :, idnt, 0) = -dxc_reaction_ni(:,:)


        DO jcompo = 1, ncompo

                                    ! Components in jacobian w/r to
                                    ! concentrations at the node
                                    ! above the current one
          jcb_syst(jcompo, jcompo, inode, -1) =
     &      (d_flxadvdif_vb(jcompo, -1) - d_flxadvdif_va(jcompo, -1))
     &      / dx_ni

                                    ! Components in jacobian w/r to
                                    ! concentrations at the current node
          jcb_syst(jcompo, jcompo, inode,  0) =
     &      jcb_syst(jcompo, jcompo, inode, 0)
     &      + phix_datime_ni(jcompo)
     &      + (d_flxadvdif_vb(jcompo, 0) - d_flxadvdif_va(jcompo, 0))
     &        / dx_ni
     &      - dxc_flxnonloc_ni(jcompo)

                                    ! Components in jacobian w/r to
                                    ! concentrations at the node
                                    ! below the current one
          jcb_syst(jcompo, jcompo, inode, 1) =
     &      (d_flxadvdif_vb(jcompo, 1) - d_flxadvdif_va(jcompo, 1))
     &      / dx_ni

        ENDDO


      ELSE
                                    ! If there is no DBL, then there is
                                    ! only a half cell at the top of
                                    ! the REACLAY
                                    ! ==================================

                                    ! Solutes: xcn - wconc = 0
        ! FVUPWIND_EQUATIONS equation:
        !  eqn_syst(jc_to_io(:), idnw) = xcn(idnw, jc_to_io(:)) - wconc(:)

        DO jsolut = 1, nsolut
          jcompo = jc_to_io(jsolut)
                                    ! Components in jacobian w/r to
                                    ! concentrations at the current node
                                    ! (<inode>, idx_dim4=0)
          jcb_syst(jcompo, jcompo, idnw, 0) = 1.0D+00
        ENDDO

                                    ! Solids: normal equations with
                                    ! thetatop weighting
        ! FVUPWIND_EQUATIONS instruction:
        !  CALL TRANSPORT
        ! &    (atimen, xzdn, idnt, xcn, xwtotn, xutotn, wconc, wfflx, ycn,
        ! &     flxadvdif_vb)
        CALL DERIVFORM_TRANSPORT
     &    (atimen, xzdn, idnt, xcn, xwtotn, xutotn, d_flxadvdif_vb)


        ! FVUPWIND_EQUATIONS instruction:
        !  CALL NONLOCAL_TRANSPORT
        ! & (jp_realm_reaclay, xzdn, idnt, xcn, wconc, flxnonloc_na)
        CALL DERIVFORM_NONLOCAL_TRANSPORT
     &    (jp_realm_reaclay, xzdn,   idnt, xcn, dxc_flxnonloc_na)

        ! FVUPWIND_EQUATIONS instruction:
        !  CALL NONLOCAL_TRANSPORT
        ! & (jp_realm_reaclay, xzdn, idntp1, xcn, wconc, flxnonloc_nb)
        CALL DERIVFORM_NONLOCAL_TRANSPORT
     &    (jp_realm_reaclay, xzdn, idntp1, xcn, dxc_flxnonloc_nb)


        ! FVUPWIND_EQUATIONS instructions:
        !  azdn  = xzdn(idnt)
        !  aphi  = xphi(idnt)
        !  ac(:) = xcn(idnt,:)
        !  CALL REACRATE(jp_realm_reaclay, azdn, aphi, ac, reaction_na)
        azdn  = xzdn(idnt)
        aphi  = xphi(idnt)
        ac(:) = xcn(idnt,:)
        CALL DREACRATE(jp_realm_reaclay, azdn, aphi, ac,
     &    dxc_reaction_na)

        ! FVUPWIND_EQUATIONS instructions:
        !  azdn  = xzdn(idntp1)
        !  aphi  = xphi(idntp1)
        !  ac(:) = xcn(idntp1,:)
        !  CALL REACRATE(jp_realm_reaclay, azdn, aphi, ac, reaction_nb)
        azdn  = xzdn(idntp1)
        aphi  = xphi(idntp1)
        ac(:) = xcn(idntp1,:)
        CALL DREACRATE(jp_realm_reaclay, azdn, aphi, ac,
     &    dxc_reaction_nb)

                                    ! - $\phi^{x}/\Delta t$ for nodes
                                    !   <idnt> and <idnt+1>
                                    !   (only required for solids here)
        phix_datime_na(jf_to_io(:)) = (1.0D+00-xphi(idnt  ))*one_datime
        phix_datime_nb(jf_to_io(:)) = (1.0D+00-xphi(idntp1))*one_datime


        ivrtx_b = idvs + 1          ! - spacing
        dx_ni = xzdv(ivrtx_b) - xzdv(idvs)

                                    ! - equations at node <idnt>
        ! FVUPWIND_EQUATIONS equations:
        !  eqn_syst(jf_to_io(:), idnt) =
        ! &   (1.0D+00-thetatop)
        ! &   * (xcn(idnt  , jf_to_io(:)) - xco(idnt  , jf_to_io(:)))
        ! &   * phix_datime_na(jf_to_io(:))
        ! & + thetatop
        ! &   * (xcn(idntp1, jf_to_io(:)) - xco(idntp1, jf_to_io(:)))
        ! &   * phix_datime_nb(jf_to_io(:))
        ! & + (flxadvdif_vb(jf_to_io(:)) - wfflx(:)) / dx_ni
        ! & - reaction_ni(jf_to_io(:)) - flxnonloc_ni(jf_to_io(:))
        !
        ! with
        !  reaction_ni(:) =
        ! &    (1.0D+00-thetatop) * reaction_na(:)
        ! &  +      thetatop      * reaction_nb(:)
        !  flxnonloc_ni(:) =
        ! &    (1.0D+00-thetatop) * flxnonloc_na(:)
        ! &  +      thetatop      * flxnonloc_nb(:)
        !
        ! where
        !  reaction_na(:)  = reaction rate at idnt^+ (this node)
        !  reaction_nb(:)  = reaction rate at idnt+1 (node + 1)
        !  flxnonloc_na(:) = non local flux rate at idnt^+ (this node)
        !  flxnonloc_nb(:) = non local flux rate at idnt+1

                                    ! Components in jacobian w/r to
                                    ! concentrations at node idnt
                                    ! --> jcb_syst(:,:, idnt, 0)
        jcb_syst(jf_to_io(:), :, idnt, 0)
     &    =  -(1.0D+00-thetatop)*dxc_reaction_na(jf_to_io(:),:)

                                    ! Components in jacobian w/r to
                                    ! concentrations at node below idnt
                                    ! --> jcb_syst(:,:, idnt, 1)
        jcb_syst(jf_to_io(:), :, idnt, 1)
     &    =  -thetatop*dxc_reaction_nb(jf_to_io(:),:)

        DO jsolid = 1, nsolid

          jcompo = jf_to_io(jsolid)

          jcb_syst(jcompo, jcompo, idnt,  0)
     &      =   jcb_syst(jcompo, jcompo, idnt,  0)
     &        + (1.0D+00-thetatop) * phix_datime_na(jcompo)
     &        + d_flxadvdif_vb(jcompo, 0) / dx_ni
     &        - (1.0D+00-thetatop) * dxc_flxnonloc_na(jcompo)

          jcb_syst(jcompo, jcompo, idnt, 1)
     &      =   jcb_syst(jcompo, jcompo, idnt,  1)
     &        + thetatop * phix_datime_nb(jcompo)
     &        + d_flxadvdif_vb(jcompo, 1) / dx_ni
     &        - thetatop * dxc_flxnonloc_nb(jcompo)

        ENDDO


      ENDIF

#ifndef SOLVSED_NO_SVC_EQUATION
                                    ! Static volume conservation equation
                                    ! to replace the equation for the most
                                    ! abundant solid at node idnt
      IF (nmatsolid > 1) THEN
        ! FVUPWIND_EQUATIONS equations:
        !  io_svc = jf_to_io(jf_svc(idnt))
        !  eqn_syst(io_svc, idnt)
        !  &  = SUM(xcn(idnt, jf_to_io(:))*apsv(:)*apsv_factor(:)) - 1.0D+00
        io_svc = jf_to_io(jf_svc(idnt))
        jcb_syst(io_svc,           : , idnt, :) = 0.0D+00
        jcb_syst(io_svc, jmf_to_io(:), idnt, 0) =
     &    apsv(jmf_to_if(:)) * apsv_factor(jmf_to_if(:))
      ENDIF
#endif

                                    ! Done with all the equations from idnw to idnt
                                    ! (If there is no DBL, there is of course only
                                    ! one such equation!)


                                    ! Prepare for next equation block
      ! FVUPWIND_EQUATIONS instruction:
      !  flxadvdif_va(:) = flxadvdif_vb(:)
      d_flxadvdif_va(:, -1) = d_flxadvdif_vb(:, 0)
      d_flxadvdif_va(:,  0) = d_flxadvdif_vb(:, 1)
      d_flxadvdif_va(:,  1) = 0.0D+00


                                    ! Equations for nodes idnt+1, ..., idnb-1
                                    ! =====================================

      DO inode = idnt + 1, idnb - 1

        ! FVUPWIND_EQUATIONS instruction:
        !  CALL TRANSPORT
        ! &  (atimen, xzdn, inode, xcn, xwtotn, xutotn, wconc, wfflx, ycn,
        ! &   flxadvdif_vb)
        CALL DERIVFORM_TRANSPORT
     &    (atimen, xzdn, inode, xcn, xwtotn, xutotn, d_flxadvdif_vb)

        ! FVUPWIND_EQUATIONS instruction:
        !  CALL NONLOCAL_TRANSPORT
        !  &  (jp_realm_reaclay, xzdn, inode, xcn, wconc, flxnonloc_ni)
        CALL DERIVFORM_NONLOCAL_TRANSPORT
     &    (jp_realm_reaclay, xzdn, inode, xcn, dxc_flxnonloc_ni)


        ! FVUPWIND_EQUATIONS instructions:
        !  azdn  = xzdn(inode)
        !  aphi  = xphi(inode)
        !  ac(:) =  xcn(inode,:)
        !  CALL REACRATE(jp_realm_reaclay, azdn, aphi, ac, reaction_ni)
        azdn  = xzdn(inode)
        aphi  = xphi(inode)
        ac(:) =  xcn(inode,:)
        CALL DREACRATE(jp_realm_reaclay, azdn, aphi, ac,
     &                              dxc_reaction_ni)

                                    ! - $\phi^{x}/\Delta t$ for node <iinode>
        phix_datime_ni(jf_to_io(:)) = (1.0D+00-aphi) * one_datime
        phix_datime_ni(jc_to_io(:)) =      aphi      * one_datime


        ivrtx_b = inode             ! - spacing
        ivrtx_a = ivrtx_b - 1
        dx_ni = xzdv(ivrtx_b) - xzdv(ivrtx_a)


        ! FVUPWIND_EQUATIONS equations:
        !  eqn_syst(:,inode) =
        ! &   (xcn(inode,:) - xco(inode,:))*phix_datime_ni(:)
        ! & + (flxadvdif_vb(:) - flxadvdif_va(:)) / dx_ni
        ! & - reaction_ni(:) - flxnonloc_ni(:)


        jcb_syst(:, :, inode, 0) = -dxc_reaction_ni(:,:)


        DO jcompo = 1, ncompo
                                    ! Components in jacobian w/r to
                                    ! concentrations at the inode
                                    ! above the current one
          jcb_syst(jcompo, jcompo, inode, -1) =
     &      (d_flxadvdif_vb(jcompo, -1) - d_flxadvdif_va(jcompo, -1))
     &      / dx_ni

                                    ! Components in jacobian w/r to
                                    ! concentrations at the current inode
          jcb_syst(jcompo, jcompo, inode,  0) =
     &      jcb_syst(jcompo, jcompo, inode, 0)
     &      + phix_datime_ni(jcompo)
     &      + (d_flxadvdif_vb(jcompo, 0) - d_flxadvdif_va(jcompo, 0))
     &        / dx_ni
     &      - dxc_flxnonloc_ni(jcompo)

                                    ! Components in jacobian w/r to
                                    ! concentrations at the inode
                                    ! below the current one
          jcb_syst(jcompo, jcompo, inode, 1) =
     &      (d_flxadvdif_vb(jcompo, 1) - d_flxadvdif_va(jcompo, 1))
     &      / dx_ni

        ENDDO

#ifndef SOLVSED_NO_SVC_EQUATION
                                    ! Static volume conservation equation
                                    ! replaces the equation for the most
                                    ! abundant solid
        IF (nmatsolid > 1) THEN
          ! FVUPWIND_EQUATIONS equations:
          !  io_svc = jf_to_io(jf_svc(inode))
          !  eqn_syst(io_svc, inode)
          !  & = SUM(xcn(inode,jf_to_io(:))*apsv(:)*apsv_factor(:)) - 1.0D+00
          io_svc = jf_to_io(jf_svc(inode))
          jcb_syst(io_svc,           : , inode, :) = 0.0D+00
          jcb_syst(io_svc, jmf_to_io(:), inode, 0) =
     &      apsv(jmf_to_if(:)) * apsv_factor(jmf_to_if(:))
        ENDIF
#endif

                                    ! Prepare next equation block
        ! FVUPWIND_EQUATIONS instruction:
        !  flxadvdif_va(:) = flxadvdif_vb(:)
        d_flxadvdif_va(:, -1) = d_flxadvdif_vb(:, 0)
        d_flxadvdif_va(:,  0) = d_flxadvdif_vb(:, 1)
        d_flxadvdif_va(:,  1) = 0.0D+00

      ENDDO

                                    ! Finally the equations for node idnb
                                    ! ===================================
      ! FVUPWIND_EQUATIONS instruction:
      !  CALL TRANSPORT
      !  &  (atimen, xzdn, idnb, xcn, xwtotn, xutotn, wconc, wfflx, ycn,
      !  &   flxadvdif_vb)
      CALL DERIVFORM_TRANSPORT
     &  (atimen, xzdn, idnb, xcn, xwtotn, xutotn, d_flxadvdif_vb)

      ! FVUPWIND_EQUATIONS instruction:
      !  CALL NONLOCAL_TRANSPORT
      !  &  (jp_realm_reaclay, xzdn, idnbm1, xcn, wconc, flxnonloc_na)
      CALL DERIVFORM_NONLOCAL_TRANSPORT
     &  (jp_realm_reaclay, xzdn, idnbm1, xcn, dxc_flxnonloc_na)

      ! FVUPWIND_EQUATIONS instruction:
      !  CALL NONLOCAL_TRANSPORT
      !  &  (jp_realm_reaclay, xzdn,   idnb, xcn, wconc, flxnonloc_nb)
      CALL DERIVFORM_NONLOCAL_TRANSPORT
     &  (jp_realm_reaclay, xzdn, idnb, xcn, dxc_flxnonloc_nb)

      ! FVUPWIND_EQUATIONS instructions:
      !  azdn  = xzdn(idnbm1)
      !  aphi  = xphi(idnbm1)
      !  ac(:) =  xcn(idnbm1,:)
      !  CALL REACRATE(jp_realm_reaclay, azdn, aphi, ac, reaction_na)
      azdn  = xzdn(idnbm1)
      aphi  = xphi(idnbm1)
      ac(:) =  xcn(idnbm1,:)
      CALL DREACRATE(jp_realm_reaclay, azdn, aphi, ac,
     &                              dxc_reaction_na)

      ! FVUPWIND_EQUATIONS instructions:
      !  azdn  = xzdn(idnb)
      !  aphi  = xphi(idnb)
      !  ac(:) =  xcn(idnb,:)
      !  CALL REACRATE(jp_realm_reaclay, azdn, aphi, ac, reaction_nb)
      azdn  = xzdn(idnb)
      aphi  = xphi(idnb)
      ac(:) =  xcn(idnb,:)
      CALL DREACRATE(jp_realm_reaclay, azdn, aphi, ac,
     &                              dxc_reaction_nb)


      phix_datime_na(jf_to_io(:)) = (1.0D+00-xphi(idnbm1)) * one_datime
      phix_datime_na(jc_to_io(:)) =      xphi(idnbm1)      * one_datime
      phix_datime_nb(jf_to_io(:)) = (1.0D+00-xphi(idnb  )) * one_datime
      phix_datime_nb(jc_to_io(:)) =      xphi(idnb  )      * one_datime


      ivrtx_a = idvb - 1            ! - spacing
      dx_ni = xzdv(idvb) - xzdv(ivrtx_a)

      ! FVUPWIND_EQUATIONS equations:
      !  eqn_syst(:,idnb) =
      ! &   thetabot
      ! &    *(xcn(idnbm1,:) - xco(idnbm1,:)) * phix_datime_na(:)
      ! & + (1.0D+00-thetabot)
      ! &    * (xcn(idnb ,:) - xco(idnb  ,:)) * phix_datime_nb(:)
      ! & + (flxadvdif_vb(:) - flxadvdif_va(:)) / dx_ni
      ! & - reaction_ni(:) - flxnonloc_ni(:)
      !
      ! where
      !  reaction_ni(:) =
      ! &  thetabot*reaction_na(:) + (1.0D+00-thetabot)*reaction_nb(:)
      !  flxnonloc_ni(:) =
      ! &  thetabot*flxnonloc_na(:) + (1.0D+00-thetabot)*flxnonloc_nb(:)

                                    ! Components in jacobian w/r to
                                    ! concentrations at idnb-1
      jcb_syst(:, :, idnb, -1)      ! --> jcb_syst(:,:, idnb, -1)
     &  =       -thetabot     *dxc_reaction_na(:,:)

                                    ! Components in jacobian w/r to
                                    ! concentrations at idnb
      jcb_syst(:, :, idnb, 0)       ! --> jcb_syst(:,:, idnt, 0)
     &  =  -(1.0D+00-thetabot)*dxc_reaction_nb(:,:)


      DO jcompo = 1, ncompo

        jcb_syst(jcompo, jcompo, idnb, -1) =
     &    jcb_syst(jcompo, jcompo, idnb, -1)
     &    + thetabot * phix_datime_na(jcompo)
     &    + (d_flxadvdif_vb(jcompo, -1) - d_flxadvdif_va(jcompo, -1))
     &      / dx_ni
     &    - thetabot * dxc_flxnonloc_na(jcompo)

        jcb_syst(jcompo, jcompo, idnb, 0) =
     &    jcb_syst(jcompo, jcompo, idnb, 0)
     &    + (1.0D+00-thetabot) * phix_datime_nb(jcompo)
     &    + (d_flxadvdif_vb(jcompo,  0) - d_flxadvdif_va(jcompo,  0))
     &      / dx_ni
     &    - (1.0D+00-thetabot) * dxc_flxnonloc_nb(jcompo)

      ENDDO

#ifndef SOLVSED_NO_SVC_EQUATION
                                    ! Static volume conservation equation
      IF (nmatsolid > 1) THEN
        ! FVUPWIND_EQUATIONS equations:
        !  io_svc = jf_to_io(jf_svc(idnb))
        !  eqn_syst(io_svc, idnb) =
        ! &  SUM(xcn(idnb,jf_to_io(:))*apsv(:)*apsv_factor(:)) - 1.0D+00
        io_svc = jf_to_io(jf_svc(idnb))
        jcb_syst(io_svc,           : , idnb, :) = 0.0D+00
        jcb_syst(io_svc, jmf_to_io(:), idnb, 0) =
     &    apsv(jmf_to_if(:)) * apsv_factor(jmf_to_if(:))
      ENDIF
#endif

                                    ! Finally, substitute DAEs for
                                    ! solutes' equations where equilibria
                                    ! have to be taken into account.
      CALL JAC_FLUX2DAE(xcn, jcb_syst, neqns_node, nvars_node)


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE FVUPWIND_JACOBIAN
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE FVUPWIND_TRANLAY_RADIODECAY(datime, yo, yn, bfflx)
!-----------------------------------------------------------------------

      ! This subroutines tracks the radioactive decay in the transit
      ! buffer layer (TRANLAY).
      !
      ! The dummy arguments yo and yn stand for different physical
      ! entities, depending on whether the optional bfflx is present
      ! during the call or not:
      ! - if bfflx *is* present, the subroutine is called at the end of
      !   a time step with yo = ysolido and yn = ysolidn (total contents);
      ! - if bfflx *is not* present, the subroutine is called within
      !   an xwtot refinement iteration with yo = yco and yn = ycn
      !   (concentrations); in this case aw_bplus < 0 as well.


      USE MOD_FVUPWIND_PARAMS,      ONLY: l_ylambda_initialized,
     &                                    FVUPWIND_INIT_YLAMBDA,
     &                                    ylambda
      USE MOD_GRIDPARAM,            ONLY: da_gpd_dcorelay
      USE MOD_INDEXPARAM,           ONLY: nsolid, jf_mud
      USE MOD_MILIEUCHARAS,         ONLY: yphi
      USE MOD_MATERIALCHARAS,       ONLY: apsv, apsv_factor
      USE MOD_PROCESSDATA,          ONLY: nproc_tranlay
      USE MOD_GAUSS


      IMPLICIT NONE


      DOUBLE PRECISION,                    INTENT(IN)  :: datime
      DOUBLE PRECISION, DIMENSION(nsolid), INTENT(IN)  :: yo
      DOUBLE PRECISION, DIMENSION(nsolid), INTENT(OUT) :: yn
      DOUBLE PRECISION, DIMENSION(nsolid), INTENT(IN)  :: bfflx

      OPTIONAL :: bfflx

      DOUBLE PRECISION, DIMENSION(nsolid, nsolid) :: WorkA
      DOUBLE PRECISION, DIMENSION(nsolid)         :: WorkX
      INTEGER                                     :: info

      INTEGER          :: j
      DOUBLE PRECISION :: one_datime        ! auxiliary variable to hold 1/datime

      DOUBLE PRECISION :: aw_bplus, diag_val


      IF (nproc_tranlay > 0) THEN
        IF (.NOT. l_ylambda_initialized) CALL FVUPWIND_INIT_YLAMBDA
      ENDIF


      IF (datime > 0.0D+00) THEN
                                    ! Transient simulation
                                    ! ====================
        IF (nproc_tranlay > 0) THEN

          one_datime = 1.0D+00/datime

          WorkA(:,:) = -ylambda(:,:)

          DO j = 1, nsolid
            WorkA(j,j) = one_datime - ylambda(j,j)
          ENDDO

          IF (PRESENT(bfflx)) THEN
            WorkX(:) = bfflx(:) + yo(:)*one_datime
          ELSE
            WorkX(:) = yo(:)*one_datime
          ENDIF

          CALL GM_GESV(WorkA, WorkX, INFO=info)
          yn(:) = WorkX(:)

        ELSE

          IF (PRESENT(bfflx)) THEN
            yn(:) = bfflx(:)*datime + yo(:)
          ELSE
            yn(:) = yo(:)
          ENDIF

        ENDIF

      ELSE
                                    ! Steady-state calculation
                                    ! ========================
        IF (PRESENT(bfflx)) THEN

          aw_bplus = SUM(bfflx(:)*apsv(:)*apsv_factor(:))

          IF (nproc_tranlay > 0) THEN

            WorkA(:,:) = -ylambda(:,:)
            diag_val = aw_bplus/(da_gpd_dcorelay * (1.0D+00-yphi))

            DO j = 1, nsolid
              WorkA(j,j) = diag_val - ylambda(j,j)
            ENDDO

            WorkX(:) = bfflx(:)

            CALL GM_GESV(WorkA, WorkX, INFO=info)
            yn(:) = WorkX(:)

          ELSE

            yn(:) = bfflx(:) * da_gpd_dcorelay/aw_bplus * (1.0D+00-yphi)

          ENDIF


        ELSE
                                    ! This can only happen during the
                                    ! iterative refinement of xwtot, because
                                    ! FVUPWIND_TRANLAY_RADIODECAY is only
                                    ! called during these without bfflx
                                    ! as an argument if aw_bplus < 0.
                                    ! In steady-state simulations,
                                    ! however, aw_bplus > 0 in the end.
                                    ! Provide pro forma values (pure clay).
            yn(:)      = 0.0D+00
            yn(jf_mud) = 1.0D+00/apsv(jf_mud)

        ENDIF


      ENDIF


      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE FVUPWIND_TRANLAY_RADIODECAY
!-----------------------------------------------------------------------
