!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


!=======================================================================
      MODULE MOD_ZONE_MASS_TOTALS
!=======================================================================

      IMPLICIT NONE

      PRIVATE

      PUBLIC :: ZONE_MASS_TOTALS_ONECOL
      PUBLIC :: ZONE_MASS_TOTALS
      PUBLIC :: REACLAY_TOTALS
      PUBLIC :: ZONE_PROCRATE_TOTALS_ONECOL
      PUBLIC :: ZONE_PROCRATE_TOTALS
                

                                    ! Numbering of the optional
                                    ! arguments in the calls below.
      INTEGER, PARAMETER :: jp_nwhich = 4

      INTEGER, PARAMETER :: jp_xm     = 1
      INTEGER, PARAMETER :: jp_ym     = 2
      INTEGER, PARAMETER :: jp_rea_xm = 3
      INTEGER, PARAMETER :: jp_fnl_xm = 4


      LOGICAL            :: l_call_internal = .FALSE.


      CONTAINS

!-----------------------------------------------------------------------
      SUBROUTINE ZONE_MASS_TOTALS_ONECOL(i_column, l_which_totals,
     &                              onecol_total_xm,
     &                              onecol_total_ym,
     &                              onecol_total_rea_xm,
     &                              onecol_total_fnl_xm)
!-----------------------------------------------------------------------

! Calculates the integrated (total) mass and the integrated reaction
! rate for each consituent, in each of the zones in the model, in
! column i_column.

! The argument <l_which_totals> can only be used when called internally
! i.e., when called within the module (indicated by the PRIVATE
! l_call_internal flag set to .TRUE.)


! Units are as follows:
!
! * for the integrated masses
!   - mol/(m2 of bulk sediment) for solutes
!   - kg/(m2 of bulk sediment) for solids
!
! * for the integrated rates
!   - mol/(m2 of bulk sediment)/yr for solutes
!   - kg/(m2 of bulk sediment)/yr for solids

!--------------
! Modules used
!--------------

      USE mod_execontrol_medusa,    ONLY: ABORT_MEDUSA
      USE mod_defines_medusa,       ONLY: jp_realm_difblay,
     &                                    jp_realm_reaclay,
     &                                    jp_stderr
      USE mod_gridparam,            ONLY: idnw, idnt, idnb,
     &                                    idvw, idvs, idvb,
     &                                    thetatop, thetabot
      USE mod_indexparam,           ONLY: nsolut, nsolid, ncompo,
     &                                    jc_to_io, jf_to_io 
      USE mod_seafloor_wdata
      USE mod_chemicalconsts,       ONLY: SETCCT
      USE mod_processcontrol
      USE mod_rreac
      USE mod_transport,            ONLY: BIRRIC, NONLOCAL_TRANSPORT
      USE mod_seafloor_central,     ONLY: N_COLUMNS_USED, GET_COLUMN,
     &                                    GET_BOUNDARY_CONDS,
     &                                    GET_MATERIALCHARAS
      USE mod_medinterfaces


!-----------------------
! Variable declarations
!-----------------------

      IMPLICIT NONE

!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
! General (global) parameters
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

!     NONE


!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
! Variables in subroutine call arguments
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

      INTEGER                             :: i_column
      LOGICAL, DIMENSION(jp_nwhich)       :: l_which_totals
      DOUBLE PRECISION, DIMENSION(ncompo) :: onecol_total_xm
      DOUBLE PRECISION, DIMENSION(ncompo) :: onecol_total_ym
      DOUBLE PRECISION, DIMENSION(ncompo) :: onecol_total_rea_xm
      DOUBLE PRECISION, DIMENSION(ncompo) :: onecol_total_fnl_xm

      INTENT(IN)  :: i_column, l_which_totals
      INTENT(OUT) :: onecol_total_xm, onecol_total_ym,
     &               onecol_total_rea_xm, onecol_total_fnl_xm

      OPTIONAL    :: l_which_totals
      OPTIONAL    :: onecol_total_xm, onecol_total_ym,
     &               onecol_total_rea_xm, onecol_total_fnl_xm


!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
! Local variables
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

      INTEGER, PARAMETER :: idntp1 = idnt + 1
      INTEGER, PARAMETER :: idnbm1 = idnb - 1

      LOGICAL, DIMENSION(jp_nwhich) :: lloc_which_totals

      INTEGER :: n_grid_seafloor
      INTEGER :: inode
      INTEGER :: jcompo, jsolut, jsolid
      INTEGER :: iflag
      
      DOUBLE PRECISION, DIMENSION(idnw:idnb)         :: xnz
      DOUBLE PRECISION, DIMENSION(idvw:idvb)         :: xvz
      DOUBLE PRECISION, DIMENSION(idnt:idnb)         :: xnphi

      DOUBLE PRECISION, DIMENSION(idnw:idnb, ncompo) :: xc
      DOUBLE PRECISION, DIMENSION(idnw:idnt)         :: xc_phis_dbl
      DOUBLE PRECISION, DIMENSION(idnt:idnb)         :: xc_phis_rcl

      DOUBLE PRECISION, DIMENSION(nsolid)            :: ysolid

      DOUBLE PRECISION, DIMENSION(idnw:idnt)         :: vn_dbl_c
      DOUBLE PRECISION, DIMENSION(idnt:idnb)         :: vn_rcl_c
      DOUBLE PRECISION, DIMENSION(ncompo)            :: area_na
      DOUBLE PRECISION, DIMENSION(ncompo)            :: area_nb
      DOUBLE PRECISION, DIMENSION(ncompo)            :: area_nc
      DOUBLE PRECISION, DIMENSION(ncompo)            :: afnl_na
      DOUBLE PRECISION, DIMENSION(ncompo)            :: afnl_nb
      DOUBLE PRECISION, DIMENSION(ncompo)            :: afnl_nc

      TYPE(WDATA_CONTAINER)                          :: wdata

      DOUBLE PRECISION, DIMENSION(nsolut)            :: wconc
      DOUBLE PRECISION, DIMENSION(nsolid)            :: wfflx
      DOUBLE PRECISION, DIMENSION(ncompo)            :: ac
      DOUBLE PRECISION                               :: azdn, aphi
      DOUBLE PRECISION                               :: onecol_total_w2t
      DOUBLE PRECISION                               :: onecol_total_t2b

!---------------------
! End of declarations
!---------------------



!------------------
! Subroutine Start
!------------------

                                    ! For internal calls, all the
                                    ! checks have been performed and
                                    ! the <l_which_totals> argument
                                    ! is always PRESENT.
      IF (l_call_internal) THEN

        lloc_which_totals(:) = l_which_totals(:)

      ELSE

        IF (PRESENT(l_which_totals)) THEN
        
          WRITE(jp_stderr, '("[ZONE_MASS_TOTALS_ONECOL] error: ", A)')
     &      'the dummy argument <i_which_totals> may only'
          WRITE(jp_stderr, '(A)')
     &      'be used for calls within the module. Aborting!'

          CALL ABORT_MEDUSA()

        ENDIF
        
        lloc_which_totals(jp_xm)     = PRESENT(onecol_total_xm)
        lloc_which_totals(jp_ym)     = PRESENT(onecol_total_ym)
        lloc_which_totals(jp_rea_xm) = PRESENT(onecol_total_rea_xm)
        lloc_which_totals(jp_fnl_xm) = PRESENT(onecol_total_fnl_xm)

      ENDIF



      CALL N_COLUMNS_USED(n_grid_seafloor)

#ifdef ALLOW_MPI
      IF (n_grid_seafloor == 0) THEN

        IF (lloc_which_totals(jp_xm)) onecol_total_xm(:)   = 0.0D+00
        IF (lloc_which_totals(jp_ym)) onecol_total_ym(:)   = 0.0D+00
        IF (lloc_which_totals(jp_rea_xm))
     &                              onecol_total_rea_xm(:) = 0.0D+00
        IF (lloc_which_totals(jp_fnl_xm))
     &                              onecol_total_fnl_xm(:) = 0.0D+00

        RETURN

      ENDIF
#endif

      IF ((i_column >= 1) .AND. (i_column <= n_grid_seafloor)) THEN

                                    ! i_column OK, but nothing to be done
        IF (.NOT. ANY(lloc_which_totals(:))) RETURN

        CALL GET_COLUMN(i_column = i_column, iflag = iflag,
     &                              xzdn = xnz(:), xzdv = xvz(:),
     &                              xphi = xnphi(:),
     &                              xc = xc(:,:), ysolid = ysolid(:))

        IF (idnw == idnt) THEN

          vn_dbl_c(idnt)        = 0.0D+00   ! not to be used

        ELSE

          vn_dbl_c(idnw:idnt-1) = xvz(idvw+1:idvs)- xvz(idvw:idvs-1)
          vn_dbl_c(idnt)        = 0.0D+00   ! not to be used

        ENDIF

        vn_rcl_c(idnt:idnb)     = xvz(idvs+1:idvb)-xvz(idvs:idvb-1)


                                    ! Load characteristic material
                                    ! parameters for material in column
                                    ! <i_column> into MOD_MATERIALCHARAS.
                                    ! REACRATE is critically dependent
                                    ! on this information. It can,
                                    ! however, not load it itself, since
                                    ! it does not know which column it
                                    ! is being called for.

        CALL GET_MATERIALCHARAS(i_column = i_column, iflag = iflag)


        IF (PRESENT(onecol_total_rea_xm)) THEN

          CALL GET_BOUNDARY_CONDS(i_column = i_column, gbcflag = iflag,
     &                              wdata = wdata,
     &                              wconc = wconc(:))
        
                                    ! Set values for chemical constants
                                    ! required for REACRATE.
          CALL SETCCT(wdata, wconc(:))
          CALL SetProcessParameters

          rreac_factor(:) = rreac_factor_max(:)

          IF (idnw /= idnt) THEN
            
            azdn  = xnz(idnw)
            aphi  = 1.0D+00
            ac(:) =  xc(idnw,:)
            CALL REACRATE(jp_realm_difblay, azdn, aphi, ac, area_nc)
  
            onecol_total_rea_xm(:) = vn_dbl_c(idnw)*area_nc(:)

            DO inode = idnw + 1, idnt - 1
              azdn  = xnz(inode)
              aphi  = 1.0D+00
              ac(:) =  xc(inode,:)
              CALL REACRATE(jp_realm_difblay, azdn, aphi, ac, area_nc)
              onecol_total_rea_xm(:) =
     &          onecol_total_rea_xm(:) + vn_dbl_c(inode)*area_nc(:)
            ENDDO

            azdn  =   xnz(idnt)
            aphi  = xnphi(idnt)
            ac(:) =    xc(idnt,:)
            CALL REACRATE(jp_realm_reaclay, azdn, aphi, ac, area_nc)

            onecol_total_rea_xm(:) =
     &        onecol_total_rea_xm(:) + vn_rcl_c(idnt)*area_nc(:)

          ELSE

            azdn  =   xnz(idnt)
            aphi  = xnphi(idnt)
            ac(:) =   xc(idnt,:)
            CALL REACRATE(jp_realm_reaclay, azdn, aphi, ac, area_na)

            azdn  =   xnz(idnt+1)
            aphi  = xnphi(idnt+1)
            ac(:) =    xc(idnt+1,:)
            CALL REACRATE(jp_realm_reaclay, azdn, aphi, ac, area_nb)

            area_nc(:) =  (1.0D+00-thetatop)*area_na(:)
     &                   +     thetatop     *area_nb(:)

            onecol_total_rea_xm(:) = vn_rcl_c(idnt)*area_nc(:)

          ENDIF


          DO inode = idnt + 1, idnb - 1

            azdn  =   xnz(inode)
            aphi  = xnphi(inode)
            ac(:) =    xc(inode,:)
            CALL REACRATE(jp_realm_reaclay, azdn, aphi, ac, area_nc)

            onecol_total_rea_xm(:) =
     &          onecol_total_rea_xm(:) + vn_rcl_c(inode)*area_nc(:)

          ENDDO


          azdn  =   xnz(idnb-1)
          aphi  = xnphi(idnb-1)
          ac(:) =    xc(idnb-1,:)
          CALL REACRATE(jp_realm_reaclay, azdn, aphi, ac, area_na)

          azdn  =   xnz(idnb)
          aphi  = xnphi(idnb)
          ac(:) =    xc(idnb,:)
          CALL REACRATE(jp_realm_reaclay, azdn, aphi, ac, area_nb)

          area_nc(:) =        thetabot     *area_na(:)
     &                 + (1.0D+00-thetabot)*area_nb(:)

          onecol_total_rea_xm(:) =
     &      onecol_total_rea_xm(:) + vn_rcl_c(idnb)*area_nc(:)

        ENDIF


        IF (PRESENT(onecol_total_fnl_xm)) THEN

          CALL GET_BOUNDARY_CONDS(i_column = i_column, gbcflag = iflag,
     &                              wconc = wconc(:),
     &                              wfflx = wfflx(:))
        
                                    ! Set the bioirrigation coefficent
                                    ! values required for NONLOCAL_TRANSPORT.
          CALL BIRRIC(xnz, xc, wconc, wfflx)

          onecol_total_fnl_xm(:) = 0.0D+00

          IF (idnw /= idnt) THEN

            CALL NONLOCAL_TRANSPORT(jp_realm_reaclay, xnz,   idnt,
     &                              xc, wconc, afnl_nc)

            onecol_total_fnl_xm(:) = vn_rcl_c(idnt)*afnl_nc(:)

          ELSE

            CALL NONLOCAL_TRANSPORT(jp_realm_reaclay, xnz,   idnt,
     &                              xc, wconc, afnl_na)

            CALL NONLOCAL_TRANSPORT(jp_realm_reaclay, xnz, idntp1,
     &                              xc, wconc, afnl_nb)

            afnl_nc(:) =  (1.0D+00-thetatop)*afnl_na(:)
     &                   +     thetatop     *afnl_nb(:)

            onecol_total_fnl_xm(:) = vn_rcl_c(idnt)*afnl_nc(:)

          ENDIF


          DO inode = idnt + 1, idnb - 1

            CALL NONLOCAL_TRANSPORT(jp_realm_reaclay, xnz, inode,
     &                              xc, wconc, afnl_nc)

            onecol_total_fnl_xm(:) =
     &          onecol_total_fnl_xm(:) + vn_rcl_c(inode)*afnl_nc(:)

          ENDDO

          CALL NONLOCAL_TRANSPORT(jp_realm_reaclay, xnz, idnbm1,
     &                              xc, wconc, afnl_na)

          CALL NONLOCAL_TRANSPORT(jp_realm_reaclay, xnz, idnb,
     &                              xc, wconc, afnl_nb)

          afnl_nc(:) =        thetabot     *afnl_na(:)
     &                 + (1.0D+00-thetabot)*afnl_nb(:)

          onecol_total_fnl_xm(:) =
     &      onecol_total_fnl_xm(:) + vn_rcl_c(idnb)*afnl_nc(:)

        ENDIF


        DO jsolut = 1, nsolut

          jcompo = jc_to_io(jsolut)

          IF (PRESENT(onecol_total_xm)) THEN

            IF (idnw /= idnt) THEN

              xc_phis_dbl(idnw:idnt-1) = xc(idnw:idnt-1,jcompo)
              xc_phis_dbl(idnt)        = 0.0D+00    ! Not used anyway

              onecol_total_w2t = SUM(vn_dbl_c(:)*xc_phis_dbl(:))

              xc_phis_rcl(idnt) = xc(idnt, jcompo) * xnphi(idnt)

            ELSE

              onecol_total_w2t = 0.0D+00

              xc_phis_rcl(idnt) =
     &          (1.0D+00-thetatop) * xc(idnt  , jcompo) * xnphi(idnt  )
     &        +       thetatop     * xc(idntp1, jcompo) * xnphi(idntp1)

            ENDIF

            xc_phis_rcl(idntp1:idnbm1) =
     &                               xc(idntp1:idnbm1, jcompo)
     &                             * xnphi(idntp1:idnbm1)
            xc_phis_rcl(idnb) =
     &                thetabot     * xc(idnbm1, jcompo) * xnphi(idnbm1)
     &        + (1.0D+00-thetabot) * xc(idnb  , jcompo) * xnphi(idnb  )

            onecol_total_t2b = SUM(vn_rcl_c(:)*xc_phis_rcl(:))

            onecol_total_xm(jcompo) =
     &        onecol_total_w2t + onecol_total_t2b

          ENDIF


          IF (PRESENT(onecol_total_ym)) THEN
            onecol_total_ym(jcompo) = 0.0D+00
          ENDIF

        ENDDO


        DO jsolid = 1, nsolid

          jcompo = jf_to_io(jsolid)

          IF (PRESENT(onecol_total_xm)) THEN

            IF (idnw /= idnt) THEN

              xc_phis_rcl(idnt) =    xc(idnt, jcompo)
     &                             * (1.0D+00-xnphi(idnt))

            ELSE

              xc_phis_rcl(idnt) =
     &          (1.0D+00-thetatop) * xc(idnt  , jcompo)
     &                             * (1.0D+00-xnphi(idnt  ))
     &        +       thetatop     * xc(idntp1, jcompo)
     &                             * (1.0D+00-xnphi(idntp1))
            ENDIF

            xc_phis_rcl(idntp1:idnbm1) =
     &                               xc(  idntp1:idnbm1, jcompo)
     &                             * (1.0D+00-xnphi(idntp1:idnbm1))
            xc_phis_rcl(idnb) =
     &                thetabot     * xc(idnbm1, jcompo)
     &                             * (1.0D+00-xnphi(idnbm1))
     &        + (1.0D+00-thetabot) * xc(idnb  , jcompo)
     &                             * (1.0D+00-xnphi(idnb  ))

            onecol_total_xm(jcompo) = SUM(vn_rcl_c(:)*xc_phis_rcl(:))

          ENDIF


          IF (PRESENT(onecol_total_ym)) THEN
            onecol_total_ym(jcompo) = ysolid(jsolid)
          ENDIF

        ENDDO

      ELSE

        WRITE(jp_stderr, '("[ZONE_MASS_TOTALS_ONECOL] error: ", A)')
     &                              'Illegal column_number!'
        WRITE(jp_stderr, '(" - is equal to ", I0)')
     &                              i_column
        WRITE(jp_stderr, '(" - should be > 0 and <= ", I0)')
     &                              n_grid_seafloor
        WRITE(jp_stderr, '("Aborting!")')

        CALL ABORT_MEDUSA()

      ENDIF


      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE ZONE_MASS_TOTALS_ONECOL
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE ZONE_MASS_TOTALS(total_xm, total_ym,
     &                              total_rea_xm, total_fnl_xm)
!-----------------------------------------------------------------------

! Calculates the integrated (total) mass and the integrated reaction
! rate for each consituent, in each of the zones in the model, in each
! column.

! Units are as follows:
!
! * for the integrated masses
!   - mol/(m2 of bulk sediment) for solutes
!   - kg/(m2 of bulk sediment) for solids
!
! * for the integrated rates
!   - mol/(m2 of bulk sediment)/yr for solutes
!   - kg/(m2 of bulk sediment)/yr for solids

!--------------
! Modules used
!--------------

      USE mod_execontrol_medusa,    ONLY: ABORT_MEDUSA
      USE mod_defines_medusa,       ONLY: jp_stderr
      USE mod_indexparam,           ONLY: ncompo
      USE mod_seafloor_central,     ONLY: N_COLUMNS_USED


!-----------------------
! Variable declarations
!-----------------------

      IMPLICIT NONE

!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
! General (global) parameters
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

!     NONE


!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
! Variables in subroutine call arguments
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

      DOUBLE PRECISION, DIMENSION(:,:) :: total_xm,
     &                                    total_ym
      DOUBLE PRECISION, DIMENSION(:,:) :: total_rea_xm
      DOUBLE PRECISION, DIMENSION(:,:) :: total_fnl_xm


      INTENT(OUT) :: total_xm, total_ym, total_rea_xm, total_fnl_xm
      OPTIONAL    :: total_xm, total_ym, total_rea_xm, total_fnl_xm


!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
! Local variables
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


      LOGICAL, DIMENSION(jp_nwhich) :: l_which_totals

      INTEGER :: i_column, n_grid_seafloor
      INTEGER :: n_errors

      DOUBLE PRECISION, DIMENSION(ncompo) :: onecol_total_xm
      DOUBLE PRECISION, DIMENSION(ncompo) :: onecol_total_ym
      DOUBLE PRECISION, DIMENSION(ncompo) :: onecol_total_rea_xm
      DOUBLE PRECISION, DIMENSION(ncompo) :: onecol_total_fnl_xm



!---------------------
! End of declarations
!---------------------



!------------------
! Subroutine Start
!------------------

      CALL N_COLUMNS_USED(n_grid_seafloor)

      l_which_totals(jp_xm)     = PRESENT(total_xm)
      l_which_totals(jp_ym)     = PRESENT(total_ym)
      l_which_totals(jp_rea_xm) = PRESENT(total_rea_xm)
      l_which_totals(jp_fnl_xm) = PRESENT(total_fnl_xm)

      n_errors = 0
      IF (l_which_totals(jp_xm)) THEN

        IF (     (SIZE(total_xm,1) /= ncompo)
     &      .OR. (SIZE(total_xm,2) /= n_grid_seafloor)) THEN
          WRITE(jp_stderr, '("[ZONE_MASS_TOTALS] error: ", A)')
     &      'incorrect dimensions for argument total_xm:'
          WRITE(jp_stderr, '(" - is (/ ", I0, ", ", I0, " /)")')
     &                              SIZE(total_xm)
          WRITE(jp_stderr, '(" - should be (/ ", I0, ", ", I0, " /)")')
     &                              ncompo, n_grid_seafloor
          n_errors = n_errors + 1
        ENDIF

      ENDIF


      IF (l_which_totals(jp_ym)) THEN

        IF (     (SIZE(total_ym,1) /= ncompo)
     &      .OR. (SIZE(total_ym,2) /= n_grid_seafloor)) THEN
          WRITE(jp_stderr, '("[ZONE_MASS_TOTALS] error: ", A)')
     &      'incorrect dimensions for argument total_ym:'
          WRITE(jp_stderr, '(" - is (/ ", I0, ", ", I0, " /)")')
     &                              SIZE(total_ym)
          WRITE(jp_stderr, '(" - should be (/ ", I0, ", ", I0, " /)")')
     &                              ncompo, n_grid_seafloor
          n_errors = n_errors + 1
        ENDIF

      ENDIF


      IF (l_which_totals(jp_rea_xm)) THEN

        IF (     (SIZE(total_rea_xm,1) /= ncompo)
     &      .OR. (SIZE(total_rea_xm,2) /= n_grid_seafloor)) THEN
          WRITE(jp_stderr, '("[ZONE_MASS_TOTALS] error: ", A)')
     &      'incorrect dimensions for argument total_rea_xm:'
          WRITE(jp_stderr, '(" - is (/ ", I0, ", ", I0, " /)")')
     &                              SIZE(total_rea_xm)
          WRITE(jp_stderr, '(" - should be (/ ", I0, ", ", I0, " /)")')
     &                              ncompo, n_grid_seafloor
          n_errors = n_errors + 1
        ENDIF

      ENDIF


      IF (l_which_totals(jp_fnl_xm)) THEN

        IF (     (SIZE(total_fnl_xm,1) /= ncompo)
     &      .OR. (SIZE(total_fnl_xm,2) /= n_grid_seafloor)) THEN
          WRITE(jp_stderr, '("[ZONE_MASS_TOTALS] error: ", A)')
     &      'incorrect dimensions for argument total_fnl_xm:'
          WRITE(jp_stderr, '(" - is (/ ", I0, ", ", I0, " /)")')
     &                              SIZE(total_fnl_xm)
          WRITE(jp_stderr, '(" - should be (/ ", I0, ", ", I0, " /)")')
     &                              ncompo, n_grid_seafloor
          n_errors = n_errors + 1
        ENDIF

      ENDIF


      IF (n_errors > 0) THEN

        WRITE(jp_stderr, '("[ZONE_MASS_TOTALS] error: ")', ADVANCE="NO")
        WRITE(jp_stderr, '("detected ", I0, " errors in the dummy' //
     &                   ' arguments. Aborting!")')

        CALL ABORT_MEDUSA()

      ENDIF


      l_call_internal = .TRUE.      ! Inform ZONE_MASS_TOTALS_ONECOL
                                    ! that the next call comes from
                                    ! inside the module so that the
                                    ! usage of the <l_which_totals>
                                    ! dummy argument is permitted.

      DO i_column = 1, n_grid_seafloor

        CALL ZONE_MASS_TOTALS_ONECOL(i_column, l_which_totals,
     &                              onecol_total_xm,
     &                              onecol_total_ym,
     &                              onecol_total_rea_xm,
     &                              onecol_total_fnl_xm)

        IF (l_which_totals(jp_xm))
     &              total_xm(:, i_column) = onecol_total_xm(:)
        IF (l_which_totals(jp_ym))
     &              total_ym(:, i_column) = onecol_total_ym(:)
        IF (l_which_totals(jp_rea_xm))
     &              total_rea_xm(:, i_column) = onecol_total_rea_xm(:)
        IF (l_which_totals(jp_fnl_xm))
     &              total_fnl_xm(:, i_column) = onecol_total_fnl_xm(:)

      ENDDO

      l_call_internal = .FALSE.     ! Reset flag.

      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE ZONE_MASS_TOTALS
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE REACLAY_TOTALS(i_column, iflag, tsolids)
!-----------------------------------------------------------------------

! Calculates the integrated (total) mass of solids in REACLAY of in
! column i_column.

! Units for tsolids are [kg/(m2 of bulk sediment)]

!--------------
! Modules used
!--------------

      USE mod_gridparam,            ONLY: idnw, idnt, idnb,
     &                                    idvw, idvs, idvb,
     &                                    thetatop, thetabot
      USE mod_indexparam,           ONLY: nsolid, ncompo, jf_to_io 
      USE mod_seafloor_central,     ONLY: N_COLUMNS_USED, GET_COLUMN


!-----------------------
! Variable declarations
!-----------------------

      IMPLICIT NONE

!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
! General (global) parameters
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

!     NONE


!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
! Variables in subroutine call arguments
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

      INTEGER,                             INTENT(IN)  :: i_column
      INTEGER,                             INTENT(OUT) :: iflag
      DOUBLE PRECISION, DIMENSION(nsolid), INTENT(OUT) :: tsolids


!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
! Local variables
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

      INTEGER, PARAMETER :: idntp1 = idnt + 1
      INTEGER, PARAMETER :: idnbm1 = idnb - 1

      INTEGER :: n_grid_seafloor
      INTEGER :: jcompo, jsolid


      DOUBLE PRECISION, DIMENSION(idvw:idnb)         :: xvz
      DOUBLE PRECISION, DIMENSION(idnt:idnb)         :: xnphi

      DOUBLE PRECISION, DIMENSION(idnw:idnb, ncompo) :: xc
      DOUBLE PRECISION, DIMENSION(idnt:idnb)         :: xc_phis

      DOUBLE PRECISION, DIMENSION(idnt:idnb)         :: xnv


!---------------------
! End of declarations
!---------------------



!------------------
! Subroutine Start
!------------------


      CALL N_COLUMNS_USED(n_grid_seafloor)

#ifdef ALLOW_MPI
      IF (n_grid_seafloor == 0) THEN
        tsolids(:) = 0.0D+00
        iflag = 0
        RETURN
      ENDIF
#endif


      IF ((i_column >= 1) .AND. (i_column <= n_grid_seafloor)) THEN

        CALL GET_COLUMN(i_column = i_column, iflag = iflag,
     &                              xzdv = xvz, xphi = xnphi, xc = xc)


        xnv(idnt:idnb) = xvz(idvs+1:idvb) - xvz(idvs:idvb-1)


        DO jsolid = 1, nsolid

          jcompo = jf_to_io(jsolid)

          IF (idnw /= idnt) THEN

            xc_phis(idnt) = xc(idnt, jcompo) * (1.0D+00-xnphi(idnt))

          ELSE

            xc_phis(idnt) =
     &          (1.0D+00-thetatop) * xc(idnt  , jcompo)
     &                             * (1.0D+00-xnphi(idnt  ))
     &        +       thetatop     * xc(idntp1, jcompo)
     &                             * (1.0D+00-xnphi(idntp1))

          ENDIF

          xc_phis(idnt+1:idnb-1) =   xc(idntp1:idnbm1, jcompo)
     &                             * (1.0D+00-xnphi(idntp1:idnbm1))

          xc_phis(idnb) =
     &                thetabot     * xc(idnbm1, jcompo)
     &                             * (1.0D+00-xnphi(idnbm1))
     &        + (1.0D+00-thetabot) * xc(idnb  , jcompo)
     &                             * (1.0D+00-xnphi(idnb  ))

          tsolids(jsolid) = SUM(xnv(:)*xc_phis(:))

        ENDDO

        iflag = 0

      ELSE

        tsolids(:) = 0.0D+00

        iflag = 1                   ! illegal i_column value

      ENDIF


      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE REACLAY_TOTALS
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE ZONE_PROCRATE_TOTALS_ONECOL(i_column,
     &                              onecol_total_pra_xm)
!-----------------------------------------------------------------------

! Calculates the integrated (total) reaction rate for each consituent
! and each process in column i_column.

! Units are as follows:
!   - mol/(m2 of bulk sediment)/yr for solutes
!   - kg/(m2 of bulk sediment)/yr for solids

!--------------
! Modules used
!--------------

      USE mod_execontrol_medusa,    ONLY: ABORT_MEDUSA
      USE mod_defines_medusa,       ONLY: jp_realm_difblay,
     &                                    jp_realm_reaclay,
     &                                    jp_stderr
      USE mod_gridparam,            ONLY: idnw, idnt, idnb,
     &                                    idvw, idvs, idvb,
     &                                    thetatop, thetabot
      USE mod_indexparam,           ONLY: nsolut, nsolid, ncompo,
     &                                    jc_to_io, jf_to_io 
      USE mod_seafloor_wdata
      USE mod_chemicalconsts,       ONLY: SETCCT
      USE mod_processdata,          ONLY: nproc
      USE mod_processcontrol
      USE mod_rreac
      USE mod_seafloor_central,     ONLY: N_COLUMNS_USED, GET_COLUMN,
     &                                    GET_BOUNDARY_CONDS,
     &                                    GET_MATERIALCHARAS
      USE mod_medinterfaces


!-----------------------
! Variable declarations
!-----------------------

      IMPLICIT NONE

!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
! General (global) parameters
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

!     NONE


!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
! Variables in subroutine call arguments
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

      INTEGER                                    :: i_column
      DOUBLE PRECISION, DIMENSION(ncompo, nproc) :: onecol_total_pra_xm

      INTENT(IN)  :: i_column
      INTENT(OUT) :: onecol_total_pra_xm


!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
! Local variables
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

      INTEGER, PARAMETER :: idntp1 = idnt + 1
      INTEGER, PARAMETER :: idnbm1 = idnb - 1

      INTEGER :: n_grid_seafloor
      INTEGER :: inode
      INTEGER :: jcompo, jsolut, jsolid
      INTEGER :: iflag
      
      DOUBLE PRECISION, DIMENSION(idnw:idnb)         :: xnz
      DOUBLE PRECISION, DIMENSION(idvw:idvb)         :: xvz
      DOUBLE PRECISION, DIMENSION(idnt:idnb)         :: xnphi

      DOUBLE PRECISION, DIMENSION(idnw:idnb, ncompo) :: xc
      DOUBLE PRECISION, DIMENSION(idnw:idnt)         :: xc_phis_dbl
      DOUBLE PRECISION, DIMENSION(idnt:idnb)         :: xc_phis_rcl

      DOUBLE PRECISION, DIMENSION(nsolid)            :: ysolid

      DOUBLE PRECISION, DIMENSION(idnw:idnt)         :: vn_dbl_c
      DOUBLE PRECISION, DIMENSION(idnt:idnb)         :: vn_rcl_c
      DOUBLE PRECISION, DIMENSION(ncompo)            :: area_dummy
      DOUBLE PRECISION, DIMENSION(ncompo, nproc)     :: aproc_na
      DOUBLE PRECISION, DIMENSION(ncompo, nproc)     :: aproc_nb
      DOUBLE PRECISION, DIMENSION(ncompo, nproc)     :: aproc_nc

      TYPE(WDATA_CONTAINER)                          :: wdata

      DOUBLE PRECISION, DIMENSION(nsolut)            :: wconc
      DOUBLE PRECISION, DIMENSION(nsolid)            :: wfflx
      DOUBLE PRECISION, DIMENSION(ncompo)            :: ac
      DOUBLE PRECISION                               :: azdn, aphi
      DOUBLE PRECISION                               :: onecol_total_w2t
      DOUBLE PRECISION                               :: onecol_total_t2b

!---------------------
! End of declarations
!---------------------



!------------------
! Subroutine Start
!------------------

      IF (nproc == 0) RETURN


      CALL N_COLUMNS_USED(n_grid_seafloor)

#ifdef ALLOW_MPI
      IF (n_grid_seafloor == 0) THEN
        onecol_total_pra_xm(:,:) = 0.0D+00
        RETURN
      ENDIF
#endif

      IF ((i_column >= 1) .AND. (i_column <= n_grid_seafloor)) THEN

        CALL GET_COLUMN(i_column = i_column, iflag = iflag,
     &                              xzdn = xnz(:), xzdv = xvz(:),
     &                              xphi = xnphi(:),
     &                              xc = xc(:,:), ysolid = ysolid(:))

        IF (idnw == idnt) THEN

          vn_dbl_c(idnt)        = 0.0D+00   ! not to be used

        ELSE

          vn_dbl_c(idnw:idnt-1) = xvz(idvw+1:idvs)- xvz(idvw:idvs-1)
          vn_dbl_c(idnt)        = 0.0D+00   ! not to be used

        ENDIF

        vn_rcl_c(idnt:idnb)     = xvz(idvs+1:idvb)-xvz(idvs:idvb-1)


                                    ! Load characteristic material
                                    ! parameters for material in column
                                    ! <i_column> into MOD_MATERIALCHARAS.
                                    ! REACRATE is critically dependent
                                    ! on this information. It can,
                                    ! however, not load it itself, since
                                    ! it does not know which column it
                                    ! is being called for.

        CALL GET_MATERIALCHARAS(i_column = i_column, iflag = iflag)

        CALL GET_BOUNDARY_CONDS(i_column = i_column, gbcflag = iflag,
     &                              wdata = wdata,
     &                              wconc = wconc(:))
        
                                    ! Set values for chemical constants
                                    ! required for REACRATE.
        CALL SETCCT(wdata, wconc(:))
        CALL SetProcessParameters

        rreac_factor(:) = rreac_factor_max(:)

        IF (idnw /= idnt) THEN
            
          azdn  = xnz(idnw)
          aphi  = 1.0D+00
          ac(:) =  xc(idnw,:)
          CALL REACRATE(jp_realm_difblay, azdn, aphi, ac,
     &                              area_dummy, aproc_nc)
  
          onecol_total_pra_xm(:,:) = vn_dbl_c(idnw)*aproc_nc(:,:)

          DO inode = idnw + 1, idnt - 1
            azdn  = xnz(inode)
            aphi  = 1.0D+00
            ac(:) =  xc(inode,:)
            CALL REACRATE(jp_realm_difblay, azdn, aphi, ac,
     &                              area_dummy, aproc_nc)
            onecol_total_pra_xm(:,:) =
     &        onecol_total_pra_xm(:,:) + vn_dbl_c(inode)*aproc_nc(:,:)
          ENDDO

          azdn  =   xnz(idnt)
          aphi  = xnphi(idnt)
          ac(:) =    xc(idnt,:)
          CALL REACRATE(jp_realm_reaclay, azdn, aphi, ac,
     &                              area_dummy, aproc_nc)

          onecol_total_pra_xm(:,:) =
     &      onecol_total_pra_xm(:,:) + vn_rcl_c(idnt)*aproc_nc(:,:)

        ELSE

          azdn  =   xnz(idnt)
          aphi  = xnphi(idnt)
          ac(:) =   xc(idnt,:)
          CALL REACRATE(jp_realm_reaclay, azdn, aphi, ac,
     &                              area_dummy, aproc_na)

          azdn  =   xnz(idnt+1)
          aphi  = xnphi(idnt+1)
          ac(:) =    xc(idnt+1,:)
          CALL REACRATE(jp_realm_reaclay, azdn, aphi, ac,
     &                              area_dummy, aproc_nb)

          aproc_nc(:,:) =  (1.0D+00-thetatop)*aproc_na(:,:)
     &                    +     thetatop     *aproc_nb(:,:)

          onecol_total_pra_xm(:,:) = vn_rcl_c(idnt)*aproc_nc(:,:)

        ENDIF


        DO inode = idnt + 1, idnb - 1

          azdn  =   xnz(inode)
          aphi  = xnphi(inode)
          ac(:) =    xc(inode,:)
          CALL REACRATE(jp_realm_reaclay, azdn, aphi, ac,
     &                              area_dummy, aproc_nc)

          onecol_total_pra_xm(:,:) =
     &      onecol_total_pra_xm(:,:) + vn_rcl_c(inode)*aproc_nc(:,:)

        ENDDO


        azdn  =   xnz(idnb-1)
        aphi  = xnphi(idnb-1)
        ac(:) =    xc(idnb-1,:)
        CALL REACRATE(jp_realm_reaclay, azdn, aphi, ac,
     &                              area_dummy, aproc_na)

        azdn  =   xnz(idnb)
        aphi  = xnphi(idnb)
        ac(:) =    xc(idnb,:)
        CALL REACRATE(jp_realm_reaclay, azdn, aphi, ac,
     &                              area_dummy, aproc_nb)

        aproc_nc(:,:) =        thetabot     *aproc_na(:,:)
     &                  + (1.0D+00-thetabot)*aproc_nb(:,:)

        onecol_total_pra_xm(:,:) =
     &    onecol_total_pra_xm(:,:) + vn_rcl_c(idnb)*aproc_nc(:,:)

      ELSE

        WRITE(jp_stderr, '("[ZONE_PROCRATE_TOTALS_ONECOL] error: ", A)')
     &                              'Illegal column_number!'
        WRITE(jp_stderr, '(" - is equal to ", I0)')
     &                              i_column
        WRITE(jp_stderr, '(" - should be > 0 and <= ", I0)')
     &                              n_grid_seafloor
        WRITE(jp_stderr, '("Aborting!")')

        CALL ABORT_MEDUSA()

      ENDIF


      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE ZONE_PROCRATE_TOTALS_ONECOL
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE ZONE_PROCRATE_TOTALS(total_pra_xm)
!-----------------------------------------------------------------------

! Calculates the integrated (total) reaction rate for each consituent
! and each process separately, in each column.

! Units are as follows:
!   - mol/(m2 of bulk sediment)/yr for solutes
!   - kg/(m2 of bulk sediment)/yr for solids

!--------------
! Modules used
!--------------

      USE mod_execontrol_medusa,    ONLY: ABORT_MEDUSA
      USE mod_defines_medusa,       ONLY: jp_stderr
      USE mod_indexparam,           ONLY: ncompo
      USE mod_processdata,          ONLY: nproc
      USE mod_seafloor_central,     ONLY: N_COLUMNS_USED


!-----------------------
! Variable declarations
!-----------------------

      IMPLICIT NONE

!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
! General (global) parameters
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

!     NONE


!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
! Variables in subroutine call arguments
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

      DOUBLE PRECISION, DIMENSION(:,:,:) :: total_pra_xm


      INTENT(OUT) :: total_pra_xm


!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
! Local variables
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


      INTEGER :: i_column, n_grid_seafloor
      INTEGER :: n_errors

      DOUBLE PRECISION, DIMENSION(ncompo, nproc) :: onecol_total_pra_xm



!---------------------
! End of declarations
!---------------------



!------------------
! Subroutine Start
!------------------

      IF (nproc == 0) RETURN


      CALL N_COLUMNS_USED(n_grid_seafloor)


      IF (     (SIZE(total_pra_xm,1) /= ncompo)
     &    .OR. (SIZE(total_pra_xm,2) /= nproc)
     &    .OR. (SIZE(total_pra_xm,3) /= n_grid_seafloor)) THEN
        WRITE(jp_stderr, '("[ZONE_PROCRATE_TOTALS] error: ", A)')
     &    'incorrect dimensions for argument total_pra_xm:'
        WRITE(jp_stderr, '(" - is (/ ", 2(I0, ", "), I0, " /)")')
     &                              SIZE(total_pra_xm)
        WRITE(jp_stderr, '(" - should be ' //
     &                             '(/ ", 2(I0, ", "), I0, " /)")')
     &                              ncompo, nproc, n_grid_seafloor
        n_errors = n_errors + 1
      ENDIF


      IF (n_errors > 0) THEN

        WRITE(jp_stderr, '("[ZONE_PROCRATE_TOTALS] error: ")',
     &                              ADVANCE="NO")
        WRITE(jp_stderr, '("detected ", I0, " errors in the dummy' //
     &                   ' arguments. Aborting!")')

        CALL ABORT_MEDUSA()

      ENDIF


      DO i_column = 1, n_grid_seafloor

        CALL ZONE_PROCRATE_TOTALS_ONECOL(i_column, onecol_total_pra_xm)

        total_pra_xm(:, :, i_column) = onecol_total_pra_xm(:,:)

      ENDDO


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE ZONE_PROCRATE_TOTALS
!-----------------------------------------------------------------------



!=======================================================================
      END MODULE MOD_ZONE_MASS_TOTALS
!=======================================================================
