!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#ifdef DEBUG
#include <debug.h>
#endif
!=======================================================================
      MODULE MOD_SEDCORE
!=======================================================================


      USE mod_defines_medusa
      USE mod_execontrol_medusa,     ONLY: ABORT_MEDUSA
      USE mod_indexparam,            ONLY: jf_mud, nsolid
      USE mod_gridparam,             ONLY: idnb, nhl, ihl1, ihln,
     &                                     da_gpd_dcorelay

#ifdef ALLOW_MPI
      USE mod_execontrol_medusa,     ONLY: MEDEXE_NPROC,
     &                                     MEDEXE_MPI_COMM,
     &                                     MEDEXE_MPI_COMM_RANK,
     &                                     MEDEXE_MPI_GETPARTITIONING,
     &                                     jp_exeproc_ncio,
     &                                     lp_exeproc_singleproc_nc
      USE mpi,                       ONLY: MPI_INTEGER,
     &                                     MPI_DOUBLE_PRECISION,
     &                                     MPI_COMM_NULL, MPI_PROC_NULL,
     &                                     MPI_STATUS_IGNORE
#endif


      IMPLICIT NONE


      PRIVATE

      PUBLIC  :: REACLAY_X_CORELAY, CORELAY_PACKMASS,
     &           CORELAY_NLAY2ERODE_CLEAR, CORELAY_NLAY2ERODE_SET,
     &           SETUP_SEDCORE_SYSTEM, SEDFIL_FINALIZE,
     &           SEDFIL_PRUNE


! Fundamental parameters and flags
! ================================

      LOGICAL :: mod_sedcore_setupdone = .FALSE.

      INTEGER, PARAMETER :: nerodlay_default = MAX(nhl/2, 10)
      INTEGER :: nerodlay = nerodlay_default

      DOUBLE PRECISION, PARAMETER :: dp_yl_over_hl_max = 1.10D+00

! Special index values and other values
      INTEGER, PARAMETER ::  jp_pt_top_emptycore       = ihl1-1
      INTEGER, PARAMETER ::  jp_pt_norecord            = -1

      DOUBLE PRECISION, PARAMETER :: dp_neverbefore    = -1.0D+31
      DOUBLE PRECISION, PARAMETER :: dp_notyet         =  1.0D+31
      DOUBLE PRECISION, PARAMETER :: dp_fillvalue      =  1.0D+32
      INTEGER, PARAMETER          :: jp_layid_phantom  =  -1


! Global grid characteristics
! ===========================

! * number of columns used in the model and to be stored into the file
      INTEGER, SAVE :: ngrid = -1
      INTEGER, SAVE :: nsedcol_ncfile = -1


! CORE LAYer and layer stack characteristics
! ==========================================

! * solids concentrations
      DOUBLE PRECISION, DIMENSION(:,:,:), ALLOCATABLE, SAVE ::
     &   corelay_solids

! * burial (creation) time
      DOUBLE PRECISION, DIMENSION(:,:),  ALLOCATABLE, SAVE ::
     &   corelay_burtime

! * ID
      INTEGER, DIMENSION(:,:),  ALLOCATABLE, SAVE ::
     &   corelay_id

! * reserved record number
      INTEGER, DIMENSION(:,:),  ALLOCATABLE, SAVE ::
     &   corelay_irec

! * LAST ID, ID of the most recent layer that was
!   previously created in the stack
      INTEGER, DIMENSION(:,:),  ALLOCATABLE, SAVE ::
     &   corelay_lastid

! * total (integrated) amount of mass gained (In-Minus-Out)
      DOUBLE PRECISION, DIMENSION(:,:), ALLOCATABLE, SAVE ::
     &   corelay_solids_total_imo

! * pointer to YOUNGEST and OLDEST layers in the stack
      INTEGER, DIMENSION(:), ALLOCATABLE, SAVE ::
     &   corelay_ihl_youngest, corelay_ihl_oldest

! * number of core layers scheduled for erosion
      INTEGER, DIMENSION(:), ALLOCATABLE, SAVE ::
     &   corelay_nlay2erode

! I/O buffer layer definitions for ERODed LAYers
! ==============================================

! * solids concentrations
      DOUBLE PRECISION, DIMENSION(:,:,:), ALLOCATABLE, SAVE ::
     &   erodlay_solids

! * time of initial creation
      DOUBLE PRECISION, DIMENSION(:,:), ALLOCATABLE, SAVE ::
     &   erodlay_burtime

! * time of erosion
      DOUBLE PRECISION, DIMENSION(:,:), ALLOCATABLE, SAVE ::
     &   erodlay_erotime

! * ID of the layer when it was created
      INTEGER, DIMENSION(:,:),  ALLOCATABLE, SAVE ::
     &   erodlay_id

! * reserved record number
      INTEGER, DIMENSION(:,:),  ALLOCATABLE, SAVE ::
     &   erodlay_irec

! * buffer stack: height (number of elements)
      INTEGER, DIMENSION(:),  ALLOCATABLE, SAVE ::
     &   erodlay_nlay

#ifdef DEBUG
! * maximum buffer stack height (for info)
      INTEGER, SAVE ::
     &   erodlay_maxlevel = 0
#endif


! Sediment core file data
! =======================

! * SEDiment CORe FILE: logical flag, .TRUE. if used
      LOGICAL, DIMENSION(2), SAVE :: sedfil_used = .FALSE.

! * SEDiment CORe FILE: logical flag, .TRUE. if internal mode
!     (without an external file) is used
      LOGICAL, SAVE :: sedfil_internal = .FALSE.

! Notice: both previous flags are set to .FALSE. initially
!   - sedfil_used is switched to .TRUE. by SEDFIL_OPEN,
!     which only executes as long as sedfil_internal == .FALSE.
!   - sedfil_internal is switched to .TRUE. by SEDFIL_NOFILE,
!     which only executes as long as sedfil_used == .FALSE.
! Accordingly, it depends on whether SEDFIL_OPEN or SEDFIL_NOFILE
! is called first.


! * SEDiment CORe FILE: identifiers for ncin and ncout:
      INTEGER, PARAMETER :: jp_ncin  = 1
      INTEGER, PARAMETER :: jp_ncout = 2


! * SEDiment CORe FILE: open/not open flag
      LOGICAL, DIMENSION(2), SAVE :: sedfil_isopen = .FALSE.

! * SEDiment CORe FILE: NetCDF ID
      INTEGER, DIMENSION(2), SAVE :: sedfil_ncid

! * SEDiment CORe FILE: NetCDF dimension and variable IDs
      INTEGER, DIMENSION(2), SAVE :: dim_col
      INTEGER, DIMENSION(2), SAVE ::  id_col

      INTEGER, DIMENSION(2), SAVE :: dim_lay
      INTEGER, DIMENSION(2), SAVE ::  id_lay

      CHARACTER(LEN=*), PARAMETER :: cp_nbcorelays = 'nlay_corelay'
      INTEGER, DIMENSION(2), SAVE :: id_nbcorelays

      CHARACTER(LEN=*), PARAMETER :: cp_nbrecs     = 'nrecs'
      INTEGER, DIMENSION(2), SAVE :: id_nbrecs

      CHARACTER(LEN=*), PARAMETER :: cp_itcorelay  = 'irec_top_corelay'
      INTEGER, DIMENSION(2), SAVE :: id_itcorelay

      CHARACTER(LEN=*), PARAMETER :: cp_ilerodlay  = 'irec_last_erodlay'
      INTEGER, DIMENSION(2), SAVE :: id_ilerodlay

      CHARACTER(LEN=*), PARAMETER :: cp_coreid     = 'core_id'
      INTEGER, DIMENSION(2), SAVE :: id_coreid

      CHARACTER(LEN=*), PARAMETER :: cp_layid      = 'lay_id'
      INTEGER, DIMENSION(2), SAVE :: id_layid

      CHARACTER(LEN=*), PARAMETER :: cp_ilastlayid = 'last_lay_id'
      INTEGER, DIMENSION(2), SAVE :: id_ilastlayid

      CHARACTER(LEN=*), PARAMETER :: cp_ipreclay   = 'irec_preclay'
      INTEGER, DIMENSION(2), SAVE :: id_ipreclay

      CHARACTER(LEN=*), PARAMETER :: cp_burtime    = 'burial_time'
      INTEGER, DIMENSION(2), SAVE :: id_burtime

      CHARACTER(LEN=*), PARAMETER :: cp_erotime    = 'erosion_time'
      INTEGER, DIMENSION(2), SAVE :: id_erotime

      CHARACTER(LEN=*), PARAMETER :: cp_prefix_hcont = 'hcont'
      INTEGER, DIMENSION(nsolid, 2), SAVE :: id_hcont

! * SEDiment CORe FILE: total number of records previously reserved
!     (already written or still in buffer)
      INTEGER, DIMENSION(2), SAVE :: sedfil_nrecs = 0

! * SEDiment CORe FILE: for each core, number records in the file
!     (all records: CORELAY, ERODLAY)
      INTEGER, DIMENSION(:,:), ALLOCATABLE, SAVE :: sedflay_nrecs

! * SEDiment CORe FILE: for each core, index of the record for
!     the latest layer that still exists only in the file
      INTEGER, DIMENSION(:,:), ALLOCATABLE, SAVE :: sedflay_irectop

! * SEDiment CORe FILE: for each core, index of the record for
!     the latest eroded layer that has been written to the file
      INTEGER, DIMENSION(:,:), ALLOCATABLE, SAVE :: sedflay_irecero

! * SEDiment CORe FILE: for each core, number of not yet eroded layers
!     in the file (sets the maximum number of records that are
!     available for transfer back).
      INTEGER, DIMENSION(:,:), ALLOCATABLE, SAVE :: sedflay_nlay

#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
      INTEGER, PARAMETER :: icol_dbg = 1
      INTEGER, PARAMETER :: jp_coldbg = 111
#endif
#endif
#endif


#ifdef ALLOW_MPI
! * MPI administrivia
      INTEGER, SAVE :: n_cprocs = 0
      INTEGER, SAVE :: i_mycomm = MPI_COMM_NULL
      INTEGER, SAVE :: i_myrank = MPI_PROC_NULL
      INTEGER, SAVE :: nsedcol_global = -1
      INTEGER, SAVE :: iproc_1stocn
      INTEGER, SAVE, DIMENSION(:), ALLOCATABLE :: nsedcol_pproc
      INTEGER, SAVE, DIMENSION(:), ALLOCATABLE :: ioffset_sedcol_pproc

      LOGICAL, SAVE :: l_oneio4many = .FALSE.
      LOGICAL, SAVE :: l_file_is_mine = .TRUE.


! * MPI parameterized messages and tags
      INTEGER, PARAMETER :: jp_mpimsg_null    =    0
      INTEGER, PARAMETER :: jp_mpimsg_go      = 1000
      INTEGER, PARAMETER :: jp_mpimsg_read    = 2000
      INTEGER, PARAMETER :: jp_mpimsg_write   = 3000
      INTEGER, PARAMETER :: jp_mpimsg_done    = 4000
      INTEGER, PARAMETER :: jp_mpimsg_send_nrecs  = 7001
      INTEGER, PARAMETER :: jp_mpimsg_recv_nrecs  = 7002

      INTEGER, PARAMETER :: jp_mpitag_reqio   = 5000
      INTEGER, PARAMETER :: jp_mpitag_data0   = 6000
      INTEGER, PARAMETER :: jp_mpitag_data1   = 6001
      INTEGER, PARAMETER :: jp_mpitag_data2   = 6002
      INTEGER, PARAMETER :: jp_mpitag_data3   = 6003
      INTEGER, PARAMETER :: jp_mpitag_data4   = 6004
      INTEGER, PARAMETER :: jp_mpitag_data5   = 6005
      INTEGER, PARAMETER :: jp_mpitag_data6   = 6006
      INTEGER, PARAMETER :: jp_mpitag_data7   = 6007
      INTEGER, PARAMETER :: jp_mpitag_nrecs   = 7000


! * arrays for the MPI I/O rank to send and receive partial arrays
!     from and to the individual ranks
      INTEGER, DIMENSION(:), ALLOCATABLE, SAVE :: iarr_mpisend_c
      INTEGER, DIMENSION(:), ALLOCATABLE, SAVE :: iarr_mpirecv_c
      INTEGER, DIMENSION(:), ALLOCATABLE, SAVE :: irec_mpirecv_c
#else
      LOGICAL, SAVE :: l_file_is_mine = .TRUE.
#endif


      CONTAINS


!======================================================================
      SUBROUTINE SEDCOREBUFFER_SETUP
!======================================================================


      USE mod_seafloor_central, ONLY: N_COLUMNS_USED


      IMPLICIT NONE


      CHARACTER(LEN=*), PARAMETER :: cfmt_modprocname_a =
     &  '("[MOD_SEDCORE/SEDCOREBUFFER_SETUP]:", A)'

      CHARACTER(LEN=*), PARAMETER :: cfmt_a = '(" - ", A)'

#ifdef ALLOW_MPI
      INTEGER :: n
#endif


#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_modprocname_a) 'Start'
#endif

      IF (mod_sedcore_setupdone) THEN
#ifdef DEBUG
        WRITE(jp_stddbg, cfmt_a) 'Setup already done -- skipping.'
        WRITE(jp_stddbg, cfmt_modprocname_a) 'Return'
        WRITE(jp_stddbg, '()')
#endif
        RETURN
      ENDIF

#ifdef ALLOW_MPI
      n_cprocs = MEDEXE_NPROC()     ! number of processes
      i_mycomm = MEDEXE_MPI_COMM()  ! communicator of Medusa
      i_myrank = MEDEXE_MPI_COMM_RANK()     ! rank of process executing this instance

                                    ! Get the number of columns in the
                                    ! mod_seafloor_central of the
                                    ! current process
      CALL N_COLUMNS_USED(ngrid)

                                    ! Now set up the file reading
                                    ! process distribution logics:
      l_file_is_mine = .TRUE.       ! - by default, each process writes
                                    !   its own files

      l_oneio4many = .FALSE.        ! - by default: no rank does I/O
                                    !   for others; this must be .FALSE.
                                    !   if n_cprocs==1 ("many" means ">1")

      IF (n_cprocs > 1) THEN        ! When running in a multi-processor
                                    ! environment with more than one process,
                                    ! then this may need to be adapted:
        IF (lp_exeproc_singleproc_nc) THEN  ! - if single-processor I/O
                                            !   for NetCDF files is requested,
          l_oneio4many = .TRUE.             !   set flag to indicate that the
                                            !   reading process reads for
                                            !   other (>1) processes than itself
          IF (i_myrank /= jp_exeproc_ncio) THEN     ! - and deactivate writing for other
            l_file_is_mine = .FALSE.                !   processes than the writing one
          ENDIF

        ELSE                        ! Multi-process environment where each
                                    ! rank controls its own files. However,
                                    ! do not open a file if there are no
                                    ! data to read. This could happen if one
                                    ! process gets atributed a domain without
                                    ! seafloor points.
          IF (ngrid == 0) l_file_is_mine = .FALSE.

        ENDIF

      ENDIF
                                    ! Get the number of columns in all
                                    ! the mod_seafloor_central's of all
                                    ! concurrent processes and the
                                    ! partitioning information
      ALLOCATE(nsedcol_pproc(0:n_cprocs-1))
      ALLOCATE(ioffset_sedcol_pproc(0:n_cprocs-1))

      CALL MEDEXE_MPI_GETPARTITIONING(nsedcol_global, iproc_1stocn,
     &                              nsedcol_pproc, ioffset_sedcol_pproc)

                                    ! If single-processor NetCDF I/O, and
                                    ! execution environment includes more
                                    ! than one processor, then allocate
                                    ! work-space memory to receive the data
                                    ! by the writing process:
      IF (l_oneio4many) THEN

        nsedcol_ncfile = nsedcol_global

      ELSE

        nsedcol_ncfile = ngrid

      ENDIF
#else
                                    ! Get the number of columns in the
                                    ! mod_seafloor_central of the
                                    ! current process
      CALL N_COLUMNS_USED(ngrid)

                                    ! In single-processor environments the
                                    ! running process always owns its files
                                    ! and reads only ngrid-long
                                    ! records
      l_file_is_mine = .TRUE.
      nsedcol_ncfile = ngrid
#endif

#ifdef ALLOW_MPI
      IF (ngrid >= 0) THEN
#else
      IF (ngrid > 0) THEN           ! With MPI, some mod_seafloor_central may be empty
#endif
                                    ! The seafloor module has been
                                    ! set up and it is safe to proceed

        ALLOCATE(corelay_solids (nsolid, ihl1:ihln, ngrid))
        ALLOCATE(corelay_burtime(        ihl1:ihln, ngrid))
        ALLOCATE(corelay_id     (        ihl1:ihln, ngrid))
        ALLOCATE(corelay_irec   (        ihl1:ihln, ngrid))
        ALLOCATE(corelay_lastid (                   ngrid, 2))


        ALLOCATE(corelay_nlay2erode(ngrid))

        corelay_nlay2erode(:) = 0


        ALLOCATE(corelay_solids_total_imo(nsolid, ngrid))

        corelay_solids_total_imo = 0.0D+00


        ALLOCATE(corelay_ihl_youngest(ngrid))
        ALLOCATE(corelay_ihl_oldest(ngrid))

                                    ! Pre-initialise pointers to the
                                    ! uppermost (youngest) historical layers:
                                    ! Set corelay_ihl_youngest to jp_pt_top_emptycore
                                    ! and corelay_ihl_oldest to jp_pt_top_emptycore --
                                    ! the historical layers are empty.
        corelay_ihl_youngest(:) = jp_pt_top_emptycore
        corelay_ihl_oldest(:)   = jp_pt_top_emptycore

        corelay_lastid(:, :) = 0    ! Pre-initialise ID of last created layer to 0


                                    ! If there is no file in use, we set nerodlay to 1
                                    ! All of the operations remain valid then

        IF (.NOT. (sedfil_used(jp_ncin).OR.sedfil_used(jp_ncout))) THEN
          nerodlay = 1
        ELSE
          nerodlay = nerodlay_default
        ENDIF


        ALLOCATE(erodlay_solids (1:nsolid, nerodlay, ngrid))
        ALLOCATE(erodlay_burtime(          nerodlay, ngrid))
        ALLOCATE(erodlay_erotime(          nerodlay, ngrid))
        ALLOCATE(erodlay_id     (          nerodlay, ngrid))
        ALLOCATE(erodlay_irec   (          nerodlay, ngrid))

        ALLOCATE(erodlay_nlay   (                    ngrid))


        erodlay_nlay(:) = 0         ! Initialise number of eroded layers to 0

#ifdef DEBUG
        erodlay_maxlevel = 0        ! Maximum level of ERODLAY fill-up
                                    ! reached during execution.
#endif

        IF (sedfil_used(jp_ncin) .OR. sedfil_used(jp_ncout)) THEN

          ALLOCATE(sedflay_nrecs(ngrid, 2))
          ALLOCATE(sedflay_nlay(ngrid, 2))
          ALLOCATE(sedflay_irectop(ngrid, 2))
          ALLOCATE(sedflay_irecero(ngrid, 2))

#ifdef ALLOW_MPI
          IF (l_oneio4many) THEN
            IF (l_file_is_mine) THEN

                                    ! ALLOCATE nsedcol_pproc,
                                    ! ioffset_sedcol_pproc.
                                    ! Get data from MOD_EXECONTROL

              n = nsedcol_ncfile
            ELSE
              n = 0
            ENDIF
          ELSE
            n = 0
          ENDIF

          ALLOCATE(iarr_mpisend_c(n))
          ALLOCATE(iarr_mpirecv_c(n))
          ALLOCATE(irec_mpirecv_c(n))
#endif

        ENDIF

                                    ! Set up and basic initialisation done
        mod_sedcore_setupdone = .TRUE.

#ifdef DEBUG
        WRITE(jp_stddbg, cfmt_a) 'Set-up finished successfully.'
#endif

      ELSE

        WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
        WRITE(jp_stderr, cfmt_a)
     &    'MOD_SEAFLOOR_CENTRAL not yet set up -- aborting'
        CALL ABORT_MEDUSA()

      ENDIF

#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_modprocname_a) 'End'
      WRITE(jp_stddbg, '()')
#endif


      RETURN


!======================================================================
      END SUBROUTINE SEDCOREBUFFER_SETUP
!======================================================================



!======================================================================
      SUBROUTINE REACLAY_X_CORELAY(atime)
!======================================================================


      USE mod_materialcharas
      USE mod_milieucharas, ONLY: xphi

      USE mod_seafloor_central, ONLY: GET_COLUMN, SAVE_COLUMN


      IMPLICIT NONE


! Dummy argument variables
! ========================

      DOUBLE PRECISION, INTENT(IN) :: atime


! Local variables
! ===============

      DOUBLE PRECISION
     &   :: HL_SolidsVol, YL_SolidsVol
      DOUBLE PRECISION, DIMENSION(1:nsolid)
     &   :: ysolid,  ytransfer
      INTEGER
     &   :: i_column
      INTEGER
     &   :: nlay2erode, ilay2erode
      INTEGER
     &   :: iflag, istatus, ilayertype, n
      INTEGER
     &   :: ihl_youngest, ihl_oldest, iel_nlay, ihl_lastid
      DOUBLE PRECISION, DIMENSION(nsolid, ihl1:ihln)
     &   :: core_solids
      DOUBLE PRECISION
     &   :: HL_burtime, EL_burtime, EL_erotime


      LOGICAL
     &   :: message_displayed = .FALSE.
      CHARACTER(LEN=*), PARAMETER
     &   :: message_text =
     &      'Core history traced if sediment file used!'


      CHARACTER(LEN=*), PARAMETER :: cfmt_modprocname_a =
     &   '("[MOD_SEDCORE/REACLAY_X_CORELAY]: ", A)'

      CHARACTER(LEN=*), PARAMETER :: cfmt_a = '(" - ", A)'

#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
      INTEGER :: ihl, iel
#endif
#endif
#endif


      IF (.NOT. mod_sedcore_setupdone) THEN
        WRITE(jp_stderr, cfmt_modprocname_a) 'Warning'
        WRITE(jp_stderr, cfmt_a)
     &    'Calling SETUP_SEDCORE_SYSTEM -- using pure internal setup.'
        CALL SETUP_SEDCORE_SYSTEM()
      ENDIF


      IF (.NOT. message_displayed) THEN
        WRITE(jp_stderr, cfmt_modprocname_a) 'Warning'
        WRITE(jp_stderr, cfmt_a) message_text
        message_displayed = .TRUE.
      ENDIF


#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
      WRITE(jp_coldbg,'(1X, A, I0, A)')
     &  '[REACLAY_X_CORELAY] Column ', icol_dbg, ' @start: '
      CALL GET_COLUMN(icol_dbg, iflag, YSOLID = ysolid)
      WRITE(jp_coldbg,*)
     &  atime,
     &  corelay_ihl_youngest(icol_dbg), corelay_ihl_oldest(icol_dbg),
     &  ysolid
#endif
#endif
#endif


#ifdef ALLOW_MPI
      CALL MSC_QUEUE_START

      IF (.NOT. l_file_is_mine) THEN
        CALL MPI_SEND(jp_mpimsg_send_nrecs, 1, MPI_INTEGER,
     &                              jp_exeproc_ncio, jp_mpitag_reqio,
     &                              i_mycomm, MPI_STATUS_IGNORE, iflag)
        CALL MPI_RECV(sedfil_nrecs(jp_ncout), 1, MPI_INTEGER,
     &                              jp_exeproc_ncio, jp_mpitag_data0,
     &                              i_mycomm, MPI_STATUS_IGNORE, iflag)
      ENDIF
#endif

      all_columns: DO i_column = 1, ngrid

                                    ! Retrieve ysolid(:) for the
                                    ! current sediment column
        CALL GET_COLUMN(i_column, iflag, YSOLID = ysolid)

        IF (iflag /= 0) THEN
          WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
          WRITE(jp_stderr, cfmt_a)
     &      '[632] Error trapped at GET_COLUMN -- aborting.'
          CALL ABORT_MEDUSA()
        ENDIF


                                    ! Use the following auxiliary
                                    ! indices and levels for convenience!
        ihl_youngest = corelay_ihl_youngest(i_column)
        ihl_oldest   = corelay_ihl_oldest(i_column)
        ihl_lastid   = corelay_lastid(i_column, jp_ncout)
        iel_nlay     = erodlay_nlay(i_column)

                                    ! H-layer volume and initial Y-layer volume
        HL_SolidsVol = da_gpd_dcorelay*(1.0D+00-xphi(idnb))
        YL_SolidsVol = SUM(ysolid(:)*apsv(:))

        nlay2erode = corelay_nlay2erode(i_column)


        IF (nlay2erode > 0) THEN

          erosion:
     &    DO ilay2erode = 1, nlay2erode

#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
            IF (i_column == icol_dbg) THEN
              WRITE(jp_coldbg,*) ' Erosion ', YL_SolidsVol, HL_SolidsVol
              WRITE(jp_coldbg,*) ' ihl_bot/top: ',
     &                              ihl_oldest, ihl_youngest
              WRITE(jp_coldbg,*) ' ilay2erode/nlay2erode: ',
     &                              ilay2erode, nlay2erode
              WRITE(jp_coldbg,*) ' ysolid: ', ysolid
            ENDIF
#endif
#endif
#endif
                                    ! If we use no sediment file, ERODLAY
                                    ! layers are immediately discarded
            IF (.NOT. sedfil_used(jp_ncout)) iel_nlay = 0

                                    ! If there is no space on the ERODLAY stack,
                                    ! then empty it first.
                                    ! Should never happen, but let us make sure!
                                    ! nerodlay > 0, hence if we do not use a file,
                                    ! the following will be skipped.
            IF (iel_nlay == nerodlay) THEN

#ifdef VERBOSE
              WRITE(jp_stderr, cfmt_modprocname_a, ADVANCE="NO")
     &                              'Warning ("should never happen")'
              WRITE(jp_stderr, cfmt_a, ADVANCE="NO")
     &                              'ERODLAY buffer full for core '
              WRITE(jp_stderr, *) i_column, ' at time ', atime
#endif

              ilayertype = 2        ! ERODLAY layers only
              n = 0

#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
              IF (i_column == icol_dbg) THEN
                WRITE(jp_coldbg,*)
     &            'ERODLAY buffer full -> SEDFIL_PUSHRECORDS'
              ENDIF
#endif
#endif
#endif
              CALL SEDFIL_PUSHRECORDS(i_column, ilayertype, n, iflag)
              iel_nlay = 0

#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
              IF (i_column == icol_dbg) THEN
                WRITE(jp_coldbg,*) ' ihl_bot/top:694 ',
     &            corelay_ihl_oldest(icol_dbg),
     &            corelay_ihl_youngest(icol_dbg)
              ENDIF
#endif
#endif
#endif

            ENDIF


            IF (ihl_youngest == jp_pt_top_emptycore) THEN
                                    ! No layers available for chemical
                                    ! erosion: either there have never been
                                    ! any (jp_pt_top_emptycore) or the last
                                    ! remaining one has just previously been
                                    ! eroded. Restore some if possible.

                                    ! Before calling SEDFIL_PULLRECORDS,
                                    ! we need to make sure that the
                                    ! youngest and oldest pointers are
                                    ! up-to-date (they may have been modified
                                    ! in a previous round here)

                                    ! First make sure ihl_oldest is OK
              ihl_oldest = jp_pt_top_emptycore

              IF (sedfil_used(jp_ncout)) THEN
                corelay_ihl_youngest(i_column) = jp_pt_top_emptycore
                corelay_ihl_oldest(i_column)   = jp_pt_top_emptycore


#ifdef VERBOSE
                                    ! This should normally not be necessary,
                                    ! since REACLAY_X_CORELAY is generally
                                    ! called at the end of a solvsed-step
                                    ! that involved calls to CORELAY_PACKMASS
                WRITE(jp_stderr, cfmt_modprocname_a, ADVANCE="NO")
     &                              'Warning (unusual behaviour)'
                WRITE(jp_stderr, cfmt_a, ADVANCE="NO")
     &                              'CORELAY buffer empty for core '
                WRITE(jp_stderr, *) i_column, ' at time ', atime
                WRITE(jp_stderr, cfmt_a, ADVANCE="NO")
     &                           'available CORELAY records from file: '
                WRITE(jp_stderr, '(I0)')
     &                              sedflay_nlay(i_column, jp_ncout)
#endif


#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
                IF (i_column == icol_dbg) THEN
                  WRITE(jp_coldbg,*)
     &                      'CORELAY buffer empty -> SEDFIL_PULLRECORDS'
                ENDIF
#endif
#endif
#endif

                n = 1               ! try to restore one CORELAY layer
                CALL SEDFIL_PULLRECORDS(jp_ncout, i_column, n, istatus)

#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
                IF (i_column == icol_dbg) THEN
                  WRITE(jp_coldbg,*) ' ihl_bot/top:759 ',
     &                              corelay_ihl_oldest(icol_dbg),
     &                              corelay_ihl_youngest(icol_dbg)
                ENDIF
#endif
#endif
#endif

#ifdef VERBOSE
                WRITE(jp_stderr, cfmt_modprocname_a) 'Warning'
                IF (istatus == 0) THEN
                  WRITE(jp_stderr, cfmt_a)
     &                              'recovered one layer from the file.'
                ELSE
                  WRITE(jp_stderr, cfmt_a)
     &                        'unable to recover a layer from the file.'
                ENDIF

                WRITE(jp_stderr, cfmt_a, ADVANCE="NO")
     &                         'remaining available records from file: '
                WRITE(jp_stderr, '(I0)')
     &                              sedflay_nlay(i_column, jp_ncout)
#endif

                                    ! Update the pointers for use here
                ihl_youngest = corelay_ihl_youngest(i_column)
                ihl_oldest   = corelay_ihl_oldest(i_column)

                                    ! Notice: both pointers may still be
                                    ! jp_pt_top_emptyrecord, e.g., if
                                    ! no layers were available in the file.
              ENDIF


            ENDIF

                                    ! Create a new ERODed LAYer
            iel_nlay = iel_nlay + 1
#ifdef DEBUG
            erodlay_maxlevel = MAX(erodlay_maxlevel, iel_nlay)
#endif

                                    ! If ihl_youngest is now still
                                    ! jp_pt_top_emptycore, then there was
                                    ! no layer available for filling up.
                                    ! We blend in a phantom layer

            IF (ihl_youngest == jp_pt_top_emptycore) THEN

              WRITE(jp_stderr, cfmt_modprocname_a) 'Warning'
              WRITE(jp_stderr, cfmt_a, ADVANCE="NO")
     &                         'blending in one phantom layer for core '
              WRITE(jp_stderr,*)
     &                         i_column, ' at time ', atime
              ytransfer(:) = 0.0D+00
              ytransfer(jf_mud) = HL_SolidsVol/apsv(jf_mud)

#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
              IF (i_column == icol_dbg) THEN
                WRITE(jp_coldbg,*) ' phantom transfer:845: ',
     &                              ytransfer
              ENDIF
#endif
#endif
#endif
                                    ! Reserve new record for the phantom layer
              sedfil_nrecs(jp_ncout) = sedfil_nrecs(jp_ncout) + 1


                                    ! Transfer the data to the new ERODed LAYer
              erodlay_id     (   iel_nlay, i_column) = jp_layid_phantom
              erodlay_irec   (   iel_nlay, i_column) =
     &                              sedfil_nrecs(jp_ncout)
              erodlay_solids (:, iel_nlay, i_column) = ytransfer(:)
              erodlay_burtime(   iel_nlay, i_column) = dp_neverbefore
              erodlay_erotime(   iel_nlay, i_column) = atime


#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
              IF (i_column == icol_dbg) THEN
                WRITE(jp_coldbg,*) 'phantom created -- ERODstack now'
                DO ihl = iel_nlay, 1, -1
                  WRITE(jp_coldbg,*) ihl, erodlay_id(ihl, i_column),
     &                              erodlay_irec(ihl, i_column),
     &                              erodlay_erotime(ihl, i_column)
                ENDDO
              ENDIF
#endif
#endif
#endif

            ELSE
                                    ! There is at least one layer left:
                                    ! transfer its contents to the Y--Layer
              ytransfer(:) = corelay_solids(:, ihl_youngest, i_column)

#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
              IF (i_column == icol_dbg) THEN
                WRITE(jp_coldbg,*) ' physical transfer:888: ', ytransfer
              ENDIF
#endif
#endif
#endif
                                    ! Transfer the data to the new ERODed LAYer
              erodlay_id       (       iel_nlay, i_column)
     &           = corelay_id  (   ihl_youngest, i_column)
              erodlay_irec     (       iel_nlay, i_column)
     &           = corelay_irec(   ihl_youngest, i_column)
              erodlay_solids   (:,     iel_nlay, i_column)
     &           = ytransfer   (:)
              erodlay_burtime  (       iel_nlay, i_column)
     &           = corelay_burtime(ihl_youngest, i_column)
              erodlay_erotime  (       iel_nlay, i_column)
     &           = atime


#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
              IF (i_column == icol_dbg) THEN
                WRITE(jp_coldbg,*) 'ERODLAY created -- ERODstack now'
                DO iel = iel_nlay, 1, -1
                  WRITE(jp_coldbg,*) iel, erodlay_id(iel, i_column),
     &                              erodlay_irec(iel, i_column),
     &                              erodlay_erotime(iel, i_column)
                ENDDO
              ENDIF
#endif
#endif
#endif

                                    ! Clear the ihl_youngest data from corelay_solids
              corelay_id     (   ihl_youngest, i_column)
     &           = -1
              corelay_irec   (   ihl_youngest, i_column)
     &           = jp_pt_norecord
              corelay_solids (:, ihl_youngest, i_column)
     &           = 0.0D+00
              corelay_burtime(   ihl_youngest, i_column)
     &           = dp_neverbefore

                                    ! and finally discard it!
              IF (ihl_youngest == ihl_oldest) THEN
                                    ! the last layer in the stack has just been eroded
                                    ! flag the stack as empty
                ihl_youngest = jp_pt_top_emptycore
                ihl_oldest   = jp_pt_top_emptycore
              ELSE                  ! there was more than one left: simply move down
                                    ! the top level indicator
                ihl_youngest = IHL_DOWN(ihl_youngest)
              ENDIF



#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
              IF (i_column == icol_dbg) THEN
                WRITE(jp_coldbg,*) 'CORELAY deleted -- COREstack now'
                ihl = ihl_youngest
                DO WHILE(ihl_youngest /= jp_pt_top_emptycore)
                  WRITE(jp_coldbg,*) ihl, corelay_id(ihl, i_column),
     &                              corelay_irec(ihl, i_column),
     &                              corelay_burtime(ihl, i_column)
                  IF (ihl == ihl_oldest) EXIT
                  ihl = IHL_DOWN(ihl)
                ENDDO
              ENDIF
#endif
#endif
#endif

            ENDIF


#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
            IF (i_column == icol_dbg) THEN
              WRITE(jp_coldbg,*) ' ihl_bot/top:969 ',
     &                              ihl_oldest, ihl_youngest
            ENDIF
#endif
#endif
#endif

                                    ! No need to add the matter from the discarded
                                    ! H-layer to the Y-layer - this has already been
                                    ! done in solvsed_onestep.
                                    ! However: update the net balance of the core
                                    ! layers. The way this is done below
                                    ! possibly leads to negative clay amount
                                    ! (in case phantom layers are transferred),
                                    ! but it helps to keep the overall mass
                                    ! balance consistent.
            corelay_solids_total_imo(:,i_column)
     &        = corelay_solids_total_imo(:,i_column) - ytransfer(:)


          ENDDO erosion

                                    ! All done: corelay_nlay2erode(i_column)
                                    ! remains to be reset to zero.
          corelay_nlay2erode(i_column) = 0


        ELSE


          burial:
     &    DO WHILE(YL_SolidsVol > dp_yl_over_hl_max*HL_SolidsVol)

#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
            IF (i_column == icol_dbg) THEN
              WRITE(jp_coldbg,*) ' Burial ', YL_SolidsVol, HL_SolidsVol
              WRITE(jp_coldbg,*) ' ihl_bot/top: ',
     &                              ihl_oldest, ihl_youngest
              WRITE(jp_coldbg,*) ' ysolid_o: ', ysolid
            ENDIF
#endif
#endif
#endif

! Make ihl_youngest point to the next free slot and adapt ihl_oldest
! if necessary (e.g., if old layers have to be deleted first)

            IF (ihl_youngest == jp_pt_top_emptycore) THEN
                                    ! The core is currently empty:
              ihl_youngest = ihl1   ! create the first layer.
              ihl_oldest   = ihl1

            ELSE                    ! The core is not empty:
                                    ! add another layer ontop
                                    ! of the most recent one.

                                    ! If CORELAY is full, make some space
              IF (CORELAY_NFILLED_IHL(ihl_oldest, ihl_youngest) == nhl)
     &        THEN

                ilayertype = 1      ! core layers only
                n = nhl/2           ! request to write half of them

                                    ! update pointers for SEDFIL_PUSHRECORDS
                corelay_ihl_youngest(i_column) = ihl_youngest
                corelay_ihl_oldest(i_column) = ihl_oldest
#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
                IF (i_column == icol_dbg) THEN
                  WRITE(jp_coldbg,*)
     &                       'CORELAY buffer full -> SEDFIL_PUSHRECORDS'
                ENDIF
#endif
#endif
#endif
                CALL SEDFIL_PUSHRECORDS(i_column, ilayertype, n, iflag)

                ihl_youngest = corelay_ihl_youngest(i_column)
                ihl_oldest   = corelay_ihl_oldest(i_column)
                                    ! update pointers for our use

                IF (n == 0) THEN    ! No records have been written
                                    ! This may happen when there is
                                    ! no SEDFIL file in use

#ifdef VERBOSE
                  WRITE(jp_stderr, cfmt_modprocname_a) 'Warning'
                  WRITE(jp_stderr, cfmt_a, ADVANCE="NO") 'at t = '
                  WRITE(jp_stderr, *) atime,
     &                           ' Unable to backup layer(s) for core ',
     &                    i_column, ' to file. One layer will get lost.'
#endif

                                    ! Move up ihl_oldest one layer
                                    ! i.e., discard that layer!
                  ihl_oldest = IHL_UP(ihl_oldest)

                ENDIF

              ENDIF

                                    ! make ihl_youngest point
                                    ! to the slot for the new layer
              ihl_youngest = IHL_UP(ihl_youngest)

            ENDIF


                                    ! Create the new layer: set its ID,
                                    ! set (reserve) its record number
                                    ! and calculate the amount of matter
                                    ! to be transferred into it
            ihl_lastid   = ihl_lastid + 1
            sedfil_nrecs(jp_ncout) = sedfil_nrecs(jp_ncout) + 1

            ytransfer(:) = ysolid(:)*(HL_SolidsVol/YL_SolidsVol)

                                    ! Now log everything into the new H-layer
            corelay_id      (   ihl_youngest, i_column) = ihl_lastid
            corelay_irec    (   ihl_youngest, i_column) =
     &                              sedfil_nrecs(jp_ncout)
            corelay_solids  (:, ihl_youngest, i_column) = ytransfer(:)
            corelay_burtime (   ihl_youngest, i_column) = atime

#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
            IF (i_column == icol_dbg) THEN
              WRITE(jp_coldbg,*) 'CORELAY created -- COREstack now'
              ihl = ihl_youngest
              DO
                WRITE(jp_coldbg,*) ihl, corelay_id(ihl, i_column),
     &                              corelay_irec(ihl, i_column),
     &                              corelay_burtime(ihl, i_column)
                IF (ihl == ihl_oldest) EXIT
                ihl = IHL_DOWN(ihl)
              ENDDO
            ENDIF
#endif
#endif
#endif

                                    ! Finally, take the matter out of the
                                    ! Y-layer and adjust the CORELAY mass balance

            ysolid(:) = ysolid(:) - ytransfer(:)

            corelay_solids_total_imo(:,i_column)
     &         = corelay_solids_total_imo(:,i_column) + ytransfer(:)

#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
            IF (i_column == icol_dbg) THEN
              WRITE(jp_coldbg,*) ' transfer out:1126: ', ytransfer
              WRITE(jp_coldbg,*) ' ihl_bot/top: ',
     &                              ihl_oldest, ihl_youngest
              WRITE(jp_coldbg,*) ' ysolid_n: ', ysolid
            ENDIF
#endif
#endif
#endif


                                    ! Prepare the next round: update YL_SolidsVol
                                    ! and check if the CORE LAYer stack is not
                                    ! full and needs to be archived.

            YL_SolidsVol = SUM(ysolid(:)*apsv(:))


          ENDDO burial


                                    ! All done. Update ysolid(:) in
                                    ! MOD_SEAFLOOR_CENTRAL.
          CALL SAVE_COLUMN(i_column, iflag, YSOLID = ysolid)
          IF (iflag /= 0) THEN
            WRITE(jp_stderr, cfmt_modprocname_a) 'Warning'
            WRITE(jp_stderr, cfmt_a)
     &               '[1144] Error trapped at SAVE_COLUMN - Continuing.'
          ENDIF


        ENDIF

                                    ! Finally update all the pointers,
                                    ! IDs and layer numbers
                                    ! N.B.: sedfil_nrecs already done
        corelay_ihl_youngest(i_column) = ihl_youngest
        corelay_ihl_oldest  (i_column) = ihl_oldest
        corelay_lastid      (i_column, jp_ncout) = ihl_lastid

        erodlay_nlay        (i_column) = iel_nlay


      END DO all_columns


#ifdef ALLOW_MPI
      IF (.NOT. l_file_is_mine) THEN
        CALL MPI_SEND(jp_mpimsg_recv_nrecs, 1, MPI_INTEGER,
     &                              jp_exeproc_ncio, jp_mpitag_reqio,
     &                              i_mycomm, iflag)
        CALL MPI_SEND(sedfil_nrecs(jp_ncout), 1, MPI_INTEGER,
     &                              jp_exeproc_ncio, jp_mpitag_data0,
     &                              i_mycomm, iflag)
      ENDIF
      CALL MSC_QUEUE_PROCESS
      CALL MSC_QUEUE_TERMINATE
#endif


#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
      WRITE(jp_coldbg,'(1X, A, I0, A)')
     &   '[REACLAY_X_CORELAY] Column ', icol_dbg, ' @end: '
      CALL GET_COLUMN(icol_dbg, iflag, YSOLID = ysolid)
      WRITE(jp_coldbg,*)
     &   atime,
     &   corelay_ihl_youngest(icol_dbg), corelay_ihl_oldest(icol_dbg),
     &   ysolid
#endif
#endif
#endif


      RETURN


!======================================================================
      END SUBROUTINE REACLAY_X_CORELAY
!======================================================================




!======================================================================
      SUBROUTINE CORELAY_PACKMASS(i_column, n_packlay, atime,
     &                              psolid, cpmflag)
!======================================================================


      USE mod_materialcharas
      USE mod_milieucharas,         ONLY: xphi
      USE mod_rreac,                ONLY: CORRECT4DECAY


      IMPLICIT NONE


! Dummy argument variables
! ========================

! i_column  : column ID
! n_packlay : number of historical layers to lump together
!             for the summing
! psolid    : mass of solids in the pack of layers
! atime     : time at which the pack of layers is requested
!             (for correction of radioactive decay)
! cpmflag   : status flag; characteristic values on exit are
!             -2 = invalid i_column (< 0 or > n_grid_seafloor)
!             -1 = invalid n_packlay (< 0 or > nhl)
!              0 = All normal
!              n>0 ==> n layers missing and substituted by clay

      INTEGER,          INTENT(IN)  :: i_column
      INTEGER,          INTENT(IN)  :: n_packlay
      DOUBLE PRECISION, INTENT(IN)  :: atime
      INTEGER,          INTENT(OUT) :: cpmflag

      DOUBLE PRECISION, DIMENSION(1:nsolid), INTENT(OUT) :: psolid


! Local variables
! ===============

      INTEGER :: i
      INTEGER :: ihl_youngest, ihl_oldest, ihl_i
      INTEGER :: n_filled, ncorelay_recs
      INTEGER :: istatus

      DOUBLE PRECISION                    :: datime
      DOUBLE PRECISION, DIMENSION(nsolid) :: dysolid

#ifdef DEBUG
      CHARACTER(LEN=*), PARAMETER ::
     &  cfmt_dbg_a = '("[mod_sedcore.F/CORELAY_PACKMASS] ' //
     &                'debug: ", A)'
#endif


! End of declarations
! ===================


      IF (.NOT. mod_sedcore_setupdone) THEN
        WRITE(jp_stderr,*) '[mod_sedcore.F/CORELAY_PACKMASS]: ' //
     &    'Calling SETUP_SEDCORE_SYSTEM. Using pure internal setup.'
        CALL SETUP_SEDCORE_SYSTEM()
      ENDIF

                                    ! First check validity of INTENT(IN)
                                    ! arguments:
                                    ! - valid column number?
      IF ((i_column < 1) .OR. (i_column > ngrid)) THEN
        psolid(:) = 0.0D+00
        cpmflag = -2
        RETURN
      ENDIF

                                    ! - valid number of layers?
      IF ((n_packlay < 1) .OR. (n_packlay > nhl)) THEN
        psolid(:) = 0.0D+00
        cpmflag = -1
        RETURN
      ENDIF

                                    ! INTENT(IN) arguments are valid: proceed.

                                    ! Initialise INTENT(OUT) arguments:
      psolid(:) = 0.0D+00
      cpmflag = n_packlay


                                    ! Check if there are enough layers
                                    ! (minimum n_packlay) in the stack
      n_filled = CORELAY_NFILLED_ICOL(i_column)

      IF (n_filled < n_packlay) THEN
                                    ! if not, fill up first.
#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
        IF (i_column == icol_dbg) THEN
          WRITE(jp_coldbg,*)
     &       'CORELAY buffer low in CORELAY_PACKMASS ' //
     &       '-> SEDFIL_PULLRECORDS'
        ENDIF
#endif
#endif
#endif
        ncorelay_recs = nhl - n_filled
        CALL SEDFIL_PULLRECORDS(jp_ncout, i_column, ncorelay_recs,
     &                               istatus)
      ENDIF


      ihl_youngest = corelay_ihl_youngest(i_column)
      ihl_oldest   = corelay_ihl_oldest(i_column)

                                    ! Notice: if RESTORE_CORELAYBOTTOM has
                                    ! not been able to find any layers to
                                    ! fill up, then ihl_youngest could
                                    ! now still be (PT_TOP_EMPTYCORE)!

      IF (ihl_youngest /= jp_pt_top_emptycore) THEN
                                    ! There is at least one layer

        ihl_i   = ihl_youngest      ! ihl_i  : index of current layer


        DO i = 1, n_packlay

          dysolid(:) = corelay_solids(1:nsolid, ihl_i, i_column)
          datime = atime - corelay_burtime(ihl_i, i_column)

#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_CORRECT4DECAY
          WRITE(jp_stddbg, cfmt_dbg_a) 'Requesting CORRECT4DECAY'
          WRITE(jp_stddbg, '("Layer, Burial Time, Erosion Time: ")',
     &                              ADVANCE="NO")
          WRITE(jp_stddbg, *)
     &      ihl_i, corelay_burtime(ihl_i, i_column), atime
          WRITE(jp_stddbg, '("Contents buried: ")', ADVANCE="NO")
          WRITE(jp_stddbg, *) dysolid
#endif
#endif
#endif
          CALL CORRECT4DECAY(datime, dysolid)
#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_CORRECT4DECAY
          WRITE(jp_stddbg, '("Contents eroded: ")', ADVANCE="NO")
          WRITE(jp_stddbg, *) dysolid
#endif
#endif
#endif

          psolid(:) = psolid(:) + dysolid

          cpmflag = cpmflag - 1     ! one more layer accounted for

                                    ! Update ihl_i value for next step
          IF (ihl_i /= ihl_oldest) THEN

            ihl_i = IHL_DOWN(ihl_i)

          ELSE                      ! If ihl_i = ihl_oldest, then we
                                    ! have just processed the
            EXIT                    ! most ancient layer that was
                                    ! (still) existing in the stack.

          ENDIF

        ENDDO

      ENDIF

                                    ! cpmflag now holds the number of
                                    ! layers that we could not include
                                    ! in the request (because there
                                    ! were not enough of them).

      IF (cpmflag /= 0) THEN
         psolid(jf_mud) = psolid(jf_mud)
     &    + (da_gpd_dcorelay*(1.0D+00-xphi(idnb))
     &       /apsv(jf_mud)) * DBLE(cpmflag)
      ENDIF


      RETURN


!======================================================================
      END SUBROUTINE CORELAY_PACKMASS
!======================================================================




!======================================================================
      SUBROUTINE CORELAY_NLAY2ERODE_CLEAR
!======================================================================


      IMPLICIT NONE


! Local variables
! ===============

      CHARACTER(LEN=*), PARAMETER ::
     &  cfmt_info_a = '("[mod_sedcore.F/CORELAY_NLAY2ERODE_CLEAR] ' //
     &                'info: ", A)'


! End of declarations
! ===================


      IF (.NOT. mod_sedcore_setupdone) THEN
        WRITE(jp_stderr,cfmt_info_a) 'MOD_SEDCORE not yet set up.'
        WRITE(jp_stderr,'(A)')
     &    'Calling SETUP_SEDCORE_SYSTEM. Using pure internal setup.'
        CALL SETUP_SEDCORE_SYSTEM()
      ENDIF


                                    ! All OK - proceed
      corelay_nlay2erode(:) = 0


      RETURN


!======================================================================
      END SUBROUTINE CORELAY_NLAY2ERODE_CLEAR
!======================================================================




!======================================================================
      SUBROUTINE CORELAY_NLAY2ERODE_SET(i_column, n_lay2erode)
!======================================================================


      IMPLICIT NONE


! Dummy argument variables
! ========================

! i_column  : column ID
! n_packlay : number of core layers in the column to schedule for erosion


      INTEGER, INTENT(IN)  :: i_column
      INTEGER, INTENT(IN)  :: n_lay2erode


! Local variables
! ===============

      CHARACTER(LEN=*), PARAMETER ::
     &  cfmt_err_a = '("[mod_sedcore.F/CORELAY_NLAY2ERODE_SET] ' //
     &               'error: ", A)'

      CHARACTER(LEN=*), PARAMETER ::
     &  cfmt_info_a = '("[mod_sedcore.F/CORELAY_NLAY2ERODE_SET] ' //
     &               'info: ", A)'


! End of declarations
! ===================


      IF (.NOT. mod_sedcore_setupdone) THEN
        WRITE(jp_stderr,cfmt_info_a) 'MOD_SEDCORE not yet set up.'
        WRITE(jp_stderr,'(A)')
     &    'Calling SETUP_SEDCORE_SYSTEM. Using pure internal setup.'
        CALL SETUP_SEDCORE_SYSTEM()
      ENDIF

                                    ! First check validity of INTENT(IN)
                                    ! arguments:
                                    ! - valid column number?
      IF ((i_column < 1) .OR. (i_column > ngrid)) THEN
        WRITE(jp_stderr,cfmt_err_a, ADVANCE="NO")
     &    'incorrect column ID i_column='
        WRITE(jp_stderr,'(I0)') i_column
        WRITE(jp_stderr,'(A)') 'Leaving corelay_nlay2erode unchanged.'
        RETURN
      ENDIF

                                    ! - valid number of layers?
      IF (n_lay2erode < 0) THEN
        WRITE(jp_stderr,cfmt_err_a, ADVANCE="NO")
     &    'negative number of layers n_lay2erode='
        WRITE(jp_stderr,'(I0)') n_lay2erode
        WRITE(jp_stderr,'(A)') 'Leaving corelay_nlay2erode unchanged.'
        RETURN
      ENDIF

                                    ! All OK - proceed
      corelay_nlay2erode(i_column) = n_lay2erode


      RETURN


!======================================================================
      END SUBROUTINE CORELAY_NLAY2ERODE_SET
!======================================================================




!======================================================================
      INTEGER FUNCTION CORELAY_NFILLED_ICOL(i_column)
!======================================================================


      IMPLICIT NONE


      INTEGER, INTENT(IN) :: i_column

      INTEGER :: ihl_youngest, ihl_oldest



      ihl_youngest = corelay_ihl_youngest(i_column)
      ihl_oldest   = corelay_ihl_oldest(i_column)


      IF (ihl_youngest == jp_pt_top_emptycore)  THEN

        CORELAY_NFILLED_ICOL = 0

      ELSEIF (ihl_oldest <= ihl_youngest) THEN

        CORELAY_NFILLED_ICOL = ihl_youngest - ihl_oldest + 1

      ELSE

        CORELAY_NFILLED_ICOL = nhl - (ihl_oldest - ihl_youngest - 1)

      ENDIF


      RETURN


!======================================================================
      END FUNCTION CORELAY_NFILLED_ICOL
!======================================================================




!======================================================================
      INTEGER FUNCTION CORELAY_NFILLED_IHL(ihl_oldest, ihl_youngest)
!======================================================================


      IMPLICIT NONE


      INTEGER, INTENT(IN) :: ihl_oldest, ihl_youngest



      IF (ihl_youngest == jp_pt_top_emptycore)  THEN

        CORELAY_NFILLED_IHL = 0

      ELSEIF (ihl_oldest <= ihl_youngest) THEN

        CORELAY_NFILLED_IHL = ihl_youngest - ihl_oldest + 1

      ELSE

        CORELAY_NFILLED_IHL = nhl - (ihl_oldest - ihl_youngest - 1)

      ENDIF


      RETURN


!======================================================================
      END FUNCTION CORELAY_NFILLED_IHL
!======================================================================




!======================================================================
      SUBROUTINE SETUP_SEDCORE_SYSTEM(cfn_ncin_sedcore,
     &                                cfn_ncout_sedcore)
!======================================================================


      IMPLICIT NONE


      CHARACTER(LEN=*), INTENT(IN), OPTIONAL :: cfn_ncin_sedcore
      CHARACTER(LEN=*), INTENT(IN), OPTIONAL :: cfn_ncout_sedcore


      LOGICAL :: l_ncin_devnull
      LOGICAL :: l_ncout_devnull

      INTEGER :: i_action

      INTEGER :: n
      INTEGER :: icore, iflag

      INTEGER :: istatus

      CHARACTER(LEN=*), PARAMETER :: cfmt_modprocname_a = 
     &  '("[MOD_SEDCORE/SETUP_SEDCORE_SYSTEM]: ", A)'

      CHARACTER(LEN=*), PARAMETER :: cfmt_a = '(" - ", A)'


#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_modprocname_a) 'Start'
#endif

      IF (mod_sedcore_setupdone) THEN
#ifdef DEBUG
        WRITE(jp_stddbg, cfmt_a) 'Setup already done'
        WRITE(jp_stddbg, cfmt_modprocname_a) 'Return'
        WRITE(jp_stddbg, '()')
#endif
        RETURN
      ENDIF

      IF (PRESENT(cfn_ncin_sedcore)) THEN
        IF (cfn_ncin_sedcore == "/dev/null") THEN
          l_ncin_devnull = .TRUE.
        ELSE
          l_ncin_devnull = .FALSE.
        ENDIF
      ELSE
        l_ncin_devnull = .TRUE.
      ENDIF


      IF (PRESENT(cfn_ncout_sedcore)) THEN
        IF (cfn_ncout_sedcore == "/dev/null") THEN
          l_ncout_devnull = .TRUE.
        ELSE
          l_ncout_devnull = .FALSE.
        ENDIF
      ELSE
        l_ncout_devnull = .TRUE.
      ENDIF


      IF (l_ncout_devnull) THEN
        IF (l_ncin_devnull) THEN
          i_action = 0              ! internal buffer only
        ELSE
          i_action = 1              ! fill internal CORELAY buffer from
                                    ! NCIN file and close NCIN
        ENDIF
      ELSE
        IF (l_ncin_devnull) THEN
          i_action = 2              ! create new NCOUT file and start from scratch
        ELSE
          IF (cfn_ncin_sedcore == cfn_ncout_sedcore) THEN
            i_action = 3            ! append new data to existing file
          ELSE
            i_action = 4            ! create new NCOUT, pre-fill with minimum
                                    ! data from NCIN, and close NCIN
          ENDIF
        ENDIF

      ENDIF

      !  i_action  open NCIN  open NCOUT
      !  0         N          N
      !  1         Y          N
      !  2         N          Y
      !  3         Y          Y=
      !  4         Y          Y

      SELECT CASE(i_action)
      CASE(0)
#ifdef DEBUG
        WRITE(jp_stddbg, cfmt_a)
     &    'Configuring MOD_SEDCORE with internal buffer only, ' //
     &    'initialized from scratch'
#endif
        sedfil_used(jp_ncin)  = .FALSE.
        sedfil_used(jp_ncout) = .FALSE.
        sedfil_internal       = .TRUE.
        CALL SEDCOREBUFFER_SETUP

      CASE(1)
#ifdef DEBUG
        WRITE(jp_stddbg, cfmt_a)
     &    'Configuring MOD_SEDCORE with internal buffer only, ' //
     &    'initialized from file "' // TRIM(cfn_ncin_sedcore) // '"'
#endif
        sedfil_used(jp_ncin)  = .TRUE.
        sedfil_used(jp_ncout) = .FALSE.
        sedfil_internal       = .TRUE.
        CALL SEDCOREBUFFER_SETUP

        CALL SEDFIL_OPEN(cfn_ncin_sedcore)
        DO icore = 1, nsedcol_ncfile
          n = nhl
          CALL SEDFIL_PULLRECORDS(jp_ncin, icore, n, iflag)
        ENDDO

        CALL SEDFIL_CLOSE(jp_ncin)


      CASE(2)
#ifdef DEBUG
        WRITE(jp_stddbg, cfmt_a)
     &    'Configuring MOD_SEDCORE with new NCOUT_SEDCORE file ' //
     &    '"' // TRIM(cfn_ncout_sedcore) // '"'
#endif
        sedfil_used(jp_ncin)  = .FALSE.
        sedfil_used(jp_ncout) = .TRUE.
        sedfil_internal       = .FALSE.
        CALL SEDCOREBUFFER_SETUP

        CALL SEDFIL_CREATE(cfn_ncout_sedcore)


      CASE(3)
#ifdef DEBUG
        WRITE(jp_stddbg, cfmt_a)
     &    'Configuring MOD_SEDCORE to append core data to ' //
     &    ' file "' // TRIM(cfn_ncout_sedcore) // '".'
#endif
        sedfil_used(jp_ncin)  = .TRUE.
        sedfil_used(jp_ncout) = .TRUE.
        sedfil_internal       = .FALSE.
        CALL SEDCOREBUFFER_SETUP

        CALL SEDFIL_OPEN(cfn_ncin_sedcore, 'APPEND')


      CASE(4)
#ifdef DEBUG
        WRITE(jp_stddbg, cfmt_a)
     &    'Configuring MOD_SEDCORE with new NCOUT_SEDCORE file ' //
     &    '"' // TRIM(cfn_ncout_sedcore) // '" to continue ' //
     &    '"' // TRIM(cfn_ncin_sedcore) // '".'
#endif
        sedfil_used(jp_ncin)  = .TRUE.
        sedfil_used(jp_ncout) = .TRUE.
        sedfil_internal       = .FALSE.
        CALL SEDCOREBUFFER_SETUP

        CALL SEDFIL_OPEN(cfn_ncin_sedcore)
        CALL SEDFIL_CREATE(cfn_ncout_sedcore)
        CALL SEDFIL_NCIN_TOP_TO_NCOUT_BOTTOM
        CALL SEDFIL_CLOSE(jp_ncin)


      END SELECT


      mod_sedcore_setupdone = .TRUE.

#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_modprocname_a) 'End'
      WRITE(jp_stddbg, '()')
#endif


      RETURN


!======================================================================
      END SUBROUTINE SETUP_SEDCORE_SYSTEM
!======================================================================




!======================================================================
      SUBROUTINE SEDFIL_CREATE(cfn_ncout_sedcore)
!======================================================================

! Opens a sediment file names <file_name>.
! Makes a reservation for a logical unit number and then
! - either creates and sets up a new one (default):
!   * initialises the pointer values and basic counters
! - or opens an existing one (action="READWRITE", or "APPEND")
!   to append new data:
!   * initialises the pointer values and basic counters
!     to reflect the state of the file
!   * does not read in any layer content (CORELAY remains empty).


      USE mod_netcdfinc,            ONLY: NF_CREATE, NF_CLOBBER,
     &                                    NF_DEF_DIM, NF_DEF_VAR,
     &                                    NF_PUT_ATT_INT,
     &                                    NF_PUT_ATT_DOUBLE,
     &                                    NF_PUT_ATT_TEXT,
     &                                    NF_PUT_VAR1_INT,
     &                                    NF_ENDDEF, NF_SYNC,
     &                                    NF_GLOBAL, NF_UNLIMITED,
     &                                    NF_INT, NF_DOUBLE,
     &                                    NF_MAX_NAME, NF_NOERR,
     &                                    HANDLE_ERRORS
      USE mod_netcdfparam


      IMPLICIT NONE


      CHARACTER(LEN=*), INTENT(IN) :: cfn_ncout_sedcore


! Local variables
! ===============

      INTEGER :: iaction, imode, ncid, istatus
      INTEGER :: i, isedcol_global
      INTEGER :: j, jsolid, jcompo
      INTEGER :: jsolid_xref, jcompo_xref
      INTEGER :: len_coldim, len_laydim


      CHARACTER(LEN = NF_MAX_NAME) :: c_varname
      INTEGER                      :: nlen

      CHARACTER(LEN=*), PARAMETER  ::
     &  c_fmterr_a = '("[mod_sedcore.F/SEDFIL_CREATE] error: ", A)'
      CHARACTER(LEN=*), PARAMETER  ::
     &  c_fmtwar_a = '("[mod_sedcore.F/SEDFIL_CREATE] warning: ", A)'


! ===================
! End of declarations
! ===================


! Check if the internal configuration has already been activated
! If so, abandon here.

      IF (sedfil_internal) THEN

        WRITE(jp_stderr, c_fmtwar_a)
     &    'internal-only configuration without external file activated'
        WRITE(jp_stderr, '(A)') 'ignoring request to open a file'
        RETURN

      ENDIF


! Check if no NCOUT SEDCORE file is currently open: if so, abort

      IF (sedfil_isopen(jp_ncout)) THEN

        WRITE(jp_stderr, c_fmterr_a)
     &    'SEDCORE_NCOUT already open. Please close the open one first.'
        CALL ABORT_MEDUSA()

      ENDIF


        !-----------------------------------------------
        ! Create the data file and put global attributes
        !-----------------------------------------------

        IF (l_file_is_mine) THEN

          istatus = NF_CREATE(TRIM(cfn_ncout_sedcore), NF_CLOBBER, ncid)
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


                                    ! File format version
          istatus = NF_PUT_ATT_INT(ncid, NF_GLOBAL,
     &                              cn_globatt_filefmt,
     &                              NF_INT, 1, iv_globatt_filefmt)
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        ENDIF


        !---------------------------------
        ! Define dimensions and variables
        !---------------------------------

        IF (l_file_is_mine) THEN

                                    ! Columns
          istatus = NF_DEF_DIM(ncid, ddn_col, nsedcol_ncfile,
     &                              dim_col(jp_ncout))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
          istatus = NF_DEF_VAR(ncid, vsn_col, NF_INT, 1,
     &                              dim_col(jp_ncout), id_col(jp_ncout))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


                                    ! Layers
          istatus = NF_DEF_DIM(ncid, ddn_lay, NF_UNLIMITED,
     &                              dim_lay(jp_ncout))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
          istatus = NF_DEF_VAR(ncid, vsn_lay,
     &                              NF_INT, 1,
     &                              dim_lay(jp_ncout), id_lay(jp_ncout))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


                                    ! Define variables for
                                    !  - number of CORELAYs per core
                                    !    (set by SEDFIL_CLOSE)
          istatus = NF_DEF_VAR(ncid, cp_nbcorelays,
     &                              NF_INT, 1, dim_col(jp_ncout),
     &                              id_nbcorelays(jp_ncout))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    !  - number of actual records per core
                                    !    (set by SEDFIL_CLOSE)
          istatus = NF_DEF_VAR(ncid, cp_nbrecs,
     &                              NF_INT, 1, dim_col(jp_ncout),
     &                              id_nbrecs(jp_ncout))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    !  - Irec of Topmost CORELAY
                                    !    (set by SEDFIL_CLOSE)
          istatus = NF_DEF_VAR(ncid, cp_itcorelay,
     &                              NF_INT, 1, dim_col(jp_ncout),
     &                              id_itcorelay(jp_ncout))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    !  - Irec of Last written ERODLAY
                                    !    (set by SEDFIL_CLOSE)
          istatus = NF_DEF_VAR(ncid, cp_ilerodlay,
     &                              NF_INT, 1, dim_col(jp_ncout),
     &                              id_ilerodlay(jp_ncout))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    !  - Layer ID of the last layer created
                                    !    in core (CORELAY or ERODLAY, but not phantom)
          istatus = NF_DEF_VAR(ncid, cp_ilastlayid,
     &                              NF_INT, 1, dim_col(jp_ncout),
     &                              id_ilastlayid(jp_ncout))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    !  - Core ID of a layer
          istatus = NF_DEF_VAR(ncid, cp_coreid,
     &                              NF_INT, 1, dim_lay(jp_ncout),
     &                              id_coreid(jp_ncout))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    !  - Layer ID in core (CORELAY or ERODLAY)
          istatus = NF_DEF_VAR(ncid, cp_layid,
     &                              NF_INT, 1, dim_lay(jp_ncout),
     &                              id_layid(jp_ncout))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    !  - Irec of the PRECeeding LAYer:
                                    !    - for a CORELAY, this is the next
                                    !      CORELAY underneath
                                    !    - for an ERODLAY, this is the
                                    !      last ERODLAY stored before
          istatus = NF_DEF_VAR(ncid, cp_ipreclay,
     &                              NF_INT, 1, dim_lay(jp_ncout),
     &                              id_ipreclay(jp_ncout))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    !  - burial time
          istatus = NF_DEF_VAR(ncid, cp_burtime,
     &                              NF_DOUBLE, 1, dim_lay(jp_ncout),
     &                              id_burtime(jp_ncout))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
          istatus = NF_PUT_ATT_DOUBLE(ncid, id_burtime(jp_ncout),
     &                              '_FillValue', NF_DOUBLE, 1,
     &                              dp_neverbefore)
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    !  - erosion time
          istatus = NF_DEF_VAR(ncid, cp_erotime,
     &                              NF_DOUBLE, 1, dim_lay(jp_ncout),
     &                              id_erotime(jp_ncout))
          istatus = NF_PUT_ATT_DOUBLE(ncid, id_erotime(jp_ncout),
     &                              '_FillValue', NF_DOUBLE, 1,
     &                              dp_notyet)
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    !  - hcont
          DO j = 1, nsolid

            jcompo = jf_to_io(j)

            c_varname = NCVARNAME_COMPO(jcompo, cp_prefix_hcont, nlen)

            istatus = NF_DEF_VAR(ncid, c_varname(1:nlen),
     &                              NF_DOUBLE, 1, dim_lay(jp_ncout),
     &                              id_hcont(j,jp_ncout))
            IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

            c_varname = TRIM(vnn(jcompo)) // ' content'
            nlen  = LEN_TRIM(c_varname)
            istatus = NF_PUT_ATT_TEXT(ncid, id_hcont(j,jp_ncout),
     &                              'long_name', nlen, c_varname)
            IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

            c_varname = TRIM(vcn(jcompo))
            nlen  = LEN_TRIM(c_varname)
            istatus = NF_PUT_ATT_TEXT(ncid, id_hcont(j,jp_ncout),
     &                              'class', nlen, c_varname)
            IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

          ENDDO


          DO j = 1, nptsolid

            jsolid      = jpf_to_if(j)
            jsolid_xref = jpf_to_ifm(j)
            jcompo_xref = jf_to_io(jsolid_xref)

            c_varname = NCVARNAME_COMPO(jcompo_xref, cp_prefix_hcont,
     &                              nlen)

            istatus = NF_PUT_ATT_TEXT(ncid, id_hcont(jsolid,jp_ncout),
     &                              'pt_xref', nlen, c_varname)
            IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

          ENDDO


        ENDIF


        !-----------------
        ! End define mode
        !-----------------

        IF (l_file_is_mine) THEN

          istatus = NF_ENDDEF(ncid)
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        ENDIF


        !-------------------------------
        ! Set coordinate variable values
        !-------------------------------

        IF (l_file_is_mine) THEN

                                    ! Set 'col' coordinate variable
                                    ! equal to its index value
#ifdef ALLOW_MPI
          IF (l_oneio4many) THEN
            DO i = 1, nsedcol_ncfile
              isedcol_global = i
              istatus = NF_PUT_VAR1_INT(ncid, id_col(jp_ncout), i,
     &                              isedcol_global)
              IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
            ENDDO
          ELSE
            DO i = 1, nsedcol_ncfile
              isedcol_global = i + ioffset_sedcol_pproc(i_myrank)
              istatus = NF_PUT_VAR1_INT(ncid, id_col(jp_ncout), i,
     &                              isedcol_global)
              IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
            ENDDO
          ENDIF
#else
          DO i = 1, nsedcol_ncfile
            isedcol_global = i
            istatus = NF_PUT_VAR1_INT(ncid, id_col(jp_ncout), i,
     &                              isedcol_global)
            IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
          ENDDO
#endif
        ENDIF

        IF (l_file_is_mine) THEN
          istatus = NF_SYNC(ncid)
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        ENDIF


        sedflay_nlay   (:, jp_ncout) = 0
        sedflay_nrecs  (:, jp_ncout) = 0
        sedflay_irectop(:, jp_ncout) = 0
        sedflay_irecero(:, jp_ncout) = 0

        sedfil_nrecs(jp_ncout) = 0

        sedfil_ncid(jp_ncout) = ncid


        sedfil_isopen(jp_ncout) = .TRUE.


      RETURN


!======================================================================
      END SUBROUTINE SEDFIL_CREATE
!======================================================================




!======================================================================
      SUBROUTINE SEDFIL_OPEN(cfn_nc_sedcore, sedfil_action)
!======================================================================

! Opens an existing SEDCORE file named cfn_nc_sedcore.
! - either only to read
!   * initialises the pointer values and basic counters
! - or to append new data (sedfil_action="READWRITE", or "APPEND")
!   * initialises the pointer values and basic counters
!     to reflect the state of the file
!   * does not read in any layer content (CORELAY remains empty).


      USE mod_netcdfinc,            ONLY: NF_OPEN,
     &                                    NF_INQ_DIMID, NF_INQ_DIMLEN,
     &                                    NF_INQ_VARID,
     &                                    NF_WRITE, NF_NOWRITE,
     &                                    NF_MAX_NAME, NF_NOERR,
     &                                    HANDLE_ERRORS
      USE mod_netcdfparam


      IMPLICIT NONE


      CHARACTER(LEN=*), INTENT(IN)           :: cfn_nc_sedcore
      CHARACTER(LEN=*), INTENT(IN), OPTIONAL :: sedfil_action


! Local variables
! ===============

      INTEGER :: iaction, imode, iplug, ncid, istatus
      INTEGER :: i, isedcol_global
      INTEGER :: j, jcompo

      INTEGER :: len_coldim, len_laydim

      CHARACTER(LEN = NF_MAX_NAME) :: c_varname
      INTEGER                      :: nlen




      CHARACTER(LEN=*), PARAMETER  ::
     &  c_fmterr_a = '("[mod_sedcore.F/SEDFIL_OPEN] error: ", A)'
      CHARACTER(LEN=*), PARAMETER  ::
     &  c_fmtwar_a = '("[mod_sedcore.F/SEDFIL_OPEN] warning: ", A)'


! ===================
! End of declarations
! ===================


! Set the action selector

      IF (PRESENT(sedfil_action)) THEN

        SELECT CASE(sedfil_action)
        CASE('READWRITE', 'APPEND')
          imode = NF_WRITE          ! Open an existing file, and
                                    ! prepare to append data do it
          iplug = jp_ncout          ! => plug the file onto NCOUT


        CASE DEFAULT
          imode   = NF_NOWRITE      ! Default action: open for reading only
          iplug = jp_ncin           ! => plug the file onto NCIN

        END SELECT

      ELSE

        imode = NF_NOWRITE          ! Default action: open for reading only
        iplug = jp_ncin             ! => plug the file onto NCIN

      ENDIF


      SELECT CASE(iplug)
      CASE(jp_ncin)
        IF (sedfil_isopen(iplug)) THEN
          WRITE(jp_stderr, c_fmterr_a)
     &      'NCIN_SEDCORE file already open -- aborting'
          CALL ABORT_MEDUSA()
        ENDIF

      CASE(jp_ncout)
        IF (sedfil_isopen(iplug)) THEN
          WRITE(jp_stderr, c_fmterr_a)
     &      'NCOUT_SEDCORE file already open -- aborting'
          CALL ABORT_MEDUSA()
        ENDIF

      END SELECT



        !--------------------
        ! Open the data file
        !--------------------

        IF (l_file_is_mine) THEN

          istatus = NF_OPEN(TRIM(cfn_nc_sedcore), imode, ncid)
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

        ENDIF


        !----------------------------------------------------
        ! Get dimension id's and lengths + consistency check
        !----------------------------------------------------

        IF (l_file_is_mine) THEN

                                    ! Columns
          istatus = NF_INQ_DIMID(ncid, ddn_col, dim_col(iplug))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
          istatus = NF_INQ_DIMLEN(ncid, dim_col(iplug), len_coldim)
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
          IF (len_coldim /= nsedcol_ncfile) THEN
            WRITE(jp_stderr, c_fmterr_a)
     &                              'incompatible number of columns'
            WRITE(jp_stderr, '(" - expected: ", I0)') nsedcol_ncfile
            WRITE(jp_stderr, '(" - found:    ", I0)') len_coldim
            WRITE(jp_stderr, '("Aborting!")')
            CALL ABORT_MEDUSA()
          ENDIF

          istatus = NF_INQ_VARID(ncid, vsn_col, id_col(iplug))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    ! Layers
          istatus = NF_INQ_DIMID(ncid, ddn_lay, dim_lay(iplug))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
          istatus = NF_INQ_DIMLEN(ncid, dim_lay(iplug), len_laydim)
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
          istatus = NF_INQ_VARID(ncid, vsn_lay, id_lay(iplug))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


                                    ! Data variables:

                                    ! 1. General core characteristics

                                    !  - number of corelay's
          istatus = NF_INQ_VARID(ncid, cp_nbcorelays,
     &                              id_nbcorelays(iplug))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    !  - number of actual records
          istatus = NF_INQ_VARID(ncid, cp_nbrecs,
     &                              id_nbrecs(iplug))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    !  - irec of topmost CORELAY
          istatus = NF_INQ_VARID(ncid, cp_itcorelay,
     &                              id_itcorelay(iplug))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    !  - irec of last written ERODLAY
          istatus = NF_INQ_VARID(ncid, cp_ilerodlay,
     &                              id_ilerodlay(iplug))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    !  - ilay of last physical layer
          istatus = NF_INQ_VARID(ncid, cp_ilastlayid,
     &                              id_ilastlayid(iplug))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


                                    ! 2. Layer related data

                                    !  - Core ID of a layer
          istatus = NF_INQ_VARID(ncid, cp_coreid,
     &                              id_coreid(iplug))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    !  - Layer ID in a core
          istatus = NF_INQ_VARID(ncid, cp_layid,
     &                              id_layid(iplug))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    !  - irec of the preceeding layer
          istatus = NF_INQ_VARID(ncid, cp_ipreclay,
     &                              id_ipreclay(iplug))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    !  - burial time
          istatus = NF_INQ_VARID(ncid, cp_burtime,
     &                              id_burtime(iplug))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    !  - erosion time
          istatus = NF_INQ_VARID(ncid, cp_erotime,
     &                              id_erotime(iplug))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

                                    !  - hcont
          DO j = 1, nsolid

            jcompo = jf_to_io(j)

            c_varname = NCVARNAME_COMPO(jcompo, cp_prefix_hcont, nlen)

            istatus = NF_INQ_VARID(ncid, c_varname(1:nlen),
     &                              id_hcont(j,iplug))
            IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

          ENDDO

        ENDIF

                                    ! Read back pointers and counters
                                    ! - adjust the number of CORELAY layers
                                    !    still available in the file
                                    !    for core <icore>
        CALL MSC_NF_GET_C_INT(ncid, id_nbcorelays(iplug),
     &                              sedflay_nlay(:, iplug))

                                    ! - adjust the number of records already
                                    !    stored in the file for core <icore>
        CALL MSC_NF_GET_C_INT(ncid, id_nbrecs(iplug),
     &                              sedflay_nrecs(:, iplug))

                                    ! - record number of the youngest
                                    !   stored CORELAY
        CALL MSC_NF_GET_C_INT(ncid, id_itcorelay(iplug),
     &                              sedflay_irectop(:, iplug))

                                    ! - record number of the most
                                    !   recently stored ERODLAY
        CALL MSC_NF_GET_C_INT(ncid, id_ilerodlay(iplug),
     &                              sedflay_irecero(:, iplug))

                                    ! - adjust the last ID of the layers
                                    !    stored in the file for core <icore>
        CALL MSC_NF_GET_C_INT(ncid, id_ilastlayid(iplug),
     &                              corelay_lastid(:, iplug))


        IF (ANY(sedflay_irectop(:,iplug) == jp_pt_norecord) .OR.
     &       ANY(sedflay_irecero(:,iplug) == jp_pt_norecord)) THEN

          WRITE(jp_stderr, c_fmterr_a)
     &            'Some cores have not been correctly initialised'
          WRITE(jp_stderr, '(A)')
     &            'File "'//TRIM(cfn_nc_sedcore)//'" is not ' //
     &            'correctly formatted -- aborting.'
            CALL ABORT_MEDUSA()

         ENDIF

                                    ! Adjust the total number of records
         sedfil_nrecs(iplug)  = len_laydim

         sedfil_ncid(iplug)   = ncid

         sedfil_isopen(iplug) = .TRUE.


      RETURN


!======================================================================
      END SUBROUTINE SEDFIL_OPEN
!======================================================================



!======================================================================
      SUBROUTINE SEDFIL_NCIN_TOP_TO_NCOUT_BOTTOM
!======================================================================


      USE mod_indexparam,           ONLY: jf_mud, nsolid
      USE mod_gridparam,            ONLY: idnw, idnt, idnb,
     &                                    thetatop, thetabot
      USE mod_seafloor_central,     ONLY: GET_COLUMN
      USE mod_materialcharas,       ONLY: apsv


      IMPLICIT NONE


      DOUBLE PRECISION :: total_xm_fullmud
      DOUBLE PRECISION :: top_hcont_mud
      DOUBLE PRECISION, DIMENSION(idnw:idnb) :: xzdn
      DOUBLE PRECISION, DIMENSION(idnt:idnb) :: xphi
      DOUBLE PRECISION, DIMENSION(idnt:idnb) :: vxzdn_rcl_c
      DOUBLE PRECISION, DIMENSION(idnt:idnb) :: phis_rcl

      INTEGER :: icore, iflag
      INTEGER :: n
      INTEGER :: irec, irec_ncin, irec_ncout
      INTEGER :: irec_ncin_prec, irec_ncout_prec
      INTEGER :: sedfil_icore, icorelayid

      DOUBLE PRECISION :: burtime, erotime
      DOUBLE PRECISION, DIMENSION(nsolid) :: hsolid


! Check if the file is actually open (if not -- fatal error)

      IF (.NOT. (sedfil_isopen(jp_ncin) .AND.
     &           sedfil_isopen(jp_ncout))) THEN

        WRITE(jp_stderr,'(A)')
     &    '[mod_sedcore.F/SEDFIL_NCIN_TOP_TO_NCOUT_BOT]: ' //
     &    'the two files must be open:'

        WRITE(jp_stderr,'(A)', ADVANCE="NO") ' - NCIN file '
        IF (sedfil_isopen(jp_ncin)) THEN
          WRITE(jp_stderr, '(A)') 'is open'
        ELSE
          WRITE(jp_stderr, '(A)') 'is *not* open'
        ENDIF

        WRITE(jp_stderr,'(A)', ADVANCE="NO") ' - NCOUT file '
        IF (sedfil_isopen(jp_ncout)) THEN
          WRITE(jp_stderr, '(A)') 'is open'
        ELSE
          WRITE(jp_stderr, '(A)') 'is *not* open'
        ENDIF

        WRITE(jp_stderr,'(A)')
     &    'Aborting.'

        CALL ABORT_MEDUSA()

      ENDIF


#ifdef ALLOW_MPI
      CALL MSC_QUEUE_START

      IF (.NOT. l_file_is_mine) THEN
        CALL MPI_SEND(jp_mpimsg_send_nrecs, 1, MPI_INTEGER,
     &                              jp_exeproc_ncio, jp_mpitag_reqio,
     &                              i_mycomm, MPI_STATUS_IGNORE, iflag)
        CALL MPI_RECV(sedfil_nrecs(jp_ncout), 1, MPI_INTEGER,
     &                              jp_exeproc_ncio, jp_mpitag_data0,
     &                              i_mycomm, MPI_STATUS_IGNORE, iflag)
      ENDIF
#endif


#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_NCIN_TO_NCOUT
        WRITE(jp_stddbg, '(A)')
     &    '[mod_sedcore.F/SEDFIL_NCIN_TOP_TO_NCOUT_BOT]: ' //
     &    'Transferring records from NCIN to NCOUT'
#endif
#endif
#endif


      DO icore = 1, ngrid
                                    ! First determine the amount of clay
                                    ! ("mud") that is required to completely
                                    ! fill up REACLAY.

        CALL GET_COLUMN(i_column = icore, iflag = iflag,
     &           xzdn = xzdn(:), xphi= xphi(:))

        vxzdn_rcl_c(idnt) = (xzdn(idnt+1)-xzdn(idnt))/2.0D+00
        vxzdn_rcl_c(idnt+1:idnb-1) =
     =        (xzdn(idnt+2:idnb)-xzdn(idnt:idnb-2))/2.0D+00
        vxzdn_rcl_c(idnb) = (xzdn(idnb)-xzdn(idnb-1))/2.0D+00

        phis_rcl(idnt) =   (1.0D+00-thetatop) * (1.0D+00-xphi(idnt  ))
     &                   +       thetatop     * (1.0D+00-xphi(idnt+1))
        phis_rcl(idnt+1:idnb-1) =  (1.0D+00-xphi(idnt+1:idnb-1))
        phis_rcl(idnb) =         thetabot     * (1.0D+00-xphi(idnb-1))
     &                   + (1.0D+00-thetabot) * (1.0D+00-xphi(idnb  ))

        total_xm_fullmud = SUM(vxzdn_rcl_c(:)*phis_rcl(:))
     &                      / apsv(jf_mud)


                                    ! Scan the NCIN file to find out
                                    ! how many of the top layers are
                                    ! required to fill REACLAY with
                                    ! clay (i.e., with the "mud" component).
        top_hcont_mud = 0.0D+00
        n = 0
        irec_ncin = sedflay_irectop(icore, jp_ncin)

        DO WHILE(n < sedflay_nrecs(icore, jp_ncin))

          CALL SEDFIL_READ_RECORD(jp_ncin, irec_ncin,
     &                              sedfil_icore, icorelayid,
     &                              irec_ncin_prec,
     &                              burtime, erotime, hsolid)

          n = n + 1

          top_hcont_mud = top_hcont_mud  + hsolid(jf_mud)

          IF (top_hcont_mud > total_xm_fullmud) EXIT

          irec_ncin = irec_ncin_prec

        ENDDO


#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_NCIN_TO_NCOUT
        WRITE(jp_stddbg, '("Column ", I0, ": ", ' //
     &    'I0, " records of ", I0, " to read in (", 2E13.6, ")")')
     &    icore, n, sedflay_nrecs(icore, jp_ncin),
     &    top_hcont_mud, total_xm_fullmud
#endif
#endif
#endif

                                    ! Now read them in again, and write
                                    ! them out into the new NCOUT file
        irec_ncin = sedflay_irectop(icore, jp_ncin)

        DO irec = n, 1, -1

          CALL SEDFIL_READ_RECORD(jp_ncin, irec_ncin,
     &                              sedfil_icore, icorelayid,
     &                              irec_ncin_prec,
     &                              burtime, erotime, hsolid)

          irec_ncout = sedfil_nrecs(jp_ncout) + irec

          IF (irec == 1) THEN
            irec_ncout_prec = 0
          ELSE
            irec_ncout_prec = irec_ncout - 1
          ENDIF

          CALL SEDFIL_WRITE_RECORD(jp_ncout, irec_ncout,
     &                              sedfil_icore, icorelayid,
     &                              irec_ncout_prec,
     &                              burtime, erotime, hsolid)

          IF (irec > 1) irec_ncin = irec_ncin_prec

        ENDDO

c~                                     ! Store the record ID upon which
c~                                     ! the oldest transcribed layer
c~                                     ! rests upon
c~         sedflay_irecbot_prec(icore) = irec_ncin_prec

                                    ! New number of erodable layers
                                    ! in NCOUT for core <icore>
        sedflay_nlay(icore, jp_ncout)  = n
                                    ! New number of total recorded layers
                                    ! in NCOUT for core <icore>
        sedflay_nrecs(icore, jp_ncout) = n

                                    ! New total number of records in NCOUT
        sedfil_nrecs(jp_ncout) = sedfil_nrecs(jp_ncout) + n

                                    ! New topmost records in NCOUT
                                    ! for core <icore>
        IF (n /= 0) THEN
                                    ! sedflay_irectop(icore, jp_ncout) is
                                    ! set to the topmost record created
                                    ! in the NCOUT file
          sedflay_irectop(icore, jp_ncout) = sedfil_nrecs(jp_ncout)
        ELSE
                                    ! If no records were available:
          sedflay_irectop(icore, jp_ncout) = 0
        ENDIF

                                    ! Update the last CORELAY ID
                                    ! for the core.
        corelay_lastid(icore, jp_ncout) = corelay_lastid(icore, jp_ncin)


      ENDDO


#ifdef ALLOW_MPI
      IF (.NOT. l_file_is_mine) THEN
        CALL MPI_SEND(jp_mpimsg_recv_nrecs, 1, MPI_INTEGER,
     &                              jp_exeproc_ncio, jp_mpitag_reqio,
     &                              i_mycomm, iflag)
        CALL MPI_SEND(sedfil_nrecs(jp_ncout), 1, MPI_INTEGER,
     &                              jp_exeproc_ncio, jp_mpitag_data0,
     &                              i_mycomm, iflag)
      ENDIF

      CALL MSC_QUEUE_PROCESS
      CALL MSC_QUEUE_TERMINATE
#endif


      RETURN


!======================================================================
      END SUBROUTINE SEDFIL_NCIN_TOP_TO_NCOUT_BOTTOM
!======================================================================



!======================================================================
      SUBROUTINE SEDFIL_CLOSE(kplug)
!======================================================================

! - close the file without any write operations


      USE mod_netcdfinc,            ONLY: NF_CLOSE, NF_NOERR,
     &                                    HANDLE_ERRORS


      IMPLICIT NONE


      INTEGER, INTENT(IN) :: kplug


      INTEGER :: istatus

#ifdef DEBUG
      CHARACTER(LEN=*), PARAMETER ::
     &  cfmt_dbg_a = '("[mod_sedcore.F/SEDFIL_CLOSE] debug: ", A)'
#endif


! Return straight away if the file is used.

      IF (.NOT. sedfil_used(kplug)) THEN
        WRITE(jp_stderr,*) '[mod_sedcore.F/SEDFIL_CLOSE]: ' //
     &    'Request to close sediment file while ' //
     &    'sedfil_used(kplug) == .FALSE.'
        WRITE(jp_stderr,*) 'Ignoring the request and returning'
        RETURN
      ENDIF


! Check if the file is actually open (if not -- fatal error)

      IF (.NOT. sedfil_isopen(kplug)) THEN
        WRITE(jp_stderr,*) '[mod_sedcore.F/SEDFIL_CLOSE]: ' //
     &    'Trying to close a sediment file that is not open ' //
     &    '-- aborting!.'
        CALL ABORT_MEDUSA()
      ENDIF


      IF (l_file_is_mine) THEN
        istatus = NF_CLOSE(sedfil_ncid(kplug))
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
      ENDIF


      sedfil_isopen(kplug) = .FALSE.


      RETURN


!======================================================================
      END SUBROUTINE SEDFIL_CLOSE
!======================================================================




!======================================================================
      SUBROUTINE SEDFIL_FINALIZE
!======================================================================

! Finalizes the NCOUT file
! - write out the CORELAY and ERODLAY contents for all the cores;
! - add the "sealing" layers
! - close the file


      USE mod_netcdfinc,            ONLY: NF_CLOSE, NF_NOERR,
     &                                    HANDLE_ERRORS


      IMPLICIT NONE


      INTEGER :: icore
      INTEGER :: ilayertype
      INTEGER :: ncid, iflag, istatus

      INTEGER :: nrecs
      INTEGER :: irec, irec_prec

#ifdef DEBUG
      CHARACTER(LEN=*), PARAMETER ::
     &  cfmt_dbg_a = '("[mod_sedcore.F/SEDFIL_FINALIZE] debug: ", A)'
#endif

! Return straight away if no file is used.

      IF (.NOT. sedfil_used(jp_ncout)) THEN
        WRITE(jp_stderr,*) '[mod_sedcore.F/SEDFIL_FINALIZE]: ' //
     &    'Request to close sediment file while ' //
     &    'sedfil_used == .FALSE.'
        WRITE(jp_stderr,*) 'Ignoring the request and returning'
        RETURN
      ENDIF


! Check if there is an open file (if not -- fatal error)

      IF (.NOT. sedfil_isopen(jp_ncout)) THEN
        WRITE(jp_stderr,*) '[mod_sedcore.F/SEDFIL_FINALIZE]: ' //
     &    'Trying to close a NCOUT SEDCORE file file although ' //
     &    'none is open, despite sedfil_used == .TRUE. -- aborting!.'
        CALL ABORT_MEDUSA()
      ENDIF


! Empty the buffers for each core

#ifdef ALLOW_MPI
      CALL MSC_QUEUE_START
#endif

      DO icore = 1, ngrid

        nrecs = CORELAY_NFILLED_ICOL(icore)

#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_CLOSEFILE
        WRITE(jp_stddbg, cfmt_dbg_a, ADVANCE="NO") 'icore, nrecs = '
        WRITE(jp_stddbg, '(I0, ", ", I0)') icore, nrecs
#endif
#endif
#endif

        ilayertype = 3             ! both CORELAY and ERODLAY stacks

#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
        IF (icore == icol_dbg) THEN
          WRITE(jp_coldbg,*) 'Closing file -> SEDFIL_PUSHRECORDS'
        ENDIF
#endif
#endif
#endif
        CALL SEDFIL_PUSHRECORDS(icore, ilayertype, nrecs, iflag)

      ENDDO


#ifdef ALLOW_MPI
      CALL MSC_QUEUE_PROCESS
      CALL MSC_QUEUE_TERMINATE
#endif


#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_CLOSEFILE
      WRITE(jp_stddbg, cfmt_dbg_a, ADVANCE="NO")
     &  'sedfil_nrecs(jp_ncout) = '
      WRITE(jp_stddbg, '(I0)') sedfil_nrecs(jp_ncout)
      WRITE(jp_stddbg, cfmt_dbg_a, ADVANCE="NO")
     & 'sedflay_nrecs(:,jp_ncout) = '
      WRITE(jp_stddbg, *) sedflay_nrecs(:, jp_ncout)
      WRITE(jp_stddbg, cfmt_dbg_a, ADVANCE="NO")
     &  'SUM(sedflay_nrecs(:,jp_ncout)) = '
      WRITE(jp_stddbg, '(I0)') SUM(sedflay_nrecs(:, jp_ncout))
#endif
#endif
#endif

                                    ! Plug in NCOUT
      ncid = sedfil_ncid(jp_ncout)


      CALL MSC_NF_PUT_C_INT(ncid, id_nbcorelays(jp_ncout),
     &                              sedflay_nlay(:, jp_ncout))

      CALL MSC_NF_PUT_C_INT(ncid, id_nbrecs(jp_ncout),
     &                              sedflay_nrecs(:, jp_ncout))

      CALL MSC_NF_PUT_C_INT(ncid, id_ilastlayid(jp_ncout),
     &                              corelay_lastid(:, jp_ncout))

      CALL MSC_NF_PUT_C_INT(ncid, id_itcorelay(jp_ncout),
     &                              sedflay_irectop(:, jp_ncout))

      CALL MSC_NF_PUT_C_INT(ncid, id_ilerodlay(jp_ncout),
     &                              sedflay_irecero(:, jp_ncout))



! All done -- close the file
      IF (l_file_is_mine) THEN
        istatus = NF_CLOSE(ncid)
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
      ENDIF



      sedfil_isopen(jp_ncout) = .FALSE.

#ifdef ALLOW_MPI
      DEALLOCATE(iarr_mpisend_c)
      DEALLOCATE(iarr_mpirecv_c)
      DEALLOCATE(irec_mpirecv_c)
#endif

#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_dbg_a, ADVANCE="NO") 'erodlay_maxlevel = '
      WRITE(jp_stddbg, '(I0)') erodlay_maxlevel
#endif


      RETURN


!======================================================================
      END SUBROUTINE SEDFIL_FINALIZE
!======================================================================




!======================================================================
      SUBROUTINE SEDFIL_PUSHRECORDS(icore, ilayertype, ncorelay_recs,
     &                               iflag)
!======================================================================

! Writes to file the contents of the <nrecs> oldest CORELAY layer
! (or less if less are available) and all the ERODLAY layers.
! Upon return, <nrecs> will be set to the number of records actually
! written, which may be lower (possibly equal to 0) if there are less
! than the requested number of layers available in CORELAY

! SEDFIL_PUSHRECORDS relies on consistent corelay_ihl_youngest
! and corelay_ihl_oldest values!

! Upon return, iflag indicates the overall success of the operations:
! iflag = -2 : ERROR -- invalid core number in icore
! iflag = -1 : ERROR -- invalid <ilayertype>
! iflag =  0 : OK
! iflag >  0 : WARNING -- less than the requested number of CORELAY
!              records could not be written (<iflag> is equal to the number
!              of missing records.)

      USE mod_netcdfinc,            ONLY: NF_SYNC, NF_NOERR,
     &                                    HANDLE_ERRORS


      IMPLICIT NONE


! Dummy argument variables
! ========================

      INTEGER, INTENT(IN)    :: icore
      INTEGER, INTENT(IN)    :: ilayertype
      INTEGER, INTENT(INOUT) :: ncorelay_recs
      INTEGER, INTENT(OUT)   :: iflag


! Local variables
! ===============

      INTEGER :: irec, irec_prec
      INTEGER :: icorelay_id
      INTEGER :: nrecs_req, nrecs_max
      INTEGER :: ihl_youngest, ihl_oldest

      INTEGER :: ncid
      INTEGER :: sedfil_icore, sedfil_corelayid
      DOUBLE PRECISION :: sedfile_burtime, sedfil_erotime
      DOUBLE PRECISION, DIMENSION(nsolid) :: sedfil_solid

      INTEGER :: i, ihl, iel, nel
      INTEGER :: iflag_corelay

      INTEGER :: istatus


! End of declarations
! ===================

! If no sediment file is used, adjust counter and flag
! and return right away (w/o warning message), as if
! simply no records could have been written
      IF (.NOT. sedfil_used(jp_ncout)) THEN
                                    ! ncorelay_recs is ignored if
                                    ! only ERODLAY layers are to
                                    ! be saved [ilayertype == 2]
        IF (ilayertype == 2) ncorelay_recs = 0

                                    ! No records could be written
        iflag         = ncorelay_recs
        ncorelay_recs = 0

        RETURN

      ENDIF


! Check if there is an open file (if not -- fatal error)

      IF (.NOT. sedfil_isopen(jp_ncout)) THEN
        WRITE(jp_stderr,*) '[mod_sedcore.F/SEDFIL_PUSHRECORDS]: ' //
     &    'Trying to write to a sediment file although none is ' //
     &    'open, despite sedfil_used == .TRUE. -- aborting!.'
        CALL ABORT_MEDUSA()
      ENDIF


! Check validity of INTENT(IN) arguments

                                    ! - valid column number?
      IF ((icore < 1) .OR. (icore > ngrid)) THEN
        ncorelay_recs = 0          ! no records written
        iflag = -2                 ! Error!
        WRITE(jp_stderr,*) '[mod_sedcore.F/SEDFIL_PUSHRECORDS]: ' //
     &    'invalid <icore> :', icore
        CALL ABORT_MEDUSA()
        RETURN
      ENDIF

                                    ! - valid layertype ?
      IF ((ilayertype < 1) .OR. (ilayertype > 3)) THEN
        ncorelay_recs = 0          ! no records written
        iflag = -1                 ! Error!
        WRITE(jp_stderr,*) '[mod_sedcore.F/SEDFIL_PUSHRECORDS]: ' //
     &    'invalid <ilayertype> :', ilayertype
        CALL ABORT_MEDUSA()
        RETURN
      ENDIF


! INTENT(IN) arguments are valid: proceed

#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
      IF (icore == icol_dbg)
     &  WRITE(jp_coldbg, '("[SWR]: push ", I0, " records [", I0, "]")')
     &  ncorelay_recs, ilayertype
#endif
#endif
#endif

! Write out layers depending on the value of
! - ilayertype = 1 --> CORELAY layers (up to <nrecs>)
! - ilayertype = 2 --> ERODLAY layers (always all)
! - ilayertype = 3 --> CORELAY and ERODLAY layers


! First write out core layers

      iflag_corelay = ncorelay_recs ! preset iflag as if no CORELAY
                                    ! records could have been written

      IF ((ilayertype == 1) .OR. (ilayertype == 3)) THEN

                                    ! Number of records requested
        nrecs_req = ncorelay_recs
                                    ! Number of records available in the
                                    ! CORELAY stack

        nrecs_max = CORELAY_NFILLED_ICOL(icore)

                                    ! Effective number of records to write
        ncorelay_recs = MIN(nrecs_max, nrecs_req)

#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
        IF (icore == icol_dbg) THEN
          WRITE(jp_coldbg, '("[SWR]: ", I0, " CORELAY available")')
     &      nrecs_max
          WRITE(jp_coldbg, '("[SWR]: push request for ",I0," CORELAY")')
     &      nrecs_req
          WRITE(jp_coldbg, '("[SWR]: push planned for ",I0," CORELAY")')
     &      ncorelay_recs
        ENDIF
#endif
#endif
#endif

        IF (ncorelay_recs > 0) THEN ! If we are able to write
                                    ! out one record at least:

                                    !  - oldest and youngest CORELAY layers
                                    !    in the buffer
          ihl_oldest = corelay_ihl_oldest(icore)
          ihl_youngest = corelay_ihl_youngest(icore)

                                    !  - irec of the current top
                                    !    CORELAY layer in the file
          irec_prec = sedflay_irectop(icore, jp_ncout)

                                    !  - erosion time of CORELAY layers: never
          sedfil_erotime = dp_notyet

#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
          IF (icore == icol_dbg)
     &      WRITE(jp_coldbg, '("[SWR@beg]: ihl o/y = ", I0, "/", I0)')
     &        ihl_oldest , ihl_youngest
#endif
#endif
#endif

          DO i = 1, ncorelay_recs

            irec      = corelay_irec(ihl_oldest, icore)


#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
            IF (icore == icol_dbg)
     &        WRITE(jp_coldbg, '("[SWR]: CLR ", I0, " ontop of ", I0)')
     &          irec, irec_prec
#endif
#endif
#endif

            CALL SEDFIL_WRITE_RECORD(jp_ncout, irec,
     &             icore,  corelay_id(ihl_oldest, icore), irec_prec,
     &             corelay_burtime(   ihl_oldest, icore),
     &             sedfil_erotime,
     &             corelay_solids  (:, ihl_oldest, icore))

                                    ! Clear CORELAY data
            corelay_id     (   ihl_oldest, icore) = -1
            corelay_irec   (   ihl_oldest, icore) = jp_pt_norecord
            corelay_burtime(   ihl_oldest, icore) = dp_neverbefore
            corelay_solids (:, ihl_oldest, icore) = 0.0D+00


            IF (ihl_oldest == ihl_youngest) THEN
                                    ! If we have just written the data for the
                                    ! youngest layer in the CORELAY stack, then
                                    ! the core is empty, and we have done all here.
              ihl_oldest = jp_pt_top_emptycore
              ihl_youngest = jp_pt_top_emptycore
              EXIT

            ELSE
                                    ! else get the index of the next oldest layer
              ihl_oldest = IHL_UP(ihl_oldest)

            ENDIF

            irec_prec = irec

          ENDDO

                                    ! Save <irec> for the last record written
                                    ! to the file for the CORELAY stack
          sedflay_irectop(icore, jp_ncout) = irec

                                    ! increase the counter of records written
          sedflay_nrecs(icore, jp_ncout) =
     &      sedflay_nrecs(icore, jp_ncout) + ncorelay_recs

                                    ! increase the counter of CORELAY records
                                    ! available in the file
          sedflay_nlay(icore, jp_ncout) =
     &      sedflay_nlay(icore, jp_ncout) + ncorelay_recs

                                    ! update the pointers to the youngest and
                                    ! oldest layers in the stack
          corelay_ihl_oldest(icore)   = ihl_oldest
          corelay_ihl_youngest(icore) = ihl_youngest

#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
          IF (icore == icol_dbg)
     &      WRITE(jp_coldbg, '("[SWR@end]: ihl o/y = ", I0, "/", I0)')
     &        ihl_oldest , ihl_youngest
#endif
#endif
#endif

        ENDIF

                                    ! set iflag_corelay to the number of
                                    ! records requested that could not be
                                    ! written
        iflag_corelay = nrecs_req - ncorelay_recs

      ELSE

        ncorelay_recs = 0           ! if ilayertype == 2, ncorelay_recs is ignored
        iflag_corelay = 0

      ENDIF


! Write out eroded layers

      IF ((ilayertype == 2) .OR. (ilayertype == 3)) THEN

                                    ! We always completely empty the buffer
        nel = erodlay_nlay(icore)

#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
        IF (icore == icol_dbg)
     &    WRITE(jp_coldbg, '("[SWR]: erodlay n = ", I0)') nel
#endif
#endif
#endif

        IF (nel > 0) THEN          ! If there is anything at all to write out, then

                                    ! Get the irec to the current top
                                    ! ERODLAY layer in the file
          irec_prec = sedflay_irecero(icore, jp_ncout)

          DO iel = 1, nel

            irec      = erodlay_irec(iel, icore)

#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
            IF (icore == icol_dbg)
     &        WRITE(jp_coldbg, '("[SWR]: ELR ", I0, " ontop of ", I0)')
     &          irec, irec_prec
#endif
#endif
#endif

            CALL SEDFIL_WRITE_RECORD(jp_ncout, irec,
     &             icore,  erodlay_id(iel, icore), irec_prec,
     &             erodlay_burtime(   iel, icore),
     &             erodlay_erotime(   iel, icore),
     &             erodlay_solids (:, iel, icore))

               irec_prec = irec

          ENDDO


          erodlay_nlay   (      icore) = 0

          erodlay_id     (   :, icore) = -1
          erodlay_solids (:, :, icore) = 0.0D+00
          erodlay_burtime(   :, icore) = dp_neverbefore
          erodlay_erotime(   :, icore) = dp_neverbefore

          sedflay_irecero(      icore, jp_ncout) = irec

          sedflay_nrecs(icore, jp_ncout) =
     &      sedflay_nrecs(icore, jp_ncout) + nel

        ENDIF


      ENDIF


      iflag = iflag_corelay


      IF (l_file_is_mine) THEN
        istatus = NF_SYNC(sedfil_ncid(jp_ncout))
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
      ENDIF


      RETURN

!======================================================================
      END SUBROUTINE SEDFIL_PUSHRECORDS
!======================================================================




!======================================================================
      SUBROUTINE SEDFIL_PULLRECORDS(kplug, icore, ncorelay_recs, iflag)
!======================================================================

! Read the <ncorelay_recs> (or less if not enough are available) most
! recently saved records for core <icore> from the file and transfer
! their data into corelay_xxx structures at the positions underneath
! the oldest layer in CORELAY.
! Upon return, <ncorelay_recs> will be set to the number of records
! actually read in, which may be lower (possibly equal to 0) if there
! are less than the requested number of records in the file or not
! enough free slots in CORELAY.

! SEDFIL_PULLRECORDS relies on consistent corelay_ihl_youngest
! and corelay_ihl_oldest values!

! Upon return, iflag indicates the overall success of the operations:
! iflag = -2 : ERROR -- invalid core number in icore
! iflag = -3 : ERROR -- invalid number of records requested
! iflag = -4 : ERROR -- trying to read data from wrong core
!              (inconsistent sedfil)
! iflag =  0 : OK
! iflag >  0 : WARNING -- less than the requested number of CORELAY
!              records could not be restored (<iflag> is equal to the
!              number of missing records.)


      IMPLICIT NONE


! Dummy argument variables
! ========================

      INTEGER, INTENT(IN)    :: kplug
      INTEGER, INTENT(IN)    :: icore
      INTEGER, INTENT(INOUT) :: ncorelay_recs
      INTEGER, INTENT(OUT)   :: iflag


! Local variables
! ===============

      INTEGER :: i, irec, irec_prec
      INTEGER :: nrecs_req, nrecs_max, nlay_empty

      INTEGER :: ihl, ihl_youngest, ihl_oldest

      INTEGER :: sedfil_icore, sedfil_corelayid
      DOUBLE PRECISION :: sedfile_burtime, sedfil_erotime
      DOUBLE PRECISION, DIMENSION(nsolid) :: sedfil_solid


! End of declarations
! ===================


! If no sediment file is used, adjust counter and flag
! and return right away (w/o warning message), as if
! simply no records could have been read
      IF (.NOT. sedfil_used(kplug)) THEN
                                    ! No records could be restored
        iflag         = ncorelay_recs
        ncorelay_recs = 0

        RETURN

      ENDIF


! Check if the file to plug in is open (if not -- fatal error)

      IF (.NOT. sedfil_isopen(kplug)) THEN
        WRITE(jp_stderr,*) '[mod_sedcore.F/SEDFIL_PULLRECORDS]: ' //
     &    'Trying to read from a sediment file although none is ' //
     &    'open -- aborting!.'
        CALL ABORT_MEDUSA()
      ENDIF



! Check validity of INTENT(IN) arguments
                                    ! - valid column number?
      IF ((icore < 1) .OR. (icore > ngrid)) THEN
        ncorelay_recs = 0          ! no records restored
        iflag = -2                 ! Error!
        WRITE(jp_stderr,*) '[mod_sedcore.F/SEDFIL_PULLRECORDS]: ' //
     &    'invalid <icore> :', icore
        CALL ABORT_MEDUSA()
        RETURN
      ENDIF

                                    ! - valid number of records/layers?
      IF ((ncorelay_recs < 0) .OR. (ncorelay_recs > nhl)) THEN
        ncorelay_recs = 0          ! no records restored
        iflag = -3                 ! Error!
        WRITE(jp_stderr,*) '[mod_sedcore.F/SEDFIL_PULLRECORDS]: ' //
     &    'invalid <ncorelay_recs> :', ncorelay_recs
        CALL ABORT_MEDUSA()
        RETURN
      ENDIF


! INTENT(IN) arguments are valid: proceed

#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
      IF (icore == icol_dbg)
     &  WRITE(jp_coldbg, '("[SWR]: pull ", I0, " records")')
     &    ncorelay_recs
#endif
#endif
#endif


! If a sediment core is in use:
      IF (ncorelay_recs == 0) THEN  ! nothing to restore: easy!
        iflag = 0                   ! done with success :-)
        RETURN
      ENDIF


! Determine the effective number of records <nrecs> to be restored

      nrecs_req = ncorelay_recs     ! number of records requested

                                    ! number of records available in the file
      nrecs_max = sedflay_nlay(icore, kplug)

                                    ! number of layers empty in CORELAY
      nlay_empty = nhl - CORELAY_NFILLED_ICOL(icore)

                                    ! number of records recoverable
      ncorelay_recs = MIN(nrecs_max, nrecs_req, nlay_empty)


#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
      IF (icore == icol_dbg)
     &  WRITE(jp_coldbg, '("[SWR]: ", I0, " CORELAY can be pulled")')
     &    ncorelay_recs
#endif
#endif
#endif

! Fill up layers below the oldest in CORELAY with data from file

                                    ! oldest and youngest layers on
                                    ! the CORELAY stack
      ihl_oldest   = corelay_ihl_oldest(icore)
      ihl_youngest = corelay_ihl_youngest(icore)

      irec = sedflay_irectop(icore, kplug)
                                    ! record number of the youngest
                                    ! <icore>-sediment layer in the file

#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
      IF (icore == icol_dbg) THEN
        WRITE(jp_coldbg, '("[SRR@beg]: ihl o/y = ", I0, "/", I0)')
     &    ihl_oldest , ihl_youngest
      ENDIF
#endif
#endif
#endif

                                    ! successively read in the latest
                                    ! <nrecs> records from the file into
                                    ! the layers older than <ihl_oldest>

      DO i = 1, ncorelay_recs

        CALL SEDFIL_READ_RECORD(kplug, irec,
     &         sedfil_icore, sedfil_corelayid, irec_prec,
     &         sedfile_burtime, sedfil_erotime, sedfil_solid)

        IF ((sedfil_icore /= icore)) THEN
                                       ! Core-ID mismatch: stop restoration

                                       ! Reset ihl_oldest
          corelay_ihl_oldest(icore) = ihl_oldest
          ncorelay_recs = i-1          ! not all records restored
          iflag = -4                   ! Error!
          CALL ABORT_MEDUSA()
        ENDIF

#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
        IF (icore == icol_dbg)
     &    WRITE(jp_coldbg, '("[SRR]: CLR ", I0, " ontop of ", I0)')
     &      irec, irec_prec
#endif
#endif
#endif

                                       ! set the layer pointer where to
                                       ! store tha data just read in
                                       ! (i.e., the next ihl_oldest one)
        IF (ihl_oldest == jp_pt_top_emptycore) THEN
          ihl_oldest   = ihl1 + ncorelay_recs - 1
          ihl_youngest = ihl_oldest
        ELSE
          ihl_oldest = IHL_DOWN(ihl_oldest)
        ENDIF

                                       ! and transfer the data into
                                       ! the CORELAY stack
        corelay_id     (   ihl_oldest, icore) = sedfil_corelayid
        corelay_irec   (   ihl_oldest, icore) = irec
        corelay_solids (:, ihl_oldest, icore) = sedfil_solid(:)
        corelay_burtime(   ihl_oldest, icore) = sedfile_burtime

        irec = irec_prec

      ENDDO

                                    ! update corelay_ihl_oldest(icore)
                                    ! and corelay_ihl_oldest(icore)
                                    ! to the current values
      corelay_ihl_oldest(icore)   = ihl_oldest
      corelay_ihl_youngest(icore) = ihl_youngest

#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
      IF (icore == icol_dbg) THEN
        WRITE(jp_coldbg, '("[SRR@end]: ihl o/y = ", I0, "/", I0)')
     &    ihl_oldest , ihl_youngest
      ENDIF
#endif
#endif
#endif


! Update bookkeeping information

                                    ! record number of the now youngest
                                    ! record available in the file for
                                    ! the core <icore>
      sedflay_irectop(icore, kplug) = irec

                                    ! adjust the number of layers remaining
                                    ! buried in the file
      sedflay_nlay(icore, kplug) = sedflay_nlay(icore, kplug)
     &                             - ncorelay_recs

                                    ! decrease the counter of records still
                                    ! present in the file
      sedflay_nrecs(icore, kplug) = sedflay_nrecs(icore, kplug)
     &                              - ncorelay_recs

! Finally set the return error flag

      iflag = nrecs_req - ncorelay_recs


      RETURN


!======================================================================
      END SUBROUTINE SEDFIL_PULLRECORDS
!======================================================================



!======================================================================
      SUBROUTINE SEDFIL_WRITE_RECORD(kplug, irec,
     &                              icore, icorelayid, irec_prec,
     &                              burtime, erotime, hsolid)
!======================================================================


      IMPLICIT NONE


      INTEGER, INTENT(IN)  :: kplug
      INTEGER, INTENT(IN)  :: irec
      INTEGER, INTENT(IN)  :: icore
      INTEGER, INTENT(IN)  :: icorelayid
      INTEGER, INTENT(IN)  :: irec_prec
      DOUBLE PRECISION, INTENT(IN) :: burtime
      DOUBLE PRECISION, INTENT(IN) :: erotime
      DOUBLE PRECISION, DIMENSION(nsolid), INTENT(IN) :: hsolid


#ifdef ALLOW_MPI
      INTEGER :: iflag
#endif


#ifdef ALLOW_MPI
      IF (l_file_is_mine) THEN
                                    ! If this rank is an I/O rank,
                                    ! it simply writes the record.

        CALL MSC_NF_PUT_RECORD(kplug, irec, icore,
     &                              icorelayid, irec_prec,
     &                              burtime, erotime, hsolid)

      ELSE
                                    ! If this rank is not an I/O rank,
                                    ! it asks the I/O rank to do the
                                    ! actual writing and sends it the
                                    ! data to write:
                                    !  - send the "Write" request
        CALL MPI_SEND(jp_mpimsg_write, 1, MPI_INTEGER,
     &                              jp_exeproc_ncio, jp_mpitag_reqio,
     &                              i_mycomm, iflag)

                                    !  - send the kplug of the file to wirite to
        CALL MPI_SEND(kplug, 1, MPI_INTEGER,
     &                              jp_exeproc_ncio, jp_mpitag_data7,
     &                              i_mycomm, iflag)

                                    !  - send the record number to write
        CALL MPI_SEND(irec, 1, MPI_INTEGER,
     &                              jp_exeproc_ncio, jp_mpitag_data0,
     &                              i_mycomm, iflag)

                                    !  - send the data to write
        CALL MPI_SEND(icore, 1, MPI_INTEGER,
     &                              jp_exeproc_ncio, jp_mpitag_data1,
     &                              i_mycomm, iflag)
        CALL MPI_SEND(icorelayid, 1, MPI_INTEGER,
     &                              jp_exeproc_ncio, jp_mpitag_data2,
     &                              i_mycomm, iflag)
        CALL MPI_SEND(irec_prec, 1, MPI_INTEGER,
     &                              jp_exeproc_ncio, jp_mpitag_data3,
     &                              i_mycomm, iflag)
        CALL MPI_SEND(burtime, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_ncio, jp_mpitag_data4,
     &                              i_mycomm, iflag)
        CALL MPI_SEND(erotime, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_ncio, jp_mpitag_data5,
     &                              i_mycomm, iflag)
        CALL MPI_SEND(hsolid, nsolid, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_ncio, jp_mpitag_data6,
     &                              i_mycomm, iflag)

      ENDIF

#else
      CALL MSC_NF_PUT_RECORD(kplug, irec, icore, icorelayid, irec_prec,
     &                              burtime, erotime, hsolid)
#endif

      RETURN

!======================================================================
      END SUBROUTINE SEDFIL_WRITE_RECORD
!======================================================================



#ifdef ALLOW_MPI
!======================================================================
      SUBROUTINE SEDFIL_WRITE_RECORD4OTHRANK(i_origrank)
!======================================================================


      IMPLICIT NONE


      INTEGER, INTENT(IN) :: i_origrank


      INTEGER  :: kplug
      INTEGER  :: irec
      INTEGER  :: icore, sedfil_icore
      INTEGER  :: sedfil_corelayid
      INTEGER  :: irec_prec
      DOUBLE PRECISION :: sedfil_burtime
      DOUBLE PRECISION :: sedfil_erotime
      DOUBLE PRECISION, DIMENSION(nsolid) :: sedfil_solid

      INTEGER :: iflag

      IF (i_origrank == i_myrank) THEN

        WRITE(jp_stderr,*) '[SEDFIL_WRITE_RECORD4OTHRANK] error: ' //
     &      '"i_origrank" must be different from "i_myrank"!'
        WRITE(jp_stderr,*) 'Aborting.'
        CALL ABORT_MEDUSA()

      ENDIF

                                    ! This rank writes for another one.
                                    ! Receive now the data to
                                    ! write from that process
                                    !  - kplug
      CALL MPI_RECV(kplug, 1, MPI_INTEGER,
     &                              i_origrank, jp_mpitag_data7,
     &                              i_mycomm, MPI_STATUS_IGNORE, iflag)
                                    !  - record number
      CALL MPI_RECV(irec, 1, MPI_INTEGER,
     &                              i_origrank, jp_mpitag_data0,
     &                              i_mycomm, MPI_STATUS_IGNORE, iflag)

                                    !  - data
      CALL MPI_RECV(icore, 1, MPI_INTEGER,
     &                              i_origrank, jp_mpitag_data1,
     &                              i_mycomm, MPI_STATUS_IGNORE, iflag)
      CALL MPI_RECV(sedfil_corelayid, 1, MPI_INTEGER,
     &                              i_origrank, jp_mpitag_data2,
     &                              i_mycomm, MPI_STATUS_IGNORE, iflag)
      CALL MPI_RECV(irec_prec, 1, MPI_INTEGER,
     &                              i_origrank, jp_mpitag_data3,
     &                              i_mycomm, MPI_STATUS_IGNORE, iflag)
      CALL MPI_RECV(sedfil_burtime, 1, MPI_DOUBLE_PRECISION,
     &                              i_origrank, jp_mpitag_data4,
     &                              i_mycomm, MPI_STATUS_IGNORE, iflag)
      CALL MPI_RECV(sedfil_erotime, 1, MPI_DOUBLE_PRECISION,
     &                              i_origrank, jp_mpitag_data5,
     &                              i_mycomm, MPI_STATUS_IGNORE, iflag)
      CALL MPI_RECV(sedfil_solid, nsolid, MPI_DOUBLE_PRECISION,
     &                              i_origrank, jp_mpitag_data6,
     &                              i_mycomm, MPI_STATUS_IGNORE, iflag)

                                    ! and write it out

      sedfil_icore = icore + ioffset_sedcol_pproc(i_origrank)

      CALL MSC_NF_PUT_RECORD(kplug, irec, sedfil_icore,
     &                              sedfil_corelayid, irec_prec,
     &                              sedfil_burtime, sedfil_erotime,
     &                              sedfil_solid)


      RETURN

!======================================================================
      END SUBROUTINE SEDFIL_WRITE_RECORD4OTHRANK
!======================================================================
#endif


!======================================================================
      SUBROUTINE SEDFIL_READ_RECORD(kplug, irec,
     &                              icore, icorelayid, irec_prec,
     &                              burtime, erotime, hsolid)
!======================================================================


      IMPLICIT NONE


      INTEGER, INTENT(IN)  :: kplug
      INTEGER, INTENT(IN)  :: irec
      INTEGER, INTENT(OUT) :: icore
      INTEGER, INTENT(OUT) :: icorelayid
      INTEGER, INTENT(OUT) :: irec_prec
      DOUBLE PRECISION, INTENT(OUT) :: burtime
      DOUBLE PRECISION, INTENT(OUT) :: erotime
      DOUBLE PRECISION, DIMENSION(nsolid), INTENT(OUT) :: hsolid


#ifdef ALLOW_MPI
      INTEGER :: iflag
#endif



#ifdef ALLOW_MPI
      IF (l_file_is_mine) THEN
                                    ! If this rank is an I/O rank,
                                    ! it simply reads the record.

        CALL MSC_NF_GET_RECORD(kplug, irec, icore,
     &                              icorelayid, irec_prec,
     &                              burtime, erotime, hsolid)

      ELSE
                                    ! If this rank is not an I/O rank,
                                    ! ask the I/O rank to do the
                                    ! actual reading and have the data
                                    ! sent back.
                                    !  - send the "Read" request
        CALL MPI_SEND(jp_mpimsg_read, 1, MPI_INTEGER,
     &                              jp_exeproc_ncio, jp_mpitag_reqio,
     &                              i_mycomm, iflag)

                                    !  - send the kplug of the file to read from
        CALL MPI_SEND(kplug, 1, MPI_INTEGER,
     &                              jp_exeproc_ncio, jp_mpitag_data7,
     &                              i_mycomm, iflag)

                                    !  - send the record number to read
        CALL MPI_SEND(irec, 1, MPI_INTEGER,
     &                              jp_exeproc_ncio, jp_mpitag_data0,
     &                              i_mycomm, iflag)

                                    !  - receive the data
        CALL MPI_RECV(icore, 1, MPI_INTEGER,
     &                              jp_exeproc_ncio, jp_mpitag_data1,
     &                              i_mycomm, MPI_STATUS_IGNORE, iflag)
        CALL MPI_RECV(icorelayid, 1, MPI_INTEGER,
     &                              jp_exeproc_ncio, jp_mpitag_data2,
     &                              i_mycomm, MPI_STATUS_IGNORE, iflag)
        CALL MPI_RECV(irec_prec, 1, MPI_INTEGER,
     &                              jp_exeproc_ncio, jp_mpitag_data3,
     &                              i_mycomm, MPI_STATUS_IGNORE, iflag)
        CALL MPI_RECV(burtime, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_ncio, jp_mpitag_data4,
     &                              i_mycomm, MPI_STATUS_IGNORE, iflag)
        CALL MPI_RECV(erotime, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_ncio, jp_mpitag_data5,
     &                              i_mycomm, MPI_STATUS_IGNORE, iflag)
        CALL MPI_RECV(hsolid, nsolid, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_ncio, jp_mpitag_data6,
     &                              i_mycomm, MPI_STATUS_IGNORE, iflag)

      ENDIF
#else
      CALL MSC_NF_GET_RECORD(kplug, irec, icore, icorelayid, irec_prec,
     &                              burtime, erotime, hsolid)
#endif


      RETURN


!======================================================================
      END SUBROUTINE SEDFIL_READ_RECORD
!======================================================================



#ifdef ALLOW_MPI
!======================================================================
      SUBROUTINE SEDFIL_READ_RECORD4OTHRANK(i_destrank)
!======================================================================


      IMPLICIT NONE


      INTEGER, INTENT(IN) :: i_destrank


      INTEGER  :: kplug
      INTEGER  :: irec
      INTEGER  :: icore
      INTEGER  :: sedfil_icore
      INTEGER  :: sedfil_corelayid
      INTEGER  :: irec_prec
      DOUBLE PRECISION :: sedfil_burtime
      DOUBLE PRECISION :: sedfil_erotime
      DOUBLE PRECISION, DIMENSION(nsolid) :: sedfil_solid

      INTEGER :: iflag

      IF (i_destrank == i_myrank) THEN

        WRITE(jp_stderr,*) '[SEDFIL_READ_RECORD4OTHRANK] error: ' //
     &      '"i_destrank" must be different from "i_myrank"!'
        WRITE(jp_stderr,*) 'Aborting.'
        CALL ABORT_MEDUSA()

      ENDIF

                                    ! This rank reads for another one:
                                    !  - get the kplug of the file to read from
      CALL MPI_RECV(kplug, 1, MPI_INTEGER,
     &                              i_destrank, jp_mpitag_data7,
     &                              i_mycomm, MPI_STATUS_IGNORE, iflag)
                                    !  - get the record number to read
      CALL MPI_RECV(irec, 1, MPI_INTEGER,
     &                              i_destrank, jp_mpitag_data0,
     &                              i_mycomm, MPI_STATUS_IGNORE, iflag)


                                    !  - read in the data
      CALL MSC_NF_GET_RECORD(kplug, irec, sedfil_icore,
     &                              sedfil_corelayid, irec_prec,
     &                              sedfil_burtime, sedfil_erotime,
     &                              sedfil_solid)

                                    !  - send the data to that rank
      icore = sedfil_icore - ioffset_sedcol_pproc(i_destrank)

      CALL MPI_SEND(icore, 1, MPI_INTEGER,
     &                              i_destrank, jp_mpitag_data1,
     &                              i_mycomm, iflag)
      CALL MPI_SEND(sedfil_corelayid, 1, MPI_INTEGER,
     &                              i_destrank, jp_mpitag_data2,
     &                              i_mycomm, iflag)
      CALL MPI_SEND(irec_prec, 1, MPI_INTEGER,
     &                              i_destrank, jp_mpitag_data3,
     &                              i_mycomm, iflag)
      CALL MPI_SEND(sedfil_burtime, 1, MPI_DOUBLE_PRECISION,
     &                              i_destrank, jp_mpitag_data4,
     &                              i_mycomm, iflag)
      CALL MPI_SEND(sedfil_erotime, 1, MPI_DOUBLE_PRECISION,
     &                              i_destrank, jp_mpitag_data5,
     &                              i_mycomm, iflag)
      CALL MPI_SEND(sedfil_solid, nsolid, MPI_DOUBLE_PRECISION,
     &                              i_destrank, jp_mpitag_data6,
     &                              i_mycomm, iflag)


      RETURN


!======================================================================
      END SUBROUTINE SEDFIL_READ_RECORD4OTHRANK
!======================================================================
#endif



!======================================================================
      SUBROUTINE MSC_NF_PUT_RECORD(kplug, irec, sedfil_icore,
     &  sedfil_corelayid, irec_prec,
     &  sedfil_burtime, sedfil_erotime, sedfil_solid)
!======================================================================


      USE mod_netcdfinc,            ONLY: NF_PUT_VAR1_INT,
     &                                    NF_PUT_VAR1_DOUBLE,
     &                                    NF_NOERR,
     &                                    HANDLE_ERRORS


      IMPLICIT NONE


      INTEGER, INTENT(IN) :: kplug
      INTEGER, INTENT(IN) :: irec
      INTEGER, INTENT(IN) :: sedfil_icore
      INTEGER, INTENT(IN) :: sedfil_corelayid
      INTEGER, INTENT(IN) :: irec_prec
      DOUBLE PRECISION, INTENT(IN) :: sedfil_burtime
      DOUBLE PRECISION, INTENT(IN) :: sedfil_erotime
      DOUBLE PRECISION, DIMENSION(nsolid), INTENT(IN) :: sedfil_solid

      INTEGER :: ncid
      INTEGER :: isolid
      INTEGER :: istatus


      ncid = sedfil_ncid(kplug)

      istatus = NF_PUT_VAR1_INT(ncid, id_lay(kplug), irec, irec)
      IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

      istatus = NF_PUT_VAR1_INT(ncid, id_coreid(kplug), irec,
     &                              sedfil_icore)
      IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

      istatus = NF_PUT_VAR1_INT(ncid, id_layid(kplug), irec,
     &                              sedfil_corelayid)
      IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

      istatus = NF_PUT_VAR1_INT(ncid, id_ipreclay(kplug), irec,
     &                              irec_prec)
      IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

      istatus = NF_PUT_VAR1_DOUBLE(ncid, id_burtime(kplug), irec,
     &                              sedfil_burtime)
      IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

      istatus = NF_PUT_VAR1_DOUBLE(ncid, id_erotime(kplug), irec,
     &                              sedfil_erotime)
      IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)


      DO isolid = 1, nsolid
        istatus = NF_PUT_VAR1_DOUBLE(ncid, id_hcont(isolid,kplug),
     &                              irec,
     &                              sedfil_solid(isolid))
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
      ENDDO


      RETURN


!======================================================================
      END SUBROUTINE MSC_NF_PUT_RECORD
!======================================================================



!======================================================================
      SUBROUTINE MSC_NF_GET_RECORD(kplug, irec,
     &  sedfil_icore, sedfil_corelayid, irec_prec,
     &  sedfil_burtime, sedfil_erotime, sedfil_solid)
!======================================================================


      USE mod_netcdfinc,            ONLY: NF_GET_VAR1_INT,
     &                                    NF_GET_VAR1_DOUBLE,
     &                                    NF_NOERR,
     &                                    HANDLE_ERRORS


      IMPLICIT NONE


      INTEGER, INTENT(IN)  :: kplug
      INTEGER, INTENT(IN)  :: irec
      INTEGER, INTENT(OUT) :: sedfil_icore
      INTEGER, INTENT(OUT) :: sedfil_corelayid
      INTEGER, INTENT(OUT) :: irec_prec
      DOUBLE PRECISION, INTENT(OUT) :: sedfil_burtime
      DOUBLE PRECISION, INTENT(OUT) :: sedfil_erotime
      DOUBLE PRECISION, DIMENSION(nsolid), INTENT(OUT) :: sedfil_solid


      INTEGER :: ncid
      INTEGER :: istatus
      INTEGER :: isolid


      ncid = sedfil_ncid(kplug)

      istatus = NF_GET_VAR1_INT(ncid, id_coreid(kplug), irec,
     &                              sedfil_icore)
      IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

      istatus = NF_GET_VAR1_INT(ncid, id_layid(kplug), irec,
     &                              sedfil_corelayid)
      IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

      istatus = NF_GET_VAR1_INT(ncid, id_ipreclay(kplug), irec,
     &                              irec_prec)
      IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

      istatus = NF_GET_VAR1_DOUBLE(ncid, id_burtime(kplug), irec,
     &                              sedfil_burtime)
      IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

      istatus = NF_GET_VAR1_DOUBLE(ncid, id_erotime(kplug), irec,
     &                              sedfil_erotime)
      IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

      DO isolid = 1, nsolid
        istatus = NF_GET_VAR1_DOUBLE(ncid, id_hcont(isolid, kplug),
     &                              irec,
     &                              sedfil_solid(isolid))
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
      ENDDO


      RETURN


!======================================================================
      END SUBROUTINE MSC_NF_GET_RECORD
!======================================================================



!======================================================================
      SUBROUTINE SEDFIL_PRUNE
!======================================================================

! - check the number of CORELAY layers and if above a treshold,
!   write out oldest contents if
! - write out all ERODLAY contents
! - for all the cores


      IMPLICIT NONE


      INTEGER :: icore
      INTEGER :: ilayertype
      INTEGER :: iflag

      INTEGER :: nfilled, nrecs


! Return stright away if no file is used
      IF (.NOT. sedfil_used(jp_ncout)) RETURN


! Check the buffers for each core

#ifdef ALLOW_MPI
      CALL MSC_QUEUE_START
#endif

      DO icore = 1, ngrid

        nfilled = CORELAY_NFILLED_ICOL(icore)

        nrecs = nhl/2

        IF (nfilled > (nhl*3)/4) THEN
          nrecs = nhl/2             ! reduce to a little more than nhl/4
          ilayertype = 3            ! both CORELAY and ERODLAY stacks
        ELSE
          nrecs = 0                 ! will be disregarded anyway
          ilayertype = 2            ! only ERODLAY stacks
        ENDIF

#ifdef DEBUG
#ifdef DEBUG_SEDCORE
#ifdef DEBUG_SEDCORE_COLUMN
        IF (icore == icol_dbg) THEN
          WRITE(jp_coldbg,*)
     &      'Prune request -> SEDFIL_PUSHRECORDS'
        ENDIF
#endif
#endif
#endif

        CALL SEDFIL_PUSHRECORDS(icore, ilayertype, nrecs, iflag)


      ENDDO


#ifdef ALLOW_MPI
      CALL MSC_QUEUE_PROCESS
      CALL MSC_QUEUE_TERMINATE
#endif


      RETURN


!======================================================================
      END SUBROUTINE SEDFIL_PRUNE
!======================================================================



!=======================================================================
      SUBROUTINE MSC_NF_PUT_C_INT(ncid, id_var, ivar_c)
!=======================================================================

! Write out one complete record provided by the array
! ivar_c(1:ngrid) into a NetCDF variable with dimension (dim_col).
! Under MPI, stitch them together if l_onewrites4many==.TRUE.


      USE mod_netcdfinc,            ONLY: NF_PUT_VAR_INT, NF_NOERR,
     &                                    HANDLE_ERRORS


      IMPLICIT NONE


      INTEGER, INTENT(IN)               :: ncid
      INTEGER, INTENT(IN)               :: id_var
      INTEGER, INTENT(IN), DIMENSION(:) :: ivar_c


      INTEGER :: istatus


#ifdef ALLOW_MPI
      IF (l_oneio4many) THEN

        CALL MPI_GATHERV(ivar_c(:),
     &    ngrid, MPI_INTEGER,
     &    iarr_mpirecv_c(:),
     &    nsedcol_pproc(:), ioffset_sedcol_pproc(:),
     &    MPI_INTEGER,
     &    jp_exeproc_ncio, i_mycomm, istatus)

        IF (i_myrank == jp_exeproc_ncio) THEN
          istatus = NF_PUT_VAR_INT(ncid, id_var, iarr_mpirecv_c(:))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        ENDIF

      ELSE

        istatus = NF_PUT_VAR_INT(ncid, id_var, ivar_c(:))
        IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)

      ENDIF
#else
      istatus = NF_PUT_VAR_INT(ncid, id_var, ivar_c(:))
      IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
#endif

      RETURN

!=======================================================================
      END SUBROUTINE MSC_NF_PUT_C_INT
!=======================================================================



!=======================================================================
      SUBROUTINE MSC_NF_GET_C_INT(ncid, id_var, ivar_c)
!=======================================================================

! Read in one complete record into the array ivar_c(1:ngrid)
! from a NetCDF variable with dimension (dim_col)
! Under MPI, scatter them if l_oneio4many==.TRUE.

      USE mod_netcdfinc,            ONLY: NF_GET_VAR_INT, NF_NOERR,
     &                                    HANDLE_ERRORS


      IMPLICIT NONE


      INTEGER, INTENT(IN)                :: ncid
      INTEGER, INTENT(IN)                :: id_var
      INTEGER, INTENT(OUT), DIMENSION(:) :: ivar_c


      INTEGER :: istatus


#ifdef ALLOW_MPI
      IF (l_oneio4many) THEN

        IF (l_file_is_mine) THEN
          istatus = NF_GET_VAR_INT(ncid, id_var, iarr_mpisend_c(:))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        ENDIF

        CALL MPI_SCATTERV(iarr_mpisend_c(:),
     &    nsedcol_pproc(:), ioffset_sedcol_pproc(:),
     &    MPI_INTEGER,
     &    ivar_c(:), ngrid,
     &    MPI_INTEGER,
     &    jp_exeproc_ncio, i_mycomm, istatus)

      ELSE

        IF (l_file_is_mine) THEN
          istatus = NF_GET_VAR_INT(ncid, id_var, ivar_c(:))
          IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
        ENDIF

      ENDIF
#else
      istatus = NF_GET_VAR_INT(ncid, id_var, ivar_c(:))
      IF (istatus /= NF_NOERR) CALL HANDLE_ERRORS(istatus)
#endif

      RETURN

!=======================================================================
      END SUBROUTINE MSC_NF_GET_C_INT
!=======================================================================



#ifdef ALLOW_MPI
!=======================================================================
      SUBROUTINE MSC_QUEUE_START
!=======================================================================


      IMPLICIT NONE


      INTEGER :: iflag
      INTEGER :: j_message

                                    ! First let all ranks come together.
      CALL MPI_BARRIER(i_mycomm, iflag)

      IF (l_oneio4many) THEN
                                    ! Multi-processor environment with
                                    ! single-rank NetCDF I/O:
        IF (l_file_is_mine) THEN
                                    !  - the current rank is the I/O rank --
          CONTINUE                  !    simply proceed

        ELSE                        !  - the current rank is not the I/O
                                    !    and has to wait for green light
                                    !    from the I/O rank
          CALL MPI_RECV(j_message, 1, MPI_INTEGER,
     &                              jp_exeproc_ncio, jp_mpitag_reqio,
     &                              i_mycomm, MPI_STATUS_IGNORE, iflag)

          IF (j_message /= jp_mpimsg_go) THEN
            ! Unexpected message
            CALL ABORT_MEDUSA()
          ENDIF

        ENDIF

      ENDIF

      RETURN

!=======================================================================
      END SUBROUTINE MSC_QUEUE_START
!=======================================================================



!=======================================================================
      SUBROUTINE MSC_QUEUE_PROCESS
!=======================================================================


      IMPLICIT NONE


      INTEGER :: i_rank
      INTEGER :: iflag
      INTEGER :: j_message


      IF (l_oneio4many) THEN
                                    ! There is one rank that does I/O for others

        IF (l_file_is_mine) THEN
                                    ! It is the current one that is the only
                                    ! one that does I/O.

          DO i_rank = 0, n_cprocs - 1

            IF (i_rank == jp_exeproc_ncio) CYCLE

            CALL MPI_SEND(jp_mpimsg_go, 1, MPI_INTEGER,
     &                              i_rank, jp_mpitag_reqio,
     &                              i_mycomm, iflag)

                                    !  - and process all its requests until
                                    !    the message "Done" is received

            process_requests: DO


              CALL MPI_RECV(j_message, 1, MPI_INTEGER,
     &                              i_rank, jp_mpitag_reqio,
     &                              i_mycomm, MPI_STATUS_IGNORE, iflag)


              SELECT CASE(j_message)

              CASE(jp_mpimsg_send_nrecs)
                CALL MPI_SEND(sedfil_nrecs(jp_ncout), 1, MPI_INTEGER,
     &                              i_rank, jp_mpitag_data0,
     &                              i_mycomm, iflag)

              CASE(jp_mpimsg_recv_nrecs)
                CALL MPI_RECV(sedfil_nrecs(jp_ncout), 1, MPI_INTEGER,
     &                              i_rank, jp_mpitag_data0,
     &                              i_mycomm, MPI_STATUS_IGNORE, iflag)

              CASE(jp_mpimsg_write)
                CALL SEDFIL_WRITE_RECORD4OTHRANK(i_rank)

              CASE(jp_mpimsg_read)
                CALL SEDFIL_READ_RECORD4OTHRANK(i_rank)

              CASE(jp_mpimsg_done)
                EXIT process_requests

              CASE DEFAULT
                WRITE(jp_stderr,'(A,I0,A,I0)')
     &            '[mod_sedcore.F/MSC_QUEUE_PROCESS]: ' //
     &            'received unexpected message ', j_message,
     &            ' from rank ', i_rank
                WRITE(jp_stderr,*) 'Aborting.'
                CALL ABORT_MEDUSA()

              END SELECT


            ENDDO process_requests

          ENDDO

        ELSE

          CONTINUE                  ! This rank does not do its own I/O.

        ENDIF

      ELSE

          CONTINUE                  ! Each rank does its own I/O.

      ENDIF


      RETURN

!=======================================================================
      END SUBROUTINE MSC_QUEUE_PROCESS
!=======================================================================



!=======================================================================
      SUBROUTINE MSC_QUEUE_TERMINATE
!=======================================================================


      IMPLICIT NONE


      INTEGER :: iflag
      INTEGER :: i_message

      IF (l_oneio4many) THEN
                                    ! Multi-processor environment with
                                    ! single-rank NetCDF I/O:
        IF (l_file_is_mine) THEN
                                    !  - the current rank is the I/O rank --
          CONTINUE                  !    nothing to do here, simply proceed.

        ELSE                        !  - the current rank is not the I/O
                                    !    and has to inform the I/O rank
                                    !    that it has finished requests.
          CALL MPI_SEND(jp_mpimsg_done, 1, MPI_INTEGER,
     &                              jp_exeproc_ncio, jp_mpitag_reqio,
     &                              i_mycomm, iflag)

        ENDIF

      ENDIF

                                    ! ... and let all ranks come together.
      CALL MPI_BARRIER(i_mycomm, iflag)


      RETURN

!=======================================================================
      END SUBROUTINE MSC_QUEUE_TERMINATE
!=======================================================================
#endif



!=======================================================================
      INTEGER FUNCTION IHL_UP(i)
!=======================================================================


      IMPLICIT NONE


      INTEGER, INTENT(IN) :: i


      IF (i == ihln) THEN
        IHL_UP = ihl1
      ELSE
        IHL_UP = i + 1
      ENDIF


      RETURN

!=======================================================================
      END FUNCTION IHL_UP
!=======================================================================



!=======================================================================
      INTEGER FUNCTION IHL_DOWN(i)
!=======================================================================


      IMPLICIT NONE


      INTEGER, INTENT(IN) :: i

      IF (i == ihl1) THEN
        IHL_DOWN = ihln
      ELSE
        IHL_DOWN = i - 1
      ENDIF

      RETURN

!=======================================================================
      END FUNCTION IHL_DOWN
!=======================================================================


!=======================================================================
      END MODULE MOD_SEDCORE
!=======================================================================


