!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


!=======================================================================
      MODULE MOD_MILIEUCHARAS
!=======================================================================

      USE MOD_GRIDPARAM, ONLY: idnt, idnb, idvs, idvb

      IMPLICIT NONE

                                    ! PUBLIC/PRIVATE declarations
                                    ! ===========================
      PRIVATE

      PUBLIC :: jp_phi_notsetup
      PUBLIC :: jp_phi_dynamic
      PUBLIC :: jp_phi_static_local, jp_phi_static_global

      PUBLIC :: jselect_phitype

      PUBLIC :: xphi, xdphi, yphi, xvphi, xvdphi
      PUBLIC :: xvtor2

      PUBLIC :: SETUP_MILIEUCHARAS, SELECTED_PHITYPE
      PUBLIC :: POROTORTUOSITY_DEF, TORTUOSITY_UPDATE

                                    ! Available options
                                    ! =================

                                    ! Numbering of the currently
                                    ! available grid variability options
                                    ! (DO NOT CHANGE!)
      INTEGER, PARAMETER :: jp_phi_notsetup      = -1
      INTEGER, PARAMETER :: jp_phi_dynamic       =  1
      INTEGER, PARAMETER :: jp_phi_static_local  =  2
      INTEGER, PARAMETER :: jp_phi_static_global =  3

                                    ! Numbering of the different
                                    ! porosity profile options
                                    ! (DO NOT CHANGE!)
      INTEGER, PARAMETER :: jp_poroprof_custom = 0
      INTEGER, PARAMETER :: jp_poroprof_const  = 1
      INTEGER, PARAMETER :: jp_poroprof_expdec = 2

                                    ! Numbering of the different
                                    ! tortuosity relationships
                                    ! (DO NOT CHANGE!)
      INTEGER, PARAMETER :: jp_torturel_custom  = 0 ! not yet implemented
      INTEGER, PARAMETER :: jp_torturel_archie  = 1
      INTEGER, PARAMETER :: jp_torturel_burfrie = 2
      INTEGER, PARAMETER :: jp_torturel_modwei  = 3


                                    ! Specific default values
                                    ! =======================

                                    ! Profile type parameters
                                    ! - const
      DOUBLE PRECISION, PARAMETER :: dp_phi0     = 0.90D+00

                                    ! Profile type parameters
                                    ! - expdec (uses dp_phi0 above as well)
      DOUBLE PRECISION, PARAMETER :: dp_phi00    = 0.70D+00
      DOUBLE PRECISION, PARAMETER :: dp_phiscale = 0.04D+00

                                    ! Tortuosity-Porosity relationship
                                    ! parameters
                                    ! - archie (Archie's law (Boudreau, 1997, p. 131))
      DOUBLE PRECISION, PARAMETER :: dp_archie_m = 2.14D+00
                                    ! - burgerfrieke (Burger-Frieke eqn.
                                    !   (Boudreau, 1997, p. 131))
      DOUBLE PRECISION, PARAMETER :: dp_burfrie_a = 3.14D+00
                                    ! - modweissberg (modified Weissberg
                                    !   relationship, (Boudreau, 1997, p. 131))
      DOUBLE PRECISION, PARAMETER :: dp_modweiss_b = 2.02D+00


                                    ! Actually used values
                                    ! ====================

                                    ! - milieu (porosity) variabilility
      INTEGER, SAVE :: jselect_phitype = jp_phi_static_global

                                    ! - porosity profiles
      INTEGER, SAVE :: jselect_poroprof = jp_poroprof_expdec

      DOUBLE PRECISION, SAVE      :: da_phi0     = dp_phi0

      DOUBLE PRECISION, SAVE      :: da_phi00    = dp_phi00
      DOUBLE PRECISION, SAVE      :: da_phiscale = dp_phiscale

                                    ! - tortuosity-porosity relationship
      INTEGER, SAVE :: jselect_torturel = jp_torturel_modwei

      DOUBLE PRECISION, SAVE      :: da_archie_m   = dp_archie_m
      DOUBLE PRECISION, SAVE      :: da_burfrie_a  = dp_burfrie_a
      DOUBLE PRECISION, SAVE      :: da_modweiss_b = dp_modweiss_b


                                    ! Processing storage
                                    ! ==================

                                    ! xphi(*): sediment porosity at grid
                                    !          nodes (i=idnt, idnb)
                                    ! xvphi(*): sediment porosity at grid
                                    !          vertices (i=idvs, idvb)
                                    ! yphi:    sediment porosity in the
                                    !          buffering layer (TRANLAY)
                                    ! zphi:    sediment porosity in the
                                    !          historical layers (CORELAY)
      DOUBLE PRECISION, SAVE, DIMENSION(idnt:idnb) ::  xphi
      DOUBLE PRECISION, SAVE                       ::  yphi
      DOUBLE PRECISION, SAVE                       ::  zphi

      DOUBLE PRECISION, SAVE, DIMENSION(idvs:idvb) ::  xvphi


                                    ! xdphi(:): xdphi(i) = (d xphi/d z)|z=xzdn(i),
                                    !          actual value of the derivative
                                    !          if xphi=fct(xzdn) analytically known
                                    ! xvdphi(:): xdphi(i) = (d xvphi/d z)|z=xzdv(i),
                                    !          actual value of the derivative
                                    !          if xvphi=fct(xzdv) analytically known
      DOUBLE PRECISION, SAVE, DIMENSION(idnt:idnb) ::  xdphi

      DOUBLE PRECISION, SAVE, DIMENSION(idvs:idvb) ::  xvdphi

                                    ! xvtor2(:): sediment tortuosity^2
                                    !          ($\theta^2$) at grid vertices (i=idvs, idvb)
                                    ! ytor2: sed't tortuosity^2 in the
                                    !          buffering joint layer (TRANLAY)
                                    ! ztor2: sed't tortuosity^2 in the
                                    !          historical layers (CORELAY)
      DOUBLE PRECISION, SAVE, DIMENSION(idvs:idvb) ::  xvtor2
      DOUBLE PRECISION, SAVE                       ::  ytor2
      DOUBLE PRECISION, SAVE                       ::  ztor2


#ifdef PHI_CUSTOM
#define PHI_CUSTOM_DECLARATIONS
#include <porosity_custom.F>
#undef PHI_CUSTOM_DECLARATIONS
#endif


      LOGICAL, SAVE               :: l_setupdone = .FALSE.


      CONTAINS


!-----------------------------------------------------------------------
      SUBROUTINE SETUP_MILIEUCHARAS
!-----------------------------------------------------------------------

!--------------
! Declarations
!--------------


      USE mod_defines_medusa,       ONLY: jp_stderr, jp_stdlog
#ifdef DEBUG
      USE mod_defines_medusa,       ONLY: jp_stddbg
#endif
      USE mod_execontrol_medusa,    ONLY: ABORT_MEDUSA
      USE mod_logunits

#ifdef ALLOW_MPI
      USE mod_execontrol_medusa,    ONLY: MEDEXE_MPI_COMM,
     &                              MEDEXE_MPI_COMM_RANK,
     &                              jp_exeproc_root
      USE mpi, ONLY: MPI_INTEGER, MPI_DOUBLE_PRECISION
#endif


      IMPLICIT NONE


!-----------------------
! Variable declarations
!-----------------------

!- - - - - - - - - - - - -  - - - - - - -
! Variables in subroutine call arguments
!- - - - - - - - - - - - -- - - - - - - -

      ! None


!- - - - - - - - - - - - - -  - - - - - - - -
! General (global) parameters and definitions
!- - - - - - - - - - - - - -- - - - - - - - -

      CHARACTER(LEN=*), PARAMETER ::
     &  cfn_milieu_cfg = "medusa_milieu_config.nml"


!- - - - - - - - -
! Local variables
!- - - - - - - - -


#ifdef ALLOW_MPI
      INTEGER :: i_mycomm, i_myrank
#endif




      CHARACTER(LEN=15) :: ctype_phi_variability
      CHARACTER(LEN=15) :: ctype_porosity_profile
      CHARACTER(LEN=15) :: ctype_tortuosity_rel

      NAMELIST /nml_milieu_options/ ctype_phi_variability,
     &                              ctype_porosity_profile,
     &                              ctype_tortuosity_rel


      DOUBLE PRECISION :: phi_0, phi_infty, phi_scale

      NAMELIST /nml_phi_const/  phi_0
      NAMELIST /nml_phi_expdec/ phi_0, phi_infty, phi_scale

      DOUBLE PRECISION :: archie_m

      NAMELIST /nml_ttp_archie/ archie_m


      INTEGER :: iu_cfg, istatus
      LOGICAL :: l_cfgfile_is_mine = .FALSE.
      LOGICAL :: l_exists

      CHARACTER(LEN=*), PARAMETER :: cfmt_modprocname_a
     &  = '("[MOD_MILIEUCHARAS/setup_milieucharas]: ", A)'

      CHARACTER(LEN=*), PARAMETER :: cfmt_a     = '(" - ", A)'
      CHARACTER(LEN=*), PARAMETER :: cfmt_a_ind = '(3X, A)'


!- - - - - - - - - - -
! End of declarations
!- - - - - - - - - - -


!----------------------------------------------------------------------
! Subroutine Start
!----------------------------------------------------------------------

#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_modprocname_a) 'Start'
#endif

      IF (l_setupdone) THEN
#ifdef DEBUG
        WRITE(jp_stddbg, cfmt_a) 'Setup already done'
        WRITE(jp_stddbg, cfmt_modprocname_a) 'Return'
        WRITE(jp_stddbg, '()')
#endif
        RETURN
      ENDIF


#ifdef ALLOW_MPI
      i_mycomm = MEDEXE_MPI_COMM()
      i_myrank = MEDEXE_MPI_COMM_RANK()
      l_cfgfile_is_mine = (i_myrank == jp_exeproc_root)
#else
      l_cfgfile_is_mine = .TRUE.
#endif

      WRITE(jp_stdlog, cfmt_modprocname_a) 'Initial report'
      WRITE(jp_stdlog, '()')
      WRITE(jp_stdlog, '(" Adopted milieu options")')
      WRITE(jp_stdlog, '(" ----------------------")')
      WRITE(jp_stdlog, '()')


      IF (l_cfgfile_is_mine) THEN

                                    ! Pre-set the default values
        jselect_phitype = jp_phi_static_global
!XXX check if grid is static, etc. static global porosity
!XXX would be inconsistent with a static local grid, etc.

        jselect_poroprof = jp_poroprof_expdec

        phi_0     = dp_phi0
        phi_infty = dp_phi00
        phi_scale = dp_phiscale

        jselect_torturel = jp_torturel_modwei

                                    ! Check if file cfn_milieu_cfg exists
        INQUIRE(FILE=cfn_milieu_cfg, EXIST=l_exists)


        IF (l_exists) THEN          ! If file exists, read it and get
                                    ! information (else stick to defaults)

#ifdef DEBUG
          WRITE(jp_stddbg, cfmt_a)
     &      'Milieu configuration file "' // cfn_milieu_cfg //
     &      '" found.'
#endif

          istatus = RESERVE_LOGUNIT(iu_cfg)
          IF (istatus /= 0) THEN
            WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
            WRITE(jp_stderr, cfmt_a)
     &       'Unable to reserve a logical unit for "iu_cfg" -- aborting'
            CALL ABORT_MEDUSA()
#ifdef DEBUG
          ELSE
            WRITE(jp_stddbg, cfmt_a, ADVANCE="NO")
     &        'Assigning logical unit number '
            WRITE(jp_stddbg, '(I0, A)') iu_cfg, ' to "iu_cfg".'
#endif
          ENDIF


          OPEN(UNIT=iu_cfg, FILE=cfn_milieu_cfg)

                                    ! Pre-set the two main options
                                    ! to "unknown" (triggering the
                                    ! use of default values if not
                                    ! modified below).
          ctype_phi_variability  = '?'
          ctype_porosity_profile = '?'
          ctype_tortuosity_rel   = '?'

          READ(iu_cfg, NML=nml_milieu_options)


#ifdef DEBUG
          WRITE(jp_stddbg, '()')
          WRITE(jp_stddbg, cfmt_a_ind) 'Milieu options requested'
          WRITE(jp_stddbg, cfmt_a_ind) '========================'
          WRITE(jp_stddbg, '()')

          WRITE(jp_stddbg, cfmt_a_ind, ADVANCE="NO")
          IF (ctype_phi_variability == '?') THEN
            WRITE(jp_stddbg, '(" * ctype_phi_variability = ", A)')
     &                              '"static_global" (DEFAULT)'
          ELSE
            WRITE(jp_stddbg, '(" * ctype_phi_variability = ", A)')
     &                   '"' // TRIM(ctype_phi_variability) // '"'
          ENDIF
          WRITE(jp_stddbg, '()')

          WRITE(jp_stddbg, cfmt_a_ind, ADVANCE="NO")
          IF (ctype_porosity_profile == '?') THEN
            WRITE(jp_stddbg, '(" * ctype_porosity_profile = ",A)')
     &                              '"expdec" (DEFAULT)'
          ELSE
            WRITE(jp_stddbg, '(" * ctype_porosity_profile = ",A)')
     &                  '"' // TRIM(ctype_porosity_profile) // '"'
          ENDIF
          WRITE(jp_stddbg, '()')

          WRITE(jp_stddbg, cfmt_a_ind, ADVANCE="NO")
          IF (ctype_tortuosity_rel == '?') THEN
            WRITE(jp_stddbg, '(" * ctype_tortuosity_rel = ",A)')
     &                              '"modweissberg" (DEFAULT)'
          ELSE
            WRITE(jp_stddbg, '(" * ctype_tortuosity_rel = ",A)')
     &                  '"' // TRIM(ctype_tortuosity_rel) // '"'
          ENDIF
          WRITE(jp_stddbg, '()')
#endif


                                    ! Milieu type
                                    ! ===========

          SELECT CASE(ctype_phi_variability)
          CASE('?')
          !--------
            CONTINUE

          CASE('static_global')
          !--------------------
            jselect_phitype = jp_phi_static_global

          CASE('static_local')
          !-------------------
            jselect_phitype = jp_phi_static_local

          CASE('dynamic')
          !------------
            jselect_phitype = jp_phi_dynamic

          CASE DEFAULT
          !-----------
            WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
            WRITE(jp_stderr, cfmt_a, ADVANCE="NO")
     &        'unknown milieu type "' //
     &        TRIM(ctype_phi_variability) // '" -- aborting'
            CALL ABORT_MEDUSA()

          END SELECT


                                    ! Porosity profile
                                    ! ================

          SELECT CASE(ctype_porosity_profile)
          CASE('?')
          !--------
            CONTINUE

          CASE('const')
          !------------
            jselect_poroprof = jp_poroprof_const

            READ(iu_cfg, NML=nml_phi_const)


          CASE('expdec')
          !------------
            jselect_poroprof = jp_poroprof_expdec

            READ(iu_cfg, NML=nml_phi_expdec)


          CASE('custom')
          !-------------
#ifdef PHI_CUSTOM
            jselect_poroprof = jp_poroprof_custom

                                    ! Carry out stage 1 of the set-up
                                    ! of the custom grid-point distribution
                                    ! (reading in the data from the
                                    ! file 'iu_cfg', currently open)
            CALL POROSITY_CUSTOM_SETUP(1, iu_cfg)
#else
            WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
            WRITE(jp_stderr, cfmt_a)
     &        '"custom" porosity profile not available -- aborting'
            CALL ABORT_MEDUSA()
#endif

          CASE DEFAULT
          !-----------
            WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
            WRITE(jp_stderr, cfmt_a)
     &        'unknown porosity profile "' //
     &        TRIM(ctype_porosity_profile) // '" -- aborting'
            CALL ABORT_MEDUSA()

          END SELECT


                                    ! Tortuosity-porosity relationship
                                    ! ================================

          SELECT CASE(ctype_tortuosity_rel)
          CASE('?')
          !--------
            CONTINUE

          CASE('archie')
          !-------------
            jselect_torturel = jp_torturel_archie
            READ(iu_cfg, NML=nml_ttp_archie)

          CASE('burgerfrieke')
          !-------------------
            jselect_torturel = jp_torturel_burfrie

          CASE('modweissberg')
          !-------------------
            jselect_torturel = jp_torturel_modwei

          CASE('custom')
          !-------------
            WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
            WRITE(jp_stderr, cfmt_a, ADVANCE="NO")
     &        '"custom" tortuosity relationship not yet available' //
     &        ' -- aborting'
            CALL ABORT_MEDUSA()

          CASE DEFAULT
          !-----------
            WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
            WRITE(jp_stderr, cfmt_a, ADVANCE="NO")
     &        'unknown tortuosity relationship "' //
     &        TRIM(ctype_tortuosity_rel) // '" -- aborting'
            CALL ABORT_MEDUSA()

          END SELECT

                                    ! Sanity checks

          !IF (... < 0.0D+00) THEN
          !  ! ... must be >= 0!
          !  ! Print out jselect_poroprof
          !  CALL ABORT_MEDUSA()
          !ENDIF


          CLOSE(UNIT=iu_cfg)


#ifdef DEBUG
          WRITE(jp_stddbg, cfmt_a, ADVANCE="NO")
     &      'Releasing logical unit number '
          WRITE(jp_stddbg, '(I0, A)', ADVANCE="NO")
     &      iu_cfg, ' (attached to "iu_cfg")'
#endif
          istatus = FREE_LOGUNIT(iu_cfg)
          IF (istatus /= 0) THEN
            WRITE(jp_stderr, cfmt_modprocname_a) 'Warning'
            WRITE(jp_stderr, cfmt_a, ADVANCE="NO")
     &        'FREE_LOGUNIT returned error code '
            WRITE(jp_stderr, '(I0, A)') istatus,
     &        ' when trying to release the logical unit number'
     &        //  ' attached to "iu_cfg" -- ignoring.'
#ifdef DEBUG
            WRITE(jp_stddbg, '()')
            WRITE(jp_stddbg, cfmt_a_ind, ADVANCE="NO")
            WRITE(jp_stddbg, '(A, I0, A)')
     &        'FREE_LOGUNIT returned error code ', istatus,
     &        ' when trying to release the logical unit number'
     &        //  ' attached to "iu_cfg" -- ignoring.'
          ELSE
            WRITE(jp_stddbg, '(" - done.")')
#endif
          ENDIF

#ifdef DEBUG
        ELSE

          WRITE(jp_stddbg, cfmt_a)
     &      'Milieu configuration file "' // cfn_milieu_cfg //
     &      '" not found - falling back to defaults'
#endif
        ENDIF

        CALL FLUSH(jp_stderr)
#ifdef DEBUG
        CALL FLUSH(jp_stddbg)
#endif

      ENDIF


#ifdef ALLOW_MPI
                                    ! Broadcast the configuration data:
                                    !  - phi variability
      CALL MPI_BCAST(jselect_phitype, 1, MPI_INTEGER,
     &                              jp_exeproc_root, i_mycomm, istatus)

                                    !  - porosity profile
      CALL MPI_BCAST(jselect_poroprof, 1, MPI_INTEGER,
     &                              jp_exeproc_root, i_mycomm, istatus)

                                    !  - tortuosity-porosity relationship
      CALL MPI_BCAST(jselect_torturel, 1, MPI_INTEGER,
     &                              jp_exeproc_root, i_mycomm, istatus)


                                    ! Broadcast the other parameter values
                                    ! ONLY if NO custom profile is used.
      IF (jselect_poroprof /= jp_poroprof_custom) THEN
                                    !  - phi_0 (all)
        CALL MPI_BCAST(phi_0, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)

        SELECT CASE(jselect_poroprof)
        CASE(jp_poroprof_expdec)
                                    !  - phi_infty (expdec)
          CALL MPI_BCAST(phi_infty, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
                                    !  - phi_scale (expdec)
          CALL MPI_BCAST(phi_scale, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
        END SELECT

      ENDIF


      IF (jselect_torturel /= jp_torturel_custom) THEN

        SELECT CASE(jselect_torturel)
          CASE(jp_torturel_archie)
                                    !  - archie (archie_m)
            CALL MPI_BCAST(archie_m, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
        END SELECT

      ENDIF
#endif


#ifdef PHI_CUSTOM
      IF (jselect_poroprof == jp_poroprof_custom) THEN
                                    ! Complete the set-up of the custom
                                    ! grid-point distributions.
        CALL PHI_CUSTOM_SETUP(2, iu_cfg) ! stage 2; iu_cfg is ignored
      ENDIF
#endif


                                    ! With custom porosity profiles,
                                    ! all has been done in PHI_CUSTOM_SETUP
                                    ! at stage 2
      IF (jselect_poroprof /= jp_poroprof_custom) THEN


        WRITE(jp_stdlog, '(" - porosity profile")')
        WRITE(jp_stdlog, '("   * profile ID: ", I0)')
     &                              jselect_poroprof

        SELECT CASE(jselect_poroprof)
        CASE(jp_poroprof_const)
          da_phi0 = phi_0
          WRITE(jp_stdlog, '("   * phi_0: ", E9.3)')
     &                              da_phi0

        CASE(jp_poroprof_expdec)
          da_phi0  = phi_0
          da_phi00 = phi_infty
          WRITE(jp_stdlog, '("   * phi_0: ", E9.3)')
     &                              da_phi0
          WRITE(jp_stdlog, '("   * phi_infty: ", E9.3)')
     &                              da_phi00
          WRITE(jp_stdlog, '("   * phi_scale: ", E9.3)')
     &                              da_phiscale

        END SELECT

        WRITE(jp_stdlog, '()')

      ENDIF


      IF (jselect_torturel /= jp_torturel_custom) THEN


        WRITE(jp_stdlog, '(" - tortuosity-porosity relationship")')
        WRITE(jp_stdlog, '("   * relationship ID: ", I0)')
     &                              jselect_torturel

        SELECT CASE(jselect_torturel)
        CASE(jp_torturel_archie)
          da_archie_m = archie_m
          WRITE(jp_stdlog, '("   * archie_m: ", E9.3)')
     &                              da_archie_m

        CASE(jp_torturel_burfrie)
          da_burfrie_a = dp_burfrie_a
          WRITE(jp_stdlog, '("   * burfrie_a: ", E9.3)')
     &                              da_burfrie_a

        CASE(jp_torturel_modwei)
          da_modweiss_b = dp_modweiss_b
          WRITE(jp_stdlog, '("   * modweiss_b: ", E9.3)')
     &                              da_modweiss_b

        END SELECT

        WRITE(jp_stdlog, '()')

      ENDIF


      WRITE(jp_stdlog, cfmt_modprocname_a) 'End of initial report'
      WRITE(jp_stdlog, '()')
      WRITE(jp_stdlog, '()')


      l_setupdone = .TRUE.

#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_modprocname_a) 'End'
      WRITE(jp_stddbg, '()')
#endif


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE SETUP_MILIEUCHARAS
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      INTEGER FUNCTION SELECTED_PHITYPE()
!-----------------------------------------------------------------------

      IMPLICIT NONE

      IF (l_setupdone) THEN
        SELECTED_PHITYPE = jselect_phitype
      ELSE
        SELECTED_PHITYPE = jp_phi_notsetup
      ENDIF

      RETURN

!-----------------------------------------------------------------------
      END FUNCTION SELECTED_PHITYPE
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE POROTORTUOSITY_DEF(xzdn, xzdv)
!-----------------------------------------------------------------------

      USE mod_gridparam, ONLY: idnw, idnt, idnb, idvw, idvs, idvb

      IMPLICIT NONE


      DOUBLE PRECISION, DIMENSION(idnw:idnb), INTENT(IN) ::  xzdn
      DOUBLE PRECISION, DIMENSION(idvw:idvb), INTENT(IN) ::  xzdv


      IF (.NOT. l_setupdone) CALL SETUP_MILIEUCHARAS


      !-------------------
      ! Porosity profiles
      !-------------------

      SELECT CASE(jselect_poroprof)
#ifdef PHI_CUSTOM
      CASE(jp_poroprof_custom)      ! custom
        CALL POROSITY_CUSTOM_DEF(xzdn)
        CALL POROSITY_CUSTOM_VRTX(xzdn)
#endif

      CASE(jp_poroprof_const)       ! constant
        xvphi(:)    = da_phi0
        xvdphi(:)   = 0.0D+00

        xphi(:)     = da_phi0
        xdphi(:)    = 0.0D+00

      CASE(jp_poroprof_expdec)      ! exponentially decreasing

        xvphi(idvs)     = da_phi0
        xvphi(idvs+1:)  = (da_phi0 - da_phi00)
     &              * EXP(-(xzdv(idvs+1:)/da_phiscale)) + da_phi00

        xvdphi(:)  = -(xvphi(:)-da_phi00)/da_phiscale

                                     ! Set xphi(:) to the average
                                     ! value of xvphi(:) in each cell.
        xphi(idnt:idnb)
     &               = da_phi00
     &                 - da_phiscale
     &                   * (xvphi(idnt:idnb) - xvphi(idnt-1:idnb-1))
     &                     / (xzdv(idnt:idnb) - xzdv(idnt-1:idnb-1))

        xdphi(:)  = -(xphi(:)-da_phi00)/da_phiscale


      END SELECT

      yphi = xvphi(idvb)
      zphi = xvphi(idvb)


      CALL TORTUOSITY_UPDATE()


      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE POROTORTUOSITY_DEF
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE TORTUOSITY_UPDATE
!-----------------------------------------------------------------------


      IMPLICIT NONE


      !---------------------
      ! Tortuosity profiles
      !---------------------

      SELECT CASE(jselect_torturel)
      CASE(jp_torturel_archie)      ! Archie's law
        xvtor2(:)  = xvphi(:)**(1.0D+00-da_archie_m)

      CASE(jp_torturel_burfrie)     ! Burger-Frieke eqn.
        xvtor2(:)  = xvphi(:) + da_burfrie_a * (1.0D+00 - xvphi(:))

      CASE(jp_torturel_modwei)      ! modified Weissberg rel.
        xvtor2(:)  = 1.0D+00 - da_modweiss_b * LOG(xvphi(:))

      END SELECT

      ytor2 = xvtor2(idvb)
      ztor2 = xvtor2(idvb)

      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE TORTUOSITY_UPDATE
!-----------------------------------------------------------------------



#ifdef PHI_CUSTOM
!     SUBROUTINE POROSITY_CUSTOM_SETUP(k_stage, iu_cfg)
!     SUBROUTINE POROSITY_CUSTOM(xzdn, xzdv)
#include <porosity_custom.F>
#endif


!=======================================================================
      END MODULE MOD_MILIEUCHARAS
!=======================================================================
