!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


!=======================================================================
      MODULE MOD_GRIDPARAM
!=======================================================================


      IMPLICIT NONE


      ! ======== Master parameters (can only be changed here) ======== !


                                    ! Number of grid NODES located in the
                                    ! interval [z_W, z_SWI[ (SWI excluded
                                    ! if z_W < z_SWI), 0 if z_W == z_SWI
      INTEGER, PARAMETER :: ndn_w2s =  0

                                    ! Number of grid NODES in the
                                    ! bioturbated part of REACLAY, i.e.,
                                    ! number of nodes located in the
                                    ! interval [z_SWI, z_Z].
      INTEGER, PARAMETER :: ndn_s2z = 21

                                    ! Number of grid NODES below the
                                    ! bioturbated part of REACLAY, i.e.,
                                    ! number of nodes located in the
                                    ! interval ]z_Z, z_B] (Z node excluded),
                                    ! set to 0 if z_B == z_Z
      INTEGER, PARAMETER :: ndn_z2b =  0

                                    ! CAVEAT AND NOTICE:
                                    ! =================
                                    ! - if a DBL is included (ndn_w2s /= 0)
                                    !   then the T node is located below
                                    !   the SWI; if no DBL is included,
                                    !   the T node is located at the SWI.
                                    ! - ndn_s2z + ndn_z2b must currently be
                                    !   greater than 1 (i.e., 2 or greater)


                                    ! Number of historical (core) layers,
                                    ! not including the transition layer
      INTEGER, PARAMETER :: nhl     = 10

                                    ! Parameters for setting the characteristic
                                    ! "center" of the top and bottom "half cells":
                                    !  - at the top (around idnw)
                                    ! xxx("center") =  (1-thetatop)*xxx(idnw)
                                    !                 +   thetatop *xxx(idnw+1)
      DOUBLE PRECISION, PARAMETER :: thetatop = 0.0D+00

                                    !  - at the bottom (around idnb)
                                    !    xxx("center") =      thetabot *xxx(idnb-1)
                                    !                    + (1-thetabot)*xxx(idnb)
      DOUBLE PRECISION, PARAMETER :: thetabot = 0.0D+00


      !----------------------------------------------------------------!
      !                                                                !
      !        Additional configuration parameters and variables       !
      !        =================================================       !
      !                                                                !
      !   Some of these could, some of them MUST NOT be changed here!  !
      !                                                                !
      !   Every sensible change can and should be made via the         !
      !   medusa_grid_config.nml file (read in at run time), for which !
      !   a template is provided in the src-med/templates directory.   !
      !                                                                !
      !----------------------------------------------------------------!


                                    ! PUBLIC/PRIVATE declarations
                                    ! ===========================
      PRIVATE

      PUBLIC :: jp_grid_notsetup
      PUBLIC :: jp_grid_dynamic
      PUBLIC :: jp_grid_static_local, jp_grid_static_global

      PUBLIC :: dp_swi_location

      PUBLIC :: da_gpd_dreaclay, da_gpd_dcorelay

      PUBLIC :: ndn_w2s, ndn_s2b, ndn, ndv, nhl
      PUBLIC :: ndo_w2s, ndo_z2b
      PUBLIC :: idnw, idnt, idnz, idnb, ihl1, ihln
      PUBLIC :: idvw, idvs,       idvb, idvaz

      PUBLIC :: thetatop, thetabot

      PUBLIC :: SETUP_GRID, SELECTED_GRIDTYPE, GRID_DEF, GRID_VTX


                                    ! Available options
                                    ! =================

                                    ! Numbering of the currently
                                    ! available grid variability options
                                    ! (DO NOT CHANGE!)
      INTEGER, PARAMETER :: jp_grid_notsetup      = -1
      INTEGER, PARAMETER :: jp_grid_dynamic       =  1
      INTEGER, PARAMETER :: jp_grid_static_local  =  2
      INTEGER, PARAMETER :: jp_grid_static_global =  3

                                    ! Numbering of the different
                                    ! grid-point distribution options
                                    ! (DO NOT CHANGE!)
      INTEGER, PARAMETER :: jp_gpd_custom   = 0     ! CUSTOM distribution of nodes
      INTEGER, PARAMETER :: jp_gpd_linear   = 1     ! LINEAR distribution of nodes
                                                    ! from top to bottom
      INTEGER, PARAMETER :: jp_gpd_qtoplbot = 2     ! Quadratic TOP, Linear BOTtom
      INTEGER, PARAMETER :: jp_gpd_ptoplbot = 3     ! Power-n TOP, Linear BOTtom
      INTEGER, PARAMETER :: jp_gpd_geoproop = 4     ! GEOmetric PROgression, OPen
      INTEGER, PARAMETER :: jp_gpd_geoprocl = 5     ! GEOmetric PROgression, CLosed
      INTEGER, PARAMETER :: jp_gpd_glogiser = 6     ! Generalised LOGIstic SERies

                                    ! Grid layouts (DO NOT CHANGE!)
      INTEGER, PARAMETER :: jp_glo_n2n = 1          ! node-to-node
      INTEGER, PARAMETER :: jp_glo_n2v = 2          ! node-to-vertex
      INTEGER, PARAMETER :: jp_glo_v2n = 3          ! vertex-to-node
      INTEGER, PARAMETER :: jp_glo_v2v = 4          ! vertex-to-vertex


                                    ! Default geometry values
                                    ! =======================

                                    ! Thickness of diffusive boundary
                                    ! layer (DBL realm,  [m])
                                    ! (only considered if ndn_w2s =/ 0)
      DOUBLE PRECISION, PARAMETER
     &  :: dp_dbl_thickness       = 0.001D+00

                                    ! Thickness of the upper sediment
                                    ! (REACLAY realm, excluding a
                                    ! possible DBL, [m])
      DOUBLE PRECISION, PARAMETER
     &  :: dp_sedmixlay_thickness = 0.10D+00

                                    ! Bioturbation depth (thickness of
                                    ! the topmost part of REACLAY that
                                    ! may be boioturbated, [m])
      DOUBLE PRECISION, PARAMETER
     &  :: dp_biotur_depth        = 0.10D+00

                                    ! Thickness of historical layers
                                    ! (CORELAY realm, [m])
      DOUBLE PRECISION, PARAMETER
     &  :: dp_historlay_thickness = 0.01D+00


                                    ! Specific default values
                                    ! =======================

                                    ! Grid point distribution parameters
                                    ! - linear: none

                                    ! - quad_lin:
      DOUBLE PRECISION, PARAMETER :: dp_qtoplbot_qtopfrac_t2z = 0.5D+00
      DOUBLE PRECISION, PARAMETER :: dp_qtoplbot_qtopfrac_z2b = 0.5D+00

                                    ! - pown_lin:
      DOUBLE PRECISION, PARAMETER :: dp_ptoplbot_ptopfrac_t2z = 0.5D+00
      DOUBLE PRECISION, PARAMETER :: dp_ptoplbot_ptopfrac_z2b = 0.5D+00
      DOUBLE PRECISION, PARAMETER :: dp_ptoplbot_pn_t2z       = 2.0D+00
      DOUBLE PRECISION, PARAMETER :: dp_ptoplbot_pn_z2b       = 2.0D+00

                                    ! - geomprog_open:
      DOUBLE PRECISION, PARAMETER :: dp_geoproxx_dreaclay1 = 0.001D+00
      DOUBLE PRECISION, PARAMETER :: dp_geoproop_progratio = 1.06D+00

                                    ! - geomprog_closed: none extra
                                    !   uses dp_geoproxx_dreaclay1

                                    ! - genlogist_ser:
      DOUBLE PRECISION, PARAMETER :: dp_glogiser_exponent   =  1.0D+00
      DOUBLE PRECISION, PARAMETER :: dp_glogiser_growthrate = 15.0D+00
      DOUBLE PRECISION, PARAMETER :: dp_glogiser_frac_f0f00 =  2.0D-04


                                    ! Actually used values
                                    ! ====================

                                    ! - grid variablility
      INTEGER, SAVE :: jselect_gridtype      = jp_grid_static_global

                                    ! - general geometry
      DOUBLE PRECISION, SAVE      ::
     &        da_gpd_ddifblay = dp_dbl_thickness
      DOUBLE PRECISION, SAVE      ::
     &        da_gpd_dreaclay = dp_sedmixlay_thickness
      DOUBLE PRECISION, SAVE      ::
     &        da_gpd_dbioturb = dp_biotur_depth
      DOUBLE PRECISION, SAVE      ::
     &        da_gpd_dcorelay = dp_historlay_thickness

                                    ! - grid-point distribution profiles
      INTEGER, SAVE :: jselect_gridptdistrib = jp_gpd_qtoplbot

      DOUBLE PRECISION, SAVE      ::
     &        da_gpd_qtoplbot_qfrac_t2z = dp_qtoplbot_qtopfrac_t2z
      DOUBLE PRECISION, SAVE      ::
     &        da_gpd_qtoplbot_qfrac_z2b = dp_qtoplbot_qtopfrac_z2b

      DOUBLE PRECISION, SAVE      ::
     &        da_gpd_ptoplbot_pfrac_t2z = dp_ptoplbot_ptopfrac_t2z
      DOUBLE PRECISION, SAVE      ::
     &        da_gpd_ptoplbot_pfrac_z2b = dp_ptoplbot_ptopfrac_z2b
      DOUBLE PRECISION, SAVE      ::
     &        da_gpd_ptoplbot_pn_t2z    = dp_ptoplbot_pn_t2z
      DOUBLE PRECISION, SAVE      ::
     &        da_gpd_ptoplbot_pn_z2b    = dp_ptoplbot_pn_z2b

      DOUBLE PRECISION, SAVE      ::
     &        da_gpd_geoproxx_dreaclay1 = dp_geoproxx_dreaclay1

      DOUBLE PRECISION, SAVE      ::
     &        da_gpd_geoproop_progratio = dp_geoproop_progratio
      DOUBLE PRECISION, SAVE      ::
     &        da_gpd_geoprocl_progratio = 0.0D+00

      DOUBLE PRECISION, SAVE      ::
     &        da_gpd_glogiser_exponent   = dp_glogiser_exponent
      DOUBLE PRECISION, SAVE      ::
     &        da_gpd_glogiser_growthrate = dp_glogiser_growthrate
      DOUBLE PRECISION, SAVE      ::
     &        da_gpd_glogiser_frac_f0f00 = dp_glogiser_frac_f0f00


! = = = = = = = No configurable anything below this line = = = = = = = !


                                    ! Location of the Sediment-Water
                                    ! Interface (SWI) on the vertical
                                    ! axis ([m]).
                                    ! !!! DO NOT CHANGE !!!
                                    ! A lot of unpredictable things can
                                    ! break if this is changed.
      DOUBLE PRECISION, PARAMETER :: dp_swi_location = 0.0D+00


                                    ! Grid point indices and numbers
                                    ! ==============================

                                    ! Master parameters
                                    ! -----------------
                                    ! - ndn_w2s : number of nodes within
                                    !             [z_W, z_SWI[ (declared above)
                                    ! - ndn_s2z : number of nodes within
                                    !             [z_SWI, z_Z] (declared above)
                                    ! - ndn_z2b : number of nodes within
                                    !             ]z_Z, z_B] (declared above)
                                    ! - nhl     : number of historical layers,
                                    !             not including the transition
                                    !             layer (declared above)


                                    ! Derived parameters
                                    ! ------------------
                                    ! - ndo_w2s : offset if a DBL is included
                                    !             ndo_w2s = 0 if ndn_w2s == 0
                                    !             ndo_w2s = 1 if ndn_w2s /= 0
                                    ! - ndo_z2b : offset if REACLAY includes a
                                    !             non-bioturbated part
                                    !             ndo_z2b = 0 if ndn_z2b == 0
                                    !             ndo_z2b = 1 if ndn_z2b /= 0

      INTEGER, PARAMETER :: ndo_w2s = (ndn_w2s+ndn_w2s)/(ndn_w2s+1)
      INTEGER, PARAMETER :: ndo_z2b = (ndn_z2b+ndn_z2b)/(ndn_z2b+1)


                                    ! - ndn_s2b : number of nodes covering
                                    !             [z_SWI, z_B]
                                    ! - ndn     : total number of nodes

      INTEGER, PARAMETER :: ndn_s2b = ndn_s2z + ndn_z2b
      INTEGER, PARAMETER :: ndn     = ndn_w2s + ndn_s2b


                                    ! - idnt    : index of the first REACLAY node
                                    !             (Index of the Depth Node at the Top)
                                    ! - idnw    : index of the node at the water
                                    !             interface, either the top node, if no
                                    !             DBL is included, or the top node of
                                    !             the DBL (Index of the Depth Node at
                                    !             the free Water interface)
                                    ! - idnz    : index locating the depth node
                                    !             where the bottom interface of
                                    !             the bioturbated region is located:
                                    !              * D_B(i)  > 0 when i < idnz
                                    !              * D_B(i)  = 0 when i > idnz
                                    !              * D_B(idnz) may or may not be
                                    !                equal to 0. This depends on
                                    !                whether the biodiffusion
                                    !                coefficient is continuous
                                    !                across x=xzdn(idnz) or not.
                                    ! - idnb    : index of the last REACLAY node
                                    !             (Index of the Depth Node at the Bottom)

      INTEGER, PARAMETER :: idnt    = 0
      INTEGER, PARAMETER :: idnw    = idnt - ndn_w2s
      INTEGER, PARAMETER :: idnz    = idnt + ndn_s2z - 1
      INTEGER, PARAMETER :: idnb    = idnt + ndn_s2b - 1


                                    ! - idvs    : index of the vertex (possibly
                                    !             virtual) at the SWI
                                    ! - idvw    : index of the virtual vertex at
                                    !             the W node
                                    ! - idvaz   : index of the vertex just above
                                    !             the Z node
                                    ! - idvb    : index of the virtual vertex at
                                    !             the B node

      INTEGER, PARAMETER :: idvs    = idnt - 1
      INTEGER, PARAMETER :: idvw    = idnw - 1
      INTEGER, PARAMETER :: idvaz   = idnz - 1
      INTEGER, PARAMETER :: idvb    = idnb

      INTEGER, PARAMETER :: ndv     = ndn + 1   ! = idvb - idvw + 1
                                                !   = idnb - (idnw - 1) + 1
                                                !   = idnt + ndn_s2b - 1 - (idnt - ndn_w2s - 1) + 1
                                                !   = ndn_s2b + ndn_w2s + 1
                                                !   = ndn + 1


                                    ! - ihl0    : index of transition layer
                                    ! - ihl1    : index of first (top) historical layer
                                    ! - ihln    : index of last (deepest) historical layer

      INTEGER, PARAMETER :: ihl0    = 0
      INTEGER, PARAMETER :: ihl1    = 1
      INTEGER, PARAMETER :: ihln    = nhl


#ifdef GRID_CUSTOM
#define GRID_CUSTOM_DECLARATIONS
#include <gridef_custom.F>
#undef GRID_CUSTOM_DECLARATIONS
#endif


      LOGICAL, SAVE               :: l_setupdone = .FALSE.


      CONTAINS


!-----------------------------------------------------------------------
      SUBROUTINE SETUP_GRID
!-----------------------------------------------------------------------

!--------------
! Declarations
!--------------


      USE mod_defines_medusa,       ONLY: jp_stderr, jp_stdlog
#ifdef DEBUG
      USE mod_defines_medusa,       ONLY: jp_stddbg
#endif
      USE mod_execontrol_medusa,    ONLY: ABORT_MEDUSA
      USE mod_logunits

#ifdef ALLOW_MPI
      USE mod_execontrol_medusa,    ONLY: MEDEXE_MPI_COMM,
     &                              MEDEXE_MPI_COMM_RANK,
     &                              jp_exeproc_root
      USE mpi, ONLY: MPI_INTEGER, MPI_DOUBLE_PRECISION
#endif


      IMPLICIT NONE


!-----------------------
! Variable declarations
!-----------------------

!- - - - - - - - - - - - -  - - - - - - -
! Variables in subroutine call arguments
!- - - - - - - - - - - - -- - - - - - - -

      ! None


!- - - - - - - - - - - - - -  - - - - - - - -
! General (global) parameters and definitions
!- - - - - - - - - - - - - -- - - - - - - - -

      CHARACTER(LEN=*), PARAMETER ::
     &  cfn_grid_cfg = "medusa_grid_config.nml"


!- - - - - - - - -
! Local variables
!- - - - - - - - -


#ifdef ALLOW_MPI
      INTEGER :: i_mycomm, i_myrank
#endif




      CHARACTER(LEN=15) :: ctype_grid_variability
      CHARACTER(LEN=15) :: ctype_gridpoint_distribution

      NAMELIST /nml_grid_options/   ctype_grid_variability,
     &                              ctype_gridpoint_distribution


      DOUBLE PRECISION :: ddifblay
      DOUBLE PRECISION :: dreaclay, dbioturb
      DOUBLE PRECISION :: dcorelay

      DOUBLE PRECISION :: qtop_frac_t2z, qtop_frac_z2b
      DOUBLE PRECISION :: ptop_frac_t2z, ptop_frac_z2b
      DOUBLE PRECISION :: pown_t2z, pown_z2b
      DOUBLE PRECISION :: dreaclay1, rreaclay
      DOUBLE PRECISION :: expon, grate, f0_f00


      NAMELIST /nml_gpd_linear/  ddifblay,
     &                           dreaclay, dbioturb,
     &                           dcorelay
      NAMELIST /nml_gpd_quad_lin/
     &                           ddifblay,
     &                           dreaclay, dbioturb,
     &                           dcorelay,
     &                           qtop_frac_t2z, qtop_frac_z2b
      NAMELIST /nml_gpd_pown_lin/
     &                           ddifblay,
     &                           dreaclay, dbioturb,
     &                           dcorelay,
     &                           ptop_frac_t2z, ptop_frac_z2b,
     &                           pown_t2z, pown_z2b
      NAMELIST /nml_gpd_geomprog_open/
     &                           ddifblay,
     &                           dcorelay,
     &                           dreaclay1, rreaclay
      NAMELIST /nml_gpd_geomprog_closed/
     &                           ddifblay,
     &                           dreaclay, dbioturb,
     &                           dcorelay,
     &                           dreaclay1
      NAMELIST /nml_gpd_genlogist_ser/
     &                           ddifblay,
     &                           dreaclay, dbioturb,
     &                           dcorelay,
     &                           expon, grate, f0_f00


      INTEGER :: iu_cfg, istatus
      LOGICAL :: l_cfgfile_is_mine = .FALSE.
      LOGICAL :: l_exists


!- - - - - - - - - - - - - -
! Standard I/O related data
!- - - - - - - - - - - - - -


      CHARACTER(LEN=*), PARAMETER :: cfmt_modprocname_a
     &  = '("[MOD_GRIDPARAM/setup_grid]: ", A)'

      CHARACTER(LEN=*), PARAMETER :: cfmt_a     = '(" - ", A)'
      CHARACTER(LEN=*), PARAMETER :: cfmt_a_ind = '(3X, A)'


!- - - - - - - - - - -
! End of declarations
!- - - - - - - - - - -


!----------------------------------------------------------------------
! Subroutine Start
!----------------------------------------------------------------------

#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_modprocname_a) 'Start'
#endif

      IF (l_setupdone) THEN
#ifdef DEBUG
        WRITE(jp_stddbg, cfmt_a) 'Setup already done'
        WRITE(jp_stddbg, cfmt_modprocname_a) 'Return'
        WRITE(jp_stddbg, '()')
#endif
        RETURN
      ENDIF

#ifdef ALLOW_MPI
      i_mycomm = MEDEXE_MPI_COMM()
      i_myrank = MEDEXE_MPI_COMM_RANK()
      l_cfgfile_is_mine = (i_myrank == jp_exeproc_root)
#else
      l_cfgfile_is_mine = .TRUE.
#endif

      WRITE(jp_stdlog, cfmt_modprocname_a) 'Initial report'
      WRITE(jp_stdlog, '()')
      WRITE(jp_stdlog, '(" Adopted grid options")')
      WRITE(jp_stdlog, '(" --------------------")')
      WRITE(jp_stdlog, '()')


      IF (l_cfgfile_is_mine) THEN

                                    ! Pre-set the default values
        jselect_gridtype = jp_grid_static_global
        IF (ndn_w2s == 0) THEN
          ddifblay       = 0.0D+00
        ELSE
          ddifblay       = dp_dbl_thickness
        ENDIF
        dreaclay         = dp_sedmixlay_thickness
        dbioturb         = dp_biotur_depth
        dcorelay         = dp_historlay_thickness

        jselect_gridptdistrib = jp_gpd_qtoplbot
        qtop_frac_t2z = dp_qtoplbot_qtopfrac_t2z
        qtop_frac_z2b = dp_qtoplbot_qtopfrac_z2b

                                    ! Check if file cfn_grid_cfg exists
        INQUIRE(FILE=cfn_grid_cfg, EXIST=l_exists)


        IF (l_exists) THEN          ! If file exists, read it and get
                                    ! information (else stick to defaults)

#ifdef DEBUG
          WRITE(jp_stddbg, cfmt_a)
     &      'Grid configuration file "' // cfn_grid_cfg // '" found.'
#endif

          istatus = RESERVE_LOGUNIT(iu_cfg)
          IF (istatus /= 0) THEN
            WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
            WRITE(jp_stderr, cfmt_a)
     &       'Unable to reserve a logical unit for "iu_cfg" -- aborting'
            CALL ABORT_MEDUSA()
#ifdef DEBUG
          ELSE
            WRITE(jp_stddbg, cfmt_a, ADVANCE="NO")
     &        'Assigning logical unit number '
            WRITE(jp_stddbg, '(I0, A)') iu_cfg, ' to "iu_cfg".'
#endif
          ENDIF


          OPEN(UNIT=iu_cfg, FILE=cfn_grid_cfg)

                                    ! Pre-set the two main options
                                    ! to "unknown" (triggering the
                                    ! use of default values if not
                                    ! modified below).
          ctype_grid_variability       = '?'
          ctype_gridpoint_distribution = '?'

          READ(iu_cfg, NML=nml_grid_options)


#ifdef DEBUG
          WRITE(jp_stddbg, '()')
          WRITE(jp_stddbg, cfmt_a_ind) 'Grid options requested'
          WRITE(jp_stddbg, cfmt_a_ind) '======================'
          WRITE(jp_stddbg, '()')

          WRITE(jp_stddbg, cfmt_a_ind, ADVANCE="NO")
          IF (ctype_grid_variability == '?') THEN
            WRITE(jp_stddbg, '(" * ctype_grid_variability = ", A)')
     &                              '"static_global" (DEFAULT)'
          ELSE
            WRITE(jp_stddbg, '(" * ctype_grid_variability = ", A)')
     &                   '"' // TRIM(ctype_grid_variability) // '"'
          ENDIF
          WRITE(jp_stddbg, '()')


          WRITE(jp_stddbg, cfmt_a_ind, ADVANCE="NO")
          IF (ctype_gridpoint_distribution == '?') THEN
            WRITE(jp_stddbg, '(" * ctype_gridpoint_distribution = ",A)')
     &                              '"quad_lin" (DEFAULT)'
          ELSE
            WRITE(jp_stddbg, '(" * ctype_gridpoint_distribution = ",A)')
     &                  '"' // TRIM(ctype_gridpoint_distribution) // '"'
          ENDIF
          WRITE(jp_stddbg, '()')
#endif


                                    ! Grid type
                                    ! =========

          SELECT CASE(ctype_grid_variability)
          CASE('?')
          !--------
            CONTINUE

          CASE('static_global')
          !--------------------
            jselect_gridtype = jp_grid_static_global

          CASE('static_local')
          !-------------------
            jselect_gridtype = jp_grid_static_local

          CASE('dynamic')
          !------------
            jselect_gridtype = jp_grid_dynamic

          CASE DEFAULT
          !-----------
            WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
            WRITE(jp_stderr, cfmt_a)
     &        'unknown grid type "' //
     &        TRIM(ctype_grid_variability) // '" -- aborting'
            CALL ABORT_MEDUSA()

          END SELECT



                                    ! Grid -point distribution
                                    ! ========================

          SELECT CASE(ctype_gridpoint_distribution)
          CASE('?')
          !--------
            CONTINUE

          CASE('linear')
          !-----------
            jselect_gridptdistrib = jp_gpd_linear

            READ(iu_cfg, NML=nml_gpd_linear)

          CASE('quad_lin')
          !---------------
            jselect_gridptdistrib = jp_gpd_qtoplbot

            READ(iu_cfg, NML=nml_gpd_quad_lin)

            ! To be tested: 0 < qtop_frac_t2z < 1
            ! To be tested: 0 < qtop_frac_z2b < 1
            !IF (... <= 0.0D+00) THEN
            !  CALL ABORT_MEDUSA()
            !ENDIF

          CASE('pown_lin')
          !---------------
            jselect_gridptdistrib = jp_gpd_ptoplbot

            READ(iu_cfg, NML=nml_gpd_pown_lin)

            ! To be tested: 0 < ptop_frac_t2z < 1
            ! To be tested: 0 < ptop_frac_z2b < 1
            ! To be tested: 0 < pown_t2z
            ! To be tested: 0 < pown_z2b
            !IF (... <= 0.0D+00) THEN
            !  CALL ABORT_MEDUSA()
            !ENDIF

          CASE('geomprog_open')
          !--------------------
            jselect_gridptdistrib = jp_gpd_geoproop

            dreaclay1 = dp_geoproxx_dreaclay1
            rreaclay  = dp_geoproop_progratio
            READ(iu_cfg, NML=nml_gpd_geomprog_open)

            ! To be tested: 0 < dreaclay1
            ! To be tested: 0 < rreaclay
            !IF (... <= 0.0D+00) THEN
            !  ! Print out jselect_gridpointdistribution
            !  CALL ABORT_MEDUSA()
            !ENDIF

          CASE('geomprog_closed')
          !----------------------
            jselect_gridptdistrib = jp_gpd_geoprocl

            dreaclay1 = dp_geoproxx_dreaclay1
            READ(iu_cfg, NML=nml_gpd_geomprog_closed)

            ! To be tested: 0 < dreaclay1 < dreaclay
            !IF (... <= 0.0D+00) THEN
            !  ! Print out jselect_gridpointdistribution
            !  CALL ABORT_MEDUSA()
            !ENDIF

          CASE('genlogist_ser')
          !--------------------
            jselect_gridptdistrib = jp_gpd_glogiser

            expon  = dp_glogiser_exponent
            grate  = dp_glogiser_growthrate
            f0_f00 = dp_glogiser_frac_f0f00
            READ(iu_cfg, NML=nml_gpd_genlogist_ser)

            ! To be tested: 0 < exponent
            ! To be tested: 0 < grate
            ! To be tested: 0 < f0_f00 < 1
            !IF (... <= 0.0D+00) THEN
            !  CALL ABORT_MEDUSA()
            !ENDIF

          CASE('custom')
          !-------------
#ifdef GRID_CUSTOM
            jselect_gridptdistrib = jp_gpd_custom

                                    ! Carry out stage 1 of the set-up
                                    ! of the custom grid-point distribution
                                    ! (reading in the data from the
                                    ! file 'iu_cfg', currently open)
            CALL GRIDEF_CUSTOM_SETUP(1, iu_cfg)
#else
            WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
            WRITE(jp_stderr, cfmt_a)
     &        '"custom" grid-point distribution not available' //
     &        ' -- aborting'
            CALL ABORT_MEDUSA()
#endif

          CASE DEFAULT
          !-----------
            WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
            WRITE(jp_stderr, cfmt_a)
     &        'unknown grid-point distribution "' //
     &        TRIM(ctype_gridpoint_distribution) // '" -- aborting'
            CALL ABORT_MEDUSA()

          END SELECT

                                    ! Sanity checks

                                    !  - ddifblay = 0 if ndn_w2s == 0 (auto-correct)
                                    !  - ddifblay > 0 if ndn_w2s > 0 (auto-correct)
          IF (ndn_w2s == 0) THEN

            IF (ddifblay /= 0.0D+00) THEN

              WRITE(jp_stderr, cfmt_modprocname_a) 'Warning'
              WRITE(jp_stderr, cfmt_a) 'Incorrect DBL thickness'
              WRITE(jp_stderr, cfmt_a_ind)
     &          ' * expected: 0.0D+00 (no DBL)'
              WRITE(jp_stderr, cfmt_a_ind, ADVANCE="NO")
     &          ' * actually: '
              WRITE(jp_stderr, '(E9.3)') ddifblay
              WRITE(jp_stderr, cfmt_a_ind) 'Resetting it to 0.0D+00'

              ddifblay = 0.0D+00

            ENDIF

          ELSE

            IF (ddifblay <= 0.0D+00) THEN

              WRITE(jp_stderr, cfmt_modprocname_a) 'Warning'
              WRITE(jp_stderr, cfmt_a) 'Incorrect DBL thickness'
              WRITE(jp_stderr, cfmt_a_ind)
     &          ' * expected: strictly positive'
              WRITE(jp_stderr, cfmt_a_ind, ADVANCE="NO")
     &          ' * actually: '
              WRITE(jp_stderr, '(E9.3)') ddifblay
              WRITE(jp_stderr, cfmt_a_ind)
     &          'Falling back to the default value!'

              ddifblay = dp_dbl_thickness

            ENDIF

          ENDIF

          ! To be tested: dreaclay > 0
          ! To be tested: 0 == dbioturb            if ndn_s2z == 0 (auto_correct)
          !               0 <= dbioturb < dreaclay if ndn_s2z > 0 and ndn_z2b >0
          !               dbioturb == dreaclay     if ndn_z2b == 0 (auto-correct)
          !               dcorelay > 0

          !IF (... < 0.0D+00) THEN
          !  ! ... must be >= 0!
          !  ! Print out jselect_gridpointdistribution
          !  CALL ABORT_MEDUSA()
          !ENDIF


          CLOSE(UNIT=iu_cfg)


#ifdef DEBUG
          WRITE(jp_stddbg, cfmt_a, ADVANCE="NO")
     &      'Releasing logical unit number '
          WRITE(jp_stddbg, '(I0, A)', ADVANCE="NO")
     &      iu_cfg, ' (attached to "iu_cfg")'
#endif
          istatus = FREE_LOGUNIT(iu_cfg)
          IF (istatus /= 0) THEN
            WRITE(jp_stderr, cfmt_modprocname_a) 'Warning'
            WRITE(jp_stderr, cfmt_a, ADVANCE="NO")
     &        'FREE_LOGUNIT returned error code '
            WRITE(jp_stderr, '(I0, A)') istatus,
     &        ' when trying to release the logical unit number'
     &        //  ' attached to "iu_cfg" -- ignoring.'
#ifdef DEBUG
            WRITE(jp_stddbg, '()')
            WRITE(jp_stddbg, cfmt_a_ind, ADVANCE="NO")
            WRITE(jp_stddbg, '(A, I0, A)')
     &        'FREE_LOGUNIT returned error code ', istatus,
     &        ' when trying to release the logical unit number'
     &        //  ' attached to "iu_cfg" -- ignoring.'
          ELSE
            WRITE(jp_stddbg, '(" - done.")')
#endif
          ENDIF

#ifdef DEBUG
        ELSE

          WRITE(jp_stddbg, cfmt_a)
     &      'Grid configuration file "' // cfn_grid_cfg //
     &      '" not found - falling back to defaults'
#endif
        ENDIF

        CALL FLUSH(jp_stderr)
#ifdef DEBUG
        CALL FLUSH(jp_stddbg)
#endif

      ENDIF


#ifdef ALLOW_MPI
                                    ! Broadcast the configuration data:
                                    !  - grid: type
      CALL MPI_BCAST(jselect_gridtype, 1, MPI_INTEGER,
     &                              jp_exeproc_root, i_mycomm, istatus)

                                    ! Broadcast the configuration data:
                                    !  - grid: grid-point distribution
      CALL MPI_BCAST(jselect_gridptdistrib, 1, MPI_INTEGER,
     &                              jp_exeproc_root, i_mycomm, istatus)

                                    ! Broadcast the other parameter values
                                    ! ONLY if NO custom profile is used.
      IF (jselect_gridptdistrib /= jp_gpd_custom) THEN
                                    !  - ddifblay (all)
        CALL MPI_BCAST(ddifblay, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
                                    !  - dreaclay (all)
        CALL MPI_BCAST(dreaclay, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
                                    !  - dbioturb (all)
        CALL MPI_BCAST(dbioturb, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
                                    !  - dcorelay (all)
        CALL MPI_BCAST(dcorelay, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)

        SELECT CASE(jselect_gridptdistrib)
          CASE(jp_gpd_qtoplbot)
                                    !  - qtop_frac_t2z (quad_lin)
           CALL MPI_BCAST(qtop_frac_t2z, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
                                    !  - qtop_frac_z2b (quad_lin)
           CALL MPI_BCAST(qtop_frac_z2b, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)

          CASE(jp_gpd_ptoplbot)
                                    !  - ptop_frac_t2z (pown_lin)
           CALL MPI_BCAST(ptop_frac_t2z, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
                                    !  - ptop_frac_z2b (pown_lin)
           CALL MPI_BCAST(ptop_frac_z2b, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
                                    !  - pown_t2z (pown_lin)
           CALL MPI_BCAST(pown_t2z, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
                                    !  - pown_z2b (pown_lin)
           CALL MPI_BCAST(pown_z2b, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)

          CASE(jp_gpd_geoproop)
                                    !  - dreaclay1 (geomprog_open)
           CALL MPI_BCAST(dreaclay1, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
                                    !  - rreaclay (geomprog_open)
           CALL MPI_BCAST(rreaclay, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)

          CASE(jp_gpd_geoprocl)
                                    !  - dreaclay1 (geomprog_closed)
           CALL MPI_BCAST(dreaclay1, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)

          CASE(jp_gpd_glogiser)
                                    !  - expon (genlogist_ser)
           CALL MPI_BCAST(expon, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
                                    !  - grate (genlogist_ser)
           CALL MPI_BCAST(grate, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
                                    !  - f0_f00 (genlogist_ser)
           CALL MPI_BCAST(f0_f00, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
        END SELECT

      ENDIF
#endif


#ifdef GRID_CUSTOM
      IF (jselect_gridptdistrib == jp_jpd_custom) THEN
                                    ! Complete the set-up of the custom
                                    ! grid-point distributions.
        CALL GRIDEF_CUSTOM_SETUP(2, iu_cfg) ! stage 2; iu_cfg is ignored
      ENDIF
#endif


                                    ! With custom grid-point distributions,
                                    ! all has been done in GRIDEF_CUSTOM_SETUP
                                    ! at stage 2
      IF (jselect_gridptdistrib /= jp_gpd_custom) THEN

        da_gpd_ddifblay = ddifblay
        da_gpd_dreaclay = dreaclay
        da_gpd_dbioturb = dbioturb
        da_gpd_dcorelay = dcorelay

        WRITE(jp_stdlog, '(" - grid-point distribution")')
        WRITE(jp_stdlog, '("   * distribution ID: ", I0)')
     &                              jselect_gridptdistrib

        WRITE(jp_stdlog, '("   * DBL thickness: ", E9.3)')
     &                              da_gpd_ddifblay
        WRITE(jp_stdlog, '("   * REACLAY thickness: ", E9.3)')
     &                              da_gpd_dreaclay
        WRITE(jp_stdlog, '("   * Bioturbated thickness: ", E9.3)')
     &                              da_gpd_dbioturb
        WRITE(jp_stdlog, '("   * CORELAY layer thickness: ", E9.3)')
     &                              da_gpd_dcorelay

        SELECT CASE(jselect_gridptdistrib)
        CASE(jp_gpd_linear)
          CONTINUE

        CASE(jp_gpd_qtoplbot)
          da_gpd_qtoplbot_qfrac_t2z = qtop_frac_t2z
          da_gpd_qtoplbot_qfrac_z2b = qtop_frac_z2b
          WRITE(jp_stdlog, '("   * qtop_frac_t2z: ", E9.3)')
     &                              da_gpd_qtoplbot_qfrac_t2z
          WRITE(jp_stdlog, '("   * qtop_frac_z2b: ", E9.3)')
     &                              da_gpd_qtoplbot_qfrac_z2b

        CASE(jp_gpd_ptoplbot)
          da_gpd_ptoplbot_pfrac_t2z = ptop_frac_t2z
          da_gpd_ptoplbot_pfrac_z2b = ptop_frac_z2b
          da_gpd_ptoplbot_pn_t2z    = pown_t2z
          da_gpd_ptoplbot_pn_z2b    = pown_z2b
          WRITE(jp_stdlog, '("   * ptop_frac_t2z: ", E9.3)')
     &                              da_gpd_ptoplbot_pfrac_t2z
          WRITE(jp_stdlog, '("   * pown_t2z: ", E9.3)')
     &                              da_gpd_ptoplbot_pn_t2z
          WRITE(jp_stdlog, '("   * qtop_frac_z2b: ", E9.3)')
     &                              da_gpd_ptoplbot_pfrac_z2b
          WRITE(jp_stdlog, '("   * pown_z2b: ", E9.3)')
     &                              da_gpd_ptoplbot_pn_z2b

        CASE(jp_gpd_geoproop)
          da_gpd_geoproxx_dreaclay1 = dreaclay1
          da_gpd_geoproop_progratio = rreaclay
          WRITE(jp_stdlog, '("   * dreaclay1: ", E9.3)')
     &                              da_gpd_geoproxx_dreaclay1
          WRITE(jp_stdlog, '("   * progratio: ", E9.3)')
     &                              da_gpd_geoproop_progratio

        CASE(jp_gpd_geoprocl)
          da_gpd_geoproxx_dreaclay1 = dreaclay1
          WRITE(jp_stdlog, '("   * dreaclay1: ", E9.3)')
     &                              da_gpd_geoproxx_dreaclay1

        CASE(jp_gpd_glogiser)
          da_gpd_glogiser_exponent   = expon
          da_gpd_glogiser_growthrate = grate
          da_gpd_glogiser_frac_f0f00 = f0_f00
          WRITE(jp_stdlog, '("   * exponent: ", E9.3)')
     &                              da_gpd_glogiser_exponent
          WRITE(jp_stdlog, '("   * growthrate: ", E9.3)')
     &                              da_gpd_glogiser_growthrate
          WRITE(jp_stdlog, '("   * frac_f0f00: ", E9.3)')
     &                              da_gpd_glogiser_frac_f0f00
        END SELECT

        WRITE(jp_stdlog, '()')

      ENDIF


      WRITE(jp_stdlog, cfmt_modprocname_a) 'End of initial report'
      WRITE(jp_stdlog, '()')
      WRITE(jp_stdlog, '()')


      l_setupdone = .TRUE.

#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_modprocname_a) 'End'
      WRITE(jp_stddbg, '()')
#endif


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE SETUP_GRID
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      INTEGER FUNCTION SELECTED_GRIDTYPE()
!-----------------------------------------------------------------------

      IMPLICIT NONE

      IF (l_setupdone) THEN
        SELECTED_GRIDTYPE = jselect_gridtype
      ELSE
        SELECTED_GRIDTYPE = jp_grid_notsetup
      ENDIF

      RETURN

!-----------------------------------------------------------------------
      END FUNCTION SELECTED_GRIDTYPE
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE GRID_DEF(xzdn, xzdv)
!-----------------------------------------------------------------------


      IMPLICIT NONE


      DOUBLE PRECISION, DIMENSION(idnw:idnb), INTENT(OUT) ::  xzdn
      DOUBLE PRECISION, DIMENSION(idvw:idvb), INTENT(OUT) ::  xzdv
      OPTIONAL :: xzdv


      INTEGER          :: i

      INTEGER          :: n_gpt
      DOUBLE PRECISION :: z_top, z_bot
      DOUBLE PRECISION :: dxzdn


      IF (.NOT. l_setupdone) CALL SETUP_GRID


      SELECT CASE(jselect_gridptdistrib)
#ifdef GRID_CUSTOM
      CASE(jp_gpd_custom)           ! Custom distribution of nodes
          CALL GRIDEF_CUSTOM(xzdn)
#endif

      CASE(jp_gpd_linear)           ! Linear Distribution of nodes

        z_top = dp_swi_location
        z_bot = dp_swi_location + da_gpd_dbioturb
        n_gpt = ndn_s2z

        IF (ndn_w2s > 0) THEN
                                    ! If a DBL is included, cover
                                    ! [z_SWI, z_Z] with a vertex-to-node
                                    ! grid (T node located below the SWI);
          xzdn(idnt:idnz) = MGP_LINEAR
     &                              (z_top, z_bot, n_gpt, jp_glo_v2n)
        ELSE
                                    ! if no DBL is included, cover
                                    ! [z_SWI, z_Z] with a node-to-node
                                    ! grid (T node located at the SWI).
          xzdn(idnt:idnz) = MGP_LINEAR
     &                              (z_top, z_bot, n_gpt, jp_glo_n2n)
        ENDIF


        IF (ndn_z2b /= 0) THEN
                                    ! Cover [Z,B] with a node-to-node grid.
          z_top = dp_swi_location + da_gpd_dbioturb
          z_bot = dp_swi_location + da_gpd_dreaclay
          n_gpt = ndn_z2b  + 1      ! Z node must be replicated, hence + 1
          xzdn(idnz:idnb) = MGP_LINEAR
     &                             (z_top, z_bot, n_gpt, jp_glo_n2n)
        ENDIF


      CASE(jp_gpd_qtoplbot)         ! Quadratic top, linear bottom

        z_top = dp_swi_location
        z_bot = dp_swi_location + da_gpd_dbioturb
        n_gpt = ndn_s2z

        IF (ndn_w2s > 0) THEN
                                    ! If a DBL is included, cover
                                    ! [z_SWI, z_Z] with a vertex-to-node
                                    ! grid (T node located below the SWI);
          xzdn(idnt:idnz) = MGP_QTOPLBOT
     &                             (z_top, z_bot, n_gpt, jp_glo_v2n,
     &                              da_gpd_qtoplbot_qfrac_t2z)
        ELSE
                                    ! if no DBL is included, cover
                                    ! [z_SWI, z_Z] with a node-to-node
                                    ! grid (T node located at the SWI).
          xzdn(idnt:idnz) = MGP_QTOPLBOT
     &                             (z_top, z_bot, n_gpt, jp_glo_n2n,
     &                              da_gpd_qtoplbot_qfrac_t2z)
        ENDIF

        IF (ndn_z2b /= 0) THEN
                                    ! Cover [Z,B] with a node-to-node grid.
          z_top = dp_swi_location + da_gpd_dbioturb
          z_bot = dp_swi_location + da_gpd_dreaclay
          n_gpt = ndn_z2b  + 1      ! Z node must be replicated, hence + 1
          xzdn(idnz:idnb) = MGP_QTOPLBOT
     &                             (z_top, z_bot, n_gpt, jp_glo_n2n,
     &                              da_gpd_qtoplbot_qfrac_z2b)
        ENDIF


      CASE(jp_gpd_ptoplbot)         ! Power-n top, linear bottom

        z_top = dp_swi_location
        z_bot = dp_swi_location + da_gpd_dbioturb
        n_gpt = ndn_s2z

        IF (ndn_w2s > 0) THEN
                                    ! If a DBL is included, cover
                                    ! [z_SWI, z_Z] with a vertex-to-node
                                    ! grid (T node located below the SWI);
          xzdn(idnt:idnz) = MGP_PTOPLBOT
     &                             (z_top, z_bot, n_gpt, jp_glo_v2n,
     &                              da_gpd_ptoplbot_pfrac_t2z,
     &                              da_gpd_ptoplbot_pn_t2z)
        ELSE
                                    ! if no DBL is included, cover
                                    ! [z_SWI, z_Z] with a node-to-node
                                    ! grid (T node located at the SWI).
          xzdn(idnt:idnz) = MGP_PTOPLBOT
     &                             (z_top, z_bot, n_gpt, jp_glo_n2n,
     &                              da_gpd_ptoplbot_pfrac_t2z,
     &                              da_gpd_ptoplbot_pn_t2z)
        ENDIF

        IF (ndn_z2b /= 0) THEN
                                    ! Cover [Z,B] with a node-to-node grid.
          z_top = dp_swi_location + da_gpd_dbioturb
          z_bot = dp_swi_location + da_gpd_dreaclay
          n_gpt = ndn_z2b  + 1      ! Z node must be replicated, hence + 1
          xzdn(idnz:idnb) = MGP_PTOPLBOT
     &                             (z_top, z_bot, n_gpt, jp_glo_n2n,
     &                              da_gpd_ptoplbot_pfrac_z2b,
     &                              da_gpd_ptoplbot_pn_z2b)
        ENDIF



c~       CASE(jp_gpd_geoproop)         ! Geometric progression, open

c~         dxzdn = da_gpd_geoproxx_dreaclay1

c~         DO i = 1, ndi_t2b

c~           xzdn(idnt+i) = xzdn(idnt+i-1) + dxzdn
c~           dxzdn = dxzdn*da_gpd_geoproop_progratio

c~         ENDDO

c~         ! xzdn(idnb) set in the loop.


c~       CASE(jp_gpd_geoprocl)         ! Geometric progression, closed

c~         IF (da_gpd_geoprocl_progratio == 0.0D+00) THEN
c~           da_gpd_geoprocl_progratio
c~      &      = DGEOMPROG_RATIO(da_gpd_dreaclay,
c~      &                              da_gpd_geoproxx_dreaclay1, ndi_t2b)
c~           IF (da_gpd_geoprocl_progratio == 0.0D+00) THEN
c~             STOP
c~           ENDIF
c~         ENDIF

c~         dxzdn = da_gpd_geoproxx_dreaclay1

c~         DO i = 1, ndi_t2b - 1

c~           xzdn(idnt+i) = xzdn(idnt+i-1) + dxzdn
c~           dxzdn = dxzdn*da_gpd_geoprocl_progratio

c~         ENDDO

c~         xzdn(idnb) = da_gpd_dreaclay

      CASE(jp_gpd_glogiser)         ! Node distribution following
                                    ! generalised logistic series
        z_top = dp_swi_location
        z_bot = dp_swi_location + da_gpd_dbioturb
        n_gpt = ndn_s2z

        IF (ndn_w2s > 0) THEN
                                    ! If a DBL is included, cover
                                    ! [z_SWI, z_Z] with a vertex-to-node
                                    ! grid (T node located below the SWI);
          xzdn(idnt:idnz) = MGP_GLOGISER
     &                              (z_top, z_bot, n_gpt, jp_glo_v2n,
     &                               da_gpd_glogiser_exponent,
     &                               da_gpd_glogiser_growthrate,
     &                               da_gpd_glogiser_frac_f0f00)
        ELSE
                                    ! if no DBL is included, cover
                                    ! [z_SWI, z_Z] with a node-to-node
                                    ! grid (T node located at the SWI).
          xzdn(idnt:idnz) = MGP_GLOGISER
     &                              (z_top, z_bot, n_gpt, jp_glo_n2n,
     &                               da_gpd_glogiser_exponent,
     &                               da_gpd_glogiser_growthrate,
     &                               da_gpd_glogiser_frac_f0f00)
        ENDIF


        IF (ndn_z2b /= 0) THEN
                                    ! Cover [Z,B] with a linear node-to-node grid.
          z_top = dp_swi_location + da_gpd_dbioturb
          z_bot = dp_swi_location + da_gpd_dreaclay
          n_gpt = ndn_z2b  + 1      ! Z node must be replicated, hence + 1
          xzdn(idnz:idnb) = MGP_LINEAR
     &                             (z_top, z_bot, n_gpt, jp_glo_n2n)
        ENDIF


      END SELECT


      IF (ndn_w2s /= 0) THEN        ! If a DBL is included, cover it
                                    ! with a linear node-to-vertex grid
        z_top = dp_swi_location - da_gpd_ddifblay
        z_bot = dp_swi_location
        n_gpt = ndn_w2s

        xzdn(idnw:idnt-1) = MGP_LINEAR(z_top, z_bot, n_gpt, jp_glo_n2v)

      ENDIF


      IF (PRESENT(xzdv)) xzdv = GRID_VTX(xzdn)


      RETURN

c~       CONTAINS

c~ !- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
c~       DOUBLE PRECISION FUNCTION DGEOMPROG_RATIO(d_tot, d_1, n)
c~ !- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

c~ ! Solve equation d_1*(1 + gpr + gpr**2 + ... gpr**(n-1)) = d_tot
c~ ! for gpr, arising if a grid with n intervals (n+1 grid-points)
c~ ! is chosen, such that d_i = x_i - x_(i-1), i = 1, ..., n are in a
c~ ! geometric progression, with d_1 given. The equation to solve,
c~ ! d_1*(1 + gpr + gpr**2 + ... gpr**(n-1)) = d_tot
c~ ! is equivalent to (1 - gpr**n)/(1 - gpr) - d_tot/d_1 = 0
c~ ! which solved here by Newton iterations.

c~       IMPLICIT NONE


c~       DOUBLE PRECISION, INTENT(IN) :: d_tot ! Total thickness of the <n> layers
c~       DOUBLE PRECISION, INTENT(IN) :: d_1   ! Thickness of the first layer
c~       INTEGER,          INTENT(IN) :: n     ! number of layers

c~       DOUBLE PRECISION :: r         ! ratio d_tot/d_1
c~       DOUBLE PRECISION :: r_n       ! short for r/n

c~       DOUBLE PRECISION :: gpr0, gpr ! geometric progression ratio
c~       DOUBLE PRECISION :: f0, f     ! function of the equation to solve
c~       DOUBLE PRECISION :: df        ! function derivative
c~       DOUBLE PRECISION :: d2f0      ! second derivative

c~       INTEGER          :: m
c~       DOUBLE PRECISION :: dgpr      ! iterative corrective


c~       r = d_tot/d_1
c~       r_n = r/DBLE(n)

c~                                     ! Start at gpr where f'(gpr) = 0
c~       gpr0 = EXP(LOG(r_n)/DBLE(n-1))

c~       ! Develop f around (gpr0,f0) to second degree and solve for gpr:

c~       ! Denote gpr -> g
c~       ! f(g) = f0 + df/dg|g0 (g-g0) + 1/2 (g-g0)^2*d2f/dg2| g0
c~       !      = f0 + 1/2  (gpr-gpr0)^2*d2f/dg2| g0
c~       ! f(g1) = 0
c~       ! => (g1-g0)^2 = -2 f0/(d2f/dg2| g0)
c~       !    g1 = g0 + SQRT(-2 f0/(d2f/dg2| g0))
c~       f0 = ((r_n) - 1.0D+00)/(gpr0 - 1.0D+00) - (r_n)*DBLE(n-1)


c~       d2f0 = DBLE(n*(n-1)) * gpr0**(n-2)/(gpr0 - 1.0D+00)

c~       gpr = gpr0 + SQRT(-(f0+f0)/d2f0)

c~       m = 0

c~       DO

c~         f = (gpr**n - 1.0D+00)/(gpr - 1.0D+00) - r
c~        df = (DBLE(n)*gpr**(N-1))/(gpr - 1.0D+00)
c~      &      - (gpr**N -1.0D+00)/((gpr - 1.0D+00)*(gpr - 1.0D+00))

c~        dgpr = -f/df
c~        gpr = gpr + dgpr

c~        m = m + 1

c~        IF (dgpr < 1.0D-9) THEN
c~          DGEOMPROG_RATIO = gpr
c~          EXIT
c~        ENDIF

c~        IF (m > 100) THEN
c~         DGEOMPROG_RATIO = 0.0D+00
c~         EXIT
c~        ENDIF

c~       ENDDO

c~       RETURN

c~ !- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
c~       END FUNCTION DGEOMPROG_RATIO
c~ !- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


!-----------------------------------------------------------------------
      END SUBROUTINE GRID_DEF
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
      FUNCTION GRID_VTX(xzdn) RESULT(xzdv)
!-----------------------------------------------------------------------


      IMPLICIT NONE


      DOUBLE PRECISION, DIMENSION(idnw:idnb), INTENT(IN) ::  xzdn

      DOUBLE PRECISION, DIMENSION(idvw:idvb)             ::  xzdv


      xzdv(idvw) = xzdn(idnw)

      xzdv(idvw+1:idvb-1)
     &  = (xzdn(idnw:idnb-1) + xzdn(idnw+1:idnb)) / 2.0D+00

      xzdv(idvb) = xzdn(idnb)

      IF (ndn_w2s /= 0) THEN        ! The vertex located at the SWI is
                                    ! not half-way between nodes T-1 and
                                    ! T if a DBL is included, and must
                                    ! therefore be reset to the SWI location.
        xzdv(idvs) = dp_swi_location

      ENDIF


      RETURN

!-----------------------------------------------------------------------
      END FUNCTION GRID_VTX
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
      FUNCTION MGP_QTOPLBOT(z_top, z_bot, n_gpt, k_type, xi_c)
     &                              RESULT(z_grid)
!-----------------------------------------------------------------------


      IMPLICIT NONE

      DOUBLE PRECISION, INTENT(IN) :: z_top
      DOUBLE PRECISION, INTENT(IN) :: z_bot
      INTEGER,          INTENT(IN) :: n_gpt
      INTEGER,          INTENT(IN) :: k_type
      DOUBLE PRECISION, INTENT(IN) :: xi_c

      DOUBLE PRECISION, DIMENSION(1:n_gpt) :: z_grid

      DOUBLE PRECISION :: denom
      INTEGER          :: i, i1, i2
      INTEGER          :: n_xi, i_off
      DOUBLE PRECISION :: z_scale, z_td
      DOUBLE PRECISION :: xi, a, b, q, q_i


      denom = ((1.0D+00 + xi_c**2)**0.5D+00 - xi_c)


      SELECT CASE(k_type)
      CASE(jp_glo_n2n)

        n_xi  = n_gpt - 1
        i_off = 1

        z_scale = (z_bot - z_top)
        z_td = z_top

        z_grid(n_gpt) = z_bot       ! Preset first and last grid points,
        z_grid(    1) = z_top       ! starting with the last, then the first,
                                    ! so that z_top is used if n_gpt = 1, ...

        i1 = 2                      ! ... and only calculate
        i2 = n_gpt - 1              ! points 2 to n_gpt-1


      CASE(jp_glo_n2v)

        n_xi  = n_gpt
        i_off = 1

        xi = DBLE(n_xi-1)/DBLE(n_xi)
        q = ((xi**2 + xi_c**2)**0.5D+00 - xi_c)/denom
        z_scale = (z_bot - z_top) * 2.0D+00/(q + 1.0D+00)
        z_td = z_top

        z_grid(    1) = z_top       ! Preset first grid point ...

        i1 = 2                      ! ... and only calculate
        i2 = n_gpt                  ! points 2 to n_gpt

      CASE(jp_glo_v2n)

        n_xi  = n_gpt
        i_off = 0

        xi = 1.0D+00/DBLE(n_xi)
        q = ((xi**2 + xi_c**2)**0.5D+00 - xi_c)/denom
        z_scale = (z_bot - z_top) * 2.0D+00/(2.0D+00 - q)
        !z_0 = z_top + (z_bot - z_top) - z_scale
        z_td = z_bot - z_scale

        z_grid(n_gpt) = z_bot       ! Preset last grid point ...

        i1 = 1                      ! ... and only calculate
        i2 = n_gpt - 1              ! points 1 to n_gpt-1

      CASE(jp_glo_v2v)

        n_xi  = n_gpt + 1
        i_off = 0

        xi = 1.0D+00/DBLE(n_xi)
        a = ((xi**2 + xi_c**2)**0.5D+00 - xi_c)/denom

        xi = DBLE(n_xi-1)/DBLE(n_xi)
        b = ((xi**2 + xi_c**2)**0.5D+00 - xi_c)/denom + 1.0D+00

        z_scale = 2.0D+00*(z_bot - z_top)/(b-a)
        z_td = z_top - a*z_scale/2.0D+00

        i1 = 1                      ! Calculate all points
        i2 = n_gpt                  ! from 1 to n_gpt

      END SELECT


      DO i = i1, i2

        xi = DBLE(i-i_off)/DBLE(n_xi)
        q = ((xi**2 + xi_c**2)**0.5D+00 - xi_c)/denom

        z_grid(i) = z_scale * q + z_td

      ENDDO


      RETURN


!-----------------------------------------------------------------------
      END FUNCTION MGP_QTOPLBOT
!-----------------------------------------------------------------------





!-----------------------------------------------------------------------
      FUNCTION MGP_PTOPLBOT(z_top, z_bot, n_gpt, k_type, xi_c, pn)
     &                              RESULT(z_grid)
!-----------------------------------------------------------------------


      IMPLICIT NONE

      DOUBLE PRECISION, INTENT(IN) :: z_top
      DOUBLE PRECISION, INTENT(IN) :: z_bot
      INTEGER,          INTENT(IN) :: n_gpt
      INTEGER,          INTENT(IN) :: k_type
      DOUBLE PRECISION, INTENT(IN) :: xi_c
      DOUBLE PRECISION, INTENT(IN) :: pn

      DOUBLE PRECISION, DIMENSION(1:n_gpt) :: z_grid

      DOUBLE PRECISION :: denom
      INTEGER          :: i, i1, i2
      INTEGER          :: n_xi, i_off
      DOUBLE PRECISION :: z_scale, z_td
      DOUBLE PRECISION :: xi, a, b, q, q_i


      denom = (1.0D+00 + xi_c**pn)**(1.0D+00/pn) - xi_c


      SELECT CASE(k_type)
      CASE(jp_glo_n2n)

        n_xi  = n_gpt - 1
        i_off = 1

        z_scale = (z_bot - z_top)
        z_td = z_top

        z_grid(n_gpt) = z_bot       ! Preset first and last grid points,
        z_grid(    1) = z_top       ! starting with the last, then the first,
                                    ! so that z_top is used if n_gpt = 1, ...

        i1 = 2                      ! ... and only calculate
        i2 = n_gpt - 1              ! points 2 to n_gpt-1


      CASE(jp_glo_n2v)

        n_xi  = n_gpt
        i_off = 1

        xi = DBLE(n_xi-1)/DBLE(n_xi)
        q = ((xi**pn + xi_c**pn)**(1.0D+00/pn) - xi_c)/denom
        z_scale = (z_bot - z_top) * 2.0D+00/(q + 1.0D+00)
        z_td = z_top

        z_grid(    1) = z_top       ! Preset first grid point ...

        i1 = 2                      ! ... and only calculate
        i2 = n_gpt                  ! points 2 to n_gpt


      CASE(jp_glo_v2n)

        n_xi  = n_gpt
        i_off = 0

        xi = 1.0D+00/DBLE(n_xi)
        q = ((xi**pn + xi_c**pn)**(1.0D+00/pn) - xi_c)/denom
        z_scale = (z_bot - z_top) * 2.0D+00/(2.0D+00 - q)
        !z_0 = z_top + (z_bot - z_top) - z_scale
        z_td = z_bot - z_scale

        z_grid(n_gpt) = z_bot       ! Preset last grid point ...

        i1 = 1                      ! ... and only calculate
        i2 = n_gpt - 1              ! points 1 to n_gpt-1


      CASE(jp_glo_v2v)

        n_xi  = n_gpt + 1
        i_off = 0

        xi = 1.0D+00/DBLE(n_xi)
        a = ((xi**pn + xi_c**pn)**(1.0D+00/pn) - xi_c)/denom

        xi = DBLE(n_xi-1)/DBLE(n_xi)
        b = ((xi**pn + xi_c**pn)**(1.0D+00/pn) - xi_c)/denom + 1.0D+00

        z_scale = 2.0D+00*(z_bot - z_top)/(b-a)
        z_td = z_top - a*z_scale/2.0D+00

        i1 = 1                      ! Calculate all points
        i2 = n_gpt                  ! from 1 to n_gpt

      END SELECT


      DO i = i1, i2

        xi = DBLE(i-i_off)/DBLE(n_xi)
        q = ((xi**pn + xi_c**pn)**(1.0D+00/pn) - xi_c)/denom

        z_grid(i) = z_scale * q + z_td

      ENDDO


      RETURN


!-----------------------------------------------------------------------
      END FUNCTION MGP_PTOPLBOT
!-----------------------------------------------------------------------





!-----------------------------------------------------------------------
      FUNCTION MGP_GLOGISER(z_top, z_bot, n_gpt, k_type,
     &                              dn, dmu, df0_f00)
     &                              RESULT(z_grid)
!-----------------------------------------------------------------------


      IMPLICIT NONE

      DOUBLE PRECISION, INTENT(IN) :: z_top
      DOUBLE PRECISION, INTENT(IN) :: z_bot
      INTEGER,          INTENT(IN) :: n_gpt
      INTEGER,          INTENT(IN) :: k_type
      DOUBLE PRECISION, INTENT(IN) :: dn
      DOUBLE PRECISION, INTENT(IN) :: dmu
      DOUBLE PRECISION, INTENT(IN) :: df0_f00

      DOUBLE PRECISION, DIMENSION(1:n_gpt) :: z_grid

      DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE :: q

      INTEGER          :: i, i1, i2
      INTEGER          :: n_xi, i_off
      DOUBLE PRECISION :: z_scale, z_td
      DOUBLE PRECISION :: xi, a, b, q_scale



      SELECT CASE(k_type)
      CASE(jp_glo_n2n)
        n_xi  = n_gpt - 1
      CASE(jp_glo_n2v)
        n_xi  = n_gpt
      CASE(jp_glo_v2n)
        n_xi  = n_gpt
      CASE(jp_glo_v2v)
        n_xi  = n_gpt + 1
      END SELECT

      ALLOCATE(q(0:n_xi))

      q(0) = 0.0D+00
      a = 1.0D+00/(df0_f00**dn) - 1.0D+00
      b = dn*dmu

      DO i = 1, n_xi-1
        xi = DBLE(i)/DBLE(n_xi)
        q(i) = q(i-1) + 1.0D+00/(1.0D+00 + a*EXP(-b*xi))**(1.0D+00/dn)
      ENDDO

      q_scale = q(n_xi-1) + 1.0D+00/(1.0D+00 + a*EXP(-b))**(1.0D+00/dn)

      q(1:n_xi-1) = q(1:n_xi-1)/q_scale
      q(n_xi)     = 1.0D+00

      SELECT CASE(k_type)
      CASE(jp_glo_n2n)

        z_scale = (z_bot - z_top)
        z_td    = z_top

        z_grid(n_gpt) = z_bot       ! Preset first and last grid points,
        z_grid(    1) = z_top       ! starting with the last, then the first,
                                    ! so that z_top is used if n_gpt = 1, ...

        i1 = 2                      ! ... and only calculate
        i2 = n_gpt - 1              ! points 2 to n_gpt-1

        i_off = 1


      CASE(jp_glo_n2v)

        z_scale = (z_bot - z_top) * 2.0D+00/(q(n_xi-1) + 1.0D+00)
        z_td    = z_top

        z_grid(    1) = z_top       ! Preset first grid point ...

        i1 = 2                      ! ... and only calculate
        i2 = n_gpt                  ! points 2 to n_gpt

        i_off = 1


      CASE(jp_glo_v2n)

        z_scale = (z_bot - z_top) * 2.0D+00/(2.0D+00 - q(1))
        z_td    = z_bot - z_scale

        z_grid(n_gpt) = z_bot       ! Preset last grid point ...

        i1 = 1                      ! ... and only calculate
        i2 = n_gpt - 1              ! points 1 to n_gpt-1

        i_off = 0

      CASE(jp_glo_v2v)

        z_scale = 2.0D+00*(z_bot - z_top)/(q(n_xi-1) - q(1) + 1.0D+00)
        z_td    = z_top - z_scale*q(n_xi-1)/2.0D+00

        i1 = 1                      ! Calculate all points
        i2 = n_gpt                  ! from 1 to n_gpt

        i_off = 0

      END SELECT


      DO i = i1, i2
        z_grid(i) = z_scale * q(i - i_off) + z_td
      ENDDO


      DEALLOCATE(q)


      RETURN


!-----------------------------------------------------------------------
      END FUNCTION MGP_GLOGISER
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      FUNCTION MGP_LINEAR(z_top, z_bot, n_gpt, k_type) RESULT(z_grid)
!-----------------------------------------------------------------------


      IMPLICIT NONE

      DOUBLE PRECISION, INTENT(IN) :: z_top
      DOUBLE PRECISION, INTENT(IN) :: z_bot
      INTEGER,          INTENT(IN) :: n_gpt
      INTEGER,          INTENT(IN) :: k_type

      DOUBLE PRECISION, DIMENSION(1:n_gpt) :: z_grid



      INTEGER          :: i, i1, i2
      INTEGER          :: n_xi, i_off
      DOUBLE PRECISION :: z_scale, z_td
      DOUBLE PRECISION :: xi, a, b



      SELECT CASE(k_type)
      CASE(jp_glo_n2n)

        n_xi  = n_gpt - 1
        i_off = 1

        z_scale = (z_bot - z_top)
        z_td    = z_top

        z_grid(n_gpt) = z_bot       ! Preset first and last grid points,
        z_grid(    1) = z_top       ! starting with the last, then the first,
                                    ! so that z_top is used if n_gpt = 1, ...

        i1 = 2                      ! ... and only calculate
        i2 = n_gpt - 1              ! points 2 to n_gpt-1

      CASE(jp_glo_n2v)

        n_xi  = n_gpt
        i_off = 1

        xi      = DBLE(n_xi-1)/DBLE(n_xi)
        z_scale = (z_bot - z_top) * xi
        z_td    = z_top

        z_grid(1) = z_top           ! Preset first grid point ...

        i1 = 2                      ! ... and only calculate
        i2 = n_gpt                  ! points 2 to n_gpt

      CASE(jp_glo_v2n)

        n_xi  = n_gpt
        i_off = 0

        xi      = 1.0D+00/DBLE(n_xi)
        z_scale = (z_bot - z_top) * 2.0D+00/(2.0D+00 - xi)
        !z_0 = z_top + (z_bot - z_top) - z_scale
        z_td    = z_bot - z_scale

        z_grid(n_gpt) = z_bot       ! Preset last grid point ...

        i1 = 1                      ! ... and only calculate
        i2 = n_gpt - 1              ! points 1 to n_gpt-1

      CASE(jp_glo_v2v)

        n_xi  = n_gpt + 1
        i_off = 0


        !xi = DBLE(1)/DBLE(n_xi)
        !a = xi
        a = 1.0D+00/DBLE(n_xi)

        !xi = DBLE(n_xi-1)/DBLE(n_xi)
        !b = xi + 1.0D+00
        b = 2.0D+00 - a

        z_scale = 2.0D+00*(z_bot - z_top)/(b-a)
        z_td    = z_top - a*z_scale/2.0D+00

        i1 = 1                      ! Calculate all points
        i2 = n_gpt                  ! from 1 to n_gpt

      END SELECT


      DO i = i1, i2

        xi = DBLE(i-i_off)/DBLE(n_xi)
        z_grid(i) = z_scale * xi + z_td

      ENDDO


      RETURN


!-----------------------------------------------------------------------
      END FUNCTION MGP_LINEAR
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      FUNCTION MGP_GEOMSERIES(r, n) RESULT(dsum)
!-----------------------------------------------------------------------

      ! This function subprogram evaluates
      ! GP_n(r) = (r**n - 1) / (r-1)
      !         = r**(n-1) + r**(n-2) + ... r + 1
      !         = (( ... ((r+1)*r + 1)*r ...)*r + 1)
      ! For n <= 0, the returned value is provisionally 0
      ! For n = 1, the returned value is 1

      IMPLICIT NONE

      DOUBLE PRECISION, INTENT(IN) :: r
      INTEGER,          INTENT(IN) :: n

      DOUBLE PRECISION :: dsum
      INTEGER :: i


      SELECT CASE(n)
      CASE(:0)
        dsum = 0.0D+00
      CASE(1)
        dsum = 1.0D+00
      CASE(2:)
        dsum = r + 1.0D+00
        DO i = n-2, 1, -1
          dsum = 1.0D+00 + r*dsum
        ENDDO
      END SELECT


      RETURN

!-----------------------------------------------------------------------
      END FUNCTION MGP_GEOMSERIES
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      FUNCTION MGP_DGEOMSERIES_DR(r, n) RESULT(dsum)
!-----------------------------------------------------------------------

      ! This function subprogram evaluates
      ! d/dr GP_n(r) = d/dr (r**n - 1) / (r-1)
      !         = (n-1)*r**(n-2) + (n-2)*r**(n-3) + ... 2*r + 1
      !         = ( ... (((n-1)*r+(n-2))*r + (n-3))*r ... + 2)*r + 1
      ! For n <= 0, the returned value is provisionally 0
      ! For n = 1, the returned value is 0

      IMPLICIT NONE

      DOUBLE PRECISION, INTENT(IN) :: r
      INTEGER,          INTENT(IN) :: n

      DOUBLE PRECISION :: dsum
      INTEGER :: i


      SELECT CASE(n)
      CASE(:1)
        dsum = 0.0D+00
      CASE(2:)
        dsum = DBLE(n-2) + DBLE(n-1)*r
        DO i = n-3, 1, -1
          dsum = DBLE(i) + r*dsum
        ENDDO
      END SELECT


      RETURN

!-----------------------------------------------------------------------
      END FUNCTION MGP_DGEOMSERIES_DR
!-----------------------------------------------------------------------





#ifdef GRID_CUSTOM
!     SUBROUTINE GRIDEF_CUSTOM(xzdn)
!     SUBROUTINE GRIDEF_CUSTOM_SETUP(k_stage, iu_cfg)
#include <gridef_custom.F>
#endif


!=======================================================================
      END MODULE MOD_GRIDPARAM
!=======================================================================
