!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


!=======================================================================
      MODULE MOD_GAUSS
!=======================================================================

      USE mod_defines_medusa,       ONLY: jp_stderr
      USE mod_execontrol_medusa,    ONLY: ABORT_MEDUSA

! Gauss elimination solver for square linear systems, with single
! or multiple righthand sides

      IMPLICIT NONE

      PRIVATE

      PUBLIC GM_GESV

! Both the single and the multiple-righthand-side versions are called
! under the common name GM_GESV -- the interface below makes it possible.

      INTERFACE GM_GESV

         MODULE PROCEDURE GM_DGESV, GM_DGESV1, GM_DGESV1M

      END INTERFACE



      CONTAINS

!-----------------------------------------------------------------------
      SUBROUTINE GM_DGESV1(a, b, ipivot, info)
!-----------------------------------------------------------------------

! Purpose
! =======
!
! Solve the linear system A*x = b for
! - a square matrix A(n,n), transmitted in <a>;
! - a 1D-array b(n), transmitted in <b>.
!

! Outcome
! =======
!
! - The result x(n) is returned in <b>, which gets overwritten.
! - Upon exit, <a> gets overwritten with the unit lower triangular and
!   upper triangular factors of A, L and R, resp., which are such that
!   where P*A = L*R, P being a permutation matrix
! - the permutation matrix P can optionally be retrieved in the compact
!   1D-representation <ipivot>.
! - <info> is an optional return flag -- return values are
!   compatible with those of the LAPACK95 LA_DGESV1 subroutine,
!   but do not cover all of the exceptions trapped by LA_DGESV1
!   (GM_DGESV1 uses automatic arrays, and allocation-related errors
!   are caught by the system, not by the subroutine itself.)
!
!
! LAPACK95 compatibility and differences
! ======================================
!
! GM_DGESV1 is a compatible plugin-replacement for the LA_DGESV1
! subroutine (la_dgesv1.f90) from the LAPACK95 library: the order,
! meaning, shape and typing and attributes of the dummy arguments
! is exactly the same.
! Unlike LA_DGESV1, GM_DGESV1 calls upon a factorization routine
! that uses a column pivot search with implicit row scaling,
! as advocated by Engel-Müllges and Uhlig (1996, p. 73).


      IMPLICIT NONE

! Argument list variables (dummy arguments)
! -----------------------------------------

      DOUBLE PRECISION, DIMENSION(:,:), INTENT(INOUT) :: a
      DOUBLE PRECISION, DIMENSION(:),   INTENT(INOUT) :: b

      INTEGER, OPTIONAL, DIMENSION(:),  INTENT(OUT)   :: ipivot
      INTEGER, OPTIONAL,                INTENT(OUT)   :: info


! Local variables
! ---------------

      INTEGER,          DIMENSION(SIZE(a,1)) :: ipivot_local
      DOUBLE PRECISION, DIMENSION(SIZE(a,2)) :: x
      INTEGER :: nra, nca, nrb, nrp
      INTEGER :: info_local

      info_local = 0

      nra = SIZE(a, 1)
      nca = SIZE(a, 2)
      nrb = SIZE(b)

      IF (PRESENT(ipivot)) THEN
        nrp = SIZE(ipivot)
      ELSE
        nrp = SIZE(a,1)
      ENDIF

      IF ((nra /= nca) .OR. (nra < 0)) THEN
        info_local = -1  ! error in the first argument
      ELSEIF (nrb /= nra) THEN
        info_local = -2  ! error in the second argument
      ELSEIF (nrp < nra) THEN
        info_local = -3  ! error in the third argument
                         ! (can only happen if actually present)
      ENDIF


      IF (info_local == 0) THEN
                 ! Proceed with the LR factorization of the
                 ! matrix A only if arguments valid
        CALL GM_DGETRF(a, ipivot_local, info_local)

        IF (info_local == 0) THEN
                 ! Solution by back-substitution
          CALL GM_DGETRS(a, ipivot_local, b, x)
        ENDIF

      ENDIF


      IF (info_local == 0) THEN
                 ! All went well: transcribe <x> into <b>
        b(1:nra) = x(1:nca)
        IF (PRESENT(ipivot)) ipivot(1:nra) = ipivot_local(:)
        IF (PRESENT(info)) info = info_local

      ELSE
                 ! There have been errors caught
        IF (PRESENT(info)) THEN
          info = info_local
        ELSE      
          WRITE(jp_stderr, '("[MOD_GAUSS/GM_DGESV1]: INFO = ", I0, ' //
     &                  ' " cannot be reported -- Aborting.")')
          CALL ABORT_MEDUSA()
        ENDIF

      ENDIF


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE GM_DGESV1
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE GM_DGESV(a, b, ipivot, info)
!-----------------------------------------------------------------------

! Purpose
! =======
!
! Solve the linear system A*x = b for
! - a square matrix A(n,n), transmitted in <a>;
! - a 2D-array b(n,m), m>0, transmitted in <b>.
!
!
! Outcome
! =======
!
! - The result x(n,m) is returned in <b>, which gets overwritten.
! - Upon exit, <a> gets overwritten with the unit lower triangular and
!   upper triangular factors of A, L and R, resp., which are such that
!   where P*A = L*R, P being a permutation matrix
! - the permutation matrix P can optionally be retrieved in the compact
!   1D-representation <ipivot>.
! - <info> is an optional return flag -- return values are
!   compatible with those of the LAPACK95 LA_DGESV1 subroutine,
!   but do not cover all of the exceptions trapped by LA_DGESV1
!   (GM_DGESV1 uses automatic arrays, and allocation-related errors
!   are caught by the system, not by the subroutine itself.)
!
!
! LAPACK95 compatibility and differences
! ======================================
!
! GM_DGESV is a compatible plugin-replacement for the LA_DGESV
! subroutine (la_dgesv.f90) from the LAPACK95 library: the order,
! meaning, shape and typing and attributes of the dummy arguments
! is exactly the same.
! Unlike LA_DGESV, GM_DGESV calls upon a factorization routine
! that uses a column pivot search with implicit row scaling,
! as advocated by Engel-Müllges and Uhlig (1996, p. 73).


      IMPLICIT NONE

! Argument list variables (dummy arguments)
! -----------------------------------------

      DOUBLE PRECISION, DIMENSION(:,:), INTENT(INOUT) :: a
      DOUBLE PRECISION, DIMENSION(:,:), INTENT(INOUT) :: b

      INTEGER, OPTIONAL, DIMENSION(:),  INTENT(OUT)   :: ipivot
      INTEGER, OPTIONAL,                INTENT(OUT)   :: info


! Local variables
! ---------------

      INTEGER, DIMENSION(SIZE(a,1))                    :: ipivot_local
      DOUBLE PRECISION, DIMENSION(SIZE(a,2),SIZE(b,2)) :: x
      INTEGER :: nra, nca, nrb, ncb, nrp
      INTEGER :: jcol
      INTEGER :: info_local

      info_local = 0

      nra = SIZE(a, 1)
      nca = SIZE(a, 2)
      nrb = SIZE(b, 1)
      ncb = SIZE(b, 2)

      IF (PRESENT(ipivot)) THEN
        nrp = SIZE(ipivot)
      ELSE
        nrp = SIZE(a,1)
      ENDIF


      IF ((nra /= nca) .OR. (nra < 0)) THEN
        info_local = -1  ! error in the first argument
      ELSEIF ((nrb /= nra) .OR. (ncb < 0)) THEN
        info_local = -2  ! error in the second argument
      ELSEIF (nrp < nra) THEN
        info_local = -3  ! error in the third argument
                         ! (can only happen if actually present)
      ENDIF


      IF (info_local == 0) THEN
                 ! Proceed with the LR factorization of the matrix A (P*A = L*R)
        CALL GM_DGETRF(a, ipivot_local, info_local)

        IF (info_local == 0) THEN
                 ! Solutions of A*X = B by back-substitution, column by column
          DO jcol = 1, ncb
            CALL GM_DGETRS(a, ipivot_local, b(:,jcol), x(:,jcol))
          ENDDO
        ENDIF

      ENDIF

      IF (info_local == 0) THEN
                 ! All went well: transcribe <x> into <b>
        b(:,:) = x(:,:)
        IF (PRESENT(ipivot)) ipivot(1:nra) = ipivot_local(:)
        IF (PRESENT(info)) info = info_local

      ELSE
                 ! There have been errors caught
        IF (PRESENT(info)) THEN
          info = info_local
        ELSE
                 ! abort if we cannot report them
          WRITE(jp_stderr, '("[MOD_GAUSS/GM_DGESV]: INFO = ", I0, ' //
     &                  ' " cannot be reported -- Aborting.")')
          CALL ABORT_MEDUSA()
        ENDIF

      ENDIF


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE GM_DGESV
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE GM_DGESV1M(a, b, ipivot, info)
!-----------------------------------------------------------------------

! Purpose
! =======
!
! Solve the linear system A*x = b for
! - a square matrix A(n,n), transmitted in <a>;
! - a 1D-array of n B(m1,m2) matrices, transmitted in <b(m1,m2, n)>.
!

! Outcome
! =======
!
! - The result x(m1,m2, n) is returned in <b>, which gets overwritten.
! - Upon exit, <a> gets overwritten with the unit lower triangular and
!   upper triangular factors of A, L and R, resp., which are such that
!   where P*A = L*R, P being a permutation matrix
! - the permutation matrix P can optionally be retrieved in the compact
!   1D-representation <ipivot>.
! - <info> is an optional return flag -- return values are
!   compatible with those of the LAPACK95 LA_DGESV1 subroutine,
!   but do not cover all of the exceptions trapped by LA_DGESV1
!   (GM_DGESV1 uses automatic arrays, and allocation-related errors
!   are caught by the system, not by the subroutine itself.)
!
!
! LAPACK95 compatibility and differences
! ======================================
!
! GM_DGESV1M is analog to GM_DGESV1.
! There is no similar counterpart in LAPACK95.


      IMPLICIT NONE

! Argument list variables (dummy arguments)
! -----------------------------------------

      DOUBLE PRECISION, DIMENSION(:,:),    INTENT(INOUT) :: a
      DOUBLE PRECISION, DIMENSION(:,:, :), INTENT(INOUT) :: b

      INTEGER, OPTIONAL, DIMENSION(:),     INTENT(OUT)   :: ipivot
      INTEGER, OPTIONAL,                   INTENT(OUT)   :: info


! Local variables
! ---------------

      INTEGER,          DIMENSION(SIZE(a,1)) :: ipivot_local
      DOUBLE PRECISION, DIMENSION(SIZE(b,1), SIZE(b,2), SIZE(a,2)) :: x
      INTEGER :: nra, nca, nrb, nrp
      INTEGER :: info_local

#ifdef DEBUG
      CHARACTER(LEN=72) :: c_fmt
      INTEGER :: i
#endif

      info_local = 0

      nra = SIZE(a, 1)
      nca = SIZE(a, 2)
      nrb = SIZE(b, 3)

      IF (PRESENT(ipivot)) THEN
        nrp = SIZE(ipivot)
      ELSE
        nrp = SIZE(a,1)
      ENDIF

      IF ((nra /= nca) .OR. (nra < 0)) THEN
        info_local = -1  ! error in the first argument
      ELSEIF (nrb /= nra) THEN
        info_local = -2  ! error in the second argument
      ELSEIF (nrp < nra) THEN
        info_local = -3  ! error in the third argument
                         ! (can only happen if actually present)
      ENDIF


      IF (info_local == 0) THEN
                 ! Proceed with the LR factorization of the
                 ! matrix A only if arguments valid
        CALL GM_DGETRF(a, ipivot_local, info_local)

        IF (info_local == 0) THEN

#ifdef DEBUG
      WRITE(*,'()')
      WRITE(*,'("LR factorization")')
      WRITE(*,'()')
      WRITE(c_fmt, '("(I4, 2X,", I0,"F8.3)")') nra
      DO i = 1, nra
        WRITE(*,c_fmt) ipivot_local(i), a(i,1:i-1), 1.0D+00
      ENDDO
      WRITE(*,'()')
      DO i = 1, nra
        WRITE(c_fmt, '("(I4, ", I0, "X,", I0,"F8.3)")')
     &    (i-1)*8+2, nra-i+1
        WRITE(*,c_fmt) ipivot_local(i), a(i,i:nra)
      ENDDO
      WRITE(*,'()')
#endif

                 ! Solution by back-substitution
          CALL GM_DGETRSM(a, ipivot_local, b, x)
        ENDIF

      ENDIF


      IF (info_local == 0) THEN
                 ! All went well: transcribe <x> into <b>
        b(:,:, 1:nra) = x(:,:, 1:nca)
        IF (PRESENT(ipivot)) ipivot(1:nra) = ipivot_local(:)
        IF (PRESENT(info)) info = info_local

      ELSE
                 ! There have been errors caught
        IF (PRESENT(info)) THEN
          info = info_local
        ELSE      
          WRITE(jp_stderr, '("[MOD_GAUSS/GM_DGESV1_ARRM]: INFO = ",' //
     &                  ' I0, " cannot be reported -- Aborting.")')
          CALL ABORT_MEDUSA()
        ENDIF

      ENDIF


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE GM_DGESV1M
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE GM_DGETRF(a, ipivot, info)
!-----------------------------------------------------------------------

      IMPLICIT NONE

! Factorize the square matrix A into an lower triangular matrix L
! (with only 1's on the diagonal) and an upper triangular matrix R
! such that P*A = L*R, where P is a permutation matrix, represented by
! the 1D array ipivot. The Gauss elimination method is used
! (algorithm 4.22 from Engeln-Müllgens and Uhlig (1996, p. 74-75).
! Some details taken from gauss.for (ibid., included CD-ROM).


! Argument list dummy variables
! -----------------------------

! a (input/output): on input, represents the matrix A of the square
!   linear system A*x = b; on output, the components of the L and
!   R triangular matrices (the diagonal of L is omitted, since it is
!   implicitly set to 1);

! ipivot (output): compact representation of the permutation matrix P
!   (ipivot(irow) = iorig <=> row <iorig> of the original matrix A
!   has become row <irow> in the LR representation of A on output;

! info (output): flag
!   = 0, successful processing
!   = i>0: row (i,:) completely zero;
!          column(:,i) requires too small a pivot (U(i,i) practically 0)

      DOUBLE PRECISION, INTENT(INOUT) :: a(:,:)
      INTEGER, INTENT(OUT)            :: ipivot(:)
      INTEGER, INTENT(OUT)            :: info


      DOUBLE PRECISION, DIMENSION(SIZE(a, 1)) :: scale_inv
      DOUBLE PRECISION rowsum, pivot, swap, factor, pivot_test
      DOUBLE PRECISION, DIMENSION(SIZE(a,2))  :: swap_row

      INTEGER irow, idiag, ipvt, jcol, jjcol

                 ! Set the relative error constant as parameter.
                 ! Note: in gauss.for from Engeln-Müllgens and
                 ! Uhlig (1996, CDROM), relerr = 8*fmachp, and
                 ! fmachp is actually equal to EPSILON(1D0)/2,
                 ! explicitly calculated. Here we use the intrinsic
                 ! Fortran 90 function.
      DOUBLE PRECISION, PARAMETER :: relerr = 4D0*EPSILON(1D0)

      INTEGER n



      n = SIZE(a,1)

      info = 0


! Initialize the ipivot array and the scales

      DO irow = 1, n

        ipivot(irow) = irow

        rowsum = SUM(DABS(a(irow,:)))

        IF (rowsum == 0.0D0) THEN
          info = irow
          WRITE(jp_stderr, '("[MOD_GAUSS/GM_DGETRF]: Row ", I0,' //
     &                  ' " completely zero")') irow
          RETURN
        ELSE
          scale_inv(irow) = 1.0D0/rowsum
        END IF
      ENDDO

                 ! If n equals 1, then we are done
      IF(n == 1) RETURN


! Proceed with the LR factorization.

      DO jcol = 1, n-1

                 ! Row index of the diagonal element in the current column
        idiag = jcol

                 ! Scale column for greatest scaled pivot element
                 !  - initialize pivot element
        pivot = DABS(a(idiag,jcol))*scale_inv(idiag)
        ipvt  = idiag

                 !  - scan to the end of the column
        DO irow = idiag+1, n

          pivot_test = DABS(a(irow,jcol))*scale_inv(irow)

          IF (pivot_test > pivot) THEN
            pivot = pivot_test
            ipvt  = irow
          END IF

        ENDDO

        IF (pivot < relerr) THEN

          info = jcol
          WRITE(jp_stderr, '("[MOD_GAUSS/GM_DGETRF]: Column ",' //
     &                  ' I0, " requires too small pivot")') jcol
          RETURN

        ELSE

          IF (ipvt /= idiag) THEN

                 ! Swap rows <idiag> and <ipvt> of A

            irow             = ipivot(idiag)
            ipivot(idiag)    = ipivot(ipvt)
            ipivot(ipvt)     = irow

            swap             = scale_inv(idiag)
            scale_inv(idiag) = scale_inv(ipvt)
            scale_inv(ipvt)  = swap

            swap_row         = a(idiag,:)
            a(idiag,:)       = a(ipvt,:)
            a(ipvt,:)        = swap_row

          END IF


! Complete the L and R components

          DO irow = idiag+1, n

            factor = a(irow,jcol)/a(idiag,jcol)
            a(irow,jcol) = factor
            a(irow,jcol+1:n) = a(irow, jcol+1:n) -
     &                         a(idiag,jcol+1:n) * factor

          ENDDO

        END IF

      ENDDO

      IF (DABS(a(n,n)) < relerr) THEN

         info = n
         WRITE(jp_stderr, '("[MOD_GAUSS/GM_DGETRF]: ", ' //
     &                 ' "Last diagonal element too small")')

      ENDIF


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE GM_DGETRF
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE GM_DGETRS(l_r, ipivot, b, x)
!-----------------------------------------------------------------------

      IMPLICIT NONE

! Calculate the solution of A*x = b, where A has been factorized
! by the subroutine GM_DGETRF to P*A = L*R, P being a permutation
! matrix (described by ipivot), and l_r contains the L and R
! triangular matrixes L and R. The adopted procedure follows
! algorithm 4.23 from Engeln-Müllgens and Uhlig (1996, p. 75).


! Argument list dummy variables
! -----------------------------

! l_r (input): LR representation of the original matrix A permuted
!    (i.e., where P*A_orig = L*R)
!
! ipivot (input): compact representation of the permutation matrix P
!    provided by GM_DGETRF
!
! b (input): right-hand side of the system
!
! x (output): solution vector
 
      DOUBLE PRECISION, DIMENSION(:,:), INTENT(IN)  :: l_r
      INTEGER,          DIMENSION(:),   INTENT(IN)  :: ipivot
      DOUBLE PRECISION, DIMENSION(:),   INTENT(IN)  :: b
      DOUBLE PRECISION, DIMENSION(:),   INTENT(OUT) :: x


! Local variables
! ---------------

      INTEGER :: ipvt, irow, jcol, k
      DOUBLE PRECISION :: dotprod
      DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE, SAVE :: c
      INTEGER :: n

! We have: P*A = L*R and we search for A*x = b
! Hence: P*A*x = P*b
! and thus L*R*x = P*b
! Solution in two stages
! 1. solve L*c = P*b for c
! 2. solve R*x = c for x


      n = SIZE(l_r, 1)

      IF (n == 1) THEN
        x(1) = b(1)/l_r(1,1)
        RETURN
      END IF


      IF (.NOT. ALLOCATED(c)) THEN

        ALLOCATE(c(n))

      ELSE

        IF (SIZE(c) < n) THEN
          DEALLOCATE(c)
          ALLOCATE(c(n))
        ENDIF

      ENDIF


! Stage 1: solve L*c = P*b for c

      ipvt = ipivot(1)
      c(1) = b(ipvt)

      DO irow = 2, n

!        dotprod = 0.0D0

!        DO k = 1, irow-1
!          dotprod = dotprod + l_r(irow, k)*c(k)
!        ENDDO

        ipvt    = ipivot(irow)
        dotprod = DOT_PRODUCT(l_r(irow, 1:irow-1),c(1:irow-1))
        c(irow) = b(ipvt) - dotprod

      ENDDO


! Stage 2: solve R*x = c for x

      x(n) = c(n)/l_r(n,n)

      DO irow = n-1, 1, -1

!        dotprod = 0.0D0

!        DO k = irow+1, n
!          dotprod = dotprod + l_r(irow,k)*x(k)
!        ENDDO

        dotprod = DOT_PRODUCT(l_r(irow,irow+1:n),x(irow+1:n))
        x(irow) = (c(irow) - dotprod)/l_r(irow,irow)

      ENDDO


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE GM_DGETRS
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE GM_DGETRSM(l_r, ipivot, b, x)
!-----------------------------------------------------------------------

      IMPLICIT NONE

! Calculate the solution of A*x = b, where A has been factorized
! by the subroutine GM_DGETRF to P*A = L*R, P being a permutation
! matrix (described by ipivot), and l_r contains the L and R
! triangular matrixes L and R. The adopted procedure follows
! algorithm 4.23 from Engeln-Müllgens and Uhlig (1996, p. 75).


! Argument list dummy variables
! -----------------------------

! l_r (input): LR representation of the original matrix A permuted
!    (i.e., where P*A_orig = L*R)
!
! ipivot (input): compact representation of the permutation matrix P
!    provided by GM_DGETRF
!
! b (input): right-hand side of the system, a 1D array of B(m,m)
!    matrices provided in a 3D-array b(m1,m2,n)
!
! x (output): solution array x(m1,m2,n) of X(m1,m2) matrices
 
      DOUBLE PRECISION, DIMENSION(:,:),   INTENT(IN)  :: l_r
      INTEGER,          DIMENSION(:),     INTENT(IN)  :: ipivot
      DOUBLE PRECISION, DIMENSION(:,:,:), INTENT(IN)  :: b
      DOUBLE PRECISION, DIMENSION(:,:,:), INTENT(OUT) :: x


! Local variables
! ---------------

      INTEGER :: ipvt, irow, jcol, k
      DOUBLE PRECISION, DIMENSION(:,:),   ALLOCATABLE, SAVE :: dotprod
      DOUBLE PRECISION, DIMENSION(:,:,:), ALLOCATABLE, SAVE :: c
      INTEGER :: m1, m2, n

! We have: P*A = L*R and we search for A*x = b
! Hence: P*A*x = P*b
! and thus L*R*x = P*b
! Solution in two stages
! 1. solve L*c = P*b for c
! 2. solve R*x = c for x


      n = SIZE(l_r, 1)
      m1 = SIZE(b, 1)
      m2 = SIZE(b, 2)

      IF (n == 1) THEN
        x(:,:, 1) = b(:,:, 1)/l_r(1,1)
        RETURN
      END IF


      IF (.NOT. ALLOCATED(c)) THEN
        ALLOCATE(c(m1,m2, n))
      ELSE
        IF (ANY(SIZE(c) /= (/ m1, m2, n /))) THEN
          DEALLOCATE(c)
          ALLOCATE(c(m1, m2, n))
        ENDIF
      ENDIF

      IF (.NOT. ALLOCATED(dotprod)) THEN
        ALLOCATE(dotprod(m1,m2))
      ELSE
        IF (ANY(SIZE(dotprod) /= (/ m1, m2 /))) THEN
          DEALLOCATE(dotprod)
          ALLOCATE(dotprod(m1, m2))
        ENDIF
      ENDIF


! Stage 1: solve L*c = P*b for c

      ipvt = ipivot(1)
      c(:,:, 1) = b(:,:, ipvt)

      DO irow = 2, n

        dotprod(:,:) = 0.0D0

        DO k = 1, irow-1
          dotprod(:,:) = dotprod(:,:) + l_r(irow, k)*c(:,:, k)
        ENDDO

        ipvt    = ipivot(irow)
!        dotprod = DOT_PRODUCT(l_r(irow, 1:irow-1),c(1:irow-1))
        c(:, :, irow) = b(:, :, ipvt) - dotprod(:,:)

      ENDDO


! Stage 2: solve R*x = c for x

      x(:,:, n) = c(:,:, n)/l_r(n,n)

      DO irow = n-1, 1, -1

        dotprod(:,:) = 0.0D0

        DO k = irow+1, n
          dotprod(:,:) = dotprod(:,:) + l_r(irow,k)*x(:,:, k)
        ENDDO

!        dotprod = DOT_PRODUCT(l_r(irow,irow+1:n),x(irow+1:n))
        x(:,:, irow) = (c(:,:, irow) - dotprod(:,:))/l_r(irow,irow)

      ENDDO


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE GM_DGETRSM
!-----------------------------------------------------------------------



!=======================================================================
      END MODULE MOD_GAUSS
!=======================================================================

