!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


! Provisional override: always assume that we have ABORT()
#ifndef HAS_ABORT
#define HAS_ABORT
#endif
!=======================================================================
      MODULE MOD_EXECONTROL_MEDUSA
!=======================================================================

      USE mod_defines_medusa, ONLY: jp_stderr, jp_stdlog
#ifdef DEBUG
      USE mod_defines_medusa, ONLY: jp_stddbg
#endif
#ifdef ALLOW_MPI
      USE mpi
#endif

      IMPLICIT NONE

      ! Insert here buffer for messages
      ! CHARACTER(LEN=<max_linelen>), DIMENSION(<max_nlines>)
      ! & :: c_errmsg

      PRIVATE

      PUBLIC :: ABORT_MEDUSA, MEDEXE_NPROC, MEDEXE_RANKFILENAME
#ifdef ALLOW_MPI
      PUBLIC :: jp_exeproc_root, jp_exeproc_ncio,
     &          lp_exeproc_singleproc_nc,
     &          MEDEXE_MPI_INIT, MEDEXE_MPI_FINALIZE,
     &          MEDEXE_MPI_COMM, MEDEXE_MPI_COMM_RANK,
     &          MEDEXE_MPI_SETTOPO,
     &          MEDEXE_MPI_GETTOPO_NDIMS, MEDEXE_MPI_GETTOPO_DIMLENS,
     &          MEDEXE_MPI_TOPO2D_PP4RANK, MEDEXE_MPI_TOPO2D_RANK4PP,
     &          MEDEXE_MPI_SETPARTITIONING, MEDEXE_MPI_GETPARTITIONING

c~       PUBLIC :: ndims_mpitopo, nlens_mpitopo,
c~      &          nsedcol_pproc, ioffset_sedcol_pproc, iproc_1stocn,
c~      &          nsedcol_seafloor_global
#endif


#ifdef ALLOW_MPI
      LOGICAL            :: l_mpi_initbymedusa = .FALSE.
      INTEGER            :: j_mpi_comm_medusa  = MPI_COMM_NULL
      INTEGER            :: n_exeproc          = 0
      INTEGER            :: j_exeproc_myrank   = MPI_PROC_NULL

      LOGICAL, PARAMETER :: lp_exeproc_singleproc_nc = .TRUE.
      INTEGER, PARAMETER :: jp_exeproc_root   = 0
      INTEGER, PARAMETER :: jp_exeproc_ncio   = jp_exeproc_root
#else
      INTEGER, PARAMETER :: n_exeproc         =  1
#endif

#ifdef ALLOW_MPI
      INTEGER, SAVE      :: ndims_mpitopo = -1
      INTEGER, SAVE, DIMENSION(:), ALLOCATABLE :: nlens_mpitopo

                                    ! - General data relevant to all
                                    !   mod_seafloor_central instances
                                    !   in the model (required for MPI
                                    !   multi-processor environments).

      INTEGER, DIMENSION(:), ALLOCATABLE, SAVE :: nsedcol_pproc
      INTEGER, DIMENSION(:), ALLOCATABLE, SAVE :: ioffset_sedcol_pproc
      INTEGER, SAVE                            :: iproc_1stocn = -1

      INTEGER, SAVE :: nsedcol_seafloor_global   = -1
#endif


      CONTAINS

!-----------------------------------------------------------------------
      SUBROUTINE ABORT_MEDUSA
!-----------------------------------------------------------------------

#ifdef ALLOW_MPI
      USE mpi
#endif
      USE mod_defines_medusa, ONLY: jp_stderr

      IMPLICIT NONE

#ifdef ALLOW_MPI
      INTEGER :: ierrcode, istatus
      LOGICAL :: l_inited
#endif

#ifdef ALLOW_MPI
      CALL MPI_INITIALIZED(l_inited, istatus)
      IF (l_inited) THEN
        CLOSE(jp_stderr)
        CLOSE(jp_stdlog)
#  ifdef DEBUG
        CLOSE(jp_stddbg)
#  endif
        ierrcode = 12321
        CALL MPI_ABORT(MPI_COMM_WORLD, ierrcode, istatus)
        STOP
      ENDIF
#endif

#ifdef HAS_ABORT
      CLOSE(jp_stderr)
      CLOSE(jp_stdlog)
#  ifdef DEBUG
      CLOSE(jp_stddbg)
#  endif
      CALL ABORT()
#else
      WRITE(jp_stderr,'()')
      WRITE(jp_stderr,'(A)') '[ABORT_MEDUSA] Aborting by STOP!'
      STOP
#endif


!-----------------------------------------------------------------------
      END SUBROUTINE ABORT_MEDUSA
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      INTEGER FUNCTION MEDEXE_NPROC()
!-----------------------------------------------------------------------

      IMPLICIT NONE

      MEDEXE_NPROC = n_exeproc

      RETURN

!-----------------------------------------------------------------------
      END FUNCTION MEDEXE_NPROC
!-----------------------------------------------------------------------




#ifdef ALLOW_MPI
!-----------------------------------------------------------------------
      SUBROUTINE MEDEXE_MPI_INIT(k_mpi_comm_host)
!-----------------------------------------------------------------------

      IMPLICIT NONE


      INTEGER, INTENT(IN), OPTIONAL :: k_mpi_comm_host

      INTEGER :: istatus


      CHARACTER(LEN=*), PARAMETER :: cfmt_err_a
     &  = '("[MEDEXE_MPI_INIT] error: ", A)' 


      IF (n_exeproc == 0) THEN

        IF (PRESENT(k_mpi_comm_host)) THEN
          j_mpi_comm_medusa = k_mpi_comm_host
        ELSE
          CALL MPI_INIT(istatus)
          j_mpi_comm_medusa = MPI_COMM_WORLD
          l_mpi_initbymedusa = .TRUE.
        ENDIF

        CALL MPI_COMM_SIZE(j_mpi_comm_medusa, n_exeproc, istatus)

        CALL MPI_COMM_RANK(j_mpi_comm_medusa, j_exeproc_myrank, istatus)

        IF ((jp_exeproc_ncio < 0) .OR.
     &      (jp_exeproc_ncio >= n_exeproc)) THEN
          
          WRITE(jp_stderr, cfmt_err_a)
     &      'Rank of NetCDF writing process outside valid range'
          WRITE(jp_stderr, '(" - should be ", I0, " to ", I0)')
     &                              0, n_exeproc-1
          WRITE(jp_stderr, '(" - is ", I0)') jp_exeproc_ncio
          WRITE(jp_stderr, '("Aborting!")')
          CALL ABORT_MEDUSA()

        ENDIF

                                    ! Set default topology: linear
        ndims_mpitopo = 1
        ALLOCATE(nlens_mpitopo(0:ndims_mpitopo-1))
        nlens_mpitopo(0) = n_exeproc

      ELSE

        WRITE(jp_stderr, cfmt_err_a) 'Init already done -- aborting!'
        CALL ABORT_MEDUSA()

      ENDIF

      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE MEDEXE_MPI_INIT
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE MEDEXE_MPI_SETTOPO(klens_mpitopo)
!-----------------------------------------------------------------------

                                    ! Re-set MPI process topology
      IMPLICIT NONE

      INTEGER, DIMENSION(:), INTENT(IN) :: klens_mpitopo

      LOGICAL, SAVE                     :: l_firstcall = .TRUE.
      INTEGER                           :: ndims
      INTEGER                           :: nproc_klens
      CHARACTER(LEN=63)                 :: cfmt
      INTEGER                           :: iflag

      CHARACTER(LEN=*), PARAMETER :: cfmt_err_a =
     &  '("[MEDEXE_MPI_SETTOPO] error: ", A)'


      IF (l_firstcall) THEN

        nproc_klens = PRODUCT(klens_mpitopo)
        ndims = SIZE(klens_mpitopo)

        IF (nproc_klens /= n_exeproc) THEN

          WRITE(jp_stderr, cfmt_err_a)
     &      'Incompatible number of MPI processes'
          WRITE(cfmt, '(A,I0,A)')
     &      '(A, I0, ', ndims, '(" * ", I0), " = ", I0)'
          WRITE(jp_stderr, cfmt)
     &      ' - expected : ', klens_mpitopo, nproc_klens
          WRITE(jp_stderr, '(A, I0)')
     &      ' - actually : ', n_exeproc
          WRITE(jp_stderr, '("Aborting!")')

                                    ! This error will be detected
                                    ! by all processes: let them reach
                                    ! this point to write out all their
                                    ! information ...
          CALL MPI_BARRIER(j_mpi_comm_medusa, iflag)

                                    ! ... and then abort.
          CALL ABORT_MEDUSA()

        ENDIF


        IF (ndims /= ndims_mpitopo) THEN
          DEALLOCATE(nlens_mpitopo)
          ALLOCATE(nlens_mpitopo(ndims))
          ndims_mpitopo = ndims
        ENDIF

        nlens_mpitopo(:) = klens_mpitopo(:)

        l_firstcall = .FALSE.

      ELSE

        WRITE(jp_stderr, cfmt_err_a)
     &    'multi-processor topology can only be reset once -- aborting.'
        CALL ABORT_MEDUSA()

      ENDIF

      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE MEDEXE_MPI_SETTOPO
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE MEDEXE_MPI_GETTOPO_NDIMS(kdims_mpitopo)
!-----------------------------------------------------------------------

      IMPLICIT NONE

      INTEGER, INTENT(OUT) :: kdims_mpitopo


      kdims_mpitopo = ndims_mpitopo

      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE MEDEXE_MPI_GETTOPO_NDIMS
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE MEDEXE_MPI_GETTOPO_DIMLENS(klens_mpitopo)
!-----------------------------------------------------------------------

      IMPLICIT NONE

      INTEGER, DIMENSION(:), INTENT(OUT) :: klens_mpitopo

      INTEGER :: ndims

      CHARACTER(LEN=*), PARAMETER :: cfmt_err_a =
     &  '("[MEDEXE_MPI_GETTOPO_DIMLENS] error: ", A)'

      ndims = SIZE(klens_mpitopo)

      IF (ndims /= ndims_mpitopo) THEN
        WRITE(jp_stderr, cfmt_err_a)
     &    'incompatible "klens_mpitopo" argument size'
        WRITE(jp_stderr, '(" - expected: ", I0)') ndims_mpitopo
        WRITE(jp_stderr, '(" - actual:   ", I0)') ndims
        WRITE(jp_stderr, '(A)') 'Aborting.'
        CALL ABORT_MEDUSA()
      ELSE
        klens_mpitopo(:) = nlens_mpitopo(:)
      ENDIF

      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE MEDEXE_MPI_GETTOPO_DIMLENS
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      INTEGER FUNCTION MEDEXE_MPI_TOPO2D_RANK4PP(ipx, jpy)
!-----------------------------------------------------------------------

      IMPLICIT NONE
                                    ! Process coordinates (ipx, jpy)
                                    ! running form 1 to npx; 1 to npy !
                                    ! Ranks run from 0 to n_exeproc-1
      INTEGER, INTENT(IN) :: ipx, jpy

      IF (ndims_mpitopo == 2) THEN
        MEDEXE_MPI_TOPO2D_RANK4PP = (jpy-1)*nlens_mpitopo(1) + ipx-1
      ELSE
        MEDEXE_MPI_TOPO2D_RANK4PP = MPI_PROC_NULL
      ENDIF

      RETURN

!-----------------------------------------------------------------------
      END FUNCTION MEDEXE_MPI_TOPO2D_RANK4PP
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE MEDEXE_MPI_TOPO2D_PP4RANK(krank, ipx, jpy)
!-----------------------------------------------------------------------

      IMPLICIT NONE
                                    ! Process coordinates (ipx, jpy)
                                    ! running form 1 to npx; 1 to npy !
                                    ! Ranks run from 0 to n_exeproc-1
      INTEGER, INTENT(IN)  :: krank
      INTEGER, INTENT(OUT) :: ipx, jpy

      IF (ndims_mpitopo == 2) THEN
        IF ((krank >= 0) .AND. (krank < n_exeproc)) THEN
          ipx = MOD(krank, nlens_mpitopo(1)) + 1
          jpy = krank/nlens_mpitopo(1) + 1
        ELSE
          ipx = -1
          jpy = -1
        ENDIF

      ELSE

        ipx = -1
        jpy = -1
      
      ENDIF

      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE MEDEXE_MPI_TOPO2D_PP4RANK
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
      SUBROUTINE MEDEXE_MPI_SETPARTITIONING(nsedcol_seafloor_central)
!-----------------------------------------------------------------------

                                    ! Re-set MPI process topology
      IMPLICIT NONE

      INTEGER, INTENT(IN) :: nsedcol_seafloor_central

      INTEGER :: i, iflag

      CHARACTER(LEN=*), PARAMETER :: cfmt_err_a =
     &  '("[MEDEXE_MPI_SETPARTITIONING] error: ", A)'

      IF (j_mpi_comm_medusa == MPI_COMM_NULL) THEN

        WRITE(jp_stderr, cfmt_err_a)
     &    'MEDEXE_MPI_INIT must be called before ' //
     &    'MEDEXE_MPI_SETPARTITIONING!'
        WRITE(jp_stderr, '("-- Aborting")')
        CALL ABORT_MEDUSA()

      ENDIF


      IF (nsedcol_seafloor_global == -1) THEN

        ALLOCATE(nsedcol_pproc(0:n_exeproc-1))
        ALLOCATE(ioffset_sedcol_pproc(0:n_exeproc-1))

        IF (n_exeproc == 1) THEN

          nsedcol_pproc(0)        = nsedcol_seafloor_central
          ioffset_sedcol_pproc(0) = 0
          nsedcol_seafloor_global = nsedcol_seafloor_central

        ELSE

          CALL MPI_ALLGATHER(nsedcol_seafloor_central, 1, MPI_INTEGER,
     &      nsedcol_pproc(:), 1, MPI_INTEGER, j_mpi_comm_medusa, iflag)

          ioffset_sedcol_pproc(0) = 0
          DO i = 1, n_exeproc-1
            ioffset_sedcol_pproc(i)
     &        = ioffset_sedcol_pproc(i-1) + nsedcol_pproc(i-1)
          ENDDO

          nsedcol_seafloor_global = ioffset_sedcol_pproc(n_exeproc-1)
     &                              + nsedcol_pproc(n_exeproc-1)

          IF (nsedcol_seafloor_global == 0) THEN
            WRITE(jp_stderr, cfmt_err_a)
     &        'No valid ocean-seafloor grid points found at all!' 
            WRITE(jp_stderr, '("-- Aborting")')
            CALL ABORT_MEDUSA()
          ENDIF

        ENDIF

        IF (lp_exeproc_singleproc_nc) THEN
                                    ! If single-processor NetCDF I/O is
                                    ! active, check if the writing process'
                                    ! mod_seafloor_central is empty and
                                    ! if so, find the first process that
                                    ! has a non empty mod_seafloor_central.

          IF (jp_exeproc_ncio < n_exeproc) THEN     ! Can only do this if the
                                                    ! selected writer's rank
                                                    ! is executing

            IF (nsedcol_pproc(jp_exeproc_ncio) > 0) THEN

              iproc_1stocn = jp_exeproc_ncio

            ELSE

              DO i = 0, n_exeproc-1
                IF (nsedcol_pproc(i) > 0) THEN
                  iproc_1stocn = i
                  EXIT
                ENDIF
              ENDDO

            ENDIF

          ENDIF

        ENDIF

      ELSE

        WRITE(jp_stderr, cfmt_err_a)
     &    'cannot reset index and offset arrays -- aborting.'
        CALL ABORT_MEDUSA()

      ENDIF

      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE MEDEXE_MPI_SETPARTITIONING
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE MEDEXE_MPI_GETPARTITIONING
     &                              (knsedcol_global, kproc_1stocn,
     &                               knsedcol_pproc, kioffset_pproc)
!-----------------------------------------------------------------------

      IMPLICIT NONE

      INTEGER,               INTENT(OUT), OPTIONAL :: knsedcol_global
      INTEGER,               INTENT(OUT), OPTIONAL :: kproc_1stocn
      INTEGER, DIMENSION(:), INTENT(OUT), OPTIONAL :: knsedcol_pproc
      INTEGER, DIMENSION(:), INTENT(OUT), OPTIONAL :: kioffset_pproc

      CHARACTER(LEN=*), PARAMETER :: cfmt_err_a =
     &  '("[MEDEXE_MPI_GETPARTITIONING] error: ", A)'

      IF (nsedcol_seafloor_global > 0) THEN

        IF (PRESENT(knsedcol_global))
     &    knsedcol_global = nsedcol_seafloor_global
        IF (PRESENT(kproc_1stocn))
     &    kproc_1stocn = iproc_1stocn
        IF (PRESENT(knsedcol_pproc))
     &    knsedcol_pproc(:) = nsedcol_pproc(:)
        IF (PRESENT(kioffset_pproc))
     &    kioffset_pproc(:) = ioffset_sedcol_pproc(:)

      ELSE

        WRITE(jp_stderr, cfmt_err_a)
     &    'index and count arrays not yet set up -- aborting.'
        CALL ABORT_MEDUSA()

      ENDIF

      RETURN 

!-----------------------------------------------------------------------
      END SUBROUTINE MEDEXE_MPI_GETPARTITIONING
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE MEDEXE_MPI_FINALIZE
!-----------------------------------------------------------------------

      IMPLICIT NONE


      INTEGER :: istatus

      IF (l_mpi_initbymedusa) THEN
        CALL MPI_FINALIZE(istatus)
      ELSE
        RETURN
      ENDIF

      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE MEDEXE_MPI_FINALIZE
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      INTEGER FUNCTION MEDEXE_MPI_COMM()
!-----------------------------------------------------------------------

      IMPLICIT NONE

      MEDEXE_MPI_COMM = j_mpi_comm_medusa

      RETURN

!-----------------------------------------------------------------------
      END FUNCTION MEDEXE_MPI_COMM
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      INTEGER FUNCTION MEDEXE_MPI_COMM_RANK()
!-----------------------------------------------------------------------

      IMPLICIT NONE

      MEDEXE_MPI_COMM_RANK = j_exeproc_myrank

      RETURN

!-----------------------------------------------------------------------
      END FUNCTION MEDEXE_MPI_COMM_RANK
!-----------------------------------------------------------------------
#endif


!-----------------------------------------------------------------------
      SUBROUTINE MEDEXE_RANKFILENAME(cfn_any)
!-----------------------------------------------------------------------

      IMPLICIT NONE


      CHARACTER(LEN=*) cfn_any



#ifdef ALLOW_MPI
      LOGICAL :: l_firstcall = .TRUE.
      INTEGER :: nlen, ilt, ipos
      CHARACTER(LEN=12) :: ctmp
      CHARACTER(LEN=33) :: cfmt = '("_", I2.2)' ! 33 = 9 + 2*12
      CHARACTER(LEN=13), SAVE :: c_rank = '_nn' ! 13 = 1 + 12
#endif

                                    ! If single-processor execution:
                                    ! no need to rank file names.
      IF (n_exeproc == 1) RETURN

#ifdef ALLOW_MPI
                                    ! Check if this module has already
      IF (n_exeproc == 0) THEN      ! been set up - if not abort.
        WRITE(jp_stderr,'(A)') '[MEDEXE_RANKFILENAME] error: ' //
     &    'MOD_EXECONTROL_MEDUSA init not yet done -- aborting!'
        CALL ABORT_MEDUSA()
      ENDIF


      IF (l_firstcall) THEN         ! Adapt format for number insertion
                                    ! at the first call
        WRITE(ctmp,'(I0)') n_exeproc
        nlen = LEN_TRIM(ctmp)
        WRITE(cfmt, '("(""_"", I",I0,".",I0,")")') nlen, nlen
        WRITE(c_rank, cfmt) j_exeproc_myrank

        l_firstcall = .FALSE.

      ENDIF

                                    ! Search for the last '.' in the name,
                                    ! we assume that it delimits the extension
      ilt = LEN_TRIM(cfn_any)

      IF (ilt == 0) RETURN          ! Return if filename is an empty string

      ipos = INDEX(cfn_any, '.', BACK=.TRUE.)

      IF (ipos > 0) THEN
                                    ! Check for a NetCDF file extension:
                                    ! NetCDF file get ranked only if
                                    ! single-processor NetCDF I/O is *not*
                                    ! selected
        SELECT CASE(cfn_any(ipos:ilt))
        CASE('.nc', '.nc4')         ! NetCDF file extensions' list: may be extended
          IF (lp_exeproc_singleproc_nc) RETURN      ! Return if single-proc NC I/O,
        END SELECT
                                    ! else, insert rank number before the extension
        cfn_any = cfn_any(1:ipos-1) // TRIM(c_rank) // cfn_any(ipos:ilt)
          
      ELSE
                                    ! No extension: append rank number
        cfn_any = cfn_any(1:ilt) // TRIM(c_rank)

      ENDIF

#endif

      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE MEDEXE_RANKFILENAME
!-----------------------------------------------------------------------



!=======================================================================
      END MODULE MOD_EXECONTROL_MEDUSA
!=======================================================================
