!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


!-----------------------------------------------------------------------
      SUBROUTINE SETUP_TRANSPORT
!-----------------------------------------------------------------------

!--------------
! Declarations
!--------------


      USE mod_defines_medusa,       ONLY: jp_stderr, jp_stdlog
#ifdef DEBUG
      USE mod_defines_medusa,       ONLY: jp_stddbg
#endif
      USE mod_execontrol_medusa,    ONLY: ABORT_MEDUSA
      USE mod_logunits

#ifdef ALLOW_MPI
      USE mod_execontrol_medusa,    ONLY: MEDEXE_MPI_COMM,
     &                              MEDEXE_MPI_COMM_RANK,
     &                              jp_exeproc_root
      USE mpi, ONLY: MPI_INTEGER, MPI_DOUBLE_PRECISION
#endif


      IMPLICIT NONE


!-----------------------
! Variable declarations
!-----------------------

!- - - - - - - - - - - - -  - - - - - - -
! Variables in subroutine call arguments
!- - - - - - - - - - - - -- - - - - - - -

      ! None


!- - - - - - - - - - - - - -  - - - - - - - -
! General (global) parameters and definitions
!- - - - - - - - - - - - - -- - - - - - - - -

      CHARACTER(LEN=*), PARAMETER ::
     &  cfn_transport_cfg = "medusa_transport_config.nml"


!- - - - - - - - -
! Local variables
!- - - - - - - - -


#ifdef ALLOW_MPI
      INTEGER :: i_mycomm, i_myrank
#endif




      CHARACTER(LEN=15)             :: ctype_biodiffusion
      CHARACTER(LEN=15)             :: ctype_bioirrigation
      CHARACTER(LEN=15)             :: ctype_upwinding


      NAMELIST /nml_transport_options/
     &  ctype_biodiffusion, ctype_bioirrigation, ctype_upwinding


      DOUBLE PRECISION :: db_0
      DOUBLE PRECISION :: db_slope
      DOUBLE PRECISION :: db_scale
      DOUBLE PRECISION :: db_halfd

      NAMELIST /nml_biodif_const/   db_0
      NAMELIST /nml_biodif_lin0z/   db_0
      NAMELIST /nml_biodif_linxz/   db_0,  db_slope
      NAMELIST /nml_biodif_quad0z/  db_0
      NAMELIST /nml_biodif_expdec/  db_0,  db_scale
      NAMELIST /nml_biodif_gaussn/  db_0,  db_scale
      NAMELIST /nml_biodif_erfc/    db_0,  db_halfd, db_scale


      DOUBLE PRECISION :: rcf_0
      DOUBLE PRECISION :: rcf_scale

      NAMELIST /nml_bioirr_expdec/ rcf_0, rcf_scale


      INTEGER :: iu_cfg, istatus
      LOGICAL :: l_cfgfile_is_mine = .FALSE.
      LOGICAL :: l_exists

      CHARACTER(LEN=*), PARAMETER :: cfmt_modprocname_a
     &  = '("[MOD_TRANSPORT/setup_transport]: ", A)'

      CHARACTER(LEN=*), PARAMETER :: cfmt_a     = '(" - ", A)'
      CHARACTER(LEN=*), PARAMETER :: cfmt_a_ind = '(3X, A)'


!- - - - - - - - - - -
! End of declarations
!- - - - - - - - - - -


!----------------------------------------------------------------------
! Subroutine Start
!----------------------------------------------------------------------

#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_modprocname_a) 'Start'
#endif

      IF (l_setupdone) THEN
#ifdef DEBUG
        WRITE(jp_stddbg, cfmt_a) 'Setup already done'
        WRITE(jp_stddbg, cfmt_modprocname_a) 'Return'
        WRITE(jp_stddbg, '()')
#endif
        RETURN
      ENDIF


#ifdef ALLOW_MPI
      i_mycomm = MEDEXE_MPI_COMM()
      i_myrank = MEDEXE_MPI_COMM_RANK()
      l_cfgfile_is_mine = (i_myrank == jp_exeproc_root)
#else
      l_cfgfile_is_mine = .TRUE.
#endif

      WRITE(jp_stdlog, cfmt_modprocname_a) 'Initial report'
      WRITE(jp_stdlog, '()')
      WRITE(jp_stdlog, '(" Adopted transport options")')
      WRITE(jp_stdlog, '(" -------------------------")')
      WRITE(jp_stdlog, '()')


      IF (l_cfgfile_is_mine) THEN

                                    ! Pre-set the default values
        jselect_biodif_profile = jp_bt_const
        db_0 = dp_bt_dcf_0

        jselect_bioirr_profile = jp_bi_none
        rcf_0 = 0.0D+00

        jselect_upwinding  = jp_upw_full


                                    ! Check if file cfn_transport_cfg exists
        INQUIRE(FILE=cfn_transport_cfg, EXIST=l_exists)


        IF (l_exists) THEN          ! If file exists, read it and get
                                    ! information (else stick to defaults)

#ifdef DEBUG
          WRITE(jp_stddbg, cfmt_a)
     &      'Transport configuration file "' // cfn_transport_cfg //
     &      '" found.'
#endif

          istatus = RESERVE_LOGUNIT(iu_cfg)
          IF (istatus /= 0) THEN
            WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
            WRITE(jp_stderr, cfmt_a)
     &       'Unable to reserve a logical unit for "iu_cfg" -- aborting'
            CALL ABORT_MEDUSA()
#ifdef DEBUG
          ELSE
            WRITE(jp_stddbg, cfmt_a, ADVANCE="NO")
     &        'Assigning logical unit number '
            WRITE(jp_stddbg, '(I0, A)') iu_cfg, ' to "iu_cfg".'
#endif
          ENDIF


          OPEN(UNIT=iu_cfg, FILE=cfn_transport_cfg)

                                    ! Pre-set the three main options
                                    ! to the default values.
          ctype_biodiffusion  = '?'
          ctype_bioirrigation = '?'
          ctype_upwinding     = '?'

          READ(iu_cfg, NML=nml_transport_options)


#ifdef DEBUG
          WRITE(jp_stddbg, '()')
          WRITE(jp_stddbg, cfmt_a_ind) 'Transport options requested'
          WRITE(jp_stddbg, cfmt_a_ind) '==========================='
          WRITE(jp_stddbg, '()')

          WRITE(jp_stddbg, cfmt_a_ind, ADVANCE="NO")
          IF (ctype_biodiffusion == '?') THEN
            WRITE(jp_stddbg, '(" * ctype_biodiffusion  = ", A)')
     &                              '""const"" (DEFAULT)'
          ELSE
            WRITE(jp_stddbg, '(" * ctype_biodiffusion  = """, A, """")')
     &                              TRIM(ctype_biodiffusion)
          ENDIF
          WRITE(jp_stddbg, '()')


          WRITE(jp_stddbg, cfmt_a_ind, ADVANCE="NO")
          IF (ctype_bioirrigation == '?') THEN
            WRITE(jp_stddbg, '(" * ctype_bioirrigation = ", A)')
     &                              '""none"" (DEFAULT)'
          ELSE
            WRITE(jp_stddbg, '(" * ctype_bioirrigation = """, A, """")')
     &                              TRIM(ctype_bioirrigation)
          ENDIF
          WRITE(jp_stddbg, '()')

          WRITE(jp_stddbg, cfmt_a_ind, ADVANCE="NO")
          IF (ctype_upwinding == '?') THEN
            WRITE(jp_stddbg, '(" * ctype_upwinding     = ", A)')
     &                              '""full"" (DEFAULT)'
          ELSE
            WRITE(jp_stddbg, '(" * ctype_upwinding     = """, A, """")')
     &                              TRIM(ctype_upwinding)
          ENDIF
          WRITE(jp_stddbg, '()')
#endif


                                    ! Biodiffusion
                                    ! ============

          SELECT CASE(ctype_biodiffusion)
          CASE('?')
          !--------
            CONTINUE

          CASE('const')
          !------------
            jselect_biodif_profile = jp_bt_const

            READ(iu_cfg, NML=nml_biodif_const)

          CASE('lin0z')
          !------------
            jselect_biodif_profile = jp_bt_lin0z

            READ(iu_cfg, NML=nml_biodif_lin0z)

          CASE('linxz')
          !------------
            jselect_biodif_profile = jp_bt_linxz

            db_slope = dp_bt_linxz_slope
            READ(iu_cfg, NML=nml_biodif_linxz)

            IF (db_slope <= 1.0D+00) THEN
              ! slope must be > 1 (else DB(x) = 0 for z < z_Z)
              ! Print out jselect_biodif_profile
              CALL ABORT_MEDUSA()
            ENDIF

          CASE('quad0z')
          !-------------
            jselect_biodif_profile = jp_bt_quad0z

            READ(iu_cfg, NML=nml_biodif_quad0z)

          CASE('expdec')
          !-------------
            jselect_biodif_profile = jp_bt_expdec

            db_scale = dp_bt_expdec_scale
            READ(iu_cfg, NML=nml_biodif_expdec)

            IF (db_scale <= 0.0D+00) THEN
              ! scale must be > 0!
              ! Print out jselect_biodif_profile
              CALL ABORT_MEDUSA()
            ENDIF

          CASE('gaussn')
          !-------------
            jselect_biodif_profile = jp_bt_gaussn

            db_0     = dp_bt_gaussn_db_0
            db_scale = dp_bt_gaussn_scale
            READ(iu_cfg, NML=nml_biodif_gaussn)

            IF (db_scale <= 0.0D+00) THEN
              ! scale must be > 0!
              ! Print out jselect_biodif_profile
              CALL ABORT_MEDUSA()
            ENDIF

          CASE('erfc')
          !-----------
            jselect_biodif_profile = jp_bt_erfc

            db_halfd = dp_bt_erfc_halfd
            db_scale = dp_bt_erfc_scale
            READ(iu_cfg, NML=nml_biodif_erfc)

            IF (db_halfd <= 0.0D+00) THEN
              ! halfd must be > 0!
              ! Print out jselect_biodif_profile
              CALL ABORT_MEDUSA()
            ENDIF

            IF (db_scale <= 0.0D+00) THEN
              ! scale must be > 0!
              ! Print out jselect_biodif_profile
              CALL ABORT_MEDUSA()
            ENDIF

          CASE('custom')
          !-------------
#ifdef BIODIFFUSION_CUSTOM
            jselect_biodif_profile = jp_bt_custom

                                    ! Carry out stage 1 of the set-up
                                    ! of the custom biodiffusion profile
                                    ! (reading in the data from the
                                    ! file 'iu_cfg', currently open)
            CALL BDIFFC_CUSTOM_SETUP(1, iu_cfg)
#else
            WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
            WRITE(jp_stderr, cfmt_a)
     &        '"custom" biodiffusion profile not available -- aborting'
            CALL ABORT_MEDUSA()
#endif

          CASE DEFAULT
          !-----------
            WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
            WRITE(jp_stderr, cfmt_a)
     &        'unknown biodiffusion profile "' //
     &        TRIM(ctype_biodiffusion) // '" -- aborting'
            CALL ABORT_MEDUSA()

          END SELECT


          IF (db_0 < 0.0D+00) THEN
            ! DB_0 must be >= 0!
            ! Print out jselect_biodif_profile
            CALL ABORT_MEDUSA()
          ENDIF


                                    ! Bioirrigation
                                    ! =============

          SELECT CASE(ctype_bioirrigation)
          CASE('?')
          !--------
            CONTINUE

          CASE('none')
          !-----------
            jselect_bioirr_profile = jp_bi_none

          CASE('expdec')
          !-------------
            jselect_bioirr_profile = jp_bi_expdec

            rcf_0     = dp_bi_rcf_0
            rcf_scale = dp_bi_expdec_scale
            READ(iu_cfg, NML=nml_bioirr_expdec)

            IF (rcf_scale <= 0.0D+00) THEN
              ! rcf_scale must be > 0!
              ! Print out jselect_bioirr_profile
              CALL ABORT_MEDUSA()
            ENDIF

          CASE('custom')
          !-------------
#ifdef BIOIRRIGATION_CUSTOM
            jselect_bioirr_profile = jp_bi_custom

                                    ! Carry out stage 1 of the set-up
                                    ! of the custom bioirrigation profile
                                    ! (reading in the data from the
                                    ! file 'iu_cfg', currently open)
            CALL BDIFFC_CUSTOM_SETUP(1, iu_cfg)
#else
            WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
            WRITE(jp_stderr, cfmt_a)
     &        '"custom" bioirrigation profile not available -- aborting'
            CALL ABORT_MEDUSA()
#endif

          CASE DEFAULT
          !-----------
            WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
            WRITE(jp_stderr, cfmt_a)
     &        'unknown bioirrigation profile "' //
     &        TRIM(ctype_bioirrigation) // '" -- aborting'
            CALL ABORT_MEDUSA()

          END SELECT

          IF (rcf_0 < 0.0D+00) THEN
            ! rcf_0 must be >= 0!
            ! Print out jselect_bioirr_profile
            CALL ABORT_MEDUSA()
          ENDIF



          SELECT CASE(ctype_upwinding)
          CASE('?')
          !--------
            CONTINUE

          CASE('full')
          !-----------
            jselect_upwinding = jp_upw_full

          CASE('expfit')
          !-------------
            jselect_upwinding = jp_upw_expfit

          CASE DEFAULT
          !-----------
            WRITE(jp_stderr, cfmt_modprocname_a) 'Fatal error'
            WRITE(jp_stderr, cfmt_a)
     &        'unknown upwinding scheme "' //
     &        TRIM(ctype_upwinding) // '" -- aborting'
            CALL ABORT_MEDUSA()

          END SELECT


          CLOSE(UNIT=iu_cfg)


#ifdef DEBUG
          WRITE(jp_stddbg, cfmt_a, ADVANCE="NO")
     &      'Releasing logical unit number '
          WRITE(jp_stddbg, '(I0, A)', ADVANCE="NO")
     &      iu_cfg, ' (attached to "iu_cfg")'
#endif
          istatus = FREE_LOGUNIT(iu_cfg)
          IF (istatus /= 0) THEN
            WRITE(jp_stderr, cfmt_modprocname_a) 'Warning'
            WRITE(jp_stderr, cfmt_a, ADVANCE="NO")
     &        'FREE_LOGUNIT returned error code '
            WRITE(jp_stderr, '(I0, A)') istatus,
     &        ' when trying to release the logical unit number'
     &        //  ' attached to "iu_cfg" -- ignoring.'
#ifdef DEBUG
            WRITE(jp_stddbg, '()')
            WRITE(jp_stddbg, cfmt_a_ind, ADVANCE="NO")
            WRITE(jp_stddbg, '(A, I0, A)')
     &        'FREE_LOGUNIT returned error code ', istatus,
     &        ' when trying to release the logical unit number'
     &        //  ' attached to "iu_cfg" -- ignoring.'
          ELSE
            WRITE(jp_stddbg, '(" - done.")')
#endif
          ENDIF

#ifdef DEBUG
        ELSE

          WRITE(jp_stddbg, cfmt_a)
     &      'Transport configuration file "' // cfn_transport_cfg //
     &      '" not found - falling back to defaults'
#endif
        ENDIF

        CALL FLUSH(jp_stderr)
#ifdef DEBUG
        CALL FLUSH(jp_stddbg)
#endif

      ENDIF


#ifdef ALLOW_MPI
                                    ! Broadcast the configuration data:
                                    !  - biodiffusion: profile type
      CALL MPI_BCAST(jselect_biodif_profile, 1, MPI_INTEGER,
     &                              jp_exeproc_root, i_mycomm, istatus)

                                    ! Broadcast the other parameter values
                                    ! ONLY if NO custom profile is used.
      IF (jselect_biodif_profile /= jp_bt_custom) THEN
                                    !  - biodiffusion (all): D_B0
        CALL MPI_BCAST(db_0, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)

        SELECT CASE(jselect_biodif_profile)
        CASE(jp_bt_linxz)           !  - biodiffusion (linxz): slope
          CALL MPI_BCAST(db_slope, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
        CASE(jp_bt_expdec, jp_bt_gaussn)
                                    !  - biodiffusion (expdec, gaussn): scale
          CALL MPI_BCAST(db_scale, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
        CASE(jp_bt_erfc)
                                    !  - biodiffusion (erfc): halfd and scale
          CALL MPI_BCAST(db_halfd, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
          CALL MPI_BCAST(db_scale, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
        END SELECT
      ENDIF
#endif


#ifdef BIODIFFUSION_CUSTOM
      IF (jselect_biodif_profile == jp_bt_custom) THEN
                                    ! Complete the set-up of the custom
                                    ! biodiffusion profile.
        CALL BDIFFC_CUSTOM_SETUP(2, iu_cfg) ! stage 2; iu_cfg is ignored
      ENDIF
#endif


#ifdef ALLOW_MPI
                                    !  - bioirrigation: profile type
      CALL MPI_BCAST(jselect_bioirr_profile, 1, MPI_INTEGER,
     &                              jp_exeproc_root, i_mycomm, istatus)

                                    ! Broadcast the other parameter
                                    ! values ONLY if NO custom
                                    ! profile is used.
      IF (jselect_bioirr_profile /= jp_bi_custom) THEN
                                    !  - bioirrigation (all): rcf_0
        CALL MPI_BCAST(rcf_0, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)

        SELECT CASE(jselect_bioirr_profile)
        CASE(jp_bi_expdec)            !  - bioirrigation (expdec): scale
          CALL MPI_BCAST(rcf_scale, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
        END SELECT
      ENDIF
#endif


#ifdef BIOIRRIGATION_CUSTOM
      IF (jselect_bioirr_profile == jp_bi_custom) THEN
                                    ! Complete the set-up of the custom
                                    ! bioirrigation profile.
        CALL BIRRIC_CUSTOM_SETUP(2, iu_cfg) ! stage 2; iu_cfg is ignored
      ENDIF
#endif



#ifdef ALLOW_MPI
                                    !  - upwinding: type
      CALL MPI_BCAST(jselect_upwinding, 1, MPI_INTEGER,
     &                              jp_exeproc_root, i_mycomm, istatus)
#endif


                                    ! With custom biodiffusion  profiles,
                                    ! all has been done in BDIFFC_CUSTOM_SETUP
                                    ! at stage 2
      IF (jselect_biodif_profile /= jp_bt_custom) THEN

        da_bt_dcf_0 = db_0

        WRITE(jp_stdlog, '(" - for biodiffusion")')
        WRITE(jp_stdlog, '("   * profile ID: ", I0)')
     &                              jselect_biodif_profile
        WRITE(jp_stdlog, '("   * D_B0: ", E9.3)') da_bt_dcf_0

        SELECT CASE(jselect_biodif_profile)
        CASE(jp_bt_linxz)
          da_bt_linxz_slope  = db_slope
          WRITE(jp_stdlog, '("   * slope: ", E9.3)') da_bt_linxz_slope
        CASE(jp_bt_expdec)
          da_bt_expdec_scale = db_scale
          WRITE(jp_stdlog, '("   * scale: ", E9.3)') da_bt_expdec_scale
        CASE(jp_bt_gaussn)
          da_bt_gaussn_scale = db_scale
          WRITE(jp_stdlog, '("   * scale: ", E9.3)') da_bt_gaussn_scale
        CASE(jp_bt_erfc)
          da_bt_erfc_halfd = db_halfd
          WRITE(jp_stdlog, '("   * halfd: ", E9.3)') da_bt_erfc_halfd
          da_bt_erfc_scale = db_scale
          WRITE(jp_stdlog, '("   * scale: ", E9.3)') da_bt_erfc_scale
        END SELECT

        WRITE(jp_stdlog, '()')

      ENDIF


      IF (jselect_bioirr_profile /= jp_bi_custom) THEN

        da_bi_rcf_0 = rcf_0

        WRITE(jp_stdlog, '(" - for bioirrigation")')
        WRITE(jp_stdlog, '("   * profile ID: ", I0)')
     &                              jselect_bioirr_profile
        WRITE(jp_stdlog, '("   * alpha_0: ", E9.3)') da_bi_rcf_0

        SELECT CASE(jselect_bioirr_profile)
        CASE(jp_bi_expdec)
          da_bi_expdec_scale = rcf_scale
          WRITE(jp_stdlog, '("   * scale: ", E9.3)') da_bi_expdec_scale
        END SELECT

        WRITE(jp_stdlog, '()')

      ENDIF


      WRITE(jp_stdlog, '(" - for upwinding")')
      WRITE(jp_stdlog, '("   * option: ", I0)') jselect_upwinding
      WRITE(jp_stdlog, '()')

      WRITE(jp_stdlog, cfmt_modprocname_a) 'End of initial report'
      WRITE(jp_stdlog, '()')
      WRITE(jp_stdlog, '()')


      l_setupdone = .TRUE.

#ifdef DEBUG
      WRITE(jp_stddbg, cfmt_modprocname_a) 'End'
      WRITE(jp_stddbg, '()')
#endif


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE SETUP_TRANSPORT
!-----------------------------------------------------------------------
