!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


!-----------------------------------------------------------------------
      SUBROUTINE TRANSPORT
     &  (atime, xnz, inode, xnc, xvw_tot, xvu_tot, wconc, wfflx, yc,
     &  flxval)
!-----------------------------------------------------------------------

! Returns flux value through
!  - the vertex below the node 'inode' if inode < idnb
!  - the virtual vertex at idnb if inode = idbn


      USE mod_defines_medusa,  ONLY: jp_stderr
      USE mod_execontrol_medusa, ONLY: ABORT_MEDUSA
      USE mod_gridparam,       ONLY: idnw, idnt, idnz, idnb,
     &                               idvw, idvs,       idvb,
     &                               dp_swi_location
      USE mod_indexparam,      ONLY: nsolut, nsolid, ncompo,
     &                               jc_to_io, jf_to_io
      USE mod_milieucharas,    ONLY: xvphi, xvtor2, xphi


      IMPLICIT NONE

      DOUBLE PRECISION, DIMENSION(idnw:idnb)         :: xnz
      DOUBLE PRECISION                               :: atime
      DOUBLE PRECISION, DIMENSION(idnw:idnb, ncompo) :: xnc
      DOUBLE PRECISION, DIMENSION(idvs:idvb)         :: xvw_tot
      DOUBLE PRECISION, DIMENSION(idvw:idvb)         :: xvu_tot
      DOUBLE PRECISION, DIMENSION(nsolut)            :: wconc
      DOUBLE PRECISION, DIMENSION(nsolid)            :: wfflx
      DOUBLE PRECISION, DIMENSION(nsolid)            :: yc
      DOUBLE PRECISION, DIMENSION(ncompo)            :: flxval
      INTEGER                                        :: inode

      INTENT(IN)  :: atime, xnz, inode, xnc, yc,
     &               xvw_tot, xvu_tot, wconc, wfflx
      INTENT(OUT) :: flxval


      INTEGER :: inode_b, inode_a, idntm1
      INTEGER :: ivrtx
      DOUBLE PRECISION :: vxzd_ab
      DOUBLE PRECISION                    :: advw_ab
      DOUBLE PRECISION                    :: advu_ab
      DOUBLE PRECISION                    :: kappaw
      DOUBLE PRECISION, DIMENSION(nsolut) :: kappau
      DOUBLE PRECISION                    :: dcfw_ab
      DOUBLE PRECISION, DIMENSION(nsolut) :: dcfu_ab
      DOUBLE PRECISION, DIMENSION(ncompo) :: anc_a
      DOUBLE PRECISION                    :: Half_OnePlusKappaw
      DOUBLE PRECISION                    :: Half_OneMinusKappaw
      DOUBLE PRECISION, DIMENSION(nsolut) :: Half_OnePlusKappau
      DOUBLE PRECISION, DIMENSION(nsolut) :: Half_OneMinusKappau

      DOUBLE PRECISION, DIMENSION(nsolut) :: sconc

                                    ! First check validity of inode
      IF ((inode < idnw) .OR. (inode > idnb)) THEN

        WRITE(jp_stderr,*) 'inode value ', inode,
     &                              ' out of permitted range ',
     &                              idnw, '-', idnb
        WRITE(jp_stderr,*) 'Aborting!'
        CALL ABORT_MEDUSA()

      ENDIF


      ivrtx = inode


      SELECT CASE(inode)

      CASE(:idnt-2)                 ! - idnt > idnw in this case
                                    ! - the inode of interest lies within the DBL
        inode_b = inode + 1

        vxzd_ab = xnz(inode_b) - xnz(inode)     ! = h_j


        ! Solids
        ! ------

        flxval(jf_to_io(:)) = wfflx(:)


        ! Solutes
        ! -------

                                    ! Free diffusion coefficient in the DBL
        dcfu_ab(:) = dcf_molion(:)

        advu_ab = xvu_tot(idvs)     ! Water transport continuity across
                                    ! the DBL: notice that, whatever the
                                    ! number of grid nodes for the DBL,
                                    ! u = xu(idnt) throughout the DBL, since
                                    ! xu = xvphi*u in REACLAY, and
                                    ! u(idnt|-) = phi(idnt|+)*u(idnt|+).

        CALL UPWIND_KAPPA_SOLUTES(advu_ab, dcfu_ab, vxzd_ab, kappau)
        Half_OnePlusKappau(:)  = (1.0D+00 + kappau(:)) * 0.5D+00
        Half_OneMinusKappau(:) = (1.0D+00 - kappau(:)) * 0.5D+00

        flxval(jc_to_io(:)) =
     &       -dcfu_ab(:)
     &        * (  xnc(inode_b, jc_to_io(:))
     &           - xnc(inode , jc_to_io(:)) )/vxzd_ab
     &      + advu_ab
     &        * (  Half_OneMinusKappau(:)*xnc(inode_b, jc_to_io(:))
     &           + Half_OnePlusKappau(:) *xnc(inode , jc_to_io(:)))


      CASE(idnt-1)                  ! Fluxes across the SWI

        idntm1 = idnt - 1
        inode_b = idnt

        ! Solids
        ! ------

        flxval(jf_to_io(:)) = wfflx(:)


        ! Solutes
        ! -------

        CALL SWI_CONC(xnz, xnc, wconc, sconc)

        advu_ab = xvu_tot(idvs)      ! Water transport (see previous case)

        IF (advu_ab < 0.0D+00) THEN

          dcfu_ab(:) = xvphi(idvs)*dcf_molion(:)/xvtor2(idvs)
          vxzd_ab = xnz(idnt) - dp_swi_location

          CALL UPWIND_KAPPA_SOLUTES(advu_ab, dcfu_ab, vxzd_ab, kappau)

          flxval(jc_to_io(:)) =
     &       -dcfu_ab(:)
     &        * (xnc(idnt, jc_to_io(:)) - sconc(:)) / vxzd_ab
     &      + advu_ab
     &        * (      -kappau(:)        * xnc(idnt, jc_to_io(:))
     &           + (1.0D+00 + kappau(:)) * sconc(:))

        ELSEIF (advu_ab > 0.0D+00) THEN

          dcfu_ab(:) = dcf_molion(:)
          vxzd_ab = dp_swi_location - xnz(inode)

          CALL UPWIND_KAPPA_SOLUTES(advu_ab, dcfu_ab, vxzd_ab, kappau)

          flxval(jc_to_io(:)) =
     &       -dcfu_ab(:)
     &        * (sconc(:) - xnc(idntm1, jc_to_io(:))) / vxzd_ab
     &      + advu_ab
     &        * (       kappau(:)        * xnc(idntm1, jc_to_io(:))
     &           + (1.0D+00 - kappau(:)) * sconc(:))

        ELSE

          vxzd_ab = -xnz(inode)

          flxval(jc_to_io(:)) =
     &       -dcf_molion(:)
     &        * (sconc(:) - xnc(idntm1, jc_to_io(:))) / vxzd_ab


        ENDIF


      CASE(idnt:idnb-1)

        inode_b = inode + 1

        vxzd_ab = xnz(inode_b) - xnz(inode)     ! = h_j

        ! Solids
        ! ------

        IF (inode < idnz) THEN
                                    ! Interval [inode, inode_b],
                                    ! with inode_b <= idnz
          dcfw_ab = (1.0D+00-xvphi(ivrtx))*dcf_biotur(ivrtx)

          advw_ab = xvw_tot(inode)

        ELSEIF (inode == idnz) THEN
                                    ! Interval [idnz, idnz+1]: need to
                                    ! use phi_s w(idnz|+) =
                                    !     phi_s w(idnz|-) - alpha D^bt (dphi_s/dz)|-
                                    ! for "xw(inode)"
                                    ! Note that xw(inode) = phi_s w(inode|-)
                                    ! by definition in this model.
          dcfw_ab = 0.0D+00

          advw_ab = xvw_tot(inode)

        ELSE                      ! IF (inode_b > idnz+1)

          dcfw_ab = 0.0D+00
          advw_ab = xvw_tot(inode)

        ENDIF
                                    ! Possible improvements here:
                                    ! * include correction by derivatives
                                    !   diffusion coefficients and advection
                                    !   coefficients in inode_b and inode
                                    !   to get better estimate of dcfw_ab
                                    !   and advw_ab in (inode_b+inode)/2
                                    ! * Use Rankine-Hugeniot relationship
                                    !   for advw_ab (see Roe, 1981)

        CALL UPWIND_KAPPA_SOLIDS(advw_ab, dcfw_ab, vxzd_ab, kappaw)
        Half_OnePlusKappaw  = (1.0D+00 + kappaw) * 0.5D+00
        Half_OneMinusKappaw = (1.0D+00 - kappaw) * 0.5D+00

        flxval(jf_to_io(:)) =
     &       -dcfw_ab * (  xnc(inode_b, jf_to_io(:))
     &                   - xnc(inode , jf_to_io(:)) )/vxzd_ab
     &      + advw_ab * (  Half_OneMinusKappaw*xnc(inode_b, jf_to_io(:))
     &                   + Half_OnePlusKappaw *xnc(inode , jf_to_io(:)))


        ! Solutes
        ! -------

        dcfu_ab(:) = xvphi(ivrtx)*dcf_molion(:)/xvtor2(ivrtx)

        advu_ab = xvu_tot(ivrtx)

        CALL UPWIND_KAPPA_SOLUTES(advu_ab, dcfu_ab, vxzd_ab, kappau)
        Half_OnePlusKappau(:)  = (1.0D+00 + kappau(:)) * 0.5D+00
        Half_OneMinusKappau(:) = (1.0D+00 - kappau(:)) * 0.5D+00

        flxval(jc_to_io(:)) =
     &       -dcfu_ab(:)
     &        *(  xnc(inode_b, jc_to_io(:))
     &          - xnc(inode , jc_to_io(:)) )/vxzd_ab
     &      + advu_ab
     &        *(  Half_OneMinusKappau(:)*xnc(inode_b, jc_to_io(:))
     &          + Half_OnePlusKappau(:) *xnc(inode , jc_to_io(:)) )


      CASE(idnb)                    ! Fluxes across the REACLAY bottom

        inode_a = idnb - 1

        ! Solutes
        ! -------
                                    ! Set to zero
                                    ! - not required for node Z when Z /= B
                                    ! - actually zero at node B
        flxval(jc_to_io(:)) = 0.0D+00


        ! Solids
        ! ------

        advw_ab = xvw_tot(ivrtx)

        IF (advw_ab > 0.0D+00) THEN

          anc_a(:) = (xnc(inode,:) + xnc(inode_a,:))/2.0D+00

          flxval(jf_to_io(:)) = advw_ab * anc_a(jf_to_io(:))

        ELSE

          flxval(jf_to_io(:)) = advw_ab * yc(:)
                                    ! flxval(jf_to_io(:)) =  advw_ab * yc(:)
                                    ! is equivalent to using
                                    ! anc_a(jf_to_io(:)) = yc(:)
                                    ! anc_b(jf_to_io(:)) = yc(:)
                                    ! In this case
                                    ! flxval(jf_to_io(:)) =
                                    !   advw_ab * (  Half_OneMinusKappa * anc_b(jf_to_io(:))
                                    !              + Half_OnePlusKappa  * anc_a(jf_to_io(:)))
                                    ! simplifies to flxval(jf_to_io(:)) =  advw_ab * yc(:)
                                    ! as (Half_OneMinusKappa+Half_OnePlusKappa) = 1
        ENDIF


        RETURN                      ! All done

      END SELECT


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE TRANSPORT
!-----------------------------------------------------------------------
