!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


!-----------------------------------------------------------------------
      DOUBLE PRECISION FUNCTION DYNVIS(d, tc, s)
!-----------------------------------------------------------------------

      USE MOD_BASICDATA_MEDUSA,     ONLY: dp_atm, dp_gearth, dp_cm2

      IMPLICIT NONE

!     Calculates dynamic viscosity of water as a function of
!      * d (DBSL/[m]), to be converted to p (pressure/[kg-force/cm2])
!      * t (temperature/[deg C])
!      ° s (salinity/[])
!     dynvis will be in centipoise [cP, 1 cP = 10^{-2} g/cm/s]
!     Taken from Boudreau (1997, p. 94) Eq. 4.4., who refers to
!     Kukulka et al. (1987), who ill-cite the units used for
!     pressure in the original publication of Matthäus (1972):
!     Boudreau (1997) and Kukulka et al. (1987) ask for [bar],
!     but it should actually be [kg-force/cm2]
!     (Checked on 26JUL2017 in the original study).
!
!     Parametrization claimed valid for
!      * 0 °C <= tc <= 30 °C
!      * 1 kg-force/cm2 <= p <= 1000  kg-force/cm2
!      * 0 <= s <= 36
!
!     Data for pure water at 1 atm for -8 °C <= tc <= 150 °C
!     cited in Kestin et al. (1978) nevertheless indicate that
!     the parametrization used here provides acceptable results
!     for tc down to -1.9 °C (the approximate freezing temperature
!     of seawater with S=35). We assume that this also extends to
!     the salinity and pressure dependencies whose effect is weaker).

      DOUBLE PRECISION, INTENT(IN) :: d, tc, s

      DOUBLE PRECISION :: p

!     Convert d (depth/[m]) to p (total pressure/[kg-f/cm2]),
!     assuming a constant seawater density of 1025 kg/m3, and using
!       1 atm = 101325 Pa ( -> dp_atm from MOD_BASICDATA_MEDUSA)
!       1 kg-force/m2 = 9.80665 N/m2 = 9.80665 Pa
!                      ( -> dp_gearth from MOD_BASICDATA_MEDUSA)
!       1 = 10^{-4} m2/cm2 ( -> dp_cm2 from MOD_BASICDATA_MEDUSA)
      p = (dp_atm/dp_gearth + d*1025.0D+00)*dp_cm2

      dynvis = 1.7910D0
     &                + tc*(-6.144D-2 + tc*(1.4510D-3 - tc*1.6826D-5))
     &                +  p*(-1.529D-4 +  p*8.3885D-8) + s*2.4727D-3
     &                + tc*(p*(6.0574D-6 - p*2.6760D-9))
     &                + s*(tc*(4.8429D-5+tc*(-4.7172D-6+tc*7.5986D-8)))

      RETURN

!-----------------------------------------------------------------------
      END FUNCTION DYNVIS
!-----------------------------------------------------------------------
