!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


!-----------------------------------------------------------------------
      SUBROUTINE DERIVFORM_TRANSPORT
     &  (atime, xnz, inode, xnc, xvw_tot, xvu_tot,
     &   Deriv_flxval)
!-----------------------------------------------------------------------

! Returns the derivative of the flux value expression through
!  - the vertex below the node 'inode' if inode < idnb
!  - the virtual vertex at idnb if inode = idbn
! with respect to the concentrations at the nodes
!  - inode-1 (--> Deriv_flxval(:,-1)
!  - inode   (--> Deriv_flxval(:, 0)
!  - inode+1 (--> Deriv_flxval(:, 1)

      USE mod_defines_medusa,  ONLY: jp_stderr
      USE mod_execontrol_medusa, ONLY: ABORT_MEDUSA
      USE mod_gridparam,       ONLY: idnw, idnt, idnz, idnb,
     &                               idvw, idvs,       idvb
      USE mod_indexparam,      ONLY: nsolut, nsolid, ncompo,
     &                               jc_to_io, jf_to_io
      USE mod_milieucharas,    ONLY: xvphi, xvtor2, xphi


      IMPLICIT NONE


      DOUBLE PRECISION, DIMENSION(idnw:idnb)         :: xnz
      DOUBLE PRECISION                               :: atime
      INTEGER                                        :: inode
      DOUBLE PRECISION, DIMENSION(idvs:idvb)         :: xvw_tot
      DOUBLE PRECISION, DIMENSION(idvw:idvb)         :: xvu_tot

      DOUBLE PRECISION, DIMENSION(idnw:idnb, ncompo) :: xnc
      DOUBLE PRECISION, DIMENSION(ncompo, -1:1)      :: Deriv_flxval

      INTENT(IN) :: atime, xnz, inode, xnc, xvw_tot, xvu_tot
      INTENT(OUT) :: Deriv_flxval

      INTEGER :: inode_b, inode_a
      INTEGER :: ivrtx
      DOUBLE PRECISION :: vxzd_ab
      DOUBLE PRECISION                    :: advw_ab
      DOUBLE PRECISION                    :: advu_ab
      DOUBLE PRECISION                    :: kappaw
      DOUBLE PRECISION, DIMENSION(nsolut) :: kappau
      DOUBLE PRECISION                    :: dcfw_ab
      DOUBLE PRECISION, DIMENSION(nsolut) :: dcfu_ab
      DOUBLE PRECISION                    :: Half_OnePlusKappaw
      DOUBLE PRECISION                    :: Half_OneMinusKappaw
      DOUBLE PRECISION, DIMENSION(nsolut) :: Half_OnePlusKappau
      DOUBLE PRECISION, DIMENSION(nsolut) :: Half_OneMinusKappau

      DOUBLE PRECISION, DIMENSION(nsolut, 0:1) ::   Deriv_sconc

      DOUBLE PRECISION, DIMENSION(ncompo) :: dxc_flxval
      DOUBLE PRECISION, DIMENSION(ncompo) :: dxcb_flxval



                                    ! First check validity of tnode
      IF ((inode < idnw) .OR. (inode > idnb)) THEN

        WRITE(jp_stderr,*) 'inode value ', inode,
     &                              ' out of permitted range ',
     &                              idnw, '-', idnb
        WRITE(jp_stderr,*) 'Aborting!'
        CALL ABORT_MEDUSA()

      ENDIF


      ivrtx = inode


      SELECT CASE(inode)

      CASE(:idnt-2)                 ! - idnt > idnw in this case, since inode_b > inode_a
                                    ! - the tnode of interest lies within the DBL
        inode_b = inode + 1

        vxzd_ab = xnz(inode_b) - xnz(inode)     ! = h_j


        ! Solids
        ! ------

        ! Transport:
        ! flxval(jf_to_io(:)) = wfflx(:)
        dxc_flxval(jf_to_io(:)) = 0.0D+00
        dxcb_flxval(jf_to_io(:)) = 0.0D+00


        ! Solutes
        ! -------

                                    ! Free diffusion coefficient in the DBL
        dcfu_ab(:) = dcf_molion(:)

        advu_ab = xvu_tot(idvs)       ! Water transport continuity across
                                    ! the DBL: notice that, whatever the
                                    ! number of grid nodes for the DBL,
                                    ! u = xu(idnt) throughout the DBL, since
                                    ! xu = xvphi*u in REACLAY, and
                                    ! u(idnt|-) = phi(idnt|+)*u(idnt|+).

        CALL UPWIND_KAPPA_SOLUTES(advu_ab, dcfu_ab, vxzd_ab, kappau)
        Half_OnePlusKappau(:)  = (1.0D+00 + kappau(:)) * 0.5D+00
        Half_OneMinusKappau(:) = (1.0D+00 - kappau(:)) * 0.5D+00

        ! Transport:
        !  flxval(jc_to_io(:)) =
        ! &  -dcfu_ab(:)
        ! &   *(  xnc(inode_b, jc_to_io(:))
        ! &     - xnc(inode , jc_to_io(:)) )/vxzd_ab
        ! & + advu_ab
        ! &   *(  Half_OneMinusKappau(:)*xnc(inode_b, jc_to_io(:))
        ! &     + Half_OnePlusKappau(:) *xnc(inode , jc_to_io(:)))

        dxc_flxval(jc_to_io(:)) =  dcfu_ab(:)/vxzd_ab
     &                              + advu_ab*Half_OnePlusKappau(:)
        dxcb_flxval(jc_to_io(:)) = -dcfu_ab(:)/vxzd_ab
     &                              + advu_ab*Half_OneMinusKappau(:)


      CASE(idnt-1)                  ! Fluxes accross the SWI


        inode_b = idnt

        ! Solids
        ! ------

        ! Transport:
        !  flxval(jf_to_io(:)) = wfflx(:)
        dxc_flxval(jf_to_io(:))  = 0.0D+00
        dxcb_flxval(jf_to_io(:)) = 0.0D+00


        ! Solutes
        ! -------

        ! Transport:
        !  CALL SWI_CONC(xnz, xnc, wconc, wfflx, xvw_tot, sconc)
        CALL DERIVFORM_SWI_CONC(xnz, xnc, Deriv_sconc)

        advu_ab = xvu_tot(idvs)      ! Water transport (see previous case)

        IF (advu_ab < 0.0D+00) THEN

          dcfu_ab(:) = xvphi(idvs)*dcf_molion(:)/xvtor2(idvs)
          vxzd_ab = xnz(idnt)

          CALL UPWIND_KAPPA_SOLUTES(advu_ab, dcfu_ab, vxzd_ab, kappau)

          ! Transport:
          !  flxval(jc_to_io(:)) =
          ! &    -dcfu_ab(:)
          ! &     * (xnc(idnt, jc_to_io(:)) - sconc(:)) / vxzd_ab
          ! &   + advu_ab
          ! &     * (      -kappau(:)        * xnc(idnt, jc_to_io(:))
          ! &        + (1.0D+00 + kappau(:)) * sconc(:)))

          dxc_flxval(jc_to_io(:)) =
     &      -dcfu_ab(:) * (-Deriv_sconc(:, 0)) / vxzd_ab
     &      + advu_ab * (1.0D+00+kappau(:)) * Deriv_sconc(:, 0)

          dxcb_flxval(jc_to_io(:)) =
     &       -dcfu_ab(:) * (1.0D+00 - Deriv_sconc(:, 1)) / vxzd_ab
     &      + advu_ab
     &          * (-kappau(:)
     &             + (1.0D+00+kappau(:)) * Deriv_sconc(:, 1))

        ELSEIF (advu_ab > 0.0D+00) THEN

          dcfu_ab(:) = dcf_molion(:)
          vxzd_ab = -xnz(inode)

          CALL UPWIND_KAPPA_SOLUTES(advu_ab, dcfu_ab, vxzd_ab, kappau)

          ! Transport:
          !  flxval(jc_to_io(:)) =
          ! &  -dcfu_ab(:)
          ! &   * (sconc(:) - xnc(idntm1, jc_to_io(:))) / vxzd_ab
          ! &  + advu_ab
          ! &    * (       kappau(:)        * xnc(idntm1, jc_to_io(:))
          ! &       + (1.0D+00 - kappau(:)) * sconc(:))

          dxc_flxval(jc_to_io(:)) =
     &       -dcfu_ab(:) * (Deriv_sconc(:, 0) - 1.0D+00) / vxzd_ab
     &      + advu_ab
     &        * ( kappau(:) + (1.0D+00-kappau(:)) * Deriv_sconc(:, 0) )

          dxcb_flxval(jc_to_io(:)) =
     &       -dcfu_ab(:) * Deriv_sconc(:, 1) / vxzd_ab
     &      + advu_ab * (1.0D+00-kappau(:)) * Deriv_sconc(: , 1)

        ELSE

          vxzd_ab = -xnz(inode)

          ! Transport:
          !  flxval(jc_to_io(:)) =
          ! &   -dcf_molion(:)
          ! &    * (sconc(:) - xnc(idntm1, jc_to_io(:))) / vxzd_ab

          dxc_flxval(jc_to_io(:)) =
     &      -dcf_molion(:) * (Deriv_sconc(:, 0) - 1.0D+00) / vxzd_ab

          dxcb_flxval(jc_to_io(:)) =
     &      -dcf_molion(:) * Deriv_sconc(:, 1) / vxzd_ab

        ENDIF


      CASE(idnt:idnb-1)

        inode_b = inode + 1

        vxzd_ab = xnz(inode_b) - xnz(inode)        ! = h_j


        ! Solids
        ! ------

        IF (inode < idnz) THEN
                                    ! Interval [inode, inode_b],
                                    ! with inode_b <= idnz
          dcfw_ab = (1.0D+00-xvphi(ivrtx))*dcf_biotur(ivrtx)

          advw_ab = xvw_tot(inode)

        ELSEIF (inode == idnz) THEN
                                    ! Interval [idnz, idnz+1]: need to
                                    ! use phi_s w(idnz|+) =
                                    !     phi_s w(idnz|-) - alpha D^bt (dphi_s/dz)|-
                                    ! for "xw(inode)"
                                    ! Note that xw(inode) = phi_s w(inode|-)
                                    ! by definition in this model.
          dcfw_ab = 0.0D+00

          advw_ab = xvw_tot(inode)

        ELSE                        ! IF (inode_b > idnz+1)

          dcfw_ab = 0.0D+00
          advw_ab = xvw_tot(inode)

        ENDIF
                                    ! Possible improvements here:
                                    ! * include correction by derivatives
                                    !   diffusion coefficients and advection
                                    !   coefficients in inode_b and inode_a
                                    !   to get better estimate of dcfw_ab
                                    !   and advw_ab in (inode_b+inode_a)/2
                                    ! * Use Rankine-Hugeniot relationship
                                    !   for advw_ab (see Roe, 1981)

        CALL UPWIND_KAPPA_SOLIDS(advw_ab, dcfw_ab, vxzd_ab, kappaw)
        Half_OnePlusKappaw  = (1.0D+00 + kappaw) * 0.5D+00
        Half_OneMinusKappaw = (1.0D+00 - kappaw) * 0.5D+00

        ! Transport:
        !  flxval(jf_to_io(:)) =
        ! &   -dcfw_ab*(  xnc(inode_b, jf_to_io(:))
        ! &             - xnc(inode  , jf_to_io(:)) )/vxzd_ab
        ! &  + advw_ab*(  Half_OneMinusKappaw*xnc(inode_b, jf_to_io(:))
        ! &             + Half_OnePlusKappaw *xnc(inode  , jf_to_io(:)))

        dxc_flxval(jf_to_io(:))  =  dcfw_ab/vxzd_ab
     &                              + advw_ab*Half_OnePlusKappaw
        dxcb_flxval(jf_to_io(:)) = -dcfw_ab/vxzd_ab
     &                              + advw_ab*Half_OneMinusKappaw


        ! Solutes
        ! -------

        dcfu_ab(:) = xvphi(ivrtx)*dcf_molion(:)/xvtor2(ivrtx)

        advu_ab = xvu_tot(ivrtx)    ! ivrtx = inode here

        CALL UPWIND_KAPPA_SOLUTES(advu_ab, dcfu_ab, vxzd_ab, kappau)
        Half_OnePlusKappau(:)  = (1.0D+00 + kappau(:)) * 0.5D+00
        Half_OneMinusKappau(:) = (1.0D+00 - kappau(:)) * 0.5D+00

        ! Transport:
        !  flxval(jc_to_io(:)) =
        ! &  -dcfu_ab(:)
        ! &   * (  xnc(inode_b, jc_to_io(:))
        ! &      - xnc(inode  , jc_to_io(:)) )/vxzd_ab
        ! & + advu_ab
        ! &   * (  Half_OneMinusKappau(:)*xnc(inode_b, jc_to_io(:))
        ! &      + Half_OnePlusKappau(:) *xnc(inode  , jc_to_io(:)) )

        dxc_flxval(jc_to_io(:))  =  dcfu_ab(:)/vxzd_ab
     &                              + advu_ab*Half_OnePlusKappau(:)
        dxcb_flxval(jc_to_io(:)) = -dcfu_ab(:)/vxzd_ab
     &                              + advu_ab*Half_OneMinusKappau(:)


      CASE(idnb)

        inode_a = idnb - 1

        ! Solutes
        ! -------
                                    ! Set to zero
                                    ! - not required for node Z when Z /= B)
                                    ! - actually zero at node B
        ! Transport:
        !  flxval(jc_to_io(:)) = 0.0D+00
        Deriv_flxval(jc_to_io(:), :) = 0.0D+00


        ! Solids
        ! ------

        advw_ab = xvw_tot(ivrtx)

        IF (advw_ab > 0.0D+00) THEN

          ! Transport:
          !  anc_a(:) = (xnc(inode,:) + xnc(inode_a,:))/2.0D+00
          !  flxval(jf_to_io(:)) = advw_ab * anc_a(jf_to_io(:))
          Deriv_flxval(jf_to_io(:), -1) = advw_ab/2.0D+00
          Deriv_flxval(jf_to_io(:),  0) = advw_ab/2.0D+00
          Deriv_flxval(jf_to_io(:),  1) = 0.0D+00

        ELSE

          ! Transport:
          !  flxval(jf_to_io(:)) =  advw_ab * yc(:)
          Deriv_flxval(jf_to_io(:), :) = 0.0D+00

        ENDIF


        RETURN                      ! All done


      END SELECT


      Deriv_flxval(:, -1) = 0.0D+00
      Deriv_flxval(:,  0) = dxc_flxval(:)
      Deriv_flxval(:,  1) = dxcb_flxval(:)


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE DERIVFORM_TRANSPORT
!-----------------------------------------------------------------------
