!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#ifdef BIRRIC_CUSTOM_DECLARATIONS

                                    ! Additional parameters and configuration
                                    ! (adjustable scales etc.) for the
                                    ! Muds D_I profile.

      DOUBLE PRECISION, PARAMETER :: dp_bi_muds_scale
     &                              = 1.8D+00*dp_cm ! = 1.8 cm


                                    ! Actually used parameter values for
                                    ! the Muds D_I profile.
      DOUBLE PRECISION, SAVE :: da_bi_muds_scale
#else
!-----------------------------------------------------------------------
      SUBROUTINE BIRRIC_CUSTOM(xzdn, xcompo, wconc, wfflx)
!-----------------------------------------------------------------------

! Bioirrigation coefficient as in Muds (Archer et al., 2002,
! Global. Biogeochem. Cycles 16, 21pp., doi:10.1029/2000GB001288)

!--------------
! Declarations
!--------------

      USE mod_basicdata_medusa,     ONLY: dp_cm, dp_cm2, dp_pi
      USE mod_gridparam             ONLY: idnw, idnt, idnb, idnz
      USE mod_indexparam            ONLY: ncompo, nsolut, nsolid,
     &                                    ic_o2, if_om
      USE mod_materialcharas,       ONLY: mol_om

#ifdef DEBUG
      USE mod_defines_medusa,       ONLY: jp_stddbg
#endif


      IMPLICIT NONE


!-----------------------
! Variable declarations
!-----------------------

!- - - - - - - - - - - - -  - - - - - - -
! Variables in subroutine call arguments
!- - - - - - - - - - - - -- - - - - - - -

      DOUBLE PRECISION, DIMENSION (idnw:idnb)           :: xzdn
      DOUBLE PRECISION, DIMENSION (idnw:idnb, ncompo)   :: xcompo
      DOUBLE PRECISION, DIMENSION (nsolut)              :: wconc
      DOUBLE PRECISION, DIMENSION (nsolid)              :: wfflx


!- - - - - - - - -
! Local variables
!- - - - - - - - -

      INTEGER          :: i

      DOUBLE PRECISION :: da_rcf_bioirr_0
      DOUBLE PRECISION :: wfflx_oc


!- - - - - - - - - - -
! End of declarations
!- - - - - - - - - - -


!----------------------------------------------------------------------
! Subroutine Start
!----------------------------------------------------------------------

#ifdef DEBUG
#ifdef DEBUG_BIRRIC
#ifdef DEBUG_BIRRIC_ENTRY_EXIT
      WRITE(jp_stddbg,*) '[BIRRIC (Muds)]: Starting'
#endif
#endif
#endif

                                    !  - [wfflx_oc] = [umolC/cm2/yr]
                                    !  - [dp_cm2] = [m2/cm]
                                    !  - 10.0D-03 mol/m3 = 10 umol/l
                                    !  - [da_rcf_bioirr_0] = [d^-1] originally
      wfflx_oc = wfflx(if_om)*om_c/(1.0D-06*mol_om/dp_cm2)

      da_rcf_bioirr_0 =
     &    11.0D+00 * (ATAN(5.0D+00*(wfflx_oc-400.0D+00)/400.0D+00)/dp_pi
     &                + 0.5D+00)
     &  -  0.9D+00
     &  + 20.0D+00 * wconc(ic_o2)/(wconc(ic_o2)+10.0D-03)
     &             * EXP(-wconc(ic_o2)/10.0D-03)
     &             * wfflx_oc / (wfflx_oc + 30.0D+00)

                                    ! Change units (to yr^-1) and make
                                    ! sure that values are not negative
      da_rcf_bioirr_0 = MAX(da_rcf_bioirr_0, 0.0D+00)/dp_day

      DO i = idnt, idnz
        rcf_bioirr(i) =
     &     da_rcf_bioirr_0 * EXP(-(xzdn(i)/da_bi_muds_scale)**2)
      ENDDO


#ifdef DEBUG
#ifdef DEBUG_BIRRIC
#ifdef DEBUG_BIRRIC_ENTRY_EXIT
      WRITE(jp_stddbg,*) '[BIRRIC (Muds)]: Exiting'
#endif
#endif
#endif


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE BIRRIC_CUSTOM
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE BIRRIC_CUSTOM_SETUP(k_stage, iu_cfg)
!-----------------------------------------------------------------------

!--------------
! Declarations
!--------------


      USE mod_defines_medusa,       ONLY: jp_stderr, jp_stdlog
      USE mod_execontrol_medusa,    ONLY: ABORT_MEDUSA

#ifdef ALLOW_MPI
      USE mod_execontrol_medusa,    ONLY: MEDEXE_MPI_COMM,
     &                              MEDEXE_MPI_COMM_RANK,
     &                              jp_exeproc_root
      USE mpi, ONLY: MPI_INTEGER, MPI_DOUBLE_PRECISION
#endif


      IMPLICIT NONE


!-----------------------
! Variable declarations
!-----------------------

!- - - - - - - - - - - - -  - - - - - - -
! Variables in subroutine call arguments
!- - - - - - - - - - - - -- - - - - - - -

      INTEGER, INTENT(IN) :: k_stage
      INTEGER, INTENT(IN) :: iu_cfg


!- - - - - - - - - - - - - -  - - - - - - - -
! General (global) parameters and definitions
!- - - - - - - - - - - - - -- - - - - - - - -

      ! None


!- - - - - - - - -
! Local variables
!- - - - - - - - -

      DOUBLE PRECISION, SAVE  :: di_scale

      NAMELIST /nml_bioirr_muds_orig/  di_scale


!- - - - - - - - - - -
! End of declarations
!- - - - - - - - - - -


!----------------------------------------------------------------------
! Subroutine Start
!----------------------------------------------------------------------


      SELECT CASE(k_stage)
      CASE(1)
                                    ! Pre-set the common default values
        di_scale    = dp_bi_muds_scale

        READ(iu_cfg, NML=nml_bioirr_muds_orig)


        IF (di_scale <= 0.0D+00) THEN
          ! scale must be > 0!
          CALL ABORT_MEDUSA()
        ENDIF


      CASE(2)

#ifdef ALLOW_MPI
                                    ! Broadcast the configuration data:
                                    !  - scale (all)
        CALL MPI_BCAST(di_scale, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
#endif

        WRITE(jp_stdlog, '(" - for bioirrigion")')
        WRITE(jp_stdlog, '("   * profile ID: ", I0)')
     &                              jselect_bioirr_profile


        da_bi_muds_scale    = di_scale

        WRITE(jp_stdlog, '("   * scale: ", E9.3)')
     &                              da_bi_muds_scale

        WRITE(jp_stdlog, '()')


      CASE DEFAULT
                                    ! Unknown stage
        CALL ABORT_MEDUSA()


      END SELECT


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE BIRRIC_CUSTOM_SETUP
!-----------------------------------------------------------------------
#endif
