!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#ifdef BDIFFC_CUSTOM_DECLARATIONS

                                    ! Numbering of the currently
                                    ! available types for the OMEXDIA
                                    ! biodiffusion profiles (DO NOT CHANGE!)
      INTEGER, PARAMETER :: jp_omexdia_dborig  = 0 ! D_B0 = f(w_0) [Default]
      INTEGER, PARAMETER :: jp_omexdia_dbconst = 1 ! D_B0 = constant


                                    ! Additional parameters and configuration
                                    ! (adjustable scales etc.) for the OMEXDIA
                                    ! D_B profile.
      DOUBLE PRECISION, PARAMETER
     &  :: dp_bt_omexdia_dcf_0     = 0.15D+00*dp_cm2_p_yr ! = 0.15 cm^2/yr
      DOUBLE PRECISION, PARAMETER
     &  :: dp_bt_omexdia_topdepth  = 5.0D+00*dp_cm ! = 5 cm
      DOUBLE PRECISION, PARAMETER
     &  :: dp_bt_omexdia_scale     = 1.0D+00*dp_cm ! = 1 cm


      INTEGER, SAVE :: jselect_omexdia_subtype = jp_omexdia_dborig


                                    ! Actually used parameter values for
                                    ! the OMEXDIA D_B profile.
      DOUBLE PRECISION, SAVE :: da_bt_omexdia_dcf_0
      DOUBLE PRECISION, SAVE :: da_bt_omexdia_topdepth
      DOUBLE PRECISION, SAVE :: da_bt_omexdia_scale

#else
!-----------------------------------------------------------------------
      SUBROUTINE BDIFFC_CUSTOM(xzdn, xzdv, xcompo, wconc, wfflx)
!-----------------------------------------------------------------------

! Biodiffusion coefficient as in OMEXDIA
! (Soetaert et al., Geochim. Cosmochim. Acta, 1996)

!--------------
! Declarations
!--------------

      USE mod_basicdata_medusa,     ONLY: dp_cm, dp_cm_p_yr, dp_cm2_p_yr
      USE mod_gridparam,            ONLY: idnw, idnt, idnz, idnb,
     &                                    idvw, idvs, idvaz, idvb
      USE mod_indexparam,           ONLY: ncompo, nsolut, nsolid
      USE mod_materialcharas,       ONLY: apsv
      USE mod_milieucharas,         ONLY: xphi

#ifdef DEBUG
      USE mod_defines_medusa,       ONLY: jp_stddbg
#endif


      IMPLICIT NONE


!-----------------------
! Variable declarations
!-----------------------

!- - - - - - - - - - - - -  - - - - - - -
! Variables in subroutine call arguments
!- - - - - - - - - - - - -- - - - - - - -

      DOUBLE PRECISION, DIMENSION(idnw:idnb)           :: xzdn
      DOUBLE PRECISION, DIMENSION(idvw:idvb)           :: xzdv
      DOUBLE PRECISION, DIMENSION(idnw:idnb, ncompo)   :: xcompo
      DOUBLE PRECISION, DIMENSION(nsolut)              :: wconc
      DOUBLE PRECISION, DIMENSION(nsolid)              :: wfflx

      INTENT(IN) :: xzdn, xzdv, xcompo, wconc, wfflx

!- - - - - - - - - - - - - -  - - - - - - - -
! General (global) parameters and definitions
!- - - - - - - - - - - - - -- - - - - - - - -

      ! None

!- - - - - - - - -
! Local variables
!- - - - - - - - -

                                    ! D_B0, the fundamental biodiffusion
                                    ! coefficient value
      DOUBLE PRECISION :: da_dcf_biotur_0
      DOUBLE PRECISION :: w_0
      INTEGER          :: i

      DOUBLE PRECISION :: azdnz


!- - - - - - - - - - -
! End of declarations
!- - - - - - - - - - -


!----------------------------------------------------------------------
! Subroutine Start
!----------------------------------------------------------------------

#ifdef DEBUG
#ifdef DEBUG_BDIFFC
#ifdef DEBUG_BDIFFC_ENTRY_EXIT
      WRITE(jp_stddbg,*) '[BDIFFC (OMEXDIA)]: Starting'
#endif
#endif
#endif


      IF (.NOT. l_setupdone) CALL SETUP_TRANSPORT

      azdnz = xzdn(idnz)

      !---------------------------------------------
      ! Biodiffusion coefficient and its derivative
      !---------------------------------------------

      SELECT CASE(jselect_omexdia_subtype)
      CASE(jp_omexdia_dborig)
                                    ! [w_0] = [m/yr]
        w_0 = SUM(wfflx(:)*apsv(:))/(1.0D+00 - xphi(idnt))

                                    ! Db_0/[cm2/d] = 15/365*(w/[cm/yr])**0.6
                                    ! Db_0/[cm2/yr] = 15*(w/[cm/yr])**0.6
        da_dcf_biotur_0
     &    = 0.15D+02*(w_0/dp_cm_p_yr)**0.6D+00 * dp_cm2_p_yr

      CASE(jp_omexdia_dbconst)
        da_dcf_biotur_0 = da_bt_omexdia_dcf_0

      END SELECT


      DO i = idvs, idvaz
        IF (xzdv(i) <= da_bt_omexdia_topdepth) THEN
          dcf_biotur(i) = da_dcf_biotur_0
          dx_dcf_biotur(i) = 0.0D+00
        ELSE
          dcf_biotur(i) = da_dcf_biotur_0
     &                    * EXP(-(xzdv(i)-da_bt_omexdia_topdepth)
     &                           /da_bt_omexdia_scale)
          dx_dcf_biotur(i) = -dcf_biotur(i)/da_bt_omexdia_scale
        ENDIF
      ENDDO

      IF (idnb == idnz) THEN      ! idnz = idvaz+1 = idnb = idvb in this case
        IF (azdnz <= da_bt_omexdia_topdepth) THEN
             dcf_biotur(idnz) = da_dcf_biotur_0
          dx_dcf_biotur(idnz) = 0.0D+00
        ELSE
             dcf_biotur(idnz) =  da_dcf_biotur_0
     &                    * EXP(-(azdnz-da_bt_omexdia_topdepth)
     &                           /da_bt_omexdia_scale)
          dx_dcf_biotur(idnz) = -dcf_biotur(idnz)/da_bt_omexdia_scale
        ENDIF
      ELSE
           dcf_biotur(idnz) =  0.0D+00
        dx_dcf_biotur(idnz) =  0.0D+00
      ENDIF

#ifdef DEBUG
#ifdef DEBUG_BDIFFC
#ifdef DEBUG_BDIFFC_ENTRY_EXIT
      WRITE(jp_stddbg,*) '[BDIFFC (OMEXDIA)]: Exiting'
#endif
#endif
#endif

      RETURN      


!-----------------------------------------------------------------------
      END SUBROUTINE BDIFFC_CUSTOM
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE BDIFFC_CUSTOM_SETUP(k_stage, iu_cfg)
!-----------------------------------------------------------------------

!--------------
! Declarations
!--------------


      USE mod_defines_medusa,       ONLY: jp_stderr, jp_stdlog
#ifdef DEBUG
      USE mod_defines_medusa,       ONLY: jp_stddbg
#endif
      USE mod_execontrol_medusa,    ONLY: ABORT_MEDUSA

#ifdef ALLOW_MPI
      USE mod_execontrol_medusa,    ONLY: MEDEXE_MPI_COMM,
     &                              MEDEXE_MPI_COMM_RANK,
     &                              jp_exeproc_root
      USE mpi, ONLY: MPI_INTEGER, MPI_DOUBLE_PRECISION
#endif


      IMPLICIT NONE


!-----------------------
! Variable declarations
!-----------------------

!- - - - - - - - - - - - -  - - - - - - -
! Variables in subroutine call arguments
!- - - - - - - - - - - - -- - - - - - - -

      INTEGER, INTENT(IN) :: k_stage
      INTEGER, INTENT(IN) :: iu_cfg


!- - - - - - - - - - - - - -  - - - - - - - -
! General (global) parameters and definitions
!- - - - - - - - - - - - - -- - - - - - - - -

      ! None


!- - - - - - - - -
! Local variables
!- - - - - - - - -


      CHARACTER(LEN=15) :: ctype_db0

      NAMELIST /nml_biodif_omexdia_options/ ctype_db0


      DOUBLE PRECISION, SAVE  :: db_0
      DOUBLE PRECISION, SAVE  :: db_topdepth
      DOUBLE PRECISION, SAVE  :: db_scale

      NAMELIST /nml_biodif_omexdia_const/ db_0, db_topdepth, db_scale
      NAMELIST /nml_biodif_omexdia_orig/        db_topdepth, db_scale


      CHARACTER(LEN=*), PARAMETER :: cfmt_info_a
     &  = '("[MOD_TRANSPORT/bdiffc_omexdia[BDIFFC_CUSTOM_SETUP]]' //
     &    ' info: ", A)'

      CHARACTER(LEN=*), PARAMETER :: cfmt_err_a
     &  = '("[MOD_TRANSPORT/bdiffc_omexdia[BDIFFC_CUSTOM_SETUP]]' //
     &    ' error: ", A)'

#ifdef DEBUG
      CHARACTER(LEN=*), PARAMETER :: cfmt_dbg_a
     &  = '("[MOD_TRANSPORT/bdiffc_omexdia[BDIFFC_CUSTOM_SETUP]]' //
     &    ' debug: ", A)'
#endif


!- - - - - - - - - - -
! End of declarations
!- - - - - - - - - - -


!----------------------------------------------------------------------
! Subroutine Start
!----------------------------------------------------------------------


      SELECT CASE(k_stage)
      CASE(1)

        ctype_db0 = '?'

        READ(iu_cfg, NML=nml_biodif_omexdia_options)

#ifdef DEBUG
#ifdef DEBUG_BDIFFC
        WRITE(jp_stddbg, cfmt_dbg_a) 'OMEXDIA options requested:'

        IF (ctype_db0 == '?') THEN
          WRITE(jp_stddbg, '(" - ctype_db0  = ", A)')
     &                              '""original""'
        ELSE
          WRITE(jp_stddbg, '(" - ctype_db0  = """, A, """")')
     &                              TRIM(ctype_db0)
        ENDIF
        WRITE(jp_stddbg, '()')
#endif
#endif


                                    ! Pre-set the common default values
        jselect_omexdia_subtype = jp_omexdia_dborig
        db_topdepth = dp_bt_omexdia_topdepth
        db_scale    = dp_bt_omexdia_scale


        SELECT CASE(ctype_db0)
        CASE('?')
          CONTINUE

        CASE('const')
          jselect_omexdia_subtype = jp_omexdia_dbconst

                                    ! Pre-set the additional default value
          db_0 = dp_bt_omexdia_dcf_0

          READ(iu_cfg, NML=nml_biodif_omexdia_const)

          IF (db_0 < 0.0D+00) THEN
            ! D_B0 must be >= 0!
            ! Print out jselect_omexdia_subtype
            CALL ABORT_MEDUSA()
          ENDIF

        CASE('original')
          jselect_omexdia_subtype = jp_omexdia_dborig

          READ(iu_cfg, NML=nml_biodif_omexdia_orig)

        CASE DEFAULT
          ! Unknow profile name
          CALL ABORT_MEDUSA()

        END SELECT

        IF (db_topdepth < 0.0D+00) THEN
          ! scale must be > =0!
          ! Print out jselect_biodif_profile
          CALL ABORT_MEDUSA()
        ENDIF

        IF (db_scale <= 0.0D+00) THEN
          ! scale must be > 0!
          ! Print out jselect_biodif_profile
          CALL ABORT_MEDUSA()
        ENDIF


      CASE(2)

#ifdef ALLOW_MPI
                                    ! Broadcast the configuration data:
                                    !  - biodiffusion: profile type
        CALL MPI_BCAST(jselect_omexdia_subtype, 1, MPI_INTEGER,
     &                              jp_exeproc_root, i_mycomm, istatus)

        SELECT CASE(jselect_omexdia_subtype)
          CASE(jp_omexdia_dbconst)  !  - D_B0 (const)
          CALL MPI_BCAST(db_0, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
        END SELECT
                                    !  - top depth (all)
        CALL MPI_BCAST(db_topdepth, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)

                                    !  - scale (all)
        CALL MPI_BCAST(db_scale, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
#endif


        WRITE(jp_stdlog, '(" - for biodiffusion")')
        WRITE(jp_stdlog, '("   * profile ID: ", I0, "[", I0, "]")')
     &                              jselect_biodif_profile,
     &                              jselect_omexdia_subtype


        SELECT CASE(jselect_omexdia_subtype)
        CASE(jp_omexdia_dbconst)      !  - D_B0 (const)
          da_bt_omexdia_dcf_0 = db_0
          WRITE(jp_stdlog, '("   * D_B0: ", E9.3)') da_bt_omexdia_dcf_0
        END SELECT


        da_bt_omexdia_topdepth = db_topdepth
        da_bt_omexdia_scale    = db_scale

        WRITE(jp_stdlog, '("   * top depth: ", E9.3)')
     &                              da_bt_omexdia_topdepth
        WRITE(jp_stdlog, '("   * scale: ", E9.3)')
     &                              da_bt_omexdia_scale

        WRITE(jp_stdlog, '()')


      CASE DEFAULT

        ! Unknown stage
        CALL ABORT_MEDUSA()


      END SELECT


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE BDIFFC_CUSTOM_SETUP
!-----------------------------------------------------------------------
#endif
