!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#ifdef BDIFFC_CUSTOM_DECLARATIONS

                                    ! Additional parameters and configuration
                                    ! (adjustable scales etc.) for the
                                    ! Muds D_B profile.
      DOUBLE PRECISION, PARAMETER :: dp_bt_muds_scale = 8.0D+00*dp_cm     ! = 8 cm


                                    ! Actually used parameter values for
                                    ! the Muds D_B profile.
      DOUBLE PRECISION, SAVE :: da_bt_muds_scale

#else
!-----------------------------------------------------------------------
      SUBROUTINE BDIFFC_CUSTOM(xzdn, xzdv, xcompo, wconc, wfflx)
!-----------------------------------------------------------------------

! Biodiffusion coefficient as in Muds (Archer et al., Global. Biogeochem.
! Cycles, 2002, 16, 21pp, doi:10.1029/2000GB001288)

!--------------
! Declarations
!--------------

      USE mod_basicdata_medusa,     ONLY: dp_cm, dp_cm2, dp_cm2_p_yr
      USE mod_gridparam,            ONLY: idnw, idnt, idnz,  idnb,
     &                                    idvw, idvs, idvaz, idvb
      USE mod_indexparam,           ONLY: ncompo, nsolut, nsolid,
     &                                    ic_o2, if_om
      USE mod_materialcharas,       ONLY: mol_om

#ifdef DEBUG
      USE mod_defines_medusa,       ONLY: jp_stddbg
#endif


      IMPLICIT NONE


!-----------------------
! Variable declarations
!-----------------------

!- - - - - - - - - - - - -  - - - - - - -
! Variables in subroutine call arguments
!- - - - - - - - - - - - -- - - - - - - -

      DOUBLE PRECISION, DIMENSION(idnw:idnb)           :: xzdn
      DOUBLE PRECISION, DIMENSION(idvw:idvb)           :: xzdv
      DOUBLE PRECISION, DIMENSION(idnw:idnb, ncompo)   :: xcompo
      DOUBLE PRECISION, DIMENSION(nsolut)              :: wconc
      DOUBLE PRECISION, DIMENSION(nsolid)              :: wfflx

      INTENT(IN) :: xzdn, xzdv, xcompo, wconc, wfflx

!- - - - - - - - -
! Local variables
!- - - - - - - - -
                                    ! D_B0, the fundamental biodiffusion
                                    ! coefficient value
      DOUBLE PRECISION :: da_dcf_biotur_0
      DOUBLE PRECISION :: wfflx_oc
      INTEGER          :: i

      DOUBLE PRECISION :: azdnz     ! azdnz = xzdn(idnz)


!- - - - - - - - - - -
! End of declarations
!- - - - - - - - - - -


!----------------------------------------------------------------------
! Subroutine Start
!----------------------------------------------------------------------

#ifdef DEBUG
#ifdef DEBUG_BDIFFC
#ifdef DEBUG_BDIFFC_ENTRY_EXIT
      WRITE(jp_stddbg,*) '[BDIFFC (Muds)]: Starting'
#endif
#endif
#endif


      IF (.NOT. l_setupdone) CALL SETUP_TRANSPORT

      azdnz = xzdn(idnz)

      !---------------------------------------------
      ! Biodiffusion coefficient and its derivative
      !---------------------------------------------

                                    !  - [wfflx_oc] = [umolC/cm2/yr]
                                    !  - [dp_cm2] = [cm2/m2]
                                    !  - 20.0D-03 mol/m3 = 20 umol/l

      wfflx_oc = wfflx(if_om)*om_c/(1.0D-06*mol_om/dp_cm2)

      da_dcf_biotur_0 = 0.0232D+00 * wfflx_oc**0.85D+00 * dp_cm2_p_yr

      DO i = idvs, idvaz
        dcf_biotur(i) =
     &      da_dcf_biotur_0 * EXP(-(xzdv(i)/da_bt_muds_scale)**2)
     &                      * wconc(ic_o2)/(wconc(ic_o2) + 20.0D-03)
        dx_dcf_biotur(i) =
     &      -dcf_biotur(i) * 2.0D+00 * xzdv(i)/da_bt_muds_scale**2
      ENDDO

      IF (idnb == idnz) THEN      ! idnz = idvaz+1 = idnb = idvb in this case
        dcf_biotur(idnz) =  
     &      da_dcf_biotur_0 * EXP(-(azdnz/da_bt_muds_scale)**2)
     &                      * wconc(ic_o2)/(wconc(ic_o2) + 20.0D-03)
        dx_dcf_biotur(idnz) =
     &      -dcf_biotur(idnz) * 2.0D+00 * azdnz/da_bt_muds_scale**2
      ELSE
           dcf_biotur(idnz) =  0.0D+00
        dx_dcf_biotur(idnz) =  0.0D+00
      ENDIF

#ifdef DEBUG
#ifdef DEBUG_BDIFFC
#ifdef DEBUG_BDIFFC_ENTRY_EXIT
      WRITE(jp_stddbg,*) '[BDIFFC (Muds)]: Exiting'
#endif
#endif
#endif

      RETURN      


!-----------------------------------------------------------------------
      END SUBROUTINE BDIFFC_CUSTOM
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE BDIFFC_CUSTOM_SETUP(k_stage, iu_cfg)
!-----------------------------------------------------------------------

!--------------
! Declarations
!--------------


      USE mod_defines_medusa,       ONLY: jp_stderr, jp_stdlog
      USE mod_execontrol_medusa,    ONLY: ABORT_MEDUSA

#ifdef ALLOW_MPI
      USE mod_execontrol_medusa,    ONLY: MEDEXE_MPI_COMM,
     &                              MEDEXE_MPI_COMM_RANK,
     &                              jp_exeproc_root
      USE mpi, ONLY: MPI_INTEGER, MPI_DOUBLE_PRECISION
#endif


      IMPLICIT NONE


!-----------------------
! Variable declarations
!-----------------------

!- - - - - - - - - - - - -  - - - - - - -
! Variables in subroutine call arguments
!- - - - - - - - - - - - -- - - - - - - -

      INTEGER, INTENT(IN) :: k_stage
      INTEGER, INTENT(IN) :: iu_cfg


!- - - - - - - - - - - - - -  - - - - - - - -
! General (global) parameters and definitions
!- - - - - - - - - - - - - -- - - - - - - - -

      DOUBLE PRECISION, SAVE  :: db_scale

      NAMELIST /nml_biodif_muds_orig/  db_scale


!- - - - - - - - -
! Local variables
!- - - - - - - - -

      ! None


!- - - - - - - - - - -
! End of declarations
!- - - - - - - - - - -


!----------------------------------------------------------------------
! Subroutine Start
!----------------------------------------------------------------------


      SELECT CASE(k_stage)
      CASE(1)
                                    ! Pre-set the common default values
        db_scale    = dp_bt_muds_scale

        READ(iu_cfg, NML=nml_biodif_muds_orig)


        IF (db_scale <= 0.0D+00) THEN
          ! scale must be > 0!
          CALL ABORT_MEDUSA()
        ENDIF


      CASE(2)

#ifdef ALLOW_MPI
                                    ! Broadcast the configuration data:
                                    !  - scale (all)
        CALL MPI_BCAST(db_scale, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
#endif

        WRITE(jp_stdlog, '(" - for biodiffusion")')
        WRITE(jp_stdlog, '("   * profile ID: ", I0)')
     &                              jselect_biodif_profile


        da_bt_muds_scale    = db_scale

        WRITE(jp_stdlog, '("   * scale: ", E9.3)')
     &                              da_bt_muds_scale
        WRITE(jp_stdlog, '()')


      CASE DEFAULT
                                    ! Unknown stage
        CALL ABORT_MEDUSA()


      END SELECT


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE BDIFFC_CUSTOM_SETUP
!-----------------------------------------------------------------------
#endif
