!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#ifdef BDIFFC_CUSTOM_DECLARATIONS

                                    ! Additional parameters and configuration
                                    ! (adjustable scales etc.) for the
                                    ! Dhakar and Burdige D_B profile.
      DOUBLE PRECISION, PARAMETER
     &  :: dp_bt_dhabur_xscale    =   1.0D+00*dp_cm
      DOUBLE PRECISION, PARAMETER
     &  :: dp_bt_dhabur_xflex     =   8.0D+00*dp_cm
      DOUBLE PRECISION, PARAMETER
     &  :: dp_bt_dhabur_dcf_0     = 150.0D+00*dp_cm2_p_kyr
      DOUBLE PRECISION, PARAMETER
     &  :: dp_bt_dhabur_dcf_infty =   0.0D+00*dp_cm2_p_kyr
      DOUBLE PRECISION, PARAMETER
     &  :: dp_bt_dhabur_o2scale   =  0.01D+00       ! 100 (mmol/L)^-1
                                                    ! 100 (mol/m3)^-1
                                                    ! = 1/(0.01 mol/m3)
                                                    ! => scale = 0.01 mol/m3
      DOUBLE PRECISION, PARAMETER
     &  :: dp_bt_dhabur_o2flex    =   4.0D-03       ! 4 µmol/L = 4D-3 mol/m3



                                    ! Actually used parameter values for
                                    ! the Dhakar and Burdige D_B profile.
      DOUBLE PRECISION, SAVE :: da_bt_dhabur_xscale
      DOUBLE PRECISION, SAVE :: da_bt_dhabur_xflex
      DOUBLE PRECISION, SAVE :: da_bt_dhabur_dcf_0
      DOUBLE PRECISION, SAVE :: da_bt_dhabur_dcf_infty
      DOUBLE PRECISION, SAVE :: da_bt_dhabur_o2scale
      DOUBLE PRECISION, SAVE :: da_bt_dhabur_o2flex
      
#else
!-----------------------------------------------------------------------
      SUBROUTINE BDIFFC_CUSTOM(xzdn, xzdv, xcompo, wconc, wfflx)
!-----------------------------------------------------------------------

! Biodiffusion coefficient as in
! Dhakar and Burdige, Am. J. Sci. 296(3), pp. 296-330, 1996.
! doi:10.2475/ajs.296.3.296

!--------------
! Declarations
!--------------

      USE mod_basicdata_medusa,     ONLY: dp_cm, dp_cm2, dp_cm2_p_yr
      USE mod_gridparam,            ONLY: idnw, idnt, idnz,  idnb,
     &                                    idvw, idvs, idvaz, idvb
      USE mod_indexparam,           ONLY: ncompo, nsolut, nsolid,
     &                                    io_o2

#ifdef DEBUG
      USE mod_defines_medusa,       ONLY: jp_stddbg
#endif

! ======================================================================
! This subroutine is currently unusable, as the calculations of the O2
! concentration approximations at the vertices and their derivatives
! are not correctly implemented (they have to be revised due to the
! grid revision).
! It will therefore abort upon it's first being called.
! Please remove this comment and the  USE mod_execontrol_medusa clause
! below once this has been fixed.
! ======================================================================

      USE mod_execontrol_medusa,    ONLY: ABORT_MEDUSA

      IMPLICIT NONE


!-----------------------
! Variable declarations
!-----------------------

!- - - - - - - - - - - - -  - - - - - - -
! Variables in subroutine call arguments
!- - - - - - - - - - - - -- - - - - - - -

      DOUBLE PRECISION, DIMENSION(idnw:idnb)           :: xzdn
      DOUBLE PRECISION, DIMENSION(idvw:idvb)           :: xzdv
      DOUBLE PRECISION, DIMENSION(idnw:idnb, ncompo)   :: xcompo
      DOUBLE PRECISION, DIMENSION(nsolut)              :: wconc
      DOUBLE PRECISION, DIMENSION(nsolid)              :: wfflx

      INTENT(IN) :: xzdn, xzdv, xcompo, wconc, wfflx

!- - - - - - - - -
! Local variables
!- - - - - - - - -

      DOUBLE PRECISION :: asdxz     ! scaled xzdn deviation
      DOUBLE PRECISION :: asdo2     ! scaled [O2] deviation
      DOUBLE PRECISION :: alimxz    ! limitation w/r to xzdn
      DOUBLE PRECISION :: alimo2    ! limitation w/r to [O2]
      DOUBLE PRECISION :: dxo2      ! d[O2]/dz

      INTEGER          :: i

      DOUBLE PRECISION :: azdnz

!- - - - - - - - - - -
! End of declarations
!- - - - - - - - - - -


!----------------------------------------------------------------------
! Subroutine Start
!----------------------------------------------------------------------

#ifdef DEBUG
#ifdef DEBUG_BDIFFC
#ifdef DEBUG_BDIFFC_ENTRY_EXIT
      WRITE(jp_stddbg,*) '[BDIFFC (Dhakar & Burdige)]: Starting'
#endif
#endif
#endif

! ======================================================================
! This subroutine is currently unusable, as the calculations of the O2
! concentration approximations at the vertices and their derivatives
! are not correctly implemented (they have to be revised due to the
! grid revision).
! It will therefore abort upon it's first being called.
! Please remove this comment and the CALL ABORT_MEDUSA line below once
! this has been fixed.
! ======================================================================

      CALL ABORT_MEDUSA()


      IF (.NOT. l_setupdone) CALL SETUP_TRANSPORT

      azdnz = xzdn(idnz)

      !---------------------------------------------
      ! Biodiffusion coefficient and its derivative
      !---------------------------------------------

      DO i = idvs, idvaz

        asdxz = (xzdv(i) - da_bt_dhabur_xflex)
     &                              / da_bt_dhabur_xscale
        asdo2 = (xcompo(i,io_o2) - da_bt_dhabur_o2flex)
     &                              / da_bt_dhabur_o2scale

        alimo2 = 1.0D+00/(1.0D+00 + EXP(-asdo2))
        alimxz = 1.0D+00/(1.0D+00 + EXP( asdxz))

        SELECT CASE(i)
        CASE(idnt)
          dxo2 = (xcompo(idnt+1,io_o2) - xcompo(idnt,io_o2))
     &           / (xzdn(idnt+1)       -   xzdn(idnt))
        CASE(idnz)
          dxo2 = (xcompo(idnz,io_o2) - xcompo(idnz-1,io_o2))
     &           / (xzdn(idnz)       -   xzdn(idnz-1))
        CASE DEFAULT
          dxo2 = (xcompo(i+1,io_o2) - xcompo(i-1,io_o2))
     &           / (xzdn(i+1)       -   xzdn(i-1))
        END SELECT


        dcf_biotur(i) =
     &      da_bt_dhabur_dcf_infty
     &        + (da_bt_dhabur_dcf_0 - da_bt_dhabur_dcf_infty)
     &          * alimo2 * alimxz

                                    ! approximate derivative:
                                    ! d[O2]/dz by finite differences
        dx_dcf_biotur(i) =
     &      (dcf_biotur(i) - da_bt_dhabur_dcf_infty)
     &       * (   alimo2 * EXP(-asdo2)/da_bt_dhabur_o2scale * dxo2
     &           - alimxz * EXP( asdxz)/da_bt_dhabur_xscale)

      ENDDO

#ifdef DEBUG
#ifdef DEBUG_BDIFFC
#ifdef DEBUG_BDIFFC_ENTRY_EXIT
      WRITE(jp_stddbg,*) '[BDIFFC (Dhakar & Burdige)]: Exiting'
#endif
#endif
#endif

      RETURN      


!-----------------------------------------------------------------------
      END SUBROUTINE BDIFFC_CUSTOM
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE BDIFFC_CUSTOM_SETUP(k_stage, iu_cfg)
!-----------------------------------------------------------------------

!--------------
! Declarations
!--------------


      USE mod_defines_medusa,       ONLY: jp_stderr, jp_stdlog
      USE mod_execontrol_medusa,    ONLY: ABORT_MEDUSA

#ifdef ALLOW_MPI
      USE mod_execontrol_medusa,    ONLY: MEDEXE_MPI_COMM,
     &                              MEDEXE_MPI_COMM_RANK,
     &                              jp_exeproc_root
      USE mpi, ONLY: MPI_INTEGER, MPI_DOUBLE_PRECISION
#endif


      IMPLICIT NONE


!-----------------------
! Variable declarations
!-----------------------

!- - - - - - - - - - - - -  - - - - - - -
! Variables in subroutine call arguments
!- - - - - - - - - - - - -- - - - - - - -

      INTEGER, INTENT(IN) :: k_stage
      INTEGER, INTENT(IN) :: iu_cfg


!- - - - - - - - - - - - - -  - - - - - - - -
! General (global) parameters and definitions
!- - - - - - - - - - - - - -- - - - - - - - -

      DOUBLE PRECISION, SAVE :: db_xscale
      DOUBLE PRECISION, SAVE :: db_xflex
      DOUBLE PRECISION, SAVE :: db_dcf_0
      DOUBLE PRECISION, SAVE :: db_dcf_infty
      DOUBLE PRECISION, SAVE :: db_o2scale
      DOUBLE PRECISION, SAVE :: db_o2flex

      NAMELIST /nml_biodif_dhabur_orig/  db_xscale, db_xflex,
     &                              db_dcf_0,   db_dcf_infty,
     &                              db_o2scale, db_o2flex

!- - - - - - - - -
! Local variables
!- - - - - - - - -

      ! None


!- - - - - - - - - - -
! End of declarations
!- - - - - - - - - - -


!----------------------------------------------------------------------
! Subroutine Start
!----------------------------------------------------------------------


      SELECT CASE(k_stage)
      CASE(1)
                                    ! Pre-set the common default values
        db_xscale    = dp_bt_dhabur_xscale
        db_xflex     = dp_bt_dhabur_xflex
        db_dcf_0     = dp_bt_dhabur_dcf_0
        db_dcf_infty = dp_bt_dhabur_dcf_infty
        db_o2scale   = dp_bt_dhabur_o2scale
        db_o2flex    = dp_bt_dhabur_o2flex

        READ(iu_cfg, NML=nml_biodif_dhabur_orig)


        IF (db_xscale <= 0.0D+00) THEN
          ! scale must be > 0!
          CALL ABORT_MEDUSA()
        ENDIF

        IF (db_xflex <= 0.0D+00) THEN
          ! x inflection must be > 0!
          CALL ABORT_MEDUSA()
        ENDIF

        IF (db_dcf_0 <= 0.0D+00) THEN
          ! dcf_0 must be > 0!
          CALL ABORT_MEDUSA()
        ENDIF

        IF (db_dcf_infty < 0.0D+00) THEN
          ! dcf_infty must be >= 0!
          CALL ABORT_MEDUSA()
        ENDIF

        IF (db_o2scale <= 0.0D+00) THEN
          ! O2 scale must be > 0!
          CALL ABORT_MEDUSA()
        ENDIF

        IF (db_o2flex <= 0.0D+00) THEN
          ! o2 inflection point must be > 0!
          CALL ABORT_MEDUSA()
        ENDIF


      CASE(2)

#ifdef ALLOW_MPI
                                    ! Broadcast the configuration data:
                                    !  - xscale
        CALL MPI_BCAST(db_xscale, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
                                    !  - xflex
        CALL MPI_BCAST(db_xflex, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
                                    !  - dcf_0
        CALL MPI_BCAST(db_dcf_0, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
                                    !  - dcf_infty
        CALL MPI_BCAST(db_dcf_infty, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
                                    !  - o2scale
        CALL MPI_BCAST(db_o2scale, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
                                    !  - o2flex
        CALL MPI_BCAST(db_o2flex, 1, MPI_DOUBLE_PRECISION,
     &                              jp_exeproc_root, i_mycomm, istatus)
#endif

        WRITE(jp_stdlog, '(" - for biodiffusion")')
        WRITE(jp_stdlog, '("   * profile ID: ", I0)')
     &                              jselect_biodif_profile


        da_bt_dhabur_xscale    = db_xscale
        da_bt_dhabur_xflex     = db_xflex
        da_bt_dhabur_dcf_0     = db_dcf_0 
        da_bt_dhabur_dcf_infty = db_dcf_infty
        da_bt_dhabur_o2scale   = db_o2scale
        da_bt_dhabur_o2flex    = db_o2flex

        WRITE(jp_stdlog, '("   * xzdn scale: ", E9.3)')
     &                              da_bt_dhabur_xscale
        WRITE(jp_stdlog, '("   * xzdn flex pt: ", E9.3)')
     &                              da_bt_dhabur_xflex
        WRITE(jp_stdlog, '("   * D_B0: ", E9.3)')
     &                              da_bt_dhabur_dcf_0
        WRITE(jp_stdlog, '("   * D_Binfty: ", E9.3)')
     &                              da_bt_dhabur_dcf_infty
        WRITE(jp_stdlog, '("   * O2 scale: ", E9.3)')
     &                              da_bt_dhabur_o2scale
        WRITE(jp_stdlog, '("   * O2 flex pt: ", E9.3)')
     &                              da_bt_dhabur_o2flex
        WRITE(jp_stdlog, '()')


      CASE DEFAULT
                                    ! Unknown stage
        CALL ABORT_MEDUSA()


      END SELECT


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE BDIFFC_CUSTOM_SETUP
!-----------------------------------------------------------------------
#endif
