!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


!-----------------------------------------------------------------------
      SUBROUTINE BDIFFC(xzdn, xzdv, xcompo, wconc, wfflx)
!-----------------------------------------------------------------------

!--------------
! Declarations
!--------------

      USE mod_gridparam,            ONLY: idnw, idnt, idnz,  idnb,
     &                                    idvw, idvs, idvaz, idvb
      USE mod_indexparam

#ifdef DEBUG
      USE mod_defines_medusa,       ONLY: jp_stddbg
#endif


      IMPLICIT NONE


!-----------------------
! Variable declarations
!-----------------------

!- - - - - - - - - - - - - -  - - - - - - - -
! General (global) parameters and definitions
!- - - - - - - - - - - - - -- - - - - - - - -

      ! None

!- - - - - - - - - - - - -  - - - - - - -
! Variables in subroutine call arguments
!- - - - - - - - - - - - -- - - - - - - -

      DOUBLE PRECISION, DIMENSION(idnw:idnb)           :: xzdn
      DOUBLE PRECISION, DIMENSION(idvw:idvb)           :: xzdv
      DOUBLE PRECISION, DIMENSION(idnw:idnb, ncompo)   :: xcompo
      DOUBLE PRECISION, DIMENSION(nsolut)              :: wconc
      DOUBLE PRECISION, DIMENSION(nsolid)              :: wfflx

      INTENT(IN) :: xzdn, xzdv, xcompo, wconc, wfflx

!- - - - - - - - -
! Local variables
!- - - - - - - - -

      INTEGER                                          :: i

!     idnz   : index locating the depth node where the bottom
!              interface of the bioturbated region is located:
!              dcf_biotur(i) != 0 when i < idnz
!              dcf_biotur(i)  = 0 when i > idnz

      DOUBLE PRECISION                                 :: azdnz
      DOUBLE PRECISION                                 :: arg

!     dcf_biotur(idnz) may or may not be equal to 0. This depends on
!     whether the biodiffusion coefficient is continuous
!     across x = azdnz (= xzdn(idnz)) or not.

! It is supposed that the biodiffusion coefficient "dcf_biotur" is
! * continuous and
! * continuously differentiable on the interval
!   [xzdn(idnt),xzdn(idnz)]
! * equal to zero on ]xzdn(idnz),xzdn(idnb)]

! dx_dcf_biotur(i) is equal to the derivative of dcf_biotur in
! x=(xzdn(i))^-, i.e., the one-sided derivative of dcf_biotur
! on the side *above* node "i", except, obviously at the top node.
! Except at the possible discontinuity point xzdn(idnz), the
! two one-sided derivatives of dcf_biotur at any given point are equal.
! It is clear the the one-sided derivative of dcf_biotur in
! x=(xzdn(i))^+, i.e., on the side *below* node "idnz" is zero!

!- - - - - - - - - - -
! End of declarations
!- - - - - - - - - - -

!----------------------------------------------------------------------
! Subroutine Start
!----------------------------------------------------------------------

#ifdef DEBUG
#ifdef DEBUG_BDIFFC
#ifdef DEBUG_BDIFFC_ENTRY_EXIT
      WRITE(jp_stddbg,*) '[BDIFFC]: Starting'
#endif
#endif
#endif


      IF (.NOT. l_setupdone) CALL SETUP_TRANSPORT

      azdnz = xzdn(idnz)

      !---------------------------------------------
      ! Biodiffusion coefficient and its derivative
      !---------------------------------------------

      SELECT CASE(jselect_biodif_profile)

      CASE(jp_bt_lin0z)
      !----------------             ! Biodiffusion coefficient according to
                                    ! D_B(z) = D_B0 * (z_Z - z)/z_Z
                                    ! Continuous D_B across z_Z = xzdn(idnz)
        DO i = idvs, idvaz
              dcf_biotur(i) =  da_bt_dcf_0*(1.0D+00-xzdv(i)/azdnz)
           dx_dcf_biotur(i) = -da_bt_dcf_0/azdnz
        ENDDO


        IF (idnb == idnz) THEN      ! idnz = idvaz+1 = idnb = idvb in this case
             dcf_biotur(idnz) =  0.0D+00
          dx_dcf_biotur(idnz) = -da_bt_dcf_0/azdnz
        ELSE
             dcf_biotur(idnz) =  0.0D+00
          dx_dcf_biotur(idnz) =  0.0D+00
        ENDIF


      CASE(jp_bt_linxz)             ! Biodiffusion coefficient according to
      !----------------             ! D_B(z) = D_B0 * (1 - (z/(z_Z*da_bt_linxz_slope))
                                    ! Discontinuous D_B across z_Z = xzdn(idnz), except if
                                    ! da_bt_linxz_slope == 1
        DO i = idvs, idvaz
             dcf_biotur(i) =  da_bt_dcf_0
     &                        * (1.0D+00 - xzdv(i)
     &                                     /(azdnz*da_bt_linxz_slope))
          dx_dcf_biotur(i) = -da_bt_dcf_0/(azdnz*da_bt_linxz_slope)
        ENDDO

        IF (idnb == idnz) THEN      ! idnz = idvaz+1 = idnb = idvb in this case
             dcf_biotur(idnz) =  da_bt_dcf_0
     &                           *(1.0D+00 - 1.0D+00/da_bt_linxz_slope)
          dx_dcf_biotur(idnz) = -da_bt_dcf_0/(azdnz*da_bt_linxz_slope)
        ELSE
             dcf_biotur(idnz) =  0.0D+00
          dx_dcf_biotur(idnz) =  0.0D+00
        ENDIF


      CASE(jp_bt_quad0z)            ! Biodiffusion coefficient according to
      !-----------------            ! D_B(z) = D_B0*(1-z/z_Z)**2
                                    ! Continuity at xzdn(idnz)
        DO i = idvs, idvaz
             dcf_biotur(i) =  da_bt_dcf_0
     &                        *(1.0D+00-xzdv(i)/azdnz)**2
          dx_dcf_biotur(i) = -da_bt_dcf_0*2.0D+00
     &                        *(1.0D+00-xzdv(i)/azdnz)/azdnz
        ENDDO

           dcf_biotur(idnz) = 0.0D+00
        dx_dcf_biotur(idnz) = 0.0D+00


      CASE(jp_bt_expdec)            ! Biodiffusion coefficient according to
      !-----------------            ! D_B(z) = D_B0*EXP(-z/da_bt_expdec_scale)
                                    ! Discontinuity at xzdn(idnz)

        DO i = idvs, idvaz
          dcf_biotur(i)    =  da_bt_dcf_0
     &                        * EXP(-xzdv(i)/da_bt_expdec_scale)
          dx_dcf_biotur(i) = -dcf_biotur(i)/da_bt_expdec_scale
        ENDDO

        IF (idnb == idnz) THEN      ! idnz = idvaz+1 = idnb = idvb in this case
             dcf_biotur(idnz) =  da_bt_dcf_0
     &                           * EXP(-azdnz/da_bt_expdec_scale)
          dx_dcf_biotur(idnz) = -dcf_biotur(idnz)/da_bt_expdec_scale
        ELSE
             dcf_biotur(idnz) =  0.0D+00
          dx_dcf_biotur(idnz) =  0.0D+00
        ENDIF


      CASE(jp_bt_gaussn)            ! Biodiffusion coefficient according to
      !-----------------            ! the Gaussian profile
                                    ! D_B(z) = D_B0*EXP(-(z/da_bt_gaussn_scale)**2)
                                    ! Discontinuity at xzdn(idnz)

        DO i = idvs, idvaz
          dcf_biotur(i) =      da_bt_dcf_0
     &                         * EXP(-(xzdv(i)/da_bt_gaussn_scale)**2)
          dx_dcf_biotur(i) =  -dcf_biotur(i)
     &                         * 2.0D+00*xzdv(i)/da_bt_gaussn_scale**2
        ENDDO

        IF (idnb == idnz) THEN      ! idnz = idvaz+1 = idnb = idvb in this case
             dcf_biotur(idnz) = da_bt_dcf_0
     &                          * EXP(-(azdnz/da_bt_gaussn_scale)**2)
          dx_dcf_biotur(idnz) =  -dcf_biotur(idnz)
     &                         * 2.0D+00*azdnz/da_bt_gaussn_scale**2
        ELSE
             dcf_biotur(idnz) =  0.0D+00
          dx_dcf_biotur(idnz) =  0.0D+00
        ENDIF


      CASE(jp_bt_erfc )             ! Biodiffusion coefficient according to
      !----------------             ! the complementary error function profile
                                    ! D_B(z) = D_B0/2*ERFC(((z-da_bt_erfc_halfd)/da_bt_erfc_scale))
                                    ! Discontinuity at xzdn(idnz)

        DO i = idvs, idvaz
          arg = (xzdv(i)-da_bt_erfc_halfd) / da_bt_erfc_scale
          dcf_biotur(i)    =  da_bt_dcf_0/2.0D+00 * ERFC(arg)
          dx_dcf_biotur(i) = -da_bt_dcf_0 * EXP(-arg**2)
     &                        / (SQRT(dp_pi)*da_bt_erfc_scale)
        ENDDO

        IF (idnb == idnz) THEN      ! idnz = idvaz+1 = idnb = idvb in this case
          arg = (azdnz-da_bt_erfc_halfd) / da_bt_erfc_scale
             dcf_biotur(idnz) =  da_bt_dcf_0/2.0D+00 * ERFC(arg)
          dx_dcf_biotur(idnz) = -da_bt_dcf_0 * EXP(-arg**2)
     &                           / (SQRT(dp_pi)*da_bt_erfc_scale)
        ELSE
             dcf_biotur(idnz) =  0.0D+00
          dx_dcf_biotur(idnz) =  0.0D+00
        ENDIF


#ifdef BIODIFFUSION_CUSTOM
      CASE(jp_bt_custom)            ! Biodiffusion coefficient according
      !-----------------            ! to a custom profile

        CALL BDIFFC_CUSTOM(xzdn, xzdv, xcompo, wconc, wfflx)
#endif


      CASE DEFAULT                  ! D_B(z) = cte = D_B0
      !-----------                  ! Here dcf_biotur(inode) = D_B(x=xzdn(inode)^-).
                                    ! Discontinuity for D_B(z) at xzdn(idnz)
        DO i = idvs, idvaz
          dcf_biotur(i)    = da_bt_dcf_0
          dx_dcf_biotur(i) = 0.0D+00
        ENDDO

        IF (idnb == idnz) THEN      ! idnz = idvaz+1 = idnb = idvb in this case
             dcf_biotur(idnz) =  da_bt_dcf_0
          dx_dcf_biotur(idnz) =  0.0D+00
        ELSE
             dcf_biotur(idnz) =  0.0D+00
          dx_dcf_biotur(idnz) =  0.0D+00
        ENDIF


      END SELECT


      DO i = idvaz+2, idvb
            dcf_biotur(i) = 0.0D+00
         dx_dcf_biotur(i) = 0.0D+00
      ENDDO


      !-----------------------------------------------------
      ! Interphase bioturbation fraction and its derivative
      !-----------------------------------------------------

      DO i = idvs, idvb
            dcf_biointer(i) = 0.0D+00
         dx_dcf_biointer(i) = 0.0D+00
      ENDDO


#ifdef DEBUG
#ifdef DEBUG_BDIFFC
#ifdef DEBUG_BDIFFC_BIODIFFUSION_INFOS
      WRITE(jp_stddbg,'(A,I0)')
     & '[BDIFFC]: Biodiffusion profile type ', jselect_biodif_profile
      IF(dcf_biotur(idnz) .NE. 0.0D+00)
     &  WRITE(jp_stddbg,'(A,I0)')
     &    '          discontinuous to 0 at node ', idnz
      IF(dcf_biotur(idnz) .EQ. 0.0D+00)
     &  WRITE(jp_stddbg,'(A,I0)')
     &    '          continuous to 0 at node ', idnz
      WRITE(jp_stddbg,'()')
      WRITE(jp_stddbg,'(A)')
     & '[BDIFFC]: Depth  Biodiffusion   derivative (in inode^-)'
      WRITE(jp_stddbg,'(A)')
     & '                 coefficient    of biod.cf.'
      WRITE(jp_stddbg,'(10X, f7.5,e13.5,2X, e13.5)')
     & (xzdv(i), dcf_biotur(i),dx_dcf_biotur(i), i=idvs,idvb)
#endif
#ifdef DEBUG_BDIFFC_ENTRY_EXIT
      WRITE(jp_stddbg,*) '[BDIFFC]: Exiting'
#endif
#endif
#endif

      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE BDIFFC
!-----------------------------------------------------------------------
