!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#include <configure.h>
!=======================================================================
 SUBROUTINE XMLCOCOGEN_LOADDB(c_xmlbaselist,                           &
    xml_compofiles, xml_procsfiles, xml_equilfiles,                    &
    xml_apiexfiles)
!=======================================================================

USE MOD_XMLCOCOGEN
USE MODMXM_STKXE
USE MODMXM_STKMX
USE MODMXM_STRUCTLOAD


IMPLICIT NONE


! Argument list variables
! -----------------------

CHARACTER(LEN=*)        :: c_xmlbaselist
TYPE(xfs_list), POINTER :: xml_compofiles
TYPE(xfs_list), POINTER :: xml_procsfiles
TYPE(xfs_list), POINTER :: xml_equilfiles
TYPE(xfs_list), POINTER :: xml_apiexfiles


! Local variables
! ----------------

CHARACTER(LEN=p_maxlen_eltname)  :: c_eltname
CHARACTER(LEN=p_maxlen_attname)  :: c_attname
CHARACTER(LEN=p_maxlen_attcntt)  :: c_attcntt


TYPE(stack_xmlevents), POINTER :: stkxe_baselist
TYPE(stack_minixml), POINTER   :: stkmx_baselist
TYPE(stkxe_ptr), DIMENSION(:), POINTER :: stkxep_solutsystlists
TYPE(stkmx_ptr), DIMENSION(:), POINTER :: stkmxp_solutsystlists
TYPE(stack_xmlevents), POINTER :: stkxe_solutsystlist
TYPE(stack_minixml), POINTER   :: stkmx_solutsystlist
INTEGER :: n_maxlen_eltname
INTEGER :: n_maxlen_attname
INTEGER :: n_maxlen_attcont

TYPE(stack_xmlevents),         POINTER :: stkxe_work

TYPE(stack_minixml),           POINTER :: stkmx_composition
TYPE(stack_minixml),           POINTER :: stkmx_solute
TYPE(stack_minixml),           POINTER :: stkmx_solid
TYPE(stack_minixml),           POINTER :: stkmx_solutsyst
TYPE(stack_minixml),           POINTER :: stkmx_member
TYPE(stack_minixml),           POINTER :: stkmx_processes
TYPE(stack_minixml),           POINTER :: stkmx_process
TYPE(stack_minixml),           POINTER :: stkmx_equilibria
TYPE(stack_minixml),           POINTER :: stkmx_equilibrium
TYPE(stack_minixml),           POINTER :: stkmx_apiextensions
TYPE(stack_minixml),           POINTER :: stkmx_apiextension

TYPE(stkmx_ptr), DIMENSION(:), POINTER :: stkmxp_composition
TYPE(stkmx_ptr), DIMENSION(:), POINTER :: stkmxp_solutes
TYPE(stkmx_ptr), DIMENSION(:), POINTER :: stkmxp_solids
TYPE(stkmx_ptr), DIMENSION(:), POINTER :: stkmxp_solutsysts
TYPE(stkmx_ptr), DIMENSION(:), POINTER :: stkmxp_members
TYPE(stkmx_ptr), DIMENSION(:), POINTER :: stkmxp_processes
TYPE(stkmx_ptr), DIMENSION(:), POINTER :: stkmxp_process
TYPE(stkmx_ptr), DIMENSION(:), POINTER :: stkmxp_equilibria
TYPE(stkmx_ptr), DIMENSION(:), POINTER :: stkmxp_equilibrium
TYPE(stkmx_ptr), DIMENSION(:), POINTER :: stkmxp_apiextensions
TYPE(stkmx_ptr), DIMENSION(:), POINTER :: stkmxp_apiextension

INTEGER :: n_composition
INTEGER :: i_child, n_children
INTEGER :: n_maxdepth, n_order

INTEGER :: i_solutsyst, n_solutsysts
INTEGER :: i_solute,    n_solutes
INTEGER :: i_solid,     n_solids
INTEGER :: i_proc,      n_procs
INTEGER :: i_equil,     n_equils
INTEGER :: i_apiex,     n_apiexs
INTEGER :: i_att,       n_atts
INTEGER :: i_order, i_ordersolsyst
TYPE(stack_recchunks), POINTER :: stkrc_eltname
TYPE(stack_recchunks), POINTER :: stkrc_attcntt
CHARACTER(LEN=p_maxlen_filename) :: c_xmlfilename
INTEGER :: nlen_returned

TYPE(xfs_list), POINTER :: xml_compofilewk
TYPE(xfs_list), POINTER :: xml_procsfilewk
TYPE(xfs_list), POINTER :: xml_equilfilewk
TYPE(xfs_list), POINTER :: xml_apiexfilewk
TYPE(xfs_list), POINTER :: xfs_test, xfs_testnext, xfs_work
INTEGER :: n_xmlcompofiles
INTEGER :: n_xmlprocsfiles
INTEGER :: n_xmlequilfiles
INTEGER :: n_xmlapiexfiles
INTEGER :: i_ordermin

LOGICAL :: l_exists
INTEGER :: n_missingfiles


! I/O related parameters

INTEGER, PARAMETER           :: jp_stderr = (CFG_STDERR)
INTEGER, PARAMETER           :: jp_stdout = (CFG_STDOUT)
CHARACTER(LEN=*), PARAMETER  :: c_fmtinf_a = '("[XMLCOCOGEN_LOADDB]: ", A)'
CHARACTER(LEN=*), PARAMETER  :: c_fmtwar_a = '("[XMLCOCOGEN_LOADDB] warning: ", A)'
CHARACTER(LEN=*), PARAMETER  :: c_fmterr_a = '("[XMLCOCOGEN_LOADDB] error: ", A)'

#ifdef CFG_DEBUG
INTEGER, PARAMETER           :: jp_stddbg = (CFG_STDDBG)
CHARACTER(LEN=*), PARAMETER  :: c_fmtdbg_a = '("DEBUG [XMLCOCOGEN_LOADDB]: ", A)'
#endif


!=====================!
! End of declarations !
!=====================!

WRITE(jp_stdout,'()') 
WRITE(jp_stdout,c_fmtinf_a) 'starting'
#ifdef CFG_DEBUG
WRITE(jp_stddbg,'()') 
WRITE(jp_stddbg,c_fmtdbg_a) 'starting'
#endif

!============!
! Operations !
!============!

n_missingfiles = 0

INQUIRE(FILE=c_xmlbaselist, EXIST=l_exists)

IF (.NOT. l_exists) THEN

  WRITE(jp_stderr,c_fmterr_a, ADVANCE='NO') ''
  WRITE(jp_stderr, '("The requested XML base list """, A, """")') TRIM(c_xmlbaselist)
  WRITE(jp_stderr, '("does not exist or is a stale symbolic link -- aborting!")')
  CALL ABORT()

ENDIF

                 ! Get the events' list for the base file
stkxe_baselist => XMLSTRUCT(c_xmlbaselist, &
                            n_maxlen_eltname, n_maxlen_attname, n_maxlen_attcont)

! Could check here if maxlen's do not exceed currently adopted parameterizations

                 ! and read in the data
stkmx_baselist => XMLLOAD(c_xmlbaselist, stkxe_baselist, n_maxdepth)


                 ! 1. get the <Composition> element and
                 !    1.1 parse for the <Solute> elements
                 !    1.2 parse for the <Solid> elements
                 !    1.3 parse for the <SoluteSystem> elements and
                 !        resolve them
                 !
                 ! 2. get the <Processes> element (if any)
                 !
                 ! 3. get the <Equilibria> element (if any)
                 !
                 ! 4. get the <APIExtensions> element (if any)


                 !=============!
                 ! Composition !
                 !=============!

                 ! Initialize the list of files to read in for the
                 ! composition
NULLIFY(xml_compofiles)
n_xmlcompofiles = 0

stkmxp_composition => STKMX_getElementNodeByName(stkmx_baselist, cp_composition)
!stkmx_composition => STKMX_getUniqueChildByName(stkmx_baselist, cp_composition)


WRITE(jp_stdout,'()')

IF (ASSOCIATED(stkmxp_composition)) THEN

  n_composition = SIZE(stkmxp_composition)

  IF (n_composition /= 1) THEN
    WRITE(jp_stderr,c_fmterr_a, ADVANCE='NO') ''
    WRITE(jp_stderr,'("Found ",I0, " <'//cp_composition//'> elements -")') n_composition
    WRITE(jp_stderr,'("there can be one at most -- aborting!")')
    CALL ABORT()
  ENDIF

  WRITE(jp_stdout,c_fmtinf_a) 'Found one <' // cp_composition // '> element.'



  stkmx_composition => stkmxp_composition(1)%ptr


  ! Solutes

  WRITE(jp_stdout,'()')
  stkmxp_solutes => STKMX_getElementNodeByName(stkmx_composition, cp_solute)
  !stkmxp_solutes => STKMX_getChildrenByName(stkmx_composition, cp_solute)

  IF (ASSOCIATED(stkmxp_solutes)) THEN
    n_solutes = SIZE(stkmxp_solutes)
    IF (n_solutes == 1) THEN
      WRITE(jp_stdout,'("<'//cp_composition//'> has 1 <'//cp_solute//'> child element")')
    ELSE
      WRITE(jp_stdout,'("<'//cp_composition//'> has ", I0, &
                      &" <'//cp_solute//'> child elements")') n_solutes
    ENDIF
  ELSE
    n_solutes = 0
    WRITE(jp_stdout,'("<'//cp_composition//'> has no <'//cp_solute//'> child element.")')
  ENDIF

  DO i_solute = 1, n_solutes
    stkmx_solute => stkmxp_solutes(i_solute)%ptr
    i_att = STKMX_getAttIdxByName(stkmx_solute, cp_file)
    IF (i_att > 0) THEN
      stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_solute, i_att)
      CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_xmlfilename, nlen_returned)
      WRITE(jp_stdout,'(" <'//cp_solute//'> ", I0, ": file=""",A,"""")') &
        i_solute, TRIM(c_xmlfilename)

      IF (n_xmlcompofiles == 0) THEN
        ! Now create the XFS_LIST for the XML composition files
        xml_compofiles  => XFSL_createRoot()
        xml_compofilewk => xml_compofiles
      ELSE
        xfs_work => xml_compofilewk
        xml_compofilewk => XFSL_addNode(xfs_work)
      ENDIF

      xml_compofilewk%fname = c_xmlfilename
      xml_compofilewk%c_type = cp_solute

      i_att = STKMX_getAttIdxByName(stkmx_solute, cp_order)
      IF (i_att > 0) THEN
        stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_solute, i_att)
        CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_attcntt, nlen_returned)
        READ(c_attcntt,*) i_order
        xml_compofilewk%i_order = i_order
      ELSE
        WRITE(jp_stdout,'(" <'//cp_solute//'> ", I0, ": ""'//cp_order//'"" &
                         &attribute missing -- using default")') i_solute
      ENDIF

      xml_compofilewk%i_system = 0 ! not part of a system

      n_xmlcompofiles = n_xmlcompofiles + 1
      
    ELSE

      WRITE(jp_stderr,c_fmterr_a, ADVANCE='NO') ''
      WRITE(jp_stderr,'(" <'//cp_solute//'> ", I0, ": ""'//cp_file//'""&
                       & attribute not found -- aborting")') i_solute
      CALL ABORT()

    ENDIF
  ENDDO


  ! Solids

  WRITE(jp_stdout,'()')
  stkmxp_solids => STKMX_getElementNodeByName(stkmx_composition, cp_solid)

  IF (ASSOCIATED(stkmxp_solids)) THEN
    n_solids = SIZE(stkmxp_solids)
    IF (n_solids == 1) THEN
      WRITE(jp_stdout,'("<'//cp_composition//'> has 1 <'//cp_solid//'> child element")')
    ELSE
      WRITE(jp_stdout,'("<'//cp_composition//'> has ", I0, &
                      &" <'//cp_solid//'> child elements")') n_solids
    ENDIF
  ELSE
    n_solids = 0
    WRITE(jp_stdout,'("<'//cp_composition//'> has no <'//cp_solid//'> child element.")')
  ENDIF

  DO i_solid = 1, n_solids
    stkmx_solid => stkmxp_solids(i_solid)%ptr
    i_att = STKMX_getAttIdxByName(stkmx_solid, cp_file)
    IF (i_att > 0) THEN
      stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_solid, i_att)
      CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_xmlfilename, nlen_returned)
      WRITE(jp_stdout,'(" <'//cp_solid//'> ", I0, ": file=""",A,"""")') &
        i_solid, TRIM(c_xmlfilename)

      IF (n_xmlcompofiles == 0) THEN
        ! Now create the XFS_LIST for the XML composition files
        xml_compofiles  => XFSL_createRoot()
        xml_compofilewk => xml_compofiles
      ELSE
        xfs_work => xml_compofilewk
        xml_compofilewk => XFSL_addNode(xfs_work)
      ENDIF

      xml_compofilewk%fname = c_xmlfilename
      xml_compofilewk%c_type = cp_solid

      i_att = STKMX_getAttIdxByName(stkmx_solid, cp_order)
      IF (i_att > 0) THEN
        stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_solid, i_att)
        CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_attcntt, nlen_returned)
        READ(c_attcntt,*) i_order
        xml_compofilewk%i_order = i_order
      ELSE
        WRITE(jp_stdout,'(" <'//cp_solid//'> ", I0, ": ""'//cp_order//'"" &
                         &attribute missing -- using default")') i_solid
      ENDIF

      xml_compofilewk%i_system = 0 ! not part of a system

      n_xmlcompofiles = n_xmlcompofiles + 1

    ELSE

      WRITE(jp_stderr,c_fmterr_a, ADVANCE='NO') ''
      WRITE(jp_stderr,'(" <'//cp_solid//'> ", I0, ": ""'//cp_file//'"" &
                       &attribute missing -- aborting")') i_solid
      CALL ABORT()

    ENDIF
  ENDDO


  ! Solute systems

  WRITE(jp_stdout,'()')
  stkmxp_solutsysts => STKMX_getElementNodeByName(stkmx_composition, cp_solutesystem)
  
  IF (ASSOCIATED(stkmxp_solutsysts)) THEN
    n_solutsysts = SIZE(stkmxp_solutsysts)
    IF (n_solutsysts == 1) THEN
      WRITE(jp_stdout,'("<'//cp_composition//'> has 1 &
                        &<'//cp_solutesystem//'> child element")')
    ELSE
      WRITE(jp_stdout,'("<'//cp_composition//'> has ",I0, " &
                        &<'//cp_solutesystem//'> child elements")') n_solutsysts
    ENDIF
                 ! Allocate space to hold the STKXE and STKMX of each solutelist file
    ALLOCATE(stkxep_solutsystlists(n_solutsysts))
    ALLOCATE(stkmxp_solutsystlists(n_solutsysts))
  ELSE
    n_solutsysts = 0
    WRITE(jp_stdout,'("<'//cp_composition//'> has no &
                      &<'//cp_solutesystem//'> child element.")')
  ENDIF


  DO i_solutsyst = 1, n_solutsysts

    stkmx_solutsyst => stkmxp_solutsysts(i_solutsyst)%ptr

    i_att = STKMX_getAttIdxByName(stkmx_solutsyst, cp_file)

    IF (i_att > 0) THEN

      stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_solutsyst, i_att)
      CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_xmlfilename, nlen_returned)

      IF (n_xmlcompofiles == 0) THEN
        ! Now create the XFS_LIST for the XML composition files
        xml_compofiles  => XFSL_createRoot()
        xml_compofilewk => xml_compofiles
      ELSE
        xfs_work => xml_compofilewk
        xml_compofilewk => XFSL_addNode(xfs_work)
      ENDIF

      xml_compofilewk%fname = c_xmlfilename
      xml_compofilewk%c_type = cp_solutesystem

      i_att = STKMX_getAttIdxByName(stkmx_solutsyst, cp_order)
      IF (i_att > 0) THEN
        stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_solutsyst, i_att)
        CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_attcntt, nlen_returned)
        READ(c_attcntt,*) i_ordersolsyst
        xml_compofilewk%i_order = i_ordersolsyst
      ELSE
        i_order = -1
        WRITE(jp_stdout,'(" <'//cp_solutesystem//'> ", I0, ": ""'//cp_order//'"" &
                          &attribute missing -- using default")') i_solutsyst
      ENDIF

      xml_compofilewk%i_system = i_solutsyst ! yes, this one is part of a system

      n_xmlcompofiles = n_xmlcompofiles + 1


      INQUIRE(FILE=TRIM(c_xmlfilename), EXIST=l_exists)
      IF (.NOT. l_exists) THEN
        WRITE(jp_stderr, '()')
        WRITE(jp_stderr, c_fmterr_a, ADVANCE='NO') ''
        WRITE(jp_stderr,'("SoluteSystem file """, A, """ not found -- aborting!")') TRIM(c_xmlfilename)
        CALL ABORT()
      ENDIF

      WRITE(jp_stdout,'(" <'//cp_solutesystem//'> ", I0, ": resolving file=""",A,"""")') &
        i_solutsyst, TRIM(c_xmlfilename)

      stkxep_solutsystlists(i_solutsyst)%ptr &
        => XMLSTRUCT(c_xmlfilename, n_maxlen_eltname, n_maxlen_attname, n_maxlen_attcont)
      stkxe_solutsystlist => stkxep_solutsystlists(i_solutsyst)%ptr

      stkmxp_solutsystlists(i_solutsyst)%ptr &
        => XMLLOAD(c_xmlfilename, stkxe_solutsystlist, n_maxdepth)
      stkmx_solutsystlist => stkmxp_solutsystlists(i_solutsyst)%ptr

      stkmxp_members => STKMX_getElementNodeByName(stkmx_solutsystlist, cp_composition)

      IF (ASSOCIATED(stkmxp_members)) THEN

        n_composition = SIZE(stkmxp_members)

        IF (n_composition /= 1) THEN
          WRITE(jp_stderr,c_fmterr_a, ADVANCE='NO') ''
          WRITE(jp_stderr,'("  Found ", I0, " <'//cp_composition//'> elements!")') n_composition
          WRITE(jp_stderr,'("  there must be exactly one -- aborting!")')
          CALL ABORT()
        ENDIF

        WRITE(jp_stdout,'(A)') '  Found one <'//cp_composition//'> element.'

        stkmx_member => stkmxp_members(1)%ptr

        IF (ASSOCIATED(stkmxp_solutes)) DEALLOCATE(stkmxp_solutes)
        NULLIFY(stkmxp_solutes)

        stkmxp_solutes => STKMX_getElementNodeByName(stkmx_member, cp_solute)

        IF (ASSOCIATED(stkmxp_solutes)) THEN

          n_solutes = SIZE(stkmxp_solutes)

          IF (n_solutes == 1) THEN
            WRITE(jp_stdout,'("  <'//cp_composition//'> has 1 &
                                &<'//cp_solute//'> child element")')
          ELSE
            WRITE(jp_stdout,'("  <'//cp_composition//'> has ",I0, " &
                                &<'//cp_solute//'> child element(s)")') n_solutes
          ENDIF

        ELSE

          n_solutes = 0

          WRITE(jp_stdout,'("  <'//cp_composition//'> has no &
                              &<'//cp_solute//'> child element.")')
          WRITE(jp_stdout,'(A)') 'Discarding the current <'//cp_solutesystem//'> '
          xfs_work => xml_compofilewk
          xml_compofilewk => XFSL_deleteTailNode(xfs_work)
          n_xmlcompofiles = n_xmlcompofiles -1

        ENDIF

        DO i_solute = 1, n_solutes
          stkmx_solute => stkmxp_solutes(i_solute)%ptr
          i_att = STKMX_getAttIdxByName(stkmx_solute, cp_file)
          IF (i_att > 0) THEN
            stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_solute, i_att)
            CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_xmlfilename, nlen_returned)
            WRITE(jp_stdout,'("  <'//cp_solute//'> ", I0, ": file=""",A,"""")') &
              i_solute, TRIM(c_xmlfilename)

            IF (n_solutes /= 1) THEN
                                    ! If we get here, n_xmlcompofiles > 0
              xfs_work => xml_compofilewk
              xml_compofilewk => XFSL_addNode(xfs_work)

              xml_compofilewk%fname = c_xmlfilename
              xml_compofilewk%c_type = cp_solute

              i_att = STKMX_getAttIdxByName(stkmx_solute, cp_order)
              IF (i_att > 0) THEN
                stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_solute, i_att)
                CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_attcntt, nlen_returned)
                READ(c_attcntt,*) i_order
                xml_compofilewk%i_order = i_ordersolsyst + i_order
              ELSE
                i_order = -1
                xml_compofilewk%i_order = i_ordersolsyst
                WRITE(jp_stdout,'("  <'//cp_solute//'> ", I0, ": ""'//cp_order//'"" &
                                 &attribute missing")') i_solute
              ENDIF

              xml_compofilewk%i_system = i_solutsyst ! yes, this one is part of a system

              n_xmlcompofiles = n_xmlcompofiles + 1

            ELSE

              WRITE(jp_stdout,c_fmtwar_a) 'Found a single <'//cp_solute//'> element:'
              WRITE(jp_stdout,'(A)') 'discarding the current <'//cp_solutesystem//'> and '
              WRITE(jp_stdout,'(A)') 'replacing it by the single <'//cp_solute//'> element itself'
              WRITE(jp_stdout,'(A)') '(ignoring any "'//cp_order//'" attribute of the &
                                     &<'//cp_solute//'> element).'

              xml_compofilewk%fname  = c_xmlfilename
              xml_compofilewk%c_type = cp_solute

            ENDIF
            
          ELSE

            WRITE(jp_stderr,c_fmterr_a, ADVANCE='NO') ''
            WRITE(jp_stderr,'(" <'//cp_solute//'> ", I0, ": ""'//cp_file//'"" &
                             &attribute missing -- aborting")') i_solute
            CALL ABORT()

          ENDIF

        ENDDO

      ELSE

        WRITE(jp_stdout,c_fmtwar_a) 'No <'//cp_composition//'> element found.'
        WRITE(jp_stdout,'(A)') 'Discarding the current <'//cp_solutesystem//'> '
        xfs_work => xml_compofilewk
        xml_compofilewk => XFSL_deleteTailNode(xfs_work)
        n_xmlcompofiles = n_xmlcompofiles -1

      ENDIF

    ELSE

      WRITE(jp_stderr,c_fmterr_a, ADVANCE='NO') ''
      WRITE(jp_stderr,'(" <'//cp_solutesystem//'> ", I0, ": ""'//cp_file//'"" &
                       &attribute missing -- aborting")') i_solutsyst
      CALL ABORT()

    ENDIF

  ENDDO


ELSE

  WRITE(jp_stderr,c_fmterr_a) 'No <'//cp_composition//'> element found.'
  WRITE(jp_stderr,'(A)') 'Cannot continue without composition information -- aborting!'
  CALL ABORT()

ENDIF

WRITE(jp_stdout,'()')

                 ! Done with the composition file list
NULLIFY(xml_compofilewk)


                 !===========!
                 ! Processes !
                 !===========!

                 ! Initialize the list of files to read in for the
                 ! process descriptions

NULLIFY(xml_procsfiles)
n_xmlprocsfiles = 0

stkmxp_processes => STKMX_getElementNodeByName(stkmx_baselist, cp_processes)

WRITE(jp_stdout,'()')

IF (ASSOCIATED(stkmxp_processes)) THEN

  n_procs = SIZE(stkmxp_processes)

  IF (n_procs /= 1) THEN
    WRITE(jp_stderr,c_fmterr_a, ADVANCE='NO')
    WRITE(jp_stderr,'("Found ", I0, " <'//cp_processes//'> elements.")') n_procs
    WRITE(jp_stderr,'("There can be one at most -- aborting!")')
    CALL ABORT()
  ENDIF
  
  WRITE(jp_stdout,'(A)') 'Found one <'//cp_processes//'> element.'


  stkmx_processes => stkmxp_processes(1)%ptr

  ! Individual <"//cp_process//"> descriptions

  stkmxp_process => STKMX_getElementNodeByName(stkmx_processes, cp_process)

  WRITE(*,'()')
  
  IF (ASSOCIATED(stkmxp_process)) THEN
    n_procs = SIZE(stkmxp_process)
    IF (n_procs == 1) THEN
      WRITE(jp_stdout,'("<'//cp_processes//'> has 1 &
                        &<'//cp_process//'> child element")')
    ELSE
      WRITE(jp_stdout,'("<'//cp_processes//'> has ", I0, " &
                        &<'//cp_process//'> child elements")') n_procs
    ENDIF
  ELSE
    n_procs = 0
    WRITE(jp_stdout,'("<'//cp_processes//'> has no &
                      &<'//cp_process//'> child element.")')
  ENDIF

  DO i_proc = 1, n_procs
    stkmx_process => stkmxp_process(i_proc)%ptr
    i_att = STKMX_getAttIdxByName(stkmx_process, cp_file)
    IF (i_att > 0) THEN
      stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_process, i_att)
      CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_xmlfilename, nlen_returned)
      WRITE(jp_stdout,'(" <'//cp_process//'> ", I0, ": file=""",A,"""")') &
        i_proc, TRIM(c_xmlfilename)

      IF (n_xmlprocsfiles == 0) THEN
        ! Now create the XFS_LIST for the XML process files
        xml_procsfiles => XFSL_createRoot()
        xml_procsfilewk => xml_procsfiles
      ELSE
        xfs_work => xml_procsfilewk
        xml_procsfilewk => XFSL_addNode(xfs_work)
      ENDIF
      xml_procsfilewk%fname = c_xmlfilename
      xml_procsfilewk%c_type = cp_process
  
      n_xmlprocsfiles = n_xmlprocsfiles + 1
      
    ELSE

      WRITE(jp_stderr,c_fmterr_a, ADVANCE='NO') ''
      WRITE(jp_stderr,'(" <'//cp_process//'> ", I0, ": ""'//cp_file//'"" &
                       &attribute missing -- aborting")') i_proc
      CALL ABORT()

    ENDIF

  ENDDO

ELSE

  WRITE(jp_stdout,c_fmtwar_a) 'No <'//cp_processes//'> element found!'

ENDIF

WRITE(jp_stdout,'()')

                 ! Done with the process file list
NULLIFY(xml_procsfilewk)


                 !============!
                 ! Equilibria !
                 !============!

                 ! Initialize the list of files to read in for the
                 ! equilibrium descriptions

NULLIFY(xml_equilfiles)
n_xmlequilfiles = 0

stkmxp_equilibria => STKMX_getElementNodeByName(stkmx_baselist, cp_equilibria)

WRITE(jp_stdout,'()')

IF (ASSOCIATED(stkmxp_equilibria)) THEN

  n_equils = SIZE(stkmxp_equilibria)

  IF (n_equils /= 1) THEN
    WRITE(jp_stderr,c_fmterr_a, ADVANCE='NO') ''
    WRITE(jp_stderr,'("Found ",I0, " <'//cp_equilibria//'> elements.")') n_equils
    WRITE(jp_stderr,'("There can be one at most -- aborting!")')
    CALL ABORT()
  ENDIF

  WRITE(jp_stdout,'(A)') 'Found one <'//cp_equilibria//'> element.'


  stkmx_equilibria => stkmxp_equilibria(1)%ptr


  ! Individual <Equilibrium> descriptions

  stkmxp_equilibrium => STKMX_getElementNodeByName(stkmx_equilibria, cp_equilibrium)

  WRITE(jp_stdout,'()')
  
  IF (ASSOCIATED(stkmxp_equilibrium)) THEN
    n_equils = SIZE(stkmxp_equilibrium)
    IF (n_equils == 1) THEN
      WRITE(jp_stdout,'("<'//cp_equilibria//'> has 1 &
                        &<'//cp_equilibrium//'> child element")')
    ELSE
      WRITE(jp_stdout,'("<'//cp_equilibria//'> has ", I0, " &
                        &<'//cp_equilibrium//'> child elements")') n_equils
    ENDIF
  ELSE
    n_equils = 0
    WRITE(jp_stdout,'("<'//cp_equilibria//'> has no &
                      &<'//cp_equilibrium//'> child element.")')
  ENDIF

  DO i_equil = 1, n_equils
    stkmx_equilibrium => stkmxp_equilibrium(i_equil)%ptr
    i_att = STKMX_getAttIdxByName(stkmx_equilibrium, cp_file)
    IF (i_att > 0) THEN
      stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_equilibrium, i_att)
      CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_xmlfilename, nlen_returned)
      WRITE(jp_stdout,'(" <'//cp_equilibrium//'> ", I0, ": file=""", A, """")') &
        i_equil, TRIM(c_xmlfilename)

      IF (n_xmlequilfiles == 0) THEN
        xml_equilfiles  => XFSL_createRoot()
        xml_equilfilewk => xml_equilfiles
      ELSE
        xfs_work => xml_equilfilewk
        xml_equilfilewk => XFSL_addNode(xfs_work)
      ENDIF
      xml_equilfilewk%fname = c_xmlfilename
      xml_equilfilewk%c_type = cp_equilibrium
  
      n_xmlequilfiles = n_xmlequilfiles + 1
      
    ELSE

      WRITE(jp_stderr,c_fmterr_a, ADVANCE='NO') ''
      WRITE(jp_stderr,'(" <'//cp_equilibrium//'> ", I0, ": ""'//cp_file//'"" &
                       &attribute missing -- aborting")') i_equil
      CALL ABORT()

    ENDIF

  ENDDO

ELSE

  WRITE(jp_stdout,c_fmtwar_a) 'No <'//cp_equilibria//'> element found!'

ENDIF

                 ! Done with the equilibrium file list
NULLIFY(xml_equilfilewk)


                 !================!
                 ! API Extensions !
                 !================!

                 ! Initialize the list of files to read in for the
                 ! API Extension descriptions

NULLIFY(xml_apiexfiles)
n_xmlapiexfiles = 0

stkmxp_apiextensions => STKMX_getElementNodeByName(stkmx_baselist, cp_apiextensions)

WRITE(jp_stdout,'()')

IF (ASSOCIATED(stkmxp_apiextensions)) THEN

  n_apiexs = SIZE(stkmxp_apiextensions)

  IF (n_apiexs /= 1) THEN
    WRITE(jp_stderr,c_fmterr_a, ADVANCE='NO') ''
    WRITE(jp_stderr,'("Found ",I0, " <'//cp_apiextensions//'> elements.")') n_apiexs
    WRITE(jp_stderr,'("There can be one at most -- aborting!")')
    CALL ABORT()
  ENDIF

  WRITE(jp_stdout,'(A)') 'Found one <'//cp_apiextensions//'> element.'


  stkmx_apiextensions => stkmxp_apiextensions(1)%ptr


  ! Individual <APIExtension> descriptions

  stkmxp_apiextension => STKMX_getElementNodeByName(stkmx_apiextensions, cp_apiextension)

  WRITE(jp_stdout,'()')
  
  IF (ASSOCIATED(stkmxp_apiextension)) THEN
    n_apiexs = SIZE(stkmxp_equilibrium)
    IF (n_apiexs == 1) THEN
      WRITE(jp_stdout,'("<'//cp_apiextensions//'> has 1 &
                        &<'//cp_apiextension//'> child element")')
    ELSE
      WRITE(jp_stdout,'("<'//cp_apiextensions//'> has ", I0, " &
                        &<'//cp_apiextension//'> child elements")') n_equils
    ENDIF
  ELSE
    n_apiexs = 0
    WRITE(jp_stdout,'("<'//cp_apiextensions//'> has no &
                      &<'//cp_apiextension//'> child element.")')
  ENDIF

  DO i_apiex = 1, n_apiexs
    stkmx_apiextension => stkmxp_apiextension(i_apiex)%ptr
    i_att = STKMX_getAttIdxByName(stkmx_apiextension, cp_file)
    IF (i_att > 0) THEN
      stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_apiextension, i_att)
      CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_xmlfilename, nlen_returned)
      WRITE(jp_stdout,'(" <'//cp_apiextension//'> ", I0, ": file=""", A, """")') &
        i_apiex, TRIM(c_xmlfilename)

      IF (n_xmlapiexfiles == 0) THEN
        xml_apiexfiles  => XFSL_createRoot()
        xml_apiexfilewk => xml_apiexfiles
      ELSE
        xfs_work => xml_apiexfilewk
        xml_apiexfilewk => XFSL_addNode(xfs_work)
      ENDIF
      xml_apiexfilewk%fname = c_xmlfilename
      xml_apiexfilewk%c_type = cp_apiextension
  
      n_xmlapiexfiles = n_xmlapiexfiles + 1
      
    ELSE

      WRITE(jp_stderr,c_fmterr_a, ADVANCE='NO') ''
      WRITE(jp_stderr,'(" <'//cp_apiextension//'> ", I0, ": ""'//cp_file//'"" &
                       &attribute missing -- aborting")') i_apiex
      CALL ABORT()

    ENDIF

  ENDDO

ELSE

  WRITE(jp_stdout,c_fmtwar_a) 'No <'//cp_apiextensions//'> element found!'

ENDIF

                 ! Done with the equilibrium file list
NULLIFY(xml_apiexfilewk)


WRITE(jp_stdout,'()')
WRITE(jp_stdout,'("Number of files in each category:")')

WRITE(jp_stdout,'(" Composition files: ", I0)')  n_xmlcompofiles
WRITE(jp_stdout,'(" Process files: ", I0)')      n_xmlprocsfiles
WRITE(jp_stdout,'(" Equilibrium files: ", I0)')  n_xmlequilfiles
WRITE(jp_stdout,'(" API Extension files: ", I0)')  n_xmlapiexfiles
#ifdef CFG_DEBUG
WRITE(jp_stddbg,'()')
WRITE(jp_stddbg,c_fmtdbg_a) 'Number of files in each category:'

WRITE(jp_stddbg,'(" Composition files: ", I0)')  n_xmlcompofiles
WRITE(jp_stddbg,'(" Process files: ", I0)')      n_xmlprocsfiles
WRITE(jp_stddbg,'(" Equilibrium files: ", I0)')  n_xmlequilfiles
WRITE(jp_stddbg,'(" API Extension files: ", I0)')  n_xmlapiexfiles
#endif

IF (n_xmlcompofiles /= 0) THEN
  WRITE(jp_stdout,'()')
  WRITE(jp_stdout,'()')
  WRITE(jp_stdout,'("Composition files")')
  WRITE(jp_stdout,'("-----------------")')
  WRITE(jp_stdout,'()')
  WRITE(jp_stdout,'("Unsorted list")')


  xml_compofilewk => xml_compofiles

  DO

    WRITE(jp_stdout,'("fname = """, A, """, order = ", I0)') &
      TRIM(xml_compofilewk%fname), xml_compofilewk%i_order
    IF (ASSOCIATED(xml_compofilewk%next)) THEN
      xml_compofilewk => xml_compofilewk%next
    ELSE
      EXIT
    ENDIF

  ENDDO



                 ! Sort the list by increasing %i_order

  i_ordermin = xml_compofiles%i_order

  IF (ASSOCIATED(xml_compofiles%next)) THEN
    xfs_test => xml_compofiles%next
  ELSE
    NULLIFY(xfs_test)
  ENDIF


  DO WHILE(ASSOCIATED(xfs_test))

    IF (ASSOCIATED(xfs_test%next)) THEN
      xfs_testnext => xfs_test%next
    ELSE
      NULLIFY(xfs_testnext)
    ENDIF

    i_order = xfs_test%i_order

    IF (i_order < i_ordermin) THEN
                 ! The current <xfs_test> provisionally becomes the root
      i_ordermin = i_order

      ! unlink xfs_test from its %prev and its %next:
      ! - forward link the %prev of xfs_test to the %next of xfs_test
      ! - backward link the %next of xfs_test to the %prev of xfs_test
      IF (ASSOCIATED(xfs_testnext)) THEN
        xfs_test%prev%next => xfs_testnext
        xfs_testnext%prev  => xfs_test%prev
      ELSE
        NULLIFY(xfs_test%prev%next)
      ENDIF

      ! Forward link the current xml_test to xml_compofiles
      ! and backward link the current xml_compofiles to xml_test
      xml_compofiles%prev => xfs_test
      xfs_test%next => xml_compofiles
      ! Then substitute the xml_compofiles with the current xml_test
      xml_compofiles => xfs_test
      ! and nullify its %prev
      NULLIFY(xml_compofiles%prev)

    ELSE

                 ! i_order is greater than the lowest value encountered
                 ! so far and will have to be inserted somewhere after
                 ! the root node.

                 ! If xfs_test%prev has an %i_order that is greater
                 ! than i_order, then proceed to the tests, else
                 ! there is nothing to do
      xfs_work => xfs_test%prev

      IF (xfs_work%i_order > i_order) THEN

        DO WHILE(ASSOCIATED(xfs_work%prev))
          xfs_work => xfs_work%prev
          IF(xfs_work%i_order > i_order) THEN
            CYCLE
          ELSE
            ! the current <xfs_test> needs to be inserted
            ! between xfs_work and xfs_work%next

            ! unlink xfs_test from its %prev and its %next:
            ! - forward link the %prev of xfs_test to the %next of xfs_test
            ! - backward link the %next of xfs_test to the %prev of xfs_test
            IF (ASSOCIATED(xfs_testnext)) THEN
              xfs_test%prev%next => xfs_testnext
              xfs_testnext%prev  => xfs_test%prev
            ELSE
              NULLIFY(xfs_test%prev%next)
            ENDIF
          
            ! insert xfs_test between xfs_work and xfs_work%next:
            ! - forward link xfs_test to xfs_work%next and xfs_work to xfs_test and
            ! - backward link xfs_work%next to xfs_test and xfs_test to xfs_work
            xfs_test%next => xfs_work%next
            xfs_test%prev => xfs_work
            xfs_test%next%prev => xfs_test
            xfs_test%prev%next => xfs_test

            ! and we are done
            EXIT
          ENDIF
        ENDDO

      ENDIF

    ENDIF

    xfs_test => xfs_testnext

  ENDDO



  xml_compofilewk => xml_compofiles
  WRITE(jp_stdout,'()')
  WRITE(jp_stdout,'("Sorted list")')


  DO

    WRITE(jp_stdout,'("fname = """, A, """, order = ", I0)', ADVANCE="NO") &
      TRIM(xml_compofilewk%fname), xml_compofilewk%i_order
    INQUIRE(FILE=TRIM(xml_compofilewk%fname), EXIST=l_exists)
    IF (l_exists) THEN
      WRITE(jp_stdout, '(": found")')
    ELSE
      WRITE(jp_stdout, '(": missing")')
      n_missingfiles = n_missingfiles + 1
    ENDIF
    IF (ASSOCIATED(xml_compofilewk%next)) THEN
      xml_compofilewk => xml_compofilewk%next
    ELSE
      EXIT
    ENDIF

  ENDDO

ENDIF


IF (n_xmlprocsfiles /= 0) THEN
  WRITE(jp_stdout,'()')
  WRITE(jp_stdout,'()')
  WRITE(jp_stdout,'("Process files")')
  WRITE(jp_stdout,'("-------------")')
  WRITE(jp_stdout,'()')


  xml_procsfilewk => xml_procsfiles

  DO

    WRITE(jp_stdout,'("fname = """, A, """")', ADVANCE="NO") TRIM(xml_procsfilewk%fname)
    INQUIRE(FILE=TRIM(xml_procsfilewk%fname), EXIST=l_exists)
    IF (l_exists) THEN
      WRITE(jp_stdout, '(": found")')
    ELSE
      WRITE(jp_stdout, '(": missing")')
      n_missingfiles = n_missingfiles + 1
    ENDIF
     
    IF (ASSOCIATED(xml_procsfilewk%next)) THEN
      xml_procsfilewk => xml_procsfilewk%next
    ELSE
      EXIT
    ENDIF

  ENDDO

ENDIF


IF (n_xmlequilfiles /= 0) THEN
  WRITE(*,'()')
  WRITE(*,'()')
  WRITE(*,'("Equilibrium files")')
  WRITE(*,'("-----------------")')
  WRITE(*,'()')


  xml_equilfilewk => xml_equilfiles

  DO

    WRITE(*,'("fname = """, A, """")', ADVANCE="NO") TRIM(xml_equilfilewk%fname)
    INQUIRE(FILE=TRIM(xml_equilfilewk%fname), EXIST=l_exists)
    IF (l_exists) THEN
      WRITE(jp_stdout, '(": found")')
    ELSE
      WRITE(jp_stdout, '(": missing")')
      n_missingfiles = n_missingfiles + 1
    ENDIF
    IF (ASSOCIATED(xml_equilfilewk%next)) THEN
      xml_equilfilewk => xml_equilfilewk%next
    ELSE
      EXIT
    ENDIF

  ENDDO

ENDIF


IF (n_xmlapiexfiles /= 0) THEN
  WRITE(*,'()')
  WRITE(*,'()')
  WRITE(*,'("API Extension files")')
  WRITE(*,'("-------------------")')
  WRITE(*,'()')


  xml_apiexfilewk => xml_apiexfiles

  DO

    WRITE(*,'("fname = """, A, """")', ADVANCE="NO") TRIM(xml_apiexfilewk%fname)
    INQUIRE(FILE=TRIM(xml_apiexfilewk%fname), EXIST=l_exists)
    IF (l_exists) THEN
      WRITE(jp_stdout, '(": found")')
    ELSE
      WRITE(jp_stdout, '(": missing")')
      n_missingfiles = n_missingfiles + 1
    ENDIF
    IF (ASSOCIATED(xml_apiexfilewk%next)) THEN
      xml_apiexfilewk => xml_apiexfilewk%next
    ELSE
      EXIT
    ENDIF

  ENDDO

ENDIF


IF (n_missingfiles > 0) THEN

  WRITE(jp_stdout,'()')
  WRITE(jp_stdout,'()')
  WRITE(jp_stderr, c_fmterr_a, ADVANCE="NO") ''
  IF (n_missingfiles == 1) THEN
    WRITE(jp_stderr, '("One of the requested XML files is missing!")')
  ELSE
    WRITE(jp_stderr, '(I0, " of the requested XML files are missing!")')
  ENDIF

  WRITE(jp_stderr, '("Please check the list in the log for information.")')
  WRITE(jp_stderr, '("Aborting!")')

  CALL ABORT()

ENDIF
  



! Read in the data

IF (n_xmlcompofiles /= 0) THEN
  WRITE(jp_stdout,'()')
  WRITE(jp_stdout,'()')
  WRITE(jp_stdout,'("Reading in the composition files")')

  xfs_work => xml_compofiles

  DO

    WRITE(jp_stdout,'(" - file """, A, """")') TRIM(xfs_work%fname)
#   ifdef CFG_DEBUG
    WRITE(jp_stddbg,'(" - file """, A, """")') TRIM(xfs_work%fname)
#   endif
    xfs_work%xe => XMLSTRUCT(xfs_work%fname, &
      n_maxlen_eltname, n_maxlen_attname, n_maxlen_attcont)
  
    xfs_work%mx => XMLLOAD(xfs_work%fname, xfs_work%xe, n_maxdepth)

    IF (ASSOCIATED(xfs_work%next)) THEN
      xfs_work => xfs_work%next
    ELSE
      EXIT
    ENDIF

  ENDDO

ELSE

  WRITE(jp_stdout,'("No composition files registered")')

ENDIF


WRITE(jp_stdout,'()')
IF (n_xmlprocsfiles /= 0) THEN
  WRITE(jp_stdout,'("Reading in the process files")')

  xfs_work => xml_procsfiles

  DO

    WRITE(jp_stdout,'(" - file """, A, """")') TRIM(xfs_work%fname)
#  ifdef CFG_DEBUG
    WRITE(jp_stddbg,'(" - file """, A, """")') TRIM(xfs_work%fname)
#  endif
    xfs_work%xe => XMLSTRUCT(xfs_work%fname, &
      n_maxlen_eltname, n_maxlen_attname, n_maxlen_attcont)
  
    xfs_work%mx => XMLLOAD(xfs_work%fname, xfs_work%xe, n_maxdepth)

    IF (ASSOCIATED(xfs_work%next)) THEN
      xfs_work => xfs_work%next
    ELSE
      EXIT
    ENDIF

  ENDDO

ELSE

  WRITE(jp_stdout,'("No process files registered")')

ENDIF


WRITE(jp_stdout,'()')
IF (n_xmlequilfiles /= 0) THEN
  WRITE(jp_stdout,'("Reading in the equilibrium files")')

  xfs_work => xml_equilfiles

  DO

    WRITE(jp_stdout,'(" - file """, A, """")') TRIM(xfs_work%fname)
#  ifdef CFG_DEBUG
    WRITE(jp_stddbg,'(" - file """, A, """")') TRIM(xfs_work%fname)
#  endif
    xfs_work%xe => XMLSTRUCT(xfs_work%fname, &
      n_maxlen_eltname, n_maxlen_attname, n_maxlen_attcont)
  
    xfs_work%mx => XMLLOAD(xfs_work%fname, xfs_work%xe, n_maxdepth)

    IF (ASSOCIATED(xfs_work%next)) THEN
      xfs_work => xfs_work%next
    ELSE
      EXIT
    ENDIF

  ENDDO

ELSE

  WRITE(jp_stdout,'("No equilibrium files registered")')

ENDIF


WRITE(jp_stdout,'()')
IF (n_xmlapiexfiles /= 0) THEN
  WRITE(jp_stdout,'("Reading in the API Extension files")')

  xfs_work => xml_apiexfiles

  DO

    WRITE(jp_stdout,'(" - file """, A, """")') TRIM(xfs_work%fname)
#  ifdef CFG_DEBUG
    WRITE(jp_stddbg,'(" - file """, A, """")') TRIM(xfs_work%fname)
#  endif
    xfs_work%xe => XMLSTRUCT(xfs_work%fname, &
      n_maxlen_eltname, n_maxlen_attname, n_maxlen_attcont)
  
    xfs_work%mx => XMLLOAD(xfs_work%fname, xfs_work%xe, n_maxdepth)

    IF (ASSOCIATED(xfs_work%next)) THEN
      xfs_work => xfs_work%next
    ELSE
      EXIT
    ENDIF

  ENDDO

ELSE

  WRITE(jp_stdout,'("No API Extension files registered")')

ENDIF


!CALL XFSL_dumpList(xml_compofiles, 99)
!WRITE(99,'()')
!WRITE(99,'()')
!CALL XFSL_dumpList(xml_procsfiles, 99)
!WRITE(99,'()')
!WRITE(99,'()')
!CALL XFSL_dumpList(xml_equilfiles, 99)


WRITE(jp_stdout,c_fmtinf_a) 'completed'
#ifdef CFG_DEBUG
WRITE(jp_stddbg,c_fmtdbg_a) 'completed'
#endif



RETURN

!===============================================================================
 END SUBROUTINE XMLCOCOGEN_LOADDB
!===============================================================================
