!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#include "configure.h"
!---+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+----0----+----1----+----2----+----3--
SUBROUTINE NORMALIZE_SOURCECODE(c_file_in, c_file_out)
! Reads in code lines from a Fortran fixed format source file
! with possibly too long lines and writes them to another
! file with problematic lines folded at spaces, '+', '-', '*', '/' '(' or ')' characters.
! It is assumed that character strings do not run over multiple lines

USE mod_medusa_cocogen, ONLY: n_lxxlcodeline, n_lmaxfixformlength

IMPLICIT NONE


CHARACTER(LEN=*), INTENT(IN) :: c_file_in, c_file_out

INTEGER, PARAMETER :: n_lengthstretch     =  7 ! must be larger than 3

INTEGER, PARAMETER :: i_unit_in = CFG_TMP1UNIT
INTEGER, PARAMETER :: i_unit_out = CFG_TMP2UNIT

CHARACTER(LEN=n_lxxlcodeline):: c_codeline
INTEGER :: n_lineno_in
INTEGER :: n_lineno_out
INTEGER :: n_linelen
INTEGER :: i, i_sinquote, i_dblquote, i_cstart, i_cend
INTEGER :: i_nobreak, i_bestbreak
INTEGER :: i_cspace, i_ccomma, i_cplus, i_cminus, i_ctimes, i_cdiv
INTEGER :: i_cparleft, i_cparright
CHARACTER(LEN=1):: c_cstartchar
LOGICAL :: l_copen



! Standard I/O related data
! -------------------------

INTEGER,          PARAMETER :: jp_stdout  = CFG_STDOUT
!~ INTEGER,          PARAMETER :: jp_stderr  = CFG_STDERR
CHARACTER(LEN=*), PARAMETER :: cp_fmtinf_a = '("[NORMALIZE_SOURCECODE]: ", A)'
!~ CHARACTER(LEN=*), PARAMETER :: cp_fmtwar_a = '("[NORMALIZE_SOURCECODE] warning: ", A)'
!~ CHARACTER(LEN=*), PARAMETER :: cp_fmterr_a = '("[NORMALIZE_SOURCECODE] error: ", A)'

#ifdef CFG_DEBUG
INTEGER,          PARAMETER :: jp_stddbg  = CFG_STDDBG
CHARACTER(LEN=*), PARAMETER :: cp_fmtdbg_a = '("DEBUG [NORMALIZE_SOURCECODE]: ", A)'
#endif


!=====================!
! End of declarations !
!=====================!


!============!
! Operations !
!============!

#ifdef CFG_VERBOSE
#if (CFG_VERBOSELEVEL == 2)
WRITE(jp_stdout,'()') 
WRITE(jp_stdout,cp_fmtinf_a) 'starting to process "' // TRIM(c_file_in) // '"'
#endif
#endif
#ifdef CFG_DEBUG
WRITE(jp_stddbg,'()') 
WRITE(jp_stddbg,cp_fmtdbg_a) 'starting to process "' // TRIM(c_file_in) // '"'
#endif

OPEN(UNIT=i_unit_in,  FILE=c_file_in)
OPEN(UNIT=i_unit_out, FILE=c_file_out)

n_lineno_in = 0
n_lineno_out = 0

l_copen = .FALSE.
i_cstart = 0

readline: DO
  READ(i_unit_in,'(A)',END=995) c_codeline
  n_lineno_in = n_lineno_in+1

  processline: DO
    IF(.NOT.l_copen) THEN
    n_linelen = LEN_TRIM(c_codeline)
    IF(n_linelen <= n_lmaxfixformlength) THEN
      WRITE(i_unit_out, '(A)') TRIM(c_codeline)
      n_lineno_out = n_lineno_out+1
      CYCLE readline
    ELSE
#    ifdef CFG_DEBUG
#    if (CFG_DEBUGLEVEL > 0)
      WRITE(jp_stddbg,cp_fmtdbg_a,ADVANCE='NO')
      WRITE(jp_stddbg,"(A,I0,A,I0)") "File '" // TRIM(c_file_in) // "': line ", &
                            n_lineno_in, " has length ", n_linelen
#    endif
#    endif
! Line too long
! check the following:
! - Comment line ? Action: copy as is
!   'c', 'C' or '*' in column 1 ?
      SELECT CASE(c_codeline(1:1))
      CASE('*', 'c', 'C')
        WRITE(i_unit_out, '(A)') TRIM(c_codeline)
        n_lineno_out = n_lineno_out+1
        CYCLE readline
      CASE DEFAULT
#      ifdef CFG_DEBUG
#      if (CFG_DEBUGLEVEL > 0)
        WRITE(jp_stddbg,"(A)") "  ... not a comment line starting with 'c', 'C' or '*' in column 1"
#      endif
#      endif
        CONTINUE
      END SELECT

!     '!' in columns 1-5 ('!' in column 6 marks a continuation line)
      i = INDEX(c_codeline(1:5), '!')
      IF(i > 0) THEN
#      ifdef CFG_DEBUG
#      if (CFG_DEBUGLEVEL > 0)
        WRITE(jp_stddbg,"(A)") "  ... comment line starting with '!' in columns 1-5: copying as is"
#      endif
#      endif
        WRITE(i_unit_out, '(A)') TRIM(c_codeline)
        n_lineno_out = n_lineno_out+1
        CYCLE readline
      ENDIF
#    ifdef CFG_DEBUG
#    if (CFG_DEBUGLEVEL > 0)
      WRITE(jp_stddbg,"(A)") "  ... not a comment line starting with '!' in columns 1-5"
#    endif
#    endif

! Check for presence of character string constant

      i_cend = 6
      i_bestbreak = 6
      i_nobreak = 6

      ENDIF

      searchnewstring: DO

      IF(.NOT.l_copen) THEN
#      ifdef CFG_DEBUG
#      if (CFG_DEBUGLEVEL > 0)
        WRITE(jp_stddbg,"(A)") '  ... no character string constant open -- checking for one'
#      endif
#      endif
        i_sinquote = INDEX(c_codeline(i_cend+1:), "'")
        i_dblquote = INDEX(c_codeline(i_cend+1:), '"')
        i          = INDEX(c_codeline(i_cend+1:), '!') ! check if there is a '!' on the same
                                                       ! line. If it preceeds any single or
                                                       ! double quote, there is no character
                                                       ! string constant on the line

        IF((i_sinquote > 0) .AND. (i_dblquote > 0)) THEN
! There are ' and " characters, so the first one set is the driving one
#        ifdef CFG_DEBUG
#        if (CFG_DEBUGLEVEL > 0)
          WRITE(jp_stddbg,"(A,I0,A,I0)",ADVANCE='NO') &
            '  ... found '' at column ', i_sinquote+i_cend, &
            ' and " at column ', i_dblquote+i_cend
#        endif
#        endif

          IF(i > 0) THEN            ! however, there is also a !
#          ifdef CFG_DEBUG
#          if (CFG_DEBUGLEVEL > 0)
            WRITE(jp_stddbg,"(A,I0)") ', and also ! at column ', i+i_cend
#          endif
#          endif
            IF((i < i_sinquote) .AND. (i < i_dblquote)) THEN
                                    ! it preceeds both ' and ", so these are
                                    ! inside a comment
              i_cstart = 0
              l_copen = .FALSE.
            ELSE
              i_cstart = MIN(i_sinquote,i_dblquote)+i_cend
              c_cstartchar = c_codeline(i_cstart:i_cstart)
              l_copen = .TRUE.
            ENDIF
          ELSE
#          ifdef CFG_DEBUG
#          if (CFG_DEBUGLEVEL > 0)
            WRITE(jp_stddbg,"()")
#          endif
#          endif
            i_cstart = MIN(i_sinquote,i_dblquote)+i_cend
            c_cstartchar = c_codeline(i_cstart:i_cstart)
            l_copen = .TRUE.
          ENDIF

        ELSEIF(i_sinquote > 0) THEN ! There is only a ' character
#        ifdef CFG_DEBUG
#        if (CFG_DEBUGLEVEL > 0)
          WRITE(jp_stddbg,"(A,I0)",ADVANCE='NO') '  ... found '' at column ', i_sinquote+i_cend
#        endif
#        endif

          IF(i > 0) THEN            ! however, there is also a !
#          ifdef CFG_DEBUG
#          if (CFG_DEBUGLEVEL > 0)
            WRITE(jp_stddbg,"(A,I0)") ' and ! at column ', i+i_cend
#          endif
#          endif

            IF((i < i_sinquote)) THEN
                                    ! the ! preceeds the ' character,
                                    ! which is thus inside a comment
              i_cstart = 0
              c_cstartchar = 'X'
              l_copen = .FALSE.
            ELSE 
                                    ! the ! is inside the character string,
                                    ! which thus starts at the ' character
              i_cstart = i_sinquote+i_cend
              c_cstartchar = "'"
              l_copen = .TRUE.
            ENDIF

          ELSE
#          ifdef CFG_DEBUG
#          if (CFG_DEBUGLEVEL > 0)
            WRITE(jp_stddbg,"()")
#          endif
#          endif

            i_cstart = i_sinquote+i_cend
            c_cstartchar = "'"
            l_copen = .TRUE.

          ENDIF

        ELSEIF(i_dblquote > 0) THEN ! There is only a " character
#        ifdef CFG_DEBUG
#        if (CFG_DEBUGLEVEL > 0)
          WRITE(jp_stddbg,"(A,I0)",ADVANCE='NO') '  ... found " at column ', i_dblquote+i_cend
#        endif
#        endif
          IF(i > 0) THEN
                                    ! however, there is also a !
#          ifdef CFG_DEBUG
#          if (CFG_DEBUGLEVEL > 0)
            WRITE(jp_stddbg,"(A,I0)") ' and ! at column ', i+i_cend
#          endif
#          endif
            IF((i < i_dblquote)) THEN
                                    ! the ! preceeds the ' character,
                                    ! which is thus inside a comment
              i_cstart = 0
              c_cstartchar = 'X'
              l_copen = .FALSE.
            ELSE
                                    ! the ! is inside the character string,
                                    ! which thus starts at the " character
              i_cstart = i_dblquote+i_cend
              c_cstartchar = '"'
              l_copen = .TRUE.
            ENDIF

          ELSE
#          ifdef CFG_DEBUG
#          if (CFG_DEBUGLEVEL > 0)
            WRITE(jp_stddbg,"()")
#          endif
#          endif

            i_cstart = i_dblquote+i_cend
            c_cstartchar = '"'
            l_copen = .TRUE.

          ENDIF

        ELSE

          i_cstart = 0
          c_cstartchar = 'X'

        ENDIF

      ELSE

#      ifdef CFG_DEBUG
#      if (CFG_DEBUGLEVEL > 0)
        WRITE(jp_stddbg,"(A)") "  ... character string constant already open"
#      endif
#      endif
        i_cstart = 6

      ENDIF


      IF(i_cstart > n_lmaxfixformlength) THEN
! There is a character string constant on the line, but only after the last allowed column;
! we ignore it for the time being
        l_copen = .FALSE.
      ENDIF

      IF(l_copen) THEN
#      ifdef CFG_DEBUG
#      if (CFG_DEBUGLEVEL > 0)
        WRITE(jp_stddbg,"(A,I0)") "  ... character string constant starting at column ", i_cstart
#      endif
#      endif
! Search for end of the character string constant
        i = i_cstart
        DO
          i = i+1
          i_cend = INDEX(c_codeline(i:), c_cstartchar)+i-1
          IF(i_cend > (i-1)) THEN
! Found first matching c_cstartchar.
! Check if there is no second one immediately following (escaped delimiter)
! in which case the end comes only afterwards
            IF(c_codeline(i_cend+1:i_cend+1) == c_cstartchar) THEN
              IF(i < (n_lmaxfixformlength-3)) THEN
                i_nobreak = i
                i_bestbreak = i+1
              ENDIF
              i = i_cend+1
              CYCLE
            ENDIF
! We have found the end of the character constant
! It extends over columns i_cstart:i_cend
            EXIT
          ELSE
! Oops, end of character string constant not on this line hmmmm
#          ifdef CFG_DEBUG
#          if (CFG_DEBUGLEVEL > 0)
            WRITE(jp_stddbg,"(A)") "  ... character string constant not closed on current line"
            WRITE(jp_stddbg,"(A)") "  ... writing out line unchanged to '" // TRIM(c_file_out)
#          endif
#          endif
            WRITE(i_unit_out, '(A)') TRIM(c_codeline)
            n_lineno_out = n_lineno_out+1
#          ifdef CFG_DEBUG
            WRITE(jp_stddbg,"(A,I0)") "  Manual action required at line ", n_lineno_out
#          endif
            WRITE(jp_stdout,'("Output file ", A,":")') '''' // TRIM(c_file_out) // ''''
            WRITE(jp_stdout,'("  manual action required at line ",I0)') n_lineno_out
            CYCLE readline
          ENDIF
        ENDDO

#      ifdef CFG_DEBUG
#      if (CFG_DEBUGLEVEL > 0)
        WRITE(jp_stddbg,"(A,I0,':',I0)") "  ... character string constant extending over columns " , &
                                        i_cstart,i_cend
#      endif
#      endif

        IF(i_cend <= n_lmaxfixformlength) THEN
#        ifdef CFG_DEBUG
#        if (CFG_DEBUGLEVEL > 0)
          WRITE(jp_stddbg,"(A)") "  ... character string constant ending before end of the line"
#        endif
#        endif
! The character string constant ends before the last allowed column
! Mark no currently opened character string constant
          l_copen = .FALSE.
! if the end of the character string constant is close to the end, break here
          IF(i_cend > (n_lmaxfixformlength-n_lengthstretch)) THEN
#          ifdef CFG_DEBUG
#          if (CFG_DEBUGLEVEL > 0)
            WRITE(jp_stddbg,"(A, I0, A)") "  ... character string constant ends after end of right stretch margin (= column ", &
              n_lmaxfixformlength-n_lengthstretch, ")"
            WRITE(jp_stddbg,"(A,I0,A)") "  ... breaking line after column ", i_cend, " (close to end)"
#          endif
#          endif
            WRITE(i_unit_out, '(A)') TRIM(c_codeline(:i_cend))
            n_lineno_out = n_lineno_out+1
            c_codeline = '     &      ' // c_codeline(i_cend+1:)
!~ #          ifdef CFG_DEBUG
!~             WRITE(jp_stddbg,"(A)") "  ... continuation line: <" // c_codeline // "> to reanalyse"
!~ #          endif
            CYCLE processline
          ELSE
! Save the end character as a potential best break point
            i_bestbreak = i_cend
            CYCLE searchnewstring
          ENDIF

        ELSE
! We have to break the line inside the character string constant
! except if the character string constant starts close to the end
#        ifdef CFG_DEBUG
#        if (CFG_DEBUGLEVEL > 0)
          WRITE(jp_stddbg,"(A)") "  ... character string constant ending after end of the line"
#        endif
#        endif
          IF(i_cstart > (n_lmaxfixformlength-n_lengthstretch)) THEN
#          ifdef CFG_DEBUG
#          if (CFG_DEBUGLEVEL > 0)
            WRITE(jp_stddbg,"(A, I0, A)") "  ... character string constant starting after end of right stretch margin (= column ", &
              n_lmaxfixformlength-n_lengthstretch, ")"
            WRITE(jp_stddbg,"(A,I0,A)") "  ... breaking line after column ", &
              i_cstart-1, " (before start of character string constant)"
#          endif
#          endif
            WRITE(i_unit_out, '(A)') TRIM(c_codeline(:i_cstart-1))
            n_lineno_out = n_lineno_out+1
            c_codeline = '     &      ' // c_codeline(i_cstart:)
!~ #          ifdef CFG_DEBUG
!~             WRITE(jp_stddbg,"(A)") "  ... continuation line: <" // c_codeline // "> to reanalyse"
!~ #          endif
            l_copen = .FALSE.  ! none open, we split before the opening
            CYCLE processline

          ELSE

            i = n_lmaxfixformlength-3
            IF(i_nobreak /= i) THEN
#            ifdef CFG_DEBUG
#            if (CFG_DEBUGLEVEL > 0)
              WRITE(jp_stddbg,"(A,I0,A)") "  ... breaking line after column ", i, " (inside character string constant)"
#            endif
#            endif
              WRITE(i_unit_out, '(A)') c_codeline(:i) // c_cstartchar // "//"
                                    ! do not TRIM c_codeline(:i) here, as trailing blanks are significant:
                                    ! they belong to the character string we are breaking
               n_lineno_out = n_lineno_out+1
               c_codeline = '     &      ' // c_cstartchar // c_codeline(i+1:)
! Mark no currently opened character string constant; the "currently open" one has been closed
               l_copen = .FALSE.
               CYCLE processline
            ELSE
#            ifdef CFG_DEBUG
#            if (CFG_DEBUGLEVEL > 0)
              WRITE(jp_stddbg,"(A,I0)") "  ... breaking line after column ", i-1
#            endif
#            endif
              WRITE(i_unit_out, '(A)') c_codeline(:i-1) // c_cstartchar // "//"
                                    ! do not TRIM c_codeline(:i-1) here, as trailing blanks are significant:
                                    ! they belong to the character string we are breaking
               n_lineno_out = n_lineno_out+1
               c_codeline = '     &      ' // c_cstartchar // c_codeline(i:)
! Mark no currently opened character string constant; the "currently open" one has been closed
               l_copen = .FALSE.
               CYCLE processline
            ENDIF
          ENDIF
        ENDIF
        


      ELSE
! No character string constant on this line starting before last allowed column
! Checking for end of line comment
        i = INDEX(c_codeline, '!')
        IF((i > 6) .AND. (i < n_lmaxfixformlength)) THEN
#        ifdef CFG_DEBUG
#        if (CFG_DEBUGLEVEL > 0)
          WRITE(jp_stddbg,"(A)") "  ... end of line comment starting before last column: copying as is"
#        endif
#        endif
          WRITE(i_unit_out, '(A)') TRIM(c_codeline)
          n_lineno_out = n_lineno_out+1
          CYCLE readline
        ENDIF
! No end of line comment either before the last possible column
#      ifdef CFG_DEBUG
#      if (CFG_DEBUGLEVEL > 0)
        WRITE(jp_stddbg,"(A)") "  ... testing for SPC, comma, +, -, *, /, (, )"
#      endif
#      endif
        i_cspace    = INDEX(c_codeline(:n_lmaxfixformlength), ' ', BACK=.TRUE.)
        i_ccomma    = INDEX(c_codeline(:n_lmaxfixformlength), ',', BACK=.TRUE.)
        i_cplus     = INDEX(c_codeline(:n_lmaxfixformlength), '+', BACK=.TRUE.)
        i_cminus    = INDEX(c_codeline(:n_lmaxfixformlength), '-', BACK=.TRUE.)
        i_ctimes    = INDEX(c_codeline(:n_lmaxfixformlength), '*', BACK=.TRUE.)
        i_cdiv      = INDEX(c_codeline(:n_lmaxfixformlength), '/', BACK=.TRUE.)
        i_cparleft  = INDEX(c_codeline(:n_lmaxfixformlength), '(', BACK=.TRUE.)
        i_cparright = INDEX(c_codeline(:n_lmaxfixformlength), ')', BACK=.TRUE.)
        i = MAX(i_cspace, i_ccomma, i_cplus, i_cminus, i_ctimes, i_cdiv, i_cparleft, i_cparright)
#      ifdef CFG_DEBUG
#      if (CFG_DEBUGLEVEL > 0)
        WRITE(jp_stddbg,"(A,I0)") "  ... found '"//c_codeline(i:i) // "' at column ", i
#      endif
#      endif
        WRITE(i_unit_out, '(A)') TRIM(c_codeline(:i))
        n_lineno_out = n_lineno_out+1
        c_codeline = '     &      ' // c_codeline(i+1:)
        CYCLE processline
      ENDIF
      ENDDO searchnewstring
! No solution ready so far:
#     ifdef CFG_DEBUG
#     if (CFG_DEBUGLEVEL > 0)
      WRITE(jp_stddbg,cp_fmtdbg_a) '  Writing out line unchanged to "' // TRIM(c_file_out) // '"'
#     endif
#     endif
      WRITE(i_unit_out, '(A)') TRIM(c_codeline)
      n_lineno_out = n_lineno_out+1
#     ifdef CFG_DEBUG
      WRITE(jp_stddbg,cp_fmtdbg_a,ADVANCE='NO') '  Manual action required at line '
      WRITE(jp_stddbg,'(I0)') n_lineno_out
#     endif
      WRITE(jp_stdout,cp_fmtinf_a,ADVANCE='NO')
      WRITE(jp_stdout,'("Output file ", A,":")') '"' // TRIM(c_file_out) // '"'
      WRITE(jp_stdout,'("  manual action required at line ",I0)') n_lineno_out
      l_copen = .FALSE.
    ENDIF
    CYCLE readline
  ENDDO processline
  CYCLE readline
  995 EXIT readline
ENDDO readline

#ifdef CFG_VERBOSE
#if (CFG_VERBOSELEVEL == 2)
WRITE(jp_stdout,cp_fmtinf_a,ADVANCE='NO')
WRITE(jp_stdout,'(A,I0,A)') 'Read in ', n_lineno_in, ' lines from file "' // TRIM(c_file_in) // '"'
WRITE(jp_stdout,cp_fmtinf_a,ADVANCE='NO')
WRITE(jp_stdout,'(A,I0,A)') 'Wrote ', n_lineno_out, ' lines to file "' // TRIM(c_file_out) // '"'
#endif
#endif


CLOSE(UNIT=i_unit_in)
CLOSE(UNIT=i_unit_out)

#ifdef CFG_VERBOSE
#if (CFG_VERBOSELEVEL == 2)
WRITE(jp_stdout,cp_fmtinf_a) 'completed'
WRITE(jp_stdout,'()') 
#endif
#endif
#ifdef CFG_DEBUG
WRITE(jp_stddbg,cp_fmtdbg_a) 'completed'
WRITE(jp_stddbg,'()') 
#endif

!=======================================================================
 END SUBROUTINE NORMALIZE_SOURCECODE
!=======================================================================
