!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#include "configure.h"
!=======================================================================
 MODULE MOD_UTILITIES
!=======================================================================

 IMPLICIT NONE

 PRIVATE :: WRITE_CONDENSED_DBLE_TOLUN, WRITE_CONDENSED_DBLE_TOCHAR


 INTERFACE WRITE_CONDENSED_DBLE

   MODULE PROCEDURE WRITE_CONDENSED_DBLE_TOLUN, &
                    WRITE_CONDENSED_DBLE_TOCHAR

 END INTERFACE

 
 CONTAINS

!-----------------------------------------------------------------------
 FUNCTION UPCASE(string) RESULT(upper)
!-----------------------------------------------------------------------
! from http://www.star.le.ac.uk/~cgp/fortran.html

CHARACTER(LEN=*), INTENT(IN) :: string
CHARACTER(LEN=LEN(string))   :: upper

INTEGER :: j


DO j = 1, LEN(string)
  IF(string(j:j) >= "a" .AND. string(j:j) <= "z") THEN
       upper(j:j) = ACHAR(IACHAR(string(j:j)) - 32)
  ELSE
       upper(j:j) = string(j:j)
  END IF
END DO

RETURN

!-----------------------------------------------------------------------
 END FUNCTION UPCASE
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 FUNCTION LOWCASE(string) RESULT(lower)
!-----------------------------------------------------------------------
! after http://www.star.le.ac.uk/~cgp/fortran.html

CHARACTER(LEN=*), INTENT(IN) :: string
CHARACTER(LEN=LEN(string))   :: lower

INTEGER :: j


DO j = 1, LEN(string)
  IF(string(j:j) >= "A" .AND. string(j:j) <= "Z") THEN
       lower(j:j) = ACHAR(IACHAR(string(j:j)) + 32)
  ELSE
       lower(j:j) = string(j:j)
  END IF
END DO

RETURN

!-----------------------------------------------------------------------
END FUNCTION LOWCASE
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 FUNCTION SPC2UNDERSCORE(c_in) RESULT(c_spc2uns)
!-----------------------------------------------------------------------

CHARACTER(LEN=*),        INTENT(IN) :: c_in
CHARACTER(LEN=LEN(c_in))            :: c_spc2uns


INTEGER :: lent, j

lent = LEN_TRIM(c_in)
c_spc2uns = c_in

DO

  j = INDEX(c_in, ' ')

  IF ((j < 1) .OR. (j > lent)) THEN
    EXIT
  ELSE
    c_spc2uns(j:j) = '_'
  ENDIF

END DO

RETURN

!-----------------------------------------------------------------------
 END FUNCTION SPC2UNDERSCORE
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
SUBROUTINE EXPAND_TOKEN(string, token, substitute, string_xp)
!-----------------------------------------------------------------------

IMPLICIT NONE

CHARACTER(LEN=*) :: string
CHARACTER(LEN=*) :: token
CHARACTER(LEN=*) :: substitute
CHARACTER(LEN=*) :: string_xp

INTEGER :: l_xp, l_sb, lt_xp, i, j, l_to

#ifdef CFG_DEBUG
CHARACTER(LEN=*), PARAMETER :: c_dbgfmt = '("DEBUG [EXPAND_TOKEN]: ",A)'
#endif



l_sb = LEN_TRIM(substitute)
string_xp = string


l_xp = LEN(string_xp)
l_to = LEN_TRIM(token)


i = INDEX(string_xp, token(1:l_to))

#ifdef CFG_DEBUG
#if (CFG_DEBUGLEVEL == 2)
WRITE(CFG_STDDBG,c_dbgfmt) ' <'//TRIM(string_xp)//'> ('// TRIM(token) // ')'
IF(i > 0) THEN
  WRITE(CFG_STDDBG,'(" token at position ", I0)') i
ELSE
  WRITE(CFG_STDDBG,'(" token not in string")')
ENDIF
#endif
#endif
DO WHILE(i > 0)
  lt_xp = LEN_TRIM(string_xp)
  j = MIN(lt_xp, l_xp-l_sb+l_to)
  string_xp(i:) = TRIM(substitute)//string_xp(i+l_to:j)
  i = INDEX(string_xp, token(1:l_to))
# ifdef CFG_DEBUG
# if (CFG_DEBUGLEVEL == 2)
  WRITE(CFG_STDDBG,c_dbgfmt) '->'//TRIM(string_xp)//'> ('// TRIM(token) // ')'
  IF(i > 0) THEN
    WRITE(CFG_STDDBG,'(" token at position ", I0)') i
  ELSE
    WRITE(CFG_STDDBG,'(" token not in string")')
  ENDIF
# endif
# endif
ENDDO

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE EXPAND_TOKEN
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE EXPAND_2TOKEN(string, token1, substitute1, &
                                  token2, substitute2, &
                                  string_xp)
!-----------------------------------------------------------------------

IMPLICIT NONE

CHARACTER(LEN=*) :: string
CHARACTER(LEN=*) :: token1, token2
CHARACTER(LEN=*) :: substitute1, substitute2
CHARACTER(LEN=*) :: string_xp

INTEGER :: l_xp, l_sb, lt_xp, i, j, l_to

#ifdef CFG_DEBUG
CHARACTER(LEN=*), PARAMETER :: c_dbgfmt = '("DEBUG [EXPAND_2TOKEN]: ",A)'
#endif

l_xp = LEN(string_xp)
string_xp = string


l_sb = LEN_TRIM(substitute1)
l_to = LEN_TRIM(token1)

i = INDEX(string_xp, token1(1:l_to))


#ifdef CFG_DEBUG
#if (CFG_DEBUGLEVEL == 2)
WRITE(CFG_STDDBG,c_dbgfmt) ' <'//TRIM(string_xp)//'> ('// TRIM(token1) // ')'
IF(i > 0) THEN
  WRITE(CFG_STDDBG,'(" token at position ", I0)') i
ELSE
  WRITE(CFG_STDDBG,'(" token not in string")')
ENDIF
#endif
#endif
DO WHILE(i > 0)
  lt_xp = LEN_TRIM(string_xp)
  j = MIN(lt_xp, l_xp-l_sb+l_to)
  string_xp(i:) = TRIM(substitute1)//string_xp(i+l_to:j)
  i = INDEX(string_xp, token1(1:l_to))
# ifdef CFG_DEBUG
# if (CFG_DEBUGLEVEL == 2)
  WRITE(CFG_STDDBG,c_dbgfmt) '->'//TRIM(string_xp)//'> ('// TRIM(token1) // ')'
  IF(i > 0) THEN
    WRITE(CFG_STDDBG,'(" token at position ", I0)') i
  ELSE
    WRITE(CFG_STDDBG,'(" token not in string")')
  ENDIF
# endif
# endif
ENDDO


l_sb = LEN_TRIM(substitute2)
l_to = LEN_TRIM(token2)

i = INDEX(string_xp, token2(1:l_to))

#ifdef CFG_DEBUG
#if (CFG_DEBUGLEVEL == 2)
WRITE(CFG_STDDBG,c_dbgfmt) ' <'//TRIM(string_xp)//'> ('// TRIM(token2) // ')'
IF(i > 0) THEN
  WRITE(CFG_STDDBG,'(" token at position ", I0)') i
ELSE
  WRITE(CFG_STDDBG,'(" token not in string")')
ENDIF
#endif
#endif
DO WHILE(i > 0)
  lt_xp = LEN_TRIM(string_xp)
  j = MIN(lt_xp, l_xp-l_sb+l_to)
  string_xp(i:) = TRIM(substitute2)//string_xp(i+l_to:j)
  i = INDEX(string_xp, token2(1:l_to))
# ifdef CFG_DEBUG
# if (CFG_DEBUGLEVEL == 2)
  WRITE(CFG_STDDBG,c_dbgfmt) '->'//TRIM(string_xp)//'> ('// TRIM(token2) // ')'
  IF(i > 0) THEN
    WRITE(CFG_STDDBG,'(" token at position ", I0)') i
  ELSE
    WRITE(CFG_STDDBG,'(" token not in string")')
  ENDIF
# endif
# endif
ENDDO

RETURN

!-----------------------------------------------------------------------
END SUBROUTINE EXPAND_2TOKEN
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
SUBROUTINE EXPAND_3TOKEN(string, token1, substitute1, &
                                 token2, substitute2, &
                                 token3, substitute3, &
                                 string_xp)
!-----------------------------------------------------------------------

IMPLICIT NONE

CHARACTER(LEN=*) :: string
CHARACTER(LEN=*) :: token1, token2, token3
CHARACTER(LEN=*) :: substitute1, substitute2, substitute3
CHARACTER(LEN=*) :: string_xp

INTEGER :: l_xp, l_sb, lt_xp, i, j, l_to

#ifdef CFG_DEBUG
CHARACTER(LEN=*), PARAMETER :: c_dbgfmt = '("DEBUG [EXPAND_3TOKEN]: ",A)'
#endif



l_xp = LEN(string_xp)
string_xp = string


l_sb = LEN_TRIM(substitute1)
l_to = LEN_TRIM(token1)

i = INDEX(string_xp, token1(1:l_to))


#ifdef CFG_DEBUG
#if (CFG_DEBUGLEVEL == 2)
WRITE(CFG_STDDBG,c_dbgfmt) ' <'//TRIM(string_xp)//'> ('// TRIM(token1) // ')'
IF(i > 0) THEN
  WRITE(CFG_STDDBG,'(" token at position ", I0)') i
ELSE
  WRITE(CFG_STDDBG,'(" token not in string")')
ENDIF
#endif
#endif
DO WHILE(i > 0)
  lt_xp = LEN_TRIM(string_xp)
  j = MIN(lt_xp, l_xp-l_sb+l_to)
  string_xp(i:) = TRIM(substitute1)//string_xp(i+l_to:j)
  i = INDEX(string_xp, token1(1:l_to))
# ifdef CFG_DEBUG
# if (CFG_DEBUGLEVEL == 2)
  WRITE(CFG_STDDBG,c_dbgfmt) '->'//TRIM(string_xp)//'> ('// TRIM(token1) // ')'
  IF(i > 0) THEN
    WRITE(CFG_STDDBG,'(" token at position ", I0)') i
  ELSE
    WRITE(CFG_STDDBG,'(" token not in string")')
  ENDIF
# endif
# endif
ENDDO


l_sb = LEN_TRIM(substitute2)
l_to = LEN_TRIM(token2)

i = INDEX(string_xp, token2(1:l_to))

#ifdef CFG_DEBUG
#if (CFG_DEBUGLEVEL == 2)
WRITE(CFG_STDDBG,c_dbgfmt) ' <'//TRIM(string_xp)//'> ('// TRIM(token2) // ')'
IF(i > 0) THEN
  WRITE(CFG_STDDBG,'(" token at position ", I0)') i
ELSE
  WRITE(CFG_STDDBG,'(" token not in string")')
ENDIF
#endif
#endif
DO WHILE(i > 0)
  lt_xp = LEN_TRIM(string_xp)
  j = MIN(lt_xp, l_xp-l_sb+l_to)
  string_xp(i:) = TRIM(substitute2)//string_xp(i+l_to:j)
  i = INDEX(string_xp, token2(1:l_to))
# ifdef CFG_DEBUG
# if (CFG_DEBUGLEVEL == 2)
  WRITE(CFG_STDDBG,c_dbgfmt) '->'//TRIM(string_xp)//'> ('// TRIM(token2) // ')'
  IF(i > 0) THEN
    WRITE(CFG_STDDBG,'(" token at position ", I0)') i
  ELSE
    WRITE(CFG_STDDBG,'(" token not in string")')
  ENDIF
#endif
#endif
ENDDO


l_sb = LEN_TRIM(substitute3)
l_to = LEN_TRIM(token3)

i = INDEX(string_xp, token3(1:l_to))

#ifdef CFG_DEBUG
#if (CFG_DEBUGLEVEL == 2)
WRITE(CFG_STDDBG,c_dbgfmt) ' <'//TRIM(string_xp)//'> ('// TRIM(token3) // ')'
IF(i > 0) THEN
  WRITE(CFG_STDDBG,'(" token at position ", I0)') i
ELSE
  WRITE(CFG_STDDBG,'(" token not in string")')
ENDIF
#endif
#endif
DO WHILE(i > 0)
  lt_xp = LEN_TRIM(string_xp)
  j = MIN(lt_xp, l_xp-l_sb+l_to)
  string_xp(i:) = TRIM(substitute3)//string_xp(i+l_to:j)
  i = INDEX(string_xp, token3(1:l_to))
# ifdef CFG_DEBUG
# if (CFG_DEBUGLEVEL == 2)
  WRITE(CFG_STDDBG,c_dbgfmt) '->'//TRIM(string_xp)//'> ('// TRIM(token3) // ')'
  IF(i > 0) THEN
    WRITE(CFG_STDDBG,'(" token at position ", I0)') i
  ELSE
    WRITE(CFG_STDDBG,'(" token not in string")')
  ENDIF
#endif
#endif
ENDDO

RETURN

!-----------------------------------------------------------------------
END SUBROUTINE EXPAND_3TOKEN
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 LOGICAL FUNCTION L_STR_EQ_DBL(c_in, d_in)
!-----------------------------------------------------------------------

IMPLICIT NONE

CHARACTER(LEN=*), INTENT(IN) :: c_in
DOUBLE PRECISION, INTENT(IN) :: d_in

DOUBLE PRECISION :: d_strvalue
LOGICAL :: l_readerr

l_readerr = .TRUE.
READ(c_in,*, ERR=1) d_strvalue
l_readerr = .FALSE.
1 CONTINUE

IF(l_readerr) THEN
  l_str_eq_dbl = .FALSE.
ELSE
  IF(d_strvalue == d_in) THEN
    l_str_eq_dbl = .TRUE.
  ELSE
    l_str_eq_dbl = .FALSE.
  ENDIF
ENDIF

RETURN

!-----------------------------------------------------------------------
 END FUNCTION L_STR_EQ_DBL
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 FUNCTION CAT4STRINGS_UNDERSCORED(string1, string2, string3, string4) RESULT(string_cat)
!-----------------------------------------------------------------------

IMPLICIT NONE

CHARACTER(LEN=*), INTENT(IN)  :: string1
CHARACTER(LEN=*), INTENT(IN)  :: string2
CHARACTER(LEN=*), INTENT(IN)  :: string3
CHARACTER(LEN=*), INTENT(IN)  :: string4
CHARACTER(LEN=LEN(string1)+LEN(string2)+LEN(string3)+LEN(string4)) &
                              :: string_cat


string_cat = TRIM(ADJUSTL(string1)) // '_' // &
             TRIM(ADJUSTL(string2)) // '_' // &
             TRIM(ADJUSTL(string3)) // '_' // &
             TRIM(ADJUSTL(string4))

RETURN

!-----------------------------------------------------------------------
 END FUNCTION CAT4STRINGS_UNDERSCORED
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
FUNCTION INDEX_COPIES_IN_ARRAY(str_arrayin) RESULT(idx_copies)
!-----------------------------------------------------------------------

IMPLICIT NONE

CHARACTER(LEN=*), DIMENSION(:), INTENT(IN) :: str_arrayin

INTEGER, DIMENSION(SIZE(str_arrayin)) :: idx_copies

INTEGER :: i, j

! Initialize each item of idx_copies to its 1D position index
idx_copies(:) = (/ (i, i=1, SIZE(str_arrayin)) /)

DO i = 1, SIZE(str_arrayin)-1

  ! If the i'th element does not have the value 'i', then, and only
  ! then, its value has been modified before.
  ! Else: we compare each of elements following it, and if we find a
  ! duplicate, we reset its position index to 'i'
  IF(idx_copies(i) == i) THEN

    DO j = i + 1, SIZE(str_arrayin)
      IF (idx_copies(j) == j) THEN
        IF (str_arrayin(i) == str_arrayin(j)) idx_copies(j) = i
      ENDIF
    ENDDO

  ENDIF

ENDDO

RETURN

!-----------------------------------------------------------------------
END FUNCTION INDEX_COPIES_IN_ARRAY
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
FUNCTION COUNT_COPIES_IN_ARRAY(str_arrayin) RESULT(n_copies)
!-----------------------------------------------------------------------

IMPLICIT NONE

CHARACTER(LEN=*), DIMENSION(:) :: str_arrayin

INTEGER, DIMENSION(SIZE(str_arrayin)) :: n_copies

INTEGER :: i, j
INTEGER :: n

n_copies(:) = 1


DO i = 1, SIZE(str_arrayin)-1

  n = 1

  IF(n_copies(i) == 1) THEN

    DO j = i + 1, SIZE(str_arrayin)
      IF (n_copies(j) /= 1) CYCLE
      IF (str_arrayin(i) == str_arrayin(j)) THEN
        n_copies(j) = 0
        n = n + 1
      ENDIF
    ENDDO

    n_copies(i) = n
    IF(n /= 1) THEN
      WHERE(n_copies(i+1:) == 0)
        n_copies(i+1:) = n
      ENDWHERE
    ENDIF

  ENDIF

ENDDO

RETURN

!-----------------------------------------------------------------------
END FUNCTION COUNT_COPIES_IN_ARRAY
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE DELIMIT_STRING_TOKENS(c_in, c_separators, i_begin, n_len)
!-----------------------------------------------------------------------

CHARACTER(LEN=*), INTENT(IN)   :: c_in
CHARACTER(LEN=*), INTENT(IN)   :: c_separators
INTEGER, DIMENSION(:), POINTER :: i_begin
INTEGER, DIMENSION(:), POINTER :: n_len


INTEGER, PARAMETER :: jp_stderr = 0
CHARACTER(LEN=LEN(c_in)) :: c_tmp

INTEGER :: i, ii, i1, i2, i3, n_words

IF(LEN_TRIM(c_separators) == 0) THEN
  WRITE(jp_stderr, '("[DELIMIT_STRING_TOKENS]: No separators provided -- aborting")')
  CALL ABORT()
ENDIF

c_tmp = ' '
n_words = 0
DO i = 1, LEN_TRIM(c_in)
  c_tmp(i:i) = ' '
  ii = INDEX(c_separators, c_in(i:i))
  IF(ii > 0) THEN
    IF (i > 1) THEN
      SELECT CASE(c_tmp(i-1:i-1))
      CASE(' ')
        CYCLE
      CASE('1')
        c_tmp(i-1:i-1) = '3'  ! one-letter word on previous character
        n_words = n_words + 1
      CASE('2')
        n_words = n_words + 1
      END SELECT
    ELSE
      CYCLE
    ENDIF  
  ELSE
    IF (i == 1) THEN
      c_tmp(i:i) = '1'
    ELSE
      SELECT CASE(c_tmp(i-1:i-1))
      CASE(' ')           ! SPC in pos=i-1
        c_tmp(i:i) = '1'  ! => new word starts at pos=i
      CASE('1')           ! new word has begun at pos=i-1
        c_tmp(i:i) = '2'  ! => provisionally end it at pos=i
      CASE('2')           ! word ended provisionally at pos=i-1
        c_tmp(i-1:i-1) = ' '
        c_tmp(i:i) = '2'  ! => report end to pos=i
      END SELECT
    ENDIF
  END IF
  IF (i == LEN_TRIM(c_in)) THEN
    SELECT CASE(c_tmp(i:i))
    CASE('1')
      c_tmp(i:i) = '3'  ! one-letter word on last character
      n_words = n_words + 1
    CASE('2')
      n_words = n_words + 1
    END SELECT
  ENDIF
END DO


IF(n_words > 0) THEN
  ALLOCATE(i_begin(n_words))
  ALLOCATE(n_len(n_words))

  ii = 1
  DO i = 1, n_words
    i1 = INDEX(c_tmp(ii:),'1')
    i3 = INDEX(c_tmp(ii:),'3')

    IF(i3 > 0) THEN
      IF(i1 > 0) THEN 
        IF(i1 < i3) THEN
          i_begin(i) = i1 + ii - 1
          i2 = INDEX(c_tmp(ii:),'2')
          n_len(i) = i2 - i1 + 1
          ii = ii + i2
        ELSE
          i_begin(i) = i3 + ii - 1
          n_len(i) = 1
          ii = ii + i3
        ENDIF
      ELSE
        i_begin(i) = i3 + ii - 1
        n_len(i) = 1
        ii = ii + i3
      ENDIF
    ELSE
      i_begin(i) = i1 + ii - 1
      i2 = INDEX(c_tmp(ii:),'2')
      n_len(i) = i2 - i1 + 1
      ii = ii + i2
    ENDIF
  ENDDO

ELSE

  NULLIFY(i_begin)
  NULLIFY(n_len)

ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE DELIMIT_STRING_TOKENS
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
 SUBROUTINE WRITE_CONDENSED_DBLE_TOLUN(kunit, dd, l_linefeed)
!-----------------------------------------------------------------------

IMPLICIT NONE

INTEGER, INTENT(IN) :: kunit
DOUBLE PRECISION, INTENT(IN) :: dd
LOGICAL, OPTIONAL, INTENT(IN) :: l_linefeed

CHARACTER(LEN=256) :: clipboard1, clipboard2
CHARACTER(LEN=1)   :: cl_char
INTEGER :: i, i1, i2, ipt, imantissa
LOGICAL :: l_has_significant_decimals

CHARACTER(LEN=*), PARAMETER :: c_d00    = 'D+00'
CHARACTER(LEN=*), PARAMETER :: c_pt0    = '.0'
CHARACTER(LEN=*), PARAMETER :: c_pt0d00 = c_pt0 // c_d00

CHARACTER(LEN=3) :: c_advance

#ifdef CFG_DEBUG
INTEGER, PARAMETER :: jp_stddbg = CFG_STDDBG
CHARACTER(LEN=*), PARAMETER  :: c_fmtdbg_a   = '("DEBUG [WRITE_CONDENSED_DBLE]: ", A)'
#endif



WRITE(clipboard1,*) dd
#ifdef CFG_DEBUG
#if (CFG_DEBUGLEVEL == 2)
WRITE(jp_stddbg,c_fmtdbg_a) 'clipboard1 = "' // TRIM(clipboard1) // '" (trimmed)'
#endif
#endif
i1 = LEN_TRIM(clipboard1)
clipboard2(1:i1) = clipboard1(1:i1)
i2 = i1
ipt = 0
DO i = 1, i1

  cl_char = clipboard2(i:i)
  SELECT CASE(cl_char)
  CASE('d','D','e','E')
    i2 = i-1
  CASE('0')
    clipboard2(i:i) = ' '
  CASE('.')
    ipt = i
  END SELECT

ENDDO

#ifdef CFG_DEBUG
#if (CFG_DEBUGLEVEL == 2)
WRITE(jp_stddbg,c_fmtdbg_a, ADVANCE='NO') 'after scan: i1, i2, ipt ='
WRITE(jp_stddbg,'(3(1X, I0))') i1, i2, ipt
WRITE(jp_stddbg,c_fmtdbg_a) 'clipboard2 = "' // clipboard2(1:i1) // '" (1:i1)'
#endif
#endif

imantissa = LEN_TRIM(clipboard2(1:i2))
#ifdef CFG_DEBUG
#if (CFG_DEBUGLEVEL == 2)
WRITE(jp_stddbg,c_fmtdbg_a, ADVANCE='NO') 'before ipt check: imantissa ='
WRITE(jp_stddbg,'(1X, I0)') imantissa
#endif
#endif

IF(imantissa == 0) THEN
  ! length of the mantissa is 0, which may happen for '0D0' or '  0D0'
  clipboard1(1:1) = '0'  ! put back one zero
  imantissa = 1
  l_has_significant_decimals = .FALSE.
ELSEIF(imantissa == ipt) THEN
  ! the last non-blank character is '.' (preceeding any possible exponent)
  ! Note: ipt /= 0 here because imantissa /= 0 here
  IF((ipt == 1) .OR. &
     (clipboard1(ipt-1:ipt-1) == ' ') .OR. &
     (clipboard1(ipt-1:ipt-1) == '-') .OR. &
     (clipboard1(ipt-1:ipt-1) == '+')     ) THEN
    ! if '.' is the first character or if it is preceeded immediately
    ! by ' ', '-', or '+', replace it by '0'
    clipboard1(ipt:ipt) = '0'
  ELSE
    ! discard the '.'
    clipboard1(ipt:ipt) = ' '
  ENDIF
  l_has_significant_decimals = .FALSE.
ELSE
  l_has_significant_decimals = .TRUE.
ENDIF

#ifdef CFG_DEBUG
#if (CFG_DEBUGLEVEL == 2)
WRITE(jp_stddbg,c_fmtdbg_a, ADVANCE='NO') 'has significant decimals = '
WRITE(jp_stddbg,'(L1)') l_has_significant_decimals
WRITE(jp_stddbg,c_fmtdbg_a, ADVANCE='NO') 'after ipt check: imantissa = '
WRITE(jp_stddbg,'(I0)') imantissa
WRITE(jp_stddbg,c_fmtdbg_a) 'clipboard1 = "' // clipboard1(1:i1) // '" (1:i1)'
#endif
#endif


c_advance='NO'
! Proceed to printing
IF (PRESENT(l_linefeed)) THEN
  IF (l_linefeed) c_advance = 'YES'
ENDIF
    

IF (i1 == i2) THEN
  ! No d, D, e, E found
  IF (l_has_significant_decimals) THEN
# ifdef CFG_DEBUG
# if (CFG_DEBUGLEVEL == 2)
  WRITE(jp_stddbg,c_fmtdbg_a) 'condensed = "' // &
      TRIM(ADJUSTL(clipboard1(1:imantissa))) // c_d00 // '"'
# endif
# endif
    WRITE(kunit, '(A)', ADVANCE=c_advance) &
      TRIM(ADJUSTL(clipboard1(1:imantissa))) // c_d00
  ELSE
#   ifdef CFG_DEBUG
#   if (CFG_DEBUGLEVEL == 2)
    WRITE(jp_stddbg,c_fmtdbg_a) 'condensed = "' // &
      TRIM(ADJUSTL(clipboard1(1:imantissa))) // c_pt0d00 // '"'
#   endif
#   endif
    WRITE(kunit, '(A)', ADVANCE=c_advance) &
      TRIM(ADJUSTL(clipboard1(1:imantissa))) // c_pt0d00
  ENDIF
ELSE
  ! d, D, e, E found: condense mantissa and append exponent part.
  IF (l_has_significant_decimals) THEN
# ifdef CFG_DEBUG
# if (CFG_DEBUGLEVEL == 2)
  WRITE(jp_stddbg,c_fmtdbg_a) 'condensed = "' // &
  TRIM(ADJUSTL(clipboard1(1:imantissa))) // clipboard1(i2+1:i1) // '"'
# endif
# endif
    WRITE(kunit, '(A)', ADVANCE=c_advance) &
      TRIM(ADJUSTL(clipboard1(1:imantissa))) // clipboard1(i2+1:i1)
  ELSE
#   ifdef CFG_DEBUG
#   if (CFG_DEBUGLEVEL == 2)
    WRITE(jp_stddbg,c_fmtdbg_a) 'condensed = "' // &
      TRIM(ADJUSTL(clipboard1(1:imantissa))) // c_pt0 // clipboard1(i2+1:i1) // '"'
#   endif
#   endif
    WRITE(kunit, '(A)', ADVANCE=c_advance) &
      TRIM(ADJUSTL(clipboard1(1:imantissa))) // c_pt0 // clipboard1(i2+1:i1)
  ENDIF

ENDIF

!-----------------------------------------------------------------------
 END SUBROUTINE WRITE_CONDENSED_DBLE_TOLUN
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE WRITE_CONDENSED_DBLE_TOCHAR(strunit, dd, l_linefeed)
!-----------------------------------------------------------------------

IMPLICIT NONE

CHARACTER(LEN=*), INTENT(OUT) :: strunit
DOUBLE PRECISION, INTENT(IN) :: dd
LOGICAL, OPTIONAL, INTENT(IN) :: l_linefeed

CHARACTER(LEN=256) :: clipboard1, clipboard2
CHARACTER(LEN=1)   :: cl_char
INTEGER :: i, i1, i2, ipt, imantissa
LOGICAL :: l_has_significant_decimals

CHARACTER(LEN=*), PARAMETER :: c_d00    = 'D+00'
CHARACTER(LEN=*), PARAMETER :: c_pt0    = '.0'
CHARACTER(LEN=*), PARAMETER :: c_pt0d00 = c_pt0 // c_d00

CHARACTER(LEN=3) :: c_advance

#ifdef CFG_DEBUG
INTEGER, PARAMETER :: jp_stddbg = CFG_STDDBG
CHARACTER(LEN=*), PARAMETER  :: c_fmtdbg_a   = '("DEBUG [WRITE_CONDENSED_DBLE]: ", A)'
#endif



WRITE(clipboard1,*) dd
#ifdef CFG_DEBUG
#if (CFG_DEBUGLEVEL == 2)
WRITE(jp_stddbg,c_fmtdbg_a) 'clipboard1 = "' // TRIM(clipboard1) // '" (trimmed)'
#endif
#endif
i1 = LEN_TRIM(clipboard1)
clipboard2(1:i1) = clipboard1(1:i1)
i2 = i1
ipt = 0
DO i = 1, i1

  cl_char = clipboard2(i:i)
  SELECT CASE(cl_char)
  CASE('d','D','e','E')
    i2 = i-1
  CASE('0')
    clipboard2(i:i) = ' '
  CASE('.')
    ipt = i
  END SELECT

ENDDO

#ifdef CFG_DEBUG
#if (CFG_DEBUGLEVEL == 2)
WRITE(jp_stddbg,c_fmtdbg_a, ADVANCE='NO') 'after scan: i1, i2, ipt ='
WRITE(jp_stddbg,'(3(1X, I0))') i1, i2, ipt
WRITE(jp_stddbg,c_fmtdbg_a) 'clipboard2 = "' // clipboard2(1:i1) // '" (1:i1)'
#endif
#endif

imantissa = LEN_TRIM(clipboard2(1:i2))
#ifdef CFG_DEBUG
#if (CFG_DEBUGLEVEL == 2)
WRITE(jp_stddbg,c_fmtdbg_a, ADVANCE='NO') 'before ipt check: imantissa ='
WRITE(jp_stddbg,'(1X, I0)') imantissa
#endif
#endif

IF(imantissa == 0) THEN
  ! length of the mantissa is 0, which may happen for '0D0' or '  0D0'
  clipboard1(1:1) = '0'  ! put back one zero
  imantissa = 1
  l_has_significant_decimals = .FALSE.
ELSEIF(imantissa == ipt) THEN
  ! the last non-blank character is '.' (preceeding any possible exponent)
  ! Note: ipt /= 0 here because imantissa /= 0 here
  IF((ipt == 1) .OR. &
     (clipboard1(ipt-1:ipt-1) == ' ') .OR. &
     (clipboard1(ipt-1:ipt-1) == '-') .OR. &
     (clipboard1(ipt-1:ipt-1) == '+')     ) THEN
    ! if '.' is the first character or if it is preceeded immediately
    ! by ' ', '-', or '+', replace it by '0'
    clipboard1(ipt:ipt) = '0'
  ELSE
    ! discard the '.'
    clipboard1(ipt:ipt) = ' '
  ENDIF
  l_has_significant_decimals = .FALSE.
ELSE
  l_has_significant_decimals = .TRUE.
ENDIF

#ifdef CFG_DEBUG
#if (CFG_DEBUGLEVEL == 2)
WRITE(jp_stddbg,c_fmtdbg_a, ADVANCE='NO') 'has significant decimals = '
WRITE(jp_stddbg,'(L1)') l_has_significant_decimals
WRITE(jp_stddbg,c_fmtdbg_a, ADVANCE='NO') 'after ipt check: imantissa = '
WRITE(jp_stddbg,'(I0)') imantissa
WRITE(jp_stddbg,c_fmtdbg_a) 'clipboard1 = "' // clipboard1(1:i1) // '" (1:i1)'
#endif
#endif


c_advance='NO'

IF (i1 == i2) THEN
  ! No d, D, e, E found
  IF (l_has_significant_decimals) THEN
# ifdef CFG_DEBUG
# if (CFG_DEBUGLEVEL == 2)
  WRITE(jp_stddbg,c_fmtdbg_a) 'condensed = "' // &
      TRIM(ADJUSTL(clipboard1(1:imantissa))) // c_d00 // '"'
# endif
# endif
    WRITE(strunit, '(A)') TRIM(ADJUSTL(clipboard1(1:imantissa))) // c_d00
  ELSE
#   ifdef CFG_DEBUG
#   if (CFG_DEBUGLEVEL == 2)
    WRITE(jp_stddbg,c_fmtdbg_a) 'condensed = "' // &
      TRIM(ADJUSTL(clipboard1(1:imantissa))) // c_pt0d00 // '"'
#   endif
#   endif
    WRITE(strunit, '(A)') TRIM(ADJUSTL(clipboard1(1:imantissa))) // c_pt0d00
  ENDIF
ELSE
  ! d, D, e, E found: condense mantissa and append exponent part.
  IF (l_has_significant_decimals) THEN
# ifdef CFG_DEBUG
# if (CFG_DEBUGLEVEL == 2)
  WRITE(jp_stddbg,c_fmtdbg_a) 'condensed = "' // &
  TRIM(ADJUSTL(clipboard1(1:imantissa))) // clipboard1(i2+1:i1) // '"'
# endif
# endif
    WRITE(strunit, '(A)') TRIM(ADJUSTL(clipboard1(1:imantissa))) // clipboard1(i2+1:i1)
  ELSE
#   ifdef CFG_DEBUG
#   if (CFG_DEBUGLEVEL == 2)
    WRITE(jp_stddbg,c_fmtdbg_a) 'condensed = "' // &
      TRIM(ADJUSTL(clipboard1(1:imantissa))) // c_pt0 // clipboard1(i2+1:i1) // '"'
#   endif
#   endif
    WRITE(strunit, '(A)') TRIM(ADJUSTL(clipboard1(1:imantissa))) // c_pt0 // clipboard1(i2+1:i1)
  ENDIF

ENDIF

!-----------------------------------------------------------------------
 END SUBROUTINE WRITE_CONDENSED_DBLE_TOCHAR
!-----------------------------------------------------------------------

!=======================================================================
 END MODULE MOD_UTILITIES
!=======================================================================
