!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#include "configure.h"
!=======================================================================
 MODULE MOD_CONFIGURE_TYPES
!=======================================================================

USE MOD_MEDUSA_COCOGEN

IMPLICIT NONE

INTEGER, PARAMETER, PRIVATE :: jp_stdout = (CFG_STDOUT)
INTEGER, PARAMETER, PRIVATE :: jp_stderr = (CFG_STDERR)

TYPE CODEBITS
  ! ------------ General information ------------
  CHARACTER(LEN=n_lmaxcomptyp)  :: type           ! type: SolubilityProduct, DiffCoeff,
                                                  ! SaturationConc, TotalConcentration,
                                                  ! WDataExtension
  CHARACTER(LEN=n_lmaxexpress)  :: units          ! units
  INTEGER                       :: nlines         ! number of lines of code
  CHARACTER(LEN=n_lmaxcodeline), &
    DIMENSION(:), POINTER       :: code           ! array with the lines of code
  CHARACTER(LEN=n_lmaxidentif)  :: varname        ! Medusa code variable name to store results
  CHARACTER(LEN=n_lmaxcodeline) :: vartype        ! Type of Medusa code variable name
  ! ------------- List connectivity -------------
  TYPE(CODEBITS), POINTER       :: next           ! pointer to next in chain
  ! ---------------------------------------------
END TYPE


TYPE CHEMREAGENT
  ! ------------ General information ------------
  CHARACTER(LEN=n_lmaxnamesgen) :: name
  CHARACTER(LEN=n_lmaxshortid)  :: shortid
  CHARACTER(LEN=n_lmaxphasid)   :: phasid
  CHARACTER(LEN=n_lmaxexpress)  :: stoech
  CHARACTER(LEN=n_lmaxshortid)  :: id
  ! ---------------------------------------------
END TYPE


TYPE RATELAW
  ! ------------ General information ------------
  CHARACTER(LEN=n_lmaxprocname) :: name
  CHARACTER(LEN=n_lmaxidentif)  :: pp_type        ! ProcessParameter Type
  ! ---------------- Parameters -----------------
  CHARACTER(LEN=n_lmaxprocname) :: xrefprocname
  INTEGER                       :: n_params
  CHARACTER(LEN=n_lmaxexpress)  :: expression
  CHARACTER(LEN=n_lmaxidentif), &
    DIMENSION(:), POINTER       :: paramname
  CHARACTER(LEN=n_lmaxexpress), &
    DIMENSION(:), POINTER       :: paramcode
  CHARACTER(LEN=n_lmaxidentif), &
    DIMENSION(:), POINTER       :: typecomponame
  CHARACTER(LEN=n_lmaxidentif), &
    DIMENSION(:), POINTER       :: xmltagname
  CHARACTER(LEN=n_lmaxidentif), &
    DIMENSION(:), POINTER       :: xmlattstocheck
  CHARACTER(LEN=n_lmaxphasid),  &
    DIMENSION(:), POINTER       :: kindofparam
  CHARACTER(LEN=n_lmaxshortid), &
    DIMENSION(:), POINTER       :: dummylabel
  INTEGER, &
    DIMENSION(:), POINTER       :: iparam_xref
  ! ---------------------------------------------
END TYPE


TYPE EQLBREL
  ! ------------ General information ------------
  CHARACTER(LEN=n_lmaxprocname) :: name
  CHARACTER(LEN=n_lmaxidentif)  :: ep_type        ! EquilibParameter Type
  ! ---------------- Parameters -----------------
  INTEGER                       :: n_params
  CHARACTER(LEN=n_lmaxexpress)  :: expression
  CHARACTER(LEN=n_lmaxidentif), &
    DIMENSION(:), POINTER       :: paramname
  CHARACTER(LEN=n_lmaxexpress), &
    DIMENSION(:), POINTER       :: paramcode
  CHARACTER(LEN=n_lmaxidentif), &
    DIMENSION(:), POINTER       :: typecomponame
  CHARACTER(LEN=n_lmaxidentif), &
    DIMENSION(:), POINTER       :: xmltagname
  CHARACTER(LEN=n_lmaxphasid),  &
    DIMENSION(:), POINTER       :: kindofparam
  CHARACTER(LEN=n_lmaxshortid), &
    DIMENSION(:), POINTER       :: dummylabel
  ! ---------------------------------------------
END TYPE


TYPE COMPOINFO
  ! ------------ General information ------------
  CHARACTER(LEN=n_lmaxnamesgen) :: name
  CHARACTER(LEN=n_lmaxcomptyp)  :: class
  CHARACTER(LEN=n_lmaxshortid)  :: shortid
  CHARACTER(LEN=n_lmaxphasid)   :: phasid
  INTEGER                       :: idx
  INTEGER                       :: i_category     ! 0 (N/A), 1, 2
  INTEGER                       :: i_system       ! id number of the system
                                                  ! that this instance of
                                                  ! the component belongs to
  INTEGER                       :: idx_xref
  TYPE(COMPOINFO), POINTER      :: xref
  ! ---------------- Parameters -----------------
  INTEGER                       :: n_params       ! Total number of parameters
  INTEGER                       :: n_checom       ! number of Chemicalcomposition
                                                  ! parameters among them
  LOGICAL                       :: l_checom_from_xref
  CHARACTER(LEN=n_lmaxidentif), &
    DIMENSION(:), POINTER       :: param_name
  CHARACTER(LEN=n_lmaxexpress), &
    DIMENSION(:), POINTER       :: param_vartype
  CHARACTER(LEN=n_lmaxidentif), &
    DIMENSION(:), POINTER       :: param_varname
  CHARACTER(LEN=n_lmaxexpress), &
    DIMENSION(:), POINTER       :: param_values
  ! ------------------- Codes -------------------
  TYPE(CODEBITS),  POINTER      :: codes
  ! ---------- Conservation Properties ----------
  INTEGER                       :: n_consrv
  CHARACTER(LEN=n_lmaxidentif), &
    DIMENSION(:), POINTER       :: consrv_name
  CHARACTER(LEN=n_lmaxexpress), &
    DIMENSION(:), POINTER       :: consrv_weights
  ! ------------- List connectivity -------------
  TYPE(COMPOINFO), POINTER      :: prev
  TYPE(COMPOINFO), POINTER      :: next
  ! ---------------------------------------------
END TYPE


TYPE SYSTEMSINFO                                  ! Example:
  ! ------------ General information ------------
  CHARACTER(LEN=n_lmaxnamesgen) :: name           ! 'DIC'
  CHARACTER(LEN=n_lmaxcomptyp)  :: class          ! 'SoluteSystem'
  CHARACTER(LEN=n_lmaxshortid)  :: shortid        ! 'dic'
  CHARACTER(LEN=n_lmaxphasid)   :: phasid
  INTEGER                       :: idx            ! := <order> of system
  INTEGER                       :: i_system       ! system id number
  INTEGER                       :: i_category     ! 1, 2 (1 only if all members are of category 1)
  ! ------------------ Members ------------------
  INTEGER                       :: n_members = 0  ! := 3
  CHARACTER(LEN=n_lmaxnamesgen), &
    DIMENSION(:), POINTER       :: member_name    ! (/ 'CO2', 'HCO3', 'CO3' /)
  ! -- Identifier names for the generated code --
  CHARACTER(LEN=n_lmaxidentif)  :: nvc_identifier ! 'nvc_' // 'dic'
  CHARACTER(LEN=n_lmaxidentif)  :: ioc_identifier ! 'ioc_' // 'dic'
  CHARACTER(LEN=n_lmaxidentif)  :: icc_identifier ! 'icc_' // 'dic'
  ! ------------- List connectivity -------------
  TYPE(SYSTEMSINFO), POINTER    :: prev
  TYPE(SYSTEMSINFO), POINTER    :: next
  ! ---------------------------------------------
END TYPE


TYPE PROCESS                                      ! Example:
  ! ------------ General information ------------
  CHARACTER(LEN=n_lmaxprocname) :: name           ! 'OrgMatterOxicRemin'
  CHARACTER(LEN=n_lmaxprocname) :: subr           ! 'MONOD1_C'
  CHARACTER(LEN=n_lmaxshortid)  :: shortid        ! 'orgmoxic'
  INTEGER                       :: idx            ! := <order> of process
  INTEGER                       :: n_realms
  CHARACTER(LEN=n_lmaxidentif), &
    DIMENSION(:), POINTER       :: c_realm_identifiers
  ! ----------- Reactants and Products ----------
  INTEGER                       :: n_reactants
  INTEGER                       :: n_products
  TYPE(CHEMREAGENT), &
    DIMENSION(:), POINTER       :: cr_reacts
  CHARACTER(LEN=n_lmaxshortid), &
    DIMENSION(:), POINTER       :: c_rid
  CHARACTER(LEN=n_lmaxshortid), &
    DIMENSION(:), POINTER       :: c_rwildcards
  TYPE(CHEMREAGENT), &
    DIMENSION(:), POINTER       :: cr_prods
  CHARACTER(LEN=n_lmaxshortid), &
    DIMENSION(:), POINTER       :: c_pid
  CHARACTER(LEN=n_lmaxshortid), &
    DIMENSION(:), POINTER       :: c_pwildcards
  INTEGER                       :: i_ref          ! index to the reference element for the
                                                  ! rate law (< 0 if a reactant, > 0 if a product)
  ! -- Identifier names for the generated code --
  CHARACTER(LEN=n_lmaxidentif)  :: pp_identifier  ! ProcessParameter Identifier
  CHARACTER(LEN=n_lmaxidentif)  :: nvp_identifier ! Identifier for the variable in MEDUSA
                                                  ! that holds the number of variable
                                                  ! components affected by the process
  CHARACTER(LEN=n_lmaxidentif)  :: iop_identifier ! Identifier for the array in MEDUSA
                                                  ! that holds the IO-identifiers of the
                                                  ! components affected by the process
  CHARACTER(LEN=n_lmaxidentif)  :: nvr_identifier ! Identifier for the variable in MEDUSA
                                                  ! that holds the number of variable
                                                  ! components entering the rate law
  CHARACTER(LEN=n_lmaxidentif)  :: ior_identifier ! Identifier for the array in MEDUSA
                                                  ! that holds the IO-identifiers of the
                                                  ! components entering the ratelaw
  CHARACTER(LEN=n_lmaxidentif)  :: pi_identifier  ! Identifier for the variable in MEDUSA
                                                  ! that holds the ProcessID
  ! ----------------- Rate Law ------------------
  TYPE(RATELAW), POINTER        :: ratelaw
  INTEGER                       :: idx_xref       ! order of an xref'd ratelaw
  ! ------------- List connectivity -------------
  TYPE(PROCESS), POINTER        :: prev
  TYPE(PROCESS), POINTER        :: next
  ! ---------------------------------------------
END TYPE


TYPE EQUILIB
  ! ------------ General information ------------
  CHARACTER(LEN=n_lmaxprocname) :: name
  CHARACTER(LEN=n_lmaxprocname) :: subr
  CHARACTER(LEN=n_lmaxshortid)  :: shortid
  INTEGER                       :: idx
  ! ----------- Reactants and Products ----------
  INTEGER                       :: n_reactants
  INTEGER                       :: n_products
  TYPE(CHEMREAGENT), &
    DIMENSION(:), POINTER       :: cr_reacts
  CHARACTER(LEN=n_lmaxshortid), &
    DIMENSION(:), POINTER       :: c_rid
  TYPE(CHEMREAGENT), &
    DIMENSION(:), POINTER       :: cr_prods
  CHARACTER(LEN=n_lmaxshortid), &
    DIMENSION(:), POINTER       :: c_pid
  ! -- Identifier names for the generated code --
  CHARACTER(LEN=n_lmaxidentif)  :: ep_identifier  ! EquilibParameter Identifier
  CHARACTER(LEN=n_lmaxidentif)  :: nve_identifier ! Identifier for the variable in MEDUSA
                                                  ! that holds the number of variable
                                                  ! components affected by the equilibrium
  CHARACTER(LEN=n_lmaxidentif)  :: ioe_identifier ! Identifier for the array in MEDUSA
                                                  ! that holds the IO-identifiers of the
                                                  ! components affected by the equilibrium
  CHARACTER(LEN=n_lmaxidentif)  :: nvr_identifier ! Identifier the variable in MEDUSA
                                                  ! that holds the number of variable
                                                  ! components entering the equilibrium relat.
  CHARACTER(LEN=n_lmaxidentif)  :: ior_identifier ! Identifier for array array in MEDUSA
                                                  ! that holds the IO-identifiers
                                                  ! of components entering the equilibrium relat.
  CHARACTER(LEN=n_lmaxidentif)  :: ei_identifier  ! Identifier for the variable in MEDUSA
                                                  ! that holds the EquilibID
  ! ---------- Equilibrium Relationship ---------
  TYPE(EQLBREL), POINTER        :: eqlbrel
  ! ------------- List connectivity -------------
  TYPE(EQUILIB), POINTER        :: prev
  TYPE(EQUILIB), POINTER        :: next
  ! ---------------------------------------------
END TYPE


TYPE APIEXTENSION
  ! ------------ General information ------------
  CHARACTER(LEN=n_lmaxnamesgen) :: name
  CHARACTER(LEN=n_lmaxshortid)  :: shortid
  INTEGER                       :: idx
  INTEGER                       :: i_category     ! 0 (N/A), 1: wdata
  ! ------------------- Codes -------------------
  TYPE(CODEBITS),  POINTER      :: codes
  ! ------------- List connectivity -------------
  TYPE(APIEXTENSION), POINTER   :: prev
  TYPE(APIEXTENSION), POINTER   :: next
  ! ---------------------------------------------
END TYPE


!=======================================================================
 CONTAINS
!=======================================================================

!-----------------------------------------------------------------------
 FUNCTION COMPOINFO_createRoot() RESULT(c)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(COMPOINFO), POINTER :: c

NULLIFY(c)
ALLOCATE(c)
NULLIFY(c%prev)

CALL COMPOINFO_initNode(c)

RETURN
!-----------------------------------------------------------------------
 END FUNCTION COMPOINFO_createRoot
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE COMPOINFO_createNext(c)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(COMPOINFO), POINTER :: c

TYPE(COMPOINFO), POINTER :: c_wk

CHARACTER(LEN=*), PARAMETER :: c_fmterr_a =  '("[COMPOINFO_createNext] error: ", A)'

NULLIFY(c_wk)

IF (ASSOCIATED(c)) THEN

  IF (.NOT. ASSOCIATED(c%next)) THEN

    ALLOCATE(c_wk)
    c_wk%prev => c
    CALL COMPOINFO_initNode(c_wk)

    c%next => c_wk

    NULLIFY(c_wk)

  ELSE

    WRITE(jp_stderr,c_fmterr_a) '<c> has already a %next node -- aborting'
    CALL ABORT()

  ENDIF

ELSE

  WRITE(jp_stderr,c_fmterr_a) '<t_compoinfo_node> not yet ASSOCIATEd -- aborting'
  CALL ABORT()

ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE COMPOINFO_createNext
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE COMPOINFO_initNode(c)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(COMPOINFO), POINTER :: c

CHARACTER(LEN=*), PARAMETER :: c_fmterr_a =  '("[COMPOINFO_initNode] error: ", A)'


IF (ASSOCIATED(c)) THEN

  c%name           = '???'
  c%class          = '???'
  c%shortid        = '???'
  c%phasid         = '??'
  c%idx            = -1
  c%idx_xref       = -1
  NULLIFY(c%xref)
  c%i_category     = 0
  c%i_system       = 0
  c%n_params       = 0
  c%n_checom       = 0
  c%l_checom_from_xref = .FALSE.
  NULLIFY(c%param_name)
  NULLIFY(c%param_vartype)
  NULLIFY(c%param_varname)
  NULLIFY(c%param_values)

  NULLIFY(c%codes)

  c%n_consrv       = 0
  NULLIFY(c%consrv_name)
  NULLIFY(c%consrv_weights)

  ! do not touch t_compoinfo_node%prev
  NULLIFY(c%next)

ELSE

  WRITE(jp_stderr,c_fmterr_a) '<t_compoinfo_node> not yet ASSOCIATEd -- aborting'
  CALL ABORT()

ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE COMPOINFO_initNode
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE COMPOINFO_deleteNode(c)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(COMPOINFO), POINTER :: c


TYPE(COMPOINFO), POINTER :: c_wk
TYPE(CODEBITS),  POINTER :: b_c, b_n

CHARACTER(LEN=*), PARAMETER :: c_fmtwar_a =  '("[COMPOINFO_deleteNode] warning: ", A)'


IF (ASSOCIATED(c)) THEN

                                     ! First unconnect c from any possible list-pre-decessor
  IF (ASSOCIATED(c%prev)) THEN

    NULLIFY(c%prev%next)
    NULLIFY(c%prev)

  ENDIF


  c_wk => c

  DO WHILE(ASSOCIATED(c_wk%next))
    c_wk => c_wk%next
  ENDDO

  DO
    IF (ASSOCIATED(c_wk%param_name))    DEALLOCATE(c_wk%param_name)
    IF (ASSOCIATED(c_wk%param_vartype)) DEALLOCATE(c_wk%param_vartype)
    IF (ASSOCIATED(c_wk%param_varname)) DEALLOCATE(c_wk%param_varname)
    IF (ASSOCIATED(c_wk%param_values))  DEALLOCATE(c_wk%param_values)

    b_c => c_wk%codes
    DO WHILE(ASSOCIATED(b_c))
      b_n => b_c%next
      DEALLOCATE(b_c%code)
      DEALLOCATE(b_c)
      NULLIFY(b_c)
      b_c => b_n
    ENDDO

    IF (ASSOCIATED(c_wk%prev)) THEN
      c_wk => c_wk%prev
      DEALLOCATE(c_wk%next)
      CYCLE
    ELSE
      DEALLOCATE(c_wk)
      NULLIFY(c_wk)
      EXIT
    ENDIF
  ENDDO

  IF (ASSOCIATED(c_wk%consrv_name))    DEALLOCATE(c_wk%consrv_name)
  IF (ASSOCIATED(c_wk%consrv_weights)) DEALLOCATE(c_wk%consrv_weights)

  NULLIFY(c)

ELSE

  WRITE(jp_stdout,c_fmtwar_a) '<c> is not ASSOCIATEd'

ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE COMPOINFO_deleteNode
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 FUNCTION COMPOINFO_lastNode(c) RESULT(c_end)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(COMPOINFO), POINTER :: c
TYPE(COMPOINFO), POINTER :: c_end

TYPE(COMPOINFO), POINTER :: c_wk

IF (ASSOCIATED(c)) THEN
  c_wk => c
  DO WHILE(ASSOCIATED(c_wk%next))
    c_wk => c_wk%next
  ENDDO
  c_end => c_wk
ELSE
  NULLIFY(c_end)
ENDIF

NULLIFY(c_wk)

RETURN
!-----------------------------------------------------------------------
 END FUNCTION COMPOINFO_lastNode
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 FUNCTION APIEXTENSION_createRoot() RESULT(a)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(APIEXTENSION), POINTER :: a

NULLIFY(a)
ALLOCATE(a)
NULLIFY(a%prev)

CALL APIEXTENSION_initNode(a)

RETURN
!-----------------------------------------------------------------------
 END FUNCTION APIEXTENSION_createRoot
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE APIEXTENSION_createNext(a)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(APIEXTENSION), POINTER :: a

TYPE(APIEXTENSION), POINTER :: a_wk

CHARACTER(LEN=*), PARAMETER :: c_fmterr_a =  '("[APIEXTENSION_createNext] error: ", A)'

NULLIFY(a_wk)

IF (ASSOCIATED(a)) THEN

  IF (.NOT. ASSOCIATED(a%next)) THEN

    ALLOCATE(a_wk)
    a_wk%prev => a
    CALL APIEXTENSION_initNode(a_wk)

    a%next => a_wk

    NULLIFY(a_wk)

  ELSE

    WRITE(jp_stderr,c_fmterr_a) '<a> has already a %next node -- aborting'
    CALL ABORT()

  ENDIF

ELSE

  WRITE(jp_stderr,c_fmterr_a) '<a> not yet ASSOCIATEd -- aborting'
  CALL ABORT()

ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE APIEXTENSION_createNext
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE APIEXTENSION_initNode(a)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(APIEXTENSION), POINTER :: a

CHARACTER(LEN=*), PARAMETER :: c_fmterr_a =  '("[APIEXTENSION_initNode] error: ", A)'


IF (ASSOCIATED(a)) THEN

  a%name           = '???'
  a%shortid        = '???'
  a%idx            = -1
  a%i_category     = 0

  NULLIFY(a%codes)

  ! do not touch a%prev
  NULLIFY(a%next)

ELSE

  WRITE(jp_stderr,c_fmterr_a) '<a> not yet ASSOCIATEd -- aborting'
  CALL ABORT()

ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE APIEXTENSION_initNode
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE APIEXTENSION_deleteNode(a)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(APIEXTENSION), POINTER :: a


TYPE(APIEXTENSION), POINTER :: a_wk
TYPE(CODEBITS),  POINTER :: b_c, b_n

CHARACTER(LEN=*), PARAMETER :: c_fmtwar_a =  '("[APIEXTENSION_deleteNode] warning: ", A)'


IF (ASSOCIATED(a)) THEN

                                     ! First unconnect a from any possible list-pre-decessor
  IF (ASSOCIATED(a%prev)) THEN

    NULLIFY(a%prev%next)
    NULLIFY(a%prev)

  ENDIF


  a_wk => a

  DO WHILE(ASSOCIATED(a_wk%next))
    a_wk => a_wk%next
  ENDDO

  DO

    b_c => a_wk%codes
    DO WHILE(ASSOCIATED(b_c))
      b_n => b_c%next
      DEALLOCATE(b_c%code)
      DEALLOCATE(b_c)
      NULLIFY(b_c)
      b_c => b_n
    ENDDO

    IF (ASSOCIATED(a_wk%prev)) THEN
      a_wk => a_wk%prev
      DEALLOCATE(a_wk%next)
      CYCLE
    ELSE
      DEALLOCATE(a_wk)
      NULLIFY(a_wk)
      EXIT
    ENDIF
  ENDDO

  NULLIFY(a)

ELSE

  WRITE(jp_stdout,c_fmtwar_a) '<a> is not ASSOCIATEd'

ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE APIEXTENSION_deleteNode
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 FUNCTION APIEXTENSION_lastNode(a) RESULT(a_end)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(APIEXTENSION), POINTER :: a
TYPE(APIEXTENSION), POINTER :: a_end

TYPE(APIEXTENSION), POINTER :: a_wk

IF (ASSOCIATED(a)) THEN
  a_wk => a
  DO WHILE(ASSOCIATED(a_wk%next))
    a_wk => a_wk%next
  ENDDO
  a_end => a_wk
ELSE
  NULLIFY(a_end)
ENDIF

NULLIFY(a_wk)

RETURN
!-----------------------------------------------------------------------
 END FUNCTION APIEXTENSION_lastNode
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE APIEXTENSION_dump(a, iunit, indent, indent_inc)
!-----------------------------------------------------------------------
IMPLICIT NONE
TYPE(APIEXTENSION), TARGET :: a
INTEGER, INTENT(IN)   :: iunit
INTEGER, INTENT(IN)   :: indent, indent_inc


INTEGER               :: i
CHARACTER(LEN=20)     :: c_fmt1
CHARACTER(LEN=254)    :: c_fmt, c_fmta, c_fmti, c_fmtia, c_fmtl
TYPE(CODEBITS), POINTER :: t_codes
WRITE(c_fmt1, '()')
IF (indent > 0) WRITE(c_fmt1, '(I0,"X, ")') indent

c_fmt   = '(' // TRIM(c_fmt1) // ' A)'
c_fmta  = '(' // TRIM(c_fmt1) // ' A, " = <", A, "> (", I0, "/", I0, ")")'
c_fmti  = '(' // TRIM(c_fmt1) // ' A, " = ", I0)'
c_fmtl  = '(' // TRIM(c_fmt1) // ' A, " = ", L1)'
c_fmtia = '(' // TRIM(c_fmt1) // ' A, "(",I0,") = <", A, "> (", I0, "/", I0, ")")'

WRITE(iunit, c_fmta) '%name',    TRIM(a%name),    LEN_TRIM(a%name),    n_lmaxprocname
WRITE(iunit, c_fmta) '%shortid', TRIM(a%shortid), LEN_TRIM(a%shortid), n_lmaxshortid
WRITE(iunit, c_fmti) '%i_category',  a%i_category
WRITE(iunit, c_fmti) '%idx',         a%idx


i = 0
IF (ASSOCIATED(a%codes)) THEN
  WRITE(iunit, c_fmt) '%codes is ASSOCIATED'
  t_codes => a%codes
  DO WHILE(ASSOCIATED(t_codes))
    i = i+1
    WRITE(iunit, c_fmta) '%codes%type', TRIM(t_codes%type), &
                           LEN_TRIM(t_codes%type), n_lmaxcomptyp
    WRITE(iunit, c_fmta) '%codes%units', TRIM(t_codes%units), &
                           LEN_TRIM(t_codes%units), n_lmaxexpress
    WRITE(iunit, c_fmta) '%codes%varname', TRIM(t_codes%varname), &
                           LEN_TRIM(t_codes%varname), n_lmaxidentif
    WRITE(iunit, c_fmta) '%codes%vartype', TRIM(t_codes%vartype), &
                           LEN_TRIM(t_codes%vartype), n_lmaxcodeline
    WRITE(iunit, c_fmti) '%codes%nlines', t_codes%nlines
    t_codes => t_codes%next
  ENDDO
ELSE
  WRITE(iunit, c_fmt) '%codes is not ASSOCIATED'
ENDIF


IF (ASSOCIATED(a%prev)) THEN
  WRITE(iunit, c_fmt) '%prev is ASSOCIATED'
ELSE
  WRITE(iunit, c_fmt) '%prev is not ASSOCIATED'
ENDIF

IF (ASSOCIATED(a%next)) THEN
  WRITE(iunit, c_fmt) '%next is ASSOCIATED'
ELSE
  WRITE(iunit, c_fmt) '%next is not ASSOCIATED'
ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE APIEXTENSION_dump
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 FUNCTION PROCESS_lastNode(p) RESULT(p_end)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(PROCESS), POINTER :: p
TYPE(PROCESS), POINTER :: p_end

TYPE(PROCESS), POINTER :: p_wk

IF (ASSOCIATED(p)) THEN
  p_wk => p
  DO WHILE(ASSOCIATED(p_wk%next))
    p_wk => p_wk%next
  ENDDO
  p_end => p_wk
ELSE
  NULLIFY(p_end)
ENDIF

NULLIFY(p_wk)

RETURN
!-----------------------------------------------------------------------
 END FUNCTION PROCESS_lastNode
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE COMPOINFO_dump(c, iunit, indent, indent_inc)
!-----------------------------------------------------------------------
IMPLICIT NONE
TYPE(COMPOINFO), TARGET :: c
INTEGER, INTENT(IN)   :: iunit
INTEGER, INTENT(IN)   :: indent, indent_inc


INTEGER               :: i
CHARACTER(LEN=20)     :: c_fmt1
CHARACTER(LEN=254)    :: c_fmt, c_fmta, c_fmti, c_fmtia, c_fmtl
TYPE(CODEBITS), POINTER :: t_codes
WRITE(c_fmt1, '()')
IF (indent > 0) WRITE(c_fmt1, '(I0,"X, ")') indent

c_fmt   = '(' // TRIM(c_fmt1) // ' A)'
c_fmta  = '(' // TRIM(c_fmt1) // ' A, " = <", A, "> (", I0, "/", I0, ")")'
c_fmti  = '(' // TRIM(c_fmt1) // ' A, " = ", I0)'
c_fmtl  = '(' // TRIM(c_fmt1) // ' A, " = ", L1)'
c_fmtia = '(' // TRIM(c_fmt1) // ' A, "(",I0,") = <", A, "> (", I0, "/", I0, ")")'

WRITE(iunit, c_fmta) '%name',    TRIM(c%name),    LEN_TRIM(c%name),    n_lmaxprocname
WRITE(iunit, c_fmta) '%class',   TRIM(c%class),   LEN_TRIM(c%class),   n_lmaxcomptyp
WRITE(iunit, c_fmta) '%shortid', TRIM(c%shortid), LEN_TRIM(c%shortid), n_lmaxshortid
WRITE(iunit, c_fmta) '%phasid',  TRIM(c%phasid),  LEN_TRIM(c%phasid),  n_lmaxphasid
WRITE(iunit, c_fmti) '%i_category',  c%i_category
WRITE(iunit, c_fmti) '%idx',         c%idx
WRITE(iunit, c_fmti) '%idx_xref',    c%idx_xref
IF (ASSOCIATED(c%xref)) THEN
  WRITE(iunit, c_fmt) '%xref is ASSOCIATED'
ELSE
  WRITE(iunit, c_fmt) '%xref is not ASSOCIATED'
ENDIF
WRITE(iunit, c_fmti) '%i_system',    c%i_system
WRITE(iunit, c_fmti) '%n_params',    c%n_params
WRITE(iunit, c_fmti) '%n_checom',    c%n_checom
WRITE(iunit, c_fmtl) '%l_checom_from_xref',    c%l_checom_from_xref

IF (ASSOCIATED(c%param_name)) THEN
  DO i = 1, SIZE(c%param_name)
    WRITE(iunit, c_fmtia) '%param_name', i, TRIM(c%param_name(i)), &
                           LEN_TRIM(c%param_name(i)), n_lmaxidentif
!  ENDDO
!ENDIF

!IF (ASSOCIATED(c%param_vartype)) THEN
!  DO i = 1, SIZE(c%param_vartype)
    WRITE(iunit, c_fmtia) '%param_vartype', i, TRIM(c%param_vartype(i)), &
                           LEN_TRIM(c%param_vartype(i)), n_lmaxexpress
!  ENDDO
!ENDIF

!IF (ASSOCIATED(c%param_varname)) THEN
!  DO i = 1, SIZE(c%param_varname)
    WRITE(iunit, c_fmtia) '%param_varname', i, TRIM(c%param_varname(i)), &
                           LEN_TRIM(c%param_varname(i)), n_lmaxidentif
!  ENDDO
!ENDIF

!IF (ASSOCIATED(c%param_values)) THEN
!  DO i = 1, SIZE(c%param_values)
    WRITE(iunit, c_fmtia) '%param_values', i, TRIM(c%param_values(i)), &
                           LEN_TRIM(c%param_values(i)), n_lmaxexpress
  ENDDO
ENDIF

i = 0
IF (ASSOCIATED(c%codes)) THEN
  WRITE(iunit, c_fmt) '%codes is ASSOCIATED'
  t_codes => c%codes
  DO WHILE(ASSOCIATED(t_codes))
    i = i+1
    WRITE(iunit, c_fmta) '%codes%type', TRIM(t_codes%type), &
                           LEN_TRIM(t_codes%type), n_lmaxcomptyp
    WRITE(iunit, c_fmta) '%codes%units', TRIM(t_codes%units), &
                           LEN_TRIM(t_codes%units), n_lmaxexpress
    WRITE(iunit, c_fmta) '%codes%varname', TRIM(t_codes%varname), &
                           LEN_TRIM(t_codes%varname), n_lmaxidentif
    WRITE(iunit, c_fmta) '%codes%vartype', TRIM(t_codes%vartype), &
                           LEN_TRIM(t_codes%vartype), n_lmaxcodeline
    WRITE(iunit, c_fmti) '%codes%nlines', t_codes%nlines
    t_codes => t_codes%next
  ENDDO
ELSE
  WRITE(iunit, c_fmt) '%codes is not ASSOCIATED'
ENDIF

WRITE(iunit, c_fmti) '%n_consrv',  c%n_consrv

IF (ASSOCIATED(c%consrv_name)) THEN
  DO i = 1, SIZE(c%consrv_name)
    WRITE(iunit, c_fmtia) '%consrv_name', i, TRIM(c%consrv_name(i)), &
                           LEN_TRIM(c%consrv_name(i)), n_lmaxidentif
    WRITE(iunit, c_fmtia) '%consrv_weights', i, TRIM(c%consrv_weights(i)), &
                           LEN_TRIM(c%consrv_weights(i)), n_lmaxexpress
  ENDDO
ENDIF

IF (ASSOCIATED(c%prev)) THEN
  WRITE(iunit, c_fmt) '%prev is ASSOCIATED'
ELSE
  WRITE(iunit, c_fmt) '%prev is not ASSOCIATED'
ENDIF

IF (ASSOCIATED(c%next)) THEN
  WRITE(iunit, c_fmt) '%next is ASSOCIATED'
ELSE
  WRITE(iunit, c_fmt) '%next is not ASSOCIATED'
ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE COMPOINFO_dump
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 FUNCTION COMPOINFO_indexName(c) RESULT(idx)
!-----------------------------------------------------------------------

IMPLICIT NONE


! Argument list and results variables
! -----------------------------------

TYPE(COMPOINFO), TARGET        :: c
INTEGER, DIMENSION(:), POINTER :: idx


! Local variables
! ---------------

INTEGER                        :: i, j, n
TYPE(COMPOINFO), POINTER       :: c_i, c_j

! First count the list elements in <c>
n = 0
c_i => c

DO WHILE(ASSOCIATED(c_i))
  n = n+1
  c_i => c_i%next
ENDDO

IF(n == 0) THEN
  NULLIFY(idx)
  RETURN
ENDIF

ALLOCATE(idx(n))
! Initialize each item of <idx> to its 1D position index
idx(:) = (/ (i, i=1, n) /)

c_i => c

DO i = 1, n-1

  ! If the idx(i) does not have the value 'i', then, and only
  ! then, its value has been modified before (i.e., it has been
  ! detected as a duplicate of a preceeding value).
  ! Else: we compare each of list elements following the i'th one,
  ! and if we find a duplicate, we reset its position index to 'i'
  IF(idx(i) == i) THEN
    c_j => c_i
    DO j = i+1, n
      c_j => c_j%next
      IF (idx(j) == j) THEN
        IF (c_i%name == c_j%name) idx(j) = i
      ENDIF
    ENDDO
  ENDIF

  c_i => c_i%next

ENDDO

RETURN

!-----------------------------------------------------------------------
 END FUNCTION COMPOINFO_indexName
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 FUNCTION COMPOINFO_getNodeByName(c, str_name) RESULT(c_node)
!-----------------------------------------------------------------------

IMPLICIT NONE


! Argument list and results variables
! -----------------------------------

TYPE(COMPOINFO), TARGET        :: c
CHARACTER(LEN=*), INTENT(IN)   :: str_name
TYPE(COMPOINFO), POINTER       :: c_node


! Local variables
! ---------------

TYPE(COMPOINFO), POINTER       :: c_i

NULLIFY(c_node)

c_i => c

DO WHILE(ASSOCIATED(c_i))

  IF (c_i%name == str_name) THEN
    c_node => c_i
    RETURN
  ENDIF

  c_i => c_i%next

ENDDO

RETURN

!-----------------------------------------------------------------------
 END FUNCTION COMPOINFO_getNodeByName
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 FUNCTION COMPOINFO_getConsrvIdxByName(c, str_name) RESULT(i_consrv)
!-----------------------------------------------------------------------

IMPLICIT NONE


! Argument list and results variables
! -----------------------------------

TYPE(COMPOINFO), POINTER       :: c
CHARACTER(LEN=*), INTENT(IN)   :: str_name
INTEGER                        :: i_consrv


! Local variables
! ---------------

INTEGER :: i
CHARACTER(LEN=*), PARAMETER :: c_fmterr_a =  '("[COMPOINFO_getConsrvIdxByName] error: ", A)'


IF(.NOT. ASSOCIATED(c)) THEN
  WRITE(jp_stderr,c_fmterr_a) '<t_compoinfo_node> not ASSOCIATEd -- aborting'
  CALL ABORT()
ENDIF

i_consrv = -1

DO i = 1, c%n_consrv
  IF(c%consrv_name(i) == str_name) THEN
    i_consrv = i
    EXIT
  ENDIF
ENDDO

RETURN

!-----------------------------------------------------------------------
 END FUNCTION COMPOINFO_getConsrvIdxByName
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 FUNCTION COMPOINFO_getParamIdxByName(c, str_name) RESULT(i_param)
!-----------------------------------------------------------------------

IMPLICIT NONE


! Argument list and results variables
! -----------------------------------

TYPE(COMPOINFO), POINTER       :: c
CHARACTER(LEN=*), INTENT(IN)   :: str_name
INTEGER                        :: i_param


! Local variables
! ---------------

INTEGER :: i
CHARACTER(LEN=*), PARAMETER :: c_fmterr_a =  '("[COMPOINFO_getParamIdxByName] error: ", A)'


IF(.NOT. ASSOCIATED(c)) THEN
  WRITE(jp_stderr,c_fmterr_a) '<t_compoinfo_node> not ASSOCIATEd -- aborting'
  CALL ABORT()
ENDIF

i_param = -1

DO i = 1, c%n_params
  IF(c%param_name(i) == str_name) THEN
    i_param = i
    EXIT
  ENDIF
ENDDO

RETURN

!-----------------------------------------------------------------------
 END FUNCTION COMPOINFO_getParamIdxByName
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 FUNCTION COMPOINFO_getNodeByShortid(c, str_shortid) RESULT(c_node)
!-----------------------------------------------------------------------

IMPLICIT NONE


! Argument list and results variables
! -----------------------------------

TYPE(COMPOINFO), TARGET        :: c
CHARACTER(LEN=*), INTENT(IN)   :: str_shortid
TYPE(COMPOINFO), POINTER       :: c_node


! Local variables
! ---------------

TYPE(COMPOINFO), POINTER       :: c_i

NULLIFY(c_node)

c_i => c

DO WHILE(ASSOCIATED(c_i))

  IF (c_i%shortid == str_shortid) THEN
    c_node => c_i
    RETURN
  ENDIF

  c_i => c_i%next

ENDDO

RETURN

!-----------------------------------------------------------------------
 END FUNCTION COMPOINFO_getNodeByShortid
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 FUNCTION COMPOINFO_indexShortid(c) RESULT(idx)
!-----------------------------------------------------------------------
IMPLICIT NONE


! Argument list and results variables
! -----------------------------------

TYPE(COMPOINFO), TARGET        :: c
INTEGER, DIMENSION(:), POINTER :: idx


! Local variables
! ---------------

INTEGER                        :: i, j, n
TYPE(COMPOINFO), POINTER       :: c_i, c_j


! First count the list elements in <c>
n = 0
c_i => c

DO WHILE(ASSOCIATED(c_i))
  n = n+1
  c_i => c_i%next
ENDDO

IF(n == 0) THEN
  NULLIFY(idx)
  RETURN
ENDIF

ALLOCATE(idx(n))
! Initialize each item of <idx> to its 1D position index
idx(:) = (/ (i, i=1, n) /)

c_i => c

DO i = 1, n-1

  ! If idx(i) does not have the value 'i', then, and only
  !   then, its value has been modified before (i.e., it has been
  !   detected as a duplicate of a preceeding value), so it is done.
  ! Else: we compare each of list elements following the i'th one,
  !   and if we find a duplicate, we reset its position index to 'i'
  IF(idx(i) == i) THEN
    IF(c_i%shortid == '???') CYCLE
    c_j => c_i
    DO j = i+1, n
      c_j => c_j%next
      IF (idx(j) == j) THEN
        IF (c_i%shortid == c_j%shortid) idx(j) = i
      ENDIF
    ENDDO
  ENDIF

  c_i => c_i%next

ENDDO

RETURN

!-----------------------------------------------------------------------
 END FUNCTION COMPOINFO_indexShortid
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 FUNCTION PROCESS_createRoot() RESULT(p)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(PROCESS), POINTER :: p

NULLIFY(p)
ALLOCATE(p)
NULLIFY(p%prev)

CALL PROCESS_initNode(p)

RETURN
!-----------------------------------------------------------------------
 END FUNCTION PROCESS_createRoot
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE PROCESS_createNext(p)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(PROCESS), POINTER :: p

TYPE(PROCESS), POINTER :: p_wk

CHARACTER(LEN=*), PARAMETER :: c_fmterr_a = '("[PROCESS_createNext] error: ", A)'

NULLIFY(p_wk)

IF (ASSOCIATED(p)) THEN

  IF (.NOT. ASSOCIATED(p%next)) THEN

    ALLOCATE(p_wk)
    p_wk%prev => p
    CALL PROCESS_initNode(p_wk)

    p%next => p_wk

    NULLIFY(p_wk)

  ELSE

    WRITE(jp_stderr,c_fmterr_a) '<p> has already a %next node -- aborting'

  CALL ABORT()

  ENDIF
ELSE
  WRITE(jp_stderr,c_fmterr_a) '<p> not yet ASSOCIATEd -- aborting'
  CALL ABORT()
ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE PROCESS_createNext
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE PROCESS_initNode(p)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(PROCESS), POINTER :: p

CHARACTER(LEN=*), PARAMETER :: c_fmterr_a = '("[PROCESS_initNode] error: ", A)'


IF (ASSOCIATED(p)) THEN

  p%idx            = -1
  p%name           = '???'
  p%subr           = '???'
  p%n_realms       = -1
  NULLIFY(p%c_realm_identifiers)

  p%n_reactants    = -1
  p%n_products     = -1
  NULLIFY(p%cr_reacts)
  NULLIFY(p%c_rid)
  NULLIFY(p%c_rwildcards)
  NULLIFY(p%cr_prods)
  NULLIFY(p%c_pid)
  NULLIFY(p%c_pwildcards)
  p%i_ref          = 0

  p%pp_identifier  = 'pp_???'  ! ProcessParameter Identifier
  p%nvp_identifier = 'nvp_???' ! Identifier for number of variable
                               ! components affected by process
  p%iop_identifier = 'iop_???' ! Identifier for array with IO-identifiers
                               ! of components affected by process
  p%nvr_identifier = 'nvr_???' ! Identifier for number of variable
                               ! components entering the rate law
  p%ior_identifier = 'ior_???' ! Identifier for array with IO-identifiers
                               ! of components entering the ratelaw
  p%pi_identifier  = 'pi_???'  ! Identifier for ProcessID

  NULLIFY(p%ratelaw)
  p%idx_xref = 0               ! order of an xref'd ratelaw

  ! do not touch p%prev
  NULLIFY(p%next)

ELSE

  WRITE(jp_stderr,c_fmterr_a) '<p> not yet ASSOCIATEd -- aborting'
  CALL ABORT()

ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE PROCESS_initNode
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE PROCESS_dump(p, iunit, indent, indent_inc)
!-----------------------------------------------------------------------
IMPLICIT NONE
TYPE(PROCESS), TARGET :: p
INTEGER, INTENT(IN)   :: iunit
INTEGER, INTENT(IN)   :: indent, indent_inc


INTEGER               :: i
CHARACTER(LEN=20)     :: c_fmt1, c_fmt2
CHARACTER(LEN=254)    :: c_fmt, c_fmta, c_fmti, c_fmtia, c_fmtna

WRITE(c_fmt1, '()')
IF (indent > 0) WRITE(c_fmt1, '(I0,"X, ")') indent
IF (indent_inc > 0) THEN
  IF (indent > 0) THEN
    WRITE(c_fmt2, '(I0,"X, ")') indent + indent_inc
  ELSE
    WRITE(c_fmt2, '(I0,"X, ")') indent_inc
  ENDIF
ELSE
  c_fmt2 = c_fmt1
ENDIF

c_fmt   = '(' // TRIM(c_fmt1) // ' A)'
c_fmta  = '(' // TRIM(c_fmt1) // ' A, " = <", A, "> (", I0, "/", I0, ")")'
c_fmti  = '(' // TRIM(c_fmt1) // ' A, " = ", I0)'
c_fmtia = '(' // TRIM(c_fmt1) // ' A, "(",I0,") ", A)'
c_fmtna = '(A, " = <", A, "> (", I0, "/", I0, ")")'

WRITE(iunit, c_fmti) '%idx', p%idx
WRITE(iunit, c_fmta) '%name',    TRIM(p%name),    LEN_TRIM(p%name),    n_lmaxprocname
WRITE(iunit, c_fmta) '%subr',    TRIM(p%subr),    LEN_TRIM(p%subr),    n_lmaxprocname
WRITE(iunit, c_fmti) '%n_realms', p%n_realms
IF (ASSOCIATED(p%c_realm_identifiers)) THEN
  DO i = 1, SIZE(p%c_realm_identifiers)
    WRITE(iunit, c_fmtia) '%c_realm_identifiers', i, '<' // TRIM(p%c_realm_identifiers(i)) // '>'
  ENDDO
ELSE
  WRITE(iunit, c_fmt) '%c_realm_identifiers is not ASSOCIATED'
ENDIF

WRITE(iunit, c_fmti) '%n_reactants', p%n_reactants
WRITE(iunit, c_fmti) '%n_products', p%n_products

IF (ASSOCIATED(p%cr_reacts)) THEN
  DO i = 1, SIZE(p%cr_reacts)
    WRITE(iunit, c_fmtia) '%cr_reacts', i, '...'
    CALL CHEMREAGENT_dump(p%cr_reacts(i), iunit, indent+indent_inc)
    IF(ASSOCIATED(p%c_rid)) THEN
      IF(LEN_TRIM(p%c_rid(i)) /= 0) THEN
        WRITE(iunit, c_fmtia, ADVANCE='NO') '%c_rid', i
        WRITE(iunit, c_fmtna) '',    TRIM(p%c_rid(i)), &
          LEN_TRIM(p%c_rid(i)),    n_lmaxshortid
      ENDIF
    ENDIF
    IF(ASSOCIATED(p%c_rwildcards)) THEN
      IF(LEN_TRIM(p%c_rwildcards(i)) /= 0) THEN
        WRITE(iunit, c_fmtia, ADVANCE='NO') '%c_rwildcards', i
        WRITE(iunit, c_fmtna) '',    TRIM(p%c_rwildcards(i)), &
          LEN_TRIM(p%c_rwildcards(i)),    n_lmaxshortid
      ENDIF
    ENDIF
  ENDDO
ENDIF

IF (ASSOCIATED(p%cr_prods)) THEN
  DO i = 1, SIZE(p%cr_prods)
    WRITE(iunit, c_fmtia) '%cr_prods', i, '...'
    CALL CHEMREAGENT_dump(p%cr_prods(i), iunit, indent+indent_inc)
    IF(ASSOCIATED(p%c_pid)) THEN
      IF(LEN_TRIM(p%c_pid(i)) /= 0) THEN
        WRITE(iunit, c_fmtia, ADVANCE='NO') '%c_pid', i
        WRITE(iunit, c_fmtna) '',    TRIM(p%c_pid(i)), &
          LEN_TRIM(p%c_pid(i)),    n_lmaxshortid
      ENDIF
    ENDIF
    IF(ASSOCIATED(p%c_pwildcards)) THEN
      IF(LEN_TRIM(p%c_pwildcards(i)) /= 0) THEN
        WRITE(iunit, c_fmtia, ADVANCE='NO') '%c_pwildcards', i
        WRITE(iunit, c_fmtna) '',    TRIM(p%c_pwildcards(i)), &
          LEN_TRIM(p%c_pwildcards(i)),    n_lmaxshortid
      ENDIF
    ENDIF
  ENDDO
ENDIF

WRITE(iunit, c_fmta) '%pp_identifier',    TRIM(p%pp_identifier), &
  LEN_TRIM(p%pp_identifier),    n_lmaxidentif
WRITE(iunit, c_fmta) '%nvp_identifier',    TRIM(p%nvp_identifier), &
  LEN_TRIM(p%nvp_identifier),    n_lmaxidentif
WRITE(iunit, c_fmta) '%iop_identifier',    TRIM(p%iop_identifier), &
  LEN_TRIM(p%iop_identifier),    n_lmaxidentif
WRITE(iunit, c_fmta) '%nvr_identifier',    TRIM(p%nvr_identifier), &
  LEN_TRIM(p%nvr_identifier),    n_lmaxidentif
WRITE(iunit, c_fmta) '%ior_identifier',    TRIM(p%ior_identifier), &
  LEN_TRIM(p%ior_identifier),    n_lmaxidentif
WRITE(iunit, c_fmta) '%pi_identifier',    TRIM(p%pi_identifier), &
  LEN_TRIM(p%pi_identifier),    n_lmaxidentif

IF(ASSOCIATED(p%ratelaw)) THEN
  WRITE(iunit, c_fmt) '%ratelaw ...'
  CALL RATELAW_dump(p%ratelaw, iunit, indent+indent_inc, indent_inc)
ELSE
  WRITE(iunit, c_fmt) '%ratelaw is not ASSOCIATED'
ENDIF

WRITE(iunit, c_fmti) '%i_ref', p%i_ref

WRITE(iunit, c_fmti) '%idx_xref', p%idx_xref

IF (ASSOCIATED(p%prev)) THEN
  WRITE(iunit, c_fmt) '%prev is ASSOCIATED'
ELSE
  WRITE(iunit, c_fmt) '%prev is not ASSOCIATED'
ENDIF

IF (ASSOCIATED(p%next)) THEN
  WRITE(iunit, c_fmt) '%next is ASSOCIATED'
ELSE
  WRITE(iunit, c_fmt) '%next is not ASSOCIATED'
ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE PROCESS_dump
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
 SUBROUTINE CHEMREAGENT_dump(cr, iunit, indent)
!-----------------------------------------------------------------------
IMPLICIT NONE
TYPE(CHEMREAGENT), TARGET :: cr
INTEGER, INTENT(IN)    :: iunit
INTEGER, INTENT(IN)    :: indent


CHARACTER(LEN=20)      :: c_fmt1
CHARACTER(LEN=254)     :: c_fmta

WRITE(c_fmt1, '()')
IF (indent > 0) WRITE(c_fmt1, '(I0,"X, ")') indent

c_fmta = '(' // TRIM(c_fmt1) // ' A, " = <", A, "> (", I0, "/", I0, ")")'

WRITE(iunit, c_fmta) '%name',    TRIM(cr%name),    LEN_TRIM(cr%name),    n_lmaxshortid
WRITE(iunit, c_fmta) '%shortid', TRIM(cr%shortid), LEN_TRIM(cr%shortid), n_lmaxnamesgen
WRITE(iunit, c_fmta) '%phasid',  TRIM(cr%phasid),  LEN_TRIM(cr%phasid),  n_lmaxphasid
WRITE(iunit, c_fmta) '%stoech',  TRIM(cr%stoech),  LEN_TRIM(cr%stoech),  n_lmaxexpress

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE CHEMREAGENT_dump
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
 FUNCTION RATELAW_createRoot() RESULT(rl)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(RATELAW), POINTER :: rl

INTEGER, PARAMETER :: jp_param0 = 0

NULLIFY(rl)
ALLOCATE(rl)
rl%n_params = jp_param0      ! needs to be set here (RATELAW_init depends on this value)

CALL RATELAW_init(rl, jp_param0)

RETURN
!-----------------------------------------------------------------------
 END FUNCTION RATELAW_createRoot
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE RATELAW_init(rl, n_params)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(RATELAW), POINTER :: rl
INTEGER, INTENT(IN)    :: n_params

CHARACTER(LEN=*), PARAMETER :: c_fmterr_a = '("[RATELAW_init] error: ", A)'


IF (ASSOCIATED(rl)) THEN

  rl%name           = '???'
  rl%pp_type        = 'PP_???'
  rl%expression     = ''
  rl%xrefprocname   = ''

  IF(rl%n_params /= 0) THEN

    rl%n_params = n_params

    DEALLOCATE(rl%paramname)
    DEALLOCATE(rl%paramcode)
    DEALLOCATE(rl%typecomponame)
    DEALLOCATE(rl%xmltagname)
    DEALLOCATE(rl%xmlattstocheck)
    DEALLOCATE(rl%kindofparam)
    DEALLOCATE(rl%dummylabel)
    DEALLOCATE(rl%iparam_xref)

  ENDIF


  IF(n_params > 0) THEN

    rl%n_params = n_params

    ALLOCATE(rl%paramname(n_params))
    ALLOCATE(rl%paramcode(n_params))
    ALLOCATE(rl%typecomponame(n_params))
    ALLOCATE(rl%xmltagname(n_params))
    ALLOCATE(rl%xmlattstocheck(n_params))
    ALLOCATE(rl%kindofparam(n_params))
    ALLOCATE(rl%dummylabel(n_params))
    ALLOCATE(rl%iparam_xref(n_params))

  ELSE

    rl%n_params = 0

    NULLIFY(rl%paramname)
    NULLIFY(rl%paramcode)
    NULLIFY(rl%typecomponame)
    NULLIFY(rl%xmltagname)
    NULLIFY(rl%xmlattstocheck)
    NULLIFY(rl%kindofparam)
    NULLIFY(rl%dummylabel)
    NULLIFY(rl%iparam_xref)

  ENDIF

ELSE

  WRITE(jp_stderr,c_fmterr_a) '<t_ratelaw_node> not yet ASSOCIATEd -- aborting'
  CALL ABORT()

ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE RATELAW_init
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE RATELAW_dump(rl, iunit, indent, indent_inc)
!-----------------------------------------------------------------------
IMPLICIT NONE
TYPE(RATELAW), TARGET :: rl
INTEGER, INTENT(IN)    :: iunit
INTEGER, INTENT(IN)    :: indent, indent_inc


INTEGER                :: i
CHARACTER(LEN=20)      :: c_fmt1
CHARACTER(LEN=254)     :: c_fmt, c_fmta, c_fmti, c_fmtia, c_fmtii

WRITE(c_fmt1, '()')
IF (indent > 0) WRITE(c_fmt1, '(I0,"X, ")') indent

c_fmt   = '(' // TRIM(c_fmt1) // ' A)'
c_fmta  = '(' // TRIM(c_fmt1) // ' A, " = <", A, "> (", I0, "/", I0, ")")'
c_fmti  = '(' // TRIM(c_fmt1) // ' A, " = ", I0)'
c_fmtia = '(' // TRIM(c_fmt1) // ' A, "(",I0,") = <", A, "> (", I0, "/", I0, ")")'
c_fmtii = '(' // TRIM(c_fmt1) // ' A, "(",I0,") = ", I0)'
WRITE(iunit, c_fmta) '%name',    TRIM(rl%name),    LEN_TRIM(rl%name),    n_lmaxprocname
WRITE(iunit, c_fmta) '%pp_type', TRIM(rl%pp_type), LEN_TRIM(rl%pp_type), n_lmaxidentif
WRITE(iunit, c_fmta) '%xrefprocname', TRIM(rl%xrefprocname), LEN_TRIM(rl%xrefprocname), n_lmaxprocname
WRITE(iunit, c_fmti) '%n_params', rl%n_params
WRITE(iunit, c_fmta) '%expression', TRIM(rl%expression), LEN_TRIM(rl%expression), n_lmaxexpress
IF (ASSOCIATED(rl%paramname)) THEN
  DO i = 1, SIZE(rl%paramname)
    WRITE(iunit, c_fmtia) '%paramname', i, TRIM(rl%paramname(i)), &
                           LEN_TRIM(rl%paramname(i)), n_lmaxidentif
  ENDDO
ELSE
  WRITE(iunit, c_fmt) '%paramname is not ASSOCIATED'
ENDIF
IF (ASSOCIATED(rl%paramcode)) THEN
  DO i = 1, SIZE(rl%paramcode)
    WRITE(iunit, c_fmtia) '%paramcode', i, TRIM(rl%paramcode(i)), &
                           LEN_TRIM(rl%paramcode(i)), n_lmaxexpress
  ENDDO
ELSE
  WRITE(iunit, c_fmt) '%paramcode is not ASSOCIATED'
ENDIF
IF (ASSOCIATED(rl%typecomponame)) THEN
  DO i = 1, SIZE(rl%typecomponame)
    WRITE(iunit, c_fmtia) '%typecomponame', i, TRIM(rl%typecomponame(i)), &
                           LEN_TRIM(rl%typecomponame(i)), n_lmaxidentif
  ENDDO
ELSE
  WRITE(iunit, c_fmt) '%typecomponame is not ASSOCIATED'
ENDIF
IF (ASSOCIATED(rl%xmltagname)) THEN
  DO i = 1, SIZE(rl%xmltagname)
    WRITE(iunit, c_fmtia) '%xmltagname', i, TRIM(rl%xmltagname(i)), &
                           LEN_TRIM(rl%xmltagname(i)), n_lmaxidentif
  ENDDO
ELSE
  WRITE(iunit, c_fmt) '%xmltagname is not ASSOCIATED'
ENDIF
IF (ASSOCIATED(rl%xmlattstocheck)) THEN
  DO i = 1, SIZE(rl%xmlattstocheck)
    WRITE(iunit, c_fmtia) '%xmlattstocheck', i, TRIM(rl%xmlattstocheck(i)), &
                           LEN_TRIM(rl%xmlattstocheck(i)), n_lmaxidentif
  ENDDO
ELSE
  WRITE(iunit, c_fmt) '%xmlattstocheck is not ASSOCIATED'
ENDIF
IF (ASSOCIATED(rl%kindofparam)) THEN
  DO i = 1, SIZE(rl%kindofparam)
    WRITE(iunit, c_fmtia) '%kindofparam', i, TRIM(rl%kindofparam(i)), &
                           LEN_TRIM(rl%kindofparam(i)), n_lmaxphasid
  ENDDO
ELSE
  WRITE(iunit, c_fmt) '%kindofparam is not ASSOCIATED'
ENDIF
IF (ASSOCIATED(rl%dummylabel)) THEN
  DO i = 1, SIZE(rl%dummylabel)
    WRITE(iunit, c_fmtia) '%dummylabel', i, TRIM(rl%dummylabel(i)), &
                           LEN_TRIM(rl%dummylabel(i)), n_lmaxshortid
  ENDDO
ELSE
  WRITE(iunit, c_fmt) '%dummylabel is not ASSOCIATED'
ENDIF
IF (ASSOCIATED(rl%iparam_xref)) THEN
  DO i = 1, SIZE(rl%iparam_xref)
    WRITE(iunit, c_fmtii) '%iparam_xref', i, rl%iparam_xref(i)
  ENDDO
ELSE
  WRITE(iunit, c_fmt) '%iparam_xref is not ASSOCIATED'
ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE RATELAW_dump
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
 FUNCTION EQLBREL_createRoot() RESULT(er)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(EQLBREL), POINTER :: er

INTEGER, PARAMETER :: jp_param0 = 0

NULLIFY(er)
ALLOCATE(er)
er%n_params = jp_param0      ! needs to be set here (RATELAW_init depends on this value)

CALL EQLBREL_init(er, jp_param0)

RETURN
!-----------------------------------------------------------------------
 END FUNCTION EQLBREL_createRoot
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE EQLBREL_init(er, n_params)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(EQLBREL), POINTER :: er
INTEGER, INTENT(IN)    :: n_params




IF (ASSOCIATED(er)) THEN

  er%name           = '???'
  er%ep_type        = 'EP_???'
  er%expression     = ''

  IF(er%n_params /= 0) THEN

    er%n_params      = n_params

    DEALLOCATE(er%paramname)
    DEALLOCATE(er%paramcode)
    DEALLOCATE(er%typecomponame)
    DEALLOCATE(er%xmltagname)
    DEALLOCATE(er%kindofparam)
    DEALLOCATE(er%dummylabel)

  ENDIF


  IF(n_params > 0) THEN

    er%n_params = n_params

    ALLOCATE(er%paramname(n_params))
    ALLOCATE(er%paramcode(n_params))
    ALLOCATE(er%typecomponame(n_params))
    ALLOCATE(er%xmltagname(n_params))
    ALLOCATE(er%kindofparam(n_params))
    ALLOCATE(er%dummylabel(n_params))

  ELSE

    er%n_params = 0

    NULLIFY(er%paramname)
    NULLIFY(er%paramcode)
    NULLIFY(er%typecomponame)
    NULLIFY(er%xmltagname)
    NULLIFY(er%kindofparam)
    NULLIFY(er%dummylabel)

  ENDIF

ELSE

  WRITE(jp_stderr,*) '[EQLBREL_init]: <t_eqlbrel_node> not yet ASSOCIATEd -- Aborting!'
  CALL ABORT()

ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE EQLBREL_init
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE EQLBREL_dump(er, iunit, indent, indent_inc)
!-----------------------------------------------------------------------
IMPLICIT NONE
TYPE(EQLBREL), TARGET :: er
INTEGER, INTENT(IN)    :: iunit
INTEGER, INTENT(IN)    :: indent, indent_inc


INTEGER                :: i
CHARACTER(LEN=20)      :: c_fmt1
CHARACTER(LEN=254)     :: c_fmt, c_fmta, c_fmti, c_fmtia

WRITE(c_fmt1, '()')
IF (indent > 0) WRITE(c_fmt1, '(I0,"X, ")') indent

c_fmt   = '(' // TRIM(c_fmt1) // ' A)'
c_fmta  = '(' // TRIM(c_fmt1) // ' A, " = <", A, "> (", I0, "/", I0, ")")'
c_fmti  = '(' // TRIM(c_fmt1) // ' A, " = ", I0)'
c_fmtia = '(' // TRIM(c_fmt1) // ' A, "(",I0,") = <", A, "> (", I0, "/", I0, ")")'

WRITE(iunit, c_fmta) '%name',    TRIM(er%name),    LEN_TRIM(er%name),    n_lmaxprocname
WRITE(iunit, c_fmta) '%ep_type', TRIM(er%ep_type), LEN_TRIM(er%ep_type), n_lmaxidentif
WRITE(iunit, c_fmti) '%n_params', er%n_params
WRITE(iunit, c_fmta) '%expression', TRIM(er%expression), LEN_TRIM(er%expression), n_lmaxexpress
IF (ASSOCIATED(er%paramname)) THEN
  DO i = 1, SIZE(er%paramname)
    WRITE(iunit, c_fmtia) '%paramname', i, TRIM(er%paramname(i)), &
                           LEN_TRIM(er%paramname(i)), n_lmaxidentif
  ENDDO
ELSE
  WRITE(iunit, c_fmt) '%paramname is not ASSOCIATED'
ENDIF
IF (ASSOCIATED(er%paramcode)) THEN
  DO i = 1, SIZE(er%paramcode)
    WRITE(iunit, c_fmtia) '%paramcode', i, TRIM(er%paramcode(i)), &
                           LEN_TRIM(er%paramcode(i)), n_lmaxexpress
  ENDDO
ELSE
  WRITE(iunit, c_fmt) '%paramcode is not ASSOCIATED'
ENDIF
IF (ASSOCIATED(er%typecomponame)) THEN
  DO i = 1, SIZE(er%typecomponame)
    WRITE(iunit, c_fmtia) '%typecomponame', i, TRIM(er%typecomponame(i)), &
                           LEN_TRIM(er%typecomponame(i)), n_lmaxidentif
  ENDDO
ELSE
  WRITE(iunit, c_fmt) '%typecomponame is not ASSOCIATED'
ENDIF
IF (ASSOCIATED(er%xmltagname)) THEN
  DO i = 1, SIZE(er%xmltagname)
    WRITE(iunit, c_fmtia) '%xmltagname', i, TRIM(er%xmltagname(i)), &
                           LEN_TRIM(er%xmltagname(i)), n_lmaxidentif
  ENDDO
ELSE
  WRITE(iunit, c_fmt) '%xmltagname is not ASSOCIATED'
ENDIF
IF (ASSOCIATED(er%kindofparam)) THEN
  DO i = 1, SIZE(er%kindofparam)
    WRITE(iunit, c_fmtia) '%kindofparam', i, TRIM(er%kindofparam(i)), &
                           LEN_TRIM(er%kindofparam(i)), n_lmaxphasid
  ENDDO
ELSE
  WRITE(iunit, c_fmt) '%kindofparam is not ASSOCIATED'
ENDIF
IF (ASSOCIATED(er%dummylabel)) THEN
  DO i = 1, SIZE(er%dummylabel)
    WRITE(iunit, c_fmtia) '%dummylabel', i, TRIM(er%dummylabel(i)), &
                           LEN_TRIM(er%dummylabel(i)), n_lmaxshortid
  ENDDO
ELSE
  WRITE(iunit, c_fmt) '%dummylabel is not ASSOCIATED'
ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE EQLBREL_dump
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
 FUNCTION SYSTEMSINFO_createRoot() RESULT(s)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(SYSTEMSINFO), POINTER :: s

NULLIFY(s)
ALLOCATE(s)
NULLIFY(s%prev)

CALL SYSTEMSINFO_initNode(s)

RETURN
!-----------------------------------------------------------------------
 END FUNCTION SYSTEMSINFO_createRoot
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE SYSTEMSINFO_createNext(s)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(SYSTEMSINFO), POINTER :: s

TYPE(SYSTEMSINFO), POINTER :: s_wk

CHARACTER(LEN=*), PARAMETER :: c_fmterr_a = '("[SYSTEMSINFO_createNext] error: ", A)'


NULLIFY(s_wk)

IF (ASSOCIATED(s)) THEN

  IF (.NOT. ASSOCIATED(s%next)) THEN

    ALLOCATE(s_wk)
    s_wk%prev => s
    CALL SYSTEMSINFO_initNode(s_wk)

    s%next => s_wk

    NULLIFY(s_wk)

  ELSE

    WRITE(jp_stderr,c_fmterr_a) '<s> has already a %next node -- aborting'
    CALL ABORT()

  ENDIF

ELSE

  WRITE(jp_stderr,c_fmterr_a) '<s> not yet ASSOCIATEd -- aborting'
  CALL ABORT()

ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE SYSTEMSINFO_createNext
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE SYSTEMSINFO_initNode(s)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(SYSTEMSINFO), POINTER :: s

CHARACTER(LEN=*), PARAMETER :: c_fmterr_a = '("[SYSTEMSINFO_initNode] error: ", A)'


IF (ASSOCIATED(s)) THEN

  s%name           = '???'
  s%class          = '???'
  s%shortid        = '???'
  s%idx            = -1
  s%i_system       = 0
  s%i_category     = 0

  s%n_members      = 0

  s%nvc_identifier = 'nvc_???'
  s%ioc_identifier = 'ioc_???'
  s%icc_identifier = 'icc_???'

  ! do not touch s%prev
  NULLIFY(s%next)

ELSE

  WRITE(jp_stderr,c_fmterr_a) '<s> not yet ASSOCIATEd -- aborting'
  CALL ABORT()

ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE SYSTEMSINFO_initNode
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 FUNCTION SYSTEMSINFO_lastNode(s) RESULT(s_end)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(SYSTEMSINFO), POINTER :: s
TYPE(SYSTEMSINFO), POINTER :: s_end

TYPE(SYSTEMSINFO), POINTER :: s_wk

IF (ASSOCIATED(s)) THEN
  s_wk => s
  DO WHILE(ASSOCIATED(s_wk%next))
    s_wk => s_wk%next
  ENDDO
  s_end => s_wk
ELSE
  NULLIFY(s_end)
ENDIF

NULLIFY(s_wk)

RETURN
!-----------------------------------------------------------------------
 END FUNCTION SYSTEMSINFO_lastNode
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE SYSTEMSINFO_dump(s, iunit, indent, indent_inc)
!-----------------------------------------------------------------------
IMPLICIT NONE
TYPE(SYSTEMSINFO), TARGET :: s
INTEGER, INTENT(IN)   :: iunit
INTEGER, INTENT(IN)   :: indent, indent_inc


INTEGER               :: i
CHARACTER(LEN=20)     :: c_fmt1
CHARACTER(LEN=254)    :: c_fmt, c_fmta, c_fmti, c_fmtia

WRITE(c_fmt1, '()')
IF (indent > 0) WRITE(c_fmt1, '(I0,"X, ")') indent

c_fmt   = '(' // TRIM(c_fmt1) // ' A)'
c_fmta  = '(' // TRIM(c_fmt1) // ' A, " = <", A, "> (", I0, "/", I0, ")")'
c_fmti  = '(' // TRIM(c_fmt1) // ' A, " = ", I0)'
c_fmtia = '(' // TRIM(c_fmt1) // ' A, "(",I0,") = <", A, "> (", I0, "/", I0, ")")'

WRITE(iunit, c_fmta) '%name',    TRIM(s%name),    LEN_TRIM(s%name),    n_lmaxprocname
WRITE(iunit, c_fmta) '%class',   TRIM(s%class),   LEN_TRIM(s%class),   n_lmaxcomptyp
WRITE(iunit, c_fmta) '%shortid', TRIM(s%shortid), LEN_TRIM(s%shortid), n_lmaxshortid
WRITE(iunit, c_fmti) '%idx',        s%idx
WRITE(iunit, c_fmti) '%i_system',   s%i_system
WRITE(iunit, c_fmti) '%i_category', s%i_category
WRITE(iunit, c_fmti) '%n_members',  s%n_members

IF (ASSOCIATED(s%member_name)) THEN
  DO i = 1, SIZE(s%member_name)
    WRITE(iunit, c_fmtia) '%member_name', i, TRIM(s%member_name(i)), &
                           LEN_TRIM(s%member_name(i)), n_lmaxnamesgen
  ENDDO
ENDIF

WRITE(iunit, c_fmta) '%nvc_identifier',    TRIM(s%nvc_identifier),    &
   LEN_TRIM(s%nvc_identifier),    n_lmaxidentif

WRITE(iunit, c_fmta) '%ioc_identifier',    TRIM(s%ioc_identifier),    &
   LEN_TRIM(s%ioc_identifier),    n_lmaxidentif

WRITE(iunit, c_fmta) '%icc_identifier',    TRIM(s%icc_identifier),    &
   LEN_TRIM(s%icc_identifier),    n_lmaxidentif

IF (ASSOCIATED(s%prev)) THEN
  WRITE(iunit, c_fmt) '%prev is ASSOCIATED'
ELSE
  WRITE(iunit, c_fmt) '%prev is not ASSOCIATED'
ENDIF

IF (ASSOCIATED(s%next)) THEN
  WRITE(iunit, c_fmt) '%next is ASSOCIATED'
ELSE
  WRITE(iunit, c_fmt) '%next is not ASSOCIATED'
ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE SYSTEMSINFO_dump
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
 FUNCTION EQUILIB_createRoot() RESULT(e)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(EQUILIB), POINTER :: e

NULLIFY(e)
ALLOCATE(e)
NULLIFY(e%prev)

CALL EQUILIB_initNode(e)

RETURN
!-----------------------------------------------------------------------
 END FUNCTION EQUILIB_createRoot
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE EQUILIB_createNext(e)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(EQUILIB), POINTER :: e

TYPE(EQUILIB), POINTER :: e_wk

CHARACTER(LEN=*), PARAMETER :: c_fmterr_a = '("[EQUILIB_createNext] error: ", A)'

NULLIFY(e_wk)

IF (ASSOCIATED(e)) THEN

  IF (.NOT. ASSOCIATED(e%next)) THEN

    ALLOCATE(e_wk)
    e_wk%prev => e
    CALL EQUILIB_initNode(e_wk)

    e%next => e_wk

    NULLIFY(e_wk)

  ELSE

    WRITE(jp_stderr,c_fmterr_a) '<e> has already a %next node -- aborting'

  CALL ABORT()

  ENDIF
ELSE
  WRITE(jp_stderr,c_fmterr_a) '<e> not yet ASSOCIATEd -- aborting'
  CALL ABORT()
ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE EQUILIB_createNext
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE EQUILIB_initNode(e)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(EQUILIB), POINTER :: e


CHARACTER(LEN=*), PARAMETER :: c_fmterr_a = '("[EQUILIB_initNode] error: ", A)'


IF (ASSOCIATED(e)) THEN

  e%idx            = -1
  e%name           = '???'
  e%subr           = '???'

  e%n_reactants    = -1
  e%n_products     = -1
  NULLIFY(e%cr_reacts)
  NULLIFY(e%c_rid)
  NULLIFY(e%cr_prods)
  NULLIFY(e%c_pid)

  e%ep_identifier  = 'ep_???'  ! EquilibriumParameter Identifier
  e%nve_identifier = 'nve_???' ! Identifier for number of variable
                               ! components affected by equilibrium
  e%ioe_identifier = 'ioe_???' ! Identifier for array with IO-identifiers
                               ! of components affected by equilibrium
  e%nvr_identifier = 'nvr_???' ! Identifier for number of variable
                               ! components entering the equilibrium relationship
  e%ior_identifier = 'ior_???' ! Identifier for array with IO-identifiers
                               ! of components entering the equilibrium relationship
  e%ei_identifier  = 'ei_???'  ! Identifier for EquilibriumID

  NULLIFY(e%eqlbrel)

  ! do not touch e%prev
  NULLIFY(e%next)

ELSE

  WRITE(jp_stderr,c_fmterr_a) '<e> not yet ASSOCIATEd -- aborting'
  CALL ABORT()

ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE EQUILIB_initNode
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 FUNCTION EQUILIB_lastNode(e) RESULT(e_end)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(EQUILIB), POINTER :: e
TYPE(EQUILIB), POINTER :: e_end

TYPE(EQUILIB), POINTER :: e_wk

IF (ASSOCIATED(e)) THEN
  e_wk => e
  DO WHILE(ASSOCIATED(e_wk%next))
    e_wk => e_wk%next
  ENDDO
  e_end => e_wk
ELSE
  NULLIFY(e_end)
ENDIF

NULLIFY(e_wk)

RETURN
!-----------------------------------------------------------------------
 END FUNCTION EQUILIB_lastNode
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE EQUILIB_dump(e, iunit, indent, indent_inc)
!-----------------------------------------------------------------------
IMPLICIT NONE
TYPE(EQUILIB), TARGET :: e
INTEGER, INTENT(IN)   :: iunit
INTEGER, INTENT(IN)   :: indent, indent_inc


INTEGER               :: i
CHARACTER(LEN=20)     :: c_fmt1
CHARACTER(LEN=254)    :: c_fmt, c_fmta, c_fmti, c_fmtia, c_fmtna

WRITE(c_fmt1, '()')
IF (indent > 0) WRITE(c_fmt1, '(I0,"X, ")') indent

c_fmt   = '(' // TRIM(c_fmt1) // ' A)'
c_fmta  = '(' // TRIM(c_fmt1) // ' A, " = <", A, "> (", I0, "/", I0, ")")'
c_fmti  = '(' // TRIM(c_fmt1) // ' A, " = ", I0)'
c_fmtia = '(' // TRIM(c_fmt1) // ' A, "(",I0,") ", A)'
c_fmtna = '(A, "= <", A, "> (", I0, "/", I0, ")")'

WRITE(iunit, c_fmti) '%idx', e%idx
WRITE(iunit, c_fmta) '%name',    TRIM(e%name),    LEN_TRIM(e%name),    n_lmaxprocname
WRITE(iunit, c_fmta) '%subr',    TRIM(e%subr),    LEN_TRIM(e%subr),    n_lmaxprocname
WRITE(iunit, c_fmta) '%shortid', TRIM(e%shortid), LEN_TRIM(e%shortid), n_lmaxshortid
WRITE(iunit, c_fmti) '%n_reactants', e%n_reactants
WRITE(iunit, c_fmti) '%n_products',  e%n_products

IF (ASSOCIATED(e%cr_reacts)) THEN
  DO i = 1, SIZE(e%cr_reacts)
    WRITE(iunit, c_fmtia) '%cr_reacts', i, '...'
    CALL CHEMREAGENT_dump(e%cr_reacts(i), iunit, indent+indent_inc)
    IF(ASSOCIATED(e%c_rid)) THEN
      WRITE(iunit, c_fmtia, ADVANCE='NO') '%c_rid', i
      WRITE(iunit, c_fmtna) '',    TRIM(e%c_rid(i)), &
        LEN_TRIM(e%c_rid(i)),    n_lmaxshortid
    ENDIF
  ENDDO
ENDIF

IF (ASSOCIATED(e%cr_prods)) THEN
  DO i = 1, SIZE(e%cr_prods)
    WRITE(iunit, c_fmtia) '%cr_prods', i, '...'
    CALL CHEMREAGENT_dump(e%cr_prods(i), iunit, indent+indent_inc)
    IF(ASSOCIATED(e%c_pid)) THEN
      WRITE(iunit, c_fmtia, ADVANCE='NO') '%c_pid', i
      WRITE(iunit, c_fmtna) '',    TRIM(e%c_pid(i)), &
        LEN_TRIM(e%c_pid(i)),    n_lmaxshortid
    ENDIF
  ENDDO
ENDIF

WRITE(iunit, c_fmta) '%ep_identifier',    TRIM(e%ep_identifier), &
  LEN_TRIM(e%ep_identifier),    n_lmaxidentif
WRITE(iunit, c_fmta) '%nve_identifier',    TRIM(e%nve_identifier), &
  LEN_TRIM(e%nve_identifier),    n_lmaxidentif
WRITE(iunit, c_fmta) '%ioe_identifier',    TRIM(e%ioe_identifier), &
  LEN_TRIM(e%ioe_identifier),    n_lmaxidentif
WRITE(iunit, c_fmta) '%nvr_identifier',    TRIM(e%nvr_identifier), &
  LEN_TRIM(e%nvr_identifier),    n_lmaxidentif
WRITE(iunit, c_fmta) '%ior_identifier',    TRIM(e%ior_identifier), &
  LEN_TRIM(e%ior_identifier),    n_lmaxidentif
WRITE(iunit, c_fmta) '%ei_identifier',    TRIM(e%ei_identifier), &
  LEN_TRIM(e%ei_identifier),    n_lmaxidentif

WRITE(iunit, c_fmt) '%eqlbrel ...'
IF(ASSOCIATED(e%eqlbrel)) THEN
  CALL EQLBREL_dump(e%eqlbrel, iunit, indent+indent_inc, indent_inc)
ELSE
  WRITE(iunit, c_fmt) '%eqlbrel is not ASSOCIATED'
ENDIF

IF (ASSOCIATED(e%prev)) THEN
  WRITE(iunit, c_fmt) '%prev is ASSOCIATED'
ELSE
  WRITE(iunit, c_fmt) '%prev is not ASSOCIATED'
ENDIF

IF (ASSOCIATED(e%next)) THEN
  WRITE(iunit, c_fmt) '%next is ASSOCIATED'
ELSE
  WRITE(iunit, c_fmt) '%next is not ASSOCIATED'
ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE EQUILIB_dump
!-----------------------------------------------------------------------


!=======================================================================
 END MODULE MOD_CONFIGURE_TYPES
!=======================================================================
