!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#ifdef CFG_MEDUSACOCOGEN
&ratelaw_config
c_name          = 'RAMP1_RAMPINH2_C'
c_pp_type       = 'PP_RAMP1_RAMPINH2_C'
n_param         = 8
c_expression    = '{#1} * [{#2}] * MIN([{#3}]/{#4}, 1) * (1 - MIN([{#5}]/{#6}, 1)) * (1 - MIN([{#7}]/{#8}, 1))'
/
! Parameter 1
&ratelaw_data
c_typecomponame  = 'RateConstant'
c_xmlattstocheck = 'type'
c_kindofparam    = 'gk'
c_dummylabel     = 'k'
/
! Parameter 2
&ratelaw_data
c_typecomponame  = 'ioProportional'
c_xmltagname     = 'Proportional'
c_kindofparam    = 'io'
/
! Parameter 3
&ratelaw_data
c_typecomponame  = 'ioRamp'
c_xmltagname     = 'RampConc'
c_kindofparam    = 'io'
/
! Parameter 4
&ratelaw_data
c_typecomponame  = 'RampMaxConstant'
c_xmlattstocheck = 'type'
c_kindofparam    = 'gk'
c_dummylabel     = 'kmax'
/
! Parameter 5
&ratelaw_data
c_typecomponame  = 'ioInhibition1'
c_xmltagname     = 'InhibitionConc'
c_kindofparam    = 'io'
/
! Parameter 6
&ratelaw_data
c_typecomponame  = 'InhibConstant1'
c_xmltagname     = 'InhibConstant'
c_xmlattstocheck = 'type'
c_kindofparam    = 'gk'
c_dummylabel     = 'kih1'
/
! Parameter 7
&ratelaw_data
c_typecomponame  = 'ioInhibition2'
c_xmltagname     = 'InhibitionConc2'
c_kindofparam    = 'io'
/
! Parameter 8
&ratelaw_data
c_typecomponame  = 'InhibConstant2'
c_xmlattstocheck = 'type'
c_kindofparam    = 'gk'
c_dummylabel     = 'kih2'
/
#endif
!---+----1----+----2----+----3----+----4----+----5----+----6----+----7-!
!=======================================================================
      MODULE MODLIB_RAMP1_RAMPINH2_C
!=======================================================================

      ! For laws of the form k * c1 * MIN(c2/c2_max, 1)
      !                             * (1 - MIN(c3/c3_inh, 1))
      !                             * (1 - MIN(c4/c4_inh, 1))


      IMPLICIT NONE

      TYPE PP_RAMP1_RAMPINH2_C
        DOUBLE PRECISION :: RateConstant       ! k
        DOUBLE PRECISION :: RampMaxConstant    ! c2_max
        DOUBLE PRECISION :: InhibConstant1     ! c3_inh
        DOUBLE PRECISION :: InhibConstant2     ! c4_inh
        INTEGER          :: ioProportional     ! io of c1 in ac(:)
        INTEGER          :: ioRamp             ! io of c2 in ac(:)
        INTEGER          :: ioInhibition1      ! io of c3 in ac(:)
        INTEGER          :: ioInhibition2      ! io of c4 in ac(:)
      END TYPE


!     Sample usage in the XML declarations:
!
!     <RateLaw reference_id="r1">
!
!        <Subroutine>RAMP1_RAMPINH2_C</Subroutine>
!        <RateConstant    type="globalconstant"/>
!        <Proportional>    OrgMatter </Proportional>
!        <RampConc>        MnO2      </RampConc>
!        <RampMaxConstant type="globalconstant"/>
!        <InhibitionConc>  O2        </InhibitionConc>
!        <InhibConstant   type="globalconstant"/>
!        <InhibitionConc2> NO3       </InhibitionConc2>
!        <InhibConstant2  type="globalconstant"/>
!
!     </RateLaw>


      CONTAINS

!-----------------------------------------------------------------------
      SUBROUTINE RAMP1_RAMPINH2_C(pp_param, ac, azdn, arate, darate_dac)
!-----------------------------------------------------------------------

      IMPLICIT NONE


! Argument list variables
! -----------------------

      TYPE(PP_RAMP1_RAMPINH2_C),             INTENT(IN)  :: pp_param
      DOUBLE PRECISION, DIMENSION(:),        INTENT(IN)  :: ac
      DOUBLE PRECISION,                      INTENT(IN)  :: azdn
      DOUBLE PRECISION,                      INTENT(OUT) :: arate
      DOUBLE PRECISION, DIMENSION(SIZE(ac)), INTENT(OUT) :: darate_dac

      OPTIONAL :: arate, darate_dac


! Local variables
! ---------------

      INTEGER          :: jo_p, jo_r, jo_i1, jo_i2
      DOUBLE PRECISION :: alaw_ct
      DOUBLE PRECISION ::          acompo_p,  arate_p,  darate_p
      DOUBLE PRECISION :: alaw_rm, acompo_r,  arate_r,  darate_r
      DOUBLE PRECISION :: alaw_ih1, acompo_i1, arate_i1, darate_i1
      DOUBLE PRECISION :: alaw_ih2, acompo_i2, arate_i2, darate_i2


! Instructions
! ------------

      jo_p  = pp_param%ioProportional
      jo_r  = pp_param%ioRamp
      jo_i1 = pp_param%ioInhibition1
      jo_i2 = pp_param%ioInhibition2

      acompo_p  = ac(jo_p)
      acompo_r  = ac(jo_r)
      acompo_i1 = ac(jo_i1)
      acompo_i2 = ac(jo_i2)
      alaw_ct   = pp_param%RateConstant
      alaw_rm   = pp_param%RampMaxConstant
      alaw_ih1  = pp_param%InhibConstant1
      alaw_ih2  = pp_param%InhibConstant2

      IF (acompo_p > 0.0D+00) THEN
        arate_p = acompo_p
      ELSE
        arate_p = 0.0D+00
      ENDIF

      IF (acompo_r <= 0.0D+00) THEN
        arate_r = 0.0D+00
      ELSEIF (acompo_r <= alaw_rm) THEN
        arate_r = acompo_r/alaw_rm
      ELSE
        arate_r = 1.0D+00
      ENDIF

      IF     (acompo_i1 <= 0.0D+00) THEN
        arate_i1 = 1.0D+00
      ELSEIF (acompo_i1 <= alaw_ih1) THEN
        arate_i1 = 1.0D+00 - (acompo_i1/alaw_ih1)
      ELSE
        arate_i1 = 0.0D+00
      ENDIF

      IF     (acompo_i2 <= 0.0D+00) THEN
        arate_i2 = 1.0D+00
      ELSEIF (acompo_i2 <= alaw_ih2) THEN
        arate_i2 = 1.0D+00 - (acompo_i2/alaw_ih2)
      ELSE
        arate_i2 = 0.0D+00
      ENDIF


      IF (PRESENT(arate)) THEN
        arate = alaw_ct * arate_p * arate_r * arate_i1 * arate_i2
      ENDIF


      IF (PRESENT(darate_dac)) THEN

        IF (acompo_p > 0.0D+00) THEN
          darate_p = 1.0D+00
        ELSEIF (acompo_p == 0.0D+00) THEN
          darate_p = 1.0D+00/2.0D+00
        ELSE
          darate_p = 0.0D+00
        ENDIF

        IF     ((acompo_r >   0.0D+00) .AND.
     &          (acompo_r <  alaw_rm)) THEN
          darate_r =  1.0D+00 / alaw_rm
        ELSEIF ((acompo_r ==  0.0D+00) .OR.
     &          (acompo_r == alaw_rm)) THEN
          darate_r =  1.0D+00 / (alaw_rm+alaw_rm)
        ELSE
          darate_r =  0.0D+00
        ENDIF

        IF     ((acompo_i1 >   0.0D+00) .AND.
     &          (acompo_i1 <  alaw_ih1)) THEN
          darate_i1 = -1.0D+00 / alaw_ih1
        ELSEIF ((acompo_i1 ==  0.0D+00) .OR.
     &          (acompo_i1 == alaw_ih1)) THEN
          darate_i1 = -1.0D+00 / (alaw_ih1+alaw_ih1)
        ELSE
          darate_i1 =  0.0D+00
        ENDIF

        IF     ((acompo_i2 >   0.0D+00) .AND.
     &          (acompo_i2 <  alaw_ih2)) THEN
          darate_i2 = -1.0D+00 / alaw_ih2
        ELSEIF ((acompo_i2 ==  0.0D+00) .OR.
     &          (acompo_i2 == alaw_ih2)) THEN
          darate_i2 = -1.0D+00 / (alaw_ih2+alaw_ih2)
        ELSE
          darate_i2 =  0.0D+00
        ENDIF


        darate_dac(:) = 0.0D+00

        darate_dac(jo_p)  = alaw_ct * darate_p  *  arate_r
     &                              *  arate_i1 *  arate_i2
        darate_dac(jo_r ) = alaw_ct *  arate_p  * darate_r
     &                              *  arate_i1 *  arate_i2
        darate_dac(jo_i1) = alaw_ct *  arate_p  *  arate_r
     &                              * darate_i1 *  arate_i2
        darate_dac(jo_i2) = alaw_ct *  arate_p  *  arate_r
     &                              *  arate_i1 * darate_i2

      ENDIF

      RETURN

!-----------------------------------------------------------------------
      END SUBROUTINE RAMP1_RAMPINH2_C
!-----------------------------------------------------------------------


!=======================================================================
      END MODULE MODLIB_RAMP1_RAMPINH2_C
!=======================================================================

