!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#ifdef CFG_MEDUSACOCOGEN
&eqlbrel_config
c_name          = 'R1R2P1P2'
c_ep_type       = 'EP_R1R2P1P2'
n_param         = 5
c_expression    = '[{#4}]*[{#5}] - {#1} * [{#2}]*[{#3}] = 0'
/
! Parameter 1
&eqlbrel_data
c_typecomponame  = 'EquilibConstant'
c_kindofparam    = 'bc'  ! equilibrium constant, to be set from the boundary conditions
c_dummylabel     = 'K'
/
! Parameter 2
&eqlbrel_data
c_typecomponame  = 'ioReactant1'
c_xmltagname     = 'Reactant1'
c_kindofparam    = 'io'
/
! Parameter 3
&eqlbrel_data
c_typecomponame  = 'ioReactant2'
c_xmltagname     = 'Reactant2'
c_kindofparam    = 'io'
/
! Parameter 4
&eqlbrel_data
c_typecomponame  = 'ioProduct1'
c_xmltagname     = 'Product1'
c_kindofparam    = 'io'
/
! Parameter 5
&eqlbrel_data
c_typecomponame  = 'ioProduct2'
c_xmltagname     = 'Product2'
c_kindofparam    = 'io'
/
#endif
!---+----1----+----2----+----3----+----4----+----5----+----6----+----7-!
!=======================================================================
      MODULE MODLIB_R1R2P1P2
!=======================================================================

      ! For equilibrium relationships
      ! P1*P2 - K * R1*R2 = 0

      IMPLICIT NONE

      TYPE EP_R1R2P1P2
        INTEGER          :: ioReactant1     ! io of R1
        INTEGER          :: ioReactant2     ! io of R2
        INTEGER          :: ioProduct1      ! io of P1
        INTEGER          :: ioProduct2      ! io of P2
        DOUBLE PRECISION :: EquilibConstant ! K value
      END TYPE


!     Sample usage in the µXML declarations:
!
!    <LawOfMassAction subr="R1R2P1P2">
!      <Reactant1>B(OH)4</Reactant1>
!      <Reactant2>HCO3</Reactant2>
!      <Product1>B(OH)3</Product1>
!      <Product2>CO3</Product2>
!   </LawOfMassAction>


      CONTAINS

!-----------------------------------------------------------------------
      SUBROUTINE R1R2P1P2(ep_param, ac, aeqrl, daeqrl_dac)
!-----------------------------------------------------------------------

      IMPLICIT NONE


! Argument list variables
! -----------------------

      TYPE(EP_R1R2P1P2),                     INTENT(IN)  :: ep_param
      DOUBLE PRECISION, DIMENSION(:),        INTENT(IN)  :: ac
      DOUBLE PRECISION,                      INTENT(OUT) :: aeqrl
      DOUBLE PRECISION, DIMENSION(SIZE(ac)), INTENT(OUT) :: daeqrl_dac

      OPTIONAL :: aeqrl, daeqrl_dac


! Local variables
! ---------------

      INTEGER          :: io_r1, io_r2, io_p1, io_p2
      DOUBLE PRECISION :: acompo_r1, acompo_r2, acompo_p1, acompo_p2
      DOUBLE PRECISION ::  aeqrl_ct
      DOUBLE PRECISION :: daeqrl_r1, daeqrl_r2, daeqrl_p1, daeqrl_p2


! Instructions
! ------------

      io_r1 = ep_param%ioReactant1
      io_r2 = ep_param%ioReactant2
      io_p1 = ep_param%ioProduct1
      io_p2 = ep_param%ioProduct2

      acompo_r1  = ac(io_r1)
      acompo_r2  = ac(io_r2)
      acompo_p1  = ac(io_p1)
      acompo_p2  = ac(io_p2)

      aeqrl_ct   = ep_param%EquilibConstant

      IF (PRESENT(aeqrl)) THEN
        aeqrl =  acompo_p1*acompo_p2 - aeqrl_ct * acompo_r1*acompo_r2
      ENDIF


      IF (PRESENT(daeqrl_dac)) THEN
        daeqrl_dac(:)     =  0.0D+00
        daeqrl_dac(io_r1) = -aeqrl_ct*acompo_r2
        daeqrl_dac(io_r2) = -aeqrl_ct*acompo_r1
        daeqrl_dac(io_p1) =  acompo_p2
        daeqrl_dac(io_p2) =  acompo_p1
      ENDIF


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE R1R2P1P2
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      SUBROUTINE SET_EQUILCT_R1R2P1P2(ep_param, ac, aeqrl_ct)
!-----------------------------------------------------------------------

      IMPLICIT NONE


! Argument list variables
! -----------------------

      TYPE(EP_R1R2P1P2),              INTENT(INOUT) :: ep_param
      DOUBLE PRECISION, DIMENSION(:), INTENT(IN)    :: ac
      DOUBLE PRECISION, OPTIONAL,     INTENT(IN)    :: aeqrl_ct


! Local variables
! ---------------

      INTEGER          :: io_r1, io_r2, io_p1, io_p2
      DOUBLE PRECISION :: acompo_r1, acompo_r2, acompo_p1, acompo_p2


! Instructions
! ------------

      io_r1 = ep_param%ioReactant1
      io_r2 = ep_param%ioReactant2
      io_p1 = ep_param%ioProduct1
      io_p2 = ep_param%ioProduct2

      acompo_r1  = ac(io_r1)
      acompo_r2  = ac(io_r2)
      acompo_p1  = ac(io_p1)
      acompo_p2  = ac(io_p2)


      IF (PRESENT(aeqrl_ct)) THEN

        ! For improved fussiness: check if equilibrium relationship
        ! fulfilled and take appropriate action if not.
        !  - set constant's value
        !  - get scale aeqrl_scale: requires ac_scale
        !  - evaluate equilibrium relationship for wconc (-> aeqrl)
        !  - check if ABS(aeqrl)/aeqrl_scale < dp_tolerance

        ep_param%EquilibConstant = aeqrl_ct

      ELSE

        ep_param%EquilibConstant
     &    =  (acompo_p1*acompo_p2) / (acompo_r1*acompo_r2)

      ENDIF


      RETURN


!-----------------------------------------------------------------------
      END SUBROUTINE SET_EQUILCT_R1R2P1P2
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
      DOUBLE PRECISION FUNCTION GET_SCALE_R1R2P1P2(ep_param, ac_scale)
!-----------------------------------------------------------------------

      IMPLICIT NONE


! Argument list variables
! -----------------------

      TYPE(EP_R1R2P1P2),              INTENT(IN) :: ep_param
      DOUBLE PRECISION, DIMENSION(:), INTENT(IN) :: ac_scale


! Local variables
! ---------------

      INTEGER          :: io_r1, io_r2
      INTEGER          :: io_p1, io_p2
      DOUBLE PRECISION :: ascale_r1, ascale_r2
      DOUBLE PRECISION :: ascale_p1, ascale_p2
      DOUBLE PRECISION :: aeqrl_ct


! Instructions
! ------------

      io_r1 = ep_param%ioReactant1
      io_r2 = ep_param%ioReactant2
      io_p1 = ep_param%ioProduct1
      io_p2 = ep_param%ioProduct2

      ascale_r1  = ac_scale(io_r1)
      ascale_r2  = ac_scale(io_r2)
      ascale_p1  = ac_scale(io_p1)
      ascale_p2  = ac_scale(io_p2)

      aeqrl_ct = ep_param%EquilibConstant

      get_scale_r1r2p1p2 =  SQRT(  aeqrl_ct*ascale_r1*ascale_r2
     &                           * ascale_p1*ascale_p2)


      RETURN


!-----------------------------------------------------------------------
      END FUNCTION GET_SCALE_R1R2P1P2
!-----------------------------------------------------------------------


!=======================================================================
      END MODULE MODLIB_R1R2P1P2
!=======================================================================
