!
!    Copyright 2007-2020 Guy Munhoven
!
!    This file is part of Medusa.
!
!    Medusa is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Affero General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    Medusa is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!    See the GNU Affero General Public License for more details.
!
!    You should have received a copy of the Affero GNU General Public
!    License along with Medusa.  If not, see <https://www.gnu.org/licenses/>.
!


#include "configure.h"
!=======================================================================
 SUBROUTINE INIT_DATACHAINS(c_xmlbaselist,                             &
                                    t_compo_chain, t_solsys_chain,     &
                                    t_process_chain, t_equilib_chain,  &
                                    t_apiext_chain)
!=======================================================================

USE MOD_MEDUSA_COCOGEN
USE MOD_XMLCOCOGEN
USE MOD_CONFIGURE_TYPES
USE MODMXM_STKMX
USE MOD_CONFIGURE,                  ONLY: n_medusaproc, n_medusaeqlb,  &
                                          n_modelsolsys, n_medusacompo,&
                                          n_medusaapiext

USE MOD_UTILITIES,                  ONLY: LOWCASE, UPCASE, &
                                          DELIMIT_STRING_TOKENS, &
                                          EXPAND_TOKEN

IMPLICIT NONE


! Argument list variables
! -----------------------

CHARACTER(LEN=*), INTENT(IN) :: c_xmlbaselist
TYPE(PROCESS), POINTER       :: t_process_chain
TYPE(COMPOINFO),   POINTER   :: t_compo_chain
TYPE(SYSTEMSINFO), POINTER   :: t_solsys_chain
TYPE(EQUILIB), POINTER       :: t_equilib_chain
TYPE(APIEXTENSION), POINTER  :: t_apiext_chain


! Local variables
! ----------------

! Local parameters

LOGICAL                      :: l_solsys_alk_isrequired = .FALSE.

TYPE(xfs_list), POINTER      :: xml_compofiles
TYPE(xfs_list), POINTER      :: xml_procsfiles
TYPE(xfs_list), POINTER      :: xml_equilfiles
TYPE(xfs_list), POINTER      :: xml_apiexfiles

TYPE(PROCESS), POINTER       :: t_process_work, t_process_curr
TYPE(COMPOINFO),   POINTER   :: t_compo_work,   t_compo_curr
TYPE(SYSTEMSINFO), POINTER   :: t_solsys_work,  t_solsys_curr
TYPE(EQUILIB), POINTER       :: t_equilib_work, t_equilib_curr
TYPE(APIEXTENSION), POINTER  :: t_apiext_work,  t_apiext_curr

TYPE(xfs_list), POINTER :: xml_work => NULL()
TYPE(xfs_list), POINTER :: xml_scan => NULL()
TYPE(stack_recchunks), POINTER :: stkrc_eltname
TYPE(stack_minixml), POINTER :: stkmx_rootelt, stkmx_work

CHARACTER(LEN=p_maxlen_filename) :: c_filename
CHARACTER(LEN=p_maxlen_eltname)  :: c_rootname
CHARACTER(LEN=p_maxlen_eltname)  :: c_type
INTEGER :: nlen_returned


CHARACTER(LEN=n_lmaxnamesgen)    :: c_namesgen
CHARACTER(LEN=n_lmaxnameslong)   :: c_nameslon
CHARACTER(LEN=n_lmaxshortid)     :: c_namessid

CHARACTER(LEN=n_lmaxprocname)    :: c_prneqnamesgen
CHARACTER(LEN=n_lmaxnameslong)   :: c_prneqnameslon
CHARACTER(LEN=n_lmaxshortid)     :: c_prneqnamessid


! Temporary string (internal file) to write formatted integers
CHARACTER(LEN=20) :: c_procnb

! Code block pointers and space
TYPE(CODEBITS), POINTER :: t_codebits_root, t_code_wk

!~ TYPE(stack_minixml), POINTER                        :: stkmx_cherea
!~ TYPE(CHEMREAGENT), DIMENSION(:), POINTER            :: tcr_reagents
!~ CHARACTER(LEN=n_lmaxshortid), DIMENSION(:), POINTER :: c_xid
!~ CHARACTER(LEN=n_lmaxshortid), DIMENSION(:), POINTER :: c_xwildcard
!~ INTEGER :: n_reactants, n_products

LOGICAL :: l_extra_copy
INTEGER :: i_system

INTEGER :: i_round, n_newcategory1

! I/O related parameters

INTEGER, PARAMETER           :: jp_stderr = (CFG_STDERR)
INTEGER, PARAMETER           :: jp_stdout = (CFG_STDOUT)
CHARACTER(LEN=*), PARAMETER  :: c_fmtinf_a = '("[INIT_DATA_CHAINS]: ", A)'
CHARACTER(LEN=*), PARAMETER  :: c_fmtwar_a = '("[INIT_DATA_CHAINS] warning: ", A)'
CHARACTER(LEN=*), PARAMETER  :: c_fmterr_a = '("[INIT_DATA_CHAINS] error: ", A)'

#ifdef CFG_DEBUG
INTEGER, PARAMETER           :: jp_stddbg = (CFG_STDDBG)
CHARACTER(LEN=*), PARAMETER  :: c_fmtdbg_a = '("DEBUG [INIT_DATA_CHAINS]: ", A)'
#endif


INTERFACE
  SUBROUTINE XMLCOCOGEN_LOADDB(c_xmlbaselist,       &
    xml_compofiles, xml_procsfiles, xml_equilfiles, &
    xml_apiexfiles)
    USE MOD_XMLCOCOGEN
    CHARACTER(LEN=*)        :: c_xmlbaselist
    TYPE(xfs_list), POINTER :: xml_compofiles
    TYPE(xfs_list), POINTER :: xml_procsfiles
    TYPE(xfs_list), POINTER :: xml_equilfiles
    TYPE(xfs_list), POINTER :: xml_apiexfiles
  END SUBROUTINE XMLCOCOGEN_LOADDB
END INTERFACE


!=====================!
! End of declarations !
!=====================!

WRITE(jp_stdout,'()')
WRITE(jp_stdout,c_fmtinf_a) 'starting'
#ifdef CFG_DEBUG
WRITE(jp_stddbg,'()')
WRITE(jp_stddbg,c_fmtdbg_a) 'starting'
#endif

!============!
! Operations !
!============!

NULLIFY(xml_compofiles)
NULLIFY(xml_procsfiles)
NULLIFY(xml_equilfiles)
NULLIFY(xml_apiexfiles)

WRITE(jp_stdout,'()')
WRITE(jp_stdout,'()')
WRITE(jp_stdout,'("Starting to load uXML data from """, A, """")') TRIM(c_xmlbaselist)
#ifdef CFG_DEBUG
WRITE(jp_stddbg,'()')
WRITE(jp_stddbg,'()')
WRITE(jp_stddbg,'("Starting to load uXML data from """, A, """")') TRIM(c_xmlbaselist)
#endif


CALL XMLCOCOGEN_LOADDB(c_xmlbaselist, &
                       xml_compofiles, xml_procsfiles, xml_equilfiles, &
                       xml_apiexfiles)


!-------------------!
! Composition files !
!-------------------!

WRITE(jp_stdout,'()')
WRITE(jp_stdout,'()')
WRITE(jp_stdout,'("Composition-file tree")')
WRITE(jp_stdout,'("=====================")')
#ifdef CFG_DEBUG
WRITE(jp_stddbg,'()')
WRITE(jp_stddbg,'()')
WRITE(jp_stddbg,'("Composition-file tree")')
WRITE(jp_stddbg,'("=====================")')
#endif

NULLIFY(t_compo_chain)
t_compo_chain => COMPOINFO_createRoot()

xml_work      => xml_compofiles

DO WHILE(ASSOCIATED(xml_work))
  c_filename = xml_work%fname
  c_type     = xml_work%c_type
  WRITE(jp_stdout,'()')
  WRITE(jp_stdout,'("File """,A,""":")') TRIM(c_filename)

                                    ! Check if the same file has been read in before
  l_extra_copy = .FALSE.
  xml_scan => xml_work%prev
  DO WHILE(ASSOCIATED(xml_scan))
    IF(xml_scan%fname == c_filename) THEN
      l_extra_copy = .TRUE.
      EXIT
    ENDIF
    xml_scan => xml_scan%prev
  ENDDO


  stkmx_rootelt => xml_work%mx
  i_system = xml_work%i_system
  stkrc_eltname => STKMX_getElementName(stkmx_rootelt)
  CALL STKRC_copyStkrcToStr(stkrc_eltname, c_rootname, nlen_returned)
  IF (nlen_returned < 0) THEN
    WRITE(jp_stdout,c_fmtwar_a) &
      'root-element name = <"' // c_rootname //'"> has been truncated'
  ENDIF

  IF (c_rootname /= c_type) THEN
    WRITE(jp_stderr,c_fmterr_a) &
      'Root element mismatch!'
    WRITE(jp_stderr,'(" - expected <",A,">")') TRIM(c_type)
    WRITE(jp_stderr,'(" - found    <",A,">")') TRIM(c_rootname)
    WRITE(jp_stderr,'("Aborting!")')
    CALL ABORT()
  ENDIF

  SELECT CASE(c_rootname)
  CASE(cp_solute, cp_solid)
    CONTINUE
  CASE(cp_solutesystem)
    xml_work => xml_work%next
    CYCLE  ! Skip for now; will be registred in the next stage
  CASE DEFAULT
    xml_work => xml_work%next
    WRITE(jp_stdout,c_fmtinf_a) &
      'Skipping (as yet) unknown <"' // c_rootname // '">'
    CYCLE
  END SELECT


  ! Read the <Names> tag
  CALL IDC_getNames(stkmx_rootelt, c_namesgen, c_nameslon, c_namessid)

  ! Register new component into the COMPOINFO tree
  t_compo_curr => IDC_registerNewCompo(t_compo_chain, c_namesgen, c_namessid, i_system)

  ! Read the <CodeBits> tag if first instance
  IF (.NOT. l_extra_copy) CALL IDC_attachCompoCodebits(stkmx_rootelt, t_compo_curr)


  ! Complete the registration
  SELECT CASE(c_rootname)
  CASE(cp_solute)
    CALL IDC_configureSolute(stkmx_rootelt, t_compo_curr, l_extra_copy)
  CASE(cp_solid)
    CALL IDC_configureSolid(stkmx_rootelt, t_compo_curr)
  END SELECT

# ifdef CFG_DEBUG
  WRITE(jp_stddbg,c_fmtdbg_a, ADVANCE='NO') &
    'Composition: %name, %idx, n_medusacompo = "'
  WRITE(jp_stddbg,'("""", A, """, ", I0, ", ", I0)') &
    TRIM(t_compo_curr%name), t_compo_curr%idx, n_medusacompo
# endif

  ! Proceed to the next one
  xml_work => xml_work%next

ENDDO


!--------------------!
! SoluteSystem files !
!--------------------!

WRITE(jp_stdout,'()')
WRITE(jp_stdout,'()')
WRITE(jp_stdout,'("SoluteSystem-file tree")')
WRITE(jp_stdout,'("======================")')
#ifdef CFG_DEBUG
WRITE(jp_stddbg,'()')
WRITE(jp_stddbg,'()')
WRITE(jp_stddbg,'("SoluteSystem-file tree")')
WRITE(jp_stddbg,'("======================")')
#endif

! t_solsys_chain: create
NULLIFY(t_solsys_chain)
t_solsys_chain => SYSTEMSINFO_createRoot()

xml_work       => xml_compofiles

DO WHILE(ASSOCIATED(xml_work))

  c_filename = xml_work%fname
  WRITE(jp_stdout,'()')
  WRITE(jp_stdout,'("File """,A,""":")', ADVANCE='NO') TRIM(c_filename)

  stkmx_rootelt => xml_work%mx
  i_system = xml_work%i_system
  stkrc_eltname => STKMX_getElementName(stkmx_rootelt)
  CALL STKRC_copyStkrcToStr(stkrc_eltname, c_rootname, nlen_returned)
  IF (nlen_returned < 0) THEN
    WRITE(jp_stdout,'()')           ! print LF
    WRITE(jp_stdout,c_fmtwar_a) &
      'root-element name = <"' // c_rootname //'"> has been truncated'
  ENDIF

  SELECT CASE(c_rootname)
  CASE(cp_solutesystem)
    WRITE(jp_stdout,'()')           ! print LF
    CONTINUE
  CASE DEFAULT
    xml_work => xml_work%next
    WRITE(jp_stdout,'(A)') ' skipped (not a '// cp_solutesystem // ' file)'
    CYCLE
  END SELECT


  ! Read the <Names> tag
  CALL IDC_getNames(stkmx_rootelt, c_namesgen, c_nameslon, c_namessid)

  ! Register names into the COMPOINFO and the SYSTEMINFO trees
  t_compo_curr  => IDC_registerNewCompo(t_compo_chain, c_namesgen, c_namessid, i_system)
  t_solsys_curr => IDC_registerNewSoluteSystem(t_solsys_chain, c_namesgen, c_namessid, i_system)

  ! Complete the configuration of the component
  CALL IDC_configureSoluteSystem(stkmx_rootelt, t_compo_curr, t_solsys_curr)


# ifdef CFG_DEBUG
  WRITE(jp_stddbg,c_fmtdbg_a, ADVANCE='NO') &
    'Composition: %name, %idx, n_medusacompo = "'
  WRITE(jp_stddbg,'("""", A, """, ", I0, ", ", I0)') &
    TRIM(t_compo_curr%name), t_compo_curr%idx, n_medusacompo
  WRITE(jp_stddbg,c_fmtdbg_a, ADVANCE='NO') &
    'SoluteSystem: %name, %idx, n_medusacompo = "'
  WRITE(jp_stddbg,'("""", A, """, ", I0, ", ", I0)') &
    TRIM(t_solsys_curr%name), t_solsys_curr%idx, n_modelsolsys
# endif

  ! Proceed to the next one
  xml_work => xml_work%next

ENDDO

                                    ! And finally add the Alkalinity
                                    ! system if required
CALL IDC_setupAlkSoluteSystem(t_compo_chain, t_solsys_chain)


!---------------!
! Process files !
!---------------!

WRITE(jp_stdout,'()')
WRITE(jp_stdout,'()')
WRITE(jp_stdout,'("Process-file tree")')
WRITE(jp_stdout,'("=================")')
#ifdef CFG_DEBUG
WRITE(jp_stddbg,'()')
WRITE(jp_stddbg,'()')
WRITE(jp_stddbg,'("Process-file tree")')
WRITE(jp_stddbg,'("=================")')
#endif

NULLIFY(t_process_chain)

IF (ASSOCIATED(xml_procsfiles)) THEN

  t_process_chain => PROCESS_createRoot()

  xml_work        => xml_procsfiles

  DO WHILE(ASSOCIATED(xml_work))

    c_filename = xml_work%fname
    c_type     = xml_work%c_type
    WRITE(jp_stdout,'()')
    WRITE(jp_stdout,'("File """,A,""":")') TRIM(c_filename)
    stkmx_rootelt => xml_work%mx
    stkrc_eltname => STKMX_getElementName(stkmx_rootelt)
    CALL STKRC_copyStkrcToStr(stkrc_eltname, c_rootname, nlen_returned)
    IF (nlen_returned < 0) THEN
      WRITE(jp_stdout,c_fmtwar_a) &
        'root-element name = <"' // c_rootname //'"> has been truncated'
    ENDIF

    IF (c_rootname /= c_type) THEN
      WRITE(jp_stderr,c_fmterr_a) &
        'Root element mismatch!'
      WRITE(jp_stderr,'(" - expected <",A,">")') TRIM(c_type)
      WRITE(jp_stderr,'(" - found    <",A,">")') TRIM(c_rootname)
      WRITE(jp_stderr,'("Aborting!")')
      CALL ABORT()
    ENDIF

    SELECT CASE(c_rootname)
    CASE DEFAULT
      CONTINUE
    END SELECT


    ! Read the <Names> tag
    CALL IDC_getNames(stkmx_rootelt, c_prneqnamesgen, c_prneqnameslon, c_prneqnamessid)

    ! Register a new process into the PROCESS tree
    t_process_curr => IDC_registerNewProcess(t_process_chain, c_prneqnamesgen, c_prneqnamessid)

    ! Complete the configuration of the process
    CALL IDC_configureProcess(stkmx_rootelt, t_compo_chain, t_process_curr)

    ! Proceed to the next one
    xml_work => xml_work%next

  ENDDO

ELSE

  WRITE(jp_stdout,'()')
  WRITE(jp_stdout,'("--Empty--")')
# ifdef CFG_DEBUG
  WRITE(jp_stddbg,'()')
  WRITE(jp_stddbg,'("--Empty--")')
# endif

ENDIF


!-------------------!
! Equilibrium files !
!-------------------!

WRITE(jp_stdout,'()')
WRITE(jp_stdout,'()')
WRITE(jp_stdout,'("Equilibrium-file tree")')
WRITE(jp_stdout,'("=====================")')
#ifdef CFG_DEBUG
WRITE(jp_stddbg,'()')
WRITE(jp_stddbg,'()')
WRITE(jp_stddbg,'("Equilibrium-file tree")')
WRITE(jp_stddbg,'("=====================")')
#endif

NULLIFY(t_equilib_chain)

IF (ASSOCIATED(xml_equilfiles)) THEN

  t_equilib_chain => EQUILIB_createRoot()

  xml_work        => xml_equilfiles


  DO WHILE(ASSOCIATED(xml_work))

    c_filename = xml_work%fname
    c_type     = xml_work%c_type

    WRITE(jp_stdout,'()')
    WRITE(jp_stdout,'("File """,A,""":")') TRIM(c_filename)

    stkmx_rootelt => xml_work%mx
    stkrc_eltname => STKMX_getElementName(stkmx_rootelt)
    CALL STKRC_copyStkrcToStr(stkrc_eltname, c_rootname, nlen_returned)
    IF (nlen_returned < 0) THEN
      WRITE(jp_stdout,c_fmtwar_a) &
        'root-element name = <"' // c_rootname //'"> has been truncated'
    ENDIF

    IF (c_rootname /= c_type) THEN
      WRITE(jp_stderr,c_fmterr_a) &
        'Root element mismatch!'
      WRITE(jp_stderr,'(" - expected <",A,">")') TRIM(c_type)
      WRITE(jp_stderr,'(" - found    <",A,">")') TRIM(c_rootname)
      WRITE(jp_stderr,'("Aborting!")')
      CALL ABORT()
    ENDIF

    SELECT CASE(c_rootname)
    CASE DEFAULT
      CONTINUE
    END SELECT


    ! Read the <Names> tag
    CALL IDC_getNames(stkmx_rootelt, c_prneqnamesgen, c_prneqnameslon, c_prneqnamessid)

    ! Register new equilibrium into the EQUILIB tree
    t_equilib_curr => IDC_registerNewEquilibrium(t_equilib_chain, c_prneqnamesgen, c_prneqnamessid)

    ! Complete the configuration of the equilibrium
    CALL IDC_configureEquilibrium(stkmx_rootelt, t_compo_chain, t_equilib_curr)

    ! Proceed to the next one
    xml_work => xml_work%next

  ENDDO

ELSE

  WRITE(jp_stdout,'()')
  WRITE(jp_stdout,'("--Empty--")')
# ifdef CFG_DEBUG
  WRITE(jp_stddbg,'()')
  WRITE(jp_stddbg,'("--Empty--")')
# endif

ENDIF


!---------------------!
! API Extension files !
!---------------------!

WRITE(jp_stdout,'()')
WRITE(jp_stdout,'()')
WRITE(jp_stdout,'("API-Extension-file tree")')
WRITE(jp_stdout,'("=======================")')
#ifdef CFG_DEBUG
WRITE(jp_stddbg,'()')
WRITE(jp_stddbg,'()')
WRITE(jp_stddbg,'("API-Extension-file tree")')
WRITE(jp_stddbg,'("=======================")')
#endif

NULLIFY(t_apiext_chain)

IF (ASSOCIATED(xml_apiexfiles)) THEN

  t_apiext_chain => APIEXTENSION_createRoot()

  xml_work        => xml_apiexfiles


  DO WHILE(ASSOCIATED(xml_work))

    c_filename = xml_work%fname
    c_type     = xml_work%c_type

    WRITE(jp_stdout,'()')
    WRITE(jp_stdout,'("File """,A,""":")') TRIM(c_filename)

    stkmx_rootelt => xml_work%mx
    stkrc_eltname => STKMX_getElementName(stkmx_rootelt)
    CALL STKRC_copyStkrcToStr(stkrc_eltname, c_rootname, nlen_returned)
    IF (nlen_returned < 0) THEN
      WRITE(jp_stdout,c_fmtwar_a) &
        'root-element name = <"' // c_rootname //'"> has been truncated'
    ENDIF

    IF (c_rootname /= c_type) THEN
      WRITE(jp_stderr,c_fmterr_a) &
        'Root element mismatch!'
      WRITE(jp_stderr,'(" - expected <",A,">")') TRIM(c_type)
      WRITE(jp_stderr,'(" - found    <",A,">")') TRIM(c_rootname)
      WRITE(jp_stderr,'("Aborting!")')
      CALL ABORT()
    ENDIF

    SELECT CASE(c_rootname)
    CASE DEFAULT
      CONTINUE
    END SELECT


    ! Read the <Names> tag
    CALL IDC_getNames(stkmx_rootelt, c_namesgen, c_nameslon, c_namessid)

    ! Register new API Extension into the APIEXTENSION tree
    t_apiext_curr => IDC_registerNewAPIExtension(t_apiext_chain, c_namesgen, c_namessid)

    ! Read the <CodeBits> tag 
    CALL IDC_attachAPIExtensionCodebits(stkmx_rootelt, t_apiext_curr)

    ! Proceed to the next one
    xml_work => xml_work%next

  ENDDO

ELSE

  WRITE(jp_stdout,'()')
  WRITE(jp_stdout,'("--Empty--")')
# ifdef CFG_DEBUG
  WRITE(jp_stddbg,'()')
  WRITE(jp_stddbg,'("--Empty--")')
# endif

ENDIF



! Categorize components into categories 0, 1, 2
!
! Initially:
!   - all solids that enter the static volume conservation equation
!     i.e., all components that have %phasid = 'if'
!     and (%class == cp_classsolid .OR. %class == cp_classorgmcnp)
!     are of category 1
!   - all other modelled components are of category 2
!   - all ignored (dummy) and parametrized components are
!     of category 0
! This has been prepared at configuration.
!
! Then, add:
!   1. from each process whose chemical reaction involves one of the
!      category 1 components already on the list
!      - all those from the rate law that have %kindofparam == 'io'
!   2. from each equilibrium whose chemical reaction involves one of
!      the category 1 components already on the list
!      - all those from the equilibrium relationship that have
!        %kindofparam == 'io'
!   3. from each solute system that includes one of the category 1
!      components already on the list,
!      - all the other members of the system
!   4. Repeat from 1. until no further additions have been registered

#ifdef CFG_DEBUG
WRITE(jp_stddbg,'()')
WRITE(jp_stddbg,'()')
WRITE(jp_stddbg,'("Primary variable detection")')
WRITE(jp_stddbg,'()')
#endif


i_round = 0

DO

  i_round = i_round + 1

  n_newcategory1 = 0

  t_process_work => t_process_chain

  DO WHILE(ASSOCIATED(t_process_work))
    CALL IDC_switchProcessCompos(t_compo_chain, t_process_work, n_newcategory1)
    t_process_work => t_process_work%next
  ENDDO


  t_equilib_work => t_equilib_chain

  DO WHILE(ASSOCIATED(t_equilib_work))
    CALL IDC_switchEquilibCompos(t_compo_chain, t_equilib_work, n_newcategory1)
    t_equilib_work => t_equilib_work%next
  ENDDO


  t_solsys_work => t_solsys_chain

  DO WHILE(ASSOCIATED(t_solsys_work))
    CALL IDC_switchSoluteSystemCompos(t_compo_chain, t_solsys_work, n_newcategory1)
    t_solsys_work => t_solsys_work%next
  ENDDO


# ifdef CFG_DEBUG
  WRITE(jp_stddbg, '(" - round ", I0, ": ")', ADVANCE='NO') i_round

  SELECT CASE(n_newcategory1)
  CASE(0)
    WRITE(jp_stddbg, '("no variables shifted to category 1 -- done")')
  CASE(1)
    WRITE(jp_stddbg, '("1 variable shifted to category 1")')
  CASE DEFAULT
    WRITE(jp_stddbg, '(I0, " variables shifted to category 1")') n_newcategory1
  END SELECT
# endif

  IF (n_newcategory1 == 0) EXIT

ENDDO


#ifdef CFG_DEBUG
WRITE(jp_stddbg,'()')
WRITE(jp_stddbg,'()')
WRITE(jp_stddbg,'("========= Component List Details =========")')

t_compo_work => t_compo_chain

DO WHILE(ASSOCIATED(t_compo_work))
  WRITE(c_procnb,'(I0)') t_compo_work%idx
  WRITE(jp_stddbg,'()')
  WRITE(jp_stddbg,'(A)') 'Component ' // TRIM(c_procnb) // ':'
  CALL COMPOINFO_dump(t_compo_work, jp_stddbg, 3, 3)

  t_compo_work => t_compo_work%next
ENDDO

WRITE(jp_stddbg,'()')
WRITE(jp_stddbg,'("==========================================")')
WRITE(jp_stddbg,'()')



WRITE(jp_stddbg,'()')
WRITE(jp_stddbg,'("======= Solute System List Details =======")')

t_solsys_work => t_solsys_chain

DO WHILE(ASSOCIATED(t_solsys_work))
  WRITE(c_procnb,'(I0)') t_solsys_work%idx
  WRITE(jp_stddbg,'()')
  WRITE(jp_stddbg,'(A)') 'Solute System ' // TRIM(c_procnb) // ':'
  CALL SYSTEMSINFO_dump(t_solsys_work, jp_stddbg, 3, 3)

  t_solsys_work => t_solsys_work%next
ENDDO

WRITE(jp_stddbg,'()')
WRITE(jp_stddbg,'("==========================================")')
WRITE(jp_stddbg,'()')




WRITE(jp_stddbg,'()')
WRITE(jp_stddbg,'("========== Process List Details ==========")')

t_process_work => t_process_chain

DO WHILE(ASSOCIATED(t_process_work))
  WRITE(c_procnb,'(i0)') t_process_work%idx
  WRITE(jp_stddbg,*)
  WRITE(jp_stddbg,'(A)') 'Process ' // TRIM(c_procnb) // ':'
  CALL PROCESS_dump(t_process_work, jp_stddbg, 3, 3)

  t_process_work => t_process_work%next
ENDDO

WRITE(jp_stddbg,'()')
WRITE(jp_stddbg,'("==========================================")')
WRITE(jp_stddbg,'()')




WRITE(jp_stddbg,'()')
WRITE(jp_stddbg,'("======== Equilibrium List Details ========")')

t_equilib_work => t_equilib_chain

DO WHILE(ASSOCIATED(t_equilib_work))
  WRITE(c_procnb,'(i0)') t_equilib_work%idx
  WRITE(jp_stddbg,*)
  WRITE(jp_stddbg,'(A)') 'Equilibrium ' // TRIM(c_procnb) // ':'
  CALL EQUILIB_dump(t_equilib_work, jp_stddbg, 3, 3)

  t_equilib_work => t_equilib_work%next
ENDDO

WRITE(jp_stddbg,'()')
WRITE(jp_stddbg,'("==========================================")')
WRITE(jp_stddbg,'()')




WRITE(jp_stddbg,'()')
WRITE(jp_stddbg,'("======= API Extension List Details =======")')

t_apiext_work => t_apiext_chain

DO WHILE(ASSOCIATED(t_apiext_work))
  WRITE(c_procnb,'(i0)') t_apiext_work%idx
  WRITE(jp_stddbg,*)
  WRITE(jp_stddbg,'(A)') 'API Extension ' // TRIM(c_procnb) // ':'
  CALL APIEXTENSION_dump(t_apiext_work, jp_stddbg, 3, 3)

  t_apiext_work => t_apiext_work%next
ENDDO

WRITE(jp_stddbg,'()')
WRITE(jp_stddbg,'("==========================================")')
WRITE(jp_stddbg,'()')
#endif


WRITE(jp_stdout,c_fmtinf_a) 'completed'
#ifdef CFG_DEBUG
WRITE(jp_stddbg,c_fmtdbg_a) 'completed'
#endif

RETURN

CONTAINS


!-----------------------------------------------------------------------
 SUBROUTINE IDC_getNames(stkmx_rootelt, c_namesgen, c_nameslon, c_namessid)
!-----------------------------------------------------------------------

IMPLICIT NONE


! Argument list variables
! -----------------------

TYPE(stack_minixml), POINTER  :: stkmx_rootelt
CHARACTER(LEN=*), INTENT(OUT) :: c_namesgen
CHARACTER(LEN=*), INTENT(OUT) :: c_nameslon
CHARACTER(LEN=*), INTENT(OUT) :: c_namessid


! Local variables
! ---------------

TYPE(stack_minixml), POINTER :: stkmx_names
TYPE(stack_minixml), POINTER :: stkmx_work
TYPE(stkrc_ptr), DIMENSION(:), POINTER ::  stkrcp_work


NULLIFY(stkmx_names)

                                      ! Find the element tag <Names>, if any
stkmx_names => STKMX_getUniqueChildEltByName(stkmx_rootelt, cp_names)

NULLIFY(stkmx_work)
NULLIFY(stkrcp_work)
stkmx_work  => STKMX_getUniqueChildEltByName(stkmx_names, cp_namesgeneric)
stkrcp_work => STKMX_getPCDatacntt(stkmx_work)
CALL STKRC_copyStkrcToStr(stkrcp_work(1)%ptr, c_namesgen)

NULLIFY(stkmx_work)
NULLIFY(stkrcp_work)
stkmx_work  => STKMX_getUniqueChildEltByName(stkmx_names, cp_nameslong)
stkrcp_work => STKMX_getPCDatacntt(stkmx_work)
CALL STKRC_copyStkrcToStr(stkrcp_work(1)%ptr, c_nameslon)

NULLIFY(stkmx_work)
NULLIFY(stkrcp_work)
stkmx_work  => STKMX_getUniqueChildEltByName(stkmx_names, cp_namesshortid)
stkrcp_work => STKMX_getPCDatacntt(stkmx_work)
CALL STKRC_copyStkrcToStr(stkrcp_work(1)%ptr, c_namessid)

NULLIFY(stkmx_work)
NULLIFY(stkrcp_work)

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE IDC_getNames
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 FUNCTION IDC_getCodeBits(stkmx_rootelt) RESULT(t_codebits_root)
!-----------------------------------------------------------------------

IMPLICIT NONE


! Argument list variables
! -----------------------

TYPE(stack_minixml), POINTER    :: stkmx_rootelt
TYPE(CODEBITS), POINTER         :: t_codebits_root


! Local variables
! ---------------

TYPE(stack_minixml), POINTER    :: stkmx_work, stkmx_work1
TYPE(stkmx_ptr), &
  DIMENSION(:), POINTER         :: stkmxp_work
CHARACTER(LEN=n_lmaxcodeline), &
  DIMENSION(:), POINTER         :: c_codeblock
TYPE(CODEBITS), POINTER         :: t_code_wk
TYPE(stack_recchunks), POINTER  :: stkrc_eltname
TYPE(stkrc_ptr),  &
  DIMENSION(:), POINTER         :: stkrcp_work
INTEGER                         :: n_codeelts
CHARACTER(LEN=p_maxlen_eltname) :: c_codebitname
INTEGER                         :: i
INTEGER                         :: i_att
TYPE(stack_recchunks), &
  POINTER                       :: stkrc_attcntt
CHARACTER(LEN=p_maxlen_attcntt) &
                                :: c_att_units
CHARACTER(LEN=p_maxlen_attcntt) &
                                :: c_att_vartype


CHARACTER(LEN=*), PARAMETER     :: c_fmtinf   = '("[[IDC_getCodeBits]]: ")'
CHARACTER(LEN=*), PARAMETER     :: c_fmtwar_a = '("[[IDC_getCodeBits]] warning: ", A)'


NULLIFY(t_codebits_root)
NULLIFY(t_code_wk)

NULLIFY(stkmx_work)
NULLIFY(stkmxp_work)

stkmx_work  => STKMX_getUniqueChildEltByName(stkmx_rootelt, cp_codebits)
IF(ASSOCIATED(stkmx_work)) THEN

  WRITE(jp_stdout,'("  - found <", A, "> tag")') TRIM(cp_codebits)

                                  ! Find the element tag names
  stkmxp_work => STKMX_getChildElementNodes(stkmx_work)
  IF(ASSOCIATED(stkmxp_work)) THEN

    n_codeelts = SIZE(stkmxp_work)

    DO i = 1, n_codeelts

      c_att_units = '-'
      c_att_vartype = cp_ftntype_dbleprec
      NULLIFY(c_codeblock)

      stkrc_eltname => STKMX_getElementName(stkmxp_work(i)%ptr)
      CALL STKRC_copyStkrcToStr(stkrc_eltname, c_codebitname)

      WRITE(jp_stdout,'("    * found <", A ,"> child ")') TRIM(c_codebitname)

                                    ! Search for the "units" attribute
      i_att = STKMX_getAttIdxByName(stkmxp_work(i)%ptr, cp_units)

      IF (i_att > 0) THEN
        stkrc_attcntt => STKMX_getAttcnttByIdx(stkmxp_work(i)%ptr, i_att)
        CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_att_units)
        WRITE(jp_stdout, c_fmtinf)
        WRITE(jp_stdout,'("      attribute ", A, "=""", A,"""")') cp_units, TRIM(c_att_units)
      ENDIF

      stkmx_work1 => STKMX_getUniqueChildEltByName(stkmxp_work(i)%ptr, cp_fortran)

      IF (ASSOCIATED(stkmx_work1)) THEN

        WRITE(jp_stdout,'("      + Processing <", A ,">")') TRIM(cp_fortran)

        WRITE(jp_stdout,'("        - checking for """, A ,""" attribute")') cp_vartype


                                    ! Search for the "vartype" attribute
        i_att = STKMX_getAttIdxByName(stkmx_work1, cp_vartype)

        IF (i_att > 0) THEN
          stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_work1, i_att)
          CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_att_vartype)
        ENDIF


        WRITE(jp_stdout,'("        - trying to load code")')

        stkrcp_work => STKMX_getCDatacntt(stkmx_work1)

        IF (ASSOCIATED(stkrcp_work)) THEN
          CALL STKRC_createSplitCopyStr(stkrcp_work(1)%ptr, c_codeblock)

          IF(ASSOCIATED(t_codebits_root)) THEN
            ALLOCATE(t_code_wk%next)
            t_code_wk => t_code_wk%next
          ELSE
            ALLOCATE(t_codebits_root)
            t_code_wk => t_codebits_root
          ENDIF

          NULLIFY(t_code_wk%next)
          t_code_wk%code => c_codeblock
          t_code_wk%type    = TRIM(c_codebitname)
          t_code_wk%units   = TRIM(c_att_units)
          t_code_wk%vartype = TRIM(c_att_vartype)

          IF (ASSOCIATED(c_codeblock)) THEN
            t_code_wk%nlines = SIZE(c_codeblock)
            WRITE(jp_stdout,'("        successfully completed")')
          ELSE
            WRITE(jp_stdout,'("        <", A,"> element empty")') TRIM(cp_fortran)
            t_code_wk%nlines = 0
            NULLIFY(t_code_wk%code)
          ENDIF

        ELSE

          IF(ASSOCIATED(t_codebits_root)) THEN
            ALLOCATE(t_code_wk%next)
            t_code_wk => t_code_wk%next
          ELSE
            ALLOCATE(t_codebits_root)
            t_code_wk => t_codebits_root
          ENDIF

          NULLIFY(t_code_wk%next)

          NULLIFY(t_code_wk%code)
          t_code_wk%nlines  = 0
          t_code_wk%type    = TRIM(c_codebitname)
          t_code_wk%units   = TRIM(c_att_units)
          t_code_wk%vartype = TRIM(c_att_vartype)

          WRITE(jp_stdout,'("        <", A,"> element empty")') TRIM(cp_fortran)
          WRITE(jp_stdout,'("        hopefully this is acceptable")')


        ENDIF

      ELSE
        WRITE(jp_stdout,'("      + no <", A, "> tag found - cannot load any code")') TRIM(cp_fortran)
      ENDIF

    ENDDO

  ELSE

    WRITE(jp_stdout,'("        <", A,"> has no child elements (???)")') TRIM(cp_codebits)

  ENDIF

ENDIF


RETURN

!-----------------------------------------------------------------------
 END FUNCTION IDC_getCodeBits
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE IDC_attachCompoCodebits(stkmx_rootelt, t_compo_curr)
!-----------------------------------------------------------------------

IMPLICIT NONE


! Argument list variables
! -----------------------

TYPE(stack_minixml), POINTER    :: stkmx_rootelt
TYPE(COMPOINFO), POINTER        :: t_compo_curr


! Local variables
! ---------------


TYPE(CODEBITS), POINTER         :: t_codebits_root
TYPE(CODEBITS), POINTER         :: t_code_wk


t_codebits_root => IDC_getCodeBits(stkmx_rootelt)

IF (ASSOCIATED(t_codebits_root)) THEN

  t_code_wk => t_codebits_root

  DO WHILE (ASSOCIATED(t_code_wk))

    SELECT CASE(t_code_wk%type)
    CASE(cp_totconc)
      t_code_wk%varname = cp_prefix_cctttcc // ADJUSTL(c_namessid)
    CASE(cp_soluprod)
      t_code_wk%varname = cp_prefix_cctksp  // ADJUSTL(c_namessid)
    CASE(cp_satuconc)
      t_code_wk%varname = cp_prefix_cctksat // ADJUSTL(c_namessid)
    CASE(cp_degrsatu)
      t_code_wk%varname = cp_prefix_cctdsat // ADJUSTL(c_namessid)
    CASE(cp_diffcoeff)
      t_code_wk%varname = cp_varname_dcfmolion
    CASE DEFAULT
      t_code_wk%varname = '???'
      !TBD Add error message here: unknown type
    END SELECT

    t_code_wk => t_code_wk%next

  ENDDO

  NULLIFY(t_code_wk)                ! Advance to the end of the t_compo_curr%codes list
  IF(ASSOCIATED(t_compo_curr%codes)) THEN
    t_code_wk => t_compo_curr%codes
    DO WHILE(ASSOCIATED(t_code_wk%next))
      t_code_wk => t_code_wk%next
    ENDDO
    t_code_wk%next => t_codebits_root
  ELSE
    t_compo_curr%codes => t_codebits_root
  ENDIF

ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE IDC_attachCompoCodebits
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 FUNCTION IDC_registerNewCompo                                         &
   (t_compo_root, c_namesgen, c_namessid, i_system)                    &
   RESULT(t_compo_activenode)
!-----------------------------------------------------------------------

USE MOD_CONFIGURE, ONLY: n_medusacompo

IMPLICIT NONE
TYPE(COMPOINFO),  POINTER      :: t_compo_root
CHARACTER(LEN=*), INTENT(IN)   :: c_namesgen
CHARACTER(LEN=*), INTENT(IN)   :: c_namessid
INTEGER                        :: i_system

TYPE(COMPOINFO), POINTER       :: t_compo_activenode

TYPE(COMPOINFO), POINTER       :: t_compo_work

NULLIFY(t_compo_activenode)

t_compo_work => COMPOINFO_lastNode(t_compo_root)

IF(t_compo_work%idx == -1) THEN
                                    ! t_compo_work points to the not yet
  t_compo_work%idx = 1              ! initialised t_compo_root element.
ELSE
  CALL COMPOINFO_createNext(t_compo_work)
  t_compo_work    => t_compo_work%next
  t_compo_work%idx = t_compo_work%prev%idx + 1
ENDIF

t_compo_work%idx_xref = 0
t_compo_work%name     = ADJUSTL(c_namesgen)
t_compo_work%shortid  = ADJUSTL(c_namessid)
t_compo_work%i_system = i_system

t_compo_activenode => t_compo_work

NULLIFY(t_compo_work)

n_medusacompo = n_medusacompo + 1

RETURN

!-----------------------------------------------------------------------
 END FUNCTION IDC_registerNewCompo
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 FUNCTION IDC_registerNewSoluteSystem                                  &
   (t_solsys_root, c_namesgen, c_namessid, i_system)                   &
   RESULT(t_solsys_activenode)
!-----------------------------------------------------------------------

USE MOD_CONFIGURE, ONLY: n_modelsolsys

IMPLICIT NONE
TYPE(SYSTEMSINFO),  POINTER    :: t_solsys_root
CHARACTER(LEN=*), INTENT(IN)   :: c_namesgen
CHARACTER(LEN=*), INTENT(IN)   :: c_namessid
INTEGER                        :: i_system

TYPE(SYSTEMSINFO), POINTER     :: t_solsys_activenode

TYPE(SYSTEMSINFO), POINTER     :: t_solsys_work

NULLIFY(t_solsys_activenode)

t_solsys_work => SYSTEMSINFO_lastNode(t_solsys_root)

IF(t_solsys_work%idx == -1) THEN
                                ! t_solsys_work points to the not yet
  t_solsys_work%idx = 1         ! initialised t_compo_chain root element.
ELSE
  CALL SYSTEMSINFO_createNext(t_solsys_work)
  t_solsys_work    => t_solsys_work%next
  t_solsys_work%idx = t_solsys_work%prev%idx + 1
ENDIF

t_solsys_work%name    = ADJUSTL(c_namesgen)
t_solsys_work%shortid = ADJUSTL(c_namessid)
t_solsys_work%i_system = i_system


t_solsys_activenode => t_solsys_work

NULLIFY(t_solsys_work)

n_modelsolsys = n_modelsolsys + 1

RETURN

!-----------------------------------------------------------------------
 END FUNCTION IDC_registerNewSoluteSystem
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE IDC_registerDummy(t_compo, c_namesgen)
!-----------------------------------------------------------------------
USE MOD_CONFIGURE, ONLY: n_medusacompo
IMPLICIT NONE
TYPE(COMPOINFO), POINTER       :: t_compo
CHARACTER(LEN=*), INTENT(IN)   :: c_namesgen
TYPE(COMPOINFO), POINTER       :: t_compo_work


! Check if t_compo already contains a component called <c_namesgen>

t_compo_work => t_compo

IF(t_compo_work%idx == -1) THEN
                                    ! t_compo_work points to the not yet
  t_compo_work%idx = 1              ! initialised t_compo_chain root element.

ELSE

  DO WHILE(ASSOCIATED(t_compo_work))
    IF (t_compo_work%name /= c_namesgen) THEN
      t_compo_work => t_compo_work%next
      CYCLE
    ELSE
      NULLIFY(t_compo_work)
      RETURN
    ENDIF
  ENDDO

  CALL COMPOINFO_createNext(t_compo_work)
  t_compo_work    => t_compo_work%next
  t_compo_work%idx = t_compo_work%prev%idx + 1

ENDIF

t_compo_work%name    = ADJUSTL(c_namesgen)
t_compo_work%shortid = '---'
t_compo_work%phasid  = 'ii'

n_medusacompo = n_medusacompo + 1


RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE IDC_registerDummy
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE IDC_configureSolute(stkmx_solute, t_compo, l_extra_copy)
!-----------------------------------------------------------------------

IMPLICIT NONE


! Argument list variables
! -----------------------

TYPE(stack_minixml), POINTER :: stkmx_solute
TYPE(COMPOINFO), POINTER     :: t_compo
LOGICAL, OPTIONAL            :: l_extra_copy


! Local variables
! ---------------

TYPE(stack_minixml), POINTER :: stkmx_checom
LOGICAL                      :: lloc_extra_copy
TYPE(stack_minixml), POINTER :: stkmx_nameselt
TYPE(stack_minixml), POINTER :: stkmx_names
TYPE(stack_minixml), POINTER :: stkmx_work, stkmx_work1, stkmx_work2
TYPE(stkmx_ptr), POINTER     :: stkmxp_work
TYPE(stkrc_ptr), &
  DIMENSION(:), POINTER      :: stkrcp_work
INTEGER :: i_att
TYPE(stack_recchunks), &
  POINTER                    :: stkrc_attcntt

CHARACTER(LEN=n_lmaxcodeline), &
  DIMENSION(:), POINTER      :: c_codeblock

CHARACTER(LEN=p_maxlen_attcntt) &
                             :: c_att_type
CHARACTER(LEN=p_maxlen_attcntt) &
                             :: c_att_class

LOGICAL                      :: l_getparams
CHARACTER(LEN=n_lmaxexpress) :: c_pcdata

INTEGER                      :: n_params, n_checom

CHARACTER(LEN=n_lmaxphasid)  :: c_phasid
CHARACTER(LEN=*), PARAMETER  :: c_fmtinf   = '("[[IDC_configureSolute]]: ")'
CHARACTER(LEN=*), PARAMETER  :: c_fmtwar_a = '("[[IDC_configureSolute]] warning: ", A)'
CHARACTER(LEN=*), PARAMETER  :: c_fmterr_a = '("[[IDC_configureSolute]] error: ", A)'


IF (PRESENT(l_extra_copy)) THEN
  lloc_extra_copy = l_extra_copy
ELSE
  lloc_extra_copy = .FALSE.
ENDIF


! Operations
! ----------

c_att_type = cp_typeignored

i_att = STKMX_getAttIdxByName(stkmx_solute, cp_type)

IF (i_att > 0) THEN
  stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_solute, i_att)
  CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_att_type)
  WRITE(jp_stdout, c_fmtinf, ADVANCE='NO')
  WRITE(jp_stdout,'("attribute ", A, "=""", A,"""")') cp_type, TRIM(c_att_type)
ELSE
  WRITE(jp_stdout, c_fmtwar_a) 'missing attribute ' // cp_type // ' -- ' &
                   // ' using default value "' // cp_typeignored //'"'
  c_att_type = cp_typeignored
ENDIF


i_att = STKMX_getAttIdxByName(stkmx_solute, cp_class)

IF (i_att > 0) THEN
  stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_solute, i_att)
  CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_att_class)
  WRITE(jp_stdout, c_fmtinf, ADVANCE='NO')
  WRITE(jp_stdout,'("attribute ", A, "=""", A,"""")') cp_class, TRIM(c_att_class)
ELSE
  WRITE(jp_stdout, c_fmtwar_a) 'missing attribute ' // cp_class // ' -- &
                     &using default value "' //TRIM(cp_classsolute) // '"'
  c_att_class = cp_classsolute
ENDIF



SELECT CASE(c_att_type)
CASE(cp_typenormal)
  IF (lloc_extra_copy) THEN
    l_getparams = .FALSE.
    c_phasid = 'xc'                 ! extra copy of the solute; possibly
                                    ! member of more than one solute system
  ELSE
    l_getparams = .TRUE.
    c_phasid = 'ic'

  ENDIF

CASE(cp_typeignored)
  l_getparams = .FALSE.
  c_phasid = 'ii'

CASE(cp_typeparamed)
  l_getparams = .FALSE.
  c_phasid = 'ip'
  ! Check if code for parameterization is present
  ! (=> %codes associated and contains a CodeBit with an accepted name)

CASE DEFAULT
  WRITE(jp_stdout,c_fmtwar_a) 'unknown type="'//TRIM(c_att_type)//'"' &
                    //' -- using default value "'//TRIM(cp_typeignored)//'"'
  c_att_type = cp_typeignored
  l_getparams = .FALSE.
  c_phasid = 'ii'

END SELECT


t_compo%class  = c_att_class
t_compo%phasid = c_phasid


IF (c_phasid == 'ic') THEN          ! Update an 'ic' component's category
                                    ! (0, 1, 2) if necessary
  SELECT CASE(c_att_class)
  CASE(cp_classsolute,cp_classorgmcnp)
    t_compo%i_category = 2          ! - all normal and OrgMatter_CNP solutes
                                    !   provisionally set to category 2.

  CASE DEFAULT
    CONTINUE                        ! - all others left in category 0

  END SELECT

ENDIF



IF (l_getparams) THEN

  ! Get number n_checom of child elements of <ChemicalComposition>
  stkmx_checom => STKMX_getUniqueChildEltByName(stkmx_solute, cp_checom)
  IF(ASSOCIATED(stkmx_checom)) THEN

    n_checom = stkmx_checom%n_children

  ELSE

    SELECT CASE(c_att_class)
    CASE(cp_classorgmcnp)
      WRITE(jp_stderr, c_fmterr_a) '<' // cp_checom // '> is mandatory for &
         &solutes with class="' // cp_classorgmcnp // '" and cannot be empty -- aborting!'
        CALL ABORT()

    CASE DEFAULT
      n_checom = 0

    END SELECT

  ENDIF

  SELECT CASE(c_att_class)
  CASE(cp_classsolute,cp_classorgmcnp)

    n_params = 1 + n_checom ! alk, n_checom composition params.

    t_compo%n_params = n_params
    t_compo%n_checom = n_checom
    ALLOCATE(t_compo%param_name(n_params))
    ALLOCATE(t_compo%param_varname(n_params))
    ALLOCATE(t_compo%param_vartype(n_params))
    ALLOCATE(t_compo%param_values(n_params))

    CALL IDC_registerAlkalinity(stkmx_solute, t_compo, 1)
!    CALL IDC_registerMolWeight(stkmx_solute, t_compo, 2)
    IF(n_checom > 0) THEN
      CALL IDC_registerChemCompo(stkmx_checom, t_compo, 2)
    ENDIF

  CASE DEFAULT
    WRITE(jp_stderr, c_fmterr_a) 'Unknown Solute class="' // TRIM(c_att_class) // '"  aborting!'
    CALL ABORT()

  END SELECT

  CALL IDC_registerConsrvProps(stkmx_solute, t_compo)

ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE IDC_configureSolute
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE IDC_configureSoluteSystem(stkmx_solsys, t_compo, t_solsys)
!-----------------------------------------------------------------------

USE MOD_CONFIGURE, ONLY: n_medusacompo

IMPLICIT NONE


! Argument list variables
! -----------------------

TYPE(stack_minixml), POINTER :: stkmx_solsys
TYPE(COMPOINFO), POINTER     :: t_compo
TYPE(SYSTEMSINFO), POINTER   :: t_solsys


! Local variables
! ---------------

INTEGER                         :: i_att
TYPE(stack_recchunks), POINTER  :: stkrc_attcntt
CHARACTER(LEN=p_maxlen_attcntt) :: c_att_type

CHARACTER(LEN=n_lmaxphasid)     :: c_phasid
TYPE(COMPOINFO), POINTER        :: t_compo_scan

CHARACTER(LEN=n_lmaxnamesgen), &
    DIMENSION(:), POINTER    :: member_name_tmp
INTEGER                         :: i_system, n_members, i_member

CHARACTER(LEN=*), PARAMETER  :: c_fmtinf   = '("[IDC_configureSoluteSystem]: ")'
CHARACTER(LEN=*), PARAMETER  :: c_fmtwar_a = '("[IDC_configureSoluteSystem] warning: ", A)'
CHARACTER(LEN=*), PARAMETER  :: c_fmterr_a = '("[IDC_configureSoluteSystem] error: ", A)'
CHARACTER(LEN=*), PARAMETER  :: c_fmterr_ai = '("[IDC_configureSoluteSystem] error: ", A, I0)'

!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
! Operations
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

i_att = STKMX_getAttIdxByName(stkmx_solsys, cp_type)

IF (i_att > 0) THEN
  stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_solsys, i_att)
  CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_att_type)
  WRITE(jp_stdout, c_fmtinf, ADVANCE='NO')
  WRITE(jp_stdout,'("attribute type=""", A,"""")') TRIM(c_att_type)
ELSE                                ! Default: take it as "normal"
                                    ! It does not make sense to have a
                                    ! <SoluteSystem> with type="ignored"
                                    ! by default, as we have chosen for
                                    ! <Solid> and <Solute>
  WRITE(jp_stdout, c_fmtwar_a) 'missing attribute type -- &
                     &using default value "' // TRIM(cp_typenormal) //'"'
  c_att_type = cp_typenormal
ENDIF

SELECT CASE(c_att_type)
CASE(cp_typenormal)
  c_phasid = 'cc'
! possible extension!
!CASE(cp_paramed)
CASE DEFAULT
  WRITE(jp_stdout,c_fmterr_a) 'unknown type="' // TRIM(c_att_type) // '" -- aborting'
  CALL ABORT()
END SELECT

t_compo%phasid      = c_phasid
t_compo%class       = cp_classsolsys

t_solsys%phasid     = c_phasid
t_solsys%class      = cp_classsolsys
t_solsys%i_category = 2


ALLOCATE(member_name_tmp(n_medusacompo))

n_members = 0
i_system = t_solsys%i_system
t_compo_scan => t_compo%prev
DO WHILE(ASSOCIATED(t_compo_scan))
  SELECT CASE(t_compo_scan%class)
  CASE(cp_classsolute)
    IF(t_compo_scan%i_system == i_system) THEN
      SELECT CASE(t_compo_scan%phasid)
      CASE('ic', 'xc')
        n_members = n_members + 1
        member_name_tmp(n_members) = ADJUSTL(t_compo_scan%name)
      CASE DEFAULT
        WRITE(jp_stderr, c_fmterr_ai, ADVANCE='NO')  'SoluteSystem ', i_system
        WRITE(jp_stderr, '(A)') ' has incorrect phasid "' // TRIM(t_compo_scan%phasid) //'" -- aborting!'
        CALL ABORT()
      END SELECT
    ENDIF
  CASE DEFAULT
    CONTINUE
  END SELECT
  t_compo_scan => t_compo_scan%prev
ENDDO

t_solsys%n_members = n_members

IF (n_members /= 0) THEN
  ALLOCATE(t_solsys%member_name(n_members))
  DO i_member = 1, n_members
    t_solsys%member_name(i_member) = member_name_tmp(i_member)
  ENDDO
ELSE
  WRITE(jp_stderr, c_fmterr_ai, ADVANCE='NO')  'SoluteSystem ', i_system
  WRITE(jp_stderr, '(A)') ' has no members -- aborting!'
  CALL ABORT()
ENDIF

t_solsys%nvc_identifier = 'nvc_' // ADJUSTL(t_solsys%shortid)
t_solsys%ioc_identifier = 'ioc_' // ADJUSTL(t_solsys%shortid)
t_solsys%icc_identifier = 'icc_' // ADJUSTL(t_solsys%shortid)


DEALLOCATE(member_name_tmp)
NULLIFY(member_name_tmp)


RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE IDC_configureSoluteSystem
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE IDC_setupAlkSoluteSystem(t_compo_root, t_solsys_root)
!-----------------------------------------------------------------------

USE MOD_CONFIGURE, ONLY: n_medusacompo, n_modelsolsys, n_solutes_withalk


IMPLICIT NONE


! Argument list variables
! -----------------------

TYPE(COMPOINFO), POINTER     :: t_compo_root
TYPE(SYSTEMSINFO), POINTER   :: t_solsys_root


! Local variables
! ---------------

TYPE(COMPOINFO), POINTER     :: t_compo_curr, t_compo_work
TYPE(SYSTEMSINFO), POINTER   :: t_solsys_curr

CHARACTER(LEN=n_lmaxnamesgen), &
    DIMENSION(:), POINTER    :: member_name_tmp


INTEGER                      :: n_params, i_param
INTEGER                      :: i_solute
DOUBLE PRECISION             :: testalk

CHARACTER(LEN=*), PARAMETER  :: c_fmtinf   = '("[[IDC_setupAlkSoluteSystem]]: ")'
#ifdef CFG_DEBUG
CHARACTER(LEN=*), PARAMETER  :: c_fmtdbg_a = '("DEBUG [[IDC_setupAlkSoluteSystem]]: ", A)'
#endif


!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
! Operations
!- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

                                    ! Detect the 'Solute' components with
                                    ! 'ic' %phasid in t_compo_chain that
                                    ! have an 'Alkalinity' parameter and
                                    ! a non-zero alkalinity value:
                                    ! - count them (--> n_solutes_withalk)
                                    ! - make a temporary list of those
                                    !   names (--> member_name_tmp)

! Temporarily allocate enough space for as many entries as there are
! nodes in t_compo_chain
ALLOCATE(member_name_tmp(n_medusacompo))

t_compo_work => t_compo_root

n_solutes_withalk = 0

DO WHILE(ASSOCIATED(t_compo_work))

  SELECT CASE(t_compo_work%class)
  CASE(cp_classsolute)
    IF(t_compo_work%phasid == 'ic') THEN
      n_params = t_compo_work%n_params
      search_params: DO i_param = 1, n_params
        IF(t_compo_work%param_name(i_param) == cp_alkalinity) THEN
          READ(t_compo_work%param_values(i_param),*) testalk
          IF(testalk /= 0D0) THEN
            n_solutes_withalk = n_solutes_withalk + 1
            member_name_tmp(n_solutes_withalk) = ADJUSTL(t_compo_work%name)
          ENDIF
          EXIT search_params
        ENDIF
      ENDDO search_params
    ENDIF
  CASE DEFAULT
    CONTINUE
  END SELECT

  t_compo_work => t_compo_work%next

ENDDO

WRITE(jp_stdout,'()')
WRITE(jp_stdout,c_fmtinf,ADVANCE='NO')
WRITE(jp_stdout,'("found ", I0, " solutes with an <", A, "> parameter")') &
  n_solutes_withalk, cp_alkalinity

                                    ! If there are solutes with alkalinity,
                                    ! register an <Alkalinity> SoluteSystem
IF (n_solutes_withalk /= 0) THEN

  c_namesgen = ADJUSTL(cp_alkalinity)
  c_namessid = ADJUSTL(cp_shortid_alk)
  i_system   = n_modelsolsys + 1

                                    ! Register it into t_compo_root ...
  t_compo_curr  => IDC_registerNewCompo(t_compo_root, c_namesgen, c_namessid, i_system)
                                    ! ... and configure it
  t_compo_curr%class   = ADJUSTL(cp_classsolsys)
  t_compo_curr%phasid  = 'cc'

                                    ! Register it into t_solsys_root ...
  t_solsys_curr => IDC_registerNewSoluteSystem(t_solsys_root, c_namesgen, c_namessid, i_system)
                                    ! ... and configure it
  t_solsys_curr%class  = ADJUSTL(cp_classsolsys)
  t_solsys_curr%phasid = 'cc'

  t_solsys_curr%n_members = n_solutes_withalk
  ALLOCATE(t_solsys_curr%member_name(n_solutes_withalk))

  DO i_solute = 1, n_solutes_withalk
    t_solsys_curr%member_name(i_solute) = member_name_tmp(i_solute)
  ENDDO

  t_solsys_curr%nvc_identifier  = 'nvc_' // cp_shortid_alk
  t_solsys_curr%ioc_identifier  = 'ioc_' // cp_shortid_alk
  t_solsys_curr%icc_identifier  = 'icc_' // cp_shortid_alk

ENDIF

DEALLOCATE(member_name_tmp)

# ifdef CFG_DEBUG
  WRITE(jp_stddbg, c_fmtdbg_a, ADVANCE='NO') &
    'Composition: %name, %idx, n_medusacompo = '
  WRITE(jp_stddbg, '("""", A, """, ", I0, ", ", I0)') &
    TRIM(t_compo_curr%name), t_compo_curr%idx, n_medusacompo
  WRITE(jp_stddbg, c_fmtdbg_a, ADVANCE='NO') &
    'SoluteSystem: %name, %idx, n_modelsolsys = '
  WRITE(jp_stddbg, '("""", A, """, ", I0, ", ", I0)') &
    TRIM(t_solsys_curr%name), t_solsys_curr%idx, n_modelsolsys
# endif

NULLIFY(t_compo_work)
NULLIFY(t_compo_curr)
NULLIFY(t_solsys_curr)



RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE IDC_setupAlkSoluteSystem
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE IDC_configureSolid(stkmx_solid, t_compo)
!-----------------------------------------------------------------------

USE MOD_CONFIGURE, ONLY: c_shortid_mud

IMPLICIT NONE


! Argument list variables
! -----------------------

TYPE(stack_minixml), POINTER :: stkmx_solid
TYPE(COMPOINFO), POINTER     :: t_compo


! Local variables
! ---------------

TYPE(stack_minixml), POINTER :: stkmx_checom
TYPE(stkrc_ptr), &
  DIMENSION(:), POINTER      :: stkrcp_work
INTEGER :: i_att
TYPE(stack_recchunks), &
  POINTER                    :: stkrc_attcntt


CHARACTER(LEN=n_lmaxcodeline), &
  DIMENSION(:), POINTER      :: c_codeblock

CHARACTER(LEN=p_maxlen_attcntt) &
                             :: c_att_type
CHARACTER(LEN=p_maxlen_attcntt) &
                             :: c_att_class
CHARACTER(LEN=p_maxlen_attcntt) &
                             :: c_att_extra
CHARACTER(LEN=p_maxlen_attcntt) &
                             :: c_att_master

TYPE(COMPOINFO), POINTER     :: t_compo_master

TYPE(CODEBITS), POINTER      :: t_code_new, t_code_wk

LOGICAL                      :: l_getparams
CHARACTER(LEN=n_lmaxphasid)  :: c_phasid

INTEGER                      :: n_params, n_checom
INTEGER                      :: n_checom_from_master
INTEGER                      :: i_param,  i_checom
INTEGER                      :: nlen

CHARACTER(LEN=n_lmaxidentif) :: c_varname

CHARACTER(LEN=*), PARAMETER  :: c_fmtinf   = '("[[IDC_configureSolid]]: ")'
CHARACTER(LEN=*), PARAMETER  :: c_fmtwar_a = '("[[IDC_configureSolid]] warning: ", A)'
CHARACTER(LEN=*), PARAMETER  :: c_fmterr_a = '("[[IDC_configureSolid]] error: ", A)'


! Operations
! ----------

c_att_type  = cp_typeignored        ! default value
c_att_class = cp_classsolid         ! default value

                                    ! Check for attribute 'type'
i_att = STKMX_getAttIdxByName(stkmx_solid, cp_type)

IF (i_att > 0) THEN
  stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_solid, i_att)
  CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_att_type)
  WRITE(jp_stdout, c_fmtinf, ADVANCE='NO')
  WRITE(jp_stdout,'("attribute type=""", A,"""")') TRIM(c_att_type)
ELSE
  WRITE(jp_stdout, c_fmtwar_a) 'missing attribute type -- &
                     &using default value "' // TRIM(cp_typeignored) //'"'
  c_att_type = cp_typeignored
ENDIF

                                    ! Check for attribute 'class'
i_att = STKMX_getAttIdxByName(stkmx_solid, cp_class)

IF (i_att > 0) THEN
  stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_solid, i_att)
  CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_att_class)
  WRITE(jp_stdout, c_fmtinf, ADVANCE='NO')
  WRITE(jp_stdout,'("attribute class=""", A,"""")') TRIM(c_att_class)
ELSE
  WRITE(jp_stdout, c_fmtwar_a) 'missing attribute class -- &
                     &using default value "' //TRIM(cp_classsolid) // '"'
  c_att_class = cp_classsolid
ENDIF

                                    ! Check for attribute 'extra'
i_att = STKMX_getAttIdxByName(stkmx_solid, cp_extra)

IF (i_att > 0) THEN

  stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_solid, i_att)
  CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_att_extra)
  WRITE(jp_stdout, c_fmtinf, ADVANCE='NO')
  WRITE(jp_stdout,'("attribute extra=""", A,"""")') TRIM(c_att_extra)

                                    ! Check if attribute extra has not
                                    ! already been used for another
                                    ! component
  SELECT CASE(c_att_extra)
  CASE(cp_extramud)
    IF (LEN_TRIM(c_shortid_mud) == 0) THEN
      c_shortid_mud = t_compo%shortid
    ELSE
      WRITE(jp_stdout, c_fmtwar_a) 'ignoring attribute extra="' // cp_extramud // '":'
      WRITE(jp_stdout, '(A)')      ' -- c_shortid_mud already set to "' // TRIM(c_shortid_mud) // '".'
    ENDIF

  CASE DEFAULT
    WRITE(jp_stdout, c_fmtwar_a) 'ignoring unknown attribute value extra="' // TRIM(c_att_extra) // '"'

  END SELECT

ENDIF


                                    ! Check for attribute 'master'
i_att = STKMX_getAttIdxByName(stkmx_solid, cp_master)

IF (i_att > 0) THEN

  stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_solid, i_att)
  CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_att_master)
  WRITE(jp_stdout, c_fmtinf, ADVANCE='NO')
  WRITE(jp_stdout,'("attribute master=""", A,"""")') TRIM(c_att_master)


                                    ! Scan preceeding entries of t_compo
                                    ! to search for the component pointed
                                    ! to by the 'master' attribute.
                                    ! It must be a solid.
    t_compo_master => t_compo%prev

    DO WHILE (ASSOCIATED(t_compo_master))

      IF (t_compo_master%name == c_att_master) THEN

        IF (t_compo_master%phasid == 'if') THEN
          t_compo%idx_xref = t_compo_master%idx
          t_compo%xref => t_compo_master
          EXIT
        ELSE
          WRITE(jp_stderr, c_fmterr_a) &
            'attribute "' // cp_master // '" of <' // cp_solid // '> '
          WRITE(jp_stderr, c_fmterr_a) &
            'points to a component with %phasid="' // TRIM(t_compo_master%phasid) // '".'
          WRITE(jp_stderr, c_fmterr_a) &
            'instead of "if" -- aborting!'
          CALL ABORT()
        ENDIF

      ENDIF

      t_compo_master => t_compo_master%prev

    ENDDO

    IF (t_compo%idx_xref == 0) THEN ! If the component indicated by "master"
                                    ! has not been found: abort.
      WRITE(jp_stderr, c_fmterr_a) &
        'attribute "' // cp_master // '" of <' // cp_solid // '> ' // &
        'points to an (as yet) unknown component "' // TRIM(c_att_master) // '".'
      WRITE(jp_stderr, '(A)') &
        'Please make sure that component "' // TRIM(c_att_master) // '" preceeds ' // &
        'component "' // TRIM(t_compo%name) // '" in the component list.'
      WRITE(jp_stderr, '(A)') &
        'Aborting!'
      CALL ABORT()
    ENDIF

ENDIF


SELECT CASE(c_att_type)
CASE(cp_typenormal)
  l_getparams = .TRUE.
  c_phasid    = 'if'

CASE(cp_typeignored)
  l_getparams = .FALSE.
  c_phasid    = 'ii'

CASE DEFAULT
  WRITE(jp_stdout, c_fmtwar_a) 'unknown type="' // TRIM(c_att_type) // '" -- &
                     &using default value "' // TRIM(cp_typeignored) //'"'
  WRITE(jp_stdout,*)
  c_att_type  = cp_typeignored
  l_getparams = .FALSE.
  c_phasid    = 'ii'

END SELECT


t_compo%class  = ADJUSTL(c_att_class)
t_compo%phasid = c_phasid


IF (c_phasid == 'if') THEN          ! Update an 'if' component's category
                                    ! (0, 1, 2) if necessary
  SELECT CASE(c_att_class)
  CASE(cp_classsolid,cp_classorgmcnp)
    t_compo%i_category = 1          ! - all normal and OrgMatter_CNP solids
                                    !   are of category 1.

  CASE DEFAULT
    t_compo%i_category = 2          ! - all others provisionally set
                                    !   to category 2
  END SELECT

ENDIF


IF (l_getparams) THEN

  stkmx_checom => STKMX_getUniqueChildEltByName(stkmx_solid, cp_checom)
  IF(ASSOCIATED(stkmx_checom)) THEN

    n_checom = stkmx_checom%n_children

  ELSE

    SELECT CASE(c_att_class)
    CASE(cp_classorgmcnp)
      WRITE(jp_stderr, c_fmterr_a) '<' // cp_checom // '> is mandatory for &
         &solids with class="' // cp_classorgmcnp // '" -- aborting!'
        CALL ABORT()

    CASE DEFAULT
      n_checom = 0

    END SELECT

  ENDIF

  SELECT CASE(c_att_class)
  CASE(cp_classsolid)

    n_params = 3 + n_checom ! alk, rho, mol_weight + n_checom composition params.

    t_compo%n_params = n_params
    t_compo%n_checom = n_checom
    ALLOCATE(t_compo%param_name(n_params))
    ALLOCATE(t_compo%param_varname(n_params))
    ALLOCATE(t_compo%param_vartype(n_params))
    ALLOCATE(t_compo%param_values(n_params))

    CALL IDC_registerAlkalinity(stkmx_solid, t_compo, 1)
    CALL IDC_registerDensity(stkmx_solid, t_compo, 2)
    CALL IDC_registerMolWeight(stkmx_solid, t_compo, 3)
    IF (n_checom > 0) THEN
      CALL IDC_registerChemCompo(stkmx_checom, t_compo, 4)
    ENDIF


  CASE(cp_classorgmcnp)

    IF (n_checom == 0) THEN
      WRITE(jp_stderr, c_fmterr_a) '<' // cp_checom // '> must not be empty for &
         &solids with class="' // cp_classorgmcnp // '" -- aborting!'
      CALL ABORT()
    ENDIF

    n_params = 3 + n_checom ! alk, rho, mol_weight + n_checom composition params.

    t_compo%n_params = n_params
    t_compo%n_checom = n_checom
    ALLOCATE(t_compo%param_name(n_params))
    ALLOCATE(t_compo%param_varname(n_params))
    ALLOCATE(t_compo%param_vartype(n_params))
    ALLOCATE(t_compo%param_values(n_params))

    CALL IDC_registerAlkalinity(stkmx_solid, t_compo, 1)
    CALL IDC_registerDensity(stkmx_solid, t_compo, 2)
    CALL IDC_registerMolWeight(stkmx_solid, t_compo, 3)
    CALL IDC_registerChemCompo(stkmx_checom, t_compo, 4)


  CASE(cp_classsolidpt)

    n_params = 0
    IF (n_checom /= 0) THEN
      WRITE(jp_stdout, c_fmtwar_a) '<' // cp_checom // '> is ignored for &
        &solids with class="' // cp_classsolidpt // '"!'
        n_checom = 0
    ENDIF

    t_compo%n_params = n_params
    t_compo%n_checom = n_checom


  CASE(cp_classsolidcolour)

    n_checom_from_master = 0
    IF (t_compo%idx_xref /= 0) THEN ! There is a master component
      IF (n_checom == 0) THEN
        n_checom_from_master = t_compo_master%n_checom
        n_checom = n_checom_from_master
        t_compo%l_checom_from_xref = .TRUE.
      ENDIF
    ENDIF

    n_params = 1 + n_checom       ! mol_weight + n_checom composition params.

    t_compo%n_params = n_params
    t_compo%n_checom = n_checom
    ALLOCATE(t_compo%param_name(n_params))
    ALLOCATE(t_compo%param_varname(n_params))
    ALLOCATE(t_compo%param_vartype(n_params))
    ALLOCATE(t_compo%param_values(n_params))


                                    ! First register the mol_weight
    CALL IDC_registerMolWeight(stkmx_solid, t_compo, 1)
    IF (t_compo%param_values(1) == cp_notfound) THEN
      IF (t_compo%idx_xref /= 0) THEN ! There is a master component
        WRITE(jp_stdout, c_fmtwar_a) 'retrieving <' // cp_phypromolwgt // '>  value &
           & from ' // cp_master // '="' // &
           TRIM(t_compo_master%name) //'"'
      DO i_param = 1, t_compo_master%n_params
        IF (t_compo_master%param_name(i_param) == cp_phypromolwgt) THEN
          t_compo%param_values(1) = t_compo_master%param_values(i_param)
          EXIT
        ENDIF
      ENDDO
      ELSE
        WRITE(jp_stderr, c_fmterr_a) 'no ' // cp_master // ' available &
          &to retrieve missing <' // cp_phypromolwgt // '> value -- aborting!'
        CALL ABORT()
      ENDIF
    ENDIF

                                    ! Then register the chemical composition
    IF (n_checom_from_master == 0) THEN
                                    ! - nothing to be copied over from the master
      IF (n_checom > 0) THEN
        CALL IDC_registerChemCompo(stkmx_checom, t_compo, 2)
      ENDIF

    ELSE
                                    ! - chemical composition to be copied over
                                    !   from the master
      i_checom = 2
      DO i_param = 1, t_compo_master%n_params
        IF (INDEX(t_compo_master%param_name(i_param), cp_prefix_checom) == 1) THEN
          t_compo%param_name(i_checom) = t_compo_master%param_name(i_param)
          c_varname = t_compo_master%param_varname(i_param)
          nlen = LEN_TRIM(t_compo_master%shortid)
          t_compo%param_varname(i_checom) = &
            TRIM(t_compo%shortid) // c_varname(nlen+1:)
          t_compo%param_vartype(i_checom) = t_compo_master%param_vartype(i_param)
          t_compo%param_values(i_checom) = t_compo_master%param_values(i_param)
          i_checom = i_checom + 1
        ENDIF
      ENDDO

    ENDIF

  CASE DEFAULT

    WRITE(jp_stderr, c_fmterr_a) 'Unknown Solid class="' // TRIM(c_att_class) // '" -- aborting!'
    CALL ABORT()

  END SELECT

  CALL IDC_registerConsrvProps(stkmx_solid, t_compo)

ENDIF


RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE IDC_configureSolid
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE IDC_registerAlkalinity(stkmx_compo, t_compo, i)
!-----------------------------------------------------------------------

IMPLICIT NONE


! Argument list variables
! -----------------------

TYPE(stack_minixml), POINTER :: stkmx_compo
TYPE(COMPOINFO), POINTER     :: t_compo
INTEGER, INTENT(IN)          :: i


! Local variables
! ---------------

TYPE(stack_minixml), POINTER :: stkmx_work
CHARACTER(LEN=n_lmaxexpress) :: c_pcdata
TYPE(stkrc_ptr), &
  DIMENSION(:), POINTER      :: stkrcp_work

stkmx_work  => STKMX_getUniqueChildEltByName(stkmx_compo, cp_alkalinity)
IF (ASSOCIATED(stkmx_work)) THEN
  stkrcp_work => STKMX_getPCDatacntt(stkmx_work)
  CALL STKRC_copyStkrcToStr(stkrcp_work(1)%ptr, c_pcdata)
ELSE
  c_pcdata = '0'
ENDIF

t_compo%param_name(i)    = cp_alkalinity
t_compo%param_varname(i) = 'eq_mol_' // TRIM(t_compo%shortid)
t_compo%param_vartype(i) = cp_ftntype_dbleprec
t_compo%param_values(i)  = TRIM(ADJUSTL(c_pcdata))

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE IDC_registerAlkalinity
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE IDC_registerDensity(stkmx_compo, t_compo, i)
!-----------------------------------------------------------------------

IMPLICIT NONE


! Argument list variables
! -----------------------

TYPE(stack_minixml), POINTER :: stkmx_compo
TYPE(COMPOINFO), POINTER     :: t_compo
INTEGER, INTENT(IN)          :: i


! Local variables
! ---------------

TYPE(stack_minixml), POINTER :: stkmx_work, stkmx_work1
CHARACTER(LEN=n_lmaxexpress) :: c_pcdata
TYPE(stkrc_ptr), &
  DIMENSION(:), POINTER      :: stkrcp_work

stkmx_work  => STKMX_getUniqueChildEltByName(stkmx_compo, cp_phypro)
stkmx_work1 => STKMX_getUniqueChildEltByName(stkmx_work, cp_phyprodensit)
stkrcp_work => STKMX_getPCDatacntt(stkmx_work1)

IF (ASSOCIATED(stkrcp_work)) THEN

  CALL STKRC_copyStkrcToStr(stkrcp_work(1)%ptr, c_pcdata)

  t_compo%param_name(i)    = cp_phyprodensit
  t_compo%param_varname(i) = 'rho_' // TRIM(t_compo%shortid)
  t_compo%param_vartype(i) = cp_ftntype_dbleprec
  t_compo%param_values(i)  = TRIM(ADJUSTL(c_pcdata))

ELSE

  t_compo%param_name(i)    = cp_phyprodensit
  t_compo%param_varname(i) = '???'
  t_compo%param_vartype(i) = '???'
  t_compo%param_values(i)  = cp_ignore

ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE IDC_registerDensity
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE IDC_registerMolWeight(stkmx_compo, t_compo, i)
!-----------------------------------------------------------------------

IMPLICIT NONE


! Argument list variables
! -----------------------

TYPE(stack_minixml), POINTER :: stkmx_compo
TYPE(COMPOINFO), POINTER     :: t_compo
INTEGER, INTENT(IN)          :: i


! Local variables
! ---------------

TYPE(stack_minixml), POINTER :: stkmx_work, stkmx_work1
CHARACTER(LEN=n_lmaxexpress) :: c_pcdata
TYPE(stkrc_ptr), &
  DIMENSION(:), POINTER      :: stkrcp_work

stkmx_work  => STKMX_getUniqueChildEltByName(stkmx_compo, cp_phypro)
stkmx_work1 => STKMX_getUniqueChildEltByName(stkmx_work, cp_phypromolwgt)
IF(ASSOCIATED(stkmx_work1)) THEN
  stkrcp_work => STKMX_getPCDatacntt(stkmx_work1)
  CALL STKRC_copyStkrcToStr(stkrcp_work(1)%ptr, c_pcdata)
ELSE
  c_pcdata = cp_notfound
  WRITE(jp_stdout, '("[IDC_registerMolWeight] warning: ", A)') &
    'no <' // cp_phypromolwgt // '> found for component "' // TRIM(t_compo%name) // '"'
ENDIF


t_compo%param_name(i)    = cp_phypromolwgt
t_compo%param_varname(i) = 'mol_' // TRIM(t_compo%shortid)
t_compo%param_vartype(i) = cp_ftntype_dbleprec
t_compo%param_values(i)  = TRIM(ADJUSTL(c_pcdata))

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE IDC_registerMolWeight
!-----------------------------------------------------------------------


!!-----------------------------------------------------------------------
! SUBROUTINE IDC_registerCNPParams(stkmx_compo, t_compo, i)
!!-----------------------------------------------------------------------

!IMPLICIT NONE


!! Argument list variables
!! -----------------------

!TYPE(stack_minixml), POINTER :: stkmx_compo
!TYPE(COMPOINFO), POINTER     :: t_compo
!INTEGER, INTENT(IN)          :: i


!! Local variables
!! ---------------

!TYPE(stack_minixml), POINTER :: stkmx_work, stkmx_work1
!CHARACTER(LEN=n_lmaxexpress) :: c_c, c_n, c_p
!TYPE(stkrc_ptr), &
!  DIMENSION(:), POINTER      :: stkrcp_work
!TYPE(stack_recchunks), &
!  POINTER                    :: stkrc_attcntt
!CHARACTER(LEN=p_maxlen_attcntt) &
!                             :: c_suffixc, c_suffixn, c_suffixp
!CHARACTER(LEN=n_lmaxshortid) :: c_shortid

!INTEGER :: i_att

!stkmx_work  => STKMX_getUniqueChildEltByName(stkmx_compo, cp_checom)

!stkmx_work1 => STKMX_getUniqueChildEltByName(stkmx_work, 'C')
!!~ i_att = STKMX_getAttIdxByName(stkmx_work1, cp_suffix)
!!~ IF (i_att > 0) THEN
!!~   stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_work1, i_att)
!!~   CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_suffixc)
!!~ ELSE
!!~   c_suffixc = '_c'
!!~ ENDIF
!stkrcp_work => STKMX_getPCDatacntt(stkmx_work1)
!CALL STKRC_copyStkrcToStr(stkrcp_work(1)%ptr, c_c)

!stkmx_work1 => STKMX_getUniqueChildEltByName(stkmx_work, 'N')
!!~ i_att = STKMX_getAttIdxByName(stkmx_work1, cp_suffix)
!!~ IF (i_att > 0) THEN
!!~   stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_work1, i_att)
!!~   CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_suffixn)
!!~ ELSE
!!~   c_suffixn = cp_suffix_n
!!~ ENDIF
!stkrcp_work => STKMX_getPCDatacntt(stkmx_work1)
!CALL STKRC_copyStkrcToStr(stkrcp_work(1)%ptr, c_n)

!stkmx_work1 => STKMX_getUniqueChildEltByName(stkmx_work, 'P')
!i_att = STKMX_getAttIdxByName(stkmx_work1, cp_suffix)
!!~ IF (i_att > 0) THEN
!!~   stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_work1, i_att)
!!~   CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_suffixp)
!!~ ELSE
!!~   c_suffixp = cp_suffix_p
!!~ ENDIF
!stkrcp_work => STKMX_getPCDatacntt(stkmx_work1)
!CALL STKRC_copyStkrcToStr(stkrcp_work(1)%ptr, c_p)

!c_shortid = t_compo%shortid

!t_compo%param_name(i)      = cp_cnp_paramname_c
!t_compo%param_varname(i)   = TRIM(c_shortid) // '_c'
!t_compo%param_vartype(i)   = cp_ftntype_dbleprec
!t_compo%param_values(i)    = TRIM(ADJUSTL(c_c))

!t_compo%param_name(i+1)    = cp_cnp_paramname_n
!t_compo%param_varname(i+1) = TRIM(t_compo%shortid) // '_n'
!t_compo%param_vartype(i+1) = cp_ftntype_dbleprec
!t_compo%param_values(i+1)  = TRIM(ADJUSTL(c_n))

!t_compo%param_name(i+2)    = cp_cnp_paramname_p
!t_compo%param_varname(i+2) = TRIM(t_compo%shortid) // '_p'
!t_compo%param_vartype(i+2) = cp_ftntype_dbleprec
!t_compo%param_values(i+2)  = TRIM(ADJUSTL(c_p))

!RETURN

!!-----------------------------------------------------------------------
! END SUBROUTINE IDC_registerCNPParams
!!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE IDC_registerChemCompo(stkmx_checom, t_compo, i)
!-----------------------------------------------------------------------

IMPLICIT NONE


! Argument list variables
! -----------------------

TYPE(stack_minixml), POINTER :: stkmx_checom
TYPE(COMPOINFO), POINTER     :: t_compo
INTEGER, INTENT(IN)          :: i


! Local variables
! ---------------

TYPE(stkmx_ptr), DIMENSION(:), POINTER :: stkmxp_elts
CHARACTER(LEN=n_lmaxexpress) :: c_elt
CHARACTER(LEN=n_lmaxexpress) :: c_stoech
TYPE(stkrc_ptr), &
  DIMENSION(:), POINTER      :: stkrcp_stoech
TYPE(stack_recchunks), &
  POINTER                    :: stkrc_elt
CHARACTER(LEN=n_lmaxshortid) :: c_shortid

INTEGER :: i_elt


stkmxp_elts => STKMX_getChildElementNodes(stkmx_checom)

DO i_elt = 1, SIZE(stkmxp_elts)

  stkrc_elt => STKMX_getElementName(stkmxp_elts(i_elt)%ptr)
  CALL STKRC_copyStkrcToStr(stkrc_elt, c_elt)

  stkrcp_stoech => STKMX_getPCDatacntt(stkmxp_elts(i_elt)%ptr)
  CALL STKRC_copyStkrcToStr(stkrcp_stoech(1)%ptr, c_stoech)

  c_shortid = t_compo%shortid

  t_compo%param_name(   i + i_elt - 1) = cp_prefix_checom // TRIM(ADJUSTL(c_elt))
  t_compo%param_varname(i + i_elt - 1) = TRIM(c_shortid) // '_' // TRIM(ADJUSTL(LOWCASE(c_elt)))
  t_compo%param_vartype(i + i_elt - 1) = cp_ftntype_dbleprec
  t_compo%param_values( i + i_elt - 1) = TRIM(ADJUSTL(c_stoech))


ENDDO


RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE IDC_registerChemCompo
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE IDC_registerConsrvProps(stkmx_compo, t_compo)
!-----------------------------------------------------------------------

IMPLICIT NONE


! Argument list variables
! -----------------------

TYPE(stack_minixml), POINTER :: stkmx_compo
TYPE(COMPOINFO), POINTER     :: t_compo


! Local variables
! ---------------

TYPE(stack_minixml),           POINTER :: stkmx_cnspro
TYPE(stkmx_ptr), DIMENSION(:), POINTER :: stkmxp_props
CHARACTER(LEN=n_lmaxidentif)           :: c_propname
CHARACTER(LEN=n_lmaxexpress)           :: c_pcdata
TYPE(stack_recchunks),         POINTER :: stkrc_propname
TYPE(stkrc_ptr), DIMENSION(:), POINTER :: stkrcp_consrvelts
TYPE(stack_minixml),           POINTER :: stkmx_work

INTEGER :: n_consrv, i


stkmx_cnspro => STKMX_getUniqueChildEltByName(stkmx_compo, cp_cnspro)
stkmxp_props => STKMX_getChildElementNodes(stkmx_cnspro)

IF(.NOT.ASSOCIATED(stkmxp_props)) RETURN

n_consrv = SIZE(stkmxp_props)

ALLOCATE(t_compo%consrv_name(n_consrv))
ALLOCATE(t_compo%consrv_weights(n_consrv))

DO i = 1, n_consrv

  stkmx_work => stkmxp_props(i)%ptr

  stkrc_propname => STKMX_getElementName(stkmx_work)
  CALL STKRC_copyStkrcToStr(stkrc_propname, c_propname)

  stkrcp_consrvelts => STKMX_getPCDatacntt(stkmx_work)
  CALL STKRC_copyStkrcToStr(stkrcp_consrvelts(1)%ptr, c_pcdata)

  t_compo%consrv_name(i)    = TRIM(ADJUSTL(c_propname))
  t_compo%consrv_weights(i) = TRIM(ADJUSTL(c_pcdata))

ENDDO

t_compo%n_consrv = n_consrv

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE IDC_registerConsrvProps
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE IDC_getChemReagentChain(stkmx_cherea, c_reagent, &
   t_compo_root, n_reagents, tcr_reagents, c_rid, c_xwildcards)
!-----------------------------------------------------------------------

IMPLICIT NONE


! Argument list variables
! -----------------------

TYPE(stack_minixml), POINTER :: stkmx_cherea
CHARACTER(LEN=*)             :: c_reagent
TYPE(COMPOINFO), POINTER     :: t_compo_root
TYPE(CHEMREAGENT), &
  DIMENSION(:), POINTER      :: tcr_reagents
CHARACTER(LEN=n_lmaxshortid), &
  DIMENSION(:), POINTER      :: c_rid
CHARACTER(LEN=n_lmaxshortid), OPTIONAL, &
  DIMENSION(:), POINTER      :: c_xwildcards
INTEGER                      :: n_reagents


! Local variables
! ---------------

TYPE(stack_minixml), POINTER :: stkmx_reagent
TYPE(stack_minixml), POINTER :: stkmx_name, stkmx_stoechcoeff
CHARACTER(LEN=n_lmaxnamesgen):: c_name
CHARACTER(LEN=n_lmaxshortid) :: c_shortid
CHARACTER(LEN=n_lmaxexpress) :: c_stoechcoeff

TYPE(stkmx_ptr), &
  DIMENSION(:), POINTER      :: stkmxp_reagents
TYPE(stkmx_ptr), &
  DIMENSION(:), POINTER      :: stkmxp_work
TYPE(stack_recchunks), &
  POINTER                    :: stkrc_attcntt

INTEGER                      :: i_att
INTEGER                      :: i_reagent
CHARACTER(LEN=n_lmaxshortid) :: c_id
CHARACTER(LEN=n_lmaxshortid) :: c_wildcard
TYPE(COMPOINFO), POINTER     :: t_compo_work
TYPE(stkrc_ptr), &
  DIMENSION(:), POINTER      :: stkrcp_work
INTEGER                      :: i_system

CHARACTER(LEN=*), PARAMETER  :: c_fmtinf   = '("[[IDC_getChemReagentChain]]: ")'
#ifdef CFG_DEBUG
CHARACTER(LEN=*), PARAMETER  :: c_fmtdbg_a = '("DEBUG [[IDC_getChemReagentChain]]: ", A)'
#endif

stkmxp_reagents => STKMX_getElementNodeByName(stkmx_cherea, c_reagent)

IF(ASSOCIATED(stkmxp_reagents)) THEN

  n_reagents = SIZE(stkmxp_reagents)

  WRITE(jp_stdout,c_fmtinf)
  IF (n_reagents == 1) THEN
    WRITE(jp_stdout,'("<'//cp_cherea//'> has 1 <'//c_reagent//'> child element")')
  ELSE
    WRITE(jp_stdout,'("<'//cp_cherea//'> has ", I0, " <'//c_reagent//'> child elements")') n_reagents
  ENDIF

  ALLOCATE(tcr_reagents(n_reagents))
  ALLOCATE(c_rid(n_reagents))
  IF(PRESENT(c_xwildcards)) ALLOCATE(c_xwildcards(n_reagents))

  DO i_reagent = 1, n_reagents

    stkmx_reagent => stkmxp_reagents(i_reagent)%ptr

    i_att = STKMX_getAttIdxByName(stkmx_reagent, cp_id)
    IF (i_att > 0) THEN
      stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_reagent, i_att)
      CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_id, nlen_returned)
      WRITE(*,'(" <'//c_reagent//'> ", I0, ": id=""",A,"""")') i_reagent, TRIM(c_id)
    ELSE
      !WRITE(*,'(" <'//c_reagent//'> ", I0, ": no ""'//cp_id//'"" attribute")') i_reagent
      c_id = ''
    ENDIF

    IF(PRESENT(c_xwildcards)) THEN
      i_att = STKMX_getAttIdxByName(stkmx_reagent, cp_wildcard)
      IF (i_att > 0) THEN
        stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_reagent, i_att)
        CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_wildcard, nlen_returned)
        WRITE(*,'(" <'//c_reagent//'> ", I0, ": wildcard=""",A,"""")') &
          i_reagent, TRIM(c_wildcard)
      ELSE
        !WRITE(*,'(" <'//c_reagent//'> ", I0, ": no ""'//cp_id//'"" attribute")') i_reagent
        c_wildcard = ''
      ENDIF
    ENDIF

    stkmx_name => STKMX_getUniqueChildEltByName(stkmx_reagent, cp_chereaname)
    stkrcp_work => STKMX_getPCDatacntt(stkmx_name)
    CALL STKRC_copyStkrcToStr(stkrcp_work(1)%ptr, c_name)

    t_compo_work => COMPOINFO_getNodeByName(t_compo_root, c_name)

    IF(.NOT.ASSOCIATED(t_compo_work)) THEN
                                    ! proceed to end of list and
                                    ! append new node with name

      c_shortid = '---'
      i_system  = 0
      t_compo_work => IDC_registerNewCompo(t_compo_root, c_name, c_shortid, i_system)

      t_compo_work%phasid = 'aa' ! automagically added

#     ifdef CFG_DEBUG
      WRITE(jp_stddbg, c_fmtdbg_a, ADVANCE='NO') &
        'Composition: %name, %idx, n_medusacompo = '
      WRITE(jp_stddbg,'("""", A, """, ", I0, ", ", I0)') &
        TRIM(t_compo_work%name), t_compo_work%idx, n_medusacompo
#     endif

    ENDIF

    stkmx_stoechcoeff => STKMX_getUniqueChildEltByName(stkmx_reagent, cp_chereastoech)
    stkrcp_work => STKMX_getPCDatacntt(stkmx_stoechcoeff)
    CALL STKRC_copyStkrcToStr(stkrcp_work(1)%ptr, c_stoechcoeff)

    tcr_reagents(i_reagent)%name    = t_compo_work%name
    tcr_reagents(i_reagent)%shortid = t_compo_work%shortid
    tcr_reagents(i_reagent)%phasid  = t_compo_work%phasid
    tcr_reagents(i_reagent)%stoech  = TRIM(ADJUSTL(c_stoechcoeff))
    c_rid(i_reagent)                = TRIM(ADJUSTL(c_id))
    IF(PRESENT(c_xwildcards)) &
      c_xwildcards(i_reagent)       = TRIM(ADJUSTL(c_wildcard))

  ENDDO

ELSE

  WRITE(jp_stdout,c_fmtinf)
  WRITE(*,'("<'//cp_cherea//'> has no <'//c_reagent//'> child element.")')

  n_reagents = 0
  NULLIFY(tcr_reagents)
  NULLIFY(c_rid)
  NULLIFY(c_xwildcards)

ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE IDC_getChemReagentChain
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
 FUNCTION IDC_registerNewProcess                                       &
   (t_process_root, c_namesgen, c_namessid)                            &
   RESULT(t_process_activenode)
!-----------------------------------------------------------------------

USE MOD_CONFIGURE, ONLY: n_medusaproc

IMPLICIT NONE
TYPE(PROCESS), POINTER       :: t_process_root
CHARACTER(LEN=*), INTENT(IN) :: c_namesgen
CHARACTER(LEN=*), INTENT(IN) :: c_namessid

TYPE(PROCESS), POINTER       :: t_process_activenode

TYPE(PROCESS), POINTER       :: t_process_work

NULLIFY(t_process_activenode)

t_process_work => PROCESS_lastNode(t_process_root)

IF(t_process_work%idx == -1) THEN
                                    ! t_compo_work points to the not yet
  t_process_work%idx = 1            ! initialised t_compo_root element.
ELSE
  CALL PROCESS_createNext(t_process_work)
  t_process_work    => t_process_work%next
  t_process_work%idx = t_process_work%prev%idx + 1
ENDIF

t_process_work%name     = ADJUSTL(c_namesgen)
t_process_work%shortid  = ADJUSTL(c_namessid)

t_process_activenode => t_process_work

NULLIFY(t_process_work)

n_medusaproc = n_medusaproc + 1

RETURN

!-----------------------------------------------------------------------
 END FUNCTION IDC_registerNewProcess
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE IDC_configureProcess(stkmx_process, t_compo_root, t_process)
!-----------------------------------------------------------------------

USE MOD_CONFIGURE,                  ONLY: n_medusaproc_difblay, &
                                          n_medusaproc_reaclay, &
                                          n_medusaproc_tranlay, &
                                          n_medusaproc_corelay

IMPLICIT NONE


! Argument list variables
! -----------------------

TYPE(stack_minixml), POINTER  :: stkmx_process
TYPE(COMPOINFO), POINTER      :: t_compo_root
TYPE(PROCESS), POINTER        :: t_process


! Local variables
! ---------------

TYPE(stack_minixml), POINTER  :: stkmx_work, stkmx_ratelaw
CHARACTER(LEN=n_lmaxexpress)  :: c_subrname
CHARACTER(LEN=n_lmaxexpress)  :: c_xrefname
CHARACTER(LEN=n_lmaxshortid)  :: c_refid
CHARACTER(LEN=n_lmaxexpress)  :: c_pcdata
TYPE(stkrc_ptr), &
  DIMENSION(:), POINTER       :: stkrcp_work


CHARACTER(LEN=p_maxlen_attcntt) &
                              :: c_att_realms
CHARACTER(LEN=p_maxlen_attcntt) &
                              :: c_realm
INTEGER                       :: n_realms, i_realm, ii_realm
INTEGER                       :: nuniq_realms, iuniq_realm
INTEGER                       :: nduplic


CHARACTER(LEN=p_maxlen_attcntt) &
                              :: c_att_master
TYPE(PROCESS), POINTER        :: t_process_master, t_process_xref
LOGICAL                       :: l_xref


CHARACTER(LEN=n_lmaxfilename) :: c_modlibname
INTEGER, PARAMETER            :: jp_moduni = (CFG_TMPUNIT)

TYPE(RATELAW), POINTER        :: t_ratelaw_work
CHARACTER(LEN=n_lmaxidentif)  :: c_name
CHARACTER(LEN=n_lmaxidentif)  :: c_pp_type
INTEGER                       :: n_param
CHARACTER(LEN=n_lmaxexpress)  :: c_expression
NAMELIST /RATELAW_CONFIG/ c_name, c_pp_type, n_param, c_expression

CHARACTER(LEN=n_lmaxidentif)  :: c_paramname
CHARACTER(LEN=n_lmaxexpress)  :: c_paramcode
CHARACTER(LEN=n_lmaxidentif)  :: c_typecomponame
CHARACTER(LEN=n_lmaxidentif)  :: c_xmltagname
CHARACTER(LEN=p_maxlen_attcntt) :: c_xmlattstocheck
CHARACTER(LEN=n_lmaxphasid)   :: c_kindofparam
CHARACTER(LEN=n_lmaxshortid)  :: c_dummylabel
NAMELIST /RATELAW_DATA/ c_paramname, c_paramcode, c_typecomponame, c_xmltagname, &
                        c_xmlattstocheck, c_kindofparam, c_dummylabel

CHARACTER(LEN=n_lmaxcodeline) :: c_oneline
TYPE(RATELAW), POINTER        :: t_ratelaw

TYPE(COMPOINFO), POINTER      :: t_compo_work
CHARACTER(LEN=n_lmaxprocname) :: c_processname

INTEGER                       :: i_param, ixref_param, i, is, ie
INTEGER                       :: n_attstocheck, i_att
CHARACTER(LEN=p_maxlen_attname) :: c_att
CHARACTER(LEN=p_maxlen_attcntt) :: c_attcntt
CHARACTER(LEN=n_lmaxidentif)  :: c_prefix

LOGICAL                       :: l_codefound
TYPE(CODEBITS), POINTER       :: t_codes_work

INTEGER, DIMENSION(:), POINTER  :: iptr_begin, nptr_len
TYPE(stack_recchunks), &
  POINTER                     :: stkrc_attcntt

INTEGER                       :: i_ref

TYPE(stack_minixml), POINTER                        :: stkmx_cherea
TYPE(CHEMREAGENT), DIMENSION(:), POINTER            :: tcr_reagents
CHARACTER(LEN=n_lmaxshortid), DIMENSION(:), POINTER :: c_xid
CHARACTER(LEN=n_lmaxshortid), DIMENSION(:), POINTER :: c_xwildcards
INTEGER :: n_reactants, n_products

CHARACTER(LEN=*), PARAMETER  :: c_fmtinf_a = '("[[IDC_configureProcess]]: ", A)'
CHARACTER(LEN=*), PARAMETER  :: c_fmtwar_a = '("[[IDC_configureProcess]] warning: ", A)'
CHARACTER(LEN=*), PARAMETER  :: c_fmterr_a = '("[[IDC_configureProcess]] error: ", A)'

#ifdef CFG_DEBUG
CHARACTER(LEN=*), PARAMETER  :: c_fmtdbg_a = '("DEBUG [[IDC_configureProcess]]: ", A)'
#endif


l_xref = .FALSE.


! Complete the initial configuration
t_process%subr    = ADJUSTL(UPCASE(t_process%name))


! Operations
! ----------

n_realms = -1
                                    ! Check for 'realms' attribute
c_att_realms = cp_realmreaclay

i_att = STKMX_getAttIdxByName(stkmx_process, cp_realms)

IF (i_att > 0) THEN

  stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_process, i_att)
  CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_att_realms)
  WRITE(jp_stdout, c_fmtinf_a, ADVANCE='NO') ''
  WRITE(jp_stdout,'("attribute ", A, "=""", A,"""")') cp_realms, TRIM(c_att_realms)

ELSE
                                    ! No 'realms' attribute?
                                    ! Then check for 'master' attribute
  i_att = STKMX_getAttIdxByName(stkmx_process, cp_master)

  IF (i_att > 0) THEN
    stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_process, i_att)
    CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_att_master)
    WRITE(jp_stdout, c_fmtinf_a, ADVANCE='NO') ''
    WRITE(jp_stdout,'("attribute ", A, "=""", A,"""")') cp_master, TRIM(c_att_master)

                                    ! Scan previous entries of t_process
                                    ! to search for the process pointed
                                    ! to by the 'master' attribute.
    t_process_master => t_process%prev

    DO WHILE (ASSOCIATED(t_process_master))

      IF (t_process_master%name == c_att_master) THEN
        n_realms = t_process_master%n_realms
        t_process%n_realms = n_realms
        IF (n_realms > 0) THEN
          ALLOCATE(t_process%c_realm_identifiers(n_realms))
          t_process%c_realm_identifiers(:) = t_process_master%c_realm_identifiers(:)
        ENDIF
        EXIT
      ENDIF

      t_process_master => t_process_master%prev

    ENDDO

    IF (n_realms < 0) THEN          ! If the process indicated by 'master'
                                    ! has not been found: abort.
      WRITE(jp_stderr, c_fmterr_a) &
        'attribute "' // cp_master // '" of <' // cp_process // '> ' // &
        'points to an (as yet) unknown process "' // TRIM(c_att_master) // '".'
      WRITE(jp_stderr, '(A)') &
        'Please make sure that process "' // TRIM(c_att_master) // '" preceeds ' // &
        'process "' // TRIM(t_process%name) // '" in the process list.'
      WRITE(jp_stderr, '(A)') &
        'Aborting!'
      CALL ABORT()
    ENDIF

  ELSE

    WRITE(jp_stdout, c_fmtwar_a) 'missing attribute ' // cp_realms // ' -- ' &
                   // ' using default value "' // cp_realmreaclay //'"'
    c_att_realms = cp_realmreaclay

  ENDIF

ENDIF

IF (n_realms < 0) THEN              ! Realms only need to be
                                    ! initialised if n_realms is still < 0.
  CALL DELIMIT_STRING_TOKENS(c_att_realms, cp_delimiters, iptr_begin, nptr_len)

  IF (ASSOCIATED(iptr_begin)) THEN
    n_realms = SIZE(iptr_begin) ! should always be >= 1 since default is "reaclay"
  ELSE
    n_realms = 0
  ENDIF


! Need to check for duplicates here
! Set nptr_len(:) of duplicates to 0 (except first occurrence)

  nduplic = 0

  DO i_realm = 1, n_realms - 1

    IF (nptr_len(i_realm) == 0) CYCLE

    is = iptr_begin(i_realm)
    ie = is +  nptr_len(i_realm) - 1
    c_realm = c_att_realms(is:ie)

    DO ii_realm = i_realm + 1, n_realms

      IF (nptr_len(ii_realm) == 0) CYCLE

      is = iptr_begin(ii_realm)
      ie = is +  nptr_len(i_realm) - 1

      IF (c_att_realms(is:ie) == c_realm) THEN

        nptr_len(ii_realm) = 0
        nduplic = nduplic + 1

        WRITE(jp_stdout, c_fmtwar_a, ADVANCE='NO') &
          'duplicate realm name "' //  TRIM(c_realm) // '" at positions '
        WRITE(jp_stdout, '(I0, " and ", I0)', ADVANCE='NO') i_realm, ii_realm
        WRITE(jp_stdout,' (A)') ' in attribute "' // &
          cp_realms // '" of <' // TRIM(cp_process) // '> -- Ignoring second occurrence.'

      ENDIF

    ENDDO

  ENDDO

  nuniq_realms = n_realms - nduplic


  t_process%n_realms = nuniq_realms

  IF (nuniq_realms > 0) THEN

    ALLOCATE(t_process%c_realm_identifiers(nuniq_realms))

    iuniq_realm = 0

    DO i_realm = 1, n_realms

      IF (nptr_len(i_realm) == 0) CYCLE

      iuniq_realm = iuniq_realm + 1

      is = iptr_begin(i_realm)
      ie = is +  nptr_len(i_realm) - 1
      c_realm = c_att_realms(is:ie)

      SELECT CASE(c_realm)
      CASE(cp_realmdifblay)
        t_process%c_realm_identifiers(iuniq_realm) = 'jp_realm_difblay'
        n_medusaproc_difblay = n_medusaproc_difblay + 1
      CASE(cp_realmreaclay)
        t_process%c_realm_identifiers(iuniq_realm) = 'jp_realm_reaclay'
        n_medusaproc_reaclay = n_medusaproc_reaclay + 1
      CASE(cp_realmtranlay)
        t_process%c_realm_identifiers(iuniq_realm) = 'jp_realm_tranlay'
        n_medusaproc_tranlay = n_medusaproc_tranlay + 1
      CASE(cp_realmcorelay)
        t_process%c_realm_identifiers(iuniq_realm) = 'jp_realm_corelay'
        n_medusaproc_corelay = n_medusaproc_corelay + 1
      CASE DEFAULT
        WRITE(jp_stderr, c_fmterr_a) &
        'unknown realm name "' //  TRIM(c_realm) // '" in attribute "' // &
        cp_realms // '" of <' // TRIM(cp_process) // '> -- aborting'
        CALL ABORT()
      END SELECT

    ENDDO

  ENDIF

ENDIF

                                    ! Chemical Reaction Tag
stkmx_cherea  => STKMX_getUniqueChildEltByName(stkmx_process, cp_cherea)

CALL IDC_getChemReagentChain(stkmx_cherea, cp_chereareact, &
  t_compo_root, n_reactants, tcr_reagents, c_xid, c_xwildcards)

t_process%n_reactants   = n_reactants
t_process%cr_reacts    => tcr_reagents
t_process%c_rid        => c_xid
t_process%c_rwildcards => c_xwildcards

CALL IDC_getChemReagentChain(stkmx_cherea, cp_chereaprod, &
  t_compo_root, n_products, tcr_reagents, c_xid, c_xwildcards)

t_process%n_products    = n_products
t_process%cr_prods     => tcr_reagents
t_process%c_pid        => c_xid
t_process%c_pwildcards => c_xwildcards


! Safety checks:
! - in the merged list t_process%cr_reacts // t_process%cr_prods,
!   there must not be any duplicates;
! - in the merged list t_process%c_rid // t_process%c_pid,
!   there must not be any duplicates (except for empty strings)
! - in the merged list t_process%c_rwildcards // t_process%c_pwildcards,
!   there must not be any duplicates (except for empty strings)

CALL IDC_resolveWildcardsProcess(t_process)


stkmx_ratelaw  => STKMX_getUniqueChildEltByName(stkmx_process, cp_ratelaw)


                                    ! Does <RateLaw> have an 'xref' attribute?
i_att = STKMX_getAttIdxByName(stkmx_ratelaw, cp_ratelawxref)

IF (i_att > 0) THEN
                                    ! Yes, it has! Get its content.
  stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_ratelaw, i_att)
  CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_xrefname, nlen_returned)

                                    ! Scan previous entries of t_process
                                    ! to search for the process pointed
                                    ! to by the 'xref' attribute.
  t_process_xref => t_process%prev

  DO WHILE (ASSOCIATED(t_process_xref))

    IF (t_process_xref%name == c_xrefname) THEN
      l_xref = .TRUE.
      t_process%idx_xref = t_process_xref%idx
      EXIT
    ENDIF

    t_process_xref => t_process_xref%prev

  ENDDO

  IF (.NOT. l_xref) THEN
                                    ! The process indicated by 'xref'
                                    ! has not been found: abort.
    WRITE(jp_stderr, c_fmterr_a) &
      'Attribute "' // cp_ratelawxref // '" of <' // TRIM(cp_ratelaw) // '> ' // &
      'points to an (as yet) unknown process "' // TRIM(c_xrefname) // '".'
    WRITE(jp_stderr, '(A)') &
      'Please make sure that process "' // TRIM(c_xrefname) // '" preceeds ' // &
      'process "' // TRIM(t_process%name) // '" in the process list.'
    WRITE(jp_stderr, '(A)') &
      'Aborting!'
    CALL ABORT()
  ENDIF

ENDIF


                                    ! Does <RateLaw> have a 'subr' attribute ?
i_att = STKMX_getAttIdxByName(stkmx_ratelaw, cp_ratelawsubr)

IF (i_att > 0) THEN
                                    ! Yes, there is one -- retrieve its content.
  stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_ratelaw, i_att)
  CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_subrname, nlen_returned)

ELSE
                                    ! No 'subr' attribute?
  IF (l_xref) THEN
                                    ! If we have found an 'xref' attribute
                                    ! above, we use its ratelaw%name
                                    ! as the rate law subroutine reference.
                                    ! (If l_xref is.TRUE., t_process_xref
                                    ! still points to the xref'd one).
    c_subrname = TRIM(t_process_xref%ratelaw%name)

  ELSE
                                    ! If we have not found any 'xref'
                                    ! attribute, we have to abort.
    WRITE(jp_stderr, c_fmterr_a) &
      'Neither "' // TRIM(cp_ratelawsubr) // '" nor "' // TRIM(cp_ratelawxref) // '" &
      &attribute found in <' // TRIM(cp_ratelaw) // '> -- aborting!'
    CALL ABORT()

  ENDIF

ENDIF
                                    ! Does <RateLaw> have a
                                    ! 'reference_id' attribute?
i_att = STKMX_getAttIdxByName(stkmx_ratelaw, cp_ratelawrefid)

IF (i_att > 0) THEN

  stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_ratelaw, i_att)
  CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_refid, nlen_returned)

ELSE

  WRITE(jp_stderr, c_fmterr_a) &
    'Mandatory "' //  TRIM(cp_ratelawrefid) // '" attribute &
    &is missing in <' // TRIM(cp_ratelaw) // '> -- aborting'
  CALL ABORT()

ENDIF
                                    ! Derive name of the modlib file
                                    ! which contains the subroutine
                                    ! that provides the current ratelaw.
c_modlibname = cp_dir_modlib  // '/' // &
               cp_prefix_modlib // TRIM(ADJUSTL(LOWCASE(c_subrname))) // '.F'
#ifdef CFG_DEBUG
WRITE(jp_stddbg,c_fmtdbg_a) 'Opening modlib file "' // TRIM(c_modlibname) // '"'
#endif
OPEN(UNIT=jp_moduni, FILE=c_modlibname, STATUS="OLD")

c_oneline = ' '
DO WHILE(c_oneline(1:1) /= '#')
  READ(jp_moduni, '(A)') c_oneline
ENDDO

n_param = 0
READ(jp_moduni, NML=RATELAW_CONFIG)

t_ratelaw => RATELAW_createRoot()
CALL RATELAW_init(t_ratelaw, n_param)

t_ratelaw%name       = TRIM(ADJUSTL(c_name))
t_ratelaw%pp_type    = TRIM(ADJUSTL(c_pp_type))
t_ratelaw%expression = c_expression
IF (l_xref) t_ratelaw%xrefprocname = ADJUSTL(c_xrefname)

IF (n_param /= 0) THEN


  DO i_param = 1, n_param

    c_paramname      = '???'
    c_paramcode      = '???'
    c_xmltagname     = '???'
    c_xmlattstocheck = ''
    c_kindofparam    = ''
    c_dummylabel     = ''
    READ(jp_moduni, NML=RATELAW_DATA)

    IF (c_xmltagname == '???') c_xmltagname = c_typecomponame
    t_ratelaw%paramname(i_param)      = ADJUSTL(c_paramname)
    t_ratelaw%paramcode(i_param)      = ADJUSTL(c_paramcode)
    t_ratelaw%typecomponame(i_param)  = ADJUSTL(c_typecomponame)
    t_ratelaw%xmltagname(i_param)     = ADJUSTL(c_xmltagname)
    t_ratelaw%xmlattstocheck(i_param) = ADJUSTL(c_xmlattstocheck)
    t_ratelaw%kindofparam(i_param)    = ADJUSTL(c_kindofparam)
    t_ratelaw%dummylabel(i_param)     = ADJUSTL(c_dummylabel)

    t_ratelaw%iparam_xref(i_param)    = 0

  ENDDO

  CLOSE(UNIT=jp_moduni)


  DO i_param = 1, n_param

                                    ! tag name to search for
    c_xmltagname = t_ratelaw%xmltagname(i_param)
                                    ! Get the element tag by the name
    stkmx_work => STKMX_getUniqueChildEltByName(stkmx_ratelaw, c_xmltagname)
                                    ! If there is no such element, ...
    IF (.NOT.ASSOCIATED(stkmx_work)) THEN
                                    ! ... issue a warning and proceed to
                                    !     the next entry,
      IF (l_xref) THEN
        WRITE(jp_stdout, c_fmtwar_a) &
          'missing <' // TRIM(c_xmltagname) // '> &
          &-- using value from process "' // TRIM(c_xrefname) // '"'
                                    ! First flag the use xref'd values
                                    ! by the idx of the xref'd process
        DO ixref_param = 1, t_process_xref%ratelaw%n_params
          IF (t_process_xref%ratelaw%xmltagname(ixref_param) == c_xmltagname) THEN
            t_ratelaw%iparam_xref(i_param) = ixref_param
            IF (t_process_xref%ratelaw%kindofparam(ixref_param) == t_ratelaw%kindofparam(i_param)) THEN
              t_ratelaw%paramname(i_param) = t_process_xref%ratelaw%paramname(ixref_param)
              t_ratelaw%paramcode(i_param) = t_process_xref%ratelaw%paramcode(ixref_param)
              ! t_ratelaw%typecomponame(i_param) may be different
              ! t_ratelaw%xmltagname(i_param) is already the same
              ! t_ratelaw%xmlattstocheck(i_param) is irrelevant
              ! t_ratelaw%kindofparam(i_param) must be the same
              ! t_ratelaw%dummylabel(i_param) may be different - leave as is
              EXIT
            ELSE
              WRITE(jp_stderr, c_fmterr_a) &
                '%kindofparam mismatch for <' // TRIM(c_xmltagname) // '>:'
              WRITE(jp_stderr, '(" - expected """, A, """")') TRIM(t_ratelaw%kindofparam(i_param))
              WRITE(jp_stderr, '(" - found """, A, """ in process """, A, """")') &
                TRIM(t_process_xref%ratelaw%kindofparam(ixref_param)), &
                TRIM(c_xrefname)
              WRITE(jp_stderr, '(A)') 'Aborting!'
              CALL ABORT()
            ENDIF
          ENDIF
        ENDDO
        IF (t_ratelaw%iparam_xref(i_param) == 0) THEN
          WRITE(jp_stdout, c_fmtwar_a) &
            'missing <' // TRIM(c_xmltagname) // '> in process " ' // &
            TRIM(c_xrefname) // '" -- using default value "' // TRIM(c_paramname) // '"'

        ENDIF
      ELSE
        WRITE(jp_stdout, c_fmtwar_a) &
          'missing <' // TRIM(c_xmltagname) // '> &
          &-- using default value "' // TRIM(c_paramname) // '"'
      ENDIF
                                    ! and continue
      CYCLE

    ENDIF
                                    ! ... else, get its content and copy it into c_pcdata
    stkrcp_work => STKMX_getPCDatacntt(stkmx_work)
    IF(ASSOCIATED(stkrcp_work)) THEN
      CALL STKRC_copyStkrcToStr(stkrcp_work(1)%ptr, c_pcdata)
    ELSE
      c_pcdata = ''
    ENDIF

                                    ! attribute names to check for
    c_xmlattstocheck = t_ratelaw%xmlattstocheck(i_param)
    IF(LEN_TRIM(c_xmlattstocheck) > 0) THEN
                                    ! If there are attributes to check for,
                                    ! retrieve them

      CALL DELIMIT_STRING_TOKENS(c_xmlattstocheck, cp_delimiters, iptr_begin, nptr_len)

      IF(ASSOCIATED(iptr_begin)) THEN
        n_attstocheck = SIZE(iptr_begin)
        check_atts: DO i = 1, n_attstocheck
          is = iptr_begin(i)
          ie = iptr_begin(i)+nptr_len(i)-1
          c_att = c_xmlattstocheck(is:ie)
          i_att = STKMX_getAttIdxByName(stkmx_work, c_att)
          IF (i_att > 0) THEN
            stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_work, i_att)
            CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_attcntt, nlen_returned)
            SELECT CASE(c_att)
            CASE('type')
              SELECT CASE(c_attcntt)
              CASE('globalconstant')
                WRITE(jp_stdout, c_fmtinf_a) 'type="globalconstant" -- processing deferred'
                t_ratelaw%paramname(i_param) = '--global constant--'
                CYCLE check_atts
              END SELECT
            CASE('code')
              SELECT CASE(c_attcntt)
              CASE('verbatim')
                t_ratelaw%paramname(i_param) = '--verbatim code--'
                t_ratelaw%paramcode(i_param) = ADJUSTL(c_pcdata)
                CYCLE check_atts
              CASE(cp_soluprod) ! 'SolubilityProduct'
                c_prefix = cp_prefix_cctksp
              CASE(cp_satuconc) ! 'SaturationConc'
                c_prefix = cp_prefix_cctksat
              CASE(cp_degrsatu) ! 'DegreeSaturation'
                c_prefix = cp_prefix_cctdsat
              CASE(cp_totconc)  ! 'TotalConcentration'
                c_prefix = cp_prefix_cctttcc
              CASE DEFAULT
                c_prefix = '???'
              END SELECT

              t_compo_work => COMPOINFO_getNodeByName(t_compo_root, c_pcdata)

              IF(ASSOCIATED(t_compo_work)) THEN

                l_codefound = .FALSE.
                t_codes_work => t_compo_work%codes
                scan_codes: DO WHILE(ASSOCIATED(t_codes_work))
                  IF(t_codes_work%type == c_attcntt) THEN
                    t_ratelaw%paramname(i_param) = TRIM(ADJUSTL(c_pcdata))
                    t_ratelaw%paramcode(i_param) = TRIM(c_prefix)//t_compo_work%shortid
                    l_codefound = .TRUE.
                    EXIT scan_codes
                  ENDIF
                  t_codes_work => t_codes_work%next
                ENDDO scan_codes

                IF(.NOT. l_codefound) THEN
                  WRITE(jp_stderr, c_fmterr_a) &
                    '<' // TRIM(c_xmltagname) // '> refers to &
                    &component "' // TRIM(c_pcdata) // '" &
                    &which does not have the requested "' // TRIM(c_attcntt) // '" code -- aborting!'
                  CALL ABORT()
                ENDIF
              ENDIF
            END SELECT
          ELSE
            WRITE(jp_stderr, c_fmterr_a) &
              '<' // TRIM(c_xmltagname) // '> &
              &does not include the required "' // TRIM(c_att) // '" attribute&
              & -- aborting!'
            CALL ABORT()
          ENDIF
        ENDDO check_atts
      ENDIF

    ELSE

      t_compo_work => COMPOINFO_getNodeByName(t_compo_root, c_pcdata)

      IF(ASSOCIATED(t_compo_work)) THEN
        SELECT CASE(t_compo_work%phasid)
        CASE('ic','if')
          t_ratelaw%paramname(i_param) = TRIM(c_pcdata)
          t_ratelaw%paramcode(i_param) = cp_prefix_io // ADJUSTL(t_compo_work%shortid)
        CASE DEFAULT
          WRITE(jp_stderr, c_fmterr_a) &
            '<' // TRIM(c_xmltagname) // '> &
            &refers to component "'// TRIM(c_pcdata) // '" &
            &which has invalid phasid "' // TRIM(t_compo_work%phasid) // '" -- aborting'

          CALL ABORT()
        END SELECT
      ELSE
        WRITE(jp_stderr, c_fmterr_a) &
            '<' // TRIM(c_xmltagname) // '> &
            &refers to unknown component "'// TRIM(c_pcdata) // '" -- aborting!'
        CALL ABORT()
      ENDIF

    ENDIF

  END DO

  t_process%ratelaw => t_ratelaw

  NULLIFY(t_ratelaw)

ENDIF

! Search for the reference_id
i_ref = 0                           ! Preset to 0 (not initialised)
DO i = 1, t_process%n_reactants
  IF (t_process%c_rid(i) /= c_refid) CYCLE
  i_ref = -i                        ! set to negative if reference == reactant
  EXIT
ENDDO

IF (i_ref == 0) THEN
  DO i = 1, t_process%n_products
    IF (t_process%c_pid(i) /= c_refid) CYCLE
    i_ref = i                       ! set to positive if reference == product
    EXIT
  ENDDO
ENDIF

IF (i_ref /= 0) THEN
  t_process%i_ref = i_ref
ELSE
  WRITE(jp_stderr, c_fmterr_a) &
    'unknown reference_id="' // TRIM(c_refid) // '" -- aborting!'
  CALL ABORT()
ENDIF

t_process%pp_identifier  = 'pp_'  // t_process%name
t_process%nvp_identifier = 'nvp_' // t_process%name
t_process%iop_identifier = 'iop_' // t_process%name
t_process%nvr_identifier = 'nvr_' // t_process%name
t_process%ior_identifier = 'ior_' // t_process%name
t_process%pi_identifier  = 'pi_'  // t_process%name


RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE IDC_configureProcess
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE IDC_resolveWildcardsProcess(t_process)
!-----------------------------------------------------------------------

IMPLICIT NONE


! Argument list variables
! -----------------------

TYPE(PROCESS), POINTER        :: t_process


! Local variables
! ---------------

CHARACTER(LEN=n_lmaxexpress)  :: c_stoech, c_stoechxp
CHARACTER(LEN=n_lmaxshortid)  :: c_wildcard, c_substitute
INTEGER                       :: i, j
INTEGER                       :: nt, ns


DO i = 1, t_process%n_reactants
  c_wildcard = t_process%c_rwildcards(i)
  !WRITE(*,*) '==== c_wildcard <' // TRIM(c_wildcard) // '> ===='
  IF (LEN_TRIM(c_wildcard) == 0) CYCLE
  c_substitute = t_process%cr_reacts(i)%shortid
  !WRITE(*,*) '==== c_substitute <' // TRIM(c_substitute) // '> ===='
  DO j = 1, t_process%n_reactants
    c_stoech = t_process%cr_reacts(j)%stoech
    !WRITE(*,*) '==== c_stoech <' // TRIM(c_stoech) // '> ===='
    CALL EXPAND_TOKEN(c_stoech, c_wildcard, c_substitute, c_stoechxp)
    !WRITE(*,*) '==== c_stoechxp <' // TRIM(c_stoechxp) // '> ===='
    t_process%cr_reacts(j)%stoech = c_stoechxp
  ENDDO
  DO j = 1, t_process%n_products
    c_stoech = t_process%cr_prods(j)%stoech
    CALL EXPAND_TOKEN(c_stoech, c_wildcard, c_substitute, c_stoechxp)
    t_process%cr_prods(j)%stoech = c_stoechxp
  ENDDO
ENDDO


DO i = 1, t_process%n_products
  c_wildcard = t_process%c_pwildcards(i)
  IF (LEN_TRIM(c_wildcard) == 0) CYCLE
  c_substitute = t_process%cr_prods(i)%shortid
  DO j = 1, t_process%n_reactants
    c_stoech = t_process%cr_reacts(j)%stoech
    CALL EXPAND_TOKEN(c_stoech, c_wildcard, c_substitute, c_stoechxp)
    t_process%cr_reacts(j)%stoech = c_stoechxp
  ENDDO
  DO j = 1, t_process%n_products
    c_stoech = t_process%cr_prods(j)%stoech
    CALL EXPAND_TOKEN(c_stoech, c_wildcard, c_substitute, c_stoechxp)
    t_process%cr_prods(j)%stoech = c_stoechxp
  ENDDO
ENDDO



RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE IDC_resolveWildcardsProcess
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
 FUNCTION IDC_registerNewEquilibrium                                   &
   (t_equilib_root, c_namesgen, c_namessid)                            &
   RESULT(t_equilib_activenode)
!-----------------------------------------------------------------------

USE MOD_CONFIGURE, ONLY: n_medusaeqlb

IMPLICIT NONE
TYPE(EQUILIB), POINTER       :: t_equilib_root
CHARACTER(LEN=*), INTENT(IN) :: c_namesgen
CHARACTER(LEN=*), INTENT(IN) :: c_namessid

TYPE(EQUILIB), POINTER       :: t_equilib_activenode

TYPE(EQUILIB), POINTER       :: t_equilib_work

NULLIFY(t_equilib_activenode)

t_equilib_work => EQUILIB_lastNode(t_equilib_root)

IF (t_equilib_work%idx == -1) THEN
                                    ! t_compo_work points to the not yet
  t_equilib_work%idx = 1              ! initialised t_compo_root element.
ELSE
  CALL EQUILIB_createNext(t_equilib_work)
  t_equilib_work    => t_equilib_work%next
  t_equilib_work%idx = t_equilib_work%prev%idx + 1
ENDIF

t_equilib_work%name     = ADJUSTL(c_namesgen)
t_equilib_work%shortid  = ADJUSTL(c_namessid)

t_equilib_activenode => t_equilib_work

NULLIFY(t_equilib_work)

n_medusaeqlb = n_medusaeqlb + 1

RETURN

!-----------------------------------------------------------------------
 END FUNCTION IDC_registerNewEquilibrium
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE IDC_configureEquilibrium(stkmx_equilib, t_compo_root, t_equilib)
!-----------------------------------------------------------------------

IMPLICIT NONE


! Argument list variables
! -----------------------

TYPE(stack_minixml), POINTER  :: stkmx_equilib
TYPE(COMPOINFO), POINTER      :: t_compo_root
TYPE(EQUILIB), POINTER        :: t_equilib


! Local variables
! ---------------

TYPE(stack_minixml), POINTER  :: stkmx_work, stkmx_lawmassa
CHARACTER(LEN=n_lmaxexpress)  :: c_subrname
CHARACTER(LEN=n_lmaxshortid)  :: c_refid
CHARACTER(LEN=n_lmaxexpress)  :: c_pcdata
TYPE(stkrc_ptr), &
  DIMENSION(:), POINTER       :: stkrcp_work

CHARACTER(LEN=n_lmaxfilename) :: c_modlibname
INTEGER, PARAMETER            :: jp_moduni = (CFG_TMPUNIT)

TYPE(EQLBREL), POINTER        :: t_eqlbrel_work
CHARACTER(LEN=n_lmaxidentif)  :: c_name
CHARACTER(LEN=n_lmaxidentif)  :: c_ep_type
INTEGER                       :: n_param
CHARACTER(LEN=n_lmaxexpress)  :: c_expression
NAMELIST /EQLBREL_CONFIG/ c_name, c_ep_type, n_param, c_expression

CHARACTER(LEN=n_lmaxidentif)  :: c_paramname
CHARACTER(LEN=n_lmaxexpress)  :: c_paramcode
CHARACTER(LEN=n_lmaxidentif)  :: c_typecomponame
CHARACTER(LEN=n_lmaxidentif)  :: c_xmltagname
CHARACTER(LEN=n_lmaxphasid)   :: c_kindofparam
CHARACTER(LEN=n_lmaxshortid)  :: c_dummylabel
NAMELIST /EQLBREL_DATA/ c_paramname, c_paramcode, c_typecomponame, c_xmltagname, &
                        c_kindofparam, c_dummylabel

CHARACTER(LEN=n_lmaxcodeline) :: c_oneline
TYPE(EQLBREL), POINTER        :: t_eqlbrel

TYPE(COMPOINFO), POINTER      :: t_compo_work
CHARACTER(LEN=n_lmaxprocname) :: c_equilibname

INTEGER                       :: i_param, i, is, ie
INTEGER                       :: n_attstocheck, i_att
CHARACTER(LEN=p_maxlen_attname) :: c_att
CHARACTER(LEN=p_maxlen_attcntt) :: c_attcntt
CHARACTER(LEN=n_lmaxidentif)  :: c_prefix

LOGICAL                       :: l_codefound
TYPE(CODEBITS), POINTER       :: t_codes_work


INTEGER, DIMENSION(:), POINTER   :: iptr_begin, nptr_len
TYPE(stack_recchunks), &
  POINTER                        :: stkrc_attcntt

TYPE(stack_minixml), POINTER                        :: stkmx_cherea
TYPE(CHEMREAGENT), DIMENSION(:), POINTER            :: tcr_reagents
CHARACTER(LEN=n_lmaxshortid), DIMENSION(:), POINTER :: c_xid
INTEGER :: n_reactants, n_products

CHARACTER(LEN=*), PARAMETER  :: c_fmterr_a = '("[[IDC_configureEquilibrium]] error: ", A)'
CHARACTER(LEN=*), PARAMETER  :: c_fmtwar_a = '("[[IDC_configureEquilibrium]] warning: ", A)'

#ifdef CFG_DEBUG
CHARACTER(LEN=*), PARAMETER  :: c_fmtdbg_a = '("DEBUG [[IDC_configureEquilibrium]]: ", A)'
#endif


t_equilib%subr    = ADJUSTL(UPCASE(t_equilib%name))



stkmx_cherea  => STKMX_getUniqueChildEltByName(stkmx_rootelt, cp_cherea)

CALL IDC_getChemReagentChain(stkmx_cherea, cp_chereareact, &
  t_compo_root, n_reactants, tcr_reagents, c_xid)

t_equilib%n_reactants = n_reactants
t_equilib%cr_reacts  => tcr_reagents
t_equilib%c_rid      => c_xid

CALL IDC_getChemReagentChain(stkmx_cherea, cp_chereaprod, &
  t_compo_root, n_products, tcr_reagents, c_xid)

t_equilib%n_products = n_products
t_equilib%cr_prods  => tcr_reagents
t_equilib%c_pid     => c_xid


stkmx_lawmassa  => STKMX_getUniqueChildEltByName(stkmx_equilib, cp_lawmassa)

! Get "subr" attribute of <LawOfMassAction>
i_att = STKMX_getAttIdxByName(stkmx_lawmassa, cp_lawmassasubr)
IF (i_att > 0) THEN
  stkrc_attcntt => STKMX_getAttcnttByIdx(stkmx_lawmassa, i_att)
  CALL STKRC_copyStkrcToStr(stkrc_attcntt, c_subrname, nlen_returned)
ELSE
  WRITE(jp_stderr, c_fmterr_a) &
    'missing "' // TRIM(cp_lawmassasubr) // '" attribute &
    &in <' // TRIM(cp_lawmassa) // '> -- aborting!'
  CALL ABORT()
ENDIF


                                    ! Derive name of the modlib file
                                    ! that contains the subroutine that
                                    ! provides the current equilibrium
                                    ! relationship.
c_modlibname = cp_dir_modlib // '/' // &
               cp_prefix_modlib // TRIM(ADJUSTL(LOWCASE(c_subrname))) // '.F'
#ifdef CFG_DEBUG
WRITE(jp_stddbg,c_fmtdbg_a) 'Opening modlib file "' // TRIM(c_modlibname) // '"'
#endif

OPEN(UNIT=jp_moduni, FILE=c_modlibname, STATUS="OLD")

c_oneline = ' '
DO WHILE(c_oneline(1:1) /= '#')
  READ(jp_moduni, '(A)') c_oneline
ENDDO

n_param = 0
READ(jp_moduni, NML=EQLBREL_CONFIG)

t_eqlbrel => EQLBREL_createRoot()
CALL EQLBREL_init(t_eqlbrel, n_param)

t_eqlbrel%name     = TRIM(ADJUSTL(c_name))
t_eqlbrel%ep_type  = TRIM(ADJUSTL(c_ep_type))
t_eqlbrel%expression = c_expression


IF (n_param /= 0) THEN

  DO i_param = 1, n_param

    c_paramname   = '???'
    c_paramcode      = '???'
    c_xmltagname  = '???'
    c_kindofparam = ''
    c_dummylabel  = ''
    READ(jp_moduni, NML=EQLBREL_DATA)

    IF (c_xmltagname == '???') c_xmltagname = c_typecomponame
    t_eqlbrel%paramname(i_param)      = ADJUSTL(c_paramname)
    t_eqlbrel%paramcode(i_param)      = ADJUSTL(c_paramcode)
    t_eqlbrel%typecomponame(i_param)  = ADJUSTL(c_typecomponame)
    t_eqlbrel%xmltagname(i_param)     = ADJUSTL(c_xmltagname)
    t_eqlbrel%kindofparam(i_param)    = ADJUSTL(c_kindofparam)
    t_eqlbrel%dummylabel(i_param)     = ADJUSTL(c_dummylabel)

  ENDDO

  CLOSE(UNIT=jp_moduni)


  DO i_param = 1, n_param
                                    ! tag name to search for
    c_xmltagname = t_eqlbrel%xmltagname(i_param)
                                    ! Get the element tag by the name
    stkmx_work => STKMX_getUniqueChildEltByName(stkmx_lawmassa, c_xmltagname)
                                    ! If there is no such element, ...
    IF (.NOT.ASSOCIATED(stkmx_work)) THEN
                                    ! ... issue a warning and proceed to
                                    !     the next entry,
      WRITE(jp_stdout, c_fmtwar_a) &
        'missing <' // TRIM(c_xmltagname) // '> &
        & -- using default value "' // TRIM(c_paramname) // '".'
                                    ! and continue
      CYCLE

    ENDIF
                                    ! ... else, get its content and copy it into c_pcdata
    stkrcp_work => STKMX_getPCDatacntt(stkmx_work)
    IF (ASSOCIATED(stkrcp_work)) THEN
      CALL STKRC_copyStkrcToStr(stkrcp_work(1)%ptr, c_pcdata)
                                    ! Then remove leading and trailing blanks,
                                    ! which are not significant.
      c_pcdata = ADJUSTL(TRIM(c_pcdata))
    ELSE
      c_pcdata = ''
    ENDIF

    ! TBD: We have to check whether the component names are all different

    IF (LEN_TRIM(c_pcdata) > 0) THEN

      t_compo_work => COMPOINFO_getNodeByName(t_compo_root, c_pcdata)

      IF (ASSOCIATED(t_compo_work)) THEN
        SELECT CASE(t_compo_work%phasid)
        CASE('ic','if')
          t_eqlbrel%paramname(i_param) = TRIM(c_pcdata)
          t_eqlbrel%paramcode(i_param) = cp_prefix_io // ADJUSTL(t_compo_work%shortid)
        CASE DEFAULT
          WRITE(jp_stderr, c_fmterr_a) &
            '<' // TRIM(c_xmltagname) // '> &
            &refers to "' //  TRIM(c_pcdata) // '" &
            &which has invalid phasid "' // TRIM(t_compo_work%phasid) // '" -- aborting!'
          CALL ABORT()
        END SELECT
      ELSE
        WRITE(jp_stderr, c_fmterr_a) &
          '<' // TRIM(c_xmltagname) // '> &
          &refers to unknown component "' // TRIM(c_pcdata) // '" -- aborting!'
        CALL ABORT()
      ENDIF

    ELSE

      WRITE(jp_stdout, c_fmtwar_a) &
        '<' // TRIM(c_xmltagname) // '> is empty &
        & -- leaving at default value and continuing with fingers crossed.'

    ENDIF

  END DO

  t_equilib%eqlbrel => t_eqlbrel

  NULLIFY(t_eqlbrel)

ENDIF

t_equilib%ep_identifier  = 'ep_'  // t_equilib%name
t_equilib%nve_identifier = 'nve_' // t_equilib%name
t_equilib%ioe_identifier = 'ioe_' // t_equilib%name
t_equilib%nvr_identifier = 'nvr_' // t_equilib%name
t_equilib%ior_identifier = 'ior_' // t_equilib%name
t_equilib%ei_identifier  = 'ei_'  // t_equilib%name


RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE IDC_configureEquilibrium
!-----------------------------------------------------------------------




!-----------------------------------------------------------------------
 FUNCTION IDC_registerNewAPIExtension                                  &
   (t_apiext_root, c_namesgen, c_namessid)                            &
   RESULT(t_apiext_activenode)
!-----------------------------------------------------------------------

USE MOD_CONFIGURE, ONLY: n_medusaapiext

IMPLICIT NONE
TYPE(APIEXTENSION), POINTER  :: t_apiext_root
CHARACTER(LEN=*), INTENT(IN) :: c_namesgen
CHARACTER(LEN=*), INTENT(IN) :: c_namessid

TYPE(APIEXTENSION), POINTER  :: t_apiext_activenode

TYPE(APIEXTENSION), POINTER  :: t_apiext_work

NULLIFY(t_apiext_activenode)

t_apiext_work => APIEXTENSION_lastNode(t_apiext_root)

IF (t_apiext_work%idx == -1) THEN
                                    ! t_apiext_work points to the not yet
  t_apiext_work%idx = 1             ! initialised t_apiext_root element.
ELSE
  CALL APIEXTENSION_createNext(t_apiext_work)
  t_apiext_work    => t_apiext_work%next
  t_apiext_work%idx = t_apiext_work%prev%idx + 1
ENDIF

t_apiext_work%name     = ADJUSTL(c_namesgen)
t_apiext_work%shortid  = ADJUSTL(c_namessid)

t_apiext_activenode => t_apiext_work

NULLIFY(t_apiext_work)

n_medusaapiext = n_medusaapiext + 1

RETURN

!-----------------------------------------------------------------------
 END FUNCTION IDC_registerNewAPIExtension
!-----------------------------------------------------------------------


!-----------------------------------------------------------------------
 SUBROUTINE IDC_attachAPIExtensionCodebits(stkmx_rootelt, t_apiext_curr)
!-----------------------------------------------------------------------

IMPLICIT NONE


! Argument list variables
! -----------------------

TYPE(stack_minixml), POINTER    :: stkmx_rootelt
TYPE(APIEXTENSION), POINTER     :: t_apiext_curr


! Local variables
! ---------------


TYPE(CODEBITS), POINTER         :: t_codebits_root
TYPE(CODEBITS), POINTER         :: t_code_wk


t_codebits_root => IDC_getCodeBits(stkmx_rootelt)

IF (ASSOCIATED(t_codebits_root)) THEN

  t_code_wk => t_codebits_root

  DO WHILE (ASSOCIATED(t_code_wk))

    SELECT CASE(t_code_wk%type)
    CASE(cp_totconc)
      t_code_wk%varname = cp_prefix_wdata // ADJUSTL(c_namessid)
    CASE DEFAULT
      t_code_wk%varname = '???'
      !TBD Add error message here: unknown type
    END SELECT

    t_code_wk => t_code_wk%next

  ENDDO

  NULLIFY(t_code_wk)                ! Advance to the end of the t_apiext_curr%codes list
  IF(ASSOCIATED(t_apiext_curr%codes)) THEN
    t_code_wk => t_apiext_curr%codes
    DO WHILE(ASSOCIATED(t_code_wk%next))
      t_code_wk => t_code_wk%next
    ENDDO
    t_code_wk%next => t_codebits_root
  ELSE
    t_apiext_curr%codes => t_codebits_root
  ENDIF

ENDIF

RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE IDC_attachAPIExtensionCodebits
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE IDC_switchProcessCompos(t_compo_root, t_process,           &
                                    n_newcategory1)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(COMPOINFO), POINTER     :: t_compo_root
TYPE(PROCESS), POINTER       :: t_process
INTEGER, INTENT(INOUT)       :: n_newcategory1

TYPE(COMPOINFO), POINTER     :: t_compo_work


INTEGER :: i
LOGICAL :: l_found_category1


IF ((.NOT. ASSOCIATED(t_compo_root)) .OR. (.NOT. ASSOCIATED(t_process))) RETURN

l_found_category1 = .FALSE.

DO i = 1, t_process%n_reactants
  t_compo_work => COMPOINFO_getNodeByName(t_compo_root, t_process%cr_reacts(i)%name)
  IF (t_compo_work%i_category == 1) THEN
    l_found_category1 = .TRUE.
    EXIT
  ENDIF
ENDDO

IF (.NOT. l_found_category1) THEN
  DO i = 1, t_process%n_products
    t_compo_work => COMPOINFO_getNodeByName(t_compo_root, t_process%cr_prods(i)%name)
    IF (t_compo_work%i_category == 1) THEN
      l_found_category1 = .TRUE.
      EXIT
    ENDIF
  ENDDO
ENDIF


IF (.NOT. l_found_category1) RETURN


DO i = 1, t_process%ratelaw%n_params

  IF (t_process%ratelaw%kindofparam(i) == 'io') THEN
    t_compo_work => COMPOINFO_getNodeByName(t_compo_root, &
                                            t_process%ratelaw%paramname(i))
    IF (t_compo_work%i_category /= 1) THEN
      t_compo_work%i_category = 1
      n_newcategory1 = n_newcategory1 + 1
    ENDIF

  ENDIF

ENDDO


RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE IDC_switchProcessCompos
!-----------------------------------------------------------------------



!-----------------------------------------------------------------------
 SUBROUTINE IDC_switchEquilibCompos(t_compo_root, t_equilib,           &
                                    n_newcategory1)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(COMPOINFO), POINTER     :: t_compo_root
TYPE(EQUILIB), POINTER       :: t_equilib
INTEGER, INTENT(INOUT)       :: n_newcategory1

TYPE(COMPOINFO), POINTER     :: t_compo_work


INTEGER :: i
LOGICAL :: l_found_category1


IF ((.NOT. ASSOCIATED(t_compo_root)) .OR. (.NOT. ASSOCIATED(t_equilib))) RETURN

l_found_category1 = .FALSE.

DO i = 1, t_equilib%n_reactants
  t_compo_work => COMPOINFO_getNodeByName(t_compo_root, t_equilib%cr_reacts(i)%name)
  IF (t_compo_work%i_category == 1) THEN
    l_found_category1 = .TRUE.
    EXIT
  ENDIF
ENDDO

IF (.NOT. l_found_category1) THEN
  DO i = 1, t_equilib%n_products
    t_compo_work => COMPOINFO_getNodeByName(t_compo_root, t_equilib%cr_prods(i)%name)
    IF (t_compo_work%i_category == 1) THEN
      l_found_category1 = .TRUE.
      EXIT
    ENDIF
  ENDDO
ENDIF


IF (.NOT. l_found_category1) RETURN


DO i = 1, t_equilib%eqlbrel%n_params

  IF (t_equilib%eqlbrel%kindofparam(i) == 'io') THEN
    t_compo_work => COMPOINFO_getNodeByName(t_compo_root, &
                                            t_equilib%eqlbrel%paramname(i))
    IF (t_compo_work%i_category /= 1) THEN
      t_compo_work%i_category = 1
      n_newcategory1 = n_newcategory1 + 1
    ENDIF

  ENDIF

ENDDO


RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE IDC_switchEquilibCompos
!-----------------------------------------------------------------------






!-----------------------------------------------------------------------
 SUBROUTINE IDC_switchSoluteSystemCompos(t_compo_root, t_solsys,       &
                                         n_newcategory1)
!-----------------------------------------------------------------------

IMPLICIT NONE

TYPE(COMPOINFO), POINTER     :: t_compo_root
TYPE(SYSTEMSINFO), POINTER   :: t_solsys
INTEGER, INTENT(INOUT)       :: n_newcategory1

TYPE(COMPOINFO), POINTER     :: t_compo_work


INTEGER :: i
LOGICAL :: l_found_category1


IF ((.NOT. ASSOCIATED(t_compo_root)) .OR. (.NOT. ASSOCIATED(t_solsys))) RETURN

                                    ! No need to continue if t_solsys
                                    ! already has category 1 -- all done
IF (t_solsys%i_category == 1) RETURN


l_found_category1 = .FALSE.

DO i = 1, t_solsys%n_members
  t_compo_work => COMPOINFO_getNodeByName(t_compo_root, t_solsys%member_name(i))
  IF (t_compo_work%i_category == 1) THEN
    l_found_category1 = .TRUE.
    EXIT
  ENDIF
ENDDO


IF (.NOT. l_found_category1) RETURN


DO i = 1, t_solsys%n_members

  t_compo_work => COMPOINFO_getNodeByName(t_compo_root, &
                                          t_solsys%member_name(i))
  IF (t_compo_work%i_category /= 1) THEN
    t_compo_work%i_category = 1
    n_newcategory1 = n_newcategory1 + 1
  ENDIF

ENDDO


t_solsys%i_category = 1


RETURN

!-----------------------------------------------------------------------
 END SUBROUTINE IDC_switchSoluteSystemCompos
!-----------------------------------------------------------------------



!=======================================================================
 END SUBROUTINE INIT_DATACHAINS
!=======================================================================

